/*
   Non-Midas test routine for ICE-PIC card

   Author: Jeff Schoen

*/

#define DEVN "ICE%s,DEVNO=%d,%s,IOM=NONE,MUXCLK=I,"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#include "icedefs.h"
#include "icelib.h"

#define FLAGS 0
#define CIRCTEST 0

int_4 testacq (PICSTRUCT *p, int_4 test);
int_4 testmap (PICSTRUCT *p);
int_4 testmtacq (PICSTRUCT *p);
int_4 testhealth (PICSTRUCT *p);
int_4 testdlbt (PICSTRUCT *p);

int main (int argc, char *argv[])
{
  PICSTRUCT ps,*p=&ps;
  char devname[200],name[8],flagstr[200],*fname,*func,zero[2]={"0"},blank[2]={""};
  int_4 flags = FLAGS;
  int_4 Mcbreak = 0;
  int_4 status,i,n;
  int_4 test,idev;

  if (argc <= 1) {
    BAIL:
    printf("Syntax:  test <devtype> <devno> <test> \n");
    printf("  <devtype> = PIC|NIC|UIC|RAM or DETECT  (this is NOT a Midas alias)\n");
    printf("  <devno>   = 0|1|2 ... (zero based index of each card in system)\n");
    printf("  <test>    = sniff|all|health|dmac|ply|map|acq|tacq|mtacq|dlbt|vhs|nvr|reset|checkflash|loadflash|setflash|#\n");
    printf("  <flags>   = VERBOSE|IOM=xxx|PM=xxx|...\n");
    exit(0);
  }
  for (i=0; argv[1][i]!=0; i++) {
    name[i] = argv[1][i];
    if (name[i]>'Z') name[i] -= 32;
  } name[i]=0;

  if (strcmp(name,"DETECT")==0) { pic_detect(0); goto DONE; }

  if (strstr("PIC,NIC,UIC,RAM",name)==0) { printf("Illegal devtype=%s Allowed=[PIC,NIC,UIC] \n",name); goto BAIL; }

  if (argc > 2 && strstr("0123456789",argv[2])==0) { printf("Illegal devno=%s Allowed=[0-9]\n",name); goto BAIL; }

  if (argc > 2) idev = atoi(argv[2]); else idev = 0;

  if (argc > 3) func = argv[3]; else func = zero;

  fname = blank;
  if (argc > 4) { 
   fname = argv[4];
   for (i=0; argv[4][i]!=0; i++) {
    flagstr[i] = argv[4][i];
    if (flagstr[i]>='a' && flagstr[i]<='z') flagstr[i] -= 32;
   } flagstr[i]=0;
  }
  else strcpy(flagstr,"NONE");

  if (strncmp(func,"load",4)==0 || strncmp(func,"init",4)==0) strcpy(flagstr,"NONE");

  sprintf(devname,DEVN, name, idev, flagstr);
  printf ("Starting ICE test=%s on dev=%s with flags=%s\n",func,devname,fname);
  status = pic_open (p, devname, &Mcbreak, flags);
  if (status<0) { printf ("Error opening device - tests aborted\n"); goto DONE; }

  if (func[0]>='0' && func[0]<='9') status = pic_test (p,atoi(func),1); 
  else if (strcmp(func,"all")==0) status = pic_test (p,0,1);
  else if (strcmp(func,"tacq")==0) status = testacq (p,-12);
  else if (strcmp(func,"xacq")==0) status = testacq (p,-11);
  else if (strcmp(func,"ply")==0) status = testacq (p,-10);
  else if (strcmp(func,"acq")==0) status = testacq (p,-9);
  else if (strcmp(func,"vhs")==0) status = testacq (p,-7);
  else if (strcmp(func,"dlbt")==0) status = testdlbt (p);
  else if (strcmp(func,"mtacq")==0) status = testmtacq (p);
  else if (strcmp(func,"reset")==0) status = pic_reset (p,0);
  else if (strcmp(func,"map")==0) status = testmap (p);
  else if (strcmp(func,"nvr")==0) status = pic_nvram (p,"",-1);
  else if (strcmp(func,"sniff")==0) status = pic_sniff (p,0);
  else if (strcmp(func,"health")==0) status = testhealth (p);
  else if (strcmp(func,"pcigen=1")==0) status = pic_setkeyl (p,0,KEY_PCIGEN,1);
  else if (strcmp(func,"pcigen=2")==0) status = pic_setkeyl (p,0,KEY_PCIGEN,2);
  else if (strncmp(func,"niop=",5)==0) {
    n = atoi(func+5);
    status = pic_setkeyl (p,0,KEY_NIOP,n);
  }
  else if (strncmp(func,"nvme=",5)==0) {
    n = atoi(func+5);
    status = pic_getkeyl (p,0,KEY_NVME);
    if (status!=n) {
      status = pic_setkeyl (p,0,KEY_NVME,n);
      pic_close (p);
      pic_open (p, devname, &Mcbreak, flags);
    }
    if (n>0) { 
      status = pic_reset (p,0); 
      if (status>=0) status = n;
      if (n&1) { i=1; pic_ioctl(p,IOCTL_RCD,p->devno,&i,4); }
      if (n&2) { i=2; pic_ioctl(p,IOCTL_RCD,p->devno,&i,4); }
    }
  }
  else if (strcmp(func,"dmac")==0) {
    i = (argc>4)? atoi(fname) : 0;
    status = pic_dmadump (p,i,0);
  }
  else if (strcmp(func,"status")==0) {
    pic_setkeyl (p,0,KEY_VERBOSE,1);
    status = pic_getkeyl (p,0,KEY_STATUS);
  }
  else if (strcmp(func,"loadvirtex")==0) {
    pic_setkeyl (p,0,KEY_VERBOSE,1);
    status = pic_loadfile (p,fname,FLG_VIRTEX);
  }
  else if (strcmp(func,"checkflash")==0) { 
    status = pic_reset (p, 0);
    pic_setkeyl (p,0,KEY_VERBOSE,1);
    status = pic_getkeyl (p,0,KEY_FLASH);
  }
  else if (strcmp(func,"loadflash")==0) {
    status = pic_reset (p, 0);
    printf("Do NOT interrupt this process ... you are reprogramming the boot loader.\n");
    pic_setkeyl (p,0,KEY_VERBOSE,1);
    status = pic_loadfile (p,fname,FLG_FLASH);
  }
  else if (strcmp(func,"initflash")==0) {
    status = pic_reset (p, 0);
    printf("Do NOT interrupt this process ... you are reprogramming the boot loader.\n");
    pic_setkeyl (p,0,KEY_VERBOSE,1);
    status = pic_loadfile (p,fname,FLG_VIRTEX);
    status = pic_loadfile (p,fname,FLG_FLASH);
  }
  else if (strcmp(func,"setflash")==0) status = pic_setkeyl (p,0,KEY_FLASH,999);
  else printf("Unsupported test=%s\n",func);

  printf ("Finished test status=0x%x\n",status);
  pic_close (p);
  DONE:
  if (status<0) exit(EXIT_FAILURE);
  else          exit(EXIT_SUCCESS);
}

int_4 testacq (PICSTRUCT *p, int_4 test) {
  int_4 dmac,nstart,nbytes,count,dir=-1;
  int_4 *data;
  int_4 i,status,pass,flags;
  DMAMAP map;

  /* reset the card */
  pic_reset (p, 0);

  flags = 0;
#if _IEEE
  flags |= FLG_BIGEND;
#endif

  /* load gate array code to gen test ramp on IOC 
     the default IOC code is loaded in the pic_reset() call 
     this call is ONLY for these special test cases */

  if (test == -7) {					/* vhs */
    pic_loadfile (p, "*_iirv", FLG_IOC);
    /* set I/O port for 32bit acquisition at 20MHz on Module 3 (1|2 mux) */
    dmac = pic_ioport (p, IOPT_MODULE, 3, -1, -1, -16, 20000000, 0.0,0,0,flags);
    count = 3*256*1024;
  } 
  else if (test == -8) {				/* 8e1 */
    pic_loadfile (p, "*_8e1t", FLG_IOC);
    flags |= FLG_DUAL;
    /* set I/O port for 1bit acquisition at 8x2MHz on Module 1 */
    dmac = pic_ioport (p, IOPT_MODULE, 1, -1, -1, 1, 16000000, 0.0,0,0,flags);
    count = 256*1024;
  } 
  else if (test == -9) {				/* acq */
    pic_loadfile (p, "*_iir", FLG_IOC);
    /* set I/O port for 32bit acquisition at 10MHz on Module 3 (1|2 mux) */
    dmac = pic_ioport (p, IOPT_MODULE, 3, -1, -1, -16, 10000000, 0.0,0,0,flags);
    count = 1024*1024;
  }
  else if (test == -10) {				/* ply */
    pic_loadfile (p, "*_oo", FLG_IOC);
    /* set I/O port for 32bit playback at 10MHz on Module 3 (1|2 mux) */
    dmac = pic_ioport (p, IOPT_MODULE, 3, -1, 1, -16, 10000000, 0.0,0,0,flags);
    count = 256*1024; dir=1;
  } 
  else if (test == -11) {				/* acq */
    pic_loadfile (p, "*_ii", FLG_IOC);
    /* set I/O port for 32bit acquisition at 10MHz on Module 1 */
    dmac = pic_ioport (p, IOPT_MODULE, 1, -1, -1, 16, 10000000, 0.0,0,0,flags);
    count = 1024*1024;
  }
  else if (test == -12) {				/* tuner */
    pic_loadfile (p, "*_iir", FLG_IOC);
    /* set I/O port for 16bit acquisition at 20MHz on Tuner 1 */
    dmac = pic_ioport (p, IOPT_TUNER, 1, -1, -1, 16, 20000000, 0.1,32,0,flags);
    count = 128*1024;
  } 
  if (dmac<=0) return -1;

  /* map the buffer to physical memory */
  nbytes = count*4;
  status = pic_mapmem (p, &map, nbytes, 1);
  if (status<0) return -1;

  /* setup the DMA handler */
  status = pic_dmasetup (p, dmac, dir, &map, -1, 0);
  data = (int_4 *)map.vaddr;

  for (i=0; i<count; i++) data[i]=i;

for (pass=0; pass<3; pass++) {

  status = pic_dmafunc (p, dmac, DMA_ONESHOT); 
  status = pic_dmafunc (p, dmac, DMA_WAIT); 
  status = pic_dmafunc (p, dmac, DMA_STOP); 
  status = pic_dmafunc (p, dmac, DMA_RESET); 

  printf("Data buffer { \n");
  for (i=0x000; i<0x008; i++) printf("%08x ",data[i]);
  printf("\n");
  for (i=0x008; i<0x010; i++) printf("%08x ",data[i]);
  printf("\n");
  for (i=0x010; i<0x018; i++) printf("%08x ",data[i]);
  printf("\n");
  for (i=0x018; i<0x020; i++) printf("%08x ",data[i]);
  printf("\n}\n");
}
  /* unmap the buffer to physical memory */
  status = pic_mapmem (p, &map, nbytes, -1);

  /* reset the card to reload the standard IOC code */
  pic_reset (p, 0);

  return 1;
}

int_4 testmap (PICSTRUCT *p) {
  int_4 i,j,nstart,*data,status,nper,nbytes=4*1024*1024,nloop=1000;
  char *buf1,*buf2;
  double t1,t2;
  DMAMAP map;
  FILE *fp;

  nloop  = pic_getintflagdef(p,"NLOOP",nloop);
  nbytes = pic_getintflagdef(p,"NTOT",nbytes);
  nper   = pic_getintflagdef(p,"NPER",nbytes);

  /* map the buffer to physical memory */
  status = pic_mapmem (p, &map, nbytes, 1);
  if (status<0) return -1;

  printf("MAP paddr=%08lx bytes=%08lx vaddr=%p   ntot=%d nper=%d \n",map.paddr,map.bytes,map.vaddr,nbytes,nper);

  buf1 = malloc(nbytes);
  buf2 = malloc(nbytes);

  t1=pic_time();
  for (i=0; i<nloop; i++) for (j=0; j<nbytes; j+=nper) memcpy(buf2+j,buf1+j,nper); 
  t2=pic_time()-t1;
  printf("Mem->Mem move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  t1=pic_time();
  for (i=0; i<nloop; i++) for (j=0; j<nbytes; j+=nper) memcpy(map.vaddr+j,buf1+j,nper); 
  t2=pic_time()-t1;
  printf("Mem->Buf move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  t1=pic_time();
  for (i=0; i<nloop; i++) for (j=0; j<nbytes; j+=nper) memcpy(buf2+j,map.vaddr+j,nper); 
  t2=pic_time()-t1;
  printf("Buf->Mem move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  t1=pic_time();
  for (i=0; i<nloop*2; i++) for (j=0; j<nbytes; j+=nper) memcpy(map.vaddr+j,map.vaddr+j+nper/2,nper/2); 
  t2=pic_time()-t1;
  printf("Buf->Buf move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  fp = fopen("/mnt/ramdisk","r");
  if (fp==NULL) goto DONE;
  fclose(fp);

  fp = fopen("/mnt/ramdisk/testfile.dat","w+");
  if (fp==NULL) { printf("Err opening ramdisk file\n"); goto DONE; }

  t1=pic_time();
  for (i=0; i<nloop; i++) { fseek(fp,0L,SEEK_SET); for (j=0; j<nbytes; j+=nper) fwrite (map.vaddr+j,nper,1,fp); }
  t2=pic_time()-t1;
  printf("Buf->File move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  t1=pic_time();
  for (i=0; i<nloop; i++) { fseek(fp,0L,SEEK_SET); for (j=0; j<nbytes; j+=nper) fread (map.vaddr+j,nper,1,fp); }
  t2=pic_time()-t1;
  printf("File->Buf move speed = %f MBy/s\n", 1e-6*nbytes*nloop/t2);

  fclose(fp);

  DONE:
  free(buf1);
  free(buf2);

  /* unmap the buffer to physical memory */
  status = pic_mapmem (p, &map, nbytes, -1);

  return 1;
}

#define MCHAN 32

int_4 maxamp (int_4 *data, int_4 n) {
  int_4 i; double a,b,c, cmax=0;
  for (i=0; i<n; i++) {
    a = (double) (data[i]);
    b = (double) (data[i]<<16);
    a /= 65536;
    b /= 65536;
    c = sqrt(a*a+b*b);
    if (c>cmax) cmax=c;
  }
  return (int)cmax;
}

/* Multiple Tuner Acquisition processor */
int_4 testmtacq (PICSTRUCT *p) {
  int_4 dmac,nstart,nbytes,bytes,n,ntun,mode;
  int_4 *data, dmaci[MCHAN];
  int_4 i,bits,dec,rate,status,index,count,flags;
  double freq;
  DMAMAP map[MCHAN];

  /* reset the card */
  pic_reset (p, 0);

  mode = pic_getintflag(p,"RESAMP");

  ntun = pic_getkeyl (p,0,KEY_CHNS);
  dec  = 64;
  bits = -16;
  rate = 40000000;
  bytes = 64*1024*4;
  freq = 0.1;

  pic_loadfile (p, "*_iir", FLG_IOC);

  for (n=0; n<ntun; n++) {
    if (n==0) flags = 0;		/* use 1st tuner as main side trigger - start last */
    else if (n==1) flags = FLG_SGO;	/* use 2nd tuner as slave side trigger - start next to last */
    else flags = FLG_RGO;		/* make all the rest non-triggers */
    if (mode==1) flags=0;
    dmac = pic_ioport (p, IOPT_TUNER, n+1, -1, -1, bits, rate, freq, dec, 0, flags);
    if (dmac<=0) { printf("Could not setup IO port for tuner %d\n",n+1); return -1; }
    status = pic_mapmem (p, map+n, bytes, 1); 		/* alloc and map a physical buffer */
    if (status<0) { printf("Could not mapmem for tuner %d\n",n+1); return -1; }
    status = pic_dmasetup (p, dmac, -1, map+n, -1, 0);	/* init DMA structs for this channel */
    if (status<0) { printf("Could not dmasetup for tuner %d\n",n+1); return -1; }
    dmaci[n]=dmac;
  } 

if (mode == 1) {
  printf("Setting resamplers in backward order\n");
  for (n=ntun-1; n>=0; n--) {
    status = pic_setkeyd (p, dmaci[n], KEY_FREQ, freq); 
    status = pic_setkeyl (p, dmaci[n], KEY_DEC, dec); 
    pic_loadfile (p, "pfir_150", FLG_FC|(n+1));
    status = pic_setkeyd (p, dmaci[n], KEY_RATIO, 1.1); 
    pic_loadfile (p, "res_15x32_80", FLG_FC|(n+1));
  }
}

  printf("Starting %d tuners in backward order\n",ntun);
  for (n=ntun-1; n>=0; n--) {
    status = pic_dmafunc (p, dmaci[n], DMA_CONTINUOUS); 
  }
  printf("Started %d tuners\n",ntun);

  for (i=0; i<5; i++) {
    usleep(1000000);
    printf("Channel check loop=%d\n",i+1);
    for (n=0; n<ntun; n++) {
      status = pic_dmastat (p, dmaci[n], &index, &count); 
      printf("Chan %d at stat=%d cycle=%d index=%d amp=%d\n",n+1,status,count,index,maxamp((int_4 *)map[n].vaddr,32));
    }
  }

  printf("Stopping %d tuners\n",ntun);
  for (n=0; n<ntun; n++) {
    status = pic_dmafunc (p, dmaci[n], DMA_STOP); 
  }

  printf("Unmapping %d tuners\n",ntun);
  for (n=0; n<ntun; n++) {
    status = pic_dmafunc (p, dmaci[n], DMA_CANCEL); 
    status = pic_mapmem (p, map+n, bytes, -1);
  }
  printf("Resetting card\n");

  /* reset the card to reload the standard IOC code */
  pic_reset (p, 0);

  return 1;
}

/* Dual Loop Back Tuner processor */
#define ISIZE 3145728
#define OSIZE 20480
#define IDEC  100
#define IRATE 10000000

int_4 testdlbt (PICSTRUCT *p) {
  int_4 dmaci1,dmaci2,dmaco1,dmaco2;
  DMAMAP mapi1,mapi2,mapo1,mapo2;
  int_4 i,j,status,flags=0,*data;
  double freq;

  /* reset the card */
  pic_reset (p, 0);
  pic_loadfile (p, "*_oo", FLG_IOC);

  /* setup modules for tuner input buffers */
  dmaci1 = pic_ioport (p, IOPT_MODULE, 1, -1,  1, 16, IRATE, 0.0, 1, 0, flags);
  if (dmaci1<=0) { printf("Could not setup IO port for module 1\n"); return -1; }
  status = pic_mapmem (p, &mapi1, ISIZE, 1);
  if (status<0) { printf("Could not mapmem for module 1\n"); return -1; }
  status = pic_dmasetup (p, dmaci1, 1, &mapi1, -1, 0);
  if (status<0) { printf("Could not dmasetup for module 1"); return -1; }

  /* setup tuners for output buffers */
  dmaco1 = pic_ioport (p, IOPT_TUNER, 1, -1, -1, -16, IRATE, 0.0, IDEC, 0, flags);
  if (dmaco1<=0) { printf("Could not setup IO port for tuner 1\n"); return -1; }
  status = pic_mapmem (p, &mapo1, OSIZE, 1);
  if (status<0) { printf("Could not mapmem for tuner 1\n"); return -1; }
  status = pic_dmasetup (p, dmaco1, -1, &mapo1, -1, 0);
  if (status<0) { printf("Could not dmasetup for tuner 1"); return -1; }

  for (i=0; i<3; i++) {

    /* load buffers */
    data = (int_4*)mapi1.vaddr;
    for (j=0; j<ISIZE/4; j++) data[j]=j;
  
    /* set frequencies */
    freq = 123.4 / (IRATE/2);
    pic_setkeyd (p, dmaco1, KEY_FREQ, freq);

    printf("Started pass #%d\n",i);
    pic_dmafunc (p,dmaco1,DMA_ONESHOT);
    pic_dmafunc (p,dmaci1,DMA_ONESHOT);

    pic_dmafunc (p,dmaci1,DMA_WAIT);
    pic_dmafunc (p,dmaco1,DMA_WAIT);

    pic_dmafunc (p,dmaci1,DMA_CANCEL);
    pic_dmafunc (p,dmaco1,DMA_CANCEL);

    pic_dmafunc (p,dmaci1,DMA_RESET);
    pic_dmafunc (p,dmaco1,DMA_RESET);
    printf("Finished pass #%d\n",i);

    /* unload buffers */
    data = (int_4*)mapi1.vaddr;

  }

  /* reset the card to reload the standard IOC code */
  status = pic_mapmem (p, &mapi1, ISIZE, -1);
  status = pic_mapmem (p, &mapo1, OSIZE, -1);
  pic_reset (p, 0);

  return 1;
}

#define SPEC_SIZE 0x00100000
#define SPEC_BLKX 0x00001000
#define SPEC_BLKY 0x00010000

int_4 testhealth (PICSTRUCT *p) {
  int_4 i,j, npass,ndata,nstart,nblock,nbytes,maxdata,maxsjump,maxbjump;
  float maxsrate,maxbrate,maxrate,minrate,minthresh,maxthresh;
  int_4 status=0, *data;
  status = pic_reset (p,0);
  if (status<0) { printf("Status=-1\n Reset Error\n"); return -1; }
  ndata  = pic_getintflagdef(p,"SIZE",SPEC_SIZE);
  nblock = pic_getintflagdef(p,"BLOCK",p->isY? SPEC_BLKY:SPEC_BLKX);
  npass  = pic_getintflagdef(p,"PASS",200);
  minthresh= pic_getintflagdef(p,"MINRATE",1500);
  maxthresh= pic_getintflagdef(p,"MAXRATE",1700);
  nbytes = ndata * 4;
  status = pic_map (p, &data, &nstart, nbytes, 1);
  if (status<0) { printf("Status=-2\n Map Error\n"); return -2; }
  for (i=0; i<npass; i++) {
    pic_spec (p, data, nstart, ndata, nblock, &maxdata, &maxsrate, &maxsjump, &maxbrate, &maxbjump, 0);
    if (i==0 || maxbrate>maxrate) maxrate=maxbrate;
    if (i==0 || maxbrate<minrate) minrate=maxbrate;
  }
  status = pic_map (p, &data, &nstart, nbytes, -1);
  /* make sure rates are OK */
  status = 0;
  if (minrate<minthresh) status = -3;
  if (maxrate>maxthresh) status = -4;
  if (minrate<minthresh && maxrate>maxthresh) status = -5;
  printf("Status=%d MaxRate=%f MinRate=%f PciDbg=%08x\n",status,maxrate,minrate,pic_getkeyl(p,-2,KEY_PCINACK));
  return status;
}
