package nxm.ice.prim;

import java.awt.Button;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Graphics;
import java.util.Vector;
import java.text.DecimalFormat;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libg.*;
import nxm.sys.lib.GPrimitive;

/*
  Displays Midas data in the form of textual lists. 

  If a template is not given, uses datalist formatting.

   @author Jeff Schoen
*/
public class view extends GPrimitive implements Keyable {

  private int controls=1, scale=0, mode=0, view=0, type=0;
  private int realtime=0, hilite=1, exit=-1, poll=0, flags=0;
  private int outtype=1, settings=-1, curmode=0, colrIdx=1;
  private long polltime = System.currentTimeMillis();

  // data content objects
  private DataFile hcb;
  private Object obj;
  private Table tab;
  // data template objects
  private Keywords tkw;
  private Table ttbl;
  // data buffers
  private Data data,datb;
  private KeyVector kv;
  private Table kvo;

  private double offset,size;
  private boolean rescale=false,isReadyF,isReadyT,isEdit,isEditing;
  private String form;
  private String currentName;
  private String templateName;
  private int th=12,ta=16,tw=8,tv;	// text height and width
  private int selRow,selCol;		// current selected cell
  private int curRow,curCol;		// current pointer cell
  private int numRows,numCols;		// config rows and columns
  private int nc,ds,dh,bh,ch,cw;
  private int sx,sw=6,sy,sh;
  private Color cbg0,cbg1,cbg2,cbs0,cbs1,cwfh;	// background and shadow colors
  private boolean showLN,useTemplate,useDataList,useTable,useAuto,isTable,isFile;
  private GWidget gedit;

  static final String exitList = "RETURN,MENU,MESSAGE";
  static final int EXIT_RETURN=0x1, EXIT_MENU=0x2, EXIT_MESSAGE=0x4, EXIT_ALL=0xF;
  static final int ONCLICK=0x1, ONMOTION=0x2;
  static final int T_CELL=1,T_ROW=2,T_COL=3;
  static final int GRID=0x1,FILL=0x2,SCROLL=0x4,CENTER=0x8,HEADER=0x10,NUMBER=0x20;
  static final int SDEF_FILE=(GRID|SCROLL|CENTER|HEADER|NUMBER);
  static final int SDEF_TABLE=(SCROLL|CENTER);
  static final String settingsList="Grid,Fill,Scroll,Center,Header,Number";

  @Override
  public int open() {

    hilite = MA.getSelectionIndex("HILITE","BACK,FORE",1);
    controls = MA.getSelectionIndex("CNT=","CLICK,CONT",1);
    poll = MA.getI("/POLL",(short)0);
    curmode = MA.getSelectionIndex("CURSOR","CELL,ROW,COL",T_CELL);
    if (curmode == -1) curmode = T_CELL;
    realtime = MA.getL("/RT",0);
    if (M.pipeMode==Midas.PINIT) exit=0; else exit=-1;
    exit = MA.getOptionMask("/EXIT",exitList,exit);

    if (MA.getState("/HEX")) flags |= BaseFile.RADIX_16;
    else if (MA.getState("/BIN")) flags |= BaseFile.RADIX_2;
    showLN = MA.getState("/L",true);
    isEdit = MA.getState("/EDIT",false);

    MW = new MWView("View",this);
    MW.open();  //set colors and fonts
    MW.setSize(800,500);
    MW.addTo(this);

    if (MW.fm != null) { // Since NeXtMidas 3.5.0. Skip if in Headless state. BUG 2817
      tw = MW.fm.charWidth('0');
      th = MW.fm.getHeight();
      ta = MW.fm.getAscent();
    }
    bh = 4;		// bottom spacing
    dh = 0;		// top spacing
    cw = tw;
    ch = dh+th+bh-2;
    tv = tw/2;
    cbg0 = MW.theme.cwms;
    cbs0 = MW.theme.cwbs; 
    cbg1 = MW.theme.cwbs;
    cbs1 = MW.theme.cwts; 
    cbg2 = MW.theme.cwbs.brighter();
    cwfh = MW.theme.cwfh;
    selRow = 0;
    selCol = 0;

    openTemplateFile(MA.getCS("TEMPLATE"));
    openAny(MA.getCS("FILE"));

    refresh(1);

    return NORMAL;
  }

  private boolean isVisible() {
    if (MW.panel==null) return false;
    java.awt.Frame frame = javax.swing.JOptionPane.getFrameForComponent(MW.panel);
    if (!MW.panel.isShowing()) return false; 
    if ((frame != null) && (frame.getExtendedState() == Frame.ICONIFIED)) return false;
    return true;
  }

  @Override
  public int process () {
    if (MW.status==MWindow.INIT)  return NOOP;	// list not up yet
    if (checkPollTime()) { refresh(); return NORMAL; }
    return NOOP;
  }

  @Override
  public int close() {
    if (hcb!=null) hcb.close();
    MW.close();
    return NORMAL;
  }

  private boolean checkPollTime() {
    if (poll<=0) return false;
    long ctm = System.currentTimeMillis();
    if ( (ctm-polltime) > (poll*1000) ) { polltime=ctm; return true; }
    return false;
  }

  private boolean autoConfig() {
    int onc = nc, onr = numRows;
    applyTemplate(); 
    return (isTable || nc!=onc || numRows!=onr);
  }

  /** Open the file to list with a refresh on clear */
  private void openAny (String name) { 
    Object obj = MR.getO(name.toUpperCase());		// check for name in results table 
    if (name.endsWith(".tbl")) obj = openTableFromFile(name);
    if (obj instanceof Table) openTable((Table)obj);
    else openFile (name);
  }

  private Table openTableFromFile (String name) {
    TextFile tf = new TextFile(M,name);
    return new Table(tf);
  }

  private void openTable (Table t) {
    tab = t;
    isTable=true;
    isFile=false;
    parseTable();
    applyTemplate();
    isReadyF=true;
    refresh();
  }
  private void closeTable () {
    tab = new Table();
  }
  private void parseTable() {
    if (kv==null) kv = new KeyVector();
    if (kvo==null) kvo = new Table();
    kv.clear();
    addTable(tab,0,"");
    size = kv.getSize();
  }
  private void addTable (Table t, int level, String prefix) {
    String pad="";
    for (int i=0; i<level; i++) pad+="  ";
    for (Table.Iterator ti=t.iterator(); ti.getNext(); ) {
      if (ti.value instanceof Table) {
	prefix += ti.key;
	Table tbl = (Table)ti.value;
	boolean topen = kvo.containsKey(prefix);
	kv.add(pad+ti.key+"@"+prefix,tbl);
	String toStr = tbl.getS("TOSTR");
	if (toStr!=null) kvo.put(prefix+"_TOSTR",toStr);
	if (topen) addTable (tbl,level+1,prefix);
      } else if (!ti.key.equals("TOSTR")) {
	kv.add(pad+ti.key,ti.value);
      }
    }
  }

  /** Open the file to list with option to refresh on clear */
  private void openFile (String name) {

    isReadyF=false;
    closeFile();				// only 1 file can be open
    if (!name.equals(currentName)) offset=0;	// not just a REOPEN
    isTable=false;
    isFile=true;

    hcb = new DataFile();
    hcb.init (M,name);
    int ioflg = isEdit? BaseFile.INOUT : BaseFile.INPUT;
    if (hcb.open(ioflg|BaseFile.OPTIONAL)) {
      size = hcb.size;
      data = hcb.getDataBuffer(1);
      datb = hcb.getDataBuffer(1);
      form = hcb.getFormat();
      autoConfig();
      isReadyF=true;
    }
    refresh();
  }

  private boolean is (int mask) {
    return (settings & mask) != 0;
  }
  private void getSettings() {
    if (settings>=0) return;
    settings = isTable? SDEF_TABLE : SDEF_FILE;
    settings = MA.getOptionMask("/SETTINGS",settingsList,settings);
  }

  private void openTemplateFile (String filename) {

    tkw=null;
    ttbl=null;
    isReadyT = false;
    templateName = filename.toLowerCase();

    useAuto     = templateName.equals("auto");
    useTable    = templateName.endsWith(".tbl");
    useDataList = templateName.equals("");
    useTemplate = !useDataList && !useAuto && !useTable;

    if (useTemplate && tkw==null) {
      DataFile thcb = new DataFile();
      thcb.init (M,templateName);
      thcb.open();
      tkw = thcb.keywords;
      thcb.close();
    }
    if (useTable && ttbl==null) {
      ttbl = Convert.o2t(templateName,this);
    }
  }

  private void applyTemplate() {

    isReadyT = false;
    getSettings();
    numRows = MW.pos.h/ch;
    if (is(HEADER)) numRows--;

    if (isTable) {
      numCols = 1;
    } else if (useDataList) {
      numCols = 1;
    } else if (useAuto) {
      numCols = hcb.getSubSize();
    } else if (useTable) {
      numCols = ttbl.getL("FIELDS");
    } else {
      tkw.setScope("SECTION=XDATALIST");
      numCols = tkw.getL("FIELDS",0);
    }
    if (columns==null || (numCols+1)>columns.length) columns = new Column[numCols+1];
    int px = 0;
    int colw0 = (useDataList||useAuto)? 5 : 3;
    boolean center = is(CENTER);
    boolean numb   = is(NUMBER);
    boolean scroll = is(SCROLL);
    nc = 1;
    String cname = "CFG";
    for (int c=0; c<=numCols; c++) {
      Column col = new Column();
      if (c==0 && !numb) {
	col.width = 0;
      } else if (c==0 && (isTable||useDataList||useAuto||useTable)) {
	col.label = "#";
	col.width = colw0;
      } else if (isTable) {
	col.width = -1;
      } else if (useDataList) {
	col.width = -1;
	if (hcb!=null && hcb.isOpen) {
	  nc = hcb.listElementsPerLine((MW.pos.w/cw)-(showLN?6:1),form,flags);
	  if (nc<1) nc = 1; nc = MA.getL("/NC",nc);
          col.label = "FILE="+hcb.getURL()+" TYPE="+hcb.getType()+" FORM="+hcb.getFormat()+" SIZE="+hcb.getSize();
	}
      } else if (useAuto) {
	String name = hcb.getRecName(c-1);
	String form = hcb.getRecFormat(c-1);
	char type = form.charAt(1);
	col.subr = name;
	col.label = name.substring(0,1)+name.substring(1).toLowerCase();
	if (type=='A') col.form = "ASCII";
	else if (type=='F'||type=='D') col.df = new DecimalFormat("#.000");
	else col.df = new DecimalFormat("##");
	col.width = 8;
	col.soff  = -1;
	col.center= center;
      } else {
	if (useTable && ttbl.getO("FIELD"+c)!=null) {
	 Table ftbl = ttbl.getTable("FIELD"+c);
	 col.subr  = ftbl.getS("SUBR","NONE").trim();
	 col.form  = ftbl.getS("DISPFORM","ASCII").trim();
	 col.label = ftbl.getS("LABEL","").trim();
	 col.width = ftbl.getL("WIDTH",colw0);
	 col.items = ftbl.getS("ITEMS");
	 cname     = ftbl.getS("COLOR","CFG").trim();
	} else if (useTemplate && tkw.setScope("SECTION=XDATALIST,FIELD="+c)==2) {
	 col.subr  = tkw.getS("SUBR","NONE").trim();
	 col.form  = tkw.getS("DISPFORM","ASCII").trim();
	 col.label = tkw.getS("LABEL","").trim();
	 col.width = tkw.getL("WIDTH",colw0);
	 cname     = tkw.getS("COLOR","CFG").trim();
	 col.items = tkw.getS("ITEMS",null);
	} else {
	 col.form  = "ASCII";
	 col.label = "";
	 col.width = colw0;
	}
	col.soff  = -1;
	col.center= (c>0) && center;
	if (col.form.equals("TOGGLE")) { 
	  col.toggle = true; col.colors = new Color[3]; col.letters = new String[3];
	  col.letters[0] = "-"; col.colors[0] = Color.RED.darker(); 
	  col.letters[1] = "*"; col.colors[1] = Color.GREEN.darker();
	  col.letters[2] = "?"; col.colors[2] = Color.BLUE.darker();
	}
	else if (col.form.equals("METER")) col.meter=true;
	else if (col.form.equals("MENU")) col.menu=true;
	else if (col.form.equals("ASCII"));
	else if (col.form.indexOf('#')>=0) col.df = new DecimalFormat(col.form);
	else     col.mf = MFormat.getNumberFormatFor(col.form);
      }
      Color color = MW.theme.getColor(cname);
      if (color==null) color = MColor.getColor(cname);
      col.color = color;
      col.pw = (col.width>=0)? (col.width+1)*cw : MW.pos.w - px - sw;
      if (c==0 && col.width==0 && scroll) col.pw = sw+cw;
      col.px = px;
      px += col.pw;
      columns[c] = col;
    }
    isReadyT = true;
  }

  private void closeFile () {
    isReadyF=false;
    if (hcb!=null && hcb.isOpen()) hcb.close();
    refresh();
  }

  /** Pad an int type, cast from a double, just add a : if too large */
  private String padi (int off, int w) {
    if (!showLN) return " ";
    String padoff = ""+off; w--;
    while (padoff.length()<w) padoff = " "+padoff;
    return padoff;
  }

  private class Column {
    int px,pw,width,soff,sbpa;
    String subr,form,label,sfmt,items;
    Color color,dcolor,colors[];
    String letters[];
    char type;
    DecimalFormat df;
    MFormat mf;
    boolean toggle,menu,meter,center;
  }
  private Column[] columns;

  private String formatCell (Data data, int ic) {
    Column col = columns[ic];
    if (col.soff<0) {
      int fn = hcb.findRec(col.subr);
      if (fn<0) M.warning("Subrecord "+col.subr+" not found in file: "+hcb.getURL());
      col.soff = hcb.getRecOffset(fn);
      col.sfmt = hcb.getRecFormat(fn);
      col.sbpa = Data.getBPA(col.sfmt);
      col.type = col.sfmt.charAt(1);
    }
    Object ocv = getValue(data,ic);
    if (col.toggle) {
      int state = (col.type!='A')? Convert.o2l(ocv) : ocv.toString().equalsIgnoreCase("On")? 1 : 0;
      if (state<0 || state>2) state=0;
      col.dcolor = col.colors[state];
      return col.letters[state];
    }
    if (col.mf != null || col.df != null) {
      double d=Convert.o2d(ocv);
      if (col.df!=null) return col.df.format(d);
      return col.mf.format(Double.valueOf(d));
    }
    if (col.form.equals("ASCII") || col.form.equals("MENU")) {
      int ls=0; for (;data.buf[col.soff+ls]>32 && ls<col.sbpa; ls++);
      return new String(data.buf,col.soff,ls);
    }
    return null;
  }

  private Object getValue (Data data, int c) {
    Column col = columns[c];
    switch (col.type) {
      case 'D': return Double.valueOf  ( Convert.unpackD(data.buf,col.soff) );
      case 'F': return Float.valueOf   ( Convert.unpackF(data.buf,col.soff) );
      case 'L': return Integer.valueOf ( Convert.unpackL(data.buf,col.soff) );
      case 'I': return Short.valueOf   ( Convert.unpackI(data.buf,col.soff) );
      case 'B': return Byte.valueOf    ( Convert.unpackB(data.buf,col.soff) );
      case 'A': return                   Convert.unpackS(data.buf,col.soff,col.sbpa);
    }
    return null;
  }

  private Object getValue (int r, int c) {
    if (isTable) return getTableLineAll(r-1);
    if (useDataList) return null;
    if (r<=0) return null;
    double index = r-1;
    if (index<0 || index>=size) return null;
    if (hcb==null || !hcb.isOpen) return null;
    if (hcb.read(datb,index,1)!=1) return null;
    if (c>0) return getValue(datb,c);
    Table t = new Table();
    for (c=1; c<=numCols; c++) {
      Column col = columns[c];
      t.put(col.subr,getValue(datb,c));
    }
    return t;
  }

  private void setValue (int r, int c, Object data) {
    if (useDataList) return;
    double index = r-1;
    if (index<0 || index>=size) return;
    if (hcb==null || !hcb.isOpen) return;
    if (hcb.read(datb,index,1)!=1) return;
    Column col = columns[c];
    if (col.type=='D') Convert.packD(datb.buf,col.soff, Convert.o2d(data));
    if (col.type=='F') Convert.packF(datb.buf,col.soff, Convert.o2f(data));
    if (col.type=='L') Convert.packL(datb.buf,col.soff, Convert.o2l(data));
    if (col.type=='A') Convert.packS(datb.buf,col.soff, col.sbpa, data.toString());
    if (hcb.write(datb,index,1)!=1) return;
  }


  private class MWView extends MWindow {
  public MWView (String name, MessageHandler mh) {
    super(name, mh,true);
  }
  public synchronized void paintComponent (Graphics g) { 
    int ls,lx,lw,pw,px=0,py=0; String s;
    if (state!=PROCESS) return;
    if (!isReadyF || !isReadyT) {
      g.setColor(cbg1);
      g.fillRect(0,0,pos.w,pos.h);
      return;
    }
    boolean fill = is(FILL);
    boolean grid = is(GRID);
    boolean scroll = is(SCROLL);
    boolean head = is(HEADER);
    boolean numb = is(NUMBER);
    int sr = head? 0 : 1, hr = head? 1 : 0;
    long aoff = ((long)offset/1000)*1000;
    for (int r=sr; r<=numRows; r++) {
      boolean labels = (r==0);
      double index = offset+(r-1)*nc;
      long arow=-1,brow=-1;
      if (!labels) {
        if (index>=size) break;
        brow = (long)(index+1);
        arow = (long)(index+1);
	if (isTable) {
        }
	else if (useDataList) { 
	  if (hcb==null || !hcb.isOpen || state!=PROCESS) break;
	  arow = (long)index;
	} else {
	  if (hcb==null || !hcb.isOpen || state!=PROCESS) break;
	  if (hcb.read(data,index,1)!=1) break;
        }
      }
      for (int c=0; c<=numCols; c++) {
        Column col = columns[c];
	boolean border = (c==0) || (r==0);
        boolean center = labels || col.center;
	boolean selected = (selRow==brow || selRow==0) && (selCol==c || selCol==0) && !(selRow==0 && selCol==0);
        px = col.px;
        pw = col.pw;
	g.setColor(border?cbg0:selected?cbg2:cbg1);
	g.fillRect(px,py,pw,ch);
	if (c==0 && !numb) continue;
	s  = null;
	     if (r==0) s = (c==0 && aoff>0)? ""+aoff+"+" : col.label;
	else if (c==0) s = padi((int)(arow-aoff),col.width);
	else if (isTable) s = getTableLine((int)index);
	else if (useDataList) s = hcb.listElements (index,nc,form,flags);
	else s = formatCell(data,c);
	if (s!=null) {
	  if (r==0 && c==0) pw-=sw;
	  ls = s.length();
	  lx = center? ((pw-ls*cw)>>1) : tv; if (lx<0) lx=0;
	  lw = useDataList? ls*tw : pw-1;
	  g.setColor(cbs1);
	  if (grid && !border && c!=numCols) g.drawLine(px+lw,py+1,px+lw,py+ch-2);
	  if (grid && !border) g.drawLine(px+1,py+ch-1,px+lw-2,py+ch-1);
	  g.setColor( selected? cwfh : (col.toggle&&!border)? col.dcolor : col.color);
	  g.drawString(s,px+lx,py+ch-bh);
	}
        px += col.pw; 
      }
      py += ch;
    } 
    g.setColor(fill?cbg0:cbg1);
    if (px<pos.w) g.fillRect(px,0,pos.w-px,pos.h);
    if (py<pos.h) g.fillRect(0,py,px,pos.h-py);
    g.setColor(cbs1);
    g.drawLine(0,0,px,0);	// top hishade
    g.drawLine(0,0,0,py);	// left hishade
    g.setColor(fill?cbg2:cbg0);
    g.drawLine(0,py,px,py);	// bottom loshade
    g.drawLine(px,0,px,py);	// right loshade
    if (scroll) {
      Column col = columns[0];
      sx = col.px+col.pw-sw; sy = ch; sh = py-ch;
      drawVSlider(sx,sy,sw,sh, offset/Math.max(1,size),0, g,theme);
    }
  }
  }

  public void drawSymbol (String s, int sx, int sy, int sr, Graphics g) {
    if (s.equals("*")) g.drawOval(sx,sy,sr,sr);
    if (s.equals("-")) g.drawRect(sx,sy,sr,sr);
  }

  private void refresh () {
    refresh(0);
  }

  private void refresh (int flag) {
    MW.refresh();
  }

  private void resize () {
    MW.resize(1);
    if (autoConfig()) refresh();
  }

  private void getCurrent () {
    curCol = -1;
    for (int c=0; c<=numCols; c++) {
      Column col = columns[c];
      if (MW.px>col.px && MW.px<col.px+col.pw) curCol = c;
    }
    curRow = MW.py/ch;
    if (!is(HEADER)) curRow++;
    if (curRow>0) curRow += (int)offset;
    if (curRow>size) curRow = -1;
  }

  private void select() {
    getCurrent();
    if (curRow<0 || curCol<0) return;
    select(curRow,curCol,true);
  }

  private void select (int r, int c) { 
    select(r,c,false);
  }

  private void editCell (int r, int c) {
    Column col = columns[c];
    Object ocv = getValue((r==0)?1:r,c);
    Object val;
    if (col.toggle) {
      if (col.type=='A') {
        val = ocv.toString().equalsIgnoreCase("On")? "Off" : "On";
      } else {
	int lval = Convert.o2l(ocv);
	val = Convert.l2o(1-lval);
      }
      if (r>0) setValue(r,c,val);
      else for (int i=1; i<=size; i++) setValue(i,c,val);
      refresh();
      return;
    }
    if (isEditing) { isEditing=false; return; }
    isEditing=true;
    String name = "Edit-"+r+"-"+c+"-"+col.subr;
    char type = col.type;
    int flags = 0;
    if (col.menu) {
      flags |= GMenu.CS;
      gedit = new GMenu (MW,name,col.items,0,flags,this);
    } else if (type=='L' || type=='F' || type=='D' || type=='T') {
      double v = Convert.o2d(ocv);
      gedit = new GValue (MW,name,v,1.0,-1.0,1.0,type,flags,this);
    } else {
      String v = ocv.toString();
      gedit = new GPrompt (MW,name,v,flags,this);
    }
  }

  private void sendCell (int r, int c, String name) {
    Table t = new Table();
    t.put("TYPE",(c==0)?"ROW":(r==0)?"COL":"CELL"); 
    t.put("ROW", Integer.valueOf(r) );   
    t.put("COLUMN", Integer.valueOf(c)); 
    t.put("NAME",columns[c].subr);
    t.put("VALUE",getValue(r,c));
    sendMessage(name,0,t);
  }

  private void select (int r, int c, boolean smsg) {
    selRow = r;
    selCol = c;
    if (selRow>=offset+numRows) offset = Math.max(0,Math.min(size,r)-numRows+1);
    if (selRow<offset+2) offset = Math.max(0,r-numRows/2);
    offset = Math.floor(offset/nc)*nc;
    refresh();
    if (smsg) sendCell(r,c,"SELECT");
  }

  private void deselect (boolean smsg) {
    isEditing=false;
    select(0,0,smsg);
  }

  private String getTableLine (int kvi) {
    if (kvi<0 || kvi>=size) return "Oops";
    String key = kv.getKey(kvi);
    Object val = kv.get(kvi);
    String text,pre;
    if (val instanceof Table) {
      Table t = (Table)val;
      int ia = key.indexOf('@');
      String prefix = key.substring(ia+1);
      key = key.substring(0,ia);
      boolean topen = kvo.containsKey(prefix);
      text = kvo.getS(prefix+"_TOSTR");
      if (text==null) text = "Table of "+t.getSize()+" entries";
      pre = topen? "- " : "+ ";
    } else {
      text = val.toString();
      pre = "  ";
    }
    return pre+key+" : "+text;
  }

  private String getTableLineAll (int kvi) {
    if (kvi<0 || kvi>=size) return "Oops";
    String key = kv.getKey(kvi);
    Object val = kv.get(kvi);
    String text;
    if (val instanceof Table) {
      Table t = (Table)val;
      int ia = key.indexOf('@');
      String prefix = key.substring(ia+1);
      key = key.substring(0,ia);
      text = t.toString();
    } else {
      text = val.toString();
    }
    return key+":"+text;
  }

  private void handleTableHit (int row) {
    int kvi = row-1;
    if (kvi<0 || kvi>=size) return;
    String key = kv.getKey(kvi);
    Object val = kv.get(kvi);
    if (val instanceof Table) {
      Table t = (Table)val;
      int ia = key.indexOf('@');
      String prefix = key.substring(ia+1);
      if (kvo.containsKey(prefix)) kvo.remove(prefix);
      else kvo.put(prefix,"Open");
      parseTable();
      applyTemplate();
    }
  }

  private void scroll (int rows) {
    double loffset = offset;
    offset = Math.max(0.0, Math.min( size+2*nc-Math.floor(numRows*nc), Math.floor(offset+rows)));
    offset = Math.floor(offset/nc)*nc;
    if (offset!=loffset) refresh();
  }

  private void getHit() { getHit(MW.px,MW.py,0,0,false); }
    
  private void getHit (int px, int py, int dx, int dy, boolean drag) {
    if (px>=sx && px<=sx+sw && py>=sy && py<sy+sh) {	// ScrollBar
      offset = (py+dy-sy)*size/sh;
      offset = Math.max(0.0, Math.min( size+2*nc-Math.floor(numRows*nc), Math.floor(offset)));
      offset = Math.floor(offset/nc)*nc;
    } else if (!drag) {					// Cell | Row | Column
      getCurrent();
      if (curRow<0 || curCol<0) return;
      if (isTable && curRow>=0 && curCol>0 && curRow==selRow && curCol==selCol) {
	handleTableHit(curRow);
      }
      else if (isEdit && curRow>=0 && curCol>0 && curRow==selRow && curCol==selCol) {
        editCell(curRow,curCol);
      }
      else {
	isEditing=false;
        select();
      }
    }
    refresh();
  }

  /** Process a message */
  @Override
  public int processMessage (Message msg) {

    /* if initializing or this Thread is not me - queue for me */
    if (MW==null || MW.status==0 || !thisIsMe()) {
      if (msg.name.equals("POINTER")) MQ.remove(msg.name);
      if (msg.name.equals("REFRESH") || msg.name.equals("RESIZE")) MQ.remove(msg.name);
      if (msg.name.startsWith("PAN") || msg.name.startsWith("DRAG")) MQ.remove(msg.name);
      MQ.put(msg);
      return NOOP;
    }
    // convert equivalences
    if (msg.name.equals("KEYPRESS")) {
      String text = (String)msg.data;
      if (text.equalsIgnoreCase("M")) msg.name = "MENU";
      if (text.equalsIgnoreCase("Enter") && (exit&EXIT_RETURN)!=0) msg.name = "EXIT";
    }
    if (msg.name.equals("BUTTON") && msg.info==2) msg.name = "MENU";

    // specials
    boolean handled = true;
    if (msg.name.equals("POINTER")) {
      if (controls==ONMOTION) {
        msg.data = null;
        sendMessage (msg);
      }
    }
    else if (msg.name.equals("BUTTON")) {
      if (msg.info==1) getHit();
      if (msg.info==3) deselect(true);
    }
    else if (msg.name.equals("DRAG")) {
      MBox mb = (MBox)msg.data;
      if (msg.info==1) getHit(mb.x,mb.y,mb.w,mb.h,true);
    }
    else if (msg.name.startsWith("EDIT-")) {
      String key = msg.name;
      int i1 = key.indexOf('-');
      int i2 = key.indexOf('-',i1+1);
      int i3 = key.indexOf('-',i2+1);
      int r = Convert.s2l(key.substring(i1+1,i2));
      int c = Convert.s2l(key.substring(i2+1,i3));
      if (r>0) setValue(r,c,msg.data);
      else for (int i=1; i<=size; i++) setValue(i,c,msg.data);
      refresh();
      sendCell(r,c,"UPDATE");
      isEditing=false;
    } else {
      handled = false;
    }
    if (handled) return NORMAL;

    // now implement flat decision tree
    switch (msg.name) {
    case "SCROLL":	scroll(msg.info); break;
    case "REFRESH":	refresh(1); break;
    case "RESIZE":	resize(); break;
    case "SELECTROW":	select(Convert.o2l(msg.data),0); break;
    case "SELECTCOL":	select(0,Convert.o2l(msg.data)); break;
    case "SELECTCELL":	Table t=Convert.o2t(msg.data); select(t.getL("ROW"),t.getL("COL")); break;
    case "DESELECT":	deselect(false); break;
    case "MENU":	new GMenu (MW,"List","Files...,Template...,Settings...,Control...,Window...,Other...,Exit",0,0,this); break;
    case "LIST": 
      switch ((String)msg.data) {
      case "FILES...":	  new GMenu (MW,"Files","Open,Close,Reread,ReOpen",0,0,this); break;
      case "TEMPLATE...": new GMenu (MW,"Template","Open,Default",0,0,this); break;
      case "CONTROL...":  new GMenu (MW,"Controls","Off,Click,Continuous,Assign...",controls+1,0,this); break;
      case "WINDOW...":	  new GMenu (MW,"Window","Toggle,Push,Pop",0,0,this); break;
      case "OTHER...":	  new GMenu (MW,"Other","Theme",0,0,this); break;
      case "SETTINGS...": new GMenu (MW,"Settings","Grid,Fill,Scroll,Center,Header,Number",settings,GMenu.TOGGLE,this); break;
      case "EXIT":	  msg.name="EXIT"; if ((exit&EXIT_MENU)!=0) MQ.put(msg); break;
      } break;
    case "FILES": 
      switch ((String)msg.data) {
      case "OPEN":     new GPrompt (MW,"OpenFile","",0,this); break;
      case "CLOSE":    closeFile(); break;
      case "REREAD":   if (isTable) { parseTable(); applyTemplate(); } refresh(); break;
      case "REOPEN":   openFile (currentName); break;
      } break;
    case "TEMPLATE": 
      switch ((String)msg.data) {
      case "OPEN":     new GPrompt (MW,"OPENTEMPLATE","",0,this); break;
      case "DEFAULT":  openFile(currentName); break;
      } break;
    case "SETTINGS":   settings=msg.info; applyTemplate(); refresh(); break;
    case "CONTROLS":   if (msg.info == 4); else controls = msg.info-1; break;
    case "WINDOW":     break;
    case "OTHER": 
      switch ((String)msg.data) {
      case "THEME": new GMenu (MW,"THEME",MWindow.themeList,0,0,this); break;
      } break;
    case "OPENTABLE":     openTable ((Table)msg.data); break;
    case "CLOSETABLE":    closeTable (); break;
    case "OPENFILE":      if (msg.info>=0) isEdit = (msg.info>0); openFile ((String)msg.data); break;
    case "CLOSEFILE":     closeFile (); break;
    case "OPENTEMPLATE":  openTemplateFile ((String)msg.data); break;
    case "APPLYTEMPLATE": openTemplateFile ((String)msg.data); break;
    case "THEME":         MW.setTheme((String)msg.data); refresh(); break;
    case "CURSOR":        curmode = msg.info; break;
    case "COLOR":         colrIdx = msg.info; break;
    case "POP":           MW.pop(-1); refresh(1); break; //Refresh with flag==1 so that Graphics context will be renewed
    case "EXIT":          if ((exit&EXIT_MESSAGE)!=0) return FINISH; break;
    case "FINISH":        return FINISH; 
    }
    return NORMAL;
  }

  public Object getCell (int row, int col) {
    return null;
  }

  @Override
  public Object setKey (String name, Object value) {
    switch (name) {
      case "CURROW":	 curRow = Convert.o2l(value); break;
      case "CURCOL":	 curCol = Convert.o2l(value); break;
      case "SELECTROW":	 select(Convert.o2l(value),0); break;
      case "SELECTCOL":	 select(0,Convert.o2l(value)); break;
      case "SELECTCELL": Table t=Convert.o2t(value); select(t.getL("ROW"),t.getL("COL")); break;
      case "DESELECT":	 deselect(false); break;
      case "ADDROW":
      case "ADDROWS":
      case "SETROW":
      case "ROWBYVALUE":
      case "REMOVEROW":	
      case "REMOVECURRROW":
      case "SELECTMROW":
      case "SELECTCELLS":
      case "SORTA":
      case "SORTD":
      case "SHOWCOL":
      case "HIDECOL":
      case "SHOWALL":
      case "HIDEALL":
      case "SETCELL":
      case "SETCELLCOLORS":
      case "SETROWCOLORS":
      case "SETCOLCOLORS": 
        M.warning("Key "+name+" not supported for this version of list"); break;
      default: return null;
    }
    return value;
  }

  private static String[] keys = {"NUMROWS","NUMCOLS","CURROW","CURCOL","CELL"};
  public String[] getKeys() { return keys; }

  @Override
  public Object getKey (String name) {
    switch (name) {
      case "NUMROWS": return numRows;
      case "NUMCOLS": return numCols;
      case "CURROW": return curRow;
      case "CURCOL": return curCol;
      case "SELROW": return selRow;
      case "SELCOL": return selCol;
      case "SELECTEDROW": return selRow;
      case "SELECTEDCOL": return selCol;
      case "CELL": return getCell(curRow,curCol);
    }
    return null;
  }

  /** Get the poll interval in seconds
   *  @return poll interval in seconds
   */
  public int getPollInterval () {
    return poll;
  }
  /** Set the poll interval in seconds
   *  @param pollInterval poll interval in seconds
   */
  public void setPollInterval (int pollInterval) {
    poll = pollInterval;
  }

}
