package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libg.GValue;

import nxm.ice.lib.Archiver;
import nxm.ice.lib.ICEPacket;
import nxm.ice.lib.MDevIce;
import nxm.ice.lib.RamDiskResource;
import nxm.ice.lib.DevIce.DmaMap;
import nxm.ice.lib.DevIce.TimeCode;
import nxm.ice.inc.PicSlave;

/**
  Synchronize a PIC acquisition into a Midas file/pipe.

  If /MULTI=n is used, this primitive controls multiple ports at once.

  @author Jeff Schoen
  @version %I%, %G%
*/
public final class sourcepic extends Primitive implements PicSlave,Keyable {

  /** List of supported replay modes */
  public static String replayList="-NShot,File,Stopped,OneShot,Continuous,"
	+"StopTop,StopNow,Spin,Archive,Restart,Abort,Finish,NewDevice,Reopen,Reconnect,Release";
  public static int rFILE=-1,rSTOPPED=0,rONESHOT=1,rCONTINUOUS=2, // replay modes
	rSTOPTOP=3,rSTOPNOW=4,rSPIN=5,rARCHIVE=6,rRESTART=7,rABORT=8,rFINISH=9,
	rNEWDEVICE=10, rREOPEN=11, rRECONNECT=12, rRELEASE=13, rRESTART_WAIT=99;
  private static int replayListOffset=-3;

  /** List of supported master/slave modes */
  public static String syncList="Master,SlaveWait,SlaveRun";
  private static int sMASTER=1,sWAIT=2,sRUN=3;

  /** List of supported archive modes */
  public static String archList="Off,Open,SnapShot,RealTime";
  private static int aOFF=0,aOPEN=1,aSNAP=2,aRT=3; 

  /** List of supported packet modes */
  public static String pktmodeList="Off,Ice,IceT,SDDS";
  private static int pOFF=0,pICE=1,pICET=2,pSDDS=3;

  /** List of supported asynchronous monitor modes */
  public static String monitorList="Off,Async,Info,Full";
  private static int mOFF=0,mASYNC=1,mINFO=2,mFULL=3;

  private long rndx,ondx,andx,ramsz;
  private int rcyc,ocyc,acyc,stats,pmin,pmax,pavg,ptot,snaptc,autors,renum;
  private int status,frame,bytes,dmamode,skip,oskip,nchan,nchano,multi,chan=0;
  private int dmac,port,bits,rate,xfer,dec,rfgain,gain,replay,block,lost,lostc,mcs,vctl;
  private int packet,apacket,pktmod,apktmod,ptype,mnbytes,tinc,tcmode,tcpp,tccnt;
  private int archmode=aOFF,archtl,archtc,archcn,ipflag,aipflag,archsf,monitor;
  private int maxGain,minGain,select=-1;
  private boolean module,tuner,core,host,fdec,flush,autoss,sgo,sss,xts,reset;
  private boolean isSlave,sdds,nodma,rtfile,nyfreq,skiponcard,w2e;
  private double freq,afreq,dfreq,rffreq,mtodo,delta,ratio,drate,wait,sfactor,maxout,fscale;
  private double tctolr,tcmaxtolr,tcows,tcofs=0,timer,timetop,timeout,timeRTF,timecur,archfs;
  private double archto,archdur=-1,archoff=-1,syncoff,timearch,agcLevel,agcWidth,agcTime,agcPeriod=0;
  private String sport,alias,slave,archfn,pmss,apmss;
  private TimeCode tc = new TimeCode(), tclast = new TimeCode();
  private Time tco, time = new Time();
  private DataFile hr,ho;
  private Archiver ha;
  private DmaMap map,smap;
  private ICEPacket pkt,apkt;
  private MDevIce pic;
  private int[] dmacs,ports,gains;
  private PicSlave picSlave;
  private Table keys = new Table();
  private float mbps=0;
  private byte[] tbuf;
  private GValue gpw;
  private Table archkw;	

  @Override
  public int open() {

    // open the PIC device
    alias = MA.getS("DEV");
    pic = new MDevIce (MA,alias);
    if (pic.open()<=0) M.error("Problem opening device: "+alias); 

    // get operational switches
    autors = MA.getL("/AUTORS");
    autoss = MA.getState("/AUTOSS");
    renum  = MA.getL("/RENUM");
    sdds   = MA.getState("/SDDS");
    rtfile = MA.getState("/RTFILE");
    multi  = MA.getL("/MULTI",1);
    nyfreq = MA.getState("/NYFREQ");
    w2e    = MA.getState("/WARN2ERR");
    archfn = MA.getS("/ARCH");
    dmacs  = new int[multi];
    ports  = new int[multi];

    // get PIC parameters
    ptype = pic.getKeyL(0,pic.KEY_PTYPE);
    port  = pic.getKeyL(0,pic.KEY_PINDEX);
    reset = !(ptype==0 || port==0 || (multi>1 && !autoss));
    reset = MA.getState("/RESET",reset);
    tinc  = pic.getKeyL(port,pic.KEY_TINC);
    tuner = (ptype == pic.IOPT_TUNER) || (ptype == pic.IOPT_TBANK);
    core  = (ptype == pic.IOPT_CORE) || (ptype == pic.IOPT_MCORE);
    module = (ptype == pic.IOPT_MODULE);
    host  = MA.getState("/HOST",module);
    nchan = (ptype==pic.IOPT_TBANK)? pic.getKeyL(port,pic.KEY_CHNS) : 1;
    nchan = MA.getL("/NCHN",nchan);
    nchano = MA.getL("/NCHNOUT",nchan);
    fdec = MA.getState("/FRAMEDEC");
    gains = new int[nchan];

    // parse packet modes
    pktmod = pOFF;
    apacket = MA.getLength("/APACKET");
    if (apacket>=0) apmss = parsePacketMode(MA.getS("/APACKET")); 
    apktmod = pktmod; 
    if (apktmod==pSDDS) apacket=-1;
    pktmod = pOFF;
    packet = MA.getLength("/PACKET");
    if (packet>=0) pmss = parsePacketMode(MA.getS("/PACKET"));
    if (pktmod==pOFF && apktmod==pSDDS) { pktmod=apktmod; pmss=apmss; }
    if (pktmod==pSDDS) packet=-1;
    ipflag = ICEPacket.FIXED | (pktmod==pICET? ICEPacket.ABSC:0);
    aipflag = ICEPacket.FIXED | (apktmod==pICET? ICEPacket.ABSC:0);

    mcs = pic.getIntFlagDef("MCS",0);
    vctl = pic.getIntFlagDef("VCTL",-1);

    // get input RAM template
    hr = MA.getDataFile ("IN");
    hr.open(hr.INOUT);
    hr.setOutput(false);
    frame = (hr.typeClass==2)? hr.getSubSize() : 1;
    if (pktmod==pSDDS) frame = frame*1088/1024;
    if (hr.bps>0) bits=hr.bps*8; else bits = -hr.bps;
    if (hr.spa==2) bits = -bits;
    bits = MA.getL("/BITS",bits);

    // get reduction 
    dec  = Math.max(1,MA.getL("DEC")); 		// happens on card
    skip = Math.max(1,MA.getL("/SKIP"));	// happens in host
    skiponcard = (module && !host && dec==1 && skip>1);
    skiponcard = MA.getState("/SKIPONCARD",skiponcard);
    skiponcard = !MA.getState("/SKIPONHOST",!skiponcard);
    if (module && host && dec>1 && !skiponcard) { skip*=dec; dec=1; }
    oskip = skiponcard? 1 : skip;

    // get resampling ratio
    delta = hr.getXDelta();
    ratio = MA.getD("/RATIO"); 
    if (ratio>0) pic.setKeyD(0,pic.KEY_RATIO,ratio);

    // get transfer parameters
    xfer = frame * MA.getL("/TL", (frame>1)? 1:4096 );
    if (M.pipeMode==Midas.PINIT) replay=rCONTINUOUS; else replay=rFILE;
    replay = MA.getSelectionIndex("/REPLAY",replayList,replay,replayListOffset);
    flush  = MA.getState("/FLUSH");
    wait   = MA.getD("/WAIT");
    timeout= MA.getD("/TIMEOUT",2.0);
    maxout = MA.getD("/MAXOUT");

    // create ICE packet 
    pkt = new ICEPacket(hr.getFormat(),xfer,ipflag);
    pkt.setChannel( (renum>0)? renum:port );

    // get the output pipe
    ho = MA.getDataFile ("OUT",hr,0);
    ho.setYDelta(delta*frame*skip);
    ho.size = (hr.size-1)/oskip + 1;
    ho.setDataRep(hr.getDataRep());

    // open output
    if (pktmod==pICE || pktmod==pICET) {
      ho.setPacketHandler(pkt);
      if (packet>0) ho.setPacket(ho.getPacket()+pmss); 
    }
    ho.open(ho.NATIVE|ho.OPTIONAL);
    if (pktmod==pSDDS) ho.keywords.putMain("PACKET",pmss);

    // force type 1000 handling
    hr.setFS(0); hr.update();
//    ho.setDFS(0);
    ramsz = (long)hr.size;
    if (ramsz%xfer!=0 || ramsz==xfer) M.info("Ram size="+ramsz+" should be an integer multiple>1 of /TL="+xfer);
    bytes = (int)(xfer*hr.dbpe);
    block = MA.getL("/BLOCK",bytes);
    if (oskip>1 || dec>1) block=-1;

    // get IO port parameters
    drate = 1.0/delta;
    if (tuner) drate = hr.spa*dec/delta;
    if (core && !fdec) drate = dec/delta;
    if (ratio>0) drate = drate/ratio;
    if (pic.getIntFlagDef("PRER2C",0)>0) drate *= 2;
    drate = MA.getD("/SRATE",drate);
    freq = MA.getD("FREQ",0.0);
    dfreq = MA.getD("/DFREQ",0.0);
    gain = MA.getL("GAIN");
    fscale = nyfreq? 1.0 : 2.0/drate;
    Table agcTable = MA.getTable("/AGC");
    if (agcTable!=null) setAGC(agcTable);
    rate = (int)Math.round(drate);
    gains = new int[nchan]; 
    for (int i=0; i<nchan; i++) gains[i]=gain;

    // get timecode parameters
    time.fromCurrent(); 
    tcmode = pic.getKeyL(0,pic.KEY_TCMODE);
    if (tcmode!=pic.TCM_OFF) pic.setKeyD(0,pic.KEY_TCOFF,time.getWSec(),time.getFSec(),-1.0);
    if (tcmode==pic.TCM_CPU||tcmode==pic.TCM_STC||tcmode==pic.TCM_ICE) tcows=0; // CPU includes year 
    else tcows = time.getYiS(); // add current year
    tco = MA.getTime("/TCOFF"); if (tco!=null) { tcows=tco.getWSec(); tcofs=tco.getFSec(); }
    tctolr = MA.getD("/TCTOLR");
    tcmaxtolr = MA.getD("/TCMAXTOLR");
    double tcps = MA.getD("/TCPS");	// time codes per second
    double sptc = (tcps>0)? 1.0/(tcps*delta) : hr.size/2/Math.max(multi,nchan);	// samples per tc
    tcpp = Math.max(1,(int)Math.round(sptc/(multi*xfer*skip)));
    tcpp = MA.getL("/TCPP",tcpp);
    tcpp *= oskip;

    // master/slave modes
    slave = MA.getS("/SLAVE","NONE");
    isSlave = !slave.equals("NONE");
    sss = slave.equals("SS");
    xts = slave.equals("XT");
    sgo = isSlave && !(sss||xts);
    slave = MA.getS("/MASTER",null);

    // initialize port
    if (reset) status = pic.reset(0); else status=1;
    if (status<0) M.error("Problem resetting device"); 

    // allocate and map memory buffer
    map = pic.mapFile(hr);
    if (map==null) M.error("Problem mapping DMA memory"); 
    if (multi>1 || !MA.getState("/PAGED",true)) map.setPaged(false);
    if (ptype == pic.IOPT_STREAM) {
      map.setPaged(false);
      map.getVirtualAddress(0,(int)map.bytes);			// map this space
    }

    // open the PIC card IO ports
    openPorts();

    // report DMA Channel number
    MR.put(MA.getU("/DMAC"),dmac);
    nodma = pic.getKeyL(dmac,pic.KEY_NODMA)>0;

    // setup framed decimation in IOC chip
    if (skiponcard) {
      pic.setKeyL(dmac,pic.KEY_FRAME,frame*nchano);
      pic.setKeyL(dmac,pic.KEY_FRAMEDEC,skip);
    }

    // setup tuner resampling 
    if (ratio>0) setRatio(ratio);

    // copy params to Keywords
    sport = pic.getPortKey();
    hr.keywords.put("PORT",sport);
    if (archfn!=null) hr.keywords.put("ARCHIVE",getArchFN());
    if (pic.getDblFlagDef("RFFREQ",0.0)>0) {
      rffreq = pic.getKeyD(dmac,pic.KEY_RFFREQ); hr.keywords.put("D:RFFREQ",rffreq); 
      rfgain = pic.getKeyL(dmac,pic.KEY_RFGAIN); hr.keywords.put("L:RFGAIN",rfgain); 
    }
    if (multi>1) hr.keywords.put("L:MULTI",multi);
    if (nchan>1) hr.keywords.put("L:NCHAN",nchan);
    if (tuner)   hr.keywords.put("D:FREQ",freq);
    hr.update();

    // initialize modes
    if (isSlave) replay=rSTOPPED; // special slave startup
    if (!MA.find("/POLL")) setPollTime(0.025);
    dmamode = pic.DMA_STOP;
    mtodo = smap.bytes;
    todo = 0; ocyc = acyc = 0; ondx = andx = 0;
    sfactor = 100.0/smap.bytes;
    if (bytes*oskip>smap.bytes) M.error("Skip cannot be greater than one buffer");
    if (rtfile) updateRTF(true);

    // setup archiving
    archtl = MA.getL("/ARCHTL",128*1024);
    archto = MA.getL("/ARCHTO",-1);
    archsf = MA.getL("/ARCHSF",0);
    archfs = MA.getD("/ARCHSFS",-1);
    archcn = MA.getL("/ARCHCHN");
    archkw = MA.getTable("/ARCHKW");
    if (archsf!=0 && archsf==1) archsf=-1;	// default to non-circular
    if (nchano>1 && (apacket>=0 || archcn>=0)) archtl = xfer;
    apkt = new ICEPacket(ho.getFormat(),archtl,aipflag);
    if (apktmod==pSDDS) archtl = archtl*1088/1024;;
    startArch(archfn,archoff,archdur);

    // now update packet fields
    setDelta(delta);

    // get status logger and progress meter
    stats  = MA.getL("/STATS");
    Object gpwo = MA.getO("/GPWR");
    if (gpwo instanceof GValue) gpw = (GValue)gpwo;
    if (gpw!=null) gpw.setValue(0.0);

    // get monitor throttling mode
    monitor = MA.getSelectionIndex("/MONITOR",monitorList,mFULL,-1);

    // get extra query list entries
    setKeys(MA.getTable("/PICKEYS"));

    return (NORMAL);
  }

  @Override
  public synchronized int process() {

    timecur = Time.current();
    if (rtfile) updateRTF(false);
    if (replay==rABORT) return(FINISH);

    if (replay==rRELEASE) {
      for (int n=0; n<multi; n++) pic.dmaFunc(dmacs[n],pic.DMA_RESET);
      replay=rSTOPPED;
    }

    // start an acquisition
    if (todo==0) {
      if (replay==rSTOPTOP || replay==rSTOPNOW) replay=rSTOPPED;
      if (replay==rSTOPPED) return(NOOP);
      if (replay==rFINISH) return(FINISH);
      if (replay==rRESTART && autors>=2) { 
	Time.sleep(0.2); pic.dmaFunc(dmac,pic.DMA_KILL);
	replay=rRESTART_WAIT; timer=Time.current();
      }
      if (replay==rRESTART_WAIT) {
	if (pic.dmaFunc(dmac,pic.DMA_ACTIVE)==0) Time.sleep(0.2);	// ready to start
	else if (Time.current()-timer < timeout+1) return (NOOP);	// wait some more
	else {
	  if (autors>=3) pic.dmaFunc(dmac,pic.DMA_BURY);
	  warning("All DMA channels on this input did not shut down");
	  Time.sleep(0.5);
	}
	replay=rCONTINUOUS;
      }
      rcyc=ocyc=acyc=lost=lostc=0; rndx=ondx=andx=0;			// init counters
      pmin=100; pmax=0; ptot=0; pkt.setCount(0); apkt.setCount(0);	// init more counters
      int newmode = (nodma)? pic.DMA_ENABLE : (replay==rONESHOT)? pic.DMA_ONESHOT : pic.DMA_CONTINUOUS;	
      if (slave!=null) setSlaveReplay(replay,newmode);			// wait for slave startup
      if (wait>0) Time.sleep(wait);					// specified startup wait
      if (stats>0) M.info("Sourcepic-"+dmac+" START");
      for (int n=multi-1; n>=0; n--) pic.dmaFunc(dmacs[n],newmode);
      dmamode=newmode;	// dont change dmamode until dmaFunc completes
      todo=mtodo; 
      snaptc=1; tc.initMetrics(); tclast.initMetrics();
      timecur = Time.current();
      timer = timetop = timearch = agcTime = timecur;
    }

    if (ha.isOpen && archto>0 && (timecur-timearch)>archto) {
      ha.checkChannels(); 
      ha.flush();
      timearch=timecur;
    }
    if (archmode==aOPEN) startArch(archfn,archoff,archdur);

    if (replay==rSPIN) return(NOOP);
    long index = pic.dmaFuncX(dmac,pic.DMA_STATUS);
    if (index==-1 && pic.getKeyL(0,pic.KEY_STATE)<0) replay=rSTOPNOW;
    if (replay==rSTOPNOW || replay==rRESTART || replay==rFINISH) todo=0; 

    // special AGC control case
    if (agcPeriod>0 && !tuner && (timer-agcTime)>agcPeriod) processWBAGC();

    // transfer data
    if (todo>0) {
      if (index<rndx) rcyc = pic.dmaFunc(dmac,pic.DMA_CYCLE);		// low overhead cycle/index counter
      if (rcyc-ocyc>10 || ocyc-rcyc>10) { 
	warning("Probable bad rcyc="+rcyc+" ocyc="+ocyc+" read");
	rcyc = pic.dmaFunc(dmac,pic.DMA_CYCLE);
      }
      rndx = index;
      int behind = rcyc - (ocyc+lost);					// check for falling behind
      if (behind>1 || (behind==1 && rndx>ondx)) { 
	warning("Falling behind "+behind+" buffers. Rcyc="+rcyc+" Ocyc="+ocyc+" Lost="+lost+" ID="+this.id); 
	lost=rcyc-ocyc; 
      }
      int tbytes = bytes*nchano;					// total bytes
      int sbytes = tbytes;						// total bytes with skip
      if (ha.isOpen) sbytes *= oskip;					// must include skip
      if (ondx+sbytes>smap.bytes) sbytes=(int)(smap.bytes-ondx);	// trim bytes to end of buffer
      if ((rcyc<ocyc+lost) || (rndx<ondx+sbytes && rcyc<=ocyc+lost)) {	// data not ready
	if (timeout<=0 || timecur-timer<timeout) return (NOOP);		// under timeout
	warning("Detected DMA stall > timeout. Rcyc="+rcyc+" Ocyc="+ocyc+" Lost="+lost);
	if (autors>0) setReplay(rRESTART);				// auto restart
	timer=timecur; return (NOOP);					// reinit stall timer
      }
      for (int n=1; n<multi; n++) { 					// now check others for multi
	index = pic.dmaFuncX(dmacs[n],pic.DMA_STATUS);
	if (index<ondx+sbytes && rcyc<=ocyc+lost) return (NOOP);	// not ready
      }
      if (tcmode!=pic.TCM_OFF) processTimeCode();			// handle TC 

      if (ho.isOpen) {
	long vaddr;
	if (ondx+tbytes>smap.bytes) {					// output data
	  M.info("Partial buffer out "+ondx+" "+smap.bytes);
	  vaddr = map.getVirtualAddress(ondx-tbytes,tbytes);		// map this space
	} else {
	  vaddr = map.getVirtualAddress(ondx,tbytes);			// map this space
	}
	if (monitor==mOFF || (monitor!=mFULL && ho.io.avail()<tbytes)){	// monitor pipe not ready
	  if (monitor==mINFO) M.info("Dropping monitor frame");
	}
	else if (nchano>1 || mcs>0) {					// channelized data
	  if (packet<0) 
	    ho.write(vaddr,0,tbytes);					// output single data
	  else for (int n=0; n<nchano; n++) {
	    if (mcs>0) pkt.setChannel(getMCSchn(vaddr,n,bytes));
	    else pkt.setChannel(n+1);					// set pkt channel
	    ho.write(vaddr,0+n*bytes,bytes);				// output banked data
	  }
	}
	else if (multi>1) for (int n=0; n<multi; n++) { 
	  pkt.setChannel( (renum>0)? renum+n : ports[n] );		// set pkt channel
	  ho.write(vaddr,0+n*(int)smap.bytes,bytes);			// output multi data
	}
	else {
	  ho.write(vaddr,0,bytes);					// output single data
	}
	if (maxout>0 && ho.seek()>=maxout) return(FINISH);		// all output done
      }

      if (ha.isOpen) {							// archive data
	index = ondx+sbytes;						// last valid index - no wrap
	tbytes = (int)(archtl*ha.dbpe);					// total bytes per atransfer
	if (nchano>1 && (apacket>=0 || archcn>=0)) tbytes *= nchano;
	for (; (acyc<ocyc) || (acyc==ocyc && andx+tbytes<=index); ) { 	// archive output
 	  long vaddr = map.getVirtualAddress(andx,tbytes);		// map this space
	  if (nchano>1) {						// channelized data
	    if (archcn==-2)
	      ha.write(vaddr,0,tbytes,-2);				// output single data
	    else if (apacket<0 && archcn<0) 
	      ha.write(vaddr,0,tbytes,-1);				// output single data
	    else for (int n=0; n<nchano; n++) {
	      apkt.setChannel(n+1);					// set pkt channel
	      if (archcn<0 || n==archcn)
	      ha.write(vaddr,0+n*bytes,bytes,n);			// output banked data
	    }
	  }
	  else if (multi==1) ha.write(vaddr,0,tbytes,-1);		// archive single data
	  else for (int n=0; n<multi; n++) {
	    apkt.setChannel( (renum>0)? renum+n : ports[n] );		// set apkt channel
	    ha.write(vaddr,0+n*(int)smap.bytes,tbytes,n);		// archive multi data
	  }
	  apkt.upCount(); apkt.decTCO(archtl);				// increment counters
	  andx += tbytes; if (andx>=mtodo) { acyc++; andx=0; }
	} 
        if (archmode==aSNAP && ha.seek()*delta>archdur) archmode=aOFF;
        if (archmode==aOFF) stopArch();
	if (ha.isFinished()) return(FINISH);				// all output done
      }

      ondx+=(bytes*nchano*oskip); todo=(mtodo-ondx); 			// advance pointers
      pkt.decTCO(xfer*skip); pkt.upCount(skip);				// update TC and count
      time.addSec( delta*xfer*skip );					// update time
      timer = timecur;
    }

    // stop an acquisition or recycle to top of buffer
    if (todo<=0) {
      // wait for raw DMA to complete, even though decimated pipe out is done
      if (replay==rONESHOT && pic.dmaFunc(dmac,pic.DMA_POLL)!=0) { todo=-1; return(NOOP); }
      ocyc++; todo=0; ondx-=smap.bytes;
      boolean partial = (replay==rSTOPNOW || replay==rRESTART || replay==rFINISH);
      if (!partial) mbps = (float)(1.e-6 * multi * smap.bytes / Math.max(0.01,timer-timetop));
      if (stats>0 && !partial) {
	int pf=0; if (pic.dmaFunc(dmac,pic.DMA_CYCLE)==(ocyc+lost)) 
	  pf = (int)Math.round(sfactor*pic.dmaFuncX(dmac,pic.DMA_INDEX));
	if (pf==100) pf=0; if (pf<pmin) pmin=pf; if (pf>pmax) pmax=pf; ptot+=pf;
	if (ocyc%stats==0) { pavg = ptot/stats;
	  String s = "CYC="+ocyc+",LOST="+lost+",FAVG="+pavg;
	  if (sdds) { int gap=pic.getKeyL(dmac,pic.KEY_GAP); s += ",GMIN="+(gap&0xFFFF)+",GMAX="+((gap>>16)&0xFFFF); }
          else if (stats>1) s += ",FMIN="+pmin+",FMAX="+pmax;
	  M.info("Sourcepic-"+dmac+" STATS={"+s+",MBPS="+mbps+"}");
          pmin=100; pmax=0; ptot=0;
        }
      }
      timetop=timer;        
      if (agcPeriod!=0 && tuner) processNBAGC();
      if (ha.isOpen && flush) ha.flush();
      int lostd = pic.dmaFunc(dmac,pic.DMA_LOST);
      if (lostd>lostc) { warning("Sourcepic-"+dmac+" LOST="+(lostd-lostc)+" card buffers"); lostc=lostd; }
      if (replay<rFILE) { replay++; todo=mtodo; return(NORMAL); }
      if (replay==rCONTINUOUS) { todo=mtodo; return(NORMAL); }
      // wait .2 sec for graceful exit of other channels, then force a DMA stall
      if (replay==rRESTART && autors>=2) { Time.sleep(0.2); pic.dmaFunc(dmac,pic.DMA_KILL); }
      if (stats>0) M.info("Sourcepic-"+dmac+" STOP");
      for (int n=0; n<multi; n++) {
        pic.dmaFunc(dmacs[n],pic.DMA_STOP);
        pic.dmaFunc(dmacs[n],pic.DMA_RESET);
      }
      dmamode=pic.DMA_STOP;
      if (slave!=null) setSlaveReplay(replay,dmamode);			// wait for slave shutdown
      if (replay==rFILE || replay==rFINISH) return(FINISH); 
      if ( replay==rONESHOT) {
        if (snaptc<0) hr.update();
        if (flush) flushRamBuffer();
        rcyc = 1;
      }
      if (replay!=rRESTART) replay=rSTOPPED;
      else if (autors>=2) replay=rRESTART_WAIT; 
      else replay=rCONTINUOUS;
      timer=timecur;
      if (ha.isOpen) ha.checkChannels();
      if (ha.isOpen) ha.discontinue();
    }

    if (gpw!=null) gpw.setValue(getProgress());
    return (NORMAL);
  }

  @Override
  public int close() {
    if (dmamode!=pic.DMA_STOP && stats>0) M.info("Sourcepic CANCEL");
    for (int n=0; n<multi; n++) {
      if (dmacs!=null && dmacs[n]>0) pic.dmaFunc(dmacs[n],pic.DMA_CANCEL);
    }
    stopArch();
    if (gpw!=null) gpw.setValue(1.0);
    if (hr!=null) hr.close();
    if (ho!=null) ho.close();
    if (map!=null) map.close();
    if (pic!=null) pic.close();
    replay = rSTOPPED;
    dmamode = pic.DMA_STOP;
    return (NORMAL);
  }

  private void warning (String text) {
    if (w2e) M.error(text);
    else M.warning(text);
  }

  private void flushRamBuffer() {
    if (hr.io instanceof RamDiskResource) return;
    int tbytes = (int)(hr.size*hr.dbpe);
    hr.setOutput(true);
    hr.seek(0.0);
    long vaddr = map.getVirtualAddress(0L,tbytes);
    hr.write(vaddr,0,tbytes);
    hr.setOutput(false);
  }

  private void updateRTF (boolean init) {
    if (init) timeRTF = Time.current();
    else if (timecur-timeRTF<0.05) return;
    timeRTF = timecur;
    if (snaptc<0) setTime(hr,tc.wsec,tc.fsec-tc.offset*delta);
    double inbytes = getBytes();
    hr.setInByte(inbytes);
    hr.ioh.write(hr.hb,80,8,80L);	// direct access to in_byte
    hr.ioh.read(hr.hb,88,8,88L);	// direct access to out_byte
    double fill = (inbytes - hr.getOutByte(0)) / map.bytes;
    if (vctl>0) pic.setKeyD(dmac,pic.KEY_VCTL,fill);
  }

  private void openPorts() {
    if (ptype==pic.IOPT_TBANK) {
      int pktlen = MA.getL("/PKTLEN",bytes);
      pic.setKeyL(port,pic.KEY_CHNS,nchan);
      pic.setKeyL(port,pic.KEY_PKTLEN,pktlen);
      pic.setKeyD(port,pic.KEY_DFREQ,dfreq*fscale);
    }
    for (int n=0; n<multi; n++) {
      // loop through ports - treat normal mode as special case of multi=1
      int flags = 0;
      if (n==0) { 
        if (sgo) flags = pic.FLG_SGO;
        if (sss) flags = pic.FLG_RGO;
        if (xts) flags = (pic.FLG_XGO|pic.FLG_TGO);
      }
      else if (n==1 && !autoss) flags = pic.FLG_SGO;
      else flags = pic.FLG_RGO;

      double nfreq = (freq+n*dfreq)*2/rate; // normalized frequency
      dmac = pic.ioPort(ptype,port,-1,-1,bits,rate,nfreq,dec,gain,flags);
      if (dmac<=0) M.error("Problem setting up IO port"); 

      smap = map.getSubMap(n,multi); // get block n of multi
      status = pic.dmaSetup(dmac,-1,smap,block,0);
      if (status<0) M.error("Problem setting up DMA channel "+status); 

      dmacs[n]=dmac;
      ports[n]=port;

      if (autoss) port += tinc;  // same side increments by 2 unless CPC!=default
      else if (n%2==0) port++;	 // B is always A+1
      else port = port-1 + tinc; // next is on same side as last
    }

    dmac = dmacs[0];
    port = ports[0];

    afreq = pic.getKeyD(dmac,pic.KEY_FREQ)/fscale; // report actual freq used
  }

  private String parsePacketMode (String ss) {
    String pmss;
    if (ss==null || ss.length()==0) pmss = "ICE";
    else if (ss.startsWith("DET")) pmss = "ICE/"+ss;
    else if (ss.equals("SDDS")) pmss = "SDDS/ICE";
    else pmss = ss;
         if (pmss.startsWith("SDDS")) { pktmod = pSDDS; }
    else if (pmss.startsWith("ICET")) { pktmod = pICET; pmss = pmss.substring(4); }
    else if (pmss.startsWith("ICE"))  { pktmod = pICE; pmss = pmss.substring(3); }
    else M.error("Illegal Packet Mode Switch String: "+ss);
    return pmss;
  }

  int tcretry=5;
  private void processTimeCode() {
         if (snaptc>0 && snaptc<tcretry) snaptc++;			// snap exception
    else if (archtc>0 && archtc<tcretry) archtc++;			// archive exception 
    else { 
      tccnt += oskip;
      if (tccnt<tcpp) return; 						// only handle TC every tcpp packets
      tccnt -= tcpp;
    }
    int tcstat = pic.tc (dmac,-1.0,delta,tc,0); 			// get current timecode
    if (tcmode!=pic.TCM_CPU && tcstat>0 && tc.wsec<100) { 		// check for ray day 1st 100 sec of year
      int days = (int)Math.round(-(tc.diff(tclast)+tcows)/86400);
      if (days==365 || days==366) tcows += days*86400; 			// bump to next year
    }
    tc.wsec += tcows;							// add year offset
    tc.fsec += tcofs;							// add year offset
    double tboff = (ocyc+lost)*mtodo; 					// byte offset at top of ram buffer
    int cskip = skiponcard? skip : 1;
    double toff = tc.offset - cskip*(tboff+ondx)/hr.bpa/nchano;		// sample offset relative to packet start
    pkt.setTC (tcmode,tcstat,toff,tc.wsec,tc.fsec);			// set in packet header
    if (tcstat == pic.TC_NOCLOCK) {
      warning("Time code NOCLOCK err detected");
      if (autors>0) setReplay(rRESTART);
    }
    if (tcstat<0 && tcstat!=-9) return;					// done for now
    if (tctolr>0 && tclast.delta>=0) {					// TC tolerance check
      double tcerr = tc.err(tclast);
      if (tcmaxtolr>0 && Math.abs(tcerr)>tcmaxtolr) {
	warning("Time code slip="+tcerr+" > maxtolr="+tcmaxtolr+". Assumed bogus and ignored");
        tclast.update(tc); return;					// update last TC for differential metrics
      }
      if (Math.abs(tcerr)>tctolr) {
        warning("Time code slip="+tcerr+" > tolr="+tctolr+" delta="+tc.delta);
        if (autors>0) setReplay(rRESTART);
      }
    }
    tc.computeMetrics(tclast,delta);	 				// calulate TC metrics
    if (snaptc>0) { snaptc=-1;
      setTime(hr,tc.wsec,tc.fsec-tc.offset*delta); hr.update();	 	// record 1st TC to ramdisk header
      setTime(ho,tc.wsec,tc.fsec-tc.offset*delta); 			// record 1st TC to output header
    }
    ho.setTimeAt(tc.wsec+(tc.fsec-toff*delta)); 			// record all TC to output header
    if (ha.isOpen) {
      toff = tc.offset - ((tboff+andx)/hr.bpa)/nchano;			// sample offset relative to packet start
      if (archtc!=0) { archtc=0;  					// to archive header
        ha.setTime(tc.wsec,tc.fsec-(tc.offset-archoff)*delta); }
      apkt.setTC (tcmode,tcstat,toff,tc.wsec,tc.fsec);			// set in arch packet header
      ha.setTimeAt(tc.wsec+(tc.fsec-toff*delta)); 			// record all TC to output header
    }
    pkt.getTC (time,0.0,delta);						// update time
    tclast.update(tc); 							// update last TC for differential metrics
  } 

  public void setArchFN  (String value) { archfn = value; }
  public void setArchDur (double value) { archdur = value; }
  public void setArchOff (double value) { archoff = value; }

  public void setSyncOff (double value) { 
    syncoff = (rcyc*mtodo + rndx)/hr.bpa;
  }

  public void setArchMode (String value) { 
    setArchMode(Parser.find(archList,value,archmode,-1));
  }
  private void setArchMode (int mode) { 
    if (mode==aOFF) stopArchiver();
    else if (archmode!=aOFF) warning("Cannot restart an archive without stopping first");
    else if (mode==aSNAP) { if (archdur<0) archdur=hr.getLength(); archmode = aOPEN; }
    else if (mode==aRT) { archdur=-1; archmode = aOPEN; }
    else archmode = mode;
  } 

  public void startArchiver (String fname) { archfn = fname; startArchiver(); }
  public void startArchiver (String fname, double dur) { archfn = fname; archdur = dur; startArchiver(); }
  public void startArchiver (String fname, double off, double dur) { archfn = fname; archoff = off; archdur = dur; startArchiver(); }
  public void startArchiver() { setArchMode(aOPEN); }
  public void stopArchiver() { archmode = aOFF; }

  private void startArch (String fname, double time, double dur) {
    stopArch();
    ha = new Archiver(this);
    if (fname==null||fname.length()==0||fname.startsWith("NULL")) return;
    if (ramsz<=archtl) M.error("Ram buffer length="+ramsz+" must be increased to at least 2x the /ARCHTL="+archtl);
    if (ramsz%archtl!=0) M.error("Ram size="+ramsz+" must be an integer multiple>1 of /ARCHTL="+archtl);
    if (MA.getL("/SDDSMULTI")>0) M.error("Switch /SDDSMULTI=n deprecated. Replace with file qualifier {FUNC=SDDSUNPACK,MULTI=n}");
    ha.init (this,fname,hr,0);
    if (multi>1) ha.setMulti(multi);
    if (nchan>1) ha.setNChan(nchan);
    if (archsf!=0) { ha.setMulti(-1); ha.setMaxLines(archsf); ha.setMaxLineSize((archfs>0)?archfs:map.bytes); }
    if (apacket>=0) { apkt.setCount(0); ha.setPacketHandler(apkt); }
    if (apacket>0) ha.setPacket( ha.getPacket()+apmss ); 
    ha.setFS(0);
    ha.open(ha.NATIVE|ha.OPTIONAL);
    if (!ha.isOpen) return;
    if (apktmod==pSDDS) ha.keywords.putMain("PACKET",apmss);
    if (ha.io instanceof FileResource && !(ha.io instanceof NFileResource)) 
	warning("UhOh. Using slower non-native file access.  Check installation.");
    archmode = (dur>0)? aSNAP:aRT;
    archtc=1; // need time code
    int tbytes = (int)(archtl*ha.dbpe);
    if (archoff>=0) {
      archoff *= hr.bpa;
      acyc = (int)(archoff/mtodo);
      andx = (long)(archoff - acyc*mtodo);
      andx = (andx/tbytes)*tbytes;
    } else {
      acyc = ocyc;
      andx = (ondx/tbytes)*tbytes;
    }
    archoff = (acyc*mtodo+andx)/hr.bpa;
    ha.keywords.put("D:ARCHOFF",archoff);
  }
  private void stopArch() {
    archmode = aOFF;
    archoff  = -1;
    if (ha!=null && ha.isOpen) ha.close();
  }

  public void setReplay (String value) { 
    if (value.equalsIgnoreCase("STOP")) value = "STOPNOW";
    if (value.equalsIgnoreCase("START")) value = "CONTINUOUS";
    int nreplay = Parser.find(replayList,value,replay,replayListOffset); 
    setReplay(nreplay);
  }
  private void setReplay (int nreplay) {
         if (nreplay==0 && replay!=0) replay = rSTOPNOW; 
    else if (nreplay==rRESTART && replay==0) replay = rCONTINUOUS; 
    else if (nreplay==rONESHOT && replay==rCONTINUOUS) replay = rSTOPTOP; 
    else if (nreplay>=rNEWDEVICE && replay!=0) warning ("Port must be Stopped to perform NewDevice|Reopen|Reconnect");
    else replay=nreplay;
    if (replay==rSTOPNOW) process(); // process now
    if (replay==rRECONNECT) { openPorts(); replay=0; }
    if (replay>=rABORT && slave!=null) setSlaveReplay(replay,pic.DMA_STOP);
    if (replay==rCONTINUOUS && nodma) replay = rSPIN;
  }
  private void setSlaveReplay (int value, int mode) {
    if (picSlave==null) picSlave = (PicSlave)M.registry.get(slave);
    if (picSlave==null) { warning("Slave="+slave+" does not exist"); return; }
    if (mode==pic.DMA_STOP && value<rABORT) value = rSTOPNOW;
    if (nodma && value==rSPIN) value = rCONTINUOUS;
    picSlave.setReplay(getReplay(value));
    for (;;) {
      int smode = picSlave.getDmaMode();
      if (smode==mode) break;
      if (smode==pic.DMA_ENABLE && mode==pic.DMA_CONTINUOUS) break;
      if (mode==pic.DMA_ENABLE && smode==pic.DMA_CONTINUOUS) break;
      Time.sleep(pollTime);
    }
  }
  public void setRate (int value) { 
    if (value==rate) return;
    if (pic.setKeyL(dmac,pic.KEY_RATE,rate)<0) {	// can't set realtime
      MA.put("/SRATE",""+value);			// impose desired rate
      setState(RESTART);				// perform a restart
    } else {
      rate=value;
    }
    fscale = nyfreq? 1.0 : 2.0/rate;
  }
  public void setFreq (double value) { 
    freq=value;
    if (ptype==pic.IOPT_TBANK) pic.setKeyD(dmac,pic.KEY_FREQ,value*fscale);
    else if (chan>0) pic.setKeyD(dmacs[chan-1],pic.KEY_FREQ,value*fscale); 
    else for (int n=0; n<multi; n++) pic.setKeyD(dmacs[n],pic.KEY_FREQ,(value+n*dfreq)*fscale); 
    afreq = pic.getKeyD(dmac,pic.KEY_FREQ)/fscale; // report actual freq used
    // if (afreq!=freq) M.info("Frequency "+freq+" rounded to "+afreq);
  }
  public void setFreqs (Data value) { 
    for (int i=0; i<value.size; i++) {
      pic.setKeyL(dmac,pic.KEY_CHAN,i+1);
      pic.setKeyD(dmac,pic.KEY_FREQ,value.getD(i)*fscale);
    }
  }
  public void setDec (int value) { 
    for (int n=0; n<multi; n++) pic.setKeyL(dmacs[n],pic.KEY_DEC,value); 
    dec = pic.getKeyL(dmac,pic.KEY_DEC);
    if (dec!=value) M.info("Decimation "+value+" rounded to "+dec);
    if (dmamode==pic.DMA_STOP);
    else if (autors>0) setReplay(rRESTART);
    //else if (tuner && tcmode!=pic.TCM_OFF) setReplay(rRESTART);
    setDelta(getDelta()); // update delta fields
  }
  private void setDelta (double value) {
    delta = value;
    if ((ipflag&ICEPacket.ABSC)!=0) {
      pkt.setAbscissa(0.0,delta);
    }
    if ((aipflag&ICEPacket.ABSC)!=0) {
      apkt.setAbscissa(0.0,delta);
    }
  }
  public void setGain (int value) { 
    if (agcPeriod>0) { // clip gain here for AGC modes 
      value = Math.max(minGain,Math.min(maxGain,value)); 
      if (value==gain) return;
    }
    gain = value; 
    if (ptype==pic.IOPT_TBANK) pic.setKeyL(dmac,pic.KEY_GAIN,gain);
    else if (chan>0) pic.setKeyL(dmacs[chan-1],pic.KEY_GAIN,gain); 
    else for (int n=0; n<multi; n++) pic.setKeyL(dmacs[n],pic.KEY_GAIN,gain); 
  }
  public void setRatio (double value) { 
    if (ptype==pic.IOPT_TBANK) pic.setKeyD(dmac,pic.KEY_RATIO,value);
    else if (chan>0) pic.setKeyD(dmacs[chan-1],pic.KEY_RATIO,value); 
    else for (int n=0; n<multi; n++) pic.setKeyD(dmacs[n],pic.KEY_RATIO,value); 
    ratio = pic.getKeyD(dmac,pic.KEY_RATIO);	// report actual ratio used
  }
  public void setSelect (int value) { 
    if (value==select) return;
    for (int i=1; i<=mcs; i++) {
      if (value>=0 && (select<0 || i==select)) { pic.setKeyL(dmac,pic.KEY_CHAN,i); pic.setKeyL(dmac,pic.KEY_ENABLE,0); }
    }
    for (int i=1; i<=mcs; i++) {
      if (value<0 || (value>0 && i==value)) { pic.setKeyL(dmac,pic.KEY_CHAN,i); pic.setKeyL(dmac,pic.KEY_ENABLE,1); }
    }
    select = value;
  }
  public void setChannel (int value) { 
    if (multi>1) {
      chan = Math.max(1,Math.min(multi,value)); 
      gain = pic.getKeyL(dmacs[chan-1],pic.KEY_GAIN);
      afreq = pic.getKeyD(dmacs[chan-1],pic.KEY_FREQ)/fscale;
      chan = Math.max(0,Math.min(multi,value)); 
    } else if (ptype==pic.IOPT_TBANK) {
      chan = value;
      pic.setKeyL(dmac,pic.KEY_CHAN,chan);
      afreq = pic.getKeyD(dmac,pic.KEY_FREQ)/fscale; // report actual freq used
    } else if (value>1) {
      M.warning("Should not be setting channel>1 when not in TBANK or MULTI modes");
    } else {
      chan = value;
    }
  }
  public void setEnable (int value) { 
    pic.setKeyL(dmac,pic.KEY_ENABLE,value);
  }
  public void setMonitor (String value) { 
    monitor = Parser.find(monitorList,value,monitor,-1); 
  }
  public void setStats (int value) {
    stats = value;
  }
  public void setMinGain (int value) { minGain = value; }
  public void setMaxGain (int value) { maxGain = value; }
  public void setAgcLevel (double value) { agcLevel = value; }
  public void setAgcWidth (double value) { agcWidth = value; }
  public void setGPW (GValue value) { gpw = value; }

  private String getReplay (int replay) { 
    return Parser.get(replayList,replay-replayListOffset);
  }
  public String getReplay() { return getReplay(replay); }
  public int getDec() { return dec; }
  public int getGain() { return gain; }
  public int getRate() { return rate; }
  public double getRatio() { return ratio; }
  public double getFreq() { return afreq; }
  public String getPort() { return sport; }
  public int getRfGain() { return rfgain; }
  public double getRfFreq() { return rffreq; }
  public int getChannel() { return chan; }
  public int getCycle() { return rcyc; }
  public long getIndex() { return rndx; }
  public double getProgress() { 
    if (maxout>0) return ho.seek()/maxout;
    if (archdur>0) return ha.getOffset()*ha.getDelta()/archdur;
    return ((double)rndx)/map.bytes;
  }
  public double getBytes() { return (((double)rcyc)*map.bytes + rndx); }
  public double getBytesOut() { return (((double)ocyc)*map.bytes + ondx); }
  public int getSkip() { return skip; }
  public String getMonitor() { return Parser.get(monitorList,monitor,-1); }
  public Time getTime() { return time; }
  public TimeCode getTimeCode() { return tclast; }
  public String getFormat() { return hr.getFormat(); }
  public int getHBLost() { return lost; }
  public int getPFull() { return pavg; }
  public int getCBLost() { return lostc; }
  public int getDmaMode() { return dmamode; }
  public double getOffset() { return ha.isOpen? ha.getOffset() : ho.getOffset(); }
  public MDevIce getDevIce() { return pic; }
  public Archiver getArchiver() { return ha; }
  public String getArchFN() { 
    if (archfn == null) return null;
    int i=archfn.indexOf('{');
    if (i>0) return archfn.substring(0,i);
    return archfn;
  }
  public String getArchMode() { return Parser.get(archList,archmode+1); }
  public double getArchDur() { return (archmode==aRT)? ha.getLength() : archdur; }
  public double getArchOff() { return (archmode==aRT)? ha.getOffset()*ha.getDelta() : archoff; }
  public double getArchTop() { return (archmode==aRT)? ha.getTimeAt(0) : 0; }
  public Table getArchKW()  { return archkw; }
  public double getSyncOff() { return syncoff; }
  public ICEPacket getPkt() { return pkt; }
  public ICEPacket getAPkt() { return apkt; }
  public float getMBPS() { return mbps; }
  public int getTCMode() { return tcmode; }
  public boolean isTuner() { return tuner; }

  public double getDelta() { 
    double delta = 1.0/rate;
    if (core) delta *= dec;
    if (tuner) delta *= (hr.spa*dec);
    if (ratio>0) delta /= ratio;
    return delta;
  }

  public double getChannelFreq (int chn) { 
    double fcny = 0.0;
    if (ptype==pic.IOPT_TBANK) {
      pic.setKeyL(dmac,pic.KEY_CHAN,chn);
      Data data = new Data(freq*2/rate);
      pic.getKey(dmac,pic.KEY_NFREQ,data);
      pic.setKeyL(dmac,pic.KEY_CHAN,chan);
      fcny = data.getD(0);
    } else {
      int n = Math.max(1,Math.min(multi,chn))-1;
      fcny = pic.getKeyD(dmacs[n],pic.KEY_FREQ); 
    }
    return fcny*rate/2;
  }

  private Table agcFreqs=null;;
  private int agcFreq=0;

  public void setAGC (Table tbl) {
    agcLevel = tbl.getD("LEVEL",0.0);
    agcWidth = tbl.getD("WIDTH",1.0);
    maxGain = tbl.getL("MAXGAIN",100);
    minGain = tbl.getL("MINGAIN",-100);
    agcPeriod = tbl.getD("PERIOD",1.0);
    agcTime = Time.current();
  }

  public void setRfGain (int value) {
    pic.setKeyL(dmac,pic.KEY_RFGAIN,value);
    rfgain=pic.getKeyL(dmac,pic.KEY_RFGAIN); 
    hr.keywords.put("L:RFGAIN",rfgain); 
    hr.update();
  }
  public void setRfFreqKWO (double value) {
    rffreq=value;
  }
  public void setRfFreq (double value) {
    if (agcLevel!=0) {	// reset AGC for RFFREQ scans
      int tfreq = (int)(Math.round(value*0.1)*10);
      if (tfreq!=agcFreq && agcFreq!=0) {
	if (agcFreqs==null) agcFreqs = new Table();
	agcFreqs.put("F"+agcFreq,gain);		// remember current Freqs gain
	Object tobj=agcFreqs.get("F"+tfreq);
	if (tobj!=null) { setGain(Convert.o2l(tobj)); agcTime=Time.current(); }		// restore new Freqs gain
      }
      agcFreq = tfreq;
    }
    pic.setKeyD(dmac,pic.KEY_RFFREQ,value);
    rffreq=pic.getKeyD(dmac,pic.KEY_RFFREQ);
    hr.keywords.put("D:RFFREQ",rffreq);
    hr.update();
    ha.addEventKey("RFFREQ",rffreq);
  }

  public Table getTable (Table t) { return new Table( pic.getKeyTable(dmac,t.toString()) ); }


  // use the keyable interface to expose all of the PIC library keys

  private void setKeys (Table kt) {
    if (kt==null || kt.size()==0) return;
    for (Table.Iterator ti=kt.iterator(); ti.getNext(); ) {
      String sval = (ti.value==null)? "QUERY" : ti.value.toString();
      if (sval.length()==0 || sval.equals("QUERY")) keys.put(ti.key,ti.value); // to show in Query
      else setKey(ti.key,ti.value); // actually set
    }
  }

  @Override
  public String[] getKeys() {
    return keys.getKeys();
  }

  @Override
  public synchronized Object setKey (String key, Object value) {
         if (key.equals("REPLAY")) setReplay(Convert.o2s(value));
    else if (key.equals("RATE"))   setRate(Convert.o2l(value));
    else if (key.equals("FREQ"))   setFreq(Convert.o2d(value));
    else if (key.equals("DEC"))    setDec(Convert.o2l(value));
    else if (key.equals("GAIN"))   setGain(Convert.o2l(value));
    else if (key.equals("RATIO"))  setRatio(Convert.o2d(value));
    else if (key.equals("RFFREQ"))  setRfFreq(Convert.o2d(value));
    else if (key.equals("RFGAIN"))  setRfGain(Convert.o2l(value));
    else if (key.equals("TABLE"))  value = pic.setKeyTable(dmac,value.toString());
    else if (key.startsWith("CHAN")) setChannel(Convert.o2l(value));
    else if (key.startsWith("SELECT")) setSelect(Convert.o2l(value));
    else {  // now expose all PIC library keys
      int ikey = pic.name2key(key);
      if (ikey<=0 && key.equals("VLAN")) ikey = pic.KEY_IPVLAN;
      if (ikey<=0 && key.equals("JOIN")) ikey = pic.KEY_IPCONN;
      if (ikey<=0 && key.equals("LEAVE")) ikey = pic.KEY_IPDISC;
      if (ikey<=0) return null;
      int ityp = pic.getKeyType(ikey);
      if (ikey==pic.KEY_IPCONN || ikey==pic.KEY_IPDISC) value = Integer.valueOf( pic.str2ip(Convert.o2s(value)) );
      if (ityp==DOUBLE) pic.setKeyD(dmac,ikey,Convert.o2d(value));
      else              pic.setKeyL(dmac,ikey,Convert.o2l(value));
      keys.put(key,value); // to show in Query
    }
    return value;
  }

  @Override
  public synchronized Object getKey (String key) {
    Object value = null;
         if (key.equals("REPLAY")) value=            getReplay();
    else if (key.equals("RATE"))   value=Convert.l2o(getRate());
    else if (key.equals("FREQ"))   value=Convert.d2o(getFreq());
    else if (key.equals("DEC"))    value=Convert.l2o(getDec());
    else if (key.equals("GAIN"))   value=Convert.l2o(getGain());
    else if (key.equals("RATIO"))  value=Convert.d2o(getRatio());
    else {  // now expose all PIC library keys
      int ikey = pic.name2key(key);
      if (ikey<=0) return null;
      int ityp = pic.getKeyType(ikey);
      if (ityp==DOUBLE) value=Convert.d2o(pic.getKeyD(dmac,ikey));
      else              value=Convert.l2o(pic.getKeyL(dmac,ikey));
    }
    return value;
  }

  short[] mcsbuf = new short[8];
  private int getMCSchn (long vaddr, int n, int bytes) {
    int off = n*bytes;
    Native.p2ja (vaddr,off, mcsbuf,0, 8);
    int chan = ((mcsbuf[0]&0x1)<<0) + ((mcsbuf[1]&0x1)<<1) + ((mcsbuf[2]&0x1)<<2) + ((mcsbuf[3]&0x1)<<3);
    if (++chan >= mcs) chan=mcs;
    return chan;
  }

  private void processWBAGC() {
    agcTime = Time.current();
    int adlm = pic.getKeyL(dmac,pic.KEY_ADLM);
    if (adlm<=-90) return;	// not valid
    double diff = adlm-agcLevel;
    int dgain = 0;
         if (diff>6) dgain=-3;
    else if (diff>1) dgain=-1;
    else if (diff<-1) dgain=1;
    else if (diff<-6) dgain=3;
    if (dgain!=0) setGain(gain+dgain);
  }

  short[] agcbuf;
  private void processNBAGC() {
    int n = (ptype==pic.IOPT_TBANK)? 256:8192;		// number of CI elements
    int loop = (ptype==pic.IOPT_TBANK)? nchan*4:1;	// number of loops through
    int bytes = n*4;	// number of bytes
    int done = (1<<nchan)-1;
    long offset = map.bytes/2;
    if (agcbuf==null) agcbuf = new short[n*2];
   for (int i=0; i<loop && done!=0; i++) {
    long vaddr = map.getVirtualAddress(offset,bytes);
    Native.p2ja (vaddr,0, agcbuf,0, bytes);
    float avg = (hr.spa==2)? calcAvgM(agcbuf,n) : calcAvg(agcbuf,n*2);
    avg = (float)(20.0*Math.log(avg));
    int gmod = 0;
    if (avg<agcLevel-agcWidth) gmod=1;
    if (avg<agcLevel-agcWidth*4) gmod=2;
    if (avg>agcLevel+agcWidth) gmod=-1;
    if (avg>agcLevel+agcWidth*4) gmod=-2;
    if (gmod==0);
    else if (ptype==pic.IOPT_TBANK) {
      int lchan=chan,lgain=gain;
      chan = getMCSchn(vaddr,0,bytes);
      int dmask = 1<<(chan-1);
      if ((dmask&done)!=0) {
        gain=gains[chan-1];
        pic.setKeyL(dmac,pic.KEY_CHAN,chan);
        setGain(gain+gmod);
        gains[chan-1]=gain;
        chan=lchan; gain=lgain;
        pic.setKeyL(dmac,pic.KEY_CHAN,chan);
        done ^= dmask;
      }
    } else {
      setGain(gain+gmod);
      done=0;
    }
    offset+=bytes;
   }
  }
  public int getChnGain (int chan) {
    return gains[Math.max(1,Math.min(nchan,chan))-1];
  }
  private float calcAvg (short[] buf, int ns) {
    float mavg=0;
    for (int i=0; i<ns; i++) {
      mavg += Math.abs(buf[i]);
    }
    return mavg/(ns*32768F);
  }
  private float calcAvgM (short[] buf, int ns) {
    float mavg=0;
    for (int i=0,j=0; i<ns; i++) {
      float ar=buf[j++], ai=buf[j++];
      mavg += Math.sqrt(ar*ar + ai*ai);
    }
    return mavg/(ns*32768F);
  }

  public static void setTime (DataFile df, double wsec, double fsec) {
    double timeu = 1e-6  * Math.floor(fsec*1e6+.5);
    int timep = (int)Math.floor((fsec-timeu)*1e12+.5);
    df.setTimeCode(wsec+timeu);
    df.keywords.putMain("TC_PREC",""+timep+"e-12");
  }

}

