package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.ice.lib.NetIO;
import nxm.ice.lib.Archiver;
import nxm.ice.lib.ICEPacket;
import nxm.ice.lib.VRTPacket;
import nxm.ice.lib.SDDSPacket;


/**
  Copies data from a devICE or other packet stream to an ordinary Midas file.

  @author Jeff Schoen
  @version $Id: sourceice.java,v 1.4 2009/05/26 13:07:53 jgs Exp $

*/
public class sourceice extends Primitive implements Keyable {

  public static String replayList="-NShot,File,Stopped,OneShot,Continuous,"
        +"StopTop,StopNow,Spin,Archive,Restart,Abort";
  public static int rFILE=-1,rSTOPPED=0,rONESHOT=1,rCONTINUOUS=2,
        rSTOPTOP=3,rSTOPNOW=4,rSPIN=5,rARCHIVE=6,rRESTART=7,rABORT=8;
  private static int replayListOffset=-3;

  public static String ptList = "NONE,ICE,SDDS,VRT,VRTL,VRTW,VRTX,VRTD";
  public static int ICE=1, SDDS=2, VRT=3, VRTL=4, VRTW=5, VRTX=6, VRTD=7;

  public static String pkthdrList = "Strip,Copy,Attach,Detach,Create";
  public static int STRIP=1,COPY=2,ATTACH=3,DETACH=4,CREATE=5;

  public static String sourceList = "Net,File,RtFile,DevIce";
  public static int NET=1,FILE=2,RTFILE=3,DEVICE=4;

  private NetIO nio;
  private Data data;
  private DataFile hi,ho,hm;
  private Archiver ha;
  private ICEPacket  opkh = null;
  private ICEPacket  ipkh = new ICEPacket();
  private VRTPacket  vpkh = new VRTPacket();
  private SDDSPacket spkh = new SDDSPacket();

  private boolean swap,wrap,net,usenat,check,attach,detach,skipframe,hasMon;
  private int replay,pt,bxoff,bxall,bxfer,bpa,bpe,errors,pkthdr,skip,iskip,frame,nper,stats,sbytes,scyc,spin,ispin,source,cons;
  private int mcs=0,chan=0,cena=-1;
  private double done,todo,drate,rtinbyte=0,rtoutbyte,rtbufsize,timer,timetop;
  private String format;
  private byte[] bdata;
  private Table keys = new Table();
  private Time time = new Time();

  public int open() {

    hi = MA.getDataFile ("IN","1000","S#,C#",0);
    net = hi.getURL().startsWith("udp:");
    source = MA.getSelectionIndex("/SOURCE",sourceList,net?NET:FILE);
    net    = (source==NET) || (source==DEVICE);
    format = MA.getFormat("FORMAT","S#,C#");
    drate  = MA.getD("RATE");

    pt     = MA.getState("/SDDS")? SDDS : ICE;
    pt     = MA.getChoice("/PT",ptList,pt+1)-1;
    check  = MA.getState("/SEQCHECK");
    usenat = MA.getState("/NATIVE",true);
    hasMon = MA.isPresent("/MON");
    mcs    = MA.getL("/MCS",0);

    todo   = MA.getD("/MAXOUT");
    frame  = MA.getL("/FRAME",0);
    spin   = MA.getL("/SPIN");
    wrap   = MA.getState("/WRAP");
    pkthdr = MA.getSelectionIndex("/PKTHDR",pkthdrList,(mcs>0)?CREATE:STRIP);

    if (source==NET) {
      nio = new NetIO(usenat);
      if (verbose) M.info("Opening ICE formatted network stream URL="+hi.getURL());
      if (nio.open(hi.getURL(),0,nio.INPUT)<=0) M.error("Problem opening ICE formatted network stream URL="+hi.getURL());
      ho = MA.getDataFile ("OUT","1000",format,0);
      bpa = hi.bpa;
      ho.setXDelta(1.0/drate);
      ho.setXUnits(Units.TIME);
      ho.setSize(Math.max(todo,4096/bpa));
      xfer = MA.getL("/TL", (frame>0)? 1 : (4096/bpa));
      if (frame>0) xfer *= frame;
      xfer = MA.getL("/PKTLEN",xfer*bpa) / bpa;
    }
    else if (source==DEVICE) {

    }
    else {
      hi.open();
      bpa = hi.bpa;
      drate = hi.getRate();
      ho = MA.getDataFile ("OUT",hi,0);
      ho.setSize(Math.max(todo,hi.getSize()));
      PacketHandler pkh = ho.getPacketHandler();
      if (pkh!=null) {
	String packet = ho.getPacket(); 
	int idet = packet.indexOf("/DET");
	if (pkthdr==STRIP) ho.setPacketHandler(null);
	else if (pkthdr==ATTACH && idet>=0) ho.setPacket(packet.substring(0,idet));
	else if (pkthdr==DETACH && idet<0) ho.setPacket("/DET");
	if (pt==ICE) xfer = ((ICEPacket)pkh).getSize();
	if (pt>=VRT) xfer = ((VRTPacket)pkh).getSize();
	if (pt==SDDS) xfer = ((SDDSPacket)pkh).getSize();
      } else {
	xfer = MA.getL("/TL", (frame>0)? 1 : (4096/bpa));
        if (frame>0) xfer *= frame;
      }
    }
    cons = MA.getL("/CL", xfer);
    setProgressFeed(ho,"SOURCEICE "+hi.getURL()+" to "+ho.getURL());
    if (source==RTFILE) {
      resyncRTFile(false);	// start at current write offset
      ho.setTime( hi.getTimeAt(rtoutbyte/bpa,null) );	// adjust time code 
    }
    skip=1;
    if (frame>0) {
      double monrate = MA.getD("/MONRATE",10.0);
      skip = (int)(drate/(frame*monrate));
      skip = MA.getL("/SKIP",skip);
      ho.setFS(frame);
      ho.setYUnits(Units.TIME);
      ho.setYDelta(frame/drate);
    }
    if (pkthdr==CREATE) {
      int ipflag = ICEPacket.FIXED | ICEPacket.ABSC;
      opkh = new ICEPacket(format,xfer,ipflag);
      ho.setPacketHandler(opkh);
    }
    if (skip>1 && !hasMon) ho.setYDelta(skip*frame/drate);
    ho.open(ho.OPTIONAL);
    ho.setDFS(0);

    hm = MA.getDataFile ("/MON",ho,0);
    if (skip>1 && hasMon) hm.setYDelta(skip*frame/drate);
    hm.open(hm.OPTIONAL);
    hm.setDFS(0);

         if (pt==SDDS) { bxoff = 56; xfer = (1024/bpa); } 
    else if (pt==ICE)  { bxoff = 64; } 
    else if (pt==VRT)  { M.error("VRT with trailer not supported"); }
    else if (pt>VRT )  { bxoff = (pt==VRTD)? 28 : 24; }
    else               { bxoff = 0; }

    bpe   = bpa*xfer;
    bxall = bxoff + bpe;
    data  = hm.getDataBuffer(bxall/bpa);
    bdata = data.buf;
    swap  = MA.getState("/SWAP",(pt==SDDS) && (bpa>=2));
    nper  = (frame>0)? frame/xfer : 1;
    skip *= nper;
    iskip = -1;
    skipframe = (skip>1) || (frame>0);

    if (M.pipeMode==Midas.PINIT) replay=rCONTINUOUS; else replay=rFILE;
    replay = MA.getSelectionIndex("/REPLAY",replayList,replay,replayListOffset);
    time.fromCurrent();

    startArchiver();

    // get status logger
    stats  = MA.getL("/STATS");

    return (NORMAL);
  }

  public int process() {
    int i=0,n=0;
    if (replay==rABORT) return(FINISH);
    if (replay==rSTOPPED) return(NOOP);
    if (source==NET) {
      i = (pkthdr==STRIP)? bxoff:0;
      n = nio.recv(0,bdata,0,bxall,0);
    }
    else if (source==RTFILE) {
      if (rtoutbyte+bpe>rtinbyte) {
        hi.ioh.read (hi.hb,0,512,0L);
        rtinbyte=hi.getInByte();
        if (rtoutbyte+bpe>rtinbyte) return NOOP;
      }
      double offset = (rtoutbyte % rtbufsize) / bpa;
      n = hi.read(data,offset,xfer);
      if (cons>xfer) { n=xfer; rtoutbyte += cons*bpa; }
      else rtoutbyte += n*bpa;
    }
    else {
      n = hi.read(data,xfer);
    }
    if (n>0) {
      sbytes += n; 
      ispin = spin;
      if (check) checkPacket();
      if (mcs>0) {
	int chan = getMCSchn(bdata);
	if (((0x1<<(chan-1)) & cena) == 0) return NORMAL; 
	opkh.setChannel(chan);
      }
      if (swap) Convert.swap2(bdata,64,512);
      if (ha!=null) {			// Archiver output
        if (!net) ha.setPktHeader( (ICEPacket)hi.getPacketHandler() );
        ha.write(0,bdata,0,n,0);
      }
      if (++iskip>=skip) iskip=0;
      boolean doskip = skipframe && iskip>=nper;
      if (hm.isOpen && !doskip) {	// Monitor output
	hm.write(data,n);
      }
      if (ho.isOpen && (hasMon || !doskip)) {	// Normal output
        if (net) ho.write(bdata,i,n-i); 
        else     ho.write(data,n);
      }
    }
    else if (n<0) { if (wrap) hi.seek(0.0); else return FINISH; }
    else if (spin>0 && --ispin<0) return NOOP;
    else return NOOP;
    done += xfer;
    timer = Time.current();
    if (stats>0 && timer-timetop>stats) {
      float mbps = (float)(1.e-6 * sbytes / Math.max(0.01,timer-timetop));
      M.info("SourceIce STATS={CYC="+scyc+",MBPS="+mbps+",SEQERR="+errors+"}");
      timetop = timer;
      sbytes = 0;
      scyc++;
    }
    if (todo>0 && done>todo) return (FINISH);
    return (NORMAL);
  }

  public int close() {
    if (nio!=null) nio.close();
    hi.close();
    hm.close();
    ho.close();
    if (ha!=null) ha.close();
    if (check) M.info("Transfer addr="+hi.getURL()+" size="+done+" errors="+errors);
    return (NORMAL);
  }

  public void setReplay (String value) {
    if (value.equalsIgnoreCase("STOP")) value = "STOPNOW";
    if (value.equalsIgnoreCase("START")) value = "CONTINUOUS";
    int nreplay = Parser.find(replayList,value,replay,replayListOffset);
    if (nreplay>replayListOffset) setReplay(nreplay);
    else M.warning("Illegal replay mode: "+value);
  }
  private void setReplay (int nreplay) {
    if (nreplay==replay) return;
         if (nreplay==0 && replay!=0) replay = rSTOPNOW;
    else if (nreplay==rRESTART && replay==0) replay = rCONTINUOUS;
    else if (nreplay==rONESHOT && replay==rCONTINUOUS) replay = rSTOPTOP;
    else replay=nreplay;
    if (nreplay==rCONTINUOUS || nreplay==rONESHOT) {
      timetop = Time.current();
      if (source==RTFILE) resyncRTFile(true);
    }
    if (nreplay==rCONTINUOUS && stats>0) M.info("SourceIce START");
    if (nreplay==rSTOPNOW && stats>0) M.info("SourceIce STOP");
    if (replay==rSTOPNOW || replay==rSTOPTOP) replay = rSTOPPED;
  }
  private String getReplay (int replay) {
    return Parser.get(replayList,replay-replayListOffset);
  }
  public String getReplay() { return getReplay(replay); }
  public int getCycle() { return 0; }
  public double getOffset() { return ha.isOpen? ha.getOffset() : ho.getOffset(); }

  private void resyncRTFile (boolean reload) {
    if (reload) hi.ioh.read (hi.hb,0,512,0L);
    rtinbyte = hi.getInByte();
    rtbufsize = hi.getDataSize();
    rtoutbyte = Math.round(rtinbyte/bpe)*bpe;
  }

  int total=0, nextseq=-1;
  private void checkPacket() {
    int seq;
    if (pt==SDDS) {
      spkh.setBuffer(bdata,0);
      seq = spkh.getCount();
    }
    else if (pt>=VRT) {
      vpkh.setBuffer(bdata,0);
      seq = vpkh.getCount();
    }
    else {
      ipkh.setBuffer(bdata,0);
      seq = ipkh.getCount();
    }
    if (nextseq>0) {
      if (seq!=nextseq && ++errors<20) M.warning("Sequence gap curr="+seq+" next="+nextseq);
    }
    if (++total%100==0) {
      if (stats<=0) M.info("Packet total="+total+" samples="+(total*xfer));
    }
    nextseq = seq+1;
    if (pt==SDDS) {
      if ((nextseq&0x1F)==0x1F) nextseq++;
      nextseq &= 0xFFFF;
    } else if (pt>=VRT) {
      nextseq &= 0xF;
    }
  }

  public void setGain (int gain) {
  }

  private int getMCSchn (byte[] buf) {
    int i = ((buf[0]&0x1)<<0) + ((buf[2]&0x1)<<1) + ((buf[4]&0x1)<<2) + ((buf[6]&0x1)<<3);
    if (++i>mcs) i=mcs;
    return i;
  }

  public void setChan (int ichan) { chan = ichan; }
  public void setEnable (int ena) {
    int mask = 0x1<<(chan-1);
    if (ena==0) cena &= ~mask; else cena |= mask;
  }

  private void startArchiver() {
    String fname = MA.getS("/ARCH");
    int nchan = MA.getL("/NCHAN",0);
    if (StringUtil.isNullOrEmpty(fname)) return;
    ICEPacket apkt = new ICEPacket(ho.getFormat(),xfer,0);
    ha = new Archiver();
    ha.init (this,fname,ho,0);
    ha.setPacketHandler(apkt);
    ha.setNChan(nchan);
    ha.setFS(0);
    ha.open(ha.NATIVE|ha.OPTIONAL);
    if (!ha.isOpen) ha=null;
  }

  // use the keyable interface to expose all of the PIC library keys
  public String[] getKeys() {
    return keys.getKeys();
  }
  public Object setKey (String key, Object value) {
    if (!thisIsMe()) MQ.put("SET."+key,0,value); // force all sets to be performed from this thread
    else if (key.equals("REPLAY")) setReplay(Convert.o2s(value));
    else if (key.equals("VLAN")||key.equals("IPVLAN")) ;
    else if (key.equals("JOIN")||key.equals("IPCONN")) ;
    else return null;
    return value;
  }
  public Object getKey (String key) {
         if (key.equals("TIME")) return time;
    else if (key.equals("CYCLE")) return new Data(0);
    else if (key.equals("STATUS")) return new Data(replay);
    else return null;
  }

}
