/*
   Non-Midas routine for USB arduino to ICE interface functions

   Author: Jeff Schoen

*/
#ifdef _LINX

#ifdef JEFF
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#endif

#define NOOPEN -1
#define NORESP -2
#define BADRESP -3

/*
  
  Common XCVR fields in PIC struct 

  xcvr=0	No XCVR flag
  xcvr=1	Valid XCVR flag	- send RFATTN to Xcvr
  xcvr=-1	Valid XCVR=bypass flag - ignore all RFATTN calls

  xena=1	Freq in range - no Xcvr bypass
  xena=0	Freq below usable range - Xcvr bypass

  xinv=0|1	Spectrum inverted

  xchn=1|2	Channel on AR/AT/AX devices

*/


int dogetstrw (char *str, int len, int fno, double wait) {
  int i,stat;
  for (i=0; i<len && wait>0; ) {
    stat = read(fno,str+i,1);
    if (stat<=0) { usleep(10000); wait-=.010; continue; }
    i += stat;
    if (str[i-1]=='\n') { str[i-1]=0; return i-1; }
  }
  return -1;
}

int doXcvrCmd (char *devn, char *cmd, char *resp) {
  int pass,fno,status=0; char *s,fname[40],func[40],str[256];
  s = strchr(devn,'-'); s = (s==NULL)? str : s+1;
  sprintf(fname,"/dev/icetty-%s",s);
  pic_xlock(1);
  fno = open(fname,O_RDWR|O_NONBLOCK);	/* non-blocking for normal Arduino failsafe reads and writes */
  if (fno<0) { printf("Err:  Xcvr %s  Cmd %s not sent\n",fname,cmd); status=NOOPEN; goto DONEX; }
  read(fno,str,160);			/* purge old data */
  write (fno, cmd, strlen(cmd)+1);	/* send command */
  if (dogetstrw(str,32,fno,1.0)<=0) { status=NORESP; goto DONE; }
  if (strstr(str,"CMD=")==NULL) { status=BADRESP; goto DONE; }
  for (pass=0;pass<3;pass++) {		/* final text output */
    if (dogetstrw(str,128,fno,2.0)==0) continue;
    if (strstr(str,"endCMD")!=NULL) break;
    status = strlen(str);
    strcpy(resp,str);
  }
  DONE:
  close(fno);
  DONEX:
  pic_xlock(-1);
  return status;
}

double doXcvrFunc (PICSTRUCT *p, char *func, double dval) {
  char *s,str[40],resp[80]; int i,ival;
  int isAR = strncmp(p->xname,"ar",2)==0;
  int isAT = strncmp(p->xname,"at",2)==0;
  int isAX = strncmp(p->xname,"ax",2)==0;
  int isLNB = strncmp(p->xname,"lnb",3)==0;
  if (p->xcvr<=0) return dval;
  if (isAR || isAT || isAX) {
    if (strcmp(func,"FREQ")==0) {
      if (dval<p->xcfb) { p->xfreq=0.0; p->xena=0; }
      else { p->xfreq = round(dval/5)*5; dval = (dval-p->xfreq) + p->xcf; p->xena=1; }
      if (isAR) p->xinv = (p->xena==0)? 0 : (p->xfreq<6000 || p->xfreq>=18000)? 1 : 0; 	/* 1-26 GHz */
      else      p->xinv = (p->xena==0)? 0 : (p->xfreq<=5500 || p->xfreq>=15500)? 1 : 0;	/* 1-18 GHz */
      sprintf(str,"freq%d=%f",p->xchn,p->xfreq);
      doXcvrCmd (p->xname,str,resp);
    }
    else if (strcmp(func,"ATTN")==0) {
      dval=fmin(30.0,fmax(0.0,dval));
      sprintf(str,"attn%d=%d",p->xchn,(int)dval);
      doXcvrCmd (p->xname,str,resp);
    }
    else if (strcmp(func,"SETUP")==0) {
      /* make sure the 10MHz reference is correct */
      ival = (findintflag("PREFX",p->config)>0)? 1 : 0;
      ival = findintflagdef("XREF",p->config,ival);
      sprintf(str,"ref=%d",ival);
      doXcvrCmd (p->xname,str,resp);
    }
    else if (strcmp(func,"ENABLE")==0) {
      /* set of RFFREQ and RFGAIN should have done this */
    }
    else if (strcmp(func,"DISABLE")==0) {
      doXcvrFunc(p,"FREQ",0.0);	/* leave in bypass */
    }
    else if (strcmp(func,"STATUS")==0) {
      i=doXcvrCmd (p->xname,"status",resp);
      if (i<0) sprintf(resp,"Bad response status=%d\n",i);
      printf("Xcvr %s\n",resp);
    }
  }
  else if (isLNB) {
    if (strcmp(func,"FREQ")==0) {
      dval = finddblflagdef("MFREQ",p->config,dval-p->xfreq);	/* is module freq fixed ? */
      if (p->xinv!=0) dval = -dval;
      vprint("Test Xcvr=%s Inv=%d LO=%f Freq=%f\n",p->xname,p->xinv,p->xfreq,dval);
    }
  } 
  else {
    printf("Unsupported XCVR type: %s\n",p->xname);
  }
  return dval;
}

int doXcvrPrep (PICSTRUCT *p, int mode) {
  switch (mode) {
    case -1: doXcvrFunc(p,"SETUP",0.0); break;
    case  1: doXcvrFunc(p,"ENABLE",0.0); break;
    case  0: doXcvrFunc(p,"DISABLE",0.0); break;
  }
  return 0;
}

int doXcvrInit (PICSTRUCT *p) {
  char *s,str[40]; int i; double d=0;
  /* defaults for no XCVR flag modes */
  p->xcf = finddblflagdef("XCF",p->config,0.0)*1000;
  p->xcfb = finddblflagdef("XCFB",p->config,0.0)*1000;
  p->xcvr = p->xchn = p->xena = p->xinv = 0;
  /* now check for XCVR flag mode */
  i = findstrflag("XCVR",p->config,str,-1);
  if (i<=0) return -1;
  if (strcmp(str,"bypass")==0) { p->xcvr=-1; return 0; }	/* use xcvr as bypassed */
  /* defaults for XCVR flag modes */
  p->xcf = finddblflagdef("XCF",p->config,2.0)*1000;
  p->xcfb = finddblflagdef("XCFB",p->config,1.4)*1000;
  p->xchn = p->pindex; /* get channel number */
  if (p->xchn>0) p->xcvr = p->xena = 1;
  s = strchr(str,':');
  if (s!=NULL) { i=strlen(s)-1; p->xinv=(s[i]=='<')? -1:0; s[i]=0; d=atof(s+1); s[0]=0; } 
  s = strchr(str,'.');
  if (strncmp(str,"lnb",3)==0) { p->xfreq=d*1000; p->xcf=0; }
  else if (s!=NULL) { p->xchn = atol(s+1); s[0]=0; }
  strcpy(p->xname,str);
  vprint("XcvrInit name=%s chn=%d inv=%d xcf=%lf xfreq=%lf\n",p->xname,p->xchn,p->xinv,p->xcf,p->xfreq);
  return 0;
}

#endif
