#define AWG_NUM_PRG_BITS      32
#define AWG_REF_OSC           10000000

#define AWG_SYSCFG_INDEX      0x00000000
#define AWG_INITCFG_INDEX     0x10000000
#define AWG_GENCFG_INDEX      0x20000000
#define AWG_ATTCFG_INDEX      0x30000000
#define AWG_DACCFG_INDEX      0x40000000
#define AWG_DACOP_INDEX       0x50000000
#define AWG_GAINCFG_INDEX     0x60000000
#define AWG_SCLKCFG_INDEX     0x70000000
#define AWG_PWRCFG_INDEX      0x80000000
#define AWG_RAMCFG_INDEX      0x90000000
#define AWG_RMSCFG_INDEX      0xA0000000
#define AWG_ENCFG_INDEX       0xB0000000
#define AWG_WAITCNT_INDEX     0xC0000000
#define AWG_DUPCNT_INDEX      0xD0000000
#define AWG_DACRESET_INDEX    0xE0000000
#define AWG_CLKINV_INDEX      0xF0000000

/* General Config Reg Bit Defs  */
#define AWG_GENCLKSEL_BIT     0x00000001             /* Select External Clock                                 */
#define AWG_16BITDATA_BIT     0x00000002             /* Select 16 Bit Data                                    */  
#define AWG_PPSTRIG_BIT       0x00000010             /* Enable PPS Triggered Output                           */

/* AWG Operation Bit Degs       */
#define AWG_DACINIT_BIT       0x00000001
#define AWG_DACPROG_BIT       0x00000002
#define AWG_DACUPD_BIT        0x00000004 

#define MAX_TUNWRD_VAL        4294967296.0           /* AWG Max Tune Word Float 2^32                          */


/* Synthetic Clock Commands     */
#define AWG_5326_SETADR       0x00000000             /* Set Address To Be Written                             */
#define AWG_5326_WRDAT        0x00400000             /* Write Data                                            */

/* RAM Storage Adress Offsets   */
#define AWG_AWGCFG_ADR        0x00000000             /* Start Address Of AWG     Config Reg Storage           */
#define AWG_ANYRCFG_ADR       0x00000040             /* Start Address Of AnyRate Config Reg Storage           */
#define AWG_FCRATEREQ_ADR     0x000000F8             /* AWG FSysclk Requested Rate, From AnyRate Part         */
#define AWG_FCRATEACTL_ADR    0x000000F9             /* AWG FSysclk Actual Rate Low, From AnyRate Part        */
#define AWG_FCRATEACTH_ADR    0x000000FA             /* AWG FSysclk Actual Rate High, From AnyRate Part       */                                          
#define AWG_CENTFREQL_ADR     0x000000FB             /* Center Frequency Low  32 Bit Word                     */ 
#define AWG_CENTFREQH_ADR     0x000000FC             /* Center Frequency High 32 Bit Word                     */  
#define AWG_INTERP_ADR        0x000000FD             /* Stored Interpolation Value                            */ 
#define AWG_RAMATTN_ADR       0x000000FE             /* Address For Current Attenuation Value                 */
#define AWG_RAMPPM_ADR        0x000000FF             /* Address For VCTCXO PPM Value                          */ 

#define AWG_RDMEM_ADR         0x10000000             /* Mem Mux Address For JTAG Read Back                    */
  
/* Debug Level                  */
#define D2AWG_DBGPRINT        0

/*---------------------------------------------------------------------------*/
int_4 d2awg_set_tcxo (PICSTRUCT *p, int_4 mport, real_8 rtcxo)
{
  int_4 itcxo = 0;

  if ((rtcxo < -1.0) || (rtcxo > 1.0))
   {  
      printf("ERR: Valid VCTCXO Range {-1.0, 1.0} \n");
      printf("ERR: Desired Value %f out of range!\n",rtcxo);
      return(-1);
   }

  itcxo = (2047.5)*(rtcxo + 1.0);
  itcxo = (itcxo < 0)    ?    (0) : (itcxo);
  itcxo = (itcxo > 4095) ? (4095) : (itcxo);
  
  /* Write to VCTCXO Control DAC Register */
   pic_wmodreg(p,mport,AWG_GAINCFG_INDEX,0x00300000|(itcxo<<4),AWG_NUM_PRG_BITS);   
   if (D2AWG_DBGPRINT >= 1) 
     printf("D2AWG(W) TCXO: %d (DAC)  \n",itcxo);
  
  return (0);
}

/*---------------------------------------------------------------------------*/
/* D2AWGR1 - Query A/D attached to AD5538 RMS Power Detector                 */
int_4 d2awg_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
/* Power Bit Settings       */
#define D2AWG_PWR_SLEEP       0x00000001
#define D2AWG_PWR_WAKE        0x00000002
#define D2AWG_PWR_ACTIVE      0x00000004

  const int_4 numavg = 4;
  const real_8 fit_coef[] = {2665.40, 32.9714};
  int_4  ii,idBm;
  real_8 pdBm;
  int_u4 mdata = 0;
  int_u4 sumpwr = 0;

 /* Enable RMS Power Meter */ 
  pic_wmodreg(p,mport,AWG_RMSCFG_INDEX,1,AWG_NUM_PRG_BITS);
  udelay(3000);

  /* Average power measurements */
  for(ii=0;ii<numavg;ii++)   /* &BILL should be 4 for averaging loop */
    {
     /* Turn Off A/D, A/D Sleep, A/D Wake, A/D Active, Turn Off A/D */
      pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,0,AWG_NUM_PRG_BITS);
      udelay(3000);
      pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,D2AWG_PWR_SLEEP,AWG_NUM_PRG_BITS);
      udelay(3000);
      pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,D2AWG_PWR_WAKE,AWG_NUM_PRG_BITS);
      udelay(5000);
      pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,D2AWG_PWR_ACTIVE,AWG_NUM_PRG_BITS);
      udelay(3000);
      pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,0,AWG_NUM_PRG_BITS);
      udelay(3000);
     /* Read A/D value from register */
      pic_rmodreg (p,mport,0x20000000,&mdata,AWG_NUM_PRG_BITS); 
      sumpwr += mdata;           
    }

  /* Power A/D back to Sleep     */
   pic_wmodreg(p,mport,AWG_PWRCFG_INDEX,D2AWG_PWR_SLEEP,AWG_NUM_PRG_BITS);
   udelay(3000);

  /* Disable RMS Power Meter     */ 
   pic_wmodreg(p,mport,AWG_RMSCFG_INDEX,0,AWG_NUM_PRG_BITS);
   udelay(3000);

  /* Computer total input power in dBm as function of A/D reading */
  pdBm = (1.0/fit_coef[1])*(((real_8)sumpwr/(real_8)numavg) - fit_coef[0]);
  pdBm = pdBm - 3.0;
  idBm = (int_4) pdBm;

  if (D2AWG_DBGPRINT >= 2) 
    printf("D2AWG(R) RFPWR: %i (dBm)  %i (ADC)  %15.2f \n",idBm,(sumpwr/numavg),pdBm);

  return(idBm);
}

/*--------------------------------------------------------------*/
int_4 d2awg_set_attn(PICSTRUCT *p, int_4 mport,int value)
{
  if ((value < 0) || (value > 31))
    {
      printf("ERR: Valid Attenuation Values 0 to 31, inclusive\n");
      printf("ERR: Desired Value %d out of range!\n",value);
      return(-1); 
    }  

 /* Store Attenuation Value */
  pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_RAMATTN_ADR,value,AWG_NUM_PRG_BITS); 
  
 /* Set Attenuation Value   */   
  pic_wmodreg(p,mport,AWG_ATTCFG_INDEX,value,AWG_NUM_PRG_BITS);
  udelay(3000);
  return(0);
} 
/*--------------------------------------------------------------*/
int_4 d2awg_get_attn (PICSTRUCT *p,int_4 mport)
{ 
  int_u4 mdata;
 /* Read Attenuation Value RAM */ 
  pic_rmodreg (p,mport,0x10000000|AWG_RAMATTN_ADR,&mdata,AWG_NUM_PRG_BITS);

  return(mdata); 
}
/*--------------------------------------------------------------*/
int_4 d2awg_initcfg_ad9957 (PICSTRUCT *p, int_4 mport)
{ 
  
  pic_wmodreg(p,mport,AWG_DACOP_INDEX,AWG_DACINIT_BIT,AWG_NUM_PRG_BITS);                     /* Init AD9957 Config Interface      */
  udelay(100);

  return 0;
}
/*--------------------------------------------------------------*/
int_4 d2awg_commitcfg_ad9957 (PICSTRUCT *p, int_4 mport)
{ 
  
  pic_wmodreg(p,mport,AWG_DACOP_INDEX,AWG_DACUPD_BIT,AWG_NUM_PRG_BITS);                      /* Init AD9957 Config Interface      */
  udelay(100);

  return 0;
}
/*--------------------------------------------------------------*/
int_4 d2awg_wmodreg_ad9957 (PICSTRUCT *p, int_4 mport,int_u4 regnum,int_u4 mswval,int_u4 lswval)
{   

#define MAXREGNUM   25
#define MAXREGSZ    64
  int_u4 regsiz[26] = {32,32,32,32,32,48,48,65,65,32,32,65,65,
                       65,64,64,64,64,64,64,64,64,32,65,16,16};               /* Bit Size AWG Regs, 65 Illegal Size, No Register  */
  int_u4 shiftdif   = 0;
  int_u4 shiftbit   = 0;

   if ((regnum > MAXREGNUM) || (regsiz[regnum] > MAXREGSZ)) 
     {
       printf("AWG Err: Write To AWG Register Out Of Range Or Write To Reserved Register.\n");
       return(-1);
     }

 /* Store AWG Cfg Reg Values */
  pic_wmodreg(p,mport,(AWG_RAMCFG_INDEX|AWG_AWGCFG_ADR)+(regnum*2),lswval,AWG_NUM_PRG_BITS);
  pic_wmodreg(p,mport,(AWG_RAMCFG_INDEX|AWG_AWGCFG_ADR)+(regnum*2)+1,mswval,AWG_NUM_PRG_BITS);  

 /* printf(" Reg Num = %d Num Bits = %d MSVAL = %x LSVAL = %x\n",regnum,regsiz[regnum],mswval,lswval); */

  /* Shift Bits To High Order For MSB Load Into AD9957 */
   shiftdif = MAXREGSZ - regsiz[regnum];
   while (shiftdif != 0)
     {        
       shiftbit = ((lswval >> 31) & 0x00000001);
       mswval   = ((mswval <<  1) & 0xFFFFFFFE) | shiftbit;
       lswval   =  (lswval <<  1) & 0xFFFFFFFE;
       shiftdif = shiftdif - 1;
     } 
     
 /*  printf(" Reg Num = %d Num Bits = %d MSVAL = %x LSVAL = %x\n",regnum,regsiz[regnum],mswval,lswval); */
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,regsiz[regnum] + 8,AWG_NUM_PRG_BITS);                /* Set Reg Shift Size + Control Byte */  
   udelay(100);  
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,regnum,AWG_NUM_PRG_BITS);                            /* Set Wr Op And Register Number     */
   udelay(100);

   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,mswval >> 24,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,mswval >> 16,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100); 
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,mswval >>  8,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,mswval >>  0,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);

   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,lswval >> 24,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,lswval >> 16,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100); 
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,lswval >>  8,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);
   pic_wmodreg(p,mport,AWG_DACCFG_INDEX,lswval >>  0,AWG_NUM_PRG_BITS);                      /* Set Reg Data MS Byte Per Trans    */
   udelay(100);
 
   pic_wmodreg(p,mport,AWG_DACOP_INDEX,AWG_DACPROG_BIT,AWG_NUM_PRG_BITS);                    /* Send AWG Config To AD9957         */
   udelay(100);

   return 0;    
}           
/*--------------------------------------------------------------*/
int_4 d2awg_rmodreg_ad9957 (PICSTRUCT *p, int_4 mport,int_u4 regnum,int_u4 *mswval,int_u4 *lswval)
{
 /* Read AWG Config Reg Value From RAM */ 
  pic_rmodreg (p,mport,(0x10000000|AWG_AWGCFG_ADR)+(regnum*2),lswval,AWG_NUM_PRG_BITS);
  pic_rmodreg (p,mport,(0x10000000|AWG_AWGCFG_ADR)+(regnum*2)+1,mswval,AWG_NUM_PRG_BITS);
  return 0;
}                
/*--------------------------------------------------------------*/
int_4 d2awg_wmodreg_si5326 (PICSTRUCT *p, int_4 mport,int_u4 value)
{   
   udelay(100);
   pic_wmodreg(p,mport,AWG_SCLKCFG_INDEX,AWG_5326_SETADR| (value & 0x0000FF00),AWG_NUM_PRG_BITS);            /* Set Reg Addr Of Wr */  
   udelay(100);
   pic_wmodreg(p,mport,AWG_SCLKCFG_INDEX,AWG_5326_WRDAT | ((value << 8) & 0x0000FF00),AWG_NUM_PRG_BITS);     /* Set Wr Reg Value   */    
   if (value == 0x00008840) 
     udelay(53000);                                                                                          /* ICAL Up To 53msec  */ 
   udelay(100);
   return 0;    
} 

/*--------------------------------------------------------------*/
int_4 d2awg_set_tenthsppm (PICSTRUCT *p, int_4 mport,int_4 ppm)
{
#define AWG_DEVICE_MAXPPM    50.0      /* DEV-Max Shift Of VCTCXO 50/10 = 5.0 PPM +/-   */
#define AWG_USER_MAXPPM      25.0      /* USER-Max Shift Of VCTCXO 25/10 = 2.5 PPM +/-  */ 
#define AWG_VCTCXODAC_V_MAX  2.5       /* Max Voltage Out Of DAC Controlling VCTCXO     */
#define AWG_VCTCXODAC_C_MAX  4096.0    /* Max Count+1 Of DAC Controlling VCTCXO         */
#define AWG_VCTCXO_MIDPT     1.65      /* VCTCXO 0 PPM Voltage                          */
 
  
  float dacval = 0.0;
  if (abs(ppm) > ((int_u4)(AWG_USER_MAXPPM)))
    {
      printf("Err: Maximum PPM Setting Of VCTCXO is +/- %d Tenths Of PPM\n",((int_u4)(AWG_USER_MAXPPM)));
      printf("Err: Valid Values Are -%d<=ppm<=%d \n",((int_u4)(AWG_USER_MAXPPM)),((int_u4)(AWG_USER_MAXPPM)));
      return(-1);  
    }
  
 /* Store PPM Value */
  pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_RAMPPM_ADR,ppm,AWG_NUM_PRG_BITS); 
  udelay(1000);

  dacval =  (AWG_VCTCXO_MIDPT/AWG_VCTCXODAC_V_MAX) *  AWG_VCTCXODAC_C_MAX;  /* VCTCXO Mid Point Voltage, 0 PPM DAC Value */  
  dacval =  dacval + ((AWG_VCTCXODAC_C_MAX/AWG_VCTCXODAC_V_MAX) * (AWG_VCTCXO_MIDPT/AWG_DEVICE_MAXPPM) * ppm) + 0.5;
  
  pic_wmodreg(p,mport,AWG_GAINCFG_INDEX,0x00300000|(((int_u4)(dacval)) << 4),AWG_NUM_PRG_BITS); 

  if (D2AWG_DBGPRINT >= 1) 
    printf("VCTCXO DAC Value = %d\n",(int_u4)(dacval));

  return(0);
}
/*--------------------------------------------------------------*/
int_4 d2awg_get_tenthsppm (PICSTRUCT *p, int_4 mport,int_4 ppm)
{
  int_u4 mdata;

 /* Read PPM Value RAM */ 
  pic_rmodreg (p,mport,0x10000000|AWG_RAMPPM_ADR,&mdata,AWG_NUM_PRG_BITS);

  return(mdata);
}
/*--------------------------------------------------------------*/
int_4 d2awg_set_clat(PICSTRUCT *p, int_4 mport,int_4 value)
 {
  /* Store New CLAT In Mem   */ 
   pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_ANYRCFG_ADR,value & 0x000000FF,AWG_NUM_PRG_BITS);

  /* Set Address Of CLAT Reg */
   value = (value & 0x000000FF) | 0x00001000;
   
   d2awg_wmodreg_si5326 (p,mport,value);

   return(0);
 }
/*--------------------------------------------------------------*/
int_4 d2awg_set_freq (PICSTRUCT *p, int_4 mport, int_u4 cfval)
{ 
  
  int_u4 fcrate,awg28reg,awg29reg,actfreq_int; 
  real_8 freq_dflt,actfreq_dflt;
 
 /* Read Fsysclk Rate                 */
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_FCRATEREQ_ADR,&fcrate,AWG_NUM_PRG_BITS);               
   
 /* Find User Center Frequency        */ 
  freq_dflt = (real_8) cfval; 
        
 /* Set Register Storage Values       */
  awg28reg = (int_u4) (((real_8)(freq_dflt/((real_8)(fcrate)))) * MAX_TUNWRD_VAL +0.5);      /* Freq Tuning Word-Reg 28    */
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_AWGCFG_ADR)+29,&awg29reg,AWG_NUM_PRG_BITS);       /* Retrieve Interp-Reg 29     */

 /* Store Actual Center Freq          */ 
  actfreq_dflt  = ((real_8) awg28reg)/((real_8) MAX_TUNWRD_VAL) * ((real_8)fcrate);       
  actfreq_int  = (int_u4) actfreq_dflt;
  pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_CENTFREQH_ADR,actfreq_int,AWG_NUM_PRG_BITS);      /* Store Integer Cent Freq    */ 
  
  if ((freq_dflt - actfreq_dflt) != 0.0)
    printf("WARN: Desired Center Freq = %e  Actual = %e\n",freq_dflt,actfreq_dflt);          /* Warning For Calc Error     */  
     
  actfreq_dflt = (actfreq_dflt - ((real_8) actfreq_int))  * 1000000.0;                       /* Get 6 Decimal Places       */
  actfreq_int  = (int_u4) actfreq_dflt;
  pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_CENTFREQL_ADR,actfreq_int,AWG_NUM_PRG_BITS);

  

  d2awg_wmodreg_ad9957 (p,mport,14,awg29reg,awg28reg);                                       /* Write New Center Freq      */  
 
  d2awg_commitcfg_ad9957(p,mport);                                                           /* Update AD9957 Regs         */
  udelay(100);

  return 0;
}
/*--------------------------------------------------------------*/
real_8 d2awg_get_freq (PICSTRUCT *p, int_4 mport)
{   
  int_u4 actfreq_int; 
  real_8 actfreq_dflt;

 /* Retrieve Decimal Actual Center Freq  */   
  pic_rmodreg(p,mport,AWG_RDMEM_ADR|AWG_CENTFREQL_ADR,&actfreq_int,AWG_NUM_PRG_BITS);       
  actfreq_dflt = ((real_8) actfreq_int)/1000000.0;  
 
 /* Retrieve Integer Actual Center Freq  */
  pic_rmodreg(p,mport,AWG_RDMEM_ADR|AWG_CENTFREQH_ADR,&actfreq_int,AWG_NUM_PRG_BITS);       
  actfreq_dflt = ((real_8) actfreq_int) + actfreq_dflt;
  
  return actfreq_dflt;
}
/*--------------------------------------------------------------*/
int_4 pic_enable_d2awg (PICSTRUCT *p, int_4 mport,int_4 enable)
{  
  if (enable != 0)
    {       
       pic_wmodreg(p,mport,AWG_ENCFG_INDEX,1,AWG_NUM_PRG_BITS);     /* Enable Output     */ 
       pic_wmodreg(p,mport,AWG_SYSCFG_INDEX,1,AWG_NUM_PRG_BITS);    /* Remove I/O  Reset */     
    }
  else
    {                                                                         
       pic_wmodreg(p,mport,AWG_ENCFG_INDEX,0,AWG_NUM_PRG_BITS);     /* Disable Output    */ 
       pic_wmodreg(p,mport,AWG_SYSCFG_INDEX,0,AWG_NUM_PRG_BITS);    /* Remove I/O  Reset */
    }

  return(0);
}

/*--------------------------------------------------------------*/
int_4 pic_d2awgr2_dump(PICSTRUCT *p, int_4 mport,int_4 flag)
{  
#define AWG_ARREF_FREQ        10000000.0             /* AnyRate Reference Frequency                           */

  int_u4 mjrdval,intnum;
  real_8 realnum,fcrate;
  char clatval;
  int_4 ii;
 
  
  printf("\n\t****************D2AWGR2 Module %d****************\n",mport); 
 /* Read Fsysclk Rate                 */
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_INTERP_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  printf("\tAWG Interpolation Value \t\t = %d\n",mjrdval);
  
 /* AWG Actual System Clock Rate      */
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_FCRATEACTH_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  fcrate = (real_8) mjrdval;
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_FCRATEACTL_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  fcrate = fcrate + (((real_8) mjrdval)/1000000.0);
  printf("\tAWG System Clock Actual Value \t\t = %.6f\n",fcrate);

 /* AWG Requested System Clock Rate   */
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_FCRATEREQ_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  printf("\tAWG System Clock Requested Value \t = %.6f\n",(real_8)mjrdval);

 /* AWG System Clock Rate Difference  */
  printf("\tAWG Clock Diff (Actual-Request) \t = %.6f\n",fcrate-((real_8)mjrdval));
  
 /* AWG Actual Center Frequency       */
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_CENTFREQH_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  realnum = (real_8) mjrdval;
  pic_rmodreg (p,mport,AWG_RDMEM_ADR|AWG_CENTFREQL_ADR,&mjrdval,AWG_NUM_PRG_BITS);
  realnum = realnum + (((real_8) mjrdval)/1000000.0);
  printf("\n\tAWG Center Frequency Actual Value \t = %.6f\n",realnum);

 /* Current AnyRate CLAT Register Val */
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR),&mjrdval,AWG_NUM_PRG_BITS);        /* CLAT Reg Adr Offset 0 */
  printf("\n\tAnyRate CLAT Register Val (HEX) \t = 0x%02x\n",mjrdval & 0x000000FF);
  clatval = (char) (mjrdval & 0x000000FF);
  printf("\tAnyRate CLAT Register Val (DECIMAL) \t = %d\n",clatval);

 /* Find CLAT Shift Granularity       */ 
  realnum = AWG_ARREF_FREQ;
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+8,&mjrdval,AWG_NUM_PRG_BITS);      /* N2HS Reg              */  
  realnum = realnum * ((real_8) (((mjrdval>>5) & 0x00000007) + 4)) ;                      /* N2HS Reg 3 Bit Value  */
  
  /* printf("N2HS = %d\n",((mjrdval>>5) & 0x00000007)+4); */ 

  intnum  = ((mjrdval & 0x0000000F) << 16);                                               /* N2LS Reg  Bits 19:16  */
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+10,&mjrdval,AWG_NUM_PRG_BITS);     /* N2LS Reg  Bits  7:0   */
  intnum  = intnum | (mjrdval & 0x000000FF);
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+9,&mjrdval,AWG_NUM_PRG_BITS);      /* N2LS Reg  Bits 15:8   */
  intnum  =  (intnum | ((mjrdval & 0x000000FF) << 8)) + 1;
  
  /* printf("N2LS = %d\n",intnum); */ 
  realnum = realnum * ((real_8)intnum);                                                   /* Fin * N2HS * N2LS     */  

  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+13,&mjrdval,AWG_NUM_PRG_BITS);     /* N31 Reg  Bits   7:0   */
  intnum  = (mjrdval & 0x000000FF);
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+12,&mjrdval,AWG_NUM_PRG_BITS);     /* N31 Reg  Bits  15:8   */
  intnum  = intnum | ((mjrdval & 0x000000FF) << 8);
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+11,&mjrdval,AWG_NUM_PRG_BITS);     /* N31 Reg  Bits  18:16  */
  intnum  = (intnum | ((mjrdval & 0x00000007) << 16)) +1;

  /* printf("N31 = %d\n",intnum); */ 
  
  if (intnum != 0)                                                                        /* Reset State           */ 
    realnum = realnum / ((real_8)intnum);                                                 /* Fosc                  */  

  /* printf("Fosc = %.6f\n",realnum); */  
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+1,&mjrdval,AWG_NUM_PRG_BITS);      /* N1HS Reg  Bits  2:0   */
  intnum  = ((mjrdval >> 5) & 0x00000007) + 4; 

  /* printf("N1HS = %d\n",intnum); */ 
                                                                      
  realnum = (((real_8) intnum) / realnum) * 1000000000.0;                                 /* N1HS/Fosc=Granularity */
  printf("\tCLAT Granularity \t\t\t = %.6f nS\n",realnum); 

  realnum = ((real_8) clatval) * realnum;
  printf("\tCurrent Shift (Granularity * CLAT) \t = %.6f nS\n",realnum);                                                        

  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+7,&mjrdval,AWG_NUM_PRG_BITS);     /* NC2LS Reg  Bits   7:0   */
  intnum  = (mjrdval & 0x000000FF);
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+6,&mjrdval,AWG_NUM_PRG_BITS);     /* NC2LS Reg  Bits  15:8   */
  intnum  = intnum | ((mjrdval & 0x000000FF) << 8);
  pic_rmodreg (p,mport,(AWG_RDMEM_ADR|AWG_ANYRCFG_ADR)+5,&mjrdval,AWG_NUM_PRG_BITS);     /* NC2LS Reg  Bits  18:16  */
  intnum  = intnum | ((mjrdval & 0x0000000F) << 16);

  /* printf("NC2LS = %d\n",intnum+1); */  

  printf("\t*************************************************\n");
  return(0);
}
/*--------------------------------------------------------------*/
int_4 pic_setup_d2awg(PICSTRUCT *p, int_4 mport, int_4 dir,int_4 bits,int_4 rate,int_4 gain,int_4 flags)
{
#define AWG_FSYSCLK_MAX  1000000000  /* Maximum DAC clock speed          */
#define AWG_FSYSCLK_MIN    60000000  /* Minimum DAC clock speed          */
#define MAX_SAMP_RATE    250000000   /* Max Sample Rate                  */
#define AWG_MAX_REG             64
#define MAX_FCRATE      1000000000   /* Max Sysclk Rate 1GHz             */
#define SI5326_LO_HOLE   940000000   /* Low Limit SI5326 Hole            */
#define SI5326_HI_HOLE   970000000   /* HI  Limit SI5326 Hole            */
#define MAX_DELAY_WAIT   268000000   /* Max Delay To Start 268e6 Clocks  */
#define AWG_LOW_INVCLK    80000000   /* Low Rate To Invert Clk           */
#define AWG_HI_INVCLK    110000000   /* High Rate To Invert Clk          */
  
   Si5326_Reg_Obj Si5326_Reg; 
   int i,ii; 
   int_u4 reffreq        = AWG_REF_OSC; 
   int clksrc            = 3;        /* 0-Syn Clk From On-Board Crystal  */
                                     /* 1-Syn Clk From Ext Ref           */                    
                                     /* 3-Clk From Ext Input--Default    */
   int_u4 d2awg_cfg_bits = 0; 
   int_u4 awggencfgreg   = 0;
   
   int_u4 qducmode       = 0;        /* Default DAC Mode                 */ 
   int_u4 fcrate;                    /* FClk Rate For Input To AWG       */            
   int_u4 interp;
   int    awgdacdup      = 0;        /* DAC Duplicate Value              */
   int    awgdelayval    = 0;        /* Pause Value Before Playback      */
   int_u4 arregadr       = 0;        /* AnyRate Reg Number               */
 
   int_u4 awgreg[AWG_MAX_REG];
   real_8 freq_dflt,actfreq_dflt;
   int_u4 actfreq_int; 
   
  /* Reset QDUC Part */
   pic_wmodreg(p,mport,AWG_DACRESET_INDEX,1,AWG_NUM_PRG_BITS);  /* Assert QDUC Part Init Reset */
   udelay(1000);
   pic_wmodreg(p,mport,AWG_DACRESET_INDEX,0,AWG_NUM_PRG_BITS);  /* Remove QDUC Part Init Reset */
   udelay(1000);  

  /* Initialize AWG Register Array Values */
   /* Initialize AWG Register Array Values */
   awgreg[0] = 0x00202000;           /* Reg 0, QDUC  Enable, SDIO Input Only    */
   awgreg[1] = 0x00000000;
   awgreg[2] = 0x00000820;           /* Reg 1, Enable PDCLK at 1:1 Ratio        */
   awgreg[3] = 0x00000000;
   awgreg[4] = 0x1F3FC000;           /* Reg 2, Bypass Refclk Input Divide       */
   awgreg[5] = 0x00000000;
   awgreg[6] = 0x000000FF;           /* Reg 3, Aux DAC 1:1 20mA Output          */
   awgreg[7] = 0x00000000;

  /* CHeck For Need To Invert Clock */           
   if ( ((rate <= AWG_HI_INVCLK) && (bits > 0)) || ((rate <= (AWG_HI_INVCLK/2)) && (bits < 0)) )
     awgreg[2] = 0x00000C20;         /* Reg 1, Invert PDCLK                     */

  /* Invert Clk at User Request     */
   if (findflag("AWGCLKINV",p->config) >= 0)                                                          
      awgreg[2] = awgreg[2] ^ 0x00000400;    
   
   for (ii=8;ii<AWG_MAX_REG;ii++)
     awgreg[ii] = 0x00000000;


   if (findflag("AWGTONE",p->config) >= 0)                                                            /* Single Tone Mode           */
     {             
      /* Find User Tone Frequency */ 
       freq_dflt = 1000000.0*finddblflag("RFFREQ",p->config); 
       freq_dflt = (freq_dflt >= 0.0) ? (freq_dflt) : (1000000.0);                                    /* Output 1MHz Tone If No Flag*/

       awgreg[0]  = 0x01000000;                                                                       /* Set Single Tone Mode       */       
       awgreg[6]  = 0x000000FF;                                                                       /* Aux DAC 1:1 20mA Output    */       
      /* awgreg[28] = (int_u4) (((float)(((float)rate)/((float)(MAX_FCRATE)))) * MAX_TUNWRD_VAL +0.5); */ /* Freq Tuning Word       */
       awgreg[28] = (int_u4) (((float)(((float)freq_dflt)/((float)(MAX_FCRATE)))) * MAX_TUNWRD_VAL +0.5);
       awgreg[29] = 0x00800000;                                                                       /* Set Amplitude Scale To Max */
       fcrate     = MAX_FCRATE;
      /* printf("Rate = %d, Freq Tuning Word = %x\n",rate,awgreg[28]);  */
     }
   else 
     {
       if ((rate > MAX_SAMP_RATE) && (bits > 0))                                                     /* DAC Mode Rate Check        */
         {     
           printf("Err: DAC Mode Max Sample Rate = %d MHz\n",MAX_SAMP_RATE/1000000);
           printf("Change Clock Rate\n");
           return(-1);
         }
       else if (((rate *2) > MAX_SAMP_RATE) && (bits < 0))                                           /* QDUC Mode Rate Check       */
         {
           printf("Err: QDUC Mode Max Sample Rate = %d MHz\n",(MAX_SAMP_RATE/2)/1000000);
           printf("Change Clock Rate\n");
           return(-1);
         }
       else if ((abs(bits) != 8) && (abs(bits) != 16))
         { 
           printf("ERR: Interpolating DAC Mode Requires 8 or 16 Bit Data\n");
           printf("ERR: Quadrature Modulator Mode Requires Complex 8 or 16 Bit Data\n");
           printf("Change Data Sample Size\n");
           return(-1);
         } 

      /* Find Start Delay For DAC or QDUC Mode */
       ii = findintflag("AWGOUTDEL",p->config); 
       if (ii > 0)
         {
           if (ii > MAX_DELAY_WAIT)
             {
               printf("ERR: Max Value For AWGOUTDEL is %d\n",MAX_DELAY_WAIT);
               return(-1);
             }                         
            awgdelayval = ii;
         }
      /* Interp Dac Mode When Bits Positive */     
       if (bits > 0)                       
         {
           ii = findintflag("AWGDACDUP",p->config);
           if (ii > 1)
             {
               if (ii > 256)
                 {
                   printf("ERR: AWGDACDUP valid value range 2 to 256\n");
                   return(-1);
                 }
               else
                 {
                   rate = rate * ii;        /* New Sample Rate of AnyRate Clock */
                   awgdacdup = ii - 1;
                 }   
             }
               
           awgreg[0] = 0x02202000;           /* Set DAC Mode,Reset CCI Filter, Reset Upon Udate */ 
         }
/*&&&&&&&&&&&&&&&&&&&*/        
      /* Look For User Interpolation Value */
       ii = findintflag("AWGINTERP",p->config);  
     
      /* Only Allow AWGINTERP = 0 For DAC Mode */
       if ((bits < 0) && (ii == 0))
         ii = -1;
                    
       if (ii < 0)                    /* NOT User Supplied-Set To Highest Value    */
         {
           rate = (rate > 0) ? (rate) : (1);       
           ii = (int_u4) ((((real_8)MAX_FCRATE)/((real_8)rate)));
           ii = (ii/4)*4;
         }

       ii = (ii > 252) ? (252) : (ii);

      /* Check Interpolation Value */
       if ((ii % 4) != 0)
         { 
           printf("ERR: Interpolation Value AWGINTERP Must Be A Multiple Of 4\n");
           return(-1);   
         }
       
      /* ii Equals Zero Whan User Has Turned Off Interpolation, AWGINTERP = 0 */
      /* Setting ii = 1, Turns Off Interpolation AND Turns Off CCI Filter     */      
       fcrate = 0;
       if (ii == 0)
         {           
           ii = 1;
           fcrate = rate * 4;
         } 
       else  
         fcrate = rate * ii;        /* Set FSysclk Rate From Interpolation Val  */     
          
       
      if ((fcrate < AWG_FSYSCLK_MIN)||(fcrate > AWG_FSYSCLK_MAX))
         { 
           printf("ERR: Invalid rate 60.0MHz < (Fsysclk = %d) < 1000.0MHz \n",fcrate);
           return(-1);   
         }

      /* Check For Si5326 Synthesizer Freq 'Hole' */       
       if ((fcrate > SI5326_LO_HOLE) && (fcrate < SI5326_HI_HOLE))
         {
            if ((ii>4) && (findflag("AWGINTERP",p->config) < 0))
              {
                ii = ii- 4;           /* Reduce Default Interpolation Factor       */
                fcrate = rate * ii;
              }  
            else
              {  
      printf("ERR: System Clock Rate After Interpolation Falls In Range Of %d to %d MHz\n",SI5326_LO_HOLE/1000000,SI5326_HI_HOLE/1000000);
      printf("ERR: This Range Of Frequencies Cannot Be Synthesized\n");
      printf("ERR: Choose Different Rate Or Interpolation Factor\n");
                return(-1);
              }        
         }

    /*   printf("Interpolation Value = %d \n",ii);  */
    /*   printf("Rate = %d \n",rate);               */
    /*   printf("fcrate = %d \n",fcrate);           */

/*&&&&&&&&&&&&&&&&&&&*/ 
      /* Find User Center Frequency  */ 
       freq_dflt = 1000000.0*finddblflag("RFFREQ",p->config); 
       freq_dflt = (freq_dflt >= 0.0) ? (freq_dflt) : (0.25*rate);        

      /* Set Register Storage Values */
       awgreg[28] = (int_u4) (((real_8)(freq_dflt/((real_8)(fcrate)))) * MAX_TUNWRD_VAL +0.5);    /* Freq Tuning Word           */
       awgreg[29] = 0x00800000 | (((int_u4) ii) << 24);                                         /* Interpolation              */

      /* Store Interpolation Value   */
       pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_INTERP_ADR,(int_u4)ii,AWG_NUM_PRG_BITS); 

      /* Store Actual Center Freq    */ 
       actfreq_dflt = ((real_8) awgreg[28])/((real_8) MAX_TUNWRD_VAL) * ((real_8)fcrate);       
       actfreq_int  = (int_u4) actfreq_dflt;
       pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_CENTFREQH_ADR,actfreq_int,AWG_NUM_PRG_BITS);    /* Store Integer Cent Freq    */ 

       if ((freq_dflt - actfreq_dflt) != 0.0)
         printf("WARN: Desired Center Freq = %.6f  Actual = %.6f\n",freq_dflt,actfreq_dflt);        /* Warning For Calc Error     */
       
       actfreq_dflt = (actfreq_dflt - ((real_8) actfreq_int))  * 1000000.0;                     /* Get 6 Decimal Places       */
       actfreq_int  = (int_u4) actfreq_dflt;
       pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_CENTFREQL_ADR,actfreq_int,AWG_NUM_PRG_BITS);

      /* Store FSyclk Rate           */
       pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_FCRATEREQ_ADR,fcrate,AWG_NUM_PRG_BITS);     
 
       if (findflag("SPECINV",p->config) >= 0)
         awgreg[29] = awgreg[29] | 0x02000000;                                                  /* Spectral Invert            */   
       if (findflag("CCIBYPASS",p->config) >= 0)
         awgreg[29] = awgreg[29] | 0x01000000;                                                  /* Inverse CCI Bypass         */                           
     }

  /* Determine Clock Source */    
   if ((i=findflag("MUXCLK",p->config)) >= 0)
    {
      if (p->config[i+6]!='=') ;
      else if (p->config[i+7]=='P') 
        {          
          clksrc = 0;
          if (findflag("PREFX",p->config) >= 0)
            {
              clksrc = 1;                            /* Set Clock Source To PLL With Ext Ref Frequency */

             /* Determine User Ref Frequency For PLL Clk Use   */
         /*   ii = findintflag("PREFX",p->config);   
              if (ii <= 1)
                printf("\nWARN: Must Specify PREFX Flag As PREFX=<freq>, Assuming %d MHz\n",A2DR9_REF_OSC/1000000); 
              else
                reffreq = ii;                
         */ 
            }      
        } 
      else if ((p->config[i+7]=='A') &&  (mport == 1));
      else if ((p->config[i+7]=='B') &&  (mport == 2));
      else if (p->config[i+7]=='N');                
      else 
        {
          printf("Illegal MUXCLK=%c flag - ignored\n",p->config[i+7]);
          printf("Allowable Clock Choices: MUXCLK=");
          if (mport != 2)  
            printf("A,N,");
          if (mport != 1)
            printf("B,N,");
          printf("P,or P w/flag PREFX\n");
        }            
   }
 
 /* Disable Module */ 
  pic_enable_d2awg (p,mport,0);         
 
 /* Assert, Then Remove Configuration Reset */ 
  pic_wmodreg(p,mport,AWG_INITCFG_INDEX,0,AWG_NUM_PRG_BITS);  /* Assert    Init Reset */
  udelay(100);   
  pic_wmodreg(p,mport,AWG_INITCFG_INDEX,1,AWG_NUM_PRG_BITS);  /* De-Assert Init Reset */

 /* Set Wait Startup Counter */
  pic_wmodreg(p,mport,AWG_WAITCNT_INDEX,awgdelayval,AWG_NUM_PRG_BITS);      

 /* Set Duplicat Reg Counter */
  pic_wmodreg(p,mport,AWG_DUPCNT_INDEX,awgdacdup,AWG_NUM_PRG_BITS); 

 /* Set General Config Register                  */ 
  if ((clksrc == 1) || (clksrc == 3))                  /* Setup Clock Source, On-Board or External Ref */ 
    awggencfgreg = awggencfgreg | AWG_GENCLKSEL_BIT;
  if (abs(bits) == 16)                                 /* Setup Number Of Bits, 8 or 16 Bits           */
    awggencfgreg = awggencfgreg | AWG_16BITDATA_BIT;
  if (findflag("MTGO",p->config) >= 0)                 /* Module Triggered Go                          */
    awggencfgreg = awggencfgreg | AWG_PPSTRIG_BIT;
  pic_wmodreg(p,mport,AWG_GENCFG_INDEX,awggencfgreg,AWG_NUM_PRG_BITS);


 /* Set VCTCXO PPM/Control Voltage  */
  ii = 0;
  if (findflag("XOTENSPPM",p->config) >= 0)
    ii = findintflag("XOTENSPPM",p->config);      
  if (d2awg_set_tenthsppm (p,mport,ii) < 0)
    return(-1);   

 /* Set Startup Attentuation */
  ii = findintflag("RFATTN",p->config);
  if (ii < 0) 
    ii=0; 
  d2awg_set_attn(p,mport,ii);


 /* Setup Clock Synthesizer */
  if (clksrc == 3)  /* Bypass to External Clock */
    {
      d2awg_wmodreg_si5326(p,mport,0x00008880);  /* Put Device In Reset     */
      d2awg_wmodreg_si5326(p,mport,0x00008800);  /* Remove Reset            */
      d2awg_wmodreg_si5326(p,mport,0x0000157C);  /* ClkSel[1..0] Reg Active */
      d2awg_wmodreg_si5326(p,mport,0x00000305);  /* Set Clk1 Input Select   */
      d2awg_wmodreg_si5326(p,mport,0x00000016);  /* Set Bypass Bit          */
    }
  else              /* Normal Clock selection */
    {                
      Si5326_Reg.debuglevel = -1;
      Si5326_Reg.fref   =  10000000.0;
      Si5326_Reg.dclk1  = ((real_8) fcrate);
      Si5326_Reg.dclk2  = ((real_8) fcrate);
      Si5326_Reg.dwght1 = 1.0;
      Si5326_Reg.dwght2 = 1.0;   
      ii = si5326_regcfg_init(&Si5326_Reg,SI5326_REGCFG_MT_AWGR1);
      ii = si5326_regcfg_bwsel(&Si5326_Reg,7);       
      ii = si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_1);       
      /* ii = si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_2);  */ /* Use For Other Module Clock */
      ii = si5326_regcfg_dividers(&Si5326_Reg);
      if (ii != 0)
        {
          printf("D2AWG Warning: AWG sample rate not possible. Using %f\n",Si5326_Reg.fclk2);

          actfreq_int = (int_u4) Si5326_Reg.fclk2; 
          pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_FCRATEACTH_ADR,actfreq_int,AWG_NUM_PRG_BITS);

          actfreq_int =  (int_u4) ((Si5326_Reg.fclk2 - ((real_8) actfreq_int)) * 1000000.0);          /* Store 6 Deimal Places */ 
          pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_FCRATEACTL_ADR,actfreq_int,AWG_NUM_PRG_BITS);
        }
      else
        {
          pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_FCRATEACTL_ADR,0,AWG_NUM_PRG_BITS);
          pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_FCRATEACTH_ADR,fcrate,AWG_NUM_PRG_BITS);
        }
      si5326_regcfg_debug(&Si5326_Reg);
      
     /* Set AnyRate Reg Values */  
      for(i=0; i<Si5326_Reg.numreg; i++)
        {
          arregadr = (int_u4) ((Si5326_Reg.regval[i] >> 8) & 0x000000FF);   

         /* Register #16, Store In Adr 0                     */ 
          if (arregadr == 16)      
            pic_wmodreg(p,mport,AWG_RAMCFG_INDEX|AWG_ANYRCFG_ADR,Si5326_Reg.regval[i] & 0x000000FF,AWG_NUM_PRG_BITS);

         /* Register #25, Store In Adr 1                     */ 
          if (arregadr == 25)      
            pic_wmodreg(p,mport,(AWG_RAMCFG_INDEX|AWG_ANYRCFG_ADR)+1,Si5326_Reg.regval[i] & 0x000000FF,AWG_NUM_PRG_BITS);

         /* Register #31 to #36 Inclusive, Store In Adr 2-7  */            
          if ((arregadr >= 31) && (arregadr <= 36))
            pic_wmodreg(p,mport,(AWG_RAMCFG_INDEX|AWG_ANYRCFG_ADR)+(arregadr-29),Si5326_Reg.regval[i] & 0x000000FF,AWG_NUM_PRG_BITS);
   
         /* Register #40 to #48 Inclusive, Store In Adr 8-16 */            
          if ((arregadr >= 40) && (arregadr <= 48))
            pic_wmodreg(p,mport,(AWG_RAMCFG_INDEX|AWG_ANYRCFG_ADR)+(arregadr-32),Si5326_Reg.regval[i] & 0x000000FF,AWG_NUM_PRG_BITS);
   
          d2awg_wmodreg_si5326(p,mport,Si5326_Reg.regval[i]);
        }
    }

  udelay(30000);    /* Wait For AnyRate To Settle */

 /* Configure AD9957 */
  d2awg_initcfg_ad9957(p,mport);

  for(ii=0;ii<4;ii++)
    d2awg_wmodreg_ad9957(p,mport,ii,awgreg[(2*ii)+1],awgreg[2*ii]); 
  for(ii=14;ii<22;ii++)
    d2awg_wmodreg_ad9957(p,mport,ii,awgreg[(2*ii)+1],awgreg[2*ii]);  

  d2awg_commitcfg_ad9957(p,mport);

#if OLDCODE
  d2awg_wmodreg_ad9957(p,mport,0x0000,0x00000000,0x00000000);  /* Reg 0, DAC Mode Enable, SDIO Input Only */
  d2awg_wmodreg_ad9957(p,mport,0x0001,0x00000000,0x00000800);  /* Reg 1, Enable PDCLK at 1:1 Ratio        */
  d2awg_wmodreg_ad9957(p,mport,0x0002,0x00000000,0x1F3FC000);  /* Reg 2, Bypass Refclk Input Divide       */
  d2awg_wmodreg_ad9957(p,mport,0x0003,0x00000000,0x000000FF);  /* Reg 3, Aux DAC 1:1 20mA Output          */
  
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x08800000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x08800000,0x40000000);   
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800001,0x0851EB85);
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800001,0x19999999);
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800000,0x08F5C28F);  /* 35MHz at 16x 1GHz */
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800000,0x66666666);  /* 400MHz at 16x 1GHz */
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x12800000,0x33333333);  /* 400MHz at 16x 1GHz */
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800000,0x59999999);  /* 350MHz at 16x 1GHz w/ spectral invert */
  d2awg_wmodreg_ad9957(p,mport,0X000E,0x10800000,0x3851EB85);  /* 220MHz at 16x 1GHz */

  d2awg_wmodreg_ad9957(p,mport,0X000E,0x11800000,0x08F5C28F);  /* 35MHz at 16x 1GHz */
  d2awg_wmodreg_ad9957(p,mport,0X000F,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0010,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0011,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0012,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0013,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0014,0x00000000,0x00000000);
  d2awg_wmodreg_ad9957(p,mport,0X0015,0x00000000,0x00000000);
#endif

  udelay(40000);

  return 0;
}
