/****************************************************************************/
/*                                                                          */
/*   Si5326 methods and routines                                            */
/*                                                                          */
/****************************************************************************/

#include "part_si5326.h"

/*---------------------------------------------------------------------------*/
/* SI5326 Generic Interface routines */               
#define SI5326_REGCFG_INITMAGIC  0x1A2B3C4D /* Init magic check               */
#define SI5326_REGCFG_CLKIN_1    0x00000001 /* Set input clock ref to CLKIN_1 */
#define SI5326_REGCFG_CLKIN_2    0x00000002 /* Set input clock ref to CLKIN_2 */
#define SI5326_REGCFG_CLKIN_BYP  0x00000004 /* Set input clock ref to External*/
#define SI5326_REGCFG_MT_LB2D    0x00000010 /* Set module type to LB2D        */
#define SI5326_REGCFG_MT_A2DR13  0x00000020 /* Set module type to A2DR13      */
#define SI5326_REGCFG_MT_A2DR11  0x00000040 /* Set module type to A2DR11      */
#define SI5326_REGCFG_MT_ESXDR1  0x00000080 /* Set module type to ESXDR1      */
#define SI5326_REGCFG_MT_AWGR1   0x00000100 /* Set module type to AWGR1       */
#define SI5326_REGCFG_MT_TGSPHY  0x00000200 /* Set module type to TGSDDS PHY  */
#define SI5326_REGCFG_MT_TGSDAT  0x00000400 /* Set module type to TGSDDS DAT  */
#define SI5326_REGCFG_MT_A2DM14  0x00000800 /* Set module type to A2DM14      */
#define SI5326_REGCFG_MT_A2DM1X  0x00000800 /* Set module type to A2DM1X      */
#define SI5326_REGCFG_MT_SNTXDR6 0x00001000 /* Set module type to SNTXDR6      */
/*---------------------------------------------------------------------------*/
/* SI5326 dump object/register configuration contents for debug */             
int32_t si5326_regcfg_debug(Si5326_Reg_Obj *P)
{
    int32_t rtn = -1;
    int32_t i;

    /* Display and check for properly initialized object */
    if (P->debuglevel > 0) {
      printf(" \n");
      printf("----<<<<< Si5326 Debug >>>>>----\n");
      printf("  initmagic: 0x%8.8X  0x%8.8X\n",
        SI5326_REGCFG_INITMAGIC,P->initmagic);

      if (P->initmagic == SI5326_REGCFG_INITMAGIC) {
        rtn = 0;
        printf("  debuglevel: %d \n",P->debuglevel);
        printf("  mtype:      %d \n",P->mtype);       
        printf("  validflag:  %d \n",P->validflag); 
        printf("  n1_hs:   %d \n",P->n1_hs);
        printf("  nc1_ls:  %d \n",P->nc1_ls);
        printf("  nc1_ls:  %d \n",P->nc2_ls);
        printf("  n2_hs:   %d \n",P->n2_hs);
        printf("  n2_ls:   %d \n",P->n2_ls);
        printf("  n31:     %d \n",P->n31);   
        printf("  n32:     %d \n",P->n32);  
        printf("  fref:    %f \n",P->fref);                   
        printf("  f3:      %f \n",P->f3);                   
        printf("  fvco:    %f \n",P->fvco); 
        printf("  dclk1:   %f \n",P->dclk1);
        printf("  fclk1:   %f \n",P->fclk1);
        printf("  dclk2:   %f \n",P->dclk2);
        printf("  fclk2:   %f \n",P->fclk2);
        printf("  dwght1:  %f \n",P->dwght1);
        printf("  dwght2:  %f \n",P->dwght2);
        printf(" \n");

        if (P->debuglevel > 1) {
          for (i=0; i<(P->numreg); i++) {
            printf("  Si5326 Register: %3i  %3d  0x%8.8X\n",i,
              ((P->regval[i] & 0x0000FF00) >> 8), P->regval[i]);
          }
          printf(" \n");
        }
      }
    }
    return (rtn);
}


/*---------------------------------------------------------------------------*/
/* SI5326 divider solution function */                     
int32_t si5326_regcfg_dividers(Si5326_Reg_Obj *P)
{
    const float64_t FOSC_MAX   = 5670000000.0;   /* 5.67GHz */
    const float64_t FOSC_MIN   = 4850000000.0;   /* 4.85GHz */
    const float64_t F3_MAX     = 2000000.0;      /* 2000kHz */
    const float64_t F3_MIN     = 100000.0;       /*  100kHz */
    const int32_t  N3x_MIN    = 1;
    const int32_t  N3x_MAX    = 524288;         /* 2^19    */
    const int32_t  N1_HS_MIN  = 4;
    const int32_t  N1_HS_MAX  = 11;
    const int32_t  NCx_LS_MIN = 1;
    const int32_t  NCx_LS_MAX = 1048576;        /* 2^20, even only except 1 */
    const int32_t  N2_HS_MIN  = 4;
    const int32_t  N2_HS_MAX  = 11;
    const int32_t  N2_LS_MIN  = 2;
    const int32_t  N2_LS_MAX  = 1048576;        /* 2^20, even only */

    /* Define optimal solutions to special / very common cases here */
    struct si5326_regcfg_case { 
      float64_t dclk1;
      float64_t dclk2;
      int32_t  n1_hs;
      int32_t  nc1_ls;
      int32_t  nc2_ls;
      int32_t  n2_hs;
      int32_t  n2_ls;
      int32_t  n31;
      int32_t  n32;
    };


    struct si5326_regcfg_case REGCFG_CASE[] = {
       { 30000000.0, 100000000.0,  9, 20,  6, 10,    270,    5,    5},
       { 30000000.0, 125000000.0,  4, 42, 10, 10,    250,    5,    5},
       { 30000000.0, 110592000.0,  8, 22,  6, 11,    240,    5,    5},
       {110592000.0, 110592000.0,  5, 10, 10,  9,   1536,   25,   25},
       {117964800.0, 117964800.0, 11,  4,  4, 11, 147456, 3125, 3125},
       {157286400.0, 157286400.0,  9,  4,  4,  9, 196608, 3125, 3125},
       {196400000.0, 196400000.0,  7,  4,  4,  7,   1964,   25,   25},
       {196608000.0, 196608000.0,  7,  4,  4,  8,  43008,  625,  625},
       {215040000.0, 215040000.0,  6,  4,  4,  9,   7168,  125,  125},
       {       -1.0,        -1.0, -1, -1, -1, -1,     -1,   -1,   -1} };

    int32_t  i;
    int32_t  n3x, n2_hs, n2_ls, n1_hs, nc1_ls, nc2_ls;
    uint32_t ureg,u1_hs,uc1_ls,uc2_ls,u2_hs,u2_ls,u_31,u_32;
    float64_t fvco_n2,f3_test;
    float64_t fvco_nc1_err, fvco_nc2_err, fvco_err_wavg, fvco_err_wavg_min;
    float64_t fvco_nc1_wght, fvco_nc2_wght;
    float64_t Fref, Fclk1, Fclk2, Fwght1, Fwght2;

    /* Check for valid object */
    if (P->initmagic != SI5326_REGCFG_INITMAGIC) return (-1);

    /* Initialize return structure */
    P->validflag = -1;
    P->n1_hs  = N1_HS_MAX;
    P->nc1_ls = NCx_LS_MAX;
    P->nc2_ls = NCx_LS_MAX;
    P->n2_hs  = N2_HS_MAX;
    P->n2_ls  = N2_LS_MAX;
    P->n31    = N3x_MAX;
    P->n32    = N3x_MAX;

    /* Local copies of object elements */
    Fref   = P->fref;
    Fclk1  = P->dclk1;
    Fclk2  = P->dclk2;
    Fwght1 = P->dwght1;
    Fwght2 = P->dwght2;

    /* Check for degenerate cases */
    if (Fref  < 2000.0) return (-1);
    if (Fclk1 < 2000.0) return (-1);
    if (Fclk2 < 2000.0) return (-1);

    /* Initialize error metrics and weightings */
    fvco_nc1_wght    = (Fwght1 < 0.0) ? (1.0) : (Fwght1);
    fvco_nc2_wght    = (Fwght2 < 0.0) ? (1.0) : (Fwght2);
    fvco_err_wavg_min = FOSC_MAX;

    /* Most common/special cases go here for fast access */
    for (i=0; REGCFG_CASE[i].n1_hs > 0; i++) {
      fvco_nc1_err = fabs(REGCFG_CASE[i].dclk1-Fclk1);
      fvco_nc2_err = fabs(REGCFG_CASE[i].dclk2-Fclk2);
      if (fabs(fvco_nc1_err+fvco_nc2_err) < 1.0e-6) {
        P->n1_hs  = REGCFG_CASE[i].n1_hs;
        P->nc1_ls = REGCFG_CASE[i].nc1_ls;
        P->nc2_ls = REGCFG_CASE[i].nc2_ls;          
        P->n2_hs  = REGCFG_CASE[i].n2_hs;
        P->n2_ls  = REGCFG_CASE[i].n2_ls;
        P->n31    = REGCFG_CASE[i].n31;
        P->n32    = REGCFG_CASE[i].n32;
        P->validflag = 0;
        goto SI5326_SETUP_DIVIDERS;  
      }
    }
    i=-1;
    
    /* Find solution that minimizes the weighted VCO frequency error */
    for (n3x=N3x_MIN; n3x<N3x_MAX; n3x++) {
      f3_test = Fref/((float64_t) n3x);
      if ((f3_test >= F3_MIN) && (f3_test <= F3_MAX)) {
        for (n2_hs=N2_HS_MAX; n2_hs>=N2_HS_MIN; n2_hs--) {
          for (n2_ls=N2_LS_MIN; n2_ls<=N2_LS_MAX; n2_ls+=2) {
            fvco_n2 = f3_test*((float64_t) n2_hs)*((float64_t) n2_ls);
            if ((fvco_n2 >= FOSC_MIN) && (fvco_n2 <= FOSC_MAX)) {
              for (n1_hs=N1_HS_MAX; n1_hs>=N1_HS_MIN; n1_hs--) {
                nc1_ls = (int32_t) (0.5 + (fvco_n2/(Fclk1*((float64_t) n1_hs))));
                nc2_ls = (int32_t) (0.5 + (fvco_n2/(Fclk2*((float64_t) n1_hs))));
                if ((nc1_ls >= NCx_LS_MIN) && (nc1_ls <= NCx_LS_MAX) &&
                    (nc2_ls >= NCx_LS_MIN) && (nc2_ls <= NCx_LS_MAX)) {
                  if (nc1_ls != 1) nc1_ls = 2*((int32_t) (nc1_ls/2));
                  if (nc2_ls != 1) nc2_ls = 2*((int32_t) (nc2_ls/2));
                  fvco_nc1_err = fabs(fvco_n2 - (Fclk1*((float64_t) n1_hs)*((float64_t) nc1_ls)));
                  fvco_nc2_err = fabs(fvco_n2 - (Fclk2*((float64_t) n1_hs)*((float64_t) nc2_ls)));
                  fvco_err_wavg = fvco_nc1_wght*fvco_nc1_err + fvco_nc2_wght*fvco_nc2_err;
                  if (fvco_err_wavg < fvco_err_wavg_min) {
                    fvco_err_wavg_min = fvco_err_wavg;
                    P->validflag = -1;
                    if (fvco_nc1_err <= 0.0) P->validflag = 1;
                    if (fvco_nc2_err <= 0.0) P->validflag = 1;
                    if (fvco_err_wavg_min <= 0.0) P->validflag = 0;
                    P->n1_hs  = n1_hs;
                    P->nc1_ls = (nc1_ls <= 0) ? (1) : (nc1_ls);
                    P->nc2_ls = (nc2_ls <= 0) ? (1) : (nc2_ls);
                    P->n2_hs  = n2_hs;
                    P->n2_ls  = n2_ls;
                    P->n31    = n3x;
                    P->n32    = n3x;
                    if (P->validflag == 0) goto SI5326_SETUP_DIVIDERS;  
                  } 
                } 
              } 
            } 
          } 
        } 
      } 
    }

    /* Set divider register configuration */
    SI5326_SETUP_DIVIDERS:  

    /* Calculate exact output frequencies */
    P->fref  = Fref; 
    P->f3    = (P->fref)/((float64_t) P->n31);
    P->fvco  = (P->f3)*((float64_t) P->n2_ls)*((float64_t) P->n2_hs);
    P->fclk1 = (P->fvco)/(((float64_t) P->n1_hs)*((float64_t) P->nc1_ls));
    P->fclk2 = (P->fvco)/(((float64_t) P->n1_hs)*((float64_t) P->nc2_ls));

#if DEBUG
    printf("Case i=%d %f %f  %d %d %d  %d %d  %d %d\n",i,P->fclk1,P->fclk2,P->n1_hs,P->nc1_ls,P->nc2_ls,P->n2_hs,P->n2_ls,P->n31,P->n32);
#endif

    /* Set divider registers using specified masks/bit offsets */
    u1_hs  = (uint32_t) (P->n1_hs - 4);
    uc1_ls = (uint32_t) (P->nc1_ls - 1);
    uc2_ls = (uint32_t) (P->nc2_ls - 1);
    u2_hs  = (uint32_t) (P->n2_hs - 4);
    u2_ls  = (uint32_t) (P->n2_ls - 1);
    u_31   = (uint32_t) (P->n31 - 1);
    u_32   = (uint32_t) (P->n32 - 1);

    for (i=0; i<(P->numreg); i++) {
      ureg = (P->regval[i])&(0x0000FF00);
      if (ureg == 0x00001900) P->regval[i] = ureg|((u1_hs << 5)&(0x000000E0));
      if (ureg == 0x00001F00) P->regval[i] = ureg|((uc1_ls >> 16)&(0x0000000F));
      if (ureg == 0x00002000) P->regval[i] = ureg|((uc1_ls >> 8)&(0x000000FF));
      if (ureg == 0x00002100) P->regval[i] = ureg|((uc1_ls)&(0x000000FF));
      if (ureg == 0x00002200) P->regval[i] = ureg|((uc2_ls >> 16)&(0x0000000F));
      if (ureg == 0x00002300) P->regval[i] = ureg|((uc2_ls >> 8)&(0x000000FF));
      if (ureg == 0x00002400) P->regval[i] = ureg|((uc2_ls)&(0x000000FF));
      if (ureg == 0x00002800) P->regval[i] = ureg|((u2_hs << 5)&(0x000000E0))|
                                                  ((u2_ls >> 16)&(0x0000000F));
      if (ureg == 0x00002900) P->regval[i] = ureg|((u2_ls >> 8)&(0x000000FF));
      if (ureg == 0x00002A00) P->regval[i] = ureg|((u2_ls)&(0x000000FF));
      if (ureg == 0x00002B00) P->regval[i] = ureg|((u_31 >> 16)&(0x00000007));
      if (ureg == 0x00002C00) P->regval[i] = ureg|((u_31 >> 8)&(0x000000FF));
      if (ureg == 0x00002D00) P->regval[i] = ureg|((u_31)&(0x000000FF));
      if (ureg == 0x00002E00) P->regval[i] = ureg|((u_32 >> 16)&(0x00000007));
      if (ureg == 0x00002F00) P->regval[i] = ureg|((u_32 >> 8)&(0x000000FF));
      if (ureg == 0x00003000) P->regval[i] = ureg|((u_32)&(0x000000FF));
    }

    return (P->validflag);
}


/*---------------------------------------------------------------------------*/
/* SI5326 register configuration function */             
int32_t si5326_regcfg_bwsel(Si5326_Reg_Obj *P, int32_t bwsel)
{
    int32_t i;
    uint32_t ureg,ubwsel;

    /* Check for valid object / ranges */
    if (P->initmagic != SI5326_REGCFG_INITMAGIC) return (-2);
    if ((bwsel < 0)||(bwsel > 15)) return (-1);

    /* BWSEL must be between 0 (MaxBW) and 15 (MinBW)          */
    /* Generally, must keep (4 < BWSEL < 10) or PLL won't lock */
    ubwsel = (uint32_t) bwsel;
    for (i=0; i<(P->numreg); i++) {
      ureg = (P->regval[i])&(0x0000FF00);
      if (ureg == 0x00000200) 
        P->regval[i] = (P->regval[i] & 0xFFFFFF0F)|((ubwsel << 4)&(0x000000F0));
    }

    return (0);
}

/*---------------------------------------------------------------------------*/
/* SI5326 register configuration function */             
int32_t si5326_regcfg_clkin(Si5326_Reg_Obj *P, int32_t clksel)
{
    int32_t i;
    uint32_t ureg;

    /* Check for valid object / ranges */
    if (P->initmagic != SI5326_REGCFG_INITMAGIC) return (-2);

    if (clksel == SI5326_REGCFG_CLKIN_1) {
      for (i=0; i<(P->numreg); i++) {
        ureg = (P->regval[i])&(0x0000FF00);
        if (ureg == 0x00000300) 
          P->regval[i] = (P->regval[i] & 0xFFFFFF3F)|(0x00000000);
      }
    }
    else if (clksel == SI5326_REGCFG_CLKIN_2) {
      for (i=0; i<(P->numreg); i++) {
        ureg = (P->regval[i])&(0x0000FF00);
        if (ureg == 0x00000300) 
          P->regval[i] = (P->regval[i] & 0xFFFFFF3F)|(0x00000040);
      }
    }
    else if (clksel == SI5326_REGCFG_CLKIN_BYP) {
      for (i=0; i<(P->numreg); i++) {
        ureg = (P->regval[i])&(0x0000FF00);
        if (ureg == 0x00000000)
          P->regval[i] = (P->regval[i] & 0xFFFFFFFD)|(0x00000002);
      }
    }
    else {
      return (-1);
    }
    return (0);
}


/*---------------------------------------------------------------------------*/
/* SI5326 register configuration init function */             
int32_t si5326_regcfg_init(Si5326_Reg_Obj *P, int32_t module_type)
{
    /* LB2D Configuration */
    int32_t si5326_lb2d_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_lb2d_regdefault[] =
      { 0x00000609,0x00008880,0x00008800,0x00000A0C,0x00000609,0x0000157C,
        0x00000315, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A00,0x0000063E };

    /* A2DR13 Configuration */
    int32_t si5326_a2dr13_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_a2dr13_regdefault[] =
      { 0x00000609,0x00008880,0x00008800,0x00000A0C,0x00000609,0x0000157C,
        0x00000315, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A04,0x00000628 };

   /* A2DR11 Configuration */
    int32_t si5326_a2dr11_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_a2dr11_regdefault[] =
      { 0x00000609,0x00008880,0x00008800,0x00000A0C,0x00000609,0x0000157C,
        0x00000315, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A04,0x00000631 };

  /* ESXDR1 Configuration */
    int32_t si5326_esxdr1_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_esxdr1_regdefault[] =
      { 0x0000063F,0x00008880,0x00008800,0x00000A00,0x0000063F,0x0000157C,
        0x00000355, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x0000072A,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A00,0x0000063F };

  /* D2AWGR1 Configuration */
    int32_t si5326_awgr1_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_awgr1_regdefault[] =
      { 0x0000063A,0x00008880,0x00008800,0x00000A0C,0x0000063A,0x0000157C,
        0x00000315, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A00,0x0000063A };

   /* TGSPHY Configuration */
    int32_t si5326_tgsphy_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_tgsphy_regdefault[] =
      { 0x00000636,0x00008880,0x00008800,0x00000A0C,0x00000636,0x0000157C,
        0x00000315,
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A00,0x00000636 };

   /* TGSDAT Configuration */
    int32_t si5326_tgsdat_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_tgsdat_regdefault[] =
      { 0x00000639,0x00008880,0x00008800,0x00000A0C,0x00000639,0x0000157C,
        0x00000315,
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A04,0x00000639 };

    /* A2DM14 Configuration */
    int32_t si5326_a2dm14_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_a2dm14_regdefault[] =
      { 0x00000609,0x00008880,0x00008800,0x00000A0C,0x00000609,0x0000157C,
        0x00000315, 
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A08,0x0000063F };

    /* SNTXDR6 Configuration */
    int32_t si5326_sntxdr6_nregdefault = 50; /* Must be < SI5326_REGCFG_MAXREGS! */
    uint32_t si5326_sntxdr6_regdefault[] =
      { 0x0000063F,0x00008880,0x00008800,0x00000A0C,0x0000063F,0x0000157C,
        0x00000355,
        0x00000015,0x000001E4,0x00000282,0x00000412,0x000005ED,0x00000729,
        0x00000800,0x000009C0,0x00000B40,0x00001000,0x00001180,0x00001200,
        0x0000132C,0x0000143E,0x000016D7,0x0000171F,0x0000183F,0x00001900,
        0x00001F00,0x00002000,0x00002100,0x00002200,0x00002300,0x00002400,
        0x00002800,0x00002900,0x00002A00,0x00002B00,0x00002C00,0x00002D00,
        0x00002E00,0x00002F00,0x00003000,0x00003700,0x0000831F,0x00008402,
        0x00008A0C,0x00008BCF,0x00008E00,0x00008F00,
        0x00008840,0x00000A00,0x0000063F }; 

    int32_t i;
    int32_t iregcnt = 0;
    uint32_t *regdefault;                       
    int32_t  *nregdefault;

    /* Initialize Register contents */
    P->numreg = 0;       
    for (i=0; i<SI5326_REGCFG_MAXREGS; i++) P->regval[i] = 0x00000000;

    /* Select appropriate register initializers */
    switch (module_type) {
      
       case (SI5326_REGCFG_MT_A2DM14):
        P->mtype = SI5326_REGCFG_MT_A2DM14;
        regdefault  = &si5326_a2dm14_regdefault[0];
        nregdefault = &si5326_a2dm14_nregdefault;
        break;
      
       case (SI5326_REGCFG_MT_ESXDR1):
        P->mtype = SI5326_REGCFG_MT_ESXDR1;
        regdefault  = &si5326_esxdr1_regdefault[0];
        nregdefault = &si5326_esxdr1_nregdefault;
        break;

      case (SI5326_REGCFG_MT_A2DR13):
        P->mtype = SI5326_REGCFG_MT_A2DR13;
        regdefault  = &si5326_a2dr13_regdefault[0];
        nregdefault = &si5326_a2dr13_nregdefault;
        break;

      case (SI5326_REGCFG_MT_A2DR11):
        P->mtype = SI5326_REGCFG_MT_A2DR11;
        regdefault  = &si5326_a2dr11_regdefault[0];
        nregdefault = &si5326_a2dr11_nregdefault;
        break;

      case (SI5326_REGCFG_MT_LB2D):  
        P->mtype = SI5326_REGCFG_MT_LB2D;  
        regdefault  = &si5326_lb2d_regdefault[0];
        nregdefault = &si5326_lb2d_nregdefault;
        break;
  
      case (SI5326_REGCFG_MT_AWGR1):  
        P->mtype = SI5326_REGCFG_MT_AWGR1;  
        regdefault  = &si5326_awgr1_regdefault[0];
        nregdefault = &si5326_awgr1_nregdefault;
        break;
 
      case (SI5326_REGCFG_MT_TGSPHY):
        P->mtype = SI5326_REGCFG_MT_TGSPHY;
        regdefault  = &si5326_tgsphy_regdefault[0];
        nregdefault = &si5326_tgsphy_nregdefault;
        break;

      case (SI5326_REGCFG_MT_TGSDAT):
        P->mtype = SI5326_REGCFG_MT_TGSDAT;
        regdefault  = &si5326_tgsdat_regdefault[0];
        nregdefault = &si5326_tgsdat_nregdefault;
        break;

      case (SI5326_REGCFG_MT_SNTXDR6):
        P->mtype = SI5326_REGCFG_MT_SNTXDR6;
        regdefault  = &si5326_sntxdr6_regdefault[0];
        nregdefault = &si5326_sntxdr6_nregdefault;
        break;

      default:
        perror("Unknown module type in func: si5326_init_registers");
        return (-1);
    }

    /* Set configuration register values */
    if (nregdefault[0] <= SI5326_REGCFG_MAXREGS) {
      iregcnt = 0;
      for (i=0; i<nregdefault[0]; i++) {
        P->regval[iregcnt] = regdefault[i];
        iregcnt ++;
      }
    }
    P->numreg = iregcnt;
    P->fclk1  = 0.0;
    P->fclk2  = 0.0;
    P->fvco   = 0.0;
    P->initmagic = SI5326_REGCFG_INITMAGIC;

    return (0);
}

