/****************************************************************************/
/*                                                                          */
/*   part_common - libraries common to parts and modules                    */
/*                                                                          */
/****************************************************************************/

#include "part_common.h"

/*----------------------------------------------------------------------------*/
/* Setbit/Getbit routines for register content manipulation                  */
/*  - Setbit writes a "uval" value into the "umask" position of the register */
/*  - Getbit reads a "uval" value from the "umask" position in the register  */
/*  - "umask" position does not need to be contiguous (helpful for SiLabs)   */
/*  - Warning: these are little endian routines, but at least it isolates    */
/*             the endian issues in one place for future updates             */

uint8_t u8_setbits(uint8_t uinit, uint8_t umask, uint8_t uval)
{
  uint8_t i;
  uint8_t im = 0x01;
  uint8_t om = 0x01;
  uint8_t rtnval = uinit;
  for (i=0; i<(8*sizeof(uint8_t)); i++) {
    if (umask & om) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      im = (im << 1);
    } 
    om = (om << 1);
  }
  return (rtnval);
}

uint8_t u8_getbits(uint8_t umask, uint8_t uval)
{
  uint8_t i;
  uint8_t im = 0x01;
  uint8_t om = 0x01;
  uint8_t rtnval = 0x00;
  for (i=0; i<(8*sizeof(uint8_t)); i++) {
    if (umask & im) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      om = (om << 1);
    } 
    im = (im << 1);
  }
  return (rtnval);
}

uint16_t u16_setbits(uint16_t uinit, uint16_t umask, uint16_t uval)
{
  uint16_t i;
  uint16_t im = 0x0001;
  uint16_t om = 0x0001;
  uint16_t rtnval = uinit; 
  for (i=0; i<(8*sizeof(uint16_t)); i++) {
    if (umask & om) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      im = (im << 1);
    } 
    om = (om << 1);
  }
  return (rtnval);
}

uint16_t u16_getbits(uint16_t umask, uint16_t uval)
{
  uint16_t i;
  uint16_t im = 0x0001;
  uint16_t om = 0x0001;
  uint16_t rtnval = 0x0000;
  for (i=0; i<(8*sizeof(uint16_t)); i++) {
    if (umask & im) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      om = (om << 1);
    } 
    im = (im << 1);
  }
  return (rtnval);
}

uint32_t u32_setbits(uint32_t uinit, uint32_t umask, uint32_t uval)
{
  uint32_t i;
  uint32_t im = 0x00000001;
  uint32_t om = 0x00000001;
  uint32_t rtnval = uinit;     
  for (i=0; i<(8*sizeof(uint32_t)); i++) {
    if (umask & om) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      im = (im << 1);
    } 
    om = (om << 1);
  }
  return (rtnval);
}

uint32_t u32_getbits(uint32_t umask, uint32_t uval)
{
  uint32_t i;
  uint32_t im = 0x00000001;
  uint32_t om = 0x00000001;
  uint32_t rtnval = 0x00000000;
  for (i=0; i<(8*sizeof(uint32_t)); i++) {
    if (umask & im) {
      rtnval = (uval & im) ? (rtnval | om) : (rtnval & (~om));
      om = (om << 1);
    } 
    im = (im << 1);
  }
  return (rtnval);
}

UDB *alloc_udb(int32_t n)
{
  UDB *b = (UDB *)NULL;
  if (n > 0) {
    b = (UDB *)malloc(n*sizeof(UDB));
    if (b != NULL) memset(b,0,n*sizeof(UDB));
  }
  return (b);
}

void free_udb(UDB *bp)
{
  if (bp != NULL) free(bp);
}

/* Converts a 64-bit float to dual whole/fractional 32-bit integer pair */
int32_t udb_f64_to_i32wf(UDB *ubuf)
{
  float64_t fv  = ubuf->f64[0];
  float64_t fnf = 0.0;
  int32_t nw = 0;
  *ubuf = udb_initializer;
  if ((fv > -2147483647.0) && (fv < 2147483647.0)) {
    nw  = (int32_t) (fv);
    fnf = 2147483648.0*(fv - ((float64_t)nw)); 
    fnf = (fv >= 0.0) ? (fnf+0.5) : (fnf-0.5);
    if (fnf > 2147483647.0) {
      nw = (nw < 2147483647) ? (nw+1) : (2147483647);
      fnf -= 2147483647.0;  
    }
    if (fnf < -2147483647.0) {
      nw = (nw > -2147483647) ? (nw-1) : (-2147483647);
      fnf += 2147483647.0;  
    }
    ubuf->i32[0] = nw;
    ubuf->i32[1] = (int32_t) fnf;
  }
  else {
    return (-1);
  }
  return (0);
}

/* Rounds 64-bit float to nearest 32-bit integer */
int32_t udb_f64_to_i32rnd(UDB *ubuf)
{
  float64_t fv = ubuf->f64[0];
  int32_t nw = 0;
  int32_t nf = 0;
  *ubuf = udb_initializer;
  if ((fv > -2147483647.0) && (fv < 2147483646.0)) {
    if (fv > 0.0) {
      nw = (int32_t) (fv+0.5);
      nf = 0;
    }
    else {
      nw = (int32_t) (fv-0.5);
      nf = 0;
    }
    ubuf->i32[0] = nw;
    ubuf->i32[1] = nf;
  }
  else {
    return (-1);
  }
  return (0);
}

/* Converts a dual whole/fractional 32-bit integer pair to 64-bit float */
int32_t udb_i32wf_to_f64(UDB *ubuf)
{
  int32_t nw = ubuf->i32[0];
  int32_t nf = ubuf->i32[1];
  *ubuf = udb_initializer;
  ubuf->f64[0] = (float64_t) ((float64_t)nw)+(((float64_t)nf)/2147483648.0);
  return (0);
}


