/****************************************************************************/
/*                                                                          */
/*  part_adf4350.c                                                          */
/*                                                                          */
/****************************************************************************/

#include "part_adf4350.h"

/* Sane defaults for ADF4350 registers                             */
/*                                                                 */
int32_t pt_adf4350_rgc_default(PT_ADF4350_REG *R)
{
  static const PT_ADF4350_REG adf4350_reg_defaults = {
/*
    {0x00000000,0x00000000,0x00000000,
     0x00000000,0x00000000,0x00000000} };

    {0x02300000,0x00000BB8,0x01A00D28,
     0x00000800,0x001540BF,0x000B0000} };
*/
    {0x04B00000,0x00004001,0x1B02B142,
     0x00001003,0x00C205FC,0x00580005} };

  int32_t i;
  for (i=0; i<ADF4350_NUM_RG; i++){
    R->rg[i] = adf4350_reg_defaults.rg[i];
  }
  return (0);
}

float64_t adf4350_pt_findsoln(PT_ADF4350_REG *S, float64_t fdes, float64_t fref)
{
  float64_t fpd, fsynth, fvco;
  float64_t pdratio, dfrac;
  int32_t  nratio;
  uint32_t U,U_idx,M,N,F;
  uint32_t bscd;
  uint32_t rcount_tmp = 0;

  fdes = fdes * 1000000.0; /* Convert to Hz */
  fref = fref * 1000000.0; /* Convert to Hz */

  /* First, test reference to make certain it is in valid range */
  if ((fref < 10000000.0)||(fref > 250000000.0)) return (-2);

  /* Always disable the reference doubler to reduce noise and jitter */
  S->rg[0x02] = u32_setbits(S->rg[0x02],ADF4350_RGM_02_REFDBL,0);

  /* Always enable reference half to force 50% duty cycle */
  S->rg[0x02] = u32_setbits(S->rg[0x02],ADF4350_RGM_02_RDIV2,1);
  fpd = 0.5*(fref);

  /* Now select the highest reference frequency below 32MHz */
  rcount_tmp = ((fpd/32000000.0)+1.0);
  rcount_tmp = (rcount_tmp > 0)? rcount_tmp : (1);
  S->rg[0x02] = u32_setbits(S->rg[0x02],ADF4350_RGM_02_RCOUNT,rcount_tmp);
  fpd = fpd/rcount_tmp;

  /* Check for valid synthesizer rage */
  if((fdes < ADF4350_VCO_FSYNTH_MIN) || (fdes > ADF4350_VCO_FMAX)) return(-1);

  /* Now compute the dividers */  
  fsynth = fdes;

  /* Compute RF divider */ 
  { U = 1; U_idx = 0x00; }
  if (fsynth <= 2200000000.0) { U = 2;  U_idx = 0x01; }
  if (fsynth <= 1100000000.0) { U = 4;  U_idx = 0x02; }
  if (fsynth <=  550000000.0) { U = 8;  U_idx = 0x03; }
  if (fsynth <=  275000000.0) { U = 16; U_idx = 0x04; }

  /* Compute nominal VCO frequency */
  fvco = (fsynth)*(U);

  /* Fix modulus at 3000 */
  M = 3000;
  S->rg[0x01] = u32_setbits(S->rg[0x01],ADF4350_RGM_01_MOD,M);

  /* Solve for nearest valid VCO frequency */
  pdratio = (fvco)/(fpd);
  nratio  = (int32_t) (pdratio);
  nratio  = (nratio > 23) ? (nratio) : (23);
  nratio  = (nratio < 65535) ? (nratio) : (65535);
  N    = nratio;
  dfrac   = pdratio - ((float64_t) nratio);
  dfrac   = (dfrac > 0.0) ? (dfrac) : (0.0);
  dfrac   = (dfrac < 1.0) ? (dfrac) : (0.0);
  F    = (uint32_t) (0.5 + dfrac*((float64_t) M));
  if (F == M) {
    F = 0;
    N = N + 1;
  }

  /* Compute the exact synthesizer frequency */
  pdratio = ((float64_t)N) + (((float64_t)F)/((float64_t)M));
  fsynth = (1.0/((float64_t)U))*(fpd*pdratio);

  /* Band select clock divider (62.5kHz is target) */
  nratio = (int32_t) (0.5+(2.0*(fpd)/62500.0));
  nratio = (nratio >=   1) ? (nratio) : (1);
  bscd = (nratio <= 255) ? (nratio) : (255);

#ifdef BOGUS
  /* Set valid / invalid solution flag */
  dfrac = fdes - fsynth;
  dfrac = (dfrac > 0.0) ? (dfrac) : (-1.0*dfrac);
  if (dfrac < 0.001) 
    P->validflag = 0;
  else if (dfrac < 500.0)
    P->validflag = 1;
  else
    P->validflag = -1;
#endif

  /* Configure remaining registers */
  S->rg[0x04] = u32_setbits(S->rg[0x04],ADF4350_RGM_04_BSCD,bscd);
  S->rg[0x04] = u32_setbits(S->rg[0x04],ADF4350_RGM_04_DIVSEL,U_idx);
  S->rg[0x03] = u32_setbits(S->rg[0x03],ADF4350_RGM_03_CLKDIV,0x0800);
  S->rg[0x03] = u32_setbits(S->rg[0x03],ADF4350_RGM_03_DIVMODE,0x00);
  S->rg[0x03] = u32_setbits(S->rg[0x03],ADF4350_RGM_03_CSR,0x00);
  S->rg[0x02] = u32_setbits(S->rg[0x02],ADF4350_RGM_02_LDF,0x00);
  if(F==0) S->rg[0x02] = u32_setbits(S->rg[0x02],ADF4350_RGM_02_LDF,0x01);
  S->rg[0x01] = u32_setbits(S->rg[0x01],ADF4350_RGM_01_PHASE,0x0000);
  S->rg[0x01] = u32_setbits(S->rg[0x01],ADF4350_RGM_01_PRESCALE,0x00);
  S->rg[0x00] = u32_setbits(S->rg[0x00],ADF4350_RGM_00_INT,N);
  S->rg[0x00] = u32_setbits(S->rg[0x00],ADF4350_RGM_00_FRAC,F);

  return (fsynth);
}

int32_t pt_adf4350_setrate(PT_ADF4350_REG *S, UDB *rate, UDB *refclk)
{
  float64_t fdes,fref,fact;
  int32_t rtn;

  fdes = rate->f64[0];
  fref = refclk->f64[0];
  fact = adf4350_pt_findsoln(S, fdes, fref)/1000000.0;
  if(fdes != fact) printf("WARNING: Desired Sample Rate = %f, Actual Sample Rate = %f\n",fdes,fact);

  S->rg[ADF4350_RG_04] = u32_setbits(S->rg[ADF4350_RG_04],ADF4350_RGM_04_OUTPOW,   0x03);
  S->rg[ADF4350_RG_04] = u32_setbits(S->rg[ADF4350_RG_04],ADF4350_RGM_04_RFEN,     0x01);
  S->rg[ADF4350_RG_04] = u32_setbits(S->rg[ADF4350_RG_04],ADF4350_RGM_04_AUXPOW,   0x03);
  S->rg[ADF4350_RG_04] = u32_setbits(S->rg[ADF4350_RG_04],ADF4350_RGM_04_AUXEN,    0x01);
  S->rg[ADF4350_RG_04] = u32_setbits(S->rg[ADF4350_RG_04],ADF4350_RGM_04_AUXSEL,   0x00);
  S->rg[ADF4350_RG_03] = u32_setbits(S->rg[ADF4350_RG_03],ADF4350_RGM_03_CLKDIV,   0x0800);
  S->rg[ADF4350_RG_03] = u32_setbits(S->rg[ADF4350_RG_03],ADF4350_RGM_03_DIVMODE,  0x00);
  S->rg[ADF4350_RG_03] = u32_setbits(S->rg[ADF4350_RG_03],ADF4350_RGM_03_CSR,      0x00);
  S->rg[ADF4350_RG_02] = u32_setbits(S->rg[ADF4350_RG_02],ADF4350_RGM_02_CPCS,     0x04);
  S->rg[ADF4350_RG_02] = u32_setbits(S->rg[ADF4350_RG_02],ADF4350_RGM_02_MUXOUT,   0x03);
  S->rg[ADF4350_RG_01] = u32_setbits(S->rg[ADF4350_RG_01],ADF4350_RGM_01_PHASE,    0x0000);
  S->rg[ADF4350_RG_01] = u32_setbits(S->rg[ADF4350_RG_01],ADF4350_RGM_01_PRESCALE, 0x00);

  if(fact>0) {
    rate->f64[0] = fact;
    rtn = 0;
  }else {
    rate->f64[0] = 0;
    rtn = -1;
  }

  return(rtn);
}

