
/****************************************************************************/
/*                                                                          */
/*    A2DM20 module interface routine                                       */
/*                                                                          */
/****************************************************************************/

#include "iomlib_a2dm20.h" 

int_4 part_wr(PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 data, int_u4 mask)
{
  int_4 timeout = 0;
  int_4 pin = 0;
  int_4 rdbk[2];

  if(func == PFUNC_PWR) pin = data;
 
  int_4 part_reg0 = data;
  int_4 part_reg1 = (pin<<16)|(func<<12)|addr;

  int_4 part_addr0 = (part_ID<<8);
  int_4 part_addr1 = (part_ID<<8)+4;
  int_4 part_busy;

  if(mask) {
    /* Write the Mask */
    pic_jpmemwr(p,mport,part_addr0,mask,4);
    pic_jpmemwr(p,mport,part_addr1,((PFUNC_RWR<<12)|0x00000FFF),4);
  }

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
    ++timeout;
  }while(part_busy); 

  /* Check for Busy */
  part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  if(part_busy) dbgerror("WRITING: WHY IS THE PART BUSY? 0x%x\n",part_ID);

  /* Perform Write */
  pic_jpmemwr(p,mport,part_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,part_addr1,part_reg1,4);

  /* Wait Until Ready */
  timeout = 0;
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
    ++timeout;
  }while(part_busy);

  return(0);
}

int_4 a2dm20_wr(PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 data, int_u4 mask)
{
  int_4 cport = 1;

  if(part_ID==AD6688_ID) {
    if(func==PFUNC_PWR){
      part_wr(p,cport,part_ID,func,0,data,mask);
    }else {
      part_wr(p,cport,part_ID,func,1,addr,0xFFFF);
      part_wr(p,cport,part_ID,func,0,data,mask);
    }
  } else if(part_ID==JESD_AVS_ID) {
    if(func==PFUNC_PWR){
      part_wr(p,cport,part_ID,func,0,data,mask);
    }else {
      part_wr(p,cport,part_ID,func,1,addr,0xFFFF);
      part_wr(p,cport,part_ID,func,0,data,mask);
    }
  } else if(part_ID==JESD_CFG_ID) {
    if(func==PFUNC_PWR){
      part_wr(p,cport,part_ID,func,0,data,0);
    }else if(addr==0xFFF) {
      part_wr(p,cport,part_ID,func,0,data,0);
    }else {
      part_wr(p,cport,part_ID,func,1,addr,0);
      part_wr(p,cport,part_ID,func,2,data,0);
    }
  } else {
    part_wr(p,cport,part_ID,func,addr,data,mask);
  }
  return(0);
}

int_4 part_rd(PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 *data, int_u4 mask)
{

  int_4 part_reg0 = 0;
  int_4 part_reg1 = (func<<12)|addr;

  int_4 part_addr0 = (part_ID<<8);
  int_4 part_addr1 = (part_ID<<8)+4;
  int_4 part_busy;

  /* Write the Mask */
  if(mask) {
    pic_jpmemwr(p,mport,part_addr0,mask,4);
    pic_jpmemwr(p,mport,part_addr1,((PFUNC_RWR<<12)|0x00000FFF),4);
    do{
      part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
    }while(part_busy);
  }

  /* Check for Busy */
  part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  if(part_busy) dbgerror("READING: WHY IS THE PART BUSY? 0x%x\n",part_ID);

  /* Perform Write */
  pic_jpmemwr(p,mport,part_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,part_addr1,part_reg1,4);

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  }while(part_busy);
  
  *data = (func == PFUNC_PWR)? (pic_jpmemrd(p,mport,part_addr1,4)>>16)&0xFFFF : pic_jpmemrd(p,mport,part_addr0,4);

  return(0);
}

int_4 part_rdstat(PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 *data, int_4 *stat, int_u4 mask)
{

  int_4 part_reg0 = 0;
  int_4 part_reg1 = (func<<12)|addr;

  int_4 part_addr0 = (part_ID<<8);
  int_4 part_addr1 = (part_ID<<8)+4;
  int_4 part_busy;

  /* Write the Mask */
  if(mask) {
    pic_jpmemwr(p,mport,part_addr0,mask,4);
    pic_jpmemwr(p,mport,part_addr1,((PFUNC_RWR<<12)|0x00000FFF),4);
    do{
      part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
    }while(part_busy);
  }

  /* Check for Busy */
  part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  if(part_busy) dbgerror("READING: WHY IS THE PART BUSY? 0x%x\n",part_ID);

  /* Perform Write */
  pic_jpmemwr(p,mport,part_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,part_addr1,part_reg1,4);

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  }while(part_busy);
  
  *data = (func == PFUNC_PWR)? (pic_jpmemrd(p,mport,part_addr1,4)>>16)&0xFFFF : pic_jpmemrd(p,mport,part_addr0,4);
  *stat = pic_jpmemrd(p,mport,part_addr1,4) & 0xFFFF;

  return(0);
}

int_4 a2dm20_rd(PICSTRUCT *p, int_4 mport, int_4 part_ID, int_4 func, int_4 addr, int_4 *data, int_u4 mask)
{
  int_4 cport = 1;

  if(part_ID==AD6688_ID) {
    part_wr(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
    part_rd(p,cport,part_ID,PFUNC_RRD,2,data,0xFFFF);
  } else if(part_ID==JESD_AVS_ID) {
    part_wr(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
    part_rd(p,cport,part_ID,PFUNC_RRD,2,data,0xFFFF);
  } else if(part_ID==JESD_CFG_ID) {
    part_wr(p,cport,part_ID,PFUNC_RWR,1,addr,0xFFFF);
    part_rd(p,cport,part_ID,PFUNC_RRD,3,data,0xFFFF);
  } else {
    part_rd(p,mport,part_ID,func,addr,data,mask);
  }
  return(0);
}

int_4 a2dm20_set_state(PICSTRUCT *p, int_4 mport, int_4 *state_ptr)
{
  int_4 i,local_crc;

  if(state_ptr == 0) return(0);

  local_crc = crc32_le(0xFFFFFFFF,state_ptr,sizeof(struct AD6688_obj));
  for(i=0;i<sizeof(struct AD6688_obj)/4;i++) {
    a2dm20_wr(p,mport,MOD_ID,PFUNC_RWR,i,*state_ptr,0);
    state_ptr++;
  }
  a2dm20_wr(p,mport,MOD_ID,PFUNC_RWR,i,local_crc,0);

  return(0);
}

int_4 a2dm20_get_state(PICSTRUCT *p, int_4 mport, int_4 *state_ptr)
{
  int_4 i;

  if(state_ptr == 0) return(0);

  for(i=0;i<sizeof(struct AD6688_obj)/4;i++) {
    part_rd(p,mport,MOD_ID,PFUNC_RRD,i,state_ptr,0);
    state_ptr++;
  }

  return(0);
}

int_4 a2dm20_get_clkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 status;
  int_4 stat_data;
  int_4 cport=1;
  int_4 extref;
  int_4 pps_stat, pps_cnt;

  a2dm20_rd(p,cport,OSC_ID,PFUNC_PRD,0,&extref,0xFFFF);
  a2dm20_rd(p,cport,MOD_ID,PFUNC_PRD,1,&stat_data,0xFFFF);
  a2dm20_rd(p,cport,MTGO_ID,PFUNC_RRD,1,&pps_stat,0xFFFF);
  a2dm20_rd(p,cport,MTGO_ID,PFUNC_RRD,2,&pps_cnt,0xFFFF);

  extref &= 0x1;
  status = 0;
  if (extref && (stat_data&0x1000)==0) status |= 0x1;
  if ((stat_data&0x2000)==0) status |= 0x2;

  return(status);
}

int_4 a2dm20_get_avmmdbg(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;

  int_4 i,k,l;
  int_4 memdat;
  int_4 dbgrst,dbgsel,trigsel,datasel,mdepth,displ,aoff,hbidbg;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 regword;
  int_4 cport = 1;
  int_4 CMDDEPTH = 64;
  FILE *file0;

  union dbgMem dbgArray0[500];

  pic_lock(p,LOCK_ALLOC);

  dbgrst  = findintflagdef("DBGRST",p->config,1);
  dbgsel  = findintflagdef("DBGSEL",p->config,0);
  aoff    = findintflagdef("DBGAOFF",p->config,0);
  trigsel = findintflagdef("TRIGSEL",p->config,0);
  datasel = findintflagdef("DATASEL",p->config,0);
  mdepth  = findintflagdef("DBGDEPTH",p->config,500);
  displ   = findintflagdef("DBGDISPL",p->config,0);
  hbidbg  = findintflagdef("HBDBG",p->config,7);

  if(mdepth>500) mdepth=500;

  printf("JJB::: Debug mdepth=%d\n",mdepth);

  if(dbgrst) {
    regword = ((trigsel&0x7)<<8) | ((datasel&0xF)<<4);
    printf("JJB::: Calling dbgsetup, %d,%d,%d, 0x%x\n",dbgsel,trigsel,datasel,regword); 
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword,0xFFFF);
    udelay(100);
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,0x800|regword,0xFFFF);
    udelay(100);
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword,0xFFFF);
  }

  a2dm20_rd(p,cport,MEM_0_ID,PFUNC_PRD,0,&stat_data,0xFFFF);
  printf("JJB::: MEM Pin Stat = 0x%x\n",stat_data); 

  k=0;l=0;
  for(i=0;i<mdepth;i+=2) {
    a2dm20_rd(p,cport,MEM_0_ID,PFUNC_RRD,i+aoff,&stat_data,0xFFFF);
    dbgArray0[k].dbgWord[0] = stat_data;
    a2dm20_rd(p,cport,MEM_0_ID,PFUNC_RRD,i+1+aoff,&stat_data,0xFFFF);
    dbgArray0[k].dbgWord[1] = stat_data;

    printf("MEMDAT:  %d = 0x%0.4x, 0x%0.4x, 0x%0.4x, 0x%0.4x\n",k,dbgArray0[k].dbgSig16.dat0,dbgArray0[k].dbgSig16.dat1,dbgArray0[k].dbgSig16.dat2,dbgArray0[k].dbgSig16.dat3);

    k=k+1; l=l+1;
  }

  printf("Sizof membuf = %d\n",sizeof(dbgArray0[0]));

  pic_lock(p,LOCK_FREE);

  return (0);
}

int_4 a2dm20_get_dbg(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;

  int_4 i,k,l;
  int_4 memdat;
  int_4 dbgrst,dbgsel,trigsel,datasel,mdepth,displ,aoff,hbidbg;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 regword;
  int_4 cport = 1;
  int_4 CMDDEPTH = 64;
  FILE *file0,*file1,*file2,*file3,*file4,*file5;

/* 
  int_4 dbgSig0[250],dbgSig1[250];
  int_4 dbgSig2[250],dbgSig3[250],dbgSig5[250];
  int_8 dbgSig4[250];
*/
  union dbgMem dbgArray0[500];
  union dbgMem dbgArray1[500];
  union dbgMem dbgArray2[500];
  union dbgMem dbgArray3[500];

  pic_lock(p,LOCK_ALLOC);

  dbgrst  = findintflagdef("DBGRST",p->config,1);
  dbgsel  = findintflagdef("DBGSEL",p->config,0);
  aoff    = findintflagdef("DBGAOFF",p->config,0);
  trigsel = findintflagdef("TRIGSEL",p->config,0);
  datasel = findintflagdef("DATASEL",p->config,0);
  mdepth  = findintflagdef("DBGDEPTH",p->config,500);
  displ   = findintflagdef("DBGDISPL",p->config,0);
  hbidbg  = findintflagdef("HBDBG",p->config,7);

  if(mdepth>500) mdepth=500;

  printf("JJB::: Debug mdepth=%d\n",mdepth);

  if(dbgrst) {
    regword = ((trigsel&0x7)<<8) | ((datasel&0xF)<<4);
    printf("JJB::: Calling dbgsetup, %d,%d,%d, 0x%x\n",dbgsel,trigsel,datasel,regword); 
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword,0xFFFF);
    udelay(100);
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,0x800|regword,0xFFFF);
    udelay(100);
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword,0xFFFF);
/*
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword|0x01,0xFFFF);
    a2dm20_wr(p,cport,MEM_1_ID,PFUNC_PWR,0,regword|0x01,0xFFFF);
    a2dm20_wr(p,cport,MEM_2_ID,PFUNC_PWR,0,regword|0x01,0xFFFF);
    a2dm20_wr(p,cport,MEM_3_ID,PFUNC_PWR,0,regword|0x01,0xFFFF);
    a2dm20_wr(p,cport,MEM_0_ID,PFUNC_PWR,0,regword,0xFFFF);
    a2dm20_wr(p,cport,MEM_1_ID,PFUNC_PWR,0,regword,0xFFFF);
    a2dm20_wr(p,cport,MEM_2_ID,PFUNC_PWR,0,regword,0xFFFF);
    a2dm20_wr(p,cport,MEM_3_ID,PFUNC_PWR,0,regword,0xFFFF);
*/
  }

  a2dm20_rd(p,cport,MEM_0_ID,PFUNC_PRD,0,&stat_data,0xFFFF);
  printf("JJB::: MEM Pin Stat = 0x%x\n",stat_data); 
/*
  file0 = fopen("d2awgm3_dbg0","ab");
  file1 = fopen("d2awgm3_dbg1","ab");
  file2 = fopen("d2awgm3_dbg2","ab");
  file3 = fopen("d2awgm3_dbg3","ab");
  file4 = fopen("d2awgm3_dbg4","ab");
  file5 = fopen("d2awgm3_dbg5","ab");
*/
  k=0;l=0;
  for(i=0;i<mdepth;i+=2) {
    a2dm20_rd(p,cport,MEM_0_ID,PFUNC_RRD,i+aoff,&stat_data,0xFFFF);
    dbgArray0[k].dbgWord[0] = stat_data;
    a2dm20_rd(p,cport,MEM_0_ID,PFUNC_RRD,i+1+aoff,&stat_data,0xFFFF);
    dbgArray0[k].dbgWord[1] = stat_data;

    printf("MEMDAT:  %d = 0x%x, 0x%x, 0x%x, 0x%x\n",k,
           dbgArray0[k].dbgSig16.dat0,dbgArray0[k].dbgSig16.dat1,dbgArray0[k].dbgSig16.dat2,dbgArray0[k].dbgSig16.dat3);
/*
    a2dm20_rd(p,cport,MEM_1_ID,PFUNC_RRD,i+aoff,&stat_data,0xFFFF);
    dbgArray1[k].dbgWord[0] = stat_data;
    a2dm20_rd(p,cport,MEM_1_ID,PFUNC_RRD,i+1+aoff,&stat_data,0xFFFF);
    dbgArray1[k].dbgWord[1] = stat_data;

    a2dm20_rd(p,cport,MEM_2_ID,PFUNC_RRD,i+aoff,&stat_data,0xFFFF);
    dbgArray2[k].dbgWord[0] = stat_data;
    a2dm20_rd(p,cport,MEM_2_ID,PFUNC_RRD,i+1+aoff,&stat_data,0xFFFF);
    dbgArray2[k].dbgWord[1] = stat_data;

    a2dm20_rd(p,cport,MEM_3_ID,PFUNC_RRD,i+aoff,&stat_data,0xFFFF);
    dbgArray3[k].dbgWord[0] = stat_data;
    a2dm20_rd(p,cport,MEM_3_ID,PFUNC_RRD,i+1+aoff,&stat_data,0xFFFF);
    dbgArray3[k].dbgWord[1] = stat_data;

    printf("MEMDAT:  %d = 0x%0.16llx, 0x%0.16llx, 0x%0.16llx, 0x%0.16llx\n",k,
           dbgArray0[k].dbgSig64.dat0,dbgArray1[k].dbgSig64.dat0,dbgArray2[k].dbgSig64.dat0,dbgArray3[k].dbgSig64.dat0);
*/
/*
    dbgSig0[l] = dbgArray[k].dbgSig16.dat0;
    dbgSig1[l] = dbgArray[k].dbgSig16.dat1;
    dbgSig2[l] = dbgArray[k].dbgSig16.dat2;
    dbgSig3[l] = dbgArray[k].dbgSig16.dat3;
    dbgSig4[l] = dbgArray[k].dbgSig1.dat0;
    dbgSig5[l] = dbgArray[k].dbgSig1.dat2;
*/
    k=k+1; l=l+1;
  }

  printf("Sizof membuf = %d\n",sizeof(dbgArray0[0]));

/*
  fwrite(&dbgSig0,sizeof(int_4),250,file0);
  fwrite(&dbgSig1,sizeof(int_4),250,file1);
  fwrite(&dbgSig2,sizeof(int_4),250,file2);
  fwrite(&dbgSig3,sizeof(int_4),250,file3);
  fwrite(&dbgSig4,sizeof(int_8),250,file4);
  fwrite(&dbgSig5,sizeof(int_4),250,file5);

  fclose(file0);
  fclose(file1);
  fclose(file2);
  fclose(file3);
  fclose(file4);
  fclose(file5);
*/
  pic_lock(p,LOCK_FREE);

  return (0);
}

void ad6688_initjesd(PICSTRUCT *p, int_4 mport)
{
/* 7 Initialize JESD link - */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_START_REG, 0x4F, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_START_REG, 0x0F, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x04, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CAL_REG, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_LOL_REG, 0x08, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_LOL_REG, 0x00, 0xFFFF);
}

int_4 ad6688_cfg(PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst, int_4 init, int_4 dbg)
{
  
  real_8 jesd_rate; 
  int_4 ad6688_reg = 0;
  int_4 Qignore,Isrc,Qsrc;
  int_4 i;

/* 1 Power down the link */
  /*  1a - 0x0571 0x15 */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_JESD_CTL1_REG, 0x15, 0xFFFF);

  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SMPL_MODE_REG, 0x01, 0xFFFF);

  switch (ad6688_inst->chip_dec) {
    case 1: ad6688_reg = AD6688_CHIP_DEC(0); break; 
    case 2: ad6688_reg = AD6688_CHIP_DEC(1); break; 
    case 3: ad6688_reg = AD6688_CHIP_DEC(8); break; 
    case 4: ad6688_reg = AD6688_CHIP_DEC(2); break; 
    case 5: ad6688_reg = AD6688_CHIP_DEC(5); break; 
    case 6: ad6688_reg = AD6688_CHIP_DEC(9); break; 
    case 8: ad6688_reg = AD6688_CHIP_DEC(3); break; 
    case 10: ad6688_reg = AD6688_CHIP_DEC(6); break; 
    case 12: ad6688_reg = AD6688_CHIP_DEC(10); break; 
    case 15: ad6688_reg = AD6688_CHIP_DEC(7); break; 
    case 16: ad6688_reg = AD6688_CHIP_DEC(4); break; 
    case 20: ad6688_reg = AD6688_CHIP_DEC(13); break; 
    case 24: ad6688_reg = AD6688_CHIP_DEC(11); break; 
    case 30: ad6688_reg = AD6688_CHIP_DEC(14); break; 
    case 40: ad6688_reg = AD6688_CHIP_DEC(15); break; 
    case 48: ad6688_reg = AD6688_CHIP_DEC(12); break; 
    default: ad6688_reg = AD6688_CHIP_DEC(0);
  } 
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CHIP_DEC_REG, ad6688_reg, 0xFFFF); 

  Qignore = 0;
  Isrc = 0;
  Qsrc = 0;
  for(i=0;i<ad6688_inst->ddc_en;++i) {
    Qignore = 0;
    if(ad6688_inst->DDC[i].src == 2) {
      Isrc = 0;
      Qsrc = 1;
    } else {
      Isrc = ad6688_inst->DDC[i].src;
      Qsrc = ad6688_inst->DDC[i].src;
    }

    switch (ad6688_inst->DDC[i].dec) {
     case 2: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(3);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 3: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(7) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 4: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(0);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 6: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(4);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 8: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(1);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 10: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(2) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 12: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(5);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 15: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(8) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 16: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(2);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 20: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(3) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 24: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(6);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 30: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(9) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 40: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(4) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     case 48: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(7);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
     default: {
       ad6688_reg = AD6688_DDC_GAIN_SEL(ad6688_inst->gain) | AD6688_DDC_CMPLX_REAL(Qignore) | AD6688_DDC_DEC_SEL1(1);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_CNTL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);

       ad6688_reg = AD6688_DDC_DEC_SEL2(0) | AD6688_DDC_QIN_SEL(Qsrc) | AD6688_DDC_IIN_SEL(Isrc);
       a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_ISEL_REG+i*AD6688_DDCx_OFFSET, ad6688_reg, 0xFFFF);
       break;
     }
    }
  }
  if(ad6688_inst->ddc_en > 2) {
    ad6688_reg = AD6688_CHIP_QIGN(Qignore) | AD6688_CHIP_MODE(3);
  } else {
    ad6688_reg = AD6688_CHIP_QIGN(Qignore) | AD6688_CHIP_MODE(ad6688_inst->ddc_en);
  }
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CHIP_MODE_REG, ad6688_reg, 0xFFFF); 

/* 2 Set JESD link configuration options */
  jesd_rate = (ad6688_inst->M * ad6688_inst->NP * 10 * ad6688_inst->fs) / (8 * ad6688_inst->chip_dec * ad6688_inst->L);
  if(p->verbose) printf("A2DM20 JESD Rate calculated as %f MHz\n",jesd_rate);
  if(jesd_rate < 1.6875e3) {
    printf("JESD Lane rate too low, something is messed up...\n");
    return(-1);
  }else if(jesd_rate < 3.375e3){
    ad6688_reg = AD6688_LANE_RATE(5); 
  }else if(jesd_rate < 6.75e3){
    ad6688_reg = AD6688_LANE_RATE(1); 
  }else if(jesd_rate < 13.5e3){
    ad6688_reg = AD6688_LANE_RATE(0); 
  }else if(jesd_rate > 15.5e3){
    printf("JESD Lane rate too high, something is messed up...\n");
    return(-1);
  }else {
    ad6688_reg = AD6688_LANE_RATE(3); 
  }
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_PLL_CNTL_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_SCR(1) | AD6688_L(ad6688_inst->L-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SCR_L_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_F(ad6688_inst->F-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_F_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_K(ad6688_inst->K-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_K_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_M(ad6688_inst->M-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_M_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_CS(ad6688_inst->CS) | AD6688_N(ad6688_inst->N-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_CS_N_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_SUBCLS(1) | AD6688_NP(ad6688_inst->NP-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_SUBCLS_NP_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_S(ad6688_inst->S-1);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_S_REG, ad6688_reg, 0xFFFF);

  ad6688_reg = AD6688_HD(0) | AD6688_CF(0);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_HD_CF_REG, ad6688_reg, 0xFFFF);

/* 3 Configure the detailed options (?) */
/* 4 Set output lane mapping (Default good) */
/* 5 set additional driver configuration options (?) */
/* 6 Power up the link */
  /*   6a - 0x0571 0x14 */
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,AD6688_JESD_CTL1_REG, 0x14, 0xFFFF);

/* Set User Test Pattern */
/*
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0551, 0x11, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0552, 0x11, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0553, 0x33, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0554, 0x33, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0555, 0x55, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0556, 0x55, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0557, 0x77, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0558, 0x77, 0xFFFF);
*/

/* PUT IN TEST MODE... */
/*
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0550, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0327, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0347, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0367, 0x00, 0xFFFF);
  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x0387, 0x00, 0xFFFF);
*/

/* 7 Initialize JESD link - */
  ad6688_initjesd(p,mport);

  return (0);
}

int_4 a2dm20_lmx2592(PICSTRUCT *p, int_4 mport, real_8 rate, int_4 div, int_4 pwrlvl, int_4 chrgpump, int_4 chrggn, int_4 vco_idac, int_4 vco_capctrl)
{
  int_4 i;
  int_4 (*lmx2582_regs)[2];
  int_4 Fpd = 160000000;
  int_4 PLL_N_PRE = 2;
  int_4 PLL_DEN = 1000000;
  real_8 Fvco_, Fvco_rem;
  int_4 PLL_N,PLL_NUM;
  int_4 cmdbuf_data;
  int_4 M,N;
  int_4 vco_sel;

  int_4 lmx2582_base_table[43][2] = {
    {0x40, 0x00AF},{0x3E, 0x0000},{0x3D, 0x0001},{0x3B, 0x0000},
    {0x30, 0x03FC},{0x2F, 0x08E0},{0x2E, 0x2021},{0x2D, 0xACA0},
    {0x2C, 0x03B9},{0x2B, 0x0020},{0x2A, 0x0000},{0x29, 0xE100},
    {0x28, 0x05F5},{0x27, 0x8104},{0x26, 0x001E},{0x25, 0x4000},
    {0x24, 0x0848},{0x23, 0x039B},{0x22, 0xC3F0},{0x21, 0x4210},
    {0x20, 0x4210},{0x1F, 0x0401},{0x1E, 0x0034},{0x1D, 0x0084},
    {0x1C, 0x2924},{0x19, 0x0000},{0x18, 0x0509},{0x17, 0x8842},
    {0x16, 0x2300},{0x14, 0x012C},{0x13, 0x0965},{0x0E, 0x0420},
    {0x0D, 0x4000},{0x0C, 0x7002},{0x0B, 0x0028},{0x0A, 0x10D8},
    {0x09, 0x0B02},{0x08, 0x1084},{0x07, 0x28B2},{0x04, 0x1943},
    {0x02, 0x0500},{0x01, 0x0809},{0x00, 0x2318}
  };

  lmx2582_regs = &lmx2582_base_table[0];

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_PWR, 0, 1, 0xFFFF);
  udelay(500000);
  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, 0x00,0x01,0xFFFF);

  Fvco_ = rate / ((real_8)(Fpd*PLL_N_PRE));
  Fvco_rem = Fvco_ - (int_4)Fvco_;

  PLL_N = (int_4)Fvco_;
  PLL_NUM = (int_4)(Fvco_rem * PLL_DEN);

  M = PLL_NUM;
  N = PLL_DEN;

  lmx2582_regs[11][1]  = (N & 0x0000FFFF);
  lmx2582_regs[12][1]  = (N & 0xFFFF0000)>>16;
  lmx2582_regs[14][1] = (PLL_N & 0xFFF)<<1;
  lmx2582_regs[7][1]  = (M & 0x0000FFFF);
  lmx2582_regs[8][1]  = (M & 0xFFFF0000)>>16;
  lmx2582_regs[5][1]  = (lmx2582_regs[5][1] & 0xFFC0) | (pwrlvl & 0x003F);
  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xC0FF) | ((pwrlvl<<8) & 0x3F00); 
  lmx2582_regs[31][1]  = (lmx2582_regs[31][1] & 0xF00F) | ((chrgpump<<2) & 0x007C) | ((chrgpump<<7) & 0x0F80) | (chrggn & 0x0003); 
  
  for (i=0; i<42; i++) { 
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, lmx2582_regs[i][0],lmx2582_regs[i][1],0xFFFF);
  } 

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  8, LMX_VCO_IDAC_OVR(1), LMX_VCO_IDAC_OVR(1));
  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, 19, LMX_VCO_IDAC(vco_idac), LMX_VCO_IDAC(0x1FF));

  if(div == 32) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 24) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 64) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(2), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 48) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(2), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else if(div == 128) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(8), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  } else if(div == 96) {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(8), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(4), LMX_SEG3(0xF));
  } else {
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG1(0), LMX_SEG1(1));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  35, LMX_SEG2(1), LMX_SEG2(0xF));
    a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR,  36, LMX_SEG3(8), LMX_SEG3(0xF));
  }

  a2dm20_wr(p, mport, LMX2582_ID, PFUNC_RWR, lmx2582_regs[i][0],lmx2582_regs[i][1], 0xFFFF);

  return(0);
}

/*---------------------------------------------------------------------------*/
/* A2DM20 - Public Interfaces for Module Initialization, Setup, and Adjust   */
/*---------------------------------------------------------------------------*/

int_4 frac2rat(real_8 x, int* M, int* N)
{
  real_8 a,b,e;
  int_4 h1,h2,k1,k2,ux;
  int_4 iter,j;

    iter = 16;
    b = x;
    h1 = 1;  h2 = 0;  k1 = 0;  k2 = 1;
    for(j=0;j<iter;j++) {
      a = (int)b;
      ux = h1;
      h1 = a*h1 + h2;
      h2 = ux;
      ux = k1;
      k1 = a*k1 + k2;
      k2 = ux;
      e = x-((real_8)h1/(real_8)k1);

      if(fabs(e*1e15) < 1) {
        *M = h1;
        *N = k1;
        return(0);
      }
      else b = 1/(b-a);
    }

    return(-1);
}

int_4 DDC_setfreq (PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  real_8 fs,fc,x,fa;
  int_4 ddc_num,cport,offset;
  int_4 M,N;
  real_8 tmp;
  int_8 ftw,maw,mbw;
  int_4 ftw_0,ftw_1,ftw_2,ftw_3,ftw_4,ftw_5;
  int_4 maw_0,maw_1,maw_2,maw_3,maw_4,maw_5;
  int_4 mbw_0,mbw_1,mbw_2,mbw_3,mbw_4,mbw_5;

  fs = ad6688_inst->fs;
  fc = ad6688_inst->DDC[mport-1].fc;
  if (findintflag("AR18",p->config)>0) fc -= 2000.0;	/* flip the 1GHz center freq */
  ddc_num = mport-1;
  cport = 1;

  x = fmod(fc,fs)/fs;
  if(frac2rat(x,&M,&N) < 0) printf("ERROR:: Tuning to %0.16f not possible... M=%d,N=%d\n",fc,M,N); 

  tmp = (real_8)M/(real_8)N;
  ftw = (int_8)(tmp * 2.81474976710656E14);
  maw = (((int_8)M % (int_8)N) * ((int_8)2.81474976710656E14 % (int_8)N)) % (int_8)N;
  mbw = (int_8)N;

  fa = (((real_8)ftw+((real_8)maw/(real_8)mbw))*fs) / 2.81474976710656E14;
  /* if(fa != fc) printf("WARNING: Desired Frequency %f MHz cannot be met, using %f MHz\n",fc,fa); */
/*
  printf("Fc = %f, Fs = %f\n",fc,fs);
  printf("x = %f\n",x);
  printf("M = %d, N = %d\n",M,N);
  printf("ftw = %lld, maw = %lld, mbw = %lld\n", ftw,maw,mbw);
  printf("Desired Freq = %f, Actual Freq = %f\n", fc,fa);
*/
  ad6688_inst->DDC[mport-1].fc = fa;

  ftw_0 = ftw & 0xFF;
  ftw_1 = (ftw & 0xFF00)>>8;
  ftw_2 = (ftw & 0xFF0000)>>16;
  ftw_3 = (ftw & 0xFF000000)>>24;
  ftw_4 = (ftw & 0xFF00000000)>>32;
  ftw_5 = (ftw & 0xFF0000000000)>>40;

  maw_0 = maw & 0xFF;
  maw_1 = (maw & 0xFF00)>>8;
  maw_2 = (maw & 0xFF0000)>>16;
  maw_3 = (maw & 0xFF000000)>>24;
  maw_4 = (maw & 0xFF00000000)>>32;
  maw_5 = (maw & 0xFF0000000000)>>40;

  mbw_0 = mbw & 0xFF;
  mbw_1 = (mbw & 0xFF00)>>8;
  mbw_2 = (mbw & 0xFF0000)>>16;
  mbw_3 = (mbw & 0xFF000000)>>24;
  mbw_4 = (mbw & 0xFF00000000)>>32;
  mbw_5 = (mbw & 0xFF0000000000)>>40;

  offset = ddc_num*AD6688_DDCx_OFFSET;

  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC0_REG+offset, ftw_0, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC1_REG+offset, ftw_1, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC2_REG+offset, ftw_2, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC3_REG+offset, ftw_3, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC4_REG+offset, ftw_4, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_PINC5_REG+offset, ftw_5, 0xFFFF);

  offset = ddc_num*AD6688_MxW_OFFSET;

  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW0_REG+offset, maw_0, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW1_REG+offset, maw_1, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW2_REG+offset, maw_2, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW3_REG+offset, maw_3, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW4_REG+offset, maw_4, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MAW5_REG+offset, maw_5, 0xFFFF);

  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW0_REG+offset, mbw_0, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW1_REG+offset, mbw_1, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW2_REG+offset, mbw_2, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW3_REG+offset, mbw_3, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW4_REG+offset, mbw_4, 0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,AD6688_DDC0_MBW5_REG+offset, mbw_5, 0xFFFF);

  return(0);
}

real_8 a2dm20_get_freq(PICSTRUCT *p, int_4 mport)
{
  real_8 freq;
  struct AD6688_obj ad6688_local;
  struct AD6688_obj *ad6688_ptr;
  UDB fs = udb_initializer;
  int_4 state_crc,local_crc;
  int_4 cport = 1;

  if(p->iom_state[0] == 0) {
    ad6688_ptr = &ad6688_local;
    a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
  }else {  
    ad6688_ptr = (struct AD6688_obj *)p->iom_state[0];
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    a2dm20_rd(p,cport,MOD_ID,PFUNC_RRD,32,&state_crc,0);
    if(local_crc != state_crc) {
      a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
      local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    }
  }

  if(ad6688_ptr->M == 2) freq = ad6688_ptr->DDC[0].fc;
  else                   freq = ad6688_ptr->DDC[mport-1].fc;

  return(freq);
}

int_4 a2dm20_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq)
{
  struct AD6688_obj *ad6688_ptr;
  struct AD6688_obj ad6688_local;
  real_8 freq_;
  int_4 local_crc,state_crc;
  UDB fs = udb_initializer;
  int_4 cport = 1;
  
  pic_lock(p,LOCK_ALLOC);

/* Get Current State... */
  if(p->iom_state[0] == 0) {
    ad6688_ptr = &ad6688_local;
    a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
  }else {  
    ad6688_ptr = (struct AD6688_obj *)p->iom_state[0];
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    a2dm20_rd(p,cport,MOD_ID,PFUNC_RRD,32,&state_crc,0);
    if(local_crc != state_crc) {
      a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
      local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    }
  }

  if(ad6688_ptr->M == 2 ) {
    ad6688_ptr->DDC[0].fc = freq;
    DDC_setfreq (p, 1, ad6688_ptr);
  } else {
    ad6688_ptr->DDC[mport-1].fc = freq;
    DDC_setfreq (p, mport, ad6688_ptr);
  }

  a2dm20_set_state(p, cport, (int_4*)ad6688_ptr);

  freq_ = a2dm20_get_freq(p,mport);
  if(freq_ != freq) printf("WARNING: Desired Frequency %0.16f MHz cannot be met, using %0.16f MHz\n",freq,freq_);

  pic_lock(p,LOCK_FREE);
  return (0);
}

double rftmp;
real_8 a2dm20_get_adcrate(PICSTRUCT *p, int_4 mport)
{
  real_8 adcrate;
  struct AD6688_obj ad6688_local;
  struct AD6688_obj *ad6688_ptr;
  UDB fs = udb_initializer;
  int_4 state_crc,local_crc;
  int_4 cport = 1;

  if(p->iom_state[0] == 0) {
    ad6688_ptr = &ad6688_local;
    a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
  }else {  
    ad6688_ptr = (struct AD6688_obj *)p->iom_state[0];
    local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    a2dm20_rd(p,cport,MOD_ID,PFUNC_RRD,32,&state_crc,0);
    if(local_crc != state_crc) {
      a2dm20_get_state(p,cport,(int_4*)ad6688_ptr);
      local_crc = crc32_le(0xFFFFFFFF,ad6688_ptr,sizeof(struct AD6688_obj));
    }
  }
  adcrate = ad6688_ptr->fs;

  return(adcrate);
}

int_4 a2dm20_get_gain(PICSTRUCT *p, int_4 mport)
{
  int_4 cport = 1;
  int_4 gain;
  int_4 gain_;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,LMH6401_0_ID,PFUNC_RRD,2,&gain_,0xFFFF);
  }else { 
    a2dm20_rd(p,cport,LMH6401_1_ID,PFUNC_RRD,2,&gain_,0xFFFF);
  }

  gain = 31 - gain_;

  pic_lock(p,LOCK_FREE);

  return(gain);
}

int_4 a2dm20_get_attn(PICSTRUCT *p, int_4 mport)
{
  int_4 cport = 1;
  int_4 attn;
  real_8 attn_;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,PE43704_0_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }else { 
    a2dm20_rd(p,cport,PE43704_1_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }

  attn_ = (real_8)attn/4;
  attn = round(attn_);

  pic_lock(p,LOCK_FREE);

  return(attn);
}

int_4 a2dm20_get_adclm(PICSTRUCT *p, int_4 mport)
{
  int_4 adclm = 0;
  int_4 adclm_ = 0;
  int_4 cport = 1;

  pic_lock(p,LOCK_ALLOC);

  if(mport == 1) {
    a2dm20_rd(p,cport,ADCLM_0_ID,PFUNC_RRD,0,&adclm_,0xFFFF);
  } else {
    a2dm20_rd(p,cport,ADCLM_1_ID,PFUNC_RRD,0,&adclm_,0xFFFF);
  }

  adclm = adclm_to_cf_i32(adclm_);

  pic_lock(p,LOCK_FREE);

  return(adclm);
}

int_4 a2dm20_get_temp(PICSTRUCT *p, int_4 mport)
{
  int_4 bufaddr;
  int_4 t1,t2;
  int_4 cmdata;
  int_4 i = 0;

  pic_lock(p,LOCK_ALLOC);

  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x18E6,0x01,0xFFFF);

  /* Read from converter (t1) */
  udelay(10000);
  a2dm20_rd(p,mport,MCP3421_0_ID,PFUNC_RRD,0,&t1,0);
  t1 = (t1 & 0xFFFF0000)>>16;

  a2dm20_wr(p,mport,AD6688_ID,PFUNC_RWR,0x18E6,0x02,0xFFFF);

  /* Read from converter (t2) */
  udelay(10000);
  a2dm20_rd(p,mport,MCP3421_0_ID,PFUNC_RRD,0,&t2,0);
  t2 = (t2 & 0xFFFF0000)>>16;

  /* calculate temperature
  temp(degrees C) = abs(t2-t1) * 3.874 - 273.15 */
  cmdata = abs(t2-t1)*3.874 - 273.15;

  /* printf("JJB::: A2DM20 Temp = %d, t1 = 0x%x, t2 = 0x%x\n",cmdata,t1,t2); */

  pic_lock(p,LOCK_FREE);
  return(cmdata);
}

int_4 a2dm20_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
  int_4 bufaddr;
  int_4 rfpwr;
  int_4 cport = 1;
  real_8 pwrdb_;
  int_4 pwrdb;
  int_4 attn;
  real_8 attn_;

  pic_lock(p,LOCK_ALLOC);

  if(mport==1) {
    a2dm20_rd(p,cport,MCP3421_1_ID,PFUNC_RRD,0,&rfpwr,0);
    a2dm20_rd(p,cport,PE43704_0_ID,PFUNC_RRD,0,&attn,0xFFFF);
  } else {
    a2dm20_rd(p,cport,MCP3421_2_ID,PFUNC_RRD,0,&rfpwr,0);
    a2dm20_rd(p,cport,PE43704_1_ID,PFUNC_RRD,0,&attn,0xFFFF);
  }

  rfpwr = (rfpwr & 0xFFFF0000)>>16;
  attn_ = (real_8)attn/4;

  pwrdb_ = 30*rfpwr*1e-3 - 70 + attn_;
  pwrdb = round(pwrdb_);

/* printf("JJB: RFPWR, %d, %f, %d, %f\n",rfpwr,pwrdb_,pwrdb,attn_); */

  pic_lock(p,LOCK_FREE);
  return(pwrdb);
}

int_4 a2dm20_set_attn (PICSTRUCT *p, int_4 mport, int_4 attn)
{
  int_4 cport = 1;

  pic_lock(p,LOCK_ALLOC);

  attn = 4*attn;
  if(mport == 1) {
    a2dm20_wr(p,cport,PE43704_0_ID,PFUNC_RWR,0,attn,0xFFFF);
  }else { 
    a2dm20_wr(p,cport,PE43704_1_ID,PFUNC_RWR,0,attn,0xFFFF);
  }

  pic_lock(p,LOCK_FREE);
  return(0);
}

int_4 a2dm20_set_gain_old (PICSTRUCT *p, int_4 mport, int_4 gain)
{
  int_4 cport = 1;
  int_4 gain_;

  pic_lock(p,LOCK_ALLOC);

  if(gain < -6) gain = -6;
  else if(gain > 26) gain = 26;

  gain_ = 26-gain;

  if(mport == 1) {
    a2dm20_wr(p,cport,LMH6401_0_ID,PFUNC_RWR,2,gain_,0xFFFF);
  }else { 
    a2dm20_wr(p,cport,LMH6401_1_ID,PFUNC_RWR,2,gain_,0xFFFF);
  }

  pic_lock(p,LOCK_FREE);
  return(0);
}

/* 
  total 63 db range, 0->31 in var attn, -6->26 in var gain amp
  must keep output of attn <= -15db to avoid compression in +15db fixed amp
  6db loss in components after fixed amp and another unknown 6dB is made up in var amp
  full scale input range of A2D is 1.9V peak-to-peak or ~6dB into 50 ohms
  apparent gain between attn and amp = +15 -2 -2 -2 -6 = +3
*/
int_4 a2dm20_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain)
{
  int_4 cport=1,rfgain=0,rfattn=31,gm=4;
  int_4 expert = findintflagdef("RFEXPERT",p->config,0);

 if (expert) {
  rfattn = -1;
  rfgain = (gain<-6)? -6 : (gain>26)? 26 : gain;	/* bounds check (-6 to 26) */
  if (rfgain!=gain) printf("A2Dm20 expert mode gain=%d out of bounds (-6,26)\n",gain);
 } else {
       if (gain<=-34){ rfattn=31; rfgain=-6; printf("A2Dm20 gain=%d < min=-35\n",gain); } /* below lowest zone */
  else if (gain<=-6) { rfattn=31; rfgain=gain-gm+rfattn; }
  else if (gain<=24) { rfgain=21; rfattn=rfgain+gm-gain; }
  else if (gain<=30) { rfattn=0 ; rfgain=gain-gm+rfattn; }
  else               { rfgain=26; rfattn=0; printf("A2Dm20 gain=%d > max=30\n",gain); }	/* above highest zone */
 }
  if (p->verbose) printf("A2DM20 GAIN expert=%d gain=%d rfattn=%d rfgain=%d\n",expert,gain,rfattn,rfgain);

  pic_lock(p,LOCK_ALLOC);
  /* dont tweak attn in expert mode */
  if (expert==0) a2dm20_wr(p,cport,(mport==2)? PE43704_1_ID:PE43704_0_ID, PFUNC_RWR, 0,4*rfattn,0xFFFF);
                 a2dm20_wr(p,cport,(mport==2)? LMH6401_1_ID:LMH6401_0_ID, PFUNC_RWR, 2,26-rfgain,0xFFFF);
  pic_lock(p,LOCK_FREE);
  return(0);
}

int_4 a2dm20_enable(PICSTRUCT *p, int_4 mport, int_4 ena)
{
  int_4 cport = 1;
  
  if (ena) a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x03,0xFFFF); 
  else if (pic_acmd(p,ADON_RD,DMA_GMCFG+(3-mport),0,0)!=0); /* other side still active */
  else a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x01,0xFFFF);

  return(0);
}

int_4 get_rststat(PICSTRUCT *p, int_4 mport)
{
  int_4 rst_dat0;
  int_4 rst_dat1;
  int_4 jesd_rst,link_val,pll_locked,digital_rst,analog_rst,waitrequest;
  int_4 rx_ready,cal_busy,lockedtodata;

  a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,1,&rst_dat0,0xFFFF);
  a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,2,&rst_dat1,0xFFFF);

  jesd_rst = JESD_STAT_RST(rst_dat0);
  link_val = JESD_STAT_RXVAL(rst_dat0);
  pll_locked = JESD_STAT_PLLLOC(rst_dat0);
  digital_rst = JESD_STAT_DIGRST(rst_dat0);
  analog_rst = JESD_STAT_ALGRST(rst_dat0);
  waitrequest = JESD_STAT_WTREQ(rst_dat0) ;
  rx_ready = JESD_STAT_RXRDY(rst_dat0) ;
  cal_busy = JESD_STAT_CALBSY(rst_dat1);
  lockedtodata = JESD_STAT_RXLOC(rst_dat1) ;

  printf("JESD RST STAT[0] = 0x%x\n",rst_dat0);
  printf("JESD RST STAT[1] = 0x%x\n",rst_dat1);

  printf("jesd_rst = 0x%x\n",jesd_rst);
  printf("link_val = 0x%x\n",link_val);
  printf("pll_locked = 0x%x\n",pll_locked);
  printf("digital_rst = %d, analog_rst = %d\n",digital_rst,analog_rst);
  printf("waitrequest = %d\n",waitrequest);
  printf("rx_ready = 0x%x\n",rx_ready);
  printf("cal_busy = 0x%x\n",cal_busy);
  printf("lockedtodata = 0x%x\n",lockedtodata);
 
  return(0);
}

int_4 xcvr_setup(PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  int_4 i,channel,offset,timecnt;
  int_4 xcvr_rddata;
  int_4 xcvr_addr,xcvr_mask,xcvr_wrdata;
  int_4 xcvr_rmw;
  int_4 config_sel;
  int_4 cmdbuf_data;
  int_4 timeout = 100;


  switch(ad6688_inst->chip_dec) {
    case 2:
      config_sel = 3;
      break;
    case 3:
      config_sel = 0;
      break;
    case 4:
      config_sel = (ad6688_inst->M == 2)? 2 : 3;
      break;
    case 6:
      config_sel = 0;
      break;
    case 8:
      config_sel = 2;
      break;
    case 12:
      config_sel = 1;
      break;
    case 16:
      config_sel = 2;
      break;
    case 24:
      config_sel = 1;
      break;
    case 48:
      config_sel = 1;
      break;
    default:
      config_sel = 0;
      break;
  }

  /* get_rststat(p, mport); */

  for(channel=0;channel<8;++channel) {
  /* Get AVMM Interface */
    xcvr_addr   = 0x000;
    xcvr_mask   = 0xFF;
    xcvr_wrdata = 0x02;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
    if(xcvr_mask == 0xFF) {
      a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_wrdata,0xFFFF);
    } else {
/* printf("%d: Reading AVMM Interface\n",channel); */
      a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF);
      xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask);
/* printf("   Writing to get AVMM Interface: 0x%x\n",xcvr_rmw); */
      a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF);
    }
/*    a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF); */
/* printf("RECONFIG WAITREQUEST: %d\n",JESD_STAT_WTREQ(cmdbuf_data)); */

    if(p->verbose==2) printf("Using PLL Configuration %d, DEC = %d\n",config_sel,ad6688_inst->DDC[0].dec);
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,0xFFF,((channel<<8)|config_sel),0xFFFF);

  /* Initiate recalibration of CDR PLL */
    xcvr_addr   = 0x100;
    xcvr_mask   = 0x02;
    xcvr_wrdata = 0x02;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
/* printf("   Reading Calibrate CDR\n"); */
    a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF);
    xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask);
/* printf("   Writing to Calibrate CDR: 0x%x\n",xcvr_rmw); */
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF);

    timecnt = 0;
    do{
      a2dm20_rd(p,mport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
      /* printf("XCVR RECONFIG CALIBRATION STAT(0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data)); */
      /* udelay(2000); */
    }while((JESD_STAT_CALBSY(cmdbuf_data) != 0) && (timecnt++ < timeout));

  /* Release AVMM Interface to PreSICE */
    xcvr_addr   = 0x000;
    xcvr_mask   = 0xFF;
    xcvr_wrdata = 0x01;
    xcvr_addr = xcvr_addr | (channel << XCVR_CFG_DPRIO_ADDR_WIDTH);
    a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_wrdata,0xFFFF);

/* printf("   Reading return AVMM Interface\n"); */
    /* a2dm20_rd(p,mport,JESD_CFG_ID,PFUNC_RRD,xcvr_addr,&xcvr_rddata,0xFFFF); */
    /* xcvr_rmw = (xcvr_rddata & ~xcvr_mask) | (xcvr_wrdata & xcvr_mask); */
/* printf("   Writing to return AVMM Interface: 0x%x\n",xcvr_rmw); */
    /* a2dm20_wr(p,mport,JESD_CFG_ID,PFUNC_RWR,xcvr_addr,xcvr_rmw,0xFFFF); */
  }

  return(0);
}

int_4 jesd_setup(PICSTRUCT *p, int_4 mport, struct AD6688_obj *ad6688_inst)
{
  int_4 jesd_ilas1;
  int_4 jesd_ilas2;
  int_4 jesd_ilas12;
  int_4 fxk;

  jesd_ilas1  = 0;
  jesd_ilas2  = 0;
  jesd_ilas12 = 0;

  jesd_ilas1 |= ((ad6688_inst->L-1) & 0x1F);
  jesd_ilas1 |= (1<<7); /* Enable SCR */
  jesd_ilas1 |= ((ad6688_inst->F-1) & 0xFF)<<8;
  jesd_ilas1 |= ((ad6688_inst->K-1) & 0x1F)<<16;
  jesd_ilas1 |= ((ad6688_inst->M-1) & 0xFF)<<24;
  
  /* printf("JESD ILAS_DATA1 = 0x%x\n",jesd_ilas1); */

  jesd_ilas2 |= ((ad6688_inst->N-1) & 0x1F);
  jesd_ilas2 |= ((ad6688_inst->CS) & 0x03)<<6;
  jesd_ilas2 |= ((ad6688_inst->NP-1) & 0x1F)<<8;
  jesd_ilas2 |= (1)<<13; /* Subclass 1 */
  jesd_ilas2 |= ((ad6688_inst->S-1) & 0x1F)<<16;
  jesd_ilas2 |= (1)<<21; /* JESD204B */
  jesd_ilas2 |= ((ad6688_inst->CF) & 0x1F)<<24;
  jesd_ilas2 |= ((ad6688_inst->HD) & 0x01)<<31;

  /* printf("JESD ILAS_DATA2 = 0x%x\n",jesd_ilas2); */

  fxk = ad6688_inst->F * ad6688_inst->K;
  jesd_ilas12 |= (fxk-1)&0x3FF;

  /* printf("JESD ILAS_DATA12 = 0x%x\n",jesd_ilas12); */

  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x25,jesd_ilas1,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x26,jesd_ilas2,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,0x30,jesd_ilas12,0xFFFF);
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,(0x54)>>2,0x3F5,0xFFFF);

  return(0);
}

int_4 a2dm20_set_jesd(PICSTRUCT *p, int_4 mport, int_4 addr, int_4 data)
{
  a2dm20_wr(p,mport,JESD_AVS_ID,PFUNC_RWR,addr,data,0xFFFF);
  return(0);
}

int_4 a2dm20_get_jesd(PICSTRUCT *p, int_4 mport, int_4 addr)
{
  int_4 x;
  a2dm20_rd(p,mport,JESD_AVS_ID,PFUNC_RWR,addr,&x,0xFFFF);
  return(x);
}

int_4 a2dm20_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{ 
  real_8 rfrate,rfratef;
  struct AD6688_obj ad6688_local;
  struct AD6688_obj *ad6688_inst;
  int_4 cmdbuf_data;
  int_4 cport = 1;
  int_4 extref;
  real_8 rffreq,rffreq2,rffreq3,rffreq4;
  int_4 rfdec,rfdecf;
  int_4 rfattn;
  int_4 rfgain;
  int_4 rfinit;
  int_4 hbt;
  int_4 lmx_div3,lmx_idac,lmx_capctrl;
  int_4 lmx_pwr;
  int_4 ddcsrc,ddcsrc2,ddcsrc3,ddcsrc4;
  int_4 testmode;
  int_4 dbg,ddc2ch;
  int_4 i,err,indx;
  int_4 lmx_div;
  int_4 a2dports;
  int_4 adclmB1 = 0; 
  int_4 adclmB2 = 0; 
  int_4 adclmD  = 0; 
  int_4 timeout = 10;
  int_4 oldM20,adcgainsel;
  int_4 pack8,pack12,pack8dbg;
  int_4 ferror,ferror_,findx;
  int_4 lsbp,ppstag,mtgo,pps_offset,tcdel;
  int_4 dbg_route;
  int_4 bus_rate,xbus;
  int_4 mcbus,mcbusx;

  for(i=0;i<M20_DECTBL_SIZE;++i) {
    rfrate = a2dm20_dec_table[i]*((real_8)rate)/1e6;
 /* printf("JJB:: Searching for rfdec... rfrate = %f, rfdec = %d\n",rfrate,a2dm20_dec_table[i]);  */
    if((rfrate >= M20_RATE_MIN) && (rfrate <= M20_RATE_MAX)) break;
    else {
      ferror = (rfrate < M20_RATE_MIN)? M20_RATE_MIN-rfrate : M20_RATE_MAX-rfrate;
      findx = ((i==0) || (fabs(ferror) < fabs(ferror_)))? i : findx;
      ferror_ = ((i==0) || (fabs(ferror) < fabs(ferror_)))? ferror : ferror_;
    }
  }
/* printf("JJB::: ferror = %d, ferror_ = %d, findx = %d, rfrate = %f, closest = %d\n",ferror,ferror_,findx,rfrate,0); */

  if(i<M20_DECTBL_SIZE) rfdec = a2dm20_dec_table[i];
  else { /* No suitable Rate/Decmiation found get as close as possible */
    rfdec = a2dm20_dec_table[findx];
    rfrate = (ferror_<0)? (real_8)M20_RATE_MAX : (real_8)M20_RATE_MIN;
    printf("WARNING: Desired rate of %f MHz cannot be met, using %f MHz instead\n",((real_8)rate)/1e6,rfrate/rfdec);
  }
#ifdef BOGUS
  else if(rfrate < M20_RATE_MIN) {
    /* rfdec = a2dm20_dec_table[0]; */
    rfrate = (real_8)M20_RATE_MIN;
  } else {
    /* rfdec = a2dm20_dec_table[M20_DECTBL_SIZE-1]; */
    rfrate = (real_8)M20_RATE_MAX;
  }
#endif

  extref = findintflagdef("PREFX",p->config,0);
  rfratef = finddblflagdef("ADCRATE",p->config,0.0);
  adcgainsel = findintflagdef("ADCGAINSEL",p->config,1);
  rffreq = finddblflagdef("RFFREQ",p->config,625.0); rftmp=rffreq;
  rffreq2 = finddblflagdef("RFFREQ2",p->config,rffreq);
  rffreq3 = finddblflagdef("RFFREQ3",p->config,rffreq);
  rffreq4 = finddblflagdef("RFFREQ4",p->config,rffreq);
  rfdecf  = findintflagdef("RFDEC",p->config,0);
  rfattn = findintflagdef("RFATTN",p->config,31);
  rfgain = findintflagdef("RFGAIN",p->config,0);
  ddcsrc = findintflagdef("DDCSRC",p->config,0);
  ddcsrc2 = findintflagdef("DDCSRC2",p->config,1);
  ddcsrc3 = findintflagdef("DDCSRC3",p->config,0);
  ddcsrc4 = findintflagdef("DDCSRC4",p->config,1);
  rfinit  = findintflagdef("RFINIT",p->config,0);
  testmode = findintflagdef("RAMP",p->config,0);
  dbg = findintflagdef("M20DBG",p->config,0);
  /* ddc2ch = findintflagdef("DDC2CH",p->config,0); */
  lmx_pwr = findintflagdef("LMXPWR",p->config,0);
  lmx_div3 = ((rfdec==3) || (rfdec==6) || (rfdec==12))? 1 : 0;
  lmx_idac = findintflagdef("LMXIDAC",p->config,300);
  lmx_capctrl = findintflagdef("LMXCAP",p->config,32);
  adclmB1 = findintflagdef("ADCLMB1",p->config,9)&0xFF;
  adclmB2 = findintflagdef("ADCLMB2",p->config,15)&0xFF;
  adclmD  = findintflagdef("ADCLMD", p->config,10)&0xFF;
  oldM20  = findintflagdef("OM20", p->config,0);
  pack8dbg = findintflagdef("PACK8DBG", p->config,0);
  lsbp = findintflagdef("LSBP", p->config,0);
  ppstag = findintflagdef("PPSTAG", p->config,0);
  mtgo = findintflagdef("MTGO", p->config,0);
  dbg_route = findintflagdef("DBG_ROUTE",p->config,0);

  pack12 = (bits==-12)? 1:0;
  pack8  = (bits==-8)? 1:0;

  pack12 = findintflagdef("PACK12", p->config,pack12);

  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  /* printf("\nBEFORE RESET:: XCVR CALIBRATION COMPLETE? (0 = DONE): 0x%x\n\n",JESD_STAT_CALBSY(cmdbuf_data)); */

/* RESET JESD IP CORE */
  a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x0000,0xFFFF);
  a2dm20_wr(p,cport,JESD_CFG_ID,PFUNC_PWR,0,0x0001,0xFFFF); /* set reconfig resets */
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x0001,0xFFFF); /* set analog and digital resets */

  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  /* printf("\nAFTER RESET:: XCVR CALIBRATION COMPLETE? (0 = DONE): 0x%x\n\n",JESD_STAT_CALBSY(cmdbuf_data)); */

  if(p->iom_state[0] == 0) {
    /* printf("IOM_STATE NULL on startup ...\n"); */
    ad6688_inst = (struct AD6688_obj *)iom_alloc_state(p,cport,sizeof(struct AD6688_obj));
    vfill (0, (int_4*)ad6688_inst, sizeof(struct AD6688_obj));
    p->iom_state[0] = (void *)ad6688_inst;
  } else {
    /* printf("IOM_STATE active setup routine...\n"); */
    ad6688_inst = (struct AD6688_obj *)p->iom_state[0];
  }
    
  /* Set Manual RFRATE, RFDEC */
  if(rfratef != 0) {
    rfrate = (rfratef > M20_RATE_MAX)? (real_8)M20_RATE_MAX : (rfratef < M20_RATE_MIN)? (real_8)M20_RATE_MIN : rfratef; 
    rfdec  = round(rfrate*1e6/(real_8)rate);
    /* printf("Manual RFRATE set to %f, requested RATE = %d, calculated RFDEC = %d\n", rfrate,rate,rfdec); */
    err = 1024;
    for(i=0;i<M20_DECTBL_SIZE;++i) {
      if(rfdec == a2dm20_dec_table[i]) {
        indx = i;
        err = 0; 
        break;
      }else if(abs(rfdec - a2dm20_dec_table[i]) < err) {
        indx = i;
        err = abs(rfdec - a2dm20_dec_table[i]);
      }
/* printf("JJB::: rfdec = %d, tbl_dec = %d, err = %d, indx = %d\n",rfdec,a2dm20_dec_table[i],err,indx); */
    }
    rfdec = a2dm20_dec_table[indx];
    if((int_4)(rfrate*1e6/rfdec) != rate) {
      printf("WARNING: Requested rate of %f MHz cannont be met, using %f MHz\n",((real_8)rate)/1e6,rfrate/rfdec);
    }
  }
  if(rfdecf != 0) {
    rfdec = rfdecf;
    err = 1024;
    for(i=0;i<M20_DECTBL_SIZE;++i) {
      if(rfdec == a2dm20_dec_table[i]) {
        indx = i;
        err = 0; 
        break;
      }else if(abs(rfdec - a2dm20_dec_table[i]) < err) {
        indx = i;
        err = abs(rfdec - a2dm20_dec_table[i]);
      }
/* printf("JJB::: rfdec = %d, tbl_dec = %d, err = %d, indx = %d\n",rfdec,a2dm20_dec_table[i],err,indx); */
    }
    rfdec = a2dm20_dec_table[indx];
    rfrate = (rfratef != 0)? rfrate : ((real_8)rate)/1e6 * rfdec;
    if(rfrate < (real_8)M20_RATE_MIN) {
      rfrate = (real_8)M20_RATE_MIN;
      printf("WARNING: Desired Rate of %f MHz cannot be met, using %f MHz\n", ((real_8)rate)/1e6, rfrate/rfdec);
    } else if(rfrate > (real_8)M20_RATE_MAX) {
      rfrate = (real_8)M20_RATE_MAX;
      printf("WARNING: Desired Rate of %f MHz cannot be met, using %f MHz\n", ((real_8)rate)/1e6, rfrate/rfdec);
    }else if((int_4)(rfrate*1e6/rfdecf) != rate) {
      printf("WARNING: Requested decmiation of %d cannont be met, using %d \n",rfdecf,rfdec);
    }
  }

  /* printf("JJB:: Setting up A2DM20...\n"); */
  /* printf("JJB:: mport=%d, dir=%d, bits=%d, rate=%d, gain=%d, flags=0x%x\n",mport,dir,bits,rate,gain,flags); */
  /* printf("JJB:: fs = %f, sizeof(fs) = %d\n", rfrate, sizeof(rfrate)); */
  /* printf("JJB:: sizeof(AD6688_obj) = %d\n", sizeof(struct AD6688_obj)); */
  /* printf("JJB:: sizeof(DDC_obj) = %d\n", sizeof(struct DDC_obj)); */
  bus_rate = (rfrate/rfdec) * 2*((pack12)? 12 : (pack8)? 8 : 16) / 8;
  xbus = (bus_rate > 2000)? 1 : 0;

  if(p->verbose) printf("A2DM20 RFRATE = %f, RFDEC = %d, RFFREQ = %f, DDCSRC = %d, DataRate = %d Byte/s,XBUS = %d\n", rfrate,rfdec,rffreq,ddcsrc,bus_rate,xbus);
  if(rfrate < (real_8)M20_RATE_MINSAFE) printf("WARNING: Running the sample rate below %d MHz may result in errors or performance degradation, currently running at %f MHz\n",M20_RATE_MINSAFE,rfrate);

  ad6688_inst->fs = rfrate; /* Sample Rate in MHz */
  ad6688_inst->gain = adcgainsel;
  ad6688_inst->chip_dec = (rfdec==48)? rfdec/2 : rfdec;
  ad6688_inst->ddc_en = 2;

  ad6688_inst->DDC[0].fc = rffreq; /* Center Frequency in MHz */
  ad6688_inst->DDC[0].dec = rfdec; 
  /* if((rfdec < 6) && (mport==2)) ad6688_inst->DDC[0].src = 1; */
  if((xbus) && (mport==2)) ad6688_inst->DDC[0].src = 1;
  else  ad6688_inst->DDC[0].src = ddcsrc;
  /* ad6688_inst.DDC[0].src = ddcsrc; */

  ad6688_inst->DDC[1].fc = rffreq2; /* Center Frequency in MHz */
  ad6688_inst->DDC[1].dec = rfdec; 
  ad6688_inst->DDC[1].src = ddcsrc2;

  ad6688_inst->DDC[2].fc = rffreq3; /* Center Frequency in MHz */
  ad6688_inst->DDC[2].dec = ad6688_inst->chip_dec;
  ad6688_inst->DDC[2].src = ddcsrc3;

  ad6688_inst->DDC[3].fc = rffreq4; /* Center Frequency in MHz */
  ad6688_inst->DDC[3].dec = ad6688_inst->chip_dec;
  ad6688_inst->DDC[3].src = ddcsrc4;

/* --------------------------------------------------- */
/* JESD CORE Rules                                     */
/* F = M * N' * S / (8*L)                              */
/* K = [17/F, min(32,1024/F)]                          */
/* --------------------------------------------------- */
 
  ad6688_inst->L = 8;
  /* ad6688_inst->M = (rfdec < 6)? 2 : 4; */
  /* ad6688_inst->F = (rfdec < 6)? 1 : 2; */
  ad6688_inst->M = (xbus)? 2 : (rfdec > 12)? 8 : 4;
  ad6688_inst->F = (xbus)? 1 : 2;
  ad6688_inst->N = 16;
  ad6688_inst->NP = 16;
  ad6688_inst->K = 32;
  ad6688_inst->CS = 0;
  ad6688_inst->SUBCLS = 1;
  ad6688_inst->S = 2;
  ad6688_inst->HD = 0;
  ad6688_inst->CF = 0;

  mcbus = ((ad6688_inst->M==8) & (rfdec<=24))? 1 : 0;
  mcbusx = (rfdec>24)? 1 : 0;

  /* if(rfdec < 6) { */
  if(xbus) {
    /*a2dm20_wr(p,cport,ROUTE_ID,PFUNC_PWR,0,0x154,0xFFFF);*/
 /* printf("JJB: Route = 0x%04x\n",(dbg_route<<13)|(mcbus<<12)|(mcbusx<<9)|(pack8dbg<<6)|(pack8<<5)|(pack12<<4)|0x100|(ppstag<<2)|(mtgo<<1)|lsbp);  */
    a2dm20_wr(p,cport,ROUTE_ID,PFUNC_PWR,0,(dbg_route<<13)|(mcbus<<12)|(mcbusx<<9)|(pack8dbg<<6)|(pack8<<5)|(pack12<<4)|0x100|(ppstag<<2)|(mtgo<<1)|lsbp,0xFFFF);
    /*a2dports=(abs(bits)==16)? 5 : 4;*/
    a2dports=5;
  } else {
    /* a2dm20_wr(p,cport,ROUTE_ID,PFUNC_PWR,0,0x054,0xFFFF); */
/* printf("JJB: Route = 0x%04x\n",(dbg_route<<13)|(mcbus<<12)|(mcbusx<<9)|(pack8dbg<<6)|(pack8<<5)|(pack12<<4)|(ppstag<<2)|(mtgo<<1)|lsbp);  */
    a2dm20_wr(p,cport,ROUTE_ID,PFUNC_PWR,0,(dbg_route<<13)|(mcbus<<12)|(mcbusx<<9)|(pack8dbg<<6)|(pack8<<5)|(pack12<<4)|(ppstag<<2)|(mtgo<<1)|lsbp,0xFFFF);
    a2dports=4;
  }

/* POWER DOWN LMX */
  a2dm20_wr(p, cport, LMX2582_ID, PFUNC_PWR, 0, 0, 0xFFFF);
  udelay(5000);

  if(extref) {
    /* printf("Using External Refrence... %d\n", extref); */
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x101,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x501,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x101,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x1101,0xFFFF);
  } else {
    /* printf("Using Internal Refrence... %d\n", extref); */
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x110,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x510,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x110,0xFFFF);
    udelay(100000);
    a2dm20_wr(p,cport,OSC_ID,PFUNC_PWR,0,0x1110,0xFFFF);
  }

i = 0;
  if(rfdec == 2) { lmx_div = 32; pps_offset = 108; }
  else if(rfdec == 3) { lmx_div = 48; pps_offset = 102; }
  else if(rfdec == 6) { lmx_div =  48; pps_offset = 121; }
  else if((rfdec == 4)&&(xbus==1)) { lmx_div = 64; pps_offset = 106; }
  else if((rfdec == 4)&&(xbus==0)) { lmx_div = 32; pps_offset = 106; }
  else if(rfdec == 8) { lmx_div = 64; pps_offset = 124; }
  else if(rfdec == 12) { lmx_div = 96; pps_offset = 120; }
  else if(rfdec == 16) { lmx_div = 64; pps_offset = 154; }
  else if(rfdec == 24) { lmx_div = 96; pps_offset = 153; }
  else if(rfdec == 48) { lmx_div = 96; pps_offset = 220; }
  else { lmx_div = 64; pps_offset = 124; }
  pps_offset = findintflagdef("TCDEL", p->config,pps_offset);

  a2dm20_lmx2592(p,cport,1e6*ad6688_inst->fs*2,lmx_div,lmx_pwr,8,0,lmx_idac,lmx_capctrl);
/* printf("JJB: rfdec = %d, pps_delay = %d, data_delay = %d\n",rfdec,(pps_offset&0x00FF),(pps_offset&0xFF00)>>8); */
  a2dm20_wr(p,cport,MTGO_ID,PFUNC_RWR,0,pps_offset,0xFFFF);

/* RESET AD6688... */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0002,0x03,0xFFFF);
  udelay(5000);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0002,0x00,0xFFFF);
  udelay(5000);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0000,0x01,0xFFFF);
  udelay(5000);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0001,0x02,0xFFFF);
  udelay(5000);
/* CLOCK DUTY CYCLE STABLIZER CIRCUITS */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x011C,0x00,0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x011E,0x00,0xFFFF);
/* Config Clock Divider */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0108,0x01,0xFFFF);
/* Config ADC Bias */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x1910,0x0F,0xFFFF);
/* Config SYSREF */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0120,0x00,0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0109,0x06,0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x010A,0x8F,0xFFFF);
/* Config Clock Delay Control */
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0110,0x06,0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0111,0x40,0xFFFF);
  a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0112,0x06,0xFFFF);
 
  cmdbuf_data = ad6688_cfg(p,cport,ad6688_inst,rfinit,dbg);

  DDC_setfreq (p, mport, ad6688_inst);
  a2dm20_set_state(p,cport,(int_4*)ad6688_inst);

  a2dm20_set_attn(p,1,rfattn);
  a2dm20_set_attn(p,2,rfattn);
  a2dm20_set_gain(p,1,rfgain);
  a2dm20_set_gain(p,2,rfgain);

  if(testmode) {
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0x0F,0xFFFF);
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,5,0xFFFF);
  }
  else {
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0550,0,0xFFFF);
    a2dm20_wr(p,cport,AD6688_ID,PFUNC_RWR,0x0327,0,0xFFFF);
  }

/* Verify CORE PLL is locked */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
  if(p->verbose==2) printf("A2DM20 CORE PLL NEEDS TO BE LOCKED: %d\n",JESD_STAT_PLLLOC(cmdbuf_data));

/* Verify Xcvr Calibration Completed */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  if(p->verbose==2) printf("A2DM20 XCVR CALIBRATION NEEDS TO COMPLETE (0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data));
  /* if(JESD_STAT_CALBSY(cmdbuf_data) != 0) return(0); */

/* Program Xcvr PLL?? */
  if(p->verbose==2) printf("A2DM20 Configure XCVR PLL\n");
  a2dm20_wr(p,cport,JESD_CFG_ID,PFUNC_PWR,0,0x0000,0xFFFF);
  xcvr_setup(p,cport,ad6688_inst);
  udelay(5000);

/* Release Xcvr Resets */
  if(p->verbose==2) printf("A2DM20 Releasing Xcvr Resets\n");
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x00,0xFFFF);

i = 0;
do {
/* Wait for rx_ready */
  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,1,&cmdbuf_data,0xFFFF);
  if(p->verbose==2) printf("A2DM20 Wait for RX READY %d: 0x%x\n",i,JESD_STAT_RXRDY(cmdbuf_data));
  ++i;
}while((JESD_STAT_RXRDY(cmdbuf_data)!=0xFF) && (i < 10));
i = 0;

/* Relese AVS Reset */
  if(p->verbose==2) printf("A2DM20 Releasing AVS Reset\n");
  a2dm20_wr(p,cport,JESD_AVS_ID,PFUNC_PWR,0,0x01,0xFFFF); 

/* Configure JESD Core */
  if(p->verbose==2) printf("A2DM20 Configuring JESD Core\n");
  jesd_setup(p,cport,ad6688_inst);

/* Release Link/Frame Resets */
  if(p->verbose==2) printf("A2DM20 Releasing Link Reset\n");
  a2dm20_wr(p,cport,MOD_ID,PFUNC_PWR,0,0x01,0xFFFF);

/* Enable SYSREF */
  if(p->verbose==2) printf("A2DM20 Enabling SYSREF\n");
  a2dm20_wr(p,cport,JESD_RST_ID,PFUNC_PWR,0,0x80,0xFFFF);

  /* Set up ADCLM */
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x01,adclmB1,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x02,adclmB2,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_0_ID,PFUNC_RWR,0x03,adclmD,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x01,adclmB1,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x02,adclmB2,0xFFFF);
  a2dm20_wr(p,cport,ADCLM_1_ID,PFUNC_RWR,0x03,adclmD,0xFFFF);

  if(p->verbose==2) {
    a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,0x011b,&cmdbuf_data,0xFFFF);
    printf("A2DM20 AD6688 Clock Status: 0x%x\n",cmdbuf_data);
    a2dm20_rd(p,cport,AD6688_ID,PFUNC_RRD,0x056F,&cmdbuf_data,0xFFFF);
    printf("A2DM20 AD6688 PLL Status: 0x%x\n",cmdbuf_data);
  }

  a2dm20_rd(p,cport,JESD_RST_ID,PFUNC_PRD,2,&cmdbuf_data,0xFFFF);
  if(p->verbose==2) printf("A2DM20 XCVR CALIBRATION STAT(0 = DONE): 0x%x\n",JESD_STAT_CALBSY(cmdbuf_data));

  return(a2dports);
}

int_4 a2dm20_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 rtn=-1;
       if (key==KEY_RFGAIN) rtn = a2dm20_get_gain (p,mport);
  else if (key==KEY_RFATTN) rtn = a2dm20_get_attn (p,mport);
  else if (key==KEY_RFPWR)  rtn = a2dm20_get_rfpwr (p,mport);
  else if (key==KEY_ADLM)   rtn = a2dm20_get_adclm (p,mport);
  else if (key==KEY_TEMP)   rtn = a2dm20_get_temp (p,mport);
  else if (key==KEY_CLKSTAT) rtn = a2dm20_get_clkstat (p,mport);
  else printf("Unhandled getkeyl=%d in a2dm20\n",key);
  return rtn;
}

real_8 a2dm20_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key)
{
  real_8 rtn=-1.0;
       if (key==KEY_RFFREQ) rtn = a2dm20_get_freq (p,mport);
  else if (key==KEY_RFBW)   rtn = a2dm20_get_adcrate (p,mport);
  else printf("Unhandled getkeyd=%d in a2dm20\n",key);
  return rtn;
}

/* already have lock because always called from pic_setkey */

int_4 a2dm20_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)
{
  int_4 rtn=0;
       if (key==KEY_RFGAIN) { a2dm20_set_gain(p,mport,i4val); }
  else if (key==KEY_RFATTN) { a2dm20_set_attn(p,mport,i4val); }
  else printf("Unhandled setkeyl=%d in lb2dm3\n",key);
  return rtn;
}

int_4 a2dm20_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)
{
  int_4 rtn=0;
       if (key==KEY_RFFREQ) { a2dm20_set_freq(p,mport,r8val); }
  else printf("Unhandled setkeyd=%d in lb2dm3\n",key);
  return rtn;
}

