/*

  Description:  Mid-level driver routines for the ICE-PIC cards
		      by Innovative Computer Engineering

  The ICE-PIC is driven by an Analog Devices 21x6x SHARC DSP and interfaced 
  to the PCI bus by a FLEX10Kxx acting as a PCI Controller Chip.  

  The ICE-PIC5+ cards use a Xilinx System-On-a-Chip (SOC) with an embedded PPC. 

  The ICE-PIC7+ cards use a Xilinx System-On-a-Chip (SOC) with a soft-core JVM.

  WARNING: Improper software mods at this level can damage chips on the board.
  Please contact the author before making changes.  For more information on the
  board's features and specifications, see the HELP files and/or the ICE web site.

  Author:  Jeff Schoen  6/96   
	   Email jeff@ice-online.com
	   Ph. 703-934-4879

  OS Support:  Linux, VMS, OSF, SGI, Solaris, Windows, and Mac

  Firmware Rev: 2.2

  Notes: Don't use the ceil() function on linux - it has problems

*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#define HOST  1
#define SHARC 0
#define ICEPM 99
#define NEWSTUFF 2

/* ICE Standard C Definitions */
#include "icedefs.h"
#include "iceconv.h"
#include "icehdr.h"
#include "packet.h"

#define DEBUG 0
#define MMFD -99
#define PGSZ 8192
#define JTAGMONITOR 0

/* special compile function cases */
#if _RDONLY
#define IOFLAGS O_RDONLY
#define MMFLAGS O_RDONLY
#define MMPROTS PROT_READ
#elif _WRONLY
#define IOFLAGS O_WRONLY
#define MMFLAGS O_RDWR
#define MMPROTS PROT_WRITE
#else
#define IOFLAGS O_RDWR
#define MMFLAGS O_RDWR
#define MMPROTS PROT_READ|PROT_WRITE
#endif

#if _LINX
#define RAW 0
#define MMAPPED 1
char* picmopenmap(int_8 base, int_8 size, char *devn);
void  picmclosemap(int_8 base, int_8 size, char *addr);
#else
#define MMAPPED 0
char* picmopenmap(int_8 base, int_8 size, char *devn) { printf("MMAP IF not supported\n"); return 0; }
void  picmclosemap(int_8 base, int_8 size, char *addr) { printf("MMAP IF not supported\n"); }
#endif

#if _darwin
#define RAW 0
#endif

#if _VMS
#include <ssdef.h>
#include <iodef.h>
#include <lckdef.h>
#include <cvtdef.h>
int iosb[2]; 
#define SYSERR(A) !(A&1)
#define SYSMSG(A) lib$signal(A)
#define _USEHOSTNAME 1
udelay (int usec) { 
  float sec = usec*1e-6; 
#if _EEEI
  int ONE=1; f_eeei2vax (&sec,&ONE);
#endif
  lib$wait (&sec);
}
double gettime() { return 0.0; }
char *getpage(int_4 size);
void freepage (char *addr, int_4 size);
#endif

#if _UNIX
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/time.h>
#include <unistd.h>
#define SYSERR(A) (A<0)
#define SYSMSG(A) perror("PERROR")
#if __cplusplus && __DECCXX
extern "C" { void usleep (int); }
#endif
double gettime() { 
  struct timeval tv; 
  gettimeofday(&tv,0); 
  return (double)tv.tv_sec + (double)tv.tv_usec*1.e-6;
} 
void udelay (int usec) {
  int udelay; struct timeval tv,tw;
  if (usec==10000) { usleep(10000); return; }	/* common loose usage */
  gettimeofday(&tv,0);
  TRYAGAIN: 
  gettimeofday(&tw,0);
  udelay = (tw.tv_sec-tv.tv_sec)*1000000 + tw.tv_usec - tv.tv_usec;
  if (usec-udelay>20000) usleep(10000);
  if (udelay < usec) goto TRYAGAIN;
}
#endif

#if _WIN
#ifdef __cplusplus
#include "StdAfx.h"
#endif
#include <windows.h>
#include <winioctl.h>
#include <fcntl.h>
#include <windef.h>
#define SYSERR(A) (A!=0&&A!=1)
#define SYSMSG(A) perror("PERROR")
void udelay (int_4 usec) { Sleep(usec/1000); }
double gettime() { return 0.0; }
#ifdef NO_INET_ADDR
#define inet_addr ice_inet_addr
int_u4 inet_addr (const char *s) { return 0; }
#endif
#define wperror(A) printf(A);printf(" errno=%d\n",GetLastError())
#define valloc malloc
#endif

#if _LINX
#define valloc malloc
#endif

#if _SOL
#include <sys/ioccom.h>
#define RAW 0
#endif

#if _SGI|_HPUX
#define RAW 0
#endif

#if _MAC
#include "macicelib.h"
#endif

/* this needs to be an ifdef because SUN defines this that way */
#ifdef _BIG_ENDIAN
#define _IEEE 1
#endif

/* ICELIB Definitions */
#include "icelib.h"

/* PCI register definitions */
#include "iceflex.h"

/* SHARC Register Definitions */
#include "icesharc.h"

/* JTAG Command definitions */
#include "icejtag.h"

/* IO Module Registers */
#include "iceioc.h"

/* PPC Defines */
#include "iceppc.h"

/* network  Defines */
#include "niolib.h"

/* fileIO  Defines */
#include "fiolib.h"

#if DEBUG
void *p_malloc (size_t size) {
  void *ptr;
  fprintf(stderr,"in  alloc %d  ",size);
  fflush(stderr);
  ptr=malloc(size);
  fprintf(stderr,"out alloc %d %p\n",size,ptr);
  fflush(stderr);
  return ptr;
}
void p_free (void *ptr) {
  fprintf(stderr,"in  free %p  ",ptr);
  fflush(stderr);
  free(ptr);
  fprintf(stderr,"out free %p\n",ptr);
  fflush(stderr);
}
#define malloc(A) p_malloc(A)
#define free(A) p_free(A)
#endif

/* Access PCI-side single 4-byte registers through a driver */
#define RD(A,B)  {io_ctl (p,IOCTL_READ,(A),&(p->X),4); B=p->X;}  
#define WR(A,B)  {p->X=B; io_ctl (p,IOCTL_WRITE,(A),&p->X,4);}

/* convert to/from card word size */
#define L2HW(A) (int_4)((A)/p->bphw)
#define HW2L(A) ((int_8)(A)*p->bphw)
#define P2HW(A) (int_4)(phys2long(A)/p->bphw)
#define HW2P(A) long2phys((int_8)(A)*p->bphw)

/* CARD DMA Transfer Control Block Definitions */
#define DMA_CADDR_ALL	(0x8000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_ALL	(0x8000)	/* 32K words each DMA channel */
#define DMA_CADDR_EXT	(0x10000+IMS)	/* Use extended memory bank only */
#define DMA_CSIZE_EXTA	(0xF000)	/* ALL=60K words each DMA channel */
#define DMA_CSIZE_EXT	(0x2000)	/* 8K words each DMA channel */
#define DMA_CSIZE_EXTV	(0x8000)	/* VHS=32K words each DMA channel */
#define DMA_CADDR_DEF	(0x8000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_DEF	(0x1000)	/* 4K words each DMA channel */
#define DMA_CADDR_MOD	(0x8000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_MOD	(0x2000)	/* 8K words each DMA channel */
#define DMA_CADDR_SER	(0xE000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_SER	(0x1000)	/* 4K words each DMA channel */
#define DMA_CADDR_GC	(0xC000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_GC	(0x1000)	/* 4K words each DMA channel */
#define DMA_CSIZE_GCI	(0x0800)	/* 2K words each DMA channel */
#define DMA_CADDR_TGC	(0xC000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_TGC	(0x2000)	/* 8K words each DMA channel */
#define DMA_CADDR_GCT	(0x10000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_GCT	(0x8000)	/* 32K words per side */
#define DMA_CADDR_INT	(0x8000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_INT	(0x1000)	/* 4K words each DMA channel */
#define DMA_CADDR_COR	(0xC000+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_COR	(0x2000)	/* 8K words each DMA channel */
#define DMA_CADDR_VHS	(0x8000+IMS)	/* Use 2nd memory bank only */
#define DMA_CADDR_VHSX	(0x5558+IMS)	/* Use 2nd memory bank only */
#define DMA_CSIZE_VHS	(0x4000)	/* 16K 48bit words each DMA channel */
#define DMA_CADDR_ISY	(0x01000000)	/* Use 16M-48M words block 16M+*/
#define DMA_CSIZE_ISY	(0x00200000)	/* 2M words (8Mby) each DMA channel */
#define DMA_CSIZE_ISYL	(0x02000000)	/* 32M words (128Mby) each DMA channel */
#define DMA_CADDR_ISYB	(0x03000000)	/* Use 48M-64M words block */
#define DMA_CSIZE_ISYB	(0x00040000)	/* 256K words (1Mby) each DMA channel */
#define DMA_CADDR_ISYC	(0x04000000)	/* Use 64M-80M words block */
#define DMA_CSIZE_ISYC	(0x00040000)	/* 256K words (1Mby) each DMA channel */
#define DMA_BLOCK_VHS	(0x1000)	/* 4K words each DMA transfer */
#define DMA_BLOCK_DEF	(0x0400)	/* 1K words each DMA transfer */
#define DMA_RELOAD_ADDR 0x40000

#define DMA_CHNS (p->isY? DMA_CHNSY:DMA_CHNSX)

int_4 tcbptab[11]    = {0x27F78,0x60,0x68,0x70,0x78,0x30,0x38,0x40,0x48,0x50,0x58};
int_4 tcbptabx[15]   = {0x47F78,0x60,0x68,0x90,0x98,0x30,0x38,0x80,0x88,0x90,0x98,0x40,0x48,0x50,0x58};
int_4 tcbptaby[17]   = {0x0, 0x14,0x16,0x18,0x1A,0x1C,0x1D,0x1E,0x1F, 0x4,0x6,0x8,0xA,0xC,0xD,0xE,0xF};
int_4 lport2dmac[7]  = { 0, 2,4,5,6,7,8 }; /* Link port to DMA channel map */
int_4 lport2dmacx[7] = { 0, 5,6,7,8,9,10 }; /* Link port to DMA channel map */
int_4 lport2dmacy[17] = { 0, 1,2,3,4,5,6,7,8, 9,10,11,12,13,14,15,16 }; /* Link port to DMA channel map */
int_4 cport2dmac[5]   = { 0, 3,4,5,6 };

int_4 mbtchiptbl[36]  = { 0, 1,2,1,2,1,2,1,2, 3,4,3,4,3,4,3,4, 5,6,5,6,5,6,5,6, 7,8,7,8,7,8,7,8, 9,10,11};
int_4 mbtchantbl[36]  = { 0, 0,0,1,1,2,2,3,3, 0,0,1,1,2,2,3,3, 0,0,1,1,2,2,3,3, 0,0,1,1,2,2,3,3, 0,0,0}; 
int_4 mbt2linktbl[7]  = { 0, 1, 2, 3, 4, 5, 6 }; 
int_4 mbt3linktbl[7]  = { 0, 5, 6, 3, 4, 1, 2 }; 
int_4 mbt4linktbl[11] = { 0, 5, 6, 5, 6, 5, 6, 5, 6, 5, 6 }; 
int_4 mbt2tadrtbl[7]  = { 0x3F00, 0x0100, 0x2000, 0x0200, 0x1000, 0x0400, 0x0800 }; 
int_4 mbt3tadrtbl[7]  = { 0x8100, 0xF700, 0xEF00, 0xFB00, 0xDF00, 0xFD00, 0xBF00 }; 
int_4 mbt4tadrtbl[17] = { 0xFF00, 0x0100, 0x0200, 0x1000, 0x2000, 0x0400, 0x0800, 0x4000, 0x8000,
				  0x5500, 0xAA00, 0xFF00, 0xCC00, 0x1100, 0x2200, 0x3300, 0xCC00 }; 

int_4 picchiptbl[4]  = { 0, 1,2, 0 };
int_4 picchantbl[4]  = { 0, 0,0, 0 }; 
int_4 piclinktbl[3]  = { 0, 5,6 }; 
int_4 pic5linktbl[5] = { 0, 3,4, 5,8 }; 
int_4 pic2tadrtbl[3] = { 0x0000, 0x0000, 0x2000 }; 
int_4 pic3tadrtbl[3] = { 0x0000, 0x0000, 0x2000 }; 
int_4 pic4tadrtbl[3] = { 0x01C0, 0x0140, 0x0180 }; 
int_4 pic5tadrtbl[4] = { 0x30000000, 0x10000000, 0x20000000, 0x30000000 }; 
int_4 pm5tadrtbl[9]  = { 0xC0000000, 0x40000000, 0x80000000, 0xC0000000, 0xC0000000,
			             0x40000000, 0x80000000, 0xC0000000, 0x80000000 }; 
int_4 coretadrtbl[9]  = { 0x00000000, 0x10000000, 0x20000000, 0x40000000, 0x80000000 };

int_4 slicchiptbl[9]  = { 0, 1,2,1,2, 1,2,1,2 }; 
int_4 slicchantbl[9]  = { 0, 0,0,1,1, 2,2,3,3 }; 
int_4 slic3chiptbl[5] = { 0, 1,1,1,1 }; 
int_4 slic3chantbl[5] = { 0, 0,1,2,3 }; 
int_4 slic3linktbl[3] = { 0, 6, 6 }; 
int_4 slic3tadrtbl[3] = { 0x0000, 0x0000, 0x0000 }; 

#define print(...) fprintf(stdout,__VA_ARGS__)
#define tprint(...) if (verify) fprintf(stderr,__VA_ARGS__)
#define vprint(...) if (p->verbose&0x1) fprintf(stderr,__VA_ARGS__)
#define v2print(...) if (p->verbose&0x2) fprintf(stderr,__VA_ARGS__)
#define v3print(...) if (p->verbose&0x4) fprintf(stderr,__VA_ARGS__)

/* FPGA Core Defines */
#define CORE_IS_ICE 1
#define CORE_IS_ICELIB 1
#define NONEXTMIDAS 1
#include "icecores.h"

/* CARD PCI Interrupt handler functions */
#define ADON_NULL	0x00		/* Add-On null */
#define ADON_WR		0x01		/* Add-On register write */
#define ADON_RD		0x02		/* Add-On register read */
#define ADON_WRM	0x03		/* Add-On register write with mask */
#define ADON_DMA	0x04		/* Add-On download DMA */
#define ADON_LIOC	0x05		/* Add-On load IOC program block */
#define ADON_DDC	0x06		/* Add-On DDC control word write */
#define ADON_MOD	0x07		/* Add-On module control word write */
#define ADON_LMOD	0x08		/* Add-On load MOD program block */
#define ADON_WRB	0x08		/* Add-On write to sys bus */
#define ADON_INIT	0x09		/* Initiate the DMA handler */
#define ADON_TC		0x0A		/* Grab the next valid Timecode */
#define ADON_QUERY	0x0B		/* Query for DMA ready */
#define ADON_NSTAT	0x0C		/* Query for MBT channel status */
#define ADON_DSTAT	0x0D		/* Query for DMA channel status */
#define ADON_WRIOC	0x0E		/* Write Control word to IOC */
#define ADON_RDIOC	0x0F		/* Read Control word from IOC */
#define ADON_WRIOB	0x10		/* Write Control word to IOB */
#define ADON_RDIOB	0x11		/* Read Control word from IOB */
#define ADON_TCPNG	0x12		/* TC Ping with delay */
#define ADON_HSTAT	0x13		/* Query for current channel status */
#define ADON_JTAG	0x14		/* Execute a JTAG command packet */
#define ADON_RBACK	0x15		/* Read Bus with ACK delay */
#define ADON_PGMC	0x15		/* Programmable clock command */
#define ADON_MPC	0x16		/* Multi-Purpose Connector Access */
#define ADON_ICAP	0x17		/* Internal Configuration Access Port */
#define ADON_SEND	0x18		/* Send Message */
#define ADON_RECV	0x19		/* Recv Message */
#define ADON_DMAX	0x1A		/* Read/Write DMA interface */
#define ADON_WMOD	0x1B		/* Write Altera IO Module */
#define ADON_RMOD	0x1C		/* Read Altera IO Module */
#define ADON_WRIOP	0x1D		/* Write Part Control word to IOC */
#define ADON_WRBLK	0x1E		/* Write block of addr/data pairs to Bus */
#define ADON_TCUP	0x1F		/* Update NIO Time Code */

#define ADON_WRJVM	0x20		/* JVM Mem Write */
#define ADON_RDJVM	0x21		/* JVM Mem Read */

#define ADON_RDBDB	0x41		/* Block Device Block Read */
#define ADON_WRBDB	0x42		/* Block Device Block Write */
#define ADON_RDBDP	0x43		/* Block Device Page Read */
#define ADON_WRBDP	0x44		/* Block Device Page Write */

#define QUERY_MSTAT	0x1		/* Query for MCFG status */
#define QUERY_NSTAT	0x2		/* Query for MBT channel status */
#define QUERY_DSTAT	0x3		/* Query for non-Master DMA status */
#define QUERY_HSTAT	0x4		/* Query for HOST DMA channel status */

#define ACMD_NOACK	0xFFFF9999	/* acmd() timeout error */

#define MTEST_AUTO   0
#define MTEST_WRITE  1
#define MTEST_READ   2
#define MTEST_BOTH   3
#define MTEST_CALIB  5
#define MTEST_CHECK  6
#define MTEST_CHECKV 7
#define MTEST_FULL   9
#define MTEST_MB2PM 11
#define MTEST_PM2MB 12
#define MTEST_MB2IO 13
#define MTEST_IO2MB 14
#define MTEST_LRS   15

#define DSG_PIC	 0
#define DSG_PAC	 1
#define DSG_SPAC 2
#define DSG_NPAC 3
#define DSG_CPAC 4
#define DSG_PICQ 5
#define DSG_TRAY 6

/*
  ADON_NSTAT status = XXXXSDDD
    where S (bit=11) is set if DMA in progress
    where DDDD is the group decimation - 1
    where XXXX is channel 15..0 allocated
*/
#define NSTAT_DEC ((nstat&0x7FFF)+1)
#define NSTAT_CHN (nstat&0xFFFF0000)
#define NSTAT_ACT ((nstat&0x8000)!=0)
#define NSTAT_MSK(x) (1<<(16+x))

/* SHARC Internal register handles */
#define SREG_MODE1	0x10000001
#define SREG_MODE2	0x10000002
#define SREG_ASTAT	0x10000003
#define SREG_STKY	0x10000004
#define SREG_IRPTL	0x10000005
#define SREG_IMASK	0x10000006
#define SREG_TPERIOD	0x10000007
#define SREG_TCOUNT	0x10000008
#define SREG_ASTATK	0x10000009

/* JVM reset addresses */
#define JVM_SYSREG	0x0008
#define JVM_RESET	0x02
#define JVM_RUN		0x01
#define JVM_LOAD	0x00

/* CrossBar module Clock resets for VHS and SSC */
#define PM_RESET_CLKUVN	0x0F82
#define PM_RESET_CLKUV	0x0F02
#define PM_RESET_CLKUSN	0x0F04
#define PM_RESET_CLKUS	0x0F84

/* Processor module reset addresses */
#define PM_RESET_CLKALL	0x0F0F
#define PM_RESET_GLOBAL	0x0F03
#define PM_RESET_CLKS	0x0F01
#define PM_RESET_CLKT	0x0F02
#define PM_RESET_CLKTV	0x0F82
#define PM_RESET_CLKP	0x0F08
#define PM_RESET_CLKPV	0x0F88
#define PM_RESET_DMAC	0x0F40
#define PM_RESET_TUN	0x0F20
#define PM_RESET_PPC	0x0F04
#define PM_RESET_PPCGO	0x0F84
#define PM_RESET_MODALL	0x0F70
#define PM_SYSOPT 	0x017B

/* Processor module testport addresses */
#define PM_TPORT_CFG 	0x0080
#define PM_TPORT_DSG 	0x0090
#define PM_TPORT_DAT  	0x0081
#define PM_TPORT_TPD  	0x0088
#define PM_TPORT_TPDX  	0x0089
#define PM_TPORT_STA  	0x008A
#define PM_TPORT_MPC  	0x008B
#define PM_TPORT_DLT  	0x0090
#define PM_TPORT_ROM  	0x00B0
#define PM_TPORT_RST  	0x00B4
#define PM_TPORT_DBG  	0x00A0

/* Processor module PPC addresses */
#define PPC_BDATA 	0x0000
#define PPC_BADDR_WR 	0x0004
#define PPC_BADDR_RD	0x000C
#define PPC_BADDR_SOCK  0x0000
#define PPC_BADDR_DSOCM 0x0000
#define PPC_BADDR_ISOCM 0x2000
#define PPC_BADDR_IBOOT 0x4000

#define JVM_BADDR_DSOCM 0x0000
#define JVM_BADDR_ISOCM 0x2010

/* getIOCsig flags */
#define IS_MUXCLK   0x100
#define IS_INTERNAL 0x200
#define IS_MUXABLE  0x300
#define IS_TEST     0x400
#define IS_INPUT    0x500
#define IS_OUTPUT   0x600
#define IS_OBMC     0x700

/* Jtag ID codes */
#define IDCODE_PF01  0x05044093
#define IDCODE_PF16  0x05058093
#define IDCODE_PF32  0x05059093
#define IDCODE_PIC5  0x01292093
#define IDCODE_PIC6  0x01EB4093
#define IDCODE_PIC7  0x03651093
#define IDCODE_PIC8  0x03822093
#define IDCODE_PIC8P 0x02cc424c
#define IDCODE_DTDM  0x0124A093
#define IDCODE_DTDMX 0x01292093
#define IDCODE_ZPPM  0x02ECE093
#define IDCODE_RFXD  0x01C22093
#define IDCODE_V5M   0x02AD6093
#define IDCODE_V6M   0x04250093
#define IDCODE_K8M   0x03822093
#define IDCODE_K8P   0x04a52093
#define IDCODE_A8M   0x02e250dd
#define IDCODE_S6M   0x04011093
#define IDCODE_S6MX  0x0401D093
#define IDCODE_SP6   0x04008093
#define IDCODE_SP6X  0x04011093
#define IDCODE_SP6Y  0x0401D093
#define IDCODE_GPS   0x04000093
#define IDCODE_INT   0x0AAAAAAA

/* V5M Variants */
#define IDCODE_V5M_LX110   0x02AD6093
#define IDCODE_V5M_LX220   0x02B06093
#define IDCODE_V5M_LX330   0x02B56093
#define IDCODE_V5M_SX240   0x02F3E093
#define IDCODE_V5M_FX100   0x032D8093
#define IDCODE_V5M_FX200   0x03334093

#define FULLSCALE   2147483648.0

#define GCNONE 0
#define GC4014 1
#define GC4016 2
#define GCFPGA 4
#define GCFDUC 8

#define GCFPGA_CLOCK 132000000
#define GCFPGB_CLOCK 166000000
#define GCFPGC_CLOCK 225000000

#define A2D_FTR 0x1000
#define IOM_CORE_WROFFSET 0x00130000
#define IOM_CORE_RDOFFSET 0x00130000
#define IOM_CORE_OFFSET   0x00070000
#define IOM_ADLM_OFFSET1  0x00000E00
#define IOM_ADLM_OFFSET2  0x00000F00

#define OIP_RCLK  0x02
#define OIP_ACLKD 0x04
#define OIP_ACLKU 0x0C
#define OIP_TRACK 0x90
#define OIP_EYESC 0x10

#define OIP_PROG  0x6F00
#define OIP_CALIB 0x6F00

#if _VMS
#define MAXICEPICS  32
#else
#define MAXICEPICS  12
#endif
#define MAXFNAME 128
#define UBREAK (*p->pbreak!=0)

#define CORE_QTFD 0x44465451
#define CORE_TFDD 0x44444654
#define CORE_MCOR 0x524F434D
#define CORE_NOOP 0x504F4F4E
#define CORE_AWG  0x00475741
#define CORE_MOD  0x00444F4D
#define CORE_DUC  0x00435544
#define CORE_UDUC 0x43554455

PICSTRUCT *psave=0;	/* Device handle for atexit() */
FILE *jtagfd=NULL;
int_u2 msg_count=0;

/* prototypes for functions w/internal linkage */
void pic_cleanup();
static int_4 findflag (char *fstr, char *astr);
static int_4 findintflag (char *fstr, char *astr);
static int_4 findintflagdef (char *fstr, char *astr, int_4 def);
static int_4 findintflagdefs (char *fstr, char *astr, int_4 def, int_4 sdef);
static real_8 finddblflag (char *fstr, char *astr);
static real_8 finddblflagdef (char *fstr, char *astr, double def);
static int_4 findstrflag (char *fstr, char *astr, char *fout, int_4 ulc);
static int_4 findstrflagx (char *fstr, char *astr, char *fout, int_4 ulc, int_4 x);
static int_4 findmaskflagdef (char *fstr, char *astr, char *list, int_4 def);
static int_4 findchoiceflagdef (char *fstr, char *astr, char *list, int_4 def);
static int_4 getbcd (char *bit, int_4 nbits, int_4 maxval, int_4 *value);
static int_4 putbcd (char *bit, int_4 nbits, int_4 value);
static void vfill (int_4 value, int_4 *array, int_4 bytes);
static int_4 ioctype (PICSTRUCT *p, char *str);
static int_4 io_ctl (PICSTRUCT *p, int_4 function, int_4 offset, int_4* buffer, int_4 bytes); 
static int_4 pic_acmd (PICSTRUCT *p, int_4 arg1, int_4 arg2, int_4 arg3, int_4 arg4);
static int_4 pic_loadvirtex (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags);
static int_4 pic_jtag (PICSTRUCT *p, int_4 port, int_4 cmd, int_4 *data, int_4 bits, int_4 delay);
static void pic_unmapcheck (PICSTRUCT *p, int_4 paddr, int_4 psize);
static int_4 pic_mapcheck (PICSTRUCT *p, int_4 paddr, int_4 psize);
static void pic_transfer (PICSTRUCT *p, int_4 *buffer, int_4 bytes, int_4 address, int_4 dir, int_4 type);
static int_4 pic_dmaqueue (PICSTRUCT *p, int_4 chan, int_4 mode);
static int_4 pic_appqueue (PICSTRUCT *p, int_4 chan, int_4 mode);
static int_4 pic_tuner_config (PICSTRUCT *p);
static int_4 pic_tuner_hsp50016 (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner_gc4014 (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner_gc4016 (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner_ad6620 (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner_gcfpga (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner_qtfpga (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static real_8 pic_tuner_freq_ (PICSTRUCT *p, real_8 fcny, int_4 port, int_4 flags);
static void pic_fload_ad6620 (PICSTRUCT *p, int_4 port, int_4 ntap, int_4 gain);
static int_4 get_ad6620_dec (int_4 dec, int_4 *val_dec2, int_4 *val_dec5, int_4 *val_deco, int_4 *val_ovsr);
static int_4 pic_sport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 sync, int_4 rate, int_4 flags);
static int_4 pic_lport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 flags);
static int_4 pic_iport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 flags);
static int_4 pic_cport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_mcport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_mport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4 pic_tport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tbport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_ubport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static int_4 pic_tuner (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
static real_8 pic_tuner_ratio (PICSTRUCT *p, int_4 port, real_8 ratio, int_4 flags);
static int_4 pic_tuner_nybw2dec (PICSTRUCT *p, int_4 port, int_4 nybw, int_4 rate, int_4 flags, real_8 *ratio);
static real_8 pic_tuner_fbwf (PICSTRUCT *p, int_4 port, real_8 fbwf, int_4 flags);
static void pic_stkupd (PICSTRUCT *p, void *data, int_4 words, int_4 dir);
static int_4 pic_fixup (PICSTRUCT *p);
static int_4 pic_cbufsz (PICSTRUCT *p, int_4 dmac);
static int_4 pic_ram (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 select, int_4 bytes, int_4 mode);
static int_4 dmass (PICSTRUCT *p, int_4 n, int_4 port, int_4 flag);
static void pic_cap (PICSTRUCT *p, int_4 set);
static int_4 pic_nv_read_b (PICSTRUCT *p, int_4 offset);
static int_4 pic_nv_read_w (PICSTRUCT *p, int_4 offset);
static int_4 pic_nv_read_l (PICSTRUCT *p, int_4 offset);
static void pic_nvread (PICSTRUCT *p);
static int_8 swaplt (int_8 data);
static int_4 swapit (int_4 data);
static int_u2 swapst (int_u2 data);
static void Swap (int_4 *buf, int_4 count, int_4 bits);
static void ConvertEndian (int_4 *buf, int_4 count, int_4 bits);
static void ConvertEEEIDouble (real_8 *buf, int_4 count);
static int_4 pagealloc (int_8 bytes, char **vaddr, int_4 mode);
static void setIOCsig (PICSTRUCT *p, int_4 flag, char *name);
static int_4 getIOCsig (PICSTRUCT *p, int_4 flag);
static int_4 getPMsig (PICSTRUCT *p, int_4 flag);
static int_4 getIOMsig (PICSTRUCT *p, int_4 flag);
static int_4 getPortSig (PICSTRUCT *p, int_4 port, int_4 flag, int_4 *sig, int_4 *ver);
static int_4 getJTAGsel (PICSTRUCT *p, int_4 port);
static int_4 getJTAGcbs (int_4 sel);
static int_4 reloadModuleCode (PICSTRUCT *p);
static void resyncIOCclocks (PICSTRUCT *p);
static int_4 setIOCframe (PICSTRUCT *p, int_4 port, int_4 bits, int_4 frame, int_4 dec, int_4 flags);
static int_4 pic_wmodreg (PICSTRUCT *p, int_4 port, int_u4 maddr, int_u4 mdata, int_4 bits);
static int_4 pic_rmodreg (PICSTRUCT *p, int_4 port, int_u4 maddr, int_u4 *mdata, int_4 bits);
static int_4 pic_wpb (PICSTRUCT *p, int_4 node, int_4 addr, int_4 data);
static int_4 pic_rpb (PICSTRUCT *p, int_4 node, int_4 addr);
static int_4 pic_setup_cxd (PICSTRUCT *p, int_4 port, int_4 dir, int_4 rate, int_4 gain);
static int_4 pic_setup_qsfp (PICSTRUCT *p, int_4 port, int_4 dir, int_4 bits, int_4 rate, int_4 flags);
static int_4 pic_test_qsfp (PICSTRUCT *p, int_4 flags);
static int_4 pic_ifup_qsfp (PICSTRUCT *p, int_4 niop, int_4 flags);
static int_4 pic_dump_qsfp (PICSTRUCT *p, int_4 flags);
static int_4 pic_tcup_qsfp (PICSTRUCT *p, int_4 dmac, int_4 stat, double offset, double delta, double soy, double fsoy);
static int_4 pic_tcupx_qsfp (PICSTRUCT *p, int_4 dmac, double soy, double fsoy);
static int_4 pic_send_qsfp (PICSTRUCT *p, int_4 isfp, int_4 ip, int_4 mode);
static int_4 qsfp_stats (PICSTRUCT *p, int_4 isfp, char *str);
static int_4 pic_setup_upac (PICSTRUCT *p, int_4 port, int_4 dir, int_4 bits, int_4 rate, int_4 flags);
static int_4 pic_setup_ioc_clk (PICSTRUCT *p, int_4 port, int_4 rate);
static int_4 pic_tc_array (PICSTRUCT *p, char *bit, real_8 soy, real_8 fsoy, int_4 flags);
static int_4 pic_detect_card (int_4 devno, int_4 flags);
static int_4 pic_detect_config (PICSTRUCT* p, char *config, int_4 flags);
static char* getCardName (int_4 type);
static char* getPmName (int_4 type);
static char* getIomName (int_4 type, int_4 rev);
static char* getPortName (int_4 index, int_4 node);
static int_4 getHost (char *name, int_4 maxlen, int_4 format);
static int_4 verify32 (PICSTRUCT *p, int_4 *buffer, int_4 nprog, int_4 address);
static int_4 verify48 (PICSTRUCT *p, int_u1 *buffer, int_4 nprog, int_4 address);
static int_4 chanfor (int_4 ic, int_4 cpc);
static int_4 test_memory (PICSTRUCT *p, int_4 port, int_4 test, int_4 pass);
static int_4 calc_port_clock_rate (PICSTRUCT *p, int_4 port);
static int_4 keyIsDouble(int_4 key);
static int_4 keyIsAscii(int_4 key);
static int_4 keyIsLong(int_4 key);
static int_4 set_iom_sequence (PICSTRUCT *p, int_4 port, int_4 mcfg, int_4 bits, int_4 rate, int_4 flags);
static int_4 set_tcgen_sequence (PICSTRUCT *p, int_4 port, int_4 count, int_4 flags);
static int_4 pic_pmstat (PICSTRUCT *p, int_4 port);
static int_4 setup_fpga_filter (PICSTRUCT *p, int_4 port, int_4 node, int_4 addrs, int_4 order, int_4 dec, int_4 taps, int_4 *coef);
static int_4 fcny2phinc (double fcny);
static int_4 setup_pkt (PICSTRUCT *p, int_4 dmac, NIC_PORT_STRUCT *np);
static int_4 dump_vrt (int_4 *lval);
static int_4 dump_sdds (int_4 *lval);
static int_4 dump_sysmon (int_4 port, int_4 *data, int_4 n);
static int_4 dump_gps (PICSTRUCT *p, int_4 side, int_4 flag, real_8 *dval);
static int_4 setupFeeds (PICSTRUCT *p, int_4 pmod, int_4 dport, int_4 dir);
static int_4 activatePM (PICSTRUCT *p, DMASTRUCT *dma, int_4 mode);
static int_4 bitsPerSample (PICSTRUCT *p, int_4 mcfg);
static int_4 calibMem (PICSTRUCT *p, int_4 port, int_4 *data, int_4 off, int_4 len);
static int_4 pic_loadlut (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags);
static int_4 pic_loadcoef (PICSTRUCT *p, int_4 port, int_4 addr, real_4 *coef, int_4 nprog, int_4 flags);
static int_4 pic_uflash (PICSTRUCT *p, int_4 *lval, int_4 len, int_4 dir);
static int_4 dmastkloc (PICSTRUCT *p, int_4 dmac);
static int_4 pic_log (PICSTRUCT *p, int_4 flag);
static int_4 dump_core (PICSTRUCT *p, int_4 port, int_4 flags);
static int_4 dump_rio (PICSTRUCT *p, int_4 port, int_4 flags);
static int_4 dump_mgt (PICSTRUCT *p, int_4 dmac);
static int_4 dump_speeds (PICSTRUCT *p, int_4 node, int_4 flags);
static int_4 pic_eyescan (PICSTRUCT *p, int_4 lanes, int_2 *buf, int_4 len);
static int_4 pic_wpbx (PICSTRUCT *p, int_4 node, int_4 addr, int_4 data);
static int_4 pic_rpbx (PICSTRUCT *p, int_4 node, int_4 addr);
static int_4 pic_drpa (PICSTRUCT *p, int_4 sel, int_4 addr);
static int_4 pic_drpw (PICSTRUCT *p, int_4 sel, int_4 data);
static int_4 pic_drpr (PICSTRUCT *p, int_4 sel);
static int_4 status_flags (PICSTRUCT *p);
static int_4 pic_enable_module (PICSTRUCT *p, int_4 mport, int_4 ena);
static int_4 pic_enable_qsfp (PICSTRUCT *p, int_4 dmac, int_4 ena);
static int_4 pic_send_vctl (PICSTRUCT *p, int_4 dmac, real_8 fill, real_8 time);
static int_4 pic_update_vctx (PICSTRUCT *p, int_4 dmac, int_4 key, real_8 dval);
static int_4 pic_nvm_stat (PICSTRUCT *p, int_4 port, int_4 test);
static int_4 pic_nio_stat (PICSTRUCT *p, int_4 port, int_4 test);
static int_4 set_awg_gain (PICSTRUCT *p, int_4 port, int_4 chan, int_4 gain);
static int_4 set_awg_mute (PICSTRUCT *p, int_4 port, int_4 chan, int_4 mute);
static int_4 set_awg_freq (PICSTRUCT *p, int_4 port, int_4 chan, double freq);
static int_4 setup_duc (PICSTRUCT *p, int_4 port, int_4 ic, int_4 rate, double ratio, int_4 gain, double fcny, int_4 sig);
static int_4 core_setup (PICSTRUCT *p, int_4 port, char *ccfg, int_4 csig);
static int_4 core_startup (PICSTRUCT *p, int_4 port, double time);
static int_4 setup_fftcore (PICSTRUCT *p, int_4 port, char *ccfg, int_4 csig, int_4 dmac, int_4 bits, int_4 rate, int_4 dec, int_4 gain);
static int_4 setTpMpc (PICSTRUCT *p, int_4 eval);
static int_4 setup_SI514 (PICSTRUCT *p, double freq);

/* RFXD Module Methods */
static int_4 rfxd_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 rate, int_4 gain);
static int_4 rfxd_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq);
static int_4 rfxd_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain);
static int_4 rfxd_set_bw (PICSTRUCT *p, int_4 mport, int_4 bw);
static int_4 rfxd_set_agctc (PICSTRUCT *p, int_4 mport, real_8 tcval);
static int_4 rfxd_set_agccf (PICSTRUCT *p, int_4 mport, real_8 cfval);
static int_4 rfxd_set_opts (PICSTRUCT *p, int_4 mport, int_4 flags);
static int_4 rfxd_set_vio( PICSTRUCT *p, int_4 port, int_4 address, int_4 data, int_4 len);
static int_4 rfxd_get_opts (PICSTRUCT *p, int_4 mport);
static int_4 rfxd_get_gain  (PICSTRUCT *p, int_4 mport);              
static int_4 rfxd_get_bw    (PICSTRUCT *p, int_4 mport);              
static int_4 rfxd_get_rfpwr (PICSTRUCT *p, int_4 mport);
static real_8 rfxd_get_freq (PICSTRUCT *p, int_4 mport);

/* LB2D Module Methods */
static int_4 lb2d_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4 lb2d_set_opts (PICSTRUCT *p, int_4 mport, int_4 flags); 
static int_4 lb2d_set_freq  (PICSTRUCT *p, int_4 mport, real_8 freq); 
static int_4 lb2d_set_attn  (PICSTRUCT *p, int_4 mport, int_4 attn);  
static int_4 lb2d_set_gain  (PICSTRUCT *p, int_4 mport, int_4 gain);  
static int_4 lb2d_set_bw    (PICSTRUCT *p, int_4 mport, int_4 bw);    
static int_4 lb2d_set_ais   (PICSTRUCT *p, int_4 mport, int_4 mode, real_8 *params);
static int_4 lb2d_get_gain  (PICSTRUCT *p, int_4 mport);              
static int_4 lb2d_get_bw    (PICSTRUCT *p, int_4 mport);              
static int_4 lb2d_get_attn  (PICSTRUCT *p, int_4 mport);              
static int_4 lb2d_get_opts  (PICSTRUCT *p, int_4 mport);              
static int_4 lb2d_get_rfpwr (PICSTRUCT *p, int_4 mport);         
static real_8 lb2d_get_freq (PICSTRUCT *p, int_4 mport);

/* LB2DM3 Module Methods */
static int_4  lb2dm3_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4  lb2dm3_enable (PICSTRUCT *p, int_4 mport, int_4 ena);
static int_4  lb2dm3_reset (PICSTRUCT *p, int_4 mport);
static int_4  lb2dm3_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key);
static real_8 lb2dm3_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key);
static int_4  lb2dm3_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val);
static int_4  lb2dm3_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val);
#if NEWSTUFF
static int_4  lb2dm3_fplan (PICSTRUCT *p, int_4 mport, real_8 *r8val, int_4 len);
#endif

/* D2AWGM3 Module Methods */
static int_4  d2awgm3_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4  d2awgm3_enable (PICSTRUCT *p, int_4 mport, int_4 ena);
static int_4  d2awgm3_reset (PICSTRUCT *p, int_4 mport);
static int_4  d2awgm3_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key);
static real_8 d2awgm3_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key);
static int_4  d2awgm3_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val);
static int_4  d2awgm3_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val);

/* A2DM14 Module Methods */
static int_4 a2dm14_setup (PICSTRUCT *p, int_4 mport, int_4 flags);
static int_4 a2dm14_setkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data);
static int_4 a2dm14_getkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data);

/* A2DM1X Module Methods */
static int_4 a2dm1x_reset (PICSTRUCT *p);
static int_4 a2dm1x_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, real_8 rate, int_4 gain, int_4 flags);
static int_4 a2dm1x_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain);
static int_4 a2dm1x_get_gain (PICSTRUCT *p, int_4 mport);              
static int_4 a2dm1x_set_freq (PICSTRUCT *p, int_4 mport, double freq);
static real_8 a2dm1x_get_freq (PICSTRUCT *p, int_4 mport);
static real_8 a2dm1x_get_rate (PICSTRUCT *p, int_4 mport);
static int_4 a2dm1x_get_adclm (PICSTRUCT *p, int_4 mport);
static int_4 a2dm1x_get_temp (PICSTRUCT *p, int_4 mport);

/* A2DM20 Module Methods */
static int_4 a2dm20_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4 a2dm20_enable (PICSTRUCT *p, int_4 mport, int_4 ena);
static int_4  a2dm20_getkeyl (PICSTRUCT *p, int_4 mport, int_4 key);
static real_8 a2dm20_getkeyd (PICSTRUCT *p, int_4 mport, int_4 key);
static int_4  a2dm20_setkeyl (PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val);
static int_4  a2dm20_setkeyd (PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val);

/* D2RF Module Methods */
static int_4 d2rf_reset (PICSTRUCT *p, int_4 mport);
static int_4 d2rf_setup (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags);
static int_4 d2rf_set_attn (PICSTRUCT *p, int_4 mport, int_4 attn);
static int_4 d2rf_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq);
static int_4 d2rf_set_bw (PICSTRUCT *p, int_4 mport, int_4 bw);
static int_4 d2rf_get_temp (PICSTRUCT *p, int_4 mport);
static int_4 d2rf_get_attn (PICSTRUCT *p, int_4 mport);
static int_4 d2rf_get_dbg (PICSTRUCT *p, int_4 mport);
static int_4 d2rf_get_clkstat (PICSTRUCT *p, int_4 mport);
static int_4 d2rf_set_opts (PICSTRUCT *p, int_4 mport, int_4 flags);
static int_4 d2rf_get_opts (PICSTRUCT *p, int_4 mport);
static real_8 d2rf_get_freq (PICSTRUCT *p, int_4 mport);

/* deprecation count */
static int_4 depcnt=0;

int_4 pic_badmagic(PICSTRUCT *p) {
  if (++depcnt<8) {
    fprintf(stderr,"PICStruct bad magic - possible memory corruption: ");
    fflush(stderr);
    fprintf(stderr,"value=%08x cnt=%d\n",p->magic,depcnt);
  }
  return -1;
}
#define CHECK_MAGIC(P) if ((P)->magic!=ICE_MAGIC) return pic_badmagic(P);

/* Rumel library routines */
#include "rumlib.h"

/* Generic IO Module  library routines */
#include "iomlib.h"

/* Time functions */
char time_str[80];
char *pic_time_str() {
  time_t t = time(NULL);
  struct tm *tmp = localtime(&t);
  strftime(time_str,80,"%Y:%m:%d::%H:%M:%S",tmp);
  return time_str;
}

double pic_time() {
  return gettime() + 631152000.0;  /* Jan-1-1970 to Midas time Jan-1-1950 */
}

int_4 pic_ioctl (PICSTRUCT *p, int_4 function, int_4 offset, int_4* buffer, int_4 bytes) {
  return io_ctl(p,function,offset,buffer,bytes);
}

char *toLower (char *s) {
  int i;
  for (i=0; i<strlen(s); i++) {
    if (s[i]>='A' && s[i]<='Z') s[i]+=32;
  }
  return s;
}

/* C helper code - these are actual functions in some math libs */
#define log2 log2_
#define powi powi_
static double log2(double value);
static double powi(int_4 base, int_4 power);
#include "icemath.h"
#define min imin
#define max imax


/* fast 16 bit count/store fixup 0x0000BBAA where
  AA is captured at sample 12*8+0
  BB is captured at sample 13*8+0
*/
int_4 fixcounto (int_4 count) {
  if ( (count&0xFF) == 0xFF) count -= 0x0100;
  count = count<<3;
  count = count - 13*8; 
  return count;
}

/* fast 28 bit count/store fixup 0xDDCCBBA0 (single clutch)
  A  is captured at sample 12*8-2
  BB is captured at sample 13*8+0  
  CC is captured at sample 14*8+0
  DD is captured at sample 15*8+0 but gets clocked at sample 15*8+2 
  delays are ibits=2 barker=7 thit1=1 thit=1 tcount=1 tload=1 tdat=-1
  we are 12 samples in by the first tload all samples moved to DD
*/
int_4 gencount1 (int_4 i) {
  i+=12;
  return (((i+12*8-2)<<4)&0xFF) | (((i+13*8+0)<<4)&0xFF00) 
	| (((i+14*8+0)<<4)&0xFF0000) | (((i+15*8-2)<<4)&0xFF000000);
}
int_4 fixcount1 (int_4 count) {
  int_4 late;
  count = (count>>4)&0x0FFFFFFF;	
  late = (count&0x6) == 0x6;
  if ((count&0xF)>=6) count-=6; else count+=10;
  if ((count&0xFF8)==0xFF8) count-=0xFF8; else count+=8;
  if ((count&0xFFFF8)==0xFFFF8 && !late) count-=0xFFFF8; else count+=8;
  count = (count-12-(15*8))&0x0FFFFFFF;
  return count;
}

/* fast 28 bit count/store fixup 0xDDCCBBA0 (double clutch)
  A  is captured at sample 12*16-2
  BB is captured at sample 13*16+0  
  CC is captured at sample 14*16+0
  DD is captured at sample 15*16+0 but gets clocked at sample 15*16+2 
  delays are ibits=2 barker=14 thit2=1 thit=1 tcount=1 tload=1 tdat=-1
  we are 19 samples in by the first tload all samples moved to DD
*/
int_4 gencount2 (int_4 i) {
  i+=19;
  return (((i+12*16-2)<<4)&0xFF) | (((i+13*16+0)<<4)&0xFF00) 
	| (((i+14*16+0)<<4)&0xFF0000) | (((i+15*16-2)<<4)&0xFF000000);
}
int_4 fixcount2 (int_4 count) {
  int_4 late;
  count = (count>>4)&0x0FFFFFFF;	
  late = (count&0xE) == 0xE;
  if ((count&0xF)>=14) count-=14; else count+=2;
  if ((count&0xFF0)==0xFF0) count-=0xFF0; else count+=16;
  if ((count&0xFFFF0)==0xFFFF0 && !late) count-=0xFFFF0; else count+=16;
  count = (count-19-(15*16))&0x0FFFFFFF;
  return count;
}

/* Fix up the TimeCode count register
   dc = 2 double clutch
   dc = 1 single clutch
   dc = 0 startup calibration
   dc < 0 startup calibration with tuner sync
*/

int_4 fixcount (PICSTRUCT *p, int_4 port, int_4 count, int_4 dc) {
  if (p->tcmode==TCM_SDD);
  else if (p->isY) {
    if (dc==1) count-=127;
    if (dc==2) count-=220;
  }
  else if (p->type==ICEPIC2) count = fixcounto(count);
  else if (p->type==ICEMBT2) count = fixcounto(count);
  else {
    if (dc< 0) count = fixcount1(count)+10-4;
    if (dc==0) count = fixcount1(count)+10-3;
    if (dc==1) count = fixcount1(count);
    if (dc==2) count = fixcount2(count);
  }
  return count;
}

/*
  bitreverse and byte swap words for JTAG big-endian delivery
*/

static int_u1 bitrev[256];

void prepBitRev() {
  int_4 i,j;
  for (j=i=0; i<256; i++) { j =
        ((i&0x1)<<7) | ((i&0x2)<<5) | ((i&0x4)<<3) | ((i&0x8)<<1) |
        ((i&0x10)>>1) | ((i&0x20)>>3) | ((i&0x40)>>5) | ((i&0x80)>>7);
    bitrev[i] = (int_u1)(j&0xFF);
  }
}
void prepBitStream (int_u1 *buf, int_4 nbuf) {
  int_4 i; int_u1 tmp;
  prepBitRev();
  for (i=0; i<nbuf; i+=4) {
    tmp=bitrev[buf[i+0]]; buf[i+0]=bitrev[buf[i+3]]; buf[i+3]=tmp;
    tmp=bitrev[buf[i+1]]; buf[i+1]=bitrev[buf[i+2]]; buf[i+2]=tmp;
  }
}
int_4 bitrev4 (int_4 datai) {
  int_4 datao=0; 
  int_u1 *bi=(int_u1*)&datai, *bo=(int_u1*)&datao;
  bo[0] = bitrev[bi[3]];
  bo[1] = bitrev[bi[2]];
  bo[2] = bitrev[bi[1]];
  bo[3] = bitrev[bi[0]];
  return datao;
}
int_4 bitrev2 (int_4 datai) {
  int_4 datao=0; 
  int_u1 *bi=(int_u1*)&datai, *bo=(int_u1*)&datao;
  bo[0] = bitrev[bi[1]];
  bo[1] = bitrev[bi[0]];
  return datao;
}
int_4 bitrev1 (int_4 data) {
  int_u1 *tmp = (int_u1*)(&data);
  *tmp = bitrev[*tmp];
  return data;
}
int_4 bitrevN (int_4 data, int_4 n) {
  int_u1 *tmp = (int_u1*)(&data);
  for (; n>0; n--,tmp++) *tmp = bitrev[*tmp];
  return data;
}
/* 
  crc = x16 + x15 + x2 + 1
  new = data^cyc[15]
  crc[0] = new
  crc[2] = crc[1]^new
  crc[15] = crc[14]^new
*/
int_4 CRC16 (int_4 crc, int_4 data, int_4 bits) {
  int_4 i,val;
  for (i=0; i<bits; i++) {
    val = ((data>>i)^(crc>>15))&0x1;
    crc = ((crc<<1)&0xFFFE) | val;
    if (val) crc ^= 0x8004;
  }
  return crc;
}
int_4 CRC32 (int_4 crc, int_4 data, int_4 bits) {
  int_4 i,val;
  for (i=0; i<bits; i++) {
    val = ((data>>i)^(crc>>31))&0x1;
    crc = ((crc<<1)&0xFFFFFFFE) | val;
    if (val) crc ^= 0xEDB88320;
  }
  return crc;
}

int_4 ioc_wr (PICSTRUCT *p, int_4 addr, int_4 data) {
  if (!p->isY) return 0;
  pic_lock (p, LOCK_ALLOC);
  WR(REG_IOC, REG_IOC_ADDR|addr);
  WR(REG_IOC, data&0xFF);
  pic_lock (p, LOCK_FREE);
  return data;
}
int_4 ioc_rd (PICSTRUCT *p, int_4 addr) {
  int_4 data;
  if (!p->isY) return 0;
  pic_lock (p, LOCK_ALLOC);
  WR(REG_IOC, REG_IOC_ADDR|addr);
  RD(REG_IOC, data); data &= 0xFF;
  pic_lock (p, LOCK_FREE);
  return data;
}

int_4 pic_relock (PICSTRUCT *p, int_4 delayms) {
  int_4 lock = p->lcount;
  if (lock==1) pic_lock(p,LOCK_FREE);
  if (delayms>0) udelay(delayms*1000);
  if (lock==1) pic_lock(p,LOCK_ALLOC);
  return 0;
} 

/* Transceiver routines */
#include "xcvrlib.c"

#include "pic9funcs.c"

int_4 core_port (int_4 core) {
  int_4 i = (core==4)? PM_TUNB: (core==3)? PM_TUNA: (core==2)? PM_CORB: PM_CORA;
  return i;
}
int_4 core_addr (int_4 core) {
  int_4 i = (core==5)? PPC_TUNAB_CTL : (core==4)? PPC_TUNB_CTL : (core==3)? PPC_TUNA_CTL : (core==2)? PPC_CORB_CTL : PPC_CORA_CTL;
  return i;
}

/* Is PM Xilinx */
int_4 isPMx (PICSTRUCT *p, int_4 index)
{
  int_4 i=(index==2)?1:0; 
  int_4 status=1;
  if (p->pmtype[i]==PMT_A8M) status=0;
  return status;
}

/* Is IOM Xilinx */
int_4 isIOMx (PICSTRUCT *p, int_4 index)
{
  int_4 i=(index==2)?1:0; 
  int_4 status=0;
  if (index==9) return 1;	/* GPS on Tray */
  if (p->mtype[i]==IOMT_GPS) status=1;
  if (p->mtype[i]==IOMT_RFXD) status=1;
  if (p->mtype[i]==IOMT_D2RF) status=1;
  if (p->mtype[i]==IOMT_RF2D) status=1;
  if (p->mtype[i]==IOMT_LB2D && p->mrev[i]>=3) status=1;
  if (p->mtype[i]==IOMT_D2AWG && p->mrev[i]==3) status=1;
  if (p->mtype[i]==IOMT_A2D && p->mrev[i]>=14 && p->mrev[i]<=18) status=1;
  if (p->mtype[i]==IOMT_D2A && p->mrev[i]>=14 && p->mrev[i]<=18) status=1;
  return status;
}

/* Does IOM have a temperature sensor */
int_4 isIOMts (PICSTRUCT *p, int_4 index)
{
  int_4 i=(index==2)?1:0; 
  int_4 status=0;
  if (p->mtype[i]==IOMT_A2D && p->mrev[i]>=14) status=1;
  if (p->mtype[i]==IOMT_SNTXD && p->mrev[i]>=6) status=1;
  if (p->mtype[i]==IOMT_D2AWG && p->mrev[i]>=3) status=1;
  return status;
}

/* Is IOM bi-directional */
int_4 isBiDir (int_4 mtypeo) {
  return (mtypeo==IOMT_DXSDDS) || (mtypeo==IOMT_DXTGSDDS) || (mtypeo==IOMT_DXMSAS) ||
         (mtypeo==IOMT_DXSNT) || (mtypeo==IOMT_DXNF) || (mtypeo==IOMT_DXFLZR) || (mtypeo==IOMT_GPS) ||
         (mtypeo==IOMT_DXSFP) || (mtypeo==IOMT_DXDSFP) || (mtypeo==IOMT_DXSFPP) || (mtypeo==IOMT_DXUFLZ) ||
         (mtypeo==IOMT_DXTGVITA) || (mtypeo==IOMT_D2RF) || (mtypeo==IOMT_D2A) || (mtypeo==IOMT_DXQSFP) || (mtypeo==IOMT_DXUPAC) ;
}

int_4 getpath (char *path, char *area) {
#if _VMS
#ifdef _XMIDAS
  sprintf(path,"XMAREA_ICE:[%s]",area);
#else
  sprintf(path,"ICEROOT:[%s]",area);
#endif
#elif _WIN
  char *str;
  str = getenv("ICEROOT");
  if (str == NULL) print("Environment variable ICEROOT not defined\n");
  sprintf(path,"%s\\%s\\",str,area);
#else
  char *str;
#ifdef _XMIDAS
  str = getenv("XMAREA_ICE");
  if (str == NULL) print("Environment variable XMAREA_ICE not defined\n");
#else
  str = getenv("ICEROOT");
  if (str == NULL) print("Environment variable ICEROOT not defined\n");
#endif
  sprintf(path,"%s/%s/",str,area);
#endif
  return strlen(path);
}

/* CODE */
int_u4 getGateway(PICSTRUCT *p,int isfp,int_u4 from,int_u4 to);

int_4 pic_open (PICSTRUCT *p, const char *cstr, int_4 *pbreak, int_4 flags) 
{
  int_4 i, j, k, lg, ip, ls, fdn, status, type, rev, nosig, internal, csig,lsig,ltmp,isRR, npm=0; 
  char *s, *str, devname[MAXFNAME], linkname[MAXFNAME], tmpstr[32], c,cl=0,cn, minor, globalflags[256];
  VAXstr devdsc; FILE *file; HALO *halo=p->halo;

  /* init the PICSTRUCT handle */
  vfill (0, (int_4 *)p, sizeof(PICSTRUCT)); 
  p->magic = ICE_MAGIC;
  p->fbwf = -1;
  strcpy(p->cname,"None");
  prepBitRev();

getGateway(p,2,0,0);

  /* construct flags */
  getpath(linkname,"cfg"); strcat(linkname,"global_flags.cfg"); file = fopen(linkname,"r");
  if (file != 0) { 
    lg=fread ((void *)globalflags, 1, 256, file);
    fclose(file);
    for (;lg>1 && globalflags[lg-1]<' ';lg--);
  } else lg=0;

  str = (char *)malloc(strlen(cstr)+lg+3+5);
  p->config = str;
  p->state = 0;
  strcpy(str,cstr);
  strcat(str,",");
  if (lg>0) { strncat(str,globalflags,lg); strcat(str,","); }
  ls = strlen(str);
  for (i=j=k=ip=0; i<ls; i++) {
    c = str[i]; cn=str[i+1];
    if (c>=8 && c<=13 && j==0) c=str[i]=',';	/* replace TABS */
    if (j==0 && islower(c)) str[i] -= 32;  /* upcase if not filename or ascii value */
    if (c=='=') { k++; if (cn=='/'||isalpha(cn)) j++; } /* start of flag value : filename */
    if (c=='{') c=str[i]='('; if (c=='}') c=str[i]=')';
    if (c=='(') { if (cl=='='||ip>0) ip++; else c=str[i]=','; }
    if (c==')') { if (ip>0) ip--; else c=str[i]=','; }
    if (c==',' || c==')' || c=='|' || c=='(') j=k=0; /* end of flag : filename */
    if (k>1) { print("Malformed config string=[%s]\n",str); return -1; }
    cl=c;
  }

  if (findintflag("TEST",str) > 0) 	flags |= FLG_TEST;
  if (findintflag("NOBRK",str) > 0) 	flags |= FLG_NOBRK;
  if (findintflag("BOOT",str) > 0) 	flags |= FLG_BOOT;
  if (findintflag("NOLOCK",str) > 0) 	flags |= FLG_NOLOCK;

  /* special case for MUXCLK=PX */
  j = findflag("MUXCLK=PX",str);
  if (j>=0) {
    for (i=strlen(str);i>j+7;i--) str[i+5]=str[i];
    str[++i]='|'; str[++i]='P'; str[++i]='R'; str[++i]='E'; str[++i]='F';
  }

  /* enable verbose status messages for debugging ? */
  nosig = findintflag("NOSIG",str);
  p->verbose = findintflag("VERBOSE",str);
  if (p->verbose > 0);
  else if (flags&FLG_VERBOSE) p->verbose=1; 
  else p->verbose=0;
  p->side  = findintflagdef("SIDE",p->config,0);

  csig=CORE_TFDD;
  if (findstrflag("TSIG",p->config,tmpstr,1)>0) {
         if (strcmp(tmpstr,"TFDD")==0) csig=CORE_TFDD;
    else if (strcmp(tmpstr,"MOD")==0)  csig=CORE_MOD;
    else if (strcmp(tmpstr,"DUC")==0)  csig=CORE_DUC;
    else if (strcmp(tmpstr,"UDUC")==0) csig=CORE_UDUC;
    else printf("Unsupported tuner core signature flag TSIG=%s\n",tmpstr);
  }
  p->tsig  = csig;
  p->isDUC = (csig==CORE_DUC || csig==CORE_UDUC || csig==CORE_AWG || csig==CORE_MOD);

  vprint("Open: %s \n",p->config);

  /* check for proper node */
  j = findflag("NODE",str);
  if (j>=0) {
    s = str+j+5;
    /* if there is a DOT in the nodename - match full domain name */
    for (i=0,fdn=0; s[i]!=','; i++) if (s[i]=='.') fdn=1;
    j = getHost(linkname,MAXFNAME,fdn);
    if (i!=j || strncmp(s,linkname,i) != 0) {
      print("Current host name %s does not match config NODE=%.*s\n",linkname,i,s);
      return (-1);
    }
  }
  p->base = 0;
  p->fd = 0;
  p->devno = findintflag("DEVNO",str);
  p->nicno = findintflag("NIC",str);
  p->ouid  = findintflag("OUID",str);
  p->flags = flags;
  jtagfd = NULL;
  devname[0]=0;
  if (p->devno==99) return 1;

  /* case sensitive cstr may be necessary for device name */
  if (p->devno<0 && (s=(char *)strstr(cstr,"PCI-DEV,"))!=NULL) {
    s += 8;
    for (i=0; s[i]!=','; i++) devname[i]=s[i]; devname[i]=0;
#if _VMS
    devname[--i]=0;	/* remove colon */
    p->devno = (int_4)devname[strlen(devname)-2] - 65;
#elif _UNIX
    /* resolve 1-level symbolic link if in /dev */
    i = readlink(devname,linkname,MAXFNAME); 
    if (i>0 && strncmp(linkname,"/dev/",5)==0) 
	{ strncpy(devname,linkname,i); devname[i]=0; }
    p->devno = (int_4)devname[strlen(devname)-1] - 48;
    if (devname[strlen(devname)-2]=='1') p->devno+=10;
#else
    p->devno = (int_4)devname[strlen(devname)-1] - 48;
    if (devname[strlen(devname)-2]=='1') p->devno+=10;
#endif
  }
  if (p->devno<0 && (s=(char *)strstr(str,"NVM-MEM,"))!=NULL) {
    s += 10;
    sscanf (s,"%llx",&p->base);
    p->devno = -2;
    p->type = ICENVM;
  }
  if (p->devno<0 && (s=(char *)strstr(str,"PCI-MEM,"))!=NULL) {
    s += 10;
    sscanf (s,"%llx",&p->base);
    p->devno = -2;
  }
  if (p->devno<0 && (s=(char *)strstr(str,"VHS-CORE,"))!=NULL) {
    p->halo = halo; p->pindex = 1; 
    p->devno = -3; nosig = (ICEPM<<4);
  }
#if _VMS
  if (p->devno >= 0) {
    if (devname[0]==0) sprintf(devname,"IC%c0",p->devno+65);
    devdsc.dtype = DSC$K_DTYPE_T;
    devdsc.dclass = DSC$K_CLASS_S;
    devdsc.c     = devname;
    devdsc.length = strlen(devname);
    /* open a channel to the device through PICDRV driver */
    status = sys$assign (&devdsc, &p->fd, 0, 0);
    if (SYSERR(status)) { if (flags!=FLG_TEST) SYSMSG(status); return(-1); } 
  }
#elif _OSF
  if (p->devno >= 0) {
    for (p->fd=0,minor='a'; p->fd<=0 && minor!='O'; minor++) {
      if (minor=='{') minor='A';   /* abc...xyz{ then ABC...LMNO */
      sprintf(devname,"/dev/ice/pic%d%c",p->devno,minor);
      p->fd = open (devname, O_RDWR|RAW, 0);
    }
    if (p->fd<=0) { if (flags!=FLG_TEST) perror("PIC Open"); return(-1); }
    fcntl (p->fd, F_SETFD, FD_CLOEXEC);
  }
#elif _UNIX
  if (findflag("ICERAM",str)>=0 || findflag("ICENIC",str)>=0 || p->nicno>=0) {
    strcpy(devname,"/dev/iceram");
    if (p->devno>=0) sprintf(devname,"/dev/iceram%d",p->devno);
    if (findflag("ICENIC",str)>=0) { p->nicno==p->devno; p->devno=-1; }
    p->fd = open (devname, IOFLAGS|RAW, 0);
    if (p->fd<=0) { perror("ICERAM Open");  if (!(flags&FLG_TEST)) return(-1); }
    flags |= FLG_NOLOCK;
    p->type = ICENIC;
    p->bphw = 4;
    goto NOPIC;
  }
  if (findflag("ICEUIC",str)>=0) { 		/* check for usb interface */
    j=findstrflag("USB",str,devname,0);
    if (j<0) sprintf(devname,"/dev/ttyACM%d",p->devno);
    else p->devno = devname[strlen(devname)-1]-'0';
    jtagfd = fopen(devname,"w+");
    vprint("Open USB device=%s ptr=%p\n",devname,jtagfd);
    if (jtagfd!=NULL && findflag("IOM3",p->config)>0) {	/* point ard to jmod=3 */
      fflush(jtagfd); fwrite ("jmod=3", 1, 7, jtagfd); fflush(jtagfd); udelay(10000);
      if (fgets(tmpstr,32,jtagfd)==NULL || strstr(tmpstr,"CMD=")==NULL 
       || fgets(tmpstr,32,jtagfd)==NULL || strstr(tmpstr,"Set JTAG")==NULL
       || fgets(tmpstr,32,jtagfd)==NULL || strstr(tmpstr,"endCMD")==NULL) 
	{ fflush(jtagfd); perror("setting Arduino jmod=3"); fclose(jtagfd); return(-1); }
       flags |= FLG_NOLOCK;
    }
    strcpy(linkname,"/dev/iceram");
    /* if (p->devno>=0) sprintf(linkname,"/dev/iceram%d",p->devno); */
    p->fd = open (linkname, IOFLAGS|RAW, 0);
    if (p->fd<=0) { perror("ICERAM Open");  if (!(flags&FLG_TEST)) return(-1); }
    p->type = ICEPIC8; p->k7 = 8;
  }
  else if (p->devno >= 0) {
    if (devname[0]==0) sprintf(devname,"/dev/icepic%d",p->devno);
    p->fd = open (devname, IOFLAGS|RAW, 0);
    if (p->fd<=0) { if (!(flags&FLG_TEST)) perror("PIC Open"); return(-1); }
    fcntl (p->fd, F_SETFD, FD_CLOEXEC);
  }
  else if (p->devno == -2) {
    p->fd = MMFD;
    p->loc = (int_4*)picmopenmap(p->base,PGSZ,"mem");
    p->devno = 0;
    vprint("Opened addr=%08llx\n",p->base);
    for (i=0; i<4; i++) vprint("Opened adr=%d data=%08x\n",i<<2,p->loc[i]);
  }
  else if (p->devno == -3) {
  }
#elif _WIN
  if (p->devno >= 0) {
    sprintf(devname,"\\\\.\\Icepic%d",p->devno);
    p->fd = CreateFile (devname,0,0,NULL,OPEN_EXISTING,0,NULL);
    if (p->fd == (void *)-1) { if (!(flags&FLG_TEST)) perror("PIC Open"); return(-1); }
  }
#elif _MAC
  if (p->devno == -2) {
    p->base = geticepicbadr((LogicalAddress)REG_PBAR);
    p->fd = MMFD;
  }
#endif
  else {
    print("Invalid DevIce string: %s\n",str);
    return (-1);
  }

#if _LINX
  /* signal (SIGQUIT,pic_cleanup); */
#elif _UNIX
  if (atexit(pic_cleanup) != 0) perror ("atexit");
#endif

  /* validate device number */
  if (p->devno==-1 || p->devno>MAXICEPICS) {
    print ("Card device number=%d indicates too many cards\n",p->devno);
    print ("Check configuration ... \n");
    return -1;
  }
  if (p->type == ICENVM) goto NOPIC;

  if (nosig==1) {
    p->state=2;
    p->flags = flags|FLG_NOLOCK;
    pic_lock (p, LOCK_OPEN);
    return(1);
  }

  /* validate the ICE card revision */
  rev = (nosig>0)? nosig : pic_nv_read_b (p,0x08); 
  i=(rev>>4); 
  if (i==ICEPM);
  else if (i<ICEPIC2 || i>ICESLIC3) {
    i=rev; rev = pic_nv_read_b (p,0x08); 
    i=(rev>>4); 
  }
  switch (i) {
  case ICEPIC3: case ICEPIC4: case ICEPIC5: case ICEPIC6: case ICEPIC7: case ICEPIC8: case ICEPIC9:
  case ICEMBT4: case ICESLIC3: case ICEPM:
    p->type = i; break;
  default:
    if ((flags&FLG_TEST)) rev=0x8F;	/* assume PM on sPAC */
    else print ("Bad card revision rev=%x type=%d \n",rev,i);
    if (!(flags&FLG_TEST)) return -1;
  }
  p->isX = (p->type==ICEPIC4||p->type==ICEMBT4);
  p->isY = (p->type==ICEPIC5||p->type==ICEPIC6||p->type==ICEPIC7||p->type==ICEPIC8||p->type==ICEPIC9);
  p->k7  = (p->type==ICEPIC7)? 7 : (p->type==ICEPIC8)? 8 : (p->type==ICEPIC9)? 9 :0;
  if (p->isY) p->ndmac=12; 
  else if (p->isX) p->ndmac=10; 
  else p->ndmac=8;
  p->rev = rev&0x0F;

  /* see if we should use 64 bit or 66 MHz interface */
  if (nosig>0) status=0x0; else RD (REG_MCSR,status);
  if (p->type==ICEPIC6||p->type==ICEPIC7||p->type==ICEPIC8) {
    p->pbr = ((status&MCSR_GENX)==MCSR_GEN3)? 1000 : ((status&MCSR_GENX)==MCSR_GEN2)? 500 : 250;
    if (p->type==ICEPIC6) p->pbr = 250;
    p->pbw = (status&MCSR_CLK66)? 2 : 1; 
    if (status&MCSR_BUS64) p->pbw *= 4;
    if (p->type==ICEPIC8 && p->rev==5 && p->pbw==2) p->pbw=16;
  } else {
    p->pbw = (status&MCSR_BUS64)?  8 : 4;
    p->pbr = (status&MCSR_CLK66)? 66 : 33;
  }
  
  /* override the 64b autodetect */
  if (findintflag("B32",str) > 0) p->pbw = 4;

  p->bphw = p->isY? 64 : 4;		/* bytes per HOST word */
  p->bpcl = (p->k7>=8)? 32 : 8;		/* bytes per CARD DMA line */
  p->bpp  = (p->k7>=8)? 256 : 64;	/* bytes per DMA packet */

  /* pick CARD memory start address */
       if (p->isY) { p->ims=0x00020000; p->ems=0x00400000; }
  else if (p->isX) { p->ims=0x00040000; p->ems=0x00800000; }
  else             { p->ims=0x00020000; p->ems=0x00400000; }

  /* Determine board clock speeds */
  p->prc_clk = p->ioc_clk = p->ext_clk = 40000000; 
  if (p->type==ICEPIC3) p->ext_clk = 65000000;
  if (p->type==ICESLIC3) p->ext_clk = 64000000;
  if (p->isX) { p->ext_clk = 100000000; p->prc_clk = 80000000; }
  if (p->isY) { p->ext_clk = 100000000; p->prc_clk = 166000000; p->ioc_clk = 200000000; }
  p->int_clk = 10000000;

  /* Parse for user crystal speed */
  i = findintflag("CCLK",str);
  if (i>0) p->ext_clk=i;

  /* parse for operational flags */
  if (findintflag("DUAL",str) > 0) 	flags |= FLG_DUAL;
  if (findintflag("CLKI",str) > 0) 	flags |= FLG_CLKI;
  if (findintflag("MSBI",str) > 0) 	flags |= FLG_MSBI;
  if (findintflag("LSBX",str) > 0) 	flags |= FLG_LSBX;
  if (findintflag("VHS",str)  > 0) 	flags |= FLG_VHS;
  if (findintflag("NCCLK",str) > 0) 	flags |= FLG_NCCLK;
  if (findintflag("RGO",str) > 0) 	flags |= FLG_RGO;
  if (findintflag("SGO",str) > 0) 	flags |= FLG_SGO;
  if (findintflag("TGO",str) > 0) 	flags |= FLG_TGO;
  if (findintflag("GGO",str) > 0) 	flags |= FLG_GGO;
  if (findintflag("XGO",str) > 0) 	flags |= FLG_XGO;
  if (findintflag("XTGO",str) > 0) 	flags |= FLG_XTGO;
  if (findintflag("XSTGO",str) > 0) 	flags |= FLG_XSTGO;
  if (findintflag("BIGEND",str) > 0) 	flags |= FLG_BIGEND;
  if ((i=findintflag("BIT",str)) >= 0) {
         if (i==0) flags |= FLG_BIT0;
    else if (i==1) flags |= FLG_BIT1;
    else if (i==4) flags |= FLG_BIT4;
    else if (i!=15) { print("Illegal BIT=%d flag.  BIT=0|3|4|15\n",i); return -1; }
  }
  p->inp = findintflag("INP",str);
  p->timeout = findintflag("TO",str);
  p->timeout = findintflagdef("TIMEOUT",str,p->timeout);
  if (p->isY && (flags&(FLG_DUAL))!=0) {
    print("DUAL flag only apply to series 3 and 4 cards. Removing flag.\n");
    flags &= ~(FLG_DUAL);
  }

  i = findstrflag("IOC",str,tmpstr,1);
       if (i>0 && tmpstr[0]=='T') internal=1;
  else if (i>2 && tmpstr[2]=='R') internal=2;
  else if (i>2 && tmpstr[2]=='W') internal=3;
  else internal=0;

  /* determine the module types */
  p->miom=2;
  p->mbits=16; 
  for (i=0; i<p->miom; i++) {
    s=tmpstr; j=findstrflagx("IOMx",str,s,1,i+1);
    if (j<0) { print("No module specified: %s\n",str); return (-1); }
    if (findflag("IOM=TEST",str)>=0) type = IOMT_TEST;
    else if (internal>0) type = IOMT_NONE;
    else if (strncmp(s,"A2D",3) == 0) type = IOMT_A2D;
    else if (strncmp(s,"T2D",3) == 0) type = IOMT_T2D;
    else if (strncmp(s,"D2T",3) == 0) type = IOMT_D2T;
    else if (strncmp(s,"CXD",3) == 0) type = IOMT_CXD;
    else if (strncmp(s,"DXC",3) == 0) type = IOMT_DXC;
    else if (strncmp(s,"GXD",3) == 0) type = IOMT_GXD;
    else if (strncmp(s,"DXG",3) == 0) type = IOMT_DXG;
    else if (strncmp(s,"UXD",3) == 0) type = IOMT_UXD;
    else if (strncmp(s,"DXU",3) == 0) type = IOMT_DXU;
    else if (strncmp(s,"LV2D",4) == 0) type = IOMT_LV2D;
    else if (strncmp(s,"D2LV",4) == 0) type = IOMT_D2LV;
    else if (strncmp(s,"DR2D",4) == 0) type = IOMT_DR2D;
    else if (strncmp(s,"CDR2D",5) == 0) type = IOMT_CDR2D;
    else if (strncmp(s,"D2CDR",5) == 0) type = IOMT_D2CDR;
    else if (strncmp(s,"DXFPQC",6) == 0) type = IOMT_DXFPQC;
    else if (strncmp(s,"FPQCXD",6) == 0) type = IOMT_FPQCXD;
    else if (strncmp(s,"DXFPDP",6) == 0) type = IOMT_DXFPDP;
    else if (strncmp(s,"FPDPXD",6) == 0) type = IOMT_FPDPXD;
    else if (strncmp(s,"UDPXD",5) == 0) type = IOMT_UDPXD;
    else if (strncmp(s,"DXUDP",5) == 0) type = IOMT_DXUDP;
    else if (strncmp(s,"SDDSXD",6) == 0) type = IOMT_SDDSXD;
    else if (strncmp(s,"DXSDDS",6) == 0) type = IOMT_DXSDDS;
    else if (strncmp(s,"TGSXD",5) == 0) type = IOMT_TGSDDSXD;
    else if (strncmp(s,"DXTGS",5) == 0) type = IOMT_DXTGSDDS;
    else if (strncmp(s,"DXTGV",5) == 0) type = IOMT_DXTGVITA;
    else if (strncmp(s,"TGVXD",5) == 0) type = IOMT_TGVITAXD;
    else if (strncmp(s,"PSE2D",5) == 0) type = IOMT_PSE2D;
    else if (strncmp(s,"D2PSE",5) == 0) type = IOMT_D2PSE;
    else if (strncmp(s,"SNTXD",5) == 0) type = IOMT_SNTXD;
    else if (strncmp(s,"DXSNT",5) == 0) type = IOMT_DXSNT;
    else if (strncmp(s,"NFXD",4) == 0) type = IOMT_NFXD;
    else if (strncmp(s,"DXNF",4) == 0) type = IOMT_DXNF;
    else if (strncmp(s,"LB2D",4) == 0) type = IOMT_LB2D;
    else if (strncmp(s,"FLZRXD",6) == 0) type = IOMT_FLZRXD;
    else if (strncmp(s,"DXFLZR",6) == 0) type = IOMT_DXFLZR;
    else if (strncmp(s,"UFLZXD",6) == 0) type = IOMT_UFLZXD;
    else if (strncmp(s,"DXUFLZ",6) == 0) type = IOMT_DXUFLZ;
    else if (strncmp(s,"MSASXD",6) == 0) type = IOMT_MSASXD;
    else if (strncmp(s,"DXMSAS",6) == 0) type = IOMT_DXMSAS;
    else if (strncmp(s,"DSFPXD",6) == 0) type = IOMT_DSFPXD;
    else if (strncmp(s,"DXDSFP",6) == 0) type = IOMT_DXDSFP;
    else if (strncmp(s,"QSFPXD",6) == 0) type = IOMT_QSFPXD;
    else if (strncmp(s,"DXQSFP",6) == 0) type = IOMT_DXQSFP;
    else if (strncmp(s,"UPACXD",6) == 0) type = IOMT_UPACXD;
    else if (strncmp(s,"DXUPAC",6) == 0) type = IOMT_DXUPAC;
    else if (strncmp(s,"SFPPXD",6) == 0) type = IOMT_SFPPXD;
    else if (strncmp(s,"DXSFPP",6) == 0) type = IOMT_DXSFPP;
    else if (strncmp(s,"SFPXD",5) == 0) type = IOMT_SFPXD;
    else if (strncmp(s,"DXSFP",5) == 0) type = IOMT_DXSFP;
    else if (strncmp(s,"RFXD",4) == 0) type = IOMT_RFXD;
    else if (strncmp(s,"ES2D",4) == 0) type = IOMT_ES2D;
    else if (strncmp(s,"D2ES",4) == 0) type = IOMT_D2ES;
    else if (strncmp(s,"E2D",3) == 0) type = IOMT_E2D;
    else if (strncmp(s,"D2E",3) == 0) type = IOMT_D2E;
    else if (strncmp(s,"D2AWG",5) == 0) type = IOMT_D2AWG;
    else if (strncmp(s,"D2A",3) == 0) type = IOMT_D2A;
    else if (strncmp(s,"GPS",3) == 0) type = IOMT_GPS;
    else if (strncmp(s,"DIODE",5) == 0) type = IOMT_DIODE;
    else if (strncmp(s,"NONE",4) == 0) type = IOMT_NONE;
    else if (strncmp(s,"AUTO",4) == 0) type = IOMT_AUTO;
    else if (strncmp(s,"TEST",4) == 0) type = IOMT_TEST;
    else if (strncmp(s,"RF2D",4) == 0) type = IOMT_RF2D;
    else if (strncmp(s,"D2RF",4) == 0) type = IOMT_D2RF;
    else { print("Bad module spec: %s\n",str); return (-1); }
    if (i==0 && type==IOMT_DIODE) type=IOMT_SFPXD;
    /* get the module revision */
    rev = -1; j--;
    if (s[j]>='0' && s[j]<='9') rev = s[j]-'0';
    if (s[j-1]=='1') rev += 10;
    if (s[j-1]=='2') rev += 20;
    if (rev>=0); /* if not specified, determine default by card type */
    else if (type==IOMT_A2D) { rev=0;
      if (p->type==ICEPIC2) rev=2;
      if (p->type==ICEPIC3) rev=3;
      if (p->type==ICESLIC3) rev=1;
      if (p->type==ICEPIC4) rev=7;
      if (p->type==ICEMBT4) rev=7;
      if (p->type>=ICEPIC5 && p->type<=ICEPIC8) rev=8;
    }
    else if (type==IOMT_LB2D) rev=2;
    else if (type==IOMT_D2AWG) rev=2;
    else if (abs(type)==IOMT_DXTGSDDS) rev=1;
    else if (abs(type)==IOMT_DXTGVITA) rev=1;
    else if (abs(type)==IOMT_DXSDDS) rev=p->isY?5:4;
    else if (type==IOMT_RFXD) {
      rev = (rev<0)? 3 : rev;
    }
    else rev=0;
    if ((type==IOMT_A2D&&rev>14)||(type==IOMT_LB2D)) p->mbits=-16;
    p->mtype[i] = type;
    p->mrev[i] = rev;
    if (!p->isY); /* Auto VHS for 5 series modules */
    else if (p->type==ICEPIC5) flags |= FLG_VHS; /* on all others, use VHS PIC5 == non-VHS PIC6 */ 
    else if (abs(type)==IOMT_D2A && rev==11) flags |= FLG_VHS;
    else if (abs(type)==IOMT_D2A && rev==20) { flags |= FLG_VHS; flags |= FLG_SSC; if (p->type>ICEPIC6) p->qdrx=3; }
    else if (abs(type)==IOMT_D2A && rev>=15) { flags |= FLG_VHS; flags |= FLG_SSC; if (p->type>ICEPIC6) p->qdrx=3; }
    else if (abs(type)==IOMT_D2AWG && rev==3) { flags |= FLG_SSC; }
    else if (abs(type)==IOMT_DXSNT) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXFLZR) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXUFLZ) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXNF) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXTGSDDS) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXTGVITA) flags |= FLG_VHS;
    else if (abs(type)==IOMT_DXQSFP && p->k7>=8) p->qdrx=3;
  }
  p->mbits = findintflagdef("MBITS",str,p->mbits);
  i = findintflagdef("VHS",p->config,-1);
  if (i==0) flags &= ~FLG_VHS; 
  if (i==1) flags |=  FLG_VHS; 
  i = findintflagdef("SSC",p->config,-1);
  if (i==0) flags &= ~FLG_SSC; 
  if (i==1) flags |=  FLG_SSC; 

  /* fixup SlimPic config */
  if (p->type == ICESLIC3 && p->mtype[0]==IOMT_NONE) {
    p->mtype[0]=p->mtype[1]=IOMT_A2D;
    p->mrev[0]=p->mrev[1]=1;
  }

  /* determine the max number of processor modules */
  if (p->type==ICEMBT4) p->mpm=1;
  else if (p->type>=ICEPIC5 && p->type<=ICEPIC8) p->mpm=2;
  else p->mpm=0;
  p->mpm = findintflagdef("MPM",str,p->mpm);

  /* determine the processor module types */
  for (i=0; i<p->mpm; i++) {
    s=tmpstr; j=findstrflagx("PMx",str,s,1,i+1);
    rev = 1;
    if (findintflag("NOPM",str)>0) type = PMT_NONE;
    else if (j<0) type = (p->type == ICEMBT4)? PMT_DTDM : PMT_NONE;
    else if (strncmp(s,"DTDMX",5) == 0) type = PMT_DTDMX;
    else if (strncmp(s,"NONE",4) == 0) type = PMT_NONE;
    else if (strncmp(s,"DTDM",4) == 0) type = PMT_DTDM;
    else if (strncmp(s,"ZPPM",4) == 0) type = PMT_ZPPM;
    else if (strncmp(s,"V5M",3) == 0)  type = PMT_V5M;
    else if (strncmp(s,"V6M",3) == 0)  type = PMT_V6M;
    else if (strncmp(s,"A8M",3) == 0)  type = PMT_A8M;
    else if (strncmp(s,"K8M",3) == 0)  type = PMT_K8M;
    else if (strncmp(s,"K8P",3) == 0)  type = PMT_K8P;
    else if (strncmp(s,"S6M",3) == 0)  type = PMT_S6M;
    else if (strncmp(s,"AUTO",4) == 0) type = PMT_AUTO;
    else if (strncmp(s,"TEST",4) == 0) type = PMT_TEST;
    else { print("Bad processor module spec: %s\n",str); return (-1); }
    if (type!=PMT_NONE) npm++;
    p->pmtype[i] = type;
    p->pmrev[i] = rev;
  }
       if (p->pmtype[0]<=IOPT_NONE && p->pmtype[1]<=IOPT_NONE) i=0;
  else if (p->pmtype[0]<=IOPT_NONE && p->pmtype[1]> IOPT_NONE) i=2;
  else if (p->pmtype[0]> IOPT_NONE && p->pmtype[1]> IOPT_NONE && p->k7) i=3;
  else if (p->pmtype[0]==p->pmtype[1]) i=3;
  else i=1;
  p->pmi = i;
  p->pmif = findintflag("PMI",str);
  p->npm = findintflagdef("NPM",str,npm);

  /* PM hypertransport speed */
  j = (p->type==ICEPIC6)? 1 : 0;
  for (i=0; i<p->mpm; i++) if (p->pmtype[i]==PMT_DTDM || p->pmtype[i]==PMT_DTDMX) j=0;
  if (findintflagdef("HS",str,j) > 0) flags |= FLG_HS;

  NOPIC:
  if (findintflag("K7",p->config)>0) p->k7=7; /* PIC6 load of PIC7 */
  if (findintflag("K8",p->config)>0) p->k7=8; /* PIC6 load of PIC8 */

  /* determine the port type */
  s = tmpstr; j = findstrflag("PORT",str,s,1);
  /* embedded processor module syntax */
  if (j>0 && strncmp(s,"PM",2)==0) { p->pmif = (s[2]-48); s+=3; }
  if (p->pmif>=0) p->pmi=p->pmif;
     
  if (j<0) p->ptype = IOPT_NONE;
  else if (strncmp(s,"LINK",4)   == 0) { p->ptype = IOPT_LINK;   i=4; }
  else if (strncmp(s,"STREAM",6) == 0) { p->ptype = IOPT_STREAM; i=6; }
  else if (strncmp(s,"MODULE",6) == 0) { p->ptype = IOPT_MODULE; i=6; }
  else if (strncmp(s,"TUNER",5)  == 0) { p->ptype = IOPT_TUNER;  i=5; }
  else if (strncmp(s,"CORE",4)   == 0) { p->ptype = IOPT_CORE;   i=4; }
  else if (strncmp(s,"MCORE",5)  == 0) { p->ptype = IOPT_MCORE;  i=5; }
  else if (strncmp(s,"MCSTREAM",8)==0) { p->ptype = IOPT_CORE;   i=8; }
  else if (strncmp(s,"TBANK",5)  == 0) { p->ptype = IOPT_TBANK;  i=5; }
  else if (strncmp(s,"CBANK",5)  == 0) { p->ptype = IOPT_CBANK;  i=5; }
  else if (strncmp(s,"INTERNAL",8)==0) { p->ptype = IOPT_INTERNAL; i=8; }
  else if (strncmp(s,"EXTERNAL",8)==0) { p->ptype = IOPT_EXTERNAL; i=8; }
  else { print("Bad port name: %s\n",str); return (-1); }

  /* add the port index */
  if (p->ptype != IOPT_NONE) {
    p->pindex = (s[i]-48); 
    if (s[i+1] >= '0' && s[i+1] <= '9') {
      p->pindex = p->pindex*10 + (s[i+1]-48);
      if (p->pindex<10) p->pmi=0;	/* case of XXX0n forces PM0 */
    }
  }
  else p->pindex = 0;

  j = (p->pindex==0 || p->mtype[1]==p->mtype[0])? 3 : (p->pindex&1)? 1 : 2;
  p->qdrx = findintflagdefs("QDRX",str,p->qdrx,j);

  if (findintflag("ALT",str) > 0) flags |= FLG_ALT;

  /* process global muxclk configuration flag */
  p->gmcs = -1;
  p->gmcr = 0;
  p->gaux = 0;
  s = tmpstr; i = findstrflag("MUXCLK",str,s,1);
  if (i >= 0) {
    flags |= FLG_MUXCLK;
    if (i==0) ;
    else if (s[0]=='I') p->gmcs = GMC_CLKI; 
    else if (s[0]=='P') p->gmcs = GMC_CLKP;
    else if (s[0]=='A') p->gmcs = GMC_CLKA;
    else if (s[0]=='B') p->gmcs = GMC_CLKB;
    else if (s[0]=='C') p->gmcs = GMC_CLKC;
    else if (s[0]=='D') p->gmcs = GMC_CLKD;
    else if (s[0]=='X') p->gmcs = GMC_CLKX;
    else if (s[0]=='1') p->gmcs = GMC_CLKA;
    else if (s[0]=='2') p->gmcs = GMC_CLKB;
    else if (s[0]=='N'){p->gmcs = GMC_CLKN; flags &= ~FLG_MUXCLK;}
    else print("Illegal MUXCLK=%s flag - ignored\n",s);
  }

  /* process global muxclk configuration flag */
  if (findintflag("INTCLK",str) > 0) p->gmcs = GMC_CLKI;
  if (findintflag("XSOE",str) > 0) p->gmcr |= GMC_XSYNCOE;
  if (findintflag("XSAC",str) > 0) p->gmcr |= GMC_XSYNCEN;
  if (findintflag("XSTRM",str) > 0) p->gmcr |= GMC_XSYNCTRM;
  if (findintflag("PWR5OFF",str) > 0) p->gaux |= GMC_PWR5OFF;
  if (flags&FLG_NCCLK) p->gmcr |= GMC_NONCONT;
  if (p->gmcs==GMC_CLKI) flags |= FLG_INTCLK;
  if (p->gmcs==GMC_CLKP) flags |= FLG_INTCLK;
  if (p->gmcs==GMC_CLKX) p->gmcr |= GMC_XSYNCEN;

  p->flags = flags;

  p->tflags=0;
  if (findintflag("UFILT",str) > 0) 	p->tflags |= FLG_UFILT;
  if (findintflag("URFIR",str) > 0) 	p->tflags |= FLG_URFIR;
  if (findintflag("UCFIR",str) > 0) 	p->tflags |= FLG_UCFIR;
  if (findintflag("NCFIR",str) > 0) 	p->tflags |= FLG_NCFIR;
  if (findintflag("PFIR4",str) > 0) 	p->tflags |= FLG_PFIR4;
  if (findintflag("AOVSR",str) > 0) 	p->tflags |= FLG_AOVSR;
  if (findintflag("POVSR",str) > 0) 	p->tflags |= FLG_POVSR;
  if (findintflag("ITDEC",str) > 0) 	p->tflags |= FLG_ITDEC;
  if (findintflag("ITCPC",str) > 0) 	p->tflags |= FLG_ITCPC;
  if (findintflag("ITFMT",str) > 0) 	p->tflags |= FLG_ITFMT;
  if (findintflag("RESAMP",str) > 0) 	p->tflags |= FLG_RESAMP;
  if (findintflag("PRESAMP",str) > 0) {	p->tflags |= FLG_PRESAMP; p->res_ratio=1.0; }
  if (findintflag("PREDLY",str) > 0)	p->tflags |= FLG_PREDELAY;
  if (findintflag("DSYNC",str) > 0) 	p->tflags |= FLG_DSYNC;
  if (findintflag("TALT1",str) > 0) 	p->tflags |= FLG_TALT1;
  if (findintflag("FSYNC",str) > 0) 	p->tflags |= FLG_FSYNC;
  if (findintflag("GSYNC",str) > 0) 	p->tflags |= FLG_GSYNC;
  if (findintflag("FIRONLY",str) > 0) 	p->tflags |= FLG_FIRONLY;
  if (findintflag("PMWBT",str) > 0) 	p->tflags |= FLG_PMWBT;
  if (findintflag("PMWBTR",str) > 0) 	p->tflags |= (FLG_PMWBT|FLG_PMWBR);

  /* Find MultiCoreStreamMode */
  p->mcs = findintflagdef("MCS",p->config,0);
  p->chan = findintflagdef("MCI",p->config,0);

  /* pointer to break flag for user interrupt handling */
  p->vbreak = 0;
  if (pbreak!=0) p->pbreak = pbreak;
  else p->pbreak = &p->vbreak;

  /* handle for cleanup on signals - check lock */
  psave = p;
  pic_lock (p, LOCK_OPEN);
  p->state = 1;
  p->tc_soff = -1;
  if (flags==(FLG_TEST|FLG_NOBRK)) return 0;	/* for detect open */
  for (i=100; i>0 && pic_lock (p,LOCK_QUERY)==1; i--) udelay(10000); /* 1 sec for lock */
  if (i==0) p->state = 2;

  /* init IOC type */
  strncpy(p->iocc,"XXXX",4);
  strncpy(p->socc,"XXXX",4);
  if (nosig>0 && p->devno!=-3) return (1);
  if (p->isY && p->state==1) {
    getPortSig(p,0,0,&lsig,&ltmp); 
    *(int_4*)p->socc = lsig;
    isRR = (lsig&0xFFFF) == 0x5252;
    if (p->k7==8) {
      RD(REG_HAR5,i); p->nvme = ((i>>16)==0x1CE0)? i&0xFFFF : -1;	/* has configured NVME ports ? */
      RD(REG_HAR6,i); p->niop = ((i>>16)==0x1CE0)? i&0xFFFF : -1;	/* has configured NIOP ports ? */
      getPortSig(p,0,1,(int_4*)(tmpstr+4),(int_4*)(tmpstr+0)); 
           if (strncmp(tmpstr,"pac",3)==0)  { p->dsgtype=DSG_PAC;  p->npm=p->mpm=0; p->niop=isRR?0:4; }
      else if (strncmp(tmpstr,"npac",4)==0) { p->dsgtype=DSG_NPAC; p->npm=p->mpm=0; p->niop=isRR?0:4; }
      else if (strncmp(tmpstr,"cpac",4)==0) { p->dsgtype=DSG_CPAC; p->npm=p->mpm=0; p->niop=isRR?0:4; }
      else if (strncmp(tmpstr,"spac",4)==0) { p->dsgtype=DSG_SPAC;                  p->niop=isRR?0:4; }
      else if (strncmp(tmpstr,"pic8q",5)==0){ p->dsgtype=DSG_PICQ; p->npm=p->mpm=0; p->niop=8; }
      else if (strncmp(tmpstr,"pic8",4)==0 && p->devno==0) {		/* PIC1 PMs on IceTray */
        if (p->nvme>=0 || p->niop>=0) p->dsgtype=DSG_TRAY;
	if (p->dsgtype==DSG_TRAY && p->niop<0) p->niop=npm*4;		/* default NIO ports */
      }
    }
    for (i=p->mpm; i<4; i++) p->pmtype[i]=PMT_NONE;
  }

  /* Find Network IO Config */
  p->niop = findintflagdef("NIOP",p->config,p->niop);
  p->nio = findchoiceflagdef("NIO",p->config,PKTMODELIST,2)-2;
  p->nioi = findflag("IPORT=NIO",p->config)>=0;
  p->pktmode = (p->nio!=0)? p->nio : 1;
  p->pktmode = findchoiceflagdef("PKTMODE",p->config,PKTMODELIST,p->pktmode+2)-2;
  p->pktmode = findchoiceflagdef("PKTY",p->config,PKTMODELIST,p->pktmode+2)-2;

  /* Find NVME Config */
  p->nvme = findintflagdef("NVME",p->config,p->nvme);

  /* Other flags */
  p->up12 = findintflagdef("UP12",p->config,0);
  p->flashport = findintflagdef("FLASHPORT",p->config,0);

  /* PM poke through */
  if (findintflagdef("ICEPM",p->config,0)>0) p->type = ICEPM;

  vprint("Open Type=%d Port=%d.%d Chns=%d Cpc=%d IOM=(%d,%d) PM=(%d,%d) Flags=%x\n",
	p->type,p->ptype,p->pindex,p->nchan,p->cpc,
	p->mtype[0],p->mtype[1],p->pmtype[0],p->pmtype[1],p->flags);

  /* Parse tuner setup */
  pic_tuner_config(p);

  /* process timecode configuration flag */
  p->tcmode=TCM_OFF;
  if (findstrflag("TC",str,tmpstr,1) > 0) pic_parsetc(p,tmpstr,0);
  p->atccalib=0.0;
  if (findstrflag("ATCCALIB",str,tmpstr,0) > 0) sscanf(tmpstr,"%lf",&p->atccalib);

  p->driver = pic_getkeyl(p,0,KEY_DRIVER);
  if (p->nvme<0) p->nvme = pic_getkeyl(p,0,KEY_NVME);
  if (p->devno == -3) p->ptype = (p->csig==CORE_TFDD)? IOPT_TUNER : IOPT_CORE;

  /* test pattern modes */
  if (findintflag("TPLRS",p->config)>0) test_memory (p,0,MTEST_LRS,0);
  p->joinTime=0.0; p->joinTimer = finddblflagdef("IGMPTIMER",p->config,-1.0);
  p->vctlTime=0.0; p->vctlTimer = finddblflagdef("VCTLTIMER",p->config,0.1); p->vctlFill=-1.0;
  if ((i=findintflag("VCTL",p->config))>0) p->vctlTimer=1.0/i;
  iom_init(p,0);
  doXcvrInit(p);

  return (1);
}

int_4 pic_tuner_config (PICSTRUCT *p) 
{
  int_4 i,mc=0,pmt=0,uid=0,active=0,cpb1,cpb2,socc,socv,ctadr; 
  int_4 csig = p->tsig;

  /* default to future base FPGA architectures */
  p->gchip=GCFPGA; p->cpb=0; p->cpc=1;
  p->pm0t = p->pm0mt = 0;
  if (p->devno>=0) active = pic_active(p,0);
  if (p->isY && active>=0) {
    mc = pic_getkeyl(p,-41,KEY_MCORES); if (mc>0 && p->mcs>0) p->cpc=mc;
    if (pic_rpb(p,0,PPC_CORA_CTL|COREG_PARAM0)==csig) p->pm0t++;
    if (pic_rpb(p,0,PPC_CORB_CTL|COREG_PARAM0)==csig) p->pm0t++;
    if (pic_rpb(p,0,PPC_TUNA_CTL|COREG_PARAM0)==csig) p->pm0mt++;
    if (pic_rpb(p,0,PPC_TUNB_CTL|COREG_PARAM0)==csig) p->pm0mt++;
    if (p->pmif>0) p->pm0mt=0;
  }

  /* Find the number of tuner chips */
  if (p->pmi>0 || p->pm0mt>0) {
    i = p->pmi; if (i==3) i=1;
    pmt = (i==0)? PMT_K8M : p->pmtype[i-1];
    uid = (p->pm0mt>0)? 0x1CE1D : (active<0)? 0 : pic_getkeyl(p,10+i,KEY_UIDCODE);
    if (i>0 && active>=0 && pic_acmd(p,ADON_RD,CHECKSUM_ADDR+i+2,0,0)==0) uid=0;	/* PM not booted yet */
    if (pmt==PMT_DTDM || pmt==PMT_DTDMX) { p->gchip=GC4016; p->cpb=8; p->cpc=4; }
    else if (uid==0x1CE1D) { p->cpb=2; p->cpc=max( pic_getkeyl(p,-(i*10+3),KEY_MCORES) , pic_getkeyl(p,-(i*10+4),KEY_MCORES) ); }
    else { p->cpb=2; p->cpc=0; }
    p->cpc=findintflagdef("NBANK",p->config,p->cpc);
  }
  else if (p->type == ICEPIC2) p->gchip=0; 
  else if (p->type == ICEPIC3) p->gchip=0; 
  else if (p->type == ICEPIC4) { p->gchip=GC4016; p->cpb=2; p->cpc=4; }
  else if (p->type == ICEMBT4) { p->gchip=GC4016; p->cpb=0; p->cpc=4; } /* NOPM */
  else if (p->type == ICESLIC3) {p->gchip=GC4014; p->cpb=(p->tflags&FLG_TALT1)? 2:1; p->cpc=4; }
  else if (p->isY) p->cpb=p->pm0t;

  /* Find active GCFPGA tuner configuration */
  if (p->gchip==GCFPGA && active>=0) {
    i = p->pmi; if (i==3) i=1;
    if (i==0 || uid==0x1CE1D) {
      ctadr = (i==0 && p->pm0mt==0)? 0x10000000 : 0x40000000;
      p->csig = pic_rpb(p,i,ctadr|COREG_PARAM0);
      p->tclk = pic_rpb(p,i,ctadr|COREG_PARAM7)*1000000;
      p->tclk = findintflagdef("TCLK",p->config,p->tclk); 
      if (p->csig==CORE_TFDD) { /* from parameter set */
        p->order = pic_rpb(p,i,ctadr|COREG_PARAM1); 
        p->mdw   = pic_rpb(p,i,ctadr|COREG_PARAM2); 
        p->nres  = pic_rpb(p,i,ctadr|COREG_PARAM3); 
        p->nps   = pic_rpb(p,i,ctadr|COREG_PARAM4); 
        p->cfxo  = pic_rpb(p,i,ctadr|COREG_PARAM5); 
      }
    }
  }

  /* Find the number of channels per chip */
  p->mchan = (int_1)(p->cpb * p->cpc);
  cpb2 = (p->cpb/2)*p->cpc; /* channels per bank back end */
  if (p->gchip!=GCFPGA) {
    p->cpc = findintflagdef("CPC",p->config,p->cpc);
    if (p->cpc!=1 && p->cpc!=2 && p->cpc!=4) print("Improper Tuner CPC= setup: %s\n",p->config); 
  }
  cpb1 = (p->cpb/2)*p->cpc; /* channels per bank front end */

  /* no need for ITDEC if cpc==1 */
  if (p->cpc==1) p->tflags &= (~FLG_ITDEC); 

  /* Find the number of useable channels */
  p->fttm = findintflagdef("FTTM",p->config, (p->ptype==IOPT_TBANK)? 1:-1);
  if (p->fttm>1) {
    p->nchan = cpb1*cpb2;
    if (p->fttm==3) p->nchan *= cpb2;
  }
  else {
    p->nchan = p->cpb*p->cpc;
    /* Dual processor module slots */
    if (p->pmi==3 && p->fttm<0) { p->nchan*=2; p->mchan*=2; } 
  }

  /* map to appropriate chan,chip,link tables */
  if (p->type == ICEMBT2) {
    p->chantbl = mbtchantbl;
    p->chiptbl = mbtchiptbl;
    p->linktbl = mbt2linktbl;
    p->tadrtbl = mbt2tadrtbl;
  }
  else if (p->type == ICEPIC3) {
    p->chantbl = picchantbl;
    p->chiptbl = picchiptbl;
    p->linktbl = piclinktbl;
    p->tadrtbl = pic3tadrtbl;
  }
  else if (p->type == ICEPIC4) {
    p->chantbl = mbtchantbl;
    p->chiptbl = mbtchiptbl;
    p->linktbl = piclinktbl;
    p->tadrtbl = pic4tadrtbl;
  }
  else if (p->gchip == GC4016) {
    p->chantbl = mbtchantbl;
    p->chiptbl = mbtchiptbl;
    p->linktbl = mbt4linktbl;
    p->tadrtbl = mbt4tadrtbl;
    p->tflags |= FLG_ITDEC;
  }
  else if (p->gchip == GCFPGA) {
    p->chantbl = mbtchantbl;
    p->chiptbl = mbtchiptbl;
    p->linktbl = pic5linktbl;
    p->tadrtbl = pic5tadrtbl;
    p->tflags |= FLG_ITDEC;
  }
  else if (p->type == ICESLIC3) {
    p->chantbl = slic3chantbl;
    p->chiptbl = slic3chiptbl;
    p->linktbl = slic3linktbl;
    p->tadrtbl = slic3tadrtbl;
    if (p->tflags&FLG_TALT1) {
      p->chantbl = slicchantbl;
      p->chiptbl = slicchiptbl;
    }
  }
  i = p->ext_clk;
  if (p->gchip==GC4016) i = (strstr(p->config,"=SDDSXD")!=0)? 104000000 : 101000000;
  p->tun_clk = findintflagdef("TCLK",p->config,i); 
  p->nchan = findintflagdef("CHNS",p->config,p->nchan);
  p->decb = 0;	/* init filter band decimation factor */
  p->ovsr[0] = p->ovsr[1] = 0;	/* init local OVSR register */
  p->cfir_gain = p->pfir_gain = p->res_gain = 1.0; /* user filter gains */
  p->ntaps = p->ctaps = 0;
  return active;
}

/*
  These functions are for Tuners and MultiCore ports.

  Special port/chip numbers, m is module #

  1m1 all A
  1m2 all B
  1m3 all A&B
  1m4 back
  1m5 front A
  1m6 front B
  1m7 front A&B
  1m8 back

   0 all
  >0 single port

*/

int_4 port2pmi (PICSTRUCT *p, int_4 *i) {
  int_4 pmi = p->pmi, nh = p->mchan>>1;
  if (*i > 100) {
    if (*i >= 120) pmi=2;
    if (*i >= 130) pmi=3;
    if (*i >= 140) pmi=4;
    return pmi; /* chip port */
  }
  if (pmi==3) {
    if (*i > 32) pmi = 2;
    else if (*i > nh) { pmi = 2; (*i)-=nh; }
    else pmi = 1;
  }
  if (pmi==2 && *i>32) (*i)-=32;
  return pmi;
}
int_4 port2chan (PICSTRUCT *p, int_4 i) {
  if (p->gchip==GCFPGA) return (i-1)>>1;
  if (i > 100) return 0; /* chip port */
  return p->chantbl[i];
}
int_4 port2chip (PICSTRUCT *p, int_4 i) {
  if (i > 100) {  /* chip port */
    i = (i-100)%10;
    if (p->gchip==GC4016) i+=8;
    return i; 
  }
  if (p->gchip==GCFPGA) {
    if (i==0) return 0;
    if (p->pm0mt>0) return (i&1)? 3:4;
    return (i&1)? 1:2;
  }
  return p->chiptbl[i];
}
int_4 chip2tadr (PICSTRUCT *p, int_4 i, int_4 pmi) {
  int_4 tadr = p->tadrtbl[i];
  if (p->gchip==GCFPGA) tadr = (p->pmi>0)? pm5tadrtbl[i] : (p->pm0mt>0)? coretadrtbl[i] : pic5tadrtbl[i];
  if (pmi<=0) pmi = p->pmi;
  if (p->isY && pmi>0 && p->gchip!=GCFPGA) tadr |= getJTAGsel(p,JTAG_PORT_PMOFF+pmi);
  return tadr;
}
int_4 chip2link (PICSTRUCT *p, int_4 i) {
  return (p->gchip==GCFPGA && p->pmi>0)? piclinktbl[i] : p->linktbl[i];
}
int_4 port2tadr (PICSTRUCT *p, int_4 i) {
  return 0;
}

int_4 pic_close (PICSTRUCT *p) 
{
  int_4 n,status;
  int_4 *vaddr, paddr, psize;
  CHECK_MAGIC(p);

  if (p->state!=0) {
    /* look for unfinished DMAs */
    for (n=0; n<DMA_CHNS; n++) if (p->dma[n].haddr != 0) {
      print ("DMA Chan #%d not closed properly before pic_close()  - cancelling\n",n+1);
      pic_dmafunc (p, n+1, DMA_CANCEL);		/* stop DMA */
      psize = HW2P(p->dma[n].hsize);
      paddr = HW2P(p->dma[n].haddr);
      vaddr = p->dma_vaddr[n];
      pic_map (p, &vaddr, &paddr, psize, -1); 	/* deallocate buffer */
    }
  }
  for (n=0; n<MAX_MCS; n++) if (p->plan[n]!=0) { core_close(p->plan[n]); p->plan[n]=NULL; }

#if _VMS
  if (p->fd != 0) sys$dassgn (p->fd);
#elif _WIN
  if (p->fd != 0) status = CloseHandle (p->fd);
#elif _MAC
  vprint("PIC close:\n");
#else
  if (p->fd==MMFD) picmclosemap(p->base,PGSZ,(char*)p->loc);
  if (p->fd > 0) if (close(p->fd) != 0) perror("PIC close");
#endif
  if (jtagfd!=NULL) { fclose(jtagfd); jtagfd!=NULL; }

  if (p->state!=0) {
    if (p->lcount > 0) {
      p->lcount = 1;
      pic_lock (p, LOCK_FREE);
    }
    pic_lock (p, LOCK_CLOSE);
  }
  psave = 0;
  p->state = 0;
  if (p->config!=0) { free(p->config); p->config=NULL; }
  if (p->ftt!=0) { free(p->ftt); p->ftt=NULL; }
  if (p->iom_buf!=0) { free(p->iom_buf); p->iom_buf=NULL; }
  for (n=0; n<2; n++) if (p->iom_state[n]!=0) { free(p->iom_state[n]); p->iom_state[n]=NULL; }
  vprint("Close: devno=%d\n",p->devno);
  return (1);
}

void pic_cleanup() { 
  if (psave != (PICSTRUCT *)0 ) pic_close (psave); 
}

int_4 pic_parsetc (PICSTRUCT *p, char *str, int_4 flags) {
  int_4 i,irbreg,tcflags; char c, *mode=p->tc_mode;
  p->flags &= ~FLG_TC;
  if (flags!=FLG_INFO) {
    for (i=0; i<strlen(str); i++) mode[i]=str[i];
    for (; i<7; i++) mode[i]=' ';
  }
       if (strncmp(mode,"SDN",3)==0) p->tcmode = TCM_SDN;
  else if (strncmp(mode,"SMS",3)==0) p->tcmode = TCM_SMS;
  else if (strncmp(mode,"DTL",3)==0) p->tcmode = TCM_DTL;
  else if (strncmp(mode,"CPU",3)==0) p->tcmode = TCM_CPU;
  else if (strncmp(mode,"ZTC",3)==0) p->tcmode = TCM_ZTC;
  else if (strncmp(mode,"STC",3)==0) p->tcmode = TCM_STC;
  else if (strncmp(mode,"IRB",3)==0) p->tcmode = TCM_IRB;
  else if (strncmp(mode,"SDD",3)==0) p->tcmode = TCM_SDD;
  else if (strncmp(mode,"ICE",3)==0) p->tcmode = TCM_ICE;
  else if (strncmp(mode,"VRT",3)==0) p->tcmode = TCM_VRT;
  else if (strncmp(mode,"OFF",3)==0) p->tcmode = TCM_OFF;
  else if (strncmp(mode,"NONE",4)==0) p->tcmode = TCM_OFF;
  else if (strncmp(mode,"FILE",4)==0) p->tcmode = TCM_FILE;
  else { 
      print("Illegal TC mode [%.7s]\n",mode); 
      return TC_BADPARAM; 
  }
  if (p->tcmode==TCM_OFF);
  else if (p->tcmode==TCM_CPU);
  else if (p->tcmode==TCM_ZTC);
  else if (p->tcmode==TCM_STC);
  else if (p->tcmode==TCM_FILE);
  else {
    p->flags &= ~FLG_BITS;
    c = mode[3];
    if      (c=='0') p->flags |= FLG_BIT0;
    else if (c=='X' && p->tcmode==TCM_IRB);
    else if (c=='S' && p->tcmode==TCM_SDD);
    else if (c=='1') p->flags |= FLG_BIT1;
    else if (c=='4') p->flags |= FLG_BIT4;
    else if (c=='3') p->flags |= FLG_BIT3;
    else if (c==' '||c==0) { 	/* valid tail delimiter */
      if (p->tcmode==TCM_SMS) p->flags |= FLG_BIT0; 
      else if (p->tcmode==TCM_IRB); 
      else p->flags |= FLG_BIT4;
    } 
    else { 
      print("Illegal TC bit specified in mode [%.7s]\n",mode); 
      return TC_BADPARAM; 
    }
    p->flags |= FLG_TC;
  }
  p->tc_modl = *(int_4*)mode;  /* save for BC test */
  /* set special timecode registers */
  irbreg = 0;
  tcflags = 0;
  if (p->tcmode==TCM_IRB) {
    if (p->isX) {
      irbreg = IRB_IRIG; 
      if (mode[4]=='P') tcflags |= FLG_TC_OPPS;
      if (mode[4]=='P') irbreg |= IRB_OPPS;
      if (mode[3]=='0') irbreg |= IRB_SWAP;
    }
    else if (p->isY) {
      if (mode[4]=='P') tcflags |= FLG_TC_OPPS;
      if (mode[3]=='0') print("PIC5+ cards do not support timecode mode IRB0P, only IRBXP\n");
    }
    else print("IRIG-B timecode only support on series 4+ cards\n");
  }
  else if (p->tcmode==TCM_CPU && mode[3]=='P') {
    tcflags = FLG_TC_OPPS;
    if (mode[4]=='X') tcflags |= FLG_TC_SWAP;
    if (!(p->isX||p->isY)) print("CPU-P timecode only support on series 4+ cards\n");
  }
  else if (p->tcmode==TCM_CPU && mode[4]=='P') {
    tcflags = FLG_TC_OPPS;
    if (mode[3]=='X') tcflags |= FLG_TC_SWAP;
    if (!(p->isX||p->isY)) print("CPU-P timecode only support on series 4+ cards\n");
  }
  if (p->tcmode==TCM_SDN || p->tcmode==TCM_DTL) tcflags |= (FLG_TC_FILL|FLG_TC_PREFILL); 
  for (i=4; i<7 && mode[i]>' '; i++) {
    if (mode[i]=='X') tcflags |= FLG_TC_VFTX;	/* use extended precision */
    if (mode[i]=='F'&& mode[i+1]=='1') tcflags |= FLG_TC_FILL; 
    if (mode[i]=='F'&& mode[i+1]=='0') tcflags &= ~(FLG_TC_FILL|FLG_TC_PREFILL); 
  }
  if (findintflag("NOTCFILL",p->config)>0) tcflags &= ~FLG_TC_PREFILL; 
  if ((tcflags&FLG_TC_OPPS)!=0 && findintflagdef(p->config,"VFT",0)>0) tcflags |= FLG_TC_OVFT;
  p->oppsoffset = findintflagdef("OPPSOFFSET",p->config,0);
  p->leapsecsoy = findintflagdef("LEAPSECDOY",p->config,0) * 86400;
  p->irbreg = irbreg;
  p->tcflags = tcflags;
  return p->tcmode;
}

int_4 findflagp (char *fstr, char *astr) 
{
  int_4 i,j,k,n,ln,delim,paren=0; char c;
  for (n=0,ln=strlen(fstr); n<ln && fstr[n]!='.'; n++);
  for (i=0; astr[i]!=0; i++) {
    c = astr[i];
    delim = (c==',' || c=='+' || c=='(' || c=='|' || c==' ' || c=='\n');	/* valid front delimiter */
    if (c=='(' && i>0) paren++;	/* mask flag syntax FLG=(A|B) */
    if (c==')') paren--;
    if ((delim||i==0) && paren==0) {
      k = delim? i+1 : 0;
      for (j=0; j<n; j++) if (astr[k+j]!=fstr[j]) goto NOMATCH;
      c = astr[k+j];
      delim = (c==',' || c=='+' || c==')' || c=='|' || c=='=' || c=='\n'); /* valid tail delimiter */
      if (delim) {
        k=k+j;
        if (n<ln) { i=findflagp(fstr+n+1,astr+k+2); k=(i>0)? k+2+i : -1; } /* subTable key past =(  */
        return (k);  
      }
      i+=(j-1);
    }
    NOMATCH:;
  }
  return (-1);
}

int_4 findflag (char *fstr, char *astr) 
{
  int_4 i = findflagp(fstr,astr);
  return (i>0)? i-strlen(fstr) : i;
}

int_4 findintflag (char *fstr, char *s) 
{
  char c; int_4 i,j,k,value=0,mult=1,base=10,off,exp=0,mexp=1;
  i = findflagp(fstr,s);
  if (i < 0) return -1;  		/* state not present */
  if (s[i]!='=') return 1;  		/* state present */
  c = s[++i]; j=i;
  if (c=='-') { mult=-1; c=s[++i]; }
  if (c=='+') { c=s[++i]; }
  if (c=='0') c=s[++i];
  if (c=='X' || c=='x') { base=16; c=s[++i]; }
  for (; (c>='0' && c<='9') || (base==16 && c>='A' && c<='F'); c=s[++i]) {
    off = (c>='A')? 55 : 48;
    value = value*base + (c-off);
  }
  if (c=='.') return (int_4)(finddblflag(fstr,s)+.1);
  if (c=='K') { i++; mult *= 1024; }
  if (c=='M') { i++; mult *= 1024*1024; }
  if (c=='G') { i++; mult *= 1024*1024*1024; }
  if (c=='E') {
    c=s[++i];
    if (c=='-') { mexp = -1; i++; }
    else if (c=='+') { i++; }
    for (; s[i] >= '0' && s[i] <= '9'; i++) exp = exp*10 + (s[i]-48); 
    mult *= pow(10.0,(double)(mexp*exp)); 
  }
  for (k=i,c=s[i]; c!=0 && c!=',' && c!='+' && c!=')' && c!='|' && c!='='; c=s[++i]); /* valid tail delimitter */
  if (i!=k) print("Unrecognized numeric input flag=%s text=%.*s\n",fstr,i-j+1,s+j);
  return value*mult;			/* value switch */
}

int_4 findintflagdefs (char *fstr, char *s, int_4 def, int_4 sdef) 
{
  int_4 i = findflagp(fstr,s);
  if (i<0) return def;
  if (s[i]!='=') return sdef;
  return findintflag(fstr,s);
}

int_4 findintflagdef (char *fstr, char *s, int_4 def) 
{
  int_4 i = findflagp(fstr,s);
  if (i<0) return def;
  return findintflag(fstr,s);
}

int_4 findmaskflagdef (char *fstr, char *s, char *list, int_4 def) 
{
  char c; int_4 i,j,k,ls,ll,lf,lm,n,all,mask,value=def,delim,mode=2;
  i = findflagp(fstr,s);
  if (i < 0) return value;  		/* state not present */
  c=s[i++]; if (c!='=') { print("Mask flag=%s must be valued in %s\n",fstr,s); return -1; }
  c=s[i];   if (c>='0' && c<='9') return findintflag(fstr,s);	/* numeric format */
            if (c!='(') { print("Bad mask flag=%s syntax FLG=(A|B|C) in %s\n",fstr,s); return -1; }
  ls=strlen(s); for (j=i; j<ls && s[j]!=')'; j++); ls=j;
  ll=strlen(list); for (k=0,all=0x1; k<ll; k++) if (list[k]==',') all |= (all<<1);
  for (;i<ls;) {
    for (j=++i; j<ls && s[j]!='|' && s[j]!=')'; j++);
    c = s[i];
         if (c=='+') { mode=1; i++; }
    else if (c=='-') { mode=0; i++; }
    else if (c=='~') { mode=-1; i++; }
    else if (mode==2) { value=0; mode=1; }
    lf = j-i;
    if (lf==7 && strncmp(s+i,"DEFAULT",7)==0) value = def;
    else if (lf==3 && strncmp(s+i,"DEF",3)==0) value = def;
    else if (lf==3 && strncmp(s+i,"ALL",3)==0) value = all;
    else if (lf==4 && strncmp(s+i,"NONE",4)==0) value = 0;
    else for (n=k=0,mask=0x1; k<ll; ) {
      for (j=k; j<ll && list[j]!=','; j++);
      lm = j-k;
      if (strncmp(s+i,list+k,lf)==0) { n++;
        if (mode==1) value |= mask;
        if (mode==0) value &= ~mask;
        if (mode==-1) value ^= mask;
      }
      k = k + lm + 1;
      mask <<= 1;
    }
    if (n==0) print("Err finding %.*s in %s for flag=%s\n",lf,s+i,list,fstr);
    i = i + lf;
  }
  return value;			/* value switch */
}

real_8 finddblflag (char *fstr, char *s) 
{
  int_4 i,dec=0; double value=0,mult=1,exp=0,mexp=1;
  i = findflagp(fstr,s);
  if (i < 0) return -1.0;  		/* state not present */
  if (s[i]!='=') return 1.0;  		/* state present */
  if (s[i+1]=='-') { mult=-1; i++; }
  for (i++; (s[i]>='0' && s[i]<='9') || s[i]=='.'; i++) {
    if (dec==1) mult /= 10;
    if (s[i]=='.') dec=1; 
    else value = value*10 + (s[i]-48);
  }
  if (s[i]=='E') { i++;
    if (s[i]=='-') { mexp = -1; i++; }
    else if (s[i]=='+') { i++; }
    for (exp=0; s[i] >= '0' && s[i] <= '9'; i++) exp = exp*10 + (s[i]-48); 
    mult *= pow(10.0,(double)(mexp*exp)); 
  }
  return value*mult;			/* value switch */
}

double finddblflagdef (char *fstr, char *s, double def) 
{
  double d = finddblflag(fstr,s);
  return (d==-1.0)? def : d;
}

int_4 findstrflag (char *fstr, char *s, char *fout, int_4 ulc) 
{
  int_4 i,j=0,p=0; char c;
  i = findflagp(fstr,s);
  if (i < 0) return -1;
  if (s[i++]!='=') return -1;
  for (;i<strlen(s);) {
    c = s[i++]; 
    if (c=='(') p++;
    if ((c==',' || c==')' || c=='|' || c=='=' || c=='\n') && p<=0) break;
    if (c==')') p--;
    if (c=='/' && ulc<0) ulc=0;
    if (ulc==0);
    else if (ulc>0 && c>='a' && c<='z') c-=32;
    else if (ulc<0 && c>='A' && c<='Z') c+=32;
    fout[j++]=c; 
  }
  fout[j]=0;
  return j;
}

/* find string flag ABCxDEF where x is a digit from 1-9 OR omitted */
int_4 findstrflagx (char *fstr, char *s, char *fout, int_4 ulc, int_4 x)
{
  int_4 i,j,k,ls=strlen(fstr); char t[16];
  if (ulc==9) goto JUSTX;
  for (i=0; i<=ls; i++) t[i] = (fstr[i]=='x')? '0'+x : fstr[i];
  k = findstrflag (t,s,fout,ulc);	/* find with index */
  if (k>=0) return k;
  for (i=j=0; i<=ls; i++) if (fstr[i]!='x') t[j++]=fstr[i];
  k = findstrflag (t,s,fout,ulc);	/* find without index */
  if (k>=0) return k;
  JUSTX:
  for (i=0; i<=ls; i++) t[i] = (fstr[i]=='x')? 'X' : fstr[i];
  k = findstrflag (t,s,fout,ulc);	/* find with replace of X in result with x */
  if (k>=0) for (i=0; i<=k; i++) if (fout[i]=='X') fout[i]='0'+x; 
  return k;
}

int_4 findchoiceflagdef (char *fstr, char *s, char *list, int_4 def) 
{
  char c,fout[40]; int_4 i,ll,lf,value=def;
  lf = findstrflag (fstr, s, fout, 1);
  if (lf < 0) return value;  					/* not present */
  c=fout[0]; if (c>='0' && c<='9') return findintflag(fstr,s);	/* numeric format */
  strcat(fout,","); lf++;
  ll=strlen(list);
  for (i=0,value=1;i<ll;i++,value++) {
    if (i+lf>ll) lf--;
    if (strncmp(list+i,fout,lf)==0) return value; 
    for (;i<ll && list[i]!=',';i++);
  }
  print("Err finding %.*s in %s for flag=%s\n",lf,fout,list,fstr);
  return def;
}

int_4 ioctype (PICSTRUCT *p, char *str) 
{
  int_4 mux = (p->flags&FLG_MUXCLK)!=0;
  int_4 sddsinput = 0;
  int_4 i,mt;

  for (i=1; i<=2; i++) { mt = p->mtype[i-1];
    if ((mt==IOMT_SDDSXD || mt==IOMT_TGSDDSXD) && findflag("RXRAWSDDS",p->config)<0) sddsinput |= i;
  }

  strcpy(str,"");
  for (i=0; i<2; i++) strcat(str, (p->mtype[i]>0)? "o":"i");

  if (p->isY) {
    for (i=1; i<=2; i++) strcat(str, (sddsinput&i)? "s":(mux)?"x":"n");
    return 0;
  }

  if (p->mtype[0]==IOMT_SDDSXD && p->mtype[1]>0); /* special case ioxs=ios */
  else if (p->mtype[0]<=0 && mux) strcat(str,"x");

       if (sddsinput==3) strcat(str,"s");
  else if (sddsinput==1) strcat(str,(p->mtype[1]>0)? "s":"sn");
  else if (sddsinput==2) strcat(str,(mux)? "s":"ns");

  if (p->type==ICESLIC3) {
    if (p->mtype[0]==IOMT_A2D&&p->mtype[1]==IOMT_A2D) strcpy(str,"iix");
    else if (p->mtype[0]>0) strcpy(str,"o");
    else strcpy(str,(mux)? "ix":"i");
  }

  return 0;
}

int_4 extjtag (PICSTRUCT *p, int_4 arg1, int_4 arg2, int_4 arg3, int_4 arg4, int_4 *data)
{
  int_4 status=0,arg0=0x12345678,stat,i,n,size=0;
  if (p->verbose==3) print("Remote access to jtag args=(%08x,%08x,%08x,%08x)\n",arg1,arg2,arg3,arg4);
  fflush(jtagfd);
  fwrite ((void *)(&arg0), 1, 4, jtagfd);	/* Sync word */
  fwrite ((void *)(&arg1), 1, 4, jtagfd);	/* 14=ADON_JTAG 15=ADON_PGMC */
 if (arg1==IOCTL_ACMD) { 
  fwrite ((void *)(data), 1, 20, jtagfd);	/* all MBX regs */
 } else {
  fwrite ((void *)(&arg2), 1, 4, jtagfd);	/* TMS bits */
  fwrite ((void *)(&arg3), 1, 4, jtagfd);	/* TDI bits */
  fwrite ((void *)(&arg4), 1, 4, jtagfd);	/* Enable Mask 0xNNNN0R0W N=ignore R=readByte[3:0]  W=writeByte[3:0] */
 }
  fflush(jtagfd);
  if (arg1==ADON_PGMC||arg1==ADON_LIOC) size=abs(arg3);
  for (i=0; i<10000 && (n=fread((void *)(&status),1,4,jtagfd))==0; i++) udelay(100);	/* TDO return bits or bulk ACK */
  stat = (n!=4)? -1 : (size!=0 && status!=size)? -2 : 0;
  if (stat<0||p->verbose==3) print("Reading n=%d status=%08x from ext JTAG device. Type=0x%x, size=%d\n", n, status, arg1, size);
  if (stat<0) return -1;
  if (size!=0) {
    size *= 4; 
    if (arg3>0) fwrite ((void *)data, 1, size, jtagfd);	/* bulk write */
    if (arg3<0) fwrite ((void *)data, 1,    4, jtagfd);	/* bulk read may have some startup bytes */
    fflush(jtagfd);
    if (arg3<0) {
      for (i=n=0; i<1000 && n<size; i++) {
        stat = fread  ((void *)(((char*)data)+n), 1, min(16,size-n), jtagfd);	/* bulk read */
        if (stat==0) udelay(1000);
        if (stat>0) n += stat;
      }
      if (n!=size) printf("Only read n=%d of %d bytes\n",n,size);
    }
  }
  return status;
}
  
/*
When the ADON gets an interrupt (by writing to OMB1), the SHARC/PPC reads the 
command from OMB1 and any arguments from OMB2, OMB3, and OMB4.  Status is
returned in IMB1.  The data FIFOs are untouched.  
*/

#define ACMD_TIMEOUT 400


int_4 pic_acmd (PICSTRUCT *p, int_4 arg1, int_4 arg2, int_4 arg3, int_4 arg4)
{
  int_4 i,value,mbx[5],wait,argchk;
  volatile int_4 status;

  if (jtagfd!=NULL && arg1==ADON_JTAG) return extjtag(p,arg1,arg2,arg3,arg4,NULL);
  if (p->state<=0 || p->devno<0) return (ACMD_NOACK);

  if (p->isY && arg2==REG_AFIFO && arg1<=2) arg2=0x00800000; 
  argchk = arg1^(arg2^(arg3^arg4));
  pic_lock (p, LOCK_ALLOC);
  RETRY:
  mbx[0] = arg1; mbx[1] = arg2; mbx[2] = arg3; mbx[3] = arg4; mbx[4] = argchk;
  status = io_ctl (p, IOCTL_ACMD, 0, mbx, 20);
  if (status == 4) { value = mbx[0]; goto DONE; }  /* responded in ioctl */
  /* wait for the SHARC/PPC to respond (0.2 or 2 seconds) */
  if (UBREAK) wait = 20; else wait = ACMD_TIMEOUT;
  for (i=0; i<wait; i++) {
    udelay(10000); 
    RD(REG_MCSR,status); 
    if ( (status&MCSR_IMB1F) != 0x0) { 
      RD(REG_IMB1, value); goto DONE;
    }
  }
  print ("ICE-PIC #%d type=%d not responding-%d \n  to %x %x %x %x %x ...\n",
	p->devno,p->type,p->lcount,arg1,arg2,arg3,arg4,argchk); 
  pic_sniff (p,0);
  value = ACMD_NOACK;
  if (UBREAK);
  else if (pic_fixup(p)>0) goto RETRY;   /* fixed it */
  print ("ICE card #%d runtime fix not allowed ... reset card\n",p->devno); 
  p->state = -2;
  DONE:
  pic_lock (p, LOCK_FREE);
  return (value);
}

int_4 checkSide (PICSTRUCT *p, int_4 side, int_4 clear) {
  int_4 i,j,n,chans,port,stkloc,dmac,dmacs[DMA_CHNSY];
  if (p->isY && side==0 && clear==0) return pic_acmd (p, ADON_RD, DMA_HPQL, 0, 0);
  for (i=j=chans=0; i<DMA_CHNS; i++) {
    stkloc = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
    if (stkloc==0) break;
    n = (stkloc-DMA_STK)/DMA_SEP;
    if (stkloc>=DMA_STK2) n = (stkloc-DMA_STK2)/DMA_SEP + DMA_CHNSX;
    port = pic_acmd (p, ADON_RD, stkloc+DMA_PORT, 0, 0);
    if ( (side==1) && (port&1)==0 ) continue;
    if ( (side==2) && (port&1)!=0 ) continue;
    chans++;
    if (!clear) continue;
    pic_stkupd (p, &p->dma[n], DMA_SEP, -1); 
    if (p->dma[n].master<0) continue;
    dmacs[j++] = n+1;
  }
  for (--j; j>=0; j--) {
    dmac = dmacs[j];
    print("Resetting DMA Channel=%d for Side=%d\n",dmac,side);
    pic_dmafunc(p,dmac,DMA_STOP);
    pic_dmafunc(p,dmac,DMA_CANCEL);
    p->dma[dmac-1].haddr = 0;
  }
  return chans;
}

int_4 pic_shutdown (PICSTRUCT *p) {
  int_4 n;
  CHECK_MAGIC(p);
  pic_lock (p, LOCK_ALLOC);
  for (n=0; n<DMA_CHNS; n++) if (p->dma[n].haddr != 0) pic_dmafunc (p, n+1, DMA_CANCEL); /* stop DMAs */
  WR(REG_MCSR, MCSR_RESET|MCSR_RIOC); 
  pic_lock (p, LOCK_FREE);
  return -1;
}

int_4 pic_disable_nvme (PICSTRUCT *p, int_4 port) {
  int_4 i;
  if (p->nvme<=0) return 0;
  for (i=1; i<3; i++) {	/* disable NVME drives */
    if ((i&p->nvme)==0) continue;
    if (p->dsgtype==DSG_NPAC) pic_nvm_stat (p,i,9);
    else if (i>p->npm);
    else if (port==10+i || port==15 || port==-1) pic_nvm_stat (p,10+i,9);
  }
  return p->nvme;
}

int_4 pic_setup_sysmon (PICSTRUCT *p, int_4 port) {
  int_4 addr,k7;
  if (p->k7==8 && port==0) {
    pic_wpb(p,port,PPC_SYSMON|0x40,0x9000);
    pic_wpb(p,port,PPC_SYSMON|0x41,0x2fd0);
    pic_wpb(p,port,PPC_SYSMON|0x42,0x2000);
    pic_wpb(p,port,PPC_SYSMON|0x48,0x4701);
    pic_wpb(p,port,PPC_SYSMON|0x03,0x0);	/* reset the SYSMON special address */
    udelay(10000);
  }
  k7 = 0;
  if (port==0 && p->k7==7 && p->rev<5) k7=1;
  if (port>JTAG_PORT_PMOFF && p->k7!=8 && p->pmtype[port-JTAG_PORT_PMOFF-1]>=PMT_ZPPM) k7=1;
  if (k7) {
    addr=0x08400000; pic_jtag (p, port, JTAG_CMD_SYSMON, &addr, 32, 0); 
    addr=0x08410000; pic_jtag (p, port, JTAG_CMD_SYSMON, &addr, 32, 0); 
    addr=0x08420000; pic_jtag (p, port, JTAG_CMD_SYSMON, &addr, 32, 0); 
  }
}


int_4 pic_reset (PICSTRUCT *p, int_4 flags)
{
  int_4 rstat=0,pstat=0,i,j,k,mc,ls,sel,vt,pmi=0,gchip,tflags,dtdm,port=0,pmall=0,stkloc,*pbreaksave,active,status,sig,ver,force,k7s,j1,j2;
  char lfstr[80], lfs[8], lfsx[80]; double qref; char *strsig=(char*)(&sig);
  CHECK_MAGIC(p);

  flags |= p->flags;
  tflags = p->tflags;

  k7s = p->k7;
  if (p->type<ICEPIC7) p->k7=0;
  if (p->type==ICENIC) return 1;
  if (p->type==ICENVM) return 1;
  if (p->type==ICEPM) return 1;

  active = pic_active(p,1);
  force = (active<0)? 1 : 0;

  if (flags&FLG_DISABLE) {	/* implement the HALT function */
    pic_lock (p, LOCK_ALLOC);
    if (active>=0) pic_disable_nvme(p,-1);
    WR(REG_MCSR, MCSR_RESET|MCSR_RIOC); 
    pic_lock (p, LOCK_FREE);
    return (3);
  }

  /* get current FPGA download checksums */
  force = findintflagdef("FORCE",p->config,force);

  if (active>=0) { 
    /* checksum versioning only applies to 319+ trees */
    if (pic_acmd (p, ADON_RD, VERSION_ADDR, 0, 0) >= 319) {
      for (i=0; i<7; i++) {
	sig = (i>=1 && i<=2) && (p->mtype[i-1]==IOMT_GPS && force!=2); /* dont reset GPS modules */
	if (force<=0 || sig) p->crcs[i] = pic_acmd (p, ADON_RD, CHECKSUM_ADDR+i, 0, 0);
      }
    }
  }

  if ( (flags&FLG_BOOT)==0 && (p->ptype!=IOPT_NONE) ) {
    /* see if the card is up and running */
    if (active<0) pic_fixup (p);
    /* check for SHARC/PPC/HOST code version match */
    status = pic_acmd (p, ADON_RD, VERSION_ADDR, 0, 0); 
    if (status == ACMD_NOACK) return -1;
    if (status!=ICE_VERSION) {
      print ("HOST software version #%d does not match card #%d - aborting\n", ICE_VERSION,status); 
      return -1;
    }
    getIOCsig(p,0);
    /* now reset the port */
    vprint ("Resetting Module/Port #%d\n",p->pindex);
    pic_ioport (p, -1, -1, 0, 0, 16, 1000000, 0.0, 64, 0, FLG_DISABLE);
    port = (p->pindex&0x1)? 1 : 2;
    pic_acmd (p, ADON_WR, RSTIME_ADDR+port, (int_4)pic_time(), 0); 
    return (2);
  }

  /* check for partial reset */
  if ( (flags&FLG_BOOT)==0 && force<=0 && p->side>0 ) {
    status = checkSide(p,p->side,1);  /* clear all on this side */
    if (status>0) print("Cleared %d channels on side=%d \n",p->side,status);
    status = checkSide(p,0,0);  /* any remaining */
    if (status>0) { print("Leaving reset of side=%d to preserve %d channels on alternate side\n",p->side,status); return (3); }
  }

  pic_lock (p, LOCK_ALLOC);

  /* hard reset - ignore break flag to STOP DMA's */
  if (flags&FLG_NOBRK) { pbreaksave = p->pbreak; p->pbreak = &p->vbreak; }

  /* clear the master control status register */
  WR(REG_MCSR, MCSR_RESET|(p->isY?MCSR_RIOC:0)); 
  udelay(10);

  /* clear the adon mailboxes */
  for (i=7; i>=0; i--) WR(REG_MBX+i*4, 0);

  /* clear the address and transfer count registers */
  WR(REG_MDTC, 0);
  WR(REG_MDAR, 0);
  WR(REG_MDARH, 0);

  /* load the bootstrap code into the SHARC/PPC/JVM processor */
  if (p->isY) {
    WR(REG_HAR0,0); 
    status = getPortSig(p,0,0,&sig,&ver);
    i = findintflagdef("TPOE",p->config,0); 
    pic_setkeyl (p, 0, KEY_TPOE, i); /* enable test port */
    WR(REG_IOC, REG_IOC_ADDR|PM_RESET_CLKS);
    RD(REG_IOC, status);
    udelay(20); /* wait for clock lock */
    status = pic_loadfile (p, "*",p->k7? FLG_JVM : FLG_PPC);
    if ((i=findstrflag("PM0FPGA",p->config,lfs,1))>0 && strncmp(lfs,strsig+2,i)!=0) {
      print ("WARNING: SoC Sig=%.4s not compatible with required PM0FPGA=%.4s. See help on SoC to reload flash\n",strsig,lfs);
      status = -1;
    }
    if (status>=0 && ver!=ICE_VERSION && findintflag("NOVERSION",p->config)<=0) {
      print ("WARNING: SoC Ver=%d not equal to Software Ver=%d. See help on SoC to reload flash\n",ver,ICE_VERSION);
      status = -1;
    }
    if (status>=0 && p->k7==7 && findintflag("NOSLR",p->config)<=0) {
      i = findintflagdef("MEMSPEED",p->config,250);
      sprintf(lfstr,"*_slr%d",i);
      status = pic_loadfile (p,lfstr,FLG_SLR);
      if (status<0) print ("WARNING: Problem configuring SL clock=%dMHz. See help on SLR.\n",i);
      udelay(10000); /* wait for clock lock */
    }
    if (status>=0) test_memory (p,0,(force>0)?MTEST_CALIB:MTEST_AUTO,0);
  } else {
    status = pic_loadfile (p, "*", FLG_SHARC);
  }
  if (status < 0) { rstat-=1; goto BAIL; } 

  /* start APP/DMA handlers */
  pic_acmd (p, ADON_INIT, 0, p->dsgtype, 0);

  /* load the bootstrap code into the IOC processor */
  if (findstrflag("IOC",p->config,lfs,-1)<0) ioctype(p,lfs);
  strcpy(lfstr,"*_"); strcat(lfstr,lfs);

  if (p->isY) {
    pic_acmd (p, ADON_WR, SOCCODE_ADDR, sig, 0); 
    getIOCsig(p,0);
    setIOCsig(p,0,lfs);
    if (p->k7==7) WR(REG_IOC, REG_IOC_ADDR|((p->flags&FLG_HS)?PM_RESET_CLKP:PM_RESET_CLKPV));
    if (p->k7==7) RD(REG_IOC, i); /* sets HyperTransport Clock speed */
    pic_wpbx (p,0,PPC_CORES_CTL|MCORE_ALL,0);
    pic_wpbx (p,0,PPC_CORES_CTL|MCORE_SYS,0);
  }
  else {
    pic_loadfile (p, lfstr, FLG_BIOC);
  }

  /* make sure there is a JTAG clock */
  if (p->isX) {
    pic_setup_ioc_clk (p, -2, p->tun_clk);
    udelay(10000); /* allow clock to stabilize */
  }

  /* load resampler filters here to reduce time locked in port setups */
  pmi = p->pmi; 
  if (p->isY) {
    gchip=p->gchip; p->pmi=0; p->gchip=GCFPGA;
    j=pic_rpb(p,0,pic5tadrtbl[1]|COREG_PARAM0); /* tuner SIG */
    k=pic_rpb(p,0,pic5tadrtbl[1]|COREG_PARAM3); /* tuner RTAPS */
    if (j==CORE_TFDD && k==8) pic_loadfile (p, "res_8x256_100", FLG_FC|0);
    if (j==CORE_TFDD && k==10) pic_loadfile (p, "res_10x2048_100", FLG_FC|0);
  }
  pmall = (pmi==3) && (p->pmtype[0]==p->pmtype[1]) && (p->pmrev[0]==p->pmrev[1]) && (p->pmtype[i]!=PMT_S6M)
        && (findflag("PM1FPGA",p->config)<0) && (findflag("PM2FPGA",p->config)<0);

  /* load default processor module code */
  for (i=0; i<p->mpm; i++) {
    p->pmi = i+1;
    port = JTAG_PORT_PMOFF+i+1;
    if (p->pmtype[i]==PMT_TEST && p->isY) test_memory (p, port, MTEST_PM2MB, 0);
    if (p->pmtype[i]<=PMT_NONE) continue;
    switch (p->pmtype[i]) {
      case PMT_DTDM:  strcpy(lfstr,"icedtdm");   j=18; k=IDCODE_DTDM;  vt=2; break;
      case PMT_DTDMX: strcpy(lfstr,"icedtdmx");  j=14; k=IDCODE_DTDMX; vt=2; break;
      case PMT_S6M:   strcpy(lfstr,"ices6m");    j=18; k=IDCODE_S6M;   vt=6; break;
      case PMT_ZPPM:  strcpy(lfstr,"icezppm");   j=18; k=IDCODE_ZPPM;  vt=5; break;
      case PMT_V5M:   strcpy(lfstr,"icev5m");    j=18; k=IDCODE_V5M;   vt=5; break;
      case PMT_V6M:   strcpy(lfstr,"icev6m");    j=18; k=IDCODE_V6M;   vt=6; break;
      case PMT_A8M:   strcpy(lfstr,"icea8m");    j=18; k=IDCODE_A8M;   vt=8; break;
      case PMT_K8M:   strcpy(lfstr,"icek8m");    j=18; k=IDCODE_K8M;   vt=8; break;
      case PMT_K8P:   strcpy(lfstr,"icek8p");    j=18; k=IDCODE_K8P;   vt=8; break;
      default:        strcpy(lfstr,"iceUnknown");j=14; k=0x00000000;   vt=2; break;
    }
    if (p->isX) pic_acmd (p,ADON_WR,DMA_JRWIOB,JTAG_CMD_USER2<<j,0);
    pic_jtag (p, port, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF;
    if (p->pmtype[i]==PMT_S6M && j==IDCODE_S6MX) { strcat(lfstr,"x"); k=IDCODE_S6MX; } /* new chip on S6M */
    ls=strlen(lfstr); 
    if (j!=k) { 
      print("Wrong PM=%d sig=%08x for PM type=%d sig=%08x\n",i+1,j,p->pmtype[i],k);
      p->pmtype[i]=PMT_NONE; continue;
    }
    strcpy(lfs, (vt>=8)? "rr" : p->isY? "hh" : "ss");
    lfsx[0]=0;
    j=findstrflagx("PMxFPGA",p->config,lfsx,-1,i+1);
    if (j>0) strcat(lfs,lfsx);
    else if (p->tflags&FLG_PMWBR) strcat(lfs,"tr");
    else if (p->tflags&FLG_PMWBT) strcat(lfs,"tf");
    status = getPortSig(p,port,0,&sig,&ver); 
    pic_jtag (p, port, JTAG_CMD_UIDCODE, &j, -32, 0); 
    vprint("PM Load Sig=%s on Sig=%.4s Ver=%d Id=%x\n",lfs,(char*)(&sig),ver,j);
    strcat(lfstr,"_"); strcat(lfstr,lfs); 
    k = (i==0 && pmall)? JTAG_PORT_PMALL : port;
    sel = getJTAGsel(p,k);
    if (lfsx[0]=='/') strcpy(lfstr,lfsx);	/* PMxFPGA has full path */
    status = pic_loadfile (p, lfstr, FLG_PM|k);
    if (status<0) {
      print("Bad FPGA Load of Sig=%s on Processor Module #%d type=%s\n",lfs,k-JTAG_PORT_PMOFF,getPmName((int_4)p->pmtype[i]));
      p->pmtype[i]=PMT_NONE; rstat-=1; continue;
    }
    pstat = (i>0 && pmall)? pstat : (status>0)? 1 : 0;
    j = findintflag("PMTPOE",p->config); 
    if (vt>2) pic_acmd (p,ADON_WRIOB,sel|PM_TPORT_CFG,(j<=0)?0:(0x10|(j-1)),1);
    if (vt>=5) {
      lfstr[ls]=0; if (findstrflag("JVM",p->config,lfs,-1)>0) { strcpy(lfstr,"_"); strcpy(lfstr,lfs); }
      /* use all characters of module name as JVM file prefix */
      status = pic_loadfile (p, lfstr, FLG_JVM|port); 
    } else {
      if (findstrflag("PPC",p->config,lfs,-1)<=0) strcpy(lfs,"def");
      /* use first 7 characters of module name as PPC file prefix */
      lfstr[7]=0; strcat(lfstr,"ppc_"); strcat(lfstr,lfs);
      status = pic_loadfile (p, lfstr, FLG_PPC|port); 
    }
    if (status<0) continue;
    if (findintflag("PMMEM",p->config)==0); else
    if (pstat>=0) test_memory (p, port, (force>0)?MTEST_CALIB:MTEST_AUTO, 0);
    if (p->isY) test_memory (p, port, MTEST_MB2PM, 0);
    if (p->isY) test_memory (p, port, MTEST_PM2MB, 0);
    pic_msg (p,port,PKTF_INIT,-1,0,0,0,1000);
    if (p->pmtype[i]==PMT_DTDM || p->pmtype[i]==PMT_DTDMX) {
      k = mbt4tadrtbl[0]; if (p->isY) k |= sel;
      p->gchip=GC4016; pic_acmd (p,ADON_WRIOB,k|0x80,0x0F,1);     /* reset tuner channels */
    } else {
      k=pic_rpb(p,i+1,pm5tadrtbl[1]|COREG_PARAM3); /* tuner RTAPS */
      p->gchip=GCFPGA; if (pstat==0); /* already done */
      else if (k==10) pic_loadfile (p, "res_10x2048_100", FLG_FC|(113+i*10)); /* load WB FPGA resampler coefs */
      else if (k==8)  pic_loadfile (p, "res_8x256_100", FLG_FC|(113+i*10)); /* load default FPGA resampler coefs */ 
    }
    j = i+1;
    pic_wpbx (p,j,PPC_CORES_CTL|MCORE_ALL,0);
    pic_wpbx (p,j,PPC_CORES_CTL|MCORE_SYS,0);
    pic_setup_sysmon (p,port);
  }
  p->pmi = pmi;
  p->gchip = gchip;
  p->tflags = tflags;

  /* initialize on-board clocks */
  resyncIOCclocks(p);

  /* reconfigure and disable any TUNER chips */
  pic_tuner_config(p);
  pic_tuner (p, -1, 16, 10000000, 0.0, 64, 0, FLG_DISABLE); 

  /* load default IO module code */
  rstat += reloadModuleCode(p);

  /* disable modules and tuner oversampling */
  pic_acmd (p, ADON_MOD, 3, 0, 0);
  pic_acmd (p, ADON_WRIOC, IOC_A|IOC_B|IOC_FRAME, 0, 2); 
  pic_acmd (p, ADON_WRIOC, IOC_A|IOC_B|IOC_PROG, 0, 1);
  pic_acmd (p, ADON_WRIOC, IOC_A|IOC_B|IOC_TCMODE, 0, 1);
  pic_acmd (p, ADON_WRIOC, IOC_A|IOC_B|IOC_TCOUNT, 0, 4); 
  if (p->isX) pic_acmd (p, ADON_WRIOC, IOC_A|IOC_B|IOC_IRIG, 0, 1); 
  if (p->isY) pic_acmd (p, ADON_WRIOC, IOC_GAUX, (int_4)p->gaux, 1); 
  if (p->isY) pic_acmd (p, ADON_WRIOC, IOC_GMC, p->gmcr,1);
  for (i=0; i<3; i++) pic_acmd (p, ADON_WR, DMA_GMCFG+i, 0, 0);
  pic_acmd (p, ADON_WR, DMA_MTAP, 0, 0);
  pic_tuner_ovsr(p,0,3,0);

  /* initialize sane boundaries for TTDM algorithms */
  if (p->isX) for (i=p->ndmac; i<DMA_CHNSX; i++) {
    stkloc = DMA_STK+i*DMA_SEP;
    pic_acmd (p, ADON_WR, stkloc+DMA_CADDR, DM_ALG_ADDR, 0);
    pic_acmd (p, ADON_WR, stkloc+DMA_CIND,  DM_ALG_ADDR, 0);
    pic_acmd (p, ADON_WR, stkloc+DMA_CSIZE, DM_ALG_SIZE, 0);
  }

  /* initialize processor DMA modes and cores */
  if (p->isY) { 
    i=(p->type==ICEPIC8 && p->pbw>1)? PPC_ROUTF_HPFDP:0;
    pic_acmd (p, ADON_WR, DMA_ROUTF, i, 0);
    pic_acmd (p, ADON_WR, PPC_DMAC_ROUTE_FLG>>2,i,0);
    pic_acmd (p, ADON_WR, PPC_DMAC_ROUTE_CLR>>2,-1,0);
    pic_acmd (p, ADON_WR, PPC_CORA_CTL>>2,0,0);
    pic_acmd (p, ADON_WR, PPC_CORB_CTL>>2,0,0);
  }

  /* stamp the version number into the PRC code */
  pic_acmd (p, ADON_WR, VERSION_ADDR, ICE_VERSION, 0); 
  i = (int_4)pic_time();
  pic_acmd (p, ADON_WR, RSTIME_ADDR+0, i, 0); 
  pic_acmd (p, ADON_WR, RSTIME_ADDR+1, i, 0); 
  pic_acmd (p, ADON_WR, RSTIME_ADDR+2, i, 0); 

  /* clear the output mailbox */
  pic_acmd (p, ADON_NULL, 0x3, 0, 0);

  /* set transfer control attributes */
  WR(REG_MCSR, MCSR_ENABLE);

  /* check for reasonable system assigned latency timer */
  if (p->pbr>=250) {		/* PCI express */
    if (findintflag("PCICAP",p->config)>0) pic_cap(p,1);
    i=pic_nv_read_l(p,0xE4); j=(i>>12)&0x7; 
    if (j==0) k=2; else k=max(1,min(3,j));
    if (j!=k || p->k7) { 
      vprint("System PCIe Read Payload=%d out of range (1,3), override=%d in effect\n",j,k);
      WR(0xE4, ((i&0xFFFF0FFF) | (k<<12)) );
    }
    i=pic_nv_read_l(p,0xE4); j=(i>>5)&0x7;
    j1=j2=0;						/* default to 128by payload only */
    if (p->type==ICEPIC8)              { j1=0; j2=1; }	/* PIC8 supports 128|256by payload */
    if (p->type==ICEPIC8 && p->rev==5) { j1=1; j2=1; }	/* PIC8P only supports 256by payload */
    if (j<j1 || j>j2) {
      k = max(j1,min(j2,j));
      vprint("System PCIe Write Payload=%d out of range (%d,%d), override=%d in effect\n",j,j1,j2,k);
      WR(0xE4, ((i&0xFFFFFF1F) | (k<<5)));
    }
    if ((j=findintflag("WMPL",p->config))>=0){
      vprint("System PCIe Write Payload override=%d in effect\n",j);
      WR(0xE4, ((i&0xFFFFFF1F) | (j<<5)));
    }
  } else {
    if ( (i=pic_nv_read_b(p,0x0D)) < 0x10) {
      vprint("Low System PCI Latency Timer = %02x, override in effect\n", i);
      WR(REG_PLAT, 0x4000);
    }
  }

  /* defaults for QSFP control on PAC */
  if (p->dsgtype>=DSG_PAC && p->dsgtype<=DSG_CPAC) {
    setTpMpc(p,0x4100);	/* 8'bz0zzzzz0   bit6 is resetn, bit0 is lpmode */
    setTpMpc(p,0x4140);	/* 8'bz1zzzzz0   out of reset */
    if (p->niop<=0) {	/* set k8tgr core ref clock for RAW QSFP capture */
      qref = 1000*finddblflagdef("QLR",p->config,10.3125)/64;	/* QSFP Line Rate in GHz */
      qref = finddblflagdef("QREF",p->config,qref);		/* QSFP ref freq in MHz */
      setup_SI514(p,qref);
    }
  }

  /* setup default NIO ports */
  if (p->dsgtype>0 && p->niop>0) {
    for (i=0; i<64; i++) {
      if (p->dsgtype<=DSG_PICQ) pic_wpb(p,0,PPC_DMAC_ROUTE_NIO,i<<8);
      if (p->dsgtype==DSG_TRAY && p->niop>=4 && p->npm>=1) pic_wpb(p,1,PPC_DMAC_ROUTE_NIO,i<<8);
      if (p->dsgtype==DSG_TRAY && p->niop>=8 && p->npm>=2) pic_wpb(p,2,PPC_DMAC_ROUTE_NIO,i<<8);
    }
    pic_ifup_qsfp (p,p->niop,0);
    if (p->dsgtype<=DSG_PICQ) status = pic_loadfile (p, "icenio", FLG_JVM_NIO|0);
    if (p->dsgtype==DSG_TRAY && p->niop>=4 && p->npm>=1) status = pic_loadfile (p, "icenio", FLG_JVM_NIO|11);
    if (p->dsgtype==DSG_TRAY && p->niop>=8 && p->npm>=2) status = pic_loadfile (p, "icenio", FLG_JVM_NIO|12);
    pic_ifup_qsfp (p,p->niop,1);
  }

  /* setup default NVME ports */
  if (p->dsgtype>0 && p->nvme>0) {
    for (i=1; i<3; i++) {
      if ((i&p->nvme)==0) continue;
      if (p->dsgtype==DSG_NPAC) k=pic_nvm_stat (p,i,8);
      else if (i<=p->npm )      k=pic_nvm_stat (p,10+i,8); 
      if (k>=0) status |= i;
    }
    for (i=0; i<8; i++) {		/* up to 8 concurrent threads ? */
      pic_acmd (p, ADON_WR, (PPC_DMA_NVME>>2)+i*2+0, (i*PIPE_SIZE(PB1M))|PIPE_CFG(PB1M), 0);
      pic_acmd (p, ADON_WR, (PPC_DMA_NVME>>2)+i*2+1, -1, 0);
    }
  }

  pic_setup_sysmon (p,0);

  pic_log(p,1);

  /* disable jtag output drivers */
  i=0; if (p->isX||p->isY) pic_wpm (p,0,PPC_JTAG_ENA,&i,4,FLG_PPC_BUS);

  /* restore hard reset switches */
  if (flags&FLG_NOBRK) p->pbreak = pbreaksave; 
  BAIL:
  pic_lock (p, LOCK_FREE);
  p->k7 = k7s;
  return ((rstat<0)?rstat:1);
}

void resyncIOCclocks (PICSTRUCT *p) {
  int_4 i,j,n,sel;
  int_4 pgmcs = (findintflag("PREFX",p->config)>0)? 1 : 0; 
  if (p->isX) {
    pic_acmd (p,ADON_WRIOC,IOC_PGMCS,pgmcs,0);
    udelay(10000);
    pic_setup_ioc_clk (p, -2, p->tun_clk);
    pic_setup_ioc_clk (p, -1, 10000000);
    udelay(10000);
    i = findintflag("TCLK",p->config); 
    pic_acmd (p,ADON_RDIOB,(i>0)?PM_RESET_CLKTV:PM_RESET_CLKT,0,1); /* reset tuner clocks */
    pic_acmd (p,ADON_RDIOB,PM_RESET_MODALL,0,1); 
    if (p->pmtype[0]!=PMT_NONE) {
      i = findintflag("PMTHROTTLE",p->config); if (i<=0) i = 70000000;
      i = PPC_CLOCK_HYPMAX | min(255, (int)(256 * ((double)i / (p->tun_clk*2/3))));
      pic_wpm (p,1,PPC_DMAC_CLOCK,&i,4,FLG_PPC_BUS);
      test_memory (p, JTAG_PORT_PMOFF+1, MTEST_MB2PM, 0);
    }
  }
  if (p->isY) { 
    pic_acmd (p,ADON_WRIOC,IOC_GMC,GMC_CLKC-1,1); 
    pic_acmd (p,ADON_WRIOC,IOC_PGMCS,pgmcs,0);
    WR(REG_IOC, REG_IOC_ADDR|((p->flags&FLG_VHS)?PM_RESET_CLKUV:PM_RESET_CLKUVN));
    RD(REG_IOC, i); /* sets HyperTransport Clock speed based on IOC_ADDR */
    WR(REG_IOC, REG_IOC_ADDR|((p->flags&FLG_SSC)?PM_RESET_CLKUS:PM_RESET_CLKUSN));
    RD(REG_IOC, i); /* sets HyperTransport Clock speed based on IOC_ADDR */
    i = findintflag("TCLK",p->config); 
    for (n=1; n<=p->mpm; n++) {
      sel = getJTAGsel(p,JTAG_PORT_PMOFF+n);
      if (p->pmtype[n-1]==PMT_DTDM || p->pmtype[n-1]==PMT_DTDMX)
        pic_acmd (p,ADON_RDIOB,sel|((i>101000000)?PM_RESET_CLKTV:PM_RESET_CLKT),0,1); /* reset tuner clocks */
    }
  }
}

int_4 isOM20 (PICSTRUCT *p) {
  PICSTRUCT pc=*p;
  pc.mtype[0]=pc.mtype[1]=IOMT_UNKNOWN;
  detect_rumel_mods(&pc);
  return (pc.mrev[0]==19)?1:0;
}

int_4 reloadModuleCode (PICSTRUCT *p) {
  /* load the bootstrap code into the MODULE processors */
  int_4 i,j,stat,status=0,mtype,mrev,mtypes[2],mrevs[2],mtypeo,modvlan,diff,wait=0,same,bidir,vsens,rios=0,internal;
  char mode,name[20]; int crcs[3];
  for (i=0; i<3; i++) crcs[i]=p->crcs[i];
  /* check if the same module on both sides */
  memcpy ((void*)(&mtype), (void*)(p->mtype), 4); /* save off mtype fields */
  if (p->isY) getIOMsig(p,0);
  for (i=0; i<2; i++) { mtypes[i]=p->mtype[i];  mrevs[i]=p->mrev[i]; }
  memcpy ((void*)(p->mtype), (void*)(&mtype), 4); /* restore orig mtype fields */
  for (i=1; i<=2; i++) {
    mtype = p->mtype[i-1];
    mrev  = p->mrev[i-1];
    mtypeo= abs(mtype); /* output form of I or O module */
    mode  = p->socc[i-1];
    bidir = isBiDir(mtypeo);
    vsens = (mtypeo==IOMT_D2A) || (mtypeo==IOMT_DXSNT) || (mtypeo==IOMT_DXTGSDDS) || (mtypeo==IOMT_DXTGVITA); /* version sensitive */
    same  = (mtypes[1]==mtypes[0]) && (mrevs[1]==mrevs[0]) && (p->socc[1]==p->socc[0]);
    internal = getIOCsig(p,IS_INTERNAL|i);
    if (bidir) same &= (abs(p->mtype[1])==abs(p->mtype[0]));
    else       same &= (p->mtype[1]==p->mtype[0]);
    if (vsens) same &= (p->mrev[1]==p->mrev[0]);
    WR(REG_IOMCRCS+(i-1)*4,0);
    /* check for proper configuration */
    if (mtype==IOMT_NONE || mtype==IOMT_TEST);
    else if ((mtypeo==IOMT_DXSFP) || (mtypeo==IOMT_DXDSFP) || (mtypeo==IOMT_DXSFPP) || (mtypeo==IOMT_DIODE) || (mtypeo==IOMT_DXQSFP) || (mtypeo==IOMT_DXUPAC));
    else if ((mtypeo != abs(mtypes[i-1])) || (!bidir && (mtype != mtypes[i-1])) || (vsens && (mrev!=mrevs[i-1]))) {
      strcpy(name,getIomName((int_4)mtypes[i-1],(int_4)mrevs[i-1]));
      print("Configuration mismatch on module=%d config=%s detect=%s\n",i,getIomName(mtype,mrev),name);
      status -= 1;
      continue;
    }
    if (i==1 && same && p->socc[0]!='R' && mtypeo!=IOMT_DXSDDS && mtype!=IOMT_RFXD && mtype!=IOMT_GPS && 
	(p->qdrx==0||p->qdrx==3) ) i=3; /* both sides */
    /* single site modules with no special setup */
    if (mtype==IOMT_A2D && mrev==9) pic_loadfile (p, "icea2dr9", FLG_MOD|i);
    if (mtype==IOMT_A2D && mrev==10) pic_loadfile (p, "icea2dr10", FLG_MOD|i);
    if (mtype==IOMT_A2D && mrev==13) pic_loadfile (p, "icea2dr13", FLG_MOD|i);
    if (mtype==IOMT_D2A && mrev<8) pic_loadfile (p, "iced2a", FLG_MOD|i);
    if (mtype==IOMT_D2A && mrev==9) pic_loadfile (p, "iced2ar9", FLG_MOD|i);
    if (mtype==IOMT_CXD) pic_loadfile (p, "icecxd", FLG_MOD|i);
    if (mtype==IOMT_CDR2D) pic_loadfile (p, "icecdr2d", FLG_MOD|i);
    if (mtype==IOMT_D2CDR) pic_loadfile (p, "iced2cdr", FLG_MOD|i);
    if (mtype==IOMT_GXD) pic_loadfile (p, "icegxd", FLG_MOD|i);  
    if (mtype==IOMT_DXFPDP) pic_loadfile (p, "icefpdpo", FLG_MOD|i);
    if (mtype==IOMT_FPDPXD) pic_loadfile (p, "icefpdpi", FLG_MOD|i);         
    if (mtypeo==IOMT_DXUDP) pic_loadfile (p, "icesdds", FLG_MOD|i);        
    if (mtypeo==IOMT_DXMSAS) rios |= (PPC_MSAS_ENABLE<<((i==2)?4:0));
    if (mtypeo==IOMT_DXDSFP) rios |= (PPC_DSFP_ENABLE<<((i==2)?4:0));
    if (mtypeo==IOMT_DXSFPP) rios |= (PPC_SFPP_ENABLE<<((i==2)?4:0));
    if (mtypeo==IOMT_DXSFP)  rios |= (PPC_SFP_ENABLE<<((i==2)?4:0));
    if (mtypeo==IOMT_DIODE)  rios |= (PPC_SFP_ENABLE<<((i==2)?4:0));
    if (mtype==IOMT_RFXD && mrev==1) pic_loadfile (p, "icerfxdr1", FLG_MOD|i);
    if (mtype==IOMT_RFXD && mrev==2) pic_loadfile (p, "icerfxdr2", FLG_MOD|i);
    if (mtype==IOMT_RFXD && mrev==3) pic_loadfile (p, "icerfxdr3", FLG_MOD|i);
    if (mtypeo==IOMT_D2RF) { pic_loadfile (p, "iced2rf_h", FLG_MOD|i); pic_loadfile (p,"iceiom",FLG_JVM|i); d2rf_reset(p,i); }
    if (mtypeo==IOMT_DXSNT && mrev<=1) pic_loadfile (p, "icesonet", FLG_MOD|i);        
    if (mtypeo==IOMT_DXUFLZ && mrev==1) pic_loadfile (p, "iceuflzr1", FLG_MOD|i);
    /* dual site modules / load on MOD 1 / i should = 3 now */
    if (mtype==IOMT_A2D && mrev==11) pic_loadfile (p, "icea2dr11", FLG_MOD|0x2);
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) { pic_loadfile (p, (p->k7<8)?"icea2dm1x_hh6":"icea2dm1x_hh", FLG_MOD|0x2); a2dm1x_reset(p); }
#if NEWSTUFF
    if (mtype==IOMT_A2D && mrev==20) {
      j=pic_loadfile (p, isOM20(p)?"icea2dm20xo_hh":(p->qdrx==0)?"icea2dm20x_hh6":"icea2dm20x_hh", FLG_MOD|0x1);
      pic_loadfile (p,"iceiomb",FLG_JVM|0x1);
      if (j!=0) a2dm20_setup(p,1,-1,-16,250e6,-30,0);	/* for PPS status and 50MHz startup */
    }
#else
    if (mtype==IOMT_A2D && mrev==20) pic_loadfile (p, isOM20(p)?"icea2dm20o_hh":(p->qdrx==0)?"icea2dm20_hh6":"icea2dm20_hh", FLG_MOD|0x1); 
#endif
    if (mtype==IOMT_DXFPQC) pic_loadfile (p, "icefpqco", FLG_MOD|0x1);
    if (mtype==IOMT_FPQCXD) pic_loadfile (p, "icefpqci", FLG_MOD|0x1);   
    if (mtype==IOMT_D2PSE)  pic_loadfile (p, "iced2pse", FLG_MOD|0x2);
    if (mtypeo==IOMT_DXNF  && p->isY) pic_loadfile (p, "icenfibr5", FLG_MOD|0x3);  /* PIC5 */
    if (mtypeo==IOMT_DXFLZR && mrev==1 && p->isY) pic_loadfile (p, "iceflzrxdr1", FLG_MOD|0x2);  /* PIC5 */
    if (mtype==IOMT_ES2D && mrev==1) pic_loadfile (p, "icees2dr1", FLG_MOD|i);
    if (mtype==IOMT_D2ES && mrev==1) pic_loadfile (p, "iced2esr1", FLG_MOD|i);
    if (mtypeo==IOMT_DXSNT && mrev==2) pic_loadfile (p, "icesntr2", FLG_MOD|0x3); /* seem to need both sides */
    if (mtypeo==IOMT_DXSNT && mrev==4) pic_loadfile (p, "icesntr4", FLG_MOD|0x3);
    if (mtypeo==IOMT_DXSNT && mrev==5 && p->isY) pic_loadfile (p, "icesntr5p5", FLG_MOD|0x3);  /* PIC5 */
    if (mtypeo==IOMT_DXSNT && mrev==5 && !p->isY) pic_loadfile (p, "icesntr5p4", FLG_MOD|0x3); /* PIC4 */
    if (mtypeo==IOMT_DXSNT && mrev==6) pic_loadfile (p, "icesntr6", FLG_MOD|0x2);
    if (mtypeo==IOMT_DXSNT && mrev==7) pic_loadfile (p, "icesntr7", FLG_MOD|0x2);
    if (mtype==IOMT_D2AWG && mrev==1) pic_loadfile (p, "iced2awgr1", FLG_MOD|i);
    if (mtype==IOMT_D2AWG && mrev==2) pic_loadfile (p, "iced2awgr2", FLG_MOD|i); 

    /* special case GPS to load JVM code after download */
    if (mtypeo==IOMT_GPS) {  
      mrev = pic_getGPSrev (p,i);			/* this can change crcs[] */
      stat = pic_loadfile (p, (mrev==2)?"icegpsx_d":"icegps_d", FLG_MOD|i);
      if (stat!=0 || crcs[i]==0) pic_setup_gps(p,i);	/* so double check initial */
    }  
    /* special case D2AWG to prevent output after download */
    if (mtype==IOMT_D2AWG && mrev==2 && internal==0)  {  
      pic_setup_d2awg(p,i,1,-16,125000000,0,0);
    }  
    /* special case 10GBE initialization for PING response */
    if (mtypeo==IOMT_DXTGSDDS) {
      if (mrev==3)      stat = pic_loadfile (p, "icetgsddsr3", FLG_MOD|0x2);
      else if (mrev==2) stat = pic_loadfile (p, "icetgsddsr2", FLG_MOD|0x2);
      else              stat = pic_loadfile (p, "icetgsdds", FLG_MOD|0x2);
      if (stat>0) pic_init_tgsdds (p);
    }
    if (mtypeo==IOMT_DXTGVITA) { 
      stat = pic_loadfile (p, "icetgvita", FLG_MOD|0x2);
      if (stat>0) pic_init_tgvita (p);
    } 
    /* special case initialization with state */
    if (mtype==IOMT_A2D && mrev==14) {
      pic_loadfile (p, (mode=='H')?"icea2dm14_h":"icea2dm14_d", FLG_MOD|i); 
      pic_loadfile (p,"iceiom",FLG_JVM|i);
      a2dm14_setup (p,i,0);
    }
    if (mtype==IOMT_LB2D && mrev<3) {
      stat = pic_loadfile (p, "icelbd2dr2", FLG_MOD|i);
      if (stat>0) lb2d_setup (p,i,-1,-16,10000000,0,0);
    }
    if (mtype==IOMT_LB2D && mrev==3) {
#if NEWSTUFF
      stat = pic_loadfile (p, "icelb2dm3x_h", FLG_MOD|i);
#else
      stat = pic_loadfile (p, "icelb2dm3_h", FLG_MOD|i);
#endif
      pic_loadfile (p,"iceiom",FLG_JVM|i);
      lb2dm3_setup(p,i,-1,-16,10000000,0,0);
    }
    if (mtype==IOMT_D2AWG && mrev==3) {
#if NEWSTUFF == 2
      stat = pic_loadfile (p, (p->qdrx&i)? "iced2awgm3y_hx":"iced2awgm3y_h", FLG_MOD|i);
#elif NEWSTUFF
      stat = pic_loadfile (p, (p->qdrx&i)? "iced2awgm3_hx":"iced2awgm3x_h", FLG_MOD|i);
#else
      stat = pic_loadfile (p, (p->qdrx&i)? "iced2awgm3_hx":"iced2awgm3_h", FLG_MOD|i);
#endif
      pic_loadfile (p,"iceiom",FLG_JVM|i);
      /* dont know why this is required here */
      if (p->qdrx&i) { d2awgm3_setup(p,i,1,-16,250000000,-100,0); pic_enable_module(p,i,1); udelay(10000); pic_enable_module(p,i,0); }
    }
    /* special Case-SDDS Vlan/Non-Vlan Download init for PING response */
    if (mtypeo==IOMT_DXSDDS) { 
      if (mrev==5) {
        pic_jtag (p, i, JTAG_CMD_IDCODE, &stat, -32, 0);
        if (stat==0x028120DD) mrev=6;
      }
      modvlan=sdds_vlan(p,i);
      /* VLAN difference keeps us from downloading both together */
      diff = !same || (i==2 && (modvlan==-1)!=(sdds_vlan(p,1)==-1));
      sprintf(name,"icesddsr%d",mrev);
      if (modvlan!=-1) strcat(name,"v");
      if (diff) stat = pic_loadfile (p, name, FLG_MOD|i);
      else if (i==1) stat = pic_loadfile (p, name, FLG_MOD|3); /* load both */
      else stat = 0;
      pic_setup_sdds(p,i,0,1000000,16); /* default to input setup */
      if (stat>0) wait = 5;
    }
    /* insure start up in disabled state */
    pic_enable_module(p,i,0); 

    /* special AWGTONE test enable */
    if (mtype==IOMT_D2AWG && mrev==3 && findintflag("AWGTONE",p->config)>0)  {  
      d2awgm3_setup(p,i,1,-16,125000000,0,0);
      pic_enable_module(p,i,1); 
    }  
  }
  if (p->isY) {
    if (p->mtype[1]==IOMT_DIODE) rios = ((PPC_SFP_ENABLE<<4)|PPC_SFP_ENABLE) ^ PPC_RIO_DIODE;
    else rios ^= ((p->rev==15)? PPC_RIO_TRXPOD : PPC_RIO_TRXPIC);
    i = findintflag("TRXPOL",p->config);
    if (i>0) rios ^= (i<<16);
    if (p->type==ICEPIC6) pic_msg (p,0,PKTF_BUS_WR,PPC_RIO_CTL,&rios,4,0,-1);
    rios = findintflagdef("RIOS",p->config,0x3311);
    if (p->type==ICEPIC6) pic_msg (p,0,PKTF_BUS_WR,PPC_RIO_STA,&rios,4,0,-1);
    udelay(10000);
    getIOCsig(p,0);
    setIOCsig(p,IOC_FLG_RESET,p->iocc); /* reset output hypertransport circuits */
    /* for in|out to make sure input clock is reset and running for holdoff logic */
    for (i=0; i<2; i++) if (p->socc[i]=='H' && p->mtype[i]!=IOMT_NONE) {
      test_memory(p,JTAG_PORT_IOOFF+i+1,MTEST_MB2IO,0);
      test_memory(p,JTAG_PORT_IOOFF+i+1,MTEST_IO2MB,0);
    }
  }
  i = findintflag("IOMWAIT",p->config); if (i>=0) wait = i;
  if (wait>0) {
    if (i<0) print("Pause %d sec for I/O module link discovery ... see IOMWAIT flag.\n",wait);
    udelay(wait*1000000);
  }
  return status;
}

int_4 pic_sniff (PICSTRUCT *p, int_4 flags)
{
  int_4 i,status,count,mcsr,rev;
  char str[80];
  CHECK_MAGIC(p);

  rev = pic_nv_read_b(p,0x08);
  if (rev>=0 && rev<=2) rev |= (ICEPIC2<<4);	/* pre version stamped EPROM */

  RD(REG_MCSR,mcsr);

  pic_lock (p, LOCK_ALLOC);	/* dont mess up pic_acmd */
  for (i=0; i<4; i++) {
    RD(REG_MBX+i*4,status);
    count = (mcsr >> (i+16)) & 0x1;
    print("OMB%d-%x = %08x = (%d)\n",i+1,count,status,status);
  }
  for (i=4; i<8; i++) {
    RD(REG_MBX+i*4,status);
    count = (mcsr >> (i+16)) & 0x1;
    print("IMB%d-%x = %08x = (%d)\n",i-3,count,status,status);
  }
  pic_lock (p, LOCK_FREE);	/* dont mess up pic_acmd */
  
  RD(REG_MDAR,status);
  RD(REG_MDTC,count);
  print("MCSR   = %08x   ", mcsr);
  if (p->isY && (mcsr&MCSR_MASTER_DAC)) {
    RD(REG_MDARH,i);
    print("DAC Addr = %08x:%08x",i,status);
  } else {
    print("Addr = %08x",status);
  }
  print("  Count = %x = (%d)\n",count,count);

  if (mcsr&MCSR_MENA) print("Master=Enabled "); 
  else                print("Master=Disabled ");
  if (mcsr&MCSR_WIDE) print("64b "); 
  if (mcsr&MCSR_MDIR) print("CARD->PCI "); 
  else                print("PCI->CARD ");
  if (mcsr&MCSR_RPRC) print("PRC=Reset "); 
  else                print("PRC=Enabled ");
  if (mcsr&MCSR_RIOC) print("IOC=Reset "); 
  else                print("IOC=Enabled ");
  print ("\n");

  print("CARD Type=%s  ",getCardName(rev>>4));
  print("PCI ChipRev=%x ",rev&0xF);
  print("Bus=%dby ",p->pbw); 
  print("Clk=%dMHz ",p->pbr); 
  print("End=%d\n",(mcsr>>8)&0x7);

  strcpy (str,"PCI->CARD FIFO = ");
  if (mcsr&MCSR_OFIFO_FULL) strcat(str,"Full ");
  if (mcsr&MCSR_OFIFO_AFULL) strcat(str,"AFull ");
  if (!(mcsr&MCSR_OFIFO_HALF)) strcat(str,"AEmpty ");
  if (mcsr&MCSR_OFIFO_EMPTY) strcat(str,"Empty ");
  print ("%s\n",str);

  strcpy (str,"CARD->PCI FIFO = ");
  if (mcsr&MCSR_IFIFO_FULL) strcat(str,"Full ");
  if (mcsr&MCSR_IFIFO_AFULL) strcat(str,"AFull ");
  if (!(mcsr&MCSR_IFIFO_HALF)) strcat(str,"AEmpty ");
  if (mcsr&MCSR_IFIFO_EMPTY) strcat(str,"Empty ");
  print ("%s\n",str);

  if (p->dsgtype>0) {
    RD(REG_HAR0,status);
    print("NVME cmd=%02x dev=%02x blocks=%04x\n",(status>>0)&0xFF,(status>>8)&0xFF,(status>>16)&0xFFFF);
  }

  /* return status is valid PBAR */
  RD(REG_PBAR,status);
  if ( (status&0xFFFF)==0 &&
       (p->base == (int_u4)0 || p->base == (int_u4)status) ) status = 1;
  else status = -1;

  return (status);

}


int_4 pic_read (PICSTRUCT *p, int_4 offset, int_4 *value)
{
  int_4 status, func;
  if (offset&0x20000000) {	/* from Add-on */
    if (offset&0x10000000) func=ADON_WRM; 
    else if (offset&0x40000000) func=ADON_RDIOB;
    else func=ADON_RD;
    *value = pic_acmd (p, func, offset&0x1FFFFFFF, 0, 0); 
  } else {			/* from PCI */
    if (offset==REG_FIFO) {
      do { udelay(1); RD(REG_MCSR,status); } 
      while (status&MCSR_IFIFO_EMPTY);
    }
    RD(offset,*value);
  }
  return (*value);
}

int_4 pic_write (PICSTRUCT *p, int_4 offset, int_4 *value)
{
  int_4 status, func, mask=0;
  if (offset&0x20000000) {	/* from Add-on */
    if (offset&0x10000000) { func=ADON_WRM; mask=-1; }
    else if (offset&0x40000000) { func=ADON_WRIOC; 
	if ((offset&0xFF)==IOC_DIV) mask=2;
	if ((offset&0xFF)==IOC_A || (offset&0xFF)==IOC_B) mask=3;
    }
    else func=ADON_WR;
    status = pic_acmd (p, func, offset&0x1FFFFFFF, *value, -1); 
  } else {			/* from PCI */
    if (offset==REG_FIFO) {
      do { udelay(1); RD(REG_MCSR,status); } 
      while (status&MCSR_OFIFO_FULL);
    }
    WR(offset, *value);
    status = *value;
  }
  return (status);
}

int_4 pic_writem (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 mask)
{
  int_4 status;
  if (offset&0x20000000) {	/* from Add-on */
    status = pic_acmd (p, ADON_WRM, offset&0x1FFFFFFF, *value, mask); 
  } else {			/* from PCI */
    RD(offset, status);
    status = (status&(~mask)) | (*value);
    WR(offset, status);
  }
  return (status);
}

int_4 pic_rapp (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 flag)
{
  if (flag&FLG_IOC) {
    offset = IOC_UREG;
    if (p->pindex==2) offset |= IOC_B; else offset |= IOC_A;
    *value = pic_acmd (p, ADON_RDIOC, offset, 0, 0); 
  } 
  else if (flag&FLG_MOD) {
  } 
  else if (flag&FLG_FC) {
  }
  return *value;
}

int_4 pic_wapp (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 flag)
{
  int_4 status=0;
  if (flag&FLG_IOC) {
    offset = IOC_UREG;
    if (p->pindex==2) offset |= IOC_B; else offset |= IOC_A;
    status = pic_acmd (p, ADON_WRIOC, offset, *value, 0); 
    vprint("App write IOC %x at %x = %x\n",*value,offset,status);
  } 
  else if (flag&FLG_MOD) {
  } 
  else if (flag&FLG_FC) {
  }
  return status;
}

int_4 pic_rfifo (PICSTRUCT *p, int_4 *value, int_4 bytes, int_4 flags)
{
  int_4 status, i=0, reg=REG_FIFO;
  while (i<bytes) {
    status = io_ctl (p, IOCTL_READ, reg, value, bytes-i);
    if (status>0) { i+=status; value+=(status>>2); }
    else if ((flags&FLG_NOWAIT) || UBREAK) return (i); 
    else udelay (0);
  }
  return (i);
}

int_4 pic_wfifo (PICSTRUCT *p, int_4 *value, int_4 bytes, int_4 flags)
{
  int_4 status, i=0, reg=REG_FIFO;
  while (i<bytes) {
    status = io_ctl (p, IOCTL_WRITE, reg, value, min(4,bytes-i));
    if (status>0) { i+=status; value+=(status>>2); }
    else if ((flags&FLG_NOWAIT) || UBREAK) return (i); 
    else udelay (0);
  }
  while (flags&FLG_SYNC) {
    RD(REG_MCSR,status);
    if (status&MCSR_OFIFO_EMPTY) break;
    udelay(100);
  }
  return (i);
}

/*
  PKTF Message structure:
  
  HEAD Struct {
    int_2 func;		- function number 1=busr 2=busw 3=memr 4=memw ...
    int_2 size;		- number of bytes in transfer
    int_4 addr;		- address for transfer
  }

*/

int_4 pic_msg (PICSTRUCT *p, int_4 node, int_4 func, int_4 addr, int_4 *data,
		int_4 sbytes, int_4 rbytes, int_4 timeout)
{
  int_4 a,d,jvm,sel,size;
  int_4 head[2];
  /* check for functions not supported by processor */
  jvm = 0;
  if (node==0) jvm = (p->k7!=0);
  else if (node>10) jvm = (p->pmtype[node-11]>=PMT_ZPPM);
  else if (node>6) jvm = (p->pmtype[node-7]>=PMT_ZPPM);
  if (jvm) {
    if (func==PKTF_MEM_CAL) {
      a = (addr>>0)&0xFF;
      d = (addr>>8)&0xFF;
      return calibMem(p,node,data,a,d);
    }
    else if (func==PKTF_MEM_TEST) {
      return calibMem(p,node,data,0,-1);
    }
  }
  else if (node>=3 && node<=4) { /* IOM */
    if (func==PKTF_BUS_RWM) {
      iom_init(p,node);
      return iom_rwm(p,node-2,addr,data,rbytes);
    }
    else print("Unsupported IOM message node=%d type=%d\n",node,func);
  }
  /* bypass msg overhead on pic8 cards */
  if (p->type==ICEPIC8 && func==PKTF_BUS_WR) {
    if (node==JTAG_PORT_PMALL) node=3;
    if (node>10 && node<13) node-=10;
    if (node<0 || node>3) printf("Unexpected pic_msg node=%d\n",node);
    sel = node<<16;
    pic_lock (p,LOCK_ALLOC);
    rbytes = pic_acmd(p,ADON_WRB,addr,*data,sel|sbytes);
    if (sbytes>4) pic_wfifo (p, data, sbytes, FLG_SYNC);
    pic_lock (p,LOCK_FREE);
    return sbytes;
  }
  size = (rbytes>0)? rbytes:sbytes;
  head[0] = func | (size<<16);
  head[1] = addr;
  if (timeout == -1) timeout = 64;
  return pic_sendrecv (p,node,head,data,sbytes,rbytes,PKT_ACMD,timeout);
}

int_4 badrcnt=0;

/*
  Node message structure:
    Key[4]	- low 24 bits is size of data section, upper is 0x55=send 0xAA=recv
    Node[4]	- port/module address
    Header[0|8]	- RMIF|ACMD header
    Data[0-N]	- command specific data
    CRC[4]	- checksum

    ACMD  = cmd[2], size[2], addr[4]
    RMIF  =  

    I/O module nodes 1&2 are not supported
*/
int_4 pic_sendrecv (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data,
		int_4 sbytes, int_4 maxbytes, int_4 flags, int_4 timeout)
{
  int_4 i, j, retry=timeout, key=0,keyf, rhead[4], rtail[4], sel, inode, qbytes=-1, rbytes=-1, chksum, usefifo;

  if (node==0 && p->isX) { if (++depcnt<3) print("Illegal node=0 in pic_send/recv, use default node=-1\n"); node=-1; }
  if (node<0) node = JTAG_PORT_PMOFF + p->pmi;
  if (node>=1 && node<=p->mpm) node += JTAG_PORT_PMOFF;
  if (node>JTAG_PORT_PMOFF && p->pmtype[node-JTAG_PORT_PMOFF-1]<=0) return 0;
  if (node==3) node=JTAG_PORT_PMALL;
  if (node>0 && node<JTAG_PORT_PMOFF) { vprint("Unsupported port=%d in pic_sendrecv()\n",node); return 0; }
  if ((flags&PKT_NOWAIT)!=0) retry = (sbytes>=0)? -1 : 0;
  sel = getJTAGsel(p,node);
  usefifo = (p->type==ICEPIC8) && (sbytes>8);
  pic_lock (p,LOCK_ALLOC);

  SEND:
  if (sbytes<0) goto RECV;
  chksum = 0;
  key = 0x55000055|(sbytes<<8);
  pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_SOCK,2);
  pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0,4); 
  pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,node,4);
  if ((flags&PKT_RMIF)!=0 || (flags&PKT_ACMD)!=0) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,head[0],4);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,head[1],4);
    if ((head[0]&0xFFFF)==PKTF_MEM_RD) qbytes = head[0]>>16;
    if ((head[0]&0xFFFF)==PKTF_BUS_RD) qbytes = head[0]>>16;
    chksum = head[0]^head[1];
  }
  /* all socket transfers occur on 4 byte boundaries */
  for (i=0,j=sbytes; j>0; i++,j-=4) {
    chksum ^= data[i];
    if (!usefifo) pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,data[i],4);
  }
  if (usefifo) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0,sbytes);
    pic_wfifo (p, data, sbytes, FLG_SYNC);
  }
  pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,chksum,4); /* msg chksum */
  pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_SOCK,2);
  pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,key,4); /* msg valid head key */

  if ((flags&PKT_NOWAIT)!=0) goto DONE;

  RECV:
  if (maxbytes<0) goto DONE; /* no wait for response */
  for ( ;retry>=0; retry--) { /* wait for response */
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_RD,PPC_BADDR_SOCK,2);
    key = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
    keyf = key&0xFF0000FF;
    if (keyf==0xAA0000AA) break; /* response ready */
    if (keyf==0x99000099 || key==0) {      /* chksum error */
      retry--;
      print("Bad PIC Node=%d checksum key=%08x in send/recv of %d/%d bytes() retry=%d\n",node,key,sbytes,qbytes,retry);
      if (sbytes>=0 && retry>0) goto SEND;
      goto DONE;
    }
    if ((flags&PKT_NOWAIT)!=0) goto DONE;
    if (retry==0) {
      if (++badrcnt>32) goto DONE;
      j = (sbytes+3)&0x00FFFFFC;
      print("Bad PIC Node=%d response key=%08x in send/recv of %d/%d bytes()\n",node,key,sbytes,qbytes);
      print(" OutBuf = 55%04x55 %08x %08x %08x ...\n",sbytes,node,head[0],head[1]);
      pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_RD,PPC_BADDR_SOCK,2);
      for (i=0; i<4; i++) rhead[i] = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
      pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_RD,PPC_BADDR_SOCK+16+j-4,2);
      for (i=0; i<4; i++) rtail[i] = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
      print(" MsgBuf = %08x %08x %08x %08x ... %08x %08x %08x %08x\n",
        rhead[0],rhead[1],rhead[2],rhead[3],rtail[0],rtail[1],rtail[2],rtail[3]);
      goto DONE;
    }
    udelay(1000);
  }
  rbytes = (key&0x00FFFF00)>>8;
  if (qbytes>=0 && rbytes>=0 && rbytes!=qbytes) 
    print("Bad PIC Node=%d response key=%08x rby=%d qby=%d sby=%d\n",node,key,rbytes,qbytes,sbytes);
  if (rbytes==0 && maxbytes<0) goto DONE; /* no response data */
  inode = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
  if ((flags&PKT_RMIF)!=0 || (flags&PKT_ACMD)!=0) {
    head[0] = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
    head[1] = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
  }
  /* all socket transfers occur on 4 byte boundaries */
  if (rbytes>maxbytes) rbytes=maxbytes;
  for (i=0,j=rbytes; j>0; j-=4) data[i++] = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);

  /* all done? set high efficiency Key for processor */
  if (rbytes >= qbytes) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_SOCK,2);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0,4); 
  }

  DONE:
  pic_lock (p,LOCK_FREE);
  return rbytes;
}

int_4 pic_send (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data, int_4 bytes, int_4 flags)
{
  int_4 status = pic_sendrecv (p,node,head,data,bytes,0,flags,64);
  if (status==0) status = bytes; /* successful send only */
  return status;
}

int_4 pic_recv (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data, int_4 maxbytes, int_4 flags)
{
  int_4 status = pic_sendrecv (p,node,head,data,-1,maxbytes,flags,8);
  return status;
}


#ifdef _XMIDAS

#if _XMVER >= 400
#define LPH lph.p
#else
#define LPH lph
#endif

int_4 pic_file (PICSTRUCT *p, HEADER *hcb, int_4 *nstart, int_4 *nbytes, int_4 mode) 
{
  int_4 status; DMAMAP map; ICEHDR *ihcb = (ICEHDR*)hcb;
  map.paddr = phys2long(*nstart); map.bytes = phys2long(*nbytes); map.offset = 0; map.vaddr = (char *)ihcb->vaddr;
  status = pic_mapfile (p,&map,hcb,mode);
  *nstart = long2phys(map.paddr); *nbytes = long2phys(map.bytes); ihcb->vaddr = (int_8)map.vaddr;
  return status;
}
int_4 pic_mapfile (PICSTRUCT *p, DMAMAP *map, HEADER *hcb, int_4 mode)
{
  int_4 status;
  CHECK_MAGIC(p);

  if (mode > 0) {				/* allocate */

    map->offset = 0;
    map->bytes = (int_8)(hcb->last_byte-hcb->first_byte+.1);
    if (sizeof(map->vaddr)==4) map->vfill = 0;
    map->devn[0] = 0;

    /* RAMDISK file */
    if (mode!=2 && hcb->LPH != 0 && hcb->disk_blk != 0) { 	
      map->paddr = ((int_8)hcb->disk_blk)*512 + (int_8)hcb->first_byte;
      map->vaddr = (char *)(hcb->LPH + (int)hcb->first_byte);
      hcb->mode = 1;
    }
    else {
      if (mode==2) status = -1;
      else status = pic_mapmem (p, map, 0, 1); 
      if (status<0) {
        print("Cannot map file - using direct PCI FIFO access (slow)\n");
        map->vaddr = (char *)malloc((int_4)map->bytes);
        map->paddr = 0;
        hcb->mode = 3;
      } else {
        hcb->mode = 2;
      }
    }

  } else {					/* deallocate */

    if (hcb->mode == 1) {
      pic_unmapcheck (p, long2phys(map->paddr), long2phys(map->bytes));
    }
    else if (hcb->mode == 2) {
      status = pic_mapmem (p, map, 0, -1);
      hcb->LPH = 0;
    }
    else if (hcb->mode == 3) {
      free ( (void *)map->vaddr );
    }

  }
  return (hcb->mode);
}
#endif

char *pic_mapram (int_8 offset, int_8 size)
{
  PICSTRUCT ph,*p=&ph; int_8 base;
  pic_open(p,"ICERAM",NULL,0);
  base = phys2long(io_ctl(p,IOCTL_QALLOC,0,0,0));
  pic_close(p);
  return picmopenmap(base+offset,size,"iceram");
}
void pic_unmapram (int_8 offset, int_8 size, char *buf)
{
  PICSTRUCT ph,*p=&ph; int_8 base;
  pic_open(p,"ICERAM",NULL,0);
  base = phys2long(io_ctl(p,IOCTL_QALLOC,0,0,0));
  pic_close(p);
  picmclosemap(base+offset,size,buf);
}

int_4 pic_map (PICSTRUCT *p, int_4 **vaddr,  
	int_4 *paddr, int_4 psize, int_4 mode)
{
  int_4 status; DMAMAP map; 
  map.vaddr = (char *)(*vaddr); map.paddr = phys2long(*paddr); map.offset = 0;
  status = pic_mapmem(p,&map,psize,mode);
  *vaddr = (int_4 *)map.vaddr; *paddr = long2phys(map.paddr); 
  return status;
}

int_4 pic_mapmem (PICSTRUCT *p, DMAMAP *map, int_4 psize, int_4 mode)
{
  int_4 status,nalloc,paddr;

  /* paddr & psize are the int_4 packed versions for ioctl args only */
  if (psize>0) map->bytes = phys2long(psize);
  else if (psize<0) {
    map->bytes = (int_8)(-psize)*4096;
    psize = long2phys(map->bytes);
  }
  else psize = long2phys(map->bytes);
  paddr = long2phys(map->paddr);

  /* mode=5 uses the RAMDISK (beg of ICERAM) for all allocations for special installations */
  if (mode==5) { map->paddr = phys2long(io_ctl(p,IOCTL_QALLOC,0,0,0))+map->offset; map->offset=0; map->vbytes=map->bytes; mode=4; }
  if (mode==-5) { mode=-4; }

  if (mode > 0) {
    if (mode<3) map->paddr = 0; 
    if (mode<4) map->offset = 0; 
    if (mode!=2) map->vaddr = 0; 
    if (sizeof(map->vaddr)==4) map->vfill = 0;
  } else {
    if (mode>-3) pic_unmapcheck (p, paddr, psize);
  }

#if _LINX
  if (mode > 0) {
    if (mode < 3) {
      status = io_ctl (p, IOCTL_MAP, 0, 0, psize);
      if (status==0) return (-1); 
      map->paddr = phys2long(status);
      map->devn[0] = 0;
    }
    if (mode==4) map->vaddr = (char *) picmopenmap (map->paddr+map->offset, map->vbytes, map->devn);
    else         map->vaddr = (char *) picmopenmap (map->paddr, map->bytes, map->devn);
    if (map->vaddr==0) return -1;
  }
  else if (mode < 0 && map->paddr!=0) {
    if (mode>-3) status = io_ctl (p, IOCTL_UNMAP, paddr, (int_4 *)map->vaddr, psize);
    if (mode==-4) picmclosemap (map->paddr+map->offset, map->vbytes, map->vaddr);
    else          picmclosemap (map->paddr, map->bytes, map->vaddr);
  }
#elif _MAC
  if (mode > 0) {
    map->vaddr = macmallocmap(map->bytes);
    if (map->vaddr == NULL) return(-1);
    map->paddr = macmopenmap(&map->vaddr,map->bytes);
  }
  else if (mode < 0) {
    if (map->paddr!=0) macmclosemap(&map->vaddr);
  }
#elif _WIN
  if (mode > 0) {
    status = io_ctl (p, IOCTL_MAP, 0, 0, psize);
    nalloc = p->ioctl.bytes;
    if (status==0 || nalloc==0) return (-1);
    map->paddr = phys2long(status);
    map->vaddr = (char *) p->ioctl.bufaddr;
  }
  else if (mode < 0) {
    if (map->paddr!=0) status = io_ctl (p, IOCTL_UNMAP, paddr, (int_4 *)map->vaddr, psize);
  } 
#else
  if (mode > 0) {
    nalloc = pagealloc (map->bytes,&map->vaddr,mode);
    if (nalloc<=0) return -1;
    status = io_ctl (p, IOCTL_MAP, 0, (int_4 *)map->vaddr, psize);
    if (status==0) { pagealloc (map->bytes,&map->vaddr,-mode); return (-1); }
    map->paddr = phys2long(status);
  }
  else if (mode < 0) {
    if (map->paddr!=0) status = io_ctl (p, IOCTL_UNMAP, paddr, (int_4 *)map->vaddr, psize);
    pagealloc (map->bytes,&map->vaddr,mode); /* deallocates memory */
  }
#endif
  return (1);
}
  

int_4 pic_mapcheck (PICSTRUCT *p, int_4 paddr, int_4 psize) 
{
  int_4 status;
#if _LINX|_WIN
  status = io_ctl (p, IOCTL_QMAP, paddr, 0, psize);
#else
  status = io_ctl (p, IOCTL_QMAP, paddr, 0, 0); 
#endif
  return (phys2long(status)>=phys2long(psize))? 1:0;
}

void pic_unmapcheck (PICSTRUCT *p, int_4 paddr, int_4 psize) 
{
  int_4 n;
  int_u4 haddr1 = P2HW(paddr);
  int_u4 haddr2 = P2HW(psize) + haddr1;
  for (n=0; n<DMA_CHNS; n++) 
  if (p->dma[n].haddr >= haddr1 && p->dma[n].haddr < haddr2) {
    if (p->dma[n].stat != DMA_STOP) {
	print("Stopping DMA #%d to unmap buffer\n",n+1);
	if (pic_dmafunc (p, n+1, DMA_CANCEL) == -1) {
	  print("Could not stop DMA - killing the SHARC/PPC\n");
	  WR(REG_MCSR, MCSR_RESET);
	}
    }
    p->dma[n].haddr = 0;
  }
}

int_4 pagealloc (int_8 bytes, char **vaddr, int_4 mode)
{
  int_4 nalloc;
  unsigned long tmp;

  tmp = (unsigned long)(*vaddr);
  nalloc = (int_4)( (bytes+PGSZ+PGSZ-1)&~(PGSZ-1) );
#if _VMS
  if (mode==1) tmp = (unsigned long)getpage (nalloc);
  if (mode==-1) freepage ( (char *)tmp, nalloc);
#else
  if (mode==1) tmp = (unsigned long)valloc(nalloc);
  if (mode==-1) free( (void *)tmp );
#endif
  /* print("PageMalloc b=%x bp=%x a=%x mode=%d\n",psize,nalloc,tmp,mode); */
  if (tmp==0) { print("Err allocating memory\n"); return (-1); }
  if (tmp&(PGSZ-1)) { print("Memory must be page aligned\n"); return (-1); }
  *vaddr = (char *)tmp;
  return nalloc;
}

void pic_transfer (PICSTRUCT *p, int_4 *buffer, int_4 bytes, 
			int_4 address, int_4 dir, int_4 type)
{
  int_4 wcount, ecount, ewcount, method, i;
  int_u4 *temp, *ubuffer; int_u2 *stemp;

  ubuffer = (int_u4 *)buffer;

  if (type == 0) {		/* 32x32 bit register data */
    wcount = bytes/4; 
    pic_lock (p, LOCK_ALLOC);
    if (dir==2) ConvertEndian (buffer,wcount,32);
    if (dir>0) for (i=0; i<wcount; i++) 
		  pic_acmd (p, ADON_WR, address+i, buffer[i], 0);
    else       for (i=0; i<wcount; i++) 
		  buffer[i] = pic_acmd (p, ADON_RD, address+i, 0, 0);
    pic_lock (p, LOCK_FREE);
    return;
  }
  else if (type == 1) {		/* 32x32 bit data  */
    wcount = bytes/4; 
    ecount = bytes/4;
    method = 0x0001;
    if (p->isX) method = 0x0101;
  } 
  else if (type == 2) {		/* 32x16 bit data */
    wcount = bytes/4; 
    ecount = bytes/2;
    method = 0x0041;
  } 
  else if (type == 3) {		/* 48x16 bit instructions */
    wcount = bytes/6; 
    ecount = bytes/2;
    method = 0x00a1;
  } 
  else if (type == 4) {		/* 48x32 bit instructions */
    wcount = bytes/6; 
    ecount = (bytes+2)/4;
    method = 0x00e1;
  } 
  else {
    print("Unsupported transfer type = %d\n",type);
    return;
  }
  if (dir<0) method |= 0x4;
  if (p->isX) method |= 0x00400; else method |= 0x200; /* dma master */   

  if (type==2 || type==3) { 	/* unpack 16 bit to 32 bit FIFO */
    temp=(int_u4 *)malloc (ecount*4); stemp = (int_u2 *)buffer;
    if (dir>0) for (i=0; i<ecount; i++) temp[i] = stemp[i]; 
    if (dir==2) ConvertEndian ((int_4 *)temp,ecount*2,16);
    bytes *= 2;
  } else { 			/* 32 bit constant data */
    temp = ubuffer;
    if (dir==2) ConvertEndian ((int_4 *)temp,ecount,32);
  }

  if (ecount>=0x10000 || wcount>=0x10000) {
    print("pic_transfer ecount=%d or wcount=%d too large\n",ecount,wcount);
    return;
  }

  ewcount = ((ecount<<16)&0xFFFF0000) | (wcount&0xFFFF);

  pic_lock (p, LOCK_ALLOC);

  pic_acmd (p, ADON_DMA, address, ewcount, method);
  if (dir>0) pic_wfifo (p, (int_4 *)temp, bytes, FLG_XFER);
  else       pic_rfifo (p, (int_4 *)temp, bytes, FLG_XFER);

  pic_lock (p, LOCK_FREE);

  if (type==2 || type==3) {	/* pack 32 bit FIFO to 16 bit */
    if (dir<0) { for (i=0; i<ecount; i++) stemp[i] = (int_u2)temp[i]; }
    free (temp);
  } else if (type==0) {		/* 32 bit constant data */
    free (temp);
  } 
}


/* 

   SHARC boot file structure
  
   256 W Boot Loader Code	(Loader Init 256 @ IMS)

     1 W Segment Type
     1 W Segment addr/size   0xAAAAAAAASSSS     
     N W Segment data

     1 W Final Init Type
     1 W Instruction for 0x20040
   256 W Load over boot loader	(Final Init 256 @ IMS)

*/

#define SEG_FINAL_INIT 0x0
#define SEG_ZERO_DM16  0x1
#define SEG_ZERO_DM32  0x2
#define SEG_ZERO_DM40  0x3
#define SEG_INIT_DM16  0x4
#define SEG_INIT_DM32  0x5
#define SEG_INIT_DM40  0x6
#define SEG_ZERO_PM16  0x7
#define SEG_ZERO_PM32  0x8
#define SEG_ZERO_PM40  0x9
#define SEG_ZERO_PM48  0xa
#define SEG_INIT_PM16  0xb
#define SEG_INIT_PM32  0xc
#define SEG_INIT_PM40  0xd
#define SEG_INIT_PM48  0xe

#define SEG_PASSUP  0x99

int_4 load2i (int_u1 *loadbyte) {
  int_4 i,j; int_u1 *b = (int_u1*)(&i);
  for (j=0; j<4; j++) b[j]=loadbyte[j];
  return i;
}
  
int_4 pic_loadsharc (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  status,n,i,j,size,nldr,retry=0,maxretry=100;
  int_4  count, address, type, *buffer, bootload[256*6];

  /* init the code word pointer */
  n = 256;		/* the AD bootloader is not used in the ICE cards */
  nldr = nprog-256;	/* this contains the special ICE bootloader */
  nprog -= 258;		/* it does not need to be re-loaded */
  flags |= p->flags;

  vprint ("Loading "); 

  /* handle bootstrapping the loader code */
  if (flags&FLG_BOOT) {
    count = 256;
    /* format the EPROM bytes into low byte of FIFO words */
    for (i=0; i<count*6; i++) bootload[i] = loadbyte[i+nldr*6];
    RETRY:
    /* reset the SHARC and any flags we need to monitor */
    WR(REG_MCSR, MCSR_RESET);
    for (i=0; i<6; i++) WR(REG_FIFO, 0); 	
    WR(REG_MCSR, MCSR_RESET);
    udelay(10); /* reset for 10+us */
    if (retry>0) { /* JEFF only if PLL problem */
      WR(REG_MCSR, (MCSR_RESET|MCSR_FLG0)); 
      udelay(10);	/* remove clock for 10+us */
      WR(REG_MCSR, MCSR_RESET);
      udelay(50);	/* allow PLL to stabilize */
    }
    /* enable the adon SHARC processor for boot load */
    WR(REG_MCSR, MCSR_ENABLE);
    vprint ("/boot/ "); 		/* download bootstrap code */
    for (i=j=0; i<count*6; ) {
      status = io_ctl (p, IOCTL_WRITE, REG_FIFO, bootload+i, (count*6-i)*4);
      if (status>0) i+=(status>>2); 
      else if (UBREAK || ++j>100) { WR(REG_MCSR, MCSR_RESET); return -1; }
      else udelay(10000);
      if (j>2 && i<200 && ++retry<maxretry) goto RETRY;
    }
    vprint ("O."); 
    for (j=0;;) {
      RD(REG_MCSR,status);  	/* wait for SHARC to initialize */	
      if ( (status&MCSR_IMB1F) != 0x0) break; 
      if (UBREAK || ++j>100) { WR(REG_MCSR, MCSR_RESET); return -1; }
      udelay(10000);
    }
    RD(REG_IMB1, status);
    vprint ("K. "); 
  }

  if (retry>=maxretry) goto BAIL;

  /* verify the boot code */
  verify48(p,loadbyte+(nldr*6),256,0);
  if ((flags&FLG_TEST)!=0) return -1;

  /* now load the code fragments as issued */
  while (n<nprog && !UBREAK) {
    i = n*6;
    type = loadbyte[i];
    if (loadbyte[i+2]==42) type = SEG_PASSUP;
    count = loadbyte[i+6] | (loadbyte[i+7]<<8);
    address =   (loadbyte[i+8])      | (loadbyte[i+9]<<8) | 
		(loadbyte[i+10]<<16) | (loadbyte[i+11]<<24);
    n += 2; i = n*6;
    if (type == SEG_FINAL_INIT) { count = 256; address = IMS; }
    if (type == SEG_PASSUP) { count = 5; address = 0x00000; }
    vprint ("%x/%d/0x%x ", type,count,address);

    if (type == SEG_INIT_PM48 || type == SEG_FINAL_INIT) {
      pic_transfer (p, (int_4 *)(loadbyte+i), count*6, address, 2, 3);
      verify48(p,loadbyte+i,count,address);
      n += count;
    } 
    else if (type == SEG_ZERO_PM48) {
      size = count*8; buffer=(int_4 *)malloc(size);
      for (j=0; j<count*2; j++) buffer[j]=0;
      pic_transfer (p, buffer, count*6, address, 2, 3);
      verify48(p,(int_u1*)buffer,count,address);
      free (buffer);
    } 
    else if (type == SEG_INIT_PM32) {
      size = count*4; buffer=(int_4 *)malloc(size);
      /* extract 32 bit words from the 48 bit loader words (skip 1st 2 bytes) */
      for (j=0; j<count; j++) buffer[j]=load2i(loadbyte+i+(j*6)+2); 
      pic_transfer (p, buffer, count*4, address, 2, 1);
      verify32(p,buffer,count,address);
      free (buffer);
      n += count;
    }
    else if (type == SEG_ZERO_PM32) {
      size = count*4; buffer=(int_4 *)malloc(size);
      for (j=0; j<count; j++) buffer[j]=0;
      pic_transfer (p, buffer, count*4, address, 2, 1);
      verify32(p,buffer,count,address);
      free (buffer);
    }
    else if (type == SEG_INIT_DM32) {
      size = count*4; buffer=(int_4 *)malloc(size);
      /* extract 32 bit words from the 48 bit loader words (skip 1st 2 bytes) */
      for (j=0; j<count; j++) buffer[j]=load2i(loadbyte+i+(j*6)+2);
      pic_transfer (p, buffer, count*4, address, 2, 1);
      verify32(p,buffer,count,address);
      free (buffer);
      n += count;
    }
    else if (type == SEG_ZERO_DM32) {
      size = count*4; buffer=(int_4 *)malloc(size);
      for (j=0; j<count; j++) buffer[j]=0;
      pic_transfer (p, buffer, count*4, address, 2, 1);
      verify32(p,buffer,count,address);
      free (buffer);
    } 
    else if (type == SEG_PASSUP) {
      n += count;
    } 
    else {
      vprint ("UH OH !!! \n");
      return (-1);
    }
  }
  BAIL:
  vprint ("done\n"); 
  if (retry>0) print("Retry reset %d times on card %d\n",retry,p->devno);
  if (retry>=maxretry) return -1;
  return (nprog);
}

int_4 verify32 (PICSTRUCT *p, int_4 *buffer, int_4 nprog, int_4 address)
{
  int_4 i,word;
  if (p->verbose!=2) return 0;
/*  print("Verifying %d 32bit code words at %x\n",nprog,address); */
  for (i=0; i<nprog; i++) {
    word = pic_acmd (p,ADON_RD,IMS|(address+i),0,0); 
    if (word!=buffer[i]) print("Bad data=%08x buf=%08x at %d\n",word,buffer[i],i);
  }
  return 0;
}

int_4 verify48 (PICSTRUCT *p, int_u1 *buffer, int_4 nprog, int_4 address)
{
  int_4 i,k,pmu,pml;
  if (p->verbose!=2) return 0;
  if (p->isX) return 0; /* this ends up in memory space that is not verifiable */
/*  print("Verifying %d 48bit code words at %x\n",nprog,address); */
  for (i=0; i<nprog; i++) { k=i*6;
    pmu =  pic_acmd (p,ADON_RD,IMS|(address+i),0,0);
    pml = (pic_acmd (p,ADON_RD,IMS|(address+i+0x1000),0,0)>>16) & 0xFFFF;
    if (buffer[k+5]!=((pmu>>24)&0xFF) || buffer[k+4]!=((pmu>>16)&0xFF) ||
        buffer[k+3]!=((pmu>>8)&0xFF) || buffer[k+2]!=(pmu&0xFF) ||
        buffer[k+1]!=((pml>>8)&0xFF) || buffer[k+0]!=(pml&0xFF) )
    print("Bad data=%08x%04x buf=%02x%02x%02x%02x%02x%02x at %d\n",pmu,pml,
	buffer[k+5],buffer[k+4],buffer[k+3],
	buffer[k+2],buffer[k+1],buffer[k+0],i);
  }
  return 0;
}

int_4 pic_ram (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 addr, int_4 bytes, int_4 mode) {
  int_4 i, j, k, sel, status, gdata, *data = (int_4*)loadbyte; char *gc = (char*)(&gdata);
  sel = getJTAGsel(p, port);
  pic_lock (p, LOCK_ALLOC);
  if (mode>=0) {
    if (port==0) {
      WR(REG_IOC, REG_IOC_ADDR|PPC_BADDR_WR);
      WR(REG_IOC, (addr&0xFF));
      WR(REG_IOC, REG_IOC_ADDR|PPC_BADDR_WR|1);
      WR(REG_IOC, (addr>>8));
      WR(REG_IOC, REG_IOC_ADDR|PPC_BDATA);
      for (i=0; i<bytes; i++) {
        WR(REG_IOC, (loadbyte[i^0x3]&0xFF) );	/* with byte swap */
        if (mode==11 && i%4==0) RD(REG_MCSR,status); /* since the 4-byte cache line read needs the pause */
        if (mode!=11 && i%16==15) RD(REG_MCSR,status); /* since we advertise infinite PCIe credits */
      }
      RD(REG_IOC,status);
    } else {
      pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,addr,2);
      for (i=0; i<bytes/4; i++) pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,swapit(data[i]),4);
    }
  }
  if (mode<=0) {
    if (port==0) {
      WR(REG_IOC, REG_IOC_ADDR|PPC_BADDR_RD);
      WR(REG_IOC, (addr&0xFF));
      WR(REG_IOC, REG_IOC_ADDR|PPC_BADDR_RD|1);
      WR(REG_IOC, (addr>>8));
      WR(REG_IOC, REG_IOC_ADDR|PPC_BDATA);
      for (i=j=0; j<bytes; i++,j+=4) {
        RD(REG_IOC,k); gc[3] = k&0xFF;
        RD(REG_IOC,k); gc[2] = k&0xFF;
        RD(REG_IOC,k); gc[1] = k&0xFF;
        RD(REG_IOC,k); gc[0] = k&0xFF;
	if (mode==-1) data[i] = gdata;
        else if (data[i]!=0) print("Readback at %x  got=%08x put=%08x\n", addr+j,gdata,data[i]);
      }
    } else {
      pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_RD,addr,2);
      for (i=j=0; j<bytes; i++,j+=4) {
        status = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4); 
        gdata = swapit(status);
        if (mode==-1) data[i] = gdata;
        else if (mode==-3) { if (gdata!=0) print("Readback at %x  got=%08x \n", addr+j,gdata); }
        else if (data[i]!=0) print("Readback at %x  got=%08x put=%08x\n", addr+j,gdata,data[i]);
      }
    }
  }
  pic_lock (p, LOCK_FREE);
  return 0;
}

/* 
  PPC boot file structure (pure ELF format - bigendian)
*/
#define MAXZEROS 0x200

int_4 pic_loadppc (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4 status,n,i,j,k,sel,size,addr,flag,doswap=0,dcache=0,vaddr,eoff,efs,ems,poff,hbt;
  ELFHDRSTRUCT elfhdr;
  ELFSECSTRUCT elfsec;
  ELFPGMSTRUCT elfpgm,elfpgm2;
  int_u1 zeros[MAXZEROS];

  for (i=0; i<MAXZEROS; i++) zeros[i]=0;
#if !_IEEE
  doswap = 1;
#endif
  if (port==1||port==2||port==3) port += JTAG_PORT_PMOFF;
  sel = getJTAGsel(p, port);

  /* reset PPC */
  if (port==0) {
    WR(REG_MCSR, MCSR_RESET);
    for (i=0; i<8; i++) WR(REG_FIFO, 0); 	
    WR(REG_MCSR, MCSR_RESET);
    WR(REG_MCSR, MCSR_RPRC);
  } else {
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_PPC,0,1); 
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_DMAC,0,1); 
  }
  if (nprog<=0) goto STARTUP;

  memcpy((void*)(&elfhdr),(void*)loadbyte,sizeof(elfhdr));
  if (doswap) Swap((void*)&elfhdr.e_version,5,32);
  if (doswap) Swap((void*)&elfhdr.e_ehsize,6,16);
  if (elfhdr.e_ident[1]!='E' || elfhdr.e_ident[2]!='L' || elfhdr.e_ident[3]!='F') {
    print("PPC load must be an ELF file\n");
    return -1;
  }
  vprint("ELF port=%d PSECs=%d Offset=%d PHdrSize=%d \n",port,elfhdr.e_phnum,elfhdr.e_phoff,elfhdr.e_phentsize);
  if (p->verbose==3) flag=0; else flag=1;

  /** loop through the program headers */
  elfpgm2.p_memsz = 0; /* init direct cache load */
  elfpgm2.p_offset = 0; /* init direct cache load */
  for (i=0; i<elfhdr.e_phnum; i++) {
    memcpy ((void*)(&elfpgm),(void*)(loadbyte+elfhdr.e_phoff+i*elfhdr.e_phentsize),sizeof(elfpgm));
    if (doswap) Swap((int_4*)&elfpgm,8,32);
    eoff = elfpgm.p_offset;
    efs  = elfpgm.p_filesz;
    ems  = elfpgm.p_memsz;
    vaddr = elfpgm.p_vaddr & 0x0FFFFFFF; /* all code is in 0xF0000000 space - fun with unsigned int */
    v2print("ELF Program=%d Type=%d VAddr=%x Paddr=%08x Offset=%08x FilSz=%08x MemSz=%08x\n",
		i,elfpgm.p_type,elfpgm.p_vaddr,elfpgm.p_paddr,eoff,efs,ems);
    if (ems-efs>MAXZEROS) {
      print("UhOh - too many zeros=%d to insert ELF program - see MAXZEROS\n",ems-efs);
    }
    else if (vaddr>=0x0d000000 && vaddr+ems<=0x0d001000) {
      poff = vaddr - 0x0d000000; 
      if (efs>0) pic_ram (p, port, loadbyte+eoff, PPC_BADDR_DSOCM+poff, efs, flag);
      if (ems>efs) pic_ram (p, port, zeros, PPC_BADDR_DSOCM+poff+efs, ems-efs, flag);
    }
    else if (vaddr>=0x0e000000 && vaddr+ems<=0x0e008000) {
      elfpgm2 = elfpgm; dcache = (ems+31)/32+1; /* number direct-to-cache lines +1 kludge */
    }
    else if (vaddr>=0x0ffff000 && vaddr+ems<=0x10000000) {
      poff = vaddr - 0x0ffff000; 
      if (efs>0) pic_ram (p, port, loadbyte+eoff, PPC_BADDR_ISOCM+poff, efs, 1);
      if (ems>efs) pic_ram (p, port, zeros, PPC_BADDR_ISOCM+poff+efs, ems-efs, 1);
    }
    else
      print("Problem Loading ELF PSec=%d Type=%d VAddr=%x Paddr=%x Offset=%x Size=%d %d\n",
			i,elfpgm.p_type,elfpgm.p_vaddr,elfpgm.p_paddr,eoff,efs,ems);
  }

  /** set number of direct cache lines to load */
  i = dcache; if (doswap) Swap (&i,1,32);
  pic_ram (p, port, (int_u1*)(&i), PPC_NBOOT&0xFFFF, 4, flag);
  STARTUP:

  if (port==0) {
    WR(REG_MCSR, MCSR_ENABLE);
  } else {
    /* handle bootstrapping the processor */
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_PPCGO,0,1); 
  }
  /* now download direct-to-cache code */
  if (dcache>0) pic_ram (p, port, loadbyte+elfpgm2.p_offset, PPC_BADDR_IBOOT, dcache*32, 11);

  if (port>0) for (j=(p->verbose==2)?0:3; j<4; j++) {
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|0xFFFC,2);
    hbt = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4);
    if (p->verbose>0) { print(" HBT%d=%08x",j,hbt); if (j==3) print("\n"); }
    if (j==3 && hbt==0) nprog=-1; /* processor didnt start */
  }
  return (nprog);
}

int_4 pic_dumpslr (PICSTRUCT *p, int_4 port)
{
  int_4 i,n,isOK=0,val,addr,mask,dats;
  int_4 uadr=(p->k7==7)? 0x02e00000 : 0x00D00000;
  int_4 sig=(p->k7==7)? 0x26 : 0x94;
  int_4 npage=13;
  int_4 nprog=256;
  int_1 data;
  data=0; ampc_write (p,uadr|0x01,&data,1);
  ampc_read   (p,uadr|0x02,(int_1*)(&dats),2);
  if ((dats&0xFF)!=sig) { print("Invalid Si5338 I2C signature=0x%04x expecting 0x??26\n",dats); return -1; }
  for (n=0; n<npage; n++) {
    data=n; ampc_write (p,uadr|0x01,&data,1);
  for (i=0; i<nprog; i++) {
    ampc_read   (p,uadr|i,(int_1*)(&dats),1);
    printf("Page=%x Reg=%x Value=%x\n",n,i,(dats&0xFF));
  }
  }
  return 0;
}

int_4 pic_loadslr (PICSTRUCT *p, int_4 port, int_4 *loadword, int_4 nprog)
{
  int_4 i,n,isOK=0,val,addr,mask,dats;
  int_4 uadr=0x02e00000;
  int_1 data;

  ampc_read   (p,uadr|0x02,(int_1*)(&dats),2);
  if ((dats&0xFF)!=0x26) { print("Invalid Si5338 I2C signature=0x%04x expecting 0x??26\n",dats); return -1; }
  ampc_writem (p,uadr|0xe6,0x10,0x10);
  ampc_writem (p,uadr|0xf1,0x80,0x80);
  for (n=0; n<nprog; n++) {
    val  = loadword[n];
    addr = (val>>16)&0xFF;
    data = (val>> 8)&0xFF;
    mask = (val>> 0)&0xFF;
    ampc_writem (p,uadr|addr,data,mask);
  }
  for (i=0,isOK=0; i<1000; i++) { ampc_read(p,uadr|0xda,&data,1); isOK = (data&0x04)==0; if (isOK) break; udelay(1000); }
  if (!isOK) { print("No Si5338 input clock lock\n"); return -1; }
  ampc_writem (p,uadr|0x31,0x00,0x80);
  ampc_writem (p,uadr|0xf6,0x02,0x02);
  udelay(25000);
  ampc_writem (p,uadr|0xf1,0x00,0x80);
  ampc_writem (p,uadr|0xf1,0x65,0xFF);
  for (i=0,isOK=0; i<1000; i++) { ampc_read(p,uadr|0xda,&data,1); isOK = (data&0x10)==0; if (isOK) break; udelay(1000); }
  if (!isOK) { print("No Si5338 PLL lock\n"); return -1; }
  ampc_read   (p,uadr|0xed,&data,1);
  ampc_writem (p,uadr|0x2f,data,0x03);
  ampc_read   (p,uadr|0xec,&data,1);
  ampc_writem (p,uadr|0x2e,data,0xff);
  ampc_read   (p,uadr|0xeb,&data,1);
  ampc_writem (p,uadr|0x2d,data,0xff);
  ampc_writem (p,uadr|0x2f,0x14,0xfc);
  ampc_writem (p,uadr|0x31,0x80,0x80);
  ampc_writem (p,uadr|0xe2,0x04,0x04);
  udelay(1000);
  ampc_writem (p,uadr|0xe2,0x00,0x04);
  ampc_writem (p,uadr|0xe6,0x00,0x1F);
  udelay(1000);

  return nprog;
}

int_4 setup_SI514 (PICSTRUCT *p, double freq) {
  int_4 i,uadr=0x0; int_1 data[4]; double m,vco;
  int_4 ls_div,hs_div,m_int,m_frac,lsdiv,lp1=2,lp2=2,tmp;
  uadr |= findintflagdef("I2CP",p->config,0x4)<<24;
  uadr |= findintflagdef("I2CA",p->config,0x55)<<17;
  if (freq>1e5) freq /= 1e6;	/* convert to MHz */
  for (hs_div=1022,lsdiv=0,ls_div=1; ls_div<32 && 2080.0/(ls_div*hs_div) > freq; lsdiv++,ls_div*=2);
  for (; hs_div>10 && 2080.0/(ls_div*hs_div) <= freq; hs_div-=2); hs_div+=2;
  m = ls_div*hs_div*freq/31.98;
  m_int = (int_4)m;
  m_frac = (m-m_int)*0x20000000+0.5;
  vco = ls_div*hs_div*freq;
       if (vco>2425) { lp1=4; lp2=4; }
  else if (vco>2332) { lp1=3; lp2=4; }
  else if (vco>2170) { lp1=3; lp2=3; }
  else if (vco>2087) { lp1=2; lp2=3; }
  else if (vco>2080) { lp1=2; lp2=2; }
  else               { lp1=2; lp2=2; }
  vprint("SI514 uadr=%08x freqMHz=%f mint=%d mfrac=%08x hsdiv=%d lsdiv=%d lp1=%d lp2=%d\n",uadr,freq,m_int,m_frac,hs_div,ls_div,lp1,lp2);
  data[0]=0x80; mpc_write (p,uadr+128,data,1);	/* Reset */
  data[0]=0x10; mpc_write (p,uadr+14,data,1);	/* OE state = high */
  data[0]=0x00; mpc_write (p,uadr+132,data,1);	/* disable OE */
  tmp=(lp1<<4)|lp2;       		  mpc_write (p,uadr+0,(int_1*)&tmp,1);	/* lp1&2 */
  tmp=(m_int<<29)|m_frac; 		  mpc_write (p,uadr+5,(int_1*)&tmp,4);	/* M_frac */
  tmp=(lsdiv<<20)|(hs_div<<8)|(m_int>>3); mpc_write (p,uadr+9,(int_1*)&tmp,3);	/* combo */
  data[0]=0x01; mpc_write (p,uadr+132,data,1);	/* FCAL */
  udelay(10000);
  data[0]=0x04; mpc_write (p,uadr+132,data,1);	/* OE */
  pic_wpb (p,0,PPC_NIO|0x10000,0x4);		/* reset eth core */
  pic_wpb (p,0,PPC_NIO|0x10000,0x0);
  return 0;
}

void wrioc (PICSTRUCT *p, int_4 addr, int_4 data, int_4 nb)
{
  int_4 i, status;
  int_4 rb = (jtagfd==NULL)? 1:0;	/* use readback to throttle */
  if (addr>=0) { 
    WR(REG_IOC, REG_IOC_ADDR|addr);
    if (rb) RD(REG_MCSR,status);
  }
  for (i=0; i<nb; i++) {
    WR(REG_IOC, data&0xFF);  
    if (rb) RD(REG_MCSR,status);
    data>>=8;
  }
}

int_4 rdioc (PICSTRUCT *p, int_4 addr, int_4 nb)
{
  int_4 i,j,k, status;
  if (addr>=0) { 
    WR(REG_IOC, REG_IOC_ADDR|addr);
  }
  for (i=j=0; i<nb; i++) {
    RD(REG_IOC, k); 
    j |= (k&0xFF)<<(i*8);
  }
  return j;
}

int_4 jvm_rwmem (PICSTRUCT *p, int_4 port, int_4 addr, int_4 *lbuf, int_4 bytes, int_4 flags)
{
  int_4 i,n,sel,iaddr,status,data=0;
  iaddr = (flags&FLG_NVME)? 0x0600 : (flags&FLG_NIO)? 0x0500 : 0x0000;
  if (flags&FLG_NVME) { if (port==2) iaddr |= 0x80; if (port<=2) port=0; }	/* NVME ports on crossbar ? */
  if (flags&FLG_NIO) { if (port==1||port==2) port+=10; }			/* NIO ports not on IOMs ? */
 while (bytes!=0) {
  n = (bytes>4)? 4 : (bytes<-4)? -4 : bytes;
  if (port==0) {
    wrioc(p, iaddr|PPC_BADDR_WR, addr, 2);
    wrioc(p, iaddr|PPC_BDATA, 0, 0);
    if (bytes>0) { data=*lbuf; for (i=0; i<n; i++) { WR(REG_IOC,(data&0xFF)); data >>= 8; } }
    else { data=0; for (i=0; i<(-n); i++) { RD(REG_IOC, status);  data |= ((status&0xFF)<<(i*8)); } }
  }
  else if (port==9) {
    iaddr |= 0x0100;				
    sel = getJTAGsel(p, port) | iaddr;
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|addr,2);
    if (bytes>0) pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,*lbuf,n);
    else data = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,-n);
  }
  else {
    if (port<=3) iaddr |= 0x0100;	/* IO Modules use page=1 now not page=0 */
    sel = getJTAGsel(p, port) | iaddr;
    if (p->k7>=8) {
     if (bytes>0) pic_acmd (p,ADON_WRJVM,addr,*lbuf,sel|n);
     else  data = pic_acmd (p,ADON_RDJVM,addr,0,sel|(-n));
    } else {
     pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|addr,2);
     if (bytes>0) pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,*lbuf,n);
     else data = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,-n);
    }
  }
  if (lbuf!=NULL && bytes<0) *lbuf = data;
  bytes -= n;
  lbuf++; addr+=4;
 }
  return data;
}

int_4 jvm_rmem (PICSTRUCT *p, int_4 port, int_4 addr, int_4 flags)             { return jvm_rwmem(p,port,addr,NULL,-4,flags); }
int_4 jvm_wmem (PICSTRUCT *p, int_4 port, int_4 addr, int_4 data, int_4 flags) { return jvm_rwmem(p,port,addr,&data,4,flags); }

int_4 pic_loadjvm (PICSTRUCT *p, int_4 port, int_2 *loadword, int_4 nprog, int_4 flags)
{
  int_4 status,i,j,k,sel,conf=0,unib,bcod,hbt,hbtl,rb,nvm,nio,addr,ncr;

  if (port==3) { return pic_loadjvm(p,1,loadword,nprog,flags) + pic_loadjvm(p,2,loadword,nprog,flags); }

  sel = getJTAGsel(p, port);
  nio = (flags&FLG_NIO)? 1:0;
  nvm = (flags&FLG_NVME)? 1:0;
  addr = nvm? 0x0600 : nio? 0x0500 : 0x0000;
  if (nvm) { if (port==2) addr |= 0x80; if (port<=2) nvm=2; }	/* NVME ports on crossbar ? */

  pic_lock (p, LOCK_ALLOC);

  /* reset JVM */
  if (port==0 || nvm==2) {
    if (nio||nvm) {
      wrioc(p, addr|JVM_SYSREG, JVM_RESET, 1);
      wrioc(p, addr|JVM_SYSREG, JVM_LOAD, 1);
    }
    else {
      WR(REG_MCSR, MCSR_RESET);
      WR(REG_MCSR, MCSR_RPRC);	/* this resets the JVM_SYSREG to zero */
      WR(REG_MCSR, MCSR_ENABLE);
    }
    wrioc(p, addr|PPC_BADDR_WR, 0x1FF8, 2);
    wrioc(p, addr|PPC_BDATA, 0x12345678, 4);
    wrioc(p,             -1, 0x87654321, 4);
    wrioc(p, addr|PPC_BADDR_WR, 0x1FF8, 2);
    wrioc(p, addr|PPC_BDATA, 0, 0);
    j=rdioc(p,-1,4); k=0x12345678; if (j!=k) { print("Error reading back DSOCM j=%08x k=%08x from port=%d\n",j,k,port); goto BAIL; }
    j=rdioc(p,-1,4); k=0x87654321; if (j!=k) { print("Error reading back DSOCM j=%08x k=%08x from port=%d\n",j,k,port); goto BAIL; }
    wrioc(p, addr|0x20, 0, 0);	/* get configured ram sizes */
    for (i=0; i<32; i+=8) { RD(REG_IOC, k); conf |= (k&0xFF)<<i; }
    wrioc(p, addr|PPC_BADDR_WR, JVM_BADDR_ISOCM, 2);
    wrioc(p, addr|PPC_BDATA, 0, 0);
  } 
  else {
    if (nio||nvm) {
      sel |= addr;
    } else {
      if (port<=3 || port==9) sel |= 0x0100;	/* IO Modules use page=1 now not page=0 */
      if (port>3 && port!=9) pic_acmd (p,ADON_RDIOB,sel|PM_RESET_DMAC,0,1); 
      if (port>3 && port!=9) pic_acmd (p,ADON_RDIOB,sel|PM_RESET_PPCGO,0,1); 
    }
    pic_acmd (p,ADON_WRIOB,sel|JVM_SYSREG,JVM_RESET,1); 
    /* test data interface */
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|0x1FF8,2);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0x12345678,4);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,0x87654321,4);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|0x1FF8,2);
    j=pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4); if (j!=0x12345678) { print("Error reading back DSOCM j=%08x from port=%d\n",j,port); goto BAIL; }
    j=pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,4); if (j!=0x87654321) { print("Error reading back DSOCM j=%08x from port=%d\n",j,port); goto BAIL; }
    conf=pic_acmd (p,ADON_RDIOB,sel|0x20,0,4);
    pic_acmd (p,ADON_WRIOB,sel|JVM_SYSREG,JVM_LOAD,1); 
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,JVM_BADDR_ISOCM,2);
  }

  i = (conf&0xFF);
  if (conf!=0 && (conf&0xFFFF)==(conf>>16) && i>=2 && i<=64) {
    ncr = (conf&0xFF) * 0x400;	/* valid embedded Code RAM size ? */
  } else {			/* default for old loads */
    ncr = (port==0)? 0x2000 : (port==1 || port==2)? 0x800 : 0x1000;
  }

  if (jtagfd!=NULL && port!=9) 
  for (i=0; i<nprog; i+=j) {
    j = min(256,nprog-i);
    extjtag (p,ADON_LIOC,0,j/2,0,(int_4*)(loadword+i));
  }
  else 
  for (i=j=0; i<nprog; i++) {
    unib = (loadword[i]>>8)&0xF;
    bcod = (loadword[i])&0xFF;
    if (unib&0x2) j++;	/* LUT load words */
    if (i-j >= ncr) { print("Load JVM port=%d ncode=%d > max=%d. Truncating.\n",port,nprog-j,ncr); break; }
    if (port==0 || nvm==2) {	/*          inc4  upper2b */
      wrioc(p, addr|0x10|(unib&0x3), bcod, 1);
    } else {
      pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA|(unib&0x3),bcod,1);
    }
  }
  if (i<nprog && port==0) print("Truncated JVM load. Must flash FPGA to current version.\n");

  /* start the processor */
  if (port==0 || nvm==2) {
    wrioc(p, addr|JVM_SYSREG, JVM_RESET, 1);
    wrioc(p, addr|JVM_SYSREG, JVM_RESET|JVM_RUN, 1);
    wrioc(p, addr|JVM_SYSREG, JVM_RUN, 1);
    wrioc(p, 0, 0, 0);
  } else {
    pic_acmd (p,ADON_WRIOB,sel|JVM_SYSREG,JVM_RESET|JVM_RUN,1); 
    pic_acmd (p,ADON_WRIOB,sel|JVM_SYSREG,JVM_RUN,1); 
  }

  vprint("Load JVM port=%d ncode=%d",port,nprog);
  for (j=0; j<4; j++) {
    hbtl = hbt;
    hbt = jvm_rmem (p,port,PPC_HBT,flags);
    vprint(" HBT%d=%08x",j,hbt);
  }
  hbt = jvm_rmem (p,port,PPC_FAT,flags);
  vprint(" FAT=%08x\n",hbt);
  if (hbt==hbtl) nprog=-1; /* processor didnt start */
  pic_lock (p, LOCK_FREE);
  return (nprog);

  BAIL:
  pic_lock (p, LOCK_FREE);
  return (-1);
}

int_4 pic_loadioc_new (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  nprep,n,i,j;
  int_u1 loadxtra[32];

  pic_lock (p, LOCK_ALLOC);

  vprint ("Loading IOC new - %d bytes ... ",nprog); 
  WR(REG_MCSR, MCSR_RIOC);
  udelay(10); /* Altera 10K spec */
  WR(REG_MCSR, 0);
  udelay(5); /* Altera 10K spec */

  n = nprog&0xFFFFFFFC;
  nprep = 16;

  /* load full blocks of program data */
  pic_transfer (p, (int_4 *)(loadbyte), n, DMA_TEMP_ADDR, 2, 0);
  verify32(p,(int_4 *)(loadbyte),n/4,DMA_TEMP_ADDR);

  /* prep buffer for xtra and 10+ clock init */ 
  for (j=0,i=n; j<nprep; i++,j++) { 
    if (i<nprog) loadxtra[j]=loadbyte[i]; 
    else loadxtra[j]=0;
  }
  pic_transfer (p, (int_4 *)loadxtra, nprep, DMA_TEMP_ADDR+n/4, 2, 0);

  vprint ("initializing ... %d ",i);
  pic_acmd (p,ADON_LIOC,DMA_TEMP_ADDR,MCSR_LIOC,(n+nprep)/4); 
  vprint ("resetting ..."); 

  pic_acmd (p,ADON_WRIOC,IOC_RAM|IOC_A,0x45,0);
  i = pic_acmd (p,ADON_RDIOC,IOC_RAM|IOC_A,0,0);
  if (i!=0x45 && p->iocc[2]!='S') print (" **** Err loading IOC code new ****\n");

  vprint ("done\n"); 
  pic_lock (p, LOCK_FREE);

  return (nprog);
}

int_4 pic_loadioc_old (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  n,i,status,nn,blk=1024;
  int_u1 loadxtra[16];

  pic_lock (p, LOCK_ALLOC);

  vprint ("Loading IOC old - %d bytes ... ",nprog);
  WR(REG_MCSR, MCSR_RIOC);
  udelay(10);

  /* enable IOC load */
  WR(REG_MCSR, MCSR_LIOC);

  /* load program data */
  n = nprog/8*8;         /* round xfer to 8-byte boundary */
  for (i=0; i<n; i+=nn) {
    nn=n-i; if (nn>blk) nn=blk;
    status = io_ctl (p, IOCTL_LIOC, REG_IOC, (int_4 *)(loadbyte+i), nn);
  }

  vprint ("initializing ... ");
  for (i=0; i<16; i++) { /* prep buffer for xtra and 10+ clock init */
    if (n+i<nprog) loadxtra[i]=loadbyte[n+i];
    else loadxtra[i]=0;
  }
  n = nprog-n+2; n = 16; /* for driver mapping 8-byte align */
  status = io_ctl (p, IOCTL_LIOC, REG_IOC, (int_4 *)loadxtra, n);

  /* disable IOC load to allow normal register access */
  WR(REG_MCSR, 0);
  vprint ("resetting ...");

  pic_acmd (p,ADON_WRIOC,IOC_RAM|IOC_A,0x45,0);
  i = pic_acmd (p,ADON_RDIOC,IOC_RAM|IOC_A,0,0);
  if (i!=0x45) print (" **** Err loading IOC code old ****\n");

  vprint ("done\n");
  pic_lock (p, LOCK_FREE);

  return (nprog);
}

int_4 pic_loadioc (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4 stat;
  if (p->type==ICEPIC2 || p->type==ICEMBT2 || p->rev<5) 
    stat = pic_loadioc_old (p,loadbyte,nprog,flags);
  else
    stat = pic_loadioc_new (p,loadbyte,nprog,flags);

  return stat;
}

/*
  Procedure for Virtex self re-program:

  Use standard JTAG to setup the bitstream
  Copy pci prep frames to DRAM
  Prep the PCI system autoload registers
   Put JTAG in bypass mode
   Load PCI configuration frames
   Put JTAG in normal mode
  Put PPC processor in reset
  Load the rest of the bitstream through PCI fifo
  Perform normal reset

  Note: mblk sized to keep ADON_JTAG calls from hitting udelay
*/

int_4 pic_loadvirtex (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  i,j,k,cmd,reg,op,jcbs,iprog, mblk=DMA_INT_SIZE/2, sel, blk, *test, vt, itick=0, status=0, id;
  int_4  rcrc[6]  = {0xFFFFFFFF,0xAA995566,0x30008001,0x00000007,0,0};
  int_4  ghigh[4] = {0x30008001,0x00000008, 0, 0};
  int_4  zeros[4] = {0, 0, 0, 0};

  sel = getJTAGsel(p,port);
  jcbs = getJTAGcbs(sel);

  /* check the module JTAG port */
  pic_jtag (p, port, JTAG_CMD_IDCODE, &i, -32, 0); i &= 0x0FFFFFFF; id=i; /* strip rev */
  pic_acmd (p, ADON_JTAG, 0, 0, 0); 
  if (i==IDCODE_DTDM || i==IDCODE_DTDMX || i==IDCODE_PIC5) vt=2;
  else if (i==IDCODE_RFXD) vt=3;
  else if (i==IDCODE_PIC6) vt=4;
  else if (i==IDCODE_PIC7) vt=7;
  else if (i==IDCODE_PIC8) vt=8;
  else if (i==IDCODE_PIC8P) vt=8;
  else if (i==IDCODE_ZPPM || i==IDCODE_V5M) vt=5;
  else if (i==IDCODE_S6M) vt=6;
  else if (i==IDCODE_S6MX) vt=6;
  else if (i==IDCODE_V6M) vt=6;
  else if (i==IDCODE_K8M) vt=8;
  else if (i==IDCODE_K8P) vt=8;
  else if (i==IDCODE_SP6) vt=6;
  else if (i==IDCODE_GPS) vt=6;
  else { print ("Bad Virtex IDCODE=%08x load aborted\n",i); return -1; }

  pic_disable_nvme (p,port);	/* disable NVME drives */

  prepBitStream((int_u1*)rcrc,6*4);
  prepBitStream((int_u1*)ghigh,4*4);

  pic_lock (p, LOCK_ALLOC);
  vprint ("Loading Virtex port=%d bytes=%d ",port,nprog);
  v2print("\n");

  /* shutdown the chip */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_CMDR, JTAG_CMD_JSHUTD<<jcbs, JTAG_MSK_CMD|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_STARTUP, 0x0, JTAG_MSK_STARTUP|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0x0, JTAG_MSK_RST|sel); 

  /* pulse the prog_b pin to clear configuration memory */
  if (vt!=7) pic_jtag (p, port, JTAG_CMD_JPROGB, 0, 0, 0);

  if (vt>6) {	/* virtex-7+ specific initing */
  }
  else if (vt==6) {	/* spartan/virtex-6 specific initing */
    pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_CMD_BYPASS<<jcbs, JTAG_MSK_CMD|sel);
    for (i=0; i<100000/32; i++) pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, 0x0, JTAG_MSK_SET32X|sel);
    pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0x0, JTAG_MSK_RST|sel);
  }
  else if (vt==5) {	/* virtex-5 specific initing */
    pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_CMD_CFGIN<<jcbs, JTAG_MSK_CMD|sel); 
    for (i=0; i<100000/32; i++) pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, 0x0, JTAG_MSK_SET32X|sel); 
    pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0x0, JTAG_MSK_RST|sel); 
  }
  else { 	/* reset the CRC register + some zeros for sync clear + global hold */
    pic_jtag (p, port, JTAG_CMD_CFGIN, rcrc, 6*32, 0);
    pic_jtag (p, port, JTAG_CMD_CFGIN, zeros, 2*32, 0);
    pic_jtag (p, port, JTAG_CMD_CFGIN, ghigh, 4*32, 0);
  }

  /* load the cfg_in command */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_CMD_CFGIN<<jcbs, JTAG_MSK_CMD|sel); 

  if (vt!=7) udelay(20000);

  test  = (int_4 *)loadbyte;
  iprog = nprog/4;
  for (i=0; i<iprog; ) {
    j   = i;
    cmd = test[i++];
    op  = (cmd>>27)&0x3;
    reg = 0xF;
    blk = 0;
    if (cmd==0xFFFFFFFF || cmd==0xAA995566) op=9;
    else if (vt>=5) blk=iprog-i;
    else if (op==0) for (; i<iprog && ((test[i]>>27)&0x3)==0; i++);
    else {
      blk = cmd&0x7FF;
      reg = (cmd>>13)&0xF;
      if (blk==0 && (test[i]>>29)==2) { blk = test[i++] & 0xFFFFF; } /* extended count register */
      if (reg==0x2 && vt==2) blk++; 
    }
    i += blk;
    v2print("BitStream Config Cmd=%08x Op=%d Reg=%x Value=%08x Words=%d\n", cmd,op,reg,test[i-1],i-j);
    prepBitStream((int_u1*)(test+j),(i-j)*4);
    if (blk<=2) for (;j<i; j++) { pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, test[j], JTAG_MSK_SET32X|sel); }
    else for (; j<i; j+=blk) {
      blk = min(mblk,i-j);
      if (jtagfd!=NULL) extjtag (p, ADON_PGMC, JTAG_TMS_SET32X,  blk, JTAG_MSK_SET32X|sel, test+j);
      else if (p->type == ICEPIC7 || p->type == ICEPIC8)  { 
        pic_acmd (p, ADON_JTAG, -10, blk, sel);
        pic_wfifo (p, test+j, blk*4, FLG_SYNC);
      }
      else {
        pic_transfer (p, test+j, blk*4, DMA_INT_ADDR, 2, 1);
        pic_acmd (p, ADON_JTAG, -11, DMA_INT_ADDR, blk|sel);
      }
      if (++itick%100==0) vprint (".");
    }
  }

  /* load dummy word of program data and reset JTAG */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, 0, JTAG_MSK_SET32X|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32,  0, JTAG_MSK_SET32|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_RST,  0x0, JTAG_MSK_RST|sel); 

  vprint (" start ");
  pic_acmd (p, ADON_JTAG, JTAG_TMS_CMDR, JTAG_CMD_JSTART<<jcbs, JTAG_MSK_CMD|sel); 
  if (vt>=7)  for (i=0; i<2000/32; i++) pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, 0x0, JTAG_MSK_SET32X|sel); 	/* extra startup */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_STARTUP, 0x0, JTAG_MSK_STARTUP|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0x0, JTAG_MSK_RST|sel); 

  /* check configuration status */
  if (port==JTAG_PORT_IOALL) {	/* multi-port load case 3==1&2 */
    for (j=1; j<=p->miom; j++) { i=pic_pmstat(p,j); vprint("stat%d=%08x ",j,i); if (i>0) status++; }
  } else if (port==JTAG_PORT_PMALL && jtagfd==NULL) {	/* multi-port load case 9==11-14 */
    for (j=1; j<=p->mpm; j++) { i=pic_pmstat(p,JTAG_PORT_PMOFF+j); vprint("stat%d=%08x ",j,i); if (i>0) status++; }
  } else {
    i=pic_pmstat(p,port); vprint("stat=%08x ",i); if (i>0) status++;
  }

  /* check the module JTAG port */
  pic_jtag (p, port, JTAG_CMD_UIDCODE, &i, -32, 0);
  vprint ("uid=%08x ",i);
  if (status==0 && vt>=6); else
  if (i!=0x1CE1D) status = -1;

  if (port>JTAG_PORT_PMOFF) {
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_GLOBAL,0,1); /* reset ALL clocks (DCM) */
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_DMAC,0,1); 
  }

  vprint ("done\n");
  if (status<=0) print("Problem loading virtex code of %d bytes on port=%d uid=%08x\n",iprog*4,port,i);

  pic_acmd (p, ADON_JTAG, 0, 0, 0); 

  pic_lock (p, LOCK_FREE);
  if (id==IDCODE_K8P) udelay(10000);

  return (status);
}

int_4 pic_loadstratix (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  i,j,jcbs,iprog, mblk=DMA_INT_SIZE*4, sel, blk, *test, itick=0;

  if (port<=0) port=1;
  port += JTAG_PORT_IOMOFF;
  sel = getJTAGsel(p,port);
  jcbs = getJTAGcbs(sel);

  /* check the module JTAG port */
  pic_jtag (p, port, JTAG_STXCMD_IDCODE, &i, -32, 0); i &= 0x0FFFFFFF; 
  if ((i&0x0F000FFF)!=0x020000dd) { 
    print ("Bad Stratix IDCODE=%08x load aborted\n",i); 
    return -1; 
  }

  pic_lock (p, LOCK_ALLOC);
  vprint ("Loading Stratix bytes=%d ",nprog);

  /* load the pgm command, goto shift-dr */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_STXCMD_PGMCMD<<jcbs, JTAG_MSK_CMD|sel); 

  /* load blocks of program data */
  iprog = nprog/4;
  for (i=0; i<nprog-4; i+=blk,loadbyte+=blk) {
    test = (int_4 *)loadbyte;
    if (nprog-4-i < mblk) blk = nprog-4-i; else blk = mblk;
    if (jtagfd!=NULL) extjtag (p,ADON_PGMC, JTAG_TMS_SET32X,  blk/4, JTAG_MSK_SET32X|sel, test);
#ifdef CANDOFASTSTRATIXLOAD
    else if (p->type == ICEPIC7 || p->type == ICEPIC8)  { 
#else
    else if (p->type == ICEPIC8 && port==1)  { 
#endif
      pic_acmd (p, ADON_JTAG, -10, blk/4, sel);
      pic_wfifo (p, test, blk, FLG_SYNC);
    }
    else {
      pic_transfer (p, test, blk, DMA_INT_ADDR, 2, 1);
      j = blk/4; if (p->isY) j |= sel;
      pic_acmd (p, ADON_JTAG, -11, DMA_INT_ADDR, j);
    }
    if (++itick%32==0) vprint (".");
  }

  /* load last word of program data, and go to Exit-DR */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32, *(int_4 *)loadbyte, JTAG_MSK_SET32|sel); 

  vprint (" start ");
  pic_acmd (p, ADON_JTAG, JTAG_TMS_XCMDR, JTAG_STXCMD_STARTUP<<jcbs, JTAG_MSK_CMD|sel); 
  for (i=0; i<100; i++) pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, -1, JTAG_MSK_SET32|sel);  /* 3200 clocks */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET8, -1, JTAG_MSK_SET8|sel); 

  /* check the module JTAG port */
  pic_jtag (p, port, JTAG_STXCMD_UIDCODE, &i, -32, 0);
  vprint ("uid=%08x ",i);
  if ((i>>16)!=0x524d) nprog = -1;
  if (nprog<0) print("Problem loading stratix code of %d bytes on port=%d uid=%08x\n",iprog*4,port,i);

  /* set address of Virtual JTAG scan chain to default address */
  if (((sel>>27)&0xC)==0xC) {
    pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_STXCMD_USER1<<jcbs, JTAG_MSK_CMD|sel); 
    pic_acmd (p, ADON_JTAG, JTAG_TMS_SET24R, 0x400000, JTAG_MSK_SET24R|sel); 
  }

  vprint ("done\n");
  pic_lock (p, LOCK_FREE);

  return (nprog);
}

#define FCMD_WRITE_STATUS  0x01
#define FCMD_PAGE_PROGRAM  0x02
#define FCMD_READ          0x03
#define FCMD_WRITE_DISABLE 0x04
#define FCMD_READ_STATUS   0x05
#define FCMD_WRITE_ENABLE  0x06
#define FCMD_SUBSEC_ERASE  0x20
#define FCMD_READ_SPSR     0x2A	
#define FCMD_READ_SFDP     0x5A	
#define FCMD_READ_EVCFG    0x65
#define FCMD_RESET_ENA     0x66
#define FCMD_READ_FLAG     0x70	
#define FCMD_WRITE_VCFG    0x81
#define FCMD_READ_VCFG     0x85
#define FCMD_READ_GPRR     0x96
#define FCMD_GLOBAL_UNLOCK 0x98
#define FCMD_RESET_MEM     0x99
#define FCMD_READ_ID       0x9F
#define FCMD_WRITE_GFRZB   0xA6
#define FCMD_READ_GFRZB    0xA7
#define FCMD_WRITE_NVCFG   0xB1
#define FCMD_READ_NVCFG    0xB5
#define FCMD_ENTER_4BYTE   0xB7
#define FCMD_DIE_ERASE     0xC4
#define FCMD_WRITE_EXTADR  0xC5
#define FCMD_BULK_ERASE    0xC7
#define FCMD_READ_EXTADR   0xC8
#define FCMD_SECTOR_ERASE  0xD8
#define FCMD_READ_NVLB     0xE2
#define FCMD_ERASE_NVLB    0xE4
#define FCMD_READ_VLB      0xE8
#define FCMD_EXIT_4BYTE    0xE9

int_4 spi_flash (PICSTRUCT *p, int_4 cmd, int_4 adr, int_4 nadr, int_u1 *loadbyte, int_4 bytes)
{
  int_u1 tmp[abs(bytes)+5];
  int_4 i,n,port=0,doff=0;
  int_4 byp = (cmd==FCMD_PAGE_PROGRAM) || (cmd==FCMD_READ);
  tmp[doff++]=cmd; /* load FLASH command into JTAG USER1 data buffer */
  for (i=0; i<nadr; i++) tmp[doff++] = adr >> ((nadr-i-1)*8);
  if (cmd==FCMD_READ_SFDP || cmd==FCMD_READ_GPRR) tmp[doff++]=0;			/* add 8 dummy clocks */
  n = abs(bytes) + doff;
  for (i=0; i<doff; i++) tmp[i]=bitrev[tmp[i]];	
  port = p->flashport;
  if (bytes>=0) { 
    memcpy((void*)(tmp+doff),(void*)loadbyte,bytes);
    if (!byp) for (i=doff; i<n; i++) tmp[i]=bitrev[tmp[i]];
    pic_jtag (p, port, JTAG_CMD_USER1, (int_4*)tmp, n*8, 0);
  } else {
    for (i=doff; i<=n; i++) tmp[i]=0;
    pic_jtag (p, port, JTAG_CMD_USER1, (int_4*)tmp, -n*8-8, 0);
    for (i=0; i<n; i++) tmp[i]=((tmp[i]>>1)|(tmp[i+1]<<7))&0xFF;
    if (!byp) for (i=doff; i<n; i++) tmp[i]=bitrev[tmp[i]];
    memcpy((void*)loadbyte,(void*)(tmp+doff),-bytes);
  }
  return 0;
}

int_4 spi_flash_wrena (PICSTRUCT *p, int_4 ena)
{
  int_u1 data[4]; int_4 wena,retry=128;
  while (--retry>0) {
    spi_flash (p,(ena>0)?FCMD_WRITE_ENABLE:FCMD_WRITE_DISABLE,0,0,data,0);
    spi_flash (p,FCMD_READ_STATUS,0,0,data,-1); 
    wena = data[0]&0x02;
    if (ena>0 && wena!=0) break;
    if (ena<0 && wena==0) break;
  }
  if (retry==0) printf("Could not set write enable=%d on flash\n",ena);
  return (retry>0);
}

int_4 spi_checkflash (PICSTRUCT *p)
{
  int_4 i,n,stat=-1,na=3; int_u1 data[20]; int_4 *ldat=(int_4*)data;
  spi_flash (p,FCMD_RESET_ENA,0,0,data,0);
  spi_flash (p,FCMD_RESET_MEM,0,0,data,0);
  n=20; spi_flash (p,FCMD_READ_ID,0,0,data,-n);    v2print("Flash ID   "); for (i=0; i<n; i++) v2print(" %02x",data[i]); v2print("\n");
  if (*ldat == 0x1018BA20 || *ldat == 0x1018BB20) stat=1; /* 128Mb, 3V || 1.8V */
  if (*ldat == 0x1019BA20 || *ldat == 0x1019BB20) stat=2; /* 256Mb, 3V || 1.8V */
  if (*ldat == 0x1021BA20 || *ldat == 0x1021BB20) stat=3; /* 1Gb,   3V || 1.8V */
  if (stat>1) { na=4; spi_flash (p,FCMD_ENTER_4BYTE,0,0,data,0); }
 if (stat>0 && p->verbose==2) {
  n=16; spi_flash (p,FCMD_READ_SFDP,0,3,data,-n);  v2print("Flash SFPD "); for (i=0; i<n; i++) v2print(" %02x",data[i]); v2print("\n");
  n=2;  spi_flash (p,FCMD_READ_NVCFG,0,0,data,-n); v2print("Flash NVCFG %02x%02x\n",data[1],data[0]); 
  n=1;  spi_flash (p,FCMD_READ_VCFG,0,0,data,-n);  v2print("Flash VCFG  %02x 7:4=DumClks 3=DisXIP 1:0=Wrap\n",data[0]);
  n=1;  spi_flash (p,FCMD_READ_EVCFG,0,0,data,-n); v2print("Flash EVCFG %02x 7=DisQSPI 6=DisDSPI 5=DisDTR 4=EnResHold 2:0=OutDrv\n",data[0]);
  n=2;  spi_flash (p,FCMD_READ_SPSR,0,0,data,-n);  v2print("Flash SPSR  %02x%02x 2=DisPassword 1=EnSectorProt\n",data[1],data[0]);
  n=1;  spi_flash (p,FCMD_READ_GFRZB,0,0,data,-n); v2print("Flash GFRZB %02x 0=DisableGlobalFreeze\n",data[0]);
  n=1;  spi_flash (p,FCMD_READ_VLB,0,na,data,-n);  v2print("Flash VLB   %02x 1=LockDownEnabled 0=WriteLockEnabled\n",data[0]);
  n=1;  spi_flash (p,FCMD_READ_NVLB,0,4,data,-n);  v2print("Flash NVLB  %02x 1=LockDownDisabled 0=WriteLockDisabled\n",data[0]);
  n=1;  spi_flash (p,FCMD_READ_FLAG,0,0,data,-n);  v2print("Flash FLAG  %02x 7=Ready 6=EraSus 5=EraErr 4=PrgErr 2=ProSus 1=ProErr 0=4byt\n",data[0]);
  n=9;  spi_flash (p,FCMD_READ_GPRR,0,0,data,-n);   v2print("Flash GPRR "); for (i=0; i<n; i++) v2print(" %02x",data[i]); v2print("\n");
  spi_flash_wrena(p,1);
  n=1;  spi_flash (p,FCMD_READ_STATUS,0,0,data,-n);v2print("Flash STAT  %02x 7=DisWSR 5=ProBotUp 6,4:2=ProRegion[3:0] 1=WrEna 0=WrProg\n",data[0]);
  spi_flash_wrena(p,-1);
  n=1;  spi_flash (p,FCMD_READ_STATUS,0,0,data,-n);v2print("Flash STAT  %02x 7=DisWSR 5=ProBotUp 6,4:2=ProRegion[3:0] 1=WrEna 0=WrProg\n",data[0]);
 }
  if (stat<0) print("Bad SPI Flash status=%x\n",*ldat);
  return stat;
}

int_4 pic_unloadflash_spi (PICSTRUCT *p, int_u1 *loadbyte, int_4 iprom, int_4 nprog, int_4 flags)
{
  int_4 i,m,n,na; int_u1 data[16];
  if ((m=spi_checkflash(p))<=0) return 0;
  na = (m>1)? 4:3;
  vprint("Unloading SPI FLASH - %d bytes ",nprog);
  if (findintflag("NOLOAD",p->config)>0);
  else for (i=0; i<nprog; i+=n,iprom+=n) {
    n=min(256,nprog-i);
    spi_flash (p,FCMD_READ,iprom,na,loadbyte+i,-n);
    if (i%0x10000==0) vprint(".");
  }
  vprint(" done\n");
  return nprog;
}

int_4 pic_loadflash_spi (PICSTRUCT *p, int_u1 *loadbyte, int_4 iprom, int_4 nprog, int_4 flags)
{
  int_4 i,j,m,n,na; int_u1 data[16];
  if ((m=spi_checkflash(p))<=0) return 0;
  na = (m>1)? 4:3;
  pic_lock (p, LOCK_ALLOC);
  vprint("Erasing SPI FLASH m=%d ",m);
  if (findintflag("NOERASE",p->config)>0);
  else if (m>1 || nprog<=0x10000) {
    for (i=0; i<nprog; i+=n) {
      j=i+iprom; n=min((j&0xFFFF)?0x1000:0x10000,nprog-i); 
      spi_flash (p,FCMD_WRITE_ENABLE,0,0,data,0);
      if (n<=0x1000) spi_flash (p,FCMD_SUBSEC_ERASE,j,na,data,0); 
      else spi_flash (p,FCMD_SECTOR_ERASE,j,na,data,0);
      do { spi_flash (p,FCMD_READ_FLAG,0,0,data,-1); } while ((data[0]&0x80)==0);
      spi_flash (p,FCMD_WRITE_DISABLE,0,0,data,0);
      vprint("."); 
    }
  }
  else {
    spi_flash (p,FCMD_WRITE_ENABLE,0,0,data,0);
    spi_flash (p,FCMD_BULK_ERASE,0,0,data,0);
    do { spi_flash (p,FCMD_READ_FLAG,0,0,data,-1); vprint("."); udelay(1000000); } while ((data[0]&0x80)==0);
    spi_flash (p,FCMD_WRITE_DISABLE,0,0,data,0);
  }
  vprint(" done\n");
  pic_lock (p, LOCK_FREE);
  if (spi_checkflash(p)<0) return 0;
  pic_lock (p, LOCK_ALLOC);
  vprint("Loading SPI FLASH ");
  if (findintflag("NOLOAD",p->config)>0)
    vfill (-1, (int_4 *)loadbyte, nprog);
  else for (i=0; i<nprog; i+=n,iprom+=n) {
    n=min(256,nprog-i);
    spi_flash (p,FCMD_WRITE_ENABLE,0,0,data,0);
    spi_flash (p,FCMD_PAGE_PROGRAM,iprom,na,loadbyte+i,n);
    do { spi_flash (p,FCMD_READ_FLAG,0,0,data,-1); } while ((data[0]&0x80)==0);
    spi_flash (p,FCMD_WRITE_DISABLE,0,0,data,0);
    if (i%0x10000==0) vprint(".");
  }
  vprint(" done\n");
  pic_lock (p, LOCK_FREE);
  return nprog;
}




int_4 pic_unloadflash (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  i,k, vt, iprom=0, nprom, nexpect=0, port=JTAG_PORT_FLASH, mblk, sel, blk, delay=15;

  /* check the platform flash JTAG port */
  if (!(p->isY||p->k7)) {
    print ("Flash only exists on PIC5+ series cards. Unload aborted\n");
    return -1;
  }
  if (p->k7) port = p->flashport;
  if (p->mtype[1]==IOMT_DIODE && nprog<=0x00020000) port=2;
  pic_jtag (p, port, JTAG_CMD_IDCODE, &i, -32, 0); i &= 0x0FFFFFFF; /* strip rev */
  pic_acmd (p, ADON_JTAG, 0, 0, 0); 

       if (i==IDCODE_PF01) { vt=1; nprom = 0x00020000; }
  else if (i==IDCODE_PF16) { vt=2; nprom = 0x00200000; }
  else if (i==IDCODE_PF32) { vt=4; nprom = 0x00400000; nexpect = nprom; }
  else if (i==IDCODE_PIC7) { vt=7; nprom = 0x01000000; nexpect = 11443568; }
  else if (i==IDCODE_PIC8) { vt=8; nprom = 0x01000000; nexpect = 16006832; }
  else if (i==IDCODE_A8M ) { vt=8; nprom = 0x02000000; nexpect = 33554432; }
  else if (i==IDCODE_K8M ) { vt=8; nprom = 0x01000000; nexpect = 16006832; }
  else if (i==IDCODE_K8P ) { vt=8; nprom = 0x02000000; nexpect = 28700672; }
  else if (i==IDCODE_INT)  { vt=8; nprom = 0x08000000; nexpect = 0; iprom=0x01002000; }	/* file is compressed still */
  else {
    print ("Bad PF IDCODE=%08x unload aborted - check JTAG select jumper\n",i); 
    return -1; 
  }
  if (nexpect==0) nexpect=nprom;

  if (nprog == -1) nprog = nexpect;
  if (port==2);
  else if (nprog == 0x10000) iprom = nprom-nprog; /* top of boot eeprom */
  else if (nprog == 0x2c00) iprom = 0x00236000; /* boot stream BRAM */
  else if (nprog > nprom) print ("Platform Flash IDCODE=%08x - nprog=%d > nprom=%d\n",i,nprog,nprom);

  if (vt>=7) { pic_unloadflash_spi (p,loadbyte,iprom,nprog,flags); goto DONESPI; }

  pic_lock (p, LOCK_ALLOC);
  vprint ("Reading PF - %d bytes ",nprog);
  mblk = (vt==1)? 512 : 1024;
  i=0; pic_jtag (p, port, 0x10000, &i, 0, 0);	/* init merge mode */
  i=3; pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ispen */
  for (i=0; i<nprog; i+=blk,iprom+=blk) {
    blk = nprog - i;  
    if (blk>=mblk) blk = mblk;
    k = iprom>>3;
    if (vt==1) pic_jtag (p, port, 0x100eb, &k, 16, 0);		/* load address */
    else       pic_jtag (p, port, 0x100eb, &iprom, 24, 0);	/* load address */
    pic_jtag (p, port, 0x100ef, (int_4*)(loadbyte+i), -8*blk, delay);  /* xsc_read */
#ifdef _XMIDAS
    if ((i%0x20000)==0) vprint ("\n");
#endif
    if ((i%0x4000)==0) vprint (".");
  }
  i=0; pic_jtag (p, port, 0x100e3, &i,  8, 0);	/* check device status */
  i=0; pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld - must be last */

  pic_jtag (p, port, JTAG_CMD_IDCODE, &i, -32, 0); i &= 0x0FFFEFFF; /* strip rev */
  if (i!=IDCODE_PF01 && i!=IDCODE_PF16 && i!=IDCODE_PF32) print ("Bad PF IDCODE=%08x after unload \n",i); 

  vprint (" done\n");
  pic_lock (p, LOCK_FREE);

  DONESPI:
  if ((vt==8) && (p->rev==0x0A)); else
  prepBitStream(loadbyte,nprog);
  pic_acmd (p, ADON_JTAG, 0, 0, 0); 

  return (nprog);
}

int_4 chk (int_u1 *b1, int_u1 *b2, int_4 n) {
  int_4 i; for (i=0; i<n; i++) if (b1[i]!=b2[i]) return 1;
  return 0;
}

int_4 pic_loadflash (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  i,j,k, vt, delay, iprom=0, nprom, nexpect=0, errs, port=JTAG_PORT_FLASH, mblk, blk, *tmp1,*tmp2, crc=0,crcr, retry, stat;
  int_u1 readbyte[256];

  /* check the platform flash JTAG port */
  if (!(p->isY||p->k7)) {
    print ("Flash only exists on PIC-5 or later series cards. Load aborted\n");
    return -1;
  }
  if (p->k7 || nprog>0x00400000) port = p->flashport;
  if (p->k7 && port>10 && p->pmtype[port-11]<=0) { print("No PM defined for port=%d\n",port); return -1; }
  if (p->mtype[1]==IOMT_DIODE && nprog<=0x20000) port=2;
  pic_jtag (p, port, JTAG_CMD_IDCODE, &i, -32, 0); i &= 0x0FFFFFFF; /* strip rev */
  pic_acmd (p, ADON_JTAG, 0, 0, 0); 
       if (i==IDCODE_PF01) { vt=1; nprom = 0x00020000; }
  else if (i==IDCODE_PF16) { vt=2; nprom = 0x00200000; }
  else if (i==IDCODE_PF32) { vt=4; nprom = 0x00400000; nexpect = nprom; }
  else if (i==IDCODE_PIC7) { vt=7; nprom = 0x01000000; nexpect = 11443568; }
  else if (i==IDCODE_PIC8) { vt=8; nprom = 0x01000000; nexpect = 16006832; }
  else if (i==IDCODE_A8M ) { vt=8; nprom = 0x02000000; nexpect = 33554432; }
  else if (i==IDCODE_K8M ) { vt=8; nprom = 0x01000000; nexpect = 16006832; }
  else if (i==IDCODE_K8P ) { vt=8; nprom = 0x02000000; nexpect = 28700672; }
  else if (i==IDCODE_INT)  { vt=8; nprom = 0x08000000; nexpect = 0; iprom=0x01002000; }	/* file is compressed still */
  else {
    print ("Bad Platform Flash IDCODE=%08x on port=%d - check File or JTAG select jumper\n",i,port); 
    return -1; 
  }
  if (port==2);
  else if (nprog == 0x10000) iprom = nprom-nprog; /* top of boot eeprom, jtag read and write */
  else if (nprog == 0x2c00) iprom = 0x00236000; /* in boot stream BRAM, read without write if no jtag jumper */
  else if (nprog > nprom) {
    print ("Platform Flash IDCODE=%08x nprog=%d > nprom=%d - check File\n",i,nprog,nprom);
    return -1; 
  }
  else if (nexpect != 0 && nprog != nexpect) {
    print ("Platform Flash IDCODE=%08x nprog=%d != nexpect=%d - check File\n",i,nprog,nexpect);
    return -1; 
  }

  tmp1 = (int_4*)loadbyte;
  if ((vt==8) && (p->rev==0x0A)); else
  prepBitStream(loadbyte,nprog);

  if (vt>=7) { pic_loadflash_spi (p,loadbyte,iprom,nprog,flags); goto DONESPI; }

  pic_lock (p, LOCK_ALLOC);
  if (iprom!=0) goto NOERASE;
  vprint ("Erasing PF ");
  i=0x0; pic_jtag (p, port, 0x10000, &i, 0, 0);		/* init merge mode */
 if (vt==4) {
  i=0xd0; pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN  bypass */
  i=0x3f; pic_jtag (p, port, 0x1aa55, &i, 24, 0);	/* XSC_UNLOCK */
  i=0x3f; pic_jtag (p, port, 0x100ec, &i, 24, 0);	/* ISC_ERASE */
  i=65; /* delay n*8+1 */
 } else if (vt==2) {
  i=0x3; pic_jtag (p, port, 0x100e8, &i, 8, 0);		/* ISPEN */
  i=0x3; pic_jtag (p, port, 0x1aa55, &i, 24, 0);	/* XSC_UNLOCK */
  i=0x3; pic_jtag (p, port, 0x100ec, &i, 24, 0);	/* ISC_ERASE */
  i=25; /* delay n*8+1 */
 } else if (vt==1) {
  i=0x34; pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0x01; pic_jtag (p, port, 0x100eb, &i, 16, 0);	/* FADDR */
  i=0x00; pic_jtag (p, port, 0x100ec, &i, 0, 0);	/* FERASE */
  i=25; /* delay n*8+1 */
 }
  while (--i>0) { 
#ifdef _XMIDAS
    if ((i%8)==0) vprint ("\n");			/* force readline output buffer flush */
#endif
    vprint("."); 
    udelay(1000000);
    stat = pic_jtag (p, port, 0x100fe, &i, 0, 0);		/* check IDCODE - actually status */
    if (stat==0xf00567ff) break;
  }
  vprint (" done\n");
  NOERASE:
  vprint ("Loading PF - %d bytes ",nprog);

  mblk = (vt==1)? 256 : 32;
 if (vt==4) {
  i=0x0;  pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld */
  i=0xd0; pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN  bypass */
  i=0x3;  pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0xffffffec; pic_jtag (p, port, 0x100f2, &i, 32, 0);	/* XSC_DATA_BTC */
  i=0;    pic_jtag (p, port, 0x100ea, &i, 0, 120);	/* ISC_PROGRAM */
  i=0xd0; pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN  bypass */
 } else if (vt==2) {
  i=3; pic_jtag (p, port, 0x100e8, &i, 8, 0);		/* ISPEN */
  i=0; pic_jtag (p, port, 0x10004, &i, 16, 0);		/* check read/write protect */
  i=0; pic_jtag (p, port, 0x100f7, &i, 16, 0);		/* check read/write protect */
  i=0xffffffe4; pic_jtag (p, port, 0x100f2, &i, 32, 0);	/* XSC_DATA_BTC */
  i=0; pic_jtag (p, port, 0x100ea, &i, 0, 120);		/* ISC_PROGRAM */
 } else if (vt==1) {
  i=0x0;  pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld */
  i=0x34; pic_jtag (p, port, 0x100e8, &i, 8, 100);	/* ISPEN */
 }

  delay = (vt==4)? 300 : (vt==2)? 400 : 2000;
  delay = findintflagdef("PDELAY",p->config,delay);

  crc=0; pic_acmd (p, ADON_NULL, 0, 0, 0); 
  for (i=0; i<nprog; i+=blk,iprom+=blk) {
    blk = nprog - i;  
    retry = 3;
    if (blk>=mblk) blk = mblk;
    REDOFLASH: 
    for (j=0; j<blk; j+=4) crc = CRC32(crc,*(int_4*)(loadbyte+i+j),32);
    k = iprom>>3;
    pic_jtag (p, port, 0x100ed, (int_4*)(loadbyte+i), 8*blk, 0); 	/* ISC_DATA_SHIFT */
    if (vt==1) pic_jtag (p, port, 0x100eb, &k, 16, 0);			/* ISC_ADDRESS_SHIFT */
    else       pic_jtag (p, port, 0x100eb, &iprom, 24, 0);		/* ISC_ADDRESS_SHIFT */
    pic_jtag (p, port, 0x100ea, &iprom, 0, delay);			/* ISC_PROGRAM */
   if (vt==4 || vt==1) {
    if (vt==1) pic_jtag (p, port, 0x100eb, &k, 16, 0);			/* ISC_ADDRESS_SHIFT */
    else       pic_jtag (p, port, 0x100eb, &iprom, 24, 0);		/* ISC_ADDRESS_SHIFT */
    pic_jtag (p, port, 0x100ef, (int_4*)(readbyte), -8*blk, 15);  	/* xsc_read */
    if (chk(readbyte,loadbyte+i,blk)) { retry--; vprint ((retry==0)?"R":"r"); if (retry>0) goto REDOFLASH; }
   }
#ifdef _XMIDAS
    if ((i%0x20000)==0) vprint ("\n");
#endif
    if ((i%0x4000)==0) vprint (".");
  }
  crcr = pic_acmd (p, ADON_NULL, 0, 0, 0); 

 if (vt==4) {
  i=0x3;    pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0xfffc; pic_jtag (p, port, 0x1000e, &i, 16, 0);	/* XSC_DATA_SUCR */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
  i=0x3;    pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0x3;    pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0xffff; pic_jtag (p, port, 0x1000c, &i, 16, 0);	/* XSC_DATA_CCB */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
  i=0;      pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld */
  i=0x3;    pic_jtag (p, port, 0x100e8, &i, 8, 0);	/* ISPEN */
  i=0xc0;   pic_jtag (p, port, 0x10009, &i, 8, 0);	/* XSC_DATA_DONE */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
 } else if (vt==2) {
  i=0;      pic_jtag (p, port, 0x100e3, &i, -8, 0);	/* check device status */
  i=0xfffc; pic_jtag (p, port, 0x1000e, &i, 16, 0);	/* XSC_DATA_SUCR */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
  i=0xffff; pic_jtag (p, port, 0x1000c, &i, 16, 0);	/* XSC_DATA_CCB */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
  i=0xcc;   pic_jtag (p, port, 0x10009, &i, 8, 0);	/* XSC_DATA_DONE */
  i=0;      pic_jtag (p, port, 0x100ea, &i, 0, 60);	/* ISC_PROGRAM */
  i=0;      pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld - must be last */
 } else if (vt==1) {
  i=0x01;   pic_jtag (p, port, 0x100eb, &i, 16, 0);	/* FADDR */
  i=0x00;   pic_jtag (p, port, 0x1000a, &i, 0, 0);	/* SERASE */
  i=0;      pic_jtag (p, port, 0x100f0, &i, 0, 50);	/* conld - must be last */
 }
  udelay(1000);	/* needed on PIC6 to alleviate flaky behavior */

  pic_jtag (p, port, JTAG_CMD_IDCODE, &i, -32, 0); i &= 0x0FFFEFFF; /* strip rev */
  if (i!=IDCODE_PF01 && i!=IDCODE_PF16 && i!=IDCODE_PF32) print ("Bad PF IDCODE=%08x after load \n",i); 
  vprint (" done\n");
  pic_lock (p, LOCK_FREE);

  DONESPI:

  if ((vt==8) && (p->rev==0x0A)); else
  prepBitStream(loadbyte,nprog);

  tmp2 = (int_4*)malloc(nprog);
  if (vt==1) udelay(1000000);
  pic_unloadflash (p,(int_u1*)tmp2,nprog,0);
  for (i=j=0,errs=0; i<nprog/4; i++) { 
    if (tmp1[i]!=tmp2[i] && ++errs<24) print("Err at i=%08x of n=%08x  r=%08x w=%08x d=%08x\n",i,nprog/4,tmp2[i],tmp1[i],tmp2[i]^tmp1[i]); 
    if (i==0 && p->flashport>10) tmp1[i]=tmp2[i]=-1;	/* crc replaced -1 in this case for pmreload */
    j = CRC32(j,tmp2[i],32); 
  }
  print("Flash loaded with %d errors. CRC=0x%x\n",errs,j);
  if (vt<7) print("Reverse CRC=0x%08x Card=0x%08x\n",crc,crcr);
  free(tmp2);
  pic_acmd (p, ADON_JTAG, 0, 0, 0); 

  return (-errs);
}

int_4 pic_pmstat (PICSTRUCT *p, int_4 port) {
  int_4 stat=0, n=port-JTAG_PORT_PMOFF, sp6=0, v6p=0, rstat[6]={0xFFFFFFFF,0xAA995566,0x2800E001,0,0,0};
  if (port>=1 && port<=2 && isIOMx(p,port)) sp6=(p->mtype[port-1]==IOMT_RFXD)? 1:2;
  if (port>=11 && port<=14 && p->pmtype[n-1]==PMT_S6M) sp6=2;
  if (port>=11 && port<=14 && p->pmtype[n-1]==PMT_V6M) v6p=1;
  if (port>=11 && port<=14 && p->pmtype[n-1]==PMT_K8M) v6p=2;
  if (port>=11 && port<=14 && p->pmtype[n-1]==PMT_K8P) v6p=2;
  if (port==0 && p->k7!=0) v6p=2;
  if (port==9 && jtagfd!=NULL) sp6=2;	/* GPS on Tray */
  if (sp6) { 
    rstat[2]=0x29012000; rstat[3]=0x20002000;
    prepBitStream((int_u1*)rstat,4*4);
    pic_jtag (p, port, 0x20000000|JTAG_CMD_CFGIN, rstat, 32*4, 0);
    pic_jtag (p, port, 0x10000000|JTAG_CMD_CFGOUT, rstat, -16, 0);
    stat = bitrev4(rstat[0]<<16);
    if (sp6==2 && (stat&0xEFFB)!=0x2738) stat |= 0x80000000; /* S6M uses INIT_B pin */
    if (sp6!=2 && (stat&0xFFFB)!=0x3738) stat |= 0x80000000;
  } else {
    if (v6p>0) { rstat[3]=rstat[2]; rstat[2]=rstat[4]=rstat[5]=0x20000000; }
    prepBitStream((int_u1*)rstat,6*4);
    pic_jtag (p, port, 0x20000000|JTAG_CMD_CFGIN, rstat, 32*6, 0);
    pic_jtag (p, port, 0x10000000|JTAG_CMD_CFGOUT, rstat, -32, 0);
    stat = bitrev4(rstat[0]);
    if ((stat&0x90EF)!=0x10ec) stat |= 0x80000000;
  }
  return stat;
}

void setflash (PICSTRUCT *p, int port) 
{
  int_4 flash[] = {0xFFFFFFFF,0xAA995566,0x20000000,0x30020001,0x00000000,0x30008001,0x0000000F,0x20000000 };
  int_4 i,sel;
  if (port>10 && p->type==ICEPIC8) {
    for (i=0; i<8; i+=1) pic_wpbx (p,port-10,PPC_ICAP,bitrevN(flash[i],4));
    udelay(500000); /* allow reload */
  } 
  else if (p->type==ICEPIC7 || p->type==ICEPIC8) {
    for (i=0; i<8; i+=2) { pic_acmd (p, ADON_ICAP, bitrevN(flash[i],4), bitrevN(flash[i+1],4), 0); udelay(10000); }
    udelay(2000000); /* allow reload */
    pic_test (p,91,0); /* reattach PCI */
  }
  else {
    sel = getJTAGsel(p, port);
    sel |= 0x80000000;	/* apply disconnect */
    pic_acmd (p, ADON_JTAG, JTAG_TMS_CMDR,  0x00ee<<12, JTAG_MSK_CMD|sel); 
  }
}

#define RLEN 16
#define RNOP 0x20000000

int_4 mem2prom (int_1 *a, int_1 *b) 
{
  int_4 i,j,k,ibit,obit,iby,oby,ibi,obi,val;
  for (i=0; i<256; i++) {
    obit = i;
    oby = obit>>3;
    obi = obit%8;
    j = obit%16; 
    k = (j<4)? 15-j*2 : (j<8)? j*2-7 : (j<12)?  22-j*2 : j*2-16 ; 
    ibit = (k*16) + 15 - (obit>>4);
    iby = ibit>>3;
    ibi = ibit%8;
    val = a[iby] & (0x1<<ibi);
    if (val!=0) continue; 
    if (oby>=24) oby+=7; else if (oby>=16) oby+=5; else if (oby>=8) oby+=2;
    b[oby] = b[oby] & ~(0x1<<obi);
  }
  return 0;
}

int_4 pic_uflash (PICSTRUCT *p, int_4 *lval, int_4 len, int_4 dir)
{
  int_4 i,j,k,l,stat=0,*ltmp,wlen=len/4;
  int_4 rstat[RLEN]={-1,-1,-1,-1,-1,0xAA995566,RNOP,0x30008001,0x4,RNOP,0x30002001,0x0,RNOP,RNOP,0x3001A000,0x50000000};
  int_1 *atmp,*btmp,ftmp[32];
  int_4 diode = (p->mtype[1]==IOMT_DIODE) && (len>0) && (len<=0x20000);

  if (len==0x10000 || diode) {
    if (dir>0) stat = pic_loadflash (p, (int_u1 *)lval, len, 0);
    else       stat = pic_unloadflash (p, (int_u1 *)lval, len, 0);
  }
  else if (len==0x800) {
    rstat[RLEN-1] |= (len/4);
    if (dir>0) {
      ltmp = (int_4*)malloc(0x10000);
#if USR_ACCESS_OK
      for (i=0; i<RLEN; i++) ltmp[i]=rstat[i];
      memcpy((void*)(ltmp+i),(void*)lval,len); i+=wlen;
      for (; i<0x4000; i++) ltmp[i]=RNOP;
      stat = pic_loadflash (p, (int_u1 *)ltmp, 0x10000, 0);
#else
      atmp=(int_1*)lval; btmp=(int_1*)ltmp;
      stat = pic_unloadflash (p, (int_u1 *)ltmp, 0x2c00, 0);
      for (i=0; i<64; i++) mem2prom(atmp+i*32,btmp+0x320+i*164);
      stat = pic_loadflash (p, (int_u1 *)ltmp, 0x2c00, 0);
#endif
      free(ltmp);
    } else {
      btmp = (int_1*)lval;
      WR(REG_IOC, REG_IOC_ADDR|PM_TPORT_RST);
      RD(REG_IOC,j);
      WR(REG_IOC, REG_IOC_ADDR|PM_TPORT_ROM);
      for (i=0; i<len; i++) { RD(REG_IOC,j); btmp[i]=(int_1)j; }
    }
  }
  else if (dir<0) {
    stat = pic_unloadflash (p, (int_u1 *)lval, len, 0);
    if (p->flashport>10) lval[0]=-1;	/* crc replaced -1 in this case for pmreload */
  }
  else print("Illegal usr flash segment size=%d\n",len);
  return stat;
}

int_4 pic_bscan (PICSTRUCT *p, int_4 port) {
  int_4 i,data[40];
  pic_jtag (p, port, JTAG_CMD_SAMPLE, data, -40*32, 0);
  for (i=0; i<40; i++) print("BScan %04d = %08x\n",i*32,data[i]);
  return 0;
}

int_4 pic_idbscan (PICSTRUCT *p, int_4 port, int_4 indx) {
  int_4 i,data[40];
  pic_jtag (p, port, JTAG_CMD_SAMPLE, data, -(indx+1)*32, 0);
  return data[indx];
}


int_4 pic_loadmod (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4  nprep,i,j, addr, blk=DMA_INT_SIZE*4;
  int_u1 loadxtra[DMA_INT_SIZE*4+16];

  addr = IOC_PROG; 
  if (port==0) port=3;
  if (port&0x1) addr |= IOC_A; 
  if (port&0x2) addr |= IOC_B; 
  vprint ("Loading MOD%d code - %d bytes ... ",port,nprog); 

  pic_lock (p, LOCK_ALLOC);

  /* clear iomodule registers */
  pic_acmd (p,ADON_MOD,port,0,0); 
  /* pulse the module's nCONFIG line to reboot FPGA */
  pic_acmd (p,ADON_WRIOC,addr,IOP_PGM|IOP_RST,0);
  udelay(10);
  pic_acmd (p,ADON_WRIOC,addr,IOP_PGM,0);
  udelay(5);

  /* load full blocks of program data */
  for (i=0; nprog-i>=blk; i+=blk) {
    pic_transfer (p, (int_4 *)(loadbyte+i), blk, DMA_INT_ADDR, 2, 1);
    pic_acmd (p,ADON_LMOD,DMA_INT_ADDR,addr,blk/4);
  }

  /* prep buffer for xtra and 10+ clock init */ 
  nprep = (nprog+16)/4*4 - i;
  for (j=0; j<nprep; i++,j++) { 
    if (i<nprog) loadxtra[j]=loadbyte[i]; 
    else loadxtra[j]=0xFF;
  }

  vprint ("initializing ... %d ",i);
  pic_transfer (p, (int_4 *)loadxtra, nprep, DMA_INT_ADDR, 2, 1);
  pic_acmd (p,ADON_LMOD,DMA_INT_ADDR,addr,nprep/4);

  pic_acmd (p,ADON_WRIOC,addr,IOP_SBIT0,0);    /* disable pgm */
  vprint ("done\n"); 
  pic_lock (p, LOCK_FREE);

  return (nprog);
}

int_4 pic_wmodreg (PICSTRUCT *p, int_4 port, int_u4 maddr, int_u4 mdata, int_4 bits)
{
  int_4  j,addr,data,word,sel; 

  if (p->verbose>=3) print ("Loading MOD %d REG a=%x d=%x\n",port,maddr,mdata); 

  pic_lock (p, LOCK_ALLOC);

  if (p->isY) {
    sel = getJTAGsel(p,port);
    pic_acmd (p, ADON_WMOD, maddr, mdata, sel); 
  }
  else {
    addr = (port==1)? IOC_PROG|IOC_A : IOC_PROG|IOC_B;
    pic_acmd (p,ADON_WRIOC,addr,IOP_PGM,0);			/* enable pgm */
    pic_acmd (p,ADON_WRIOC,addr,IOP_PGM|IOP_CLK|IOP_CBIT0,0);	/* start bit */
    
    word = (int)mdata;						/* load register */
    for (j=0; j<bits; j++) {
      data = ((word>>j)&0x1)? IOP_SBIT0 : IOP_CBIT0; 
      pic_acmd (p,ADON_WRIOC,addr,IOP_PGM|IOP_CLK|data,0);	/* data bit */
    }
    pic_acmd (p,ADON_WRIOC,addr,IOP_PGM|IOP_CLK|IOP_CBIT0,0);	/* stop bit */
    pic_acmd (p,ADON_WRIOC,addr,IOP_PGM|IOP_CLK|IOP_SBIT0,0);	/* extra clock */
    pic_acmd (p,ADON_WRIOC,addr,IOP_SBIT0,0);			/* disable pgm */
  }
  pic_lock (p, LOCK_FREE);
  return (1);
}

int_4 pic_rmodreg (PICSTRUCT *p, int_4 port, int_u4 maddr, int_u4 *mdata, int_4 bits)
{
  int_4  j,addr,word,sel; 

  pic_lock (p, LOCK_ALLOC);

  if (p->isY) {
    sel = getJTAGsel(p,port);
    *mdata = pic_acmd (p, ADON_RMOD, maddr, 0, sel); 
  }
  else {
    print("Readback not supported until 5 series cards\n");
  }
  pic_lock (p, LOCK_FREE);

  if (p->verbose>=2) print ("Reading MOD %d REG a=%x d=%x\n",port,maddr,*mdata); 

  return (1);
}

int_4 pic_getstrflag (PICSTRUCT *p, char *name, char *str, int_4 flags)
{
  return findstrflag(name,p->config,str,flags);
}

int_4 pic_getintflag (PICSTRUCT *p, char *name)
{
  return findintflag(name,p->config);
}

int_4 pic_getintflagdef (PICSTRUCT *p, char *name, int_4 def)
{
  return findintflagdef(name,p->config,def);
}

real_8 pic_getdblflag (PICSTRUCT *p, char *name)
{
  return finddblflag(name,p->config);
}

real_8 pic_getdblflagdef (PICSTRUCT *p, char *name, real_8 def)
{
  return finddblflagdef(name,p->config,def);
}

int_4 pmrestart (PICSTRUCT *p, int_4 pm) {
  int_4 port = JTAG_PORT_PMOFF+pm;
  int_4 sel = getJTAGsel(p,port);
  vprint("Restarting PM=%d with PROM load\n",pm); 
  pic_acmd (p,ADON_RDIOB,sel|PM_RESET_GLOBAL,0,1); /* reset ALL clocks (DCM) */
  pic_acmd (p,ADON_RDIOB,sel|PM_RESET_DMAC,0,1); 
  return 1;
}
int_4 ispmreloadable (PICSTRUCT *p, int_4 port, int crc) {
  int_4 data[2],stat=0,fcrc,m,na,ofp;
  m = pic_getkeyl(p,port,KEY_UIDCODE);
  if (m!=0x1CE1D) return 0;
  ofp = p->flashport;
  p->flashport=port;
  m = spi_checkflash(p);
  if (m<=0) goto DONE; na = (m>1)? 4:3;
  spi_flash (p,FCMD_READ,0,na,(int_u1*)data,-8);	/* get 1st 8 bytes of flash */
  fcrc = bitrev4(data[0]);				/* CRC was written to 1st 4 bytes */
  v2print("Flash port=%d read %08x %08x CRC flash=%08x file=%08x\n",port,data[0],data[1],fcrc,crc);
  if (fcrc!=crc) goto DONE;
  stat=1;
  DONE:
  p->flashport=ofp;
  return stat;
}
int_4 pmreload (PICSTRUCT *p, int_4 port, int crc) {
  int_4 status=0, sel=getJTAGsel(p,port);
  if (findintflagdef("PMRELOAD",p->config,-1)==0) return 0;
  if (port==15) {
    if (ispmreloadable(p,11,crc)+ispmreloadable(p,12,crc)==2) status = 3;
  } else {
    if (ispmreloadable(p,port,crc)==1) status = (port==12)?2:1;
  }
  if (status>0) {
    pic_disable_nvme (p,port);	/* disable NVME drives */
    vprint("Reloading PM port=%d from PROM with crc=%08x\n",port,crc); 
    setflash(p,port);
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_GLOBAL,0,1); /* reset ALL clocks (DCM) */
    pic_acmd (p,ADON_RDIOB,sel|PM_RESET_DMAC,0,1); 
  }
  return status;
}

char syncpatt[8] = {-1,-1,-1,-1,-86,-103,85,102};

int_4 pic_loadfile (PICSTRUCT *p, char *name, int_4 flags)
{
  int_4 i,j,pm,ndata,ndatam,ndataf,nalloc,status,port,addr,mods,crc,isbit=0,sig,sigi,ver;
  char fname[MAXFNAME],lfstr[MAXFNAME],*cname,*sigb;
  int_u1 *data,*tmp; int_4 *lval;
  FILE *file;
  ICEHDR hcb;
  CHECK_MAGIC(p);

  port = flags&0xFF;
  if (port==0xFF) port = -1;
  addr = (flags>>8)&0xFF;
  mods = (flags>>16)&0xFF;
  flags = flags&0xFF000000;
  if (p->isY && flags == FLG_IOC) {
    vprint("Setting IOC load type: %s\n",name); 
    setIOCsig(p,0,name+2);
    resyncIOCclocks(p);
    return 0;
  }

  getpath (fname,"dat");

  if (name[0]=='*') {
    strcat(fname,"ice");
    cname = getCardName((int)p->type); /* upper case - to lower case */
    for (i=0; i<strlen(cname); i++) if (cname[i]>='A' && cname[i]<='Z') cname[i] += 32; 
    strcat(fname,cname);
    name++;
  }
  if (name[0]=='/') fname[0]=0; /* path is included */
  for (i=0, j=strlen(fname); name[i]!=0 && name[i]>' '; i++,j++) {
    fname[j] = name[i]; if (fname[j]>='A' && fname[j]<='Z') fname[j] += 32;
  } fname[j]=0;

  if (flags == FLG_MOD) {	/* pick up IO Module load file signature */
    j=findstrflagx("IOMxFPGA",p->config,lfstr,-1,port);
    if (j<=0); else if (lfstr[0]=='/') strcpy(fname,lfstr); else strcat(fname,lfstr);
    j = strlen(fname);
  }
  isbit = (j>4) && strcmp(fname+j-4,".bit")==0;
  if (!isbit) strcat(fname,".prm");

  vprint("Opening load file=%s ",fname); 
  file = fopen (fname,"rb");
  if (file == 0) {
    vprint(" not found\n"); 
    print("Error opening load file=%s\n",fname);
    return -1;
  }
  status = fread ((void *)(&hcb), 1, 512, file);

  crc=0;
  if (isbit) {
    data = (int_u1*)(&hcb); j=0;
    for (i=0; i<512 && chk(data+i,(int_u1*)syncpatt,8)!=0; i++) {
      if (data[i]==101) memcpy((void*)(&j),(void*)(data+i+1),4); 
    }
    ndataf = ndata = swapit(j);
    fseek(file,(long)i,SEEK_SET);
    vprint(" BIT size=%d\n",ndataf); 
  }
  else if (strncmp(hcb.version,"BLUE",4)==0 && strncmp(hcb.head_rep,"EEEI",4)==0 && strncmp(hcb.data_rep,"EEEI",4)==0) {
    for (i=0; i<L_keywords; i++) { /* prep main keywords string for parsing by findflag */
      if (hcb.keywords[i]==0) hcb.keywords[i]=',';
    } hcb.keywords[L_keywords-1]=0;
    ConvertEEEIDouble (&hcb.data_size, 2);
    ndataf = (int) hcb.data_size;
    ndata = findintflagdef("ICSZ",hcb.keywords,ndataf);
    crc = findintflagdef("CRC",hcb.keywords,0);
    vprint(" sizef=%d size=%d crc=%08x\n",ndataf,ndata,crc); 
  }
  else {
    vprint(" no Midas header\n"); 
    print("Load file=%s is not an EEEI Midas file\n",fname);
    return -1;
  }
  if (crc!=0) {	/* check for early out */
    i = -1;
    if (flags==FLG_IOC) i = 0;
    if (flags==FLG_MOD && port>=1 && port<=2) i = port;
    if (flags==FLG_MOD && port==3) i = 1;
    if (flags==FLG_PM && port>=1 && port<=4) i = port+2;
    if (flags==FLG_PM && port>=11 && port<=14) i = port-8;
    if (flags==FLG_PM && port==15) i = 3;
    if (i>=0 && crc==p->crcs[i]) { 
      vprint("Port=%d CRC=%08x already loaded\n",port,crc);
      pic_acmd (p,ADON_WR,CHECKSUM_ADDR+i,crc,0);
      fclose(file); return 0;
    }
    if (flags==FLG_PM && p->k7>=8 && port>10) {
      status = pmreload(p,port,crc);
      if (status>0) {
	if (status&1) { p->crcs[3]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+3,crc,0); }
	if (status&2) { p->crcs[4]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+4,crc,0); }
        fclose(file); return status;
      }
    }
  }
  if (flags != FLG_FLASH) nalloc = ndata;
  else if (ndata <= 0x020000) nalloc = 0x020000;
  else if (ndata <= 0x200000) nalloc = 0x200000;
  else if (ndata <= 0x400000) nalloc = 0x400000;
  else nalloc = ndata;
  data = (int_u1 *)malloc (nalloc);
  status = fread ((void *)data,1,ndataf,file);
  if (status == ndataf);
  else if (isbit && ndata-status < 256) ndata = status; /* .bit file EOF often a bit early */
  else {
    print("Bad data=%d read=%d from load file: %s\n",ndata,status,fname);
    return -1;
  }
  fclose (file);
  if (ndata != ndataf) { /* 2 pass huffman uncompress */
    ndatam = findintflagdef("ICSZM",hcb.keywords,ndataf);
    tmp = (int_u1 *)malloc (nalloc+512);
    pic_compress(data,ndataf,tmp,ndatam,-1);
    pic_compress(tmp,ndatam,data,ndata,-1);
    free(tmp);
  }
  if (isbit) Swap((int_4*)data,ndata/4,32);
  for (;ndata<nalloc;) data[ndata++]=0xFF;
  lval = (int_4*)data;
  for (i=crc=0; i<ndata/4; i++) crc = CRC32(crc,lval[i],32);

  status = 0;
  if (port==99) {
    status = crc;
  }
  else if (flags == FLG_SHARC) {
    status = pic_loadsharc (p, data, ndata/6, FLG_BOOT);
  } 
  else if (flags == FLG_IOC || flags == FLG_BIOC) {
    setIOCsig(p,0,name+1);
    if (crc==p->crcs[0]) { 
      pic_acmd (p, ADON_MOD, 3, 0, 0);
      pic_acmd (p,ADON_WRIOC,IOC_PROG|IOC_A|IOC_B,0,0);
      vprint("IOC CRC=%08x already loaded\n",crc);
    }
    else {
      status = pic_loadioc (p, data, ndata, FLG_BOOT);
      p->crcs[1]=p->crcs[2]=0; /* IOMs must be reloaded now */
    }
    if (flags == FLG_IOC) { /* reloading */ 
      resyncIOCclocks(p);
      reloadModuleCode(p);
    }
    p->crcs[0]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+0,crc,0);
  } 
  else if (flags == FLG_PM) {
    pm = (port>JTAG_PORT_PMOFF)? port-JTAG_PORT_PMOFF : port;
    if (port<=0) pm = p->pmi;
    if (port==JTAG_PORT_PMALL && crc==p->crcs[2+1] && crc==p->crcs[2+2]) { vprint("PM ports=All CRC=%08x already loaded\n",crc); }
    else if (port!=JTAG_PORT_PMALL && crc==p->crcs[2+pm]) { vprint("PM port=%d CRC=%08x already loaded\n",port,crc); }
    else status = pic_loadvirtex (p, port, data, ndata, FLG_BOOT);
    if (status>=0) for (i=1; i<=p->mpm; i++) if (port==JTAG_PORT_PMALL || pm==i) { p->crcs[2+i]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+2+i,crc,0); }
  }
  else if (flags == FLG_VIRTEX) {
    if (port<0) port = 0;
    status = pic_loadvirtex (p, port, data, ndata, FLG_BOOT);
  }
  else if (flags == FLG_PPC) {
    if (p->type==ICEMBT4) port = 1; 
    status = pic_loadppc (p, port, data, ndata, FLG_BOOT);
  }
  else if (flags == FLG_JVM) {
    i = (mods==2)? FLG_NVME : (mods==1)? FLG_NIO : FLG_BOOT;
    status = pic_loadjvm (p, port, (int_2 *)data, ndata/2, i);
  }
  else if (flags == FLG_SLR) {
    for (i=0; i<mods; i++) data[4*(addr+i)+1] = (p->regX>>(i*8))&0xFF;
    status = pic_loadslr (p, port, (int_4 *)data, ndata/4);
  }
  else if (flags == FLG_FC) {
         if (hcb.format[1]=='I') status = pic_loadfc (p, port, (int_2 *)data, ndata/2, FLG_EEEI);
    else if (hcb.format[1]=='L') status = pic_loadfc (p, port, (int_2 *)data, ndata/4, FLG_EEEI|FLG_LONG);
    else if (hcb.format[1]=='F') status = pic_loadfc (p, port, (int_2 *)data, ndata/4, FLG_EEEI|FLG_FLOAT);
    else print("Unallowable format=%.2s in file=%s for filter tap load\n",hcb.format,fname);
  } 
  else if (flags == FLG_FLASH) {
    if (p->flashport>10) lval[0]=crc;	/* override flash[0]=-1 with crc for pmreload from flash check */
    if (p->type==ICEPIC6 && p->rev==0x0F && strstr(fname,"icepod")==0) 
      print("Err: ICEPOD flash file: %s must contain the string icepod.\n",fname);
    else if (port==0) status = pic_loadflash (p, data, ndata, 0);
    else { print("LoadFlash for %s CRC32=0x%08x\n",name,crc); status=crc; }
  } 
  else if (flags == FLG_MOD) {
    if (port==3 && crc==p->crcs[1] && crc==p->crcs[2]) { vprint("IOM port=Both CRC=%08x already loaded\n",crc); }
    else if (port!=3 && crc==p->crcs[port]) { vprint("IOM port=%d CRC=%08x already loaded\n",port,crc); }
    else if (p->isY) {
      if (isIOMx(p,port)) {
        if (p->mtype[0]==IOMT_A2D && p->mrev[0]==18) pic_acmd (p,ADON_WRIOC, IOC_A|IOC_B|IOC_FLAGS, 0,0); /* disable clocks for a2dm1x */
        status = pic_loadvirtex (p, port, data, ndata, FLG_BOOT);
        if (p->mtype[0]==IOMT_A2D && p->mrev[0]==18) setIOCsig(p,0,p->iocc);
      } 
      else status = pic_loadstratix (p, port, data, ndata, 0);
    }
    else status = pic_loadmod (p, port, data, ndata, 0);
    if (status>=0 && port&1) { p->crcs[1]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+1,crc,0); }
    if (status>=0 && port&2) { p->crcs[2]=crc; pic_acmd (p,ADON_WR,CHECKSUM_ADDR+2,crc,0); }
  }
  else if (flags == FLG_STRATIX) {
    status = pic_loadstratix (p, port, data, ndata, 0);
  }
  else if (flags == FLG_LUT) {
    flags = findintflag("FLAGS",hcb.keywords);
    status = pic_loadlut (p, port, data, ndata, flags);
  }
  else if (flags == FLG_COEF) {
    status = pic_loadcoef (p, port, addr, (real_4*)data, ndata/4, mods);
  }
  else {
    print("Unsupported loadfile flags=%08x on file=%s\n",flags,fname);
  }
  free (data);

  return status;
}

int_4 checkModuleMode (PICSTRUCT *p, int_4 port) {
  int_4 mtype = p->mtype[port-1], amtype=abs(mtype), mrev = p->mrev[port-1];
  char mmode = p->socc[port-1], mode = 'H';
  if (amtype==IOMT_GPS) mode=mmode; 
  if (amtype==IOMT_D2A && mrev==8) mode='S'; 
  if (amtype==IOMT_D2A && mrev==14) mode=(mmode=='H')?'H':'D'; 
  if (amtype==IOMT_D2A && mrev>=15) mode=(mmode=='H')?'H':'D'; 
  if (mtype==IOMT_DR2D || mtype==IOMT_RFXD) mode='D'; 
  if (amtype==IOMT_D2E || amtype==IOMT_D2T) mode='S'; 
  if (amtype==IOMT_D2LV || amtype==IOMT_D2PSE) mode='S'; 
  if (amtype==IOMT_DXSFP || amtype==IOMT_DXDSFP) mode='R'; 
  if (amtype==IOMT_DXMSAS || amtype==IOMT_DXSFPP) mode='R'; 
  if (amtype==IOMT_DIODE || amtype==IOMT_DXQSFP) mode='R'; 
  if (amtype==IOMT_DXUPAC) mode='R'; 
  if (mmode==mode || mtype==IOMT_NONE || mtype==IOMT_TEST) return 0;
  print("Incorrect SOC Load=%.4s for Module=%d Type=%s\n",p->socc,port,getIomName(mtype,mrev));
  return -1;
}

void setIOCsig (PICSTRUCT *p, int_4 flags, char *name) {
  int_4 i,j,iocc,flg; 
  for (i=0; i<4; i++) {
    if (i>=(int_4)strlen(name)) p->iocc[i]=' '; 
    else if (name[i]>='a' && name[i]<='z') p->iocc[i]=name[i]-32;
    else p->iocc[i]=name[i];
  }
  iocc = *((int_4 *)p->iocc);
  iocc = pic_acmd (p, ADON_WR, IOCCODE_ADDR, iocc, 0); 
  if (p->isY) for (i=0; i<2; i++) {
    flg = 0;
    if (checkModuleMode(p,i+1) == 0) {
      if (p->mtype[i]==IOMT_NONE);
      else if (p->mtype[i]==IOMT_UNKNOWN);
      else flg |= IOC_FLG_GOOD;
    }
    j = (i==1 && p->iocc[3]==' ')? 2 : i+2;
    if (p->iocc[0]=='T') {
      flg |= IOC_FLG_LOOP;
      flg |= ((p->iocc[1]=='1') == (i==0))? IOC_FLG_IN : IOC_FLG_OUT;
    }
    if (p->iocc[i]=='I' || p->iocc[i]=='U' || p->mtype[i]==IOMT_TEST) flg |= IOC_FLG_IN;
    if (p->iocc[i]=='O' || p->iocc[i]=='U' || p->mtype[i]==IOMT_TEST) flg |= IOC_FLG_OUT;
    if (p->iocc[j]=='S') flg |= IOC_FLG_SDDS; 
    if (p->iocc[j]=='R') flg |= IOC_FLG_RAMP;
    if (p->iocc[j]=='W') flg |= IOC_FLG_WHITE;
    if (isIOMx(p,i+1))   flg |= IOC_FLG_ISX;
    if (p->qdrx&(i+1))   flg |= IOC_FLG_QDRX;
    if (findintflagdef("IRBF",p->config,0)>0) flg |= IOC_FLG_IRBF;
    if (findintflagdef("TOUT",p->config,0)>0) flg |= IOC_FLG_TOUT;
    if (findintflagdef("OHOS",p->config,0)>0) flg |= IOC_FLG_OHOS;
    if (findintflagdef("NOFC",p->config,0)>0) flg |= IOC_FLG_NOFC;
    if (findintflagdef("GGO",p->config,-1)==0) flg |= IOC_FLG_MGGO;
    if (findintflagdef("ZFILL",p->config,0)>0) flg |= IOC_FLG_ZFILL;
    if (p->mtype[i]==IOMT_A2D && p->mrev[i]==20) flg |= (IOC_FLG_NOFC|IOC_FLG_QDRM);
    if (findintflagdef("QDRM",p->config,-1)==0) flg &= ~IOC_FLG_QDRM;
    if (findintflagdef("SEQF",p->config,0)>0) { flg &= ~IOC_FLG_QDRM; flg |= IOC_FLG_SEQF; }
    if (p->mtype[i]==IOMT_D2AWG && p->mrev[i]==3) flg |= IOC_FLG_OHOS;
    if (p->mtype[i]==IOMT_A2D && p->mrev[i]==20 && i==1 && isOM20(p)) flg |= IOC_FLG_M20F;
    if (p->type==ICEPIC8 && (p->flags&FLG_VHS)==0) flg |= IOC_FLG_CSEL;
    if ((flg&(IOC_FLG_LOOP|IOC_FLG_RAMP|IOC_FLG_WHITE))!=0) flg &= ~IOC_FLG_QDRX;
    j = IOC_FLAGS | ((i==0)? IOC_A : IOC_B);
    if (flags!=0) pic_acmd (p,ADON_WRIOC, j, flg|flags,0); 
    pic_acmd (p,ADON_WRIOC, j, flg,0); 
    vprint("IOC port=%d flags=%08x\n",i+1,flg);
  }
}

int_4 getIOCsig (PICSTRUCT *p, int_4 flag) {
  int_4 ip,jp,iocc,socc,port,mtype,mrev; char c;
  port = flag&0xFF;
  flag = flag&0xFF00;
  ip = (port==2)? 1:0;
  mtype = p->mtype[ip];
  mrev = p->mrev[ip];
  if (strncmp(p->iocc,"XXXX",4)==0) {
    iocc = pic_acmd (p, ADON_RD, IOCCODE_ADDR, 0, 0); 
    *((int_4 *)p->iocc) = iocc;
  }
  c = p->iocc[3];
  jp = (c=='R'||c=='W'||c=='X'||c=='Z')? ip+2 : 2;
  if (p->isY && strncmp(p->socc,"XXXX",4)==0) {
    socc = pic_acmd (p, ADON_RD, SOCCODE_ADDR, 0, 0); 
    *((int_4 *)p->socc) = socc;
  }
  switch (flag) {
  case IS_MUXABLE:
    if (p->iocc[0]=='T') return 0;
    if (p->iocc[0]!=p->iocc[1]) return 0;
    /* still need muxclk to be muxable */
  case IS_MUXCLK:
    if (p->iocc[ip]=='O') return 1;	/* output port */
    if (p->iocc[jp]=='X') return 1;
    if (p->iocc[jp]=='Z') return 1;
    /* the internal modes are also muxclk */
  case IS_INTERNAL:
    if (p->gmcs==0) return 0; /* explicit MUXCLK=N */
    if (p->iocc[jp]=='W') return 2;
    if (p->iocc[jp]=='R') return 1;
  case IS_TEST:
    if (p->iocc[0]=='T') return 1;
    break;
  case IS_INPUT:
    if (p->iocc[ip]=='I') return 1;	
    if (ip==0 && p->iocc[0]=='T' && p->iocc[1]=='1') return 1;
    if (ip==1 && p->iocc[0]=='T' && p->iocc[1]=='2') return 1;
    break;
  case IS_OUTPUT:
    if (p->iocc[ip]=='O') return 1;	
    if (ip==0 && p->iocc[0]=='T' && p->iocc[1]=='2') return 1;
    if (ip==1 && p->iocc[0]=='T' && p->iocc[1]=='1') return 1;
    break;
  case IS_OBMC:
    if (abs(mtype)==IOMT_D2A && mrev>=9) return 1; 
    if (abs(mtype)==IOMT_D2ES) return 1; 
    if (abs(mtype)==IOMT_D2AWG) return 1;
    if (abs(mtype)==IOMT_DXTGSDDS) return 1;
    if (abs(mtype)==IOMT_DXTGVITA) return 1;
    if (abs(mtype)==IOMT_D2RF) return 1;
    if (mtype==IOMT_LB2D) return 1;
    if (mtype==IOMT_RFXD) return 1;
    break;
  }
  return 0;
}

int_4 getPMsig (PICSTRUCT *p, int_4 flag) {
  int_4 i,j,k,sel,poff=JTAG_PORT_PMOFF+1;
  for (i=0; i<4; i++) p->pmtype[i]=PMT_NONE;
  for (i=0; i<p->mpm; i++) {
    p->pmrev[i]=0;
    p->pmtype[i]=PMT_DTDMX;
    pic_jtag (p, i+poff, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF; /* strip rev */
    if (j==IDCODE_DTDMX) { p->pmtype[i]=PMT_DTDMX; continue; }
    p->pmtype[i]=PMT_S6M;
    pic_jtag (p, i+poff, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF; /* strip rev */
    if (j==IDCODE_K8M)  { p->pmtype[i]=PMT_K8M; k=pic_getkeyl(p,-(JTAG_PORT_PMOFF+1+i),KEY_HWREV); p->pmrev[i]=(k==7)?0:k; continue; }
    if (j==IDCODE_K8P)  { p->pmtype[i]=PMT_K8P; continue; }
    if (j==IDCODE_S6M)  { p->pmtype[i]=PMT_S6M; continue; }
    if (j==IDCODE_S6MX) { p->pmtype[i]=PMT_S6M;  p->pmrev[i]=2; continue; }
    p->pmtype[i]=PMT_A8M;
    pic_jtag (p, i+poff, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF; /* strip rev */
    if (j==IDCODE_A8M)  { p->pmtype[i]=PMT_A8M; continue; } 
    p->pmtype[i]=PMT_DTDM;
    pic_jtag (p, i+poff, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF; /* strip rev */
    if (j==IDCODE_DTDM) { p->pmtype[i]=PMT_DTDM; continue; }
    if (j==IDCODE_ZPPM) { p->pmtype[i]=PMT_ZPPM; continue; }
    if (j==IDCODE_V5M)  { p->pmtype[i]=PMT_V5M;  continue; }
    if (j==IDCODE_V5M)  { p->pmtype[i]=PMT_V5M;  continue; }
    if (j==IDCODE_V6M)  { p->pmtype[i]=PMT_V6M;  
      sel = getJTAGsel(p, i+poff);
      k = pic_acmd (p,ADON_RDIOB,sel|0x0088,0,1); 
      p->pmrev[i]=(k==0x8C)?2:0; continue;
    }
    if (j!=0x0FFFFFFF) vprint("Bad PM signature read=%08x\n",j);
    p->pmtype[i]=PMT_NONE;
  }
  return 0;
}

int_4 getSP6id (PICSTRUCT *p, int_4 side) {
  int_4 lval,id=0;
  int_4 bsid,bsidx;
  pic_jtag (p, side, JTAG_CMD_UIDCODE, &lval, -32, 0);
  if (lval == 0x0001CE1D && findintflag("NOSP6ID",p->config)<=0) {
    id = (pic_jpmemrd(p,side,0x7F00,4) >> 24)&0xFF;
  } else {
    bsid  = pic_idbscan(p,side,0) & 0xFFFFFF;
    bsidx = pic_idbscan(p,side,5) & 0xFFFFFF;
    vprint("BSID side=%d sig=%08x\n",side,bsid);
         if (bsid==0x7db7db && (bsid&0x00104900)==0x00100100) id = 0xAE; /* D2RF */
    else if (bsid==0x6fffdb && (bsid&0x00104900)==0x00004900) id = 0xAD; /* LB2D */
/*    else if (bsid==0x6db6db && (bsid&0x00104900)==0x00000000 && bsidx==0xb6db6d && (bsidx&0x00412400)==0x00000000) id = 0xAB; /* D2AWG */
/*    else if (bsid==0x6db6db && (bsid&0x00104900)==0x00000000 && bsidx==0xf6db6d && (bsidx&0x00412400)==0x00400000) id = 0xAF; /* A2DM1X */
    else {
      pic_loadfile (p, "icesp6_h", FLG_MOD|side);
      id = (pic_jpmemrd(p,side,0x7F00,4) >> 24)&0xFF;
    }
  }
  return id;
}

int_4 pic_getGPSrev (PICSTRUCT *p, int_4 port) {
  int_4 uid,bsid=0,rev=0,rate;
  pic_jtag (p, port, JTAG_CMD_UIDCODE, &uid, -32, 0);
  if (uid==0x1CE1D) {				/* FPGA already loaded */
    rev = pic_jpmemrd (p,port,0x4000,1);
    rate = pic_jpmemrd (p,port,0x4004,1);
    if (rate>30 && rate<60) return rev;		/* still check ocxo */
  }
  pic_loadfile (p, "icegpsx_d", FLG_MOD|port);	/* assume new */
  rate = pic_jpmemrd (p,port,0x4004,1);
  if (rate>30 && rate<60) return 2;		/* check ocxo */

  pic_loadfile (p, "icegps_d", FLG_MOD|port);	/* try old */
  rate = pic_jpmemrd (p,port,0x4004,1);
  if (rate>30 && rate<60) return 1;		/* check ocxo */

  v2print("Unknown GPS scrape uid=%08x rev=%d rate=%d - assume rev=1\n",uid,rev,rate); 
  rev=1; 
  return rev;
}

int_4 getIOMsig (PICSTRUCT *p, int_4 flag) {
  int_4 i,j,id, stat, rios,tlock,rlock,trate;
  for (i=0; i<p->miom; i++) {
    p->mtype[i]=IOMT_UNKNOWN;
    if (p->mtype[i]==IOMT_UNKNOWN || p->mtype[i]==IOMT_NONE) {
      p->mtype[i]=IOMT_RFXD; /* force length of command register */
      pic_jtag (p, i+1, JTAG_CMD_IDCODE, &j, -32, 0); 
      if (p->verbose==3) printf("JTag Mod=%d IDCode=%08x\n",i+1,j);
      j &= 0x0FFFFFFF; /* strip rev */
      if (j==IDCODE_RFXD) { p->mtype[i]=IOMT_RFXD; continue; }
      if (j==IDCODE_SP6)  { p->mtype[i]=IOMT_A2D; p->mrev[i]=14; continue; }
      if (j==IDCODE_SP6X) { p->mtype[i]=IOMT_A2D; p->mrev[i]=18; p->mtype[0]=IOMT_A2D; p->mrev[0]=18; continue; } /* M18 Devel Module */
      if (j==IDCODE_SP6Y) { 
        id = getSP6id(p,i+1);
        if (id==0xAF) { p->mtype[i]=IOMT_A2D; p->mrev[i]=18; p->mtype[0]=IOMT_A2D; p->mrev[0]=18; continue; } /* M18 Prod Module */
        if (id==0xAE) { p->mtype[i]=IOMT_D2RF; p->mrev[i]=0; continue; } /* D2RF/RF2D Detection */
        if (id==0xAD) { p->mtype[i]=IOMT_LB2D; p->mrev[i]=3; continue; } /* LB2DM3 Detection */
        if (id==0xA7) { p->mtype[i]=IOMT_LB2D; p->mrev[i]=3; continue; } /* LB2DM3 Detection */
        if (id==0xAB) { p->mtype[i]=IOMT_D2AWG; p->mrev[i]=3; continue; } /* D2AWGM3 Detection */
        if (id==0x8B) { p->mtype[i]=IOMT_D2AWG; p->mrev[i]=3; continue; } /* D2AWGM3 Detection */
        if (id==0x9B) { p->mtype[i]=IOMT_D2AWG; p->mrev[i]=3; continue; } /* D2AWGM3 Detection */
      }
      if (j==IDCODE_GPS)  { p->mtype[i]=IOMT_GPS; p->mrev[i]=pic_getGPSrev(p,i+1); continue; }
      p->mtype[i]=IOMT_UNKNOWN;
    }
  }
  if (findintflag("NOIOM",p->config)>0) stat = 0;
  else stat = detect_rumel_mods(p);
  if (p->type==ICEPIC6) {	/* check the RIO ports */
    rios = PPC_RIO_ENABLE;
    pic_msg (p,0,PKTF_BUS_WR,PPC_RIO_CTL,&rios,4,0,-1);
    udelay(10000);
    pic_msg (p,0,PKTF_BUS_RD,PPC_RIO_CTL,&rios,0,4,-1);
    for (i=0; i<p->miom; i++) {
      if (p->mtype[i]==IOMT_NONE) {
        tlock = (rios>>(i*16+0)) & 0xF;
        rlock = (rios>>(i*16+4)) & 0xF;
        trate = (rios>>(i*16+8)) & 0xFF;
        if (tlock==0xF && abs(trate-0xA8)<=1) { 
	  p->mtype[i] = (rlock==0xF)?IOMT_MSASXD:IOMT_DXMSAS; }
        else if (tlock==0xF && abs(trate-0xAF)<=1) { 
	  p->mtype[i] = (rlock==0xF)?IOMT_SFPPXD:IOMT_DXSFPP; }
        else if (tlock&0x1 && abs(trate-0x8C)<=1) { rlock&=3;
	  p->mtype[i] = (rlock==3)?IOMT_DSFPXD:(rlock==1)?IOMT_SFPXD:IOMT_DXDSFP; }
      }
    }
    rios = PPC_RIO_DISABLE;
    pic_msg (p,0,PKTF_BUS_WR,PPC_RIO_CTL,&rios,4,0,-1);
    if (p->mtype[1]==IOMT_DXDSFP || p->mtype[1]==IOMT_SFPXD) { /* check for DIODE module */
      i=p->mtype[1]; p->mtype[1]=IOMT_DIODE;
      pic_jtag (p, 2, JTAG_CMD_IDCODE, &j, -32, 0); j &= 0x0FFFFFFF; 
      if (j==IDCODE_PF01) p->mtype[0]=IOMT_SFPXD; else p->mtype[1]=i; /* ok? IOM1=SFPXD,IOM2=DIODE else revert */
    }
  }
  if (p->type==ICEPIC8) {	/* check the RIO ports */
    for (i=0; i<p->miom; i++) if (p->socc[i]=='R') { p->mtype[i]=(p->rev==4)?IOMT_UPACXD:IOMT_QSFPXD; p->mrev[i]=0; }
  }
  if (p->mrev[0]==19) p->mrev[0]=p->mrev[1]=20;	/* old M20 acts as just M20 from here on out */
  return stat;
}

int_4 getPortSig (PICSTRUCT *p, int_4 port, int_4 flag, int_4 *sig, int_4 *ver) {
  int_4 data[2], i,n, sel; 
  char *lsig;
  *sig = 0;
  *ver = 0;

  if (p->isY);
  else if (p->type==ICEMBT4 && port==(JTAG_PORT_PMOFF+1));
  else return 0;

  if (port>JTAG_PORT_PMOFF && p->pmtype[port-JTAG_PORT_PMOFF-1]<=PMT_NONE) return 0;

  if (port>JTAG_PORT_PMOFF) {
    sel = getJTAGsel(p, port);
    *ver = pic_acmd (p,ADON_RDIOB,sel|0x0080,0,4);	/* PM test port addr */
    *sig = pic_acmd (p,ADON_RDIOB,sel|0x0084,0,4); 
  } else {
    pic_lock (p,LOCK_ALLOC);
    WR(REG_IOC, REG_IOC_AINC|(flag?PM_TPORT_DSG:PM_TPORT_CFG));		/* MB test port addr $dsg vs $hsv */
    *ver=0; for (n=0; n<4; n++) { RD(REG_IOC, i); *ver |= ((i&0xFF)<<(n*8)); }
    *sig=0; for (n=0; n<4; n++) { RD(REG_IOC, i); *sig |= ((i&0xFF)<<(n*8)); }
    pic_lock (p,LOCK_FREE);
  }
  lsig = (char*)sig;
  for (i=0; i<4; i++) if (lsig[i]<' ' || lsig[i]>'~') lsig[i]='~';
  return 0;
}

int_4 feed2dmac (PICSTRUCT *p, int_4 feed) 
{
  int_4 port = (feed&0xE);
  int_4 pmod = (feed>>4)&0xF;
  int_4 dmac = (feed&1);
       if (port==PM_CORA           ) dmac += (3 + pmod*2);
  else if (port==PM_HYPA && pmod==0) dmac +=  1;
  else if (port==PM_TUNA && pmod==0) dmac +=  5 + dmac;
  else if (port==PM_TUNA && pmod>0 ) dmac += (13 + (pmod-1)*32);
  else dmac = 0;
  return dmac;
}

int_4 pic_ioport (PICSTRUCT *p, int_4 ptype, int_4 pindex, int_4 dmac, 
	int_4 dir, int_4 bits, int_4 rate, real_8 fcny, 
	int_4 dec, int_4 gain, int_4 flags)
{
  int_4 m,n,side,pside,dmac2,bits2,mcfg1,mcfg2,maxrate,stat,g0,feed,ovsr,i;
  double pfcny,ratio;
  char muxclks[] = "NICABXPDQ";
  DMASTRUCT *dma;
  CHECK_MAGIC(p);

  if (ptype == -1) ptype  = p->ptype; 
  if (pindex == -1) pindex  = p->pindex; 
  side = (pindex&1)?1:2;
  pside = (p->inp>0)? p->inp : side;

  pic_lock (p,LOCK_ALLOC);

  /* turn of QDR when needed for test modes */
  if (p->qdrx>0 && (flags&FLG_DISABLE)==0) {
    if (getIOCsig(p,IS_TEST)) p->qdrx=0;
    if (getIOCsig(p,IS_INTERNAL|1)) p->qdrx=0x2;
    if (getIOCsig(p,IS_INTERNAL|2)) p->qdrx=0x1;
    if (p->flags&FLG_LSBX) { printf("Turning off QDRX=%d for LSBX flag operation.\n",p->qdrx); p->qdrx=0x0; }
  }

  if ((ptype==IOPT_TUNER || ptype==IOPT_TBANK || ptype==IOPT_CORE || ptype==IOPT_MCORE) 
      && (p->tflags&FLG_PRESAMP) && (flags&FLG_DISABLE)==0) {
    ovsr = ((p->tflags&FLG_POVSR)==0)? 1 : p->ovsr[pside-1];
    p->res_dec = (int)(1.0/p->res_ratio + 0.5);
    p->res_ovsr = (rate/ovsr>60000000)? 1:2;
    p->res_ovsr = findintflagdef("PREOVSR",p->config,p->res_ovsr); 
    stat = pic_rpb (p,0,pic5tadrtbl[pside]);
    if ((stat&0x1)==0) { /* not running yet */
      pic_tuner_ratio (p, pside, p->res_ratio, 0);
      g0 = findintflagdef("PREGAIN",p->config,0);
      m = findintflagdef("MBITS",p->config,16);
      n = findintflagdef("NBITS",p->config,16);
      pfcny = finddblflagdef("PRETUNE",p->config,(m<0)?0.0:0.5);
      stat = pic_tuner_gcfpga (p, pside, n, rate/ovsr, pfcny, 1, g0, FLG_PRESAMP);
    }
  }

  if (ptype == IOPT_NONE) {
   if ( (flags&FLG_DISABLE) == 0) {
    print("Writing test ramp for acquisition/playback \n");
    pic_test (p,3,0);
    dmac = 1; 
    p->flags |= FLG_TEST;
   }
  }
  else if (ptype == IOPT_MODULE) {
         if (p->isY) maxrate = 520000000;		/* max IOPort rate */ 
    else if (p->isX) maxrate = 2*70*1000000*3/4;	/* max 4-series */ 
    else             maxrate = 2*38*1000000*3/4;	/* max 3-series */
    p->mbits = bits;
    dmac2=0;
    if (pindex==3) { 						/* mux port setup */
      if (bits>0) bits2 = bits/2; else bits2 = -bits;
      dmac2 = pic_mport (p, 2, -1, dir, bits2, rate, gain, flags|FLG_MUX|FLG_SGO);
      dmac  = pic_mport (p, 1, -1, dir, bits2, rate, gain, flags|FLG_MUX);
      /* slave the B side to the A side's main or secondary dma channel */
      if (dmac<=0 || (flags&FLG_DISABLE)!=0);
      else if (p->dma[dmac-1].slave==0) p->dma[dmac-1].slave = dmac2; 
      else p->dma[p->dma[dmac-1].slave-1].slave = dmac2;
      if (p->isY) { p->dma[dmac2-1].mcfg &= ~IOM_LEN;  p->dma[dmac2-1].enb = 0; }	/* all is DMAd through A side */
    } 
    else if (p->isY==0 && (p->flags&FLG_VHS)!=0 && dir>0 && 	/* VHS dual output port setup */
	((p->flags&FLG_DUAL)!=0 || rate/8*bits>maxrate)) { 
      if (pindex==1) {
        dmac2 = pic_mport (p, 2, -1, dir, bits, rate, gain, flags|FLG_DUAL|FLG_SGO);
        dmac  = pic_mport (p, 1, -1, dir, bits, rate, gain, flags|FLG_DUAL);
      } else {
        dmac2 = pic_mport (p, 1, -1, dir, bits, rate, gain, flags|FLG_DUAL|FLG_SGO);
        dmac  = pic_mport (p, 2, -1, dir, bits, rate, gain, flags|FLG_DUAL);
        p->dma[dmac-1].mcfg |= IOM_UOPT;
        p->dma[dmac2-1].mcfg |= IOM_UOPT;
      }
      /* slave the B side to A side's secondary dma channel */
      if (dmac<=0 || (flags&FLG_DISABLE)!=0);
      else p->dma[p->dma[dmac-1].slave-1].slave = dmac2;
    } 
    else if (p->isY==0 && (p->flags&FLG_VHS)!=0 && dir<0 &&  	/* VHS dual input port setup */
        getIOCsig(p,IS_MUXABLE) && ((p->flags&FLG_DUAL)!=0 || rate/8*bits>maxrate)) { 
      if (pindex==1) {
        dmac2 = pic_mport (p, 2, -1, dir, bits, rate, gain, flags|FLG_DUAL|FLG_ALT|FLG_SGO);
        dmac  = pic_mport (p, 1, -1, dir, bits, rate, gain, flags|FLG_DUAL);
      } else {
        dmac2 = pic_mport (p, 2, -1, dir, bits, rate, gain, flags|FLG_DUAL|FLG_SGO);
        dmac  = pic_mport (p, 1, -1, dir, bits, rate, gain, flags|FLG_DUAL|FLG_ALT);
      }
      /* slave the B side to A side's secondary dma channel */
      if (dmac<=0 || (flags&FLG_DISABLE)!=0);
      else p->dma[p->dma[dmac-1].slave-1].slave = dmac2;
    } 
    else {							/* single port setup */
      if ((p->flags&FLG_VHS)!=0 && rate/8*bits>maxrate && !p->isY)  
	print("VHS rate max is %d Mby/s without MUXCLK \n", maxrate/1000000); 
      dmac = pic_mport (p, pindex, dmac, dir, bits, rate, gain, flags);
    }
    /* if auto slaved, watch for XTGO->XSTGO case */
    if (dmac2 != 0 && (p->flags&FLG_XSTGO)==FLG_XTGO) p->dma[dmac2-1].mcfg &= ~(IOM_SGO);
    pic_acmd (p,ADON_WRIOC,(pindex<<6)|IOC_FRAME,0,1);
  }
  else if (ptype == IOPT_STREAM) {
    dmac = pic_sport (p, pindex, dmac, dir, bits, bits, rate, flags);
  }
  else if (ptype == IOPT_TUNER) {
    dmac = pic_tport (p, pindex, dmac, dir, bits, rate, fcny, dec, gain, flags);
  }
  else if (ptype == IOPT_TBANK) {
    if (p->ptype == IOPT_NONE) {
      /* TBPORTs badly need the disable step done already if default port */
      pic_tbport (p, pindex, dmac, dir, bits, rate, fcny, dec, gain, flags|FLG_DISABLE);
    }
    dmac = pic_tbport (p, pindex, dmac, dir, bits, rate, fcny, dec, gain, flags);
    p->tbflags=flags;
  }
  else if (ptype == IOPT_LINK) {
    if (rate*bits > 20000000*8 && (flags&FLG_TUNER)==0 && !p->isY) flags |= FLG_HS;
    dmac = pic_lport (p, pindex, dmac, dir, bits, flags);
  }
  else if (ptype == IOPT_INTERNAL || ptype == IOPT_EXTERNAL) {
    dmac = pic_iport (p, pindex, dmac, dir, bits, flags);
  }
  else if (ptype == IOPT_CORE) {
    dmac = pic_cport (p, pindex, dmac, dir, bits, rate, fcny, dec, gain, flags);
  }
  else if (ptype == IOPT_MCORE) {
    dmac = pic_mcport (p, pindex, dmac, dir, bits, rate, fcny, dec, gain, flags);
  }
  else {
    print ("Unsupported port type identifier: %d \n",ptype); 
    dmac = -1;
  }

  pic_lock (p,LOCK_FREE);
  if (flags&FLG_DISABLE) return dmac;

  /* finish setting the dma channel parameters structure */
  if (dmac>=1 && dmac<=DMA_CHNS) {
    /* fill in the generic parameters */
    n = dmac-1;
    if (p->dma[n].port > 0) pindex = p->dma[n].port; /* port redefinition */
    if (pic_getkeyl(p,dmac,KEY_NODMA)>1) p->dma[n].mcfg &= ~IOM_PORTS;
    p->dma[n].type = ptype;
    p->dma[n].port = pindex;
    p->dma[n].bits = bits;
    p->dma[n].rate = rate;
    p->dma[n].dec  = dec ;
    p->dma[n].gain = gain;
    p->dma[n].fcny = fcny2phinc(fcny);
    while (p->dma[n].slave > 0) {
      n = p->dma[n].slave-1;
      p->dma[n].type = ptype;
      p->dma[n].port = pindex;
      if (ptype==IOPT_MODULE) p->dma[n].master = dmac;
      if (ptype==IOPT_MODULE) p->dma[dmac-1].master = -1;
    }
  }

  if (ptype==IOPT_LINK) return (dmac); /* embedded ioport call */

  p->presamp = 0;
  if (p->isY && ptype!=IOPT_MODULE && p->tcmode!=TCM_OFF) {
    dma = p->dma + (dmac-1);
    side = (dma->mcfg>>28)&0x3;
    if (side==3) side=1; /* mux module mode */
    p->feedrdec=1.0;
    if (p->tflags&FLG_PRESAMP) p->presamp=(FEED_TUNER|pside);
    for (i=0; i<4 && side==0; i++) {	/* resolve special routing */
      feed = dma->feed;
      if (feed==0 && dma->master>0) feed = p->dma[dma->master-1].feed;
      feed = (feed>>16)&0xFF; /* select input port */
      if (feed==PM_HYPA || feed==PM_HYPB) { /* IO module on base card ? */
	side = (feed&1)? 2:1;
        p->presamp |= side;
      } else if (feed<=1) {			/* from HOST */
	side = (pindex&1)? 1:2;
      } else if (feed==0x10 || feed==0x20) {	/* from IPORT=NIO */
	side = (pindex&1)? 1:2;
      } else {
	n = feed2dmac(p,feed);
	if (n<=0) { print("No feed=%x path to IO Module on dmac=%d. Must setup in forward order.\n",feed,dmac); break; }
	dma = p->dma + (n-1);
	if (dma->type==0) {
          pic_stkupd (p, &dma->type, 1, -1); 
          pic_stkupd (p, &dma->feed, 1, -1);
          pic_stkupd (p, &dma->dec,  1, -1);
          pic_stkupd (p, &dma->bits, 1, -1);
          pic_stkupd (p, &dma->port, 1, -1);
          pic_stkupd (p, &dma->mcfg, 1, -1);
	}
        if (dma->dec>0) { p->feedrdec *= dma->dec; }
        if (dma->bits==-bits) { p->presamp |= FEED_RCFLIP; }
        if (dma->type==IOPT_TUNER && dma->port>=1 && dma->port<=2) { 
	  ratio = 1.0;
	  stat = pic_rpb (p,0,pic5tadrtbl[dma->port-1]|(3<<2)); /* GCFPGA_RATIO */
	  if (stat!=0) ratio = ((double)(0x10000000)) / ((double)(stat));
	  stat = pic_rpb (p,0,pic5tadrtbl[dma->port-1]|(4<<2)); /* GCFPGA_RMON */
	  if (stat!=0) ratio = ((double)(stat>>16)) / ((double)(stat&0xFFFF));
	  p->feedrdec /= ratio;
	  p->presamp |= FEED_TUNER;
	}
	side = (dma->mcfg>>28)&0x3; if (side==3) side=1;
      }
    }
  }

  /* setup the global clock register */
  if (findintflag("XSTP",p->config)>0) p->gmcr |= GMC_XSYNCTP;
  if ((p->gmcs>0 || p->gmcr!=0) && (flags&FLG_INFO)==0) {
    if (p->gmcs<0) p->gmcs = (dir>0)? GMC_CLKI:GMC_CLKN;
    vprint("Set GMCR(DG|NC|EN|OE|INV)=%02x GMCS=%c\n",p->gmcr,muxclks[p->gmcs]);
    pic_acmd (p,ADON_WRIOC,IOC_GMC,p->gmcr|max(0,p->gmcs-1),1); 
    if (p->mtype[0]==IOMT_A2D && 
        (strncmp(p->tc_mode,"CPU0P",5)==0 || p->tcmode==TCM_IRB)) {
      mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+1, 0, 0); 
      mcfg2 = mcfg1|IOM_COE|IOM_MCOE;
      if (mcfg2!=mcfg1) pic_acmd (p, ADON_MOD, 1, mcfg2, 0); 
    }
    if (p->gmcs==GMC_CLKA && p->mtype[0]==IOMT_A2D) {
      mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+1, 0, 0); 
      mcfg2 = mcfg1|IOM_MCEN;
      if (mcfg2!=mcfg1) pic_acmd (p, ADON_MOD, 1, mcfg2, 0); 
    }
    if (p->gmcs==GMC_CLKB && p->mtype[1]==IOMT_A2D) {
      mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+2, 0, 0); 
      mcfg2 = mcfg1|IOM_MCEN;
      if (mcfg2!=mcfg1) pic_acmd (p, ADON_MOD, 2, mcfg2, 0); 
    }
    if (p->gmcs==GMC_CLKX && !p->isY && rate>75000000) {
      print("MUXCLK=X is not supported above 75MHz on %s card\n",getCardName(p->type));
      return -1;
    }
    if (p->gmcs!=GMC_CLKN && p->gmcs!=GMC_CLKA && p->gmcs!=GMC_CLKB 
				&& dir<0 && getIOCsig(p,IS_INTERNAL|side)==0) {
      n = p->mtype[side-1];
      if (n<0 && n!=IOMT_A2D && n!=IOMT_LB2D) 
      print("Warn: MUXCLK=%c setting probably not appropriate for this IO module side=%d type=%s\n",
	muxclks[p->gmcs], side, getIomName((int_4)p->mtype[side-1],(int_4)p->mrev[side-1]));
    }
  }

  /* setup the network output core */
  if (p->nio!=0 && ptype!=IOPT_STREAM) {
    if (ptype!=IOPT_MODULE) {
      ratio = (p->res_ratio>0)? p->res_ratio : 1.0;
      if (dec>1) ratio /= dec;
      if (p->mbits<0) ratio /= 2;
      rate = (int_4)(rate*ratio+.5);
    }
    if (p->nioi) pic_setup_qsfp (p,dmac, dir,bits,rate,flags|((dir>0)?FLG_NIO:0));
    else         pic_setup_qsfp (p,dmac,-dir,bits,rate,flags|((dir<0)?FLG_NIO:0));
  }
  
  return (dmac);
}



int_4 pic_mport (PICSTRUCT *p, int_4 port, int_4 dmac, 
		int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{
  int_4 i,status=0,mcfg,mcfg1,mtype,mrev,enclk,enclko,running=0;
  int_4 lport,lportd,iport,stat,brate,maxrate,mport,lbits,obmc,ract,rmin,rmax;
  int_4 lap,laq,lbp,lbq,internal,muxclk,dcm,delay,idelay,clkdly;
  char lfs[4];

  if (port==0) {
    mcfg = IOM_TEN|IOM_SEN;
    if (dmac<=0);
    else if (dir>0) p->dma[dmac-1].mcfg = 0;
    else            p->dma[dmac-1].mcfg |= mcfg;
    return 0;
  }

  /* set the MODULE address in the control word */
       if (p->type == ICEPIC3)  { lap=1; laq=3; lbp=2; lbq=4; }
  else if (p->type == ICEPIC4)  { lap=1; laq=3; lbp=2; lbq=4; }
  else if (p->type == ICEMBT4)  { lap=1; laq=3; lbp=2; lbq=4; }
  else if (p->type == ICEPIC5)  { lap=1; laq=1; lbp=2; lbq=2; }
  else if (p->type == ICEPIC6)  { lap=1; laq=1; lbp=2; lbq=2; }
  else if (p->type == ICEPIC7)  { lap=1; laq=1; lbp=2; lbq=2; }
  else if (p->type == ICEPIC8)  { lap=1; laq=1; lbp=2; lbq=2; }
  else if (p->type == ICEPIC9)  { lap=1; laq=1; lbp=2; lbq=2; }
  else if (p->type == ICEPIC2)  { lap=3; laq=5; lbp=4; lbq=6; }
  else if (p->type == ICEMBT2)  { lap=5; laq=6; lbp=3; lbq=4; }
  else if (p->type == ICESLIC3) { lap=3; laq=1; lbp=4; lbq=2; }
  else { print("Unhandled card type %d in pic_mport\n",p->type); return -1; }

  mport = (port&1)? 1 : 2;
  if (mport == 1) { lport = lap; lportd = laq; } 
  if (mport == 2) { lport = lbp; lportd = lbq; }

  flags |= p->flags;
  if (p->inp>0 && p->inp!=mport) flags |= FLG_ALT;

  iport = mport-1;
  if (flags&FLG_ALT) iport=1-iport;
  if (p->isY && p->inp>0 && (flags&FLG_SGO)) flags ^= (FLG_SGO|FLG_RGO); /* swap SGO/RGO for INP=X override of normal SGO/RGO setups */
  mtype = p->mtype[iport];
  mrev = p->mrev[iport];
  enclk = (iport==0 && p->gmcs==GMC_CLKA) || (iport==1 && p->gmcs==GMC_CLKB);
  gain = findintflagdef("MGAIN",p->config,gain); /* override module gain */
  rate = findintflagdef("MRATE",p->config,rate); /* override module rate */

  p->iomparam[iport].bits = bits;
  p->iomparam[iport].rate = rate;
  p->iomparam[iport].gain = gain;
  p->iomparam[iport].dir  = dir;

  mcfg = mport<<28;
  if (flags&FLG_DISABLE) goto DISABLE;

  /* auto swap of bi-directional modules */
  if (dir*mtype<0 && isBiDir(abs(mtype)) && findintflag("BIDIR",p->config)>0) {
     mtype = p->mtype[iport] = -mtype; 
     if (findstrflag("IOC",p->config,lfs,-1)<0) { ioctype(p,lfs); setIOCsig(p,0,lfs); }
  }

  muxclk   = getIOCsig(p,IS_MUXCLK|mport);	/* use mux clocks */
  internal = getIOCsig(p,IS_INTERNAL|mport);	/* non-module internal mux clocks */
  obmc     = getIOCsig(p,IS_OBMC|mport) && !internal;	/* modules with on-board mux clocks */

  if (p->xchn>0) doXcvrPrep(p,-1);

  if (internal) flags |= FLG_VHS;

  /* if _iir|_t1|_t2 act as if no module */
  if (getIOCsig(p,IS_TEST)) mtype = IOMT_TEST;
  else if (internal) mtype = IOMT_NONE;

  /* handle the mux clock selection */
  if (dir>0 || muxclk) {
    if (p->gmcs>=0);
    else if ((flags&FLG_SGO)!=0);
    else if (dir>0) {
      if ((p->type==ICEPIC2 || p->type==ICEMBT2)) {
        if (mport==2) p->gmcs = GMC_CLKB; else p->gmcs = GMC_CLKA;
        /* if dual D2E, clock is only in A side, auto switch to it */
        if (mport==2 && p->iocc[0]=='O' && p->iocc[1]=='O') p->gmcs = GMC_CLKA;
      } 
      else if (internal) p->gmcs = GMC_CLKI; 
      else p->gmcs = GMC_CLKX;
    } 
    else if (internal) p->gmcs = GMC_CLKI; 
    else if (iport==1) p->gmcs = GMC_CLKB; 
    else p->gmcs = GMC_CLKA;
  }
  /* check for muxclk equal to module clock */
  enclko = (iport==0 && p->gmcs==GMC_CLKA) || (iport==1 && p->gmcs==GMC_CLKB);

  if (dir>0 && mtype<0 && mtype!=IOMT_TEST && p->iocc[iport]=='O') {
    print("Using IOC output with input module for internal use. Disabling buffers.\n");
    mtype = IOMT_NONE;
  }
  else if (dir>0) mcfg |= IOM_PLY;
  if (dir>0) mcfg |= IOM_COE;

  status = pic_acmd (p, ADON_RD, DMA_GMCFG+mport, 0, 0); 
  running = (status&IOM_ENA)!=0 || (flags&FLG_RGO)!=0;  /* dont need to init if RGO */

  if (mtype==IOMT_NONE || mtype==IOMT_TEST);
  else if (dir>0 && mtype<0) { print("Err: Output not supported on %s module\n",getIomName(mtype,mrev)); return (-1); }
  else if (dir<0 && mtype>0) { p->flags = FLG_LOOP;
    print("Note: Looping back output module=%s stream\n",getIomName(mtype,mrev));
    status = pic_lport (p, lport, dmac, dir, 32, flags); 
    return status;
  }
  else if (dir<0 && mtype>0) { print("Err: Input not supported on %s module\n",getIomName(mtype,mrev)); return (-1); }

  stat=0;
  switch (mtype) {

  case IOMT_NONE:
    break;

  case IOMT_TEST:
    if (dir>0) {
      flags |= FLG_INTCLK;
      p->flags |= FLG_INTCLK;
    } 
    else if (dir<0) {
      mcfg |= 0;
    }
    break;

  case IOMT_A2D:
    ract = rate/1000000; 
    switch (mrev) {
    case 1: rmin=1;   rmax=100;
      mcfg |= IOM_MSBI; 	/* this is a SLIC3A */
      break;
    case 5: rmin=1;   rmax=105;
      break;
    case 6: rmin=25;  rmax=210;
      if (bits==8 && enclk!=0) { bits=16; rate=rate/2; } /* bytes are packed into 16 bit words */
      break;
    case 7: rmin=1;  rmax=105;
      break;
    case 8: rmin=2;  rmax=125;
      break;
    case 9: rmin=10; rmax=250;
      if (!running) stat=pic_setup_a2dr9(p,mport,dir,bits,rate,gain,flags); 
      break;
    case 10: rmin=10; rmax=500;
      if (!running) stat=pic_setup_a2dr10(p,mport,dir,bits,rate,gain,flags); 
      break;
    case 11: rmin=100; rmax=1500;
      if (!running) stat=pic_setup_a2dr11(p,2,dir,8,rate,gain,flags); 
      mcfg |= IOM_MUX; /* half of data on B side */
      mcfg |= 0x2<<28; /* copy mcfg to B side */
      if ((p->tcflags&FLG_TC_OPPS) && (rate<200e6)) print("Warn: A2Dr11 has degraded 1-PPS timecode performance below 200MHz sample rate\n");
      break;
    case 13: rmin=40; rmax=250;
      if (!running) stat=pic_setup_a2dr13(p,mport,dir,bits,rate,gain,flags);
      break;
    case 14: rmin=1; rmax=250;
      if (bits>0 && p->socc[iport]=='D' && ract>findintflagdef("A2DTRANS",p->config,125)) mcfg |= IOM_DUAL; /* even real data captured as complex */
      if (!running) stat=a2dm14_setup (p,mport,flags|((mcfg&IOM_DUAL)?FLG_DUAL:0));
      break;
    case 15: case 16: case 17: case 18: rmin=100; rmax=1600;
      if (running) stat=findintflagdef("A2DPORTS",p->config,2);
      else stat=a2dm1x_setup (p,mport,dir,bits,(real_8)rate,gain,flags);
      if (stat==3||stat==5) mcfg |= (IOM_MUX|(0x3<<28)); /* use both port resources as data word mux */
      if ((stat==3||stat==5) && mport==2 && dmac==2) dmac = 1; /* force DMA channel */
      if (stat==3 && bits<0) bits = -bits; /* muxed by R/I on module so fake like real data */
      break;
    case 20: rmin=100; rmax=3000;
      if (running) stat=findintflagdef("A2DPORTS",p->config,2);
      else stat=a2dm20_setup(p,mport,dir,bits,rate,gain,flags);
      if (stat==3 || stat==5) mcfg |= (IOM_MUX|(0x3<<28)); /* use both port resources as data word mux */
      if (stat==5 && mport==2 && dmac==2) dmac = 1; /* force DMA channel */
      if (mport==2 && pic_dmaqueue(p,-1,0)!=1) mcfg |= (0x1<<28); /* must autostart A side */
      break;
    default:  rmin=5; rmax=65;
    }
    if (bits<0) {
      rmin/=2; rmax/=2;
      if (mrev<13) print("Warn: %s does not support complex data output\n",getIomName(mtype,mrev));
    }
    if (ract<rmin || ract>rmax) print("Warn: %s rate=%dMHz out of specified range [%dMHz,%dMHz]\n", getIomName(mtype,mrev),ract,rmin,rmax);
    mcfg |= (IOM_COE|IOM_MCOE);  	/* enable output drivers */
    if ((flags&FLG_MUXCLK)==0 || enclk!=0) mcfg |= IOM_MCEN; /* enable clock circuit */
    break;

  case IOMT_D2A:
    if (mrev < 8) {    
      if ((flags&FLG_MUXCLK)==0 || enclko) {  /* enable clock circuit */
        stat=pic_setup_d2a(p,mport,dir,bits,rate,gain,flags);
      }  
      if (p->gmcs<0) p->gmcs = GMC_CLKX;
    }
    else if (mrev == 9) stat=pic_setup_d2ar9(p,mport,dir,bits,rate,gain,flags);
    break;

  case IOMT_D2AWG:
    if (!running && mrev!=3) stat=pic_setup_d2awg(p,mport,dir,bits,rate,gain,flags);
    if (!running && mrev==3) stat=d2awgm3_setup(p,mport,dir,(bits<0)?-16:16,rate,gain,0);
    break;

  case IOMT_DR2D:
    break;

  case IOMT_E2D:
    break;

  case IOMT_LV2D:
    break;

  case IOMT_D2E:
    break;

  case IOMT_D2LV:
    break;

  case IOMT_T2D:
    /* unfortunately, this module uses MCEN for output enable */
    mcfg |= (IOM_MCOE|IOM_MCEN);
    break;

  case IOMT_D2T:
    break;

  case IOMT_CDR2D: case IOMT_D2CDR:
       
    if (dir>0) {
      print("D2CDR output not supported \n"); return (-1);
    } 
    else if (dir<0) {
      pic_setup_cdr2d (p,mport,dir,rate,gain);
    }
    break;

  case IOMT_CXD: case IOMT_DXC:
    /* enables are both low for normal operation */
    if (dir>0) {
      print("CXD output not supported yet\n"); return (-1);
    } 
    else if (dir<0) {
      pic_setup_cxd (p,mport,dir,rate,gain);
    }
    break;

  case IOMT_GXD:  case IOMT_DXG:
    /* enables are both low for normal operation */
    break;

  case IOMT_UXD:  case IOMT_DXU:
    /* enables are both low for normal operation */
    break;

  case IOMT_DXFPQC: case IOMT_FPQCXD:  /* FPQC */
    stat=pic_setup_fpqc(p,mport,dir,rate,gain);
    break;

  case IOMT_DXFPDP: case IOMT_FPDPXD: /* FPDP */
    bits=16;
    rate=rate * 2; /* User Should Have Specified FPDP Bus Rate, We Multiply To Equal IOC Clk Rate */ 
    stat=pic_setup_fpdp(p,mport,dir,rate,gain);
    break;

  case IOMT_DXUDP: case IOMT_UDPXD: 
    bits=16;
    rate=60000000; 
    if (!running) stat=pic_setup_udp(p,mport,dir,rate,gain);
    break; 

  case IOMT_SDDSXD: 
    if (p->isY && p->gmcs>0) muxclk = 1; /* turn on throttling circuit */
  case IOMT_DXSDDS: 
    if (bits==32) { bits=16; rate*=2; }
    if (p->isX && dir>0 && bits!=16 && findflag("TXRAWDATA",p->config)>=0) { rate/=(16/bits); bits=16; }
    if (!running) stat=pic_setup_sdds(p,mport,dir,rate,abs(bits));
    break;

  case IOMT_DXNF: case IOMT_NFXD: 
    if (!running) stat=pic_setup_ngcfiber(p,mport,dir,rate,gain);
    break; 

  case IOMT_D2ES: case IOMT_ES2D: 
    if (mrev==1) {
      if (!running) stat=pic_setup_esxdr1(p,mport,dir,rate,gain,bits);
    }
    break; 

  case IOMT_DXUFLZ: case IOMT_UFLZXD:
    if (!running) stat=pic_setup_uflz(p,mport,dir,rate,gain);   
    break;

  case IOMT_DXFLZR: case IOMT_FLZRXD:
    if (!running) stat=pic_setup_flzr(p,mport,dir,rate,gain);
    mcfg |= IOM_MUX; /* half of data on B side */
    mcfg |= 0x2<<28; /* copy mcfg to B side */
    break;

  case IOMT_DXTGSDDS: case IOMT_TGSDDSXD:
    i = findintflagdef("TGPORTS",p->config,2);
    if (!running) stat=pic_setup_tgsdds(p,mport,dir,rate,(bits<0)?-2*bits:bits,i);
    if (stat==0) mcfg |= (IOM_MUX|IOM_BMUX|(0x2<<28));
    if (stat==3) mcfg |= (IOM_MUX|(0x2<<28));
    break;

  case IOMT_DXTGVITA: case IOMT_TGVITAXD:
    i = findintflagdef("TGPORTS",p->config,2);
    if (!running) stat=pic_setup_tgvita(p,mport,dir,rate,(bits<0)?-2*bits:bits,i);
    if (stat==0) mcfg |= (IOM_MUX|IOM_BMUX|(0x2<<28));
    if (stat==3) mcfg |= (IOM_MUX|(0x2<<28));
    pic_config_tgvita(p,mport, 0,0, 0,0, 0,0, findintflagdef("VRTSIZE",p->config,1472) );
    break;

  case IOMT_DXRF: case IOMT_RFXD: 
    if (!running) stat=rfxd_setup(p,mport,dir,rate,gain);
    break; 

  case IOMT_DXSNT: case IOMT_SNTXD: 
    if (mrev<=1) {
      if (bits==8) { bits=16; rate=rate/2; }
      if (!running) stat=pic_setup_sonet(p,mport,dir,rate,gain);
    }
    else if (mrev==2) {
      if (bits==8) { bits=16; rate=rate/2; }
      if (!running) stat=pic_setup_sonetr2(p,mport,dir,rate,gain);
    }
    else if (mrev==4) {
      if ((p->type == ICEPIC4 || p->type == ICEMBT4) && (bits==8)) { bits=16; rate=rate/2; }
      if (!running) stat=pic_setup_sonetr4(p,mport,dir,rate,gain);
      if (stat == 2) {            
        mcfg |= IOM_MUX; /* half of data on B side */ /* For Full Rate OC192 */
        mcfg |= 0x2<<28; /* copy mcfg to B side    */
      } 
    }
    else if (mrev==5) {
      if ((p->type == ICEPIC4 || p->type == ICEMBT4) && (bits==8)) { bits=16; rate=rate/2; }
      if (!running) stat=pic_setup_sonetr5(p,mport,dir,rate,gain);
    } 
    else if (mrev==6) {     
      if (!running) stat=pic_setup_sonetr6(p,mport,dir,rate,gain);
      if (stat == 2) {            
        mcfg |= IOM_MUX; /* half of data on B side */ /* For Full Rate OC192 */
        mcfg |= 0x2<<28; /* copy mcfg to B side    */
      }         
    }
    else if (mrev==7) {     
      if (!running) stat=pic_setup_sonetr7(p,mport,dir,rate,gain);
      if (stat == 2 && p->qdrx<1) {            
        mcfg |= IOM_MUX; /* half of data on B side */ /* For Full Rate OC192 */
        mcfg |= 0x2<<28; /* copy mcfg to B side    */
      }         
    }
    break; 

  case IOMT_D2PSE: 
    if (bits==8) { bits=16; rate=rate/2; } 
    if (!running) stat=pic_setup_d2pse(p,mport,dir,rate,gain);
    break; 

  case IOMT_LB2D:
    ract = rate/1000000; rmin=10; rmax=250;
    if (!running && mrev<3) stat=lb2d_setup(p,mport,dir,(bits<0)?-16:16,rate,gain,flags);
    if (!running && mrev==3) stat=lb2dm3_setup(p,mport,dir,(bits<0)?-16:16,rate,gain,0);
    if (ract<rmin || ract>rmax) print("Warn: %s rate=%dMHz out of specified range [%dMHz,%dMHz]\n", getIomName(mtype,mrev),ract,rmin,rmax);
    break;

  case IOMT_MSASXD:
  case IOMT_DXMSAS:
    if (bits<0) bits*=2; rate*=(bits/8); bits=8;
    break;

  case IOMT_DSFPXD:
  case IOMT_DXDSFP:
    break;

  case IOMT_SFPXD:
  case IOMT_DXSFP:
  case IOMT_DIODE:
    break;

  case IOMT_D2RF: 
  case IOMT_RF2D: 
    if (!running) stat=d2rf_setup(p,mport,dir,(bits<0)?-16:16,rate,gain,0);
    break;

  case IOMT_SFPPXD:
/*      pic_setup_sfpp(p,2,dir,rate,gain,flags,bits);  */
    mcfg |= IOM_MUX; /* half of data on B side */
    mcfg |= 0x2<<28; /* copy mcfg to B side */
    break;

  case IOMT_QSFPXD:
  case IOMT_DXQSFP:
    pic_setup_qsfp (p,mport,dir,bits,rate,flags);
    break;

  case IOMT_UPACXD:
  case IOMT_DXUPAC:
    pic_setup_upac (p,mport,dir,bits,rate,flags);
    break;

  default:
    print("Err: Unsupported MODULE type: %d\n",p->mtype[iport]); 
    stat = -1;
  }

  /* bail on bad setup status */
  if (stat<0) return(-1);

  /* complex data on 5+ modules via rising=I and falling=Q clock edges */
  if (p->isY && bits<0 && (flags&FLG_MUX)==0) {
    mcfg |= IOM_DUAL;
    bits = -bits;
  }

  /* set the FPGA bit operations registers */
  if      (bits==16) mcfg |= IOM_16B; 
  else if (bits==32) mcfg |= IOM_16B;
  else if (bits==12 && (p->qdrx&mport)!=0) mcfg |= IOM_16B;
  else if (bits==8) mcfg |= IOM_08B;
  else if (bits==4 && p->isY) mcfg |= IOM_04B;
  else if (bits==1) mcfg |= IOM_01B;
  else { print("Illegal number of bits=%d for MODULE=%d with QDRX=%d\n",bits,port,p->qdrx); return (-1); }
  if ((p->qdrx&mport)!=0 && p->up12) mcfg |= IOM_UP12;

  /* which bit (default=15) (FLG_BIT1 is a 2bit combo flag) */
  if ((flags&FLG_BIT1)==FLG_BIT1) mcfg |= IOM_B01;
  else if (flags&FLG_BIT0) mcfg |= IOM_B00;
  else if (flags&FLG_BIT4) mcfg |= IOM_B04;
  else if (flags&FLG_BIT3) mcfg |= IOM_B03;

#ifdef _XMIDAS
  if (bits==1) flags |= FLG_BIGEND; /* default X-Midas SP is big endian */
#endif

  /* set up the clock */
  if (p->gmcs<=0);
  else if (dir>=0 && findflag("OPORT",p->config)>0 && strstr(p->config,"OPORT=MOD")==NULL);
  else if (dir<=0 && findflag("IPORT",p->config)>0 && strstr(p->config,"OPORT=MOD")==NULL);
  else pic_timer (p, mport, rate); 

  /* check for proper MUX IOC code */
  if (dir<=0 && !muxclk) {
    if ((flags&FLG_MUX)!=0 && !p->isY) 
      print("Warn: Complex mux with channel clock skew >5ns requires use of MUXCLK flag in pic_reset()\n"); 
    if (p->gmcs>0) print("Warn: Use of MUXCLK|INTCLK requires MUXCLK flag in pic_reset()\n");
  }

  /* auto invert input clock to compensate for FPGA I/O delays */
  delay=idelay=clkdly=dcm=0;
  if (dir>0) {		/* clock inversions for output - if any */
  }      		
  else if (p->isY) {	/* no clock inversions for 5+ series */
    if (p->socc[iport]=='S' || p->socc[iport]=='D') {
      dcm = findintflag("DCM",p->config);
      if ((mcfg&IOM_DUAL)==0) {     /* normal single edge clock */
        if (p->type==ICEPIC8 && rate>100000000) mcfg ^= IOM_CLKI;
        if (p->type==ICEPIC7 && rate>100000000) mcfg ^= IOM_CLKI;
        if (p->type==ICEPIC6 && rate>100000000) mcfg ^= IOM_CLKI;
        if (p->type==ICEPIC5 && rate>100000000) { mcfg ^= IOM_CLKI; clkdly=1; }
      } else if (rate> 75000000 && dcm!=0) {  /* fast DDR - use DCM */
        dcm = 1;
      } else {	                    /* slow DDR - use CLKDLY to adjust phase by 90 */
        clkdly = (p->type==ICEPIC5)? 7:5;
      }
    }
  }
  else if (p->isX) {
    if (mtype==IOMT_A2D) { 	/* clk out/in 12.5ns delay - at (20,60,100) */
      if (rate>25000000) mcfg ^= IOM_CLKI; 
      if (rate>75000000) mcfg ^= IOM_CLKI; 
    } else if (mtype==IOMT_UDPXD || mtype==IOMT_SDDSXD || (mtype==IOMT_SNTXD && mrev>0)) { 	
      mcfg ^= IOM_CLKI;
      if ((flags&FLG_VHS)==0) flags |= FLG_DUAL;
    } else if (muxclk) {
      if (rate>30000000) { mcfg ^= IOM_CLKI; delay = (1e9/rate/2); }
    } else {		
      if (rate>40000000) { mcfg ^= IOM_CLKI; delay = (1e9/rate/2); }
    }
    /* SPEC data=1.2 dA=5.6 dB=6.2 mdA=6.0 mdB=13.0 ns */
    /* REAL data=1 dA=3 dB=6 mdA=3 mdB=7 ns */
    if (!muxclk) idelay=3;
    else if (p->gmcs==GMC_CLKA) idelay=3;
    else if (p->gmcs==GMC_CLKB) idelay=7;
  }
  else {
    if (mtype==IOMT_A2D) { 	/* clk out/in 12.5ns delay - at (20,60,100) */
      if (rate>25000000) mcfg ^= IOM_CLKI; 
    } else if (muxclk) { 	/* muxclk in dd=7ns delay  */
      if (rate>30000000) mcfg ^= IOM_CLKI; 
    }
    idelay=7;
  }

  /* force clock on rising edge */
  if (findintflag("CLKRE",p->config) > 0) { mcfg |= IOM_CLKI; delay = 0; }

  /* implement clock delays */
  clkdly = findintflagdef("CLKDLY",p->config,clkdly);
  if (p->isY);
  else if (p->isX || muxclk) {
    if (clkdly>0) delay += clkdly;
    delay -= idelay;
         if (delay <= 2) clkdly=DLY_TAP0;
    else if (delay <= 3) clkdly=DLY_TAP1;
    else if (delay <= 5) clkdly=DLY_TAP2;
    else                 clkdly=DLY_TAP3;
    if (flags&FLG_CLKI) clkdly |= DLY_INV; /* invert input clock ? */
         if (muxclk) pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_M|clkdly,1);
    else if (mport==1) pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_A|clkdly,1);
    else if (mport==2) pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_B|clkdly,1);
  }

  /* invert MSB ? */
  if (flags&FLG_MSBI) mcfg ^= IOM_MSBI;

  /* insert LSB ? */
  if (flags&FLG_LSBX) mcfg |= IOM_LSBX;

  /* big endian bit ordering ? */
  if ((flags&FLG_BIGEND)!=0 && bits==1) mcfg |= (p->isY? IOM_FLIP:IOM_01BB);

  /* wait for slave GO ? */
  if (flags&FLG_SGO) mcfg |= IOM_SGO;

  /* wait for trigger GO ? */
  if (flags&FLG_TGO) mcfg |= IOM_TGO;

  /* wait for gated GO ? */
  if (flags&FLG_GGO) mcfg |= IOM_GGO;

  /* external gated GO ? */
  if (flags&FLG_XGO) mcfg |= IOM_XGO;

  /* complex channel mux ? */
  if (flags&FLG_MUX) mcfg |= IOM_MUX;

  /* very high speed mode ? */
  if (flags&FLG_VHS && (flags&FLG_TUNER)==0 && !p->isY) mcfg |= IOM_VHS;

  /* alternate IO port ? */
  if (flags&FLG_ALT) mcfg |= IOM_ALT;

  /* non-continuous output clock - JEFF kludge for now */
  if (dir>0 && (flags&FLG_NCCLK)) mcfg |= IOM_UOPT;

  /* unique option flag */
  if (findintflag("UOPT",p->config) > 0) mcfg |= IOM_UOPT;

  /* block mux option flag */
  if (findintflag("BMUX",p->config) > 0) mcfg |= IOM_BMUX;

  /* loopback case ? */
  if (dir>0 && dmac==0) mcfg |= IOM_ALT;

  /* set the FPGA bit operations registers */
  if ((flags&FLG_RGO)==0) mcfg |= IOM_ENA;

  /* muxclk mod on 5 series ? */
  if (p->isY) {
    if (flags&FLG_LOOP) mcfg |= IOM_VOPT;
    if (muxclk && p->gmcs!=0) mcfg |= IOM_CLKM;
    if (muxclk && dir>0 && enclko) mcfg &= ~IOM_CLKM;
    if (muxclk && obmc) mcfg &= ~IOM_CLKM;
    if (muxclk && mtype==IOMT_A2D && mrev==8) mcfg &= ~IOM_CLKM; /* fed through muxclk */
    if (p->iocc[iport+2]=='S' && rate<=120000000 && (mcfg&IOM_GGO)==0 && !p->k7) mcfg |= IOM_CLKM; /* SDDS smoother */
    if (findintflag("NOCLKM",p->config)>0) mcfg &= ~IOM_CLKM;	/* disable smoother */
    if (findintflag("RXNOSEQFILL",p->config)>0) mcfg |= IOM_VOPT; /* disable 16 packet filler */
    if ((flags&FLG_SGO) && (flags&FLG_TGO)) mcfg &= ~IOM_SGO;
    if (p->socc[iport]=='H') {
      if (mcfg&IOM_CLKI) { print("Clock inversion not supported with packetized modules\n");  mcfg &= ~IOM_CLKI; }
      if ((mcfg&IOM_TGO)==0);
      else if (mcfg&IOM_XGO) print("XTGO function is coarse alignment only on IO module type=%s\n",getIomName(mtype,mrev));
      else                   print("TGO function is not supported on IO module type=%s\n",getIomName(mtype,mrev));
    } 
  }

  /* link port, tuner output, and tuner sync enables */
  if ((flags&FLG_TUNER)!=0) {
    if (port<=8) mcfg |= IOM_TEN; /* dedicated tuner ports */
    if (!p->isY) mcfg |= IOM_SEN;
    if (p->gchip==GC4016) mcfg |= (IOM_TEN|IOM_SEN);
    if (p->gchip==GCFPGA && p->pmi>0 && p->tcmode!=TCM_OFF) mcfg |= (IOM_TEN|IOM_SEN);
  }
  else mcfg |= IOM_LEN;

  /* initialize the IOC dynamic nodes */
  if (p->isY && !running) {
    if (set_iom_sequence (p,mport,mcfg,bits,rate,flags)<0) return -1;
    if (p->socc[iport]=='S' || p->socc[iport]=='D') {
      if (mtype==IOMT_A2D && mrev==14) mcfg |= IOM_CLKI;
      i = IOC_TAPS|((port&1)?IOC_A:IOC_B);
      clkdly = (clkdly<=0)? 0 : (0x1<<(clkdly-1));
      if (dcm>0) clkdly = -1;  /* enable DCM */
      pic_acmd (p,ADON_WRIOC, i, clkdly, 0); 
      i = IOC_PROG|((port&1)?IOC_A:IOC_B);
      pic_acmd (p,ADON_WRIOC,i,IOP_RCLK,0); 
      clkdly = (rate>125000000)? 60 : 0;
      clkdly = findintflagdef("CLKSKEW",p->config,(dcm>0)? clkdly:0);
      for (;clkdly>0; clkdly--) pic_acmd (p,ADON_WRIOC,i,IOP_ACLKU,0); 
      for (;clkdly<-1; clkdly++) pic_acmd (p,ADON_WRIOC,i,IOP_ACLKD,0); 
      delay = ((mcfg&(IOM_GGO|IOM_TGO))!=0 && (mcfg&IOM_XGO)==0)? 1:0;
      i = findintflag("PRETRIG",p->config); if (i>0) delay += i;
      i = (mtype==IOMT_A2D && mrev<9)? delay+4 : delay;  /* adjust for pipe delay of A2D samples */
      delay = (i<<4)|delay;
    } else { /* for 'H' and 'R' */
      delay = findintflagdef("PREDELAY",p->config,0); 
    }
    i = IOC_DELAY|((port&1)?IOC_A:IOC_B);
    pic_acmd (p,ADON_WRIOC, i, delay, 0); 
  }

  DISABLE:

  /* set non-module resource bits */
  if ((flags&FLG_TUNER)!=0) { 
         if (p->type == ICEPIC2) mcfg |= IOM_PORT_S1;
    else if (p->type == ICEPIC3) mcfg |= IOM_PORT_L3;
    goto FINISH; /* bail now if not processing module data through link ports */
  }

  /* need to use 2x link output clock for higher rates */
  brate = (rate/1000000)*bits/8;
  if (p->up12) brate = (brate*12)/16;
  if (p->isY && (mcfg&IOM_DUAL)) brate *= 2;
  if (p->isY) maxrate = 500;
  else if (p->isX) maxrate = 70;
  else maxrate = 38;
  if (p->isY);
  else if (brate > maxrate/2) flags |= FLG_HS;

  /* force 80MHz link clock except in test modes and IOC nibble modes */
  /* current nibble modes are E321 E1X16 , need better scheme */
  if (p->isY);
  else if (p->mtype[iport]==IOMT_TEST || p->iocc[0]=='E');
  else if (dir<0) flags |= FLG_HS; 

  /* set up link port for follow on DMA */
  if (mcfg&IOM_VHS) lbits=48; else lbits=32;
  status = pic_lport (p, lport, dmac, dir, lbits, flags); 
  if (status<0) return (-1);
  p->dma[status-1].slave = 0;  /* no slave yet */
  mcfg |= p->dma[status-1].mcfg; /* get resource bits */

  if (p->isY) {
         if (p->socc[iport]=='S') maxrate = 320;
    else if (p->socc[iport]=='D') maxrate = 600;
    else if (p->socc[iport]=='R') maxrate = 1200;
    else if (p->type==ICEPIC8 && p->socc[2]!=' ') maxrate = (flags&FLG_VHS)? 1000 : 660;
    else if (p->type==ICEPIC8) maxrate = (flags&FLG_VHS)? 800 : 600;	/* only the default hh gen load has this limit */
    else if (p->type==ICEPIC7) maxrate = (flags&FLG_VHS)? 800 : 520;
    else if (p->type==ICEPIC6) maxrate = (flags&FLG_VHS)? 800 : 520;
    else if (p->type==ICEPIC5) maxrate = (flags&FLG_VHS)? 520 : 260;
    else                       maxrate = 520;
    if ((mcfg&IOM_MUX)!=0 && (flags&FLG_MUX)==0) maxrate *= 2;
    if (p->qdrx&port) maxrate *= 2;
  }
  else {
    if ((brate>maxrate) || (flags&FLG_DUAL)!=0 || (flags&FLG_VHS)!=0) {
      vprint("Applying dual link port option ...\n");
      if ((flags&FLG_DUAL)!=0 || (mcfg&IOM_VHS)==0) mcfg |= IOM_DUAL;
      stat = pic_lport (p, lportd, -1, dir, lbits, flags); 
      if (stat<0) return (-1);
      p->dma[stat-1].slave = 0;        /* no slave yet */
      p->dma[status-1].slave = stat; /* slave upper to lower */
      mcfg |= p->dma[stat-1].mcfg; /* get resource bits */
      p->dma[stat-1].mcfg = 0;   /* leave it to the master */
      maxrate *= 2;
    }
    if ((mcfg&IOM_VHS)!=0 && (mcfg&IOM_DUAL)!=0) maxrate *= 2;
    if (flags&FLG_VHS) maxrate = (int)(0.75*maxrate);
  }

  if (dir!=0 && dmac!=0 && brate>maxrate) { 
    print ("WARNING: Xfer rate of %d Mby/s per channel is > spec of %d Mby/s. ",brate,maxrate);
    if ((flags&FLG_VHS)==0) print ("See VHS flag.\n"); else print ("\n");
  }

  mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+mport, 0, 0); 

  if (flags&FLG_DISABLE) {		/* disable */
    if ((mcfg1&0xC)==0) pic_acmd (p, ADON_MOD, mport, 0, 0); 
  }
  else if (dmac == 0) {		/* no dma */
    if (mcfg&IOM_ENA) pic_enable_module(p,mport,1);
    /* download the iomod configuration word now */
    if ((mcfg1&0xC)==0) pic_acmd (p, ADON_MOD, mport, mcfg, 0); 
    pic_acmd (p, ADON_WR, DMA_GMCFG+mport, mcfg, 0); 
  } 
  else {
    /* reset the modules to insure proper slave sync-up */
    if ((mcfg1&0xC)==0) pic_acmd (p, ADON_MOD, mport, 0, 0); 
  }
  dmac = status;
  vprint ("mcfg = %08x MTYPE=%d\n", mcfg, p->mtype[iport]);

  FINISH:
  /* store the iomod_cfg in the dma structure for dmafunc */
  if (dmac>0) {
    p->dma[dmac-1].mcfg = mcfg;
    p->dma[dmac-1].flags = flags;
  }

  return (status);
}

/*
  IOM Sequencer for 5+ series

  There are too many modes to code in FPGA so we put mechanisms in FPGA
  to route data and develop a sequence pattern here to control them.

  In SDDS, the first 8 sequence values are for 16b data, then 4 for 8b, and 4 for 4b.

  {mixsel[8][2], mixena[8], ulnmsk[2],swpsel,outsel, vival,woena,wival,voena}

  outer = wodatL,wodatH,muxCB,muxCI (muxx,outsel)
  muxer = vodat,outer,waves,ymixi (muxsel)
  muxed = muxer,~muxer (swpsel) (nibble swap)
  mixin = muxed,-muxed (bflip) (spectral flip)
  mixer = mixin0,mixin1,mixin2,mixin3 (mixsel,mixena) (per 8by input, 4by output)

*/

int_4 set_iom_sequence (PICSTRUCT *p, int_4 port, int_4 mcfg, int_4 bits, int_4 rate, int_4 flags) {
  int_4 i,n,ibits,ebits,gbits,alt,mux,tmp,test,loop,sdds,swap,flip,data[16],re,iport,minc,b16,b8,a14,calib,nodma; 
  char ptype,pmode,mtype,mrev;

  /* port config and test patterns */
  ptype = (port==2)? p->socc[1] : p->socc[0];
  pmode = (port==2 && p->iocc[3]!=' ')? p->iocc[3] : p->iocc[2];
  test  = (pmode=='R') || (pmode=='W');
  sdds  = (pmode=='S') && (mcfg&IOM_GGO)==0;
  loop  = (p->iocc[0] == 'T');
  mtype = (port==2)? p->mtype[1] : p->mtype[0];
  mrev  = (port==2)? p->mrev[1]  : p->mrev[0];
  nodma = findintflagdef("NODMA",p->config,0); 		/* read word */

  /* are we using the alternate input modes ? */
  alt   = (mcfg&IOM_ALT)!=0 || p->iocc[0]=='T';
  mux   = (mcfg&IOM_MUX)!=0 && (mcfg&IOM_BMUX)==0;
  if (sdds && mux && (mcfg&IOM_UOPT)!=0) sdds=0;
  calib = (flags==FLG_TEST);
  b16   = (!calib && mtype==IOMT_LB2D) || 
	  (!calib && mtype==IOMT_D2RF) || 
	  (!calib && mtype==IOMT_D2AWG && mrev>=3) ||
	  (!calib && mtype==IOMT_A2D && mrev==14 && ptype=='H') || sdds;
  a14   = (!calib && mtype==IOMT_A2D && mrev==14 && ptype=='D' && (mcfg&IOM_DUAL)!=0);
  b8    = (!calib && mtype==IOMT_A2D && mrev==11);

  /* internal bits per word */
  tmp = (mcfg&IOM_BITS);
  ibits = (tmp==IOM_16B)? 16 : (tmp==IOM_08B)? 8 : (tmp==IOM_04B)? 4 : 1;
  if (bits!=0) ibits = abs(bits);
  if (mcfg&IOM_DUAL) ibits = -ibits;

  /* should we clock with the rising edge */
  re = (mcfg&IOM_CLKI)?1:0;
  if (test) re = (pmode=='R')? 1:0;
  else if (ptype=='H' || ptype=='R') re = 1;	/* always RE ordered */
  else if (ibits<0) re = 1-re;			/* DDR data uses rising edge clocks */

  /* external bits per word */
  ebits = -16;					/* default to CI data ports */
  if (test || mtype==0);
  else if (b16) ebits = (ibits<0)? -16:16;
  else if (b8)  ebits = (ibits<0)? -8:8;
  else if (ptype=='H' || ptype=='R') ebits = ibits; /* already packed - RE ordered */
  if (test || loop) ebits = -16;		/* test vector is CI {lrs,ramp} */

  if (mcfg&IOM_PLY) {
    ebits = findintflagdef("LBEBITS",p->config,ebits);
    ibits = findintflagdef("LBIBITS",p->config,ibits);
  }
  i = findintflag("LBCF",p->config);
  if (i>1) ibits=ebits=(ibits*i);

  /* samples per internal 64b word */
  minc = 64/ibits;
  if (minc<0) minc = -minc/2;
  if (mux) minc = minc/2;
  minc = max(1,minc);

  /* enable data byte swapping */
  swap = !calib && (findintflag("SWAP",p->config)>0 || (flags&FLG_BIGEND)!=0);

  /* spectral flip */
  flip = !calib && (findintflag("SPINV",p->config)>0 || findintflag("SPFLIP",p->config)>0);
  if (flip==0);
  else if (ibits<0 && !a14) { flip = 0x00;  re = 1-re; }
  else if (abs(ebits)==16) flip = 0x0C;
  else if (abs(ebits)==8)  flip = 0x0A;
  else                     flip = 0x80;

  /* if ebits != -16 then data is packed into 32b words */
  n = 0;
  gbits = 0;
  if (mcfg&IOM_PLY) {			/* output modes */
    if (test) {
      if (ibits==-16) for (;n<2;n++) data[n] = 0x00E40F08;
      if (ibits==16)  for (;n<4;n++) data[n] = (re?0x00440F08:0x00EE0F08);
      if (ibits==-32) for (;n<1;n++) data[n] = 0x00E40F08;
      if (ibits==32)  for (;n<2;n++) data[n] = (re?0x00440F08:0x00EE0F08);
    } else if (p->qdrx&port) {		/* QDRX output modes */
      for (;n<1;n++) data[n] = 0x0F08 | ((0xE4+n*0)<<16);
    } else if (mux) {
      if (ibits==16) gbits |= 0x10;
      if (ibits==ebits)            for (;n<1;n++) data[n] = 0x0F08 | ((0xE4+n*0)<<16);
      if (ibits==32 && ebits==-16) for (;n<1;n++) data[n] = 0x0F08 | ((0xEE +n*0)<<16);
      if (ibits==16 && ebits==-16) for (;n<2;n++) data[n] = 0x0F08 | ((0x44+n*0xAA)<<16);
      if (ibits==8  && ebits==-16) for (;n<4;n++) data[n] = 0x0A08 | ((0x00+n*0x44)<<16);
    } else if (swap) {
      if (ibits==4  && ebits==-16) for (;n<16;n++)data[n] = 0x0A48 | ((((n>>1)%4)*0x44)<<16) | ((n&8)?0x10:0) | ((n&1)?0x20:0);
      if (ibits==1  && ebits==-16) for (;n<16;n++)data[n] = 0x0A48 | ((((n>>1)%4)*0x44)<<16) | ((n&8)?0x10:0) | ((n&1)?0x20:0);
      if (ibits==4  && ebits==4)   for (;n<2;n++) data[n] = 0x0F28 | ((0xE4+(n%1)*0x00)<<16) | ((n&1)?0x10:0);
      else if (ibits==ebits)       for (;n<2;n++) data[n] = 0x0F08 | ((0xB1+(n%1)*0x00)<<16) | ((n&1)?0x10:0);
    } else {
      if (ibits==ebits)            for (;n<2;n++) data[n] = 0x0F08 | ((0xE4+(n%1)*0x00)<<16) | ((n&1)?0x10:0);
      if (ibits==32 && ebits==-16) for (;n<2;n++) data[n] = 0x0F08 | ((0xEE +(n%1)*0xAA)<<16) | ((n&1)?0x10:0);
      if (ibits==16 && ebits==-16) for (;n<4;n++) data[n] = 0x0F08 | ((0x44+(n%2)*0xAA)<<16) | ((n&2)?0x10:0);
      if (ibits==8  && ebits==-16) for (;n<8;n++) data[n] = 0x0A08 | ((0x00+(n%4)*0x44)<<16) | ((n&4)?0x10:0);
      if (ibits==4  && ebits==-16) for (;n<16;n++)data[n] = 0x0A48 | ((((n>>1)%4)*0x44)<<16) | ((n&8)?0x10:0) | ((n&1)?0:0x20);
      if (ibits==1  && ebits==-16) for (;n<16;n++)data[n] = 0x0A48 | ((((n>>1)%4)*0x44)<<16) | ((n&8)?0x10:0) | ((n&1)?0:0x20);
      if (ibits==-8 && ebits==-16) for (;n<4;n++) data[n] = 0x0A08 | ((0x40+(n%2)*0x88)<<16) | ((n&2)?0x10:0);
      if (ibits==8  && ebits== 16) for (;n<4;n++) data[n] = 0x0A08 | ((0x40+(n%2)*0x88)<<16) | ((n&2)?0x10:0);
      if (ibits==-16 && ebits==-8) for (;n<2;n++) data[n] = 0xDDDD0004 | (0x0300<<(n*2))     | ((n&1)?0x10:0) | ((n&1)?0x08:0);
    }
    if (n>0 && nodma!=2) data[0] |= 0x4; 		/* read word */
  }
  else if (p->qdrx&port) {			/* QDRX input modes */
    for (;n<2;n++) data[n] = (re?0xE4E40001:0x4E4E0001) | (0x0F00<<(n*4));
  }
  else if (sdds) {			/* SDDS input modes */
    if (mux) {
    } else if ((mcfg&IOM_UOPT)!=0) { 	/* raw packets with optional swap */
      if (swap) for (;n< 8;) { data[n++]=0xB1B10F01; data[n++]=0xB1B1F003; } /* SI */
                for (;n<12;) { data[n++]=0xE4E40F01; data[n++]=0xE4E4F003; } /* SB or SI noswap */
      if (swap) for (;n<16;) { data[n++]=0xE4E40F21; data[n++]=0xE4E4F023; } /* SN swap */
                for (;n<16;) { data[n++]=0xE4E40F01; data[n++]=0xE4E4F003; } /* SN noswap */
    } else if (p->tcmode==TCM_SDN || p->tcmode==TCM_DTL) { 
      if (ibits==16) {
       for (;n< 8;) { data[n++]=0x11110300; data[n++]=0xBBBB0C01; data[n++]=0x11113000; data[n++]=0xBBBBC003; }
       for (;n<16;) data[n++]=0;
      }
      if (ibits==8) {
       for (;n< 8;) { data[n++]=0x00000100; data[n++]=0xAAAA0201; data[n++]=0x00000400; data[n++]=0xAAAA0801; 
                      data[n++]=0x00001000; data[n++]=0xAAAA2001; data[n++]=0x00004000; data[n++]=0xAAAA8003; }
       for (;n<16;) data[n++]=0;
      }
    } else {
      if (abs(ibits)==16) {
        if (swap) for (;n< 8;) { data[n++]=0xE4E40F01; data[n++]=0xE4E4F003; }
        for (;n< 8;) { data[n++]=re?0xB1B10F01:0x1B1B0F01; data[n++]=re?0xB1B1F003:0x1B1BF003; }
        for (;n<12;) { data[n++]=re?0xC840AA03:0x8C04AA03; }
        for (;n<16;) { data[n++]=0x44002240; data[n++]=0x44008862; data[n++]=0xCC882240; data[n++]=0xCC888863; }
      }
      if (abs(ibits)==8) {
        for (;n< 8;) { data[n++]=0x88880301; data[n++]=0x88880C01; data[n++]=0x88883001; data[n++]=0x8888C003; }
        for (;n<12;) { data[n++]=re?0xE4E40F01:0xB1B10F01; data[n++]=re?0xE4E4F003:0xB1B1F003; }
        for (;n<16;) { data[n++]=0xFA505540; data[n++]=0xFA50AA63; }
      }
      if (abs(ibits)==4) {
        for (;n< 8;) { data[n++]=0x55550101; data[n++]=0x55550201; data[n++]=0x55550401; data[n++]=0x55550801; 
                       data[n++]=0x55551001; data[n++]=0x55552001; data[n++]=0x55554001; data[n++]=0x55558003; }
        for (;n<12;) { data[n++]=0xCCCC0301; data[n++]=0xCCCC0C01; data[n++]=0xCCCC3001; data[n++]=0xCCCCC003; }
        if (swap) for (;n<16;) { data[n++]=0xE4E40F01; data[n++]=0xE4E4F003; }
        for (;n<16;) { data[n++]=0xE4E40F21; data[n++]=0xE4E4F023; }
      }
    }
  }
  else if (swap) {			/* swapped byte input modes */
    if (mux) {
    } else {
      if (ibits==16  && ebits==-16) for (;n<4;n++) data[n] = (re?0x11110001:0xBBBB0001) | (0x0300<<(n*2));
      if (ibits==-16 && ebits==-16) for (;n<2;n++) data[n] = (re?0xB1B10001:0x1B1B0001) | (0x0F00<<(n*4));
      if (ibits==16  && ebits==16)  for (;n<2;n++) data[n] = (              0xB1B10001) | (0x0F00<<(n*4));
      if (ibits==4   && ebits==-16) for (;n<16;n++)data[n] = (re?0x55550001:0xFFFF0001) | (0x0100<<(n/2)) | ((n%2==0)?0x40:0xA0);
      if (ibits==1   && ebits==-16) for (;n<16;n++)data[n] = (re?0xFFFF0001:0xFFFF0001) | (0x0100<<(n/2)) | ((n%2==0)?0x40:0xA0);
    }
    if (n>0) data[n-1] |= 0x2; 		/* write word */
  }
  else {				/* input modes */
    if (mux) {
      if (ibits==32 && ebits==-16) for (;n<1;n++) data[n] = (re?0xEE440001:0x44EE0001) | (0xCC00<<(n*8));
      if (ibits==16 && ebits==-16) for (;n<2;n++) data[n] = (re?0x44440001:0xEEEE0001) | (0x0F00<<(n*4));
      if (ibits==8  && ebits==-16) for (;n<4;n++) data[n] = (re?0x55550001:0xFFFF0001) | (0x0300<<(n*2));
      if (ibits==16 && ebits==16)  for (;n<1;n++) data[n] = (              0xEE440001) | (0xFF00<<(n*8));
      if (ibits==8  && ebits==16)  for (;n<2;n++) data[n] = (              0xF4F40001) | (0x0F00<<(n*4));
      if (ibits==8  && ebits==8)   for (;n<1;n++) data[n] = (              0xFA500001) | (0xFF00<<(n*8));
      if (ibits==16 && ebits==8)   for (;n<2;n++) data[n] = (n&1)?0xCC88AA03:0x4400AA02;
    } else {
      if (ibits==ebits           ) for (;n<2;n++) data[n] = (re?0xE4E40001:0x4E4E0001) | (0x0F00<<(n*4));
      if (ibits==16 && ebits==-16) for (;n<4;n++) data[n] = (re?0x44440001:0xEEEE0001) | (0x0300<<(n*2));
      if (ibits==8  && ebits==-16) for (;n<8;n++) data[n] = (re?0x55550001:0xFFFF0001) | (0x0100<<(n*1));
      if (ibits==-8 && ebits==-16) for (;n<4;n++) data[n] = (re?0xDDDD0001:0x77770001) | (0x0300<<(n*2));
      if (ibits==8  && ebits==16)  for (;n<4;n++) data[n] = (              0xDDDD0001) | (0x0300<<(n*2));
      if (ibits==16 && ebits==8)   for (;n<1;n++) data[n] = (              0xC8400001) | (0xAA00<<(n*8));
      if (ibits==4  && ebits==-16) for (;n<16;n++)data[n] = (re?0x55550001:0xFFFF0001) | (0x0100<<(n/2)) | ((n%2==1)?0x40:0xA0);
      if (ibits==1  && ebits==-16) for (;n<16;n++)data[n] = (re?0xFFFF0001:0xFFFF0001) | (0x0100<<(n/2)) | ((n%2==1)?0x40:0xA0);
      if (ibits==32 && ebits==-16) for (;n<2;n++) data[n] = (re?0x44440001:0xEEEE0001) | (0x0C00<<(n*4));
      if (ibits==-32 && ebits==-16) for (;n<1;n++) data[n] = (re?0xEE440001:0x44EE0001) | (0xCC00<<(n*4));
    }
    if (n>0) data[n-1] |= 0x2; 		/* write word */
  }
  if (n==0) {
    print("Case ibits=%d ebits=%d not supported yet with IOC mcfg=%08x\n",ibits,ebits,mcfg);
    return -1;
  }
  for (i=0; i<16; i++) {
    data[i] |= gbits;
    if (i>=n) data[i] = data[i%n];	/* reproduce sequence */
  }
      
  tmp = 0;
  if ((mcfg&IOM_MUX)!=0 && (flags&FLG_MUX)==0) port=3;
  if (port&0x1) tmp |= IOC_A;
  if (port&0x2) tmp |= IOC_B;
  if (port==3 && flip>0) {
    pic_acmd (p,ADON_WRIOC,IOC_A|IOC_FLIP,0x0,1);
    pic_acmd (p,ADON_WRIOC,IOC_B|IOC_FLIP,0xF,1);
  } else {
    pic_acmd (p,ADON_WRIOC,tmp|IOC_FLIP,flip,1);
  }
  i = findintflagdef("PKTBLK",p->config,0); /* output clumper */
  n = min(255,rate/390625); /* (256/100MHz) for SDDS smoother */
  pic_acmd (p,ADON_WRIOC,tmp|IOC_MINC,(n<<16)|(i<<8)|minc,1);
  for (i=15; i>=0; i--) {
    pic_acmd (p,ADON_WRIOC,tmp|IOC_SEQ,data[i],4);
    v3print("Port=%d minc=%d Seq#%x=%08x\n", port, minc, i, data[i]);
  }
  i = findintflagdef("EMT",p->config,0x4400); /* envelope Meas and Track */
  pic_acmd (p,ADON_WRIOC,tmp|IOC_EMT,i,2);
  return 0;
}

int_4 set_tcgen_sequence (PICSTRUCT *p, int_4 port, int_4 count, int_4 flags) {
  int_4 i, n=19, data[20], tmp=IOC_TCGEN;

  data[0] = 0x030F;	/* barker */
  data[1] = 0x0304;
  data[2] = 0x1396;	/* sec */
  data[3] = 0x1255;
  data[4] = 0x1394;	/* min */
  data[5] = 0x1253;
  data[6] = 0x1392;	/* hrs */
  data[7] = 0x1121;
  data[8] = 0x1393;	/* day */
  data[9] = 0x1392;
  data[10] = 0x5131;
  data[11] = 0x2397;	/* msec */
  data[12] = 0x2398;
  data[13] = 0x2399;
  data[14] = 0x0300;	/* vft */
  data[15] = 0x0300;
  data[16] = 0x8308;
  data[17] = 0x0;
  data[18] = count;

  if (flags==1) data[0] &= ~0x0001;	/* no Fill Bit */

  if (port&0x1) tmp |= PPC_HYPA_CTL;
  if (port&0x2) tmp |= PPC_HYPB_CTL;
  for (i=0; i<n; i++) {
    pic_acmd (p,ADON_WR,tmp>>2,data[i],4);
    v2print("TcGenSeq#%d=%08x \n", i, data[i]);
  }
  return 0;
}

int_4 pic_lport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir, int_4 bits, int_4 flags)
{
  int_4 n,lb,side,maxport,mask,value,status,nodma,dmac_def,dmacx,rev,lctl,lcom,chained,mcfg,reg,lx,feed,alt,mux;

  flags |= p->flags;
  if (dmac==0) nodma=1; else nodma=0;
  side = (p->inp>0)? p->inp : (dmac&1)? 1 : 2;

  maxport = p->isY? 12 : 6;
  if (port<1 || port>maxport) { 
    print("Err: LPORT number=%d out of range [1-%d]\n",port,maxport); 
    return (-1); 
  }

  /* select default DMA channel */
  if (p->isX) {
    lctl = 0xcc; lcom = 0xce; 
    dmac_def = lport2dmacx[port];	
    chained = (findflag("MEM=EXT",p->config) >= 0); 
  } else if (p->isY) {
    lctl = 0x0; lcom = 0x0; 
    dmac_def = lport2dmacy[port];	
    chained = 0;
  } else {
    lctl = 0xc6; lcom = 0xc7; 
    dmac_def = lport2dmac[port];	
    chained = (dmac==7 || dmac==8); 
  }

  if (dmac <= 0) dmac = dmac_def;
#ifdef JEFF
  if (dmac != dmac_def && !p->isX) { 
    print("Err: LPORT DMA channel %d must be %d\n",dmac,dmac_def); 
    return (-1); 
  }
#endif
  lb = port-1;

  /* set DMA enable parameters for DMA_FUNC */
  n = dmac-1;
  value = 0x3;
  if (chained) value |= 0x4;
  alt = 0;
  p->dma[n].dir = dir;
  if (nodma) {
    p->dma[n].reg = 0;
    p->dma[n].enb = 0;
  } 
  else if (p->isY) {
    value = 0;
    reg = 0;
    feed = p->dma[n].feed;
    dmacx = (dmac>4 && dmac<9 && (feed&0xF)!=0)? dmac+4 : dmac; /* Cores on PM */
    if (dir>0) alt = (feed==0)? 0 : ((dmac-1)^(feed>>24))&0x1; /* side!=oport */
    else       alt = (feed==0)? 0 : ((dmac-1)^(feed>>16))&0x1; /* side!=iport */
    if (dir>0) switch (dmacx) {
      case  1: value = PPC_ROUTE_R2HA;                 reg = PPC_ISTAT(PM_HYPA); break;
      case  2: value = PPC_ROUTE_R2HB;                 reg = PPC_ISTAT(PM_HYPB); break;
      case  3: value = PPC_ROUTE_R2CA|PPC_ROUTE_CA2HA; reg = PPC_OSTAT(PM_CORA); break;
      case  4: value = PPC_ROUTE_R2CB|PPC_ROUTE_CB2HB; reg = PPC_OSTAT(PM_CORB); break;
      case  5: value = PPC_ROUTE_R2TA|PPC_ROUTE_TA2HA; reg = PPC_OSTAT(PM_TUNA); break;
      case  6: value = PPC_ROUTE_R2TB|PPC_ROUTE_TB2HB; reg = PPC_OSTAT(PM_TUNB); break;
      case  7: value = PPC_ROUTE_R2TA|PPC_ROUTE_TA2HA; reg = PPC_OSTAT(PM_TUNA); break;
      case  8: value = PPC_ROUTE_R2TB|PPC_ROUTE_TB2HB; reg = PPC_OSTAT(PM_TUNB); break;
      case  9: value = PPC_ROUTE_R2CA|PPC_ROUTE_CA2TA; reg = PPC_OSTAT(PM_CORA); break;
      case 10: value = PPC_ROUTE_R2CB|PPC_ROUTE_CB2TA; reg = PPC_OSTAT(PM_CORB); break;
      case 11: value = PPC_ROUTE_R2CA|PPC_ROUTE_CA2TB; reg = PPC_OSTAT(PM_CORA); break;
      case 12: value = PPC_ROUTE_R2CB|PPC_ROUTE_CB2TB; reg = PPC_OSTAT(PM_CORB); break;
    } 
    else switch (dmac) {
      case  1: value = PPC_ROUTE_HA2R;                 reg = PPC_OSTAT(PM_HYPA); break;
      case  2: value = PPC_ROUTE_HB2R;                 reg = PPC_OSTAT(PM_HYPB); break;
      case  3: value = PPC_ROUTE_CA2R|(alt?PPC_ROUTE_HB2CA:PPC_ROUTE_HA2CA); reg = PPC_OSTAT(PM_CORA); break;
      case  4: value = PPC_ROUTE_CB2R|(alt?PPC_ROUTE_HA2CB:PPC_ROUTE_HB2CB); reg = PPC_OSTAT(PM_CORB); break;
      case  5: case  9: value = PPC_ROUTE_TA2R|(alt?PPC_ROUTE_HB2TA:PPC_ROUTE_HA2TA); reg = PPC_OSTAT(PM_TUNA); break;
      case  6: case 10: value = PPC_ROUTE_TA2R|(alt?PPC_ROUTE_HA2TA:PPC_ROUTE_HB2TA); reg = PPC_OSTAT(PM_TUNA); break;
      case  7: case 11: value = PPC_ROUTE_TB2R|(alt?PPC_ROUTE_HB2TB:PPC_ROUTE_HA2TB); reg = PPC_OSTAT(PM_TUNB); break;
      case  8: case 12: value = PPC_ROUTE_TB2R|(alt?PPC_ROUTE_HA2TB:PPC_ROUTE_HB2TB); reg = PPC_OSTAT(PM_TUNB); break;
    }
    if ((p->tflags&FLG_PRESAMP)==0);
    else if (reg==PPC_OSTAT(PM_TUNA)) value = (side==2)? PPC_ROUTE_TA2R|PPC_ROUTE_HB2CB|PPC_ROUTE_CB2TA : PPC_ROUTE_TA2R|PPC_ROUTE_HA2CA|PPC_ROUTE_CA2TA;
    else if (reg==PPC_OSTAT(PM_TUNB)) value = (side==2)? PPC_ROUTE_TB2R|PPC_ROUTE_HB2CB|PPC_ROUTE_CB2TB : PPC_ROUTE_TB2R|PPC_ROUTE_HA2CA|PPC_ROUTE_CA2TB;
    else { print("Flag PRESAMP not supported for this configuration\n"); return -1; }
    p->dma[n].reg = reg;
    p->dma[n].enb = value;
    /* check for cross module options and modify routes */
    if (dmac>=3 && dmac<=12 && p->dma[n].feed!=0) activatePM (p,p->dma+n,0);
  } 
  else if (p->isX) {
    if (lb>2) { lctl++; lb-=3; }
    p->dma[n].reg = lctl;
    p->dma[n].enb = value<<(lb*10);
  } 
  else {
    p->dma[n].reg = lctl;
    p->dma[n].enb = value<<(lb*4);
  }
  p->dma[n].enbx = p->dma[n].enb;
  p->dma[n].flags = flags;
  mcfg = 0;
  if (p->isY) {
    lx = (dmac-1)>>1;
    if (lx==0) mcfg = IOM_PORT_L1;
    if (lx==1) mcfg = IOM_PORT_L2;
  } else {
    lx = (port-1)>>1;
    if (lx==0) mcfg = IOM_PORT_L1;
    if (lx==1) mcfg = IOM_PORT_L2;
    if (lx==2) mcfg = IOM_PORT_L3;
    if ((port&1)!=alt) mcfg |= 0x1<<28; else mcfg |= 0x2<<28;
  }
  p->dma[n].mcfg = mcfg;

  if ((flags&FLG_INFO)!=0) return (dmac); /* setup only ? */
  if (p->isY) return (dmac);
  /* the rest is for pre PIC5 cards only */

  /* disable link port and its DMA */
  if (p->isX) mask = (0x3FF)<<(lb*10); else mask = (0xF)<<(lb*4);
  status = pic_acmd (p, ADON_WRM, lctl, 0, mask);  

  /* disable port & exit ? */
  if ((flags&FLG_DISABLE)!=0) return (dmac);

  if (!p->isX && !p->isY) {
    /* can only write at 1x clock unless ADSP1.2x rev */
    if ( (flags&FLG_HS) != 0) {
      status = pic_acmd (p, ADON_WRM, SREG_MODE2, 0, 0);
      rev = (status>>28)&0x3; if (rev<2) {
        print("ADSP rev %x does not support 2x link clock output\n",rev);
        flags ^= FLG_HS; 
      }
    }
    /* set the LCOM register for 2-D DMA at 2x clock (if DMA) */
    /* NOTE: 2-D DMA is bit 18 of the LCOM reg - set during boot */
    mask = (1<<(12+lb));
    if (flags&FLG_HS) value = mask; else value = 0;
    status = pic_acmd (p, ADON_WRM, lcom, value, mask); 
    vprint ("LCOM = %08x %08x  LB = %d\n",value, status, lb);
  }

  /* OK - setup link port */
  if (dir>0) value=0x8;			/* output */ 
  else       value=0x0;			/* input */
  if (nodma) value = value | 0x1;	/* enable now if no DMA */
  if (p->isX) {
    value |= 0x220;  /* 8-bit, clock = core */
    if (!chained) value |= 0x080; /* 2D DMA */
    if (bits==48) value |= 0x10; 
    value = value<<(lb*10); 
    mask = (0x3FF)<<(lb*10); 
  } else {
    value = (value&0xF)<<(lb*4);
    if (bits==48) value |= (1<<(24+lb)); 
    mask = ((0xF)<<(lb*4))|(1<<(24+lb));
  }
  status = pic_acmd (p, ADON_WRM, lctl, value, mask);  
  vprint ("LCTL port=%d v=%08x m=%08x s=%08x\n", port, value, mask, status);

  return (dmac);
}


int_4 pic_iport (PICSTRUCT *p, int_4 port, int_4 feed, int_4 dir, int_4 bits, int_4 flags)
{
  int_4 n,dmac;
  dmac = 32+port;
  n = dmac-1;
  p->dma[n].feed = feed;
  p->dma[n].reg  = 0;
  p->dma[n].enb  = 0;
  p->dma[n].enbx = 0;
  p->dma[n].mcfg = 0;
  p->dma[n].flags = flags;
  return dmac;
}

void lsub (char *ccfg, char *name, int_4 value) {
  char *s, ctmp[512]; int ln=strlen(name);
  s = strstr(ccfg,name);
  if (s==NULL) return;
  strcpy(ctmp,s+ln);
  sprintf(s,"%d",value);
  strcat(s,ctmp);
}
  
#define UPSTAPS 16
#define UPSTSHF (2<<8)
int_4 loadUPStaps (PICSTRUCT *p, int_4 pmod, int_4 addr, int_4 ups) {
  int_4 i,j=addr; float ftap[UPSTAPS*4]; int_4 itap[UPSTAPS*2]; 
  firkais_generate (FIRKAIS_LOW,0.9/2,0.0,0.05/2,90.0,ftap,UPSTAPS*2,0);
  for (i=0; i<UPSTAPS*2; i++) itap[i] = round(UPSTSHF*ftap[i+i]*0x8000); /* complex to real + shifts */
  for (i=0; i<UPSTAPS*2; i++) pic_wpb (p,pmod,j|MCORE_TAPS,itap[i]);
  return 0;
}

int_4 pic_cport (PICSTRUCT *p, int_4 port, int_4 dmac, 
	int_4 dir, int_4 bits, int_4 rate, double fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i,j,status,side,lport,core,mcores,pmod,sysreg,sysflg,frame=1,mport,mbits,pmtype;
  int_4 aflags,mcfg,feed,dport,iport,oport,route,csig,crate,lc=0,ln=0,mci,ups,mcfb;
  char ccfg[256],cname[16],fname[128];

  side = (port&0x1)? 1 : 2;
  pmod = port/10;
  core = port%10;
  if (pmod==0) pmod = p->pmi;
  if (pmod==3) pmod = 1;
  port = pmod*10 + core;
  pmtype = (pmod>0 && pmod<=p->mpm)? p->pmtype[pmod-1] : 0;
  mcores = 2;
  if (pmod==0 || pmtype>3) mcores = 4;

  vprint("Core Port=%d (PM%dCORE%d) bits=%d dec=%d gain=%d dir=%d\n",port,pmod,core,bits,dec,gain,dir);

  if (p->isY && core>=1 && core<=mcores) {
         if (pmod==0) dmac = 2 +              core;
    else if (core<=2) dmac = 4 + (pmod-1)*2 + core;
    else              dmac = 8 + (pmod-1)*2 + side;
    lport = dmac;
  }
  else if (p->isX && pmod==1 && core>=1 && core<=2) {
    lport = p->linktbl[side];
    dmac = 2+side;
  }
  else { print("Core Port=%d (PM%dCORE%d) not supported on this card\n",port,pmod,core); return -1; }

  j = core_addr(core);
  csig  = pic_rpb(p,pmod,j|MCORE_SYS|COREG_PARAM0); 
  crate = pic_rpb(p,pmod,j|MCORE_SYS|COREG_PARAM7)*1000000; 
  ups   = findintflagdef("UPS",p->config,(dir>0 && rate>crate)?2:0); /* upsampler mode */
  mcfb  = findintflagdef("MCFB",p->config,0); /* multicore feedback */
  mci   = findintflagdef("MCI",p->config,-1); /* multicore index */
  if (p->mcs>0) {			/* enable MultiCoreStreaming */
    if (csig!=CORE_MCOR) print("Multi-Core-Streaming enabled on non-MCS port sig\n");
    i = 0;
    if (mcfb==0) i |= 0x0008;		/* bulk 1kBy xfer */
    if (findintflagdef("ICSEL",p->config,0)>0) i |= 0x00400100;	else	/* channel select tags in + swap */
    if (findintflagdef("ICTAG",p->config,(dir>0)?1:0)>0) i |= 0x0100;	/* channel data tags in */
    if (findintflagdef("OCTAG",p->config,(dir>0)?0:1)>0) i |= 0x0200;	/* channel data tags out */
    if (dir<0) i |= 0x2000;		/* add throttle flag if input - only applies to OUTADDR mode */
    if (ups>0) i |= 0x4000;		/* UPS output mode */
    if (mcfb>0) i |= 0x8000;		/* multicore feedback */
    pic_wpb (p,pmod,j|MCORE_SYS,i);
    pic_wpb (p,pmod,j|MCORE_ALL,0x0);	/* initially disable all ports */
    i |= (p->mcs<<16);			/* add num active channels */
    pic_wpb (p,pmod,j|MCORE_SYS,i);
    pic_wpb (p,pmod,j|MCORE_RATE,rate);
    i = findintflagdef("MCGAIN",p->config,gain);
    set_awg_gain(p,port,0,i);		/* multicore combiner gain */
    p->dec5 = findintflagdef("NFGAIN",p->config,-100);
    set_awg_gain(p,port,99,p->dec5);	/* multicore noise gain */
  }
  else if (csig==CORE_MCOR) {
    i = 0x00010000;
    pic_wpb (p,pmod,j|MCORE_SYS,i);
    i = findintflagdef("MCGAIN",p->config,0);
    set_awg_gain(p,port,0,i);		/* multicore combiner gain */
    p->dec5 = findintflagdef("NFGAIN",p->config,-100);
    set_awg_gain(p,port,99,p->dec5);	/* multicore noise gain */
    p->mcs=-2;
  }
  if (ups>0) loadUPStaps(p,pmod,j,ups);

  aflags = FLG_TUNER;
  flags |= p->flags;
/*JGS  if (dir<0 && getIOCsig(p,IS_OUTPUT|side) && findflag("IPORT",p->config)<0) { aflags |= FLG_RGO; dir=0; } /* output module */

  /* parse input/output ports */
  dport = (core==4)? PM_TUNB : (core==3)? PM_TUNA : (side==2)? PM_CORB : PM_CORA;
  feed  = setupFeeds (p,pmod,dport,dir);
  iport = (feed>>16)&0xF;
  oport = (feed>>24)&0xF;
  p->dma[dmac-1].feed = feed;
  p->dma[dmac-1].port = port;

  /* parse module port */
  mbits = (dir>0)? bits : p->mbits;
  mport = 0;
  if (dir<0 && (iport==PM_HYPA || iport==PM_HYPB)) mport = (iport==PM_HYPB)? 2 : 1;
  if (dir>0 && (oport==PM_HYPA || oport==PM_HYPB)) mport = (oport==PM_HYPB)? 2 : 1;

  /* set up the link port DIR=0 BITS=32 */
  dmac = pic_lport (p, lport, dmac, dir, 32, flags); 
  if (dmac<0) return (-1);
  p->dma[dmac-1].port = port;

  /* set up the Module configuration */
  status = pic_mport (p, mport, dmac, dir, mbits, rate, 0, flags|aflags);
  if (status<0) return (-1);

  /* initialize CORE system register - borrows system register bitmap from IO modules */
  sysreg = findintflagdef("CORESYS",p->config,CORE_DEC);
  sysflg = findintflagdef("COREFLG",p->config,sysflg);
  lc = findstrflag("CORE",p->config,ccfg,0);
  if (lc>0) {
    ln = findstrflag("NAME",ccfg,cname,0);
    /* assumes noopengine if no NAME specified */
    sysflg = findintflagdef("FLAGS",ccfg,sysflg);
    frame = findintflagdef("FRAME",ccfg,frame);
    dec = findintflagdef("DEC",ccfg,dec);
  }

  if (dir>0) {
    sysreg |= CORE_PLY;
    sysreg |= getFmt(bits)<<8;
    sysreg |= getFmt(mbits)<<12;
    if (lc<0) {
      /* check for rate|bits tradeoff if using port for data conduit */
      if (dec<=1 && bits==mbits && bits>0 && rate>100e6) { rate/=(32/bits); bits=mbits=-16; }
      if (mport==0) sysflg |= 0x1; 	/* default noop output throttling if oport not IO module */
      if (mport>0) rate += max(0x200000,rate/10); /* add 1MHz cushion if actually throttled by IOM */
    }
  } else {
    sysreg |= getFmt(mbits)<<8;
    sysreg |= getFmt(bits)<<12;
  }
  if (csig==CORE_MCOR && (flags&FLG_LOOP)) {
    pic_wpb (p,pmod,j|MCORE_SYS,0x40);
  }

  /* disable port & exit ? */
  if (flags&FLG_DISABLE) return (dmac);

  /* set generic core parameters */
  pic_setkeyl (p,dmac,KEY_CORE+COREG_DEC,max(0,dec-1));
  pic_setkeyl (p,dmac,KEY_CORE+COREG_GAIN,gain);
  pic_setkeyl (p,dmac,KEY_CORE+COREG_RATE,rate);
  pic_setkeyl (p,dmac,KEY_CORE+COREG_FREQ,R2L(fcny));
  pic_setkeyl (p,dmac,KEY_CORE+COREG_FRAME,frame);
  pic_setkeyl (p,dmac,KEY_CORE+COREG_FLAG,sysflg);

  /* set specific core parameters */
  csig  = pic_rpb(p,pmod,j|COREG_PARAM0); 

  /* special case for flags directed core */
  if (lc>0 && ln>0) {
    p->dmaci = dmac;
    /* pick up special run-time parameter args */
    lsub (ccfg,"C_RATE",rate);
    lsub (ccfg,"C_BITS",bits);
    lsub (ccfg,"C_DEC" ,dec);
    lsub (ccfg,"C_GAIN",gain);
    core_setup(p,port,ccfg,csig);
  }

  if (findintflag("NFFT",p->config)>0) {
    print("Warning: New flag syntax for FFT core is FFT=(NFFT=x,NAVG=x,NMAX=x,FPPF=x,WIND=x,FLAGS=x|y|z)\n");
  }

  /* special case for FFT core */
  if (findstrflag("FFT",p->config,ccfg,1)>0) { strcpy(p->cname,"FFT");
    setup_fftcore (p,port,ccfg,csig,dmac,bits,rate,dec,gain);
  }

  /* special case for filter core */
  if (findstrflag("FFIR",p->config,fname,-1)>0) { strcpy(p->cname,"FFIR");
    p->deco = dec;
    pic_loadfile (p, fname, FLG_FC|port); 
    pic_setkeyl(p,-port,KEY_GAIN,gain);
  }

  /* special case for hilbert core */
  if (findstrflag("HFIR",p->config,fname,-1)>0) { strcpy(p->cname,"HFIR");
    sysreg |= 0x0800; /* add complex input */
    pic_loadfile (p, fname, FLG_FC|port); 
    pic_setkeyl(p,-port,KEY_GAIN,gain);
  }

  /* don't enable individual MCS channel if */
  if (p->mcs>0 && mci==0); else	
  /* enable CORE system register */
  pic_setkeyl (p,dmac,KEY_CORE,sysreg);

  return (dmac);
}

int_4 pic_mcport (PICSTRUCT *p, int_4 port, int_4 dmac, 
	int_4 dir, int_4 bits, int_4 rate, double fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i,n,lc=0,ln=0,csig,status,side,lport,core,pmod,sysreg,mport,mbits,pmtype,aflags,mcfg,feed,dport,iport,oport,route,sdmac;
  char ccfg[256],cname[8],fname[40]; double ratio; int_4 chip,chan;

  side = (port&0x1)? 1 : 2;
  pmod = (port-1)/32+1;
  core = (port-1)%32+1;
  if (pmod==1) pmod = p->pmi;
  if (pmod==3) pmod = 1;
  port = (pmod-1)*32 + core;
  pmtype = (pmod>0 && pmod<=p->mpm)? p->pmtype[pmod-1] : 0;
  if (pmod==0 && p->type==ICEPIC8) { pmtype = PMT_K8M; port = core; }
  if (pmtype==PMT_DTDM || pmtype==PMT_DTDMX) pmtype=0; /* multicore not supported on DTDM modules */

  if (p->isY && pmtype>0 && core>=1 && core<=32) {
    if (pmod==0) {
      sdmac = 8 + (side-1)*2 + 1;
      lport = sdmac;
      dmac = 12 + core;
    } else {
      sdmac = 8 + (pmod-1)*2 + side;
      lport = sdmac;
      dmac = 12 + (pmod-1)*32 + core;
    }
  }
  else { 
    print("MultiCore Port=%d (PM%dMCORE%d) not supported on this card\n",port,pmod,core); 
    return -1;
  }

  vprint("MultiCore Port=%d (PM%dMCORE%d) bits=%d dec=%d gain=%d dir=%d\n",port,pmod,core,bits,dec,gain,dir);

  aflags = FLG_TUNER;

  /* parse input/output ports */
  dport = (side==2)? PM_TUNB : PM_TUNA;
  feed  = setupFeeds (p,pmod,dport,dir);
  iport = (feed>>16)&0xF;
  oport = (feed>>24)&0xF;
  p->dma[dmac-1].feed = feed;
  p->dma[dmac-1].port = port;

  /* parse module port */
  mbits = p->mbits;
  mport = 0;
  if (dir<0 && (iport==PM_HYPA || iport==PM_HYPB)) mport = (iport==PM_HYPB)? 2 : 1;
  if (dir>0 && (oport==PM_HYPA || oport==PM_HYPB)) mport = (oport==PM_HYPB)? 2 : 1;

  /* handle output combiner cores */
if (dir>0 && p->isDUC) {
  if (p->mcs==0) p->mcs=-1;
  chan  = ((port-1)>>1)&0xF;
  chip  = (port&1)? 1 : 2;
  dmac = p->ndmac + port;
  sdmac = 4+chip;
  pic_lport (p, chip, sdmac, dir, bits, flags); 
  p->dma[dmac-1].reg = p->dma[sdmac-1].reg;
  p->dma[dmac-1].enb = p->dma[sdmac-1].enb;
} else {
  /* set up the link port */
  sdmac = pic_ioport (p, IOPT_LINK, lport, sdmac, -1, bits, rate, fcny, dec, gain, flags|aflags);
  if (sdmac <= 0) return (-1); 
  p->dma[sdmac-1].type = IOPT_MCORE;
  p->dma[sdmac-1].master = -1;
  p->dma[sdmac-1].feed = feed;
  p->dma[dmac-1].master = sdmac;
  p->dma[dmac-1].reg = p->dma[sdmac-1].reg;
  pic_dma (p, sdmac, -1, 0, 0, 1024*1024, 0, aflags);

  /* set up the Module configuration */
  status = pic_mport (p, mport, dmac, dir, mbits, rate, 0, flags|aflags);
  if (status<0) return (-1);
}

  /* initialize CORE system register - borrows system register bitmap from IO modules */
  sysreg = CORE_DEC;
  if (dir>0) {
    sysreg |= CORE_PLY;
    sysreg |= getFmt(bits)<<8;
    sysreg |= getFmt(mbits)<<12;
  } else {
    sysreg |= getFmt(mbits)<<8;
    sysreg |= getFmt(bits)<<12;
  }
  sysreg |= ((core-1)>>1)<<16;
  pic_setkeyl (p,dmac,KEY_MCORE+COREG_SYS,sysreg);

  /* disable port & exit ? */
  if (flags&FLG_DISABLE) return (dmac);

  lc = findstrflag("CORE",p->config,ccfg,0);
  if (lc>0) ln = findstrflag("NAME",ccfg,cname,0);

  /* special case for flags directed core */
  if (lc>0 && ln>0) {
    csig = pic_getkeyl (p,dmac,KEY_MCORE+COREG_PARAM0);
    p->dmaci = dmac;
    /* pick up special run-time parameter args */
    lsub (ccfg,"C_RATE",rate);
    lsub (ccfg,"C_BITS",bits);
    lsub (ccfg,"C_DEC" ,dec);
    lsub (ccfg,"C_GAIN",gain);
    core_setup(p,port,ccfg,csig);
  } else {
    /* standard registers */
    pic_setkeyl (p,dmac,KEY_MCORE+COREG_DEC,max(0,dec-1));
    pic_setkeyl (p,dmac,KEY_MCORE+COREG_GAIN,gain);
    pic_setkeyl (p,dmac,KEY_MCORE+COREG_RATE,rate);
    pic_setkeyl (p,dmac,KEY_MCORE+COREG_FREQ,R2L(fcny));
    pic_setkeyl (p,dmac,KEY_MCORE+COREG_FLAG,flags);
  }

  /* enable CORE system register */
  pic_setkeyl (p,dmac,KEY_MCORE+COREG_SYS,sysreg);

  return (dmac);
}

int_4 parseioport (PICSTRUCT *p, char *key, int_4 dport)
{
  char value[20]; int_4 i=0;
  int_4 len  = findstrflag(key,p->config,value,1);
  int_4 side = (dport&0x1)?1:0;
  int_4 mod  = (dport>>4);
  if (len<=0) return dport; /* default port */
  if (strcmp(value,"NONE")==0) return 0;
  if (strcmp(value,"HOST")==0) return dport;
  if (strncmp(value,"PM",2)==0) { mod = value[2]-'0'; i=3; }
       if (strncmp(value+i,"NIO",3)==0) dport=PM_PROC;
  else if (strncmp(value+i,"CORE",4)==0) dport=PM_CORA;
  else if (strncmp(value+i,"TUNER",5)==0) dport=(mod==0)?PM_CORA:PM_TUNA;
  else if (strncmp(value+i,"MCORE",5)==0) dport=PM_TUNA;
  else if (strncmp(value+i,"TBANK",5)==0) dport=PM_TUNA;
  else if (strncmp(value+i,"MODULE",6)==0) dport=PM_HYPA;
  else if (strncmp(value+i,"THRUCORE",8)==0) dport=PM_CORHYPA;
  else print("Illegal I/O port type flag=%s value=%s\n",key,value);
       if (value[len-1]=='1');
  else if (value[len-1]=='2') dport++;
  else if (value[len-1]=='3' && dport==PM_CORA) dport=PM_TUNA;
  else if (value[len-1]=='4' && dport==PM_CORA) dport=PM_TUNB;
  else if (value[len-1]=='X') dport += side;
  else if (dport==PM_PROC);
  else print("Illegal I/O port index flag=%s value=%s\n",key,value);
       if (value[len-2]=='1') dport |= 0x10;
  else if (value[len-2]=='2') dport |= 0x20;
  else dport |= (mod<<4);
  return dport;
}

int_4 setupFeeds (PICSTRUCT *p, int_4 pmod, int_4 dport, int_4 dir) {
  int_4 aport = (p->isY && p->inp>0 && findflag("IPORT",p->config)<0)? p->inp-1 : dport;
  int_4 iport = (pmod==0 && dir>0)? PM_HOST : ((aport&0x1)? PM_HYPB:PM_HYPA)|(pmod<<4);
  int_4 oport = (pmod==0 && dir<=0)? PM_HOST : ((dport&0x1)? PM_HYPB:PM_HYPA)|(pmod<<4);
  iport = parseioport(p,"IPORT",iport);
  oport = parseioport(p,"OPORT",oport);
  return (oport<<24) | (iport<<16) | (dport<<8) | pmod;
}

int_4 activatePM (PICSTRUCT *p, DMASTRUCT *dma, int_4 mode) {
  int_4 oport = (dma->feed>>24)&0xF;
  int_4 omod  = (dma->feed>>28)&0xF;
  int_4 iport = (dma->feed>>16)&0xF;
  int_4 imod  = (dma->feed>>20)&0xF;
  int_4 cport = (dma->feed>> 8)&0xF;
  int_4 pmod  = (dma->feed>> 0)&0xF;
  int_4 dport = findintflag("DELAYPORT",p->config);
  int_4 route = 0;
  int_4 routf = 0;
  int_4 routx = 0;
  int_4 rcmd  = (mode>0)? PPC_DMAC_ROUTE_SET:PPC_DMAC_ROUTE_CLR;
  int_4 dlyline = findintflagdef("DELAYLINE",p->config,0);
  int_4 prer2c = findintflagdef("PRER2C",p->config,0);
  int_4 tmp[4],addr,mask,csig;
  if (dma->dir<0 && !(iport==PM_HYPA||iport==PM_HYPB)) dma->enb &= ~PPC_ROUTE_HX2X;
  if (dma->dir>0 && !(oport==PM_HYPA||oport==PM_HYPB)) dma->enb &= ~PPC_ROUTE_X2HX;
  if (mode==0) vprint("Activate=%d dir=%d iport=%d:%d cport=%d:%d oport=%d:%d enb=%08x\n",
	mode,dma->dir,imod,iport,pmod,cport,omod,oport,dma->enb);
  if (pmod==0 && cport==PM_CORA && iport==PM_CORB) {
    routf = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0);
    pic_acmd (p, ADON_WR, DMA_ROUTF, routf|PPC_ROUTF_CB2CA, 0);
  }
  if (pmod==0 && cport==PM_CORB && iport==PM_CORA) {
    routf = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0);
    pic_acmd (p, ADON_WR, DMA_ROUTF, routf|PPC_ROUTF_CA2CB, 0);
  }
  if (pmod==0 && dma->dir>0 && iport==PM_HOST) return dma->enb;
  if (dport<=0) dport = dma->port;
  if (imod!=0 && imod!=pmod) {
    /* add cross module route */
    routf = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0); 
    if (pmod==2 && imod==1) { routf |= PPC_ROUTF_TA2TB; dma->enb &= ~PPC_ROUTE_X2TB; }
    if (pmod==1 && imod==2) { routf |= PPC_ROUTF_TB2TA; dma->enb &= ~PPC_ROUTE_X2TA; }
    pic_stkupd (p, &dma->enb, 1, 1);
    pic_acmd (p, ADON_WR, DMA_ROUTF, routf, 0); 
    /* connect source port to crossbar */
    routx = (iport==PM_TUNB)? PPC_ROUTE_TB2HB : (iport==PM_TUNA)? PPC_ROUTE_TA2HA : 
	    (iport==PM_CORB)? PPC_ROUTE_CB2HB : (iport==PM_CORA)? PPC_ROUTE_CA2HA : 0;
    pic_wpb (p,imod,rcmd,routx);
    /* connect dest port to crossbar */
    iport = (iport&1)? PM_HYPB:PM_HYPA;
  }
  if (pmod==0 && dma->dir<0 && oport==PM_HOST) return dma->enb;
  if (dma->dir>0 && pmod>0 && (oport==PM_HYPA||oport==PM_HYPB)) {
    /* add direct PM to IOM route */
    routf = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0); 
    mask = (pmod==2)? PPC_ROUTF_TB2HX : PPC_ROUTF_TA2HX;
    if (mode>=0) routf |= mask; /* else routf &= ~mask; */
    pic_acmd (p, ADON_WR, DMA_ROUTF, routf, 0); 
    /* startup input cores */
    if (findflag("IPORT",p->config)<0 || findflag("IPORT=HOST",p->config)>0) {
      addr = (oport==PM_HYPB)? PPC_CORB_CTL:PPC_CORA_CTL;
      csig = pic_rpb (p,0,addr|MCORE_SYS|COREG_PARAM0);
      if (csig==CORE_MCOR) { 
	pic_wpb (p,0,addr|MCORE_SYS,0x40);
      } else {
	pic_wpb (p,0,addr+COREG_FLAG,0);
	pic_wpb (p,0,addr+COREG_SYS,(mode>0)?0x8803:0x8800);
      }
    }
  }
  if (pmod>0 && findintflag("CTAG",p->config)>0) {
    routf = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0); 
    pic_acmd (p, ADON_WR, DMA_ROUTF, routf|PPC_ROUTF_TXTAG, 0); 
  }
  if (imod==0 && pmod>0 && (iport==PM_CORA||iport==PM_CORB||iport==PM_TUNA||iport==PM_TUNB)) {
    /* connect source port to crossbar */
    if (pmod==1) routx = (iport==PM_CORB)? PPC_ROUTE_CB2TA : (iport==PM_CORA)? PPC_ROUTE_CA2TA : 0;
    if (pmod==2) routx = (iport==PM_CORB)? PPC_ROUTE_CB2TB : (iport==PM_CORA)? PPC_ROUTE_CA2TB : 0;
    dma->enb &= ~PPC_ROUTE_HX2TX; dma->enb |= routx;
    /* connect dest port to crossbar */
    iport = (iport&1)? PM_HYPB:PM_HYPA;
  }
  if (mode==0 && pmod>0) return route; /* early out for setup */
  if (cport==PM_CORA && oport==PM_HYPA) route |= PPC_ROUTE_CA2HA;
  if (cport==PM_CORB && oport==PM_HYPA) route |= PPC_ROUTE_CB2HA;
  if (cport==PM_CORA && oport==PM_HYPB) route |= PPC_ROUTE_CA2HB;
  if (cport==PM_CORB && oport==PM_HYPB) route |= PPC_ROUTE_CB2HB;
  if (cport==PM_CORA && oport==PM_HOST) route |= PPC_ROUTE_CA2R;
  if (cport==PM_CORB && oport==PM_HOST) route |= PPC_ROUTE_CB2R;
  if (cport==PM_CORA && iport==PM_TUNA) route |= PPC_ROUTE_TA2CA;
  if (cport==PM_CORB && iport==PM_TUNB) route |= PPC_ROUTE_TB2CB;
  if (iport==PM_HYPA && cport==PM_CORA) route |= (dlyline?0:PPC_ROUTE_HA2CA);
  if (iport==PM_HYPB && cport==PM_CORA) route |= (dlyline?0:PPC_ROUTE_HB2CA);
  if (iport==PM_HYPA && cport==PM_CORB) route |= (dlyline?0:PPC_ROUTE_HA2CB);
  if (iport==PM_HYPB && cport==PM_CORB) route |= (dlyline?0:PPC_ROUTE_HB2CB);
if (dma->type == IOPT_TBANK) {
  if (cport==PM_TUNA && oport==PM_CORHYPA) route |= PPC_ROUTE_CA2HA;
  if (cport==PM_TUNB && oport==PM_CORHYPB) route |= PPC_ROUTE_CB2HB;
                                           route |= (PPC_ROUTE_TA2R|PPC_ROUTE_TB2R);
} else {
  if (cport==PM_TUNA && oport==PM_CORA) route |= PPC_ROUTE_TA2CA;
  if (cport==PM_TUNB && oport==PM_CORB) route |= PPC_ROUTE_TB2CB;
  if (cport==PM_TUNA && oport==PM_CORHYPA) route |= (PPC_ROUTE_TA2CA|PPC_ROUTE_CA2HA);
  if (cport==PM_TUNB && oport==PM_CORHYPB) route |= (PPC_ROUTE_TB2CB|PPC_ROUTE_CB2HB);
  if (cport==PM_TUNA && oport==PM_HYPA) route |= PPC_ROUTE_TA2HA;
  if (cport==PM_TUNB && oport==PM_HYPA) print("Route TB2HA not supported\n");
  if (cport==PM_TUNA && oport==PM_HYPB) print("Route TA2HB not supported\n");
  if (cport==PM_TUNB && oport==PM_HYPB) route |= PPC_ROUTE_TB2HB;
  if (cport==PM_TUNA && oport==PM_HOST) route |= PPC_ROUTE_TA2R;
  if (cport==PM_TUNB && oport==PM_HOST) route |= PPC_ROUTE_TB2R;
}
  if (iport==PM_HYPA && cport==PM_TUNA) route |= (dlyline?0:PPC_ROUTE_HA2TA);
  if (iport==PM_HYPB && cport==PM_TUNA) route |= (dlyline?0:PPC_ROUTE_HB2TA);
  if (iport==PM_HYPA && cport==PM_TUNB) route |= (dlyline?0:PPC_ROUTE_HA2TB);
  if (iport==PM_HYPB && cport==PM_TUNB) route |= (dlyline?0:PPC_ROUTE_HB2TB);
  if (iport==PM_CORA && cport==PM_TUNA) route |= PPC_ROUTE_CA2TA;
  if (iport==PM_CORB && cport==PM_TUNB) route |= PPC_ROUTE_CB2TB;

if (dma->type == IOPT_TBANK) {
  if (p->gchip==GCFPGA) {
    /* both sides use TA as front end, TB as back end */
    if (iport==PM_HYPB && p->fttm>1) { route &= ~PPC_ROUTE_HB2TB; route |= PPC_ROUTE_HB2TA; }
    if (iport==PM_HYPA && p->fttm<2 && (dma->port%10)==3) route |= PPC_ROUTE_HA2TB;
    if (iport==PM_HYPB && p->fttm<2 && (dma->port%10)==3) route |= PPC_ROUTE_HB2TB;
  } else {
    if (iport==PM_HYPA && (dma->port%10)==3) route |= PPC_ROUTE_HA2TB;
    if (iport==PM_HYPB && (dma->port%10)==3) route |= PPC_ROUTE_HB2TB;
  }
}
  if (pmod==0) dma->enb = route;
if (prer2c) {
  if (route&PPC_ROUTE_HX2TA) pic_setkeyl(p,-(pmod*10+1),KEY_CORE,0x1); /* enable R2C */
  if (route&PPC_ROUTE_HX2TB) pic_setkeyl(p,-(pmod*10+2),KEY_CORE,0x1); /* enable R2C */
  if (route&PPC_ROUTE_HA2TA) route = (route^PPC_ROUTE_HA2TA)|PPC_ROUTE_HA2CA|PPC_ROUTE_CA2TA;
  if (route&PPC_ROUTE_HB2TB) route = (route^PPC_ROUTE_HB2TB)|PPC_ROUTE_HB2CB|PPC_ROUTE_CB2TB;
  if (route&PPC_ROUTE_HA2TB) route = (route^PPC_ROUTE_HA2TB)|PPC_ROUTE_HA2CB|PPC_ROUTE_CB2TB;
  if (route&PPC_ROUTE_HB2TA) route = (route^PPC_ROUTE_HB2TA)|PPC_ROUTE_HB2CA|PPC_ROUTE_CA2TA;
}
  if (pmod==0) return route;
  if (dma->type==IOPT_CORE  && dport>10) dport -= (pmod*10);
  if (dma->type==IOPT_MCORE && dport>10) dport -= (pmod*10);
  if (dma->type==IOPT_TBANK && dport>10) dport -= (pmod*10);
  if (dma->type==IOPT_TUNER) dport = (dport&1)? 1 : 2;

  if (dma->master > 0) return route;

  tmp[0] = dlyline? (0x1<<(iport+16))|route : route;
  tmp[1] = dlyline? (cport<<12)|(iport<<8)|0x0F : 0;
  tmp[2] = dlyline;
  vprint("Activate=%d Port=%d Type=%d Enb=%08x Route=%08x on Processor=%d\n",mode,dport,dma->type,dma->enb,route,pmod);
  pic_msg (p,JTAG_PORT_PMOFF+pmod,(mode>0)?PKTF_RUN:PKTF_STOP,(dma->type<<16)|dport,tmp,12,0,-1);
  return route;
}

int_4 pic_sport_ (PICSTRUCT *p, int_4 port, int_4 dmac, 
  		int_4 dir, int_4 bits, int_4 sync, int_4 rate, int_4 flags)
{
  int_4 n,chn,sdmac,lport,sysreg,side,iport,stat,nstat,aflags=FLG_TUNER;
  if (flags&FLG_DISABLE) return 0;
  side = (port&1)? 1:2;
  iport = port-1;
  sdmac = 8 + side;
  lport = sdmac;
  dmac = 12 + port;
  chn = (port-1)/2;
  vprint("STREAM Channel port=%d side=%d dmac=%d sdmac=%d\n",port,side,dmac,sdmac); 
  n = dmac-1;
  p->dmaci = dmac;
  p->dma[n].dir  = dir;
  p->dma[n].reg  = 0;
  p->dma[n].enb  = 0;
  p->dma[n].enbx = 0;
  p->dma[n].mcfg = (side<<28)|0x08000001;
  p->dma[n].port = port;
  p->dma[n].flags = flags;
  return dmac;
  p->dma[sdmac-1].type = IOPT_MCORE;
  p->dma[sdmac-1].master = -1;
  p->dma[sdmac-1].feed = 0x1;
  p->dma[sdmac-1].port = side;
  p->dma[sdmac-1].mcfg = 0;
  p->dma[sdmac-1].dec = 1;
  p->dma[sdmac-1].enb = (side==2)? PPC_ROUTE_TB2R : PPC_ROUTE_TA2R;
  p->dma[sdmac-1].reg = (side==2)? PPC_OSTAT(PM_TUNB) : PPC_OSTAT(PM_TUNA);
  p->dma[n].master = sdmac;
  p->dma[n].reg = p->dma[sdmac-1].reg;
  nstat = pic_acmd (p, ADON_NSTAT, sdmac, 0, 0);
  if ((NSTAT_MSK(chn) & nstat) != 0) { printf("Stream %d already in use\n",port); return 0; }
  if (nstat!=0) aflags |= FLG_INFO;
  pic_dma (p, sdmac, -1, 0, 0, 1024*1024, 0, aflags);
/*  pic_setup_qsfp (p,side,dir,bits,rate,flags); */
  sysreg = (chn<<12) | (iport<<8) | 0x0001;
  stat = pic_acmd(p,ADON_WRB,(side==2)?PPC_TUNB_CTL:PPC_TUNA_CTL,sysreg,4);
  return dmac;
}

int_4 pic_sport (PICSTRUCT *p, int_4 port, int_4 dmac,
                int_4 dir, int_4 bits, int_4 sync, int_4 rate, int_4 flags)
{
  int_4 i,n,chn,sdmac,lport,sysreg,side,iport,stat,nstat;
  int_4 rawbits,qsfphold,chena;
  int_4 rxusrsys,rxusroffset,rxusrsz,rxusrhdr;
  int_4 rxusripadr,rxusrprtcl;
  int_4 aflags=FLG_TUNER;

  if (flags&FLG_DISABLE) return 0;

  side = (port&1)? 1:2;
  iport = port-1;
  sdmac = 8 + side;
  lport = sdmac;
  dmac = 12 + port;
  chn = (port-1)/2;
  vprint("STREAM Channel port=%d side=%d dmac=%d sdmac=%d\n",port,side,dmac,sdmac);
  n = dmac-1;
  p->dmaci = dmac;
  p->dma[n].dir  = dir;
  p->dma[n].reg  = 0;
  p->dma[n].enb  = 0;
  p->dma[n].enbx = 0;
  p->dma[n].mcfg = 0;
  p->dma[n].port = port;
  p->dma[n].flags = flags;

  if (p->nicno>=0) { return dmac; }	/* in NIC mode */

  if (p->dsgtype>0) {	/* in xPAC mode */
    pic_setup_qsfp (p,dmac,dir,bits,rate,flags);
    return dmac;
  }
  /* in PICxQ mode */
  p->dma[n].mcfg = (side<<28)|0x08000001;
  p->dma[sdmac-1].type = IOPT_MCORE;
  p->dma[sdmac-1].master = -1;
  p->dma[sdmac-1].feed = 0x1;
  p->dma[sdmac-1].port = side;
  p->dma[sdmac-1].mcfg = 0;
  p->dma[sdmac-1].dec = 1;
  p->dma[sdmac-1].enb = (side==2)? PPC_ROUTE_TB2R : PPC_ROUTE_TA2R;
  p->dma[sdmac-1].reg = (side==2)? PPC_OSTAT(PM_TUNB) : PPC_OSTAT(PM_TUNA);
  p->dma[n].master = sdmac;
  p->dma[n].reg = p->dma[sdmac-1].reg;
  nstat = pic_acmd (p, ADON_NSTAT, sdmac, 0, 0);
  if ((NSTAT_MSK(chn) & nstat) != 0) { printf("Stream %d already in use\n",port); return 0; }
  if (nstat!=0) aflags |= FLG_INFO;
  pic_dma (p, sdmac, -1, 0, 0, 1024*1024, 0, aflags);
/*  pic_setup_qsfp (p,side,dir,bits,rate,flags); */

  sysreg = 0;
  i = (side==2)?PPC_TUNB_CTL:PPC_TUNA_CTL;
  stat = pic_acmd(p,ADON_WRB,i,sysreg,4);
  stat = pic_acmd(p,ADON_WRB,i|(9<<20),(side==2)?0xAA:0x55,4);

  qsfphold = findintflagdef("QSFPHOLD",p->config,0x0FFF0000);
  rawbits = findintflagdef("RXRAWBITS",p->config,0);
  rxusrsys = findintflagdef("RXUSRSYS",p->config,0);
  rxusrsys |= rawbits<<1;
  rxusrsys |= findintflagdef("RXENDCONV",p->config,0)<<3;
  rxusroffset = findintflagdef("RXUSROFFSET",p->config,14);
  /* rxusroffset = (rxusroffset<-4)?0:rxusroffset+4; */
  rxusrsz = findintflagdef("RXUSRSZ",p->config,0);
  /* rxusrsz = (rxusrsz<0)?0:rxusrsz; */
  /* rxusrsys |= (rxusrsz<0)?1<<2:0; */
  rxusrhdr = findintflagdef("RXUSRHDR",p->config,0);
  /* rxusrdstadr = findintflagdef("RXUSRDSTADDR",p->config,0); */
  rxusrprtcl = findintflagdef("RXUSRPRTCL",p->config,0);
  rxusripadr = convert_ip_addr("RXUSRIPADDR",p->config);
  rxusripadr = ((rxusripadr&0xFF)<<24)|((rxusripadr&0xFF00)<<8)|((rxusripadr&0xFF0000)>>8)|((rxusripadr&0xFF000000)>>24);

  if(findintflagdef("RXUDPDATA",p->config,0) > 0) {
    rxusrprtcl = 0x11;
    rxusroffset = 98;
  }
  if(findintflagdef("RXRAWDATA",p->config,0) > 0) {
    rxusrprtcl = 0;
    rxusroffset = 0;
  }

  i = ((side==2)?PPC_TUNB_CTL:PPC_TUNA_CTL)|(port<<20);
  stat = pic_acmd(p,ADON_WRB,i,rxusrsys|1,4);
  stat = pic_acmd(p,ADON_WRB,i|4,rxusroffset,4);
  stat = pic_acmd(p,ADON_WRB,i|8,rxusrsz,4);
  stat = pic_acmd(p,ADON_WRB,i|12,rxusrhdr,4);
  stat = pic_acmd(p,ADON_WRB,i|20,rxusrprtcl,4);
  stat = pic_acmd(p,ADON_WRB,i|24,rxusripadr,4);
  stat = pic_acmd(p,ADON_WRB,i|28,qsfphold,4);
  stat = pic_acmd(p,ADON_WRB,i,rxusrsys,4);

  sysreg = (rawbits<<31) | (chn<<12) | (iport<<8) | 0x0001;
  i = ((side==2)?PPC_TUNB_CTL:PPC_TUNA_CTL);
  stat = pic_acmd(p,ADON_WRB,i,sysreg,4);

  return dmac;
}

int_4 checkTunerConfig (PICSTRUCT *p) {
  int stat,i,n=5;
  if (p->mchan>0 || p->nchan>0) return 0;
  for (i=1; i<n && (stat=pic_tuner_config(p))<0; i++) printf("Trouble getting tuner Config. Try %d of %d\n",i,n);
  return stat;
}


int_4 pic_tport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir,
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 status,nstat,lport,sport,mport,pport,dmac_lp,dmac_sp,dmac_po,side,pmod,tqd;
  int_4 chan,chip,sdmac,disable,n,aflags,adec,mrate,ovsr,flag;
  int_4 feed,dport,iport,oport,mbits,reg;
  char fname[40]; double ratio;

  side  = (port&0x1)? 1 : 2;
  disable = (flags&FLG_DISABLE)? 1 : 0;
  checkTunerConfig(p);

  if (port<0) port=0;
  if (port==2 && p->mchan==1); /* HHTQ */
  else if (port>p->mchan) { 
    print("Channel #%d out of range [1-%d]\n",port,p->mchan);
    return -1;
  }

       if (abs(bits)==16);
  else if (abs(bits)==8  && p->gchip==GCFPGA);
  else if (abs(bits)==32 && p->gchip>=GC4016);
  else {
    print("Tuner channels do not support format bits=%d on this card\n",bits);
    return -1;
  }

  pport = port;
  pmod = port2pmi(p,&pport); /* auto processor module index */
  if (pmod==2) port = 32+pport;
  vprint("Tuner Port=%d (PM%dTUNER%d) bits=%d dec=%d gain=%d\n",port,pmod,pport,bits,dec,gain);

  /* parse input/output ports */
  if (pmod==0 && p->pm0mt==0) 
       dport = (port&1)? PM_CORA:PM_CORB;
  else dport = (port&1)? PM_TUNA:PM_TUNB;
  feed  = setupFeeds (p,pmod,dport,dir);
  iport = (feed>>16)&0xF;
  oport = (feed>>24)&0xF;
  if (dir>0) mport = (oport==PM_HYPA)? 1 : (oport==PM_HYPB)? 2 : 0;
  else       mport = (iport==PM_HYPA)? 1 : (iport==PM_HYPB)? 2 : 0;
  mbits = p->mbits;

  /* apply standard flags */
  if ((ovsr=findintflag("OVSR",p->config))>=0)  pic_tuner_ovsr (p,ovsr,side,0);  
  if (!disable) {
    if (p->gchip==GCFPGA) p->deco = dec;
    if (findstrflag("CFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
    if (findstrflag("PFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
    if (findstrflag("RFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
  }

  pic_tuner_ovsr (p,0,0,FLG_INFO);  /* retrieve ovsr */
  ovsr = p->ovsr[side-1];  
  if (dir>0) {
    p->tflags |= FLG_SWAPIO;	/* mbits/obits swap */
    mrate = rate/dec;
    if (bits>0 && mbits<0) mrate = mrate/2;
  } else {
    mrate = rate;
    if ((p->tflags&FLG_POVSR)!=0 && ovsr>1) mrate = rate/ovsr;
  }

  aflags = FLG_TUNER; 
  if (getIOCsig(p,IS_OUTPUT|mport) && dir<0) { aflags |= FLG_RGO; dir=0; } /* output module */

  if (disable==0) {
    adec = pic_tuner_dec (p,dec,port,0);  /* validate tuner decimation */
    if (adec!=dec) print("Initial port=%d tuner decimation=%d not available. Actual dec=%d.\n",port,dec,adec);
  }

  if (p->isDUC) {
    if (p->mcs==0) p->mcs=-1;
    chan  = ((port-1)>>1)&0xF;
    chip  = (port&1)? 1 : 2;
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, flags);
    dmac = p->ndmac + port;
    sdmac = 4+chip;
    pic_lport (p, chip, sdmac, dir, bits, flags); 
    p->dma[dmac-1].reg = p->dma[sdmac-1].reg;
    p->dma[dmac-1].enb = p->dma[sdmac-1].enb;
    return dmac;
  }

  if (p->type == ICEPIC2) {
    sport = mport;
    /* set up tuner chip */
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, flags);
    if (status < 0) return (-1);
    /* set up serial port for follow on DMA */
    bits = abs(bits);
    dmac_sp = pic_sport (p, sport, dmac, -1, bits, bits, rate, flags|aflags); 
    if (dmac_sp < 0) return (-1);
    /* set up the I/O Module DMA=0 DIR=0 BITS=16 */
    status = pic_mport (p, mport, dmac_sp, dir, 16, mrate, 0, flags|aflags);
    if (status < 0) return (-1);
    if (disable) return 0;
    return (dmac_sp);
  }
  else if (p->type == ICEPIC3) {
    lport = p->linktbl[port];
    sdmac = lport2dmac[lport];
    /* set up tuner chip */
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, flags);
    if (status < 0) return (-1);
    /* set up link port for follow on DMA */
    bits = abs(bits);
    dmac_lp = pic_ioport (p, IOPT_LINK, lport, sdmac, -1, bits, rate, fcny, dec, gain, flags|aflags);
    if (dmac_lp != sdmac) return (-1);
    p->dma[sdmac-1].port = port;
    /* set up the I/O Module DMA=0 DIR=0 BITS=16 */
    status = pic_mport (p, mport, dmac_lp, dir, 16, mrate, 0, flags|aflags);
    if (status < 0) return (-1);
    if (disable) return 0;
    return (dmac_lp);
  } 
  else if (p->gchip == GCFPGA && pmod==0 && p->pm0mt==0) {
    lport = p->linktbl[port];
    sdmac = lport2dmacy[lport];
    /* set up tuner chip */
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, flags);
    if (status < 0) return (-1);
    /* set up link port for follow on DMA */
    bits = abs(bits);
    p->dma[sdmac-1].feed = feed;
    dmac_lp = pic_ioport (p, IOPT_LINK, lport, sdmac, dir, bits, rate, fcny, dec, gain, flags|aflags);
    if (dmac_lp != sdmac) return (-1);
    p->dma[sdmac-1].port = port;
    if (status>0) { tqd = status; /* setup WB prefilter */
      dmac_po = dmac_lp+1; p->dma[dmac_po-1].port = port+1;
      p->dma[dmac_lp-1].enb &= ~PPC_ROUTE_HA2CA;
      p->dma[dmac_lp-1].enb |=  PPC_ROUTE_HA2CB;
      p->dma[dmac_lp-1].enbx &= ~PPC_ROUTE_HA2CA;
      p->dma[dmac_lp-1].enbx |=  PPC_ROUTE_HA2CB;
      pic_acmd (p, ADON_WR, DMA_ROUTF, PPC_ROUTF_CB2CA, 0);
    }
    /* set up the I/O Module DMA=0 DIR=0 BITS=16 */
    status = pic_mport (p, mport, dmac_lp, dir, mbits, mrate, 0, flags|aflags);
    if (status < 0) return (-1);
    if (disable) return 0;
    return (dmac_lp);
  }
  else {
    if ((p->gchip==GC4016) && (p->cpc==2) && ((port-1)&0x6)==0x2) port += 2;  /* auto increment 2nd channel */
    chan  = (p->type==ICESLIC3)? p->chantbl[port] : ((port-1)>>1)&0xF;
    chip  = port2chip(p,pport); if (pmod>0 && p->gchip!=GCFPGA) chip = 8+side;
    lport = chip2link(p,chip); if (pmod==2) lport += 2; if (p->isY) lport += 4;
    if (p->isX) sdmac = lport2dmacx[lport];
    else if (p->isY) sdmac = lport2dmacy[lport];
    else sdmac = lport2dmac[lport];
    p->dma[sdmac-1].feed = feed;
    /* set up the DMA channel for this port */
    dmac = p->ndmac + port; 	
    p->dma[dmac-1].master = sdmac;	
    p->dma[dmac-1].flags = flags;
    p->dma[dmac-1].port = port;
    p->dma[dmac-1].rate = rate;
    p->dma[dmac-1].bits = bits;
    p->dma[dmac-1].dec = dec;
    p->dma[dmac-1].dir = dir;
    /* get the current tuner state */
    nstat = pic_acmd (p, ADON_NSTAT, sdmac, 0, 0);
    vprint ("Current tport=%d cpc=%d NSTAT=%08x for dec=%d\n",port,p->cpc,nstat,dec);
    /* check channel availability */
    if (p->cpc<=16 && (NSTAT_MSK(chan) & nstat) != 0) {
      print ("WARN: Overriding Channel %d already in use\n",port);
    }
    if (p->gchip==GCFPGA);
    else if (p->cpc==1 && (chan&0x3)!=0) {
      print ("ERROR: Channel %d not available with CPC=1\n",port); return 0;
    }
    else if (p->cpc==2 && (chan&0x1)!=0) {
      print ("ERROR: Channel %d not available with CPC=2\n",port); return 0;
    }
    /* perform initial setup of master lport */
    if (disable) aflags |= FLG_INFO;
    else if (nstat==0) {	/* set up tuner chips globally */
      if (pmod==0) pmod=4;
      status = pic_tuner (p, (pmod<<16)|chip, bits, rate, fcny, dec, gain, flags);
      if (status < 0) return (-1);
    }
    else if (NSTAT_DEC!=dec && (p->tflags&FLG_ITDEC)==0) { 
      print ("ERROR: Mixed decimation on same tuner chip\n"); return 0;
    }
    else aflags |= FLG_INFO;	/* make pic_ioport just query link port */
	
    /* set up the Module configuration */
    status = pic_mport (p, mport, dmac, dir, mbits, mrate, 0, flags|aflags);
    if (status < 0) return (-1);

    if (disable) return 0;
    
    /* set up the link port DIR=0 BITS=16 */
    if (p->isX && p->type!=ICESLIC3 && chip<=2) aflags |= FLG_HS;
    dmac_lp = pic_ioport (p, IOPT_LINK, lport, sdmac, -1, bits, rate, fcny, dec, gain, flags|aflags);
    if (dmac_lp != sdmac) return (-1); 
    n = sdmac-1; 
    if (p->isY) {
      p->dma[dmac-1].reg = p->dma[n].reg;
      if (p->tcmode!=TCM_OFF) p->dma[n].mcfg |= IOM_TEN;
    }
    p->dma[n].type = IOPT_TUNER;	
    p->dma[n].port = chip;
    p->dma[n].master = -1;
    p->dma[n].mcfg |= (p->dma[dmac-1].mcfg&IOM_BITS);
    /* setup the master link DMA for internal aquisition */
    pic_dma (p, sdmac, -1, 0, 0, 1024*1024, 0, aflags);

    /* slic3s lport=6 is actually on the 1 side */
    if (p->type==ICESLIC3 && (p->tflags&FLG_TALT1)==0) {
      p->dma[n].mcfg -= 0x10000000;
      p->dma[n].mcfg |= IOM_SEN;
    }

    /* set the frequency and gain for individual channels */
    flag = (NSTAT_DEC!=dec)? FLG_DTUN : FLG_MTUN;
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, flag);
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, FLG_FTUN);
    status = pic_tuner (p, port, bits, rate, fcny, dec, gain, FLG_GTUN);

    return (dmac);
  }

}


int_4 pic_tbport (PICSTRUCT *p, int_4 port, int_4 dmac, int_4 dir,
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i,j,k,l,n,m,pktlen,status,side,lport,mport,aflags,mcfg,ovsr,irate,mrate,cpb,cpc1,cpc2,disable,mbits,mindec1,mindec2;
  int_4 m1,m2,m3,adec,d1,d2,d3,x2,p2,g2,tbits,phinc,dphinc,dport,feed,iport,oport,pmod,pmport,beport,cpb1,cpb2,cflg,sys,csig,crate,ups;
  int_4 vt = (p->gchip==GCFPGA);
  int_4 prer2c = findintflagdef("PRER2C",p->config,0);
  char fname[40];
  double f,dfreq,ratio;
  FTTSTRUCT ftt;

  disable = (flags&FLG_DISABLE)? 1 : 0;
  checkTunerConfig(p);

  if (port<0) port=3;
  pmod = port/10;
  if (pmod==0) pmod = p->pmi;
  if (pmod==3) pmod = 1;
  pmport = port%10;
  beport = pmod*10 + 4;
  port = pmod*10 + pmport;
  mbits = p->mbits;

  if (p->fttm>1 && (p->tflags&FLG_RESAMP)!=0) {
    print("TunerBank FTTM>1 only supports PRESAMP flag, not RESAMP.\n"); return -1; }

  if (p->fttm==0 && power2ge(dec)!=dec) {
    print("TunerBank FTTM=0 only supports decimation=%d by powers of 2.\n",dec); return -1; }

       if (pmport==1 || pmport==3) side=1;
  else if (pmport==2 || pmport==4) side=2;
  else { print("Illegal port=%d in pic_tbport()\n",port); return -1; }

  vprint("TunerBank Port=%d (PM%dTBANK%d) bits=%d dec=%d gain=%d\n",port,pmod,pmport,bits,dec,gain);

  vfill (0, (int_4 *)(&ftt), sizeof(FTTSTRUCT));
  p->tflags |= FLG_TBANK;

  if (dir<0) aflags = FLG_TUNER;
  if (getIOCsig(p,IS_OUTPUT|side) && dir<0) { aflags |= FLG_RGO; dir=0; } /* output module */

  if (p->isY) lport = dmac = (pmod==0)? cport2dmac[pmport] : (6+2*pmod) + side;
  else { lport = p->linktbl[side]; dmac  = lport2dmacx[lport]; }

  if ((ovsr=findintflag("OVSR",p->config))>=0)  pic_tuner_ovsr (p,ovsr,pmport,0);  

  pic_tuner_ovsr (p,0,0,FLG_INFO);  /* retrieve ovsr */
  ovsr = p->ovsr[side-1];  
  mrate = rate;
  if ((p->tflags&FLG_POVSR)!=0 && ovsr>1) mrate = rate/ovsr;

  /* parse input/output ports */
  dport = core_port(pmport);
  if (pmod>0) dport = (pmport&1)? PM_TUNA:PM_TUNB;
  feed = setupFeeds (p,pmod,dport,dir);
  iport = (feed>>16)&0xF;
  oport = (feed>>24)&0xF;
  if (dir>0) mport = (port&1)? 1 : 2;
  else mport = (iport==PM_HYPA)? 1 : (iport==PM_HYPB)? 2 : 0;
  p->dma[dmac-1].feed = feed;
  p->dma[dmac-1].port = port;

  /* set up the Module configuration */
  status = pic_mport (p, mport, dmac, dir, mbits, mrate, 0, flags|aflags);
  if (status<0) return (-1);
  mcfg = p->dma[dmac-1].mcfg;
    
  /* set up the link port */
  dmac = pic_lport (p, lport, dmac, dir, 32, flags); 
  if (dmac<0) return (-1);
  p->dma[dmac-1].mcfg |= mcfg;

  /* set up the tuner channels */
  if (abs(bits)!=16 && abs(bits)!=32) { 
    print("TunerBank channels only support 16|32 bit data formats. Bits = %d\n",bits);
    return -1;
  }
  tbits = (p->fttm>1)? 16 : bits;

  if (prer2c) {
    mbits = -16;
    rate = rate/2;
  }
  mbits = findintflagdef("NBITS",p->config,mbits);
  dfreq = p->dfreq;

  /* output inverse tuner bank setup */
  if (dir>0) {
    if (disable) return 0;
    j = core_addr(pmport);
    csig = pic_rpb(p,pmod,j|COREG_PARAM0); 
    crate = pic_rpb(p,pmod,j|MCORE_SYS|COREG_PARAM7)*1000000; 
    ups = findintflagdef("UPS",p->config,(rate>crate)?2:0); /* upsampler mode */
    if (ups>0) loadUPStaps(p,pmod,j,ups);
    m1 = p->nchan/2;
    i = 0x00800409;		/* tbnk|ibnk|bulk */
    i |= (m1<<16);		/* num channels */
    if (ups>1) i |= 0x4000;	/* UPS output mode */
    pic_wpb (p,pmod,j|MCORE_SYS,i);
    port = pmod*10+port;
    set_awg_gain(p,port,0,gain);
    p->dec5 = findintflagdef("NFGAIN",p->config,-100);
    p->mcs = m1;
    set_awg_gain(p,port,99,p->dec5);	/* multicore noise gain */
    for (n=0; n<p->mcs; n++) setup_duc (p, port, n, rate, p->res_ratio, gain, fcny+n*dfreq, csig);
    pic_wpb (p,pmod,j|MCORE_SYS,i);
    return dmac;
  }

  /* determine back end CPC */
  cpb  = p->cpb;
  cpc1 = p->cpc;
  cpc2 = vt? p->cpc : 4;
  cpc2 = findintflagdef("FTTC2",p->config,cpc2);
  cpb1 = (cpb/2)*cpc1;
  cpb2 = (cpb/2)*cpc2;

  /* apply standard flags */
  if (!disable && p->fttm>1) {
    k = p->tflags;
    i = (p->gchip==GCFPGA)? pmod*10+1 : port;
    pic_loadfile (p, "cfir_90", FLG_FC|i); 
    pic_loadfile (p, "pfir_90", FLG_FC|i); 
    p->tflags = k;
  } 
  p->cpc = cpc2;
  if (!disable && p->fttm>0) {
    i = (p->fttm>1)? (pmod*10)+8 : port;
    if (findstrflag("CFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|i); 
    if (findstrflag("PFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|i); 
    if (findstrflag("RFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|i); 
  }
  p->cpc = cpc1;

  if (disable) {
    p->chan = -1;
    if (p->fttm<1) return 0; /* no tuners, all done */
    status = pic_tuner (p, port, tbits, rate, fcny, dec, gain, flags);
    if (p->fttm<2) return 0; /* no back end, all done */
    p->chan = -1;
    p->cpc = cpc2;
    status = pic_tuner (p, beport, bits, rate, 0.0, 16, 0, FLG_BACKEND|flags);
    p->cpc = cpc1;
    return 0;
  }

  if ((p->tflags&FLG_POVSR)!=0) ovsr = 1; 
  else if (ovsr>1) { dec *= ovsr; rate *= ovsr; dfreq /= ovsr; }

  /* set constant channel spacing mode */
  if (p->fttm<0 && dfreq!=0) p->fttm=1;
  /* width = (m2-1)*chsp + chbw < 0.90 * 1/d1 */

  m1 = findintflag("FTTN1",p->config);
  m2 = findintflag("FTTN2",p->config);
  d1 = findintflag("FTTD1",p->config);
  d2 = findintflag("FTTD2",p->config);
  m3 = 1;
  d3 = 1;
  mindec1 = vt? 4 : 4*cpc1;
  mindec2 = vt? 16 : 4*cpc2;

  if (p->fttm == 0) {
    d1 = power2ge(dec);
    m1 = p->nchan;
    m2 = 1;
    d2 = 1;
  }
  else if (p->fttm > 1) {
    if (d2<0) { 
     d2=mindec2-2;
     do { d2+=2; if (p->fttm==3) d3=d2; } while (dec%(d2*d3*2)!=0 && (d2*d3)<dec/mindec1);
    }
    if (d1<0) d1 = max(mindec1,dec/(d2*d3)) & 0xFFFE;
    if (m2<0 && dfreq>0) m2 = (int)((0.90/d1-1.0/dec/d3)/dfreq+1);
    if (m2<0 && dfreq<=0) m2 = cpb2;
    m2 = min(cpb2,m2);
    if (p->fttm==3) m3=m2;
    if (m1<0) m1 = (p->nchan-1)/(m2*m3)+1;
    if ((m1&0x1)!=0 && pmport!=3) m1++;
    m1 = min(cpb1,m1);
  } else {
    m1 = p->nchan;
    d1 = dec;
    m2 = 1;
    d2 = 1;
    m1 = min(cpb*cpc1,m1);
  }
  if (pmport==3) m1 = max(2,m1);

  p2 = findintflag("FTTP2",p->config);
  if (p2<0) p2 = B256K/d2;
  if (p->gchip==GCFPGA) p2 *= 2;
  p2 = p2-(p2%p->bpp); /* insure transfers on bpp by boundaries */
  x2 = p2*d2;

  if (d1*d2*d3 != dec) {
    print("Err: Unable to factor TunerBank dec=%d into dec1*2*3=%d|%d|%d (16+)\n",dec,d1,d2,d3);
    return -1;
  }
  if (m1*m2*m3 < p->nchan && p->fttm>0) {
    print("Err: Unable to obtain desired TunerBank channels=%d max=%d with FTTM=%d D1=%d D2=%d D3=%d N1=%d N2=%d N3=%d\n",
      m1*m2*m3,p->nchan,p->fttm,d1,d2,d3,m1,m2,m3); p->nchan = m1*m2*m3;
    return -1;
  }

  adec = pic_tuner_dec (p,d1,pmport,0);  /* validate tuner decimation */
  if (adec!=d1) print("Initial port=%d tuner dec1=%d not available\n",port,d1);

  /* save off parameters for use under the hood */
  p->d1 = d1;
  p->d2 = d2;
  p->d3 = d3;
  p->m1 = m1;
  p->m2 = m2;
  p->m3 = m3;
  p->ofreq = 0;

  /* setup PPC FTT controller structure */
  ftt.stages = max(1,p->fttm);
  ftt.bits = bits;

  /* set the speed step = (i/16)*101MHz */
  irate = (rate/d1) * m1 * 1.2;
  if (ftt.stages>=2) irate *= 2.5;
  j = p->tun_clk / 16;
  for (i=1; i<16 && i*j<irate; i++);
  ftt.speed = i;
  i = findintflag("SPEED",p->config);
  if (i>0) ftt.speed = i;

    ftt.tb[0].mchan = m1;
    ftt.tb[0].nchan = p->nchan;
    ftt.tb[0].dec   = d1;
    ftt.tb[0].cpc   = cpc1;
    ftt.tb[0].xfer  = x2;
    ftt.tb[0].prod  = p2;
    ftt.tb[0].pstrt = vt? 0 : B16M;
    i = (vt || cpc1<4)? PB2M : PB1M;
    if (p->fttm==0) for (i=PB1M; (1<<i) > B256M/max(1,m1/8); i--);
    ftt.tb[0].pcfg  = PIPE_CFG(i);
    ftt.tb[0].psize = PIPE_SIZE(i);
    ftt.tb[0].pmask = PIPE_MASK(i);

  if (ftt.stages>=2) {
    j = vt? B64M : (ftt.stages==2)? B32M : B16M;
    for (i=PB256K; i>PB64K && m1*m2*PIPE_SIZE(i)>j; i--);
    ftt.tb[0].nchan = m1;
    ftt.tb[1].mchan = m2;
    ftt.tb[1].nchan = p->nchan;
    ftt.tb[1].dec   = d2;
    ftt.tb[1].cpc   = cpc2;
    ftt.tb[1].xfer  = min(B64K,PIPE_SIZE(i)/2);
    ftt.tb[1].prod  = (ftt.stages==2)?0:ftt.tb[1].xfer/d3;
    ftt.tb[1].pstrt = ftt.tb[0].pstrt + m1*ftt.tb[0].psize;
    ftt.tb[1].pcfg  = PIPE_CFG(i);
    ftt.tb[1].psize = PIPE_SIZE(i);
    ftt.tb[1].pmask = PIPE_MASK(i);
    p->ofreq = 0.5*(m2-1)*dfreq*m3 + 0.5*(m3-1)*dfreq;
    ftt.tb[1].phinc = (int)(0.5*(1.0-d1*(m2-1)*dfreq*m3)*FULLSCALE + 0.499999);
    for (ftt.tb[1].tphinc=0,i=0; i<ftt.tb[0].xfer/2; i++) ftt.tb[1].tphinc+=ftt.tb[1].phinc; 
    ftt.tb[1].dphinc = (int)((d1*dfreq*m3)*FULLSCALE + 0.499999);
    for (ftt.tb[1].tdphinc=0,i=0; i<ftt.tb[0].xfer/2; i++) ftt.tb[1].tdphinc+=ftt.tb[1].dphinc; 
    if (ovsr>1) p->ofreq *= ovsr;
    if (p->gchip==GC4016) for (i=0,j=NBANK; i<NBANK; i++,j++) ftt.taddr[i] = PPC_TADDR(j);
  }
  if (ftt.stages>=3) {
    for (i=PB64K; i>PB4K && m1*m2*m3*PIPE_SIZE(i)>B16M; i--);
    ftt.tb[1].nchan = m1*m2;
    ftt.tb[2].mchan = m3;
    ftt.tb[2].nchan = p->nchan;
    ftt.tb[2].dec   = d3;
    ftt.tb[2].cpc   = cpc2;
    ftt.tb[2].xfer  = min(B16K,PIPE_SIZE(i)/2);;
    ftt.tb[2].prod  = 0;
    ftt.tb[2].pstrt = ftt.tb[1].pstrt + m1*m2*ftt.tb[1].psize;
    ftt.tb[2].pcfg  = PIPE_CFG(i);
    ftt.tb[2].psize = PIPE_SIZE(i);
    ftt.tb[2].pmask = PIPE_MASK(i);
    ftt.tb[2].phinc = (int)(0.5*(1.0-d1*d2*(m3-1)*dfreq)*FULLSCALE + 0.499999);
    for (ftt.tb[2].tphinc=0,i=0; i<ftt.tb[1].xfer/2; i++) ftt.tb[2].tphinc+=ftt.tb[2].phinc; 
    ftt.tb[2].dphinc = (int)((d1*d2*dfreq)*FULLSCALE + 0.499999);
    for (ftt.tb[2].tdphinc=0,i=0; i<ftt.tb[1].xfer/2; i++) ftt.tb[2].tdphinc+=ftt.tb[2].dphinc; 
  }
  p->phinc = ftt.tb[1].phinc;
  p->dphinc = ftt.tb[1].dphinc;

  /* force frame size for banked output */
  pktlen = p->pktlen;
  if (pktlen<=0) pktlen = B16K;
  i = ftt.tb[ftt.stages-1].psize-B1K;
  if ((pktlen%p->bpp) != 0 || pktlen>i) {
    print("Err: Tuner Bank packet|frame size=%d must be a multiple of %d bytes and < %d.\n",p->pktlen,p->bpp,i);
    return -1;
  }

  for (i=0; i<3; i++) {
    ftt.tb[i].cons = ftt.tb[i].xfer;
    if (i+1 <  ftt.stages) ftt.tb[i].xfer += ( vt? B16K+(B1K*ftt.tb[i+1].dec) : B8K );	/* xtra to flush FIFOs */
    if (i+1 == ftt.stages) ftt.tb[i].xfer = ftt.tb[i].cons = (p->fttm==0)? pktlen*8:pktlen;
  }

  vprint("FTT Port=%d Stages=%d Dec=%d Speed=%d MRate=%d\n",port,ftt.stages,dec,ftt.speed,mrate);
  for (i=0; i<ftt.stages; i++)
    vprint("FTT%d N=%d M=%d Dec=%d Xfer=%d Cons=%d Prod=%d PipeStr=%08x PipeSz=%08x phinc=%08x dphinc=%08x Cpc=%d\n",i,
    ftt.tb[i].nchan,ftt.tb[i].mchan,ftt.tb[i].dec,ftt.tb[i].xfer,ftt.tb[i].cons,ftt.tb[i].prod,
    ftt.tb[i].pstrt,ftt.tb[i].psize,ftt.tb[i].phinc,ftt.tb[i].dphinc,ftt.tb[i].cpc);

  /* setup output configurations */
  i = oport;
  if (i==PM_CORHYPA) i=PM_CORA;
  if (i==PM_CORHYPB) i=PM_CORB;

  /* download FTT controller structure */
  if (pmport==2) {
    ftt.omodb = i;
    ftt.omoda = i-1;
  } else {
    ftt.omoda = i;
    ftt.omodb = i+1;
  }
  ftt.lstage = ftt.busy = -1;
  ftt.dca[0] = ftt.dca[1] = 0;
  if (p->fttm==0) ftt.stages = -ftt.stages;
  if (p->fttm==0) ftt.tb[0].mchan = ftt.tb[0].nchan = p->nchan/8;	/* 8 per memory block */
  i = (p->fttm==1 && (port%10)==2 && p->isY)? 0x180:0;
  if (pmod>0) pic_wpm (p, pmod, PPC_FTT_ADDR+i, (int_4 *)(&ftt), sizeof(FTTSTRUCT), 0);
  if (pmod>0 && i>0) pic_wpm (p, pmod, PPC_FTT_ADDR, (int_4 *)(&ftt), 4, 0);	/* update STAGES */

  if (ovsr>1) { dec /= ovsr; rate /= ovsr; }

  /* init channelizer core */
  if (p->fttm==0) {
    i = pmod*10 + 2 + pmport; 
    n = (mbits>0)? dec : dec/2;	/* FFT size */
    n = dec;			/* FFT size */
    m = (pmport==3)? m1 : m1/2;
    k = pow2sizem1(n)>>16;	/* pow2 */
    j = fftscale((gain/6),k);
    sys = (getFmt(bits)<<12) | (getFmt(mbits)<<8) | 0x00;
    if (pmport==3) sys |= CORE_XAB;	/* flags for SPLIT or XAB output */
    cflg = 0x1;				/* flags for PKT output */
    ratio = p->res_ratio;
    if (ratio<=0) ratio=1;
    if (bits>0) ratio*=1;
    pic_setkeyl (p,-i,KEY_CORE+COREG_SYS,sys);
    pic_setkeyl (p,-i,KEY_CORE+COREG_USER1,0x7);			/* PPFtaps-1 */
    if (findstrflag("FPPF",p->config,fname,-1)<0) sprintf(fname,"botfppf_%d",n);
    pic_loadfile (p, fname, FLG_COEF_PP8|(COREG_USER2<<8)|i); 
    if (findstrflag("RFIR",p->config,fname,-1)<0) sprintf(fname,"botrfir_4k");
    if (strcmp(fname,"none")==0 && ratio==2) cflg |= 0x2; 		/* flag for RAW output */
    else pic_loadfile (p, fname,  FLG_COEF|(COREG_USER3<<8)|i); 
    if (findintflag("CTAG",p->config)>0) cflg |= 0x4;
         if (pktlen==0x2000) cflg |= 0x300;
    else if (pktlen==0x1000) cflg |= 0x200;
    else if (pktlen==0x0800) cflg |= 0x100;
    else if (pktlen==0x0400) cflg |= 0x000;
    else print("Illegal packet length=%d for FTTM=0 mode\n",pktlen);
    pic_setkeyl (p,-i,KEY_CORE+COREG_SYS,sys);
    pic_setkeyl (p,-i,KEY_CORE+COREG_DEC,m-1);				/* nout-1 */
    pic_setkeyl (p,-i,KEY_CORE+COREG_GAIN,fftscale((gain/6),k));	/* FFT scale */
    pic_setkeyl (p,-i,KEY_CORE+COREG_RATIO,R2L(((bits>0)?0.5:1.0)/ratio));
    pic_setkeyl (p,-i,KEY_CORE+COREG_FRAME,pow2sizem1(n));
    pic_setkeyl (p,-i,KEY_CORE+COREG_FLAG,cflg);			/* PKT|REAL output */
    pic_setkeyl (p,-i,KEY_CORE+COREG_FREQ,R2L(-fcny-1.0/n));		/* odd bin shift */
    pic_setkeyl (p,-i,KEY_CORE+COREG_FREQ,R2L(-fcny));			/* even bin shift */
    pic_setkeyl (p,-i,KEY_CORE+COREG_SYS,sys|CORE_ENA);
    vprint("FTTM=0 dec=%d nfft=%d bits=%d k=%d j=%x d1=%d fcny=%f rat=%f cflg=%x\n",dec,n,bits,k,j,d1,fcny,ratio,cflg);
    return dmac; /* no tuners, all done */
  }

  /* init front end tuners */
  p->chan = -1;
  if ( pic_tuner (p, port, tbits, rate, fcny, dec, gain, flags) < 0) return (-1);
  p->chan = 0;
  if ( pic_tuner (p, port, tbits, rate, fcny, dec, gain, FLG_FTUN) < 0) return (-1);

  if (p->fttm<2) return dmac; /* no back end, all done */
    
  /* init back end tuners */
  g2 = findintflagdef ("FTTG2",p->config,0);
  p->cpc = cpc2;
  p->chan = -1;
  if ( pic_tuner (p, beport, bits, rate, 0.0, d2, g2, FLG_BACKEND|flags) < 0) return (-1);
  p->chan = 0;
  if ( pic_tuner (p, beport, bits, rate, 0.0, d2, g2, FLG_BACKEND|FLG_FTUN) < 0) return (-1);
  p->cpc = cpc1;

  return (dmac);
}


int_4 pic_timer (PICSTRUCT *p, int_4 port, int_4 rate) 
{
  int_4 div, actual,rmin,rmax, clock=0,mode=0,iport,mtype,mrev,lbcf;

       if (rate==0);
  else if (port==FLG_ICLK) mode=1;
  else if (port==FLG_PCLK) mode=2;
  else if (p->gmcs==GMC_CLKP) mode=2;
  else if (p->gmcs==GMC_CLKQ) mode=2;
  else if (p->gmcs==GMC_CLKD) mode=3;
  else if (p->flags&FLG_INTCLK) mode=1;
  iport = (port==2)? 1 : 0;

  if (p->gmcs==GMC_CLKC && rate!=p->ext_clk) 
    print("Warn: MUXCLK=C rate Desired=%d Actual=%d Hz\n",rate,p->ext_clk); 

  if (mode==0) return 0;

  lbcf = findintflag("LBCF",p->config);
  if (lbcf>1) rate /= lbcf;

  mtype = p->mtype[iport];
  mrev  = p->mrev[iport];

  if (mode==2) { 
    if (p->isY) {
      rmin=p->k7? 1000000/64:200000/64;
      rmax=p->k7? 200000000:150000000;
      if (p->gmcs==GMC_CLKP && rate<rmin*64) p->gmcs = GMC_CLKQ;
      if (getIOCsig(p,IS_INTERNAL|port)>0); /* use mainboard clock regardless of module */
      else if (getIOCsig(p,IS_OBMC|port)>0) return 0; /* uses clock on module */
    } else {
      rmin=100000; 
      rmax=120000000;
    }
  }
  else if (mode==3) { 
    clock=p->ext_clk; 
    if (p->isX||p->isY) rmin=clock/64; 
    else rmin=clock/16; 
    rmax=clock;
  }
  else { 
    clock=p->ioc_clk; 
    rmin=clock/4096;
    rmax=clock;
  }

  if (rate<rmin || rate>rmax) { 
    print("Err: On-Board Clock Freq=%d out of range [%d-%dHz]\n", rate, rmin, rmax); 
    return (-1); 
  }

  if (mode==2) { 
    actual = pic_setup_ioc_clk (p, 1, rate);
    vprint ("IOM_RATEp = %08x ACTUAL = %d\n", rate, actual);
  } else {
    div = (int)( (double)clock / (double)rate + .5);
    div = max( 0, div-1);
    actual = (int)( (double)clock/(double)(div+1) );
    vprint ("IOM RATE=%d DIV=%d ACTUAL=%d\n", rate, div, actual);
    pic_acmd (p,ADON_WRIOC,IOC_DIV,div,2); 
  }
  if (actual!=rate) print("Warn: MUXCLK rate Desired=%d Actual=%d\n",rate,actual);
  p->int_clk=actual;
  return (actual);
}

int_4 core_defconfig (HALO *plan) {
  int_4 stat;
  stat = core_config(plan,0,0x0);       /* func */
  stat = core_config(plan,1,0x0808);	/* fmts */
  stat = core_config(plan,1,0x0000);    /* fmts */
/*  stat = core_config(plan,2,1);       /* flow */
/*  stat = core_config(plan,3,6);       /* impl */
  stat = core_config(plan,4,0x0);       /* flgs */
  stat = core_config(plan,5,0x0);       /* scid */
  stat = core_init(plan);
  return stat;
}

int_4 core_startup (PICSTRUCT *p, int_4 port, double time) {
  HALO *plan = p->plan[port];
  double dval;
  if (plan==NULL) return 0;
  /* test if core has a startup key */
  if (core_getkey(plan,"D:STARTUP?",&dval,8)>=0) {
    core_setkey(plan,"D:STARTUP",&time,8);
  }
  return 0;
}

int_4 core_setup (PICSTRUCT *p, int_4 port, char *ccfg, int_4 sig) {
  HALO *plan; char cname[20]; int stat;
  char *csig = (char*)(&sig); char *s;
  int ln = findstrflag("NAME",ccfg,cname,0);
  if (ln<0) { printf("Missing core NAME parameter in CORE=%s\n",ccfg); return -1; }
  s = strchr(cname,';'); if (s!=NULL) ln = (int)(s-cname);
  if (strncmp(cname,csig,imin(ln,4))!=0) printf("Core NAME=%s does not match Core SIG=%.4s\n",cname,csig); 
  plan = core_alloc(cname,"ICEP",p);
  vprint("Load Core=%s on port=%d with cname=%s csig=%.4s plan=%p\n", ccfg,port,cname,csig,plan);
  core_defconfig(plan);
  core_setkeytable(plan,ccfg);
  stat = core_open(plan);
  strcpy(p->cname,cname);
  p->plan[0] = plan;
  return stat;
}

int_4 setup_fftcore (PICSTRUCT *p, int_4 port, char *ccfg, int_4 csig, int_4 dmac, int_4 bits, int_4 rate, int_4 dec, int_4 gain) {
  int_4 i,j,k,n,cx,over,flgs,p2,cadr,ntop,nwin;
  real_4 *fir,thresh; char fname[128],mfppcfg[128];
    cx= (p->mbits<0);
    n = findintflag("NFFT",ccfg);
    i = findintflagdef("NAVG",ccfg,dec);
    j = findintflagdef("NMAX",ccfg,dec/i);
    k = pow2sizem1(n)>>16; /* pow2 */
    flgs = (bits>0)? 0x0C:0x00;
    if (p->mbits>0) flgs |= 0x40;
    flgs = findmaskflagdef("FLAGS",ccfg,"INV,AC,MAG,MAX,EXP,PPF,REAL,PKP",flgs);
    over = (rate>1000000000)? 8 : (rate>500000000)? 4 : (rate>250000000)? 2 : 1;
    over = findintflagdef("OVER",ccfg,over);
    pic_setkeyl (p,dmac,KEY_CORE+COREG_DEC,pow2sizem1(i)); /* navg */
    pic_setkeyl (p,dmac,KEY_CORE+COREG_FRAME,pow2sizem1(n)); /* nfft */
    pic_setkeyl (p,dmac,KEY_CORE+COREG_USER1,pow2sizem1(cx?n:n/2)); /* nout */
    pic_setkeyl (p,dmac,KEY_CORE+COREG_USER2,pow2sizem1(j/over)); /* nmax */
    pic_setkeyl (p,dmac,KEY_CORE+COREG_USER3,pow2sizem1(1)); /* nexp */
    i = fftscale( (gain/6)+(cx?0:1), k);
    pic_setkeyl (p,dmac,KEY_CORE+COREG_GAIN,i);
    if (over>1) {
      p2 = k-3; /* assume 8 complex 16 bit samples per 256b DMA line */
      if (abs(bits)==8) p2--;	/* complex 8 bit */
      if (bits>0) p2--;	/* real data */
      p2++;	/* fudge factor for now */
      flgs |= (p2<<8);
      if (over==1) flgs |= ((p2+0)<<12);
      if (over==2) flgs |= ((p2+1)<<12);
      if (over==4) flgs |= ((p2+2)<<12);
      if (over==8) flgs |= ((p2+3)<<12);
      vprint("FFT using overlap=-%d to match FPGA load capabilites. Take/Skip flags=%08x\n",over,flgs);
    }
    pic_setkeyl (p,dmac,KEY_CORE+COREG_FLAG,flgs);
    if (flgs&0x80) {		/* Peak Finder Enabled */
      nwin = findintflagdef("NWIN",ccfg,8);
      ntop = findintflagdef("NTOP",ccfg,128);
      thresh = (real_4)finddblflagdef("THRESH",ccfg,2.0);
      sprintf(mfppcfg,"(NAME=MFPP,L:FRAME=%d,L:NTOP=%d,L:WIDTH=%d,F:THRESH=%f,S:FLAGS=TAG)",n,ntop-1,nwin,thresh);
      p->coreAddrX = 0x10000;	/* address offset of MFPP core parameters within FFT core */
      core_setup(p,port,mfppcfg,csig);
      p->coreAddrX = 0x0;
    }
    if (findstrflag("FPPF",ccfg,fname,-1)>0) {
      pic_setkeyl (p,dmac,KEY_CORE+COREG_FLAG,flgs|0x20);
      pic_loadfile (p, fname, (cx?FLG_COEF_PP4F:FLG_COEF_PP4)|(COREG_COEF<<8)|port); 
    }
    if (findstrflag("WIND",ccfg,fname,-1)>0) {
      fir = malloc(n*4);
      firwind_generate(fname,1.0,fir,n); /* use window to rotate CX FFT to +- nyquist */
      for (i=0,j=1; i<n; i++,j=-j) pic_setkeyl (p,dmac,KEY_CORE+COREG_COEF,R2LC18(j*fir[i])|i);
      free(fir);
    }
    else { /* window==NONE but use window to rotate CX FFT to +- nyquist */
      for (i=0,j=1; i<n; i++,j=-j) pic_setkeyl (p,dmac,KEY_CORE+COREG_COEF,R2LC20(j)|i);
    }
  return 0;
}

int_4 keyIsDouble (int_4 key) {
  switch (key) {
    case KEY_FREQ: case KEY_DFREQ: case KEY_NFREQ: case KEY_MFREQ: case KEY_NTPOFF: case KEY_RATIO: case KEY_NRATIO:
    case KEY_RFFREQ: case KEY_RFBW: case KEY_TCOFF: case KEY_RFAIS: case KEY_ATCCALIB: case KEY_TEMP: case KEY_TCXO:
    case KEY_AGCTC: case KEY_AGCCF: case KEY_GPS: case KEY_DRATE: case KEY_MTOFF: case KEY_VCTL: case KEY_STIME: case KEY_FBWF:
      return 1;
    default:
      return 0;
  }
}

int_4 keyIsAscii (int_4 key) {
  switch (key) {
    case KEY_CONFIG: case KEY_NAME: case KEY_CNAME: case KEY_TABLE: case KEY_QSTAT: return 1;
    default: return 0;
  }
}

int_4 keyIsLong (int_4 key) {
  if (keyIsDouble(key)) return 0;
  if (keyIsAscii(key)) return 0;
  return 1;
}

int_1 pic_getkeytype (PICSTRUCT *p, int_4 key) {
  if (keyIsDouble(key)) return 'D';
  else if (keyIsAscii(key)) return 'A';
  else return 'L';
}

int_4 pic_getkeysize (PICSTRUCT *p, int_4 key) {
  switch (key) {
    case KEY_PKTHDR: return 16;
    case KEY_CONFIG: return 10;
    case KEY_GPS:    return 6;
    case KEY_SYSMON: return 4;
    case KEY_TCOFF:  return 3;
    case KEY_CUID:   return 2;
    case KEY_RTCLK:  return 2;
    case KEY_NAME:   return 2;
    case KEY_FPGADSG:return 2;
    default:         return 1;
  }
}

int_4 parsetable (PICSTRUCT *p, int_4 dmac, char *tval, int_4 len, int_4 dir) 
{
  int lval,lkey,core=0,i,j=0,o=0,n=0,ls,lb,lk,lv,skip; float fval; double dval;
  char line[80],key[40],value[40],type,*s;
  char out[1024];

  i=0; if (tval[0]=='{' || tval[0]=='(') i++;
  ls=strnlen(tval,len); if (tval[ls-1]=='}' || tval[ls-1]==')') ls--;
  out[o++]='{';

  for (;i<ls;i+=(lb+1)) {
    skip=0;
    while (i<ls && tval[i]<=' ') i++;	/* trim leading white space */
    s = index(tval+i,','); j = (s==NULL)? ls : (s-tval);
    lb=j-i; strncpy(line,tval+i,lb); line[lb]=0;
    s = index(line,'='); j = (s==NULL)? lb : (s-line);
    lk = max(0,j); lv = max(0,lb-lk-1);
    if (lk>0) strncpy(key,line,lk); key[lk]=0;
    if (lv>0) strncpy(value,line+lk+1,lv); value[lv]=0;
    if (lk==4 && strcmp(key,"CORE")==0) { core=core_s2d(value); p->chan=core; n=core-1; skip=1; }
    if (lk==4 && strcmp(key,"NAME")==0 && core>0 && lv>0) { p->plan[n] = core_alloc(value,"ICEP",p); core_defconfig(p->plan[n]); skip=1; }
    if (lk==4 && strcmp(key,"OPEN")==0 && core>0)         { core_open(p->plan[n]); skip=1; }
    if (skip) { strncpy(out+o,line,lb); o+=lb; out[o++]=','; continue; }
    if (core>0) {
      type = key[0]; 
      lkey = -2;
    } else if (key[1]==':') {
      type = key[0]; 
      lkey = pic_name2key(p,key+2);
    } else {
      type = ' ';
      lkey = pic_name2key(p,key);
    }
    if (lkey==-1) { printf("Unknown table key name=%s\n",key); continue; }
    if (type==' ') type = pic_getkeytype (p,lkey);
    if (core>0 && n>MAX_MCS) { printf("Core=%d out of range\n",core); continue; }
    if (core>0 && p->plan[n]==NULL) { printf("Core=%d not instatiated\n",core); continue; }
    if (dir>=0 && lv>0) {
      if (core>0) {
	     if (type=='S') {                       core_setkey(p->plan[n],key,value,lv); }
	else if (type=='D') { dval=core_s2d(value); core_setkey(p->plan[n],key,&dval,8); }
	else if (type=='F') { fval=core_s2d(value); core_setkey(p->plan[n],key,&fval,4); }
	else                { lval=core_s2d(value); core_setkey(p->plan[n],key,&lval,4); }
      } else {
	     if (type=='S') {                       pic_setkey (p,dmac,lkey,value,lv); }
	else if (type=='D') { dval=core_s2d(value); pic_setkeyd(p,dmac,lkey,dval); }
	else                { lval=core_s2d(value); pic_setkeyl(p,dmac,lkey,lval); }
      }
    }
    if (dir<=0) {
      strncpy(out+o,key,lk); o+=lk; out[o++]='=';
			/* if the get fails dval & lval will be left alone */
      if (core>0) {
	strcat(key,"?");
	     if (type=='S') { core_getkey(p->plan[n],key,value,40); }
	else if (type=='D') { if (core_getkey(p->plan[n],key,&dval,8)>=0) sprintf(value,"%f",dval); }
	else if (type=='F') { if (core_getkey(p->plan[n],key,&fval,4)>=0) sprintf(value,"%f",fval); }
	else                { if (core_getkey(p->plan[n],key,&lval,4)>=0) sprintf(value,"%d",lval); }
      } else {
             if (type=='S') { pic_getkey(p,dmac,lkey,value,40); }
        else if (type=='D') { pic_getkey(p,dmac,lkey,&dval,8); sprintf(value,"%f",dval); }
        else                { pic_getkey(p,dmac,lkey,&lval,4); sprintf(value,"%d",lval); }
      }
      lv = strlen(value);
      strncpy(out+o,value,lv); o+=lv; out[o++]=','; 
    }
  }
  out[o-1]='}'; out[o]=0;
  vprint("SetTable: dmac=%d dir=%d len=%d str=%s\n",dmac,dir,o,out);
  if (core>0) p->chan=0;
  if (dir<=0) { if (o<len) len=o; strncpy(tval,out,len); tval[len]=0; }
  return len;
}

int_4 pic_setkeyd (PICSTRUCT *p, int_4 dmac, int_4 key, double dval) 
{
  if (!keyIsDouble(key)) print("ERR: pic_setkeyd used on key=%d not of type double\n",key);
  return pic_setkey(p,dmac,key,(void *)(&dval), 8);
}

int_4 pic_setkeyl (PICSTRUCT *p, int_4 dmac, int_4 key, int_4 lval) 
{
  if (!keyIsLong(key)) print("ERR: pic_setkeyl used on key=%d not of type long\n",key);
  return pic_setkey(p,dmac,key,(void *)(&lval), 4);
}

int_4 pic_setkeyt (PICSTRUCT *p, int_4 dmac, char *tval) 
{
  return parsetable(p,dmac,tval,strlen(tval),0);
}

int_4 pic_setkey (PICSTRUCT *p, int_4 dmac, int_4 key, void *vval, int_4 len) 
{
  int_4 stat=1,mtype,mrev,maddr,port,ovsr,side,pside,bits,i,j,k,m,addr,mcfg,*lval;
  double fcny, rdec, *dval, dtmps[4];
  int_4 n = max(0,dmac-1);
  DMASTRUCT *dma = p->dma+n;
  APPSTRUCT *app=0;
  DMAMAP *map;
  int_8 ulval;
  CHECK_MAGIC(p);

  lval = (int_4 *)vval;
  dval = (double *)vval;

  if (dmac>0) port = dma->port; 
  else if (dmac<0) port = -dmac;
  else port = p->pindex;
  if (port<=0) port=1;
  pside = (port&1)? 1:2;
  side  = (p->inp>0)? p->inp : pside;
  mtype = p->mtype[side-1];
  mrev  = p->mrev[side-1];

  if (dma->appp != 0) {
    m = (dma->appp-APP_STK)/APP_SEP;
    app = p->app+m;
  } 
  else m=-1;

  if ((p->verbose&0x2)==0);
  else if (keyIsDouble(key)) print("SetKey %s on dmac=%d port=%d to %f\n",pic_key2name(p,key),dmac,port,*dval);
  else if (key!=KEY_INBYTE)  print("SetKey %s on dmac=%d port=%d to 0x%08x\n",pic_key2name(p,key),dmac,port,*lval);

  pic_lock(p,LOCK_ALLOC);
  
  if (key>=KEY_MCORE) {
    m = (port-1)/32+1; i = ((port-1)%32)>>1; if (m==1 && p->pmi!=3) m=p->pmi;
    if (m==99) j = (key-KEY_MCORE) + ((port&1)?PPC_CORA_CTL:PPC_CORB_CTL) + i*MCORE_CHN;
    else j = (key-KEY_MCORE) + ((port&1)?PPC_TUNA_CTL:PPC_TUNB_CTL) + i*MCORE_CHN;
    v2print("Key MCORE=%d PM=%d addr=%08x data=%08x\n",port,m,j,*lval);
    pic_wpb (p,m,j,*lval);
  }
  else if (key>=KEY_CORE) {
    maddr = (key-KEY_CORE);
    if (dmac==1 || dmac==2 || port>=50 || p->pmi<0 || p->ptype==IOPT_MODULE) {
      v2print("Key CORE=%d IOM=%d addr=%08x data=%08x len=%d\n",port,side,maddr,*lval,len);
      if (isIOMx(p,side)) i = pic_jpmemwr(p,side,maddr|IOM_CORE_OFFSET,*lval,4);
      else                i = pic_setkey_module(p,side,maddr|IOM_CORE_WROFFSET,*lval);
    } else if (dmac=-8 && port==8) {
      maddr |= PPC_TRACER;
      pic_wpb (p,0,maddr,*lval);
      v2print("Key CORE=%d CDBG addr=%08x data=%08x len=%d\n",port,maddr,*lval,len);
    } else if (dmac=-9 && port==9) {
      maddr |= PM_TPORT_DBG;
      wrioc (p,maddr,*lval,1);
      v2print("Key CORE=%d CDBG addr=%08x data=%08x len=%d\n",port,maddr,*lval,len);
    } else {
      m = (port-1)/10; i = port-m*10; if (m==0) m=p->pmi; if (m==4) m=0; /* node & core */
      j = (key-KEY_CORE) + core_addr(i);
      if (p->mcs==-99) j |= MCORE_SYS; if (p->mcs!=0) j |= ( (p->chan<=0)? MCORE_ALL|MCORE_TRIM : (p->chan-1)*MCORE_CHN );
      j |= p->coreAddrX;
      v2print("Key CORE=%d PM=%d addr=%08x data=%08x len=%d\n",port,m,j,*lval,len);
      if (len<=4) pic_wpb (p,m,j,*lval);
      else if (p->devno==-3) for (len/=4,i=0; i<len; i++) pic_wpb (p,m,j,lval[i]);
      else for (len/=4,i=0; i<len; i+=k) {
	k = min(64,len-i);
	pic_msg (p,m,PKTF_BUS_WR,j,lval+i,k*4,0,-1);
	pic_relock(p,0);
      }
    }
  }
  else switch (key) {

  case KEY_TABLE:
    stat = pic_setkeyt(p,dmac,(char*)vval);
    break;

  case KEY_RATE:
    if (dma->type==IOPT_MODULE && !p->isY) {
      if (p->isX) i=70000000; else i=38000000;
      if (dma->slave!=0) i*=2;
      if (dma->slave!=0 && p->dma[dma->slave-1].slave!=0) i*=2;
      if (dma->bits<0) bits = -2*dma->bits; else bits = dma->bits;
      if ((*lval)/8*bits > i) {
	print("Rate %d Hz is out of range for this setup\n",*lval);
	stat = -1; break;
      }
    }
    dma->rate = *lval;
    stat = pic_timer (p, port, dma->rate);
    if (dma->type==IOPT_TUNER || dma->type==IOPT_TBANK) {
      stat = pic_tuner (p, port, dma->bits, dma->rate, 0.0, dma->dec, dma->gain, FLG_RTUN);
    }
    if (dma->type==IOPT_CORE) {
      dma->rate = *lval;
      pic_setkeyl (p,dmac,KEY_CORE+COREG_RATE,dma->rate);
    }
    if (stat>=0) pic_stkupd (p, &dma->rate, 1, 1);
    break;

  case KEY_NYBW:
    if (dma->type==IOPT_TUNER ||  dma->type==IOPT_TBANK) {
      dma->dec = pic_tuner_nybw2dec (p, port, *lval, dma->rate, 0, &fcny);
      if (fcny!=1.0) stat = pic_tuner_ratio (p, port, fcny, 0);
      stat = pic_tuner (p, port, dma->bits, dma->rate, 0.0, dma->dec, dma->gain, FLG_DTUN);
    }
    break;

  case KEY_FBWF:
    p->fbwf = *dval;
    break;

  case KEY_FREQ:
    fcny = *((double *)vval); 
    if (dma->type==IOPT_TBANK && dma->dir>0) { 
      set_awg_freq(p,port,p->chan,fcny);
    }
    else if (dma->type==IOPT_TBANK && p->fttm==0) { i=port+2;
      pic_setkeyl (p,-i,KEY_CORE+COREG_FREQ,R2L(-fcny-1.0/p->d1));
      pic_setkeyl (p,-i,KEY_CORE+COREG_FREQ,R2L(-fcny));
    }
    else if (dma->type==IOPT_TUNER || dma->type==IOPT_TBANK) {
      stat = pic_tuner (p, port, dma->bits, dma->rate, fcny, dma->dec, dma->gain, FLG_FTUN);
    }
    else if (dma->type==IOPT_CORE) pic_setkeyl (p,dmac,KEY_CORE+COREG_FREQ,R2L(fcny));
    else if (dma->type==IOPT_MCORE) pic_setkeyl (p,dmac,KEY_MCORE+COREG_FREQ,R2L(fcny));
    if (stat>=0) pic_stkupd (p, &dma->fcny, 1, 1);
    break;

  case KEY_DFREQ:
    p->dfreq = *((double *)vval);
    if (p->tflags&FLG_PRESAMP) p->dfreq /= p->res_ratio;
    break;

  case KEY_DEC:
    /* JEFF need to check if running */
    if (dma->type==IOPT_TBANK && p->mcs==0) {
      if (*lval == dma->dec) break;
      print("ERR: Cannot change TunerBank decimation on the fly\n");
      break;
    }
    dma->dec = *lval;
    if (dma->type==IOPT_TUNER ||  dma->type==IOPT_TBANK) {
      dma->dec = pic_tuner_dec (p, *lval, port, 0);
      i = (dma->type==IOPT_TUNER && p->gchip==GCFPGA && p->nps<=1 && (dma->dec==1 || *lval==1))? FLG_MTUN : FLG_DTUN;
      stat = pic_tuner (p, port, dma->bits, dma->rate, 0.0, dma->dec, dma->gain, i);
    }
    if (dma->type==IOPT_MODULE && p->frame!=0) pic_setkeyl(p,dmac,KEY_FRAMEDEC,*lval);
    if (dma->type==IOPT_CORE) pic_setkeyl (p,dmac,KEY_CORE+COREG_DEC,*lval-1);
    if (dma->type==IOPT_MCORE) pic_setkeyl (p,dmac,KEY_MCORE+COREG_DEC,*lval-1);
    if (stat>=0) pic_stkupd (p, &dma->dec, 1, 1);
    break;

  case KEY_NFGAIN:
    p->dec5 = *lval;
    set_awg_gain(p,port,99,p->dec5);
    break;

  case KEY_MUTE:
    if (dma->type==IOPT_TBANK && dma->dir>0) set_awg_mute(p,port,p->chan,*lval);
    break;


  case KEY_MGAIN:
    if (dma->type!=IOPT_MODULE) { 
      dma = p->dma+(side-1); /* module input to this port */
      dma->type=IOPT_MODULE;
    }
    if (mtype==IOMT_D2AWG && mrev==3) { d2awgm3_setkeyl (p,side,KEY_MGAIN,*lval); break; }
  case KEY_GAIN:
    dma->gain = *lval;
    if (dma->type==IOPT_TBANK && dma->dir>0) { 
      set_awg_gain(p,port,p->chan,dma->gain);
    }
    else if (dma->type==IOPT_TBANK && p->fttm==0) { i=port+2;
      j = fftscale((dma->gain/6),pow2ge(dma->dec));
      pic_setkeyl (p,-i,KEY_CORE+COREG_GAIN,j);
    }
    else if (dma->type==IOPT_CORE && p->mcs!=0 && dma->dir>0) { 
      set_awg_gain(p,port,p->chan,dma->gain);
    }
    else if (dma->type==IOPT_TUNER || dma->type==IOPT_TBANK) {
      stat = pic_tuner (p, port, dma->bits, dma->rate, 0.0, dma->dec, dma->gain, FLG_GTUN);
    }
    else if (p->tflags&FLG_FIRONLY) {
      i = (int)(0x1000 * pow(2.0,dma->gain/6.0) / p->cfir_gain); 
      if (i>=0x8000) { print("Filter output scaling max=8 exceeded\n"); i=0x7FFF; }
      m = (port-1)/10; if (m==0) m=p->pmi; 
      j = ((pside==2)?PPC_CORB_CTL:PPC_CORA_CTL) + (9<<2);
      pic_wpb (p,m,j,i);
    }
    else if (dma->type==IOPT_CORE) pic_setkeyl (p,dmac,KEY_CORE+COREG_GAIN,*lval);
    else if (dma->type==IOPT_MCORE) pic_setkeyl (p,dmac,KEY_MCORE+COREG_GAIN,*lval);
    else if (mtype==IOMT_CXD) stat = pic_setup_cxd (p,side,dma->dir,dma->rate,dma->gain);
    else if (mtype==IOMT_A2D && mrev==9) pic_setup_a2dr9_gain(p,side,dma->gain);
    else if (mtype==IOMT_A2D && mrev==13) pic_setup_a2dr13_gain(p,side,dma->gain);
    else if (mtype==IOMT_A2D && mrev==14) a2dm14_setkey(p,side,KEY_GAIN,&dma->gain);
    else if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) a2dm1x_set_gain (p,side,dma->gain);
    else if (mtype==IOMT_A2D && mrev==20) a2dm20_setkeyl (p,side,KEY_RFGAIN,dma->gain);
    else if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyl (p,side,KEY_RFGAIN,dma->gain);
    if (stat>=0) pic_stkupd (p, &dma->gain, 1, 1);
    pic_update_vctx(p,dmac,key,i2d(dma->gain));
    break;

  case KEY_ENABLE:
    if (p->mcs!=0) {
      k = p->chan; p->chan=0; j = pic_getkeyl(p,dmac,KEY_CORE+COREG_SYS);
      if (*lval>0) j |= CORE_ENA; else j &= ~CORE_ENA;
      p->chan = k; pic_setkeyl(p,dmac,KEY_CORE+COREG_SYS,j);
    }
    break;

  case KEY_MBITS:
    p->mbits = *lval;
    break;

  case KEY_EMT:
    addr = IOC_EMT | ((side&1)? IOC_A:IOC_B);
    pic_acmd (p,ADON_WRIOC,addr,*lval,2);
    break;

  case KEY_TODO:
    dma->todo = *lval;
    pic_stkupd (p, &dma->todo, 1, 1);
    break;

  case KEY_INBYTE_W64:
    dma->hind = dma->haddr + *lval;
    WR(((REG_AHIRX+n)<<2)&0xFFF,dma->hind);
    break;

  case KEY_INBYTE:
    dma->hind = dma->haddr + L2HW(*lval);
    WR(((REG_AHIRX+n)<<2)&0xFFF,dma->hind);
    break;

  case KEY_VCTL:
    p->vctlFill = *dval;
    break;

  case KEY_DMACHK:
    map = (DMAMAP*)vval;
    pic_stkupd (p, &dma->haddr, 48, -1);
    p->dma_vaddr[n] = (int_4*)(map->vaddr+map->offset);
    p->dma_vindx[n] = HW2L(dma->hind-dma->haddr);
    p->dma_vcycx[n] = dma->hcycle;
    break;

  case KEY_DMAOFF:
    p->dma_vindx[n] += *lval;
    break;

  case KEY_STATE:
    p->state = *lval;
    break;

  case KEY_RFFREQ:
         if (mtype==IOMT_RFXD) rfxd_set_freq (p,side,*dval);
    else if (mtype==IOMT_LB2D && mrev<3) lb2d_set_freq (p,side,*dval);
    else if (mtype==IOMT_LB2D && mrev==3) lb2dm3_setkeyd (p,side,KEY_RFFREQ,*dval);
    else if (mtype==IOMT_D2RF) d2rf_set_freq (p,side,*dval);
    else if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) a2dm1x_set_freq (p,side,*dval);
    else if (mtype==IOMT_A2D && mrev==20) a2dm20_setkeyd (p,side,KEY_RFFREQ,*dval);
    else if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyd(p,side,KEY_RFFREQ,*dval);
    else print("Set RF freq to %f\n",*dval);
    pic_update_vctx(p,dmac,KEY_RFFREQ,*dval);
    break;

  case KEY_RFGAIN:
         if (mtype==IOMT_RFXD) rfxd_set_gain (p,side,*lval);
    else if (mtype==IOMT_LB2D && mrev<3) lb2d_set_gain (p,side,*lval);
    else if (mtype==IOMT_LB2D && mrev==3) lb2dm3_setkeyl (p,side,KEY_RFGAIN,*lval);
    else if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyl (p,side,KEY_RFGAIN,*lval);
    else if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) a2dm1x_set_gain(p,side,*lval);
    else if (mtype==IOMT_A2D && mrev==20) a2dm20_setkeyl(p,side,KEY_RFGAIN,*lval);
    else print("Set RF gain to %d\n",*lval);
    pic_update_vctx(p,dmac,KEY_RFGAIN,i2d(*lval));
    break;

  case KEY_RFATTN:
         if (mtype==IOMT_LB2D && mrev<3) lb2d_set_attn (p,side,*lval);
    else if (mtype==IOMT_D2AWG && mrev<3) d2awg_set_attn (p,side,*lval);
    else if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyl (p,side,KEY_RFATTN,*lval);
    else if (mtype==IOMT_D2RF) d2rf_set_attn(p,side,*lval);
    else if (mtype==IOMT_A2D && mrev==20) a2dm20_setkeyl(p,side,KEY_RFATTN,*lval);
    else print("Set RF attenuation to %d\n",*lval);
    break;

  case KEY_RFBW:
         if (mtype==IOMT_RFXD) rfxd_set_bw (p,side,(int_4)((*dval)*1000000));
    else if (mtype==IOMT_LB2D && mrev<3) lb2d_set_bw (p,side,(int_4)((*dval)*1000000));
    else if (mtype==IOMT_LB2D && mrev==3) lb2dm3_setkeyd (p,side,KEY_RFBW,*dval);
    else if (mtype==IOMT_D2RF) d2rf_set_bw (p,side,(int_4)((*dval)*1000000));
    else print("Set RF bandwidth to %d\n",*lval);
    break;

  case KEY_MTOFF:
    if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyd (p,side,KEY_MTOFF,*dval);
    break;

  case KEY_RFOPTS:
         if (mtype==IOMT_RFXD) rfxd_set_opts (p,side,*lval);
    else if (mtype==IOMT_LB2D && mrev<3) lb2d_set_opts (p,side,*lval);
    else if (mtype==IOMT_LB2D && mrev==3) lb2dm3_setkeyl (p,side,KEY_RFOPTS,*lval);
    else if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_setkeyl (p,side,KEY_RFOPTS,*lval);
    else if (mtype==IOMT_D2RF) d2rf_set_opts (p,side,*lval);
    else print("Set RF options to %08x\n",*lval);
    break;

  case KEY_RFCAIS:
    dval = (double*)dtmps;
    /* calculate parameters fall through to sets */

  case KEY_RFAIS:
         if (mtype==IOMT_RFXD) ; /* rfxd_set_aic (p,side,dval); */
    else if (mtype==IOMT_LB2D && mrev<3) lb2d_set_ais (p,side,3,dval); 
    else print("Set RF AIC to (%f,%f) (%f,%f)\n",dval[0],dval[1],dval[2],dval[3]);
    break;

  case KEY_TCXO:
    if (mtype==IOMT_D2AWG) d2awg_set_tcxo (p,side,*dval);
    break;

  case KEY_AGCTC:
    if (mtype==IOMT_RFXD) rfxd_set_agctc (p,side,*dval);
    break;

  case KEY_AGCCF:
    if (mtype==IOMT_RFXD) rfxd_set_agccf (p,side,*dval);
    break;

  case KEY_CHAN:
    p->chan = *lval;
    if (dma->type==IOPT_TBANK && p->chan>0 && p->mcs==0) {
      m = (port%10==3)? p->chan : pside + (p->chan-1)*2;
      if (p->pmi==2 || port>20) m += 32;
      if (p->fttm==1) dma->fcny = -pic_getkeyl(p,-m,KEY_MCORE+COREG_FREQ);
      if (p->tflags&FLG_ITDEC) dma->dec = pic_getkeyl(p,-m,KEY_MCORE+COREG_DEC)>>16;
    }
    if (dma->type==IOPT_TBANK && p->chan>0 && p->mcs!=0) {
      dma->fcny = -pic_getkeyl(p,dmac,KEY_CORE+COREG_FREQ);
      if (p->tflags&FLG_ITDEC) dma->dec = pic_getkeyl(p,dmac,KEY_CORE+COREG_DEC)>>16;
    }
    break;

  case KEY_CHNS:
    p->nchan = *lval;
    i = p->pindex % 10;
    j = p->pindex / 10;
    if (dmac==1 || dmac==2 || j==0 || i==1 || i==2) p->nchan *= 2;  /* for one side */
    break;

  case KEY_PKTLEN:
    p->pktlen = *lval;
    break;

  case KEY_FRAME:
    p->frame = *lval;
    break;
  case KEY_FRAMEDEC:
    if (dma->type==IOPT_CORE && findintflag("SKIPONCORE",p->config)>0) {
      pic_setkeyl (p,dmac,KEY_CORE+COREG_FRAME,p->frame-1);
      pic_setkeyl (p,dmac,KEY_CORE+COREG_DEC,max(0,*lval-1));
    }
    else {
      if ((dma->mcfg&IOM_DUAL) && (dma->mcfg&IOM_VHS) && !p->isY) port = 3;
      dma->frame = setIOCframe(p,port,dma->bits,p->frame,*lval,0x3);
      pic_stkupd (p, &dma->frame, 1, 1);
    }
    break;

  case KEY_CHAIN:
    if (*lval > 0 && *lval <= 1024) {	/* by index */
      pic_dmachain(p,dmac,0,0,DMA_ONESHOT,*lval);
    } else {				/* by address */
      dma->chain = *lval;
      pic_stkupd (p, &dma->chain, 1, 1);
    }
    break;

  case KEY_RATIO:
    fcny = *((double *)vval); 
    if (dmac==0) {   /* preload for PRESAMP mode */
      p->res_ratio = fcny;
    }
    else if (p->tflags&FLG_PRESAMP) {
      stat = pic_tuner_ratio (p, port, fcny, 0);
    }
    else if ((p->tflags&FLG_RESAMP)==0) {
      if (fcny!=1) print("WARN: Non-unity resample ratio=%f specified without RESAMP flag\n",fcny);
    }
    else if (dma->type==IOPT_TUNER || (dma->type==IOPT_TBANK && p->fttm>0)) {
      stat = pic_tuner_ratio (p, port, fcny, 0);
    }
    break;

  case KEY_OVSR:
    ovsr = *lval;
    if (dmac<=3) side=dmac;
    pic_tuner_ovsr (p,ovsr,side,0);  
    break;

  case KEY_DELAY:
    pic_msg (p,-1,PKTF_MODIFY,0,lval,4,0,-1);
    break;

  case KEY_CLKI:
    pic_stkupd (p, &dma->mcfg, 1, -1);
    mcfg = pic_acmd (p, ADON_RD, DMA_GMCFG+side, 0, 0); 
    if (*lval>0) { dma->mcfg |= IOM_CLKI; mcfg |= IOM_CLKI; p->flags |= FLG_CLKI; }
    else { dma->mcfg &= (~IOM_CLKI); mcfg &= (~IOM_CLKI); p->flags &= (~FLG_CLKI); }
    pic_stkupd (p, &dma->mcfg, 1, 1);
    pic_acmd (p, ADON_WR, DMA_GMCFG+side, mcfg, 0); 
    pic_acmd (p, ADON_MOD, side, mcfg, 2);
    break;

  case KEY_TCMODE:
    p->tcmode = (int_1)(*lval);
    break;

  case KEY_ATCCALIB:
    p->atccalib = *dval;
    break;

  case KEY_BIT:
    p->flags &= ~FLG_BITS;
         if (*lval==0) p->flags |= FLG_BIT0;
    else if (*lval==1) p->flags |= FLG_BIT1;
    else if (*lval==4) p->flags |= FLG_BIT4;
    break;

  case KEY_FLGBSET:
    p->flags |= (*lval);
    break;
  case KEY_FLGBCLR:
    p->flags &= ~(*lval);
    break;
  case KEY_TFLGBSET:
    p->tflags |= (*lval);
    break;
  case KEY_TFLGBCLR:
    p->tflags &= ~(*lval);
    break;

  case KEY_MSBI:
    pic_stkupd (p, &dma->mcfg, 1, -1);
    mcfg = pic_acmd (p, ADON_RD, DMA_GMCFG+side, 0, 0); 
    if (*lval>0) { dma->mcfg |= IOM_MSBI; mcfg |= IOM_MSBI; p->flags |= FLG_MSBI; }
    else { dma->mcfg &= (~IOM_MSBI); mcfg &= (~IOM_MSBI); p->flags &= (~FLG_MSBI); }
    pic_stkupd (p, &dma->mcfg, 1, 1);
    pic_acmd (p, ADON_WR, DMA_GMCFG+side, mcfg, 0); 
    pic_acmd (p, ADON_MOD, side, mcfg, 2);
    break;

  case KEY_ALG:
    break;
  case KEY_FEED:
    dma->feed = *lval;
    pic_stkupd (p, &dma->feed, 1, 1);
    break;
  case KEY_TCOFF:
    p->tc_wsec = (len>=8)?  dval[0] : 0;
    p->tc_fsec = (len>=16)? dval[1] : 0;
    p->tc_soff = (len>=24)? dval[2] : 0;
    ulval = (int_8)(p->tc_wsec*4.0e9); /* in 250pSec hacks */
    ulval += (int_8)(p->tc_fsec*4.0e9); /* wsec/fsec accurate */
    if (mtype==IOMT_DXSDDS) pic_set_sddstxtcoffset (p,side,ulval);
    vprint("Key TCOFF soff=%f wsec=%f fsec=%f\n",p->tc_soff,p->tc_wsec,p->tc_fsec);
    break;
  case KEY_SWAP:
    dma->mcsr &= 0xFFFFF8FF;
    dma->mcsr |= (((*lval)&0x7)<<8);
    pic_stkupd (p, &dma->mcsr, 1, 1);
    break;

  case KEY_USER1:
    pic_jtag (p, dmac, JTAG_CMD_USER1, lval, 32, 0);
    break;
  case KEY_USER2:
    pic_jtag (p, dmac, JTAG_CMD_USER2, lval, 32, 0);
    break;
  case KEY_UIDCODE:
    pic_jtag (p, dmac, JTAG_CMD_UIDCODE, lval, 32, 0);
    break;
  case KEY_LOG:
    pic_log(p,0);
    break;

  case KEY_TRACE:
    pic_setkey(p, dmac, KEY_CORE+0xFFC, lval, 4);
    break;

  case KEY_FLASH:
    if (len>4) pic_uflash (p,lval,len,1);
    else if (*lval==999) setflash(p,port);
    else print("CAREFUL - this would reboot the FPGA if value was 999\n");
    break;

  case KEY_ROUTE:
    if (dmac==0 && p->isX) dmac=1;
    if (*lval==-1) i = pic_wpb (p,dmac%10,PPC_DMAC_ROUTE_RST,*lval);
    else           i = pic_wpb (p,dmac%10,PPC_DMAC_ROUTE_SET,*lval);
    break;

  case KEY_ROUTF:
    if (p->isY) pic_acmd (p, ADON_WR, DMA_ROUTF, *lval, 0);
    break;

  case KEY_TIMEOUT:
    p->timeout = lval[0];
    break;

  case KEY_HUNG:
    pic_acmd (p, ADON_MOD, side, 0, 0); 
    break;

  case KEY_TPOE:
    i = *lval;
    if (!p->isY) break;
    if (port>=7) pic_acmd (p,ADON_WRIOB,getJTAGsel(p,port)|PM_TPORT_CFG,(i<=0)?0:(0x10|(i-1)),1);
    else { ioc_wr(p,PM_TPORT_CFG, (i>0)?0x10+i-1 : (i<0)? 0x08 : 0 ); }
    break;

  case KEY_MPCIO:
    ioc_wr(p,PM_TPORT_MPC,*lval);
    break;

  case KEY_HWREV:
    if (!p->isY) break;
    if (findintflag("FACTORY",p->config)!=0x1CE) { print("HW Revision can only be changed at the ICE factory\n"); break; }
    if (p->type==ICEPIC7) ampc_write(p,0x00A00078,(int_1*)lval,4);
    if (p->type==ICEPIC8) ampc_write(p,0x00A0FFF8,(int_1*)lval,4);
    break;

  case KEY_JTAGPORT:
    if (!p->isY) break;
    i = 0x80000000 | (0x000F0000 & getJTAGsel(p,*lval));
    print("Setting JTAG enable mask to %08x\n",i);
    pic_wpm (p,0,PPC_JTAG_ENA,&i,4,FLG_PPC_BUS);
    break;

  case KEY_TPD:
    ioc_wr(p,PM_TPORT_DAT, *lval);
    break;
  case KEY_TPSR:
    if (!p->isY) break;
    ioc_wr(p,PM_TPORT_CFG, (*lval<0)? 0 : 0x08 );
    ioc_wr(p,PM_TPORT_DAT, *lval);
    break;

  case KEY_GPSMODE:
    if (mtype!=IOMT_GPS) { print("No GPS module configured on side=%d mtype=%d\n",side,mtype); break; }
    pic_jpmemwr (p,side,0x0100,*lval,1);
    pic_jpmemwr (p,side,0xFD000004,*lval,4);
    break;

  case KEY_IOCCLK:
    pic_setup_ioc_clk(p,2,*lval);
    break;
  case KEY_ICLK:
    pic_timer (p,FLG_ICLK,*lval);
    break;
  case KEY_PCLK:
    pic_timer (p,FLG_PCLK,*lval);
    break;
  case KEY_PMINDEX:
    p->pmi = *lval;  
    break;

  case KEY_PCIGEN:
  case KEY_PCILANES:
    if (!p->k7) { print("Only supported on PIC7+ cards\n"); break; }
    i = findintflagdef("PLCTRL",p->config,-1);
    if (i>0) goto PL_DIRECT;
    i = 0x23; /* application directed change of rate and width */
    if (key==KEY_PCILANES); else if (*lval!=p->pbr/250) p->pbr=*lval*250;
    else { print("PCIe bus already at rate=%d\n",p->pbr); break; }
         if (p->pbr==250) i |= 0x00;
    else if (p->pbr==500) i |= 0x10;
    else { print("Illegal PCIe bus rate=%d\n",p->pbr); break; }
    if (key==KEY_PCIGEN); else if (*lval!=p->pbw) p->pbw=*lval;
    else { print("PCIe bus already at width=%d\n",p->pbw); break; }
         if (p->pbw==1) i |=0x00;
    else if (p->pbw==2) i |=0x04;
    else if (p->pbw==4) i |=0x08;
    else if (p->pbw==8) i |=0x0C;
    else { print("Illegal PCIe bus width=%d\n",p->pbw); break; }
    PL_DIRECT:
    j = getJTAGsel(p,0); 
    print("PCIe directed change pbr=%d pbw=%d pl_ctrl=0x%02x\n",p->pbr,p->pbw,i);
    pic_drpa (p,j,0x0);
    pic_drpw (p,j,i);
    udelay(250000);
    break;

  case KEY_RMPL:
    i = pic_nv_read_l(p,0xE4);
    i = (i&0xFFFF0FFF) | (((*lval)&0x7) << 12);
    WR(0xE4,i);
    break;
  case KEY_WMPL:
    i = pic_nv_read_l(p,0xE4);
    i = (i&0xFFFFFF1F) | (((*lval)&0x7) << 5);
    WR(0xE4,i);
    break;
 
  case KEY_PFIFO:
    WR(REG_FIFO, *lval); 
    break;
  case KEY_AFIFO:
    pic_acmd (p, ADON_WR, i, *lval, 0); 
    break;

  case KEY_CLKDLY:
    i = (*lval)&0xF;
    pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_M|i,1);
    if (side==1) pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_A|i,1);
    if (side==2) pic_acmd(p,ADON_WRIOC,IOC_CLKDLY,DLY_B|i,1);
    break;

  case KEY_BREAK:
    *p->pbreak = (*lval);
    break;
  case KEY_VERBOSE:
    i = *lval;
    if (i==-1) i = max(1,p->verbose);
    p->verbose = i;
    break;

  case KEY_IPSOCK:
    p->ipsock = *lval;
    break;
  case KEY_IPADDR:
    if (p->type==ICENIC) p->ipaddr = *lval;
    else set_module_HW_IP_addr (p, side, (int_u4)(*lval),0); /* & BILL */ 
    break;
  case KEY_IPCONN:
    if (p->type==ICENIC) p->ipconn = *lval;
    else if (p->niop>0) pic_send_qsfp (p,dmac,*lval,1);
    else { i = pic_acmd (p,ADON_RDIOC,IOC_STAT|(side<<6),0,1); module_join_leave_group (p, side, 1, (int_u4)(*lval), (i&IOC_STAT_ENB)? 1:0 ); }
    break;
  case KEY_IPDISC:
    if (p->type==ICENIC) p->ipaddr = *lval;
    else if (p->niop>0) pic_send_qsfp (p,dmac,*lval,2);
    else { i = pic_acmd (p,ADON_RDIOC,IOC_STAT|(side<<6),0,1); module_join_leave_group (p, side, -1, (int_u4)(*lval), (i&IOC_STAT_ENB)? 1:0 ); }
    break;
  case KEY_IPVLAN:
    if (p->type==ICENIC) p->ipvlan = *lval;
    else sdds_set_active_vlan (p, side, (int_u2) (*lval));
    break;
  case KEY_IPCFG:
    setup_pkt (p,dmac,(NIC_PORT_STRUCT *)lval);
    break;
  case KEY_PING:
    pic_send_qsfp (p,dmac,*lval,0);
    break;

  case KEY_DCSBN:
    if (mtype==IOMT_LB2D && mrev==3) lb2dm3_setkeyl (p,side,KEY_DCSBN,*lval);
    break;

  case KEY_NVME:
    p->nvme = *lval;
    WR (REG_HAR5,0x1CE00000|p->nvme);
    break;
  case KEY_NVMS:
    pic_nvm_stat(p,dmac,*lval);
    break;
  case KEY_NIOP:
    p->niop = *lval;
    WR (REG_HAR6,0x1CE00000|p->niop);
    break;
  case KEY_NIOS:
    pic_nio_stat(p,dmac,*lval);
    break;

  case KEY_PKTHDR:
    print("Not supported\n");
    break;

  default:
    if (key>=KEY_ARGS && key<KEY_ARGS+16) {
      if (m<0) { print("No APP for DMA chan#%d\n",dmac); break; }
      i = key-KEY_ARGS;
      app->args[i] = *lval;
      pic_stkupd (p, &app->args[i], 1, 1);
    }
    else if (key>=KEY_APP) {
      print("Key APP not implemented yet\n");
    }
    else if (key>=KEY_MOD) {
      maddr = key-KEY_MOD; 
      if (isIOMx(p,side)) i = pic_jpmemwr(p,port,maddr,*lval,4);
      else                i = pic_setkey_module(p,port,maddr,*lval);
    }
    else if (key>=KEY_IOC) {
      addr = key-KEY_IOC; 
      if (len<0) len=1;
      if (len>4) len=4;
      if (port&1) addr|=IOC_A; else addr|=IOC_B;
      pic_acmd (p,ADON_WRIOC,addr,*lval,len);
    }
    else {
      vprint("Illegal key ID=%d on DMAC=%d in setKey() routine\n",key,dmac);
      stat = -1;
    }
  }
  pic_lock(p,LOCK_FREE);
  return stat;
}

int_4 pic_setcorel (PICSTRUCT *p, int_4 port, int_4 addr, int_4 data)
{
  int_4 pmi,cpi,side,stat=0;
  if (port<0) port = 40 - port; /* convert -1x/-2x syntax to 5x/6x syntax */
  pmi = port/10;
  cpi = port%10;
  if (pmi==5 || pmi==6) {
    side = (pmi==6)? 2:1;
    if (isIOMx(p,side)) stat = pic_jpmemwr(p,side,addr|IOM_CORE_OFFSET,data,4);
    else                stat = pic_setkey_module(p,side,addr|IOM_CORE_WROFFSET,data);
    v2print("Key CORE=%d IOM=%d addr=%08x data=%08x stat=%d\n",port,side,addr,data,stat);
  }
  else {
    addr |= core_addr(cpi);
    stat = pic_wpb (p,pmi,addr,data);
    v2print("Key CORE=%d PM=%d addr=%08x data=%08x stat=%d\n",port,pmi,addr,data,stat);
  }
  return stat;
}

int_4 pic_setcore (PICSTRUCT *p, int_4 port, int_4 addr, int_4 *data, int_4 ndata)
{
  int_4 pmi,cpi,side,stat=0,i;
  if (port<0) port = 40 - port; /* convert -1x/-2x syntax to 5x/6x syntax */
  pmi = port/10;
  cpi = port%10;
  if (pmi==5 || pmi==6) {
    side = (pmi==6)? 2:1;
    for (i=0; i<ndata/4; i++,addr+=4) {
      if (isIOMx(p,side)) stat = pic_jpmemwr(p,side,addr|IOM_CORE_OFFSET,data[i],4);
      else                stat = pic_setkey_module(p,side,addr|IOM_CORE_WROFFSET,data[i]);
    }
  }
  else {
    addr |= core_addr(cpi);
    stat = pic_msg (p,pmi,PKTF_BUS_WR,addr,data,ndata,0,-1);
  }
  return stat;
}

int_4 setIOCframe (PICSTRUCT *p, int_4 port, int_4 bits, int_4 frame, int_4 dec, int_4 flag) 
{
  int_4 fword=0,i,j,pbs,fper,status=0,side=(port&1)?1:2;
  if (p->type==ICEPIC1 || p->type==ICEPIC2 || p->type==ICEMBT2) {
    print("IOC Framed dec not supported on this card type\n");
    status = 1;
  }
  else if (p->isY) {
    if (bits<0) bits = -bits*2;
    frame *= (bits/8);
    pbs = p->k7? 9 : 6; /* packet bit shift */
    fper = 0x1<<pbs;
    if (frame%fper != 0) print("Err: IOC Frame=%d must be modulo %d in (bytes)\n",frame,fper);
    if (dec>0xFFFF) { print("Err: IOC Frame dec=%d limited to 64K-1\n",dec); dec=0xFFFF; }
    status = (dec<<16)|(frame>>pbs);
    if (p->k7>=8 && findintflag("HXFD",p->config)>0) {
      status = 0x10000000|(((frame>>6)-1)<<16)|(dec-1);
      pic_wpb (p,0,PPC_DMAC_ROUTE_FDC|side,status);
      status=0;
    }
    if (bits!=0) frame /= (bits/8);
  }
  else if (p->isX) {
    if ((frame&0xFF) != 0) print("Err: IOC Frame=%d must be 256*N\n",frame);
    if ((frame&0xFFF00000) != 0) print("Err: IOC Frame=%d must be < 1M\n",frame);
    if ((p->flags&FLG_VHS) && (frame%6)!=0) print("Err: IOC Frame=%d with VHS must be modulo 6\n",frame);
    if (dec>1024) { print("Err: IOC Frame dec=%d must be <= 1024\n",dec); dec=1024; }
    fword =  (dec<=0)? 0 : dec-1;
    if (flag&0x1) pic_acmd (p,ADON_WRIOC,(port<<6)|IOC_FRAME,fword>>8,1);
    if (flag&0x1) pic_acmd (p,ADON_WRIOC,(port<<6)|IOC_FRAME,fword>>0,1);
    fword =  (frame>0)? 0x8000 | ((frame>>8)-1) : 0;
    if (findintflag("FRAMEDELAY",p->config)>0) fword |= 0x4000;
    if (flag&0x2) pic_acmd (p,ADON_WRIOC,(port<<6)|IOC_FRAME|1,fword>>8,1);
    if (flag&0x2) pic_acmd (p,ADON_WRIOC,(port<<6)|IOC_FRAME|1,fword>>0,1);
    status = frame;
  }
  else {
    /* 4 to 19 matches the shift algorithm in the IOC */
    for (i=4; frame>0 && i<=19 && (0x2<<i)!=frame; i++);
    if (i==20) print("Err: IOC Frame=%d != 2^N from 32 to 512k\n",frame);
    if (dec>1024) { print("Err: IOC Frame dec=%d must be <= 1024\n",dec); dec=1024; }
    if (dec<=0) dec=1; if (dec>1024) dec=1024; j=dec-1;
    if (frame!=0) fword = ((i-4)<<12)|0x0800|j; 
    if (findintflag("FRAMEDELAY",p->config)>0) fword |= 0x0400;
    pic_acmd (p,ADON_WRIOC,(port<<6)|IOC_FRAME,fword,2);
    status = frame;
  }
  vprint("Set IOC %d word=0x%x frame=%d dec=%d\n",port,fword,frame,dec);
  return status;
}

int_4 pic_active (PICSTRUCT *p, int_4 flags) 
{
  int_4 i,j,m;
  if (p->type==ICENIC || p->type==ICENVM) return 0;
  /* clear outstanding locks in case of previous unclean exit */
  j = (p->timeout<0)? 200 : p->timeout*100;
  if (p->lcount<=0) for (i=j; i>=0; i--) {
    if (pic_lock (p, LOCK_QUERY) == 0) break;
    else if (i>0) udelay(10000);
    else if (flags==1) { /* from RESET */
      print("Resetting uncleared semaphore lock on ICEPIC %d\n",p->devno); 
      pic_lock (p, LOCK_CLEAR);
    } else { 
      return -2;	/* signal locked */
    }
  }
  RD(REG_IMB4,i);
  udelay(10000);
  RD(REG_IMB4,m);
  if (i==m) i = -1; /* not booted */
  else i = checkSide(p,p->side,0); 
  return i;
}

int_4 pic_getkeyv (PICSTRUCT *p, int_4 dmac, int_4 key) 
{
  int_4 status=0, side=(dmac&1)?1:2, i=side-1;
  char *s,stmp[256];
  FILE *file;
  getpath (stmp,"tbl/iom");
  strcat (stmp,toLower(getIomName(p->mtype[i],p->mrev[i])));
  strcat (stmp,".tbl");
  file = fopen(stmp,"r");
  if (file==NULL) {
    vprint ("Could not open getKeyValid file=%s\n",stmp);
    return -3;	/* no file */
  }
  status = fread(stmp,1,256,file); stmp[255]=0;
  fclose(file);
  status = findstrflag("KEYS",stmp,stmp,1);
  if (status<0) {
    vprint ("No KEYS string in getKeyValid file=%s\n",stmp);
    return -2;	/* unknown */
  }
  s = pic_key2name(p,key);
  status = findintflag(s,stmp);
  return status;
}

int_4 pic_getkeyl (PICSTRUCT *p, int_4 dmac, int_4 key) 
{
  int_4 lval; 
  if (!keyIsLong(key)) print("ERR: pic_getkeyl used on key not of type long\n");
  pic_getkey (p, dmac, key, (void *)(&lval), 4);
  return lval;
}

double pic_getkeyd (PICSTRUCT *p, int_4 dmac, int_4 key) 
{
  double dval;
  if (!keyIsDouble(key)) print("ERR: pic_getkeyd used on key not of type double\n");
  pic_getkey (p, dmac, key, (void *)(&dval), 8);
  return dval;
}

int_4 pic_getkeyt (PICSTRUCT *p, int_4 dmac, char *tval, int_4 tlen)
{
  return parsetable(p,dmac,tval,tlen,-1);
}

int_4 pic_getkey (PICSTRUCT *p, int_4 dmac, int_4 key, void *vval, int_4 len) 
{
  int_4 stat=1,port,mtype,mrev,maddr,i,j,m,sel,drp,side,addr,*lval,ovsr,tmp,head[2];
  double fcny, ftmp, *dval;
  char stmp[20],*s;
  int_4 ltmp[4];
  int_4 n = max(0,dmac-1);
  DMASTRUCT *dma = p->dma+n;
  APPSTRUCT *app=0;
  CHECK_MAGIC(p);

  lval = (int_4 *)vval;
  dval = (double *)vval;

  if (dmac>0) port = dma->port; 
  else if (dmac<0) port = -dmac; 
  else port = p->pindex;
  if (port<=0) port=1;

  side  = (port&1)? 1:2;
  side  = (p->inp>0)? p->inp : side;
  mtype = p->mtype[side-1];
  mrev  = p->mrev[side-1];
  
  if (dma->appp != 0) {
    m = (dma->appp-APP_STK)/APP_SEP;
    app = p->app+m;
  } 
  else m=-1;

  /* non locked keys */
  switch (key) {
  case KEY_ACTIVE:
    *lval = pic_active(p,dmac);
    break;
  case KEY_STATE:
    *lval = p->state;
    break;
  case KEY_STATUS:
    pic_detect_card((int_4)p->devno,0);
    if (p->isY) pic_getkeyl (p,0,KEY_SYSMON);
    if (p->k7) pic_getkeyl (p,4,KEY_SYSMON);
    for (i=0; i<2; i++) { if (isIOMts(p,i+1)) pic_getkeyl (p,(1+i),KEY_SYSMON); }
    for (i=0; i<p->mpm; i++) { if (p->pmtype[i]>=PMT_ZPPM) pic_getkeyl (p,(11+i),KEY_SYSMON); }
    *lval = checkSide(p,0,0); 
    break;
  case KEY_DETECT:
    *lval = (dmac<0)? pic_detect(1) : pic_detect_card(dmac,0);
    break;
  case KEY_CONFIG:
    pic_detect_config(p,(char*)vval,0);
    break;
  case KEY_DRIVER:
    i = io_ctl (p, IOCTL_VERSION, 0, lval, 4); if (i!=4) *lval = 0;
    break;
  case KEY_QALLOC:
    *lval = io_ctl (p, IOCTL_QALLOC, 0, 0, 0);
    break;
  case KEY_NAME:
    if (dmac==0) dmac=side;
    i = (dmac-1)&0x1;
    if (dmac==5) strcpy((char*)lval,getCardName(p->type));
    if (dmac>=1 && dmac<=2) strcpy((char*)lval,getIomName(p->mtype[i],p->mrev[i]));
    if (dmac>=7 && dmac<=8) strcpy((char*)lval,getPmName(p->pmtype[i]));
    if (dmac>=11 && dmac<=14) strcpy((char*)lval,getPmName(p->pmtype[i]));
    break;
  case KEY_SPEEDS:
    j=dump_speeds(p,max(0,dmac),0);
    if (dmac<0) for (i=0; i<2; i++) if (p->pmtype[i]>0) j+=dump_speeds(p,11+i,0);
    *lval = j;
    break;
  case KEY_EYESCAN:
    if (p->k7) len = pic_eyescan (p, dmac, (int_2*)lval, len);
    else { print("PCIe Eye Scan only supported on PIC7+ cards\n"); stat = -1; }
    break;
  case KEY_BREAK:
    *lval = *p->pbreak;
    break;
  default:
    stat = 0;
  }
  if (stat!=0) return stat;
  stat=1;

  pic_lock (p, LOCK_ALLOC);

  if (key>=KEY_MCORE) {
    m = (port-1)/32+1; i = ((port-1)%32)>>1; if (m==1 && p->pmi!=3) m=p->pmi;
    if (m==99) j = (key-KEY_MCORE) + ((port&1)?PPC_CORA_CTL:PPC_CORB_CTL) + i*MCORE_CHN;
    else j = (key-KEY_MCORE) + ((port&1)?PPC_TUNA_CTL:PPC_TUNB_CTL) + i*MCORE_CHN;
    *lval = pic_rpb(p,m,j); stat=4;
    v2print("Key MCORE=%d PM=%d addr=%08x data=%08x stat=%d\n",port,m,j,*lval,stat);
  }
  else if (key>=KEY_CORE) {
    maddr = (key-KEY_CORE);
    if (dmac==1 || dmac==2 || port>=50 || p->pmi<0 || p->ptype==IOPT_MODULE) {
      if (isIOMx(p,side)) *lval = pic_jpmemrd(p,side,maddr|IOM_CORE_OFFSET,4);
      else *lval = pic_getkey_module(p,side,maddr|IOM_CORE_RDOFFSET);
      v2print("Key CORE=%d IOM=%d addr=%08x data=%08x\n",port,side,maddr,*lval);
    } else if (dmac=-8 && port==8) { 
      maddr |= PPC_TRACER;
      *lval = pic_rpb (p, 0, maddr); 
      v2print("Key CORE=%d CDBG addr=%08x data=%08x\n",port,maddr,*lval);
    } else if (dmac=-9 && port==9) {
      maddr |= PM_TPORT_DBG;
      *lval = rdioc (p,maddr,4);
      v2print("Key CORE=%d CDBG addr=%08x data=%08x\n",port,maddr,*lval);
    } else {
      m = (port-1)/10; i = port-m*10; if (m==0) m=p->pmi; if (m==4) m=0; /* node & core */
      if (dmac>0 && dma->type==IOPT_MCORE) i+=2;
      j = maddr + core_addr(i);
      if (p->mcs==-99) j |= MCORE_SYS; if (p->mcs!=0 && p->chan>1) j |= (p->chan-1)*MCORE_CHN;
      j |= p->coreAddrX;
      *lval = pic_rpb(p,m,j); stat=4;
      v2print("Key CORE=%d PM=%d addr=%08x data=%08x stat=%d\n",port,m,j,*lval,stat);
    }
  }
  else switch (key) {

  case KEY_TABLE:
    stat = pic_getkeyt(p,dmac,(char*)vval,len);
    break;
  case KEY_RATE:
    *lval = dma->rate;
    break;
  case KEY_DRATE:
    *dval = dma->rate;
    if (mtype==IOMT_D2AWG && mrev==3) *dval = d2awgm3_getkeyd (p,side,KEY_DRATE);
    if (mtype==IOMT_LB2D && mrev==3) *dval = lb2dm3_getkeyd (p,side,KEY_DRATE);
    break;
  case KEY_FREQ:
    fcny = (double)dma->fcny / FULLSCALE;
    *dval = fcny;
    break;
  case KEY_DFREQ:
    *dval = p->dfreq;
    break;
  case KEY_DEC:
    *lval = dma->dec;
    break;
  case KEY_GAIN:
    *lval = dma->gain;
    if (dma->type==IOPT_TUNER || dma->type==IOPT_TBANK) break;
  case KEY_MGAIN:
    *lval = dma->gain;
    if (mtype==IOMT_A2D && mrev==13) *lval = pic_get_a2dr13_gain(p,port);
    if (mtype==IOMT_A2D && mrev==14) a2dm14_getkey(p,port,KEY_GAIN,lval);
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) *lval = a2dm1x_get_gain(p,port);
    if (mtype==IOMT_A2D && mrev==20) *lval = a2dm20_getkeyl(p,port,KEY_RFGAIN);
    if (mtype==IOMT_D2AWG && mrev==3) *lval = d2awgm3_getkeyl(p,port,KEY_MGAIN);
    break;
  case KEY_NFGAIN:
    *lval = p->dec5;
    break;
  case KEY_MFREQ:
    fcny = 0.0;
    if (dma->type==IOPT_TUNER || dma->type==IOPT_TBANK) {
      if (p->mbits<0) fcny = .5 * dma->rate;
    } else {
      if (dma->bits<0) fcny = .5 * dma->rate;
    }
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=20) fcny=0;	/* no hilbert here */
    *dval = finddblflagdef("MFREQ",p->config,fcny);
    break;
  case KEY_CHAN:
    *lval = p->chan;
    break;
  case KEY_TODO:
    pic_stkupd (p, &dma->todo, 1, -1);
    *lval = dma->todo;
    break;
  case KEY_MBITS:
    *lval = p->mbits;
    break;
  case KEY_FTTM:
    *lval = p->fttm;
    break;
  case KEY_STIME:
    *dval = p->zeroTime;
    break;

  case KEY_NTPOFF:
    *dval = (double)p->ntpoffset;
    if (mtype==IOMT_D2AWG && mrev==3) *dval = d2awgm3_getkeyd(p,side,KEY_NTPOFF);
    if (mtype==IOMT_A2D && mrev==20) *dval = a2dm20_getkeyd(p,side,KEY_NTPOFF);
    break;

  case KEY_NFREQ:
    *dval = pic_tuner_freq_(p,*dval,port,0);
    break;
  case KEY_NDEC:
    *lval = pic_tuner_dec(p,*lval,port,0);
    break;
  case KEY_NRATIO:
    *dval = pic_tuner_ratio(p,port,*dval,FLG_INFO);
    break;

  case KEY_RFFREQ:
    *dval = 0;
    if (mtype==IOMT_RFXD) *dval = rfxd_get_freq (p,side);
    if (mtype==IOMT_LB2D && mrev<3) *dval = lb2d_get_freq (p,side);
    if (mtype==IOMT_LB2D && mrev==3) *dval = lb2dm3_getkeyd (p,side,KEY_RFFREQ);
    if (mtype==IOMT_D2AWG && mrev<3) *dval = d2awg_get_freq (p,side);
    if (mtype==IOMT_D2AWG && mrev==3) *dval = d2awgm3_getkeyd(p,side,KEY_RFFREQ);
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) *dval = a2dm1x_get_freq (p,side);
    if (mtype==IOMT_A2D && mrev==20) *dval = a2dm20_getkeyd (p,side,KEY_RFFREQ);
    if (mtype==IOMT_D2RF) *dval = d2rf_get_freq (p,side);
    break;
  case KEY_RFGAIN:
    *lval = 0;
    if (mtype==IOMT_RFXD) *lval = rfxd_get_gain (p,side);
    if (mtype==IOMT_LB2D && mrev<3) *lval = lb2d_get_gain (p,side);
    if (mtype==IOMT_LB2D && mrev==3) *lval = lb2dm3_getkeyl (p,side,KEY_RFGAIN);
    if (mtype==IOMT_A2D && mrev==20) *lval = a2dm20_getkeyl(p,port,KEY_RFGAIN);
    if (mtype==IOMT_D2AWG && mrev==3) *lval = d2awgm3_getkeyl(p,side,KEY_RFGAIN);
    break;
  case KEY_RFBW:
    *dval = 0;
    if (mtype==IOMT_RFXD) *dval = rfxd_get_bw (p,side) / 1000000.0;
    if (mtype==IOMT_LB2D && mrev<3) *dval = lb2d_get_bw (p,side) / 1000000.0;
    if (mtype==IOMT_LB2D && mrev==3) *dval = lb2dm3_getkeyd (p,side,KEY_RFBW);
    if (mtype==IOMT_D2AWG && mrev==3) *dval = d2awgm3_getkeyd (p,side,KEY_RFBW);
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) *dval = a2dm1x_get_rate (p,side);
    if (mtype==IOMT_A2D && mrev==20) *dval = a2dm20_getkeyd (p,side,KEY_RFBW);
    break;
  case KEY_RFATTN:
    *lval = 0;
    if (mtype==IOMT_LB2D && mrev<3) *lval = lb2d_get_attn (p,side);
    if (mtype==IOMT_D2AWG && mrev<3) *lval = d2awg_get_attn (p,side);
    if (mtype==IOMT_D2AWG && mrev==3) *lval = d2awgm3_getkeyl(p,side,KEY_RFATTN);
    if (mtype==IOMT_D2RF) *lval = d2rf_get_attn (p,side);
    if (mtype==IOMT_A2D && mrev==20) *lval = a2dm20_getkeyl (p,side,KEY_RFATTN);
    break;
  case KEY_MTOFF:
    if (mtype==IOMT_D2AWG && mrev==3) *dval = d2awgm3_getkeyd(p,side,KEY_MTOFF);
    break;
  case KEY_RFPWR:
    *lval = 0;
    if (mtype==IOMT_LB2D && mrev<3) *lval = lb2d_get_rfpwr (p,side);
    if (mtype==IOMT_LB2D && mrev==3) *lval = lb2dm3_getkeyl (p,side,KEY_RFPWR);
    if (mtype==IOMT_RFXD) *lval = rfxd_get_rfpwr (p,side);
    if (mtype==IOMT_D2AWG && mrev<3)  *lval = d2awg_get_rfpwr (p,side);
    if (mtype==IOMT_D2AWG && mrev==3) *lval = d2awgm3_getkeyl (p,side,KEY_RFPWR);
    if (mtype==IOMT_A2D && mrev==20) *lval = a2dm20_getkeyl (p,side,KEY_RFPWR);
    break;
  case KEY_RFOPTS:
    *lval = 0;
    if (mtype==IOMT_RFXD) *lval = rfxd_get_opts (p,side);
    if (mtype==IOMT_LB2D && mrev<3) *lval = lb2d_get_opts (p,side);
    if (mtype==IOMT_LB2D && mrev==3) *lval = lb2dm3_getkeyl (p,side,KEY_RFOPTS);
    if (mtype==IOMT_D2RF) *lval = d2rf_get_opts (p,side);
    break;
  case KEY_TCXO:
/*    if (mtype==IOMT_D2AWG) *dval = d2awg_get_tcxo (p,side); */
    break;

  case KEY_TPD:
    *lval = ioc_rd(p,PM_TPORT_TPD);
    break;
  case KEY_TPDX:
    *lval = ioc_rd(p,PM_TPORT_TPDX);
    break;
  case KEY_TPSR:
    *lval = ioc_rd(p,PM_TPORT_STA);
    break;
  case KEY_MPCIO:
    *lval = ioc_rd(p,PM_TPORT_MPC);
    break;
  case KEY_CUID:
    lval[0] = lval[1] = 0;
    if (!p->isY) break;
    if (p->type==ICEPIC7) ampc_read(p,0x00A000F8,(int_1*)lval,8);
    if (p->type==ICEPIC8) ampc_read(p,0x00D00011,(int_1*)lval,8);
    break;
  case KEY_HWREV:
    lval[0] = 0;
    if (!p->isY) break;
    if (port==5) { *lval=ioc_rd(p,PM_TPORT_MPC); }
    else if (port==10) { *lval = ioc_rd(p,0x008B); }
    else if (port>=11) { sel = getJTAGsel(p, port); i=pic_acmd (p,ADON_RDIOB,sel|0x008B,0,1); *lval = i&0xFF; }
    else if (p->type==ICEPIC7) ampc_read(p,0x00A00078,(int_1*)lval,4);
    else if (p->type==ICEPIC8) ampc_read(p,0x00A0FFF8,(int_1*)lval,4);
    break;
  case KEY_EVCNT:
    lval[0] = 0;
    if (p->type==ICEPIC8) ampc_read(p,0x00D0000D,(int_1*)lval,8);
    break;
  case KEY_RTCLK:
    lval[0] = lval[1] = 0;
    if (p->type==ICEPIC8) ampc_read(p,0x0020000D,(int_1*)lval,7);
    break;

  case KEY_PKTLEN:
    *lval = p->pktlen;
    break;
  case KEY_FRAME:
    *lval = dma->frame;
    break;
  case KEY_CHAIN:
    *lval = pic_dmachain(p,dmac,-1,0,0,0);
    break;
  case KEY_RATIO:
    *dval = p->res_ratio;
    break;
  case KEY_OVSR:
    pic_tuner_ovsr (p,0,0,FLG_INFO);  /* retrieve ovsr */
    if (dmac==1||dmac==2) side=dmac;
    *lval = p->ovsr[side-1];  
    break;
  case KEY_ALG:
    *lval = 0;
    break;
  case KEY_FEED:
    *lval = dma->feed;
    break;
  case KEY_ATCCALIB:
    *dval = p->atccalib;
    break;

  case KEY_CTYPE:
    *lval = p->type;  
    break;
  case KEY_PTYPE:
    if (dmac>0) *lval = dma->type;
    else *lval = p->ptype;  
    break;
  case KEY_PINDEX:
    if (dmac>0) *lval = dma->port;
    else *lval = p->pindex;  
    break;
  case KEY_MTYPE:
    *lval = mtype;  
    break;
  case KEY_MREV:
    *lval = p->mrev[side-1];  
    break;

  case KEY_PMTYPE:
    i = (dmac<0)? -dmac : max(1,min(2,p->pmi));
    *lval = p->pmtype[i-1];  
    break;
  case KEY_PMINDEX:
    *lval = p->pmi;  
    break;
  case KEY_GCHIP:
    *lval = p->gchip;  
    break;
  case KEY_CPC:
    *lval = p->cpc;  
    break;
  case KEY_CHNS:
    *lval = p->nchan;                    /* for both sides */
    if (dmac==1 || dmac==2) *lval /= 2;  /* for one side */
    break;
  case KEY_MCHNS:
    *lval = p->mchan;  
    break;
  case KEY_TINC:
    i = p->mchan/4;
    if (i==0 || p->gchip==GCFPGA) *lval = 2;
    else *lval = 8/p->cpc;
    break;
  case KEY_TCMODE:
    *lval = (int)p->tcmode;
    break;
  case KEY_BIT:
         if ((p->flags&FLG_BITS)==FLG_BIT0) *lval = 0;
    else if ((p->flags&FLG_BITS)==FLG_BIT1) *lval = 1;
    else if ((p->flags&FLG_BITS)==FLG_BIT4) *lval = 4;
    else *lval = 15;
    break;
  case KEY_FLAG:
    if (dma->flags==0) pic_stkupd (p, &dma->flags, 1, -1);
    *lval = dma->flags;
    break;
  case KEY_DRVFLG:
    *lval = status_flags(p);
    break;
  case KEY_MCFG:
    if (dma->mcfg==0) pic_stkupd (p, &dma->mcfg, 1, -1);
    *lval = dma->mcfg;
    break;
  case KEY_CSIZE:
    *lval = dma->csize*4;
    break;
  case KEY_CXFER:
    *lval = dma->cxfer*4;
    break;
  case KEY_NODMA:
    *lval = 0;
    i = (dmac!=0)? dma->dir : -1;
    if (i==0 || findflag("NODMA",p->config)>=0) *lval = 1;
    if (i<0) { j=findstrflag("OPORT",p->config,stmp,1); if (j>0 && strcmp(stmp,"HOST")!=0 && strcmp(stmp,"THRUCORE")!=0) *lval = 1; }
    if (i>0) { j=findstrflag("IPORT",p->config,stmp,1); if (j>0 && strcmp(stmp,"HOST")!=0 && strcmp(stmp,"THRUCORE")!=0) *lval = 1; }
    break;
  case KEY_DMACHK:
    *lval = pic_acmd (p, ADON_RD, dmastkloc(p,dmac)+DMA_STAT, 0, 0);
    break;
  case KEY_DUMP:
    if (p->ptype==IOPT_CORE) dump_core (p,port,0);
    else if (p->ptype==IOPT_MCORE) dump_core (p,port,0);
    else if (p->ptype==IOPT_TUNER) dump_core (p,port,0);
    else if (mtype==IOMT_GPS) *lval = dump_gps (p,side,0,NULL);
    else if (isIOMx(p,side)) *lval = iom_dump_state(p,side);
    else pic_module_dump (p,side,0);
    break;
  case KEY_MGTDUMP:
    if (dmac>=0) dump_mgt (p,dmac);
    if (dmac<0) dump_rio (p,-dmac,0);
    break;
  case KEY_GPS:
    if (mtype==IOMT_GPS) dump_gps (p,side,0,dval);
    break;

  case KEY_CBUFSZ:
    *lval = pic_cbufsz(p,dmac);  
    break;

  case KEY_PRCCLK:
    *lval = p->prc_clk;  
    break;
  case KEY_IOCCLK:
    *lval = p->ioc_clk;  
    break;
  case KEY_TUNCLK:
    *lval = p->tun_clk;  
    break;
  case KEY_ICLK: case KEY_PCLK:
    *lval = p->int_clk;  
    break;
  case KEY_CCLK:
    *lval = p->ext_clk;  
    break;
  case KEY_MCLK:
    *lval = (int_4)calc_port_clock_rate(p,port);
    break;

  case KEY_PCICLK:
    *lval = p->pbr;
    break;
  case KEY_PCIBUS:
    *lval = p->pbw*8;
    break;
  case KEY_PCIBW:
    *lval = p->pbw*p->pbr;  
    break;
  case KEY_PCIGEN:
    *lval = p->pbr/250;
    break;
  case KEY_PCILANES:
    *lval = p->pbw;
    break;
  case KEY_PCIREV:
    *lval = p->rev;  
    break;
  case KEY_PFIFO:
    RD(REG_FIFO, i); *lval = i;
    break;
  case KEY_AFIFO:
    *lval = pic_acmd (p, ADON_RD, i, 0, 0); 
    break;
  case KEY_PCINACK:
    if (p->type==ICEPIC6 || p->type==ICEPIC7 || p->type==ICEPIC8) { 
      WR(REG_IOC, REG_IOC_ADDR|0x10000);
      RD(REG_IOC,i); *lval=i;
      WR(REG_IOC, REG_IOC_ADDR);
      if (dmac==-2) break; /* Quiet mode */
      tmp = (int_4)pic_time() - pic_acmd (p, ADON_RD, RSTIME_ADDR, 0, 0);
      /* 7-series 0=malformed 1=rxoverflow 2=recverr 3=protocolerr 4=badtlp 5=baddlp 6=replayro 7=replayto */
      if (p->type==7) print("PCIe-%d Errs = All:%08x  Link:%02x Card:%03x Root:%03x  at %s\n", p->devno,i, (i>>20)&0xF,(i>>16)&0xF,(i>>12)&0xF,pic_time_str());
      else print("PCIe-%d Errs = All:%08x  Link:%02x Card:%03x Root:%03x  at %s\n", p->devno,i, (i>>24)&0xFF,(i>>12)&0xFFF,i&0xFFF,pic_time_str());
    }
    else print("PciNACK only supported on series 6+\n"); 
    break;

  case KEY_VERSION:
    *lval = pic_acmd (p, ADON_RD, VERSION_ADDR, 0, 0); 
    break;
  case KEY_FPGAREV:
    getPortSig(p,dmac,0,&tmp,lval);
    break;
  case KEY_FPGASIG:
    getPortSig(p,dmac,0,lval,&tmp);
    break;
  case KEY_FPGADSG:
    getPortSig(p,dmac,1,&tmp,lval); lval[1]=tmp;
    break;
  case KEY_DSGTYPE:
    *lval = p->dsgtype;
    break;
  case KEY_NVME:
    *lval = p->nvme;
    break;
  case KEY_NIOP:
    *lval = p->niop;
    break;
  case KEY_NPM:
    *lval = p->npm;
    break;
  case KEY_NIOS:
    *lval = pic_nio_stat(p,dmac,-1);
    break;

  case KEY_IOCCODE:
    *lval = pic_acmd (p, ADON_RD, IOCCODE_ADDR, 0, 0); 
    break;
  case KEY_PMCODE:
    getPortSig(p,JTAG_PORT_PMOFF+side,0,lval,&tmp);
    break;
  case KEY_RSTIME:
    i = (dmac<=0)? 0 : ((dmac&0x1)? 1 : 2);
    *lval = pic_acmd (p, ADON_RD, RSTIME_ADDR+i, 0, 0); 
    break;

  case KEY_IDCODE:
    pic_jtag (p, dmac, JTAG_CMD_IDCODE, lval, -32, 0);
    break;
  case KEY_UIDCODE:
    pic_jtag (p, dmac, JTAG_CMD_UIDCODE, lval, -32, 0);
    break;
  case KEY_USER1:
    pic_jtag (p, dmac, JTAG_CMD_USER1, lval, -32, 0);
    break;
  case KEY_USER2:
    pic_jtag (p, dmac, JTAG_CMD_USER2, lval, -32, 0);
    break;
  case KEY_PMSTAT:
    *lval = pic_pmstat (p, dmac);
    break;
  case KEY_PMHBT:
    pic_ram (p, dmac, (int_u1*)(&i), PPC_BADDR_DSOCM+PPC_HBTOFF, 4, -1);
    *lval = swapit(i);
    break;

  case KEY_HUNG:
    *lval = 0;
    for (i=0; i<DMA_CHNS; i++) {
      m = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (m==0) break;
      addr = pic_acmd (p, ADON_RD, m+DMA_CINDP, 0, 0);
      ovsr = pic_acmd (p, ADON_RD, addr, 0, 0);
      if (pic_acmd (p, ADON_RD, addr, 0, 0) != ovsr) continue;
      udelay(1000);
      if (pic_acmd (p, ADON_RD, addr, 0, 0) != ovsr) continue;
      udelay(10000);
      if (pic_acmd (p, ADON_RD, m+DMA_TODO, 0, 0) == 0) continue;
      ovsr = pic_acmd (p, ADON_RD, m+DMA_HINDEX, 0, 0);
      if (pic_acmd (p, ADON_RD, m+DMA_HINDEX, 0, 0) != ovsr) continue;
      for (j=0; j<20; j++) { udelay(10000);
        if (pic_acmd (p, ADON_RD, m+DMA_HINDEX, 0, 0) != ovsr) break;
      }
      if (j<20) continue;
      m = (m-DMA_STK)/DMA_SEP+1;
      print("Channel %d appears hung\n",m);
      pic_dmadump(p,m,0);
      (*lval)++;
    }
    break;

  case KEY_TRACE:
    for (i=0; i<(len/4); i++) lval[i] = pic_getkeyl(p, dmac, KEY_CORE+0xFFC);
    break;

  case KEY_FLASH:
    m = len; /* check for just checksum value */
    if (m==4) { len = -1; lval = (int_4 *)malloc(0x05000000); }
    len = pic_uflash (p, lval, len, -1);
    for (i=j=0; i<len/4; i++) j = CRC32(j,lval[i],32);
    if (m==4) { *(int_4 *)vval = j; free((void*)lval); }
    break;

  case KEY_BSCAN:
    *lval = pic_bscan (p, dmac);
    break;

  case KEY_CRC32:
    for (i=j=0; i<len/4; i++) j = CRC32(j,lval[i],32);
    lval[0] = j;
    break;

  case KEY_SEQERR:
    addr = 0x1 | ((side&1)? IOC_A:IOC_B);
    i = pic_acmd (p,ADON_RDIOC,addr,0,1);
    if (p->isY) i >>= 16;
    *lval = i&0xFF;
    break;

  case KEY_SEQFILL:
    addr = 0x2 | ((side&1)? IOC_A:IOC_B);
    i = pic_acmd (p,ADON_RDIOC,addr,0,1);
    if (p->isY) i >>= 24;
    *lval = i&0xFF;
    break;

  case KEY_GAP:
    addr = 0x8 | ((side&1)? IOC_A:IOC_B);
    *lval = pic_acmd (p,ADON_RDIOC,addr,0,4);
    if (p->verbose!=0) print("PKT GAP min=%d max=%d\n",(*lval)&0xFFFF,((*lval)>>16)&0xFFFF);
    break;

  case KEY_EMT:
    addr = 0x0 | ((side&1)? IOC_A:IOC_B);
    i = pic_acmd (p,ADON_RDIOC,addr,0,4);
    *lval = (i>>8)&0xFF; /* as 8 bit data 0-128 */
    break;

  case KEY_ADLM:
    i = -100;
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18) i = a2dm1x_get_adclm(p,side);
    if (mtype==IOMT_A2D && mrev==20) i = a2dm20_getkeyl(p,side,KEY_ADLM);
    if (mtype==IOMT_LB2D && mrev==3) i = lb2dm3_getkeyl(p,side,KEY_ADLM);
    *lval = i;
    break;

  case KEY_PKTHDR:
    if (!p->isY) print("Only supported on series 5+ cards\n");
    addr = IOC_PKTHDR | ((side&1)? PPC_HYPA_CTL : PPC_HYPB_CTL);
    if (p->pktmode==PKT_VRT) {
      i = pic_rpm (p,0,addr,lval,32,FLG_PPC_BUS);
      if (p->verbose!=0) dump_vrt(lval);
    } else {
      i = pic_rpm (p,0,addr,lval,64,FLG_PPC_BUS);
      if (p->verbose!=0) dump_sdds(lval);
    }
    break;

  case KEY_TEMP:
  case KEY_SYSMON:
    port = abs(dmac);
    i = (port>JTAG_PORT_PMOFF && p->pmtype[port-JTAG_PORT_PMOFF-1]==PMT_S6M)? 1:0;
    if ((port==0 && p->type==ICEPIC6 && !p->k7) || i>0) {	/* temp sensor */
      if (i>0) {
        sel = getJTAGsel(p,port);
        pic_acmd (p,ADON_WRIOB,sel|0xA500,31,1);
        *lval = pic_acmd (p, ADON_JTAG, JTAG_TMS_GET32X, 0, JTAG_MSK_GET32|sel); 
      }
      else if (p->type==ICEPIC6) pic_jtag (p, 5, JTAG_CMD_BYPASS, lval, -32, 0); 
      else *lval = -1;
      if (*lval == -1) ftmp = -1.0;
      else { j = (bitrev4(*lval)>>8)&0xFFFF; ftmp = j/128.0; }
      if (key==KEY_TEMP) *dval = ftmp;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port,ftmp,*lval);
    }
    else if (port==0 && p->rev==0xA) {  /* Temp A8 Module */
      i = ioc_rd(p,0x0500);
      lval[0] = ((693*i*4)/1024)-265;
      if (key==KEY_TEMP) *dval = (double)lval[0];
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, lval[0]*1.0, i);
    }
    else if ((port==0 && p->k7 && p->rev<5) || (port>JTAG_PORT_PMOFF && p->pmtype[port-JTAG_PORT_PMOFF-1]>=PMT_ZPPM)) { /* Xilinx V5+ SYSMON */
      drp = 0; 
      if (p->k7==8 && port==0) {
        tmp = pic_acmd(p,ADON_RBACK,PPC_SYSMON,0,8);
	     if ((tmp>>16)==0x1CE0) { vprint("SysMon DRP IF error status=%08x. Staying on JTAG.\n",tmp); }
	else if ((tmp>>20)==0x1CE) drp=1;	/* test for SYSMON DRP */
      }
      j = pic_getkeysize (p, key);
      for (i=0; i<j; i++) {
	if (drp) {
          if (i>0) tmp = pic_acmd(p,ADON_RBACK,PPC_SYSMON|i,0,8);
	  ltmp[i] = tmp & 0xFFFF;
	} else {			/* read from SYSMON JTAG */
	  addr = 0x04000000 + (i<<16);
	  pic_jtag (p, port, JTAG_CMD_SYSMON, &addr, 32, 0); 
	  pic_jtag (p, port, JTAG_CMD_SYSMON, ltmp+i, -32, 0); 
	}
        if (i<(len>>2)) lval[i]=ltmp[i];
      }
      tmp = ltmp[0];
      if (p->k7==7) tmp-=0x28a; /* 5deg */
      if (p->k7==8) tmp-=0x79e; /* 15deg */
      ftmp = 504.0*tmp/0x10000-273.15; /* 12b A2D left justified in 16b register 0.123 deg K per LSB or 504degK full scale  */
      if (key==KEY_TEMP) {
        *dval = ftmp;
      } else {
	print("Node %d SYSMON  ",port);
	print(" Temp = %.2lf  ",ftmp);
	if (p->verbose==2) print(" DieTemp = %.2lf  ",504.0*ltmp[0]/0x10000-273.15);
	print(" Vint = %.2lf  ",3.0*ltmp[1]/0x10000);
	print(" Vaux = %.2lf\n",3.0*ltmp[2]/0x10000);
      }
    } 
    else if ((port==1 || port==2) && p->mtype[port-1]==IOMT_D2AWG && p->mrev[port-1]==3) {
      i = d2awgm3_getkeyl(p,port,KEY_TEMP) & 0xFF;	/* boadr<<16 | junction */
      if (i>50) i = 50 + (i-50)/3;	/* derate for juction temperature make 110=70 */
      if (key==KEY_TEMP) *dval = i;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, i*1.0, i);
    }
    else if (port==1 && p->mtype[0]==IOMT_A2D && p->mrev[0]>=15 && p->mrev[0]<=18) {
      i = a2dm1x_get_temp(p,port);
      i = i>>4;
      if (key==KEY_TEMP) *dval = i * 0.0625;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, i*0.0625, i);
    }
    else if (port==1 && p->mtype[0]==IOMT_A2D && p->mrev[0]==20) {
      i = a2dm20_getkeyl(p,port,KEY_TEMP);
      ftmp = (i-17.272)/1.1818;		/* derate for juction temperature make 90=70 */
      if (key==KEY_TEMP) *dval = ftmp;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, ftmp, i);
    }
    else if (port==1 && abs(p->mtype[0])==IOMT_DXSNT && p->mrev[0]>=6) {
      i = pic_getkey_module(p,port,0x3);
      if (key==KEY_TEMP) *dval = (double)i;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, i*1.0, i);
    }
    else if (port==1 && abs(p->mtype[0])==IOMT_DXTGSDDS && p->mrev[0]>=3) {
      i = pic_getkey_module(p,port,0x3);
      if (key==KEY_TEMP) *dval = (double)i;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, i*1.0, i);
    }
#if D2RF_TEMP
    else if ((port==1 || port==2) && p->mtype[port-1]==IOMT_D2RF) {
      i = d2rf_get_temp(p,port);
      if (key==KEY_TEMP) *dval = i;
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, i*1.0, i);
    }
#endif
    else if (port==4 && p->k7==7) {	/* temp sensor on PIC7 board */
      lval[0]=0; ampc_read(p,0x04900000,(int_1*)lval,2);
      i = (lval[0]<<24) | ((lval[0]<<8)&0xFF0000);
      if (key==KEY_TEMP) *dval = i * (0.0625/0x00100000);
      else print("Node %d SYSMON   Temp = %.2lf\n",port, i*(0.0625/0x00100000));
    }
    else if (port==4 && p->k7==8) {	/* temp sensor on PIC8 board */
      i = ioc_rd(p,0x0300);
      lval[0] = i-17;
      if (key==KEY_TEMP) *dval = (double)lval[0];
      else print("Node %d SYSMON   Temp = %.2lf   Hex=%08x\n",port, lval[0]*1.0, i);
    }
    else {
      if (key==KEY_TEMP) *dval = -1.0;
    }
    if (key!=KEY_TEMP);
    else if (*dval==-1);
    else if (*dval<-20.0 || *dval>=110.0) *dval = -3;
    else if (*dval<0.0) *dval = 0.0;
    break;

  case KEY_TIMEOUT:
    lval[0] = p->timeout;
    break;

  case KEY_ROUTE:
    if (dmac==0 && p->isX) dmac=1;
    *lval = pic_rpb (p,dmac%10,PPC_DMAC_ROUTE);
    break;

  case KEY_ROUTF:
    if (p->isY) *lval = pic_rpb (p,dmac%10,PPC_DMAC_ROUTE_FLG);
    else *lval = 0;
    break;

  case KEY_ROUTIOS:
    if (p->isY) *lval = pic_rpb (p,dmac%10,PPC_DMAC_ROUTE_IOS);
    else *lval = 0;
    break;

  case KEY_DEVNO:
    *lval = p->devno;  
    break;
  /* 8 deprecated keys */
  case KEY_MTYPE1:
    *lval = p->mtype[0];  
    break;
  case KEY_MTYPE2:
    *lval = p->mtype[1];  
    break;
  case KEY_MREV1:
    *lval = p->mrev[0];  
    break;
  case KEY_MREV2:
    *lval = p->mrev[1];  
    break;
  case KEY_PMTYPE1:
    *lval = p->pmtype[0];  
    break;
  case KEY_PMTYPE2:
    *lval = p->pmtype[1];  
    break;
  case KEY_PM1CODE:
    getPortSig(p,JTAG_PORT_PMOFF+1,0,lval,&tmp);
    break;
  case KEY_PM2CODE:
    getPortSig(p,JTAG_PORT_PMOFF+2,0,lval,&tmp);
    break;

  case KEY_CNAME:
    s = (char*)lval;
    i = (p->ptype==IOPT_MCORE)? KEY_MCORE : KEY_CORE;
    stat = pic_getkey(p,-port,i+COREG_PARAM0,lval,4);
    s[4]=0; for (i=3; i>=0; i--) if (s[i]<=' ') s[i]=0;
    break;

  case KEY_QSTAT:
    stat = qsfp_stats(p,dmac,(char*)vval);
    break;

  case KEY_MCORES:
    m = (port-1)/10; i = port-m*10; if (m==0) m=p->pmi;   /* node & core */
    if (m==4) m=0;
    j =  MCORE_SYS + COREG_PARAM0 + PPC_MADDR(i+3);
    stat = pic_rpb(p,m,j+0);
    if (stat==CORE_MCOR) *lval = pic_rpb(p,m,j+4);
    else *lval = 0;
    break;

  default:
    if (key>=KEY_ARGS && key<KEY_ARGS+16) {
      if (m<0) { print("No APP for DMA chan#%d\n",dmac); break; }
      i = key-KEY_ARGS;
      pic_stkupd (p, &app->args[i], 1, -1);
      *lval = app->args[i];
    }
    else if (key>=KEY_MOD) {
      maddr = key-KEY_MOD; 
      if (isIOMx(p,side)) *lval = pic_jpmemrd(p,port,maddr,4);
      else *lval = pic_getkey_module(p,port,maddr);
    }
    else if (key>=KEY_IOC) {
      addr = key-KEY_IOC; 
      if (port&1) addr|=IOC_A; else addr|=IOC_B;
      *lval = pic_acmd (p,ADON_RDIOC,addr,0,len);
    }
    else {
      vprint("Illegal key ID=%d on DMAC=%d in getKey() routine\n",key,dmac);
      stat = -1;
    }
  }
  pic_lock (p, LOCK_FREE);
  return stat;

}
int_4 dump_speeds (PICSTRUCT *p, int_4 node, int_4 flags)
{
  int_4 i,j,pcie,route,cnt,rate=2,delay,tdelay,total=0; double speed; char string[32];

  rate = findintflagdef("RATE",p->config,rate);
  delay = 1000000/rate;
  tdelay = 50000;
  pic_rpm (p,node,PPC_DMAC_STATUS,&route,4,FLG_PPC_BUS);
  if (node<=10) sprintf(string,"XBar=%s",getCardName(p->type));
  else { node-=10; sprintf(string,"PM%d=%s",node,getPmName((int_4)p->pmtype[node-1])); }
  printf("IcePIC-%d %s Route=%08x\n",p->devno,string,route);
  for (i=0; i<32; i++) {
    j = i&0xF;
    if (j>10) continue;
    pcie = (node==0) && (j==1 || j==8);
    pic_wpb (p,node,PPC_DMAC_ROUTE_SPD,i); 
    udelay(tdelay);
    cnt = pic_rpb (p,node,PPC_DMAC_ROUTE_SPD);
    if ((cnt&0x0FFFFFFF)==0 && !pcie) continue;
    pic_wpb (p,node,PPC_DMAC_ROUTE_SPD,i); 
    udelay(delay);
    cnt = pic_rpb (p,node,PPC_DMAC_ROUTE_SPD);
    speed = (cnt&0x0FFFFFFF);
    if (speed==0) continue;
    speed *= (0x40 << (cnt>>28));
    speed *= rate;
    speed *= 1e-6;
    printf("  Chan=%d Port=%s Speed=%f Mby/s\n",i,getPortName(i,node),speed);
    total += (int)speed;
  }
  return total;
}

int_4 dump_rio (PICSTRUCT *p, int_4 port, int_4 flags)
{
  int_4 i,j,off,lvala,lvalb,lvalc,stat;
  off = (0x1<<21);
  stat = pic_msg (p,port,PKTF_BUS_RD,off+4*4,&lvala,0,4,-1);
  stat = pic_msg (p,port,PKTF_BUS_RD,off+5*4,&lvalb,0,4,-1);
  printf("RIO Port=%d Sig=%08x Pattern=%08x\n",port,lvala,lvalb);
  for (i=0; i<2; i++) {
    stat = pic_msg (p,port,PKTF_BUS_RD,off+(i+2)*4,&lvala,0,4,-1);
    stat = pic_msg (p,port,PKTF_BUS_RD,off+(i+6)*4,&lvalb,0,4,-1);
    stat = pic_msg (p,port,PKTF_BUS_RD,off+(i+0)*4,&lvalc,0,4,-1);
    print("RIO Port=%d Stat=%08x : TxPkts=%02x : RxPkts=%02x : RxErrs=%02x : Data=%08x [", i,lvala,(lvala)&0xFF,(lvala>>8)&0xFF,(lvala>>16)&0xFF,lvalb);
    for (j=0; j<4; j++) { stat = pic_msg (p,port,PKTF_BUS_RD,off+(8+i*4+j)*4,&lvala,0,4,-1); print(" %08x ",lvala); }
    printf("]\n");
  }
  return 0;
}
 
int_4 dump_core (PICSTRUCT *p, int_4 port, int_4 flags) 
{
  int_4 i,j,n,o,w,s,stat,stat2,mc=0,vals[8],cport,mcport,isCore,isMCore;
  n = findintflagdef("CPN",p->config,8);
  o = findintflagdef("CPO",p->config,0);
  w = findintflagdef("CPW",p->config,4);
  s = findintflagdef("CPS",p->config,4);
  isCore = (p->ptype==IOPT_CORE);
  isMCore = (p->ptype==IOPT_MCORE);
  cport = ((p->pmi==0)?40:p->pmi*10) + port;
  mcport = ((p->pmi==0)?0:(p->pmi-1)*32) + port;
  if (isCore) mc=pic_getkeyl(p,-cport,KEY_MCORES);
  print("dumpCore ptype=%d pmi=%d port=%d cport=%d mc=%d\n",p->ptype,p->pmi,port,cport,mc);
  if (isCore && mc>0) {
    for (i=1; i<8; i++) vals[i] = pic_getkeyl(p,-cport,KEY_CORE+MCORE_SYS+COREG_PARAM0+(i<<2));
    print("Core=%d MC=%d NC=%d IC=%d UPS=%d CLKX=%d CLKF=%d\n",port,mc,vals[1],vals[2],vals[4],vals[6],vals[7]);
  }
  if (isCore) for (i=0; i<8; i++) {
    stat = pic_getkeyl(p,-cport,KEY_CORE+COREG_PARAM0+(i<<2));
    if (i==0) print("Core=%d Param=%d Value=0x%08x Sig=%.4s\n",port,i,stat,(char*)(&stat));
    else      print("Core=%d Param=%d Value=0x%08x Int=%d\n",port,i,stat,stat);
  }
/* if (mc>0) return 0; */
  i = findintflagdef("DBG",p->config,0);
  if (i>0) { n=i; o=0x0800; }
  if (isCore) for (i=0; i<n; i++) {
    if (w==8) {
      j = o + (i<<3);
      stat = pic_getkeyl(p,-cport,KEY_CORE+j);
      stat2 = pic_getkeyl(p,-cport,KEY_CORE+j+4);
      print("Core=%d Register=%d:%d Value=0x%08x:%08x\n",port,i,j,stat2,stat);
    } else {
      j = o + i*s;
      stat = pic_getkeyl(p,-cport,KEY_CORE+j);
      print("Core=%d Register=%d:%d Value=0x%08x %d\n",port,i,j,stat,stat);
    }
  }
  else for (i=0; i<n; i++) {
      j = o + i*s;
      stat = pic_getkeyl(p,-mcport,KEY_MCORE+j);
      print("MCore=%d Register=%d:%d Value=0x%08x %d\n",port,i,j,stat,stat);
  }
  return 0;
}

int_4 dump_sdds (int_4 *lval) {
  ICESDDSPACKETSTRUCT *sdds = (ICESDDSPACKETSTRUCT *)lval;
  int_4 i;
  print("SDDS Packet Hex Dump\n");
  print(" ICE Keys = %04hx\n",sdds->keys);
  print(" UDP Port = %04hx\n",swapst(sdds->port));
  print(" UDP Addr = %08x\n",swapit(sdds->addr));
  print(" Format   = %02hhx\n",sdds->fid);
  print(" Bits     = %02hhx\n",sdds->bits);
  print(" Frame    = %04hx\n",swapst(sdds->frame));
  print(" TC info  = %04hx\n",swapst(sdds->tcinfo));
  print(" TC corr  = %04hx\n",swapit(sdds->tccorr));
  print(" TC tics  = %08x %08x\n",swapit(lval[5]),swapit(lval[4]));
  print(" TC extp  = %08x\n",swapit(sdds->tcext));
  print(" SSC info = "); for (i=0;i<12;i++) print("%02hhx ",sdds->sscinfo[i]); print("\n");
  print(" SSD info = "); for (i=0;i< 4;i++) print("%02hhx ",sdds->ssdinfo[i]); print("\n");
  print(" AAD info = "); for (i=0;i<20;i++) print("%02hhx ",sdds->aadinfo[i]); print("\n");
  return 0;
}

real_8 vctx_b2d (int_4 *data, int_4 fpp) {
  int_8 ldata = *(int_8*)data;
  real_8 sdata = (real_8)swaplt(ldata);
  return sdata/(1<<fpp);
}
real_4 vctx_b2f (int_4 *data, int_4 fpp) {
  real_4 sdata = (real_4)swapit(*data);
  return sdata/(1<<fpp);
}
int_4 vctx_b2l (int_4 *data) {
  int_4 sdata = swapit(*data);
  return sdata;
}

int_4 dump_vrt (int_4 *lval) {
  VRTPACKETSTRUCT *vrt = (VRTPACKETSTRUCT*)lval;
  VRTCONTEXTSTRUCT *vctx = (VRTCONTEXTSTRUCT*)lval;
  int_4 hdrbe=*lval, hdrle=swapit(hdrbe), len=(hdrle&0xFFFF), mask, *ival;
 if ((hdrbe&0xF0)==0x40) {	/* context packet */
  print("VRT Context Packet LE Dump Len=%d\n",len);
  print(" Header  = %08x\n",swapit(vctx->header));
  print(" StreamID = %08x\n",swapit(vctx->streamID));
  if ((hdrbe&0xF8)==0x48) { ival = lval+7;	/* has OUID */
   print(" OrgID    = %08x\n",swapit(vctx->orgUID));
   print(" PktCC    = %08x\n",swapit(vctx->pktinfoCC));
   print(" TS ISec = %08x\n",swapit(vctx->tsis));
   print(" TS FSec = %016lx\n",swaplt(vctx->tsfs));
  } else ival = lval+5; 
  mask = bitrev4(swapit(*ival)); ival++;
  if (mask&0x0001) { print(" ChX     = Context has changes\n"); }
  if (mask&0x0002) { print(" RPI     = %f\n",vctx_b2l(ival));    ival+=1; }
  if (mask&0x0004) { print(" BW-Hz   = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0008) { print(" IF-Hz   = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0010) { print(" RF-Hz   = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0020) { print(" RFo-Hz  = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0040) { print(" IFo-Hz  = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0080) { print(" RefL-dB = %f\n",vctx_b2f(ival,7));  ival+=1; }
  if (mask&0x0100) { print(" Gain-dB = %f\n",vctx_b2f(ival,7));  ival+=1; }
  if (mask&0x0200) { print(" OR-Cnt  = %d\n",vctx_b2l(ival));    ival+=1; }
  if (mask&0x0400) { print(" SR-Hz   = %f\n",vctx_b2d(ival,20)); ival+=2; }
  if (mask&0x0800) { print(" TAdj-s  = %f\n",vctx_b2f(ival,0));  ival+=1; }
  if (mask&0x1000) { print(" TCal-s  = %f\n",vctx_b2f(ival,0));  ival+=1; }
  if (mask&0x2000) { print(" Temp-C  = %f\n",vctx_b2f(ival,6));  ival+=1; }
  if (mask&0x4000) { print(" DevId   = %08x\n",vctx_b2l(ival));  ival+=1; }
  if (mask&0x8000) { print(" Event   = %08x\n",vctx_b2l(ival));  ival+=1; }
  if (mask&0x10000){ print(" FMT-w1  = %08x\n",vctx_b2l(ival));  ival+=1; }
  if (mask&0x10000){ print(" FMT-w2  = %08x\n",vctx_b2l(ival));  ival+=1; }
  if (mask&0x20000){ print(" GPS     = %d\n",vctx_b2l(ival));    ival+=1; }
 } else {			/* other packet */
  print("VRT Data Packet BE Hex Dump Len=%d\n",len);
  print(" Header   = %08x\n",vrt->header);
  print(" StreamID = %08x\n",vrt->streamID);
  print(" OrgID    = %08x\n",vrt->orgUID);
  print(" PktCC    = %08x\n",vrt->pktinfoCC);
  print(" TS ISec  = %08x\n",vrt->tsis);
  print(" TS FSec  = %016lx\n",vrt->tsfs);
  print(" Trailer  = %08x\n",vrt->trailer);
 }
  return 0;
}

int_4 dump_sysmon (int_4 port, int_4 *data, int_4 n) {
  print("Node %d SYSMON  ",port);
  print(" Temp = %lf  ",504.0*data[0]/0x10000-273.15);
  print(" Vint = %lf  ",3.0*data[1]/0x10000);
  print(" Vaux = %lf\n",3.0*data[2]/0x10000);
  return 0;
}

int_4 status_flags (PICSTRUCT *p) {
  int_4 i,flg,flags,mtype,mrev;
  flags = 0;
  flg = ioc_rd(p,0x0400);      /* Main Board xsync 0=100Hz 1=10MHz 2=1PPS */
  if (flg&0x1) flags |= 0x4;
  if (flg&0x2) flags |= 0x1;
  /* if (flg&0x4) flags |= 0x2; PPS NIY */
  for (i=0; i<p->miom; i++) {
    mtype = p->mtype[i];
    mrev = p->mrev[i];
    if (mtype==IOMT_D2RF) {		/* IO Module 0=10MHz 1=PPS */
      flg = d2rf_get_clkstat (p, i+1);
      flags |= (flg&0xF)<<(i*4+8);
    }
    if (mtype==IOMT_LB2D && mrev==3) {
      flg = lb2dm3_getkeyl (p, i+1, KEY_CLKSTAT);
      flags |= (flg&0xF)<<(i*4+8);
    }
    if (mtype==IOMT_D2AWG && mrev==3) {
      flg = d2awgm3_getkeyl (p, i+1, KEY_CLKSTAT);
      flags |= (flg&0xF)<<(i*4+8);
    }
    if (mtype==IOMT_A2D && mrev==14) {
      a2dm14_getkey (p, i+1, KEY_CLKSTAT, &flg);
      flags |= (flg&0xF)<<(i*4+8);
    }
    if (mtype==IOMT_A2D && mrev==20) {
      flg = a2dm20_getkeyl(p,i+1,KEY_CLKSTAT);
      flags |= (flg&0xF)<<(i*4+8);
    }
  }
  return flags;
}

int_4 ptemp (PICSTRUCT *p, int_4 port) {
  int_4 status,sel;
  status = (int_4)pic_getkeyd(p,port,KEY_TEMP); 
  if (status == -1) return status;	/* not handled */
  if (status>=0) return status;
  status = (int_4)pic_getkeyd(p,port,KEY_TEMP); /* try again */
  if (status>=0) return status;
  if (port==0 && p->k7==8 && msg_count<=16) { 
    pic_setup_sysmon (p,0);
    vprint("WARNING: Rebooting SYSMON to fix spurious temp reading on devno=%d node=%d\n", p->devno,port);
    udelay(10000);
  }
  status = (int_4)pic_getkeyd(p,port,KEY_TEMP); 
  if (status>=0) return status;
  msg_count++;
  if (msg_count==0)  print("WARNING: Enabling ICEPIC spurious temp readings after 64k ignores on devno=%d node=%d\n", p->devno,port);
  if (msg_count<=16) print("WARNING: ICEPIC spurious temp reading on devno=%d node=%d status=%d\n", p->devno,port,status);
  if (msg_count==16) print("WARNING: Ignoring future ICEPIC spurious temp readings on devno=%d node=%d\n", p->devno,port);
  return -3;
}

int_4 pic_log (PICSTRUCT *p, int_4 flag) {
  int_4 i,status=0,skip=0,qvals[4],lvals[9];
  CHECK_MAGIC(p);
  if (findintflag("NOLOG",p->config)>0) return 0;
  io_ctl (p, IOCTL_QLOG, 0, qvals, 16);
  lvals[0] = (int_4)gettime();
  if (flag==0 && (lvals[0]-qvals[0])<5) return -1; /* dont overlog temp */
  pic_lock(p,LOCK_ALLOC);
  lvals[1] = ptemp(p,0);
  lvals[2] = ptemp(p,1);
  lvals[3] = ptemp(p,2);
  lvals[4] = ptemp(p,11);
  lvals[5] = ptemp(p,12);
  lvals[6] = status_flags(p);	/* status flags */
  lvals[7] = checkSide(p,0,0);	/* DMA active count */
  status = io_ctl (p, IOCTL_LOG, 0, lvals, 32);
  pic_lock(p,LOCK_FREE);
  if (lvals[1]>MAX_PIC_TEMP || lvals[2]>MAX_IOM_TEMP || lvals[3]>MAX_IOM_TEMP || lvals[4]>MAX_PM_TEMP || lvals[5]>MAX_PM_TEMP) {
    print("WARNING: ICEPIC devno=%d component overtemp pic=%d iom1=%d iom2=%d pm1=%d pm2=%d.\n", p->devno,lvals[1],lvals[2],lvals[3],lvals[4],lvals[5]);
    if (findintflag("NOTEMP",p->config)<=0) { print("WARNING: Shutting down.\n"); p->state = pic_shutdown(p); }
  }
  return 0;
}

/*
 address map bits 
 [9:1] drp address
 [12:10] port number 0-7
 [13] All ports write or read dig monitor
 [15:14] MGT address
*/
int_4 pic_drpa (PICSTRUCT *p, int_4 sel, int_4 addr) {
  int_4 jtmp, jcbs = getJTAGcbs(sel);
  pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0, JTAG_MSK_RST|sel); 
  pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, JTAG_CMD_USER2<<jcbs, JTAG_MSK_CMD|sel); 
  jtmp = 0x00060002 | ((addr&0xFF00)<<12) | ((addr&0xFF)<<4); /* IOB Page+Addr */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, jtmp, JTAG_MSK_SET32X|sel); 
  return 0;
}
int_4 pic_drpc (PICSTRUCT *p, int_4 sel) {
  pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0, JTAG_MSK_RST|sel); 
  return 0;
}
int_4 pic_drpw (PICSTRUCT *p, int_4 sel, int_4 data) {
  int_4 jtmp = 0x00080008 | ((data&0xFF00)<<12) | ((data&0x00FF)<<4);  /* IOB Data Write with auto address increment */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, jtmp, JTAG_MSK_SET32X|sel); 
  return 0;
}
int_4 pic_drpr (PICSTRUCT *p, int_4 sel) {
  int_4 jtmp=0x00090009;  /* IOB Data Read 2 bytes with auto address increment */
  pic_acmd (p, ADON_JTAG, JTAG_TMS_GET16X, 0x0001, JTAG_MSK_GET16X|sel);	/* preload the DRP register */
  jtmp = pic_acmd (p, ADON_JTAG, JTAG_TMS_GET32X, jtmp, JTAG_MSK_GET32X|sel);
  return ((jtmp>>8)&0xFF) | ((jtmp>>16)&0xFF00);
}
int_4 pic_drprb (PICSTRUCT *p, int_4 sel) {
  int_4 jtmp=0x00010001;  /* IOB Data Read 1 byte with NO auto address increment - preload DRP register */
  jtmp = pic_acmd (p, ADON_JTAG, JTAG_TMS_GET32X, jtmp, JTAG_MSK_GET32X|sel);
  return ((jtmp>>24)&0xFF);
}
int_4 pic_drpar (PICSTRUCT *p, int_4 sel, int_4 addr, int_4 len) {
  pic_drpa(p,sel,addr);
  return (len==1)? pic_drprb(p,sel) : pic_drpr(p,sel);
}
int_4 pic_drprx (PICSTRUCT *p, int_4 sel, int_4 i2, int_4 i1) {
   int_4 val = pic_drpr(p,sel);
   return (val>>i1) & ((2<<(i2-i1))-1);
}

int_4 dump_mgt (PICSTRUCT *p, int_4 dmac) {

  int_4 addr,val,cfgo,i,j,nl=8,k8=(p->k7==8);
  int_4 port = ((dmac%10)==0)?0:(dmac%10)+JTAG_PORT_PMOFF;
  int_4 sel = getJTAGsel(p,port); 
  int_4 DMON_CFG,RXLPM_CFG,RXDFE_CFG,ADAPT_CFG;

  if (p->k7==0) { print("MGT dump only supported on K7/K8 devices\n"); return 0; }

  DMON_CFG	= (k8?0x3A:0x3A)<<1;
  RXLPM_CFG	= (k8?0x32:0x32)<<1;
  RXDFE_CFG	= (k8?0x54:0x54)<<1;
  ADAPT_CFG	= (k8?0x92:0x92)<<1;

  addr=0x2000;			/* all-lane write and/or individual lane digital monitor read */
  if (dmac>=10) addr|=0x8000; 	/* RIO ports vs PCIe ports */
  else if (dmac>0) addr|=0xC000;

  print("MGT dump port=%d dmac=%d addr=%08x sel=%08x\n",port,dmac,addr,sel);

  pic_drpa (p,sel,addr|DMON_CFG); cfgo = 0; /*pic_drpr (p,sel) & ~0x0100; */
  pic_drpa (p,sel,addr|DMON_CFG); pic_drpw (p,sel,cfgo|0x0100);	/* enable digital monitor */
  pic_drpa (p,sel,addr|ADAPT_CFG); pic_drpw (p,sel,0x0000);	/* enable LPM digital monitor */

  pic_drpa (p,sel,addr|RXLPM_CFG); pic_drpw (p,sel,0x0020);	/* enable HF Gain */
  print("LPM HF Gains  "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,2)); } print("\n");
  pic_drpa (p,sel,addr|RXLPM_CFG); pic_drpw (p,sel,0x0028);	/* enable LF Gain */
  print("LPM LF Gains  "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,2)); } print("\n");
  pic_drpa (p,sel,addr|RXLPM_CFG); pic_drpw (p,sel,0x0030);	/* enable Baseline wander */
  print("LPM Wander    "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,0)); } print("\n");
  pic_drpa (p,sel,addr|RXLPM_CFG); pic_drpw (p,sel,0x0038);	/* enable AGC Gain */
  print("LPM AGC Gains "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,4,0)); } print("\n");

  pic_drpa (p,sel,addr|ADAPT_CFG); pic_drpw (p,sel,0x0200);	/* enable DFE digital monitor */

  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0020);	/* enable Baseline wander */
  print("DFE Wander    "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,0)); } print("\n");
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0021);	/* enable LF Gain */
  print("DFE LF Gains  "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,2)); } print("\n");
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0022);	/* enable VP Peak */
  print("DFE VP Peaks  "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,0)); } print("\n");
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0024);	/* enable AGC Gain */
  print("DFE AGC Gains "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,4,0)); } print("\n");
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0023);	/* enable Tap 1 */
  print("DFE Tap 1     "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,6,0)); } print("\n");
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0025);	/* enable Tap 2 */
  print("DFE Tap 2     "); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,5,0)); } print("\n");
 for (j=3; j<12; j++) {
  pic_drpa (p,sel,addr|RXDFE_CFG); pic_drpw (p,sel,0x0023+j);	/* enable Tap 3 */
  print("DFE Tap %x     ",j); for (i=0;i<nl;i++) { pic_drpa(p,sel,addr|(i<<10)); print(" %04x",pic_drprx (p,sel,4,0)); } print("\n");
 }
  pic_drpa (p,sel,addr|DMON_CFG); pic_drpw (p,sel,cfgo); /* disable digital monitor */
  pic_drpc (p,sel);

  return 0;
}

int_4 pic_eyescan (PICSTRUCT *p, int_4 dmac, int_2 *buf, int_4 len) {
  int_4 ix,iy,ut,vert,dvert,horz,dhorz,lane,scale=0,gen=1,retry,stat,drpx;
  int_4 i,j,k,errs,samps,n=len/2,addr=0,addri,gridp,k8=(p->k7==8),k9=0,loop;
  int_4 port = ((dmac%10)==0)?0:(dmac%10)+JTAG_PORT_PMOFF;
  int_4 sel = getJTAGsel(p,port); 
  int_4 lanes = (buf[0]&0x0F);
  int_4 lane0 = (buf[0]&0xF0)>>4;
  int_4 grid  = buf[0]>>8;
  int_4 errco[8],timeout=8;
  double telapse=pic_time();

  if ((port==0 && p->k7==0) || (port>JTAG_PORT_PMOFF && p->pmtype[port-JTAG_PORT_PMOFF-1]<PMT_K8M)) 
    { print("EYEScan port=%d only supported on K7/K8 devices\n",port); return 0; }

  drpx = findintflagdef("DRPX",p->config,1);
  if (port>JTAG_PORT_PMOFF && p->pmtype[port-JTAG_PORT_PMOFF-1]==PMT_K8P) k9=1;

  int_4 ES_QUALIFIER      = (k8?0x3F:0x2C)<<1;
  int_4 ES_QUAL_MASK      = (k8?0x44:0x31)<<1;
  int_4 ES_SDATA_MASK     = (k8?0x49:0x36)<<1;
  int_4 ES_VERT_OFFSET    = (k8?0x97:0x3B)<<1;
  int_4 ES_HORZ_OFFSET    = (k8?0x4F:0x3C)<<1;
  int_4 ES_CONTROL        = (k8?0x3C:0x3D)<<1;
  int_4 ES_ERROR_COUNT    = (k8?0x151:0x14F)<<1;
  int_4 ES_SAMPLE_COUNT   = (k8?0x152:0x150)<<1;
  int_4 ES_CONTROL_STATUS = (k8?0x153:0x151)<<1;
  int_4 ES_QUALIFIER_X    = (k8?0xE7:0x2C)<<1;
  int_4 ES_QUAL_MASK_X    = (k8?0xEC:0x31)<<1;
  int_4 ES_SDATA_MASK_X   = (k8?0xF1:0x36)<<1;

  if (lanes==0) {
    addr = findintflagdef("DRPA",p->config,addr);
    pic_drpa (p,sel,addr);
    for (i=0; i<n; i++)  buf[i] = pic_drpr (p,sel);
    grid = 0;
  } else {
    n /= lanes;
    while (grid*grid<n) grid++;
    gridp = grid+1;
    if (dmac<10) {
      stat = pic_drpar (p,sel,addr,1); /* read MSCR[15:8] */
      gen = (stat>>4)&0x3; if (gen==0) gen=1;
    }
    if (dmac>0) gen=2;
    gen = findintflagdef("GEN",p->config,gen);
    dvert = 256;
    dhorz = (gen==3)? 32 : (gen==2)? 64 : 128;
    if (k8) dhorz*=2;	/* range in ultrascale is 2x */

    addr = (0x2000);	/* all lanes */
    if      (dmac>9) addr |= 0x8000; /* RIO vs PCIe ports */
    else if (dmac>0) addr |= 0xC000; /* NIO vs RIO */
    addr = findintflagdef("DRPA",p->config,addr);
    scale = findintflagdef("EYESCALE",p->config,0);

    pic_drpa (p,sel,addr|ES_SDATA_MASK);
    for (i=0; i<5; i++) pic_drpw (p,sel,(i<=1)?0x0000:(i==2)?0xFF00:0xFFFF);
    pic_drpa (p,sel,addr|ES_QUAL_MASK);
    for (i=0; i<5; i++) pic_drpw (p,sel,0xFFFF);
    pic_drpa (p,sel,addr|ES_QUALIFIER);
    for (i=0; i<5; i++) pic_drpw (p,sel,0x0000);
   if (k9) {
    pic_drpa (p,sel,addr|ES_QUAL_MASK_X);
    for (i=0; i<5; i++) pic_drpw (p,sel,0xFFFF);
    pic_drpa (p,sel,addr|ES_QUALIFIER_X);
    for (i=0; i<5; i++) pic_drpw (p,sel,0x0000);
   }
    pic_drpa (p,sel,addr|ES_CONTROL);
    pic_drpw (p,sel,k8?0x0300:0xE300);
  }
  for (i=0; i<lanes; i++) errco[i]=0;
  for (i=0; i<n*lanes; i++) buf[i]=0;
  loop=0;
  vprint("EyeScan Start:  port=%d lane0=%d lanes=%d grid=%d addr=%x scale=%d stat=%x sel=%08x\n",
		port,lane0,lanes,grid,addr,scale,stat,sel);
  if (k8) {
   if (drpx==0) {
    pic_drpa (p,sel,0x0002);
    pic_drpw (p,sel,0x01);	/* individual mode */
   }
   for (lane=0; lane<lanes; ) {
    vert = horz = ut = 0;
    addri = addr | ((lane+lane0)<<10); if (drpx) addri ^= 0x2000;
    pic_drpa (p,sel,addri|ES_VERT_OFFSET);
    pic_drpw (p,sel,(ut<<9)|((vert&0x17F)<<2));
    pic_drpa (p,sel,addri|ES_HORZ_OFFSET);
    pic_drpw (p,sel,(horz&0x0FFF)<<(k8?4:0));
    pic_drpa (p,sel,addri|ES_CONTROL);
    pic_drpw (p,sel,(0x0700|scale));
    udelay(10000);
    k = pic_drpar(p,sel,addri|ES_CONTROL_STATUS,1);
    pic_drpa (p,sel,addri|ES_ERROR_COUNT);
    errs = pic_drpr (p,sel);
    if (errs>0 && loop<32) {
      loop++;
      pic_drpa (p,sel,addri|ES_CONTROL);
      pic_drpw (p,sel,0x0000);
    } else {
      if (loop>0) vprint("EyeScan Reboot: lane=%d loop=%d k=%x err=%x addr=%x\n",lane+lane0,loop,k,errs,addri);
      lane++; loop=0;
    }
    pic_drpa (p,sel,addri|ES_CONTROL);
    pic_drpw (p,sel,0x0300);
   }
   if (drpx==0) {
    pic_drpa (p,sel,0x0002);
    pic_drpw (p,sel,0x00);	/* normal mode */
   }
  }

  for (iy=0; iy<grid; iy++) {
  for (ut=0; ut<2; ut++) {
    vert = (dvert/gridp)*(iy+1) - (dvert/2);
    if (vert<0) vert = vert^0x7F;
    pic_drpa (p,sel,addr|ES_VERT_OFFSET);
    if (k8) pic_drpw (p,sel,(ut<<9)|((vert&0x17F)<<2));
    else    pic_drpw (p,sel,(scale<<11)|(ut<<8)|(vert&0xFF));

  for (ix=0; ix<grid; ix++) {
    horz = (dhorz/gridp)*(ix+1) - (dhorz/2);
    pic_drpa (p,sel,addr|ES_HORZ_OFFSET);
    pic_drpw (p,sel,(horz&0x0FFF)<<(k8?4:0));
    pic_drpa (p,sel,addr|ES_CONTROL);
    pic_drpw (p,sel,k8?(0x0700|scale):0xE301);
    for (lane=0; lane<lanes; lane++) {
      if (errco[lane]>100) continue;
      addri = addr | ((lane+lane0)<<10); if (drpx) addri ^= 0x2000;
      for (i=0; i<timeout; i++) {
        k = pic_drpar(p,sel,addri|ES_CONTROL_STATUS,1);
        if ((k&7)==0x5) break; /* END and DONE bits */
	udelay(100);
      }
      if (i>=timeout) errco[lane]++;
      pic_drpa (p,sel,addri|ES_ERROR_COUNT);
      errs = pic_drpr (p,sel);
      /* samps = pic_drpr (p,sel); */
      buf[lane*n+iy*grid+ix] += (errs>>2);
    }
    pic_drpa (p,sel,addr|ES_CONTROL);
    pic_drpw (p,sel,k8?0x0300:0xE300);
  }}}
  pic_drpa (p,sel,addr|ES_CONTROL);
  pic_drpw (p,sel,k8?0x0100:0xE000); /* leave K8 eyescan enabled for gen2 reset issue */
  pic_drpc (p,sel);
  telapse=pic_time()-telapse;
  vprint("EyeScan Complete: time=%f errs=",telapse);
  for (lane=0; lane<lanes; lane++) vprint("%d,",errco[lane]);
  vprint("\n");
  return len;
}

/*
  Lower Word
  0x000n - read port and write
  0x0m00 - extra write port mask 

  Upper Word
  0x00r0 - clock rate 100/(r+1)/2
  0x1000 - drive jtag connector
  0xi000 - 2 bit instruction register depth
  0x8000 - disconnect and wait before action

  Currently r=0 : 50
  Currently r=1 : 25
  Currently r=2 : 16
  Currently r=3 : 12.5
  Currently r=4 : 10
  Currently r=5 : 9
  Currently r=6 : 8
  Currently r=7 : 6.25
  Currently r=9 : 5

  Currently i=0 : 6
  Currently i=1 : 10
  Currently i=2 : 14
  Currently i=3 : 18
*/

int_4 getJTAGsel (PICSTRUCT *p, int_4 port) {
  int_4 sel=0,i=0,j=0;
  
  switch (port) {
    case 0: sel=0x1010; i = 2; break;	/* Card FPGA Virtex-2Pro|4 */
    case 1: sel=0x0041;        break;	/* IOM 1 */
    case 2: sel=0x0042; j = 1; break;	/* IOM 2 */
    case 3: sel=0x0341;        break;	/* IOM 1&2 */
    case 4: sel=0x1040;        break;	/* Card EPROM */
    case 5: sel=0x1098;        break;	/* Card Temperature Sensor */
    case 7: sel=0x0403;        break;	/* PM 1 and 3 */
    case 8: sel=0x0804; j = 1; break;	/* PM 2 and 4 */
    case 9: sel=0x0049;        break;	/* IOM 3 on Tray */
    case 10: sel=0x1010; i = 2; break;	/* PM 0 Card FPGA */
    case 11: sel=0x0003;        break;	/* PM 1 */
    case 12: sel=0x0004; j = 1; break;	/* PM 2 */
    case 13: sel=0x0005; j = 2; break;	/* PM 3 */
    case 14: sel=0x0006; j = 3; break;	/* PM 4 */
    case 15: sel=0x0C03;        break;	/* PM 1&2 and 3&4 */
    default: print("Illegal JTAG port=%d\n",port); break;
  }
  if (port==0 && p->k7) i=0;	/* Card FPGA Kintex-7 */
  if (port==0 && p->k7==8) sel = 0x1030;
  if (port==0 && p->k7==8 && p->rev==0x0A) i=3; 
  if (port>=1 && port<=3) { /* IOM */
    i = isIOMx(p,j+1)? 0:1;
    if ((abs(p->mtype[0])==IOMT_DXSNT)    && (p->mrev[0]==7)) sel = (sel & 0xFFFFFF0F) | 0x00000010; /* 25MHz */
    if ((abs(p->mtype[0])==IOMT_DXSNT)    && (p->mrev[0]==6)) sel = (sel & 0xFFFFFF0F) | 0x00000010; /* 25MHz */
    if ((abs(p->mtype[0])==IOMT_DXTGSDDS) && (p->mrev[0]>=2)) sel = (sel & 0xFFFFFF0F) | 0x00000010; /* 25MHz */
    if ((abs(p->mtype[0])==IOMT_DXTGVITA) && (p->mrev[0]==1)) sel = (sel & 0xFFFFFF0F) | 0x00000010; /* 25MHz */
    if ((    p->mtype[0] ==IOMT_A2D)      && (p->mrev[0]==20)) { sel = (sel & 0xFFFFFF0F) | 0x00000010; i=3; } /* 25MHz - JTag */
  }
  if (port>=10 && port<=15) { /* PM */
         if (p->pmtype[j]==PMT_S6M)   { sel |= 0x0020;        } /* 16 MHz, spec=33, unstable at 25 */
    else if (p->pmtype[j]==PMT_DTDM)  { sel |= 0x0020; i = 1; } /* 16 MHz, spec=33, unstable at 25 on PIC4 */
    else if (p->pmtype[j]==PMT_DTDMX) { sel |= 0x0010; i = 2; } /* 25 MHz, spec=33 */
    else if (p->pmtype[j]==PMT_A8M)   { sel |= 0x0010; i = 3; } /* 25 MHz, spec=33 */
    else if (p->pmtype[j]==PMT_K8M)   { sel |= 0x0010;        } /* 25 MHz, spec=33 */
    else if (p->pmtype[j]==PMT_K8P)   { sel |= 0x0010;        } /* 25 MHz, spec=33 */
    else    /* ZPPM/V5M/V6M */        { sel |= 0x0000; i = 1; } /* 50 MHz, spec=66 */
  }
  sel = (i<<29)|(sel<<16);
#if JTAGMONITOR
  sel |= 0x10000000;
#endif
  return sel;
}

int_4 getJTAGcbs (int_4 sel) {
  int_4 icl = (sel>>27)&0xC;
  int_4 crb = (icl==0xC)? 10 : icl + 6; /* control register bits - special Altera case=C */
  return 28 - crb;   			/* control register bit shift */
}

int_4 pic_jtag (PICSTRUCT *p, int_4 port, int_4 cmd, int_4 *data, int_4 bits, int_4 delay) {
  int_4 i,n,jcbs,sel,merge,msk,arst=0,stat=0;
  if (port>=0);
  else if (p->isX) port=JTAG_PORT_PMOFF+1;
  else if (p->isY) port=0;
  sel = getJTAGsel(p,port);
  jcbs = getJTAGcbs(sel);
  merge = (cmd&0xFFFF0000) == 0x10000;
  if (merge) { cmd &= 0xFFFF; merge=9; }
  else merge = (cmd>>28)&0xF;
  v3print("Jtag p=%d cmd=%08x bits=%d delay=%d\n",port,cmd,bits,delay);
  if (port==2 && p->mtype[1]==IOMT_DIODE) port=-4; /* EPROM on DIODE module */
  pic_lock (p, LOCK_ALLOC);
  if (p->isY && port==0) {		/* Virtex base board VIRTEX */
    if (p->rev == 0x0A) {
      switch (cmd) {
        case JTAG_CMD_IDCODE: cmd=JTAG_STXCMD_IDCODE; break;
        case JTAG_CMD_USER1: 
          /* USER1 Commands directed to SPI Flash */
          pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  JTAG_STXCMD_USER1<<jcbs, JTAG_MSK_CMD|sel); 
          pic_acmd (p, ADON_JTAG, JTAG_TMS_SET16, (0x2000)<<2, JTAG_MSK_SET16|sel); 
          cmd=JTAG_STXCMD_USER0;
          break;
        case JTAG_CMD_USER2: 
          /* USER1 Commands directed to SPI Flash */
          pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  JTAG_STXCMD_USER1<<jcbs, JTAG_MSK_CMD|sel); 
          pic_acmd (p, ADON_JTAG, JTAG_TMS_SET16, (0x1000)<<2, JTAG_MSK_SET16|sel); 
          cmd=JTAG_STXCMD_USER0;
          break;
      }
    }
    if (merge&1) pic_acmd (p, ADON_JTAG, JTAG_TMS_XCMD, cmd<<jcbs, JTAG_MSK_CMD|sel); 
    else         pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  cmd<<jcbs, JTAG_MSK_CMD|sel); 
  } 
  else if (p->isY && port==5) {		/* Virtex base board TMP122 Temperature sensor */
    *data = pic_acmd (p, ADON_JTAG, 0xFF0000FF, 0x00000000, JTAG_MSK_GET32X|sel); 
    bits=0; /* already done */
  } 
  else if (p->isY && abs(port)==4) {		/* Virtex base board EPROM */
    jcbs = (port<0)? 28-8 : 28-16;
    if (cmd==JTAG_CMD_IDCODE) cmd = 0x00FE;
    if (cmd==JTAG_CMD_UIDCODE) cmd = 0x00FD;
    if (cmd==0) { 
      stat=pic_acmd (p, ADON_JTAG, JTAG_TMS_RSTC, -1, JTAG_MSK_RSTC|sel|0xF00); 
      goto DONE;
    }
    msk = (merge!=0)? JTAG_FRTIMASK : -1;
    if (delay<=0) {
      if (bits==0) stat=pic_acmd (p, ADON_JTAG, msk&JTAG_TMS_CMDR, cmd<<jcbs, JTAG_MSK_CMD|sel|0xF00); 
      else         stat=pic_acmd (p, ADON_JTAG, msk&JTAG_TMS_CMD,  cmd<<jcbs, JTAG_MSK_CMD|sel|0xF00);
    } else {
      stat=pic_acmd (p, ADON_JTAG, msk&JTAG_TMS_CMDR,  cmd<<jcbs, JTAG_MSK_CMD|sel|0xF00); 
      udelay(delay);
      if (bits!=0) pic_acmd (p, ADON_JTAG, JTAG_TMS_R2SD, 0, JTAG_MSK_R2SD|sel); 
    }
    if (bits==0) goto DONE;
  } 
  else if (port>=JTAG_PORT_PMOFF) {	/* Virtex based Processor module */
    if (isPMx(p,port-JTAG_PORT_PMOFF)) {
      if (cmd==JTAG_CMD_SYSMON) sel |= 0x00200000; /* cant operate at 50MHz */
    } else {
      if (cmd==JTAG_CMD_IDCODE) cmd = JTAG_STXCMD_IDCODE;
      if (cmd==JTAG_CMD_UIDCODE) cmd = JTAG_STXCMD_UIDCODE;
    }
    if (merge&1) pic_acmd (p, ADON_JTAG, JTAG_TMS_XCMD, cmd<<jcbs, JTAG_MSK_CMD|sel); 
    else         pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  cmd<<jcbs, JTAG_MSK_CMD|sel); 
  }
  else if (port>=1 && port<=3) { 	/* I/O Modules */
    if (isIOMx(p,port)) {		/* Spartan based */
      if (merge&1) pic_acmd (p, ADON_JTAG, JTAG_TMS_XCMD, cmd<<jcbs, JTAG_MSK_CMD|sel); 
      else         pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  cmd<<jcbs, JTAG_MSK_CMD|sel); 
    } else {				/* Stratix based IO module */
      if (cmd==JTAG_CMD_IDCODE) cmd = JTAG_STXCMD_IDCODE;
      if (cmd==JTAG_CMD_UIDCODE) cmd = JTAG_STXCMD_UIDCODE;
      pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD, cmd<<jcbs, JTAG_MSK_CMD|sel); 
    }
  }
  else if (jtagfd!=NULL) {		/* USB recovery cable */
    if (merge&1) pic_acmd (p, ADON_JTAG, JTAG_TMS_XCMD, cmd<<jcbs, JTAG_MSK_CMD|sel); 
    else         pic_acmd (p, ADON_JTAG, JTAG_TMS_CMD,  cmd<<jcbs, JTAG_MSK_CMD|sel); 
  }
  else {		/* Unsupported jtag port */
    print("Unsupported JTAG port=%d\n",port);
  }
  if (jtagfd!=NULL && abs(bits)>128) {
    n = (abs(bits)-32)/32; if (bits<0) n=-n;
    if (bits>0) extjtag (p,ADON_PGMC, JTAG_TMS_SET32X, n, JTAG_MSK_SET32X|sel, data);
    else        extjtag (p,ADON_PGMC, JTAG_TMS_GET32X, n, JTAG_MSK_GET32X|sel, data);
    bits-=(n*32); data+=abs(n);
  }
  for (;bits>256; bits-=(n*32), data+=n)  {
    n = min(1024,bits-32)/32;
    pic_transfer (p, data, n*4, DMA_INT_ADDR, 1, 1);
    pic_acmd (p, ADON_JTAG, -11, DMA_INT_ADDR, sel|n); 
  }
  for (;bits<-256; bits+=(n*32), data+=n)  {
    n = min(1024,-bits-32)/32;
    pic_transfer (p, data,   4, DMA_INT_ADDR, 1, 0);	/* preload possible instructions before read */
    pic_acmd (p, ADON_JTAG, -12, DMA_INT_ADDR, sel|n); 
    pic_transfer (p, data, n*4, DMA_INT_ADDR, -1, 1);
  }
  for (;bits>32; bits-=32, data++)  pic_acmd (p, ADON_JTAG, JTAG_TMS_SET32X, *data, JTAG_MSK_SET32X|sel); 
  for (;bits<-32; bits+=32, data++) *data = pic_acmd (p, ADON_JTAG, JTAG_TMS_GET32X, *data, JTAG_MSK_GET32X|sel); 
  arst = (merge==0) && (bits>-32) && (bits<32) && (bits!=0);
  if (bits==0);
  else if (bits==  8) pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_SET8R :JTAG_TMS_SET8 ), *data, (arst?JTAG_MSK_SET8R: JTAG_MSK_SET8 )|sel); 
  else if (bits== 16) pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_SET16R:JTAG_TMS_SET16), *data, (arst?JTAG_MSK_SET16R:JTAG_MSK_SET16)|sel); 
  else if (bits== 24) pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_SET24R:JTAG_TMS_SET24), *data, (arst?JTAG_MSK_SET24R:JTAG_MSK_SET24)|sel); 
  else if (bits== 32) pic_acmd (p, ADON_JTAG,                       JTAG_TMS_SET32 , *data,                       JTAG_MSK_SET32 |sel); 
  else if (bits== -8) *data = pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_GET8R :JTAG_TMS_GET8) , *data, (arst?JTAG_MSK_GET8R :JTAG_MSK_GET8 )|sel); 
  else if (bits==-16) *data = pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_GET16R:JTAG_TMS_GET16), *data, (arst?JTAG_MSK_GET16R:JTAG_MSK_GET16)|sel); 
  else if (bits==-24) *data = pic_acmd (p, ADON_JTAG, (arst?JTAG_TMS_GET24R:JTAG_TMS_GET24), *data, (arst?JTAG_MSK_GET24R:JTAG_MSK_GET24)|sel); 
  else if (bits==-32) *data = pic_acmd (p, ADON_JTAG,                       JTAG_TMS_GET32 , *data,                       JTAG_MSK_GET32 |sel); 
  else print("Bad number of bits=%d in pic_jtag\n",bits);
  if (merge==9) pic_acmd (p, ADON_JTAG, JTAG_TMS_SD2R, 0, JTAG_MSK_SD2R|sel); 
  else if (merge&2);
  else if (arst);
  else pic_acmd (p, ADON_JTAG, JTAG_TMS_RST, 0, JTAG_MSK_RST|sel); 
  DONE:
  pic_lock (p, LOCK_FREE);
  return stat;
}


int_4 pic_tuner (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 stat=-1,dmac,n,m,pmod,pmport,ovsr; real_8 f;

  if (p->tflags&FLG_PRESAMP && p->res_ratio!=0 && (flags&FLG_BACKEND)==0) {
    ovsr = (port%2==0)? p->ovsr[1]:p->ovsr[0];
    if (ovsr==0) ovsr=1;
    if (p->tflags&FLG_POVSR) fcny *= ovsr;
    if (p->mbits<0) fcny = fcny/p->res_ratio + 0.5;
    else fcny = (fcny-.5)/p->res_ratio + 0.5;
    if (p->tflags&FLG_POVSR) fcny /= ovsr;
    rate = rate*p->res_ratio + 0.5;
  }

  dmac = p->ndmac+port;
  if (p->tflags&FLG_TBANK) { 
    pmod = port/10;
    pmport = port%10;
    if (p->isY) dmac = ((pmod==0)?2:6+2*pmod) + ((pmport==2)?2:1);
    else dmac = lport2dmacx[p->linktbl[(pmport==2)?2:1]];
  }

  if ((p->tflags&FLG_TBANK) && (p->chan==0) && (flags&FLG_FTUN)) {  /* loop over all channels in bank */
    m = (p->fttm>1)? p->m1 : p->nchan;
    if (pmport<3) m /= 2;
    if (pmport==4) m = p->m2;
    for (stat=0,p->chan=1; p->chan<=m && stat>=0; p->chan++) {
      f =  fcny+(p->chan-1)*p->dfreq*p->m2*p->m3;
      if (p->gchip == GC4016) stat = pic_tuner_gc4016 (p, port, bits, rate, f, dec, gain, flags);
      if (p->gchip == GCFPGA) stat = pic_tuner_gcfpga (p, port, bits, rate, f, dec, gain, flags);
    }
    p->chan = 0;
  }
  else if ((p->tflags&FLG_TBANK) && (p->fttm>1) && (flags&FLG_FTUN)) { 	/* by full channel number ? */
    m = p->chan;
    p->chan = ((p->chan-1)/(p->m2*p->m3)) + 1;
    f =  fcny - (m-1-(p->chan-1)*p->m2*p->m3)*p->dfreq;
    if (p->gchip == GC4016) stat = pic_tuner_gc4016 (p, port, bits, rate, f, dec, gain, flags);
    if (p->gchip == GCFPGA) stat = pic_tuner_gcfpga (p, port, bits, rate, f, dec, gain, flags);
    p->chan = m;
  }
  else if (p->type == ICEPIC2) { 
    stat = pic_tuner_hsp50016 (p, port, bits, rate, fcny, dec, gain, flags);
    dmac = port;
  }
  else if (p->type == ICEPIC3) {
    stat = pic_tuner_ad6620 (p, port, bits, rate, fcny, dec, gain, flags);
    dmac = lport2dmac[p->linktbl[port]];
  }
  else if (p->gchip == GC4014) {
    stat = pic_tuner_gc4014 (p, port, bits, rate, fcny, dec, gain, flags);
  }
  else if (p->gchip == GC4016) {
    stat = pic_tuner_gc4016 (p, port, bits, rate, fcny, dec, gain, flags);
    if (port<0 && p->mchan>32) /* global reset of 2nd module */
      stat = pic_tuner_gc4016 (p, -2, bits, rate, fcny, dec, gain, flags);
  }
  else if (p->gchip == GCFPGA) {
    stat = pic_tuner_gcfpga (p, port, bits, rate, fcny, dec, gain, flags);
    if (p->isDUC) dmac = p->ndmac + port;
    else if (p->pmi==0 && port<=2) dmac = 2+port;
  }

  n = dmac-1;
  if (stat>=0 && port>0 && n<DMA_CHNS && (flags&FLG_BACKEND)==0) {
    if (flags&FLG_DTUN) {
      p->dma[n].dec = dec; pic_stkupd (p, &p->dma[n].dec, 1, 1);
      m = p->dma[n].master-1; if (m>=0) {
        p->dma[m].dec = dec; pic_stkupd (p, &p->dma[m].dec, 1, 1); }
    }
    if (flags&FLG_GTUN) {
      p->dma[n].gain = gain; pic_stkupd (p, &p->dma[n].gain, 1, 1);
    }
    if (flags&FLG_FTUN) { 
      p->dma[n].fcny = fcny2phinc(fcny); pic_stkupd (p, &p->dma[n].fcny, 1, 1);
    }
  }
  return stat;
}


int_4 pic_tuner_hsp50016 (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i, j, n, oform, ospec, obits, hdfgain, mode, twos, ovsr;
  int_4 slotlen, hdfshift, cdec, hdfdec, phaseoff, gain1, gain2, multi;
  int_u4 sphaseinc, dphaseinc, ephaseinc;
  int_4 portmask, portmodule, status, cdecmin, cdecmax, factor;
  int_4 i1=1, i2=7, needclock=0, needmulti=0, mcfg1,mcfg2;
  double gaincomp, gainfine;
  struct { int_4 h,l; } cw[8], cw6[DMA_CHNS];

  if (flags&FLG_RTUN) return 0;

  ovsr = (port&0x1)? p->ovsr[0] : p->ovsr[1];
  if (ovsr > 1) {
    if ((p->tflags&FLG_POVSR)!=0) rate = rate/ovsr;
    else { dec = dec*ovsr; fcny = fcny/ovsr; }
    gain = gain + (int)(log2((double)ovsr)*6+.5);
  }
  if (ovsr>0) {
    if (rate*ovsr<=p->tun_clk);
    else print("Err: HSP50016 Tuner Oversampling=%d not supported at %d Hz input rate\n",ovsr,rate);
  }

  if (p->type == ICEPIC2) {
    if (port < 0) port = 3;	/* all channels */
    n = port-1;
    portmask = port;
    portmodule = port;
    multi = 0;
  }

  if (flags&FLG_DISABLE) { 
    /* just tri-state all of the outputs */
    i1=i2=6; 
    cw[6].h = 0xD81C0; 
    cw[6].l = 0x02001;
    goto LOAD; 
  }

  /* validate tuner frequency (fc/fs * 2^33) = (fc/ny * 2^32) */
  if (fcny<-1.0 || fcny>1.0) { 
    print("Err: Tuner Freq %f out of range [0-1]\n",fcny); return (-1); }
  sphaseinc = (int_u4)( fcny*4294967296.0 + .499999);
  ephaseinc = sphaseinc; 
  dphaseinc = 0;

  /* frequency update only */
  if (flags&FLG_FTUN) {
    i1=i2=1;
    cw[1].h = 0x30000 | (sphaseinc>>16);		/* word 1 + update */
    cw[1].l = 0x00001 | (sphaseinc&0xFFFF)<<4;		/* CW Mode */
    goto LOAD;
  }

  /* validate the decimation factor */
  cdecmin = 64+8;
  cdecmax = 131072;
  if (p->mchan>2) cdecmin = (64+8+4) * (p->mchan/2);
  cdec = dec*2;  /* from normal to complex view */
  if (cdec<cdecmin || cdec>cdecmax) { 
    print("Err: Tuner dec=%d out of range [%d-%d]\n",dec,cdecmin/2,cdecmax/2); 
    return (-1); 
  }
  if ( (cdec&0x3) != 0) { 
    cdec &= 0xFFFC; print("Warn: Tuner decimation rounded to %d\n",cdec/2); 
  }

  /* input format (offset/twos compliment) */
  twos=1;

  /* validate output format (negative bits = complex) */
  if (bits<0) { mode=2; ospec=0x0; }
  else        { mode=1; ospec=0x1; }
  obits = abs(bits);
  if      (obits ==  16) oform=0x0; 
  else if (obits ==  32) oform=0xE; 
  else { print("Err: Tuner Illegal output format bits\n"); return (-1); }

  /* calculate automatic parameters */
  gain1 = (gain+6000+5)/6 - 1000; 
  gain2 = gain - (gain1*6); 
  hdfdec = (cdec>>2);  /* complex/real output shift */
  factor = (int)( 5.0*log2((double)hdfdec) +.999 );
  gaincomp = powi(2,factor) / powi(hdfdec,5);
  gaincomp = 1; /* NOT compensate for non-pow2 loss causes spur at fs/4 */
  gainfine = pow( 10.0, (double)gain2 * 0.5 * 0.1);
  hdfgain  = max(1, min(0xFFFF, (int)((gaincomp*gainfine)*0x8000) ));
  hdfshift = 75 - factor + gain1;

  vprint("HDF shift=%08x gain=%08x %f %f\n",
				hdfshift,hdfgain,gaincomp,gainfine);
  slotlen = (obits+2)*2 + 1;
  if (multi>0) slotlen = max(slotlen,cdec/2/multi);
  phaseoff = 0;

  /* build the control words h=(39-20), l=(19-0) */
  cw[0].h = 0x10000;	/* update control words */
  cw[0].l = 0x00000;
  cw[1].h = 0x20000 | (sphaseinc>>16);
  cw[1].l = 0x00009 | (sphaseinc&0xFFFF)<<4;	/* CW Mode with TEST */
  cw[2].h = 0x40000 | (ephaseinc>>20);;
  cw[2].l = 0x00000 | (ephaseinc&0xFFFFF);
  cw[3].h = 0x60000 | (slotlen&0x3FFF)>>2;
  cw[3].l = 0x00000 | (phaseoff&0x3FFFF) | (slotlen&0x3)<<18;
  cw[4].h = 0x80000 | (dphaseinc&0xFFFFFF)>>13 | ospec<<11;
  cw[4].l = 0x00000 | (dphaseinc&0x1FFF)<<7 | hdfshift<<1;
  cw[5].h = 0xA0000 | ((hdfdec-1)&0x8FFF)<<1 | (hdfgain&0x8000)>>15;
  cw[5].l = 0x00001 | (hdfgain&0x7FFF)<<5 | oform<<1;  /* MSB 1st */
  cw[6].h = 0xC81D1;  /* I&Q=I, Cont 50% IQCLK, fall edge, IQSTRB pre */
  cw[6].h = 0xC81D9;  /* I&Q=I, Cont 50% IQCLK, fall edge, IQSTRB pre/actLow */
  cw[6].l = 0x20001 | (twos<<13);  /* enb I output, IQCLK=CLK/2 */
  cw[7].h = 0xE0000;
  cw[7].l = 0x00002;  /* wait for ram full */

  /* decimation or gain update only */
  if ( (flags&FLG_DTUN)!=0 || (flags&FLG_GTUN)!=0 ) {
    i1=4;i2=5;  cw[5].h |= 0x10000; /* add update */
    if (multi<=0 || (flags&FLG_DTUN)==0) goto LOAD; 
    /* we must change decimation for all multi-channels at the same time */
  }

  if (multi>0) { 
    needmulti = 1; 	/* these control parameters affect all channels */
    cw[6].h = 0xC81C3;  /* I&Q=triI, triIQCLK falling edge, IQSTRB pre */
    cw[6].h = 0xC81CB;  /* I&Q=triI, triIQCLK falling edge, IQSTRB pre/actLow */
    cw[6].l = 0x60001 | (twos<<13);  /* enb I output, IQCLK=CLK/2 */
    portmask = 0;
    for (j=0; j<multi; j++) {
      cw6[j].l = cw[6].l;
      cw6[j].h = cw[6].h | (j<<9);
      if (j==0) cw6[j].h |= 0x00010;	/* enable clock */
      if (portmodule&0x1) portmask |= p->tadrtbl[j+j+1];
      if (portmodule&0x2) portmask |= p->tadrtbl[j+j+2];
    }
  }
  p->ntaps = 121;
  needclock = 1;	/* and must be synchronized by a reset */

  LOAD:

  pic_lock (p, LOCK_ALLOC);

  if (needclock==0) {			/* really, are you sure ? */
    status = pic_acmd (p, ADON_RD, DMA_GMCFG+portmodule, 0, 0);
    if (!(status&IOM_ENA)) needclock = 1;
  }

  /* set Module bits to enable BPCLK to DDC */
  if (needclock) {
    if (port&0x1) mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+1, 0, 0);
    if (port&0x1) pic_acmd (p, ADON_MOD, 1, mcfg1|IOM_TEN, 0); 
    if (port&0x2) mcfg2 = pic_acmd (p, ADON_RD, DMA_GMCFG+2, 0, 0);
    if (port&0x2) pic_acmd (p, ADON_MOD, 2, mcfg2|IOM_TEN, 0); 
  }

  /* send the control words */
  for (i=i1; i<=i2; i++) {
    vprint("DDC word#%d h=%08x l=%08x portmask=%08x\n",
			i,cw[i].h,cw[i].l,portmask); 
    pic_acmd (p, ADON_DDC, portmask, cw[i].h, cw[i].l);  
  }

  /* send multi-channel specific control words */
  if (needmulti) {
    for (j=0; j<multi; j++) {
      portmask = 0;
      if (portmodule&0x1) portmask |= p->tadrtbl[j+j+1];
      if (portmodule&0x2) portmask |= p->tadrtbl[j+j+2];
      vprint("DDC word#%d h=%08x l=%08x portmask=%08x\n",
			6,cw6[j].h,cw6[j].l,portmask); 
      status = pic_acmd (p, ADON_DDC, portmask, cw6[j].h, cw6[j].l);  
      if (status != portmask) print("Hey %x %x\n",status,portmask);
    }
  }

  /* assert RESET, clear Module bits to disable BPCLK to DDC, lift RESET */
  if (needclock) {
    pic_acmd (p, ADON_WRIOC, IOC_TCS|portmodule, IOC_DT_RS|IOC_DT_RSEN,0); 
    if (port&0x1) pic_acmd (p, ADON_MOD, 1, mcfg1, 0); 
    if (port&0x2) pic_acmd (p, ADON_MOD, 2, mcfg2, 0); 
    pic_acmd (p, ADON_WRIOC, IOC_TCS|portmodule, 0|IOC_DT_RSEN, 0); 
  }

  pic_lock (p, LOCK_FREE);

  return 0;
}

#define GC4014_SM   0x00
#define GC4014_DM   0x01
#define GC4014_DB0  0x02
#define GC4014_DB1  0x03
#define GC4014_SC   0x04
#define GC4014_CG   0x05
#define GC4014_OF   0x06
#define GC4014_OM   0x07
#define GC4014_BC   0x08
#define GC4014_CFC  0x09
#define GC4014_CB0  0x0A
#define GC4014_CB1  0x0B
#define GC4014_TM   0x0C
#define GC4014_PM   0x0D
#define GC4014_STAT 0x0E
#define GC4014_CHK  0x0F
#define GC4014_FREQ 0x10
#define GC4014_PHA  0x14
#define GC4014_GAIN 0x16
#define GC4014_CC   0x17
#define GC4014_CS   0x18

int_4 pic_tuner_gc4014 (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 ic, chan, chip, tadr, ovsr;
  int_4 gain1, gain2, portmodule, mcfg;
  int_4 phaseinc, value, complex, gaincomp, cpc, pdec;
  int_4 status, cdec, decmin, decmax, scale, scaleu, scalel;
  int_4 val_pm, val_dm, val_cc;
  double dgain;

  if (flags&FLG_RTUN) return 0;
  if (port < 0) port=0; 

  /* initialize oversampling ratio */
  ovsr = (port&0x1)? p->ovsr[0] : p->ovsr[1];
  if (ovsr > 1) {
    if ((p->tflags&FLG_POVSR)!=0) rate = rate/ovsr;
    else { dec = dec*ovsr; fcny = fcny/ovsr; }
    gain = gain + (int)(log2((double)ovsr)*6+.5);
  }
  if (ovsr>0 && rate*ovsr>p->tun_clk)
    print("Err: GC4014 Tuner Oversampling=%d not supported at %d Hz input rate\n",ovsr,rate);
  if (ovsr<=0) ovsr=1;

  if (flags&FLG_ATUN) {		/* chan specific */
    chan = p->chantbl[port];
    chip = p->chiptbl[port];
  } else {			/* chip specific */
    chan = 0;
    chip = port;
  }
  tadr = p->tadrtbl[chip];
  cpc  = p->cpc;
  if (cpc==1) chan=0; 
  else if (cpc==2 && chan!=0) chan=1; 

  pdec = 2;
  val_pm = 0x40;
  if (p->tflags&FLG_NCFIR) val_pm |= 0x10;
  if (p->tflags&FLG_PFIR4) { val_pm |= 0x20; pdec=4; }
  status = 0;

  pic_lock (p, LOCK_ALLOC);

  if (flags&FLG_DISABLE) { 
    pic_acmd (p,ADON_WRIOB,tadr|GC4014_BC,0x00,0); 	/* turn off outputs */
    for (ic=0; ic<4; ic++) {
      value = val_pm|(4+ic);
      pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,value,0);
      value = 0x80; /* power down */
      pic_acmd (p,ADON_WRIOB,tadr|GC4014_CC,value,0);
    }
    goto DONE;
  }

  if (bits!=16 && bits!= -16) {
    print("Err: GC4014 Illegal output format bits\n"); 
    return -1; 
  }
  complex = (bits<0);


  /* set page to this channel */
  value = val_pm|(4+chan);
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,value,0);

  if (flags&FLG_FTUN) goto FREQONLY; 	
  if (flags&FLG_GTUN) goto DECONLY; 	
  if (flags&FLG_DTUN) goto DECONLY; 	

  /* validate tuner frequency (fc/fs * 2^32) = (fc/ny * 2^31) */
  FREQONLY:
  if (fcny<-1.0 || fcny>2.0) { 
    print("Err: Tuner Freq %f out of range [-1,2]\n",fcny); 
    status = -1; goto DONE;
  }
  phaseinc = fcny2phinc(fcny);
  if (complex) phaseinc = -phaseinc;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_FREQ,phaseinc,4);
  if (flags&FLG_FTUN) goto DONE; 	/* frequency update only */ 

  /* validate the decimation factor */
  DECONLY:
  decmin = max(16,8*cpc);
  decmax = 32768;
  if (dec<decmin || dec>decmax) { 
    print("Err: Tuner dec=%d out of range [%d-%d]\n",dec,decmin,decmax); 
    status = -2; goto DONE;
  }
  cdec = dec*4/pdec;  		/* from normal to complex view */
  if ( (cdec&0x3) != 0) { 
    cdec &= 0xFFFC; print("Warn: Tuner decimation rounded to %d\n",cdec*pdec/4); 
  }

  /* (scale+6*bigscale) <= ( 56 - 4log2(N) + log2(nzero+1) ) */
  scale = max(0,(int_4)(56 - 4.0*log2((double)cdec/4)));
  scaleu = scale/6;
  scalel = scale - scaleu*6;
  dgain = powi(cdec/4,4) * powi(2,scaleu*6+scalel-56);

  if (flags&FLG_GTUN) goto GAINONLY;		 /* dec update only */   

  /* CIC decimation */
  value = (cdec/4)-1; 
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_DB0,value,2);

  /* CIC scale control and channel gain */
  value = scaleu<<4 | scalel;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_SC,value,0);

  /* channel gain and control */
  GAINONLY:
  gain = min(47+12,gain);
  gain1 = max(0,min(7,gain/6)); 	/* COARSE gain */
  gain2 = gain - (gain1*6); 
  /* add factor of 2 for ICE/GC def of unity gain */
  gaincomp = (int_4)(32 * pow(2.0,gain2/6.0) * 2 / dgain);
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_GAIN,gaincomp&0xFF,0);
  value = (gain1&0x7)<<4; /* coarse gain */
  val_cc = value;
  if (p->type==ICEMBT2) value |= 0x1; /* B input */
  else if ((p->tflags&FLG_TALT1)==0 && (chan&0x1)!=0) value |= 0x1; /* B input SLIC3 */
  else value |= 0x2; /* C input SLIC3 */
  if (flags&FLG_ATUN) pic_acmd (p,ADON_WRIOB,tadr|GC4014_CC,value,0);

  if (flags&FLG_GTUN) goto DONE; 		/* gain update only */
  if (flags&FLG_DTUN) goto DONE;		 /* dec update only */   

  /* sync mode = SI for DEC_SYNC only */
  value = 0x01;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,value,0);

  /* decimation mode = LINK_MODE|EN_DOUBLER|REAL(opt) */
  value = 0x90; if (!complex) value |= 0x1; 
  if (p->tflags&FLG_UFILT) value |= 0x2;	/* user filter */
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_DM,value,0);
  val_dm = value;

  /* channel gain (unused bits) */
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CG,0x00,0);

  /* output format */
  value = 0x01;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_OF,value,0);

  /* output mode */
  if (cpc==1) value = 0x00; 
  else if (cpc==2) value = 0x04; 
  else value = 0x08;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_OM,value,0);

  /* blanking control */
  value = 0x40;
  if (flags&FLG_DISABLE) value = 0x0; /* turn off outputs */
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_BC,value,0);

  /* channel flush */
  value = 0x55;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CFC,value,0);

  /* counter mode */
  value = 0xff;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CB0,value,0);
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CB1,value,0);

  /* test mode */
  value = 0x0;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_TM,value,0);

  /* status */
  value = 0x0;
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_STAT,value,0);

  /* multi-channel setup */
  for (ic=0; ic<4; ic++) {
    value = val_pm|(4+ic);
    pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,value,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4014_PHA,0,2);
    value = val_cc;
    if (ic>=cpc || (flags&FLG_DISABLE)!=0) value |= 0x80; /* power down */
    if (p->type==ICEMBT2) value |= 0x1; /* B input */
    else if ((p->tflags&FLG_TALT1)==0 && (chan&0x1)!=0) value |= 0x1; /* B input SLIC3 */
    else value |= 0x2; /* C input SLIC3 */
    pic_acmd (p,ADON_WRIOB,tadr|GC4014_CC,value,0); 
    pic_acmd (p,ADON_WRIOB,tadr|GC4014_CS,0x5f,0);
  }

  /* run the clock to synchronize and flush */
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,0x03,0);
  if (port&0x1) portmodule=1; else portmodule=2;
  mcfg = pic_acmd (p, ADON_RD, DMA_GMCFG+portmodule, 0, 0);
  pic_acmd (p, ADON_MOD, portmodule, mcfg|IOM_TEN, 0); 
  /* complex mode to initialize IQ phase circuit - bug */ 
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_DM,0x80,0); 
  udelay(10000); /* insure ample clocks */
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_DM,val_dm,0);
  pic_acmd (p, ADON_MOD, portmodule, mcfg, 0);
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,0x01,0);

  DONE:
  pic_lock (p, LOCK_FREE);
  return (status);
}

int_4 pic_dump_gc4014 (PICSTRUCT *p, int_4 port) 
{
  int_4 chan, chip, tadr, value;

  if (port < 0) port=0; 
  chan = p->chantbl[port];
  chip = p->chiptbl[port];
  tadr = p->tadrtbl[chip];

  /* set page to this channel */
  value = pic_acmd (p,ADON_RDIOB,tadr|GC4014_PM,0,0);
  value = (value&0xF0)|(4+chan);
  pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,value,0);

  print("GC4014 Port=%d Chip=%d Channel=%d Parameters:\n",port,chip,chan);

  print("GC4014_SM   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_SM,0,0) );
  print("GC4014_DM   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_DM,0,0) );
  print("GC4014_DB0  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_DB0,0,2) );
  print("GC4014_SC   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_SC,0,0) );
  print("GC4014_CG   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CG,0,0) );
  print("GC4014_OF   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_OF,0,0) );
  print("GC4014_OM   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_OM,0,0) );
  print("GC4014_BC   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_BC,0,0) );
  print("GC4014_CFC  = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CFC,0,0) );
  print("GC4014_CB0  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CB0,0,2) );
  print("GC4014_TM   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_TM,0,0) );
  print("GC4014_PM   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_PM,0,0) );
  print("GC4014_STAT = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_STAT,0,0) );
  print("GC4014_CHK  = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CHK,0,0) );
  print("GC4014_GAIN = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_GAIN,0,0) );
  print("GC4014_CC   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CC,0,0) );
  print("GC4014_CS   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_CS,0,0) );
  print("GC4014_PHA  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_PHA,0,2) );
  print("GC4014_FREQ = %08x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4014_FREQ,0,4) );
  return 1;
} 

#define GC4016_GR   0
#define GC4016_SR   1
#define GC4016_PM   2
#define GC4016_CHK  3
#define GC4016_GS   4
#define GC4016_CS   5
#define GC4016_CB0  6
#define GC4016_CB1  7

#define GC4016_PHA  16
#define GC4016_FREQ 18

#define GC4016_CR   16
#define GC4016_FS   17
#define GC4016_NS   18
#define GC4016_ZP   19
#define GC4016_DS   20
#define GC4016_DB0  21
#define GC4016_CSC  23
#define GC4016_IQ   24
#define GC4016_CFIR 25
#define GC4016_PFIR 26
#define GC4016_INP  27
#define GC4016_PKC  28
#define GC4016_PKR  29
#define GC4016_FGB0 30

#define GPFIR_GAIN 1.149429
int_4 gpfir[32]={31,136,208,107,-123,-181,69,277,46,-353,
		-235,358,483,-246,-750,-24,967,470,-1046,
		-1081,884,1817,-360,-2608,-689,3365,
		2594,-3992,-6527,4407,23277,32767};
#define GCFIR_GAIN 0.989944
int_4 gcfir[11]={5,-8,-172,-192,806,1493,-1889,-6182,1085,21109,32767};

#define GPFIRW_GAIN 0.6320
int_4 gpfirw[32]={-32,-234,98,-24,-74,165,-199,138,13,-198,
		329,-323,149,145,-437,577,-456,73,445,
		-867,955,-576,-213,1138,-1778,1706,-659,
		-1331,3909,-6474,8360,32767};
#define GCFIRW_GAIN 0.9154
int_4 gcfirw[11]={47,-102,-1184,-1274,1487,3243,-1798,-7760,15,20939,32767};

#define GPFIRW_GAIN_90 1.020096
#define GCFIRW_GAIN_90 1.074203

int_4 chanfor (int_4 ic, int_4 cpc) {
  int_4 chan;
  if (cpc==4) chan=ic;
  else if (cpc==2 && ic>=2) chan=2;
  else chan=0;
  return chan;
}

int_4 index2port (PICSTRUCT *p, int_4 index) {
  int_4 port;
  if (p->cpc==4) return index;
  port = ((index-1)&0x1E)*(4/p->cpc) + ((index-1)&0x1) + 1;
  return port;
}

int_4 pic_tuner_gc4016 (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i, j, k, ic, pmi, chan, chip, tadr, cadr, portmodule, mcfg, all, ovsr;
  int_4 phaseinc, value, complex, mixedmode, itdec, gaincomp, cpc, pdec;
  int_4 status, cdec, decmin, decmax, scale, scaleu, scalel;
  int_4 decratio, cicscale, shift, coarse, *fc, ds[4], tbank=0;
  double dgain,sgain;

  if (flags&FLG_RTUN) return 0;
  if (port<0) {
    pmi = (port==-2)? 2:p->pmi;
    port = 0;
  } else if (p->tflags&FLG_TBANK) { 
    pmi = port/10;
    port = port%10;
  } else {
    pmi = (port>>16);
    port = port&0xFFFF;
    if (port>32) { port-=32; pmi=2; }
  }

  if (flags&FLG_ATUN) {		/* chan specific */
    if (p->tflags&FLG_TBANK) { 
      /* port 1&2&3 are up front, port 4 is back end */
      if (port==4) fcny = 0;
      else fcny = fcny + p->ofreq;
      if (port==3) port = index2port(p,p->chan);
      else if (port==4) port = NBANK + index2port(p,p->chan);
      else port = index2port(p, port+(p->chan-1)*2);
    }
    chan = p->chantbl[port];
    chip = p->chiptbl[port];
  }
  else {			/* chip specific */
    chan = -1;
    chip = port;
    if (p->tflags&FLG_TBANK) {
      if (port==0);
      else if (p->fttm>1) chip += 12;	/* address front end chips on this side */
      else chip += 8;			/* address all chips on this side */
      tbank=1;
    }
  }
  tadr = chip2tadr(p,chip,pmi);
  cpc  = p->cpc;
  complex = (bits<0);
  mixedmode = (p->tflags&(FLG_ITCPC|FLG_ITFMT))!=0;
  if (complex) bits = -bits;
  if (p->tflags&FLG_ITDEC) itdec=1; else itdec=0; 
  if (p->tflags&FLG_PFIR4) pdec=4; else pdec=2;
  /* if (p->tflags&FLG_NCFIR); */

  /* initialize oversampling ratio */
  ovsr = (port&0x1)? p->ovsr[0] : p->ovsr[1];
  if (p->fttm>1 && ((tbank && port==4) || port>NBANK)) ovsr=1;
  if (ovsr > 1) {
    if ((p->tflags&FLG_POVSR)!=0) rate = rate/ovsr;
    else { dec = dec*ovsr; fcny = fcny/ovsr; }
    gain = gain + (int)(log2((double)ovsr)*6+.5);
  }
  if ((p->tflags&FLG_TBANK)!=0 && (flags&FLG_DISABLE)==0 && p->fttm>1) { 
    /* yes these decimation parameters are post OVSR */
    dec = ((tbank && port==4) || (port>NBANK))? p->d2 : p->d1;
  }
  if (rate>=p->tun_clk)
    print("Err: GC4016 Tuner Input rate=%d Hz not supported with TCLK=%d Hz\n",rate,p->tun_clk);
  if (ovsr>0 && rate*ovsr>p->tun_clk) 
    print("Err: GC4016 Tuner Oversampling=%d not supported at %d Hz input rate\n",ovsr,rate);
  if (ovsr<=0) ovsr=1;

  if (bits!=16 && bits!=32) {
    print("Err: GC4016 Illegal output format bits=%d for CPC=%d\n",bits,cpc); 
    return (-1); 
  }
  if (bits==32 && itdec==0 && cpc!=1) {
    print("Err: GC4016 32-bit Output mode requires ITDEC or CPC=1\n"); 
    return (-1); 
  }
  if ((p->tflags&FLG_RESAMP)!=0 && !complex && (flags&FLG_DISABLE)==0) {
    print("Err: GC4016 Port=%d Bits=%d Resampler only supported with complex output data\n",port,bits); 
    return (-1); 
  }

  pic_lock (p, LOCK_ALLOC);

  if ((flags&FLG_DISABLE)!=0) { 
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0xF8,0); /* turn off outputs */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_GS,0x00,0); /* general syncs off */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CS,0x00,0); /* count syncs off */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CB0,0xff,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CB1,0xff,0);
    /* initialize output control */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,98<<1,0);
    pic_acmd (p,ADON_WRIOB,tadr|16,0x7D,0); 
    pic_acmd (p,ADON_WRIOB,tadr|17,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|18,0x4B,0); /* output mode - no REAL_ONLY modes */
    pic_acmd (p,ADON_WRIOB,tadr|19,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|20,0x01,0);
    pic_acmd (p,ADON_WRIOB,tadr|21,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|22,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|23,0x33221100,4);
    pic_acmd (p,ADON_WRIOB,tadr|28,0x00,0);
    /* initialize resampler pages */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,64<<1,0);
    pic_acmd (p,ADON_WRIOB,tadr|16,0x3,0); /* cpc=4 */
    pic_acmd (p,ADON_WRIOB,tadr|18,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|19,0x15,0);
    pic_acmd (p,ADON_WRIOB,tadr|20,0xE4,0); /* cpc=4 */
    pic_acmd (p,ADON_WRIOB,tadr|21,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|22,0x00,0);
    pic_acmd (p,ADON_WRIOB,tadr|23,0xE4,0);
    if (p->type==ICEPIC4) cadr=0x30; else cadr=0x80;
    pic_acmd (p,ADON_WRIOB,tadr|cadr,0x1F,1); 
  }

  status = 0;

  if (flags&FLG_FTUN) goto FREQONLY; 	
  if (flags&FLG_GTUN) goto DECONLY; 	
  if (flags&FLG_DTUN) goto DECONLY; 	

  /* validate tuner frequency (fc/fs * 2^32) = (fc/ny * 2^31) */
  FREQONLY:
  if (fcny<-1.0 || fcny>2.0) { 
    print("Err: Tuner Freq %f out of range [-1,2]\n",fcny); 
    status = -1; goto DONE;
  }
  phaseinc = fcny2phinc(fcny);
  for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==chan || tbank) {
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+6)<<1,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_FREQ,phaseinc,4);
  }
  if (cpc!=4) {  /* pulse the frequency sync */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CS,0xC0,0); /* oneshot level sync */
    udelay(10000); /* insure ample clocks */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CS,0x00,0); /* oneshot reset */
  }
  if (flags&FLG_FTUN) goto DONE; /* frequency update only */ 

  DECONLY:
  /* validate the decimation factor */
  decmin = 4*cpc;
  decmax = 2048*cpc;
  if (dec<decmin || dec>decmax) { 
    print("Err: Tuner dec=%d out of range [%d-%d]\n",dec,decmin,decmax); 
    status = -2; goto DONE;
  }
  cdec = dec*4/pdec;  		/* from normal to complex view */
  if ( (cdec&0x3) != 0) { 
    cdec &= 0xFFFC; print("Warn: Tuner decimation rounded to %d\n",cdec*pdec/4); 
  }

  if ((p->tflags&FLG_RESAMP)!=0) {
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,64<<1,0);
    p->nres = (pic_acmd (p,ADON_RDIOB,tadr|17,0x0,1) & 0x3F) + 1; 
  }
  else p->nres=7;

  if (flags&FLG_GTUN) goto GAINONLY;		 /* gain update only */   

  /* CIC decimation */
  if ((flags&FLG_DTUN)!=0) {
    for (ic=0; ic<4; ic++) if (itdec==0 || chanfor(ic,cpc)==chan) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
      ds[ic] = pic_acmd (p,ADON_RDIOB,tadr|GC4016_DS,0,1);
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x44,1);
    }
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CS,0xC0,0); /* oneshot level sync */
  } 
  
  decratio = (cdec/cpc)-1; 
  for (ic=0; ic<4; ic++) if (itdec==0 || chanfor(ic,cpc)==chan || tbank) {
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_DB0,0x7000|decratio,2);
  }

  if ((flags&FLG_DTUN)!=0) {
    udelay(10000); /* insure ample clocks */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CS,0x00,0); /* oneshot level sync */
    for (ic=0; ic<4; ic++) if (itdec==0 || chanfor(ic,cpc)==chan) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,ds[ic],1);
    }
  }  

  GAINONLY:
  /* channel gain: (shift+scale+6*bigscale) <= ( 62 - 5log2(N) + log2(nzero+1) ) */
  if (cpc==1) cdec *= 2; 
  shift=4; if (cdec>3100) shift=3;
  scale = max(0,(int)(62 - shift - 5.0*log2((double)cdec/4)));
  if (gain>=12) coarse = min(7,(gain-6)/6); else coarse = 0;
  scale = min(scale,6*7+5);
  scaleu = scale/6;
  scalel = scale - scaleu*6;
  dgain = powi(cdec/4,5) * powi(2,shift+coarse+scaleu*6+scalel-62);
  cicscale = 0x40 | (scaleu<<3) | scalel; /* round 20 bits */

  sgain = 1.0;
  /* compensate for CFIR gain */
  if (p->fttm>1 && port!=4) sgain *= GCFIRW_GAIN_90;
  else if ((p->tflags&FLG_UCFIR)!=0) sgain *= p->cfir_gain;
  else if (cpc==1) sgain *= GCFIRW_GAIN;
  else sgain *= GCFIR_GAIN;

  /* compensate for PFIR gain */
  if (p->fttm>1 && port!=4) sgain *= GPFIRW_GAIN_90;
  else if ((p->tflags&FLG_UFILT)!=0) sgain *= p->pfir_gain;
  else if (cpc==1) sgain *= GPFIRW_GAIN;
  else sgain *= GPFIR_GAIN;

  /* compensate for resampler filter gain */
  if (p->fttm>1 && port!=4);
  else if ((p->tflags&FLG_RESAMP)!=0) sgain *= p->res_gain;

  sgain = 2.0/sgain; /* add factor of 2 for ICE/GC def of unity gain */
  gaincomp = (int_4)(1024 * pow(2.0,gain/6.0) * sgain / dgain);
  for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==chan || tbank) {
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CSC,cicscale,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_FGB0,min(gaincomp,0x3FFF),2);
    value = coarse<<4; if (cpc==1 && complex==0 && (ic&1)==0) value |= 0x2;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CFIR,value,0);
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CR,0x08|shift,0);
  }


  if (flags&FLG_GTUN) goto DONE; 				/* gain update only */
  if ((flags&FLG_DTUN)!=0 && mixedmode==0) goto DONE;		 /* dec update only */   

  /* FPGA data massage register */
  if (p->type==ICEPIC4) cadr=0x30; else cadr=0x80;
  if (bits==32) value=0xE0; else value=0x60;
  pic_acmd (p,ADON_WRIOB,tadr|cadr,value,1); 

  /* input data type register (just chip 1 and 2) */
  if (abs(p->mbits)==8) value=0x1; else value=0x0;
  if (p->isY) pic_wpm (p,pmi,PPC_TMODEX|((tadr>>6)&0xC),&value,4,FLG_PPC_BUS);

  /* common output page parameters */
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,98<<1,0);
  if (itdec && bits!=32) value=0x08; else value=0x00;
  pic_acmd (p,ADON_WRIOB,tadr|17,value,0);
  if (bits==32) value=0x08; else value=0x01;
  pic_acmd (p,ADON_WRIOB,tadr|20,value,0);
  if (cpc==1) i=0; else if (cpc==2) i=0x22002200; else i=0x33221100;
  pic_acmd (p,ADON_WRIOB,tadr|23,i,4);

  /* common resampler page parameters */
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,64<<1,0);
  pic_acmd (p,ADON_WRIOB,tadr|16,cpc-1,0);
  if (bits==32 && itdec) value=0x75; 
  else if (bits==32) value=0x35; 
  else if (itdec) value=0x05; 
  else value=0x15;
  pic_acmd (p,ADON_WRIOB,tadr|19,value,0);
  if (cpc==1) value=0x00; else if (cpc==2) value=0x50; else value=0xE4; 
  pic_acmd (p,ADON_WRIOB,tadr|20,value,0);

  /* initialize resampler coefficients (for bypassing) */
  if ((p->tflags&FLG_RESAMP)==0 || (flags&FLG_DISABLE)) {
    pic_acmd (p,ADON_WRIOB,tadr|17,0x46,0); /* active nmult was set in loadfc() */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,65<<1,0);
    for (ic=0;ic<4;ic++) pic_acmd (p,ADON_WRIOB,(tadr|0x10)+ic*4,0x04000000,4);
    value=1024; 
    for (i=32;i<64;i++) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,i<<1,0);
      for (j=16;j<32;j+=4) { pic_acmd (p,ADON_WRIOB,tadr|j,value,4); value=0; }
    }
  } else if ((p->tflags&FLG_URFIR)==0) {
    i = pic_acmd (p,ADON_RDIOB,(tadr&0xFFFF0300)|17,0,0); /* read active nmult was set in loadfc() */
    j = (port==0)?3:(port&1)?1:2; if (pmi>0) j += 100+(pmi*10);
    k = findintflagdef("NDELAY",p->config,32);
    if ( (i&0x40)!=0 ) pic_loadfile (p, (k==64)?"res_8x64_80": (k==16)?"res_19x16_80":"res_15x32_80", FLG_FC|j); 
  }

  /* multi-channel setup */
  all = (!itdec||tbank)? 1 : 0;
  for (ic=0; ic<4; ic++) {
    /* mixed FMT can operate on individual channels within a chip, mixed CPC cannot */
    if (mixedmode!=0 && flags!=0 && (p->tflags&FLG_ITCPC)==0 && chanfor(ic,cpc)!=chan) continue;

    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
    if (cpc==4) value=0x77; else value=0x47;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_FS,value,0);	/* freq|phase */
    if (cpc==4) value=0x00;  				/* Split I/Q */
    else if (ic&1) value=0x50; else value=0x30;
    if (complex); 
    else if (cpc==4) value|=0x06;			/* NEG_CTL */
    else if (cpc==2) { if (ic&1) value|=0x0A; else value|=0x05; }
    else if (cpc==1) { if (ic==1||ic==2) value|=0x0F; }
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_IQ,value,0);	
    if (cpc==1 && ic<2) value=0x02;
    else if (complex);
    else if (cpc==2 && (ic&1)==0) value=0x02;
    else if (cpc==4) value=0x04;
    else value=0x00;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PFIR,value,0);

    if (mixedmode!=0 && flags!=0 && (p->tflags&FLG_ITCPC)==0) continue;

    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+6)<<1,0);
    if (cpc!=4 && (ic&0x1)!=0) value=0x4000; else value=0x0000;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PHA,value,2);

    if (cpc==1) fc=gcfirw; else fc=gcfir;
    if ((p->tflags&FLG_UCFIR)==0) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+0)<<1,0);
      for (j=0;j<8;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),fc[j],2); 
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+1)<<1,0);
      for (j=0;j<3;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),fc[j+8],2); 
    }
    if (cpc==1) fc=gpfirw; else fc=gpfir;
    if ((p->tflags&FLG_UFILT)==0) for (i=0;i<4;i++) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+2+i)<<1,0);
      for (j=0;j<8;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),fc[j+i*8],2);
    }
    if (mixedmode!=0 && flags!=0) continue;

    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
    if ((p->tflags&FLG_DSYNC)!=0) value=0x72; else value=0x22;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_NS,value,0);	/* dither|nco */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_ZP,0x20,0);	/* zpad|nzero */
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x77,0);	/* flush|dec sync */
    if (all) pic_acmd (p,ADON_WRIOB,tadr|GC4016_DB0,0x7000|decratio,2);
    if (all) pic_acmd (p,ADON_WRIOB,tadr|GC4016_CSC,cicscale,0);

    if (p->type==ICEPIC4) value=0x10; else value=0x11;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_INP,value,0);	/* input port */
    if (all) pic_acmd (p,ADON_WRIOB,tadr|GC4016_FGB0,gaincomp&0x3FFF,2);
    if (flags&FLG_DISABLE) value=0x8C; else value=0x08|shift;
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_CR,value,0);
  }

  if (mixedmode!=0 && flags!=0) goto DONE;

  /* insure enabled/disabled */
  if (flags&FLG_DISABLE) value=0xF8; else value=0x08;
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,value,0);

  /* run the clock to synchronize and flush */
 if (!p->isY) {
  if (port&0x1) portmodule=1; else portmodule=2;
  mcfg = pic_acmd (p, ADON_RD, DMA_GMCFG+portmodule, 0, 0);
  pic_acmd (p, ADON_MOD, portmodule, mcfg|IOM_TEN, 0); 
  udelay(10000); /* insure ample clocks */
  pic_acmd (p, ADON_MOD, portmodule, mcfg, 0);
 }

  DONE:
  pic_lock (p, LOCK_FREE);
  return (status);
}


int_4 pic_dump_gc4016 (PICSTRUCT *p, int_4 port) 
{
  int_4 pmod, chan, chip, tadr;

  if (port<0) port=0; 
  pmod = port2pmi(p,&port); 
  chan = port2chan(p,port);
  chip = port2chip(p,port);
  tadr = chip2tadr(p,chip,pmod);

  print("GC4016 PM=%d Port=%d Chip=%d Tadr=%08x Channel=%d Parameters:\n",pmod,port,chip,tadr,chan);

  print("GC4016_GR   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_GR,0,0) );
  print("GC4016_GS   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_GS,0,0) );
  print("GC4016_CHK  = %01x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_CHK,0,1) );
  print("GC4016_CB0  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_CB0,0,2) );
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*chan+7)<<1,0);
  print("GC4016_CR   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_CR,0,0) );
  print("GC4016_FS   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_FS,0,0) );
  print("GC4016_NS   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_NS,0,0) );
  print("GC4016_ZP   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_ZP,0,0) );
  print("GC4016_DS   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_DS,0,0) );
  print("GC4016_DB0  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_DB0,0,2) );
  print("GC4016_CSC  = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_CSC,0,0) );
  print("GC4016_IQ   = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_IQ,0,0) );
  print("GC4016_CFIR = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_CFIR,0,0) );
  print("GC4016_PFIR = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_PFIR,0,0) );
  print("GC4016_INP  = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_INP,0,0) );
  print("GC4016_FGB0 = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_FGB0,0,2) );
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*chan+6)<<1,0);
  print("GC4016_PHA  = %04x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_PHA,0,2) );
  print("GC4016_FREQ = %08x\n", pic_acmd (p,ADON_RDIOB,tadr|GC4016_FREQ,0,4) );
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,64<<1,0);
  print("GC4016_NMULT= %02x\n", pic_acmd (p,ADON_RDIOB,tadr|17,0,0) );
  print("GC4016_FSR  = %02x\n", pic_acmd (p,ADON_RDIOB,tadr|19,0,0) );
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,65<<1,0);
  print("GC4016_RATIO= %08x\n", pic_acmd (p,ADON_RDIOB,(tadr|0x10)+chan*4,0,4) );
  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,98<<1,0);
  print("GC4016_TAG  = %08x\n", pic_acmd (p,ADON_RDIOB,tadr|23,0,4) );
  return 1;
} 

void applyCICcomp(int_4 dec, real_4 *filt, int_4 n) {
  real_4 alpha = (dec==2)? 0.4F : (dec==3)? 0.45F : 0.5F;
  real_4 c0 = -alpha/2, c1 = 1.0F+alpha, c2 = -alpha/2;
  int_4 i;
  /* quick and dirty 3 tap convolve */
  real_4 f0, f1 = filt[n-1], f2=filt[0], fx=f2;
  for (i=0; i<n; i++) {
    f0 = f1; f1 = f2; f2 = (i==n-1)? fx : filt[i+1];
    filt[i] = f0*c0 + f1*c1 + f2*c2;
  }
}

/* ADON_WR left shifts by 2 */
#define GCFPGA_SYS    (0<<2)
#define GCFPGA_DEC    (1<<2)
#define GCFPGA_GAIN   (2<<2)
#define GCFPGA_RMON   (3<<2)
#define GCFPGA_RATIO  (4<<2)
#define GCFPGA_OVSR   (5<<2)
#define GCFPGA_FREQ   (6<<2)
#define GCFPGA_POFF   (7<<2)
#define GCFPGA_LUT    (8<<2)
#define GCFPGA_CFIRS (10<<2)
#define GCFPGA_CFIRT (11<<2)
#define GCFPGA_PFIRS (12<<2)
#define GCFPGA_PFIRT (13<<2)
#define GCFPGA_RFIRS (14<<2)
#define GCFPGA_RFIRT (15<<2)

#define GCFPGA_SYS_ENABLE 0x01
#define GCFPGA_SYS_PLAY   0x02
#define GCFPGA_SYS_REDUCE 0x04
#define GCFPGA_SYS_MASK   0x08
#define GCFPGA_SYS_TUNER  0x10
#define GCFPGA_SYS_SPACER 0x20
#define GCFPGA_SYS_RESAMP 0x40
#define GCFPGA_SYS_DEMOD  0x80

int_4 setupTunerChip (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags, real_8 ratio)
{
  int_4 stat=0;
  if (p->tflags&FLG_RESAMP && p->nres>0) pic_loadfile (p, (p->nres==10)? "res_10x2048_100":"res_8x256_100", FLG_FC|0);
  stat = pic_tuner_gcfpga (p, port, bits, rate, fcny, dec, gain, flags);
  if (p->tflags&FLG_RESAMP && p->nres>0) pic_tuner_ratio(p,port,ratio,flags);
  printf("setupTunerChip nps=%d order=%d nres=%d pfd=%d\n",p->nps,p->order,p->nres,p->dec2);
  return stat;
}


int_4 pic_tuner_gcfpga (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i, abits, mbits, side, chip, chan=0, tadr, ovsr, sysreg, pmod,core, tqd,cf1,htf,res, fport,global,dofilt;
  int_4 phaseinc, complex, gaincomp, decratio, coarse, tclk, tbank, itdec, lut, nps=0, n4t, nrt=0, csig, order=1, ocfir, mdw=8;
  int_4 status, cdec, decmin, decmid, decmax, scale, tflags,cgchip,cpmi,ctadr,ntap,ndec,nchan,mdec;
  double dgain,sgain,fgain,ftmp,fscl,fbwf,ratio;
  float *fbuf; int_4 *lbuf;
  char fname[40];
  HALO *plan;

  status = 0;
  if (flags&FLG_RTUN) return status;
  n4t = (findintflag("N4T",p->config)>0); /* sub NOOP for TFDD */
  if (findflag("ZFSO4",p->config)>=0) fcny+=0.5;

  if (port < 0) port=1; 
  side  = (port&0x1)? 1 : 2;
  ovsr  = p->ovsr[side-1];

  if (flags==FLG_PRESAMP) {
    tflags = p->tflags; p->tflags = FLG_RESAMP;
    cgchip = p->gchip;  p->gchip = GCFPGA;
    cpmi = p->pmi;  p->pmi = 0;
    dec  = p->res_dec;
    if (dec==1) gain -= 3;
    ovsr = p->res_ovsr;
    pmod = 0;
  }
  pmod = port>>16;
  port = port&0xFF;
  chip = side;
  tbank = (p->tflags&FLG_TBANK)!=0;
  itdec = (p->tflags&FLG_ITDEC)!=0;
  nchan = 1;
  global = (pmod>0);
  if (tbank) {			/* processor multicore tuner bank */
    pmod = port/10;
    chip = port%10;
    chan = p->chan;
    if (flags&FLG_ATUN) {
      if (chip==4) fcny = 0; else fcny += p->ofreq;
    }
    if (chan<=0) tadr = (p->fttm>1 && chip==1)? MCORE_EVEN : (p->fttm>1 && chip==2)? MCORE_ODD : MCORE_ALL;
    else if (p->fttm<2 && chip==3) { tadr = ((chan-1)>>1) * MCORE_CHN; chip = (chan&1)? 1:2; }
    else if (p->fttm>1 && chip<=2) tadr = (side-1)*MCORE_CHN + (chan-1)*2*MCORE_CHN;
    else tadr = (chan-1)*MCORE_CHN;
    if (p->fttm<=1);		/* single pass same side */
    else if (chip==4) { chip=2;	ovsr=1; } /* back end is B side */
    else chip=1;                /* front end is A side */
    tadr |= chip2tadr(p,chip,pmod);
    nchan = (p->mcs>0)? p->mcs : p->nchan;
  }
  else if (global && pmod==4) {		/* base card multicore global */
    tadr = ((chip==2)? PPC_TUNB_CTL:PPC_TUNA_CTL) + MCORE_ALL;
    pmod=0; chan=-1;
  }
  else if (global) {		/* processor multicore global */
    tadr = chip2tadr(p,chip,pmod) | MCORE_ALL;
    port += (pmod-1)*32;
  }
  else if (p->pmi>0) {		/* processor multicore channel */
    core = port;
    pmod = port2pmi(p,&core);
    chan = port2chan(p,core);
    chip = port2chip(p,core);
    tadr = chip2tadr(p,chip,pmod) + (chan*MCORE_CHN);
  }	
  else if (p->pm0mt>0) {	/* base card multicore */
    chan = (port-1)>>1;
    tadr = ((chip==2)? PPC_TUNB_CTL:PPC_TUNA_CTL) + (chan*MCORE_CHN);
  }
  else if (p->isDUC) {		/* base card multicore */
    chan = (port-1)>>1;
    tadr = ((chip==2)? PPC_TUNB_CTL:PPC_TUNA_CTL) + (chan*MCORE_CHN);
  }
  else {			/* base card core */
    chip = (port==3)? 0 : port;
    tadr = pic5tadrtbl[chip];
  }
  ctadr = tadr&0xFF000000; /* the 1st multicore for readback */

  /* out for QTuner */
  if (pmod==0 && p->type==ICEPIC6 && p->socc[1+side]=='Q') {
    return pic_tuner_qtfpga (p, port, bits, rate, fcny, dec, gain, flags);
  }
  if ((flags&FLG_DISABLE)!=0) {
    pic_wpb (p,pmod,tadr|GCFPGA_SYS,0); /* put in reset */
    return 0;
  }

  /* out for DUC */
  if (p->isDUC) {
    if (p->mcs==0) p->mcs=-1;
    p->pindex = 2+side;
    plan = p->plan[chan];
    ratio = dec; if (p->res_ratio>0) ratio *= p->res_ratio;
    dgain = gain;
    if (flags&FLG_DISABLE);
    else if (plan!=0 && (flags&FLG_GTUN)!=0) status = core_setkey(plan,"D:GAIN",&dgain,8);
    else if (plan!=0 && (flags&FLG_FTUN)!=0) status = core_setkey(plan,"D:FCNY",&fcny,8);
    else status = setup_duc (p, p->pindex, chan, rate, ratio, gain, fcny, p->tsig);
    return status;
  }

  if (p->csig==0) { print("Err: No core signature loaded in the FPGA\n"); return(-1); }
  csig=p->csig; order=p->order; mdw=p->mdw; nps=p->nps; tclk=p->tclk;
  if (csig==CORE_TFDD || csig==CORE_QTFD || csig==CORE_NOOP);
  else { print("Err: Incorrect core signature=%.4s loaded in the FPGA\n",(char*)(&csig)); return(-1); }

  /* port for filter loads */
  fport = (global || tbank)?  100+(pmod*10)+chip : port;

  /* handle really WB input */
  tqd = 1;
  if (port==1 && csig==CORE_QTFD && rate>tclk) {
    for (i=dec; (rate/tqd)>tclk && tqd<=16; tqd*=2);
    if ((flags&FLG_ATUN)==0) pic_tuner_qtfpga (p, 2, -16, rate, fcny, tqd/2, -6, flags);
    rate /= tqd; dec /= tqd; gain += 6;
    if (dec*tqd != i) { print("Unable to achieve dec=%d, must be multiple of %d\n",i,tqd); return 0; }
  }

  /* initialize oversampling ratio */
  if (nps>0) tclk *= nps; 
  if (ovsr<=0) ovsr=1;
  if (p->tflags&FLG_FIRONLY) ovsr=2;
  if (ovsr > 1) {
    if ((p->tflags&FLG_POVSR)!=0) rate = rate/ovsr;
    else { dec = dec*ovsr; fcny = fcny/ovsr; }
    gain = gain + (int)(log2((double)ovsr)*6+.5);
  }
  if (tbank && p->fttm>1) { /* yes these are post OVSR */
    dec = (chip==2)? p->d2 : p->d1;
  }
  if (ovsr>0 && rate*ovsr>tclk) 
    print("Err: GCFPGA Tuner Oversampling=%d not supported at %d Hz input rate with tclk=%d\n",ovsr,rate*tqd,tclk);

  /* find number of bits in module port */
  mbits = p->mbits;
  if (p->tflags&FLG_PRESAMP) mbits = 16;
  if (flags&FLG_BACKEND) mbits = 16;
  if (tqd>1) mbits = -16;

  /* for tuners feeding an output port - reverse the I/O bits parameters */
  if (p->tflags&FLG_SWAPIO) { i=mbits; mbits=bits; bits=i; }

  complex = (bits<0);
  abits = abs(bits);
  if (abits!=8 && abits!=16 && abits!=32) {
    print("Err: GCFPGA Illegal output format bits=%d\n",bits); 
    return (-1); 
  }

  /* validate tuner frequency (fc/fs * 2^32) = (fc/ny * 2^31) */
  if (fcny<-1.0 || fcny>2.0) { 
    print("Warn: Tuner freq %f out of range [-1,2]. Now set to zero.\n",fcny); 
    fcny = 0;
  }

  /* initialize CORE system register - borrows system register bitmap from IO modules */
  sysreg = 0;
  if (p->tflags&FLG_RESAMP) sysreg |= GCFPGA_SYS_RESAMP;
  if ((p->tflags&FLG_FIRONLY)==0) sysreg |= GCFPGA_SYS_TUNER;
  if (tbank && p->fttm>1 && chip==2) sysreg |= GCFPGA_SYS_MASK;
  if (n4t) sysreg = CORE_DEC;
  sysreg |= getFmt(mbits)<<8;
  sysreg |= getFmt(bits)<<12;

  /* LUT demod status */
  lut = (findflag("LUT",p->config) > 0);
  if (lut) sysreg |= GCFPGA_SYS_DEMOD;
  if (lut && !complex) dec /= 2;

  /* validate the decimation factor */
  decmin = (order>2)? 1 : (order>1)? 2 : (p->tflags&FLG_RESAMP)? 8 : 4;
  decmid = (order>2)? 2 : (order>1)? 4 : 8; if (nps>0) decmid = nps;
  decmax = 0x1<<mdw;
  if (pmod==0 && p->socc[3]=='N') decmid = 1;
  if (dec<decmin || dec>decmax) { 
    print("Err: Tuner dec=%d out of range [%d-%d]\n",dec*tqd,decmin*tqd,decmax*tqd); 
    return (-2);
  }
  cdec = (dec==1)? 4 : dec*2;  		/* from normal to complex view */
  if ( (cdec&0x3) != 0 && dec>3) { 
    cdec &= 0xFFFC; print("Warn: Tuner decimation rounded to %d\n",cdec/2); 
  }
  decratio = cdec/4; 
  p->deco = dec;

  /* default filter taps for TC calcs */
  htf = (dec<decmid);
  cf1 = (dec==1) && (order<4);
  res = (p->tflags&FLG_RESAMP)!=0;
  if ((p->tflags&FLG_UCFIR)==0 || p->ctaps==0) p->ctaps=(cf1||htf)?15:31; 
  if ((p->tflags&FLG_UFILT)==0 || p->ntaps==0) p->ntaps=(cf1||htf)?31:63;

  pic_lock (p, LOCK_ALLOC);

  if (flags&FLG_FTUN) goto FREQONLY; 	
  if (flags&FLG_GTUN) goto GAINONLY; 	
  if (flags&FLG_DTUN) goto DECONLY; 	

  /* put in reset */
  pic_wpb (p,pmod,tadr|GCFPGA_SYS,sysreg);

  if (flags&FLG_MTUN) p->tflags &= ~(FLG_UFILT|FLG_UCFIR);

  /* load filters */
  i = p->tflags;
  if ((p->tflags&FLG_UCFIR)==0 && nps<=0) pic_loadfile (p, cf1?"cfir_80f":htf?"cfir_80e":res?"dfir_25":"cfir_80", FLG_FC|fport); 
  if ((p->tflags&FLG_UFILT)==0 && nps<=0) pic_loadfile (p, cf1?"pfir_80f":htf?"pfir_80e":"pfir_80", FLG_FC|fport); 
  if (flags==FLG_PRESAMP) {
    if (findstrflag("PRECFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
    if (findstrflag("PREPFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
    if (findstrflag("PRERFIR",p->config,fname,-1)>0) pic_loadfile (p, fname, FLG_FC|port); 
  }
  p->tflags = i;
  /* allow CICs to flush during filter load */

  if (flags&FLG_MTUN) goto DECONLY; 	

  FREQONLY:
  phaseinc = -fcny2phinc(fcny);
  if (tqd>1) { /* split phase increment between pretuner and back end */
    pic_wpb (p,pmod,pic5tadrtbl[2]|GCFPGA_FREQ,phaseinc);
    phaseinc &= 0x00FFFFFF;
    for (i=1; i<tqd; i*=2) phaseinc <<= 1;
  }
  if (nps>0) {
    for (i=0; i<nps; i++) {
      pic_wpb (p,pmod,tadr|GCFPGA_FREQ+i,phaseinc*nps);
      pic_wpb (p,pmod,tadr|GCFPGA_POFF+i,phaseinc*i);
    }
  }
  else {
    pic_wpb (p,pmod,tadr|GCFPGA_FREQ,phaseinc);
  }
  if (flags&FLG_FTUN) goto DONE; /* frequency update only */ 

  /* initialize phase and prime DDS */
  pic_wpb (p,pmod,tadr|GCFPGA_POFF,0);

  DECONLY:
  if (n4t) decratio = dec;
  mdec = (dec<<16) | max(0,decratio-1);
  pic_wpb (p,pmod,tadr|GCFPGA_DEC,mdec);
  if (tbank && chan<=0) for (i=0; i<nchan; i++) pic_wpb (p,pmod,ctadr|(i*MCORE_CHN)|GCFPGA_DEC,mdec); /* for mcs readback RAM */
  if (nps>0) {
    fbwf = (p->fbwf<0)? 0.88 : p->fbwf;
    ndec = 2;
    ocfir = order+p->cfxo;	/* does build have deeper CFIR order */
    ntap = (2<<ocfir) * ndec / nps;
    dofilt = global || !tbank || ((flags&FLG_MTUN)==0);
    if (dofilt) {
      ftmp = 0.80/ndec;
      if (dec<=3 && ntap>=64) { ftmp=0.95/ndec; if (p->fbwf<0) fbwf=0.95; }
      ftmp = finddblflagdef("CFIRF",p->config,ftmp);
      fbuf = (float*)malloc(ntap*2*4); lbuf = (int_4*)fbuf;
      firkais_generate (FIRKAIS_LOW,ftmp,0.0,0.0,(dec==1)?60.0:80.0,fbuf,ntap,0);
      for (i=0; i<ntap; i++) fbuf[i] = fbuf[i+i];
      if (flags&FLG_DTUN) pic_wpb (p,pmod,tadr|GCFPGA_SYS,sysreg);
      i=setup_fpga_filter (p,port,pmod,tadr|GCFPGA_CFIRS,(nps<<8)|ocfir,ndec,-ntap,lbuf);
      free(fbuf);
    }
    p->ctaps = ntap; p->decb = ndec;
    ndec = (dec<=3)? dec : 2;
    ntap = (dec<=3)? dec*16 : 64;
    if (dofilt) {
      ftmp = (dec==1)? fbwf : (dec<=3)? 0.85*fbwf/ndec : fbwf/ndec;
      ftmp = finddblflagdef("PFIRF",p->config,ftmp);
      fbuf = (float*)malloc(ntap*2*4); lbuf = (int_4*)fbuf;
      firkais_generate (FIRKAIS_LOW,ftmp,0.0,0.0,60.0,fbuf,ntap,0);
      fscl = (ndec==1)? 0.75 : 1.0;
      for (i=0; i<ntap; i++) fbuf[i] = fscl*fbuf[i+i];
      if (decratio>1) applyCICcomp(decratio,fbuf,ntap);
      i=setup_fpga_filter (p,port,pmod,tadr|GCFPGA_PFIRS,order,ndec,-ntap,lbuf);
      free(fbuf);
    }
    p->ntaps = ntap; p->dec2 = ndec;
    p->cfir_gain = p->cfir_gain = 1.0;
  }

  GAINONLY:
  /* channel gain: (shift+scale+6*bigscale) <= ( 62 - 5log2(N) + log2(nzero+1) ) */
  if (mbits<0) gain -= 6; /* compensate for twice energy in CIC filter */
  coarse = (gain>=0)? (gain/6) : (gain-5)/6; 
  scale = (int)(5.0*log2((double)cdec/4)+0.99) - coarse;
  dgain = powi(cdec/4,5) * powi(2,-scale);
  scale = max(0,scale+5); /* from 24b to 18b data */
  sgain = 1.0;
  if (p->cfir_gain != 1.0) sgain *= p->cfir_gain; /* compensate for CFIR gain */
  if (p->pfir_gain != 1.0) sgain *= p->pfir_gain; /* compensate for PFIR gain */
  fgain = pow(2.0,gain/6.0) / (sgain*dgain);
  if (fgain>=2.0 && scale>0) { fgain/=2; scale--; }
  gaincomp = (int_4)(0x8000 * fgain);
  vprint("Port=%d Chan=%d Dec=%d Dgain=%f Sgain=%f Coarse=%d Fine=%f Comp=%08x\n",port,chan,dec,dgain,sgain,scale,fgain,gaincomp);
  if (gaincomp>=0x10000) gaincomp = 0xffff;
  pic_wpb (p,pmod,tadr|GCFPGA_GAIN,(scale<<16)|gaincomp);

  if (flags&FLG_GTUN) goto DONE; 		/* gain update only */
  if (flags&FLG_DTUN) goto DDONE; 		/* dec update only */
  if (flags&FLG_MTUN) goto DONE;		/* dec update only */   

  /* reset the resampler registers */
  pic_wpb (p,pmod,tadr|GCFPGA_RATIO,p->res_inc);
  pic_wpb (p,pmod,tadr|GCFPGA_RMON,p->res_mon);

  /* load demodulator LUT */
  if (lut) {
    strcpy(fname,"lut_");
    findstrflag("LUT",p->config,fname+4,-1);
    pic_wpb (p,pmod,tadr|GCFPGA_SYS,sysreg|GCFPGA_SYS_ENABLE);
    i = (pmod>0)? 10*pmod+3+((port-1)%8) : port; 
    pic_loadfile (p, fname, FLG_LUT|i); 
    pic_wpb (p,pmod,tadr|GCFPGA_SYS,sysreg);
  }

  /* set oversampling */
  pic_wpb (p,pmod,tadr|GCFPGA_OVSR,ovsr);

  /* global multicore register */
  if (pmod>0 || (pmod==0 && p->cpc>1) ) {
    i = 0x0;  
    if (tbank) {
      i = 0x8;  /* bulk 512by xfer =0x8 */
      if (flags&FLG_BACKEND) i |= 0x0;  /* bulk 512by xfer =0x8 */
      else if (p->fttm>1 && (port%10)!=3) i |= 0x4; /* add split mode */
      chan = p->m1;
      if (p->fttm<2) chan >>= 1;
      else if (flags&FLG_BACKEND) chan = p->m2;
      i |= (chan<<16);
      tadr |= MCORE_TRIM; /* for later calls to sysreg - addr only active channels */
      if (p->mcs>0) i |= 0x200;	/* need channel output tags */
    }
    if (findintflag("SWAP2",p->config)>0) i |= 0x20;
    i = findintflagdef("MCOPT",p->config,i);
    pic_wpb (p,pmod,(tadr&0xFF000000)|MCORE_SYS,i);
  }

  /* remove reset */
  vprint("FPGA Tuner port=%d pmi=%d tadr=%08x sysreg=%08x dec=%d ovsr=%d phinc=%08x\n",port,pmod,tadr,sysreg,dec,ovsr,phaseinc);
  DDONE:
  if ((pmod==0 && p->pm0mt==0) || tbank) sysreg |= GCFPGA_SYS_ENABLE;
  pic_wpb (p,pmod,tadr|GCFPGA_SYS,sysreg);

  DONE:
  if (flags==FLG_PRESAMP) {
    p->tflags = tflags;
    p->gchip  = cgchip;
    p->pmi  = cpmi;
  }
  pic_lock (p, LOCK_FREE);
  return (tqd/2);
}

int_4 pic_tuner_qtfpga (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 i, status, abits, side, cport, tadr, sysreg, pmod, mbits, phaseinc, tapsin,tapcos,tapcx;
  int_4 ndiv=256, decmin=1, decmax=8, decratio, cdec;
  double fgain,amp,phase;
  char fname[40];

  status = 0;
  if (flags&FLG_RTUN) return status;

  abits = abs(bits);

  if ((flags&FLG_DISABLE)!=0) {
    pic_setcorel (p,port,GCFPGA_SYS,0);
    return 0;
  }

  if (rate>GCFPGB_CLOCK*8) {
    print("Err: QTFPGA Tuner not supported at %d Hz input rate\n",rate);
    return (-1); 
  }

  if (abits!=8 && abits!=16) {
    print("Err: QTFPGA Illegal output format bits=%d\n",bits); 
    return (-1); 
  }

  /* validate tuner frequency (fc/fs * 2^32) = (fc/ny * 2^31) */
  if (fcny<-1.0 || fcny>2.0) { 
    print("Err: Tuner Freq %f out of range [-1,2]\n",fcny); 
    return (-1);
  }

  /* find number of bits in module port */
  mbits = p->mbits;

  /* initialize CORE system register - borrows system register bitmap from IO modules */
  sysreg = 0;
  sysreg |= getFmt(mbits)<<8;
  sysreg |= getFmt(bits)<<12;

  /* validate the decimation factor */
  if (dec<decmin || dec>decmax) { 
    print("Err: Tuner dec=%d out of range [%d-%d]\n",dec,decmin,decmax); 
    return (-2);
  }
  decratio = (dec>=8)? 8 : (dec>=4)? 4 : (dec>=2)? 2 : 1;	/* from normal to complex view */
  if (decratio != dec) print("Warn: Tuner decimation rounded to %d\n",decratio); 

  cdec = decratio*2;
  if (cdec==8 && mbits!=8) {
    print("Warn: QTuner dec==8 must use input format MBITS=8. Current MBITS=%d\n",mbits);
    return (-3);
  }

  pic_lock (p, LOCK_ALLOC);

  if (flags&FLG_FTUN) goto FREQONLY; 	
  if (flags&FLG_GTUN) goto GAINONLY; 	
  if (flags&FLG_DTUN) goto DECONLY; 	

  /* put in reset */
  pic_setcorel (p,port,GCFPGA_SYS,sysreg);

  FREQONLY:
  phaseinc  = -fcny2phinc(fcny);
  if (phaseinc&0x00800000) phaseinc += 0x01000000;
  pic_setcorel (p,port,GCFPGA_FREQ,phaseinc);
  if (flags&FLG_FTUN) goto DONE;

  DECONLY:
  pic_setcorel (p,port,GCFPGA_DEC,cdec-1);

  /* load decimation specific filter */
  i = p->tflags;
  if (findstrflag("QFIR",p->config,fname,-1)<=0) sprintf(fname,"qtfir_%d",cdec);
  pic_loadfile (p, fname, FLG_COEF|(0x34<<8)|port); /* Addr of PFIRT */
  p->tflags = i;
  if (flags&FLG_DTUN) goto DONE; 

  GAINONLY:
  fgain = pow(2.0,gain/6.0);

  /* load sincos LUT with embedded gain */
  amp = 0x2000 * fgain;
  pic_setcorel (p,port,GCFPGA_SYS,sysreg|GCFPGA_SYS_ENABLE); /* allow auto increment */
  for (i=0; i<ndiv; i++) {
    phase = (i*6.283185307179586/ndiv);
    tapsin = (int)(amp*sin(phase));
    tapcos = (int)(amp*cos(phase));
    tapcx  = ((tapsin<<16)&0xFFFF0000) | (tapcos&0xFFFF);
    pic_setcorel (p,port,GCFPGA_PFIRS,tapcx);
  }
  if (flags&FLG_GTUN) goto DONE; 
  pic_setcorel (p,port,GCFPGA_SYS,sysreg);

  /* remove reset */
  vprint("FPGA Tuner sysreg=%08x dec=%d phinc=%08x\n",sysreg,decratio,phaseinc);
  pic_setcorel (p,port,GCFPGA_SYS,sysreg|GCFPGA_SYS_ENABLE); 

  DONE:
  pic_lock (p, LOCK_FREE);
  return (0);
}

int_4 pic_dump_gcfpga (PICSTRUCT *p, int_4 port) 
{
  int_4 core, pmod, chan, chip, tadr;

  core = port;
  pmod = port2pmi(p,&core);
  chan = port2chan(p,core);
  chip = port2chip(p,core);
  tadr = chip2tadr(p,chip,pmod);

  if (pmod==0 && p->pm0mt>0) tadr = (chip==2)? PPC_TUNB_CTL:PPC_TUNA_CTL;
  if (pmod>=0) tadr += (chan*MCORE_CHN);

  print("GCFPGA PM=%d Port=%d Chip=%d Tadr=%08x Channel=%d Parameters:\n",pmod,port,chip,tadr,chan);

  print("GCFPGA_SYS   = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_SYS)  );
  print("GCFPGA_OVSR  = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_OVSR) );
  print("GCFPGA_FREQ  = %08x\n",~pic_rpb (p,pmod,tadr|GCFPGA_FREQ) );
  print("GCFPGA_DEC   = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_DEC)  );
  print("GCFPGA_GAIN  = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_GAIN) );
  print("GCFPGA_RATIO = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_RATIO));
  print("GCFPGA_RMON  = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_RMON) );
  print("GCFPGA_POFF  = %08x\n", pic_rpb (p,pmod,tadr|GCFPGA_POFF) );
  return 1;
} 


#define AD6620_FCRAM 0x000
#define AD6620_DRAM  0x100
#define AD6620_MCR   0x300
#define AD6620_NCR   0x301
#define AD6620_NSCR  0x302
#define AD6620_FREQ  0x303
#define AD6620_POFF  0x304
#define AD6620_SCL2  0x305
#define AD6620_DEC2  0x306
#define AD6620_SCL5  0x307
#define AD6620_DEC5  0x308
#define AD6620_SCLO  0x309
#define AD6620_DECO  0x30A
#define AD6620_FCOFF 0x30B
#define AD6620_NTAPS 0x30C
#define AD6620_RSV   0x30D
#define AD6620_AHI   0x3FF

int_4 pic_tuner_ad6620 (PICSTRUCT *p, int_4 port, 
	int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags)
{
  int_4 gain1,gain2, complex,real, fullup, status, cdec, cdeci;
  int_4 val_mcr, val_ncr, val_nscr, val_poff, val_fcoff, val_rsv, val_ntaps;
  int_4 val_freq, val_scl2, val_dec2, val_scl5, val_dec5, val_sclo, val_deco;
  double ol_cic2, ol_cic5, clock;
  int_4 n, maxtaps, usetaps, mtap, tdec, ovsr, disable;
  char fcname[24];

  if (port < 0) port=0; 
  if (port==0) port=3;
  status = 0;
  complex = (bits < 0);
  fullup = !(flags&(FLG_FTUN|FLG_GTUN|FLG_DTUN|FLG_RTUN));
  disable = (flags&FLG_DISABLE)!=0;

  /* initialize oversampling ratio */
  ovsr = (port&0x1)? p->ovsr[0] : p->ovsr[1];
  if (!disable && ovsr<=1) {
    if (rate>16000000) ovsr=0; else ovsr=1;
  }
  if (ovsr > 1) {
    if ((p->tflags&FLG_POVSR)!=0) rate = rate/ovsr;
    else { dec = dec*ovsr; fcny = fcny/ovsr; }
    gain = gain + (int)(log2((double)ovsr)*6+.5);
  }
  if (ovsr>0) {
         if (rate<=16000000);
    else if (rate*ovsr<=p->tun_clk);
    else print("Err: AD6620 Tuner Oversampling=%d not supported at %d Hz input rate\n",ovsr,rate);
  }

  /* validate tuner frequency (fc/fs * 2^32) = (fc/ny * 2^31) */
  if (fcny<-1.0 || fcny>1.0) { 
    print("Err: AD6620 Tuner Freq %f out of range [0-1]\n",fcny); 
    status = -1; goto DONE;
  }
  val_freq = fcny2phinc(fcny);
  if (disable) val_freq = 0x87654321;

  if (flags&FLG_FTUN) goto DOIT2; /* frequency update only */

  val_mcr = 0x00;	/* single channel real */
  val_ncr = (p->tflags&FLG_DSYNC)? 0x0 : 0x06;	/* disable NCO phase & amplitude dither */
  val_nscr= 0xffffffff;	/* sync mask */
  val_poff = 0x0000;	/* NCP phase offset */

  gain1 = (gain+6000+5)/6 - 1000; 
  gain2 = gain - (gain1*6); 
  val_sclo = max(0,min(7,4-gain1));

  if (flags&FLG_GTUN) goto DOIT; 	/* gain update only */

  /* get the tuner decimation */
  cdeci = dec*2; /* from normal to complex view */ 
  cdec = get_ad6620_dec (cdeci,&val_dec2,&val_dec5,&val_deco,&ovsr);
  p->dec2=(int_1)val_dec2; p->dec5=(int_1)val_dec5; p->deco=(int_1)val_deco;
  dec = cdec/2; 
  if (cdec != cdeci) print("Warn: Tuner decimation rounded to %d\n",dec); 
  if (!complex) val_deco = val_deco/2;

  val_scl2 = (int)(log2(powi(val_dec2,2)) +.999) - 2;
  ol_cic2 = powi(val_dec2,2) / powi(2,2+val_scl2);
  if (val_dec2==1) { val_scl2=0; ol_cic2=1; }
  val_scl5 = (int)(log2(powi(val_dec5,5)*ol_cic2) +.999) - 5; 
  ol_cic5 = powi(val_dec5,5) * ol_cic2 / powi(2,5+val_scl5);
  if (val_dec5==1) { val_scl5=0; ol_cic5=1; }

  val_rsv   = 0x0;
  vprint ("TUNER %d dec=%d|%d|%d \n",port,val_dec2,val_dec5,val_deco);

  if (flags&FLG_DTUN) goto DOIT;  	/* decimation update only */

  DOIT:
  pic_lock (p, LOCK_ALLOC);

  /* set oversampling factor */
  if (!disable && ovsr!=p->ovsr[port-1]) {
    pic_tuner_ovsr (p,ovsr,port,0);  
  }

  if (fullup) {
    p->decb = 0; /* init filter band decimation factor */
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CEEN|IOC_DT_CE,0);
    if (complex) real=0; else real=IOC_DT_RM; 
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_RMEN|real,0);
    /* pulse the reset pin and then set the chip select */
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_RSEN|IOC_DT_RS,0);
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_RSEN|0,0);
    /* clear the data ram */
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN|IOC_DT_CS,0);
    pic_acmd (p,ADON_WRIOB,AD6620_AHI,0x1,1); 
    /* for (i=0; i<255; i++) pic_acmd (p, ADON_WRIOB, AD6620_DRAM+i, 0, 4); */
  }

  /* calculate max filter taps */
  maxtaps = 255;
  if (port==1 || port==2) {
    n = lport2dmac[p->linktbl[port]]-1;
    /* if oversampling, the tap clock is cclk */
    if (p->ovsr[port-1]>0) clock=p->ext_clk; else clock=(double)rate;
    if (complex) tdec=dec*2; else tdec=dec;
    if (p->ovsr[port-1]>1) tdec = tdec/p->ovsr[port-1];
    maxtaps = (int_4)(clock / ((double)rate/tdec) / 1.1);
    maxtaps = (maxtaps/2)*2-1;
  }
       if (maxtaps>=255) usetaps=255;
  else if (maxtaps>=127) usetaps=127;
  else if (maxtaps>=63) usetaps=63;
  else if (maxtaps>=31) usetaps=31;
  else usetaps=15;
  vprint("Maxtaps = %d UseTaps = %d\n",maxtaps,usetaps);

  mtap = pic_acmd (p, ADON_RD, DMA_MTAP, 0, 0);
  if (port==1||port==3) p->ntaps = (mtap&0xFFFF);
  if (port==2||port==3) p->ntaps = (mtap>>16);

  /* load new filter ? */
  if (disable);
  else if (p->tflags&FLG_UFILT);
  else if (flags&(FLG_FTUN|FLG_GTUN));
  else if (p->decb != p->deco || usetaps != p->ntaps) { 
    sprintf(fcname,"fc_def%dx%d",p->deco,usetaps);
    pic_loadfile (p, fcname, FLG_FC|port); 
    p->decb = p->deco;
  }

  val_fcoff = 0x0;
  val_ntaps = p->ntaps;

  /* calculate filter offsets */
  if (maxtaps<val_ntaps && !disable) {
    val_fcoff = (p->ntaps-maxtaps)/2;
    val_ntaps = maxtaps;
    vprint("Maxtaps = %d of %d off=%d at dec=%d\n",
	val_ntaps,p->ntaps,val_fcoff,tdec);
  }

  DOIT2:
  if (flags&FLG_FTUN) pic_lock (p, LOCK_ALLOC);

  /* write the high address byte for dynamic register access */
  pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN|IOC_DT_CS,0);
  pic_acmd (p, ADON_WRIOB, AD6620_AHI, 0x3, 1); 

  if (flags&FLG_FTUN) { 	/* frequency update only */
    pic_acmd (p, ADON_WRIOB, AD6620_FREQ, val_freq, 4); 
  } 
  else if (flags&FLG_GTUN) { 	/* gain update only */
    pic_acmd (p, ADON_WRIOB, AD6620_SCLO, val_sclo, 1); 
    pic_fload_ad6620 (p, port, val_ntaps, gain2);
  } 
  else if (flags&(FLG_DTUN|FLG_RTUN)) { /* dec update only */
    pic_acmd (p, ADON_WRIOB, AD6620_SCL2, val_scl2, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DEC2, val_dec2-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_SCL5, val_scl5, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DEC5, val_dec5-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_SCLO, val_sclo, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DECO, val_deco-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_FCOFF, val_fcoff, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_NTAPS, val_ntaps-1, 1); 
  } 
  else {			/* full reset update */
    pic_acmd (p, ADON_WRIOB, AD6620_MCR, val_mcr|0x1, 1); /* hold soft reset */
    pic_acmd (p, ADON_WRIOB, AD6620_NCR, val_ncr, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_NSCR, val_nscr, 4); 
    pic_acmd (p, ADON_WRIOB, AD6620_FREQ, val_freq, 4); 
    pic_acmd (p, ADON_WRIOB, AD6620_POFF, val_poff, 2); 
    pic_acmd (p, ADON_WRIOB, AD6620_SCL2, val_scl2, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DEC2, val_dec2-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_SCL5, val_scl5, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DEC5, val_dec5-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_SCLO, val_sclo, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_DECO, val_deco-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_FCOFF, val_fcoff, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_NTAPS, val_ntaps-1, 1); 
    pic_acmd (p, ADON_WRIOB, AD6620_RSV, val_rsv, 1); 
    if (gain2!=0) pic_fload_ad6620 (p, port, val_ntaps, gain2);
    if (!disable) /* lift soft reset */
    pic_acmd (p, ADON_WRIOB, AD6620_MCR, val_mcr, 1);
  }

  /* release the chip select */
  pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN,0);

  pic_lock (p, LOCK_FREE);

  DONE:
  return status;
}

int_4 pic_dump_ad6620 (PICSTRUCT *p, int_4 port) 
{

  if (port<0) port=0;
  pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN|IOC_DT_CS,0);
  pic_acmd (p,ADON_WRIOB, AD6620_AHI, 0x3, 1); 

  print("AD6620_MCR  = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_MCR,0,1) );
  print("AD6620_NCR  = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_NCR,0,1) );
  print("AD6620_NSCR = %08x\n", pic_acmd (p,ADON_RDIOB,AD6620_NSCR,0,4) );
  print("AD6620_FREQ = %08x\n", pic_acmd (p,ADON_RDIOB,AD6620_FREQ,0,4) );
  print("AD6620_POFF = %04x\n", pic_acmd (p,ADON_RDIOB,AD6620_POFF,0,2) );
  print("AD6620_SCL2 = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_SCL2,0,1) );
  print("AD6620_DEC2 = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_DEC2,0,1) );
  print("AD6620_SCL5 = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_SCL5,0,1) );
  print("AD6620_DEC5 = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_DEC5,0,1) );
  print("AD6620_SCLO = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_SCLO,0,1) );
  print("AD6620_DECO = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_DECO,0,1) );
  print("AD6620_FCOFF= %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_FCOFF,0,1) );
  print("AD6620_NTAPS= %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_NTAPS,0,1) );
  print("AD6620_RSV  = %02x\n", pic_acmd (p,ADON_RDIOB,AD6620_RSV,0,1) );
/*
  pic_acmd (p,ADON_WRIOB, AD6620_AHI, 0x0, 1); for (i=0; i<16; i++)
  print("AD6620_FILT_%d  = %08x\n",i,pic_acmd (p,ADON_RDIOB,AD6620_FCRAM+i,0,3) );
*/
  pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN,0);

  return 1;
} 


int_4 get_ad6620_dec (int_4 dec,
	int_4 *val_dec2,int_4 *val_dec5,int_4 *val_deco,int_4 *val_ovsr)
{
  int_4 tdec,mdec,ddec,sdec, dec2,dec5,deco;
  
  sdec = mdec = dec;
       if (dec<=512*2) deco=2;
  else if (dec<=512*4) deco=4;
  else if (dec<=512*8) deco=8;
  else if (dec<=512*16) deco=16;
  else if (dec<=512*32) deco=32;
  else deco=64;
  if (*val_ovsr==0) dec2=2; else dec2=1;
  for (; dec2<=16; dec2++) {
    for (dec5=1; dec5<=32; dec5++) {
      tdec = dec2*dec5*deco;
      ddec = dec-tdec;
      if (ddec>=0 && ddec<mdec) {
	mdec = ddec;
	*val_dec2 = dec2;
	*val_dec5 = dec5;
	*val_deco = deco;
	sdec = tdec;
        if (ddec<=1) goto DONE;
      }
    }
  }
  DONE:
  return sdec;
}

int_4 pic_loadfc (PICSTRUCT *p, int_4 port, 
		int_2 *data, int_4 ndata, int_4 flags)
{
  int_4 i, j, ic, core, chan, chip, tadr, value, val_pm, ntaps, mtap, tbank=0, fcore, pmod;
  int_2 loadshort[ndata]; 
  int_4 loadint[ndata]; 
  int_4 *ldata = (int_4 *)data;
  real_4 *rdata = (real_4 *)data;
  int_u1 *loadbyte = (int_u1 *)loadshort;
  int_4 cpc, nmult, ndelay, order, useint;
  double gain = 1.0;

  if (port<0) port=p->pindex; /* port=0 is all tuners */
  fcore = (p->ptype == IOPT_CORE) && !(port<=2 && (p->tflags&FLG_PRESAMP)!=0);
  tbank = ((p->tflags&FLG_TBANK)!=0)? 1:0;
  if (tbank && port<100) port += 100;

 if (fcore) {
  pmod = port/10;
  core = port%10;
  chan = chip = 0;
  if (pmod==0) pmod = p->pmi;
  if (pmod==3) pmod = 1;
  tadr = (core&1)? PPC_CORA_CTL:PPC_CORB_CTL;
 } else {
  core = port;
  pmod = port2pmi(p,&core);
  chan = port2chan(p,core);
  chip = port2chip(p,core);
  tadr = chip2tadr(p,chip,pmod);
  if (pmod==0 && p->pm0mt>0) {	/* base card multicore */
   tadr = (chip==2)? PPC_TUNB_CTL : (chip==1)? PPC_TUNA_CTL : PPC_TUNA_CTL|PPC_TUNB_CTL;
   if (chan<0) tadr |= MCORE_ALL;
   else tbank=1;
  }
  else if (pmod==0) {
   if (p->cpc>1 && chan<0) tadr |= MCORE_ALL;
   if (p->cpc==0 && port==0) tadr |= MCORE_ALL;	/* RESAMP filter init */
   if (p->mcs>0) tbank=1;
  }
 }
  cpc  = p->cpc;
  useint = ( (p->gchip==GCFPGA) || fcore);

  ntaps = ndata;
  if (useint) {
    for (i=0; i<ndata; i++) loadint[i]   = (flags&FLG_FLOAT)? rint(rdata[i]*0x10000) : (flags&FLG_LONG)? ldata[i] : data[i];
  } else {
    for (i=0; i<ndata; i++) loadshort[i] = (flags&FLG_FLOAT)? rint(rdata[i]*0x10000) : (flags&FLG_LONG)? ldata[i] : data[i];
    if (flags&FLG_LONG) print ("Filter taps file format=SL only supported on FPGA tuners\n");
    /* need native format to sum the coefficients FLG_EEEI=fromLoadfile */
    if ((flags&FLG_EEEI)!=0) ConvertEndian ((void*)loadshort,ntaps,16);
  }

  if (p->type == ICEMBT2 || p->type == ICESLIC3) {
    if (ndata == 63) {		/* check symmetry */
      for (i=1; i<32; i++) if (loadshort[31+i]!=loadshort[31-i]) ndata=0;
    } 
    if (ndata != 32 && ndata != 63) {
      print("Filter file must be a 63 tap symmetric filter\n");
      return -1;
    }
  } 
  else if (useint) {
    if (fcore);
    else if (ndata%2 == 0) { /* check even symmetry */
      ndata = ndata/2;
      for (i=0;i<ndata;i++)
        if (abs(loadint[ndata+i]-loadint[ndata-1-i])>2) ndata=0;
    } else {		/* check odd symmetry */
      ndata = (ndata+1)/2;
      for (i=1;i<ndata;i++)
        if (abs(loadint[ndata-1+i]-loadint[ndata-1-i])>2) ndata=0;
    } 
    for (i=0,value=0;i<ntaps;i++) value += loadint[i];    /* sum coefficients */
    if (fcore && value==0) value = 32768;  /* antisymmetric ? */
    value = abs(value);
         if (fcore)                              gain=p->cfir_gain=(double)value/65536;
    else if (ndata==8 || ndata==11 || ndata==15) gain=p->cfir_gain=(double)value/65536;
    else if (ndata==8 || ndata==16 || ndata==24 || ndata==32 || ndata==64) gain=p->pfir_gain=(double)value/65536;
    else if (ndata==1024 || ndata==10240)        gain=p->res_gain=(double)1;
    else {
      print("Filter file taps=%d must be a CFIR=15|21|31, PFIR=31|63|127, or RES=2048|20480 tap symmetric filter.\n",ntaps);
      return -1;
    } 
  } 
  else if (p->gchip == GC4016) {
    if (ndata == 63 || ndata == 21) {       /* check PFIR, CFIR symmetry */
      ndata = (ndata+1)/2;
      for (i=1;i<ndata;i++)
        if (loadshort[ndata-1+i]!=loadshort[ndata-1-i]) ndata=0;
    } 
    else if (ndata > 256 && ndata%2 == 0) { /* check RES symmetry */
      ndata = ndata/2;
      for (i=0;i<ndata;i++)
        if (loadshort[ndata+i]!=loadshort[ndata-1-i]) ndata=0;
    } 
    for (i=0,value=0;i<ndata;i++) value += loadshort[i];    /* sum coefficients */
    if (ndata==32) gain=p->pfir_gain=(double)(2*value-loadshort[31])/65536;
    else if (ndata==11) gain=p->cfir_gain=(double)(2*value-loadshort[10])/65536;
    else if (ndata>=128) { 
      ndelay = findintflag("NDELAY",p->config);
      if (ndelay<0) ndelay=32;
      if ((2*ndata)%ndelay != 0) {
        print("The number of RES taps must be an integer multiple of NDELAY.\n");
        return -1;
      }
      nmult=2*ndata/ndelay;
      if (nmult < 6 || (cpc == 1 && nmult < 7)) {
        print("NMULT must not be less than 7 for one channel, or less than 6 for"
		   " two or more channels per chip.\n"); return -1;
      }
      if (nmult >= 64/cpc) {
        print("NMULT must be less than 64/CPC.\n"); return -1;
      }
      /* resampler coeffs are 12 bit so *16 to account for <<4 */
      gain=p->res_gain=(double)(2*16*2*value)/(32768*ndelay);
    } 
    else {
      print("Filter file taps=%d must be a CFIR=21, PFIR=63, or RES>256 tap symmetric filter.\n",ntaps);
      return -1;
    } 
  }

  /* the rest of this routine assumes a EEEI buffer, make sure it is */
  if (!useint) ConvertEndian ((void*)loadshort,ntaps,16);

  vprint("Loading Filter port=%d pm=%d tadr=%08x ntaps=%d gain=%f\n",port,pmod,tadr,ntaps,gain);

  pic_lock (p, LOCK_ALLOC);

  /* set page to this channel */
  if (p->gchip == GC4014) {
    value = pic_acmd (p,ADON_RDIOB,tadr|GC4014_PM,0,0);
    val_pm = 0xF0 & value;
    if (flags&FLG_NCFIR) val_pm |= 0x10; else val_pm &= 0xEF;
    for (j=0; j<4; j++) {
      value = val_pm|(8+j);
      pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,value,0);
      for (i=0; i<16; i++,loadbyte++) {
	value = (int_4)(*loadbyte);
        pic_acmd (p,ADON_WRIOB,tadr|(16+i),value,0);
      }
    }
  }
  else if (p->type == ICEPIC3) {
    i = (ndata+1)/2*4;
    /* cause transfer on IEEE will flip */
    ConvertEndian ((void*)loadshort,i/4,32); 
    if (port==1 || port==3)
    pic_transfer (p, (void*)loadshort, i, DM_FILT1, 1, 0);
    if (port==2 || port==3)
    pic_transfer (p, (void*)loadshort, i, DM_FILT2, 1, 0);

    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN|IOC_DT_CS,0);
    pic_fload_ad6620 (p, port, ndata, 0);
    pic_acmd (p,ADON_WRIOC,IOC_TCS|port,IOC_DT_CSEN,0);

    /* report to static SHARC registers */
    mtap = pic_acmd (p, ADON_RD, DMA_MTAP, 0, 0);
    if (port==1||port==3) mtap = (mtap&0xFFFF0000) + ndata;
    if (port==2||port==3) mtap = (mtap&0xFFFF) + (ndata<<16);
    mtap = pic_acmd (p, ADON_WR, DMA_MTAP, mtap, 0);
    p->ntaps = ndata;
  }
  else if (fcore) {
    i = (p->deco>0)? p->deco : 1;
    p->tflags |= FLG_FIRONLY;
    order = pic_rpb(p,pmod,tadr|COREG_PARAM1);
    setup_fpga_filter (p,port,pmod,tadr|GCFPGA_PFIRS,order,i,ntaps,loadint);
    pic_setkeyl(p,-port,KEY_GAIN,0);
  }
  else if (p->gchip == GCFPGA) {
    order = pic_rpb(p,pmod,tadr|COREG_PARAM1);
    value = findintflag("PFIR_DEC",p->config);
    if (pmod>0 || tbank) tadr |= ( (port>100)? MCORE_ALL : chan*MCORE_CHN ); /* address all channels */
    if ((ndata == 8 || ndata == 11 || ndata == 15) && value<0) {	/* load CFIR filter taps */
      p->tflags |= FLG_UCFIR;
      i = (order>=3 && p->deco<=1 && ndata==8)? 1 : 2;	/* decimation in filter */
      p->ctaps = setup_fpga_filter (p,port,pmod,tadr|GCFPGA_CFIRS,order,i,ntaps,loadint);
    }
    else if (ndata==8 || ndata == 16 || ndata == 24 || ndata==32 || ndata == 64 || ndata==128) {	/* load PFIR filter taps */
      p->tflags |= FLG_UFILT;
      p->ntaps = setup_fpga_filter (p,port,pmod,tadr|GCFPGA_PFIRS,order,(value<0)?2:value,ntaps,loadint);
    }
    else {                      /* load RESAMPLER filter taps (just first half of symmetric) */
      p->tflags |= (FLG_URFIR|FLG_RESAMP);
      v2print("FPGA filter load port=%d taps=%d pmod=%d tadr=%x\n",port,ntaps,pmod,tadr);
      pic_wpb (p,pmod,tadr|GCFPGA_RFIRS,0x0); 
      for (i=0; i<ndata; i++) pic_wpb (p,pmod,tadr|GCFPGA_RFIRT,IX2L(loadint[i])); 
      pic_wpb (p,pmod,tadr|GCFPGA_RFIRS,0x10); /* FIFO afull in samples */
    }
  }
  else if (p->gchip == GC4016) {
    if (ndata == 32) {			/* load PFIR filter taps */
      p->tflags |= FLG_UFILT;
     for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==chan || tbank) {
      for (i=0;i<4;i++) {
        pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+2+i)<<1,0);
        for (j=0;j<8;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),loadshort[j+i*8],2);
      }
     }
    }
    else if (ndata == 11) {		/* load CFIR filter taps */
      p->tflags |= FLG_UCFIR;
     for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==chan || tbank) {
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+0)<<1,0);
      for (j=0;j<8;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),loadshort[j],2);
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+1)<<1,0);
      for (j=0;j<3;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),loadshort[j+8],2);
     }
    }
    else {                      /* load RESAMPLER filter taps */
      p->tflags |= (FLG_URFIR|FLG_RESAMP);
      for (i=ndata;i<256;i++) loadshort[i]=0;       /* zero unused coefficients */
      for (i=0;i<32;i++) {
        pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(32+i)<<1,0);
        for (j=0;j<8;j++) pic_acmd (p,ADON_WRIOB,tadr|(16+j*2),loadshort[i*8+j],2); 
      }
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,64<<1,0);
      pic_acmd (p,ADON_WRIOB,tadr|17,nmult-1,0);
    }
  }
  else {
    print("Ntap=%d filters not supported on this card type=%s\n",ntaps,getCardName(p->type));
  }
  pic_lock (p, LOCK_FREE);
  return ndata;
}

int_4 setup_fpga_filter (PICSTRUCT *p, int_4 port, int_4 pmod, int_4 addrs, int_4 order, int_4 dec, int_4 taps, int_4 *coef)
{
  int_4 i,odd,sym,anti,staps,otap,ttaps,tpc,cps,odec,otpc,aload,bload,acc,aco,seq,htaps,otaps;
  int_4 aadr,ashft,badr,bshft,cadr,cshft,radr,rshft,idec,tcnt,ccnt, addrt;
  int_4 sequence[32],b1,side,nps;
  real_4 *fcoef = (real_4*)coef;

  if (taps<0) {
    taps = -taps;
    for (i=0; i<taps; i++) coef[i] = R2LC20(fcoef[i]);
  } else {
    for (i=0; i<taps; i++) coef[i] = I2L(coef[i]);
  }

  odd  = (taps%2 == 1);
  sym  = 1;
  anti = 1;

  nps  = (order>>8)&0xFF;
  if (nps==0) nps=1;
  b1   = (order>1 && nps<=1);   /* use prev cell for turnaround */
  order &= 0xFF;

  if (nps>1 && dec%nps!=0) printf("FPGA filter dec=%d must be a multiple of nps=%d parallel filter build\n",dec,nps);

  /* parse taps */
  for (i=0; i<taps/2; i++) {
    if (abs(coef[i]-coef[taps-1-i])>0x10000) sym = 0;
    if (abs(coef[i]+coef[taps-1-i])>0x10000) anti = 0;
  }
  if (anti) sym=1;
  if (!sym) odd=0;
  staps = taps;
  if (sym) staps/=2; 
  if (odd) staps++;
  otap = order;
  ttaps = 1<<otap;
if (dec*ttaps/nps==staps) {
  tpc = dec/nps;
} else {
  for (i=1; i<dec; i*=2) otap++;  /* disallow degenerate case of short decimating filters */
  ttaps = 1<<otap;
  for (; ttaps<staps; ttaps+=ttaps,otap++);
  tpc = 1<<(otap-order);
}
  cps = (tpc+dec-1)/dec;
  htaps =  sym? ttaps : ttaps/2;
  otaps = htaps / dec;

  v2print("Filter order=%d dec=%d tpc=%d cps=%d ttaps=%d staps=%d sym=%d odd=%d anti=%d nps=%d\n",
					order,dec,tpc,cps,ttaps,staps,sym,odd,anti,nps);
  dec /= nps;
  odec = dec;
  otpc = tpc;
  tpc--;
  cps--;
  dec--;
  /* form sequence */
  aadr=tpc; ashft=0;
  badr=dec; bshft=0;
  cadr=tpc; cshft=0;
  radr=1+b1*tpc; rshft=0;
  idec=0;
  if (odd) radr--;
  for (tcnt=0; tcnt<=tpc; tcnt++) {
    ccnt = tcnt&cps;
    aload = (ccnt==0);
    bload = (ccnt==cps);
    acc = (tcnt!=0);
    aco = (tcnt==tpc);
    if (bload) {
      if (idec==0) rshft=tpc-dec-(1-b1);
      else rshft+=2;
      if (++idec>dec) idec=0;
    }
    seq = ((radr+rshft)<<18) | ((cadr+cshft)<<12) | ((badr+bshft)<<6) | ((aadr+ashft));
    if (aload) seq |= 0x01000000;
    if (bload) seq |= 0x02000000;
    if (acc)   seq |= 0x04000000;
    if (aco)   seq |= 0x08000000;
    v2print("Filter Sequence %d=%08x aadr=%02x badr=%02x cadr=%02x radr=%02x aload=%d bload=%d acc=%d aco=%d\n",
                tcnt,seq,aadr+ashft,badr+bshft,cadr+cshft,radr+rshft,aload,bload,acc,aco);
    sequence[tcnt] = seq;
    if (aload) { ashft++; radr++; }
    if (bload) { bshft++; radr--; }
    aadr-=odec; if (aadr<0)   aadr+=tpc;
    badr+=odec; if (badr>tpc) badr-=(tpc+2);
    cadr-=odec; if (cadr<0)   cadr+=tpc;
  }

  side = (port&0x1)? 1:2;
  v2print("FPGA filter load port=%d taps=%d pmod=%d tadr=%x\n",port,taps,pmod,addrs);
  addrt = addrs+4;

  /* QTuner filter load */
  if (pmod==0 && p->type==ICEPIC6 && p->socc[1+side]=='Q') {
    for (i=0; i<staps; i++) pic_wpb (p,pmod,addrt,coef[i]|i); 
    return taps;
  }

  /* load sequence */
  seq = 0x80000000;
  if (anti)  seq |= 0x10000000;
  if (!sym)  seq |= 0x20000000;
  pic_wpb (p,pmod,addrs,seq|(tpc<<16)|otaps);
  for (tcnt=tpc; tcnt>=0; tcnt--) pic_wpb (p,pmod,addrs,sequence[tcnt]);
  
  /* load taps */
  if (odd) coef[staps-1] /= 2;
  if (sym) {
    for (i=staps-1; i>=0; i--) pic_wpb (p,pmod,addrt,coef[i]|i); 
    for (i=staps; i<ttaps; i++) pic_wpb (p,pmod,addrt,0);
  } else {
    for (i=staps; i<ttaps; i++) pic_wpb (p,pmod,addrt,0);
    for (i=taps-1; i>=0; i--) pic_wpb (p,pmod,addrt,coef[i]|i); 
  }
  if (sym) ttaps *= 2;
  if (odd) ttaps -= 1;
  return ttaps;
}

int_4 pic_loadlut (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags)
{
  int_4 i,pi,chan=0,pmi,tadr;
  int_4 *data = (int_4 *)loadbyte;

  pi   = port%10;
  pmi  = port/10;
  if (pi>4) { chan = (pi-1)/2 - 1; pi -= 2; }
  tadr = core_addr(pi);
  tadr |= 0x100*chan;
  tadr |= GCFPGA_LUT; 

  /* load sequence */
  vprint("Loading LUT of %d bytes into port=%d pmi=%d tadr=%08x\n",nprog,port,pmi,tadr);
  pic_wpm (p,pmi,tadr|0x1,data,nprog,FLG_PPC_BUS); /* table data addr|0x1 = to single address */
  pic_wpb (p,pmi,tadr,flags); /* operational flags */

  return nprog;
}

int_4 pic_loadcoef (PICSTRUCT *p, int_4 port, int_4 addr, real_4 *coef, int_4 ncoef, int_4 flags)
{
  int_4 i,j,k,rank,frame,cpi,pmi,tadr;
  int_4 data[ncoef];

  cpi  = port%10;
  pmi  = port/10;

  if (flags&0x1) { for (i=0; i<ncoef; i+=2) coef[i] = -coef[i]; }	/* Flip */

  if (flags&0x2) {	/* Polyphase */
    rank = (flags&0x4)? 8:4;
    frame=ncoef/rank;
    for (i=j=0; j<rank; j++) {
      for (k=0; k<frame; k++,i++) data[i] = R2LC20(coef[i])|k;
    }
  } else {		/* Normal */
    for (i=0; i<ncoef; i++) data[i] = R2LC20(coef[i])|i;
  }

  if (flags&0x1) { for (i=0; i<ncoef; i+=2) coef[i] = -coef[i]; }	/* Flip */

  /* load sequence */
  vprint("Loading COEF of %d taps into pmi=%d cpi=%d addr=%08x\n",ncoef,pmi,cpi,addr);
  if (pmi==5 || pmi==6) {
    for (i=0; i<ncoef; i++) pic_setcorel (p,port,addr,data[i]);
  } else {
    addr |= core_addr(cpi);
    pic_wpm (p,pmi,addr|0x1,data,ncoef*4,FLG_PPC_BUS); /* table data addr|0x1 = to single address */
  }

  return ncoef;
}

void pic_fload_ad6620 (PICSTRUCT *p, int_4 port, int_4 ntap, int_4 gain) 
{
  int_4 addr,mult;
  addr = DM_FILT2;
  if (port&0x1) addr = DM_FILT1;
  mult = (int)( 0.5 + 16.0*pow(2.0,gain/6.0) );

  /* write the high address byte for coefficient ram access */
  pic_acmd (p, ADON_WRIOB, AD6620_AHI, 0x0, 1); 

  /* move data into tuners */
  pic_acmd (p, ADON_DDC, addr, ntap, mult); 

  /* write the high address byte for dynamic register access */
  pic_acmd (p, ADON_WRIOB, AD6620_AHI, 0x3, 1); 
  pic_acmd (p, ADON_WRIOB, AD6620_NTAPS, ntap-1, 1); 
}


real_8 pic_tuner_freq (PICSTRUCT *p, real_8 fcny, int_4 flags)
{
  int_u4 uphaseinc;
  int_8 lphaseinc; 
  if (fcny<-1.0) fcny = -1.0;
  if (fcny>2.0) fcny = 2.0;
  if (p->type == ICEPIC2) {
    uphaseinc = (int_u4)( fcny*4294967296.0 + .499999);
    if (uphaseinc==0 && fcny>0.5) fcny = 1.0;
    else fcny = (uphaseinc/4294967296.0); 
  } else {
    lphaseinc = (int_8)( (fcny+2)*FULLSCALE + .499999);
    fcny = (lphaseinc/FULLSCALE) - 2; 
  }
  return fcny;
}

int_4 fcny2phinc (double fcny) {
  return (int_4)(int_8)( (fcny+2)*FULLSCALE + .499999);
}

real_8 pic_tuner_freq_ (PICSTRUCT *p, real_8 fcny, int_4 port, int_4 flags)
{
  int_4 ovsr,povsr=1,i,j;
  double ratio,dfreq,ofreq;

  if (port<0) port = p->pindex;
  ovsr = findintflag("OVSR",p->config); /* user specified ovsr */
  ratio = finddblflag("RATIO",p->config); /* user specified PRESAMP ratio */
  if (ratio<0 && (p->tflags&FLG_PRESAMP) && p->res_ratio!=0) ratio = p->res_ratio;
  if (ovsr<0) { /* retrieve current ovsr */
    pic_tuner_ovsr (p,0,0,FLG_INFO); 
    if (port&0x1) ovsr=p->ovsr[0]; else ovsr=p->ovsr[1];
  }
  if ((p->tflags&FLG_POVSR)!=0) { povsr=ovsr; ovsr=0; }

  if (ratio>0) fcny = ((fcny*povsr-.5)/ratio + 0.5)/povsr;
  if (ovsr>1) fcny = fcny/ovsr;
  if (p->fttm>0) {
    dfreq = p->dfreq;
    ofreq = p->ofreq;
    if (ovsr>1) dfreq = dfreq/ovsr;
    if (ovsr>1) ofreq = ofreq/ovsr;
    i = max(0,p->chan-1);
    if (p->fttm==2) {
      j = i/p->m2; i = i%p->m2;
      fcny = pic_tuner_freq(p,fcny+j*dfreq*p->m2+ofreq,0); /* stage 1 */
      fcny += (((p->phinc+i*p->dphinc)/FULLSCALE-.5) / p->d1); /* stage 2 */
    } else {
      fcny = pic_tuner_freq(p,fcny+i*dfreq,0); /* stage 1 */
    }
  } else {
    fcny = pic_tuner_freq(p,fcny,flags);
  }
  if (ovsr>1) fcny = fcny*ovsr;
  if (ratio>0) fcny = ((fcny*povsr-.5)*ratio + 0.5)/povsr;

  return fcny;
}

int_4 pic_tuner_dec (PICSTRUCT *p, int_4 dec, int_4 port, int_4 flags)
{
  int_4 chan,chip,lport,sdmac,nstat,mask,ovsr;
  int_4 tmp1,tmp2,tmp3,mindec,maxdec,ngc;

  if (port<0) port = p->pindex;
  pic_tuner_ovsr (p,0,0,FLG_INFO);  /* retrieve ovsr */
  if (port&0x1) ovsr=p->ovsr[0]; else ovsr=p->ovsr[1];
  if ((p->tflags&FLG_POVSR)!=0) ovsr=0; 

  if (p->type == ICEPIC2) {
    if (p->tflags&FLG_AOVSR) {
      tmp1 = (35/dec)+1;
      if (tmp1<=1) tmp1=0;
      if (tmp1!=ovsr) { ovsr=tmp1; pic_tuner_ovsr (p,ovsr,port,0); }
    }
    if (ovsr>1) dec = dec*ovsr;
    mindec = 36; maxdec=65536;
    if (ovsr>1) mindec = (mindec+ovsr-1)/ovsr*ovsr;
    dec = max(mindec,min(maxdec,dec));
    if ((dec&1)!=0) dec++;
    if (ovsr>1) for (dec=dec/ovsr; (dec*ovsr)<mindec || ((dec*ovsr)&0x1)!=0; dec++);
  } 
  else if (p->type == ICEPIC3) {
    chip = p->chiptbl[port];
    lport = p->linktbl[chip];
    sdmac = lport2dmac[lport];
    if (ovsr>1) dec = dec*ovsr;
    dec = dec*2; /* from normal to complex view */
    dec = get_ad6620_dec (dec,&tmp1,&tmp2,&tmp3,&ovsr);
    dec = dec/2; /* from complex to normal view */
  } 
  else if (p->gchip == GCFPGA) {	/* FPGA tuner */
    mindec = p->isDUC? 1 : (p->order>2)? 1 : (p->order>1)? 2 : (p->tflags&FLG_RESAMP)? 8 : 4;
    maxdec = p->isDUC? 4096 : 0x1<<p->mdw;
    if (p->tflags&FLG_AOVSR) {
      mindec = 2;
      tmp1 = ((mindec-1)/dec)+1;
      if (tmp1<=1) tmp1=0;
      if (tmp1!=ovsr) { ovsr=tmp1; pic_tuner_ovsr (p,ovsr,port,0); }
    }
    if (ovsr>1) dec = dec*ovsr;
    if (ovsr>1) mindec = (mindec+ovsr-1)/ovsr*ovsr;
    dec = max(mindec,min(maxdec,dec));
    if (dec>3 && (dec&1)!=0 && p->isDUC==0) dec++;
    if (ovsr>1) for (dec=dec/ovsr; (dec*ovsr)<mindec || ((dec*ovsr)&0x1)!=0; dec++);
  }
  else {
    ngc = (p->isX||p->isY);
    mindec = ngc? 16*p->cpc/4 : max(16,8*p->cpc);
    if (p->tflags&FLG_AOVSR) {
      tmp1 = ((mindec-1)/dec)+1;
      if (tmp1<=1) tmp1=0;
      if (tmp1!=ovsr) { ovsr=tmp1; pic_tuner_ovsr (p,ovsr,port,0); }
    }
    if (ovsr>1) dec = dec*ovsr;
    maxdec = ngc? p->cpc*2048 : 32768;
    if (ovsr>1) mindec = (mindec+ovsr-1)/ovsr*ovsr;
    dec = max(mindec,min(maxdec,dec));
    if ((dec&1)!=0) dec++;
    /* get the current tuner state if channels are shared */
    if ((p->tflags&FLG_ITDEC)==0) {
      chan = (port-1)>>1;
      chip = p->chiptbl[port];
      lport = p->linktbl[chip];
      sdmac = p->isY? lport2dmacy[lport] : p->isX? lport2dmacx[lport] : lport2dmac[lport];
      nstat = pic_acmd (p, ADON_NSTAT, sdmac, 0, 0);
      mask = NSTAT_MSK(chan) ^ 0xFFFF0000;
      if ((mask&nstat) != 0) dec = NSTAT_DEC*max(1,ovsr);
/*      if (NSTAT_ACT && ngc) dec = NSTAT_DEC*max(1,ovsr); */
    }
    if (ovsr>1) for (dec=dec/ovsr; (dec*ovsr)<mindec || ((dec*ovsr)&0x1)!=0; dec++);
  }
  return dec;
}

int_4 pic_tuner_ovsr (PICSTRUCT *p, int_4 factor, int_4 port, int_4 flags)
{
  int_4 ovsr,maxovsr,dtdmY,pmod;

  ovsr = pic_acmd (p, ADON_RD, DMA_OVSR, 0, 0);
  p->ovsr[0] = ovsr&0xFFFF;
  p->ovsr[1] = ovsr>>16;
  ovsr = findintflag("OVSR",p->config);
  if (ovsr>0)  p->ovsr[0] = p->ovsr[1] = ovsr;
  if (flags==FLG_INFO) return ovsr;
  dtdmY = p->isY && (p->gchip==GC4016);
  pmod = port2pmi(p,&port);

  if (p->type==ICEPIC3) maxovsr=2; else maxovsr=255;
  if (factor < 0 || factor > maxovsr) {
    print("Tuner Oversampling factor %d is out of range [0-%d]\n", factor,maxovsr);
    return -1;
  }
  if (port <= 0) {
    if (p->pindex<=0) port=3;
    else if (p->pindex&1) port=1;
    else if (p->pindex>0) port=2;
  }
  if (port==1 || port==3) { 
    p->ovsr[0] = (int_2)factor;
    if (dtdmY) pic_wpm (p,pmod,PPC_TOVSRA,&factor,4,FLG_PPC_BUS);
    else if (p->isY); /* done in pic_tuner_gcfpga */
    else pic_acmd (p,ADON_WRIOC,IOC_OVSR|IOC_A,factor,0);
  }
  if (port==2 || port==3) {
    p->ovsr[1] = (int_2)factor;
    if (dtdmY) pic_wpm (p,pmod,PPC_TOVSRB,&factor,4,FLG_PPC_BUS);
    else if (p->isY); /* done in pic_tuner_gcfpga */
    else pic_acmd (p,ADON_WRIOC,IOC_OVSR|IOC_B,factor,0);
  }

  ovsr = (p->ovsr[1]<<16) | p->ovsr[0];
  pic_acmd (p, ADON_WR, DMA_OVSR, ovsr, 0);
  vprint("Set OverSampling Ratio to A=%d B=%d\n",p->ovsr[0],p->ovsr[1]);

  return 0;
}

real_8 pic_tuner_ratio (PICSTRUCT *p, int_4 port, double ratio, int_4 flags)
{
  int_4 ic, modulus, all, pmod, cpc, chan, gchip, chip, tadr, reg, n, m, mon, dec, pre, rfx;
  double d, tolr=0.0000001;

  if (flags&FLG_RTUN) return 0;
  pre = (p->tflags&FLG_PRESAMP)!=0;
  gchip = pre? GCFPGA : p->gchip;
  if (pre) ratio *= p->res_dec;

  modulus = (gchip==GCFPGA)? 0x10000000 : 0x04000000;
  reg = (int_4)(modulus/ratio + .5);

  /* find an M over N solution to within given tolerance */
  for (m=1; m<10000; m++) {
    d = m/ratio;
    n = (int)(d+.5);
    if (fabs(d-n)<tolr) break;
  }
  if (gchip!=GCFPGA || m>=10000 || findintflag("NORESMON",p->config)==1) { 
    n=m=0;
    p->res_ratio = (double)modulus / (double)reg;
  } else {
    p->res_ratio = (double)m / (double)n;
  }
  mon = (m<<16)|n;
  p->res_mon = mon;
  p->res_inc = reg;
  vprint("Resampler Ratio: Port=%d Ratio=%f m/n=%d/%d Reg=%08x Actual=%f\n", port,ratio,m,n,reg,p->res_ratio);

  if (flags==FLG_INFO) return p->res_ratio;

  if (port < 0) port=0;
  if (p->tflags&FLG_TBANK && port<100) port+=100;
  if (p->tflags&FLG_TBANK) all=1; 
  else if (p->tflags&FLG_ITDEC) all=0; 
  else all=1; 
  cpc = p->cpc;

  if (pre) {
    pmod = 0;
    chan = 0;
    chip = (port&0x1)?1:2;
  } else {
    pmod = port2pmi(p,&port);
    chan = port2chan(p,port);
    chip = port2chip(p,port);
    if ((p->tflags&FLG_TBANK)==0);
    else if (p->fttm>1);
    else if (gchip!=GCFPGA) chip+=8;
  }
  tadr = chip2tadr(p,chip,pmod);
  if (pmod==0 && p->cpc>1 && p->chan<=0) tadr |= MCORE_ALL;

  pic_lock (p, LOCK_ALLOC);
  if (gchip==GCFPGA) {
    if (p->tflags&FLG_TBANK && chip==3) tadr |= 0xC0000000;
    if (pmod>0) tadr += (all? MCORE_ALL : MCORE_CHN*chan);
    if (pmod==0 && p->mcs>0 && p->chan>0) tadr += MCORE_CHN*(p->chan-1);
    if (pre) tadr = pic5tadrtbl[chip];
    pic_wpb (p,pmod,tadr|GCFPGA_RATIO,reg);
    pic_wpb (p,pmod,tadr|GCFPGA_RMON,mon);
    rfx = 16; /* adjust RFIFX bounds for max throughput */
    if (ratio>1 && ratio<2.4) rfx = (int)(17.5+(2-ratio)*10);
    rfx = findintflagdef("RFX",p->config,rfx);
  } else {
    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,65<<1,0);
    for (ic=0; ic<4; ic++) if (all || chanfor(ic,cpc)==chan) {
      pic_acmd (p,ADON_WRIOB,(tadr|0x10)+ic*4,reg,4);
    }
  }
  pic_lock (p, LOCK_FREE);
  if (p->tflags&FLG_PRESAMP) p->res_ratio /= p->res_dec;
  return p->res_ratio;
}

int_4 pic_tuner_nybw2dec (PICSTRUCT *p, int_4 port, int_4 nybw, int_4 rate, int_4 flags, real_8 *ratio)
{
  int cx = (p->mbits<0)? 2 : 1;
  double irate=rate, orate=nybw;
  int dec; double rdec, drate;
  rdec = irate / orate;
  dec = pic_tuner_dec (p, (int)(rdec/cx+.01), port, 0);
  drate = irate/(dec*cx);
  *ratio = orate/drate;
  vprint("Set port=%d ifs=%d ofs=%d fbwf=%f rdec=%f dec=%d ratio=%f\n",port,rate,nybw,p->fbwf,rdec,dec,*ratio);
  return dec;
}
real_8 pic_tuner_fbwf (PICSTRUCT *p, int_4 port, double fbwf, int_4 flags)
{
  return fbwf;
}

/*
 TestPort Multi-Purpose Connector IO Pins
 eval == 0xEEVV - 8bits of OutputEnable=E and Value=V
*/
int_4 setTpMpc (PICSTRUCT *p, int_4 eval) {
  int_4 bit,mix;
  WR(REG_IOC, REG_IOC_ADDR|PM_TPORT_MPC);
  for (bit=0; bit<8; bit+=2) {	/* addressable 2 bits at a time */
    mix = (bit<<4) | (((eval>>bit)&0x300)>>6) | ((eval>>bit)&0x3);
    WR(REG_IOC,mix);
  }
  WR(REG_IOC, REG_IOC_ADDR|0);
  return 0;
}

/*
 Multi-Purpose Connector IO Pins 
 Currently implements 1 to 4 I2C interfaces 
 Each port uses 4 wires as  [sda_oe,scl_oe,sda,scl]
 SSA is 1=Start -1=Stop 0=Ack
 DWR writes 8 bits with ack
 DRD reads 8 bits with ack
 on MB chan 3-0 status returned in one byte as DDCCBBAA data|clock pairs
 on IOM chan 0 status returned as xxAAxxxx data|clock pair

 addr=SSAAOODD
  SS= 0,2,4,6 for port number
  AA is I2C Address << 1 with read bit
  OO is Command Address
  DD is Command Data
*/

#define MPCDELAY udelay(5)

int_4 mpc_acc (PICSTRUCT *p, int_4 sel, int_4 data, int_4 dir)
{
  int_4 node,page;
  if (sel<8) {	/* on main board */
    if (dir>0) WR(REG_IOC,(sel<<4)|data)
    else  { RD(REG_IOC,data); data >>= sel; } /* shift cur sel data bit to 0x2 */
  } 
  else {	/* on IO module  node=1|2 */
    node = sel>>4;
    page = sel & 0x0F;
    if (dir>0) pic_jpmemwr (p, node, page<<8, data, 1);
    else { data = pic_jpmemrd (p, node, page<<8, 1); data >>= 4; } /* shift cur sel data bit to 0x2 */
  }
  MPCDELAY;
  return data;
}

void mpc_ssa (PICSTRUCT *p, int_4 sel, int_4 data)
{
  int_4 i;
  if (data>0) {
    mpc_acc(p,sel,0xF,1);
    mpc_acc(p,sel,0xD,1);
  }
  else if (data<0) {
    mpc_acc(p,sel,0xD,1);
    mpc_acc(p,sel,0xF,1);
  }
  else {
    mpc_acc(p,sel,0x5,1);
  }
}

void mpc_dwr (PICSTRUCT *p, int_4 sel, int_4 data)
{
  int_4 i,ii,j=0;
  mpc_acc(p,sel,0xC,1);
  for (ii=0; ii<8; ii++) { /* data */
    j = (data&0x80)? 2:0;
    mpc_acc(p,sel,0xC|j,1);
    mpc_acc(p,sel,0xD|j,1);
    mpc_acc(p,sel,0xC|j,1);
    data <<= 1;
  }
  mpc_acc(p,sel,0x4,1);
  mpc_acc(p,sel,0x5,1);
  mpc_acc(p,sel,0xC,1);
}

int_4 mpc_drd (PICSTRUCT *p, int_4 sel, int_4 nack)
{
  int_4 i,ii,j=0,data=0;
  for (ii=0; ii<8; ii++) { /* data */
    mpc_acc(p,sel,0x4,1);
    mpc_acc(p,sel,0x5,1);
    i = mpc_acc(p,sel,0x0,-1);
    if (i&0x2) data |= (0x1<<(7-ii));
  }
  j = nack? 2 : 0;
  mpc_acc(p,sel,0x4,1);
  mpc_acc(p,sel,0xC|j,1);
  mpc_acc(p,sel,0xD|j,1);
  mpc_acc(p,sel,0xC|j,1);
  mpc_acc(p,sel,0xC,1);
  return data;
}

int_4 mpc_read (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len)
{
  int_4 i,sel,uaddr;
  sel=(addr>>24)&0xFF;
  uaddr=(addr>>16)&0xFE;
  addr&=0xFF;
  if (sel<8) { WR(REG_IOC, REG_IOC_ADDR|PM_TPORT_MPC); MPCDELAY; }
  mpc_ssa(p,sel,1);
  mpc_dwr(p,sel,uaddr);
  mpc_dwr(p,sel,addr);
  mpc_ssa(p,sel,-1);
  mpc_ssa(p,sel,1);
  mpc_dwr(p,sel,uaddr|0x1); /* set the read bit */
  for (i=0; i<len; i++) data[i] = mpc_drd(p,sel,(i==len-1)?1:0);
  mpc_ssa(p,sel,-1);
  return 0;
}

int_4 mpc_write (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len)
{
  int_4 i,sel,uaddr;
  sel=(addr>>24)&0xFFF;
  uaddr=(addr>>16)&0xFE;
  addr&=0xFF;
  if (sel<8) { WR(REG_IOC, REG_IOC_ADDR|PM_TPORT_MPC); MPCDELAY; }
  mpc_ssa(p,sel,1);
  mpc_dwr(p,sel,uaddr);
  mpc_dwr(p,sel,addr);
  for (i=0; i<len; i++) mpc_dwr(p,sel,data[i]);
  mpc_ssa(p,sel,-1);
  return 0;
}

int_4 mpc_writem (PICSTRUCT *p, int_4 addr, int_1 data, int_1 mask)
{
  int_1 datar,dataw,datarb,diff, imask=~mask, retry=3;
  if (mask==0) return data;
  RETRY:
  if (mask&0xFF==0xFF) {
    datar = 0;
    dataw = data;
  } else {
    mpc_read (p,addr,&datar,1);
    dataw = (datar & imask) | (data & mask);
  }
  mpc_write (p,addr,&dataw,1);
  mpc_read (p,addr,&datarb,1);
  diff = dataw^datarb;
  if (diff!=0) print("writem a=%08x r=%02hhx d=%02hhx m=%02hhx w=%02hhx rb=%02hhx diff=%02hhx try=%d\n",addr,datar,data,mask,dataw,datarb,diff,retry);
  if (diff!=0 && --retry>0) goto RETRY;
  return (int_4)dataw;
}

int_4 ampc_read (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len)
{
   int_4 i, ldata;
   for (i=0; i<len; i++) { 
     if (i%4==0) ldata = pic_acmd (p,ADON_MPC,addr+i,0,-min(4,len-i));
     data[i]=(ldata&0xFF);
     ldata=ldata>>8;
   }
   return len;
}

int_4 ampc_write (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len)
{
   int_4 i, j, ldat, ldata=0; 
   for (i=j=0; i<len; ) {
     if (i%4==0) { ldata=0; j=i; }
     ldat = data[i];
     ldata |= (ldat&0xFF)<<((i-j)*8); i++;
     if (i%4==0 || i==len) pic_acmd (p,ADON_MPC,addr+j,ldata,i-j); 
   }
   return len;
}

int_4 ampc_writem (PICSTRUCT *p, int_4 addr, int_1 data, int_1 mask)
{
  int_1 datar,dataw,datarb,diff, imask=~mask, retry=3;
  if (mask==0) return data;
  RETRY:
  if (mask&0xFF==0xFF) {
    datar = 0;
    dataw = data;
  } else {
    ampc_read (p,addr,&datar,1);
    dataw = (datar & imask) | (data & mask);
  }
  ampc_write (p,addr,&dataw,1);
  ampc_read (p,addr,&datarb,1);
  diff = dataw^datarb;
  if (diff!=0) print("writem a=%08x r=%02hhx d=%02hhx m=%02hhx w=%02hhx rb=%02hhx diff=%02hhx try=%d\n",addr,datar,data,mask,dataw,datarb,diff,retry);
  if (diff!=0 && --retry>0) goto RETRY;
  return (int_4)dataw;
}

#define TEST_ADDR DMA_INT_ADDR
#define TEST_SIZE DMA_INT_SIZE

#define SPEC_SIZE 0x00100000
#define SPEC_BLKX 0x00001000
#define SPEC_BLKY 0x00010000

int_4 pic_test (PICSTRUCT *p, int_4 test, int_4 flags)
{
  int_4 i,j,k,k1,k2,py,n,nb,count,address,errors,lerrors;
  int_4 status,*array,mdata[2],mask;
  int_4 ndata,nbytes,nblock,nstart,*data;
  int_4 maxdata,maxsjump,maxbjump;
  int_4 verify=0,ioc,pci,port,freg,fdat;
  float maxsrate, maxbrate;
  int_u1 fdata[20], *fdat_ptr;
  double d;

  pci = 1;
  ioc = 1;

  if (p->ptype != IOPT_NONE) {
    print("Cannot run tests on specific card ports. Check configuration.\n");
    return -1;
  }

  p->flags &= ~(FLG_VHS|FLG_DUAL);	/* disable improper flags */

  if (flags==1) verify=1; else verify=0;
  port = (p->isY)? 0 : JTAG_PORT_PMOFF+1; 
  i = findintflag("TP",p->config);
  n = findintflag("TPCNT",p->config);
  if (i>=0) port = i;
  if (test==91) pic_lock (p, LOCK_CLEAR);
  pic_lock (p, LOCK_ALLOC);
  errors = lerrors = 0;

  if (test == 1 || test == 0) {			/* reset test */
    tprint("Testing RESET     ... ");
    status = pic_reset (p,0);
    if (status != 1) errors++;
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
    if (test == 0 && status != 1) return -1;
  }

  if (test == 2 || test == 0) {			/* mailbox test */
    tprint("Testing MBX/REG   ... ");
    count = TEST_SIZE;
    for (i=0; i<count; i+=4) {
      k=i+1;	WR (REG_IMB1,k);
      k=-(i+3);	WR (REG_OMB3,k);
      k=i+2;	WR (REG_IMB2,k);
      k=-(i+4);	WR (REG_OMB4,k);
      k=i<<2;	WR (REG_MDTC,k);
      k=i<<4;	WR (REG_MDAR,k);
      k=-(i+3);	RD (REG_OMB3,j); if (j!=k) { RD (REG_OMB3,k2); errors++; tprint("OMB3 got=%x expected=%x  %x\n",j,k,k2); }
      k=-(i+4);	RD (REG_OMB4,j); if (j!=k) { RD (REG_OMB4,k2); errors++; tprint("OMB4 got=%x expected=%x  %x\n",j,k,k2); }
      k=i+1;	RD (REG_IMB1,j); if (j!=k) { RD (REG_IMB1,k2); errors++; tprint("IMB1 got=%x expected=%x  %x\n",j,k,k2); }
      k=i+2;	RD (REG_IMB2,j); if (j!=k) { RD (REG_IMB2,k2); errors++; tprint("IMB2 got=%x expected=%x  %x\n",j,k,k2); }
      k=i<<2;	RD (REG_MDTC,j); if (j!=k) { RD (REG_MDTC,k2); errors++; tprint("MDTC got=%x expected=%x  %x\n",j,k,k2); }
      k=i<<4;	RD (REG_MDAR,j); if (j!=k) { RD (REG_MDAR,k2); errors++; tprint("MDAR got=%x expected=%x  %x\n",j,k,k2); }
    }
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
  }

  if (test == 3 || test == 0) {			/* memory test */
    tprint("Testing CARD mem  ... ");
    count = TEST_SIZE; 
    address = TEST_ADDR;
    for (i=0; i<count; i++) {
      pic_acmd (p,ADON_WR,address+i,-1,0);
    }
    for (i=0; i<count; i++) {
      if (i&0x1) k=-i; else k=i;
      j = pic_acmd (p,ADON_WR,address+i,k,0);
      if (j!=k) { errors++; tprint ("MEM Write Got %x expected %x\n",j,k); }
    }
    for (i=0; i<count; i++) {
      j = pic_acmd (p,ADON_RD,address+i,0,0);
      if (i&0x1) k=-i; else k=i;
      if (j!=k) { errors++; tprint ("MEM Got %x expected %x\n",j,k); }
    }
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
  }

  if ((test == 4 || test == 0) && pci) {	/* fifo test */
    tprint("Testing FIFO      ... ");
    py = p->isY;
    count = TEST_SIZE;
    WR(REG_MCSR, MCSR_RFIFO);
    for (i=0; i<count; i++) {
      if (i&0x1) k=-i; else k=i;
      if (py) pic_acmd (p,ADON_WR,REG_AFIFO,k,0); else WR (REG_FIFO,k);
      RD (REG_FIFO,j);
      if (j!=k) { errors++; tprint ("FIFO1 Got %x expected %x\n",j,k); }
    }
    WR(REG_MCSR, MCSR_RFIFO);
    /* now keep two entry in FIFO at all times */
    if (py) pic_acmd (p,ADON_WR,REG_AFIFO,0,0); else WR (REG_FIFO,0);
    if (py) pic_acmd (p,ADON_WR,REG_AFIFO,-1,0); else WR (REG_FIFO,-1);
    for (i=0; i<count; i++) {
      if (i&0x1) k=-(i+2); else k=(i+2);
      if (py) pic_acmd (p,ADON_WR,REG_AFIFO,k,0); else WR (REG_FIFO,k);
      RD (REG_FIFO,j);
      if (i&0x1) k=-i; else k=i;
      if (j!=k) { errors++; tprint ("FIFO2 Got %x expected %x\n",j,k); }
    } 
    WR(REG_MCSR, MCSR_RFIFO);
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
  }

  if (test == 5 || test == 0) {			/* IOC test */
    tprint("Testing IOC Reg   ... ");
    mask = p->isY? 0xFFFFFFFF : 0xFF;
    nb   = p->isY? 4 : 1;
    for (n=0; n<100 && errors<10; n++) {
      count = min(TEST_SIZE,32);		/* 32 bytes of RAM per side starts at IOC_RAM=0x20 */
      for (i=0; i<count; i+=4) {
        k = i+(n&0x1); k1 = -k*5; k2 = -k & mask;
        pic_acmd (p,ADON_WRIOC,(IOC_RAM|IOC_A)+i,k1,nb);
        pic_acmd (p,ADON_WRIOC,(IOC_RAM|IOC_B)+i,k2,nb);
      }
      if  (p->isX && (p->iocc[2]=='S' || p->iocc[3]=='S')) continue;
      for (i=0; i<count; i+=4) {
        k = i+(n&0x1); k1 = -k*5; k2 = -k & mask;
        j = pic_acmd (p,ADON_RDIOC,(IOC_RAM|IOC_A)+i,0,1);
        if (j!=k1) k = pic_acmd (p,ADON_RDIOC,(IOC_RAM|IOC_A)+i,0,nb); 
        if (j!=k1) { errors++; tprint("IOC IOM A got=%08x expect=%08x at=%x n=%d loop=%d\n",j,k1,i,(k==j)?2:1,n); }
        j = pic_acmd (p,ADON_RDIOC,(IOC_RAM|IOC_B)+i,0,1);
        if (j!=k2) k = pic_acmd (p,ADON_RDIOC,(IOC_RAM|IOC_B)+i,0,nb); /* */
        if (j!=k2) { errors++; tprint("IOC IOM B got=%08x expect=%08x at=%x n=%d loop=%d\n",j,k2,i,(k==j)?2:1,n); }
      }
    }
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
  }

  if ((test == 6 || test == 0) && !p->k7) {			/* dma test */
    tprint("Testing CARD DMA  ... ");
    count = TEST_SIZE;
    address = TEST_ADDR;
    array = (int_4 *)malloc(TEST_SIZE*4);
    for (i=0; i<count; i++) array[i]=i;
    pic_transfer (p, array, count*4, address, 1, 1);
    for (i=0; i<count; i++) array[i]=0;
    pic_acmd (p, ADON_WR, DMA_QUE, 0, 0);
    pic_dma (p, 1, -1, array, 0x0, count*4, 0, FLG_TEST);
    pic_dmafunc (p, 1, DMA_ONESHOT);
    pic_dmafunc (p, 1, DMA_WAIT);
    pic_acmd (p, ADON_WR, DMA_QUE, 0, 0);
    for (i=0; i<count; i++) { j = array[i];
      if (j!=i) { errors++; vprint ("DMA Got %x expected %x of %x\n",j,i,count); }
    }
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
    free(array);
  }

  if ((test == 7 || test == 0) && pci) {	/* PCI bus spec test */
    tprint("Testing PCI bus DMA lanes=%d ... \n",p->pbw);
    ndata  = findintflagdef("SIZE",p->config,SPEC_SIZE);
    nblock = findintflagdef("BLOCK",p->config,p->isY? SPEC_BLKY:SPEC_BLKX);
    nbytes = ndata * 4;
    status = pic_map (p, &data, &nstart, nbytes, 1);
    if (status < 0) {
      tprint ("\nCould not map buffer of size %d: Test aborted\n",nbytes);
      errors++; goto BADMAP;
    }
    for (i=0; i<ndata; i++) data[i]=0xFFFFFFFF;
    if (port!=2) {
      pic_spec (p, data, nstart, ndata, -nblock, &maxdata, &maxsrate, &maxsjump, &maxbrate, &maxbjump, verify);
    }
    if (port!=1) {
      pic_spec (p, data, nstart, ndata, nblock, &maxdata, &maxsrate, &maxsjump, &maxbrate, &maxbjump, verify);
    }
    if (p->verbose==1) {
      print("Top -> ");
      for (i=0; i<8; i++) print("%08x ",data[i]);
      print("\nMid -> "); 
      for (i=0; i<8; i++) print("%08x ",data[maxdata/2+i]);
      print("\nEnd -> "); if (maxdata>8)
      for (i=-8; i<0; i++) print("%08x ",data[maxdata+i]);
      print("\n"); 
    }
    status = pic_map (p, &data, &nstart, nbytes, -1);
    BADMAP:
    lerrors = errors;
  }

  if (test == 8) {				/* main test */
    print("Testing POD6 PCIe settings ... \n");
    n=findintflag("TXPA",p->config);
    if (n>0) { j=0; j=findintflagdef("TXPD",p->config,j); WR(REG_IOC, REG_IOC_AINC|0x0200|(n<<1)); for (i=0; i<2; i++) WR(REG_IOC, (j>>(8*i))&0xFF); }
    for (n=0x40; n<0x80; n+=1) {
      WR(REG_IOC, REG_IOC_AINC|0x0200|(n<<1));
      for (i=0,j=0; i<2; i++) { RD(REG_IOC, status);  j |= ((status&0xFF)<<(i*8)); }
      printf("DRP a=%04x d=%04x\n",n,j);
    }
  }

  if (test == 9) {				/* test of the day */
    tprint("Test of the day-1    ... \n");
    if (port<=0) port=1;
    address = IOC_PROG|((port==2)?IOC_B:IOC_A); /* 1=IOMA 2=IOMB */
    pic_acmd (p,ADON_WRIOC,address,IOP_RCLK,0); udelay(10000);
    print("Port=%d Cnt=%d Addr=%08x ... \n",port,n,address);
    pic_lock (p, LOCK_FREE);
    if (n>=0) for (i=0; i<n; i++) { pic_acmd (p,ADON_WRIOC,address,IOP_ACLKU,0); udelay(10000); }
    else for (i=0; i<(-n); i++)   { pic_acmd (p,ADON_WRIOC,address,IOP_ACLKD,0); udelay(10000); }
    pic_lock (p, LOCK_ALLOC);
    address = IOC_TAPS|((port==2)?IOC_B:IOC_A); /* 1=IOMA 2=IOMB */
    n = (n<=0)? 0 : (0x1<<(n-1));
    pic_acmd (p,ADON_WRIOC, address, n, 0); 
  }


  if (test == 10) {                             /* test of the day */
    iom_init(p,port);
    for (i=0; i<8; i++) {
      k = iom_rd (p,port,OIP_CALIB|(i<<2),4);
      k1 =  ((k>> 8)&0xF) + ((k>>12)&0xF);
      k2 =  ((k>>16)&0xF) + ((k>>20)&0xF);
      printf("port=%d calib=%08x w1=%d w2=%d\n",port,k,k1,k2);
    }
  }

  if (test == 11) {				/* test of the day */
    iom_init(p,port);
    iom_wr(p,port,OIP_PROG,0,0x4);
    for (i=0; i<64; i++) {
      j = (i<16||i>=48);
      iom_wr(p,port,OIP_PROG,(j)?OIP_ACLKU:OIP_ACLKD,1);
      iom_exec(p,port);
      mdata[0] = iom_rd (p,port,OIP_CALIB,4);
      if (j==0) printf("port=%d i=%+d calib=%08x\n",port,i-32,mdata[0]);
    }
    iom_wr(p,port,OIP_PROG,0,0x0);
  }

  if (test == 12) {				/* test of the day */
    iom_init(p,port);
    mdata[0] = iom_rd (p,port,OIP_CALIB,4);
    printf("port=%d calib=%08x\n",port,mdata[0]);
  }

  if (test == 13) {				/* test of the day */
    i=0x0100000F; pic_wpm(p,port,0xfd813F48,&i,4,0);
    i=0x0180000F; pic_wpm(p,port,0xfd813F4C,&i,4,0);
    pic_wpb(p,port,PPC_DMAC_ROUTE_SET,PPC_ROUTE_HP2R);
  }

  if (test == 14) {				/* test of the day */
   i = IOC_PROG|((port&1)?IOC_A:IOC_B);
   print("Adjust IOC port=%d taps=%d\n",port,n);
   if (n==0) pic_acmd (p,ADON_WRIOC,i,IOP_RCLK,0); 
   for (;n>0; n--) pic_acmd (p,ADON_WRIOC,i,IOP_ACLKU,0); 
   for (;n<-1; n++) pic_acmd (p,ADON_WRIOC,i,IOP_ACLKD,0); 
  }

  if (test == 15) {				/* test of the day */
   i = IOC_PROG|((port&1)?IOC_A:IOC_B);
   print("Adjust OIP port=%d taps=%d\n",port,n);
   if (n==0)        { mdata[0]=OIP_PROG; mdata[1]=OIP_RCLK;   pic_msg (p,port,PKTF_BUS_RWM,OIP_CALIB,mdata,8,1*4,10000); }
   for (;n>0; n--)  { mdata[0]=OIP_PROG; mdata[1]=OIP_ACLKU;  pic_msg (p,port,PKTF_BUS_RWM,OIP_CALIB,mdata,8,1*4,10000); }
   for (;n<-1; n++) { mdata[0]=OIP_PROG; mdata[1]=OIP_ACLKD;  pic_msg (p,port,PKTF_BUS_RWM,OIP_CALIB,mdata,8,1*4,10000); }
  }

  if (test == 16) {				/* test of the day */
    i = findintflagdef("MPCA",p->config,0xF8);
    n = findintflagdef("MPCN",p->config,8);
    mdata[0] = findintflagdef("MPCD0",p->config,0);
    mdata[1] = findintflagdef("MPCD1",p->config,0);
    mpc_write(p,i,(int_1*)mdata,n);
  }

  if (test == 17) {				/* test of the day */
    if (n<0) n=1;
    print("Inject PCIe NACK reg=0x%02x\n",n);
    WR(REG_IOC, REG_IOC_ADDR|0x10000);
    WR(REG_IOC, n&0xFF); 
    WR(REG_IOC, REG_IOC_ADDR);
  }

  if (test == 18 || test == 1819) {		/* test of the day */
    if (p->isY) test_memory (p, port, MTEST_MB2PM, 0);
  }
  if (test == 19 || test == 1819) {		/* test of the day */
    if (p->isY) test_memory (p, port, MTEST_PM2MB, 0);
  }

  if (test == 20) {				/* test of the day */
    print("Port=%d n=%x\n",port,n);
    pic_acmd (p,ADON_WRIOB,getJTAGsel(p,port)|0xC000,n,1);
  }

  if (test == 22) {				/* test of the day */
    print("SRCFG Port=%d n=%x\n",port,n);
    pic_wpb (p,port,PPC_DMAC_SRCFG,n);
  }
  if (test == 23) {				/* test of the day */
    n = findintflagdef("MR0",p->config,SD4_MRS_CL15BL8);
    n = findintflagdef("MR1",p->config,SD4_EMRS1_NORM);
    print("MEM REGn Port=%d n=%x\n",port,n);
    mdata[0]=n; pic_msg (p,port,PKTF_MEM_INIT,0,mdata,4,0,-1); 
  }
  if (test == 24) {				/* test of the day */
    d = finddblflagdef("QSFPREF",p->config,156.25);
    setup_SI514(p,d);
  }
  if (test == 25) {				/* test of the day */
    status = pic_loadfile (p, "icenvme", FLG_JVM_NVME|port);
  }
  if (test == 26) {				/* test of the day */
    status = pic_loadfile (p, "icenio", FLG_JVM_NIO|port);
  }

  if (test == 30) {
    freg = findintflagdef("FREG",p->config,0);
    fdat = findintflagdef("FDAT",p->config,0);
    n    = findintflagdef("NBYTES",p->config,0);

    if (freg==FCMD_WRITE_NVCFG) {
      printf("Don't Do This...\n");
    } else {
      fdat_ptr = (int_u1*)&fdat;
      for(i=0;i<20;i++) fdata[i] = 0;
      for(i=0;i<abs(n);i++) fdata[i] = *(fdat_ptr++);
      spi_flash (p,FCMD_WRITE_ENABLE,0,0,fdata,0);
      spi_flash (p,freg,0,0,fdata,n);
      spi_flash (p,FCMD_WRITE_DISABLE,0,0,fdata,0);
      if (n<0) printf("Flash STATUS "); for (i=0; i<abs(n); i++) printf(" %02x",fdata[i]); printf("\n");
    }
  } 

  if (test>=70 && test<=81) {
    pic_lock (p, LOCK_FREE);
    errors = test_memory (p,port,test-70,0);
    pic_lock (p, LOCK_ALLOC);
  }

  if (test == 91) {			/* reattach card through driver */
    tprint("ReAttaching card ... \n");
    status = io_ctl (p, IOCTL_REATTACH, 0, 0, 0);
    i = pic_nv_read_b(p,0x08);
    tprint("EPROM Revision %02x\n",i&0xFF);
  }

  if (test == 92) {				/* 64-bit loopback test */
    tprint("Performing 64-bit pin loopback test ... \n");
    if (p->type==ICEPIC2 || p->type==ICEMBT2 || p->type==ICESLIC3) { 
      tprint("Not supported on series 2 cards \n"); return 0; }
    if (p->pbw==8) { 
      tprint("Should be on a 32-bit bus for this \n"); return 0; }
    count = 1000;
    for (i=0; i<count; i++) {
      if (i&0x1) k=0x55555555; else k=0xAAAAAAAA;
      WR(REG_OMB4, k);
      WR(REG_MCSR, MCSR_RESET|MCSR_WIDE);
      if ((i&0xF)==0) { k=0; WR(REG_MCSR, MCSR_RESET|MCSR_WIDE|MCSR_TEST);}
      WR(REG_OMB4, 0x333);
      WR(REG_MCSR, MCSR_RESET);
      RD(REG_OMB4, status);
      if (status != k) { vprint("Oops got %x not %x\n",status,k); errors++; }
    }
  }

  if (test == 93) {			/* override 64-bit bus register */
    WR(REG_MCSR,MCSR_BUS64);
  }
  if (test == 94) {				/* mem test all offsets */
    errors = pic_dumpslr (p,0);
  }

  if (test == 96) {				/* test of the day */
    tprint("Performing timecode counter tests ... \n");
    for (i=0; i<0x10000000; i++) {
      j = gencount1(i); k = fixcount1(j);
      if (k!=i) { print("Bad fixcount1 at i=%08x j=%08x k=%08x\n",i,j,k); errors++; }
    }
    print("Single clutch %d bad of %d\n",errors,i); errors=0;
    for (i=0; i<0x10000000; i++) {
      j = gencount2(i); k = fixcount2(j);
      if (k!=i) { print("Bad fixcount2 at i=%08x j=%08x k=%08x\n",i,j,k); errors++; }
    }
    print("Double clutch %d bad of %d\n",errors,i);
  }

  if (test == 97) {				/* processor mem test */
    errors = pic_dump_qsfp (p,-1);
  }
  if (test == 98) {				/* processor mem test */
    errors = pic_test_qsfp (p,-1);
  }

  if (test == 99) {				/* processor mem test */
    if (port>0 && (p->pmtype[port-7]<=PMT_NONE || p->pmtype[port-7]==PMT_S6M)) {
      tprint("Bypass PRC DDR DRAM test\n");
    } else {
      i = test_memory (p,port,MTEST_BOTH,0); errors+=i;
      j = test_memory (p,port,MTEST_FULL,0); errors+=j;
      tprint("Testing PRC DDR DRAM chips ... ");
      tprint("%d+%d errs \n",i,j); lerrors = errors;
    }
  }

  if (test == 100) {				/* mem test all offsets */
    errors = spi_checkflash (p);
  }

  if (test > 100 && test < 165) {		/* tuner register dump */
    if (p->type==ICEPIC3) pic_dump_ad6620 (p, test-100);
    if (p->gchip==GC4016) pic_dump_gc4016 (p, test-100);
    if (p->gchip==GC4014) pic_dump_gc4014 (p, test-100);
    if (p->gchip==GCFPGA) pic_dump_gcfpga (p, test-100);
  }

#include "pic9tests.c"

  if (test > 1000 && test < 1200) {		/* timecode generator test */
    status = (test-1000);
    tprint("Output test white/tc=sdn0 gen at %d MHz with MUXCLK=P\n",status);
    pic_timer (p,FLG_PCLK,status*1000000);
    count = (status*1000)-1;
    pic_acmd (p,ADON_WRIOC,IOC_GMC,GMC_CLKP-1,1); 
    status = IOM_ENA|IOM_PLY|IOM_COE|IOM_LEN;
    getIOCsig(p,0);
    if (p->isY) {
      set_iom_sequence (p,3,status,0,count,0);
      set_tcgen_sequence (p,3,count, port);
      pic_acmd (p, ADON_MOD, 3, status|IOM_LSBX, 0); 
    } else {
      pic_acmd (p, ADON_WRIOC, IOC_AB|0x10,count,3); 
      pic_acmd (p, ADON_MOD, 3, status, 0); 
      pic_acmd (p, ADON_MOD, 2, status|IOM_UOPT, 0); 
    }
  }

  if (test > 2000 && test < 2200) {		/* timecode glitch test */
    status = (test-2000);
    tprint("Output glitch at %d MHz with MUXCLK=P\n",status);
    status = (status*1000)-1;
    pic_acmd (p,ADON_WRIOC,IOC_AB|0x10,status+1,2); 
    udelay(1000);
    pic_acmd (p,ADON_WRIOC,IOC_AB|0x10,status,2); 
  }

  if (test == 0) {                              /* ACMD checksum test */
    tprint("Testing Checksum  ... ");
    if (p->isY) j=0;
    else j = pic_acmd (p, ADON_RD, MSGR6, 0, 0);
    errors += j;
    tprint("%d errs \n",errors-lerrors); lerrors = errors;
  }

  pic_lock (p, LOCK_FREE);

  return (errors);
}

/*
  Port Memory Map access through the PPC only - see iceppc.h
  Supports naturally aligned 16|32|64 byte cache line read/writes
  Only _A1 and _A2 support 64 byte read/writes
*/

#define MEM_TEST_A1 0x00100040
#define MEM_TEST_A2 0x001000C0
#define MEM_TEST_A3 0x00108000

int_4 testpatt1[] = {0x01234567,0x76543210,0x12345678,0x87654321,0x23456789,0x98765432,0x34567890,0x09876543,0x45678901,0x10987654,0x56789012,0x21098765,0x67890123,0x32109876,0x78901234,0x43210987};
int_4 testpatt2[] = {0x1E1E1E1E,0x2D2D2D2D,0x3C3C3C3C,0x4B4B4B4B,0x5A5A5A5A,0x69696969,0x78787878,0x87878787,0x96969696,0xA5A5A5A5,0xB4B4B4B4,0xC3C3C3C3,0xD2D2D2D2,0xE1E1E1E1,0xF0F0F0F0,0x0F0F0F0F};
int_4 testpatt3[] = {0x11111111,0x22222222,0x33333333,0x44444444,0x55555555,0x66666666,0x77777777,0x88888888,0x99999999,0xAAAAAAAA,0xBBBBBBBB,0xCCCCCCCC,0xDDDDDDDD,0xEEEEEEEE,0xFFFFFFFF,0x00000000};

int_4 testlrs[] = {0x6B34359A,0xB8D2DC69,0xDD466BA3,0x6F18BA8C,0xC862E431,0x1B8690C3,0x6E19370C,0xC463E232,
		   0x0B8E88C7,0x2E37171B,0xBEDB5C6D,0xF56B77B5,0xD5AAEAD5,0x50A9AB55,0x48A5A752,0x28939749,
		   0xA84C5126,0x9B304A98,0x60BE305F,0x8EF9C77C,0x2FE217F1,0xC58A5FC5,0x10278B13,0x409B204D};

/* write processor bus */
int_4 pic_wpbx (PICSTRUCT *p, int_4 node, int_4 addr, int_4 data) {
  int_4 head[2],sel,stat=0;
  if (p->type==ICEPIC8) {
    if (node>=1 && node<=p->mpm) node += JTAG_PORT_PMOFF;
    if (node==3 || node==5) node = JTAG_PORT_PMALL;
    sel = (node==0)? 0 : getJTAGsel(p,node);
    stat = pic_acmd(p,ADON_WRB,addr,data,sel|4);
  } else if (p->devno==-3) {
    stat = FPGA_syswr(p->halo,addr,data);
  } else if (node==0) {
    stat = pic_acmd(p,ADON_WR,(addr>>2),data,4);
  } else {
    head[0] = (0x4<<16)|PKTF_BUS_WR;
    head[1] = addr;
    stat = pic_sendrecv (p,node,head,&data,4,0,PKT_ACMD,64);
    if (stat==0) stat=4;
  }
  return stat;
}

int_4 pic_wpb (PICSTRUCT *p, int_4 node, int_4 addr, int_4 data) {
  int sel=node<<16;
  if (p->type==ICEPIC8) {
    if (node<0 || node>3) printf("Unexpected node=%d in pic_wpb\n",node);
    return pic_acmd(p,ADON_WRB,addr,data,sel|4);
  }
  return pic_wpbx(p,node,addr,data);
}

/* read processor bus */
int_4 pic_rpbx (PICSTRUCT *p, int_4 node, int_4 addr) {
  int_4 head[2],stat=0;
  if (p->devno==-3) return FPGA_sysrd(p->halo,addr);
  if (node==0 && (addr&0xFF000000)!=0) {
    stat = pic_acmd(p,ADON_RD,(addr>>2),0,4);
  } else {
    if (node==3) node=1; /* read from 1st of two nodes */
    head[0] = (0x4<<16)|PKTF_BUS_RD;
    head[1] = addr;
    pic_sendrecv (p,node,head,&stat,0,4,PKT_ACMD,8);
  }
  return stat;
}

int_4 pic_rpb (PICSTRUCT *p, int_4 node, int_4 addr) {
  int sel=node<<16,stat;
  if (p->type==ICEPIC8) {
    if (node<0 || node>3) printf("Unexpected node=%d in pic_rpb\n",node);
    stat = pic_acmd(p,ADON_WRB,addr,0,sel|0);	/* WRB of 0 bytes performs a read */
    if (stat==0xffff9999) printf("Timeout pic_rpb node=%d addr=%08x\n",node,addr);
    return stat;
  }
  return pic_rpbx(p,node,addr);
}

/* write port memory map */
int_4 pic_wpm (PICSTRUCT *p, int_4 node, int_4 addr, int_4 *data, int_4 bytes, int_4 flags) {
  int_4 n,stat,status=0,head[2],maxby=1024;
  if (flags&FLG_CTL_BUS) maxby=4;
  while (bytes>0) {
    n = min(bytes,maxby);
    head[0] = ((flags&FLG_PPC_BUS)!=0)? PKTF_BUS_WR : PKTF_MEM_WR;
    head[0] |= (n<<16);
    head[1] = addr;
    if (flags&FLG_CTL_BUS) {
      pic_jpmemwr (p, node, addr, *data, n);
      stat = n;
    } else {
      stat = pic_send (p,node,head,data,n,PKT_ACMD);
    }
    if (stat<0) return stat; 
    status += stat;
    if ((flags&FLG_PPC_BUS)==0) addr += n;
    data += (n>>2);
    bytes -= n;
  }
  return status;
}

/* read port memory map */
int_4 pic_rpm (PICSTRUCT *p, int_4 node, int_4 addr, int_4 *data, int_4 bytes, int_4 flags) {
  int_4 n,stat,status=0,head[2],maxby=1024;
  if (flags&FLG_CTL_BUS) maxby=4;
  while (bytes>0) {
    n = min(bytes,maxby);
    head[0] = ((flags&FLG_PPC_BUS)!=0)? PKTF_BUS_RD : PKTF_MEM_RD;
    head[0] |= (n<<16);
    head[1] = addr;
    if (flags&FLG_CTL_BUS) {
      *data = pic_jpmemrd (p, node, addr, n);
      stat = n;
    } else {
      stat = pic_sendrecv (p,node,head,data,0,n,PKT_ACMD,8);
    }
    if (stat<0) return stat; 
    status += stat;
    if ((flags&FLG_PPC_BUS)==0) addr += n;
    data += (n>>2);
    bytes -= n;
  }
  return status;
}

/* adjust the hypertransport clock on the processor module */
int_4 clkadj (PICSTRUCT *p, int_4 port, int_4 clkadr, int_4 value, int_4 n) {
  int_4 data[256];
  data[0]=clkadr; data[1]=value; 
  if (n>0) pic_msg (p,port,PKTF_BUS_RWM,0,data,8,n*4,1000);
  return n;
}

/* parse the calibration results to find the center */
int_4 calibrate (PICSTRUCT *p, char *func, int_4 port, int_4 *data, int_4 n, int_4 verbose) {
  int_4 i,j,k,q0,errs,lerrs=0,tpp,minwin,minstr;
  if (verbose<0) verbose = p->verbose;
  tpp = (n>128)? 256 : n>32? 64 : 32;
  minwin = tpp/16;
  minstr = tpp/64;
  for (q0=-1,i=k=0; i<n; i++) {
    if (data[i]==-1) continue;
    if (verbose&4) print("Set %s port=%d DDR clock tap=%d errs=%08x\n",func,port,i,data[i]);
    errs = data[i]&0xFF;
    if (q0<0 && errs<2) q0=i;
    if (q0>=0 && errs>0 && i-q0<minwin) q0=-1;
    if (q0>=0 && errs>=2 && lerrs>0) break;
    lerrs = errs; k = i;
  }
  j = (q0+k-1)/2; /* center of window */
  if (q0>=0) q0 = k-q0;
  if (verbose&2) {
    print("Set %s port=%d DDR clock tap window:",func,port);
    for (i=0; i<n; i++) {
      errs = data[i]&0xFF;
      if (i%64==0) print("\n   ");
      if (i==j) print("#");
      else if (i<minstr) print("^");
      else if (data[i]==-1) print(".");
      else if (errs<10) print("%d",errs);
      else print("+");
    }
    print("\n");
  }
  k = q0*256/tpp; /* based on 32, 64 or 256 tap make width look like 256 tap */ 
  if (verbose&1) print("Set %s port=%d DDR clock taps=%d width=%d\n",func,port,j,k);
  data[0] = j;
  data[1] = k;
  return j;
}

int_4 ndram_init[3]  = { ND_EMRS_NORMD,ND_EMRS_NORM,ND_MRS_CL4BL4 };

int_4 sdram_init[5]  = { SD_PRECHARGEALL,SD_EMRS_NORM,(SD_MRS_CL25BL4|SD_MRS_RSTDLL),SD_MRS_CL25BL4,SD_PRECHARGERDZ };

int_4 sdram2_init[9] = { SD_PRECHARGEALL,SD2_EMRS2_NORM,SD2_EMRS3_NORM,SD2_EMRS1_NORMDQ, /* the OCDEY (usually OCDEX) helps only on some V6M modules */
			(SD2_MRS_CL3BL4|SD_MRS_RSTDLL),SD_PRECHARGEALL,SD2_EMRS1_OCDEN,SD2_EMRS1_OCDEY, SD_PRECHARGERDY };

int_4 sdram3_init[7] = { SD3_PRECHARGEALL,SD3_EMRS2_CWL7,SD3_EMRS3_NORM,SD3_EMRS1_NORM,
			(SD3_MRS_CL11BL8|SD_MRS_RSTDLL),SD3_ZQCL,SD3_PRECHARGEALL };

int_4 sdram4_init[10] = { SD4_PRECHARGEALL,SD4_EMRS3_NORM,SD4_MR6,SD4_MR5,SD4_MR4,SD4_EMRS2_CWL9,SD4_EMRS1_NORM,
			(SD4_MRS_CL9BL8|SD4_MRS_RSTDLL),SD4_ZQCL,SD4_PRECHARGEALL };

int_4 resetMem (PICSTRUCT *p, int_4 port, int_4 flag) 
{
  int_4 i,j,k,stat=0,len,data[16],tst,n=port-JTAG_PORT_PMOFF;
  int_4 k7 = (port==0 && p->k7==7), k8 = (p->k7==8);

  if (k7 || k8) {
    tst = (port==0)? 0x000 : 0x000;
    tst = findintflagdef((port==0)?"MEMTST":"PMEMTST",p->config,tst);
    k = findintflagdef("MEMSPEED",p->config,250);
    /* memcfg = (ttt[19:16],rrec[15:14],rstr[13:8],wtrn[7:4],rtrn[3:0]} */
    j = k8? 0x8F64 : (k==333)? 0x2177 : 0x4354;
    j = findintflagdef("MEMCFG",p->config,j);
    if (tst>0) j |= (tst&0xFF00)<<8;
    pic_wpbx (p,port,PPC_DMAC_SRCFG,j);
    stat=j;
  }

  if (flag==0 && (port==0 || k8!=0)) {
    clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RMCM,1);
    udelay(10);
    clkadj(p,port,PPC_DMAC_CLOCK,0x0,1);
    udelay(10);
    clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RDLC,1);
    udelay(10000);
    clkadj(p,port,PPC_DMAC_CLOCK,0x0,1);
    udelay(10);
  }

  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RMEM,1);
  udelay(250);
  clkadj(p,port,PPC_DMAC_CLOCK,0x0,1);
  udelay(500);

  if (port==0 && p->type==ICEPIC6) {
    pic_msg (p,port,PKTF_MEM_INIT,0,sdram2_init,sizeof(sdram2_init),0,-1);
  }
  else if (k7) {
    len=sizeof(sdram3_init); 
    for (i=0; i<len/4; i++) data[i]=sdram3_init[i];
    if ((tst&0x10) && p->k7==7) data[2]=SD3_EMRS3_TEST;
    if ((tst&0x10) && k8) data[1]=SD4_EMRS3_TEST;
    pic_msg (p,port,PKTF_MEM_INIT,0,data,len,0,-1); 
  }
  else if (k8) {
    for (i=0;i<160;i++) {
      stat = pic_msg (p,port,PKTF_BUS_RD,(0x1<<20),&tst,0,4,-1);
      if (tst==0);		/* old version - just wait */
      else if ((tst>>24)&0x1==1) break;	/* calibration done */
      udelay(10000);
    }
    vprint("Memory calibration complete on port=%d srcfg=%08x time=%d\n",port,tst,i);
    if (i>=160) printf("Problem with memory calibration on port=%d srcfg=%08x\n",port,tst);
    /* pic_msg (p,port,PKTF_MEM_INIT,0,sdram4_init,sizeof(sdram4_init),0,-1); */
  }
  else if (port==0 && p->type==ICEPIC5) {
    pic_msg (p,port,PKTF_MEM_INIT,0,sdram_init,sizeof(sdram_init),0,-1);
  }
  else if (n>0 && (p->pmtype[n-1]==PMT_DTDM||p->pmtype[n-1]==PMT_DTDMX)) {
    pic_msg (p,port,PKTF_MEM_INIT,0,ndram_init,sizeof(ndram_init),0,-1);
  }
  else if (n>0 && (p->pmtype[n-1]==PMT_ZPPM||p->pmtype[n-1]==PMT_V5M||p->pmtype[n-1]==PMT_V6M)) {
    pic_msg (p,port,PKTF_MEM_INIT,0,sdram2_init,sizeof(sdram2_init),0,-1);
  }
  return stat;
}

#define TLEN 16
#define TLENP 16

int_4 calibMem (PICSTRUCT *p, int_4 port, int_4 *data, int_4 off, int_4 len) 
{
  int_4 i, j, k, m, errs, tap, tapmin, winmin, tst, retry=0, mretry=1, srcfg;
  int_4 testi[TLEN], testo[TLEN], mdata[256], taps[4];
  int_4 k7 = (port==0 && p->k7==7), k8 = (p->k7==8);

  for (i=j=0; j<TLEN; i++,j+=2) { testi[j+0] = testpatt3[i]^0x00000000; testi[j+1] = testpatt3[i]^0xFFFFFFFF; }

  if (len<0) {
    pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A1,testi,TLEN*4,0,-1);
    pic_msg (p,port,PKTF_MEM_RD,MEM_TEST_A1,testo,0,TLEN*4,-1);
    for (errs=0,j=0; j<TLEN; j++) if (testo[j]!=testi[j]) errs++;
    data[0] = errs;
    data[1] = TLEN*4;
    data[2] = MEM_TEST_A1;
    data[3] = testo[0];
    return 0;
  }

  mretry = findintflagdef("MEMRETRY",p->config,mretry);
  tst = findintflagdef("MEMTST",p->config,0);

  RETRY:
  if (k7) srcfg=resetMem(p,port,1);
  retry++;
  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RSCP,1);
  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_ASCPD,off);

  if (tst&0x8) for (j=0; j<TLEN; j++) { testi[j+0] = 0x00010001<<j; }
  if ((tst&0x10)==0) pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A1,testi,TLEN*4,0,-1);

  for (i=0; i<len; i++) {
    if (tst&0x2) for (j=0; j<TLEN; j+=2) { testi[j+0] = testi[j+0]^0xFFFFFFFF; testi[j+1] = testi[j+1]^0xFFFFFFFF; }
    if (tst&0x4) pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A1,testi,TLEN*4,0,-1);
    pic_msg (p,port,PKTF_MEM_RD,MEM_TEST_A1,testo,0,TLEN*4,-1);
    clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_ASCPU,1);
    for (errs=0,j=0; j<TLEN; j++) if (testo[j]!=testi[j]) errs++;
    if (tst>0) { print("I=%d ",i); for (j=0; j<TLENP; j++) print(" %08x",testo[j]); print("\n"); }
    data[i] = errs;
  }
  for (i=len; i<256; i++) data[i]=-1;
  tap = calibrate (p,"MEM",port,data,len,0);
  if (data[1]<40 && retry<mretry) { calibrate (p,"MEM",port,data,len,-1); goto RETRY; }
  if (retry>1) vprint("Retry memory calibration %d times\n",retry);

  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RSCP,1);
  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_ASCPD,off);
  tap = findintflagdef("MEMTAPS",p->config,tap);
  clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_ASCPU,tap);
  if (k7) pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A1,testi,TLEN*4,0,-1);

  return 0;
}

int_4 linktrack (PICSTRUCT *p, int_4 port, int_4 toff, int_4 tmax, int_4 verbose) {
  int i,j,k; int w=0,twin; char c;
  int_4 tpp = (tmax>128)? 256 : tmax>32? 64 : 32;
  if (verbose) printf("Set OIM port=%d AutoTrack window ",port);
  port -= 2;
  iom_init(p,port);
  iom_wr(p,port,OIP_PROG,OIP_TRACK,4);	/* track on */
  iom_dly(p,port,2000);
  iom_wr(p,port,OIP_PROG,0,4);	/* track off to measure new window */
  iom_exec(p,port);
  for (i=0; i<64; i++) {
    j = (i<16||i>=48);
    iom_wr(p,port,OIP_PROG,(j)?OIP_ACLKU:OIP_ACLKD,1);
    k = iom_rd (p,port,OIP_CALIB,4) & 0xFF;
    c = (k<10)? '0'+k : '+';
    if (k==0) w++;
    if (j==0 && (verbose&2)) printf("%c",c);
  } 
  twin = w*256/tpp; 
  if (verbose) printf(" width=%d\n",twin);
  iom_wr(p,port,OIP_PROG,OIP_RCLK,4);	/* restart */
  for (i=0; i<toff; i++) iom_wr(p,port,OIP_PROG,OIP_ACLKU,4);
  iom_wr(p,port,OIP_PROG,OIP_TRACK,4);	/* track on */
  iom_exec(p,port);
  return twin;
}

int_4 linkcal (PICSTRUCT *p, int_4 port, char *func, int_4 clkadr, int_4 ctladr, int_4 rcnt, int_4 toff, int_4 *data, int_4 tmax) {
  int_4 i,n,w,twin,tadj,trst,tup,tdn,cport=port;
  if (tmax==0) { data[0]=0; vprint("Set %s port=%d clock AUTO\n",func,port); return 0; } /* MGT based IF */
  if (port<0) { cport=-port; port=0; }
       if (func[0]=='I') { trst=IOP_RCLK; tdn=IOP_ACLKD; tup=IOP_ACLKU; }		/* IOM */
  else if (func[0]=='O') { trst=OIP_RCLK; tdn=OIP_ACLKD; tup=OIP_ACLKU; }		/* IOM */
  else { trst=PPC_CLOCK_RSCV; tdn=PPC_CLOCK_ASCVD; tup=PPC_CLOCK_ASCVU; };	/* PM */
  if (rcnt>0) { data[0]=clkadr; data[1]=trst; pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,rcnt*4,10000); }
  if (toff<0) { data[0]=clkadr; data[1]=tdn;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,-toff*4,10000); }
  if (toff>0) { data[0]=clkadr; data[1]=tup;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,toff*4,10000); }
  if (tmax>0) { data[0]=clkadr; data[1]=tup;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,tmax*4,10000); }
  tadj = calibrate (p,func,cport,data,tmax,-1); twin = data[1];
  tadj = findintflagdef("IOMTAPS",p->config,tadj+findintflagdef("IOMTAPOFF",p->config,0));
  if (rcnt>0) { data[0]=clkadr; data[1]=trst; pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,rcnt*4,10000); }
  if (toff<0) { data[0]=clkadr; data[1]=tdn;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,-toff*4,10000); }
  if (toff>0) { data[0]=clkadr; data[1]=tup;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,toff*4,10000); }
  if (tadj>0) { data[0]=clkadr; data[1]=tup;  pic_msg (p,port,PKTF_BUS_RWM,ctladr,data,8,tadj*4,10000); }
  if (strcmp(func,"OIM")==0) twin=linktrack(p,port,toff+tadj,tmax,p->verbose);
  if (twin<25) return -1;
  return 0;
}

#define MTEST_SIZE 64

#define RIO3TRY 16
#define RIO3DLY1 10000
#define RIO3DLY 1000

int_4 rio3_buswr (PICSTRUCT *p, int_4 port, int_4 adr, int_4 val) {
  return pic_wpbx (p,port,0x00200000+adr,val);
}

int_4 rio3_busrd (PICSTRUCT *p, int_4 port, int_4 adr) {
  return pic_rpbx (p,port,0x00200000+adr);
}

int_4 rio3_dataisgood (int_4 *d) {
  int_4 d1=d[0],d2=d[1],d3=d[2],d4=d[3];
  if (d2!=d1 || d3!=d1 || d4!=d1) return 0;
  if (d1!=0xEE11FF00 && d1!=0xCC33DD22 && d1!=0xAA55BB44 && d1!=0x88779966 &&
      d1!=0x66997788 && d1!=0x44BB55AA && d1!=0x22DD33CC && d1!=0x00FF11EE) return 0;
  return 1;
}

int_4 rio3_reboot_xb (PICSTRUCT *p, int_4 port, int_4 iport) {
  int i,j,k=0,ii,d[4],i2=(iport&1);
  rio3_buswr (p,0,i2?4:0,0x37);	/* reset the XB RIO */
  rio3_buswr (p,0,i2?4:0,0x36);	/* reset the XB RIO */
  rio3_buswr (p,0,i2?4:0,0x34);	/* reset the XB RIO O */
  rio3_buswr (p,0,i2?4:0,0x24);	/* reset the XB RIO O */
  rio3_buswr (p,port,0x0,0x37);	/* reset the PM RIO */
  rio3_buswr (p,port,0x0,0x36);	/* reset the PM RIO */
  rio3_buswr (p,port,0x0,0x34);	/* reset the PM RIO O */
  rio3_buswr (p,port,0x0,0x24);	/* reset the PM RIO O */
  rio3_buswr (p,port,0x0,0x20);	/* reset the PM RIO I */
  rio3_buswr (p,0,i2?4:0,0x20);	/* reset the XB RIO I */
  for (i=0; i<RIO3TRY; i++) {
    rio3_buswr (p,port,0x0,0x60);	/* reset the XB RIO I */
    rio3_buswr (p,port,0x0,0x20);
    rio3_buswr (p,port,0x0,0x00);
    udelay(RIO3DLY);
    rio3_buswr (p,port,0x0,0x08);	/* reset the XB RIO I */
    udelay(RIO3DLY);
    j=rio3_busrd (p,port,8);
    k=rio3_busrd (p,port,24);
    for (ii=0; ii<4; ii++) d[ii]=rio3_busrd(p,port,32+ii*4);
    if (((j>>24)&0xFF)==0xFC && rio3_dataisgood(d)) break;
  }
  vprint("Reset RIO GEN3 MB2PM #%d port=%d:%d %08x %08x    %08x %08x %08x %08x\n",i,port,iport,j,k,d[0],d[1],d[2],d[3]);
  if (i>=RIO3TRY) return -1;
  return 0;
}

int_4 rio3_reboot_pm (PICSTRUCT *p, int_4 port, int_4 iport) {
  int i=0,j,k=0,ii,jx,d[4],i2=(iport&1);
  XBREBOOTED:
  for (; i<RIO3TRY; i++) {
    rio3_buswr (p,0,i2?4:0,0x60);	/* reset the XB RIO I */
    rio3_buswr (p,0,i2?4:0,0x20);	/* reset the XB RIO I */
    rio3_buswr (p,0,i2?4:0,0x00);
    udelay(RIO3DLY);
    rio3_buswr (p,0,i2?4:0,0x08);	/* reset the XB RIO I */
    udelay(RIO3DLY);
    j=rio3_busrd (p,0,i2?12:8);
    k=rio3_busrd (p,0,i2?28:24);
    for (ii=0; ii<4; ii++) d[ii]=rio3_busrd(p,0,i2*16+32+ii*4);
    if (((j>>24)&0xFF)==0xFC && rio3_dataisgood(d)) break;
  }
  vprint("Reset RIO GEN3 PM2MB #%d port=%d:%d %08x %08x    %08x %08x %08x %08x\n",i,port,iport,j,k,d[0],d[1],d[2],d[3]);
  if (i>=RIO3TRY) return -1;
  if (findintflag("SCRAM",p->config)==0) return 0;	/* scrambler off */
  rio3_buswr (p,0,i2?4:0,0x80);	/* start the XB RIO scrambler */
  rio3_buswr (p,port,0x0,0x80);	/* start the PM RIO scrambler */
  udelay(RIO3DLY);
  /* recheck data status with scrambler ON */
  j=rio3_busrd (p,0,i2?12:8); for (ii=0; ii<4; ii++) d[ii]=rio3_busrd(p,0,i2*16+32+ii*4); if (rio3_dataisgood(d)<0) j|=0x1;
  jx=rio3_busrd (p,port,8);   for (ii=0; ii<4; ii++) d[ii]=rio3_busrd(p,port,32+ii*4);    if (rio3_dataisgood(d)<0) jx|=0x1;
  vprint("Reset RIO GEN3 PMxMB #%d port=%d:%d %08x %08x\n",i,port,iport,j,jx);
  if (j!=0xFF000000 || jx!=0xFF000000) {	/* bad scrambler startup */
    for (ii=0; ii<8 && rio3_reboot_xb(p,port,iport)<0; ii++);
    if (ii<8) goto XBREBOOTED;
  }
  return 0;
}

int_4 test_memory (PICSTRUCT *p, int_4 port, int_4 test, int_4 pass) 
{
  int_4 i,j,k,n,pm,iport,trs,r0,r1,s0,s1,tmax,toff,errs=0,retry=0,mretry=8,nrw=3,data[256],tl=64,mcfg,tsize,tword,sel;
  if (port<0) port = JTAG_PORT_PMOFF+1;
  pm = port;
  if (port>JTAG_PORT_PMOFF) pm-=JTAG_PORT_PMOFF;
  iport=(port>JTAG_PORT_PMOFF)? ((pm-1)&0x1)+CB_PMOFF:port-1;
  nrw = findintflagdef("RETRYWARN",p->config,nrw);
  if (test==MTEST_LRS) {
    for (i=j=0; i<8; i++) { testpatt1[i]=testlrs[j++];  testpatt2[i]=testlrs[j++];  testpatt3[i]=testlrs[j++]; }
    return 0;
  }
  if (p->k7==8 && p->rev==5 && test<=MTEST_FULL) return 0;
  if (test<=MTEST_AUTO) {
    /* check read clock delay */
    if (p->type!=ICEPIC8 || findintflag("MEMCALIB",p->config)>0) test = MTEST_CALIB;
    else if (test_memory(p,port,MTEST_CHECK,0)>0) test = MTEST_CALIB;
  }
  if (port>JTAG_PORT_PMOFF && p->pmtype[pm-1]==PMT_S6M) {
    if (test==MTEST_CHECK || test==MTEST_CHECKV || test==MTEST_CALIB || test==MTEST_FULL)  return 0;
    if (test==MTEST_READ || test==MTEST_WRITE || test==MTEST_BOTH) return 0;
  }
  if (test==MTEST_WRITE || test==MTEST_BOTH) {
    tsize = p->k7? MTEST_SIZE : 32;
    pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A1,testpatt1,tsize,0,-1);
    pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A2,testpatt2,tsize,0,-1);
    pic_msg (p,port,PKTF_MEM_WR,MEM_TEST_A3,testpatt3,tsize,0,-1);
  }
  if (test==MTEST_READ || test==MTEST_BOTH) {
    tsize = p->k7? MTEST_SIZE : 32; tword = tsize/4;
    pic_msg (p,port,PKTF_MEM_RD,MEM_TEST_A1,data,0,tsize,-1);
    for (i=0; i<tword; i++) if (data[i]!=testpatt1[i]) errs++;
    if (p->verbose>0 || errs>0) for (i=0; i<tword; i+=2) {
      s0=testpatt1[i]; r0=data[i]; s1=testpatt1[i+1]; r1=data[i+1];
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A1+(i+0)*4,s0,r0,s0^r0);
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A1+(i+1)*4,s1,r1,s1^r1);
    }
    pic_msg (p,port,PKTF_MEM_RD,MEM_TEST_A2,data,0,tsize,-1);
    for (i=0; i<tword; i++) if (data[i]!=testpatt2[i]) errs++;
    if (p->verbose>0 || errs>0) for (i=0; i<tword; i+=2) {
      s0=testpatt2[i]; r0=data[i]; s1=testpatt2[i+1]; r1=data[i+1];
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A2+(i+0)*4,s0,r0,s0^r0);
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A2+(i+1)*4,s1,r1,s1^r1);
    }
    pic_msg (p,port,PKTF_MEM_RD,MEM_TEST_A3,data,0,tsize,-1);
    for (i=0; i<tword; i++) if (data[i]!=testpatt3[i]) errs++;
    if (p->verbose>0 || errs>0) for (i=0; i<tword; i+=2) {
      s0=testpatt3[i]; r0=data[i]; s1=testpatt3[i+1]; r1=data[i+1];
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A3+(i+0)*4,s0,r0,s0^r0);
      print ("PM%d Mem %x %08x %08x %08x\n",pm,MEM_TEST_A3+(i+1)*4,s1,r1,s1^r1);
    }
  }
  if (test==MTEST_CALIB) {
    resetMem(p,port,0);
    trs=1; i=0; j=160, r0=1; /* default to timing recovery, no tap backoff, normal window, post reset */
    if (port==0) {
      if (p->type==ICEPIC5) { trs = 0; i=10; } /* could not route mclk on PIC5 */
      if (p->type==ICEPIC6 && p->socc[3]!='Q') trs = 0; /* only on HHTQ for now */
      if (p->type==ICEPIC6) { trs = 0; i=10; } 
      if (p->type==ICEPIC7) { trs = 0; j=32; r0=0; }
      if (p->type==ICEPIC8) return 0;
    }
    else if (p->pmtype[pm-1]==PMT_DTDM){ i=10; trs=0; }	/* mclk not stable on DTDM */
    else if (p->pmtype[pm-1]==PMT_V5M) { i=40; } 	/* back off a bit */
    else if (p->pmtype[pm-1]==PMT_V6M) { i=50; j=100; } /* back off a bit */
    else if (p->pmtype[pm-1]==PMT_K8M) return 0;
    else if (p->pmtype[pm-1]==PMT_K8P) return 0;
    pic_msg (p,port,PKTF_MEM_CAL,0x00020000|(j<<8)|i,data,0,1024,10000);
    j = calibrate (p,"MEM",port,data,j,-1);
    errs = data[1]; /* actually the width */
    /* enable automatic timing recovery system for memory read clock */
    j = PPC_CLOCK_TRSP; 
    if (port==0 && p->type==ICEPIC6) j |= 0x00960000;
    else  j |= 0x00C90000; /* default timing window */
    trs = findintflagdef("TRSM",p->config,trs);
    if (trs) pic_msg (p,port,PKTF_BUS_WR,PPC_DMAC_CLOCK,&j,4,0,-1);
    if (r0) resetMem(p,port,0);
  }
  if (test==MTEST_CHECK) {
    if (p->k7==8) { 
      clkadj(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_RMEM,1);
      udelay(250);
      clkadj(p,port,PPC_DMAC_CLOCK,0x0,1);
      udelay(500);
    }
    i = pic_msg (p,port,PKTF_MEM_TEST,0x00100000,data,0,16,10000);
    if (i<0) errs = -1; else errs = data[0];
  }
  if (test==MTEST_CHECKV) {
    i = pic_msg (p,port,PKTF_MEM_TEST,0x00100000,data,0,20,10000);
    if (i<0) errs = -1; else errs = data[0];
    print ("PM%d MemTest pass=%d bytes=%d msec=%d errs=%d\n",pm,pass,data[1],i,data[0]);
    if (errs>0) print ("PM%d MemTest i=%08x k1=%08x k2=%08x\n",pm,data[2],data[3],data[4]);
  }
  if (test==8) {
    tl=128*4;
    pic_acmd (p,ADON_WRIOB,PPC_BADDR_WR,PPC_BADDR_DSOCM,2);	/* reset counter */
    for (i=0; i<tl/4; i++) pic_acmd (p,ADON_WRIOB,PPC_BDATA,i*65536,4);
    pic_acmd (p,ADON_WRIOB,PPC_BADDR_RD,PPC_BADDR_DSOCM,2);	/* reset counter */
    for (i=0; i<tl/4; i++) {
      j = pic_acmd (p,ADON_RDIOB,PPC_BDATA,0,4);
      print("Readback at %x  got=%08x put=%08x\n", i,j,i);
    }
  }
  if (test==MTEST_FULL) {
    i = pic_msg (p,port,PKTF_MEM_TEST,0x01f00000,data,0,16,10000);
    errs = data[0];
    print ("PM%d MemTest pass=%d bytes=%d msec=%d errs=%d\n",pm,pass,data[1],i,data[0]);
    if (errs>0 && pass<=0 && p->verbose==2) {
      print ("PM%d MemTest last err data %x at %x\n",pm,data[3],data[2]);
      tl=64; i = (data[2]&0xFFFFFFF0);
      pic_msg (p,port,PKTF_MEM_RD,i,data,0,tl,8);
      for (j=0; j<(tl>>2); i+=8) {
        s0=i; r0=data[j++]; s1=~i; r1=data[j++];
        print ("PM%d Mem %x s=%08x r=%08x x=%08x\n",pm,i,s0,r0,s0^r0);
        print ("PM%d Mem %x s=%08x r=%08x x=%08x\n",pm,i,s1,r1,s1^r1);
      }
    }
  }
  if (test==MTEST_MB2PM && p->isY) {
    RETRY_MB2PM: retry++;
    pic_wpbx (p,0,PPC_MADDR(iport),0x00010000); /* reset the MB HYPO */
    pic_wpbx (p,0,PPC_MADDR(iport),0x00000000);
    if (p->type==ICEPIC8) for (i=0; i<32 && rio3_reboot_xb(p,port,iport)<0; i++);
    if (p->isY) pic_msg (p,port,PKTF_INIT,0,0,0,0,1000);
    toff=0; tmax=240; trs=1; r1=1;
    if (p->pmtype[pm-1]==PMT_S6M) { trs=0; toff=0; tmax=63; r1=1; }
    if (p->pmtype[pm-1]==PMT_V6M) { trs=0; tmax=127; }
    if (p->pmtype[pm-1]==PMT_DTDM) { trs=0; tmax=192; }
    if (p->pmtype[pm-1]==PMT_K8M) { trs=0; tmax=0; }
    if (p->pmtype[pm-1]==PMT_K8P) { trs=0; tmax=0; }
    toff = findintflagdef("TOFF",p->config,toff);
    j=linkcal (p,port,"PMI",PPC_DMAC_CLOCK,PPC_HYPA_CTL,r1,toff,data,tmax);
    if (j<0 && retry<mretry) goto RETRY_MB2PM;
    if (retry>nrw) print("Retry PMI link calib on port=%d n=%d times. Final width=%d\n",port,retry,data[1]);
    /* enable automatic timing recovery system for memory read clock */
    trs = findintflagdef("TRSV",p->config,trs);
    if (trs) pic_wpbx(p,port,PPC_DMAC_CLOCK,PPC_CLOCK_TRSV);
  }
  if (test==MTEST_MB2PM && p->isX) {
    /* startup test patterns */
    pic_acmd (p,ADON_WRIOC,IOC_PGMCB,1,0);
    data[0] = 0;
    pic_msg (p,port,PKTF_LNK_CAL,0x00000400,data,4,1024,10000);
    j = calibrate (p,"PMI",port,data,256,-1);
    pic_acmd (p,ADON_WRIOC,IOC_PGMCB,0,0);
    pic_msg (p,port,PKTF_INIT,0x0,data,0,0,1000);
  }
  if (test==MTEST_PM2MB) {
    r1=1; toff=0; tmax=64;	/* uses 64 tap IDELAY */
    if (p->type==ICEPIC5) tmax=256; /* uses 256 tap DCM */
    if (p->type==ICEPIC6) { toff=12; tmax=52; }
    if (p->type==ICEPIC7) tmax=32;
    if (p->type==ICEPIC8) tmax=0;
    toff = findintflagdef("TOFF",p->config,toff);
    RETRY_PM2MB: retry++;
    pic_acmd (p, ADON_WR, DMA_ROUTF, PPC_ROUTF_TXFDP, 0);
    data[0]=B1M; data[1]=-1;      pic_msg (p,0,PKTF_MEM_WR,PPC_DMA_XO+iport*8,data,8,0,-1);	/* setup dummy DMA address */
    pic_wpbx(p,0,PPC_DMAC_ROUTE_SET,PPC_ROUTE_M2R(iport));					/* set the ROUTE from port to mem */
    pic_wpbx(p,port,PPC_DMAC_ROUTE_CNT,0x1);							/* set the training mode bit */
    pic_msg (p,port,PKTF_INIT,0,0,0,0,1000);
    if (p->type==ICEPIC8) for (i=0; i<32 && rio3_reboot_pm(p,port,iport)<0; i++);
    pic_wpbx (p,0,PPC_MADDR(iport),0x00020000); 						/* reset just the MB HYPI */
    pic_wpbx (p,0,PPC_MADDR(iport),0x80000001);
    j=linkcal (p,-port,"PMO",PPC_MADDR(iport)|IOC_PROG,PPC_DMAC_CALIB,r1,toff,data,tmax);
    if (j<0 && retry<mretry) goto RETRY_PM2MB;
    pic_msg (p,port,PKTF_INIT,0,0,0,0,1000);
    if (retry>nrw) print("Retry PMO link calib on port=%d n=%d times. Final width=%d\n",port,retry,data[1]);
    pic_wpbx(p,port,PPC_DMAC_ROUTE_CNT,0x0);							/* clr the training mode bit */
    pic_wpbx(p,0,PPC_DMAC_ROUTE_CLR,PPC_ROUTE_M2R(iport));					/* clr the ROUTE from port to mem */
    pic_wpbx(p,0,PPC_MADDR(iport),0x00000001);	
    pic_wpbx(p,0,PPC_DMAC_ROUTE_RST,-1);							/* reset the DMA controller routes */
  }
  if (test==MTEST_MB2IO) {
    r1=1; toff=0; tmax=0;
    i = port-JTAG_PORT_IOOFF;		 /* port index */
    if (p->mtype[i-1]==IOMT_GPS) return 0;
    if (p->mtype[i-1]==IOMT_D2RF) { tmax=63; toff=0; }
    if (p->mtype[i-1]==IOMT_D2AWG && p->mrev[i-1]==3) { tmax=47; toff=0; }
    if (tmax>0) {
      mcfg = IOM_ENA|IOM_LEN|IOM_GGO|IOM_DUAL;
      RETRY_MB2IO: retry++;
      pic_acmd (p,ADON_WRIOC,(i==1)?IOC_A:IOC_B,mcfg,0);				/* start port */
      j=linkcal (p,port,"OIM",OIP_PROG,OIP_CALIB,r1,toff,data,tmax);			/* run calibration loop */
      if (j<0 && retry<mretry) goto RETRY_MB2IO;
      pic_acmd (p,ADON_WRIOC,(i==1)?IOC_A:IOC_B,0,0);					/* stop port */
    }
  }
  if (test==MTEST_IO2MB) {
    r1=1; toff=0; tmax=256;
    if (p->type==ICEPIC7) tmax=31;	/* max tap before wraparound on K7 */
    if (p->type==ICEPIC8) tmax=128;	/* max tap before wraparound on K8 */
    if (p->type==ICEPIC8 && (p->flags&FLG_VHS)==0) toff=72;	
    i = port-JTAG_PORT_IOOFF; 		/* port index */
    mcfg = IOM_LEN|IOM_GGO|IOM_DUAL;
    RETRY_IO2MB: retry++;
    if (p->mtype[0]==IOMT_A2D && p->mrev[0]==18 && i==1) a2dm1x_reset(p);
    if (p->mtype[i-1]==IOMT_LB2D && p->mrev[i-1]==3) lb2dm3_reset(p,i);
    if (p->mtype[i-1]==IOMT_D2AWG && p->mrev[i-1]==3) return 0;
    if (p->mtype[i-1]==IOMT_D2RF) return 0;
    if (p->mtype[i-1]==IOMT_GPS) return 0;
    pic_acmd (p, ADON_WR, DMA_ROUTF, PPC_ROUTF_HXFDP, 0);
    set_iom_sequence (p,i,mcfg,0,0,FLG_TEST);
    pic_acmd (p,ADON_WRIOC,(i==1)?IOC_A:IOC_B,mcfg|IOM_ENA,0);					/* start port */
    data[0]=B1M; data[1]=-1;      pic_msg (p,0,PKTF_MEM_WR,PPC_DMA_XO+iport*8,data,8,0,-1);	/* setup dummy DMA address */
    data[0]=PPC_ROUTE_M2R(iport); pic_msg (p,0,PKTF_BUS_WR,PPC_DMAC_ROUTE_SET,data,4,0,-1);	/* set the ROUTE from port to mem */
    data[0]=mcfg|IOM_ENA;         pic_msg (p,0,PKTF_BUS_WR, PPC_MADDR(iport),data,4,0,-1);	/* set the IOC reg */
    j=linkcal (p,-port,"IOM",PPC_MADDR(iport)|IOC_PROG,PPC_DMAC_CALIB,r1,toff,data,tmax);	/* run calibration loop */
    data[0]=PPC_ROUTE_M2R(iport); pic_msg (p,0,PKTF_BUS_WR,PPC_DMAC_ROUTE_CLR,data,4,0,-1);	/* clr the ROUTE from port to mem */
    pic_acmd (p,ADON_WRIOC,(i==1)?IOC_A:IOC_B,mcfg,0);						/* stop port */
    pic_acmd (p,ADON_WRIOC,(i==1)?IOC_A:IOC_B,0,0);						/* stop port */
    data[0]=-1; pic_msg (p,0,PKTF_BUS_WR,PPC_DMAC_ROUTE_RST,data,4,0,-1);			/* reset the DMA controller routes */
    if (j<0 && retry<mretry) goto RETRY_IO2MB;
  }
  return errs;
}

int_4 pic_spec (PICSTRUCT *p, 
  int_4 *data, int_4 nstart, int_4 ndata, int_4 nblock, int_4 *Ntotal,
  float *Maxsrate, int_4 *Maxsjump, float *Maxbrate, int_4 *Maxbjump, int_4 flags) 
{
  int_4 i,j,dir,pbr,pcie,mcsr0,mcsr,waddr,uaddr,count,curjump,maxrate,maxsjump,maxbjump,maxdata,nbjump,status=-1;
  int_8 laddr;

  *Maxsjump = *Maxbjump = *Ntotal = 0;
  *Maxsrate = *Maxbrate = 0; 

  RD(REG_MCSR, i);
  mcsr0 = i&0xF;
  if (nblock<0) {
    dir = 1;
    nblock = -nblock;
    mcsr = mcsr0|MCSR_MASTER_RDM|MCSR_TEST;
  } else {
    dir = -1;
    mcsr = mcsr0|MCSR_MASTER_WR|MCSR_TEST;
  }
  if (p->type<=4) mcsr |= MCSR_RFIFO;
  j = findintflagdef("SPECDIR",p->config,0);
  if (j!=0 && j!=dir) return 0;	/* special mode */
  pic_lock (p, LOCK_ALLOC);
  WR(REG_MCSR, mcsr0);
  pbr = p->pbr;
  if (p->k7) pbr=500; /* internal write rate of PCIE cores */
  if (p->k7==8) pbr=1000;
  if (p->k7==8 && p->rev==1) pbr=500;
  if (p->k7==8 && p->rev==2) pbr=500;
  if (p->k7==8 && p->rev==4) pbr=500;
  if (p->k7==8 && p->rev==5) pbr=2000;
  if (p->type==ICEPIC6) pbr=250;	
  pcie = (p->type>=6 && p->type<=8);
  i = findintflag("PPLM",p->config);
  if (pcie && i>=0 && i<=3) mcsr |= (i<<12);
  laddr = phys2long(nstart);
  waddr = (laddr&0xFFFFFFFF);
  WR(REG_MDAR, waddr);
  uaddr = (laddr>>32);
  WR(REG_MDARH, uaddr);
  ndata -= 64; /* minus four packets to not overrun 2**N packet transfer length */
  if (uaddr!=0) {
    mcsr |= MCSR_MASTER_DAC;
  }
  if (p->pbw==8  || pcie) {
    nbjump = nblock*2;
  } else {
    nbjump = nblock*4;
  }
  if (p->pbw==8 && !pcie) {
    WR(REG_MDTC, ndata*2);
    mcsr |= MCSR_WIDE;
  } else {
    WR(REG_MDTC, ndata*4);
  }
  WR(REG_MCSR, mcsr);
  pic_lock (p, LOCK_FREE);
  i=0;
  j = findintflag("SWAMP",p->config);
  do {
    if (j>0) {
      udelay(100);
      if (++i>10000) goto BAIL;
    } else {
      udelay(10000);
      if (++i>100) goto BAIL;
    }
    RD(REG_MCSR,mcsr);
  } while (mcsr&MCSR_MENA);


  if (dir<0) {
    ConvertEndian (data,ndata,32);
    maxsjump = 0;
    j = pcie? 128 : 2;
    count = ndata-j;
    for (i=0; i<count; i+=j) {
      curjump = data[i] - data[i+j];
      if (curjump > maxsjump) maxsjump = curjump;
      if (curjump < 0 || data[i] <= 0) break;
      ndata = i;
    }
    maxbjump = 0;
    if (ndata<nblock) nblock=(ndata&0xFFFFF000);
    if (nblock==0) nblock=256;
    for (i=0; i<ndata-nblock; i+=nblock/4) {
      curjump = data[i] - data[i+nblock];
      if (curjump > maxbjump) maxbjump = curjump;
    }
    maxdata = data[0] - data[ndata];
    if (maxdata==0) maxdata = 1000000000;
    if (maxbjump==0) maxbjump = 1000000000;
    maxrate = pbr*4;
    *Maxsrate = (float)maxrate*ndata/(maxdata/4);
    *Maxsjump = maxsjump/4;
    *Maxbrate = (float)maxrate*nblock/(maxbjump/4);
    *Maxbjump = (maxbjump-nbjump)/4;
    *Ntotal = ndata;
  }
  else if (pcie) {
    pic_lock (p, LOCK_ALLOC);
    WR(REG_IOC,REG_IOC_ADDR);
    RD(REG_IOC,i); 
    pic_lock (p, LOCK_FREE);
    i = 0x01000000 - (i>>8)&0x00FFFFFF; /* 250Mhz clocks to complete - PIC7 counts by 2 with pbr=500 */
    *Maxsrate = *Maxbrate = (float)pbr/i * (ndata*4);
    *Maxsjump = *Maxbjump = 0;
    *Ntotal   = ndata;
  } else {
    RD(REG_MDAR,i); i = i-waddr;
    *Maxsrate = *Maxbrate = (float)pbr*i/(0x400000/p->pbw);
    *Maxsjump = *Maxbjump = 0;
    *Ntotal   = i/4;
  }
  status=1;
  BAIL:
  WR(REG_MCSR, mcsr0);
  if (flags&0x1) {
    if (dir>0) print ("Output SRate = %f Mby/s  Ndata  = %d\n", *Maxsrate, *Ntotal);
    if (dir<0) print ("Input  SRate = %f Mby/s  Ndata  = %d  MaxJump = %d\n", *Maxsrate, *Ntotal, *Maxsjump);
    if (dir<0) print ("Input  BRate = %f Mby/s  BlockSize = %d  MaxJump = %d\n", *Maxbrate, nblock, *Maxbjump);
  }
  if ((mcsr0&0xF)==0) pic_log(p,0);
  return status;
}

int_4 pic_cbufsz (PICSTRUCT *p, int_4 dmac) 
{
  int_4 size=DMA_CSIZE_DEF*4, gchips=0, ptype=p->ptype;
  if (dmac>0) return p->dma[dmac-1].csize*4;
       if (ptype == IOPT_STREAM) size = DMA_CSIZE_SER*4;
  else if (ptype == IOPT_INTERNAL) size = DMA_CSIZE_INT*4;
  else if (ptype == IOPT_EXTERNAL) size = DMA_CSIZE_EXT*4;
  else if (ptype == IOPT_TUNER) {
    if (p->type==ICEPIC2) size = DMA_CSIZE_SER*4;
    else if (p->type==ICEPIC3) size = DMA_CSIZE_SER*4;
    else { 
      size = DMA_CSIZE_GC*4;
      gchips = p->mchan/4;
      if (gchips==1) size*=4;
      if (gchips==2) size*=2;
    }
  }
  else if (ptype == IOPT_MODULE) {
    if (p->flags&FLG_VHS && !p->isY) size = DMA_CSIZE_VHS*6;
    else size = DMA_CSIZE_MOD*4;
  }
  return size;
}

int_4 pic_dma (PICSTRUCT *p, int_4 dmac, int_4 dir, 
	int_4 buffer[], int_u4 nstart, int_4 nbytes, int_4 nblock, int_4 flags)
{
  DMAMAP map;
  map.vaddr = (char *)buffer;
  map.paddr = nstart;
  map.bytes = nbytes;
  map.offset = 0;
  return pic_dmasetup (p,dmac,dir,&map,nblock,flags);
}

int_4 pic_dmasetup (PICSTRUCT *p, int_4 dmac, int_4 dir, 
	DMAMAP *map, int_4 nblock, int_4 flags)
{
  int_4 words, block, n,m,i,j, stkloc, iper, mux, xinc, slot, status, bpa; 
  int_4 caddr, csize, cbase, nmux, imuxoff, tcbcp[8], ptype, pindex, chip, side, cwsize; 
  int_4 hwsize, mblock, gchip, gchips=0, hsize, ndiv, chained, width, cper, nodma;
  DMASTRUCT *dma;
  APPSTRUCT *app = p->app;
  TCB2DSTRUCT *tcb = p->tcb, *tcbn=0, tcbtmp;
  CHECK_MAGIC(p);

  int_8 count = map->bytes;
  int_8 paddr = map->paddr+map->offset;
  char* vaddr = map->vaddr+map->offset;
  real_8 spw;

  stkloc = dmastkloc(p,dmac);
  if (stkloc<0) return (-1);

  n = dmac-1; 
  dma = p->dma+n;
  flags |= p->flags;
  ptype = dma->type;
  pindex = dma->port;
  if (flags&FLG_TUNER) ptype = IOPT_TUNER;
  if (ptype!=IOPT_MODULE || p->isY) flags &= (~FLG_VHS);
  hwsize = (p->isY)? 64 : 4;
  cwsize = (p->isY)? 4 : (flags&FLG_VHS)? 6 : 4;
  gchip = p->gchip;
  if (gchip!=0) gchips = p->mchan/4;
  chained = (!p->isX && !p->isY && (n==6 || n==7));
  if (pindex&1) side=1; else side=2;
  bpa = (dma->bits<0)? -dma->bits/4 : dma->bits/8;
  if (bpa<=0) bpa=1;
  nodma = findintflagdef("NODMA",p->config,0);

  /* configuration for multiplexed DMA channels */
  if (flags&FLG_MUX) mux=p->xmux;
  else if (flags&FLG_VHS) mux=1;
  else if (ptype==IOPT_MODULE && pindex==3) mux=1;
  else if (ptype==IOPT_MODULE && (dma->mcfg&IOM_DUAL)!=0 && !p->isY) mux=1;
  else mux = 0;
  if (mux==1) {
    p->xinc = 2;
    if (p->isY);
    else if ((dma->mcfg&(IOM_VHS |IOM_MUX))==(IOM_VHS |IOM_MUX)) p->xinc=4;
    else if ((dma->mcfg&(IOM_DUAL|IOM_VHS))==(IOM_DUAL|IOM_VHS)) p->xinc=4;
    else if ((dma->mcfg&(IOM_DUAL|IOM_MUX))==(IOM_DUAL|IOM_MUX)) p->xinc=4;
  }

  /* enforce word boundaries */
  hsize = (int_4)(count/hwsize);
  if (count%hwsize != 0) {
#ifdef _LONG64
    print("Count %ld must be %d byte aligned\n",count,hwsize);
#else
    print("Count %lld must be %d byte aligned\n",count,hwsize);
#endif
    return (0);
  }
  words  = (int_4)(count/cwsize);
  if (count%cwsize != 0) {
#ifdef _LONG64
    print("Count %ld must be %d byte aligned\n",count,cwsize);
#else
    print("Count %lld must be %d byte aligned\n",count,cwsize);
#endif
    return (0);
  }

  /* determine memory buffer location */
  nmux = 1; 
  imuxoff = 0;
  csize = findintflag("CSIZE",p->config);
  cbase = findintflagdef("CBASE",p->config,DMA_CADDR_ISY);
  if (flags&FLG_TEST) {	/* test mode uses the whole 2nd block */
    csize  = TEST_SIZE;
    caddr  = TEST_ADDR;
    dma->reg = -1;	/* no internal source */
  }
  else if (p->isY) {
    if (csize>0) csize*=256;
    else {
      /* tuner bank can burst 4M chunks faster than 1-lane PCI */
      csize = (n<16)? DMA_CSIZE_ISY : DMA_CSIZE_ISYB;
      if (p->k7) csize = DMA_CSIZE_ISY;
      if (findintflag("SKIPONCARD",p->config)<=0)
        while (csize>0x800 && csize*cwsize>count) csize=csize/2;
      if (dma->rate<1000000 && csize>0x4000) csize=0x4000;
    }
  if (p->k7) {
    if (n<12) caddr = DMA_CADDR_ISY + DMA_CSIZE_ISY*n;
    else      caddr = DMA_CADDR_ISYC + DMA_CSIZE_ISY*(n-12);
  } else {
    if (n<12) caddr = DMA_CADDR_ISY + DMA_CSIZE_ISY*n;
    else      caddr = DMA_CADDR_ISYB + DMA_CSIZE_ISYB*(n-12);
  }
    if (csize>DMA_CSIZE_ISY) caddr = cbase + csize*n;
    if (csize>DMA_CSIZE_ISYL && n>=8) caddr = cbase + csize*(n-4);
    if (caddr%csize!=0) caddr &= ~(csize-1);
    i = findintflag("COFFS",p->config); if (i>=0) caddr = i*256;
    if (findintflag("DIO",p->config)>0) dir = dir*2;
    if (p->nio!=0 && dma->type!=IOPT_STREAM) { dma->flags |= FLG_NIO; if (dir>0) dir=3; }
    if (nodma==4 && dir>0) dir=4;
  }
  else if (csize>0) {                /* user specified allocation (in kBytes) */
    csize = min(csize*256,DMA_CSIZE_ALL);
    caddr = findintflag("COFFS",p->config);
    if (caddr>0) caddr=caddr*256; else caddr=0;
    if (flags&FLG_VHS) caddr = DMA_CADDR_VHSX + min(caddr,DMA_CSIZE_VHS-csize);
    else caddr = DMA_CADDR_ALL + min(caddr,DMA_CSIZE_ALL-csize);
  }
  else if (findflag("MEM=ALL",p->config) >= 0) {
    csize = DMA_CSIZE_ALL;
    caddr = DMA_CADDR_ALL;
  }
  else if (findflag("MEM=EXT",p->config) >= 0) {
    if (flags&FLG_VHS) {
      csize = DMA_CSIZE_VHS;
      caddr = DMA_CADDR_VHSX;
    } else {
      csize = DMA_CSIZE_EXTA;
      caddr = DMA_CADDR_EXT;
    }
    if (p->isX) chained=1;
  }
  else if (p->type==ICEPIC2 && ptype==IOPT_TUNER) {
    csize = DMA_CSIZE_SER;
    caddr = DMA_CADDR_SER + (pindex-1)*csize;
  } 
  else if (p->type==ICEPIC3 && ptype==IOPT_TUNER) {
    csize = DMA_CSIZE_SER;
    caddr = DMA_CADDR_SER + (pindex-1)*csize;
  } 
  else if ((p->gchip==GC4016 && p->type!=ICEPIC4) && ptype==IOPT_TUNER) {
    if (dma->master>0) { 	/* muxed chip acq */
      chip = p->chiptbl[pindex];
      csize = DMA_CSIZE_GCT/16; /* JEFF each chan too small for max BW output */
      caddr = DMA_CADDR_GCT + ((pindex-1)>>1)*csize;
      csize *= (4/p->cpc);
      caddr += (side-1)*DMA_CSIZE_GCT;
    } else {			/* demux single chan */
      chip = pindex;	
      csize = DMA_CSIZE_TGC;
      caddr = DMA_CADDR_TGC + (side-1)*csize;
    }
  } 
  else if (p->type==ICEPIC4 && ptype==IOPT_TUNER && (p->tflags&FLG_ITDEC)!=0) {
    if (dma->master>0) { 	/* muxed chip acq */
      chip = p->chiptbl[pindex];
      csize = DMA_CSIZE_GCT/4;
      caddr = DMA_CADDR_GCT + ((pindex-1)>>1)*csize;
      csize *= (4/p->cpc);
      caddr += (side-1)*DMA_CSIZE_GCT;
    } else {			/* demux single chan */
      chip = pindex;	
      csize = DMA_CSIZE_TGC;
      caddr = DMA_CADDR_TGC + (side-1)*csize;
    }
  } 
  else if (gchip && ptype==IOPT_TUNER) {	/* Graychip Tuner */
    if (dma->master>0) {
      nmux = p->cpc;
      chip = p->chiptbl[pindex];
      if (gchips==1) imuxoff = (pindex-1)&0x3; 
      else imuxoff = ((pindex-1)>>1)&0x3; 
      if (nmux==2 && imuxoff>1) imuxoff=1;	/* 2nd half alternate accounting */
    }
    else chip = pindex;				/* muxed chip acq */
    csize = DMA_CSIZE_GC;
    if (gchips==1) csize *= 4;
    if (gchips==2) csize *= 2; 
    caddr = DMA_CADDR_GC + (chip-1)*csize;
    if (chip==5) caddr = DMA_CADDR_MOD;
    if (chip==6) caddr = DMA_CADDR_MOD+DMA_CSIZE_MOD;
  } 
  else if (ptype==IOPT_INTERNAL) {		/* Internal Buffer */
    csize = DMA_CSIZE_INT;
    caddr = DMA_CADDR_INT + (pindex-1)*csize;
  } 
  else if (ptype==IOPT_EXTERNAL) {		/* External Buffer */
    csize = DMA_CSIZE_EXT;
    caddr = DMA_CADDR_EXT + (pindex-1)*csize;
  } 
  else if (pindex==3 && ptype==IOPT_TBANK) {	/* TunerBank Buffer */
    csize = DMA_CSIZE_GCT;
    caddr = DMA_CADDR_GCT;
  } 
  else if (ptype==IOPT_TBANK) {			/* TunerBank Buffer */
    csize = DMA_CSIZE_GCT;
    caddr = DMA_CADDR_GCT + (side-1)*csize;
  } 
  else if (ptype==IOPT_CORE) {			/* Core Buffer */
    csize = DMA_CSIZE_COR;
    caddr = DMA_CADDR_COR + (side-1)*csize;
  } 
  else if (flags&FLG_VHS) { 			/* Very High Speed mode */
    if (p->isX) caddr  = DMA_CADDR_VHSX;
    else        caddr  = DMA_CADDR_VHS;
    if (p->xinc==4) {			/* use entire block as 48 bit */
      csize  = DMA_CSIZE_VHS;
    } else {				/* use half block as 48 bit */
      csize  = DMA_CSIZE_VHS/2;
      caddr += (pindex-1)*csize;
    }
  }
  else if (pindex==3 && ptype==IOPT_MODULE) {	/* AB mux */
    csize = DMA_CSIZE_MOD*2;
    caddr = DMA_CADDR_MOD;
  } 
  else if ((pindex&0x1) != 0) {			/* A side */
    csize = DMA_CSIZE_MOD;
    caddr = DMA_CADDR_MOD;
  } 
  else {					/* B side */
    csize = DMA_CSIZE_MOD;
    caddr = DMA_CADDR_MOD + DMA_CSIZE_MOD;
  }

  if (csize>words && !p->isY) csize=words; 	/* small buffer case */
  ndiv = 4;
  width = (p->pbw==8)? 2:1;
  if (paddr==0 || nmux>1) width=1;	/* non-master fifo or scatter/gather */

  if (p->isX && width==2 && p->pbr>=66 && 
     (findflag("RXRAWDATA",p->config)>=0 || findflag("RXRAWSDDS",p->config)>=0)) {
    vprint("Setting PCI write bus width=32b to workaround SHARC burst problem\n");
    width = 1;
  }

  /* find evenly divisible csize */
  if (p->isY && (flags&FLG_TEST)==0) {
/*    ndiv = csize/((p->type==ICEPIC6||p->type==ICEPIC7||p->type==ICEPIC8)?4096:1024); */
  }
  else if (words%(csize/nmux) != 0 && dma->master<=0) {		
    for (i=csize; i>=csize/2; i-=nmux) {
      if (words%(i/nmux)!=0) continue; 
      ndiv=4; if (i%(ndiv*width)==0) break;
      ndiv=5; if (i%(ndiv*width)==0) break;
      ndiv=3; if (i%(ndiv*width)==0) break;
    }
    vprint("Dropped CSIZE=%d to %d/%d for HSIZE=%d\n",csize,i,ndiv,words); 
    if (i>=csize/2) csize=i;
    else { print("Could not find CSIZE for HSIZE=%d\n",words); return -1; }
  }

  /* re-calculate a decent DMA blocking ratio (8by boundaries) */
  block = mblock = csize/ndiv;
  if (nblock<0) {
    spw = dma->dec*(4.0/dma->rate)/bpa;	/* seconds per word to calc PCIe response time to 1ms */
    while (block>=0x1000 && (block&0x3)==0 && (block*spw>0.001)) block>>=1;
  }
  else if (nblock>0 && (nblock>>2)<=mblock) {
    for (block=(nblock>>2); csize%block!=0 && block<mblock; block++);
  }
  i = findintflag("BLOCK",p->config);
  if (i>0) block = (i>>2);	/* set BLOCK */

  if (ptype==IOPT_MODULE && dma->master>0) {
    m = dma->master-1;
    csize = p->dma[m].csize;
    caddr = p->dma[m].caddr;
    block = p->dma[m].block;
  }

  /* test for valid buffer ratios */
  cper = (csize*cwsize/nmux);
  iper = (int_4)(count/cper);
  if (p->isY);
  else if (iper > 0x10000) {
    print("Host buffer must be less than (64K*%d) bytes\n",cper);
    return (0);
  }
  else if (iper*cper != hsize*hwsize) {
    print("Host buffer must be a multiple of %d bytes\n",csize*cwsize);
    return (0);
  }
  dma->iper = iper;
  iper = csize / block;
  if (p->isY);
  else if (iper*block != csize) {
    print("SHARC/PPC buffer must be a multiple of %d bytes\n",block*cwsize);
    return (0);
  }

  if (mux>0) {
    xinc=p->xinc;
    p->xmux=mux+1;
    /* offset the address for the slaved port */
    if (xinc==4 && mux==2) imuxoff=2;
    else if (xinc==4 && mux==3) imuxoff=1;
    else imuxoff = (mux-1);
  }
  else xinc=1;

  /* setup slot and TCB for DMA */
  if (ptype==IOPT_TUNER && dma->master>0) slot = dma->master;
  else if (dmac>p->ndmac) slot = 0;	/* 5-8 are PM0 core feeds to PMs */
  else slot = dmac;

  if (p->isY && slot>0) {
    if (dir>0) slot+=8;
    if (p->npm==0 && dmac==6) slot++;
    if (p->npm>0 && dir>0 && dmac>=5 && dmac<=8) slot = side+10;
  }

  /* set up the SHARC/PPC/host DMA handler structure */
  dma->caddr  = dma->cindex = caddr + imuxoff;
  dma->csize  = csize;
  dma->haddr  = dma->hindex = L2HW(paddr);
  dma->hsize  = hsize;
  dma->block  = block;
  dma->stat   = 0; 
  dma->dir    = dir; 
  dma->todo   = 0; 
  dma->hindp  = stkloc + DMA_HINDEX;
  dma->hcycle = dma->ccycle = dma->miss = dma->err = 0;
  dma->multi 	= nmux; 
  i = dmac - p->ndmac - 1; 
  if (p->isY && dir<0 && dmac>8 && dmac<=12) i = (dmac>10)? dmac-11+32 : dmac-9; /* TBank/MCore uses 1st tuner slots */
       if (p->isY) dma->tcbp = (i<0)? (PPC_DMA_XI>>2)+tcbptaby[slot] : (dir>0)? (PPC_DMA_TI>>2)+i*2 : (PPC_DMA_TO>>2)+i;
  else if (p->isX) dma->tcbp = tcbptabx[slot];
  else             dma->tcbp = tcbptab[slot];
  if (p->isY && dir<0 && dmac>8 && dmac<=12 && ptype==IOPT_TUNER) dma->tcbp = 0; /* tuner channel handle - no actual DMA */
  if (p->isY && dir<0 && p->dsgtype>0 && ptype==IOPT_STREAM) dma->tcbp = (PPC_DMA_NIO>>2)+i*2; /* debug NIO stream DMA */
  dma->cindp  = dma->tcbp;
  dma->cind   = dma->caddr; 
  p->dma_vaddr[n] = (int_4 *)vaddr;
  p->dma_vindx[n] = 0;
  p->dma_vcycx[n] = 0;

  if (p->type==ICENIC) return(1);
  if (p->type==ICENVM) return(1);

  /* set up the slaved DMA channels */
  if (mux>0 && mux<xinc) pic_dmasetup (p,dma->slave,dir,map,nblock,FLG_MUX);

  /* cindp to the last slave if low-rate data? NOW always due to bursty data */
  for (i=n; p->dma[i].slave>0; ) {
    i = p->dma[i].slave-1;
    if (p->isY) p->dma[i].cindp = dma->tcbp;
    else dma->cindp = p->dma[i].tcbp;
  }

  /* are we really controlling this TCB */
  if (n<p->ndmac || p->isY) tcbn = tcb+n; 

  /* enable fast look-up for registers from PCI chip */
  if (p->type==ICEPIC2||p->type==ICEMBT2||p->type==ICESLIC3);
  else if (p->driver<338);  /* only 338+ drivers have access to the upper PCI window */
  else if (p->isY) dma->hindp = REG_AHIRY+n; 
  else if (n<32) dma->hindp = REG_AHIRX+n;
  dma->hind   = dma->haddr; 

  /* transfer direction >0 means from PCI to CARD and out */
  if (p->isX) dma->cntrl = 0x00000501; 
  else if (flags&FLG_VHS) dma->cntrl = 0x000002E1; /* 48 bit with 48/32 packing */
  else dma->cntrl = 0x00000201;

  /* setup the PCI MCSR register */
  dma->mcsr = (dir>0 && dir<3)? MCSR_MASTER_RDM : MCSR_MASTER_WR;
  if (p->pbr>133) {  /* PCIe */
    dma->mcsr &= 0xFFFFCFFF;
    i = findintflag("PPLM",p->config);
    if (i>=0 && i<=3) dma->mcsr |= (i<<12);
  }
#if _IEEE
  if (p->rev<=2) {  /* early version */
    if (abs(dma->bits)>8) dma->mcsr |= 0x200;
    if (abs(dma->bits)>16) dma->mcsr |= 0x400;
  } else {
    if (abs(dma->bits)>8) dma->mcsr |= MCSR_BIGE0;
    if (abs(dma->bits)>16) dma->mcsr |= MCSR_BIGE1;
  }
#endif
  if (p->isY) {
    if (width==2) dma->mcsr |= MCSR_WIDE;
    else          dma->mcsr |= MCSR_BIGE2;
    if (paddr>>32 != 0) dma->mcsr |= MCSR_MASTER_DAC;
  }
  else if (p->isX) {
    if (dir>0 && p->rev==9) dma->mcsr = MCSR_MASTER_RDMX;
    if (width==2) {
      dma->cntrl = 0x00000401; 
      dma->mcsr |= MCSR_WIDE;
    } else {
      dma->mcsr |= MCSR_BIGE2;
    }
    if (flags&FLG_VHS) dma->mcsr |= MCSR_C6TO4;
    if (dma->master<0 && dma->slave>0) dma->mcsr |= MCSR_SLAVE_SYNC;
  }
  if (dma->haddr==0)  dma->mcsr &= (~MCSR_MENA);
  if (dir<=0) dma->cntrl |= 0x4; /* enable dma output */

  if (gchip==GC4014 && ptype==IOPT_TUNER) {
    if (block==csize/4) dma->block *= 2;  /* extend transfer size */
    if (dma->bits<=0);	/* complex data */
    else dma->cindp = APP_STK + (dma->master-1)*APP_SEP + APP_INDI;
  }

  if (gchip==GC4016 && ptype==IOPT_TUNER && (p->tflags&FLG_ITDEC)!=0) {
    if (p->isY);
    else dma->cindp = stkloc + DMA_CIND; 
  }

  /* if host size = 0, make Host buffer match the Card buffer */
  if (dma->hsize == 0) dma->hsize = dma->csize * cwsize / hwsize;

  /* set up the CARD/peripheral DMA handler structure */
  if (tcbn==0);
  else if (p->isY) {
    for (m=10; PIPE_SIZE(m) < (dma->csize<<2) && m<32; m++);
    if (m<32 && PIPE_SIZE(m)==(dma->csize<<2)) { /* power of two buffer */
      tcbn->xs = (dma->caddr<<2) | PIPE_CFG(m);
      tcbn->xi = -1; /* forever */
    }
    else {
      print("Cannot support non-power2 CSIZE=%d \n",dma->csize);
      return 0;
    }
  }
  else {
    tcbn->xs = dma->caddr;	/* X start */
    tcbn->xi = xinc;		/* X increment */
    tcbn->xc = dma->csize/xinc;	/* X count */
    tcbn->cp = 0;		/* Chain pointer */
    tcbn->yc = 0xFFFF;		/* Y count -> forever */
    tcbn->yi = -dma->csize;	/* Y increment -> back to top */
  }

  /* set up the TCB chain pointers if necessary */
  if (chained) {
    if (n>=14) return 0;
    tcbn->cp = (DMA_TCBCP+(8*n)+7)&0x1FFFF; 
    tcbcp[0] = 0;
    tcbcp[1] = 0;
    tcbcp[2] = tcbn->yi;
    tcbcp[3] = tcbn->yc;
    tcbcp[4] = tcbn->cp;
    tcbcp[5] = tcbn->xc;
    tcbcp[6] = tcbn->xi;
    tcbcp[7] = tcbn->xs;
  }

  /* test mode initializes the II register to one full block plus slop */
  if (flags&FLG_TEST) {
    tcb[n].xs = dma->cindex + dma->csize + 32;
    dma->cindp = stkloc + DMA_CIND; 
    dma->cind  = (dma->cindex + dma->csize + 32)<<2;
  }

  /* set up initial xfer block */
  dma->cxfer = dma->block;
  dma->cxfes = dma->block/nmux;
  dma->hxfer = dma->hxfes = dma->cxfes*cwsize/hwsize;

  /* add 3 extra words to keep PCI write fifo pipeline full */
  if (dir<0 && dma->haddr!=0 && !p->isY) {
    if (p->isX) i=8; else i=3;
    if (dma->mcsr&MCSR_WIDE) i*=2;
    dma->hxfes += i;
    dma->cxfes += i*hwsize/cwsize;
  }

  /* for VHS on 21160, 6byte words are transfered out using 4byte accounting */
  if (p->isX && (flags&FLG_VHS)!=0) dma->cxfes = dma->hxfes;
#ifdef _LONG64
  vprint ("DMA bytes=%08lx addr=%08lx chan=%d csize=%d block=%d\n", count, paddr, dmac, csize, block);
#else
  vprint ("DMA bytes=%08llx addr=%08llx chan=%d csize=%d block=%d\n", count, paddr, dmac, csize, block);
#endif
  if (flags&FLG_INFO) return (1);

  /* make sure this channel is not in the queue */
  pic_dmaqueue(p, dmac, -1);
    
  /* download the TCB chain pointer parameters if needed */ 
  if (chained) pic_transfer (p, tcbcp, 8*4, DMA_TCBCP+(8*n), 1, 0);

  /* download the CARD side DMA parameters */ 
  if (tcbn!=0 && dma->tcbp!=0 && (dma->reg!=0 || p->isY)) {
    i = (dma->tcbp>=(PPC_DMA_XI>>2) && dma->tcbp<(PPC_DMA_TX>>2))? 8:4;
    if (p->isY && dir>0 && dma->tcbp>=(PPC_DMA_TI>>2)) i=8;
    if (p->isY && dma->tcbp>=(PPC_DMA_NIO>>2)) i=8;
    i = p->isY? ((dma->dir==-2)?8:i) : sizeof(tcbtmp);
    pic_transfer (p, (int_4 *)tcbn, i, dma->tcbp, 1, 0);
  }

  /* add the TCBP for network input */
  if (p->isY && dir==3 && p->dsgtype>0 && ptype!=IOPT_STREAM) {
    dma->cind = (PPC_DMA_NIO>>2)+(dmac-1)*2; /* debug NIO stream DMA */
    if (p->dsgtype<DSG_TRAY) pic_transfer (p, (int_4 *)tcbn, 8, dma->cind, 1, 0);
  }

  /* download the CARD/host DMA handler structure */
  pic_transfer (p, (int_4 *)(&p->dma[n]), sizeof(p->dma[n]), stkloc, 1, 0); 

  /* download the CARD side APP parameters */ 
  if (m<APP_CHNS && dma->alg != 0) 
  pic_transfer (p, (int_4 *)(&app[m]), sizeof(app[m]), dma->appp, 1, 0);

  /* if already running or this is a reconnect then leave 
     early to avoid pre-cue which can mess up NSTAT logic */
  if (gchip==GC4016 && ptype==IOPT_TUNER) {
    i = pic_acmd (p, ADON_RD, DMA_GMCFG+side, 0, 0);
    if (i&IOM_SEN) { vprint("Avoiding pre-cue on dmac=%d stat=%08x\n",dmac,i); return (1); }
  }

  /* place this DMA channel in the processing queue to allocate channel parameters */
  pic_lock (p, LOCK_ALLOC);
  status = pic_dmaqueue(p, dmac, 1);
  if (status>=0 && ((ptype==IOPT_TUNER)||(ptype==IOPT_STREAM)) && dma->master>0) status = pic_dmaqueue(p, dma->master, 1);
  if (status<0) print("Err: Out of DMA queue slots\n");
  else if (status==0) status=1; /* already queued is OK status */
  pic_lock (p, LOCK_FREE);

  return (status);
}

int_4 pic_dmaqueue (PICSTRUCT *p, int_4 chan, int_4 mode)
{
  int_4 i,j,status,stkloc,appchan,n=chan-1;
  DMASTRUCT *dma = p->dma+n;

  stkloc = dmastkloc(p,chan);

  if (mode==0) {
    for (i=0; i<DMA_CHNS; i++) {
      j = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (j==0) break;
      if (j==stkloc) return 1;
      if (chan==-1 && (((j-DMA_STK)/DMA_SEP)%2)==0) return 1;	/* odd channel 1-based */
      if (chan==-2 && (((j-DMA_STK)/DMA_SEP)%2)==1) return 1;	/* even channel 1-based */
    }
    return 0;
  }
  if (stkloc<0) return -1;

  if (mode > 0) {
    if ( p->isY && dma->hindp>=REG_AHIRY) pic_acmd (p, ADON_WR, dma->hindp, 0, 0);
    if (!p->isY && dma->hindp>=REG_AHIRX) WR((dma->hindp<<2)&0xFFF,0);
  }

  pic_lock (p, LOCK_ALLOC);

  if (dma->appp!=0) {
    appchan = (dma->appp-APP_STK)/APP_SEP+1;
    pic_appqueue (p,appchan,mode); 
  }

  if (mode > 0) {			/* add to queue */
    status = -1;
    for (i=0; i<DMA_CHNS && status<0; i++) {
      j = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (j==stkloc) status=0; /* already in queue */
      else if (j==0) {	/* add to queue */
        pic_acmd (p, ADON_WR, DMA_QUE+i+1, 0, 0);
        pic_acmd (p, ADON_WR, DMA_QUE+i, stkloc, 0);
        status=1;
      }
    }
  }
  else {				/* remove from queue */
    status = 0;
    for (i=0; i<=DMA_CHNS; i++) {
      j = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (status == 1) pic_acmd (p, ADON_WR, DMA_QUE+i-1, j, 0);
      if (j == stkloc) status=1;	/* found it */
      if (j == 0) break;
    }
  }
  i = checkSide(p,0,0);	io_ctl (p, IOCTL_LOG, 0, &i, 4);  /* report active channels */
  pic_lock (p, LOCK_FREE);
  return (status);
}

int_4 pic_appqueue (PICSTRUCT *p, int_4 chan, int_4 mode)
{
  int_4 i,j,stkloc,status,n=chan-1;

  if (n<0 || n>=APP_CHNS) {
    print("ERROR: Queued APP channel %d out of range\n",chan);
    return -1;
  }
  stkloc = APP_STK+n*APP_SEP;

  if (pic_getkeyl(p,chan,KEY_NODMA)>0) return 1;

  if (p->app[n].dmai!=0 && mode>0) { 
    /* init the algorithm data index pointer */
    pic_acmd (p, ADON_WR, stkloc+APP_INDI, p->app[n].indi, 0);
    pic_acmd (p, ADON_WR, stkloc+APP_INDO, p->app[n].indo, 0);
  }

  pic_lock (p, LOCK_ALLOC);

  if (mode > 0) {			/* add to queue */
    status = -1;
    for (i=0; i<APP_CHNS && status<0; i++) {
      j = pic_acmd (p, ADON_RD, APP_QUE+i, 0, 0);
      if (j==stkloc) status=0; /* already in queue */
      else if (j==0) {	/* add to queue */
        pic_acmd (p, ADON_WR, APP_QUE+i+1, 0, 0);
        pic_acmd (p, ADON_WR, APP_QUE+i, stkloc, 0);
        status=1;
      }
    }
    if (i==APP_CHNS) status = -1;
  }
  else {				/* remove from queue */
    status = 0;
    for (i=0; i<=APP_CHNS; i++) {
      j = pic_acmd (p, ADON_RD, APP_QUE+i, 0, 0);
      if (status == 1) pic_acmd (p, ADON_WR, APP_QUE+i-1, j, 0);
      if (j == stkloc) status=1;	/* found it */
      if (j == 0) break;
    }
  }
  pic_lock (p, LOCK_FREE);
  return (status);
}

int_4 pic_dmachain (PICSTRUCT *p, int_4 dmac, int_4 index, DMAMAP *map, int_4 todo, int_4 nextindex)
{
  int_4 stack[4],stkloc,stklocn,dmc,n=dmac-1;

  dmc = p->isY? DM_CHAIN3 : ((dmac&0x1)!=0)? DM_CHAIN1 : DM_CHAIN2;

  if (index == -1) { /* get the current next DMA chain pointer */
    pic_stkupd (p, &p->dma[n].chain, 1, -1);
    if (p->dma[n].chain<dmc) index=0;
    else index = (p->dma[n].chain-dmc)/4 + 1;
    return index;
  }

  stkloc = dmc + (index-1)*4;
  stklocn = dmc + (nextindex-1)*4;
  if (nextindex==0) stklocn = 0;

  if (index==0) { /* set current DMA chain pointer */
    p->dma[n].chain = stklocn; 
    pic_stkupd (p, &p->dma[n].chain, 1, 1);
    p->dma[n].todo = todo; 
    pic_stkupd (p, &p->dma[n].todo, 1, 1);
  }

  if (map != 0) {
    stack[0] = L2HW(map->paddr+map->offset);
    stack[1] = L2HW(map->bytes);
    stack[2] = todo;
    stack[3] = stklocn;
    pic_transfer (p, (int_4 *)stack, sizeof(stack), stkloc, 1, 0); 
  }

  if (index==1) { 
    p->dma[n].chain = stkloc; 
    pic_stkupd (p, &p->dma[n].chain, 1, 1);
  }

  return 1;
}

int_4 pic_dmadump (PICSTRUCT *p, int_4 dmac, int_4 flags) {
  int_4 i,offset,dmap,appp,tcbp,cindp,channel,chain,index,miss,v[48];

  if (dmac==0) {
    for (i=0; i<DMA_CHNS; i++) {
      offset = DMA_QUE + i;
      channel=index=miss=0;
      dmap = pic_acmd (p, ADON_RD, offset, 0, 0);
      if (dmap==0||dmap==ACMD_NOACK) break;
      if ((dmap>=DMA_STK && dmap<DMA_STK+DMA_SEP*DMA_CHNSX) || 
          (dmap>=DMA_STK2 && dmap<DMA_STK2+DMA_SEP*DMA_CHNS2)) {
        if (dmap>=DMA_STK2) channel = (dmap-DMA_STK2)/DMA_SEP + 1 + DMA_CHNSX;
        else channel = (dmap-DMA_STK)/DMA_SEP + 1;
        cindp = pic_acmd (p, ADON_RD, dmap+DMA_CINDP, 0, 0);
        miss  = pic_acmd (p, ADON_RD, dmap+DMA_MISS, 0, 0);
        appp  = pic_acmd (p, ADON_RD, dmap+DMA_APPP, 0, 0);
        if (cindp>0) index = pic_acmd (p, ADON_RD, cindp, 0, 0);
      }
      print("DMA Stack=%08x  Chan=%02d  Index=%08x  at %08x  Miss=%d\n",dmap,channel,index,offset,miss);
    }
    if (p->isX) for (i=0; i<APP_CHNS; i++) {
      offset = APP_QUE + i;
      channel=index=0;
      appp = pic_acmd (p, ADON_RD, offset, 0, 0);
      if (appp==0||appp==ACMD_NOACK) break;
      if (appp>=APP_STK && appp<APP_STK+APP_SEP*APP_CHNS) {
        channel = (appp-APP_STK)/APP_SEP + 1;
        index = pic_acmd (p, ADON_RD, appp+APP_INDI, 0, 0);
      }
      print("APP Stack=%08x  Chan=%02d  Index=%08x  at %08x\n",appp,channel,index,offset);
    }
  }
  else if (dmac>0 && dmac<=DMA_CHNS) {
    dmap = dmastkloc(p,dmac);
    tcbp = pic_acmd (p, ADON_RD, dmap+DMA_TCBP, 0, 0);
    print ("DMA Chan %02d Parameters (in Hex):\n",dmac);
    if (p->isY && tcbp!=0) {
      for (i=0; i<2; i++) v[i] = pic_acmd (p, ADON_RD, tcbp+i, 0, 0); 
      miss  = (v[0]&0xF) + 12;
      v[4] = v[0]&0xFFFFFFF0;
      v[5] = v[4]&~PIPE_MASK(miss);
      v[6] = PIPE_SIZE(miss);
      print(" X-addr %08x   X-Size %08x   X-ind  %08x   X-End  %08x\n",v[5]>>2,v[6]>>2,v[0]>>2,v[1]>>2);
    }
    else if (tcbp>=0x30 && tcbp<=0x98) {
      for (i=0; i<6; i++) v[i] = pic_acmd (p, ADON_RD, tcbp+i, 0, 0); 
      print(" X-ind  %08x   X-inc  %08x   X-cnt  %08x \n",v[0],v[1],v[2]);
      print(" Chain  %08x   Y-cnt  %08x   Y-inc  %08x \n",v[3],v[4],v[5]);
    }
    else if (tcbp!=0) {
      print(" *** INVALID ***\n");
    }
    for (i=0; i<48; i++) v[i] = pic_acmd (p, ADON_RD, dmap+i, 0, 0); 
    print(" Haddr  %08x   Hsize  %08x   Hindex %08x   Hcycle %08x\n",v[0],v[1],v[2],v[3]);
    print(" Caddr  %08x   Csize  %08x   Cindex %08x   Ccycle %08x\n",v[4],v[5],v[6],v[7]);
    print(" Stat   %08x   Dir    %08x   ToDo   %08x   Block  %08x\n",v[8],v[9],v[10],v[11]);
    print(" Cntrl  %08x   Tcbp   %08x   Reg    %08x   Mcsr   %08x\n",v[12],v[13],v[14],v[15]);
    print(" Enb    %08x   Enbx   %08x   Mcfg   %08x   Mcfgx  %08x\n",v[16],v[17],v[18],v[19]);
    print(" Err    %08x   Miss   %08x   Iper   %08x   Multi  %08x\n",v[20],v[21],v[22],v[23]);
    print(" Hxfer  %08x   Hxfes  %08x   Cxfer  %08x   Cxfes  %08x\n",v[24],v[25],v[26],v[27]);
    print(" Cindp  %08x   Hindp  %08x   Cind   %08x   Hind   %08x\n",v[28],v[29],v[30],v[31]);
    print(" Type   %08x   Port   %08x   Bits   %08x   Dec    %08x\n",v[32],v[33],v[34],v[35]);
    print(" Rate   %08x   Gain   %08x   Fcyn   %08x   Frame  %08x\n",v[36],v[37],v[38],v[39]);
    print(" Master %08x   Slave  %08x   Feed   %08x   TCOff  %08x\n",v[40],v[41],v[42],v[43]);
    print(" Alg    %08x   Appp   %08x   Chain  %08x   Flags  %08x\n",v[44],v[45],v[46],v[47]);
    for (chain=v[46]; (chain>=DM_CHAIN1 && chain<DM_CHAIN2+DM_CHAIN_SIZE) ||
		      (chain>=DM_CHAIN3 && chain<DM_CHAIN3+DM_CHAIN_SIZE); chain=v[3]) {
      for (i=0; i<4; i++) v[i] = pic_acmd (p, ADON_RD, chain+i, 0, 0); 
      print("DMA Chain parameters at %08x\n",chain);
      print(" Haddr  %08x   Hsize  %08x   ToDo   %08x   Chain  %08x\n",v[0],v[1],v[2],v[3]);
      if (v[3]<=chain) break; /* avoid circular reference */
    }
  }
  else if (dmac<0 && dmac >= -APP_CHNS) {
    dmac = -dmac;
    appp = APP_STK + (dmac-1)*APP_SEP;
    for (i=0; i<16; i++) v[i] = pic_acmd (p, ADON_RD, appp+i, 0, 0); 
    print(" Ena    %08x   Alg    %08x   XferI  %08x   XferO  %08x\n",v[0],v[1],v[2],v[3]);
    print(" DmaI   %08x   DmaO   %08x   IndI   %08x   IndO   %08x\n",v[4],v[5],v[6],v[7]);
    print(" Arg1   %08x   Arg2   %08x   Arg3   %08x   Arg4   %08x\n",v[8],v[9],v[10],v[11]);
    print(" IcI    %08x   IcO    %08x   Tc     %08x   Tcbp   %08x\n",v[12],v[13],v[14],v[15]);
  }
  return dmac;
}

int_4 dmastkloc (PICSTRUCT *p, int_4 dmac) 
{
  int_4 stkloc=-1;
  if (dmac>DMA_CHNS) print("DMA channel %d out of range [1-%d]\n",dmac,DMA_CHNS);
  else if (dmac>DMA_CHNSX) stkloc = DMA_STK2+(dmac-1-DMA_CHNSX)*DMA_SEP;
  else if (dmac>0) stkloc = DMA_STK+(dmac-1)*DMA_SEP;
  return stkloc;  
}

int_4 loaddmacs (PICSTRUCT *p, int_4 dmac)
{
  int_4 n = dmac-1;
  int_4 stkloc = dmastkloc(p,dmac);
  pic_transfer (p, (int_4 *)(&p->dma[n]), sizeof(p->dma[n]), stkloc, -1, 0); 
  if (p->dma[n].slave > 0) loaddmacs (p,p->dma[n].slave);
  return 0;
}

#ifdef _LINX
#include <pthread.h>
#define MAXOOP 72
void *netRun (void *vp) {
  PICSTRUCT *p = (PICSTRUCT*)vp;
  int_4 n=p->dmaci-1;
  DMASTRUCT *dma = p->dma+n;
  ICEPACKETSTRUCT *pkt = (ICEPACKETSTRUCT *) p->pkt;
  VRTPACKETSTRUCT *vpkt = (VRTPACKETSTRUCT *) p->pkt;
  VRTLPACKETSTRUCT *vlpkt = (VRTLPACKETSTRUCT *) p->pkt;
  VRTWPACKETSTRUCT *vwpkt = (VRTWPACKETSTRUCT *) p->pkt;
  VRTXPACKETSTRUCT *vxpkt = (VRTXPACKETSTRUCT *) p->pkt;
  VRTDPACKETSTRUCT *vdpkt = (VRTDPACKETSTRUCT *) p->pkt;
  SDDSPACKETSTRUCT *spkt = (SDDSPACKETSTRUCT *) p->pkt;
  int_4 *pdata,*qdata,status=0,tmp[16],oop[MAXOOP*2],noop=-1;
  int_4 xfer = p->pktlen, sfer,yfer;
  int_4 seq, i, j, keys;
  int_4 bpa = (dma->bits<0)? -dma->bits*2/8 : dma->bits/8;
  int_4 spw = 4/bpa;
  int_4 phb = (p->pktmode==PKT_UDP)? 0 : (p->pktmode==PKT_VRTX)? 20 : 
	      (p->pktmode==PKT_VRTL||p->pktmode==PKT_VRTW)? 24 : 
	      (p->pktmode==PKT_VRT||p->pktmode==PKT_VRTD)?  28 : (p->pktmode==PKT_SDDS)? 56 : 64;
  int_4 ptb = (p->pktmode==PKT_VRT||p->pktmode==PKT_VRTX)? 4 : 0;	/* has trailer */
  int_4 phw = phb>>2;
  int_4 txfer = (dma->dir<0)? 9000 : phb+xfer+ptb;
  int_4 noseq = findintflagdef("NOSEQ",p->config,0);
  double time = gettime();
  double dtmp, dout = (time-p->pretcount[n]) * dma->rate * bpa;
  int_4 *ptmp = (int_4 *)malloc(txfer); /* tmp space for endian converts */

  while (dma->todo!=0) {
    qdata = p->dma_vaddr[n] + dma->hindex; 		/* where the data resides */
    if ((dma->hindex==0 && phw>0) || (dma->hindex-phw+(txfer>>2) > dma->hsize)) { 
      pdata = ptmp;					/* use ptmp buffer */
      if (dma->dir>0) memcpy((void*)(pdata+phw),qdata,xfer);
    } else { 
      pdata = qdata-phw; 				/* where the packet resides */
      if (phb>0) memcpy((void*)tmp,pdata,phb);		/* save off area for packet header */
    }
    sfer = 0;
    if (dma->dir>0) {
      dtmp = ((double)dma->hsize*dma->hcycle + dma->hindex)*4;
      if (dtmp>=dout) break;				/* past real-time output rate */
      if (p->pktmode==PKT_ICE) {			/* insert time code */
       if (p->tc_soff>=0) {				/* insert time code */
	pkt->tcwsec = p->tc_wsec;
	pkt->tcfsec = p->tc_fsec;
	pkt->tcoff  = p->tc_soff - ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
	pkt->tcstatus = TC_OK;
       } else pkt->tcstatus=0;
      }
      if (phb>0) memcpy(pdata,(void*)p->pkt,phb);
      status = nioSend (p->sock, 0, pdata,txfer, 0);
      if (status==txfer) sfer = xfer;
           if (p->pktmode==PKT_ICE) pkt->count++;
      else if (p->pktmode==PKT_SDDS) spkt->frame = swapst(swapst(spkt->frame)+1);
      else if (p->pktmode>=PKT_VRT) vpkt->header = (vpkt->header&0xFFFFF0FF) | ((vpkt->header+0x100)&0xF00);
    }
    else {
      status = nioRecv (p->sock, 0, pdata,txfer, 0);
      if (status==0 && p->threadid2!=0) break; 
      if (status==0) { usleep(1000); continue; }
      if (status<0) break;
      sfer = status-(phb+ptb);	
      if (p->pktmode==PKT_UDP) {
	seq = p->pktseq;
      }
      else if (p->pktmode==PKT_ICE) {
        pkt = (ICEPACKETSTRUCT *) pdata;
	keys = pdata[0];
        seq = pkt->count;
	if (keys != 0x00676665) { printf("Not ICE Packet Sig=%08x\n",keys); status=0; continue; }
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = pkt->tcwsec;
          p->tc_fsec = pkt->tcfsec;
          if (pkt->tcstatus<0) p->tc_soff = pkt->tcstatus;	/* error status ? if not then offset */
          else p->tc_soff = pkt->tcoff + ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode>=PKT_VRT && (pdata[0]&0xF0)==0x40) {	/* this is a Context Packet */
	if (p->verbose&1) dump_vrt(pdata);
        if (pdata!=ptmp) memcpy(qdata-phw,(void*)tmp,phb);	/* copy back area used for packet header */
	continue;
      }
      else if (p->pktmode==PKT_VRT) {
        vpkt = (VRTPACKETSTRUCT *) pdata;
        p->pktseq &= 0xF;
        seq = (vpkt->header>>8)&0xF;
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = vpkt->tsis;
          p->tc_fsec = 1.0e-12 * vxpkt->tsfs;
          p->tc_soff = ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode==PKT_VRTL) {
        vlpkt = (VRTLPACKETSTRUCT *) pdata;
        p->pktseq &= 0xF;
        seq = (vlpkt->header>>8)&0xF;
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = vlpkt->tsis;
          p->tc_fsec = 1.0e-12 * vxpkt->tsfs;
          p->tc_soff = ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode==PKT_VRTW) {
        vwpkt = (VRTWPACKETSTRUCT *) pdata;
        p->pktseq &= 0xF;
        seq = (vwpkt->header>>8)&0xF;
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = vwpkt->tsis;
          p->tc_fsec = 1.0e-12 * vxpkt->tsfs;
          p->tc_soff = ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode==PKT_VRTX) {
        vxpkt = (VRTXPACKETSTRUCT *) pdata;
        p->pktseq &= 0xF;
        seq = (vxpkt->header>>8)&0xF;
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = swapit(vxpkt->tsis);
          p->tc_fsec = 1.0e-12 * vxpkt->tsfs;
          p->tc_soff = ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode==PKT_VRTD) {
        vdpkt = (VRTDPACKETSTRUCT *) pdata;
        p->pktseq &= 0xF;
        seq = (vdpkt->header>>8)&0xF;
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = swapit(vdpkt->tsis);
          p->tc_fsec = 1.0e-12 * vdpkt->tsfs;
          p->tc_soff = ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw;
        }
      }
      else if (p->pktmode==PKT_SDDS) {
        spkt = (SDDSPACKETSTRUCT *) pdata;
  /*      if ((pkt->count&0x1F)==0x1F) pkt->count++; */
        p->pktseq &= 0xFFFF;
        seq = swapst(spkt->frame);
        if (noop<0) { noop=0; p->pktseq=seq; }
        if (seq==p->pktseq) {				/* next sequence in order - process timecode */
          p->tc_wsec = spkt->tctics;
          p->tc_fsec = spkt->tctics;
          p->tc_soff = -1; /* ((double)dma->hcycle*(double)dma->hsize+dma->hindex)*spw; */
        }
      }
      else {
	printf("Network PKTMODE=%d not coded yet\n",p->pktmode);
	break;
      }
    }
    status = sfer;
    if (pdata==ptmp) {
      i = (dma->hsize-dma->hindex)<<2;
      if (dma->dir<0) memcpy(qdata,pdata+phw,min(status,i));
      if (dma->dir<0 && status>i && dma->todo!=1) memcpy(p->dma_vaddr[n],pdata+phw+(i>>2),status-i);
    } else {
      memcpy(qdata-phw,(void*)tmp,phb);			/* copy back area used for packet header */
    }
    if (dma->dir>0);
    else if (seq==p->pktseq) {				/* next sequence in order */
      p->pktseq++;
    } else if (noseq) {
      vprint("Seq seq=%08x count=%08x addr=%08x len=%d:%d ty=%d\n",seq,p->pktseq,p->ipconn,phb,status,p->pktmode);
      p->pktseq++;
    } else {						/* next sequence not in order */
      i = seq - p->pktseq;
      if (i<0 || i>MAXOOP*2) {
        print("Seq gap=%d at seq=%08x on adr=%08x\n",i,p->pktseq,p->ipconn);
        p->pktseq = seq+1; 
	noop = 0;
      } else {
        yfer = i*(sfer>>2);
	pdata = qdata+yfer;
	if (dma->hindex+yfer >= dma->hsize) pdata -= dma->hsize;
        memcpy(pdata,qdata,sfer);			/* copy to proper slot */
        if (noop>=MAXOOP) { 				/* decide seq missing - zero fill buffer */
          print("Seq missing at seq=%08x on adr=%08x\n",p->pktseq,p->ipconn);
	  vfill (0, qdata, sfer);
          p->pktseq++; 
        } else {					/* add to reorder buffer */ 
          vprint("Seq reorder oop[%d]=%08x on adr=%08x\n",noop,seq,p->ipconn);
	  status = 0;
        }
	oop[noop++] = seq; 
      }
    }
    for (i=0; i<noop; i++) {				/* work off the OOP list */
      if (oop[i]==p->pktseq) {
	status += sfer; noop--; p->pktseq++;
	for (j=i; j<noop; j++) oop[j]=oop[j+1];
	i=-1;
      }
    }
    dma->hindex += (status>>2);
    if (dma->hindex<dma->hsize) continue;
    dma->hindex -= dma->hsize; dma->hcycle++;
    if (dma->todo>0) dma->todo--;
    if (dma->todo==0) dma->stat=0;
  } 
  if (ptmp!=NULL) free(ptmp);
  return NULL;
}
int_4 netStart (PICSTRUCT *p, char *addr) {
  int_4 flags=NIO_UDP|NIO_BIGBUF, status=0, n=p->dmaci-1;
  DMASTRUCT *dma = p->dma+n;
  flags |= (dma->dir>0)? NIO_OUTPUT:NIO_INPUT;
  p->sock = nioOpen(addr,0,flags);
  if (p->sock<=0) { print("Err: Opening socket for %s\n",addr); return -1; }
  dma->stat=1;
  if (dma->dir>0 || p->threadid2!=0) return status;
  p->threadid = 0;
  status = pthread_create(&p->threadid,NULL,netRun,(void*)p); 
  if (status!=0) print("Err: Opening socket recv thread for %s\n",addr);
  return status;
}
int_4 netStop (PICSTRUCT *p) {
  int_4 status=0, n=p->dmaci-1;
  DMASTRUCT *dma = p->dma+n;
  dma->stat=0;
  if (p->threadid!=0) {
    status = pthread_join(p->threadid,NULL); 
    if (status!=0) print("Err: Stopping socket recv thread\n");
    p->threadid = 0;
  }
  if (p->sock>0) { nioClose(p->sock); p->sock=0; }
  return status;
}
#else
int_4 netStart (PICSTRUCT *p, char *addr) { print("Not supported on this platform\n"); return -1; }
int_4 netStop (PICSTRUCT *p) { print("Not supported on this platform\n"); return -1; }
#endif

int_4 pic_dmafunc_stream (PICSTRUCT *p, int_4 dmac, int_4 func)
{
  int_4 port,status=0;
  int_4 i,n=max(0,dmac-1);
  int_u1 *pc;
  char *s,nic[40],mca[40],addr[64];
  DMASTRUCT *dma = p->dma+n;
  ICEPACKETSTRUCT *pkt = (ICEPACKETSTRUCT *) p->pkt;
  VRTPACKETSTRUCT *vpkt = (VRTPACKETSTRUCT *) p->pkt;
  VRTLPACKETSTRUCT *vlpkt = (VRTLPACKETSTRUCT *) p->pkt;
  VRTWPACKETSTRUCT *vwpkt = (VRTWPACKETSTRUCT *) p->pkt;
  VRTXPACKETSTRUCT *vxpkt = (VRTXPACKETSTRUCT *) p->pkt;
  VRTDPACKETSTRUCT *vdpkt = (VRTDPACKETSTRUCT *) p->pkt;
  SDDSPACKETSTRUCT *spkt = (SDDSPACKETSTRUCT *) p->pkt;
  int_4 ab = abs(dma->bits);
  int_4 sid = findintflagdef("SID",p->config,0);
  int_4 ouid = (p->ouid>0)? p->ouid : OUID_ICE;
  switch (func) {
  case DMA_ENABLE:
  case DMA_ONESHOT:
  case DMA_CONTINUOUS:
    dma->hindex = 0;
    dma->hcycle = 0;
    dma->todo = func;
    pc = (int_u1*)(&p->ipconn);
    findstrflag("ADDR",p->config,nic,1);
    if (p->ipsock<=0) p->ipsock = findintflagdef("IPSOCK",p->config,7000);
    port=(dma->dir>0)? 8000+(p->nicno*10)+dma->port : p->ipsock;
    port=findintflagdef("NPORT",p->config,port);
    i=findstrflag("IPCONN",p->config,mca,1);
    if (i>0) sprintf(addr,"udp:%s:%d/%s:%d",nic,port,mca,p->ipsock);
    else sprintf(addr,"udp:%s:%d/%d.%d.%d.%d:%d",nic,port,pc[0],pc[1],pc[2],pc[3],p->ipsock);
    p->tc_soff = -1;
    p->tc_wsec = 0.0;
    p->tc_fsec = 0.0;
    p->pretcount[n]=gettime();
    p->pktlen = findintflagdef("PKTLEN",p->config,1024);
    p->pktseq = 0;
    for (i=0; i<16; i++) p->pkt[i]=0;
    if (p->pktmode==PKT_ICE) {
      pkt->keys[0]=101; pkt->keys[1]=102; pkt->keys[2]=103; pkt->keys[3]=0;
      pkt->rep=(p->pktmode==PKT_ICE)?'E':'I';
      pkt->bpa=(dma->bits<0)? -dma->bits*2/8 : dma->bits/8;
      pkt->mode=(dma->bits<0)? 'C' : 'S';
      pkt->type=(ab==32)? 'L' : (ab==16)? 'I' : 'B';
      pkt->count=0;
      pkt->elem= p->pktlen/pkt->bpa;
    } else if (p->pktmode==PKT_VRT) {
      vpkt->header = 0x1C600000 + ((p->pktlen+32)>>2);
      vpkt->streamID = sid;
      vpkt->orgUID = ouid;
      vpkt->pktinfoCC = ((dma->bits<0)? 0xA200:0x8200) | (ab-1);
      for (i=0;i<4;i++) p->pkt[i] = swapit(p->pkt[i]);
    } else if (p->pktmode==PKT_VRTL) {
      vlpkt->header = 0x08600000 + ((p->pktlen+24)>>2);
      vlpkt->orgUID = ouid;
      vlpkt->pktinfoCC = (sid<<16)|((dma->bits<0)? 0xA200:0x8200) | (ab-1);
      for (i=0;i<3;i++) p->pkt[i] = swapit(p->pkt[i]);
    } else if (p->pktmode==PKT_VRTW) {
      vwpkt->header = 0x08600000 + ((p->pktlen+24)>>2);
      vwpkt->orgUID = ouid;
      vwpkt->pktinfoCC = (0<<16)|((dma->bits<0)? 0xE200:0xC200) | (ab-1);
      for (i=0;i<3;i++) p->pkt[i] = swapit(p->pkt[i]);
    } else if (p->pktmode==PKT_VRTX) {
      vxpkt->header = 0x14600000 + ((p->pktlen+24)>>2);
      vxpkt->streamID = sid; 
      for (i=0;i<2;i++) p->pkt[i] = swapit(p->pkt[i]);
    } else if (p->pktmode==PKT_VRTD) {
      vdpkt->header = 0x18600000 + ((p->pktlen+28)>>2);
      vdpkt->streamID = sid; 
      vdpkt->orgUID = OUID_DIFI;
      vdpkt->pktinfoCC = ((dma->bits<0)? 0xA200:0x8200) | (ab-1);
      for (i=0;i<4;i++) p->pkt[i] = swapit(p->pkt[i]);
    } else if (p->pktmode==PKT_SDDS) {
      spkt->fid = (ab<=4)? 0x80 : (ab<=8)? 0x81 : (ab<=16)? 0x82 : 0x00;
      spkt->bits = ab;
      spkt->frame = 0;
    }
    p->dma_vindx[n] = 0;
    p->dma_vcycx[n] = 0;
    status = netStart(p,addr);
    break;
  case DMA_STATUS:
  case DMA_STATUS_W64:
    if (dma->dir>0 || p->threadid2!=0) netRun(p);
    if (func==DMA_STATUS) status = dma->hindex<<2;
    else                  status = dma->hindex>>4;
    break;
  case DMA_CYCLE:
    status = dma->hcycle;
    break;
  case DMA_CANCEL:
  case DMA_STOP:
    dma->todo = 0;
    netStop(p);
    break;
  case DMA_POLL:
    status=dma->stat;
    break;
  }
  return status;
}

int_8 pic_dmafuncx (PICSTRUCT *p, int_4 dmac, int_4 func)
{
  int_4 n = max(0,dmac-1);
  DMASTRUCT *dma = p->dma+n;
  int_4 stkloc;
  int_8 status;
  if (func==DMA_STATUS) {
    status = pic_dmafunc(p,dmac,DMA_STATUS_W64);
    status = status<<6;
  }
  else if (func==DMA_INDEX) {
    if (p->type==ICENIC) {
      status = dma->hindex;
    } else {
      stkloc = dmastkloc(p,dmac);
      status = (int_u4)(pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0)-dma->haddr);
    }
    status = p->isY? status<<6 : status<<2;
  }
  else {
    status = pic_dmafunc(p,dmac,func);
  }
  return status;
}

int_4 core_enable (PICSTRUCT *p, int dmac, int ena) {
  int_4 i,m;
  int_4 n = max(0,dmac-1);
  DMASTRUCT *dma = p->dma+n;
  int_4 nodma = findintflagdef("NODMA",p->config,0);
  int_4 core = dma->port;
  /* handle CORE fifo resets */
  m = 0; /* decide if actually a core */
  if (dma->type==IOPT_CORE) m=KEY_CORE;
  if (dma->type==IOPT_MCORE) m=KEY_MCORE;
  if (dma->type==IOPT_TUNER && dma->dir>0 && p->mcs!=0 && p->chan>0) { m=KEY_CORE; core = (p->pmi*10) + ((dma->port&1)?3:4); }
  if (dma->type==IOPT_CORE && dma->dir>0 && p->mcs>0 && p->chan==0 && nodma==0) m = 0;	/* dont start all */
  if (m>0) {
    i = pic_getkeyl (p,-core,m);
    if (i&CORE_ENA) {
      if (ena==0) pic_setkeyl (p,-core,m,i&(~CORE_ENA));
    } else {
      if (ena==1) pic_setkeyl (p,-core,m,i|CORE_ENA);
    }
  }
  return 0;
}

int_4 pic_dmafunc (PICSTRUCT *p, int_4 dmac, int_4 func)
{
  int_4 i,j,m,port,status,stkloc,timer,dmacs,fill;
  int_4 nstat,nstart,nbytes,offset,*array,head[2],data[2];
  double fcny,qto,time,soy,fsoy; int_u4 ustat;
  int_4 n = max(0,dmac-1);
  DMASTRUCT *dma = p->dma+n;
  CHECK_MAGIC(p);

  if (p->type==ICENIC) return pic_dmafunc_stream(p,dmac,func);

  stkloc = dmastkloc(p,dmac);
  if (stkloc<0) return -1;
  if (p->state<0) return -1;

  if (dma->type==0 && (p->flags&FLG_TEST)==0) {
    /* channel not ours, retrieve parameters from CARD */
    pic_transfer (p, (int_4 *)dma, sizeof(*dma), stkloc, -1, 0); 
    dma->haddr=0; /* but dont do exit checking */
  }
  port = dma->port;
  status = 0;
  timer = 0;

  switch (func) {

  case DMA_WAIT:
    do {
      if (dma->haddr == 0) pic_dmafunc (p, dmac, DMA_STATUS); 
      status = pic_acmd (p, ADON_RD, stkloc+DMA_TODO, 0, 0);
      if (status == DMA_ONDEMAND) {
	pic_stkupd (p, &dma->hind, 1, -1); pic_stkupd (p, &dma->hindex, 1, -1);
	pic_stkupd (p, &dma->cind, 1, -1); pic_stkupd (p, &dma->cindex, 1, -1);
        if (dma->hind == dma->hindex && dma->cind == dma->cindex) {
          dma->todo = DMA_STOP;  pic_stkupd (p, &dma->todo, 1, 1); 
        }
      }
      if (UBREAK || status == ACMD_NOACK) return (-3);
      if (p->timeout>=0 && ++timer>p->timeout*100) return (-1);
      udelay(10000);
    } while (status!=0);
    if (dma->dir>0) {  
      do {
        status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0);
        if (UBREAK || status == ACMD_NOACK) return (-3);
        udelay(100);
      } while (status!=0);
      /* pause long enough for IOC output buffers to empty */
      i = dma->bits/8; if (i<0) i=-i;
      timer = (int_4)((p->isY?4096:256)*1.0e6/(dma->rate*i));
      if (timer>10) udelay(timer);
    }
    status = pic_dmafunc (p, dmac, DMA_LOST);
    if (status!=0) return -2;
    break;

  case DMA_STATUS:
  case DMA_STATUS_W64:
    /* go after non-master DMA data */
    if (dma->haddr == 0 && p->dma_vaddr[n] != 0) {
      i=0; array = p->dma_vaddr[n];
      pic_lock(p, LOCK_ALLOC);
      TRY4MORE:
      offset = pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0);
      if (offset<0 || offset>=dma->hsize) { 
        print("Bad DSTAT offset=%x size=%x\n",offset,dma->hsize);
	offset = 0;
      } 
      status = pic_acmd (p, ADON_DSTAT, stkloc, 0, 0);
      if (status == dma->hxfes) {
        if (p->isY) { offset<<=4; status<<=4; }
        if (dma->dir<0) pic_rfifo (p, array+offset, status*4, 0);
        else            pic_wfifo (p, array+offset, status*4, 0);
	if (++i<16) goto TRY4MORE;
      }
      else if (status != 0) print("Bad DSTAT status = %x %x\n",status,offset);
      pic_lock(p, LOCK_FREE);
    }
    if (dma->hindp >= REG_AHIRX) { 
      RD((dma->hindp<<2)&0xFFF,status);    /* low overhead access */
     if (p->isY) {
      for (j=0; status==0 && j<2; j++) RD((dma->hindp<<2)&0xFFF,status); /* RAM contention retry */
      if (status==0) status = pic_acmd (p, ADON_RD, dma->hindp, 0, 0); /* internal access to fast register */
     }
      if (status==0) status = pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0); /* internal - not started yet */
      ustat = status;
      if (ustat<dma->haddr || (ustat>dma->haddr+dma->hsize && dma->chain==0)) { /* range check - no chain */
        RD((dma->hindp<<2)&0xFFF,i);
        j = pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0);
	print("Got AHIR dmac=%d status=%08x stat2=%08x hindx=%08x start=%08x end=%08x\n", n+1,status,i,j,dma->haddr,dma->haddr+dma->hsize);
      }
    } else {
      status = pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0);
    }
    if (status == ACMD_NOACK) return -1;
    fill = status-dma->hind;
    if (fill<0) fill += dma->hsize;
    status -= dma->haddr;  	/* buffer offset */
    if (p->isY) {
      if (func==DMA_STATUS_W64);/* already in 64by words */
      else status <<= 6;	/* convert 64by to 1by */
    } else {
      if (func==DMA_STATUS_W64) status >>= 4; /* convert 4by to 64by */
      else status <<= 2;	/* convert 4by to 1by */
    }
    if (p->joinTime>0.0) {
      time = pic_time();
      qto = (status==0 && dma->hcycle==0)? 1.0 : p->joinTimer;
      if (qto<0) p->joinTime=0.0;
      else if (time-p->joinTime > qto) pic_enable_qsfp(p,dmac,1);
    }
    if (p->vctlTime>0.0) {
      time = pic_time();
      qto = p->vctlTimer;
      if (qto<0) p->vctlTime=0.0;
      else if (time-p->vctlTime < qto);	/* not ready yet */
      else if (p->vctlFill>=0) pic_send_vctl(p,dmac,p->vctlFill,time);
      else pic_send_vctl(p,dmac,((double)fill)/dma->hsize,time);
    }
    if (p->plan[0]!=NULL && p->pollTimer>0.0) {
      qto = pic_time(); if (qto-p->pollTimer > 1.0) { core_poll(p->plan[0]); p->pollTimer=qto; }
    }
    break;

  case DMA_INDEX:
    status = pic_acmd (p, ADON_RD, stkloc+DMA_HINDEX, 0, 0);
    status -= dma->haddr;  	/* buffer offset */
    status = p->isY? status<<6 : status<<2;
    break;

  case DMA_CYCLE:
    status = pic_acmd (p, ADON_RD, stkloc+DMA_HCYCLE, 0, 0);
    if (status != dma->hcycle) pic_log(p,0);
    dma->hcycle = status;
    break;

  case DMA_POLL:
    if (dma->haddr == 0) pic_dmafunc (p,dmac,DMA_STATUS);
    status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0);
    break;

  case DMA_ENABLE: 
    func = DMA_SPIN;
    pic_transfer (p, (int_4 *)(&p->dma[n]), sizeof(p->dma[n]), stkloc, 1, 0); 
    status = pic_acmd (p, ADON_WR, stkloc+DMA_DIR, 0, 0);
    goto DMA_NSHOT;

  case DMA_ONDEMAND: 
    if (!p->isY) { print("Throttle mode=ONDEMAND not supported on this card"); return -1; }
    status = pic_acmd (p, ADON_WR, stkloc+DMA_ALG, 0x02000080+(n<<2), 0); 	/* JVM addr of inbyte */
    if (findintflag("NOPREFILL",p->config)>0) pic_acmd (p, ADON_WR, stkloc+DMA_STAT, -1, 0); 
    pic_setkeyl (p, dmac, KEY_INBYTE, 0);
  case DMA_ONESHOT: 
  case DMA_CONTINUOUS: 
  case DMA_LOADNSPIN: 
  case DMA_LOOPNSPIN: 
    DMA_NSHOT:
    pic_lock(p, LOCK_ALLOC);
    vprint ("DMAFUNC Start chan=%d, port=%d func=%d type=%d feed=%x\n",dmac,port,func,dma->type,dma->feed);
    if (dma->slave>0) pic_dmafunc (p, dma->slave, DMA_SPIN);
    /* update Time on QSFP packets */
    time = finddblflagdef("CTIME",p->config,0.0);
    if (abs(time)<60.0) time = pic_time() + time;		/* CTIME is an offset */
    soy  = floor(time); fsoy = time-soy;
    if (findintflag("MTGO",p->config)>0) { soy+=1; fsoy=0; }
    if (findflag("MTGOOFF",p->config)>0) { soy+=1; }
    p->zeroTime = soy+fsoy;
    if (p->nio!=0 && dma->type!=IOPT_STREAM && dma->dir<0) { pic_tcupx_qsfp (p,n+1,soy,fsoy); }
    /* update Time and perform startup on cores */
    core_startup(p,0,soy+fsoy-631152000);
    dma->todo = func;
    if (dma->feed!=0 && dma->type!=IOPT_INTERNAL) activatePM (p,dma,1);
    core_enable(p,dmac,1);   /* handle CORE fifo resets */
    /* if host DMA, check to make sure this buffer is fully mapped */
    if (dma->haddr!=0 && !pic_mapcheck(p,HW2P(dma->haddr),HW2P(dma->hsize))) {
      print ("dma_func start err: DMA Buffer is not mapped.\n"); status = -1;
    } 
    else if ((dma->type==IOPT_TUNER||dma->type==IOPT_MCORE) && dma->master>0) {
      dmacs = dma->master;
      nstat = pic_acmd (p, ADON_NSTAT, dmacs, 0, 0); 
      if (!NSTAT_ACT) pic_dmafunc (p, dmacs, DMA_SPIN); 
      pic_dmaqueue(p, dmac, 1);
      status = dmass (p, n, port, func);
    } else {
      pic_dmaqueue(p, dmac, 1);
      status = dmass (p, n, port, func);
    }
    pic_lock(p, LOCK_FREE);
    if (p->plan[0]!=NULL) { if (core_poll(p->plan[0])>0) p->pollTimer=pic_time(); } 	/* start core poll timer */
    p->dma_vindx[n] = 0;
    p->dma_vcycx[n] = 0;
    break;

  case DMA_RESHOT:
    vprint ("DMAFUNC ReShot chan=%d, port=%d func=%d\n",dmac,port,func);
    pic_lock(p, LOCK_ALLOC);
    if (dma->slave>0) pic_dmafunc (p, dma->slave, func);
    pic_acmd (p, ADON_WR, DMA_GMCFG, 0, 0); 
    status = pic_acmd (p, ADON_WR, stkloc+DMA_TODO, 1, 0); /* kick it */
    pic_lock(p, LOCK_FREE);
    break;

  case DMA_CANCEL:
#if JEFFO
    if (p->nio!=0 && dma->type==IOPT_STREAM) pic_setup_qsfp (p,dmac,dma->dir,0,0,FLG_DISABLE);
    if (p->nio!=0 && dma->type!=IOPT_STREAM) pic_setup_qsfp (p,dmac,-dma->dir,0,0,FLG_DISABLE|((dma->dir<0)?FLG_NIO:0));
#endif
    status = pic_acmd (p, ADON_RD, stkloc+DMA_TODO, 1, 0);
    if (dma->todo==0 && status==0) { pic_dmaqueue(p, dmac, -1); break; }

  case DMA_STOP:
    vprint ("DMAFUNC Stop chan=%d, port=%d func=%d\n",dmac,port,func);
    pic_lock(p, LOCK_ALLOC);
    dmass (p, n, port, func);
    dma->todo = 0; status = 0;
    pic_dmaqueue(p, dmac, -1);
    if (dma->slave>0) pic_dmafunc (p, dma->slave, func);
    if ((dma->type==IOPT_TUNER||dma->type==IOPT_MCORE||dma->type==IOPT_STREAM) && dma->master>0) {
      nstat = pic_acmd (p, ADON_NSTAT, dma->master, 0, 0); 
      if (NSTAT_CHN==0) pic_dmafunc (p, dma->master, func);
      if (NSTAT_CHN==0) pic_dmafunc (p, dma->master, DMA_RESET);
    }
    if (dma->feed!=0 && dma->type!=IOPT_INTERNAL) activatePM (p,dma,-1);
    if (p->plan[0]!=NULL && p->pollTimer>0.0) { core_poll(p->plan[0]); p->pollTimer=0; }
    core_enable(p,dmac,0);   /* handle CORE fifo resets */
    pic_lock(p, LOCK_FREE);
    p->dma_vindx[n] = 0;
    p->dma_vcycx[n] = 0;
    break;

  case DMA_RESET: 
    if (dma->slave>0) pic_dmafunc (p, dma->slave, func);
    vprint ("DMAFUNC Reset chan=%d, port=%d func=%d\n",dmac,port,func);
    pic_lock(p, LOCK_ALLOC);
    pic_dmaqueue(p, dmac, -1);
    /* reset the Tuner chip to synchronize next acquisition start */
    if (dma->type==IOPT_TUNER && dma->master==0) { 
      fcny = (double)dma->fcny / FULLSCALE;
      pic_tuner (p, port, dma->bits, dma->rate, fcny, dma->dec, dma->gain, 0);
    }
    /* reset the Tuner bank to synchronize next acquisition start */
    if (dma->type==IOPT_TBANK) { 
      fcny = (double)dma->fcny / FULLSCALE;
      pic_tbport (p, port, -1,dma->dir, dma->bits, dma->rate, fcny, dma->dec, dma->gain, p->tbflags);
    }
    /* reset the master channel to synchronize next acquisition start */
    if (dma->type!=IOPT_TUNER || dma->master<=0 || p->isY) {
      i = p->isY? ( (dma->tcbp>=(PPC_DMA_XI>>2) && dma->tcbp<(PPC_DMA_TX>>2))? 8:4 ) : sizeof(p->tcb[n]);
      pic_transfer (p, (int_4 *)(&p->tcb[n]), i, dma->tcbp, 1, 0);
    }
    dma->todo = 0;    
    dma->hcycle = dma->ccycle = dma->miss = dma->err = 0;
    pic_transfer (p, (int_4 *)(&p->dma[n]), sizeof(p->dma[n]), stkloc, 1, 0); 
    /* reset any associated ALG */
    m = (dma->appp!=0)? (dma->appp-APP_STK)/APP_SEP : -1;
    if (m>=0 && m<APP_CHNS && dma->alg!=0) 
      pic_transfer (p, (int_4 *)(&p->app[m]), sizeof(p->app[m]), dma->appp, 1, 0);
    pic_lock(p, LOCK_FREE);
    break;

  case DMA_ACTIVE:
    status = 0;
    pic_lock(p, LOCK_ALLOC);
    for (i=0; i<DMA_CHNS; i++) {
      stkloc = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (stkloc==0) break;
      j = pic_acmd (p, ADON_RD, stkloc+DMA_PORT, 0, 0);
      if ( ((j^port)&0x1) != 0) continue;
      j = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0);
      if (j!=0) status += 1;
    }
    pic_lock(p, LOCK_FREE);
    vprint ("DMAFUNC Active chan=%d, port=%d nactive=%d\n",dmac,port,status);
    break;

  case DMA_SPIN:
    if (p->type==ICESLIC3 && dma->type==IOPT_TUNER && (dma->mcfg&IOM_SGO)!=0) break;
    if (dma->slave>0) pic_dmafunc (p, dma->slave, DMA_SPIN);
    vprint ("DMAFUNC Spin chan=%d, port=%d func=%d\n",dmac,port,func);
    if (dma->feed!=0 && dma->type!=IOPT_INTERNAL) activatePM (p,dma,1);
    core_enable(p,dmac,1);   /* handle CORE fifo resets */
    pic_dmaqueue(p, dmac, 1);
    dmass (p, n, port, func);
    break;

  case DMA_LOOP:
    vprint ("DMAFUNC Loop chan=%d, port=%d func=%d\n",dmac,port,func);
    pic_dma (p, dmac, -1, 0, 0, 256*1024*1024, 0, 0);
    pic_dmaqueue(p, dmac, 1);
    dmass (p, n, port, func);
    break;

  case DMA_LOAD:
    vprint ("DMAFUNC Load chan=%d, port=%d func=%d\n",dmac,port,func);
    if (p->dma_vaddr[n] == 0) { print("ERR: DMA LOAD Vaddr=0\n"); break; }
    nbytes = dma->csize*4;
    if (p->isY && p->type!=ICEPIC5) goto LOAD_DMAX;

    array = (int_4 *)malloc(nbytes); /* tmp space for endian converts */
    memcpy((void*)array,(void*)p->dma_vaddr[n],nbytes);
    if (p->isX && abs(dma->bits)==16) ConvertEndian (array,dma->csize*2,16);
    ConvertEndian (array,dma->csize,32);
    for (i=0; i<dma->csize; i+=j) {
      j = min(0x8000,dma->csize-i);
      pic_transfer (p, array+i, j*4, dma->caddr+i, 1, 1);
    }
    free(array);
    break;

    LOAD_DMAX:
    status = pic_map (p, &array, &nstart, nbytes, 1);
    memcpy((void*)array,(void*)p->dma_vaddr[n],nbytes);
    status = pic_acmd (p, ADON_DMAX, (int_4)(phys2long(nstart)>>6), dma->caddr<<2, nbytes);
    status = pic_map (p, &array, &nstart, nbytes, -1);
    break;

  case DMA_LOST:
    status = pic_acmd (p, ADON_RD, stkloc+DMA_MISS, 0, 0);
    if (dma->slave>0) status += pic_dmafunc (p, dma->slave, DMA_LOST);
    break;

  case DMA_KILL:
    status = 0;
    if (port&0x1) m=1; else m=2;
    vprint ("DMAFUNC Kill chan=%d, port=%d side=%d\n",dmac,port,m);
    pic_acmd (p, ADON_MOD, m, 0, 0); 
    pic_acmd (p, ADON_WR, DMA_GMCFG+m, 0, 0);
    break;

  case DMA_BURY:
    status = 0;
    if (port&0x1) m=0; else m=1;
    vprint ("DMAFUNC Bury chan=%d, port=%d side=%d\n",dmac,port,m);
    pic_lock(p, LOCK_ALLOC);
    for (i=0; i<DMA_CHNS; i++) {
      stkloc = pic_acmd (p, ADON_RD, DMA_QUE+i, 0, 0);
      if (stkloc==0) break;
      n = (stkloc-DMA_STK)/DMA_SEP; dmacs = n+1;
      if (n%2!=m) continue; /* different side */
      j = pic_acmd (p, ADON_RD, stkloc+DMA_MASTER, 0, 0);
      if (j<0) continue; /* master channels get cancelled elsewhere */
      loaddmacs(p,dmacs);
      pic_dmafunc(p,dmacs,DMA_STOP);
      pic_dmafunc(p,dmacs,DMA_RESET);
      print("Buried chan=%d\n",dmacs);
      i=0; /* start over */
    }
    pic_lock(p, LOCK_FREE);
    break;

  default:
    if (func>=1) goto DMA_NSHOT;
    print ("Unsupported function %d in pic_dmafunc()\n",func);
    status = -1;
  }
  return (status);
}

int_4 wait4sync (PICSTRUCT *p, int_4 side, int_4 mcfg1) {
  int_4 i,stat;
  pic_acmd (p, ADON_MOD, side, mcfg1, 0); 
  /* make sure sync has worked its way through the IOC Fifo */
  for (i=0; i<100; i++) {
    stat = pic_acmd(p,ADON_RDIOC,IOC_STAT|(side<<6),0,1);
    if ((stat&IOC_STAT_SEN)==0) break; 
    udelay(10000);
  }
  /* and pipeline stages */
  if (p->type!=ICEPIC4) udelay(10000);
  return 0;
}

int_4 dmass (PICSTRUCT *p, int_4 n, int_4 port, int_4 func)
{
  int_4 i,j,k,ic,nc,spc,side,aside,tside,mside,mcfg,mcfg1,mcfg2,stat,stkloc,status,timer,premcfg,pport;
  int_4 pmi,chip,chan,tadr,sadr,val_pm,mask,running,cpc,cadr,needtime,needsync,lastsync,nodma,isqsfp;
  DMASTRUCT *dma = p->dma+n;

  stat = -1;	/* assume failure */
  stkloc = dmastkloc(p,n+1);
  needsync = 0;
  mask = IOM_COE|IOM_MCEN|IOM_MCOE;
  nodma = pic_getkeyl(p,n+1,KEY_NODMA);
  isqsfp = (dma->type==IOPT_STREAM && dma->dir<0) || (p->nio!=0 && dma->type!=IOPT_STREAM && dma->dir>0);

  mcfg = dma->mcfg;
  side = (mcfg>>28)&0x3;
  if (side==0) side = (port&0x1)? 1:2;
  mside = side;
  if (side==3) side=(dma->port==2)?2:1;
  tside = (port&0x1)? 1:2;
  pmi = -1;
  if ((dma->type==IOPT_TUNER || dma->type==IOPT_MCORE) && p->gchip>0) {
    if (dma->master<0) { chip = port; chan = -1; pmi = (p->npm<=0)? 0 : (n==10||n==11)? 2:1; }
    else { pport=port; pmi=port2pmi(p,&pport); chip=port2chip(p,pport); chan=port2chan(p,pport); }
  }
  else if (dma->type==IOPT_TBANK) {
    pmi  = (p->npm<=0)? 0 : (port>20)? 2 : (port>10)? 1 : p->pmi;
    chip = (port%10) + ((p->fttm>1)?12:8);
    chan = -1;
  }
  else if (dma->type==IOPT_CORE) {
    pmi  = (p->npm<=0)? 0 : (port>20)? 2 : (port>10)? 1 : p->pmi;
  }
  else chip=chan=0;
  if (p->type==ICEPIC4) cadr=0x30; else cadr=0x80;
  cpc = p->cpc;

  pic_lock(p, LOCK_ALLOC);

  if (func == DMA_STOP || func == DMA_CANCEL) {

    mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+side, 0, 0);

    status = pic_acmd (p, ADON_RD, stkloc+DMA_TODO, 0, 0);
    status = pic_acmd (p, ADON_WR, stkloc+DMA_TODO, 0, 0);
    status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0);
    for (timer=0; status!=0 && status!=ACMD_NOACK && timer<100; timer++) {	
      udelay(10000); 
      status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0);
      if (UBREAK)  timer += 10;
    }
    if (dma->dir>0) udelay(10000); 
    /* this ORs all the mcfg bits from the still active channels */
    mcfg2 = pic_acmd (p, ADON_QUERY, side, 0, QUERY_MSTAT);
    /* keep alt port enabled if using INP flags */
    if (p->inp==side) mcfg2 |= (mask&mcfg1);
    vprint("DMASS Stop %d p=%d c=%d s=%d mcfgs %08x %08x %08x\n",
	n+1, port, chip, mside, mcfg1, mcfg, mcfg2);
    if (mcfg1==0) mcfg2 = mcfg1; /* DMA_KILL signal */

    /* handle tuner master channel shutdown */
    if (chip>0 && dma->master<=0) {
      tadr = chip2tadr(p,chip,pmi);
      if (p->gchip==GC4016) {
	for (ic=0; ic<4; ic++) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x77,0); /* dec/flush sync */
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_NS,(p->tflags&FLG_DSYNC)?0x72:0x22,0);
	}
	pic_acmd (p,ADON_WRIOB,tadr|cadr,0x1F,1); /* disable channel */
      } 
      else if (p->gchip==GC4014) {
        pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,0x3,0); 
        pic_acmd (p,ADON_WRIOB,tadr|GC4014_CFC,0xFF,0);
      }
      if (p->tflags&FLG_PRESAMP) { 
        stat = pic_rpb (p,0,pic5tadrtbl[side]);
        pic_wpb (p,0,pic5tadrtbl[side],stat&~GCFPGA_SYS_ENABLE);
      }
    }

    /* handle tuner channel shutdown */
    if (chip>0 && dma->master>0) {
      tadr = chip2tadr(p,chip,pmi);
      if (p->gchip==GC4016 && (p->tflags&FLG_ITDEC)!=0) {
	for (ic=0; ic<4; ic++) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	  if (chanfor(ic,cpc)==chan) {
            pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x77,0); /* dec/flush sync */
	    pic_acmd (p,ADON_WRIOB,tadr|GC4016_NS,(p->tflags&FLG_DSYNC)?0x72:0x22,0);
            pic_acmd (p,ADON_WRIOB,tadr|cadr,(1<<ic),1); 
	  }
        }
        /* wait for demux algorithm to flush old results + 0x200 samples */
        i = pic_acmd (p, ADON_RD, p->dma[dma->master-1].cindp, 0, 0);
        udelay(10000);
        i = pic_acmd (p, ADON_RD, p->dma[dma->master-1].cindp, 0, 0) - i; 
        if (i<0) i+= p->dma[dma->master-1].csize;
        if (i==0); /* its already stopped */
        else if (i>0x200); /* its already flushed */
        else udelay( min(100000,0x200*10000/i) );
      }
      else if (p->gchip==GCFPGA) {
	tadr += MCORE_CHN*chan;
        stat = pic_rpb (p,pmi,tadr|GCFPGA_SYS);
        pic_wpb (p,pmi,tadr|GCFPGA_SYS,stat&~GCFPGA_SYS_ENABLE);
      }
    }

    if (mcfg2!=mcfg1) {
      if (mside==1||mside==3) pic_acmd (p, ADON_WR, DMA_GMCFG+1, mcfg2, 0); 
      if (mside==2||mside==3) pic_acmd (p, ADON_WR, DMA_GMCFG+2, mcfg2, 0); 
      pic_acmd (p, ADON_MOD, mside, mcfg2, 0);
    }
    if (dma->dir>0 && !p->isY) { /* flush link port output FIFOs */
      if (p->isX) i=0x00100401; else i=0x00111111;
      pic_acmd (p, ADON_WRM, dma->reg, i, dma->enb); 
      pic_acmd (p, ADON_WRM, dma->reg, 0, dma->enb); 
    }
    if (dma->dir>0 && p->mcs!=0) {
      tadr = core_addr(port);
      pic_wpb (p,pmi,tadr|MCORE_ENA,0);
    }
    /* kick off NIO inputs */
    if (isqsfp) pic_enable_qsfp (p,n+1,0);

    /* reset the I/O module */
    if ((mcfg2&IOM_ENA)==0) pic_enable_module(p,side,0);
    
  } else {

    /* enable module on alternate port if appropriate */
    if ((mcfg&IOM_ALT)!=0) {
      aside = 3-side; 
      mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+aside, 0, 0);
      if ((mcfg1&IOM_PLY)!=0) mcfg2=mcfg1;
      else mcfg2 = (mcfg1&(~mask)) | (mcfg&mask); 
      vprint("DMASS Start %d Alt Port mcfgs %08x %08x %08x\n",n+1,mcfg1,mcfg&mask,mcfg2);
      pic_acmd (p, ADON_MOD, aside, mcfg2, 0); 
      pic_acmd (p, ADON_WR, DMA_GMCFG+aside, mcfg2, 0); 
      mcfg = mcfg&(~mask);
    }
    mcfg1 = pic_acmd (p, ADON_RD, DMA_GMCFG+side, 0, 0);
    mcfg2 = mcfg1 | mcfg;
    vprint("DMASS Start %d p=%d c=%d s=%d mcfgs %08x %08x %08x\n",
	n+1, port, chip, side, mcfg1, mcfg, mcfg2);
    if (((mcfg&mcfg1)&IOM_PORTS)!=0 && nodma==0) {
      print("Port resource contention - Mcfg0=%08x Mcfg1=%08x - Startup aborted.\n",mcfg,mcfg1); goto DONE;
    }
    /* enable clock output early to initialize A2D pipeline registers, etc */
    if ((mcfg2&IOM_COE)!=0 && (mcfg1&IOM_ENA)==0 && (p->gmcr&GMC_NONCONT)==0) {
      premcfg = mcfg2 & (~(IOM_ENA|IOM_TEN));
      pic_acmd (p, ADON_MOD, side, premcfg, 0); 
    }
    /* check for data width consistency on PIC5+ series */
    if (p->isY && (mcfg1&IOM_ENA)!=0 && (mcfg>>28)!=0 && (mcfg1&IOM_BITS) != (mcfg&IOM_BITS)) {
      print("Cannot mix input widths on series 5+ cards. Startup aborted..\n");
      print("Note: Tuners default to 16bit input. See MBITS=n flag.\n"); goto DONE;
    }
    if (dma->dir>0 && p->mcs!=0) {
      tadr = core_addr(port);
      pic_wpb (p,pmi,tadr|MCORE_ENA,CORE_ENA);
    }
    /* enable the I/O module */
    if ((mcfg1&IOM_ENA)==0 && (mcfg2&IOM_ENA)!=0) pic_enable_module(p,side,1);
    
    /* see if module port is already running */
    status = pic_acmd (p,ADON_RDIOC,IOC_STAT|(side<<6),0,1);
    if (chip==0) running = (status&IOC_STAT_ENB)!=0; 
    else         running = (status&IOC_STAT_SEN)!=0;
    /* wait for last tuner sync/start to occur if raw port is running */
    if (chip!=0 && !running && (mcfg1&IOM_SEN)!=0 && (status&IOC_STAT_ENB)!=0) {
      i = 0x1000*(1000000.0/dma->rate); udelay(min(100000,i));
      status = pic_acmd (p,ADON_RDIOC,IOC_STAT|(side<<6),0,1);
      running = (status&IOC_STAT_SEN)!=0;
      if (!running) print("Delayed %duS but no data yet, check rate, starting anyway\n",i);
    }

    /* handle tuner master channel startup */
    if (chip>0 && dma->master<0) {
      /* dont resync GC's already running - this messes up timecode */
      if (running && p->gchip==GC4014) for (i=tside; i<=p->cpb; i+=2) if (i!=chip) {
	tadr = p->tadrtbl[i]; 
	status = pic_acmd (p,ADON_RDIOB,tadr|GC4014_CFC,0,0);
	if (status==0) continue; /* already desynced */
	vprint("Disable sync line to chip=%d stat=%x\n",i,status);
	pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,0,0); 
	pic_acmd (p,ADON_WRIOB,tadr|GC4014_CFC,0,0);
	val_pm = 0xF0 & pic_acmd (p,ADON_RDIOB,tadr|GC4014_PM,0,0);
	for (ic=0; ic<p->cpc; ic++) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,val_pm|(4+ic),0);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CS,0x0F,0);
	}
      } 
      /* lower the sync to restart GC on TC sync pulse */
      if (running && p->gchip==GC4014) { 
	mcfg1&=IOM_NOSEN; mcfg2&=IOM_NOSEN;
        wait4sync(p,side,mcfg1);
      }
      tadr = chip2tadr(p,chip,pmi);
      if (p->gchip==GC4014) { 
	pic_acmd (p,ADON_WRIOB,tadr|GC4014_SM,0x1,0); 
	pic_acmd (p,ADON_WRIOB,tadr|GC4014_CFC,0x55,0);
	val_pm = 0xF0 & pic_acmd (p,ADON_RDIOB,tadr|GC4014_PM,0,0);
	for (ic=0; ic<p->cpc; ic++) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4014_PM,val_pm|(4+ic),0);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4014_CS,0x5F,0);
	}
      } 
      else if (p->gchip==GC4016 && (p->tflags&FLG_ITDEC)==0) { 
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x58,0); 
        udelay(10000);
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x08,0); 
	for (ic=0; ic<4; ic++) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x22,0); /* dec/flush sync */
	}
	pic_acmd (p,ADON_WRIOB,tadr|0x30,0x90,1); /* enable channel */
      } 
      else if (p->gchip==GC4016) {
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x58,0); 
        udelay(10000);
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x08,0); 
	/* kludge: chan 1 must run 1st for some reason */
	for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==0) {
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x00,0); } /* dec/flush sync */
	udelay(10000); /* insure ample clocks */
	for (ic=3; ic>=0; ic--) if (chanfor(ic,cpc)==0) {
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x77,0); } /* dec/flush sync */
      }
    }
    if (chip>0 && dma->master<=0) {
      if (p->tflags&FLG_PRESAMP) { 
        stat = pic_rpb (p,0,pic5tadrtbl[side]);
        pic_wpb (p,0,pic5tadrtbl[side],stat|GCFPGA_SYS_ENABLE);
      }
    }

    /* handle tuner channel startup */
    if (chip>0 && dma->master>0) {
      if (p->gchip==GC4016 && (p->tflags&FLG_ITDEC)!=0) {
	needsync=1; val_pm=0;
	dma->cind = dma->caddr; pic_stkupd (p, &dma->cind, 1, 1);
	if (running) for (i=1; i<=p->cpb; i++) {
	  if (p->isY && p->inp>0) ; /* both sides for shared input case */
	  else if ( ((i^side)&0x1) != 0) continue; /* other side - leave alone */
          tadr = chip2tadr(p,i,pmi);
	  for (ic=0; ic<4; ic++) if (i!=chip || chanfor(ic,cpc)!=chan) {
	    pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	    status = pic_acmd (p,ADON_RDIOB,tadr|GC4016_DS,0,1);	
	    if (status==0x22) {
	      pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x00,0);
	      pic_acmd (p,ADON_WRIOB,tadr|GC4016_NS,(p->tflags&FLG_DSYNC)?0x70:0x00,0);
	    }
	    if (status!=0x77 && i==chip) val_pm++;
	  }
	}
        tadr = chip2tadr(p,chip,pmi);
	if (chan==0 && val_pm==0) { /* 1st chan really running 1st - resync */
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x58,0); 
	  udelay(10000);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x08,0); 
	}
	if (running) { mcfg1&=IOM_NOSEN; wait4sync(p,side,mcfg1); }
	for (ic=0; ic<4; ic++) if (chanfor(ic,cpc)==chan) {
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	  pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x22,0);	/* dec/flush sync */
	  pic_acmd (p,ADON_WRIOB,tadr|cadr,0x80|(1<<ic),1);	/* enable channel */
	}
      }
      else if (p->gchip==GCFPGA || dma->type==IOPT_MCORE) {
	needsync = (mcfg&IOM_TEN)!=0;
        tadr = chip2tadr(p,chip,pmi) + MCORE_CHN*chan;
        stat = pic_rpb (p,pmi,tadr|GCFPGA_SYS);
        if (needsync && running) { mcfg1&=IOM_NOSEN; wait4sync(p,side,mcfg1); }
        pic_wpb (p,pmi,tadr|GCFPGA_SYS,stat|GCFPGA_SYS_ENABLE);
      }
      else {
	if ((mcfg1&IOM_SEN)==0) needsync=1;
      }
    }
    if (chip>0 && dma->master==0 && p->gchip==GCFPGA) {
      if (running) { mcfg1&=IOM_NOSEN; wait4sync(p,side,mcfg1); }
      needsync=1;
    }

    /* handle tuner bank startup */
    if (chip>0 && dma->master==0 && p->gchip==GC4016) {
      tadr = chip2tadr(p,chip,pmi);
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x58,0); 
      udelay(10000);
      pic_acmd (p,ADON_WRIOB,tadr|GC4016_GR,0x08,0); 
      if (running) { mcfg1&=IOM_NOSEN; wait4sync(p,side,mcfg1); }
      for (ic=0; ic<4; ic++) {
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_PM,(8*ic+7)<<1,0);
	pic_acmd (p,ADON_WRIOB,tadr|GC4016_DS,0x22,0);	/* dec/flush sync */
	pic_acmd (p,ADON_WRIOB,tadr|cadr,0x80|(1<<ic),1); /* enable channel */
      }
      needsync=1;
    }

    sadr = (side==1)? (IOC_A|IOC_TCOUNT) : (IOC_B|IOC_TCOUNT);
    if (needsync) mcfg2 |= IOM_SEN;
    if (p->tcmode!=TCM_OFF) {
      pic_acmd (p, ADON_WRIOC, sadr, 0xFF, 1); 
      lastsync = pic_acmd (p, ADON_RDIOC, sadr, 0, 4); 
    }

    /* setup timecode modes */
    if (p->tcmode!=TCM_OFF && !(dma->type==IOPT_MODULE && dma->master>0)) { 
      i=0; if (dma->rate<50000000) i |= IOC_TC_NODC; 
      if (p->tcmode==TCM_IRB)     i |= IOC_TC_IRIG;
      if (p->tcmode==TCM_SDD)     i |= IOC_TC_SDDS;
      if (p->tcflags&FLG_TC_OPPS) i |= IOC_TC_OPPS;
      if (p->tcflags&FLG_TC_SWAP) i |= IOC_TC_SWAP;
      if (p->tcflags&FLG_TC_PREFILL) i |= IOC_TC_FILL1;
      if (p->tcmode==TCM_SDN || p->tcmode==TCM_DTL) { 
        i |= IOC_TC_SDN; i |= 0x2000;
        if (dma->rate<72000) print("SDN/DTL Timecode rate=%d not supported below 72kHz\n",dma->rate);
      }
      j = (p->isY && (dma->mcfg&IOM_BMUX))? 3 : side;
      vprint("Writing TCMODE side=%d reg=%04x IRIG-B reg=%02x\n",j,i,p->irbreg);
      pic_acmd (p,ADON_WRIOC,(j<<6)|IOC_TCMODE,i,0); 
      if (p->isX) pic_acmd(p,ADON_WRIOC,IOC_IRIG,p->irbreg,1);  
    }

    if (p->type==ICESLIC3 && (p->tflags&FLG_TALT1)==0 && chip!=0 && 
        dma->master>0 && (chan&1)!=0 && (mcfg&IOM_ENA)!=0) { 	/* auto start side A */
      premcfg = (mcfg2&0x0FFFFFFF)|IOM_SGO;
      status = pic_acmd (p,ADON_RDIOC,IOC_STAT|(1<<6),0,1);
      if ((status&IOC_STAT_ENB)==0) pic_acmd (p, ADON_MOD, 1, premcfg, 0);
    }

    /* set the register contents for IOC startup */
    if (mcfg2!=mcfg1 && dma->reg!=0) pic_acmd (p, ADON_WR, DMA_GMCFG, mcfg2, 0);
    else pic_acmd (p, ADON_WR, DMA_GMCFG, 0, 0);

    if (nodma) dma->reg=0; 

    /* handle route flags on PIC5+ cards */
    if (p->isY && dma->reg!=0) {
      /* status = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0); */
    }

    /* kick on the port */
    if (dma->tcbp==DMA_TCBTMP) pic_acmd (p, ADON_WR, DMA_TCBTMP+TCB_YC, 0xFFFF, 0);
    status = pic_acmd (p, ADON_WR, stkloc+DMA_TODO, func, 0); 
    status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0); 

    /* wait for response */
    for (timer=0; status==0 && timer<100; timer++) {	
      udelay(10000); 
      status = pic_acmd (p, ADON_RD, stkloc+DMA_STAT, 0, 0); 
      /* check for short ONESHOT finishing between polls */
      if (func==DMA_ONESHOT && timer>10 && status==0) 
        status = pic_acmd (p, ADON_RD, stkloc+DMA_HCYCLE, 0, 0);
      if (UBREAK || dma->dir>0) timer+=10;
    }
    if (mcfg2!=mcfg1) {
      if (dma->reg==0) pic_acmd (p, ADON_MOD, mside, mcfg2, 0); 
      if (mside==1||mside==3) pic_acmd (p, ADON_WR, DMA_GMCFG+1, mcfg2, 0); 
      if (mside==2||mside==3) pic_acmd (p, ADON_WR, DMA_GMCFG+2, mcfg2, 0); 
    }

    /* kick on NIO inputs */
    if (isqsfp) pic_enable_qsfp (p,n+1,1);

    vprint("DMASS Started %d p=%d c=%d s=%d mcfgs %08x %08x %08x  r=%d\n",
	n+1, port, chip, side, mcfg1, mcfg, mcfg2, running);

    /* handle timecode startup synchronization issues */
    dma->tcoff = 0;
    needtime = (dma->master<0 && (p->tflags&(FLG_PRESAMP|FLG_PREDELAY)));
    needtime |= (running && dma->master<=0 && (dma->type==IOPT_MODULE||dma->type==IOPT_CORE||dma->type==IOPT_MCORE));
    if (p->tcmode!=TCM_OFF && (needsync || needtime) ) {
      if (needsync && chip!=0 && dma->master>0 && (p->tflags&FLG_ITDEC)==0) dma = p->dma+(dma->master-1);
      i = 0x1080*(1000000.0/dma->rate); udelay(min(100000,i));	/* wait for startup */
      if (!p->isY) for (j=10; j>0; j--) {	/* wait extra to verify startup */
	k = pic_acmd (p, ADON_RDIOC,IOC_STAT|(side<<6),0,1);
	i = pic_acmd (p, ADON_RDIOC, sadr, 0, 4); 
	if (i!=lastsync) break;
	if (needsync && (k&IOC_STAT_SEN)!=0) break;
	if (!needsync && !running && (k&IOC_STAT_ENB)!=0) break;
	if (j>=1) { vprint("Waiting ... %d %d\n",i,lastsync); }
	udelay(10000);
      }
      if (p->isY) {
	ic = bitsPerSample(p,dma->mcfg);	/* bits per sample */
	k  = pic_acmd (p, ADON_RD, DMA_GSTAT, 0, 0); 
	dma->tcoff = k*(p->bpcl*8/ic);
        if ((p->tflags&FLG_PRESAMP)!=0 || (p->tflags&FLG_PREDELAY)!=0) {
	  if (dma->master>0) { /* get the PM input stream offset */
	    if (running) pic_stkupd (p, &p->dma[dma->master-1].tcoff, 1, -1);
	    if (running) pic_stkupd (p, &p->dma[dma->master-1].hcycle, 1, -1);
	    dma->tcoff = p->dma[dma->master-1].tcoff;
	    k = ((pmi==2)?PPC_TUNB_CTL:PPC_TUNA_CTL) + 0x10 + (side-1)*4;
            i = pic_acmd (p, ADON_RD, (k|0x8)>>2, 0,0);
	    j = p->dma[dma->master-1].hcycle; /* wrap around count */
	    p->pretcount[n] = ((double)i + (double)j*0x100000000LL)*(p->bpp*8/ic);
	  }
          else if (dma->master==-1) { /* store the PM input stream offset */
	    dma->hind = k;
	    pic_stkupd (p, &dma->hind, 1, 1);
  	  }
  	}
      }
      else if (j==0) dma->tcoff = 0;
      else dma->tcoff = fixcount(p,port,i, (needsync)? -1:0 );
      pic_stkupd (p, &dma->tcoff, 1, 1);
      v2print("SB dmac=%d sync=%d Count=%d last=%d tcoff=%08x j=%08x\n", n+1,needsync,i,lastsync,dma->tcoff,j);
      if (!needsync && running && p->gchip==0) {
	print("WARN: PIC2/PIC3 Series TimeCode circuits require Module to be started before Tuners\n");
	dma->tcoff = -999;
      }
    }
  }
  stat = 0;
  
  DONE:
  pic_lock(p, LOCK_FREE);
  return stat;
}

int_4 bitsPerSample (PICSTRUCT *p, int_4 mcfg)
{
  int_4 bps, bcfg = mcfg&IOM_BITS;
       if (bcfg==IOM_16B) bps = 16;
  else if (bcfg==IOM_08B) bps = 8;
  else if (bcfg==IOM_04B) bps = 4;
  else bps = 1;
  if (p->isY && p->mbits<0) bps *= 2; /* complex data */
  return bps;
}

int_4 pic_dmastat (PICSTRUCT *p, int_4 dmac, int_4 *index_, int_4 *cycle_)
{
  int_4 n=dmac-1,status,stkloc,index,cycle;
  DMASTRUCT *dma = p->dma+n;
  CHECK_MAGIC(p);

  if (p->type==ICENIC) {
    index = dma->hindex>>2;
    cycle = dma->hcycle;
    status = 1;
  }
  else {
    stkloc = dmastkloc(p,dmac);
    pic_lock (p, LOCK_ALLOC);
    /* DMA_STAT is the upper 16 bits of HSTAT */
    status = pic_acmd (p, ADON_HSTAT, stkloc, 0, 0) >> 16;
    /* read 1st to stop driver from issuing dummy write for consecutive address issue */
    RD(REG_IMB3, cycle); 
    RD(REG_IMB2, index); 
    index = (index-p->dma[n].haddr)*p->bphw;
    pic_lock (p, LOCK_FREE);
  }

  *index_ = index;
  *cycle_ = cycle;
  return status;
}

int_4 pic_dmaxfer (PICSTRUCT *p, int_4 dmac, void *buffer, int_4 bytes, int_4 flags)
{
  int_4 n=dmac-1,status; void *bufferp;
  DMASTRUCT *dma = p->dma+n;
  status = pic_dmaxptr (p, dmac, &bufferp, bytes, flags);
  if (p->dma[n].dir<0 && status>0) {
    memcpy (buffer, bufferp, status); 
    if (bytes>status) memcpy ((char*)buffer+status,(void *)p->dma_vaddr[n], bytes-status); 
    status=bytes;
  }
  if (p->dma[n].dir>0 && status>0) {
    memcpy (bufferp, buffer, status); 
    if (bytes>status) memcpy ((void *)p->dma_vaddr[n], (char*)buffer+status, bytes-status); 
    status=bytes;
  }
  return status;
}

int_4 pic_dmaxptr (PICSTRUCT *p, int_4 dmac, void **buffer, int_4 bytes, int_4 flags)
{
  int_4 n=dmac-1,cycle,upause,tpause=0,stat;
  int_8 status,lsize;
  DMASTRUCT *dma = p->dma+n;
  int_4 bpw = p->isY? 6 : 2;	/* bits to shift for words to bytes */
  int_8 index = p->dma_vindx[n];
  int_8 isize = ((int_8)dma->hsize)<<bpw;
  int_4 ixfer = (dma->dir>0)? (dma->hxfer)<<bpw : 0;
  int_4 bps = dma->bits/8; if (bps<0) bps = -2*bps; bps = max(1,bps); /* bytes per sample */
  if (dma->todo==DMA_ONDEMAND) pic_setkeyl (p, dmac, KEY_INBYTE, index);
  for (;;) {
    status = pic_dmafuncx (p, dmac, DMA_STATUS);
    if (status<0 || UBREAK) return -2;				/* error */ 
    if (dma->dir>0 && p->type==ICENIC && dma->stat==0) return -1;
    lsize=0;
    if (dma->dir>0) {
      if (status <= index) lsize = isize;			/* looped around - add one buffer */
    } else {
      if (status < index) lsize = isize;			/* looped around - add one buffer */
    }
    if (status+lsize >= index+bytes+ixfer) break;		/* next transfer ready */
    if (status<=0 && pic_dmafunc(p,dmac,DMA_POLL)==0) {		/* DMA no longer active */
      if (dma->dir>0 && (flags&FLG_NOWAIT)==0) pic_dmafunc (p, dmac, DMA_WAIT); /* wait for output FIFOs to flush */
      return -1;
    }
    if (flags&FLG_NOWAIT) return 0;				/* non-blocking mode */
    upause = (int)((1000000.0*dma->dec*bytes)/((double)dma->rate*bps)); /* roughly one transfer length */
    upause = min(50000,max(10000,upause));			/* min of 10ms for Unix, max of 1/20 sec */
    udelay(upause);						/* not ready yet - pause */
    tpause += upause;						/* total for timeout */
    if (p->timeout>0 && tpause>p->timeout*1000000) return 0;
  }
  *buffer = (void *)((char*)p->dma_vaddr[n]+index);
  v2print("DMAXPTR dmac=%d index=%llx bytes=%x isize=%llx\n",dmac,index,bytes,isize);
  index += bytes;
  if (index>=isize) { 
    cycle = p->dma_vcycx[n]+1; index -= isize;
    if (flags&FLG_NOWRAP) { bytes -= index; index=0; }		/* cant wrap at bottom of buffer */
    stat = pic_dmafunc (p, dmac, DMA_CYCLE);
    if (dma->dir<0 && stat!=cycle && dma->todo!=0) print("Fell behind %d buffers.\n",stat-cycle);
    p->dma_vcycx[n]=stat;
  }
  p->dma_vindx[n] = index;
  return bytes;
}

void pic_stkupd (PICSTRUCT *p, void *data, int_4 words, int_4 dir)
{
  int_4 i,offset;
  if ((char *)data < (char *)p->app) {
    offset = DMA_STK + (((char *)data - (char *)(p->dma)) >> 2);
    if (offset>=DMA_STK1) offset += (DMA_STK2-DMA_STK1);
  } else {
    offset = APP_STK + (((char *)data - (char *)(p->app)) >> 2);
  }
  if (dir>0) {
    for (i=0; i<words; i++)
      pic_acmd (p, ADON_WR, offset+i, ((int_4*)data)[i], 0);
  } else {
    for (i=0; i<words; i++)
      ((int_4*)data)[i] = pic_acmd (p, ADON_RD, offset+i, 0, 0);
  }
}

int_4 pic_fixup (PICSTRUCT *p)
{
  int_4 status;
  RD(REG_IMB4,status);
  if (status==0x88888888) {
    print("Performing ICE card #%d initial reset ...\n",p->devno);
    status = pic_reset (p, FLG_NOBRK|FLG_BOOT);
    return status;
  }
  return(0);
}

/* 
  SMS/SDN/DTL Timecode formats

 Time code is the time at the sample ONE BIT PRIOR to the Barker code [SDN],
 at the start of the first Barker bit [SMS], or at the midpoint (high-to-low
 edge of clock) of the first Barker bit [Datolite].


       Bit(s)          Data
       ------          -------------------------------
        < 1            Fill bits: SMS = 0, SDN = 1
        1-7            7 bit Barker code, 1110010 = 114
        8-14           BCD seconds, LSB first
       15-21           BCD minutes
       22-27           BCD hours
       28-37           BCD days
  For SDN only:
       38-49           BCD milliseconds
       50-60           Binary VFT (very fine time), MSB first, units = 5 nsec
  VFT Extended Precision/Range:
       61-63           Bits -1,-2,-3 of VFT; inverted polarity
          64           Spare
       65-67           Bits 13,12,11 of VFT; inverted polarity

    * NOTE - this scheme can only address 81us above a 1ms hack, min sample rate = 12345 Hz
    * NOTE - barker can easily reappear in timecode string, require 4 bits of fill for SDN/DTL
    * NOTE - with 4 bit fill, minimum supported rate is 71kHz
 

  IRIG-B Timecode format  B012

 Time code is once per second.  Format is pairs of BCD numbers

       Bit(s)          Data
       ------          -------------------------------
       02-08           BCD seconds, LSBit first, LSDigit first
       11-18           BCD minutes
       21-27           BCD hours  
       28-30           must be 001 - use to check for offset by 1 or 2
       31-38           BCD days
       41-44           BCD days*100


  SDDS Timecode format

 Time code is once per packet.  Format is binary numbers

       Bit(s)          Data Big Endian 
       ------          -------------------------------
       00-04           status bits        
       05-15                   
       16-31           # (250psec) hacks from 1ms ptr to SSC        
       32-96           # (250psec) time increments

 IOC tvalid bits: 7=val 6=cyc 5=hit 4=nclk 3=den 2=ply 1=enb 0=ena

*/

int_4 pic_tc (PICSTRUCT *p, int_4 dmac, real_8 *offset, real_8 *delta,
 			real_8 *soy, real_8 *fsoy, int_4 flags)
{
  char abits[128],*bit,fill,opps,tmp; 
  int_4 i,j,k,n,nn,ioff=0,status,nbits,retry=3,tvalid=0,stkloc,mstkloc;
  int_4 ccycle,cindex,hcycle,hindex,current,msgr[4],port,tcount=0,dir;
  int_4 dec,bits,mbits,nambig,mcindex,mccycle,mhcycle,nmux,ptype,d2cor;
  int_4 rate,verify,ovsr,spa,ospa,tcoff,tuner,cwsize,amask,feed,presamp,skip;
  real_8 tc,ftc,tcoffset,corr,tcc,ftcc,cursamp,mcursamp,spw,rtcoffset,ddec,rdec,ambig; 
  real_8 res_ratio; int_4 res_dec,res_ovsr;
  int_8 sddval,sddtmp,sddtps;
  DMASTRUCT *dma,*dmam;

  verify = (p->verbose==2)? 1:0;

  /* default returns */
  *soy = *fsoy = 0.0;
  if ((flags&FLG_TCINTERP)==0) *offset = 0.0;
  if (dmac<1 || dmac>DMA_CHNS || p->tc_mode[0]==0) return TC_INACTIVE;

  /* BC for code that just set p->tc_mode directly */
  if (*(int_4*)p->tc_mode != p->tc_modl) pic_parsetc(p,0,FLG_INFO);
  
  switch (p->tcmode) {
    case TCM_SDN: nbits = 67; corr = 1; break;
    case TCM_SMS: nbits = 37; corr = 0; break;
    case TCM_DTL: nbits = 67; corr = -0.5; break;
    case TCM_CPU: nbits =  0; corr = 0; break;
    case TCM_ZTC: nbits = -1; corr = 0; break;
    case TCM_STC: nbits = -1; corr = 0; break;
    case TCM_IRB: nbits = 48; corr = 0; break;
    case TCM_SDD: nbits = 96; corr = 0; break;
    case TCM_ICE: nbits = 0; corr = 0; break;
    case TCM_VRT: nbits = 0; corr = 0; break;
    case TCM_FILE: nbits = 0; corr = 0; break;
    default: return TC_BADPARAM;
  }

  n    = dmac-1; 
  dma  = p->dma+n;
  nn   = dma->master-1;
  dmam = p->dma+nn;
  dec  = dma->dec;
  bits = dma->bits;
  ptype= dma->type;
  rate = dma->rate;
  dir  = dma->dir;

  if (dir>0) return TC_INACTIVE;

  port = (dma->mcfg>>28)&0x3;
  if (port==0 && p->inp>0) port = (p->inp&0x1)? 1:2;
  if (port==0 || port==3) port = (dma->port&0x1)? 1:2;

  fill = (p->tcflags&FLG_TC_FILL)? 1 : 0;
  opps = (p->tcflags&FLG_TC_OPPS)? 1 : 0;
  tuner = (ptype==IOPT_TUNER) || (ptype==IOPT_TBANK);
  cwsize = (ptype==IOPT_MODULE && !p->isY && (p->flags&FLG_VHS)!=0)? 6 : 4;
  presamp = (ptype!=IOPT_MODULE && (p->tflags&FLG_PRESAMP)!=0)? 1 : 0;

  if (opps) {
    if (nbits==0) nbits=1;
    if (p->oppsoffset!=0) corr -= p->oppsoffset;
    else if (p->mtype[port-1]==IOMT_A2D) {
      if (!p->isY) corr -= 4;   /* four sample pipeline diff in A2Ds */
      if (p->irbreg&IRB_SWAP) corr -= 1; /* one sample buffer in A2D */
    }
    if (p->mtype[port-1]==IOMT_A2D && p->mrev[port-1]==14 && (p->tcflags&FLG_TC_OVFT)!=0) opps=2;
    if (p->mtype[port-1]==IOMT_A2D && p->mrev[port-1]==20) opps=3;
  }

  spa = (bits<0)? 2:1;
  if (dec==0) dec=1;

  if (tuner) {
    dec = dec*spa; 	/* complex output dec in time - not BW */   
    ovsr = p->ovsr[port-1];  if (ovsr<=0) ovsr=1;
    ddec = dec;
    if (p->tflags&FLG_POVSR) ddec /= ovsr; 
    rdec = ddec;
    if (findflag("PRER2C",p->config)>=0) rdec*=2;
  } else {
    ovsr = 1;
    rdec = ddec = dec;
  }
  if (ptype!=IOPT_MODULE && (p->tflags&FLG_RESAMP || p->tflags&FLG_PRESAMP)) rdec /= p->res_ratio; 
  nmux = max(1,dma->multi);

  if (presamp) {		/* embedded PRESAMP follows side */
    res_dec = p->res_dec;
    res_ovsr = p->res_ovsr;
    res_ratio = p->res_ratio;
  }
  else if ((p->presamp&FEED_PORT)!=0) {	/* feed based PRESAMP */
    port = p->presamp&FEED_PORT;
    res_dec = (int_4)(p->feedrdec+.5);
    res_ovsr = 1;
    res_ratio = res_dec/p->feedrdec;
    rdec *= p->feedrdec;
    if (p->presamp&FEED_RCFLIP) { rdec*=2; res_ratio/=2; }
    presamp = (p->presamp&FEED_TUNER)? 1 : 2;
  }

  /* TC based on slave dma channel */
  if (tuner && dma->master>0 && nbits>0 && (p->tflags&FLG_ITDEC)==0 && !p->isY) { 
    nn = n; n = dma->master-1; dma = p->dma+n; dmam = p->dma+nn;
    mstkloc = DMA_STK+nn*DMA_SEP;
  } else {
    nn = mstkloc = 0;
  }
  stkloc = dmastkloc(p,n+1);

  if (p->tcmode==TCM_ZTC || p->tcmode==TCM_STC) {
    tc=ftc=tcoffset=0.0; 
    goto GOTTC; 
  }

  if (p->tcmode==TCM_CPU && !opps) { 
    pic_lock (p, LOCK_ALLOC);
    status = pic_acmd (p, ADON_HSTAT, stkloc, 0, 0);
    current = (status&0xFFFF)+dma->caddr;
    ftc = pic_time();
    RD(REG_IMB2, hindex); hindex -= dma->haddr;
    RD(REG_IMB3, hcycle); 
    pic_lock (p, LOCK_FREE);
    if (hcycle==0 && current==dma->caddr) return TC_INACTIVE;
    tc = floor(ftc); ftc = ftc-tc;
    hcycle = hcycle*nmux;
    i = (hindex*nmux/cwsize)*4;
    ccycle = i / dma->csize;
    cindex = i - ccycle*dma->csize + dma->caddr;
    goto GETTC;
  }

  if (ptype==IOPT_STREAM && p->dsgtype!=DSG_PICQ) {
    tc = p->tc_wsec; 
    ftc = p->tc_fsec;
    tcoffset = p->tc_soff;
    if (tcoffset<0) return (int_4)tcoffset;	/* error status if negative */
    goto GOTTC;
  }

  if (dma->tcoff== -999) return TC_BADSTART;

  RETRY:
#if DEBUG
  if (retry!=3) verify=1; 
#endif
  pic_lock (p, LOCK_ALLOC);
  if (p->isY) mstkloc = p->tcmode; /* PPC needs this info */
  tvalid = pic_acmd (p, ADON_TC, stkloc, port, mstkloc);
  if (tvalid==0) {  		/* no new timecode bits in buffer */
    pic_lock (p, LOCK_FREE);
    if (--retry>0 && p->tcmode!=TCM_CPU && p->tcmode!=TCM_IRB) { udelay(10000); goto RETRY; }
    status = pic_acmd (p, ADON_HSTAT, stkloc, 0, 0);
    status = (status&0xFFFF)? TC_NOBITS : TC_INACTIVE;
    tprint("TC-%d port=%d retried out status=%d\n",dmac,port,status);
    return status;
  }
  if (p->tcmode==TCM_CPU) ftc = pic_time();
  cindex  = pic_acmd (p, ADON_RD, DMA_TCDAT+0, 0, 0);
  ccycle  = pic_acmd (p, ADON_RD, DMA_TCDAT+1, 0, 0);
  hcycle  = pic_acmd (p, ADON_RD, DMA_TCDAT+2, 0, 0);
  msgr[0] = pic_acmd (p, ADON_RD, DMA_TCDAT+3, 0, 0);
  msgr[1] = pic_acmd (p, ADON_RD, DMA_TCDAT+4, 0, 0);
  msgr[2] = pic_acmd (p, ADON_RD, DMA_TCDAT+5, 0, 0);
  tcount  = pic_acmd (p, ADON_RD, DMA_TCDAT+6, 0, 0);
  current = pic_acmd (p, ADON_RD, DMA_TCDAT+7, 0, 0);
  if (!p->isY) { tmp=tvalid; tvalid=current; current=tmp; }
  tprint("TC-%d current=%08x tcount=%08x tvalid=%08x retry=%d port=%d mstk=%x\n",
		dmac,current,tcount,tvalid,retry,port,mstkloc);
  if (nn != 0) {
   mcindex  = pic_acmd (p, ADON_RD, DMA_TCDAT+8, 0, 0);
   mccycle  = pic_acmd (p, ADON_RD, DMA_TCDAT+9, 0, 0);
   mhcycle  = pic_acmd (p, ADON_RD, DMA_TCDAT+10, 0, 0);
  }
  pic_lock (p, LOCK_FREE);

  /* check for false startup OPPS in 360 release */
  if (p->isY && opps && tcount==0x77) return TC_NOBITS;

  /* adjust for different accounting method on newer cards */
  if (p->isY) { hindex = current-dma->haddr; current = cindex; }

  /* check for port not active yet - but TC exist from previous run */
  if ((tvalid&IOC_STAT_ENB)==0 && dma->todo != 0) {
    udelay(10000); 	/* ten millisecond */
    if (--retry > 0) goto RETRY; 
    return TC_INACTIVE;
  }

  /* check for SDN embedded in SDDS packet boundary issues */
  if (p->isY && p->tcmode==TCM_SDN && p->socc[port-1]=='H' && (tcount&0x1FF)==0) {
    udelay(10000); 	/* ten millisecond */
    if (--retry > 0) goto RETRY; 
    return TC_BADSTART;
  }

  tprint("TC-%d CS=%08x CI=%08x CC=%08x HC=%08x\n",dmac,current,cindex,ccycle,hcycle);
  if (nn!=0) tprint("TC-%d MCI=%08x MCC=%08x MHC=%08x \n",dmac,mcindex,mccycle,mhcycle);
  tprint("TC-%d Data = %08x %08x %08x %08x %08x\n",dmac,msgr[0],msgr[1],msgr[2],tcount,tvalid);

  /* handle SDDS timecode fields */
  if (p->tcmode==TCM_SDD) {
    msgr[0] = swapit(msgr[0]);
    msgr[1] = swapit(msgr[1]);
    msgr[2] = swapit(msgr[2]);
    sddval = ((int_8)msgr[1])<<32 | (int_u4)msgr[2];
    sddtps = 1000000000; sddtps *= 4; /* 250ps tics per second */
    sddtmp = (sddval/sddtps);
    tc = (double)sddtmp;
    ftc = (250.0e-12)*(double)(sddval-sddtmp*sddtps);
    if (p->leapsecsoy>0 && tc>p->leapsecsoy) tc -= 1; /* apply mid year leap second */
    goto GETTC;
  }
  if (p->tcmode==TCM_CPU) goto GETTC;

  /* get packed bits into char bit array */
  i = 0;
  for (j=0; j<3; j++) { 	
    if (j==0 && p->isY && (tvalid&IOC_STAT_TDC)!=0)  /* repack 1st double clutch word */
      for (k=0; k<32; k+=2) abits[i++] = (msgr[j]>>k) & 0x1;
    else
      for (k=0; k<32; k++)  abits[i++] = (msgr[j]>>k) & 0x1;
  }
  if (verify) {
    print("TC-%d Bits=[",dmac); for (j=0;j<80;j++) print("%d",abits[j]); print("]\n");
  }

  /* check for the barker code */
  for (i=-1; i<24; i++) { 
    bit = abits+i;
    if (p->tcmode==TCM_IRB && p->isY) goto GOTBARKER;
    if (bit[1]==1 && bit[2]==1 && bit[3]==1 && bit[4]==0 &&
        bit[5]==0 && bit[6]==1 && bit[7]==0) goto GOTBARKER; 
  }
  if (--retry > 0) goto RETRY; 
  if (verify) {
    print("TC-%d RBits=[",dmac); for (j=0;j<80;j++) print("%d",abits[j]); print("]\n");
  }
  return TC_NOBARKER;

  GOTBARKER:
  ioff = i+1;

  /* test validity of optional precision */
  for (i=nbits; bit[i]==fill && i>49; i--);
  if (i<=60 && nbits>60) nbits = 60;
  if (i<=49 && nbits>49) nbits = 49;
  tc = ftc = 0.0;

  tprint("TC-%d Barker at bit=%d, nbits=%d\n",dmac,ioff,nbits); 

  /* handle IRIG-B timecode fields */
  if (p->tcmode==TCM_IRB) {
    if (!p->isY) i=8;
    else if (!opps) i=32;
    else for (i=34; i>32 && (bit[i+27]!=0 || bit[i+28]!=0 || bit[i+29]!=1); i--);
    bit += i;
    status  = getbcd (bit+1, 4, 9, &i);    tc += i;		/* seconds */
    status += getbcd (bit+6, 3, 6, &i);    tc += i*10;		/* seconds*10 - Allow for leap second == 60 */
    status += getbcd (bit+10, 4, 9, &i);   tc += i*60;		/* minutes */
    status += getbcd (bit+15, 3, 5, &i);   tc += i*600;		/* minutes*10 */
    status += getbcd (bit+20, 4, 9, &i);   tc += i*3600;	/* hours */
    status += getbcd (bit+25, 2, 2, &i);   tc += i*36000;	/* hours*10 */
    status += getbcd (bit+30, 4, 9, &i);   tc += (i-1)*86400;	/* day */
    status += getbcd (bit+35, 4, 9, &i);   tc += i*864000;	/* day*10 */
    status += getbcd (bit+40, 2, 3, &i);   tc += i*8640000;	/* day*100 */
    tc += 1;	/* decoded value is attached to the next seconds marker */
    if (p->irbreg&IRB_OPPS) corr += 6; /* isX correction ? */
    else if (!opps) ftc = 0.011;  /* double sync is 11mS past the nominal 1PPS */
    goto FINISH;
  }

  /* handle the standard SMS timecode fields */
  status  = getbcd (bit+8, 7, 59, &i);    tc += i;		/* seconds */
  status += getbcd (bit+15, 7, 59, &i);   tc += i*60;		/* minutes */
  status += getbcd (bit+22, 6, 23, &i);   tc += i*3600;		/* hours */
  status += getbcd (bit+28, 10, 366, &i); tc += (i-1)*86400;	/* days */

  if (nbits<49) goto FINISH;	/* no SDN */
  status += getbcd (bit+38, 12, 999, &i);  ftc += i*1.0e-3;	/* millisec */ 

  if (nbits<60) goto FINISH;	/* no VFT */
  for (i=0,k=50; k<=60; k++) i = (i<<1) + bit[k];   ftc += i*5.0e-9;

  if (nbits<67 || (p->tcflags&FLG_TC_VFTX)==0) goto FINISH;	/* no extended VFT */
  for (i=0,k=61; k<=63; k++) i = (i<<1) + (1-bit[k]);   ftc += i*625.0e-12;
  for (i=0,k=65; k<=67; k++) i = (i<<1) + (1-bit[k]);   ftc += i*10240.0e-9;

  FINISH:
#if DEBUG
  print("TCBits=["); for (j=0;j<80;j++) print("%d",abits[j]); 
  print("] %d %d %x\n",status,retry,tvalid);
#endif
  if (status != 0) {
    vprint("Tc-%d Bata = %08x %08x %08x %08x %08x\n",dmac,msgr[0],msgr[1],msgr[2],tcount,tvalid);
  }
  if (status == 0); 		 	/* OK decode */
  else if (--retry > 0) goto RETRY; 	/* decoding error - retry */
  else return TC_BCDERR; 		/* decoding error */

  GETTC:
  if (!p->isY) current |= IMS;	/* CARD internal mem / DMA offset */
  if (nmux>1) {  /* round multi track buffer offsets for easy math */
    current = (current-dma->caddr)/nmux*nmux + dma->caddr;
    mcindex = (mcindex-dma->caddr)/nmux*nmux + dma->caddr;
  }

  /* calculate current acquisition sample number */
  spw = (32.0 / abs(bits)) / spa; 
  if (p->isY) {
    skip = (dma->frame!=0)? (dma->frame>>16) : 1;
    cursamp =  (double)hcycle*(double)dma->hsize + hindex;
    cursamp *= skip*(p->bphw/4); /* convert host words to card words */
  } else {
    cursamp = (double)ccycle*(double)dma->csize + (double)(current-dma->caddr);
    if (current<cindex) cursamp += dma->csize;	/* next buffer */
    if (cwsize!=4) cursamp = cursamp*cwsize/4;
    cursamp += (double)hcycle*(double)dma->hsize;
  }
  if (nmux>1) cursamp = floor(cursamp/nmux);
  cursamp *= spw;		/* multiply by samples per word */
  if (ptype != IOPT_MODULE) cursamp *= rdec;  /* decimation in tuner|core */
  if (ptype == IOPT_TBANK) {	/* multiple channels in output */
    if ((dma->port%10)==3) cursamp /= p->nchan; 
    else cursamp /= (p->nchan/2); 
  }

  /* resync tcount - bytes 0,1,2,3 sampled 1 wordclock apart */
  amask = 0x0FFFFFFF;
  ambig = 0x10000000;
  if (p->type==ICEPIC2||p->type==ICEMBT2) {
    tcoff = 0;
    tcount = fixcounto (tcount);
    ambig = 0x80000; 
  } else {
    if (nn==0) tcoff = dma->tcoff;
    else tcoff = pic_acmd (p, ADON_RD, stkloc+DMA_TCOFF, 0, 0);
    tcount = fixcount (p,port,tcount,(tvalid&IOC_STAT_TDC)?2:1);
    if (p->isY) { 
      if ( p->tcmode==TCM_SDD || p->tcmode==TCM_SDN);
      else if (p->socc[port-1]=='H') { 
	if (tuner) mbits = bitsPerSample(p,dma->mcfg); 
	else mbits = (bits<0)? -2*bits : bits;
	i = (mbits==8)? 2 : (mbits==16)? 1 : 0;	/* samples per count shift */
	j = (p->qdrx&port)? 2:1; if (j==2) tcount <<= 1; /* dual packet paths */
	if (isIOMx(p,port) || opps==3) tcount -= (((p->up12?341:256)*j)>>2);	/* from previous packet */
	tcount += (opps?j:2);
	tcount <<= i; 
	if (opps) { k = (msgr[0]>>12)&0xFF; if (p->up12) k=k*341/256; }  /* packet byte count offset */
	if (opps) tcount += ( k >> (2-i) ) << (j-1); /* packet byte count offset to word offset */
      } else {	/* S or D */
	if (opps) tcount += 8;	/* offset in FPGA count circuit */
        if ((tuner?p->mbits:bits)<0) {	/* complex module input ? */
          if (dma->mcfg&IOM_DUAL);
        } else {
          if (dma->mcfg&IOM_DUAL) tcount <<= 1; /* DDR mode real input */
        }
      }
      if (dma->mcfg&IOM_MUX) tcount <<= 1;
      if (presamp) cursamp += (p->pretcount[n]/res_ratio); /* offset between presamp output and tuner startup */
      if (p->tflags&FLG_PREDELAY) cursamp += p->pretcount[n];
      ambig = 0x100000000LL;
      amask = 0xFFFFFFFF;
    }
    else if (p->isX) {
      if (opps) tcount += ((p->tcflags&FLG_TC_SWAP)? 8:7);
    }
  }
  /* now calculate the sample number (0-based) of the barker code */
  /* referenced to the first input sample of the resource */
  tcount = (tcount-tcoff) & amask;
  if (p->isY && p->tcmode==TCM_SDD && (tuner?p->mbits:bits)<0) { tcount>>=1;  ambig/=2; } /* complex SDDS input */
  nambig = (int_4)(((cursamp-tcount)/ambig)+1.5) - 1;
  tcoffset = ambig*nambig + tcount + ioff - corr;

  /* fixup for TC=CPU */
  if (nbits<=0) tcoffset = cursamp;

  GOTTC:
  rtcoffset = 0; /* resource in->out delay in units of output sample rate */
  if (tuner) {
    if (ptype==IOPT_TBANK && p->fttm==0) {
      if (p->res_ratio!=0.0)  rtcoffset = 0; /* need polyphase filter and 64 tap resampler */
    }
    else if (p->type == ICEPIC2) {
      /* includes integration in HDC, 121 tap FIR, and 3rd sample startup */
      /* InSamp = (OutSamp*dec + 32.75*dec - 1.5) if real */
      if (spa==1) rtcoffset = 1.5/ddec - 32.75;
      /* InSamp = (OutSamp*dec + 15.875*dec - 1.5) if complex */
      if (spa==2) rtcoffset = 1.5/ddec - 15.875;
    } 
    else if (p->type == ICEPIC3) { 
      if (p->dec2>4) d2cor=0; 
      else if (p->dec2==3) d2cor=3; 
      else if (p->dec2==1) d2cor=2; 
      else d2cor=4;
      rtcoffset = (-256 + 3 + d2cor + (p->dec2*p->dec5*(2+p->ntaps*.5)) - p->dec2*255.5 )/(ddec*ovsr);
      /* add output pipeline delay 1 real or 1/2 complex sample */
      rtcoffset += (1.0/spa); 
    } 
    else if (p->gchip == GC4014) {
      /* GC4014 output includes startup transients - see DataSheet Sec 3.10 */
      /* IOC sync generation starts tuner on sync input sample 0 , dec = 4N  */
      /* InSamp = N-2 + (OutSamp-21)*4N (+5)	if complex	*/
      /* formulas barely helped - this is empirical but exact */
      if (p->type==ICESLIC3) i=4; else i=2;
      rtcoffset = ((double)i)/ovsr/ddec + 41.5/spa;
      if (spa==1) rtcoffset += 1;
    } 
    else if (p->gchip == GC4016) {
      /* GC4016 output includes startup transients - see DataSheet Sec 3.10 */
      /* IOC sync generation starts tuner on sync input sample 0 , dec = 4N  */
      /* InSamp = N-2 + (OutSamp-21)*4N (+5)	if complex	 */
      /* delay = 2.5N + 0.5N*CTAP(21) + N*PTAP(63) + 2N*NMULT(7) */
      /* formulas barely helped - this is empirical but exact */

      if (ptype==IOPT_TBANK && p->fttm>1) {  /* apply 1st stage TBANK parameters */
        ospa = spa; spa = 1;
        ddec = ((double)p->d1/ovsr);
      }

      rtcoffset = 1.5/ovsr/ddec;
      if (p->cpc==1) i=21; else i=42; 
      if (ddec*ovsr<=i*spa) rtcoffset -= (1.0/(ddec*ovsr));
      if (spa==1) rtcoffset += 1;
      /* handle recombination delays when ganging channels */
      if (p->cpc==4) rtcoffset += 42.75/spa;	
      if (p->cpc==2) rtcoffset += 44.75/spa;	
      if (p->cpc==1) rtcoffset += 91.50/spa;
      /* account for half of (resampler filter - one 1/32 tap) */
      if (p->tflags&FLG_RESAMP) rtcoffset = (rtcoffset + 0.5*p->nres - 1.0/64.0)*p->res_ratio; 

      if (ptype==IOPT_TBANK && p->fttm>1) {  /* apply 2nd stage TBANK parameters */
        rtcoffset -= 64*2;
        spa = (p->fttm==2)? ospa:1;
        ddec = p->d2*spa;
        rtcoffset /= ddec;
        rtcoffset += 1.5/ddec;
        if (ddec<=42*spa) rtcoffset -= (1.0/ddec);
        if (spa==1) rtcoffset += 1;
        rtcoffset += 42.75/spa;
        ddec = ((double)p->d1/ovsr)*p->d2*spa;
      }

      if (ptype==IOPT_TBANK && p->fttm>2) {  /* apply 3rd stage TBANK parameters */
        rtcoffset -= 64*2;
        spa = ospa;
        ddec = p->d3*spa;
        rtcoffset /= ddec;
        rtcoffset += 1.5/ddec;
        if (ddec<=42*spa) rtcoffset -= (1.0/ddec);
        if (spa==1) rtcoffset += 1;
        rtcoffset += 42.75/spa;
        ddec = ((double)p->d1/ovsr)*p->d2*p->d3*spa;
      }
      /* account for 64 tap/word (256by) transient masking in the DTDM modules */
      if (p->type != ICEPIC4) rtcoffset -= 64*spw;
    }
    else if (p->gchip == GCFPGA) {
      if (ptype==IOPT_TBANK && p->fttm>1) {		/* apply 1st stage TBANK parameters */
        ospa = spa; spa = 1;
        ddec = ((double)p->d1/ovsr);
      }
      if (p->nps>0) {
        rtcoffset = (double)(p->ctaps-1)/spa/ddec;	/* decimating CFIR */
	rtcoffset -= ((ddec==4)? 38.5 : 6.5)/ddec;	/* unknown corr factor */
        rtcoffset += ((3.5/ddec) + (3.5/spa))*p->decb;	/* CIC */
      } else {
        /* Emulates a GC4016 but has no apparent mode changes */
        rtcoffset = (3.5/ovsr/ddec) + (4.25/spa);	/* CIC */
        if (ddec==spa && ovsr==1) rtcoffset = rtcoffset*2-(1.0/spa);	/* bypass CIC and no CFIR dec */
        rtcoffset += (double)(p->ctaps-1)/2/2/spa;	/* dec=2 CFIR */
      }
      if (p->tflags&FLG_RESAMP) {
	/* JGS */
	i = findintflagdef("ICORR",p->config,0);
	corr = i - 1;
	corr = (p->nres==8)? corr/256 : corr/2048;
        rtcoffset += ((double)p->nres+corr)/2/spa;	/* RFIR */
        rtcoffset *= p->res_ratio;			/* all blocks to here are resampled */
        rtcoffset += (double)(2)/2/spa;			/* RFIR */
      }
      if (p->nps>0) rtcoffset += (double)(p->ntaps-1)/p->dec2/spa;	/* decimating PFIR */
      else rtcoffset += (double)(p->ntaps-1)/2/spa;	/* PFIR */
      if (spa==1) rtcoffset += 1;

      if (ptype==IOPT_TBANK && p->fttm>1) { 		/* apply 2nd stage TBANK parameters */
        spa = (p->fttm==2)? ospa:1;
        ddec = p->d2*spa;
        rtcoffset /= ddec;
        rtcoffset += (3.5/ddec) + (4.25/spa);	  /* CIC */
        rtcoffset += (double)(p->ctaps-1)/2/2/spa;/* CFIR */
        rtcoffset += (double)(p->ntaps-1)/2/spa;  /* PFIR */
        rtcoffset -= (128/spa);                   /* mask */
        if (spa==1) rtcoffset += 1;
        ddec = ((double)p->d1/ovsr)*p->d2*spa;
      }

    }
  }
  else if (ptype == IOPT_CORE) {
    rtcoffset = 0;
  }
  if (p->atccalib != 0.0) rtcoffset += p->atccalib;

  if (presamp==1) {
    i = res_dec * res_ovsr;
    corr = 3.5/i + 4.25;
    if (i==1) corr = corr*2-1.0;	/* bypass CIC */
    j = (i==1)? 15:31;
    corr += (double)(j-1)/2/2;
    corr += (10)/2;
    corr *= res_ratio*res_dec;
    corr += (2)/2;
    j = (i==1)? 31:63;
    corr += (double)(j-1)/2 + 1.0;
    corr -= p->pretcount[n];
    rtcoffset += corr/ddec;
  }

  /* add the post resource rate correction */
  if (rdec==1.0) rtcoffset += tcoffset;
  else rtcoffset += (tcoffset/rdec);

  /* adjust sample pointer based on master/slave startup offset */
  if (nn != 0 && nbits>0) {
    mcursamp =  (double)mhcycle*(double)dmam->hsize + 
		(double)mccycle*(double)(dmam->csize/nmux) + 
		(double)( (current-dma->caddr)/nmux );
    if (current<mcindex) mcursamp += (dma->csize/nmux); /* next buffer */
    mcursamp *= rdec;  /* decimation in tuner */
    mcursamp *= spw;	/* multiply by samples per word */
    rtcoffset -= (cursamp-mcursamp)/rdec;	 /* in terms of slave chan */
    tprint("TC-%d Cursamp=%f rtcoff=%f Diff=%f mhcyc=%d mccyc=%d \n",
	dmac,cursamp,rtcoffset,cursamp-mcursamp,mhcycle,mccycle);
    cursamp = mcursamp;
  }

  /* handle CPU 1PPS timecode fields */
  if (p->tcmode==TCM_CPU && opps) {
    ftcc = (*delta)*(cursamp-tcoffset); 
    if (ptype!=IOPT_MODULE) ftcc /= rdec;
    ftc -= ftcc;
    tc  = floor(ftc+.5);
    p->ntpoffset = ftc-tc;
    tprint("NTP Offset = %f\n",p->ntpoffset);
    if (fabs(p->ntpoffset)>0.25 && findintflag("OKNTPERR",p->config)<=0) return TC_NTPERR;
    ftc = 0;
  }
  if (opps==2) { /* has VFT register */
    i=pic_jpmemrd (p,port,A2D_FTR,2);
    if (i&0x8000) rtcoffset -= (1/(spa*rdec));
    ftc = (i&0xFFF)*1.25e-9 - 7.5e-9;
  }

  /* interpolate if called for */
  if ((flags&FLG_TCINTERP)!=0) {
    ftcc = (*offset - rtcoffset)*(*delta);
  } else {
    ftcc = 0.0; *offset = rtcoffset;
  }

  /* apply correction terms in sec&frac */
  tcc = (int)ftcc; ftcc -= tcc;
  *soy = tc + tcc; *fsoy = ftc + ftcc;
  i = (int_4)(*fsoy); 
  if (*fsoy<0) i--;
  *soy += i; *fsoy -= i; 

  if (p->tcmode==TCM_STC) {
    *soy += p->zeroTime;
  }

  tprint("TC-%d Tc  = %.3f + %.9f at index %d, sample %.0f|%.0f\n", dmac,tc,ftc,ioff,tcoffset,*offset);
  tprint("TC-%d SoY = %.1f + %.9f = %.3f\n", dmac, *soy, *fsoy, (*soy + *fsoy) );

  if (p->nio!=0 && findintflag("NOTCUP",p->config)<=0) {
    i = abs(bits);
    j = (bits<0)? 'C':'S';
    k = (i==8)? 'B' : (i==32)? 'L' : 'I';
    i = 0x05000000 | (p->tcmode<<16) | (j<<8) | k;
    pic_tcup_qsfp (p,port,i,*offset,*delta,*soy,*fsoy);
  }

  /* check no clock bit and flag for a restart */
  if (rate>200000 && (tvalid&IOC_STAT_TNCLK)!=0) {
    if (findflag("OKNC",p->config) >= 0) return TC_OKNC;
    return TC_NOCLOCK;  
  }

  return TC_OK;
}

int_4 pic_buffer (PICSTRUCT *p, DMAMAP *map, int_4 offset,
	int_1 *bbuf, int_4 length, int_4 mode) 
{
  int_4 i,tmp,mask=0,*bufi,*bufo;
  if (mode&0x1) mask = 0x00010001;
  bbuf = (int_1*) map->vaddr;
  if (mode&0x2) {
    bufi = (int_4*)(bbuf+length);
    bufo = (int_4*)(bbuf+(length*2));
    for (i=length/4; i>0; i--) {
      tmp = *(--bufi);
      *(--bufo) = ((tmp<<0)&0xFF000000) | ((tmp>>8)&0xFF00) | mask;
      *(--bufo) = ((tmp<<16)&0xFF000000) | ((tmp<<8)&0xFF00) | mask;
    }
  } else {
    bufi = (int_4*)(bbuf);
    for (i=0; i<length/2; i++) bufi[i] = bufi[i]|mask;
  }
  return 0;
}

int_4 pic_tc_insert (PICSTRUCT *p, int_2 *buffer, int_4 length, 
	real_8 delta, real_8 soy, real_8 fsoy, int_4 flags)
{
  int_4 i, j, off, inc, tcbits;
  real_8 s, fs, fm; char bit[68];

  inc = (0.001/delta)+.999;
  s = soy; fs = fsoy;
  fm = fs-.001*floor(fs*1000);
  if (fm>0) {
    off = (.001-fm)/delta+1;
    fs += off*delta;
  } else off=0;

  for (i=off; i<length; i+=inc) {
    for (;fs>=1; s+=1,fs-=1); 
    fm = fs-.001*floor(fs*1000);
    if (fm>delta) { i--; fs-=delta; }
    tcbits = pic_tc_array (p,bit,s,fs,0);
    if (i+tcbits<length) for (j=0; j<tcbits; j++) {
      buffer[i+j] = (buffer[i+j]&0xFFFE)|bit[j];
    }
    fs += inc*delta;
  }
  return 0;
}

int_4 pic_tc_array (PICSTRUCT *p, char *bit, real_8 soy, real_8 fsoy, int_4 flags)
{
  int_4 sec, i, k, verify;

  /* set barker code */
  bit[0] = 1; /* fill so bit[1] is the start of the Barker code */
  bit[1]=1; bit[2]=1; bit[3]=1; bit[4]=0; bit[5]=0; bit[6]=1; bit[7]=0;

  verify = (p->verbose==2)? 1:0;

  sec = (int)(soy);
  i = sec/86400; sec-=(86400*i); putbcd (bit+28, 10, i);	/* days */
  i = sec/3600; sec-=(3600*i); putbcd (bit+22, 6, i);		/* hours */
  i = sec/60; sec-=(60*i); putbcd (bit+15, 7, i);		/* min */
  i = sec; putbcd (bit+8, 7, i);				/* sec */

  i = (int)(fsoy*1000); putbcd (bit+38, 12, i); 		/* msec */
  i = (int)((fsoy*1000-i) * 1600000 + .5);		/* # 625 psec hacks */
  for (k=63; k>=61; k--,i>>=1) bit[k] = 1-(i&0x1);	/* ext prec VFT */
  for (k=60; k>=50; k--,i>>=1) bit[k] = (i&0x1);	/* VFT */
  for (k=67; k>=65; k--,i>>=1) bit[k] = 1-(i&0x1);	/* ext prec VFT */
  bit[64]=1;

  if (verify) {
    print("putBCD of s=%f f=%f\n", soy,fsoy);
    print("TCBits=["); for (i=0;i<68;i++) print("%d",bit[i]); print("]\n");
  }
  return 68;
}

/* old timecode interface */
int_4 pic_timecode_dep=0;
int_4 pic_timecode (PICSTRUCT *p, int_4 dmac, char *mode, real_8 offset, 
 real_8 delta, int_4 dec, int_4 bits, int_2 *buffer, real_8 *soy, real_8 *fsoy)
{
  pic_parsetc(p,mode,0);
  p->dma[dmac-1].dec = dec;
  p->dma[dmac-1].bits = bits;
  if (pic_timecode_dep++ == 0)
  print("PIC_TIMECODE has been deprecated - use PIC_TC()\n");
  return pic_tc (p, dmac, &offset, &delta, soy, fsoy, FLG_TCINTERP);
}

int_4 getbcd (char *bit, int_4 nbits, int_4 maxval, int_4 *value)
{
  int_4 i,j,bcd=0,total=0,mult=1;

  *value = 0;
  for (i=j=0; i<nbits; i++,j++) {
    bcd += (bit[i] << j);
    if ( j==3 || i==nbits-1 ) { 
      if (bcd>9) return 1;	/* bad digit */
      total += bcd*mult; 	/* new total */
      mult *= 10; bcd = 0; j = -1; 
    }
  }
  /* vprint ("BCD total=%d \n",total); */
  if (total > maxval) return 1;
  *value = total;
  return 0;
}

int_4 putbcd (char *bit, int_4 nbits, int_4 value)
{
  int_4 i,j,k,value2;

  for (k=0; k<nbits;) {
    value2 = value/10;
    i = value - value2*10;
    for (j=0; j<4 && k<nbits; j++,k++,i>>=1) bit[k] = i&0x1;
    value = value2;
  }
  return 0;
}

int_4 calc_port_clock_rate (PICSTRUCT *p, int_4 port) {
  int_4 cycles,count,acycles,count1,count2,tcport,rate;
  if (port&1) tcport=IOC_A; else tcport=IOC_B;
  cycles = 256*1024;
  pic_lock (p,LOCK_ALLOC);
  acycles = pic_acmd(p,ADON_TCPNG,tcport,0,cycles);  
  RD(REG_IMB2,count2);
  RD(REG_IMB3,count1);
  pic_lock (p,LOCK_FREE);
  vprint("Got %d cy=%x acy=%x c1=%08x c2=%08x \n",port,cycles,acycles,count1,count2);
  acycles = 0x10000000-acycles;
  count1 = fixcount (p,port,count1,0);
  count2 = fixcount (p,port,count2,0);
  count = count2-count1;
  if (count<0) count += 0x10000000;
  vprint("Got %d cy=%x acy=%x c=%x c1=%08x c2=%08x \n",port,cycles,acycles,count,count1,count2);
  rate = (int)( (double)p->prc_clk*count/acycles );
  return rate;
}

#define MAXNAMES 80

static char regnames[MAXNAMES][8]={
  "FIFO","HIR8","MCSR","PLAT","PBAR","PVID","MDAR","MDTC",
  "OMB1","OMB2","OMB3","OMB4","IMB1","IMB2","IMB3","IMB4",
  "AFIFO","AMDAU","AMCSR","APLAT","APBAR","AIOC","AMDAR","AMDTC",
  "AIMB1","AIMB2","AIMB3","AIMB4","AOMB1","AOMB2","AOMB3","AOMB4",
  "STC0","SRC0","STX0","SRX0","STD0","NULL","SRD0","LCTL0",
  "STC1","SRC1","STX1","SRX1","STD1","NULL","SRD1","LCTL1",
  "SYSCON","VIRPT","WAIT","SYSTAT","LCTL","LCOM","LAR","LCOMX",
  "MSGR0","MSGR1","MSGR2","MSGR3","MSGR4","MSGR5","MSGR6","MSGR7",
  "MODE1","MODE2","ASTAT","STKY","IRPTL","IMASK","TPERIOD","TCOUNT",
  "IOMA","IOMB","IOCDIV","IOCGMC","NULL","NULL","NULL","NULL"
};
static int_4 regoffsets[MAXNAMES]={
  0x40, 0xA0, 0x48, 0x0C, 0x10, 0x00, 0x58, 0x5C,
  0x60, 0x64, 0x68, 0x6C, 0x70, 0x74, 0x78, 0x7C,
  0x00, 0x44, 0x48, 0x0C, 0x10, 0x54, 0x58, 0x5C,
  0x60, 0x64, 0x68, 0x6C, 0x70, 0x74, 0x78, 0x7C,
  0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xCC,
  0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xCD,
  0x00, 0x01, 0x02, 0x03, 0xC6, 0xC7, 0xC8, 0xCE,
  0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
  0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
  0x40, 0x80, 0x20, 0x30, 0x00, 0x00, 0x00, 0x00 
};

int_4 pic_name2offset (PICSTRUCT *p, char *name, int_4 flags)
{
  int_4 i,offset=-1,inc=0; char fname[80];

  for (i=0; name[i]!=0 && name[i]>' '; i++) {
    fname[i] = name[i]; if (fname[i]>='a' && fname[i]<='z') fname[i] -= 32;
    if (fname[i]=='+'||fname[i]=='-') { inc = strtol(name+i,NULL,10); break; }
  } fname[i]=0;
  if (fname[0]>='0' && fname[0]<='9') {
    offset = strtol(fname,NULL,0);
    offset |= 0x20000000;
    return offset+inc;
  }
  for (i=0; i<MAXNAMES; i++) {
    if (strcmp(fname,regnames[i])==0) {
      offset = regoffsets[i];
      if (i==16) offset = (offset/4)+EMS_BU;
      if (i>16 && i<32) offset = (offset/4)+(p->isY?0x02000000>>2:EMS_B1);
      if (i>=16) offset |= 0x20000000;
      if (i>=64 && i<72) offset |= 0x10000000;
      if (i>=72) offset |= 0x40000000;
      return offset+inc;
    }
  }
  return -1;
}

#define MAXKEYS 195

static char keynames[MAXKEYS][10]={
  "RATE","FREQ","DEC","GAIN","OVSR","FRAME","ALG","FEED",
  "TYPE","CTYPE","PTYPE","PINDEX","MTYPE","MTYPE1","MTYPE2","MCHNS",
  "CPC","CHNS","PCICLK","PCIBUS","PCIBW","PCIREV","STATUS","VERSION",
  "PRCCLK","IOCCLK","ICLK","PCLK","CCLK","MCLK","CLKI","MSBI",
  "IOC","IOCIOM","IOCALG","IOCRAM","MOD","APP","MCLK","CBUFSZ",
  "TCOFF","SWAP","BIT","BITS","IOCTYPE","TINC","DRIVER","TCMODE",
  "IDCODE","UIDCODE","USER1","USER2","PMSTAT","BSCAN","CHAIN","RATIO",
  "FLGBSET","FLGBCLR","TFLGBSET","TFLGBCLR","CLKDLY","IPADDR","IPCONN","IPDISC",
  "IPVLAN","BREAK","CHAN","ACTIVE","PM1CODE","PM2CODE","IOCCODE","PMINDEX",
  "PFIFO","AFIFO","NFREQ","NDEC","PMTYPE","PMTYPE1","PMTYPE2","DELAY",
  "FLASH","VERBOSE","DETECT","QALLOC","DFREQ","PKTLEN","CORE","MCORE",
  "FLAG","MCFG","HUNG","TPOE","CRC32","TIMEOUT","SEQERR","SEQFILL",
  "PKTHDR","ROUTE","RFFREQ","RFGAIN","RFATTN","RFBW","RFPWR","RFOPTS",
  "ATCCALIB","JTAGPORT","FPGASIG","FPGAREV","GCHIP","MREV1","MREV2","PMHBT",
  "TPD","TPDX","RMPL","WMPL","NRATIO","INBYTE","SYSMON","ROUTF",
  "NODMA","STATUS","MREV","RFAIS","RFCAIS","TODO","GAP","RSTIME",
  "TEMP","STATE","DMACHK","DMAOFF","TCXO","DUMP","AGCTC","AGCCF",
  "MBITS","TUNCLK","MGAIN","MFREQ","NTPOFF","JOIN","LEAVE","CONFIG",
  "TPSR","MPCIO","CUID","GPS","PCINACK","EMT","EYESCAN","GPSMODE",
  "LOG","PCIGEN","PCILANES","MCORES","ADLM","NAME","DRATE","FTTM",
  "HWREV","EVCNT","RTCLK","MGTDUMP","IPSOCK","OPTS","IPCFG","DRVFLG",
  "DCSBN","MTOFF","FRAMEDEC","ENABLE","SPEEDS","FPGADSG","NFGAIN","TABLE",
  "TRACE","DSGTYPE","NVME","NIOP","NVMS","NIOS","DEVNO","MUTE",
  "CSIZE","CXFER","STIME","VCTL","NPM","NYBW","FBWF","PING",
  "CNAME","QSTAT","NULL"
};
static int_4 keyindex[MAXKEYS]={
  KEY_RATE,KEY_FREQ,KEY_DEC,KEY_GAIN,KEY_OVSR,KEY_FRAME,KEY_ALG,KEY_FEED,
  KEY_CTYPE,KEY_CTYPE,KEY_PTYPE,KEY_PINDEX,KEY_MTYPE,KEY_MTYPE1,KEY_MTYPE2,KEY_MCHNS,
  KEY_CPC,KEY_CHNS,KEY_PCICLK,KEY_PCIBUS,KEY_PCIBW,KEY_PCIREV,KEY_STATUS,KEY_VERSION,
  KEY_PRCCLK,KEY_IOCCLK,KEY_ICLK,KEY_PCLK,KEY_CCLK,KEY_MCLK,KEY_CLKI,KEY_MSBI,
  KEY_IOC,KEY_IOCIOM,KEY_IOCALG,KEY_IOCRAM,KEY_MOD,KEY_APP,KEY_MCLK,KEY_CBUFSZ,
  KEY_TCOFF,KEY_SWAP,KEY_BIT,KEY_BITS,KEY_IOCTYPE,KEY_TINC,KEY_DRIVER,KEY_TCMODE,
  KEY_IDCODE,KEY_UIDCODE,KEY_USER1,KEY_USER2,KEY_PMSTAT,KEY_BSCAN,KEY_CHAIN,KEY_RATIO,
  KEY_FLGBSET,KEY_FLGBCLR,KEY_TFLGBSET,KEY_TFLGBCLR,KEY_CLKDLY,KEY_IPADDR,KEY_IPCONN,KEY_IPDISC,
  KEY_IPVLAN,KEY_BREAK,KEY_CHAN,KEY_ACTIVE,KEY_PM1CODE,KEY_PM2CODE,KEY_IOCCODE,KEY_PMINDEX,
  KEY_PFIFO,KEY_AFIFO,KEY_NFREQ,KEY_NDEC,KEY_PMTYPE,KEY_PMTYPE1,KEY_PMTYPE2,KEY_DELAY,
  KEY_FLASH,KEY_VERBOSE,KEY_DETECT,KEY_QALLOC,KEY_DFREQ,KEY_PKTLEN,KEY_CORE,KEY_MCORE,
  KEY_FLAG,KEY_MCFG,KEY_HUNG,KEY_TPOE,KEY_CRC32,KEY_TIMEOUT,KEY_SEQERR,KEY_SEQFILL,
  KEY_PKTHDR,KEY_ROUTE,KEY_RFFREQ,KEY_RFGAIN,KEY_RFATTN,KEY_RFBW,KEY_RFPWR,KEY_RFOPTS,
  KEY_ATCCALIB,KEY_JTAGPORT,KEY_FPGASIG,KEY_FPGAREV,KEY_GCHIP,KEY_MREV1,KEY_MREV2,KEY_PMHBT,
  KEY_TPD,KEY_TPDX,KEY_RMPL,KEY_WMPL,KEY_NRATIO,KEY_INBYTE,KEY_SYSMON,KEY_ROUTF,
  KEY_NODMA,KEY_STATUS,KEY_MREV,KEY_RFAIS,KEY_RFCAIS,KEY_TODO,KEY_GAP,KEY_RSTIME,
  KEY_TEMP,KEY_STATE,KEY_DMACHK,KEY_DMAOFF,KEY_TCXO,KEY_DUMP,KEY_AGCTC,KEY_AGCCF,
  KEY_MBITS,KEY_TUNCLK,KEY_MGAIN,KEY_MFREQ,KEY_NTPOFF,KEY_IPCONN,KEY_IPDISC,KEY_CONFIG,
  KEY_TPSR,KEY_MPCIO,KEY_CUID,KEY_GPS,KEY_PCINACK,KEY_EMT,KEY_EYESCAN,KEY_GPSMODE,
  KEY_LOG,KEY_PCIGEN,KEY_PCILANES,KEY_MCORES,KEY_ADLM,KEY_NAME,KEY_DRATE,KEY_FTTM,
  KEY_HWREV,KEY_EVCNT,KEY_RTCLK,KEY_MGTDUMP,KEY_IPSOCK,KEY_OPTS,KEY_IPCFG,KEY_DRVFLG,
  KEY_DCSBN,KEY_MTOFF,KEY_FRAMEDEC,KEY_ENABLE,KEY_SPEEDS,KEY_FPGADSG,KEY_NFGAIN,KEY_TABLE,
  KEY_TRACE,KEY_DSGTYPE,KEY_NVME,KEY_NIOP,KEY_NVMS,KEY_NIOS,KEY_DEVNO,KEY_MUTE,
  KEY_CSIZE,KEY_CXFER,KEY_STIME,KEY_VCTL,KEY_NPM,KEY_NYBW,KEY_FBWF,KEY_PING,
  KEY_CNAME,KEY_QSTAT,KEY_NULL
};
static char keyunknown[8] = "UNKNOWN";

int_4 pic_name2key (PICSTRUCT *p, char *name)
{
  int_4 i,inc=0; char fname[16],finc[32];
  for (i=0; i<16 && name[i]!=0 && name[i]>' '; i++) {
    fname[i] = name[i]; if (fname[i]>='a' && fname[i]<='z') fname[i] -= 32;
    if (fname[i]=='+'||fname[i]=='-') { 
      strncpy(finc,name+i,30); /* protect from no NULL terminator */
      inc = strtol(finc,NULL,0);
      break;
    }
  }
  fname[i]=0;

  for (i=0; i<MAXKEYS; i++) {
    if (strcmp(fname,keynames[i])==0) return keyindex[i]+inc;
  }
  return -1;
}

static char keyname[32];
char *pic_key2name (PICSTRUCT *p, int_4 key)
{
  int_4 i;
  for (i=0; i<MAXKEYS; i++) {
    if (keyindex[i]==key) return keynames[i];
  }
  if (key>=KEY_MCORE) sprintf(keyname,"MCORE+0x%x",key-KEY_MCORE);
  else if (key>=KEY_CORE) sprintf(keyname,"CORE+0x%x",key-KEY_CORE);
  else sprintf(keyname,"UNKNOWN:%d",key);
  return keyname;
}

#define CC 0
#define LTPI 3.14159265
#define TW 16
#define TW2 TW*TW*4

struct {
  int_2 d0,f0,ddi,ddr;
} tab[TW2];

#define CXD_REF_FREQ 17184000
#define CXD_NUM_PRG_BITS 32

int_4 pic_setup_cxd (PICSTRUCT *p, int_4 mport, int_4 dir, int_4 rate, int_4 gain) 
{
  int_u4 reg=0,actual;
  
  /* Calculate Programmable Clock Config Bits */  
  reg = calc_synthetic_clkbits(CXD_REF_FREQ,rate,&actual);

       if (rate== 2048000) reg |= 0x01;
  else if (rate== 8448000) reg |= 0x02;
  else if (rate==32768000) reg |= 0x03;
  else if (rate==34368000) reg |= 0x03;
  else if (rate== 1544000) reg |= 0x05;
  else { print("CXD input rate %d not supported\n",rate); return -1; }
  
       if (gain <= -9) reg |= 0x10;
  else if (gain <= -6) reg |= 0x00;
  else if (gain <= -3) reg |= 0x20;
  else                 reg |= 0x30;
  
  vprint("Setup CXD port=%d reg=%x rate=%d\n",mport,reg,actual);
  pic_wmodreg(p,mport,0,reg,CXD_NUM_PRG_BITS);
  return 0;
}

static int_u4 calc_cy22150_clkbits (int_4 fref, int_u4 freq, int_u4 *actual)
{
  int_4 q,qt, pb,po,pt;
  int_4 res, reg=0, pd, ierr,err;

  int_4 fmin=100000000, ftarget=250000000, fmax=400000000;

  pd = max(2,min(125,ftarget/freq));
  
  /* P,Q with minimal error */
  err = freq;
  for (q=0; q<128; q++) { 
    qt=q+2;
    for (pb=0; pb<1024; pb++) {
      for (po=0; po<2; po++) {
        pt = 2*(pb+4)+po;
        res = (int)((((double)fref/qt) * pt) / pd);
        ierr = (res<freq)? (freq-res) : (res-freq);
	if (ierr<err) { 
          err=ierr; 
	  *actual=res;
	  reg = (pd<<24) | (q<<16) | (po<<15) | pb;
        }
      }
    }
  }
  return reg;
}

static int_u4 calc_fpga_clkbits (real_8 fvco, int_u4 freq, int_u4 *actual)
{
  int_u4 reg;
  reg = max(1,freq/1000000);
  *actual = reg*1000000;
  reg *= 1024*1024;
  return reg;
}

static int_u8 calc_si5338_clkbits (real_8 fvco, int_u4 freq, int_u4 *actual)
{
  int_4 a,b,c;
  int_8 p1,p2,p3;
  int_4 res,ierr,err;
  int_u8 reg;

  a = (int_4)(fvco/freq);
  
  /* P,Q with minimal error */
  err = freq;
  for (c=1; c<1024; c++) { 
    for (b=0; b<=c; b++) {
      res = (int_4)( (fvco / ((double)a + ((double)b/(double)c)))+.5 );
      ierr = (res<freq)? (freq-res) : (res-freq);
      if (ierr<err) { 
        err=ierr; 
	*actual=res;
	p1 = (int_8)(a*c+b)*128/c - 512;
	p2 = b*128 % c;
	p3 = c;
	reg = (p3<<48) | (p2<<18) | p1;
        if (ierr==0) goto DONE;
      }
    }
  }
  DONE:
  return reg;
}

#define IOC_REF_FREQX 10000000
#define IOC_REF_FREQY 10000000
#define IOC_REF_VCO   2.5e9
#define FPGA_REF_VCO  200.0e6

int_4 pic_setup_ioc_clk (PICSTRUCT *p, int_4 port, int_4 rate)
{
  int_u4 reg,i,actual,oactual;
  int_4 q, pb,po,pd, pt,cs,cp;
  int_u8 regX;

  /* Calculate Programmable Clock Config Bits */
  if (p->gmcs==GMC_CLKQ) rate *= 64;
  if (p->k7)       reg = calc_fpga_clkbits(FPGA_REF_VCO,rate,&actual);
  else if (p->isY) reg = calc_cy22150_clkbits(IOC_REF_FREQY,rate,&actual);
  else             reg = calc_synthetic_clkbits(IOC_REF_FREQX,rate,&actual);
  if (p->gmcs==GMC_CLKQ) actual /= 64;

  if (port<0) { port = -port; oactual = 0; }
  else if (port==1) oactual = pic_acmd (p, ADON_RD, DMA_GICLK, 0, 0);

  vprint("IOC clock %d reg=%08x rate=%d actual=%d\n",port,reg,rate,actual);

  if (actual!=oactual) {
    if (p->k7) {
      pic_wpb (p,0,PPC_DMAC_PCLK,(int_4)reg);
    }
    else if (p->isY) {
      /* variables */
      pd = (reg>>24)&0xFF;
      q = (reg>>16)&0xFF;
      po = (reg>>15)&0x01;
      pb = reg&0x3FF;
      pt = 2*(pb+4)+po;
      /* charge pump */
           if (pt<=44) cp=0;
      else if (pt<=479) cp=1;
      else if (pt<=639) cp=2;
      else if (pt<=799) cp=3;
      else if (pt<=1023) cp=4;
      /* clock source */
      if (pd>=4) cs=1;
      else if (pd==3) { cs=3; pd=6; }
      else if (pd==2) { cs=2; pd=4; }
      else if (pd==1) { cs=0; pd=8; }
      v2print("Cy-22150 pd=%d po=%d pb=%d q=%d cs=%d cp=%d\n",pd,po,pb,q,cs,cp);
      pic_acmd (p, ADON_PGMC, 0x09, 0x01, 1);		/* CLKOE control */
      pic_acmd (p, ADON_PGMC, 0x0C, pd, 1);		/* DIV1 src and divider */
      pic_acmd (p, ADON_PGMC, 0x12, 0x20, 1);		/* Oscillator drive */
      pic_acmd (p, ADON_PGMC, 0x13, 0x00, 1);		/* Input load capacitor */
      pic_acmd (p, ADON_PGMC, 0x40, 0xC0|(cp<<2)|(pb>>8), 1);	/* Charge pump  & PB counter */
      pic_acmd (p, ADON_PGMC, 0x41, (pb&0xFF), 1);	/* Charge pump  & PB counter */
      pic_acmd (p, ADON_PGMC, 0x42, (po<<7)|q, 1);	/* PO & Q counter */
      pic_acmd (p, ADON_PGMC, 0x44, cs<<5, 1);		/* Crosspoint switch matrix */
      pic_acmd (p, ADON_PGMC, 0x45, 0x00, 1);		/* Crosspoint switch matrix */
      pic_acmd (p, ADON_PGMC, 0x46, 0x3F, 1);		/* Crosspoint switch matrix */
      pic_acmd (p, ADON_PGMC, 0x47, 0x08, 1);		/* DIV2 src and divider */
    } 
    else if (p->isX) {
      for (i=1; i<4; i++) { pic_acmd (p,ADON_WRIOC,IOC_PGMC+port,(reg>>(i*8)),0); udelay(10); }
    }
    if (port==1) pic_acmd (p, ADON_WR, DMA_GICLK, actual, 0);
  }

  return actual;
}

/*
    Utility routines
*/

int_4 getHost (char *name, int_4 maxlen, int_4 fdn) {
  int_4 i,len; char *s; 
#if _UNIX
  gethostname(name,maxlen);
  len = strlen(name);
#elif _VMS
  VAXstr devdsc; int_4 stat; unsigned short slen;
  devdsc.dtype = DSC$K_DTYPE_T;
  devdsc.dclass = DSC$K_CLASS_S;
  devdsc.c      = name;
  devdsc.length = maxlen;
  stat = lib$get_hostname(&devdsc,&slen,0);
  if (SYSERR(stat)) len=0; else len=slen;
#endif
  name[len]=0;
  for (i=0; i<len; i++) name[i]=toupper(name[i]);
  /* if !fdn, just match the short name ie.  MYHOST of myhost.mydomain */
  if (fdn==0 && len>0) {
    s = (char *)strstr(name,":.");
    if (s!=0) { 
      /* if DECNET last part has nodename */
      s = (char *)strrchr(name,'.'); 
      if (s!=0) strcpy(name,s+1); 
    }
    s = (char *)strchr(name,'.');
    if (s!=0) s[0]=0;
    len = strlen(name);
  }
  return len;
}

char *getCardName(int_4 type) {
  static char name[8];
       if (type==ICEPIC1) strcpy(name,"PIC1");
  else if (type==ICEPIC2) strcpy(name,"PIC2");
  else if (type==ICEPIC3) strcpy(name,"PIC3");
  else if (type==ICEPIC4) strcpy(name,"PIC4");
  else if (type==ICEPIC5) strcpy(name,"PIC5");
  else if (type==ICEPIC6) strcpy(name,"PIC6");
  else if (type==ICEPIC7) strcpy(name,"PIC7");
  else if (type==ICEPIC8) strcpy(name,"PIC8");
  else if (type==ICEPIC9) strcpy(name,"PIC9");
  else if (type==ICEMBT2) strcpy(name,"MBT2");
  else if (type==ICEMBT4) strcpy(name,"MBT4");
  else if (type==ICESLIC3) strcpy(name,"SLIC3");
  else strcpy(name,"Unknown");
  return name;
}

char *getPmName(int_4 type) {
  static char name[8];
       if (type==PMT_DTDM) strcpy(name,"DTDM");
  else if (type==PMT_DTDMX) strcpy(name,"DTDMX");
  else if (type==PMT_S6M) strcpy(name,"S6M");
  else if (type==PMT_ZPPM) strcpy(name,"ZPPM");
  else if (type==PMT_V5M) strcpy(name,"V5M");
  else if (type==PMT_V6M) strcpy(name,"V6M");
  else if (type==PMT_A8M) strcpy(name,"A8M");
  else if (type==PMT_A9M) strcpy(name,"A9M");
  else if (type==PMT_K8M) strcpy(name,"K8M");
  else if (type==PMT_K8P) strcpy(name,"K8P");
  else if (type==PMT_NONE) strcpy(name,"NONE");
  else if (type==PMT_TEST) strcpy(name,"TEST");
  else strcpy(name,"Unknown");
  return name;
}

char *getIomName(int_4 type, int_4 rev) {
  static char name[16],srev[16];
       if (type==IOMT_A2D) strcpy(name,"A2D");
  else if (type==IOMT_E2D) strcpy(name,"E2D");
  else if (type==IOMT_T2D) strcpy(name,"T2D");
  else if (type==IOMT_CXD) strcpy(name,"CXD");
  else if (type==IOMT_DR2D) strcpy(name,"DR2D");
  else if (type==IOMT_GXD) strcpy(name,"GXD");
  else if (type==IOMT_UXD) strcpy(name,"UXD");
  else if (type==IOMT_VXD) strcpy(name,"VXD");
  else if (type==IOMT_FPDPXD) strcpy(name,"FPDPXD");
  else if (type==IOMT_UDPXD) strcpy(name,"UDPXD");
  else if (type==IOMT_SNTXD) strcpy(name,"SNTXD");
  else if (type==IOMT_NFXD) strcpy(name,"NFXD");
  else if (type==IOMT_RFXD) strcpy(name,"RFXD");
  else if (type==IOMT_LV2D) strcpy(name,"LV2D");
  else if (type==IOMT_SDDSXD) strcpy(name,"SDDSXD");
  else if (type==IOMT_TGSDDSXD) strcpy(name,"TGSXD");
  else if (type==IOMT_TGVITAXD) strcpy(name,"TGVXD");
  else if (type==IOMT_FLZRXD) strcpy(name,"FLZRXD");
  else if (type==IOMT_UFLZXD) strcpy(name,"UFLZXD");
  else if (type==IOMT_MSASXD) strcpy(name,"MSASXD");
  else if (type==IOMT_DSFPXD) strcpy(name,"DSFPXD");
  else if (type==IOMT_QSFPXD) strcpy(name,"QSFPXD");
  else if (type==IOMT_UPACXD) strcpy(name,"UPACXD");
  else if (type==IOMT_SFPXD) strcpy(name,"SFPXD");
  else if (type==IOMT_SFPPXD) strcpy(name,"SFPPXD");
  else if (type==IOMT_LB2D) strcpy(name,"LB2D");
  else if (type==IOMT_ES2D) strcpy(name,"ES2D");
  else if (type==IOMT_RF2D) strcpy(name,"RF2D");

  else if (type==IOMT_D2A) strcpy(name,"D2A");
  else if (type==IOMT_D2E) strcpy(name,"D2E");
  else if (type==IOMT_D2T) strcpy(name,"D2T");
  else if (type==IOMT_DXC) strcpy(name,"DXC");
  else if (type==IOMT_D2DR) strcpy(name,"D2DR");
  else if (type==IOMT_DXG) strcpy(name,"DXG");
  else if (type==IOMT_DXU) strcpy(name,"DXU");
  else if (type==IOMT_DXV) strcpy(name,"DXV");
  else if (type==IOMT_DXFPDP) strcpy(name,"DXFPDP");
  else if (type==IOMT_DXUDP) strcpy(name,"DXUDP");
  else if (type==IOMT_DXSNT) strcpy(name,"DXSNT");
  else if (type==IOMT_DXNF) strcpy(name,"DXNF");
  else if (type==IOMT_DXRF) strcpy(name,"DXRF");
  else if (type==IOMT_D2LV) strcpy(name,"D2LV");
  else if (type==IOMT_DXSDDS) strcpy(name,"DXSDDS");
  else if (type==IOMT_DXTGSDDS) strcpy(name,"DXTGS");
  else if (type==IOMT_DXTGVITA) strcpy(name,"DXTGV");
  else if (type==IOMT_D2PSE) strcpy(name,"D2PSE");
  else if (type==IOMT_DXFLZR) strcpy(name,"DXFLZR");
  else if (type==IOMT_DXUFLZ) strcpy(name,"DXUFLZ");
  else if (type==IOMT_DXMSAS) strcpy(name,"DXMSAS");
  else if (type==IOMT_DXDSFP) strcpy(name,"DXDSFP");
  else if (type==IOMT_DXQSFP) strcpy(name,"DXQSFP");
  else if (type==IOMT_DXUPAC) strcpy(name,"DXUPAC");
  else if (type==IOMT_DXSFP) strcpy(name,"DXSFP");
  else if (type==IOMT_DXSFPP) strcpy(name,"DXSFPP");
  else if (type==IOMT_D2ES) strcpy(name,"D2ES");
  else if (type==IOMT_D2AWG) strcpy(name,"D2AWG");
  else if (type==IOMT_GPS) strcpy(name,"GPS");
  else if (type==IOMT_DIODE) strcpy(name,"DIODE");
  else if (type==IOMT_D2RF) strcpy(name,"D2RF");

  else if (type==IOMT_NONE) strcpy(name,"NONE");
  else if (type==IOMT_TEST) strcpy(name,"TEST");
  else if (type==IOMT_UNKNOWN) strcpy(name,"UNKNOWN");
  else strcpy(name,"Unknown");

  if (rev>0 && type!=0 && type<90) { 
    sprintf(srev,"r%d",rev);
    strcat(name,srev); 
  }
  return name;
}

char *getPortName(int_4 index, int node) 
{
  int pm = (node>0)? 1:0;
  int pmp = (node>0)? 1:0;
  int port = index&0xF;
  int inp = index>=16;
  static char name[8];
  switch (port) {
  case 0:  strcpy(name,"PROC"); break;
  case 1:  strcpy(name,"PCIE"); break;
  case 2:  strcpy(name,pm?"XBAR1":"IOM1"); break;
  case 3:  strcpy(name,pm?"XBAR2":"IOM2"); break;
  case 4:  strcpy(name,"CORE1"); break;
  case 5:  strcpy(name,"CORE2"); break;
  case 6:  strcpy(name,pmp?"MCOR1":"PM1"); break;
  case 7:  strcpy(name,pmp?"MCOR2":"PM2"); break;
  case 8:  strcpy(name,"DRAM"); inp=1-inp; break;
  case 9:  strcpy(name,"DIO"); break;
  case 10: strcpy(name,"NIO"); break;
  default: strcpy(name,"???"); break;
  }
  strcat(name,inp?"-I":"-O");
  return name;
}

int_4 pic_type2name (PICSTRUCT *p, int_4 mode, int_4 type, int_4 rev, char *name)
{
       if (mode==T2N_CARD) strcpy (name, getCardName(type) );
  else if (mode==T2N_PM  ) strcpy (name, getPmName(type) );
  else if (mode==T2N_IOM ) strcpy (name, getIomName(type,rev) );
  else if (mode==T2N_PORT) strcpy (name, getPortName(type,rev) );
  else return -1;
  return strlen(name);
}

/*
    Non-Volatile RAM interface routines
*/
int_4 pic_nvm_stat (PICSTRUCT *p, int_4 port, int_4 test)
{
  int_4 i,j,iadr=0x0800,reg[128],status,hbt,maxtl,maxgb,sgl,tpd,off,gen,lanes,enabled=0;
  if (port<0) port = findintflagdef("TP",p->config,11);
  tpd = findintflagdef("TPD",p->config,0);
  status = jvm_rmem(p,port,0x18,FLG_NVME);
  if (test==9) vprint("Disabling NVME controller on port=%d\n",port);
  if (findintflag("FORCE",p->config)==2) status = -1;
  if (status!=0x1ce00000) { 
    if (test==9) return 0;
    if (test!=0&&test!=8) { print("Err running NVME test=%d on active status=%08x\n",test,status); return -1; }
    for (i=0; i<5; i++) {
      jvm_wmem(p,port,0x1C,0,FLG_NVME);
      status = pic_loadfile (p, "icenvme", FLG_JVM_NVME|port); 
      if (status>0) break; udelay(200000); 
    }
    if (i>=5) { print("Err booting NVME JVM processor on port=%d retry=%08x\n",port,status); return -1; }
    for (i=0; i<200; i++) { status=jvm_rmem(p,port,0x1C,FLG_NVME); if (((status>>16)&0xFF)!=0) break; udelay(10000); }
    if (i>=200) { print("Err booting NVME PCIe interface on port=%d status=%08x\n",port,status); return -1; }
    test=6; jvm_rwmem(p,port,0x18,&test,1,FLG_NVME);
    for (i=0; i<200; i++) { status=jvm_rmem(p,port,0x1C,FLG_NVME); if ((status>>24)==1) break; udelay(10000); }
    if (i>=200) { print("Err enabling NVME controller on port=%d status=%08x\n",port,status); return -1; }
    test=8;
  }
  if (test==0) return 0;

  if (test>=70) jvm_rwmem(p,port,0x14,&tpd,4,FLG_NVME);
  jvm_rwmem(p,port,0x18,&test,1,FLG_NVME);
  for (i=0; i<100 && jvm_rmem(p,port,0x18,FLG_NVME)!=0x1ce00000; i++) udelay(10000); 
  if (test==9) return 0;
  if (i>=100) print("Err completing NVME controller test=%d on port=%d\n",test,port);

  if (test==8) {
    status = jvm_rmem(p,port,0x10,FLG_NVME);
    lanes = (status>>28)&0xF;
    gen   = (status>>24)&0xF; if (gen==4) gen=3;
    for (j=0; j<80; j++) reg[j] = jvm_rmem (p, port, iadr+(j*4), FLG_NVME);
    maxtl = (reg[19]>>8)&0xFF; maxtl = 0x1000 * (1<<maxtl);
    maxgb = (reg[71]<<2) | ((reg[70]>>30)&0x3);	
    sgl   = reg[79];	
    if (p->verbose&0x11) print("PCI-NVME Status port=%d Lnk=G%dL%d Vid=%04x SN=%.20s MN=%.28s maxTL=%d maxGB=%d SGL=%x\n",
	port,gen,lanes,reg[0]&0xFFFF,(char*)(reg+1),(char*)(reg+6),maxtl,maxgb,sgl);
    off = (port-11)*8;
    WR(REG_HAR8+off+0,reg[70]);	/* for icebd gendisk */
    WR(REG_HAR8+off+4,reg[71]);	/* for icebd gendisk */
  } else {
    hbt = jvm_rmem(p,port,0xFFFC,FLG_NVME);
    print("PCI-NVME Debug Register Space for test=%d port=%d stat=%08x hbt=%08x\n",test,port,status,hbt);
    for (i=0x0; i<0x100; i+=16) {
      for (j=0; j<4; j++) reg[j] = jvm_rmem (p, port, iadr+i+(j*4), FLG_NVME);
      print(" Addr=%04x Data=%08x_%08x_%08x_%08x\n",i,reg[3],reg[2],reg[1],reg[0]);
    }
  }
  return 0;
}


int_4 pic_nvram (PICSTRUCT *p, char *fname, int_4 mode)
{
       if (mode<=0) pic_nvread(p);
  else if (mode>=1) pic_nvm_stat(p,-1,mode);
  else print("NVRAM mode=%d not supported\n",mode);
  return 0;
}

int_4 pic_rcfg (PICSTRUCT *p, int_4 addr)
{
  int data;
  int status = io_ctl (p, IOCTL_RCFG, addr, &data, 4);
  return data;
}

void pic_cap (PICSTRUCT *p, int_4 set)
{
  int_4 i,cp,cpx,cpy,cpz,addr; 
  addr = pic_rcfg(p, 0x34) & 0xFF;
  for (i=0; i<10 && addr!=0; i++) {
    cp  = pic_rcfg(p, addr);
    cpx = pic_rcfg(p, addr+4);
    cpy = pic_rcfg(p, addr+8);
    cpz = 0;
    if ((cp&0xFF)==0x10 && set) { /* capabilites register */
      RD(0xE4, cpz);
      cpz = (cpz&0xFFFF0000) | (cpy&0x0000FFFF);
      WR(0xE4, cpz);
    }
    vprint("Cap at %02x = %08x : %08x : %08x    E4=%08x\n",addr,cp,cpx,cpy,cpz);
    addr = (cp>>8)&0xFF;
  }
}

void pic_nvread (PICSTRUCT *p)
{
  int_4 i,j,k,cp,cpx,cpy,addr; char text[40];
  int_4 type = pic_nv_read_b(p,0x08) >> 4;
  print("PCI Config Space\n");
  print("CardType = %s\n",getCardName(type));
  print("Vendor ID = %04x\n", pic_nv_read_w(p, 0x00) );
  print("Device ID = %04x\n", pic_nv_read_w(p, 0x02) );
  print("Command = %04x\n",   pic_nv_read_w(p, 0x04) );
  print("Status  = %04x\n",   pic_nv_read_w(p, 0x06) );
  print("Revision ID = %02x\n", pic_nv_read_b(p, 0x08) );
  print("Class Code = %06x\n", ( pic_nv_read_l(p,0x08)>>8)&0xFFFFFF );
  print("Latency Timer = %02x\n", pic_nv_read_b(p, 0x0D) );
  print("Header Type = %02x\n", pic_nv_read_b(p, 0x0E) );
  print("Self Test = %02x\n", pic_nv_read_b(p, 0x0F) );
  print("BaseAddr-0 = %08x\n", pic_nv_read_l(p, 0x10) );
  print("BaseAddr-1 = %08x\n", pic_nv_read_l(p, 0x14) );
  print("BaseAddr-2 = %08x\n", pic_nv_read_l(p, 0x18) );
  print("BaseAddr-3 = %08x\n", pic_nv_read_l(p, 0x1C) );
  print("BaseAddr-4 = %08x\n", pic_nv_read_l(p, 0x20) );
  print("BaseAddr-5 = %08x\n", pic_nv_read_l(p, 0x24) );
  print("ConfigPtr  = %02x\n", pic_nv_read_b(p, 0x34) );
  print("Interrupt Line = %02x\n", pic_nv_read_b(p, 0x3C) );
  print("Interrupt Pin = %02x\n", pic_nv_read_b(p, 0x3D) );
  print("Min-Grant = %02x\n", pic_nv_read_b(p, 0x3E) );
  print("Max-Latency = %02x\n", pic_nv_read_b(p, 0x3F) );
  if (!p->isY) return;
  if (type==ICEPIC5) {
    print("PCI-X Cmd    = %08x\n", pic_nv_read_l(p, REG_XCMD) );
    print("PCI-X Status = %08x\n", pic_nv_read_l(p, REG_XSTA) );
  }
  if (type==ICEPIC8) pic_cap (p,0);
  cp = pic_nv_read_b(p, 0x34) & 0xFF;
  if (type==ICEPIC6 || type==ICEPIC7 || type==ICEPIC8) 
  for (i=0; i<11; i++) {
    j = pic_nv_read_l(p, cp+i*4);
    text[0]=0;
    if (i==3) sprintf(text,"(cMPL=%d)", 0x80<<((j>>0)&0x7) );
    if (i==4) sprintf(text,"(rMPL=%d wMPL=%d)", 0x80<<((j>>12)&0x7), 0x80<<((j>>5)&0x7) );
    if (i<2) print("PM    Capability (%d) = %08x\n", i, j);
    else     print("PCIe  Capability (%d) = %08x %s\n", i-2, j, text);
  }
}

int_4 pic_nv_read_l (PICSTRUCT *p, int_4 offset)
{
  int_4 status;
  RD( offset, status);
  return (status);
}
int_4 pic_nv_read_w (PICSTRUCT *p, int_4 offset)
{
  int_4 status;
  RD( (offset&0xFFFC), status);
  status >>= (offset&0x3)*8;
  return (status&0xFFFF);
}
int_4 pic_nv_read_b (PICSTRUCT *p, int_4 offset)
{
  int_4 status;
  RD( (offset&0xFFFC), status);
  status >>= (offset&0x3)*8;
  return (status&0xFF);
}

void vfill (int_4 value, int_4 *array, int_4 bytes) 
{
  int_4 i, size=bytes/4;
  for (i=0; i<size; i++) array[i] = value;
}

void pic_print_pm (PICSTRUCT *p, int_4 offset)
{
  int_4 lower,upper;
  upper = pic_acmd (p, ADON_RD, offset+0, 0, 0);
  lower = pic_acmd (p, ADON_RD, offset+4096, 0, 0);
  lower &= 0xFFFF0000;
  print ("Addr=%x, PM=%08x %08x\n",offset,upper,lower);
}

void pic_prettyprint (PICSTRUCT* p)
{
  print("PICSTRUCT Address: 0x%lx\n", (long unsigned int)p);
  print("  type     : %d\n", p->type);
  print("  ptype    : %d\n", p->ptype);
  print("  pindex   : %d\n", p->pindex);
  print("  mtype    : %d,%d\n", p->mtype[0], p->mtype[1]);
  print("  pmtype    : %d,%d\n", p->pmtype[0], p->pmtype[1]);
  print("  flags    : %d\n", p->flags);
  print("  devno    : %d\n", p->devno);
  print("  lcount   : %d\n", p->lcount);
  print("  fd       : %d\n", p->fd);
  print("  function : %d\n", p->ioctl.function);
  print("  offset   : %d\n", p->ioctl.offset);
  print("  buffaddr : 0x%lx\n",(long unsigned int)p->ioctl.bufaddr);
  print("  bytes    : %d\n", p->ioctl.bytes);
  print("  status   : %d\n", p->ioctl.status);
  print("  base     : %llx\n", p->base);
  print("  size     : %d\n", p->size);
  print("  \n");
}

int_4 pic_detect_config (PICSTRUCT* p, char *config, int_4 flags) 
{
  int_4 i, status, devno, active, breakflag=0,sig,ver,npm; char string[160];
  PICSTRUCT psave;
  memcpy ((void*)(&psave), (void*)p, sizeof(PICSTRUCT));
  devno  = p->devno;
  sprintf(string,"ICEPIC,DEVNO=%d,IOM=NONE,PM=NONE,",devno);
  status = pic_open (p,string,&breakflag,0);
  if (status<0) return -1;
  getPortSig(p,0,0,&sig,&ver);
  if (sig==0x7E7E7E7E) {
    sprintf(config,"ICEPM,DEVNO=%d,IOM=NONE,",devno);
    goto DONE;
  }
  active = pic_active(p,0);
  if (active<0) status = pic_reset(p,FLG_BOOT);
  getIOMsig(p,0);
  getPMsig(p,0);
  sprintf(config,"ICEPIC,DEVNO=%d,BIDIR=1,",devno);
  for (i=0; i<p->miom; i++) { sprintf(string,"IOM%d=%s,",i+1,getIomName((int_4)p->mtype[i],(int_4)p->mrev[i])); strcat(config,string); }
  for (i=0; i<p->mpm; i++) { sprintf(string,"PM%d=%s,",i+1,getPmName((int_4)p->pmtype[i])); strcat(config,string); }
  npm=0; for (i=0;i<4;i++) if (p->pmtype[i]>0) npm++;
  if (npm>2) strcat(config,"QPM,");
  DONE:
  memcpy ((void*)p, (void*)(&psave), sizeof(PICSTRUCT));
  return strlen(config);
}

int_4 pic_ioctl_version (int_4 index)
{
  int value = -1;
#if _LINX
  short fd; int_4 status,lbuf[6]; PICIOCTL pio;
  fd = open("/dev/iceram",O_RDWR,0);
  if (fd>0) {
    pio.function = IOCTL_VERSION;
    pio.offset = 0;
    pio.bufaddr = (void *)lbuf;
    pio.bytes = 8;
    status = ioctl(fd,_IOR('f',0x1,PICIOCTL),&pio);
    if (status==0) value = lbuf[index];
    close(fd);
  }
#endif
  return value;
}

int_4 pic_detect (int_4 flag) 
{
  int_4 i,n=0,numdev,flags,status,lstat; char string[80];
  PICSTRUCT *p = (PICSTRUCT *)malloc(sizeof(PICSTRUCT));
  numdev = pic_ioctl_version(1);
  if (flag==1 && numdev>=0) return numdev;
  if (numdev<0) numdev=12;
  for (i=0; i<numdev; i++) {
    sprintf(string,"ICEPIC,DEVNO=%d,IOM=NONE,PM=NONE,",i);
    status = pic_open (p,string,NULL,FLG_TEST|FLG_NOBRK);
    if (status<0);
    else if (flag==0) {	/* detect */
      n++; pic_detect_card (i,0);
    }
    else if (flag==1) {	/* numdev */
      n++;
    }
    else if (flag==2) {	/* active */
      lstat = pic_lock (p, LOCK_QUERY);
      if (lstat!=0) n=-2;
      else n += pic_active(p,0);
    }
    if (status>=0) pic_close(p);
  }
  free((void*)p);
  return n;
}

int_4 pic_detect_card (int_4 devno, int_4 flags) 
{
  int_4 i, n=0, ver, sig, rev, typ, dsg[2], bw, br, mcsr, active, status, breakflag=0; char string[80];
  PICSTRUCT *p = (PICSTRUCT *)malloc(sizeof(PICSTRUCT));
  sprintf(string,"ICEPIC,DEVNO=%d,IOM=NONE,PM=NONE,",devno);
  status = pic_open (p,string,&breakflag,FLG_TEST);
  if (status>=0) {
    if (flags&FLG_TEST) goto DONE;
    rev = pic_nv_read_b(p,0x08);
    if (rev>=0 && rev<=2) rev |= (ICEPIC2<<4);	/* pre version stamped EPROM */
    typ = rev>>4; rev &= 0xF;
    p->mpm = p->isY? 2 : (p->type==ICEMBT4)? 1 : 0;
    active = pic_active(p,0);
    print("CARD #%d Type=%s  ",devno,getCardName(typ));
    if (typ>=ICEPIC8) {
      getPortSig(p,0,1,dsg+1,dsg+0);
      print(" Dsg=%.8s  ",(char*)dsg);
    }
    if (active>0) print ("(Up/Active) Ndmac=%d",active);
    else if (active==0) print ("(Up/Idle)");
    else if (active<0) print ("(Down)");
    print("\n  Interface  ");
    RD(REG_MCSR,mcsr);
    if (typ==9) {
      br = ((mcsr&MCSR_GENX)==MCSR_GEN3)? 4 : ((mcsr&MCSR_GENX)==MCSR_GEN2)? 3 : ((mcsr&MCSR_GENX)==MCSR_GEN1)? 2 : 1;
      bw = ((mcsr&MCSR_LANEX)==MCSR_LANE16)? 16 : ((mcsr&MCSR_LANEX)==MCSR_LANE8)? 8 : ((mcsr&MCSR_LANEX)==MCSR_LANE4)? 4 : 1;
      print("Type=PCIe ChipRev=%x Bus=%dLane Clk=Gen%d ",rev,bw,br);
    } else if (typ>=6 && typ<=8) {
      br = ((mcsr&MCSR_GENX)==MCSR_GEN3)? 3 : ((mcsr&MCSR_GENX)==MCSR_GEN2)? 2 : 1;
      if (typ==6) br = 1; 
      bw = (mcsr&MCSR_CLK66)? 2 : 1; 
      if (mcsr&MCSR_BUS64) bw *= 4;
      if (typ==ICEPIC8 && rev==5 && bw==2) bw=16;
      print("Type=PCIe ChipRev=%x Bus=%dLane Clk=Gen%d ",rev,bw,br);
    } else {
      bw = (mcsr&MCSR_BUS64)? 64 : 32;
      br = (mcsr&MCSR_CLK66)? 66 : 33;
      print("Type=PCI ChipRev=%x Bus=%db Clk=%dMHz ",rev,bw,br);
    }
    print ("Endian=%d ",(mcsr>>8)&0x7);
    status = pic_getkeyl(p,0,KEY_DRIVER);
    print ("Driver=%d ",status);
    if (active==0) { 			/* alive with nobody using */
      print ("\n  Modules    ");
      getPMsig(p,0);
      getIOMsig(p,0);
      if (p->isY) {
        print ("IOM1=%s ",getIomName(p->mtype[0],p->mrev[0]));
        print ("IOM2=%s ",getIomName(p->mtype[1],p->mrev[1]));
      } else {
        print ("IOM1=N/A IOM2=N/A ");
      }
      for (i=0; i<p->mpm; i++) {
        status = getPortSig(p,JTAG_PORT_PMOFF+i+1,0,&sig,&ver);
        if (p->pmtype[i]<=0) print ("PM%d=%s ",i+1,getPmName((int_4)p->pmtype[i]));
        else if (p->pmrev[i]<=0) print ("PM%d=%s:%d:%.4s ",i+1,getPmName((int_4)p->pmtype[i]),ver,(char*)(&sig));
        else print ("PM%d=%sr%d:%d:%.4s ",i+1,getPmName((int_4)p->pmtype[i]),p->pmrev[i],ver,(char*)(&sig));
      }
    }
    print ("\n  FirmWare ");
    if (p->isY) { 
      status = getPortSig(p,0,0,&sig,&ver);
      print ("  SoC Ver=%d Sig=%.4s ",ver,(char*)(&sig));
    } else {
      print ("  NONE");
    }
    print ("\n  SoftWare   ");
    if (active>=0) {
      getIOCsig(p,0);
      ver = pic_getkeyl(p,0,KEY_VERSION);
      print ("%s=%d ",p->isY?"PPC":"Sharc",ver);
      print ("IOC=%.4s ",p->iocc);
    }
    print ("\n");
    DONE:
    pic_close(p);
    status = 1;
  }
  free((void*)p);
  return status;
}

/*
    Driver interface code
*/
#define MM_MCSR (REG_MCSR>>2)

int_4 mmioctl (PICSTRUCT *p) {
  int_4 i=0, n=p->ioctl.bytes>>2, off=p->ioctl.offset>>2, fifo=(p->ioctl.offset==REG_FIFO);
  volatile int_4 *ploc = (volatile int_4 *)(p->loc);
  volatile int_4 *lbuf = (volatile int_4 *)(p->ioctl.bufaddr);

  switch (p->ioctl.function) {
    case IOCTL_VERSION: 
      lbuf[i++] = ICE_VERSION;
      break;
    case IOCTL_READ: 
      for (;i<n;i++) {
        if (fifo) while (ploc[MM_MCSR]&MCSR_IFIFO_EMPTY) udelay(20);
        lbuf[i] = ploc[off];
      }
      break;
    case IOCTL_WRITE:
      for (;i<n;i++) {
        if (fifo) while (ploc[MM_MCSR]&MCSR_OFIFO_FULL) udelay(20);
        ploc[off] = lbuf[i];
      }
      break;
    case IOCTL_ACMD:
      for (i=4;i>=0;i--) ploc[(REG_OMBX>>2)+i] = lbuf[i]; 
      for (i=0;;) {
        if (ploc[MM_MCSR]&MCSR_IMB1F) break;
        udelay(20);
      } 
      lbuf[i++] = ploc[REG_IMB1>>2]; 
      break;
    case IOCTL_LOG:
    case IOCTL_QLOG:
      break;
    default: print("Unhandled MMIOCTL function=%d\n",p->ioctl.function);
  }
  p->ioctl.status = i<<2;
  return 0;
}

int_4 io_ctl (PICSTRUCT *p, int_4 function, int_4 offset, int_4* buffer, int_4 bytes) 
{
  int_4 status;
  if (jtagfd!=NULL && function>=IOCTL_READ && function<=IOCTL_ACMD) {
    buffer[0]=extjtag(p,function,offset,buffer[0],bytes,buffer);
    return 4;
  }
  if (p->devno<0 || p->type==ICENIC || p->type==ICENVM) {	/* not real PIC cards */
    if (function>=IOCTL_READ && function<=IOCTL_ACMD) return 0;	/* fake HW access - allow other functions */
    if (p->devno==-3) return 0;		/* VHS Core */
  }
#if _VMS
  unsigned long inadr[2];
  if (function==IOCTL_MAP || function==IOCTL_UNMAP) {
    inadr[0] = (unsigned long)(buffer);
    inadr[1] = inadr[0] + bytes - 1;
    if (function==IOCTL_MAP  ) status = sys$lckpag (inadr, 0, 0);
    if (function==IOCTL_UNMAP) status = sys$ulkpag (inadr, 0, 0);
    if (SYSERR(status)) SYSMSG(status);
  }
  status = sys$qiow(0,p->fd,IO$_NOP,iosb,0,0, 
	buffer, bytes, function, offset, 0, 0);
  if (SYSERR(status)) { SYSMSG(status); p->ioctl.status = 0; }
  else p->ioctl.status = iosb[1];

#elif _WIN
  int_4 inbytes=sizeof(PICIOCTL), outbytes=sizeof(PICIOCTL);
  p->ioctl.function = function;
  p->ioctl.offset   = offset;
  p->ioctl.bufaddr  = buffer;
  p->ioctl.bytes    = bytes;
  if (function==IOCTL_ACMD || function==IOCTL_WRITE || function==IOCTL_LIOC) {
    inbytes+=bytes; memcpy ((void *)p->ioctldata,(void *)buffer,bytes); }
  else if (function==IOCTL_READ || function==IOCTL_VERSION) outbytes+=bytes; 
  status = DeviceIoControl (p->fd, 
    (ULONG)CTL_CODE(ICEPIC_DEVICE_TYPE,0x800,METHOD_BUFFERED,FILE_ANY_ACCESS), 
         &p->ioctl,inbytes, &p->ioctl,outbytes, (PULONG)(&outbytes),NULL);
  if (SYSERR(status)) { SYSMSG(status); p->ioctl.status=0; }
  outbytes -= sizeof(PICIOCTL);
  if (outbytes==4) buffer[0] = p->ioctldata[0];
  else if (outbytes>0) memcpy ((void *)buffer,(void *)p->ioctldata,outbytes); 

#else 
  p->ioctl.function = function;
  p->ioctl.offset   = offset;
  p->ioctl.bufaddr  = buffer;
  p->ioctl.bytes    = bytes;
#if _MAC
  status = macmmioctl(p);
#elif _SOL
  status = ioctl (p->fd, _IOWR('f',0x1, sizeof(PICIOCTL)), &p->ioctl);
#else
  if (p->fd==MMFD) status = mmioctl(p);
  else status = ioctl (p->fd, _IOR('f',0x1, PICIOCTL), &p->ioctl);
#endif
  if (SYSERR(status)) { SYSMSG(status); p->ioctl.status=0; }

#endif
  return (p->ioctl.status);
}


/*
   Endian Conversion routines
*/
void ConvertEndian (int_4 *buf, int_4 count, int_4 bits) {
#if _IEEE
  Swap(buf,count,bits);
#endif
}

void Swap (int_4 *buf, int_4 count, int_4 bits) {
  int_4 i; char c; char *cbuf = (char *)buf;
  if (bits==16) for (i=0; i<count*2; i+=2) {
    c=cbuf[i+0]; cbuf[i+0]=cbuf[i+1]; cbuf[i+1]=c;
  }
  if (bits==32) for (i=0; i<count*4; i+=4) {
    c=cbuf[i+0]; cbuf[i+0]=cbuf[i+3]; cbuf[i+3]=c;
    c=cbuf[i+1]; cbuf[i+1]=cbuf[i+2]; cbuf[i+2]=c;
  }
  if (bits==64) for (i=0; i<count*8; i+=8) {
    c=cbuf[i+0]; cbuf[i+0]=cbuf[i+7]; cbuf[i+7]=c;
    c=cbuf[i+1]; cbuf[i+1]=cbuf[i+6]; cbuf[i+6]=c;
    c=cbuf[i+2]; cbuf[i+2]=cbuf[i+5]; cbuf[i+5]=c;
    c=cbuf[i+3]; cbuf[i+3]=cbuf[i+4]; cbuf[i+4]=c;
  }
}

int_8 swaplt (int_8 data) {
  char c, *cbuf=(char*)(&data);
  c=cbuf[0]; cbuf[0]=cbuf[7]; cbuf[7]=c;
  c=cbuf[1]; cbuf[1]=cbuf[6]; cbuf[6]=c;
  c=cbuf[2]; cbuf[2]=cbuf[5]; cbuf[5]=c;
  c=cbuf[3]; cbuf[3]=cbuf[4]; cbuf[4]=c;
  return data;
}

int_4 swapit (int_4 data) {
  char c, *cbuf=(char*)(&data);
  c=cbuf[0]; cbuf[0]=cbuf[3]; cbuf[3]=c;
  c=cbuf[1]; cbuf[1]=cbuf[2]; cbuf[2]=c;
  return data;
}

int_u2 swapst (int_u2 data) {
  char c, *cbuf=(char*)(&data);
  c=cbuf[0]; cbuf[0]=cbuf[1]; cbuf[1]=c;
  return data;
}

void ConvertEEEIDouble (real_8 *buf, int_4 count) {
#if _VAX
  int_4 i; for (i=0; i<count; i++) 
    cvt$convert_float(&buf[i],CVT$K_IEEE_T,&buf[i],CVT$K_VAX_D,0);
#else 
  ConvertEndian ((int_4 *)buf, count, 64);
#endif
}



/*
  Driver lock mechanism 
*/

#if _VMS
int_4 lockinit=0;
typedef struct { short status, na; int id; } LKSB;
volatile LKSB *lksb, lksbs[MAXICEPICS];

int_4 pic_lock (PICSTRUCT *p, int_4 mode) 
{
  int_4 i,stat;
  VAXstr devdsc; 
  char lockname[80];
  unsigned short len;

  if (p->flags&FLG_NOLOCK) return 0;
  lksb = &lksbs[p->devno];

  switch (mode) {

  case LOCK_OPEN:
    p->lcount = 0;
    if (lockinit==0) { lockinit=1; for (i=0;i<MAXICEPICS;i++) lksbs[i].id=0; }
    if (lksb->id!=0) break;
    devdsc.dtype = DSC$K_DTYPE_T;
    devdsc.dclass = DSC$K_CLASS_S;
    devdsc.c      = lockname;
#if _USEHOSTNAME
    stat = getHost (lockname,80,0);
    if (stat>20) { stat=20; 
      print("WARNING: HostName %s truncated to 20 char for VMS lock\n",lockname); }
    if (stat<=0) sprintf(lockname,"ICEPIC-%d",p->devno);
    else sprintf(lockname,"%.*s_ICEPIC-%d",stat,lockname,p->devno);
#else
    sprintf(lockname,"ICEPIC-%d",p->devno);
#endif
    devdsc.length = (int_4)strlen(lockname);
    vprint("Lockname = %s\n",lockname);
    lksb->status = lksb->na = lksb->id = 0;
    if ( !sys$enqw (0, 0, lksb, LCK$M_SYNCSTS, &devdsc, 0,0,0,0,0,0) ) {
      print("Unable to allocate resource lock\n");
      return (-1);
    }
    break;

  case LOCK_ALLOC:
    if (++p->lcount != 1) break;	/* already acquired */
    stat = sys$enqw (0, LCK$K_EXMODE, lksb, LCK$M_CONVERT, 0,0,0,0,0,0,0);
    if (!stat) { SYSMSG(stat); p->lcount--; return 1; }
    break;  

  case LOCK_FREE:
    if (--p->lcount != 0) break;	/* still acquired */
    stat = sys$enqw (0, LCK$K_NLMODE, lksb, LCK$M_CONVERT, 0,0,0,0,0,0,0);
    if (!stat) { SYSMSG(stat); p->lcount++; return 1; }
    break;  

  case LOCK_QUERY:
    return 0;	/* not a problem on VMS */
    break;

  case LOCK_CLEAR:
    return 0;	/* not a problem on VMS */
    break;

  case LOCK_CLOSE:
    /* released on image exit */
    p->lcount = 0;
    break;  

  default:
    print ("Bad lock mode %d\n",mode);
  }

  /* vprint("Lock mode=%d stat=%d count=%d\n",mode,stat,p->lcount); */
  return (p->lcount);
}

#elif _WIN
volatile HANDLE hsems[MAXICEPICS];

int_4 pic_lock (PICSTRUCT *p, int_4 mode) { 

  HANDLE hsem;
  DWORD waitResult;
  int lockid;
  int_4 threadid;
  char semname[40];

  if (p->flags&FLG_NOLOCK) return 0;

  lockid = p->devno;
  hsem = hsems[lockid];
  threadid = (int_4)GetCurrentThreadID();

  switch (mode) {

  case LOCK_OPEN:
    sprintf(semname,"IcepicSem%d",lockid);
    hsem = CreateSemaphore(NULL,1L,1L,semname);
    if (hsem==NULL) { perror("semCreate - IPC"); }
    hsems[lockid]=hsem;
    p->lcount = 0;
    break;  

  case LOCK_ALLOC:
    if (p->lcount>0 && threadid==p->tid); /* already acquired */
    else waitResult = WaitForSingleObject(hsem,10000L);
    p->tid = threadid;
    p->lcount++;
    break;  

  case LOCK_FREE:
    if (--p->lcount != 0) break;	/* still acquired */
    if (!ReleaseSemaphore(hsem,1L,NULL)) { perror("semRelease - IPC"); }
    break;  

  case LOCK_QUERY:
    return 0;	/* not a problem on WIN */

  case LOCK_CLEAR:
    p->lcount = 0;
    return 0;	/* not a problem on WIN */
    break;  

  case LOCK_CLOSE:
    p->lcount = 0;
    break;  

  default:
    print ("Bad lock mode %d\n",mode);
  }
  return (p->lcount);
}

#else

#include <sys/syscall.h>
#ifndef SEMVMX
#define SEMVMX 32767
#endif

#if _OSF|_SOL|_HPUX
union semun {
  int              val;         /* value for SETVAL */
  struct semid_ds *buf;         /* buffer for IPC_STAT & IPC_SET */
  ushort          *array;       /* array for GETALL & SETALL */
};
#elif _LINX
  union semun {
    int val;                    /* value for SETVAL */
    struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
    unsigned short int *array;  /* array for GETALL, SETALL */
    struct seminfo *__buf;      /* buffer for IPC_INFO */
  };
#endif

int  semid = -1;		/* Semaphore set ID */
key_t semkey = 0x1CE;			/* Key used to create semaphore set */

int_4 pic_lock (PICSTRUCT *p, int_4 mode) 
{
  int	      lockid;
  int         status=0;
  int_4	      threadid;
  struct sembuf op[2];
  char errstr[64];
  union semun semctl_arg;		/* For semctl functions */

  if (p->flags&FLG_NOLOCK) return 0;

  lockid = p->devno;

#ifdef SYS_gettid
  threadid = (int_4)syscall(SYS_gettid);
#else
  threadid = (int_4)getpid();
#endif

  if (semid<0) {
    if ((semid = semget(semkey, MAXICEPICS, 0666 | IPC_CREAT)) < 0) {
      perror("semget - IPC_CREAT");
    }
  }

  switch (mode) {

  case LOCK_OPEN:
    p->lcount = 0;
    break;  

  case LOCK_ALLOC:
    if (p->lcount>0 && threadid==p->tid) goto ACQUIRED; /* already acquired */
    op[0].sem_num = op[1].sem_num = lockid;
    op[0].sem_op=0; op[1].sem_op  = SEMVMX;
    op[0].sem_flg = op[1].sem_flg = 0;
    while (semop(semid, &op[0], 2) < 0) {
      if (errno == EINTR);  /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EAGAIN) udelay(10000); /* try later */
      else {
	sprintf(errstr,"semop - WRLCK failure semid=%d id=%d p=%p",semid,lockid,p);
	perror(errstr);
	return -1;	/* shouldn't get here */
      }
    }
    ACQUIRED:
    p->tid = threadid;
    p->lcount++;
    break;  

  case LOCK_FREE:
    if (--p->lcount != 0) break;	/* still acquired */
    op[0].sem_num = lockid;
    op[0].sem_op  = -1*SEMVMX;
    op[0].sem_flg = 0;
    while (semop(semid, &op[0], 1) < 0) {
      if (errno == EINTR);    /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EINVAL) {  /* Lock already taken from us */
	print("Lock cleared prior to exclusive lock release attempt\n");
	break;	/* out of while */
      }
      else {
        perror("semop - WRUNLCK");
	return -1; 
      }
    }
    break;  

  case LOCK_QUERY:
    semctl_arg.val = 0;
    status = semctl(semid, lockid, GETVAL, semctl_arg);
    if (status<0) {
      perror("semctl - IPC_QUERY");
      return -1;
    }
    else if (status>0) return 1;
    else return 0;
    break;  

  case LOCK_CLEAR:
    semctl_arg.val = 0;
    if ( semctl(semid, lockid, SETVAL, semctl_arg) < 0) {
      perror("semctl - IPC_CLEAR");
    }
    p->lcount = 0;
    break;  

  case LOCK_CLOSE:
    p->lcount = 0;
    break;  

  default:
    print ("Bad lock mode %d\n",mode);
  }
  return (p->lcount);
}

/*
  This is used to lock the loadbuf array used by IceCores which is independent of the card lock
  so we added an extra lock ID after the last possible card MAXICEPICS and auto clear after 10 seconds
*/

int_4 pic_xlock (int_4 mode) 
{
  int lockid;
  struct sembuf op[2];
  union semun semctl_arg;		/* For semctl functions */
  int timeout = 2*100;			/* 2 sec auto clear */

  if (semid<0) {
    if ((semid = semget(semkey, MAXICEPICS, 0666 | IPC_CREAT)) < 0) {
      perror("semget - IPC_CREAT");
    }
  }
  lockid = MAXICEPICS-1;
  
  if (mode>0) {				/* alloc */
    semctl_arg.val = 0;
    while (timeout>0 && semctl(semid, lockid, GETVAL, semctl_arg)!=0) {
      udelay(10000); timeout--; 
    }
    if (timeout<=0) {
      semctl_arg.val = 0;
      semctl(semid, lockid, SETVAL, semctl_arg);	/* clear */
      printf("xLock cleared prior to exclusive lock attempt\n");
    }
    op[0].sem_num = op[1].sem_num = lockid;
    op[0].sem_op=0; op[1].sem_op  = SEMVMX;
    op[0].sem_flg = op[1].sem_flg = 0;
    while (semop(semid, &op[0], 2) < 0) {
      if (errno == EINTR);  /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EAGAIN) { udelay(10000); timeout--; } /* try later */
      else { perror("semop - WRLCK id=X"); return -1; }	/* shouldn't get here */
      semctl_arg.val = 0;
      if (timeout<=0) semctl(semid, lockid, SETVAL, semctl_arg);	/* clear */
    }
  } else {				/* free */
    op[0].sem_num = lockid;
    op[0].sem_op  = -1*SEMVMX;
    op[0].sem_flg = 0;
    while (semop(semid, &op[0], 1) < 0) {
      if (errno == EINTR);    /* semop int by a signal (probably SIGCHLD) */
      else if (errno == EINVAL) {  /* Lock already taken from us */
	printf("Lock cleared prior to exclusive lock release attempt\n");
	break;	/* out of while */
      }
      else {
        perror("semop - WRUNLCK");
	return -1; 
      }
    }
  }
  return 0;
}
#endif

/*
    Memory Utilities
*/

#if _VMS
char *getpage(int_4 size)
{
  int seclen, status, passaddr[2];
  seclen = (size + 511) / 512;
  status = sys$expreg(seclen, passaddr, 0, 0);
  if (!status || passaddr[1]-passaddr[0]+1 < size) {
    print("Mem alloc of %d bytes failed\n", size);
    lib$signal(status); return (0);
  }
  return ((char *) passaddr[0]); 
}

void freepage(char *addr, int_4 size)
{
  int seclen, status, passaddr[2];
  passaddr[0] = (int) addr;
  passaddr[1] = passaddr[0] + size - 1;
  status = sys$deltva(passaddr, 0, 0);
  if (!status) lib$signal(status);
}
 
#endif

#ifdef _XMIDAS
/*
    Fortran interface code
*/

#if _VMS
#define STRI	VAXstr *str
#define STRO	str->c
#else
#define STRI	char *str
#define STRO	str
#endif

#if _UNIX
#define m$pic_open     m$pic_open_
#define m$pic_close    m$pic_close_
#define m$pic_sniff    m$pic_sniff_
#define m$pic_detect   m$pic_detect_
#define m$pic_reset    m$pic_reset_
#define m$pic_test     m$pic_test_
#define m$pic_read     m$pic_read_
#define m$pic_write    m$pic_write_
#define m$pic_writem   m$pic_writem_
#define m$pic_wapp     m$pic_wapp_
#define m$pic_rapp     m$pic_rapp_
#define m$pic_rfifo    m$pic_rfifo_
#define m$pic_wfifo    m$pic_wfifo_
#define m$pic_file     m$pic_file_
#define m$pic_map      m$pic_map_
#define m$pic_mapmem   m$pic_mapmem_
#define m$pic_mapfile  m$pic_mapfile_
#define m$pic_getkey   m$pic_getkey_
#define m$pic_setkey   m$pic_setkey_
#define m$pic_getkeytype  m$pic_getkeytype_
#define m$pic_getkeysize  m$pic_getkeysize_
#define m$pic_loadfc   m$pic_loadfc_
#define m$pic_loadfile m$pic_loadfile_
#define m$pic_dma      m$pic_dma_
#define m$pic_dmasetup m$pic_dmasetup_
#define m$pic_tuner_freq m$pic_tuner_freq_
#define m$pic_tuner_dec m$pic_tuner_dec_
#define m$pic_tuner_ovsr m$pic_tuner_ovsr_
#define m$pic_dmadump  m$pic_dmadump_
#define m$pic_dmafunc  m$pic_dmafunc_
#define m$pic_dmafuncx m$pic_dmafuncx_
#define m$pic_dmastat  m$pic_dmastat_
#define m$pic_sport    m$pic_sport_
#define m$pic_lport    m$pic_lport_
#define m$pic_mport    m$pic_mport_
#define m$pic_tport    m$pic_tport_
#define m$pic_ioport   m$pic_ioport_
#define m$pic_timer    m$pic_timer_
#define m$pic_timecode m$pic_timecode_
#define m$pic_tc       m$pic_tc_
#define m$pic_nvram    m$pic_nvram_
#define m$pic_spec     m$pic_spec_
#define m$pic_name2key m$pic_name2key_
#define m$pic_name2offset m$pic_name2offset_
#define m$pic_prettyprint m$pic_prettyprint_
#define m$pic_str2ip   m$pic_str2ip_
#endif

int_4 m$pic_open  (PICSTRUCT *p, STRI, int_4 *pbreak, int_4 *flags) { 
 return pic_open (p, STRO, pbreak, *flags); } 

int_4 m$pic_close (PICSTRUCT *p) { 
 return pic_close (p); }

int_4 m$pic_reset (PICSTRUCT *p, int_4 *mode) { 
 return pic_reset (p, *mode); }

int_4 m$pic_sniff (PICSTRUCT *p, int_4 *mode) { 
 return pic_sniff (p, *mode); }

int_4 m$pic_detect (int_4 *flags) { 
 return pic_detect (*flags); }

int_4 m$pic_test  (PICSTRUCT *p, int_4 *test, int_4 *flags) { 
 return pic_test (p, *test, *flags); }

int_4 m$pic_read  (PICSTRUCT *p, int_4 *offset, int_4 *data) { 
 return pic_read  (p, *offset, data); }

int_4 m$pic_write (PICSTRUCT *p, int_4 *offset, int_4 *data) { 
 return pic_write (p, *offset, data); }

int_4 m$pic_writem (PICSTRUCT *p, int_4 *offset, int_4 *data, int_4 *mask) { 
 return pic_writem (p, *offset, data, *mask); }

int_4 m$pic_rapp (PICSTRUCT *p, int_4 *offset, int_4 *data, int_4 *flag) { 
 return pic_rapp (p, *offset, data, *flag); }

int_4 m$pic_wapp (PICSTRUCT *p, int_4 *offset, int_4 *data, int_4 *flag) { 
 return pic_wapp (p, *offset, data, *flag); }

int_4 m$pic_rfifo (PICSTRUCT *p, int_4 *data, int_4 *bytes, int_4 *flags) { 
 return pic_rfifo  (p, data, *bytes, *flags); }

int_4 m$pic_wfifo (PICSTRUCT *p, int_4 *data, int_4 *bytes, int_4 *flags) { 
 return pic_wfifo  (p, data, *bytes, *flags); }

int_4 m$pic_ioport (PICSTRUCT *p, int_4 *ptype, int_4 *pindex, int_4 *dmac, 
			int_4 *dir, int_4 *bits, int_4 *rate, real_8 *fcny, 
			int_4 *dec, int_4 *gain, int_4 *flags) { 
 return pic_ioport (p, *ptype, *pindex, *dmac, *dir, *bits, *rate, *fcny, 
			*dec, *gain, *flags); }

real_8 m$pic_tuner_freq (PICSTRUCT *p, real_8 *fcny, int_4 *flags) {
 return pic_tuner_freq_ (p, *fcny, -1, *flags); }

int_4 m$pic_tuner_dec (PICSTRUCT *p, int_4 *dec, int_4 *port, int_4 *flags) {
 return pic_tuner_dec (p, *dec, *port, *flags); }

int_4 m$pic_tuner_ovsr (PICSTRUCT *p,int_4 *factor,int_4 *port,int_4 *flags) { 
 return pic_tuner_ovsr (p, *factor, *port, *flags); }

int_4 m$pic_file (PICSTRUCT *p, HEADER *hcb, 
			int_4 *start, int_4 *count, int_4 *flags) { 
 return pic_file (p, hcb, start, count, *flags); } 

int_4 m$pic_mapfile (PICSTRUCT *p, DMAMAP *map, HEADER *hcb, int_4 *mode) { 
 return pic_mapfile (p, map, hcb, *mode); } 

int_4 m$pic_map (PICSTRUCT *p, int_4 **vaddr, int_4 *paddr, int_4 *psize, int_4 *mode) { 
 return pic_map (p, vaddr, paddr, *psize, *mode); } 

int_4 m$pic_mapmem (PICSTRUCT *p, DMAMAP *map, int_4 *psize, int_4 *mode) { 
 return pic_mapmem (p, map, *psize, *mode); } 

int_4 m$pic_dma (PICSTRUCT *p, int_4 *dmac, int_4 *dir, int_4 *vaddr, 
		int_4 *start, int_4 *count, int_4 *block, int_4 *flags) { 
 return pic_dma (p, *dmac, *dir, vaddr, *start, *count, *block, *flags); }

int_4 m$pic_dmasetup (PICSTRUCT *p, int_4 *dmac, int_4 *dir, DMAMAP *map, int_4 *block, int_4 *flags) { 
 return pic_dmasetup (p, *dmac, *dir, map, *block, *flags); }

int_4 m$pic_setkey (PICSTRUCT *p, int_4 *dmac, int_4 *key, void *vval, int_4 *len) {
 return pic_setkey (p, *dmac, *key, vval, *len); }

int_4 m$pic_getkey (PICSTRUCT *p, int_4 *dmac, int_4 *key, void *vval, int_4 *len) {
 return pic_getkey (p, *dmac, *key, vval, *len); }

int_4 m$pic_getkeytype (PICSTRUCT *p, int_4 *key) {
 return (int_4)pic_getkeytype (p, *key); }

int_4 m$pic_getkeysize (PICSTRUCT *p, int_4 *key) {
 return pic_getkeysize (p, *key); }

int_4 m$pic_dmafunc (PICSTRUCT *p, int_4 *dmac, int_4 *flags) {
 return pic_dmafunc (p, *dmac, *flags); } 

#if _PFC
int_4 m$pic_dmafuncx (PICSTRUCT *p, int_4 *dmac, int_4 *flags) {
 return pic_dmafunc (p, *dmac, *flags); } 
#else
int_8 m$pic_dmafuncx (PICSTRUCT *p, int_4 *dmac, int_4 *flags) {
 return pic_dmafuncx (p, *dmac, *flags); } 
#endif

int_4 m$pic_dmadump (PICSTRUCT *p, int_4 *dmac, int_4 *flags) {
 return pic_dmadump (p, *dmac, *flags); } 

int_4 m$pic_dmastat (PICSTRUCT *p, int_4 *dmac, int_4 *index, int_4 *cycle) {
 return pic_dmastat (p, *dmac, index, cycle); } 

int_4 m$pic_loadfc (PICSTRUCT *p, int_4 *port, int_2 *data, int_4 *ndata, int_4 *flags) {
 return pic_loadfc (p, *port, data, *ndata, *flags); } 

int_4 m$pic_loadfile  (PICSTRUCT *p, STRI, int_4 *flags) { 
 return pic_loadfile (p, STRO, *flags); } 

int_4 m$pic_timer  (PICSTRUCT *p, int_4 *port, int_4 *freq) { 
 return pic_timer (p, *port, *freq); }

int_4 m$pic_timecode (PICSTRUCT *p, int_4 *dmac, STRI, real_8 *offset, 
			real_8 *delta, int_4 *dec, int_4 *bits, int_2 *buf, real_8 *soy, real_8 *fsoy) { 
 return pic_timecode (p, *dmac, STRO, *offset, *delta, *dec, *bits, buf, soy, fsoy); }

int_4 m$pic_tc (PICSTRUCT *p, int_4 *dmac, real_8 *offset, real_8 *delta, 
			real_8 *soy, real_8 *fsoy, int_4 *flags) { 
 return pic_tc (p, *dmac, offset, delta, soy, fsoy, *flags); }

void m$pic_nvram  (PICSTRUCT *p, STRI, int_4 *flags) { 
 pic_nvram (p, STRO, *flags); } 

void m$pic_prettyprint (PICSTRUCT* p) {
 pic_prettyprint(p); }

int_4 m$pic_spec (PICSTRUCT *p, 
  int_4 *data, int_4 *nstart, int_4 *ndata, int_4 *nblock, int_4 *Ntotal,
  float *Maxsrate, int_4 *Maxsjump, float *Maxbrate, int_4 *Maxbjump, int_4 *flags) {
  return pic_spec (p, data, *nstart, *ndata, *nblock, Ntotal,
			Maxsrate, Maxsjump, Maxbrate, Maxbjump, *flags); 
}

int_4 m$pic_name2key (PICSTRUCT *p, STRI) { 
 return pic_name2key (p, STRO); } 

int_4 m$pic_name2offset  (PICSTRUCT *p, STRI, int_4 *flags) { 
 return pic_name2offset (p, STRO, *flags); } 

int_4 m$pic_str2ip  (PICSTRUCT *p, STRI) { 
 return pic_str2ip (p, STRO); } 

#endif

#if _LINX

#include "ramlib.c"

int_4 getrampagesize() {
  RAMSTRUCT *rs = getramstruct();
  return (rs->ram_alloc>0)? (rs->ram_page<<10) : 0x00001000;
}

/*************************************************************************/
/*  LINUX interface routines for initializing MM bus communications      */
/*************************************************************************/


#include <sys/mman.h>

char *picmopenmap (int_8 base, int_8 size, char *devn)
{
  char *addr;
  char devname[16];
  int mem_fd,poff,poffs;
  size_t pages = (size_t)size;
  off_t page = (off_t)base;
  off_t pagemask = getrampagesize()-1;

  /* construct the backing device name */
  if (devn[0]==0) strcpy(devname,"/dev/pmem");
  else { strcpy(devname,"/dev/"); strcat(devname,devn); }

  /* the O_NONBLOCK is to distinguish this open inside the driver */
  if ((mem_fd = open(devname,MMFLAGS|O_NONBLOCK)) < 0) {
    print("MOPENMAP: cant open %s\n",devname);
    perror ("mopenmap open");
    return (0);
  }

  /* page align */
  poff  = page & pagemask;
  page  = page ^ poff;
  pages = pages + poff;
  pages = pages + pagemask;
  poffs = pages & pagemask;
  pages = pages ^ poffs;

  addr = (char *) mmap( (void *)0, pages, MMPROTS, MAP_SHARED, mem_fd, page );

#if DEBUG
  print("Page %d %016llx Pages %d %08x Addr=%zx\n",sizeof(page),page,sizeof(pages),pages,addr);
#endif

  if (addr == (char *)-1) {
    print("Page %d %016llx Pages %d %08x Addr=%zx\n",sizeof(page),page,sizeof(pages),pages,addr);
    perror ("MOPENMAP mmap");
    return (0);
  }

  if ( close (mem_fd) != 0) 
    perror("MOPENMAP close fd");

  /* report address of interest - not page aligned address */
  addr = addr + poff;

  return (addr);

}

void picmclosemap (int_8 base, int_8 size, char *addr)
{
  int poff,poffs;
  size_t pages = (size_t)size;
  off_t page = (off_t)base;
  off_t pagemask = getrampagesize()-1;

  /* page align */
  poff  = page & pagemask;
  page  = page ^ poff;
  pages = pages + poff;
  pages = pages + pagemask;
  poffs = pages & pagemask;
  pages = pages ^ poffs;

  addr = addr - poff;

  if ( munmap ((void *)addr, pages) != 0) 
    perror("MCLOSEMAP munmap");
}

#endif

/*
  These interfaces read and write 1-4 consecutive bytes on a JTAG port using
  the ICE bscanif module on the recieve FPGA.  The WRIOB function runs 
  code on the ICEPIC base card that uses the USER2 jtag command to implement
  a one byte page write, one byte address write, 1-4 byte data write, and 
  1-4 byte data read (at the current page and address++).

  The page is bits [15:8] of addr, address is bits[7:0] of addr.

  If bit 15 of the addr is set, the operation is sent to the SPI interface,
  otherwise the operation is sent to the byte parallel interface.

  The address and data are bit-reversed for use by the bscanif SPI interface

  The port argument is 1=IOM1, 2=IOM2, 7=PM1, 8=PM2, 9=IOM3.
*/
int_4 pic_jpmemrd (PICSTRUCT *p, int_4 port, int_4 addr, int_4 len) {
  int_4 spi = (port<=2||port==9) && ((addr&0x8000)!=0);
  int_4 jvm = (port<=2||port==9) && ((addr&0xFF000000)==0xFD000000);
  int_4 cor = (port<=2||port==9) && ((addr&0x000F0000)==IOM_CORE_OFFSET);
  int_4 data=0,datab,i,sel=getJTAGsel(p,port); 
  if (jvm) {
    sel |= 0x0100;
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|addr,2);
    data = pic_acmd (p,ADON_RDIOB,sel|PPC_BDATA,0,len);
  } else if (cor) {
    data = pic_acmd (p,ADON_RDIOB,sel|0x4000|(addr&0x3FFF),0,len);  /* page=0x4* addr=** == core space */
  } else if (spi) {
    if (addr&0x4000) {
      pic_acmd (p,ADON_WRIOB,sel|(addr&0xFF00)|0x00,bitrev1(addr),1); /* address - then read with auto increment */
      for (i=0; i<len; i++) data |= (bitrev1(pic_acmd(p,ADON_RDIOB,sel|(addr&0xFF00)|bitrev1(0xA0),0,1)) << (i*8));
    } else {
      data = pic_acmd (p,ADON_RDIOB,bitrev1(sel|addr),0,len); 
      data = (len==4)? bitrev4(data) : (len==2)? bitrev2(data) : bitrevN(data,len);
    }
  } else {
    data = pic_acmd (p,ADON_RDIOB,sel|addr,0,len);
  }
  return data;
}

int_4 pic_jpmemwr (PICSTRUCT *p, int_4 port, int_4 addr, int_4 data, int_4 len) {
  int_4 spi = (port<=2||port==9) && ((addr&0x8000)!=0);
  int_4 jvm = (port<=2||port==9) && ((addr&0xFF000000)==0xFD000000);
  int_4 cor = (port<=2||port==9) && ((addr&0x000F0000)==IOM_CORE_OFFSET);
  int_4 rfxd = (port<=2) && (p->mtype[port-1]==IOMT_RFXD);
  int_4 sel = getJTAGsel(p,port); 
  /* RFXD use for debug and setting up internal variables */
  if (rfxd && ((addr & 0xff) >= 0x80) && ((addr & 0xff) <= 0x8f) ) {
    print("Entered pic_jpmemwr: 0x%8.8x \n",addr);
    rfxd_set_vio (p, port, addr, data, len);
    return data;
  }
  if (jvm) {
    sel |= 0x0100;
    pic_acmd (p,ADON_WRIOB,sel|PPC_BADDR_WR,PPC_BADDR_DSOCM|addr,2);
    pic_acmd (p,ADON_WRIOB,sel|PPC_BDATA,data,4);
  } else if (cor) {
    pic_acmd (p,ADON_WRIOB,sel|0x4000|(addr&0x3FFF),data,4);	/* page=0x4* addr=** == core space */
  } else if (spi) {  
    if (addr&0x4000) { /* si5326 mode a=00 d=addr, a=40 d=wdata */
      pic_acmd (p,ADON_WRIOB,sel|(addr&0xFF00)|0x00,bitrev1(addr),1); /* address - then write with auto increment */
      for (;len>0; len--,addr+=1,data>>=8) pic_acmd (p,ADON_WRIOB,sel|(addr&0xFF00)|bitrev1(0x60),bitrev1(data),1);
    } else {
      data = (len==4)? bitrev4(data) : (len==2)? bitrev2(data) : bitrevN(data,len);
      pic_acmd (p,ADON_WRIOB,sel|bitrev1(addr),data,len);      
    }
  } else {
    if (rfxd) len=1; /* RDH kludge */
    pic_acmd (p,ADON_WRIOB,sel|addr,data,len); 	
  }
  return data;
}

int_4 pic_jpmemmsk (PICSTRUCT *p, int_4 port, int_4 addr, int_4 dand, int_4 dxor, int_4 len) {
  int_4 data = pic_jpmemrd (p,port,addr,len);
  data = (data&dand)^dxor;
  pic_jpmemwr (p,port,addr,data,len);
  return data;
}

/* Simple non-optimized CRC routines for data integrity checks  */

/* crc16_le() - Bitwise little-endian CCITT 16-bit CRC            */
int_u2 crc16_le(int_u2 crc, void *pbuf, size_t len)
{
    unsigned char *p = (unsigned char *) pbuf;
    int i;
    while (len--) {
      crc ^= *p++;
      for (i = 0; i < 8; i++)
        crc = (crc >> 1) ^ ((crc & 0x0001) ? CRC16POLY_LE : 0);
    }
    return crc;
}

/* crc16_be() - Bitwise big-endian CCITT 16-bit CRC  */
int_u2 crc16_be(int_u2 crc, void *pbuf, size_t len)
{
    unsigned char *p = (unsigned char *) pbuf;
    int i;
    while (len--) {
      crc ^= *p++ << 8;
      for (i = 0; i < 8; i++)
        crc = (crc << 1) ^ ((crc & 0x8000) ? CRC16POLY_BE : 0);
    }
    return crc;
}

/* crc32_le() - Bitwise little-endian Ethernet CRC32   */
int_u4 crc32_le(int_u4 crc, void *pbuf, size_t len)
{
    unsigned char *p = (unsigned char *) pbuf;
    int i;
    while (len--) {
      crc ^= *p++;
      for (i = 0; i < 8; i++)
        crc = (crc >> 1) ^ ((crc & 0x00000001) ? CRC32POLY_LE : 0);
    }
    return crc;
}

/* crc32_be() - Bitwise big-endian Ethernet CRC32      */
int_u4 crc32_be(int_u4 crc, void *pbuf, size_t len)
{
    unsigned char *p = (unsigned char *) pbuf;
    int i;
    while (len--) {
      crc ^= *p++ << 24;
      for (i = 0; i < 8; i++)
        crc = (crc << 1) ^ ((crc & 0x80000000) ? CRC32POLY_BE : 0);
    }
    return crc;
}

/* dummy calls to remove compiler warnings */
int_4 dummycalls() {
  pagealloc(0,(char**)0,0);
  return 0;
}

static int_4 setup_pkt(PICSTRUCT *p, int_4 dmac, NIC_PORT_STRUCT *np) 
{
  printf("NIC myip=%08x mcip=%08x\n",np->myip,np->mcip);
  return 0;
}

/* compression libraries */
#include "huffman.c"

int_4 pic_compress (int_u1 *ibuf, int_4 ilen, int_u1 *obuf, int_4 olen, int_4 dir)
{
  if (dir>0) olen = Huffman_Compress (ibuf,obuf,(int_u4)ilen);
  else Huffman_Uncompress (ibuf,obuf,(int_u4)ilen,(int_u4)olen);
  return olen;
}

/* FPGA Core libraries */
#include "fpgalib.c"

/* Parts module libraries */
#include "partlib.c"

/* I/O module libraries */
#include "pktlib.c"
#include "iomlib.c"

/* Rumel  module libraries */
#include "rumlib.c"

/* LGS module libraries */
#include "lgslib.c"

/* networking libraries */
#include "niolib.c"

/* fileIO libraries */
#include "fiolib.c"

/* ena=1 enable, ena=0 disable 
 - here for implicit declarations of enable_xxx routines */
int_4 pic_enable_module (PICSTRUCT *p, int_4 mport, int_4 ena) {
  int_4 i,mtype,mrev,mtypeo;
  int_4 retval = 0;

  if (p->xchn>0) doXcvrPrep(p,ena);

  for (i=1; i<3; i++) {
    if (i==1 && mport==2) continue;
    if (i==2 && mport==1) continue;
    if (p->flags&FLG_LOOP) continue;
    mtype  = p->mtype[i-1];
    mrev   = p->mrev[i-1];
    mtypeo = abs(mtype); /* output form of I or O module */

    if (mtypeo==IOMT_DXTGSDDS) retval= pic_enable_tgsxdr1(p, i,ena);
    if (mtypeo==IOMT_DXTGVITA) retval = pic_enable_tgvitar1(p, i,ena);
    if (mtypeo==IOMT_DXFLZR && mrev==1) pic_enable_flzrxdr1 (p, i,ena);
    if (mtype==IOMT_LB2D && mrev<3) lb2d_enable (p, i,ena);
    if (mtype==IOMT_LB2D && mrev==3) lb2dm3_enable (p, i,ena);
    if (mtype==IOMT_A2D && mrev==9)  pic_enable_a2dr9 (p, i,ena);
    if (mtype==IOMT_A2D && mrev==10) pic_enable_a2dr10 (p, i,ena);
    if (mtype==IOMT_A2D && mrev==11) pic_enable_a2dr11 (p, 2,ena);
    if (mtype==IOMT_A2D && mrev==13) pic_enable_a2dr13 (p, i,ena);
    if (mtype==IOMT_A2D && mrev==14) a2dm14_enable (p,i,ena);
    if (mtype==IOMT_A2D && mrev>=15 && mrev<=18)  a2dm1x_enable (p,i,ena);
    if (mtype==IOMT_A2D && mrev==20) a2dm20_enable (p,i,ena);
    if (mtype==IOMT_D2A && mrev<8) pic_reset_d2a (p, i);
    if (mtype==IOMT_D2A && mrev==9) pic_enable_d2ar9 (p, i,ena);
    if (mtype==IOMT_D2AWG && mrev==3) d2awgm3_enable (p, i,ena);
    if (mtype==IOMT_D2AWG && mrev!=3) pic_enable_d2awg (p, i,ena);
    if (mtype==IOMT_D2RF)  d2rf_enable (p,i,ena);
    if (mtypeo==IOMT_D2CDR && ena==0) pic_reset_cdr2d (p, i);
    if (mtypeo==IOMT_DXUDP && ena==0) pic_reset_sdds (p, i);
    if (mtype==IOMT_D2PSE && ena==0)  pic_reset_d2pse (p, i);
    if (mtypeo==IOMT_DXNF && mrev==5 && ena==0)  pic_reset_ngcfiber(p, i);  /* PIC5 */
    if (mtypeo==IOMT_DXSNT && mrev<=1 && ena==0) pic_reset_sonetr1 (p, i);
    if (mtypeo==IOMT_DXSNT && mrev==2 && ena==0) pic_reset_sonetr2 (p, i); /* seem to need both sides */
    if (mtypeo==IOMT_DXSNT && mrev==4) pic_enable_sonetr4 (p, i,ena);
    if (mtypeo==IOMT_DXSNT && mrev==5) pic_enable_sonetr5 (p, i,ena);
    if (mtypeo==IOMT_DXSNT && mrev==6) pic_enable_sonetr6 (p, i,ena);
    if (mtypeo==IOMT_DXSNT && mrev==7) pic_enable_sonetr7 (p, i,ena);
    if (mtypeo==IOMT_DXSDDS) sdds_enable(p, i,ena);
    if (mtypeo==IOMT_D2ES) pic_enable_esxdr1 (p,i,ena);
    if (mtypeo==IOMT_DXUFLZ) pic_enable_uflzxdr1 (p,i,ena);
    if (mtypeo==IOMT_DXQSFP) pic_enable_qsfp (p, i,ena);
  }
  return retval;
}

int_4 setup_duc (PICSTRUCT *p, int port, int ic, int rate, double ratio, int gain, double fcny, int sig) {
  HALO *plan;
  int stat,mci=ic+1; char csig[8];
  double fs=rate, fsym=fs/ratio, freq=fcny*rate, dgain=gain, fbwf=0.85;
  if (sig==0) sig = p->tsig;
  strncpy (csig,(char*)(&sig),4); csig[4]=0;
  vprint("DUC csig=%.4s port=%d chan=%d rate=%d ratio=%f gain=%d fcny=%f\n", csig,port,mci,rate,ratio,gain,fcny);
  plan = core_alloc(csig,"ICEP",p);
  p->plan[ic] = plan;
  core_defconfig(plan);
  stat = core_config(plan,6,mci);	/* mcid */
  stat = core_setkey(plan,"D:FS",&fs,8);
  stat = core_setkey(plan,"D:FSYM",&fsym,8);
  stat = core_setkey(plan,"D:FREQ",&freq,8);
  stat = core_setkey(plan,"D:GAIN",&dgain,8);
  stat = core_setkey(plan,"D:FBWF",&fbwf,8);
  stat = core_open(plan);
  return 0;
}

HALO* planForChan (PICSTRUCT *p, int_4 chan) {
  HALO *plan; int_4 i=0;
  if (chan>0 && chan<=MAX_MCS) i=chan-1;
  if (p->plan[i]==0) i=0;
  plan = (HALO*)p->plan[i];
  if (plan==NULL) { print("Error finding plan for chan=%d\n",chan); return 0; }
  plan->ph->chan=chan;
  return plan;
}

int_4 set_awg_freq (PICSTRUCT *p, int_4 port, int_4 chan, double fcny) {
  HALO *plan = planForChan(p,chan);
  return core_setkey(plan,"D:FCNY",&fcny,8);
}

int_4 set_awg_gain (PICSTRUCT *p, int_4 port, int_4 chan, int_4 gain) {
  HALO *plan;
  int_4 stat=0,i,j,pmod=port/10;
  double dgain = (double)gain;
  port -= pmod*10;
  if (chan==0) {			/* after combiner */
    i = (int_4) (pow(2.0,dgain/6)*0x100);
    j = core_addr(port);
    pic_wpb (p,pmod,j|MCORE_SCL,i);
  } else if (chan==99) {		/* noise channel */
    dgain -= 18;	/* shift offset */
    i = (int_4) (pow(2.0,dgain/6)*0x10000);
    j = core_addr(port);
    pic_wpb (p,pmod,j|MCORE_NFSCL,i);
  } else {				/* individual channel */
    plan = planForChan(p,chan);
    i=core_setkey(plan,"D:GAIN",&dgain,8);
  }
  return stat;
}

int_4 set_awg_mute (PICSTRUCT *p, int_4 port, int_4 chan, int_4 mute) {
  HALO *plan = planForChan(p,chan);
  int_4 stat=0,i,j,pmod=port/10;
  port -= pmod*10;
  plan->ph->chan=chan;
  i=core_setkey(plan,"L:MUTE",&mute,4);
  return stat;
}

int_4 pic_query (PICSTRUCT *p, int_4 type, int_4 info, void *buf, int_4 len) {
  int_4 status=-1;
  int_4 mport = (info>0)? info : p->pindex;
  if (type==QUERY_FPLAN && p->mtype[mport-1]==IOMT_LB2D) {
#if NEWSTUFF
    status = lb2dm3_fplan(p,mport,(real_8*)buf,len);
#endif
  }
  if (status<0) printf ("Unhandled query type=%d info=%d port=%d\n",type,info,mport);
  return status;
}

int_4 pic_nio_stat (PICSTRUCT *p, int_4 port, int_4 test)
{
  if (test<=0) dump_qsfp(p,port);
  return test;
}

