/*
  routines for calling FPGA functions as an attached processor 
*/

#define FPGA_DMASZ_O 0x0100000
#define FPGA_DMASZ_I 0x0200000

int_4 FPGA_open (void *plan, void *config) {
  HALO *halo = (HALO*)plan; PICSTRUCT *p;
  int_4 status,ipm,opm,ipo,opo,icn,ocn,mc,coreio;

  if (strncmp(config,"subCore",7)==0) { printf("subCore call shouldnt happen\n"); return 0; }	/* already done by multicore wrapper */
  if (strncmp(config,"libCore",7)==0) return 0;	/* already done by icelib setup */

  p = halo->ph = (PICSTRUCT *)malloc( sizeof(PICSTRUCT) ); 
  p->halo = halo;

  status = pic_open (p, config, 0, 0);
  if (status<0) { printf("Err opening device=%s\n",(char*)config); return -1; }

  if (strncmp(config,"VHS-CORE",8)==0) return 0;	/* all done by icelib setup */

  /* config has the form IPORT=PM0CORE1,OPORT=PM1CORE1 */
  ipo=findflag("IPORT",p->config); ipm=p->config[ipo+8]-'0'; icn=p->config[ipo+13]-'0'; if (ipm>3) ipm=-1;
  opo=findflag("OPORT",p->config); opm=p->config[opo+8]-'0'; ocn=p->config[opo+13]-'0'; if (opm>3) opm=-1;

  coreio = findintflagdef("COREIO",p->config,1);
  if (coreio==0) {
    p->pmi    = opm;
    p->pindex = ocn;
    p->dmaci  = p->dmaco = (opm<0)? -ocn : -(opm*10+ocn);
    return 0;
  }

  p->pmi=ipm; p->config[ipo]='X';
  status = pic_ioport (p, IOPT_CORE, icn, -1, 1, -16, 100000000, 0.0, 1, 0, FLG_LOOP);
  if (status<0) { printf("Err opening output channel dmao=%d\n",status); return -1; }
  p->dmaco = status; p->config[ipo]='I';
  status = pic_setkeyl (p, p->dmaco,KEY_CORE+0,0x8803);

  p->pmi=opm; p->config[opo]='X';
  status = pic_ioport (p, IOPT_CORE, ocn, -1, -1, -16, 100000000, 0.0, 1, 0, 0); 
  if (status<0) { printf("Err opening input channel dmai=%d\n",status); return -1; }
  p->dmaci = status; p->config[opo]='O';
  status = pic_setkeyl (p, p->dmaci,KEY_CORE+0,0x0);

  status = pic_mapmem (p, &p->mapi, FPGA_DMASZ_I, 1);
  if (status<0) { printf("Err mapping input buffer of 0x%x bytes\n",FPGA_DMASZ_I); return -1; }
  status = pic_mapmem (p, &p->mapo, FPGA_DMASZ_O, 1);
  if (status<0) { printf("Err mapping output buffer of 0x%x bytes\n",FPGA_DMASZ_O); return -1; }

  status = pic_dmasetup (p, p->dmaci, -1, &p->mapi, -1, 0);
  status = pic_dmasetup (p, p->dmaco,  1, &p->mapo, -1, 0);

  /* for sys_wr and wpb routines */
  p->pmi    = opm;
  p->pindex = ocn;

  v2print("FPGA_open config=%s dmaco=%d:%d:%d dmaci=%d:%d:%d\n",p->config,ipm,icn,p->dmaco,opm,ocn,p->dmaci);
  return 0; 
}

int_4 FPGA_syswr (void *plan, int_4 addr, int_4 value) { 
  HALO *pt = (HALO*)plan; PICSTRUCT *p = pt->ph;
  if (p==NULL || p->state<=0) return -1;
  v2print("FPGA_syswr dmac=%d chan=%d addr=%08x value=%08x\n",p->dmaci,p->chan,addr,value);
  if (p->devno==-3) return (*pt->syswr) (plan,addr,value);
  return pic_setkey (p,p->dmaci,KEY_CORE+addr,&value,4);
}

int_4 FPGA_syswrblk (void *plan, int_4 addr, int_4 *value, int_4 size) { 
  PICSTRUCT *p = ((HALO*)plan)->ph;
  if (p==NULL || p->state<=0) return -1;
  v2print("FPGA_syswrblk dmac=%d chan=%d addr=%08x len=%08x\n",p->dmaci,p->chan,addr,size);
  return pic_setkey (p,p->dmaci,KEY_CORE+addr,value,size*4);
}

int_4 FPGA_sysrd (void *plan, int_4 addr) { 
  HALO *pt = (HALO*)plan; PICSTRUCT *p = pt->ph; int_4 value;
  if (p==NULL || p->state<=0) return -1;
  if (p->devno==-3) value = (*pt->sysrd) (plan,addr);
  else value = pic_getkeyl (p,p->dmaci,KEY_CORE+addr);
  v2print("FPGA_sysrd dmac=%d chan=%d addr=%08x value=%08x\n",p->dmaci,p->chan,addr,value);
  return value;
}

int_4 FPGA_lock (void *plan, int_4 mode) { 
  HALO *pt = (HALO*)plan; PICSTRUCT *p = pt->ph; int_4 value;
  if (p->devno==-3) return 0;
  v2print("FPGA_locki mode=%d\n",mode);
  value = pic_xlock (mode);
  v2print("FPGA_locko mode=%d\n",mode);
  return value;
}

typedef struct {
  int_4 len,bib,woff,roff,wboff,rboff,wbpe,rbpe,wper,rper,wok,rok,mtl,mask,usenative,chn,cvt;
  int_1 *b;
} Stream_;

int_4 FPGA_process (void *plan, void *si, void *sip, void *so, void *sop, int_4 hbw) { 
  PICSTRUCT *p = ((HALO*)plan)->ph;
  int_4 status=0,n; Stream_ *s;
  if (p->dmaco>0) {
    if (p->dma[p->dmaco-1].todo==0) status = pic_dmafunc (p, p->dmaco, DMA_ONDEMAND);
    s = (Stream_*)si;
    n = s->bib; if (n+s->roff>s->len) n = s->len-s->roff;
    status = pic_dmaxfer (p, p->dmaco, (int_4*)(s->b+s->roff), n, 0);
    v2print("FPGA dmac=%d bib=%d n=%d len=%d stat=%d\n",p->dmaco,s->bib,n,s->len,status);
    if (status>0) { s->bib -= status; s->roff += status; if (s->roff>=s->len) s->roff -= s->len; }
  }
  if (p->dmaci>0) {
    if (p->dma[p->dmaci-1].todo==0) status = pic_dmafunc (p, p->dmaci, DMA_CONTINUOUS);
    s = (Stream_*)so;
    n = s->len-s->bib; if (n+s->woff>s->len) n = s->len-s->woff;
    status = pic_dmaxfer (p, p->dmaci, (int_4*)(s->b+s->woff), n, FLG_NOWAIT);
    v2print("FPGA dmac=%d bob=%d n=%d len=%d stat=%d\n",p->dmaci,s->bib,n,s->len,status);
    if (status>0) { s->bib += status; s->woff += status; if (s->woff>=s->len) s->woff -= s->len; }
  }
  return status;
}

int_4 FPGA_close (void *plan) {
  PICSTRUCT *p = ((HALO*)plan)->ph;
  int_4 status;
  if (p->dmaci>0) status = pic_dmafunc (p, p->dmaci, DMA_CANCEL);
  if (p->dmaco>0) status = pic_dmafunc (p, p->dmaco, DMA_CANCEL);
  if (p->dmaci>0) status = pic_mapmem (p, &p->mapi, FPGA_DMASZ_I, -1);
  if (p->dmaco>0) status = pic_mapmem (p, &p->mapo, FPGA_DMASZ_O, -1);
  if (p->plan!=0) return 0;	/* dont close here if embedded */
  pic_close (p);
  free((void*)p);
  return 0;
}

int_4 FPGA_wpb (PICSTRUCT *p, int_4 node, int_4 addr, int_4 data) { 
  int csel = (p->pindex==4)? 0x80 : (p->pindex==3)? 0x40 : (p->pindex==2)? 0x20 : 0x10;
  if ((addr&0xFF000000)==0) addr |= (csel<<24);
  return pic_wpb(p,node,addr,data);
}

