package nxm.ice.lib;

import nxm.sys.inc.PacketHandler;
import nxm.sys.inc.DataTypes;
import nxm.sys.lib.*;
import nxm.ice.lib.CoreCommon;

/**
  Handler for VITA-49 Radio Transport packets.

  <pre>

  VRT Packet Word Placement

    0 HDR Header (Type:4,C:1,T:1,RR:2,TSI:2,TSF:2,Seq:4,Size:16)
    1 SID Stream ID 
    2 OUID Organizational Unique ID
    3 CID  Information Class : Packet Class
    4 TSIS TimeStamp - Integer Seconds
  6:5 TSFS TimeStamp - Fractional Seconds
    7 TLR Trailer
    
    X Data Payload (X=360 qwords for 1500MTU)

    Packet qword 7 moved after payload at transmit

  Header Types:
    0cxs c=Context x=Extension s=WithStreamID
    
    Packet Class (Pack:1,Type:2,Format:5,Resv:2,Size:6)

    Ice PCC = Packet Class Code (PT:1,BE:1,DST:2,DIF:4,DIS:8) = 0x100F for LittleEndian CI

    Data Item Size = (bitsPerSample-1)

    Data Sample Type
    00 Real
    01 Complex Cartesian
    10 Complex Polar

    Data Item Format
    u0000 Fixed Point - u=Unsigned|Signed
    u0eee VRT float w/ e-bit exponent
    01110 IEEE Single Precision Floating Point
    01111 IEEE Double Precision Floating Point

  Information Class


  TimeStamp
    TSI 0=None 1=UTC 2=GPS 3=Other
    TSF 0=None 1=Count 2=pSec 3=FreeRunningCount
  
  </pre>

  @author Jeff Schoen
  @version $Id: VRTPacket.java,v 1.28 2012/10/15 20:32:56 jgs Exp $

*/
public class VRTPacket implements PacketHandler,Cloneable {

  private int O_HDR=0, O_SID=4, O_OUID=8, O_CID=12, O_TSIS=16, O_TSFS=20, O_TAIL=28;

  public static int OUID_ICE=0x104D77, OUID_DIFI=0x6A621E;

  public static int TSI_NONE=0, TSI_UTC=1, TSI_GPS=2, TSI_POSIX=3;
  public static int TSF_NONE=0, TSF_CNT=1, TSF_PSEC=2, TSF_FRC=3;

  public static String flagsList = "TRIM,BIGE,NOID,CTX,NOSID,DIFI";
  public static int FLG_NONE=0x0,FLG_TRIM=0x1,FLG_BIGE=0x2,FLG_NOUID=0x4,FLG_CTX=0x8,FLG_NOSID=0x10,FLG_DIFI=0x20;

  public byte[] buf = new byte[32];
  public byte[] cbuf;

  private int headerOffset=0, headerLength=32, trailerOffset=28, trailerLength=4, fixedBytes=1440;
  private int hdr, sid, ouid, cid, ctx, chdr, size, bits, fbits, fb, tb, count, ipoff, rhdrs, whdrs, flags;
  private int dataLength, totalLength, fdataLength;
  private String format;

  byte hrep=DataTypes.IEEE, drep=DataTypes.EEEI;

  private boolean isInput=false,needsContext=false;
  private double npktoff;

  public VRTPacket() {
    this("SB",0,null);
  }
  public VRTPacket (String format, String variant) {
    this(format,0,variant);
  }
  public VRTPacket (String format, int flags) {
    this(format,flags,null);
  }

  public VRTPacket (String format, int flags, String variant) {

         if (variant==null);
    else if (variant.equals("VRT") ) flags |= 0;
    else if (variant.equals("VRTL")) flags |= FLG_NOSID | FLG_TRIM;
    else if (variant.equals("VRTW")) flags |= FLG_NOSID | FLG_TRIM | FLG_BIGE;
    else if (variant.equals("VRTX")) flags |= FLG_NOUID | FLG_CTX  | FLG_BIGE;
    else if (variant.equals("VRTD")) flags |= FLG_DIFI  | FLG_TRIM | FLG_BIGE;
    else System.out.println("Unknown VRT variant: "+variant);
    this.flags = flags;

    formHeaderBlock();
    setFormat(format);
    setCount(0);
    setSID(1);
    setOUID(((flags&FLG_DIFI)!=0)?OUID_DIFI:OUID_ICE);
    setTC(0.0,0.0);
    setSize(1024);
  }

  private void packL (int off, int data) {
    if (off<0) return;
    Convert.packL(buf,off,data,hrep);
  }
  private void packX (int off, long data) {
    if (off<0) return;
    Convert.packX(buf,off,data,hrep);
  }

  private int unpackL (int off) {
    if (off<0) return 0;
    return Convert.unpackL(buf,off,hrep);
  }
  private long unpackX (int off) {
    if (off<0) return 0;
    return Convert.unpackX(buf,off,hrep);
  }

  private void formHeaderBlock() {
    int n=0;
    O_SID=O_OUID=O_CID=O_TSIS=O_TSFS=O_TAIL=-1;	// invalidate offsets
    hdr=0x00600000;
    O_HDR=n; n+=4;		// locate HDR
    if ((flags&FLG_NOSID)==0) {
      hdr |= 0x10000000;
      O_SID=n; n+=4;		// locate SID
    }
    if ((flags&FLG_NOUID)==0) {
      hdr |= 0x08000000;
      O_OUID=n; n+=4;		// locate OUID
      O_CID=n; n+=4;		// locate CID
    }
    O_TSIS=n; n+=4;		// locate TSIS
    O_TSFS=n; n+=8;		// locate TSFS
    if ((flags&FLG_TRIM)==0) {
      hdr |= 0x04000000;
      O_TAIL=n; n+=4;		// locate TAIL
      trailerLength=4;
      setTrailer(0xe4060000); 
    } else {
      trailerLength=0;
    }
    Convert.packL(buf,0,hdr,hrep);
    headerLength=n;
  }

  public VRTPacket (String format, int size, int flags) {
    this(format,flags);
    this.setSize(size);
  }

  /* init configuration from first packet header */
  private void initFrom (DataFile df) {
    setRep((byte)df.getDataRep().charAt(0));
    if (df.hp!=null) {
      df.hp.seek((long)df.getDataStart());
      df.hp.read (this.buf,headerOffset,headerLength);
      df.hp.seek((long)df.getDataStart());
    } else {
      df.io.seek((long)df.getDataStart());
      df.io.read (this.buf,headerOffset,headerLength);
      df.io.seek((long)df.getDataStart());
    }
    setInternals();
  }

  /* sets the data payload representation */
  public void setFlags (int flags) {
    this.flags = flags;
  }

  /* sets the local parameters to match header */
  private void setInternals () {
    hdr = Convert.unpackL(buf,O_HDR,hrep);
    headerLength=4;		// hdr
    trailerLength=0;
    O_SID=O_OUID=O_CID=O_TSIS=O_TSFS=O_TAIL=-1;
    if ((hdr&0x10000000)!=0) {	// has SID
      O_SID=headerLength; headerLength+=4;
    }
    if ((hdr&0x08000000)!=0) {	// has CID
      O_OUID=headerLength; headerLength+=4;
      O_CID=headerLength; headerLength+=4;
      cid = Convert.unpackL(buf,O_CID,hrep);
      bits = (cid&0x3F)+1;
      if ((cid&0x2000)!=0) bits *= 2;
    }
    if ((hdr&0x00F00000)!=0) {	// has TS
      O_TSIS=headerLength; headerLength+=4;
      O_TSFS=headerLength; headerLength+=8;
    }
    if ((hdr&0x04000000)!=0) {	// has trailer
      O_TAIL=headerLength; headerLength+=4;
      trailerLength=4; 
    }
    trailerOffset = headerLength-trailerLength;
    totalLength = (hdr&0xFFFF)<<2;
    dataLength = totalLength-headerLength;
    fdataLength = fbits*dataLength/bits;
    size = dataLength*8/bits;
  }

  /* sets the data payload representation */
  public void createContextFor (int ctx, int rate, int freq, int gain) {
    int n=0;
    cbuf = new byte[256];
    long lrate = ((long)rate)<<20;	// sample rate in Hertz
    long lfreq = ((long)freq)<<20;	// sample rate in Hertz
    int  lgain = gain<<7;
    int  lform = 0x200003CF;	
    if (ctx==0) return;
    if (ctx==1) ctx = ((flags&FLG_NOUID)!=0)? 0x1010592 : 0x000513;
    int  lctx = CoreCommon.brev4(ctx&0xFFFFC);
    chdr = 0x41600006;
    Convert.packL(cbuf,n*4,chdr,hrep); n++;
    Convert.packL(cbuf,n*4,sid,hrep); n++;
    if (((ctx>>23)&1)==0) {	// has OUID
      chdr = 0x49600008;
      Convert.packL(cbuf,n*4,ouid,hrep); n++;
      Convert.packL(cbuf,n*4,0,hrep); n++;
    }
    n+=3;	// TC fields
    Convert.packL(cbuf,n*4,lctx,hrep); n++;
    if (((ctx>>4)&1)==1) { Convert.packX(cbuf,n*4,lfreq,hrep); n+=2; }
    if (((ctx>>8)&1)==1) { Convert.packL(cbuf,n*4,lgain,hrep); n+=1; }
    if (((ctx>>10)&1)==1) { Convert.packX(cbuf,n*4,lrate,hrep); n+=2; }
    if (((ctx>>16)&1)==1) { Convert.packL(cbuf,n*4,lform,hrep); n+=1; Convert.packL(cbuf,n*4,0,hrep); n+=1; }
    chdr = (chdr&0xFFFFFF00) + n;
    Convert.packL(cbuf,0*4,chdr,hrep);
    this.ctx = ctx;
  }

  public int getNextContext() {
    int len = (chdr&0xFF);
    Convert.packL(cbuf,0,chdr,hrep);
    System.arraycopy(buf,O_TSIS, cbuf,(((ctx>>20)&1)!=0)?8:16, 12);
    int cnt = (chdr>>16)&0xF; cnt++;
    chdr = (chdr&0xFFF0FFFF) | ((cnt&0xF)<<16);
    return len<<2;
  }

  /* sets the data payload representation */
  public void setRep (byte rep) {
    this.drep = rep;
  }
  /* get representation of data payload */
  public byte getRep() {
    return drep;
  }
  /* get header length in bytes */
  public int getHeaderLength() {
    return headerLength;
  }
  /* get header length in bytes */
  public int getDataLength() {
    return dataLength;
  }
  /* get trailer length in bytes */
  public int getTrailerLength() {
    return trailerLength;
  }

  /* sets the data payload size int elements */
  public void setSize (int elem) {
    size = elem;
    setBytes(elem*bits/8);
  }
  /* get Size of data payload in elements */
  public int getSize() {
    return size;
  }

  /* sets the data payload size in bytes */
  public void setBytes (int bytes) {
    size = bytes*8/bits;
    dataLength = bytes;
    fdataLength = fbits*dataLength/bits;
    totalLength = headerLength+dataLength;
    int pktsize=(totalLength+3)>>2;
    hdr = (hdr&0xFFFF0000) | pktsize;
    packL(O_HDR,hdr);
  }
  /* get Size of data payload in bytes */
  public int getBytes() {
    return dataLength;
  }

  /* sets Stream ID */
  public void setTrailer (int trailer) {
    packL(O_TAIL,trailer);
  }

  /* sets Stream ID */
  public void setSID (int sid) {
    if (O_SID>0) packL(O_SID,sid);
    else if (O_CID>0) {
      cid = (cid&0xFFFF) | (sid<<16);
      packL(O_CID,cid);
    }
    this.sid = sid;
  }
  /* gets Stream ID */
  public int getSID() {
    if (O_SID>0) return unpackL(O_SID);
    if (O_CID>0) return unpackL(O_CID)>>16;
    return 0;
  }

  /* sets Organizational Unique ID */
  public void setOUID (int ouid) {
    packL(O_OUID,ouid);
    this.ouid = ouid;
  }
  /* gets Organizational Unique ID */
  public int getOUID() {
    return unpackL(O_OUID);
  }

  /* sets Information Class ID */
  public void setICID (int icid) {
    setCID( (cid&0xFFFF) | (icid<<16) );
  }
  /* gets Information Class ID */
  public int getICID() {
    return (cid>>16)&0xFFFF;
  }

  /* sets Class ID */
  public void setCID (int cid) {
    this.cid = cid;
    packL(O_CID,cid);
  }
  /* gets Class ID */
  public int getCID() {
    return cid;
  }

  /* packet number since start of xfer */
  public void setCount (int count) {
    this.count = count;
    hdr = (hdr&0xFFF0FFFF) | ((count&0xF)<<16);
    packL(O_HDR,hdr);
  }
  /* get the current packet count */
  public int getCount() {
    if (isInput) {
      int hdr = unpackL(O_HDR);
      count = (hdr>>16)&0xF;
    }
    return count;
  }
  /* up the count by one */
  public void upCount() {
    count++;
    setCount(count);
  }

  /* set format/bits from Midas format digraph */
  public void setFormat (String format) {
    if (format.equals(this.format)) return;
    this.format = format;
    if (!DataFile.checkFormat(format,"S#,C#")) System.out.println("Unallowed format="+format);
    char mode=format.charAt(0), type=format.charAt(1);
    int bps = Data.getBPS(type);
    fbits = bits = (bps<0)? -bps : bps*8;
    fb = (bits-1);
    tb = 0x80; // link efficient
    tb |= (mode=='C')? 0x10:0x00;	// complex vs real
         if (type=='F') tb |= 0x0E;
    else if (type=='D') tb |= 0x0F;
    else                tb |= 0x00;
    if ((flags&FLG_BIGE)!=0) tb |= 0x40;	// big endian
    cid = (cid&0xFFFF0000)|(tb<<8)|fb;
    packL(O_CID,cid);
  }

  /* get the current Format */
  public String getFormat() {
    return format;
  }

  /* set number of bits per sample */
  public void setBits (int bits) {
    this.bits = bits;
    int fb = (bits-1);
    cid = (cid&0xFFFFFF00) | (fb&0x3F);
    packL(O_CID,cid);
  }
  /* get the number of bits per sample */
  public int getBits() {
    return bits;
  }

  /* set timecode */
  public void setTC (double wsec, double fsec) {
    int  isec = (int)(wsec+Time.J1950TOJ1970);
    long psec = (long)(fsec*1e12+.5);
    if (psec<0) { isec--; psec+=1e12; }
    if (O_TSIS>0) packL(O_TSIS,isec); 
    if (O_TSFS>0) packX(O_TSFS,psec); 
    hdr = (hdr&0xFF0FFFFF) | (TSI_UTC<<22) | (TSF_PSEC<<20);
    packL(O_HDR,hdr);
  }

  /* get timecode from ICE/VITA formatted header */
  public Time getTC() {
    int  isec = (O_TSIS<0)? 0 : Convert.unpackL(buf,O_TSIS,hrep); 
    long psec = (O_TSFS<0)? 0 : Convert.unpackX(buf,O_TSFS,hrep); 
    double wsec = isec; wsec+=Time.J1970TOJ1950;
    double fsec = psec*1e-12;
    return new Time (wsec,fsec);
  }

/*
  BE Packing Modes
     |       0       |       1       |       2       |       3       |
 16  |F E D C B A 9 8 7 6 5 4 3 2 1 0|F E D C B A 9 8 7 6 5 4 3 2 1 0|
 12  |F E D C B A 9 8 7 6 5 4|F E D C B A 9 8 7 6 5 4|F E D C B A 9 8
  8  |F E D C B A 9 8|F E D C B A 9 8|F E D C B A 9 8|F E D C B A 9 8|
  6  |7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 
  4  |7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|


  LE Packing Modes
     |       3       |       2       |       1       |       0       |
 16  |F E D C B A 9 8 7 6 5 4 3 2 1 0|F E D C B A 9 8 7 6 5 4 3 2 1 0|
 12   B A 9 8 7 6 5 4|F E D C B A 9 8 7 6 5 4|F E D C B A 9 8 7 6 5 4|
  8  |F E D C B A 9 8|F E D C B A 9 8|F E D C B A 9 8|F E D C B A 9 8|
  6   3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|7 6 5 4 3 2|
  4  |7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|7 6 5 4|

*/

  /* pack to different word sizes */
  private void pack (byte[] buf, int boff, int n, int ibits, int obits) {
    int i,j,iadr=0,oadr=0,ibit,obit,ibyt,obyt,diff=ibits-obits; // > 0
    if (drep==DataTypes.IEEE) {
      for (i=0; i<n; i++) {
        for (j=0, iadr=i*ibits, oadr=i*obits; j<obits; j++,iadr++,oadr++) {
          ibyt=iadr>>3; ibit=(~iadr)&0x7; 
          obyt=oadr>>3; obit=(~oadr)&0x7;
          if ((buf[ibyt]&(1<<ibit))==0) buf[obyt]&=~(1<<obit); else buf[obyt]|=(1<<obit); 
        }
      }
    } else {
      for (i=0; i<n; i++) {
        for (j=0, iadr=i*ibits+diff, oadr=i*obits; j<obits; j++,iadr++,oadr++) {
          ibyt=iadr>>3; ibit=iadr&0x7; 
          obyt=oadr>>3; obit=oadr&0x7;
          if ((buf[ibyt]&(1<<ibit))==0) buf[obyt]&=~(1<<obit); else buf[obyt]|=(1<<obit); 
        }
      }
    }
  }

  /* unpack to different word sizes */
  private void unpack (byte[] buf, int boff, int n, int ibits, int obits) {
    int i,j,iadr,oadr,ibit,obit,ibyt,obyt,diff=ibits-obits; // < 0
    if (drep==DataTypes.IEEE) {
      for (i=n; i>0; i--) {
        for (j=0, iadr=i*ibits-1-diff, oadr=i*obits-1; j<obits; j++,iadr--,oadr--) {
          ibyt=iadr>>3; ibit=(~iadr)&0x7;
          obyt=oadr>>3; obit=(~oadr)&0x7;
          if (j+diff<0 || (buf[ibyt]&(1<<ibit))==0) buf[obyt]&=~(1<<obit); else buf[obyt]|=(1<<obit); 
        }
      }
    } else {
      for (i=n; i>0; i--) {
        for (j=obits-1, iadr=i*ibits-1, oadr=i*obits-1; j>=0; j--,iadr--,oadr--) {
          ibyt=iadr>>3; ibit=iadr&0x7;
          obyt=oadr>>3; obit=oadr&0x7;
          if (j+diff<0 || (buf[ibyt]&(1<<ibit))==0) buf[obyt]&=~(1<<obit); else buf[obyt]|=(1<<obit); 
        }
      }
    }
  }

  public int ctx2pkt (long pdata) {
    int len = getNextContext();
    Native.ja2p (cbuf,0, pdata,0, len);
    return len;
  }

  public int buf2pkt (byte[] dbuf, int off, long pdata, int len) {
    int hlen = headerLength-trailerLength;
    Native.ja2p (buf,0,    pdata,0, hlen);
    Native.ja2p (dbuf,off, pdata,hlen, len);
    if (bits==16 && (flags&FLG_BIGE)!=0) NetIO.swap2(pdata+hlen,pdata+hlen,len);
    if (bits==12) { NetIO.cvt16to12(pdata+hlen,pdata+hlen,len); len=(len/4)*3; }
    Native.ja2p (buf,hlen, pdata,hlen+len, trailerLength);
    return headerLength+len;
  }

  /** {@inheritDoc} */
  public String getConfiguration (DataFile df) {
    String config = "VRT";
    if (df.hp!=null) {
      config += "/DET";
      if (df.hp.aux!=null && df.hp.aux.length()>0) config += "="+df.hp.aux;
    }
    return config;
  }

  /** {@inheritDoc} */
  public void setFileName (DataFile df, FileName filename) {
    DataFile hp = new DataFile();
    hp.init(null,filename,"3000","NH",df.getPacketHandlerFlags());
    String srec="HDR|SL,SID|SL,OUID|SL,PIC|SL,TCIS|SL,TCFS|SX,TLR|SL";
    hp.setSubRecords(srec);
    if (hrep==DataTypes.IEEE) hp.setDataRep("IEEE");
    if (hrep==DataTypes.EEEI) hp.setDataRep("EEEI");
    df.hp=hp;
  }

  /** {@inheritDoc} */
  public double naturalDataOffset (DataFile df, double boffset) {
    int bpp = fdataLength;
    double npkt = Math.floor(boffset/bpp);
    return npkt*bpp;
  }

  /** {@inheritDoc} */
  public double dataToPacketOffset (DataFile df, double boffset) {
    double npkt = Math.floor(boffset/fdataLength);
    double bpkthdr = npkt*headerLength;
    if (bits!=fbits) boffset = boffset*bits/fbits;
    ipoff = (int)(boffset-npkt*dataLength); 			// save offset into packet
    if (df.hp==null) {
      boffset += bpkthdr;
      if (ipoff>0) boffset += (headerLength-trailerLength);	// skip header
    } else {
      if (ipoff>0) npkt++;  					// skip header
      npktoff = npkt;
    }
    return boffset;
  }

  /** {@inheritDoc} */
  public double packetToDataOffset (DataFile df, double boffset) {
    int bpp = dataLength;
    if (df.hp==null) bpp += headerLength;
    double npkt = Math.floor(boffset/bpp);
    return npkt*fdataLength;
  }

  /** {@inheritDoc} */
  public void open (DataFile df) {
    isInput = df.isInput;
    setFormat(df.getFormat());
    if (df.hp!=null) df.hp.open();
    if (isInput) initFrom(df);
  }

  /** {@inheritDoc} */
  public void seek (DataFile df, double boffset) {
    boffset = dataToPacketOffset (df,boffset);
    if (df.hp!=null) df.hp.seek(npktoff);
    if (df.io!=null) df.io.seek((long)(df.getDataStart()+boffset));
  }

  /** {@inheritDoc} */
  public int read (DataFile df, byte[] buf, int boff, int bytes, long lbuf) {
    int nbp=getBytes(), nb, fboff=boff;
    int ndo=(bits!=fbits)? bytes*bits/fbits : bytes;
    int nelem=bytes*8/fbits;
    for (rhdrs=0; ndo>0; ndo-=nb) {
      if (ipoff==0) { 						 		// past packet header
        if (df.hp!=null) df.hp.read (this.buf,headerOffset,headerLength);	// read packet header
        else             df.io.read (this.buf,headerOffset,headerLength-trailerLength);	// read packet header less trailer
        rhdrs++;
      }
      nb = Math.min(ndo,nbp-ipoff);
      if (buf!=null) nb = df.io.read(buf,boff,nb);
      else           nb = df.io.read(lbuf,boff,nb);
      if (nb<0) return nb;
      boff+=nb; ipoff+=nb; if (ipoff==nbp) ipoff=0; // signal next packet header
      if (ipoff==0 && df.hp==null) df.io.read (this.buf,trailerOffset,trailerLength);	// read packet trailer
    }
    if (bits!=fbits) unpack (buf,fboff, nelem, bits,fbits);
    return bytes;
  }

  /** {@inheritDoc} */
  public int write (DataFile df, byte[] buf, int boff, int bytes, long lbuf) {
    int nbp=getBytes(), nb;
    int ndo=(bits!=fbits)? bytes*bits/fbits : bytes;
    int nelem=bytes*8/fbits;
    if (needsContext) {
      int nby = getNextContext();
      df.io.write (this.cbuf,0,nby);	// write packet header
      needsContext=false;
    }
    if (bits!=fbits) pack (buf,boff, nelem, fbits,bits);
    if (bits==16 && (flags&FLG_BIGE)!=0) Convert.swap2(buf,boff,bytes>>1);
    for (whdrs=0; ndo>0; ndo-=nb) {
      if (ipoff==0) { 								// past packet header
        if (df.hp!=null) df.hp.write (this.buf,headerOffset,headerLength);	// write packet header
        else             df.io.write (this.buf,headerOffset,headerLength-trailerLength);	// write packet header
        whdrs++;
      }
      nb = Math.min(bytes,nbp-ipoff);
      if (buf!=null) nb = df.io.write(buf,boff,nb);
      else           nb = df.io.write(lbuf,boff,nb);
      if (nb<0) return nb;
      boff+=nb; ipoff+=nb; if (ipoff==nbp) ipoff=0; 			// signal next packet header
      if (ipoff==0 && df.hp==null) df.io.write (this.buf,trailerOffset,trailerLength);	// write packet trailer
    }
    return bytes;
  }

  /** {@inheritDoc} */
  public void close (DataFile df) {
    if (df.hp!=null) df.hp.close();
  }

  /** {@inheritDoc} */
  public boolean hasHeader() {
    return (rhdrs>0);
  }

  /** {@inheritDoc} */
  public String listHeader() {
    Time t = getTC();
    String ts = (t!=null)? t.toString(12) : null;
    String list = "VRT-Packet OUID="+Convert.l2x(getOUID())+" Stream="+getSID()+
	" Count="+getCount()+" Size="+getSize()+" Bits="+getBits()+" TC="+ts;
    return list;
  }

  /** {@inheritDoc} */
  public PacketHandler cloneOf() {
    PacketHandler ph=null;
    try { ph = (PacketHandler)this.clone(); }
    catch (Exception e) {}
    return ph;
  }

  /* copy bytes from a user buffer into the 64by header buffer */
  public void setBuffer (byte[] buffer, int boff) {
    System.arraycopy(buffer,boff, buf,headerOffset, headerLength);
  }

  /* copy bytes from the 64by header buffer into a user buffer */
  public void getBuffer (byte[] buffer, int boff) {
    System.arraycopy(buf,headerOffset, buffer,boff, headerLength);
  }

}
