package nxm.ice.lib;

public class IcePIC9 extends IceJVM {

  public final static int CF_ALLPM=0xC0000000;    // Command Fifo PM select both
  public final static int CF_SLOTS=0x00000028;    // Command Fifo slots address
  public final static int CF_SNIFF=0x00000020;    // Command Fifo sniff address
  public final static int CF_XFER =0x00000030;    // Command Fifo read|write address
  public final static int CF_RESET=0x00000028;    // Command Fifo reset address

  // normal static vars
  public static int runMode;
  public static int lastDir;
  public static int baddr;
  public static int dmaqp;
  public static int dsgType;

  public static void run() {
    int tick,stat,chns;
    reset();
    resetCmdFifo(1);
    for (tick=1;;tick=tick+1) {
      stat = arBus(REG_AMCSR);
      if ((stat&MCSR_OMB1F)!=0) processMbx();
      stat = arMem(MSG);
      if ((stat>>24)==0x55) processMessages();
      stat = arMem(dmaqp);
      if (stat!=0) processDMA(stat<<2);
      else { awMem(DMA_HPQL,(dmaqp-DMA_QUE)>>2); dmaqp = DMA_QUE; awBus(REG_AOMB4,tick); }
      stat = arBus(REG_AHAR0);
      if ((stat&0x40)!=0) processNVME(stat);
      awMem(HBT,tick);
    }
  }

  public static void processDMAx() {
    int stat;
    stat = arMem(dmaqp);
    if (stat!=0) processDMA(stat<<2);
    else { awMem(DMA_HPQL,(dmaqp-DMA_QUE)>>2); dmaqp = DMA_QUE; }
  }

  public static int isBus (int addr) {
    addr &= 0xFF000000;
    if (addr==0xFD000000) return 0; // 0xFD000000 is memory cutout - not sys bus
    if (addr!=0) return 1; // everything else with the select of mask 0xFF000000 is sys bus
    return 0;
  }

  public static void resetFifo() {
    int r1=arBus(REG_AMCSR);
    if ((r1&MCSR_OFIFO_EMPTY)!=0) return;
    while ((r1&MCSR_MENA)!=0) r1=arBus(REG_AMCSR); 
    awBus(REG_AMCSR,r1|MCSR_RFIFO);
  }

  public static void resetCmdFifo (int mode) {
    awBus(CF_ALLPM|CF_RESET,(mode<0)?1:mode);
    awBus(CF_ALLPM|CF_RESET,(mode<0)?0:mode);
  }

  public static void processMbx() {

    int r0,r1,r2,r3,r4,r5;
    int i,j,k,i2;
    DMAStruct dma;

    r1 = arBus(REG_AIMB1);
    r2 = arBus(REG_AIMB2);
    r3 = arBus(REG_AIMB3);
    r4 = arBus(REG_AIMB4);

    i2 = r2<<2;

    r0 = r3;
    switch (r1) {
    case 0: 	// adon_null
      break;
    case 1: 	// adon_wr
      if (isBus(i2)!=0) awBus(i2,r3);
      else              awMem(i2,r3);
      break;
    case 2: 	// adon_rd
      if (isBus(i2)!=0) r0=arBus(i2);
      else              r0=arMem(i2);
      wait4dma (MCSR_MENA);
      break;
    case 3:	// adon_rwm
      r0=arMem(i2);
      r0 &= r1;
      r3 &= r4;
      r0 |= r3;
      awMem(i2,r0);
      break;
    case 4: 	// adon_dma
      r0 = r3&0xFFFF;
      resetFifo();
      awBus(REG_AOMB1,r0);
      if ((r4&0x4)!=0) wfifo (i2,r0);
      else             rfifo (i2,r0);
      return;
    case 7: 	// adon_mod
      r0 = r3 | (r2<<28);
      writemod(r0);
      break;
    case 8: 	// adon_wrb
      r5 = r4>>16;
      if (r4<=4) {	// local bus write
	if (r4==0) r0=arBus(r2);
	else       awBus(r2,r3);
      } else {		// this may take a while
        r4 = r4&0xFFFF;
	if (r4>4) resetFifo();
	if (r4==0) r0=adon_rdbf(r2,r3,r4,r5);
        awBus(REG_AOMB1,r0);		// release ADON command
	if (r4==0) return;
	if (r5==0) for (;r4>0;r4-=4) awBus(r2,rafifo());	// local bus from FIFO
	else if (r5<=3) r0=adon_wrbf(r2,r3,r4,r5);		// PM bus via command FIFO
	else            r0=adon_wrbj(r2,r3,r4,r5<<16);		// PM bus via JTAG
        return; 
      }
      break;
    case 9: 	// adon_init
      if (r2>=0) {
        initMC();
        runMode=1;
        resetCmdFifo(-1);
      } else {
        runMode=0;
      }
      dsgType = r3;
      break;
    case 10: 	// adon_tc
      r0 = adon_tc(i2,r3,r4);
      break;
    case 11: 	// adon_mstat
      r0 = adon_mstat(r2);
      break;
    case 12: 	// adon_nstat
      r0 = adon_nstat(r2);
      break;
    case 13: 	// adon_dstat
      dma = (DMAStruct)getObjectAt(i2);
      if (dma.todo!=0) { 
        dma.stat = -dma.stat;
        r0 = dma.hxfer;
      }
      else r0 = 0;
      break;
    case 14:    // adon_wrioc
    case 15:    // adon_rdioc
      r2 = ((r2&0xC0)<<(HYP_SHF-6)) | IOC_ADDR | (r2&0x3F);
      if (r1==14) awBus(r2,r3);
      else   r0 = arBus(r2);
      break;
    case 16:	// adon_wriob
      resetFifo();
      awBus(REG_AOMB1,0);
      r0 = rwiob(r2,r3,r4,false);
      return; //break;
    case 17:	// adon_rdiob
      r0 = rwiob(r2,r3,r4,true);
      break;
    case 19:	// adon_hstat
      dma = (DMAStruct)getObjectAt(i2);
      awBus(REG_AOMB3,dma.hcycle);
      awBus(REG_AOMB2,dma.hindex);
      r0 = (dma.cindex-dma.caddr);
      if (r0>0xFFFF) r0 = 0xFFFF;
      r0 |= (dma.stat<<16);
      break;
    case 20:	// adon_jtag
      if (r2==-10) {
	resetFifo();
        awBus(REG_AOMB1,0);	// early release for FIFO
        for (i=0; i<r3; i++) dojtagx (0,rafifo(),r4|0x000F); // write block from FIFO
        return;
      }
      else if (r2==-11 || r2==-12) {
        j = r4&0x0000FFFF;
        k = r4&0xFFFF0000;
        r0=arMem(r3<<2);
        for (i=0; i<j; i++,r3++) {
          if (r2==-11) { dojtag (0,arMem(r3<<2),k|0x000F); }			// write block
          if (r2==-12) { r0=dojtag (0,r0,k|0x0F0F); awMem(r3<<2,r0); r0=0; }	// read block
        }
      } else {
        r0 = dojtag(r2,r3,r4);
      }
      break;
    case 21:	// adon_rback
      r0 = arBusAck(r2,r4);
      break;
    case 22:	// adon_mpio
      r0 = dompio (r2,r3,r4);
      break;
    case 23:	// adon_icap
      awBus(REG_AOMB1,r0);
      udelay(10000);
      awBus(ICAP_BASE,r2);
      awBus(ICAP_BASE,r3);
      return;
    case 24:	// adon_send
      break;
    case 25:	// adon_recv
      break;
    case 26:	// adon_dmax
      r0 = r4;
      wait4dma(MCSR_MENA);	// wait till last one has finished
      if (r4<0) {
	r0 = -r4;
        setupDMAi(PM_HOST,r3|0x1F,r3+r0);
        awBus(DMAC_ROUTE_SET,ROUTE_R2HP);
        r5 = MCSR_MASTER_WR;
      } else {
        awBus(DMAC_HPCNT,0);
        r5 = MCSR_MASTER_RDM;
      }
      awBus(REG_AMDAR,r2<<6);
      r2 = r2>>26;
      if (r2>0) r5 |= MCSR_MASTER_DAC;
      awBus(REG_AMDAU,r2);
      awBus(REG_AMDTC,r0);
      awBus(REG_AMCSR,r5);
      wait4dma(MCSR_QSTAT);	// wait till this one has started
      if (r4>0) {
        setupDMAo(PM_HOST,r3|0x1F,r3+r0);
        awBus(DMAC_ROUTE_SET,ROUTE_HP2R);
      }
      wait4dma(MCSR_MENA);	// wait till this one has finished
      break;
    case 27:	// adon_wmod
    case 28:	// adon_rmod
      r0 = (r1==27)? 0x0000A5A5 : 0x0000D2D2;	// write : read indicator
      r5 = r4|JTAG_MSK_WMOD32;
      dojtag(JTAG_TMS_WMOD32,        r0,r5);
      dojtag(JTAG_TMS_WMOD32,        r2,r5);			// maddr
      dojtag(JTAG_TMS_WMOD32,0x0000B4B4,r5);
   r0=dojtag(JTAG_TMS_RMOD32,        r3,JTAG_MSK_RMOD32|r4);	// mdata
      dojtag(JTAG_TMS_WMOD32,0xFFFFFFFF,r5);
      dojtag(JTAG_TMS_WMOD32,0xFFFFFFFF,JTAG_MSK_WMOD32);	// disable
      break;
    case 29:	// adon_wriop
      resetFifo();
      awBus(REG_AOMB1,0);
      r0 = adon_wriop(r2,r3,r4);
      return;
    case 30:	// adon_wrblk
      maddr=i2;
      for (;r4>0;r4--) {
	aBus(mdata);
	wBus(mdata);
      }
      break;
    case 31:	// adon_tcup
      awBus(r3,0);
      maddr=i2;
      awBus(r4,mdata); r4+=4;		// tc status
      r0 = arBus(r3+11);
      awBus(r4,mdata); r4+=4;		// tc off frac
      awBus(r4,mdata-r0); r4+=8;	// tc off whole
      awBus(r4,mdata); r4+=4;		// tc wsec
      awBus(r4,mdata); 			// tc fsec
      awBus(r3,1);
      break;
    case 32:	// jvm_wmem
      r0=jvm_rwmem(r2,r3,r4,1);
      break;
    case 33:	// jvm_rmem
      r0=jvm_rwmem(r2,r3,r4,-1);
      break;
    }
    awBus(REG_AOMB1,r0);
  }

  public static void processDMA (int stkloc) {
    int i,j,stat,diff,space,xfer,cxfer;
    int start,stop,nhindex,ncindex,hindex,csize,clast;
    int mcsr,ccur,mask;
    DMAStruct dma = (DMAStruct)getObjectAt(stkloc);
    int dir = dma.dir;	// <0 is input to host, >0 is output from host
    dmaqp = dmaqp+4;
    // need to stop port ?
    if (dma.todo==0) {
      if (dma.stat>0) portStop(dma);
      return;
    }
    // need to start port ?
    if (dma.stat==0 && (dir<=0 || dma.todo==DMA_SPIN)) portStart(dma);

    // handle SPIN case
    if (dma.todo==DMA_SPIN) {
      if (dma.master==-1 && dma.hind!=0) {
        i = arBus(dma.hind<<2);
        j = 0xFF000000;
        if ((i&j)==0 && (dma.hindex&j)!=0) dma.hcycle++;
        dma.hindex = i;
      }
      return;
    }

    // get current index pointer
    i = (dir==3)? dma.cind : dma.cindp;
    stat = arMemSync(i<<2);

    // handle DIO input case
    if (dir == -2) {
      if (stat<dma.hindex) {
        if (stat<=1) stat=dma.haddr;
        dma.hcycle++;
        if (dma.todo>0) dma.todo--;
      }
      dma.hindex = stat;
      updateHIP(dma,stat);
      return;
    }

    // num 4by words to transfer this pass
    stop = dma.haddr + dma.hsize;
    xfer = stop - dma.hindex;
    diff = 0x04000000 - (dma.hindex&0x03FFFFFF); // 4Gby in packets
    if (xfer > diff) xfer = diff;         // dont cross a 4G boundary
    if (dma.frame != 0) {
      if (dma.appp==0) dma.appp = (dma.frame&0xFFFF)<<3;
      i = dma.appp; if (i<0) i=-i;
      if (xfer>i) xfer = i;
    }
    if (xfer > dma.hxfer) xfer = dma.hxfer;
    cxfer = (xfer<<4);
    csize = dma.csize;
    clast = dma.caddr+csize;
  
    // where does the Host index end up
    nhindex = dma.hindex + xfer;
    if (nhindex >= stop) nhindex = dma.haddr;
  
    // where does the Card index end up
    ncindex = dma.cindex + cxfer;
    if (ncindex >= clast) ncindex -= csize;

    // check for data available
    ccur = (stat>>2)&0x3FFFFFF0;
    diff = ccur - dma.cindex;					// diff is avail Idata|Ospace
    if (diff<0 || (diff==0 && dma.stat==0 && dma.ccycle==0 && dir!=3)) diff += csize;  // handle wrap around
    space = csize-diff;						// space is avail Ispace|Odata

   while (true) {			// transfer loop for breaking out 

    if (dir<0) {
      if (diff<cxfer) return;					// not ready yet
      if (space<(csize>>3) && dma.frame==0) dma.miss++;		// signal miss if only 1/8 buffer is avail
    }
    else if (dir==2);						// DIO
    else if (dir==3) {						// NIO
      if (dma.stat!=1) {
        if (dma.stat==0) { dma.stat=3; portStart(dma); }	// enable port to start NIO
        if (diff<(csize>>1)) return;				// wait for 1/2 buffer full
	awBus(DMAC_ROUTE_SET,dma.enb);
        dma.stat=1;	
      }
      if (diff<cxfer) { 					// possible falling behind
//	dma.miss++;
//	awBus(DMAC_ROUTE_CLR,dma.enb);
//	dma.stat=2;
      }
      diff = (arMemSync(dma.cindp<<2)>>2) - dma.cindex;		// IO port reader - current PCI
      if (diff>=0 && diff<cxfer) return;			// not ready to transfer yet
      dir = -3;
    }
    else if (dma.todo == DMA_ONDEMAND) {
      if (dma.stat<0) portStart(dma); 
      if (diff!=0 && diff<=cxfer) return;			// wait for CARD buffer space
      dma.hind = arBusSync(dma.alg);				// AHIRX+n is KEY_INBYTE
      i = dma.hind - dma.hindex;
      if (i<0) i += dma.hsize;
      if (i<xfer) return;					// wait for HOST buffer pointer
    }
    else if (dma.todo == DMA_LOADNSPIN) {
      if (dma.stat==0 && dma.ccycle>0) portStart(dma);
      if (diff<cxfer) return;					// not ready yet
      if (dma.stat!=0) dir=0;
    }
    else if (dma.todo == DMA_LOOPNSPIN) {
      if (dma.stat==0) portStart(dma);				// restart
      if (diff<cxfer) return;					// not ready yet
      dir=0;
    }
    else {
      if (dma.stat==0 && space>=(csize>>1)) portStart(dma);	// start output at 1/2 buffer full
      if (diff<cxfer+0x100) return;				// not ready yet (ram prefetch of up to 1Kby)
      if (dma.stat!=0 && space<(csize>>2)) dma.miss++;		// signal miss if only 1/4 buffer is avail
    }
    if (dma.frame != 0) {					// framed decimation bookeeping
      if (dma.appp > 0) {
        dma.appp -= xfer;
        if (dma.appp==0) dma.appp = ((dma.frame&0xFFFF)<<3)*(1-(dma.frame>>16));
      } else {
        dma.appp += xfer;
        nhindex = dma.hindex;
        break;
      }
    }
    if (dma.todo == DMA_LOOP) break;

    // start the Host port master DMA
    start = (dma.cindex<<2) | (stat&0x1F);
    stop  = (ncindex<<2);
    hindex = dma.hindex;

    // fifo reset
    if (dir != lastDir) {
      if (lastDir<=0) wait4dma (MCSR_MENA);
      else            wait4route (ROUTE_HP2R);
      awBus(DMAC_HPCNT,0);
    }
    // input fifo preload
    if (dir==0);
    else if (dir<0) {
      if ((j=xfer&0x1F)!=0) {	// round fifo memory transfers to 2Kby - backup and overwrite to keep fifos happy
        i = 0x20-j; xfer += i; hindex -= i; start -= (i<<6);
        if ( start<(dma.caddr<<2) ) start += (csize<<2);
      }
      if (wait4route(ROUTE_R2HP) < 0) { unlockDMA(dma); return; }
      setupDMAi(PM_HOST,start,stop);
      wait4dma(MCSR_MENA);	// previous DMA all scheduled
      awBus(DMAC_ROUTE_SET,ROUTE_R2HP);
    } else if (dir==2) {
      wait4dma(MCSR_MENA);	// previous DMA all queued
    } else {
      if (wait4route(ROUTE_HP2R) < 0) { unlockDMA(dma); return; }
      awBus(DMAC_HPCNT,0);
    }
    // start the PCI transfer
    if (dir!=0) {
      mcsr = dma.mcsr;
      awBus(REG_AMDAR,hindex<<6);
      awBus(REG_AMDAU,hindex>>26);
      awBus(REG_AMDTC,xfer<<6);
      awBus(REG_AMCSR,mcsr);
      wait4dma(MCSR_QSTAT);	// wait till this one has started
    }
  
    // output fifo unload
    if (dir==1) {
      setupDMAo(PM_HOST,start,stop);
      awBus(DMAC_ROUTE_SET,ROUTE_HP2R);
      if (dma.todo == DMA_ONDEMAND) {
        if (wait4route(ROUTE_HP2R) < 0) unlockDMA(dma); 
        awMem((dma.cindp+1)<<2,ncindex<<2);			// sets the new STOP point, start is already at last STOP
        // mask = arBus(DMAC_STATUS) & dma.enb & ROUTE_R2X;	// is my RAM still running ?
        //if (mask!=0); else						// already running
        if (dma.stat>0) awBus(DMAC_ROUTE_SET,dma.enb); 		// possibly restart with new segment + debug
        else if (diff==0 || space<(csize>>1));			// not ready yet
        else portStart(dma); 					// start when half full
      }
    }
    
    updateHIP(dma,dma.hindex);
    lastDir = dir;
    break;
   }
    // update card loop pointers
    if (ncindex < dma.cindex) dma.ccycle++;
    dma.cindex = ncindex;
  
    // update host loop pointers
    if (nhindex < dma.hindex) {
      dma.hcycle++;
      if (dma.todo>=1) dma.todo--;
      if (dma.todo==0 && dma.chain>0) {	// load next transfer
        i = dma.chain<<2;
        dma.haddr = arMem(i); i+=4;
        dma.hsize = arMem(i); i+=4;
        dma.todo  = arMem(i); i+=4;
        dma.chain = arMem(i); 
        nhindex = dma.haddr;
      }
      // check for card buffer loss
      stat = readmod(dma.mcfg);
      if ((stat&IOC_STAT_ERR)!=0) {
        dma.err = (stat>>8)&0xFF;
        if (dma.err==0) dma.err=0x100;
      }
    }
    dma.hindex = nhindex;
  }

  public static void portStart (DMAStruct dma) {
    int i, ro,r, cadr, gmcfg;
    int side = getside(dma);
    int dir = dma.dir;
    ro = arBus(DMAC_STATUS);
    r = ro | dma.enb;
    i = arMem(DMA_ROUTF) & ~(ROUTF_HXFDP|ROUTF_CXFDP|ROUTF_TXFDP); // calculate dma flags
   if ((dma.feed&0xF)!=0) {	// processor module
    if ((r&ROUTE_X2TA)!=0  && (ro&ROUTE_X2TA)==0 ) awBus(PMA_CTL,0x01);
    if ((r&ROUTE_X2TB)!=0  && (ro&ROUTE_X2TB)==0 ) awBus(PMB_CTL,0x01);
    if ((r&ROUTE_XA2TA)!=0 && (ro&ROUTE_XA2TA)==0) awBus2z(PMA_CTL+0x10);
    if ((r&ROUTE_XB2TA)!=0 && (ro&ROUTE_XB2TA)==0) awBus2z(PMA_CTL+0x14);
    if ((r&ROUTE_XA2TB)!=0 && (ro&ROUTE_XA2TB)==0) awBus2z(PMB_CTL+0x10);
    if ((r&ROUTE_XB2TB)!=0 && (ro&ROUTE_XB2TB)==0) awBus2z(PMB_CTL+0x14);
   }
//    if ( ((r&ROUTE_HAANY)!=0 && (r&ROUTE_HBANY)!=0) ) i |= (ROUTF_HXFDP);
//    if ( !((r&ROUTE_TAANY)!=0 && (r&ROUTE_TBANY)!=0) ) i |= (ROUTF_TXFDP);
    if ( (r&ROUTE_TAANY)==0 || (r&ROUTE_TBANY)==0 ) i |= (ROUTF_TXFDP);
    if ( ((r&(ROUTE_HA2X|ROUTE_X2HA))!=0) != ((r&(ROUTE_HB2X|ROUTE_X2HB))!=0) ) i |= (ROUTF_HXFDP);
    if ((dma.mcfg&IOM_MUX) != 0) {
      i |= (ROUTF_HXFDP);
      if ( (r&(ROUTE_HX2CX|ROUTE_CX2HX)) != 0) i |= ROUTF_CXFDP;
    }
    if (dsgType<DSG_TRAY && (dma.flags&FLG_NIO)!=0) {
      if (dir<0) i |= ROUTF_M2NIO;
      else if ((dma.flags&FLG_SGO)!=0);
      else { i |= ROUTF_NIO2M; awBus(DMAC_ROUTE_SET,ROUTE_PR2R); }	// add static routes
    }
    if (dir==-2) {
      awMem((dma.cindp^0x10)<<2,(dma.todo==1)?0:dma.haddr);
      i |= ROUTF_DIO2P;
    }
    awMem(DMA_ROUTF,i);
    awBus(DMAC_ROUTE_FLG,i);		// add dma flags 
    cadr = (side==2)? HYPB_CTL+4 : HYPA_CTL+4;
    gmcfg = arMem(DMA_GMCFG);
    if (gmcfg!=0) awBus (cadr-3, 0);	// hold output
    r = arBusSync(cadr);		// read the same twice
    if (side==3 && (dma.mcfg&0x80000000)!=0) r += arBus(HYPB_CTL+4); 		// block mux add B side count
    if (dma.stat==0) awBus(DMAC_ROUTE_SET,dma.enb);	// add static routes
    if (dma.stat==-1) awBus(DMAC_ROUTE_SET,dma.enb&(~ROUTE_R2X));	// add static routes
    writemod(gmcfg);			// new mcfg, also disables hold
    awMem(DMA_GSTAT,r);
    awMem(DMA_GMCFG,0);
    if (dma.stat!=3) dma.stat = 1;
    if (dir<0) {
      updateHIP(dma,0);
    } else {
      if (dma.todo>0 || dma.todo==DMA_ONDEMAND) dma.chain=-1;	// for exact cutoff handler
      if (dma.todo==DMA_RESHOT) { dma.stat= -1; dma.todo=1; }
    }
  }

  public static void portStop (DMAStruct dma) {
    int r,ro,gmcfg,pmena;
    int enb = dma.enb;		// full disconnect
    int side = getside(dma);
    ro = arBus(DMAC_STATUS);
    if (dma.dir>0 && dma.chain<0 && dma.feed==0) {
      if (dma.chain==-1) { dma.chain=-2; awMem((dma.cindp+1)<<2,dma.cindex<<2); }
      if ((enb&arBus(DMAC_STATUS))!=0) return;	// output not finished yet
    }
    if (dma.dir>0 && (enb&ROUTE_R2X)!=0) {
      awBus(DMAC_ROUTE_CLR,(enb&ROUTE_R2X));	// stop RAM reader first
      udelay(100);	// let clear out of read queue - up to 2K bytes
    }
    dma.stat = 0;
    r = getenb();	// get route for all remaining channels
    enb &= (r^ro);	// only clear routes nobody else is using 
   if ((dma.feed&0xF)!=0) {	// processor module
    pmena = (dsgType>0)? 0x1 : 0x0;	// leave enabled for NIO & NVME
    if ((r&ROUTE_X2TA)==0  && (ro&ROUTE_X2TA)!=0) awBus(PMA_CTL,pmena);
    if ((r&ROUTE_X2TB)==0  && (ro&ROUTE_X2TB)!=0) awBus(PMB_CTL,pmena);
   }
    awBus ((side==2)? HYPB_CTL+1 : HYPA_CTL+1, 0);	// hold output
    awBus(DMAC_ROUTE_CLR,enb);	// static routes
    if (side==3) side=1;
    gmcfg = arMem(DMA_GMCFG+(side<<2));
    writemod(gmcfg);		// new mcfg, also disables hold
    updateHIP(dma,0);
    if (r==0) {		// clear flags
      ro = arMem(DMA_ROUTF);
      ro &= ~(ROUTF_M2NIO|ROUTF_NIO2M|ROUTF_P2DIO|ROUTF_DIO2P);
      awMem(DMA_ROUTF,ro);
      awBus(DMAC_ROUTE_FLG,ro);		// add dma flags 
    }
  }

  public static int getenb () {
    int enb = arMem(DMA_ROUTD);
    for (int i=DMA_QUE; ; i=i+4) {
      int stkloc = arMem(i);
      if (stkloc==0) return enb;
      stkloc = stkloc<<2;
      DMAStruct dma = (DMAStruct)getObjectAt(stkloc);
      if (dma.stat!=0) {
	enb |= dma.enb;
//	if ((dma.flags&FLG_NIO) enb |= 0x00010000;
      }
    }
  }

  public static int getside (DMAStruct dma) {
    int side = (dma.mcfg>>28)&0x3;
    if (side==0) side = ((dma.port&1)!=0)? 1:2;
    return side;
  }

  public static void updateHIP (DMAStruct dma, int index) {
    if (dma.hindp<=0x28000) return;	// in cached structures - no HIP
    awBus(dma.hindp<<2, index);
  }

  public static void unlockDMA (DMAStruct dma) {
    int mcsr = arBus(REG_AMCSR);
    if ((mcsr&MCSR_MENA)!=0) {
      mcsr = mcsr & ~MCSR_MENA;
      awBus(REG_AMCSR,mcsr);
    }
    dma.err = dma.err+1;
//    dma.todo = 0;
  }

  public static int adon_mstat (int side) {
    int mcfg = 0;
    int mask = side<<28; 
    for (int i=DMA_QUE; ; i=i+4) {
      int stkloc = arMem(i);
      if (stkloc==0) return mcfg;
      stkloc = stkloc<<2;
      DMAStruct dma = (DMAStruct)getObjectAt(stkloc);
      if (dma.stat==0) continue;
      if ((dma.mcfg&mask)!=0) mcfg |= dma.mcfg;
    }
  }

  public static int adon_nstat (int dmac) {
    int stat = 0;
    int stki = (0x27000 + ((dmac-1)*DMA_SEP) ) << 2;
    for (int i=DMA_QUE; ; i=i+4) {
      int stkloc = arMem(i);
      if (stkloc==0) break;
      stkloc = stkloc<<2;
      DMAStruct dma = (DMAStruct)getObjectAt(stkloc);
      if (stkloc==stki) {
        stat |= (dma.dec-1);
        if (dma.todo!=0) stat |= 0x8000;
      }
      else if (dma.master==dmac) {
        int mask = ((dma.port-1)>>1) & 0xF;
        stat |= (0x1 << (16+mask));
      }
    }
    return stat;
  }

  public static int wait4route (int mask) {
    for (int i=0;i<0xFFF0;i++) {
      if ((arBus(DMAC_STATUS)&mask)==0) return i;
      if (i>0x1000) udelay(10);
    }
    awBus(DMAC_ROUTE_CLR,mask);
    return -1;
  }

  public static int wait4dma (int mask) {
    for (int i=0;;i++) {
      int mcsr = arBus(REG_AMCSR);
      if ((mcsr&mask)==0) return i;
      if ((mcsr&MCSR_TEST)!=0) return i;
      if (i>0xFFF0) return -1;
      if (i>0x1000) udelay(10);
    }
  }

  public static void initMC () {
    memz(0,0x1FFF);
    resetDMA();
  }

  public static void wfifo (int addr, int count) {
    int dram = (addr&0xFFF00000);
    int rmax = 64;
    maddr = (dram>0)? DMA_TMP : addr;
    for (int i=0; i<count; ) {
      int stat = arBusSync(REG_AMCSR);
      if ((stat&MCSR_IFIFO_FULL)==0) {
        if (dram>0 && (i&0xF)==0x0) {
	  maddr = DMA_TMP;
          ioadr = addr;
          ioctl = IOCTL_RD|64;
          addr  = addr+rmax;
	  maddr = DMA_TMP;
        }
        int data = mdata;
        awBus(REG_AFIFO,data);
        i++;
      }
    }
  }

  public static int rafifo () {
    while((arBus(REG_AMCSR)&MCSR_OFIFO_EMPTY)!=0);
    return arBus(REG_AFIFO);
  }

  public static void rfifo (int addr, int count) {
    int dram = (addr&0xFFF00000);
    int rmax = 64;
    maddr = (dram>0)? DMA_TMP : addr;
    for (int i=0; i<count; ) {
      int stat = arBusSync(REG_AMCSR);
      if ((stat&MCSR_OFIFO_EMPTY)==0) {
        int data = arBus(REG_AFIFO);
        mdata = data;
	i++;
        if (dram>0 && (i&0xF)==0x0) {
	  maddr = DMA_TMP;
          ioadr = addr;
          ioctl = IOCTL_WR|64;
          addr  = addr+rmax;
	  maddr = DMA_TMP;
        }
      }
    }
  }

  protected static void writemod (int iom) {
    if (iom==0) return;
    int addr = (((iom>>28)&0x3)<<HYP_SHF) | IOC_ADDR;	// data bits[29:28] are address, shift to module address
    awBus(addr,iom);
  }

  protected static int readmod (int iom) {
    if (iom==0) return 0;
    int addr = (((iom>>28)&0x3)<<HYP_SHF) | IOC_ADDR;	// data bits[29:28] are address, shift to module address
    return arBus(addr);
  }

  public static void dojtagx (int tms, int tdi, int ena) {
    awBus(JTAG_TMS,tms);
    while ( (arBus(JTAG_STAT)&0x1) != 0);	// wait for last scan to finish
    awBus(JTAG_TDI,tdi);
    awBus(JTAG_ENA,ena);
  }

  public static int dojtag (int tms, int tdi, int ena) {
    int i, tdo=tdi;
    awBus(JTAG_TMS,tms);
    awBus(JTAG_TDI,tdi);
    awBus(JTAG_ENA,ena);
    while ( (arBus(JTAG_STAT)&0x1) != 0);	// wait for scan to finish
    if ((ena&0x0F00)!=0) tdo = arBus(JTAG_BASE);
    return tdo;
  }

  public static int dompio (int addr, int data, int len) {
    int i,k;
    int sel=(addr>>24)&0xFF;
    int uaddr=(addr>>16)&0xFE;
    mpc_ssa(sel,1);
    mpc_dwr(sel,uaddr);
    if ((addr&0x8000)!=0) mpc_dwr(sel,(addr>>8)&0x7F);
    mpc_dwr(sel,addr&0xFF);
    if (len<0) {
      len = -len;
      data = 0;
      mpc_ssa(sel,-1);
      mpc_ssa(sel,1);
      mpc_dwr(sel,uaddr|0x1);	// set the read bit
      for (i=0; i<len; i++) { k=(i==len-1)?1:0; data |= (mpc_drd(sel,k) << (i<<3)); }
    }
    else if (len>0) {
      for (i=0; i<len; i++) { mpc_dwr(sel,data); data = data>>8; }
    }
    mpc_ssa(sel,-1);
    return data;
  }

  public static int mpc_acc (int sel, int data, int dir) {
    if (dir>0) { awBus(JTAG_STAT,(sel<<4)|data); }
    else  { data = arBus(JTAG_STAT) >> (sel+8); }
    udelay(3);	// ~85kHz for 3 MPC acc per cycle at 200MHz JVM clock
    return data;
  }

  public static void  mpc_ssa (int sel, int mode) {
    if (mode>0) {
      mpc_acc(sel,0xF,1);
      mpc_acc(sel,0xD,1);
    }
    else if (mode<0) {
      mpc_acc(sel,0xD,1);
      mpc_acc(sel,0xF,1);
    }
    else {
      mpc_acc(sel,0x5,1);
    }
  }

  public static void  mpc_dwr (int sel, int data) {
    int i,j;
    mpc_acc(sel,0xC,1);
    for (i=0; i<8; i++) { 
      j = ((data&0x80)!=0)? 2:0;
      mpc_acc(sel,0xC|j,1);
      mpc_acc(sel,0xD|j,1);
      mpc_acc(sel,0xC|j,1);
      data <<= 1;
    }
    mpc_acc(sel,0x4,1);
    mpc_acc(sel,0x5,1);
    mpc_acc(sel,0xC,1);
  }

  public static int  mpc_drd (int sel, int nack) {
    int i,j,data=0;
    for (i=0; i<8; i++) { 
      mpc_acc(sel,0x4,1);
      mpc_acc(sel,0x5,1);
      j = mpc_acc(sel,0x0,-1);
      if ((j&0x2)!=0) data |= (0x1<<(7-i));
    }
    j = (nack!=0)? 2 : 0;
    mpc_acc(sel,0x4,1);
    mpc_acc(sel,0xC|j,1);
    mpc_acc(sel,0xD|j,1);
    mpc_acc(sel,0xC|j,1);
    mpc_acc(sel,0xC,1);
    return data; 
  }

  // compare of synchronous current and config DMA registers
  protected static int dmaSync (int addr) {
    maddr = addr;
    int data = mdata;
    data -= mdata;
    return data;
  }

  protected static void reset () {
    awBus(DMAC_ROUTE_RST,-1);
    awBus(DMAC_HPCNT,0);
    dmaqp = DMA_QUE;
    awMem(dmaqp,0);
    awMem(MSG,0);
    runMode=0;
  }

  protected static void resetDMA () {
    awBus(DMAC_ROUTE_RST,-1);
    udelay(20);
    awBus(DMAC_ROUTE_RST,-1);
    memz(DMA_XI,16);
    memz(DMA_XO,16);
    memz(DMA_TO,64);
    memz(DMA_NI,16);
    memz(DMA_NO,16);
    awBus(DMAC_ROUTE_FLG,0);
  }
    
  // sets up DMA out of block into RAM
  static void setupDMAa (int i, int cur_addr, int cfg_addr) {
    maddr = DMA_AO + (i<<3);
    mdata = cur_addr; 
    mdata = cfg_addr; 
  }

  static int waitDMAa (int i, int done_addr, int timeout) {
    while (true) {
      maddr = DMA_AO + (i<<3);
      if (mdata >= done_addr) break;
      timeout = timeout-1;
      if (timeout<=0) break;
    }
    return timeout;
  }

  // sets up DMA out of block into RAM
  static void setupDMAo (int i, int cur_addr, int cfg_addr) {
    maddr = DMA_XO + (i<<3);
    mdata = cur_addr; 
    mdata = cfg_addr; 
  }

  // sets up DMA into block from RAM
  static void setupDMAi (int i, int cur_addr, int cfg_addr) {
    maddr = DMA_XI + (i<<3);
    mdata = cur_addr; 
    mdata = cfg_addr; 
  }
  
  static int getDMAi (int i) {
    maddr = DMA_XI + (i<<3);
    return mdata;
  }
  
  // sets up DMA out of tuner/mcore into RAM
  static void setupDMAt (int i, int cur_addr) {
    maddr = DMA_TO + (i<<2);
    mdata = cur_addr; 
  }
  
  static void user2jcmd (int jsel) {
    int icl = (jsel>>27) & 0xC;
    int jcbs = 28 - 6 - icl;
    if (icl==0xC) { // Altera based
      dojtag(JTAG_TMS_CMD,(JTAG_STXCMD_USER0<<18)|0xFF,jsel|JTAG_MSK_CMD);
    } else {
      dojtag(JTAG_TMS_CMD,(JTAG_CMD_USER2<<jcbs)|0xFF,jsel|JTAG_MSK_CMD);
    }
  }

// code for common IceMC 

  public static void processMessages() {
    maddr = MSG+4;
    int node = mdata;
    int head = mdata;
    int addr = mdata;
    int func = head & 0x3F;
    int olen = (head >> 16) & 0xFFFF;
    int size = olen >> 2;
    int data = MSG+16;
    switch (func) {
    case 1: busr (addr,data,size); break;
    case 2: busw (addr,data,size); olen=0; break;
    case 3: memr (addr,data,size); break;
    case 4: memw (addr,data,size); olen=0; break;
    case 6: initAlgorithm(); break;
//    case 7: modifyAlgorithm(data,size); break;
    case 8: case 9:
      int port = addr & 0xFFFF;
      int type = addr>>16;
      int route = mdata;
      if (func==8) openAlgorithm(port,type,route);
      else        closeAlgorithm(port,type,route);
      break;
    case 14: busrwm (addr,data,size); break;
    case 15: memi (data,size); olen=0; break;
    }
    maddr = MSG;
    mdata = 0xAA0000AA | (olen<<8);
  }

  public static void initAlgorithm () {
    resetDMA();
    runMode = ALG_INIT;
  }

  public static void openAlgorithm (int port, int type, int route) {
    awBus(DMAC_ROUTE_SET,route);
  }

  public static void processAlgorithm () {
  }

  public static void closeAlgorithm (int port, int type, int route) {
    awBus(DMAC_ROUTE_CLR,route);
    if (arBus(DMAC_CTL)==0) resetDMA();
  }

  public static void modifyAlgorithm (int data, int size) {
  }

  protected static void awBus2z (int addr) {
    saddr = addr;
    sdata = 0;
    saddr = addr+8;
    sdata = 0;
  }

  public static void user2done(int jsel) {
    dojtag(JTAG_TMS_RST,-1,jsel|JTAG_MSK_RST);
    awBus(JTAG_ENA,0);
  }

  public static int rwiob (int r2, int r3, int r4, boolean rw) {
    int i,j,ja,r0=rw?0:r3;
    int jsel = r2&0xFFFF0000; r2 &= 0xFFFF;
    boolean g2 = ((jsel>>16)&0x7)>2;
    if ((jsel&0x10000000)!=0) {			// main board
     if (rw) {
      if (r2==BDATA) { r0 = arMem(baddr); baddr+=4; }
     } else {
      for (i=0; i<r4; i=i+4) {
	if (r4>4) r3 = rafifo();
        if (r2==BADDR_WR || r2==BADDR_RD) baddr = r3;
        else if (r2==BDATA) { awMem(baddr,r3); baddr+=4; }
      } 
     }
    } else {
     user2jcmd(jsel);
     if (g2) ja = 0x00060002|((r2&0xFF00)<<12)|((r2&0xFF)<<4);
     else    ja = (rw?0x0E000300:0x02000300)|(r2>>8)|((r2&0xFF)<<16);
     if (rw) {
      dojtag(JTAG_TMS_SET32X,ja,jsel|JTAG_MSK_SET32X);	// IOB Page+Addr
      i=0; do {
        if (g2) j = 0x0009;			// IOB Data Read with auto address increment
        else    j = (i+1==r4)? 0x0500:0x0C00;	// IOB Data Read with auto address increment
        j = dojtag(JTAG_TMS_GET16X,j,jsel|JTAG_MSK_GET16X);
	if (g2) j = j>>8;
        r0 |= ((j&0xFF) << (i<<3));
      } while (++i<r4);
     } else {
      for (i=0; i<r4; i++) {
	if ((i&3)==0) {
	  dojtag(JTAG_TMS_SET32X,ja,jsel|JTAG_MSK_SET32X);	// IOB Page+Addr
	  if (r4>4) r3 = rafifo();
	}
        if (g2) j = 0x0008 | ((r3&0xFF)<<4);	// IOB Data Write with auto address increment
        else    j = 0x0800 | (r3&0xFF);		// IOB Data Write with auto address increment
        dojtag(JTAG_TMS_SET16X,j,jsel|JTAG_MSK_SET16X);
        r3 = r3>>8;
      } 
     }
     user2done(jsel);
    }
    return r0;
  }

  public static int adon_wriop (int r2, int r3, int r4) {
    int i,j,ja,r0=r3;
    int jsel = r2&0xFFFF0000; r2 &= 0xFFFF;
    user2jcmd(jsel);
    ja = 0x02000300|(r2>>8)|((r2&0xFF)<<16);
    r4 = r4<<1;
    for (i=0; i<r4; i++) {
      if ((i&3)==0) {
        if ((i&4)==0) {
	  dojtag(JTAG_TMS_SET32X,ja,jsel|JTAG_MSK_SET32X);	// IOB Page+Addr
	  r3 = rafifo();
	} else {
	  r3 = r0;
	}
      }
      j = 0x0800 | (r3&0xFF);		// IOB Data Write with auto address increment
      dojtag(JTAG_TMS_SET16X,j,jsel|JTAG_MSK_SET16X);
      r3 = r3>>8;
    } 
    user2done(jsel);
    return r0;
  }

  public static void prep_rwdata (int jsel, int addr, boolean by4) {
    int jmsk = jsel|JTAG_MSK_SET32X;
    dojtag(JTAG_TMS_SET32X,0x00060042,jmsk);			// Write addr of BADDR Register
    dojtag(JTAG_TMS_SET32X,0x00080008|(addr<<4),jmsk);		// Address of processor write 
    dojtag(JTAG_TMS_SET32X,by4?0x00060102:0x00060002,jmsk);	// Write addr of BDATA Register
  } 

  public static int adon_rddata (int jsel) {
    int i,j,r0,r3=0;
    for (i=0; i<32; i+=16) {
      j = 0x000D000D;						// IOB Data Read with auto address increment x4
      r0 = dojtag(JTAG_TMS_GET32X,j,jsel|JTAG_MSK_GET32X);
      r3 |= (((r0>>8)&0xFF)|((r0>>16)&0xFF00)) << i;
    }
    return r3;
  }

  public static void adon_wrdata (int jsel, int r3) {
    int i,j;
    for (i=0; i<2; i++) {
      j = 0x000C000C | ((r3&0xFF00)<<12) | ((r3&0xFF)<<4);	// IOB Data Write with auto address increment x4
      dojtag(JTAG_TMS_SET32X,j,jsel|JTAG_MSK_SET32X);
      r3 = r3>>16;
    }
  }

  // write processor bus using command FIFO
  public static int adon_wrbf (int r2, int r3, int r4, int node) {
    int cfaddr = (node<<30)|CF_XFER;				// address TA or TB or both command FIFOs
    int stat = r4;
    awBus(cfaddr,0x1ce10004);
    wBus(r2);
    wBus(r4);
    if (r4<=4) wBus(r3); 
    else for (;r4>0;r4-=4) awBus(cfaddr,rafifo());		// local bus from FIFO
    return stat;
  }

  // read response from command FIFO
  public static int adon_resp (int node, int timeout) {
    int slots,cmd,need=0,cfaddr=(node<<30);
    for (;timeout>0;timeout--) {
      slots = arBusSync(cfaddr|CF_SLOTS) & 0xFFF;
      if (slots>0) {            	// command FIFO available
	cmd = arBus(cfaddr|CF_XFER);
	if (need>0) return cmd;		// return data word
	if (cmd==0x1CE30002) need=1;	// ready for data word
      }
      else udelay(10);
    }
    return 0xFFFF9999;
  }

  // read processor bus using command FIFO
  public static int adon_rdbf (int r2, int r3, int r4, int node) {
    int cfaddr = (node<<30)|CF_XFER;				// address TA or TB or both command FIFOs
    awBus(cfaddr,0x1ce20002);
    wBus(r2);
    return adon_resp(node,8000);
  }

  // write processor bus using JTAG 
  public static int adon_wrbj (int r2, int r3, int r4, int jsel) {
    int i,j,stat=r4,jkey=0x55000077|(r4<<8);
    user2jcmd(jsel);
    prep_rwdata(jsel,4,false);					// address just after cmd
    adon_wrdata (jsel,r2);					// WRB system bus address
    if (r4<=4) adon_wrdata (jsel,r3);				// WRB system bus single data
    else for (;r4>0;r4-=4) adon_wrdata (jsel,rafifo());		// WRB system bus multiple data
    prep_rwdata(jsel,0,true);					// address cmd register with autoinc x4
    adon_wrdata (jsel,jkey);					// write special WRB command
    for (i=0; i<1000; i++) {
      j = adon_rddata (jsel);					// read special WRB command status
      if ((j&0xFF) == 0xAA) break;
      if (i>100) stat=-i;
    }
    user2done(jsel);
    return stat;
  }

  public static int adon_tc (int dmap, int port, int mode) {
    DMAStruct dma = (DMAStruct)getObjectAt(dmap);
    int r0, r1=0, r2, stat,i2;
    if ((dma.mcfg&IOM_BMUX)!=0) {
      i2 = IOC_ADDR|HYPA_CTL;
      stat = arBus(i2+IOC_STAT);	// try A side
      if ((stat&IOC_STAT_TVAL)==0) { i2 = IOC_ADDR|HYPB_CTL; r1=128; }	// try B side
    } else {
      i2 = IOC_ADDR | (((port&0x1)!=0)? HYPA_CTL : HYPB_CTL);
    }
    stat = arBus(i2+IOC_STAT);
    if ((stat&IOC_STAT_TVAL)==0) return 0;
    int cindex = dma.cindex;
    int ccycle = dma.ccycle;
    int hindex = dma.hindex;
    int hcycle = dma.hcycle;
    maddr = DMA_TCDAT;
    mdata = cindex;
    mdata = ccycle;
    mdata = hcycle;
    if (mode == TCM_SDDS) {
      r2 = i2+IOC_PKTHDR+8;
      if ((stat&IOC_STAT_TCYC)==0) r2=r2+64;
      r0 = 0x03&arBus(r2); r2=r2+4;	// format field of header
      mdata = arBus(r2); r2=r2+4;	// SDDS TC #1
      mdata = arBus(r2); r2=r2+4;	// SDDS TC #2
      mdata = arBus(r2); r2=r2+4;	// SDDS TC #3
      r1     += arBus(r2);		// tcount + B side muxoff in words
      if (r0==0)      r1 <<= 3;
      else if (r0==1) r1 <<= 2;
      else            r1 <<= 1;
      mdata = r1;			// tcount in samples
    }
    else {
      r2 = i2+IOC_RAM;
      if ((stat&IOC_STAT_TCYC)==0) r2=r2+16;
      mdata = arBus(r2); r2=r2+4;	// msgr 0
      mdata = arBus(r2); r2=r2+4;	// msgr 1
      mdata = arBus(r2); r2=r2+4;	// msgr 2
      mdata = arBus(r2); r2=r2+4;	// tcount
    }
    mdata = hindex;
    return stat;
  }

  public static int jvm_rwmem (int addr, int data, int flags, int dir) {
    int bytes = flags&0xFF;
    int sel = flags&0xFFFFFF00;
    rwiob (sel|BADDR_WR,BADDR_DSOCM|addr,2,false);
    if (dir>0) rwiob (sel|BDATA,data,bytes,false);
    else data = rwiob (sel|BDATA,0,bytes,true);
    return data;
  }
  public static int  jvm_rmem (int addr,           int flags) { return jvm_rwmem(addr,  0 ,flags,-1); }
  public static int  jvm_wmem (int addr, int data, int flags) { return jvm_rwmem(addr,data,flags,1); }

  public static void processNVME (int cmd) {
    int device,blocks,nblk,nblk2,dblkl,dblku,soff,sblkl,sblku,mblkl,mblku;
    int idev,route=0,dport,dport2,cmds,timeout;
    int cfa,invme=0,cmstart,cmstop,cmstart2,cmstop2;
    int maxblk=256,maxblk2=512;
    boolean dual;

    cmds   = arBusSync(REG_AHAR0);	// command
    blocks = cmds>>16;			// blocks to transfer
    device = (cmds>>8)&0xFF;		// device minor ID
    cmd    = cmds&0xFF;			// command type
    dblkl  = arBus(REG_AHAR0+4);	// lower disk block address
    mblkl  = arBus(REG_AHAR0+8);	// lower mem block address
    dblku  = arBus(REG_AHAR0+12);	// upper mem|disk block address
    soff   = arBus(REG_AHAR0+16);	// partition and superblock offset
    mblku  = dblku>>16;
    dblku  = dblku&0xFFFF;
    dport2 = PM_TUNB;

   while (blocks>0) {

    nblk = blocks;
    dual = false;

    if (device==3) {		// handle striping - always by 2 with stripesize of 128K or 2^17 or 256 blocks
      idev  = (dblkl>>8)&0x1;
      sblkl = (dblkl&0xFF);
      dual  = (idev==0) && (sblkl==0) && (blocks>=maxblk2);
      if (sblkl+nblk>maxblk) nblk = maxblk-sblkl; 	// cant write past a stripe boundary
      sblkl = ((dblku<<31)|((dblkl>>1)&0x7FFFFF00)|sblkl) + soff;
      sblku = (dblku>>1);
    } else {			// no striping
      idev  = device-1;
      if (nblk>maxblk) nblk=maxblk;
      sblkl = dblkl + soff;
      sblku = dblku;
    }
    if (sblkl>=0 && sblkl<soff) sblku++;	// wrapped
    nblk2 = nblk;

    if (idev==1) {
      dport = PM_TUNB;
      route = ROUTE_R2TB;
      cfa   = 0x80000034;
    } else if (dual) {
      dport = PM_TUNA;
      route = ROUTE_R2TA|ROUTE_R2TB;
      cfa   = 0xC0000034;	// command FIFO address - both
      nblk2  = nblk<<1;
    } else {
      dport = PM_TUNA;
      route = ROUTE_R2TA;
      cfa   = 0x40000034;
    }

    cmstart  = ((invme&0x3)+1) << 20;
    cmstop   = cmstart + (nblk<<9);
    cmstart2 = cmstart + 0x20008;
    cmstop2  = cmstart + (nblk2<<9);
    cmstart |= 0x8;		// need to set CirBuf size > transfer

    if (invme==0) {		// if first pass
      wait4dma(MCSR_MENA);	// wait till HP queue is cleared
      wait4route(ROUTE_HP2R);
    }

    // read block
    if (cmd==0x41) {
      setupDMAa(idev,cmstart,-1);
      if (dual) setupDMAa(1,cmstart2,-1);
      awBus(cfa,0x1CE0003D);	// send read to NVME command Fifo address
      wBus(nblk);
      wBus(sblkl);
      wBus(sblku);
      timeout=10000;
              while (waitDMAa(idev,cmstop,0x10)==0 && timeout>0) { timeout-=1; lastDir=0; processDMAx(); } // while waiting 
      if (dual) while (waitDMAa(1,cmstop2,0x10)==0 && timeout>0) { timeout-=1; lastDir=0; processDMAx(); }
      if (timeout<=0) { cmds |= 0x80; break; }

      wait4dma(MCSR_MENA);	// wait till last one has finished
      awBus(DMAC_HPCNT,0);
      awBus(REG_AMDAR,(mblkl<<9));
      awBus(REG_AMDAU,(mblku<<9)|(mblkl>>23));
      awBus(REG_AMDTC,nblk2<<9);
      setupDMAi(PM_HOST,cmstart,cmstop2+0x100);
      awBus(DMAC_ROUTE_SET,ROUTE_R2HP);
      awBus(REG_AMCSR,MCSR_MASTER_WR);
      lastDir=-1;
    }

    // write block
    if (cmd==0x42) {
      awBus(DMAC_HPCNT,0);
      awBus(REG_AMDAR,(mblkl<<9));
      awBus(REG_AMDAU,(mblku<<9)|(mblkl>>23));
      awBus(REG_AMDTC,nblk2<<9);
      awBus(REG_AMCSR,MCSR_MASTER_RDM);
      lastDir=1;
      wait4dma(MCSR_QSTAT);	// wait till this one has started
      setupDMAo(PM_HOST,cmstart,cmstop2);
      awBus(DMAC_ROUTE_SET,ROUTE_HP2R);
      wait4route(ROUTE_HP2R);
    processDMAx();		// see if anything else to do
    wait4dma(MCSR_MENA);	// wait till HP queue is cleared
    wait4route(ROUTE_HP2R);
      wait4route(route);
      awBus(cfa,0x1CE0003E);	// send write to NVME command Fifo address
      wBus(nblk);
      wBus(sblkl);
      wBus(sblku);
      setupDMAi(dport,cmstart,cmstop);
      if (dual) setupDMAi(dport2,cmstart2,cmstop2);
      awBus(DMAC_ROUTE_SET,route);
    }

    // prep for next stripe
    blocks -= nblk2;
    mblkl  += nblk2;
    dblkl  += nblk2;
    if (dblkl>=0 && dblkl<nblk2) dblku++;	// wrapped
    invme  += (dual?2:1);

   }
    if (cmd==0x41) wait4dma(MCSR_MENA);	
    if (cmd==0x42) wait4route(route);
    awBus(REG_AHAR0,cmds^0x40);
    lastDir=0;
  }

}
