package nxm.ice.lib;

import nxm.ice.lib.IceHW;

/*
   IceJVM is the base class for all JVM instatiations in Ice hardware.

   By default, JVM addresses are local to the JVM space

   NOTE: these are the non-standard behaviors of this compiler

   The saddr,sdata registers are special and export a system bus
   The maddr,mdata registers are special and export a memory bus
   The ioadr,ioctl registers are special and export an I/O DMA bus

   The >>> and >> operators both act as >> in verilog, or as unsigned

   The x++ and x-- operators only use the lower 17 bits and do not change the upper 15 bits.

*/

public class IceJVM implements IceHW {

  // special static vars recognized by Ice JVM loader
  public static int saddr,sdata; // system bus
  public static int maddr,mdata; // memory bus
  public static int ioadr,ioctl; // IO bus

  // special native routine to access structures in memory
  public native static Object getObjectAt (int addr);

  // need to access run method to have entry in Constant Pool - not called
  static void main() { run(); }

  // this needs to be over-ridden in sub class
  static void run() { }

  // micro delay routine in clock ticks
  protected static void udelay (int usec) {
    for (int i=0; i<usec; i=i+1) {
      for (int j=0; j<16; j=j+1);
    }
  }

  // integer divide pseudo routine
  static int _idiv (int num, int denom) {
    boolean neg = (num<0);
    if (num<0) num = -num;
    if (denom<0) { denom = -denom; neg = !neg; }
    int a=num, b=0, i=0;
    while (a!=0) { a=a>>4; i=i+4; }		// find leftmost non-zero nibble
    while (i-- > 0) {				// appends one bit from numerator to a
      a = (a << 1) + ((num & (1 << i)) >> i);
      b = b << 1;
      if (a >= denom) { a=a-denom; b=b+1; }
    }
    return neg? -b : b;
  }

  static void stubs() {
    int a=0,b=0;
    _idiv(a,b);
  }

  // external special access routines to communicate outside the JVM

  // write a word to the 32b system bus
  protected static void awBus (int addr, int data) {
    saddr = addr;
    saddr = addr;
    sdata = data;
  }

  protected static void aBus (int addr) {
    saddr = addr;
  }
  protected static void wBus (int data) {
    sdata = data;
  }
  protected static int rBus () {
    return sdata;
  }

  // read a word from the 32b system bus
  protected static int arBus (int addr) {
    saddr = addr;
    saddr = addr;
    saddr = addr;
    saddr = addr; // make sure address is presented for ample clocks
    return sdata;
  }

  // read a word from the 32b system bus twice
  protected static int arBusAck (int addr, int timeout) {
    int data;
    saddr = addr;
    data = sdata;
    for (int i=0; i<timeout; i=i+1);
    return sdata;
  }

  // read the same word twice from the 32b system bus
  protected static int arBusSync (int addr) {
    int data,datal;
    saddr = addr;
    saddr = addr;
    data = sdata;
    do {
      datal = data;
      data = sdata;
    } while (data!=datal);
    return data;
  }

  // write a word to the 32b memory bus
  protected static void awMem (int addr, int data) {
    maddr = addr;
    mdata = data;
  }

  // read a word from the 32b memory bus
  protected static int arMem (int addr) {
    maddr = addr;
    return mdata;
  }

  // read a word from the 32b memory bus
  protected static int arMemSync (int addr) {
    int data,datal;
    maddr = addr;
    data = mdata;
    do {
      datal = data;
      maddr = addr;
      data = mdata;
    } while (data!=datal);
    return data;
  }

  // higher level processMessage functions
  
  // read multiple words from the system bus into a message buffer
  static void busr (int addr, int madr, int size) {
    saddr = addr;
    maddr = madr;
    for (int i=0; i<size; i++) mdata = sdata;
  }

  // write multiple words to the system bus from a message buffer
  static void busw (int addr, int madr, int size) {
    saddr = addr;
    maddr = madr;
    for (int i=0; i<size; i++) sdata = mdata;
  }

  // read/write multiple words to the system bus from a message buffer
  static void busrwm (int addr, int madr, int size) {
    saddr = addr;
    maddr = madr;
    int waddr = mdata;
    int wdata = mdata;
    maddr = madr;
    for (int i=0; i<size; i++) {
      saddr = waddr;
      saddr = waddr;
      sdata = wdata;
      udelay(10);
      saddr = addr;
      udelay(10);
      mdata = sdata;
    }
  }

  final static int rmax=64;
  final static int wmax=64;

  // read multiple words from the memory bus into a message buffer
  static void memr (int addr, int madr, int size) {
    int data; 
    if ((addr&0xFFF00000) > 0) {	// DRAM
      for (size<<=2; size>rmax; size-=rmax) {
        maddr = madr;
        ioadr = addr;
        ioctl = IOCTL_RD|rmax;
	madr+=rmax; addr+=rmax;
      }
      maddr = madr;
      ioadr = addr;
      ioctl = IOCTL_RD|size;
    } else {				// BRAM
      for (int i=0; i<size; i++) {
        maddr = addr; data = mdata; addr+=4;
        maddr = madr; mdata = data; madr+=4;
      }
    }
  }

  // write multiple words to the memory bus from a message buffer
  static void memw (int addr, int madr, int size) {
    int data;
    if ((addr&0xFFF00000) > 0) {	// DRAM
      for (size<<=2; size>wmax; size-=wmax) {
        maddr = madr;
        ioadr = addr;
        ioctl = IOCTL_WR|wmax;
	madr+=wmax; addr+=wmax;
      }
      maddr = madr;
      ioadr = addr;
      ioctl = IOCTL_WR|size;
    } else {				// BRAM
      for (int i=0; i<size; i++) {
        maddr = madr; data = mdata; madr+=4;
        maddr = addr; mdata = data; addr+=4;
      }
    }
  }

  // read multiple addresses from the message buffer and write to DRAM controller for initialization
  static void memi (int madr, int size) {
    for (int i=0; i<size; i++) {
      maddr = madr;
      ioadr = mdata;
      ioctl = IOCTL_WR|0x4;
      madr += 4;
      udelay(50);
    }
  }

  // zero multiple words on the memory bus
  static void memz (int madr, int size) {
    maddr = madr;
    for (int i=0; i<size; i=i+1) mdata = 0;
  }

}
