package nxm.ice.lib;

import nxm.ice.lib.IceHW;

/* 
            WR    RD
  saddr[16] SYS / Status
  saddr[17] IRB / Ram
  saddr[18] VCO
  saddr[19] OPPS Output Delay in 6ns ticks

  Status = 0x0CBBBBBA - C=currentTenths B=10MHzTickLoc A=160MHzTickLoc
  VCO    = 0x0000AAAA - A=VoltageOffset 0 to 0xFFFF

  OCXO has stability of +-50ppb and tuning range of +-(5 to 10)ppm with V from 1.4 to 2.4 volts  (~15ppm/V)
  D2A  is 16 bit from 0 to 2.5 V   (2.5V/2^16)

  so 1/160 ppm would be fixed by (1/160)*(1V/15ppm)(2^16/1V) or 27 point D2A diff
  since we average the differences by 64 and use 32 point D2A diff, we shift the the dvco = acc_dif>>1

  empirically, this shift is about 4 times too weak, so we use acc_dif<<1

  Cfg register
  bit 0 - GoodConfig
  bit 1 - APWR
  bit 2 - AFVP
  bit 3 - FixedMode
  bit 4 - PTST
  bit 5 - NPPS
  bit 6 - DPPS - Direct pps from CW125
  bit 7 - External Operation

  this now relates 160MHz tick offsets to dVCO 

  Status:
  0 - startup, wait for N satellites in view, flywheel at initial VCO settings
  1 - rough track VCO setting
  2 - fine track VCO setting
  3 - fine track good
  4 - fine track good still
  5 - normal operation with phase tracking correction 
  6 - normal operation with phase tracking correction averaged at least one minute

 */
public class IceGPS extends IceIOM {

  // normal static vars
  public static int lvco,loff,mode,calib;
  public static int dfvco,dpvco,cvco;		// calib VCO dFreq and dPhase and calib
  public static int acc_off,acc_dif,acc_sav;	// accumulators
  public static int nos;			// number of satellites
  public static int dop;			// dilution of position
  public static int sog;			// speed over ground
  public static int count;			// count of seconds at a given stage
  public static int week,sow,alt,lat,lon;	// for xfer to host
  public static int mrev,debug;
  public static int mindop;

  public static void run() {

    // legacy 
    loff = 0x0;
    lvco = 0x94000000;
    mode = 0;
    dop  = 10;
    sog  = 100;
    nos  = 0;
    count = 0;
    mindop = 16;		// 16 essentially disables this check
    awBus(0x10000,0x1); 	// initial sync

    awBus(0x80000,0x16); 	// OPPS offset delay to match 10MHz filter (6ns ticks) 
				// theoretical group delay of filter at 10MHz is 136ns or (22*6nS)
				// align the 10M to negative this amount so the signals after delay are aligned
    int cfg    = arMem(0x0008);
    mrev       = (cfg>>24)&0xFF;
    lvco = (mrev<=1)? 0x88000000 : 0x94000000;		// rev 1 and VCO initial offsets

    for (int beat=1;;beat++) {
      awMem(HBT,beat);					// heartbeat monitor
      if (mode==0 && beat<1000) continue;		// startup heartbeat
      int status = getStatus();				// poll for counter to advance
      int tenths = getTenths(status);
      if (tenths==0) awMem(0x0100, 0);
      if (tenths==1) {
        processTick (status);				// update VCO register
        while (getStatusTenths()==1) udelay(10);	// wait for > .1 sec in
      }
      if (tenths==3) {
        processNEMA (status);				// update IRIG register
        while (getStatusTenths()==3) udelay(10); 	// wait for > .3 sec in
      }
    }

  }

  public static int getStatus() {
    return arBusSync(0x10000);
  }
  public static int getTenths (int status) {
    return (status>>28)&0x7;
  }
  public static int getStatusTenths() {
    return getTenths(getStatus());
  }

  public static void processTick (int status) {

    int off = (status&0x0FFFFFFF);
    // make 2's complement difference
    if ( off > 80000000 ) off = off-160000000;	// not hex
    awMem(0x0001C, status);
    int cfg    = arMem(0x0008);
    int offset = (cfg>> 8)&0xFF;
    int navg   = (cfg>>16)&0xFF;
    mrev       = (cfg>>24)&0xFF;

    if ((cfg&0x80)!=0) { 	// Manual mode
      int cmd = arMem(0x000C);
      if (cmd!=0) {
        lvco=cmd;
	count = -4;
      }
      mode=8;
      awMem(0x000C,0);
    }
    if (mode==0 && (cfg&0x08)!=0) { count=-4; mode=-1; } // FIXED mode

    // adjust for FPGA pipeline delays (2 x 10MHz ticks + 1 160MHz)
    off = off - offset;

    // force resync if 10uS out of range 
    if ( (mode==6) && ((off > 1600) || (off < -1600)) ) count = -4;

    // get the current difference
    int dif = off - loff;

    if (count <= -4 && mode==6) lvco = cvco;	// resync while tracking
    if (count <= -4) awBus(0x10000, 0x1);	// resync

    if (count>=0 && count<64) {
      setOff(count,off);
      setDif(count,dif);
    }
    count += 1; 
    if (nos>6 && dop<mindop) {			// tracking enough satellites
      if (mode==0) { mode=1; count=-4; }	// enter calibration freq #1
    }
    if (mode>0 && (nos<3 || dop>mindop)) {	// lost it
      mode=0;					// start over
    }

    if (mode>=1 && mode<=4 && count>=navg) {
      acc_dif = getAccDif(navg);
      dpvco = 0;
           if (acc_dif > 40)   { dpvco = -0x100; mode=1; }
      else if (acc_dif > 20)   { dpvco = -0x80;  mode=1; }
      else if (acc_dif > 2)    { dpvco = -0x10;  mode=2; }
      else if (acc_dif < -40)  { dpvco =  0x100; mode=1; }
      else if (acc_dif < -20)  { dpvco =  0x80;  mode=1; }
      else if (acc_dif < -2)   { dpvco =  0x10;  mode=2; }
      else if ( ++mode > 4)    { cvco=lvco; mode=6; }	// resync and track
      lvco = lvco + (dpvco<<16);
      count = -4;
    }   
    else if (mode==6 && count>=0) {
      acc_off = getAccOff(1);
      acc_dif = getAccDif(1);
      dpvco = 0;
      if (acc_off > 0 && acc_dif >= 0) { dpvco = -0x02; }
      if (acc_off < 0 && acc_dif <= 0) { dpvco =  0x02; }
      lvco = lvco + (dpvco<<16);
      count=0;
    }
    else if (mode<0 && count>0) {
      count = 0;
    }

    loff = off;

    // update VCO value
    awBus(0x40000,lvco>>16);

    // debug information
    awMem(0x00020, (mode<<24)|((count&0xFF)<<16)|(dop<<8)|nos);
    awMem(0x00024, status);
    awMem(0x00028, off);
    awMem(0x0002C, dif);
    awMem(0x00030, acc_off);
    awMem(0x00034, acc_dif);    
    awMem(0x00038, lvco);
    awMem(0x0003C, calib);
    awMem(0x00040, dpvco);
    awMem(0x00044, dfvco);
  }

  public static void setOff (int i, int off) {
    awMem(0x0200+(i<<2),off);
  }
  public static int getAccOff (int navg) {
    int i,avg=0;
    for (i=0; i<navg; i++) {
      avg = avg + arMem(0x0200+(i<<2));
    }
    return avg;
  }

  public static void setDif (int i, int dif) {
    awMem(0x0300+(i<<2),dif);
  }
  public static int getAccDif (int navg) {
    int i,avg=0;
    for (i=0; i<navg; i++) {
      avg = avg + arMem(0x0300+(i<<2));
    }
    return avg;
  }

  public static void copyNEMA (int len) {
    awMem(0x0100, 1);
    awMem(0x0104, (week<<16)|(dop<<12)|(mode<<8)|nos );
    awMem(0x0108, sow );
    awMem(0x010C, alt );
    awMem(0x0110, lat );
    awMem(0x0114, lon );
    awMem(0x0118, len );
    awMem(0x0100, 2);
  }

  public static void processNEMA (int status) {
    int iadr,nadr,len;
    // NMEA returns around 10 lines
    for (iadr=nadr=0x20000; iadr<0x20400; iadr=nadr) {
      nadr = checkLine(iadr); 
      if (nadr==0) break;
    }
    len = iadr-1-0x20000;
    copyNEMA(len);
  }

  // returns int of decimal * 10^dp decimal places
  public static int parseDec (int iadr, int nadr, int dp) {
    boolean neg=false,pd=false; // past decimal
    int c,val=0; 
    for (;iadr<nadr;) {
      c = arBus(iadr++);
      if (c==46) pd=true;
      else if (c==43);		// +
      else if (c==45) neg=true;	// -
      else if (c<48) break;	// NaN
      else if (c>57) break;	// NaN
      else {
	val = val*10 + (c-48);
	if (pd) dp--;
      }
      if (pd && dp<=0) break; 
    }
    if (neg) val = -val;
    return val;
  }

  public static int checkLine ( int iadr ) {
    // get the relevant characters
    int a = arBus(iadr+3);
    int b = arBus(iadr+4);
    int c = arBus(iadr+5);
    int nadr;

    if ( a==82 && b==77 && c==67 ) {		 // GPRMC (get position & SOG)
      // get speed over ground in knots 
      iadr = nextField(iadr, 7);
      nadr = nextField(iadr, 1);
      sog  = parseDec(iadr,nadr,0);
      awMem(0x0006C,sog);
    }
    else if ( a==71 && b==71 && c==65 ) {	 // GPGGA - geographic fix
      // latitude
      iadr = nextField(iadr, 2);
      nadr = nextField(iadr, 1);
      lat = parseDec(iadr,nadr,4);
      if (arBus(nadr)==83) lat=-lat;	// S
      awMem(0x00050,lat);
      // longitude
      iadr = nextField(nadr, 1);
      nadr = nextField(iadr, 1);
      lon = parseDec(iadr,nadr,4);
      if (arBus(nadr)==87) lon=-lon;	// W
      awMem(0x00054,lon);
      // altitude
      iadr = nextField(nadr, 4);
      nadr = nextField(iadr, 1);
      alt = parseDec(iadr,nadr,2);
      awMem(0x00058,alt);
    }
    else if ( a==71 && b==83 && c==65 ) {	 // GPGSA - DOP and active satellites
      iadr = nextField(iadr, 15);
      nadr = nextField(iadr, 1);
      dop = parseDec(iadr,nadr,0);
      if (dop>0xF) dop=0xF;
      awMem(0x0005C,dop);
    }
    else if ( a==71 && b==83 && c==86 ) {	 // GPGSV - Sattelites in View
      // Satellite information is 3 fields
      iadr = nextField(iadr, 3);
      nadr = nextField(iadr, 1);
      nos = parseDec(iadr,nadr,0);
      awMem(0x00060,nos);
    }
    else if ( a==76 && b==89 && c==84 ) {	 // POLYT (get week sow lat lon alt)
      // $POLYT,003640.010,060180,2200.0103682,000,2215.0103725,-2372461,-1071.077,25,2215008,99999,99999,*62
      int sec=0, day=0, mon=0, year=0;
      // field 1 is HHMMSS
      iadr = nextField(iadr, 1);
      nadr = nextField(iadr, 1);
     if ( nadr-iadr >= 6) { // check for a valid field length
      sec = arBCD(iadr, 5);
      sec += 10*arBCD(iadr, 4);
      sec += 60*arBCD(iadr, 3);
      sec += 600*arBCD(iadr, 2);
      sec += 3600*arBCD(iadr, 1);
      sec += 36000*arBCD(iadr, 0);
     }
      // get DDMMYY (1 field away from nadr)
      iadr = nextField(iadr, 1);
      nadr = nextField(iadr, 1);
     if ( nadr-iadr >= 6) { // check for a valid field length
      day  = 10*arBCD(iadr,0) + arBCD(iadr,1); // day 
      mon  = 10*arBCD(iadr,2) + arBCD(iadr,3); // month
      year = 10*arBCD(iadr,4) + arBCD(iadr,5); // year
     }
      // update the IRIG-B values
      if (sec!=0 && year!=0) processIRB (year,mon,day,sec);
      // GPS week
      iadr = nextField(iadr, 2);
      nadr = nextField(iadr, 1);
      week = parseDec(iadr,nadr,0);
      if (mrev<=1) week += 1024;	// rev 1 GPS week rollover
      awMem(0x00064,week);
      // GPS seconds of week
      iadr = nextField(nadr, 1);
      sow  = parseDec(nadr,iadr,3);
      awMem(0x00068,sow);
    }
    else if ( a==76 && b==89 && c==83 ) {	// stop at POLYS
      return 0;
    }
    return nextLine(iadr, 200);
  }

  public static void processIRB ( int year, int mon, int day, int sec ) {
    int irb0=0,irb1=0,irb2=0;
    int cn,dn;
    int doy = getDoY(day,mon,year);

    // set readback fields for host monitor
    awMem(0x00010,year);
    awMem(0x00014,doy);
    awMem(0x00018,sec);

    sec = sec + 1; // increment for next tic 
    if (sec>=86400) {
      doy = doy+1;
      sec = sec-86400;
    }

    // set 96 bit IRIG-B shift register
    irb2 |= (sec<<8);     // binary seconds field

    dn=36000; for (cn=0; sec>=dn; cn++,sec-=dn); irb0 |= (cn<<23); // tens hours
    dn=3600;  for (cn=0; sec>=dn; cn++,sec-=dn); irb0 |= (cn<<18); // ones hours
    dn=600;   for (cn=0; sec>=dn; cn++,sec-=dn); irb0 |= (cn<<14); // tens minutes
    dn=60;    for (cn=0; sec>=dn; cn++,sec-=dn); irb0 |= (cn<<9);  // ones minutes
    dn=10;    for (cn=0; sec>=dn; cn++,sec-=dn); irb0 |= (cn<<6);  // tens seconds
    cn = sec;          irb0 |= (cn<<1);  // ones seconds

    dn=100;   for (cn=0; doy>=dn; cn++,doy-=dn); irb1 |= (cn<<4);  // hund days
    dn=10;    for (cn=0; doy>=dn; cn++,doy-=dn); irb1 |= (cn<<0);  // tens days
    cn = doy;          irb0 |= (cn<<27); // ones days

    dn=10;  for (cn=0; year>=dn; cn++,year-=dn); irb1 |= (cn<<18); // tens year
    cn = year;           irb1 |= (cn<<13); // ones year

    awBus(0x20000,irb0);
    awBus(0x20004,irb1);
    awBus(0x20008,irb2);
  }

  /** Parse bus for the NMEA start character ($)

      @param adr starting address
      @param len maximum length of characters
      @return starting address of NMEA string
   */
  public static int nextLine ( int adr, int len ) {
    // increment adr (looking for the next line, not this one) and 
    // check for the characters: '$' and '#'
    for ( adr++; (arBus(adr)!=36) && (arBus(adr)!=35) && (arBus(adr)!=0) && len>0; adr++, len-- );
    return adr;
  }


  /** Look for a specific character in a stream 

      @param iadr starting address
      @param eind ending address
      @param ac character
      @return address of chracter
   */
  public static int findChar ( int iadr, int eind, int ac ) {
    for ( ; (eind>iadr) && (arBus(iadr)!=ac); iadr++ );
    return iadr;
  }

  public static int arBCD (int adr, int off) {
    int bcd = arBus(adr+off)-48;	// char 48 is a 0
    return bcd;
  }

  public static int nextField (int adr, int cc) {
    for (;cc>0;) {
      int c = arBus(adr++);
      if (c==44) cc--;	// char 44 is a , 
      if (c==42) cc--;	// char 42 is a * 
      if (c==0) break;	// end of string
    }
    return adr;
  }

  public static int getDoY (int day, int mon, int year) {
    if (mon>1) day += 31; // add Jan
    if (mon>2) day += 28;
    if (mon>3) day += 31;
    if (mon>4) day += 30;
    if (mon>5) day += 31;
    if (mon>6) day += 30;
    if (mon>7) day += 31;
    if (mon>8) day += 31;
    if (mon>9) day += 30;
    if (mon>10) day += 31;
    if (mon>11) day += 30; // add Nov
    if (mon>2 && (year&0x3)==0) day += 1; // leap year
    return day;
  }

  public static int addBCD (int word, int bcd, int shft) {
    return word | (bcd<<shft);
  }

}
