package nxm.ice.lib;

import nxm.sys.inc.*;
import nxm.sys.lib.*;

/**
  Helper to parse and generate indexed filenames for Sink/Sourcepic ArchSFN modes. 
  It looks for the last numeric characters bordered by non-alphabetic characters
  to auto generate the format string needed to generate the progression.
  It can also be over-ridden by the user.

  @author Jeff Schoen
  @version %I%, %G%
*/
public class ArchSFN {

  private String format;
  private String quals;
  private int indx,indx0,indxt,nsf,xfer=1;
  private boolean legit;
  private Archiver.ToC toc;
  private Object ref;
  private double clen,flen,tlen;
  private double rate=1,top,toff,dur=-1;

  /** Class for Archive filename handling */
  public ArchSFN () {
  }

  /** Class for Archive filename handling 
  @param ref	Midas reference
  @param fn	Root filename
  @param nsf	Number of seperate file
  */
  public ArchSFN (Object ref, String fn, int nsf) {
    this.ref = ref;
    setSize(nsf);
    setSFN(fn);
  }

  public boolean setSFN (String fn) {
    int i,j=-1,k=-1;
    int p = fn.indexOf("%");
    int q = fn.indexOf("{");	// qualifiers
    int t = fn.indexOf("(");	// trimmers
    if (t>0 && (t<q || q<0)) q=t;
    if (q>0) {
      quals = fn.substring(q);
      fn = fn.substring(0,q);
    }
    int s = fn.lastIndexOf("/")+1;
    // look for a table of contents
    char cl='.';
    for (i=fn.length()-1; i>s; i--) {
      char c = fn.charAt(i);
      if (k<0 && isDigit(c) && isDelimit(cl)) k=i;
      if (k>0 && isDigit(c)) j=i; 
      if (k>0 && !isDigit(c)) break;
      cl = c;
    }
    int n = (k<0 || j<0)? 0 : k-j+1;
    if (p>=0) {			// format specified
      format = fn;
      indx = indx0 = 0;
    }
    else if (nsf==0 && n>0) {	// format implied
      boolean lz = (n>1 && fn.charAt(j)=='0'); // leading zeros
      String form = lz? "%0"+n+"d" : "%d";
      format = fn.substring(0,j)+form+fn.substring(k+1);
      indx = indx0 = Convert.s2l(fn.substring(j,k+1));
    }
    else {			// format default generation
      format = fn+"_%d";
      indx = indx0 = 1;
      // look for TOC file
      toc = new Archiver.ToC(ref,fn+"_toc",nsf,-1,DataFile.INPUT|DataFile.OPTIONAL);
      if (toc.isOpen()) {	// get current stats
	nsf = toc.size();
	tlen = toc.getLength(); clen=0;
        indx = 0; nextIndex();
        if (toc.tocr.getRate()>0) rate = toc.tocr.getRate();
        System.out.println("Using TOC file="+toc.df.getURL()+" entries="+nsf+" time="+tlen);
      }
      else toc = null;
    }
    indxt=indx0;
    legit = true;
    return legit;
  }

  public int nextIndex() {
   while (true) {
    indx = indx+1;
    if (nsf>0 && indx-indx0>=nsf) indx=indx0; // wrap if last index is specified
    if (toc!=null) {
      long cleft = (long)((dur-clen)*rate);
      if (dur>0 && cleft<xfer) indx=indxt;
      if (indx==indxt) clen=0;
      toc.read(indx-1);
      double t=toc.tocr.getTime(); 
      double d=toc.tocr.getLength();
      if (dur>0 && (t+d<top || t>top+dur)) continue;	// not from this event
    }
    break;
   }
    return indx;
  }

  public String nextSFN() {
    indx = nextIndex();
    return getSFN();
  }

  public String getSFN() {
    return getSFN(indx);
  }

  public String getSFN (int index) {
    String fn = legit? String.format(format,index) : format;
    if (fn.startsWith("http:")) fn += ".tmp";
    if (toc!=null) {
      int a=0,b=0;
      double t=toc.tocr.getTime(); 
      flen = toc.tocr.getLength();
      if (index==indxt) flen -= toff;
      clen += flen;
      if (toc.tocr.getRate()>0) rate = toc.tocr.getRate();
      if (index==indxt && toff>0) a = xfer*(int)(toff*rate/xfer);
      if (dur>0 && clen>dur) b = xfer*(int)((clen-dur)*rate/xfer);
      if (a!=0 || b!=0) fn += "("+a+":-"+b+")";
    }
    if (quals!=null) fn += quals;
    return fn;
  }

  public void setTL (int xfer) {
    this.xfer = xfer;
  }

  public void setTop (double top) {
    this.top = top;
    if (top<0) return;
    int i=0;
    double t=0, d=1;
    if (toc!=null) {
      for (i=0; i<nsf; i++) {
	toc.read(i); 
	t=toc.tocr.getTime(); 
	d=toc.tocr.getLength();
	if (t<=top && top<t+d) break; 
      }
      if (i==nsf) { System.out.println("ArchSFN top time="+t+" not found. Starting at 1st record."); i=0; toc.read(i); }
      indx=indxt=i+1;
      clen=0;
      toff = top-t;
    }
  }

  public void setDur (double dur) { 
    this.dur = dur;
  }

  public int getIndex () { return indx; }
  public void setIndex (int index) { indx=index; }
  public int getSize () { return nsf; }
  public void setLast (int index) { nsf=index-indx0+1; }
  public int getLast () { return nsf+indx0-1; }
  public void setSize (int size) { nsf=size; }
  public int getIndex0 () { return indx0; }
  public void setIndex0 (int index) { indx0=index; }
  public String getFormat () { return format; }
  public void setFormat (String form) { if (form!=null) format=form; }
  public boolean isLegit() { return legit; }
  public int resetIndex() { indx=indx0; return indx; }
  public boolean isFirst() { return (indx==indx0); }
  public boolean isTop() { return (indx==indxt); }
  public int getOffset () { return indx-indx0; }
  public double getLength () { return (dur>0)? dur : tlen; }
  public double getLengthTo (double offset) { return clen - flen + offset/rate; }
  public double getTimeTop () { return top; }
  public double getTimeDur () { return (dur>0)? dur : tlen; }

  private boolean isAlpha (char c) { return (c>='a' && c<='z') || (c>='A' && c<='Z'); }
  private boolean isDigit (char c) { return (c>='0' && c<='9'); }
  private boolean isDelimit (char c) { return (c=='.' || c=='-') || c=='_' || c=='/'; }
}
