/**
  Description:	Header file for data packet headers 

  Author:	Jeff Schoen

  $Revision:	$
  $Modtime:	$

*/

/* 
  Network overhead per packet
  12by - dead time between packets
   8by - ethernet preamble (0x1FB,0x55,0x55,0x55,0x55,0x55,0x55,0x5D)
  14by - ethernet header
  20by - IP header
   8by - UDP header
 ~64by - Data Packet header
 ~1Kby - Data Packet data
   4by - ethernet trailer FCS
   4by - ethernet postamble (0x1FD,0x07,0x07,0x07)

  Total network+ethernet+UDP overhead per packet = 70by

  For 1k by SDDS or ICE packets
  Total of ~134by/1024by = 13% overhead
  Max 1G throughput = 125 Mby / 1.12 = 111 Mby/sec data rate
  Max 10G throughput = 1250 Mby / 1.12 = 1110 Mby/sec data rate
*/

typedef struct {
  real_8 start,delta;
} ICEPKTX;

typedef struct {
  int_4 ramphys,ramsize,dataoffset,datasize;
} ICEPKTR;

/* 
  ICE Packet Structure
*/
typedef struct {
  int_1	 keys[4];	/* barker syncs[2:0] and packet type[3] */
  int_4	 count;		/* packet number since start of xfer */
  int_4	 elem;		/* data elements in packet */
  int_2	 user;		/* user defined field */
  int_2	 channel;	/* port channel number */
  int_1	 id1,id2;	/* subsystem IDs */
  int_1	 rep;		/* data rep (Vax,Ieee,Eeei) */
  int_1	 bpa;		/* bytes per atom=(mode,type) */
  int_1	 mode;		/* data format mode (Complex, Scalar) */
  int_1	 type;		/* data format type (Pack,Byte,Int,Long) */
  int_1	 tcmode;	/* timecode mode */
  int_1	 tcstatus;	/* timecode status */
  double tcoff;		/* sample offset from data[0] to TC */
  double tcwsec;	/* timecode 1950 whole seconds */
  double tcfsec;	/* timecode fractional seconds */
 union {
  real_8 dval[2];	/* user fields */
  real_4 fval[4];	/* user fields */
  int_4	 lval[4];	/* user fields */
  int_2	 ival[8];	/* user fields */
  int_1	 bval[16];	/* user fields */
  ICEPKTX X;
  ICEPKTR R;
 } u;
} ICEPACKETSTRUCT;

#define ICE_HDR_SZ 64

typedef struct {
  int_1 func; 		/* defines the method to call */
  int_1 flag; 		/* is the protocol version number and options mask */
  int_1 info; 		/* channel number for fast routing of monitored properties */
  int_1 rep;  		/* packet data representation (VAX|IEEE|EEEI) */
  int_1 seq;  		/* sequence number (0-127) for Reliable packets */
  int_1 try_;  		/* number of resends for Reliable packets */
  int_1 rpt;  		/* report sequence number */
  int_1 adj;  		/* number of bytes in adjunct header */
} RMIFPACKETSTRUCT;

/* 
  SDDS Packet Structure

  56by - SDDS header
  1Kby - SDDS data

*/
typedef struct {
  int_u1 fid;		/* format identifier {SF,SOS,PP,OF,SS,DM:3} */
  int_u1 bits;		/* bits per data word {CX,SNP,VW,BPS:5} 4|8|16 */
  int_u2 frame;		/* frame sequence number (0x1F=parity packet) */
  int_u2 tcinfo;	/* Time Tag info {1msV,ttV,sscV,00,1msPtr[10:0]} */
  int_u2 tccorr;	/* Time from sample to 1msec hack in 250psec tics (min SR=60kHz) */ 
  int_u8 tctics;	/* Number of 250psec tics since jan 1 to timecode hack */
  int_u4 tcext;		/* Timecode extended precision (250psec/2^32) */
  int_u1 sscinfo[12];	/* SSC information {DFDT:32,F:64} */
  int_u1 ssdinfo[4];	/* SSD information */
  int_u1 aadinfo[20];	/* AAD information */
  int_u1 data[1024];	/* packet data */
} SDDSPACKETSTRUCT;

#define SDDS_HDR_SZ 56
#define SDDS_PKT_SZ 1080

typedef struct {
  int_u2 keys;		/* barker packet sync 0x55AA */
  int_u2 port;		/* UDP source|destination port (0|29495 or 0x0|0x7337) */
  int_u4 addr;		/* UDP source|destination address */
			/* now starts the actual SDDS UDP payload */
  int_u1 fid;		/* format identifier {SF,SOS,PP,OF,SS,DM:3} */
  int_u1 bits;		/* bits per data word {CX,SNP,VW,BPS:5} 4|8|16 */
  int_u2 frame;		/* frame sequence number (0x1F=parity packet) */
  int_u2 tcinfo;	/* Time Tag info {1msV,ttV,sscV,00,1msPtr[10:0]} */
  int_u2 tccorr;	/* Time from sample to 1msec hack in 250psec tics (min SR=60kHz) */ 
  int_u8 tctics;	/* Number of 250psec tics since jan 1 to timecode hack */
  int_u4 tcext;		/* Timecode extended precision (250psec/2^32) */
  int_u1 sscinfo[12];	/* SSC information {DFDT:32,F:64} */
  int_u1 ssdinfo[4];	/* SSD information */
  int_u1 aadinfo[20];	/* AAD information */
  int_u1 data[1024];	/* packet data */
} ICESDDSPACKETSTRUCT;

#define ICESDDS_HDR_SZ 64
#define ICESDDS_PKT_SZ 1088


/* 
  VITA-49 Packet Structure

  70by - network overhead

  28by - VRT header
 ~1Kby - VRT data (variable 1024,1440,2048,4096,8192,8640)
   4by - VRT trailer

  Total of 102by/1024by = 10% overhead
  Total of 102by/1440by = 7% overhead
  Total of 102by/8640by = 1% overhead

  Max 1G throughput     1.250 (8B10B) = 1G/8 = 125 Mby / 1.06 = 118 Mby/sec data rate
  Max 10G throughput 10.3125 (64B66B) = 10G/8 = 1250 Mby / 1.06 = 1180 Mby/sec data rate 

  The ICE VRT format uses the following Vita-49 spec items to build a Packet Class Code
  that directly specifies the data formats instead of a lookup table entry.

  Packet Header word (Type:4,C:1,T:1,RR:2,TSI:2,TSF:2,Seq:4,Size:16)
  Type:
    0 - Signal Data
    1 - Signal Data w/ Stream Identifier
    4 - Context Packet
    6 - Command Packet
   C: has Class Identifier ( OUID + ICC + PCC )
   T: has Trailer
  RR: reserved

  TSI = TimeStamp Integer Seconds (0=unsed 1=UTC 2=GPS 3=POSIX)
  TSF = TimeStamp Fractional Seconds (0=unsed 1=sampleCount 2=picoseconds 3=freeRunningCount)

  DIS = Data Item Size (bits-1)	
  DIF = Data Item Format (0=int 14=float 15=double)
  DST = Data Sample Type (0=real 1=cxCartesian 2=cxPolar)
  BE  = BigEndian? 1:0
  PT  = PackingType (0=ProcessorEfficient 1=LinkEfficient)

  PCC = Packet Class Code (PT:1,BE:1,DST:2,DIF:4,DIS:8) 
	= 0x100F for LittleEndian CI 
	= 0xD00F for LinkEfficient BigEnd CI
	= 0xD00B for LinkEfficient BigEnd CJ or Complex 12bit

  ICC = Informational Class Code (unused) = 0
  SID = Stream Identifier (16|32 bit)

  Note: Header size field is in 4by words and includes the header and payload words

*/
#define PACKED __attribute__((__packed__))

typedef struct PACKED {
  int_4 header;		/* Header = 0x1C600108 for 1K pktsize */
  int_4 streamID;	/* Stream Identifier (ie. channel number) */
  int_4 orgUID;		/* Organizational Unique ID = 0x00104D77 for ICE */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) */
  int_4 tsis;		/* Time Stamp Integer Seconds (GPS type) J1970 */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (GPS type) */
  int_1 data[1024];	/* Packet data (def size=1024 but is variable) */
  int_4 trailer;	/* Event trailer (struct member only usable for def size) */
} VRTPACKETSTRUCT;

#define VRT_HDR_SZ 32
#define VRT_PKTM_SZ 1024	/* max power 2 in 1500 MTU */
#define VRT_PKTD_SZ 1440	/* max in 1500 MTU even for 12|16 bit data */
#define VRT_PKTE_SZ 8640	/* max in 9000 MTU even for 12|16 bit data */

/* 
  The VRTL or VRT-Lite version removes the trailer and puts the SID in the ICC field to maintains 8byte alignment
*/
typedef struct PACKED {
  int_4 header;		/* Header = 0x08600106 for 1K pktsize */
  int_4 orgUID;		/* Organizational Unique ID = 0x00104D77 for ICE */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) (optional 16bit SID in ICC field) */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (pSec type) */
  int_1 data[1024];	/* Packet data (def size=1024 but is variable) */
} VRTLPACKETSTRUCT;

/* 
  The VRTW or VRT-WagonWheel version removes the trailer, uses LinkEfficient packing,
    and BE data 256 x CI, 384 x Cb12, or 512 x CB    so PCC = 0xD00F, 0xD00B, or 0xD007
*/
typedef struct PACKED {
  int_4 header;		/* Header = 0x08600106 for 1K pktsize */
  int_4 orgUID;		/* Organizational Unique ID = 0x104D77 for ICE */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (pSec type) */
  int_1 data[1024];	/* Packet data (def size=1024 max but is variable (Cb8=512*2=1024 Cb12=384*3=1152 Cb16=256*4=1024) */
} VRTWPACKETSTRUCT;

/* 
  The VRTX version of VITA is for compatibility with AST formats
  The context packets are produced once per second by default 
*/
typedef struct PACKED {
  int_4 header;		/* Header = 0x14600106 */
  int_4 streamID;	/* Stream Identifier */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (pSec type) */
  int_1 data[1024];	/* Packet data */
  int_4 trailer;	/* Event trailer */
} VRTXPACKETSTRUCT;

/* 
  The VRTD version of VITA is for compatibility with DIFI formats
  Note: the header fields do not lend themselves to 8-byte alignment on the host system
  The context packets are produced once per second by default 
*/
typedef struct PACKED {
  int_4 header;		/* Header = 0x18600106 for 1K packet size */
  int_4 streamID;	/* Stream Identifier */
  int_4 orgUID;		/* Organizational Unique ID = 0x6A621E for DIFI */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) = 0x00030006 */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (pSec type) */
  int_1 data[1024];	/* Packet data */
} VRTDPACKETSTRUCT;

typedef struct PACKED {
  int_4 header;		/* Header = 0x49600008 with 0 data fields */
  int_4 streamID;	/* Stream Identifier */
  int_4 orgUID;		/* Organizational Unique ID = 0x104D77 for ICE */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (pSec type) */
  int_4 indicator;	/* Bit mask of populated fields */
  int_4 data[24];
} VRTCONTEXTSTRUCT;

typedef struct PACKED {	/* Timing Flow Control Packet */
  int_4 header;		/* Header = 0x68600012 */
  int_4 streamID;	/* Stream Identifier */
  int_4 orgUID;		/* Organizational Unique ID = 0x6A621E for DIFI */
  int_4 pktinfoCC;	/* Packet and Informational Class Codes (ICC:16,PCC:16) = 0x00030006 */
  int_4 tsis;		/* Time Stamp Integer Seconds (UTC type) */
  int_8 tsfs;		/* Time Stamp Fractional Seconds (count|pSec? type) */
  int_4 indicator;	/* Bit mask of populated fields 0xA0081000 */
  int_4 messageID;	/* sequence count:8 */
  int_4 controlleeID;	/* */
  int_4 controllerID;	/* */
  int_4 indicator2;	/* Bit mask of populated fields 0x40100002 */
  int_4 indicator3;	/* Bit mask of populated fields 0x00000002 */
  int_4 refPoint;	/* Reference Point 0x00000064 */
  int_8 tsfempto;	/* */
  int_8 bufSize;	/* Buffer size in bytes */
  int_4 bufFill;	/* Buffer Fill:12 Over:1 AFull:1 AEmpty:1 Under:1 */
} VRTINFOSTRUCT;

/*
  Ethernet Structures
*/

typedef struct {
  int_1 destAddr[6];	/* MAC address of dest NIC 01-00-5E-XX-XX-XX */
  int_1 srcAddr[6];	/* MAC address of source NIC 01-00-5E-XX-XX-XX */
  int_2 type;		/* 0x0800=IP 0x0806=ARP 0x8035=RARP 0x88F7=PTP 0x88CC=LLDP 0x8808=EFC(ethflowctrl) */
} EHDRSTRUCT;

typedef struct {
  int_2 htype;		/* 10 hardware type 0100 */
  int_2 ptype;		/* 12 protocol type 0008 */
  int_1 hal;		/* 14 hardware address length 06 */
  int_1 pal;		/* 15 protocol address length 04 */
  int_2 opcode;		/* 16 oP Code 0100 */
  int_1 srcAddr[6];	/* 18 MAC address of source NIC 01-00-5E-XX-XX-XX */
  int_1 srcProAddr[4];	/* 1E Protocol address of source */
  int_1 trgAddr[6];	/* 22 cMAC address of target NIC 01-00-5E-XX-XX-XX */
  int_1 trgProAddr[4];	/* 28 Protocol address of target */
} ARPSTRUCT;

typedef struct {
  int_1 versionLength;	/* 0x45   IP version = 4, header length in 32b words = 5 */
  int_1 typeOfService;	/* 0x00   precedence, throughput, reliability */
  int_2 totalLength;	/* 0x0454 for SDDS packet + IPhdr + UDPhdr  or 1080 + 20 + 8 = 1108 */
  int_2 identification;	/* 0x0000 unique identifier */
  int_2 fragmentOffset;	/* 0x4000 do not fragment bit:3 + frag offset:13 */
  int_1 timeToLive;	/* 32     max hops before discard */
  int_1 protocol;	/* transport layer protocol 1=ICMP 2=IGMPv2 6=TCP 8=EGP 17=UDP 89=OSPF */
  int_2 headerChecksum;	/* calculate from IP header only */
  int_1 srcAddr[4];	/* IP source addr */
  int_1 destAddr[4];	/* IP dest addr: 224.0.0.1 is IGMP broadcast, > 224.0.0.1 and < 240.0.0.0 is multicast */
} IPHDRSTRUCT;

typedef struct {
  int_1 type;		/* ipv4 8=ping 0=pong  ipv6 128=ping 129=pong */
  int_1 code;		/* 0 */
  int_2 checksum;
  int_2 ident;		/* 0x1ce */
  int_2 seqno;		/* 1 */
} ICMPHDRSTRUCT;

typedef struct {
  int_1 type;		/* 3=join 5=leave */
  int_1 code;		/* 0=public 1=private */
  int_2 checksum;
  int_4 group;		/* multicast address */
} IGMPHDRSTRUCT;

typedef struct {
  int_2 srcPort;	/* source port = 0 */
  int_2 destPort;	/* destination port = 29495 or 0x7337 */
  int_2 length;		/* length of UDP datagram including UDP header ex. SDDS = 1088 or (8+56+1K) */
  int_2 checksum;	/* 0 = not computed */
} UDPHDRSTRUCT;

/*
  Processor/IO Module Hypertransport packet
*/
typedef struct {
  int_1 key; 		/* Start Key = 0xA5 */
  int_1 cmd; 		/* Command = 0x00 for data */
  int_1 len; 		/* Data len in number of 64by packets */
  int_1 chn;  		/* Channel Number */
} HYPERPACKETSTRUCT;

/*
  File Exchange Packet
*/
typedef struct {
  int_4 key;		/* Header Sync word */
  int_4 state;		/* Current state */
  int_4 offset;		/* Current offset */
  int_4 avail;		/* available buffer space */
  int_4 length;		/* available buffer space */
  int_4 sequence;	/* packet sequence number */
  double time;		/* time of last transmission */
  char fromfile[128];	/* filename path root */
  char tofile[128];	/* filename path root */
  char chksum[128];	/* file checksum */
  char data[4096-384-4*12];	/* fill a 4K page */
} FEXPACKETSTRUCT;
