/*
  Basic types and Memory Configuration for ICE-PPC engines.

    Jeff Schoen - Innovative Computer Engineering
*/

#ifndef _iceppc_h
#define _iceppc_h

#if !HOST

/* Type definitions */
typedef signed char     int_1;
typedef unsigned char   int_u1;
typedef short           int_2;
typedef unsigned short  int_u2;
typedef int             int_4;
typedef unsigned int    int_u4;
typedef long long       int_8;
typedef float   	real_4;
typedef double  	real_8;
typedef char *  	pointer;

/* System cache management */
void zCache (int_4 addr);
void fCache (int_4 addr);
void fCache2 (int_4 addr);
void iCache (int_4 addr);
void lCache (int_4 addr);
void liCache (int_4 addr);

/* System control bus routines (in boot.S) */
void aBus (int_4 addr);
void wBus (int_4 data);
void awBus (int_4 addr, int_4 data);
int_4 rBus();
int_4 arBus(int_4 addr);
int_4 rSync(int_4 *addr);
void udelay(int_4 usec);
void stall();
void zero (int_4 *data, int_4 size);
void memf (int_4 fill, int_4 *data, int_4 size);

/* Algorithm access routines */
int_4 initAlgorithm();
int_4 openAlgorithm(int_4 port, int_4 type, int_4 route);
int_4 processAlgorithm();
int_4 closeAlgorithm(int_4 port, int_4 type, int_4 route);
int_4 modifyAlgorithm(int_4 *msg, int_4 lmsg);
void  setRunMode(int_4 mode);

/* DMA service routines */
void initDMA();
void resetDMA();
void setupDMAo (int_4 i, int_4 cur_addr, int_4 cfg_addr);
void setupDMAi (int_4 i, int_4 cur_addr, int_4 cfg_addr);
void setupDMAt (int_4 i, int_4 cur_addr);

/* DMA service structures */
typedef struct {
  int_4 key;
  int_4 node;
  int_4 head[2];
  int_4 data[256];
} MESSAGE;

/* BIG endian HEADER for PPC */
typedef struct {
  int_2 size;
  int_2 func;
  int_4 addr;
} HEADER;

#define DMACHANNELS 64
/* DMA channel structures */
typedef struct {
  int_u4 haddr, hsize, hindex, hcycle;
  int_u4 caddr, csize, cindex, ccycle;
  int_4 stat,  dir,   todo,   block;
  int_4 cntrl, tcbp,  reg,    mcsr;
  int_4 enb,   enbx,  mcfg,   mcfgx;
  int_4 err,   miss,  iper,   multi;
  int_4 hxfer, hxfes, cxfer,  cxfes;
  int_4 cindp, hindp, cind,   hind;
  int_4 type,  port,  bits,   dec;
  int_4 rate,  gain,  fcny,   frame;
  int_4 master,slave, feed,   tcoff;
  int_4 alg,   appp,  chain,  flags;
} DMASTRUCT;

/* DMA modes */
#define DMA_LOADNSPIN  -19
#define DMA_ONDEMAND   -17
#define DMA_LOOP       -14
#define DMA_RESHOT     -11
#define DMA_CONTINUOUS -2
#define DMA_SPIN       -1
#define DMA_ONESHOT    1

/* graychip registers */
#define GC4016_GR   0
#define GC4016_PM   2
#define GC4016_CR   16
#define GC4016_FS   17
#define GC4016_NS   18
#define GC4016_DS   20
#define GC4016_IQ   24
#define GC4016_PFIR 26

#define GC4016_PHA  16
#define GC4016_FREQ 18

/* ioport types */
#define IOPT_TUNER      4
#define IOPT_CORE       7
#define IOPT_MCORE      8
#define IOPT_TBANK      9
#define IOPT_CBANK      10

/* Time Code codes */
#define TCM_SDDS 7

/* PCI Interface register definitions */
#define PMS 		0x02000000

#define MCSR_RPRC       0x00000001      
#define MCSR_RIOC       0x00000002      
#define MCSR_RFIFO      0x00000004      
#define MCSR_RMBX       0x00000008      

#define MCSR_MENA       0x00000010      
#define MCSR_MDIR       0x00000020      
#define MCSR_WIDE       0x00000040      
#define MCSR_TEST       0x00000080      

#define MCSR_QSTAT      0x00000100      
#define MCSR_LT128      0x00000200      
#define MCSR_LT8        0x00000400      
#define MCSR_LIOC       0x00000800
#define MCSR_MASTER_DAC 0x00000800 

#define MCSR_FLG0	0x00001000
#define MCSR_FLG1	0x00002000
#define MCSR_GEN2       0x00002000
#define MCSR_CLK66      0x00004000
#define MCSR_BUS64      0x00008000

#define MCSR_MASTER_WR  0x00000034 
#define MCSR_MASTER_RDM 0x00001014 

#define MCSR_RFIFO      0x00000004      /* Reset FIFO */
#define MCSR_OMB1F      0x00010000      /* Output Mailbox 1 Full */
#define MCSR_IMB1F      0x00100000      /* Input Mailbox 1 Full */
#define MCSR_OMBXF      0x000F0000      /* Output Mailbox 1-4 Full */
#define MCSR_IMBXF      0x00F00000      /* Input Mailbox 1-4 Full */

#define MCSR_OFIFO_FULL         0x01000000      /* P2C FIFO Full */
#define MCSR_OFIFO_AFULL        0x02000000      /* P2C FIFO Almost Full */
#define MCSR_OFIFO_HALF         0x04000000      /* P2C FIFO Half Full */
#define MCSR_OFIFO_EMPTY        0x08000000      /* P2C FIFO Empty */
#define MCSR_IFIFO_FULL         0x10000000      /* C2P FIFO Full */
#define MCSR_IFIFO_AFULL        0x20000000      /* C2P FIFO Almost Full */
#define MCSR_IFIFO_HALF         0x40000000      /* C2P FIFO Half Full */
#define MCSR_IFIFO_EMPTY        0x80000000      /* C2P FIFO Empty */

#define REG_AFIFO       (PMS|(0x00<<2))           /* Add-on FIFO */
#define REG_AMDARH      (PMS|(0x11<<2))           /* Master Address High */
#define REG_AMDAU       (PMS|(0x11<<2))           /* Master Address High */
#define REG_AMCSR       (PMS|(0x12<<2))           /* Master Control Status */
#define REG_AMDAR       (PMS|(0x16<<2))           /* Master Address Low */
#define REG_AMDTC       (PMS|(0x17<<2))           /* Master Transfer Count */
#define REG_AMBX        (PMS|(0x18<<2))           /* Offset to Mailboxes */
#define REG_AIMB1       (PMS|(0x18<<2))           /* Input Mailbox #1 */
#define REG_AIMB2       (PMS|(0x19<<2))           /* Input Mailbox #2 */
#define REG_AIMB3       (PMS|(0x1A<<2))           /* Input Mailbox #3 */
#define REG_AIMB4       (PMS|(0x1B<<2))           /* Input Mailbox #4 */
#define REG_AOMB1       (PMS|(0x1C<<2))           /* Output Mailbox #1 */
#define REG_AOMB2       (PMS|(0x1D<<2))           /* Output Mailbox #2 */
#define REG_AOMB3       (PMS|(0x1E<<2))           /* Output Mailbox #3 */
#define REG_AOMB4       (PMS|(0x1F<<2))           /* Output Mailbox #4 */
#define REG_AHIRX       (PMS|(0x20<<2))           /* Host Index Registers */
#define REG_AHIRY       (PMS|(0x80<<2))           /* Host Index Registers */

#define REG_AHIR8       (PMS| 0xA0    )           /* Debug Registers */

#define REG_APECMD      (PMS+(0x40<<2))           /* Adon Packet Engine Cmd */
#define REG_APEACK      (PMS+(0x41<<2))           /* Adon Packet Engine Ack */


#endif

/*
  PM Memory - access through the PPC only

  Memory control upper nibble:  Config,RW,blen[1:0]

  Mode registers are driven by the RAS or upper address bits.

  PPC addr to NDRAM  addr bits:  crbl xxED CBA9 8765 4321 0|654 32bb 10xx

  PPC addr to SDRAM1 addr bits:  crbl CBA9 8765 4321 0|987 6543 bb21 0xxx

  PPC addr to SDRAM2 addr bits:  crbl CBA9 8765 4321 0|987 6543 bb21 0xxx

  PPC addr to SDRAM3 addr bits:  crbl CBA9 8765 4321 0|987 654b bb32 10xx
*/
#define PPC_MEMCTL      0x80000000

#define SDC(BA,A) (0x80000000|(BA<<6)|(A<<15))
#define SDD(BA,A) (0x80000000|(BA<<6)|(A<<16))

/* Network DRAM */
#define ND_MRS_CL3BL4   0x80000308 /* CAS lat=3, Burst Len=4 */
#define ND_MRS_CL4BL4   0x80000408 /* CAS lat=4, Burst Len=4 */
#define ND_MRS_CL4BL2   0x80000404 /* CAS lat=4, Burst Len=2 */
#define ND_EMRS_NORMD   0x80000014 /* normal driver strength, disable DLL */
#define ND_EMRS_STRONG  0x80000018 /* strong driver strength, enable DLL */
#define ND_EMRS_NORM    0x80000010 /* normal driver strength, enable DLL */
#define ND_EMRS_WEAK    0x80000410 /* weak driver strength, enable DLL */
#define ND_EMRS_WEAKEST 0x80000418 /* weakest driver strength, enable DLL */

/* SDRAM all */
#define SD_PRECHARGEALL 0x82002000 /* precharge by addr[13] with A10=all */
#define SD_PRECHARGERDY 0x82003000 /* precharge by addr[13] with A10=all (RDY=addr[12] is an ICE ready) */
#define SD_PRECHARGERDZ 0x82001000 /* precharge by addr[13] with A10=all (RDY=addr[12] is an ICE ready) */
#define SD_MRS_RSTDLL   0x00800000 /* reset DLL */

/* SDRAM 1 */
#define SD_MRS_CL20BL4  0x80110000 /* CAS lat=2.0, Burst Len=4 */
#define SD_MRS_CL30BL4  0x80190000 /* CAS lat=3.0, Burst Len=4 */
#define SD_MRS_CL25BL2  0x80308000 /* CAS lat=2.5, Burst Len=2 */
#define SD_MRS_CL25BL4  0x80310000 /* CAS lat=2.5, Burst Len=4 */
#define SD_MRS_CL25BL8  0x80318000 /* CAS lat=2.5, Burst Len=8 */
#define SD_EMRS_NORMD   0x80008040 /* normal driver strength, disable DLL */
#define SD_EMRS_NORMN   0x80000040 /* normal driver strength, enable DLL */
#define SD_EMRS_NORM    0x80010040 /* reduced driver strength, enable DLL */

/* SDRAM 2 */
#define SD2_MRS_CL3BL4   SDC(0,0x0632) /* CAS lat=3, Burst Len=4, Write Rec=4 */
#define SD2_MRS_CL4BL4   SDC(0,0x0642) /* CAS lat=4, Burst Len=4, Write Rec=4 */
#define SD2_MRS_CL5BL4   SDC(0,0x0652) /* CAS lat=5, Burst Len=4, Write Rec=4 */
#define SD2_EMRS1_WEAK   SDC(1,0x000A) /* weak drv */
#define SD2_EMRS1_ODT    SDC(1,0x0048) /* ODT 150ohm */
#define SD2_EMRS1_NORMDQ SDC(1,0x040A) /* weak drv, enable DLL, disable DQS# */
#define SD2_EMRS1_OCDEN  SDC(1,0x078A) /* weak drv, enable DLL, disable DQS#, OCD enter */
#define SD2_EMRS1_OCDEX  SDC(1,0x040A) /* weak drv, enable DLL, disable DQS#, OCD exit */
#define SD2_EMRS1_OCDEY  SDC(1,0x0788) /* norm drv, enable DLL, disable DQS#, OCD enter ? */
#define SD2_EMRS2_NORM   SDC(2,0x0000) /* normal */
#define SD2_EMRS3_NORM   SDC(3,0x0000) /* normal */

/* SDRAM 3 */
#define SD3_MRS_CL13BL8 SDC(0,0x0014)  /* CAS lat=13, Burst Len=8, Write Rec=16 */
#define SD3_MRS_CL12BL8 SDC(0,0x0004)  /* CAS lat=12, Burst Len=8, Write Rec=16 */
#define SD3_MRS_CL11BL8 SDC(0,0x0070)  /* CAS lat=11, Burst Len=8, Write Rec=16  def */
#define SD3_MRS_CL10BL8 SDC(0,0x0060)  /* CAS lat=10, Burst Len=8, Write Rec=16 */
#define SD3_MRS_CL9BL8  SDC(0,0x0050)  /* CAS lat=9, Burst Len=8, Write Rec=16 */
#define SD3_MRS_CL8BL8  SDC(0,0x0040)  /* CAS lat=8, Burst Len=8, Write Rec=16 */
#define SD3_MRS_CL5BL8  SDC(0,0x0810)  /* CAS lat=5, Burst Len=8, Write Rec=8 */
#define SD3_EMRS1_ODT40 SDC(1,0x0044)  /* normal DLL=On  ODT=40 AddLat=0 WriteLev=0 Drive=40ohm */
#define SD3_EMRS1_ODT60 SDC(1,0x0004)  /* normal DLL=On  ODT=60 AddLat=0 WriteLev=0 Drive=40ohm */
#define SD3_EMRS1_NORM  SDC(1,0x0000)  /* normal DLL=On  ODT=Off AddLat=0 WriteLev=0 Drive=40ohm */
#define SD3_EMRS1_NODLL SDC(1,0x0045)  /* normal DLL=Off ODT=60 AddLat=0 WriteLev=0 */
#define SD3_EMRS2_CWL5  SDC(2,0x0000)  /* normal CWL=5 dynODT=Off */
#define SD3_EMRS2_CWL7  SDC(2,0x0410)  /* normal CWL=7 dynODT=120 def */
#define SD3_EMRS2_CWL8  SDC(2,0x0018)  /* normal CWL=8 dynODT=Off */
#define SD3_EMRS2_CWL9  SDC(2,0x0020)  /* normal CWL=9 dynODT=Off */
#define SD3_EMRS3_NORM  SDC(3,0x0000)  /* normal */
#define SD3_EMRS3_TEST  SDC(3,0x0004)  /* read test */
#define SD3_ZQCL	SDC(4,0x0400)  /* ZQCL A10=long|short */
#define SD3_PRECHARGEALL  0x82002000   /* precharge by acfg with addr[13] &&  A10<<15=all */

/* SDRAM 4 */
#define SD4_MRS_CL9BL8W SDD(0,0x0200)  /* CAS lat=9,  Burst Len=8, Write Rec=12 */
#define SD4_MRS_CL9BL8  SDD(0,0x0000)  /* CAS lat=9,  Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL10BL8 SDD(0,0x0004)  /* CAS lat=10, Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL11BL8 SDD(0,0x0010)  /* CAS lat=11, Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL12BL8 SDD(0,0x0014)  /* CAS lat=12, Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL13BL8 SDD(0,0x0020)  /* CAS lat=13, Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL14BL8 SDD(0,0x0024)  /* CAS lat=14, Burst Len=8, Write Rec=10 */
#define SD4_MRS_CL15BL8 SDD(0,0x0630)  /* CAS lat=15, Burst Len=8, Write Rec=16 */
#define SD4_EMRS1_NORMX SDD(1,0x0301)  /* normal DLL=Off  ODT=Off AddLat=0 WriteLev=0 Drive=40ohm  DLL A0=0:On,1:Off On has more delay */
#define SD4_EMRS1_NORM  SDD(1,0x0604)  /* normal DLL=Off  ODT=Off AddLat=0 WriteLev=0 Drive=240ohm  DLL A0=0:On,1:Off On has more delay */
#define SD4_EMRS2_CWL9  SDD(2,0x0000)  /* normal CWL=9 dynODT=120 TRR=1 */
#define SD4_EMRS2_CWL10 SDD(2,0x0008)  /* normal CWL=10 dynODT=120 TRR=1 */
#define SD4_EMRS2_CWL11 SDD(2,0x0010)  /* normal CWL=11 dynODT=120 TRR=1 */
#define SD4_EMRS3_NORM  SDD(3,0x0000)  /* normal */
#define SD4_EMRS3_TEST  SDD(3,0x0007)  /* read test */
#define SD4_MR4         SDD(4,0x0000)  /* Mode4 register */
#define SD4_MR5         SDD(5,0x0000)  /* Mode5 register */
#define SD4_MR6         SDD(6,0x0014)  /* Mode6 register Vrefdq=.7Vdd */
#define SD4_ZQCL	SDD(7,0x0000)  /* ZQCL A10=long|short */
#define SD4_PRECHARGEALL 0x84002000    /* precharge by acfg with addr[13] &&  A10<<16=all */
#define SD4_MRS_RSTDLL   0x01000000    /* reset DLL */

/* RocketIO enables & polarity swaps TXP|RXP|PROTO|PWRDN */
#define PPC_RIO_ENABLE	0x000022FF
#define PPC_RIO_DISABLE	0x00000000
#define PPC_RIO_TRXPIC	0xFDED0000
#define PPC_RIO_TRXPOD	0x66FF0000
#define PPC_RIO_DIODE	0x66FF0000
#define PPC_MSAS_ENABLE	0x0900020F
#define PPC_SFPP_ENABLE	0x0F0F060F
#define PPC_DSFP_ENABLE	0x03030503
#define PPC_SFP_ENABLE	0x01010401

#define PPC_RIO_CTL	0x00200000
#define PPC_RIO_STA	0x00200004
#define PPC_AUX_CTL	0x00400000
#define PPC_JTAG	0x00800000
#define PPC_JTAG_MSK	0x00800000
#define PPC_JTAG_DAT	0x00800004
#define PPC_JTAG_ENA	0x00800008
#define PPC_JTAG_STAT	0x0080000C

/* Module indices */
#define PM_DMAC 0	/* DMA Controller */
#define PM_PROC 0	/* PowerPC Processor */
#define PM_HOST 1	/* Host Port */
#define PM_HYPA 2	/* HyperTransportIF  A-Side */
#define PM_HYPB 3	/* HyperTransportIF  B-Side */
#define PM_CORA 4	/* Core A-Side */
#define PM_CORB 5	/* Core B-Side */
#define PM_TUNA 6	/* TunerBank/Processor A-Side */
#define PM_TUNB 7	/* TunerBank/Processor B-Side */
#define PM_PMA  6	/* TunerBank/Processor A-Side */
#define PM_PMB  7	/* TunerBank/Processor B-Side */
#define PM_TUNC 1	/* TunerBank C-Side */
#define PM_TCTL 7	/* TunerBank Control */
#define PM_CORHYPA 8	/* Core->Hyp A-Side */
#define PM_CORHYPB 9	/* Core->Hyp A-Side */

#define CB_IOOFF 2
#define CB_PMOFF 6

/* Packet Processor Addresses */
#define PE_STATUS    0
#define PE_CONFIG    0
#define PE_SYSREG    0x80000000
#define PE_SYSTAT    0x80000004
#define PE_CBUS      0x40000000

#define PE_WSTAT     0x04000000
#define PE_WDLP      0x04000110
#define PE_WSOSP     0x04000120
#define PE_WTOSP     0x04000130
#define PE_WTLP      0x04000210
#define PE_WTLPX     0x04000220
#define PE_WTLPD     0x04000200
#define PE_WTLPDI    0x04000280

#define PE_WTLPCID   0x04000210
#define PE_WTLPTAG   0x04000220
#define PE_WTLPADL   0x04000230
#define PE_WTLPADH   0x04000240
#define PE_WTLPDAT   0x04000250

#define PE_RSTAT     0x08000000
#define PE_RDAT      0x08000000
#define PE_RCNT      0x08000004
#define PE_RWAD      0x08000008
#define PE_RRAD      0x0800000C

/* Special Symbol codes */
#define SYM_SKP	0x11C
#define SYM_FTS	0x13C
#define SYM_SDP	0x15C
#define SYM_IDL	0x17C
#define SYM_COM	0x1BC
#define SYM_PAD	0x1F7
#define SYM_STP	0x1FB
#define SYM_END	0x1FD
#define SYM_EDB	0x1FE

/* Common MultiCore sysreg bit register */
#define MCORE_CHN   0x010000
#define MCORE_ALL   0x410000
#define MCORE_EVEN  0x420000
#define MCORE_ODD   0x440000
#define MCORE_BOTH  0x460000
#define MCORE_TRIM  0x480000
#define MCORE_SYS   0x800000
#define MCORE_SCL   0x800004
#define MCORE_NFSCL 0x800008
#define MCORE_RATE  0x80000C
#define MCORE_TAPS  0x800010
#define MCORE_FCYS  0x800014
#define MCORE_ENA   0x800018

/* Module Bit Offset  0xXX000000 */
#define PPC_MSHF(A)  (A+24)
#define PPC_HYP_SHF  (PM_HYPA+24)

/* Module Address  0xXX000000 */
#define PPC_MADDR(A)  (0x1<<(A+24))
#define PPC_DMAC_CTL  PPC_MADDR(PM_DMAC)
#define PPC_HYPA_CTL  PPC_MADDR(PM_HYPA)
#define PPC_HYPB_CTL  PPC_MADDR(PM_HYPB)
#define PPC_CORA_CTL  PPC_MADDR(PM_CORA)
#define PPC_CORB_CTL  PPC_MADDR(PM_CORB)
#define PPC_TUNA_CTL  PPC_MADDR(PM_TUNA)
#define PPC_TUNB_CTL  PPC_MADDR(PM_TUNB)
#define PPC_TUNAB_CTL PPC_MADDR(PM_TUNA)|PPC_MADDR(PM_TUNB)
#define PPC_PMA_CTL   PPC_MADDR(PM_TUNA)
#define PPC_PMB_CTL   PPC_MADDR(PM_TUNB)

#define PPC_CORE1_CTL  PPC_MADDR(PM_CORA)
#define PPC_CORE2_CTL  PPC_MADDR(PM_CORB)
#define PPC_CORE3_CTL  PPC_MADDR(PM_TUNA)
#define PPC_CORE4_CTL  PPC_MADDR(PM_TUNB)
#define PPC_CORES_CTL  0xF0000000

#define PPC_JTAG       0x00800000
#define PPC_ICAP       0x00700000
#define PPC_SYSMON     0x00600000
#define PPC_NIO        0x00500000
#define PPC_AUX        0x00400000
#define PPC_TRACER     0x00300000
#define PPC_RIO        0x00200000
#define PPC_SYSCLK     0x00100000

#define PPC_ICAP7      0x00080000

#define PPC_DMAC_STATUS    (PPC_MADDR(PM_DMAC)|0x8000)
#define PPC_DMAC_ROUTE     (PPC_MADDR(PM_DMAC)|0x8000)
#define PPC_DMAC_ROUTE_RST (PPC_MADDR(PM_DMAC)|0x8000)
#define PPC_DMAC_ROUTE_FLG (PPC_MADDR(PM_DMAC)|0x8004)
#define PPC_DMAC_ROUTE_SET (PPC_MADDR(PM_DMAC)|0x8008)
#define PPC_DMAC_ROUTE_CLR (PPC_MADDR(PM_DMAC)|0x800C)
#define PPC_DMAC_ROUTE_CNT (PPC_MADDR(PM_DMAC)|0x8010)
#define PPC_DMAC_ROUTE_SPD (PPC_MADDR(PM_DMAC)|0x8014)
#define PPC_DMAC_ROUTE_NIO (PPC_MADDR(PM_DMAC)|0x8018)
#define PPC_DMAC_ROUTE_FDC (PPC_MADDR(PM_DMAC)|0x801C)
#define PPC_DMAC_ROUTE_IOS (PPC_MADDR(PM_DMAC)|0x801C)
#define PPC_DMAC_HPCNT     (PPC_MADDR(PM_DMAC)|0x8010)
#define PPC_DMAC_CLOCK     (PPC_MADDR(PM_DMAC)|0x4000)
#define PPC_DMAC_CALIB     (PPC_MADDR(PM_DMAC)|0x2000)
#define PPC_DMAC_SRCFG     (PPC_MADDR(PM_DMAC)|0x2000)
#define PPC_DMAC_PCLK      (PPC_MADDR(PM_DMAC)|0x1000)

/* Tuner Address  0x8000XX00 */
#define PPC_TBANKA   (PPC_MADDR(PM_TCTL)|0x5500)
#define PPC_TBANKB   (PPC_MADDR(PM_TCTL)|0xAA00)
#define PPC_TBANKC   (PPC_MADDR(PM_TCTL)|0xCC00)
#define PPC_TBANKX   (PPC_MADDR(PM_TCTL)|0xFF00)
#define PPC_TBANKZ   (PPC_MADDR(PM_TCTL)|0x0000)
#define PPC_TOVSRA   (PPC_MADDR(PM_TCTL)|0x0010)
#define PPC_TOVSRB   (PPC_MADDR(PM_TCTL)|0x0020)
#define PPC_TMODEX   (PPC_MADDR(PM_TCTL)|0x0030)
#define PPC_TADDR(A) (PPC_MADDR(PM_TCTL)|( ((A&0x1)+1) << ( 8 + ((A&0x8)>>1) + ((A&0x10)>>3) ) ) )

/* DMA Controller Routing - see vdefs.h */
#define PPC_ISTAT(A) (0x1<<(A+00))
#define PPC_OSTAT(A) (0x1<<(A+16))

/* Peripheral address */
#define PPC_BRAM      0xFD000000
#define PPC_IRAM      0xFF000000
#define PPC_PRAM      0x10000000
#define PPC_DRAM      0x00000000

/* DSOCM Allocation */
#define PPC_MSG       0xFD000000
#define PPC_NBOOT     0xFD000820
#define PPC_ALG       0xFD000840

#define PPC_DMA       0xFD810000	/* this 2K ram is shadowed in the base 4K for speed */
#define PPC_DMA_TO    0xFD810000	/* to tuner RAM */
#define PPC_DMA_TI    0xFD810E00	/* from tuner RAM */
#define PPC_DMA_XI    0xFD810F00	/* from RAM */
#define PPC_DMA_XO    0xFD810F40	/* to RAM */
#define PPC_DMA_TX    0xFD810F70	/* TX to RAM */
#define PPC_DMA_NVME  0xFD810F80	/* NVME to RAM */
#define PPC_DMA_NIO   0xFD810FC0	/* NIO to RAM */

#define PPC_VERSION   0xFD00FFE0
#define PPC_SIGNATURE 0xFD00FFE4
#define PPC_MSGR6     0xFD00FFF4
#define PPC_FAT       0xFD00FFF8
#define PPC_HBT       0xFD00FFFC

#define PPC_VOFF      (PPC_VERSION-PPC_BRAM)
#define PPC_HBTOFF    (PPC_HBT-PPC_BRAM)

#define PPC_FTT_ADDR  0x480
#define PPC_ALG_ADDR  0x600
#define FTTSTRUCTLOC (FTTSTRUCT*)PPC_FTT_ADDR

#define PPC_STAT      0x80000000

/* send/recv buffer defines */
#define PPC_BDATA      0x0000
#define PPC_BADDR_WR   0x0004
#define PPC_BADDR_RD   0x000C

/* DMA clock adjust definitions */
#define PPC_CLOCK_RSCV  0x00000001
#define PPC_CLOCK_ASCVD 0x00000002
#define PPC_CLOCK_ASCVU 0x00000102
#define PPC_CLOCK_RSCP  0x00000004
#define PPC_CLOCK_ASCPD 0x00000008
#define PPC_CLOCK_ASCPU 0x00000108
#define PPC_CLOCK_TRSV  0x00000020
#define PPC_CLOCK_TRSP  0x00000040
#define PPC_CLOCK_RMEM  0x00000010
#define PPC_CLOCK_XRST  0x00000100
#define PPC_CLOCK_RMCM  0x01000000
#define PPC_CLOCK_RDLC  0x02000000

/* HyperTransport clock throttle */
#define PPC_CLOCK_HYPMAX 0x00000200

/* System run states */
#define PPC_ALG_INIT 0
#define PPC_ALG_OPEN 1
#define PPC_ALG_PROCESS 2
#define PPC_ALG_CLOSE 3
#define PPC_ALG_DONE 0

/* DMA routing definitions */
#define PPC_ROUTE_DEF   (PPC_ROUTE_TA2HA|PPC_ROUTE_TB2HB|PPC_ROUTE_HA2TA|PPC_ROUTE_HB2TB)

#define PPC_ROUTE_STATIC  0x0000FFFF
#define PPC_ROUTE_DYNAMIC 0xFFFF0000

#define PPC_ROUTE_CA2HA 0x00000001
#define PPC_ROUTE_CB2HA 0x00000002
#define PPC_ROUTE_CA2HB 0x00000004
#define PPC_ROUTE_CB2HB 0x00000008
#define PPC_ROUTE_HA2CA 0x00000010
#define PPC_ROUTE_HB2CA 0x00000020
#define PPC_ROUTE_HA2CB 0x00000040
#define PPC_ROUTE_HB2CB 0x00000080

#define PPC_ROUTE_TA2HA 0x00000100
#define PPC_ROUTE_CA2TA 0x00000200
#define PPC_ROUTE_CB2TB 0x00000400
#define PPC_ROUTE_TB2HB 0x00000800
#define PPC_ROUTE_HA2TA 0x00001000
#define PPC_ROUTE_HB2TA 0x00002000
#define PPC_ROUTE_HA2TB 0x00004000
#define PPC_ROUTE_HB2TB 0x00008000

#define PPC_ROUTE_M2R(A) (0x1<<(A+16))
#define PPC_ROUTE_PR2R  0x00010000
#define PPC_ROUTE_HP2R  0x00020000
#define PPC_ROUTE_HA2R  0x00040000
#define PPC_ROUTE_HB2R  0x00080000
#define PPC_ROUTE_CA2R  0x00100000
#define PPC_ROUTE_CB2R  0x00200000
#define PPC_ROUTE_TA2R  0x00400000
#define PPC_ROUTE_TB2R  0x00800000

#define PPC_ROUTE_R2M(A) (0x1<<(A+24))
#define PPC_ROUTE_R2PR  0x01000000
#define PPC_ROUTE_R2HP  0x02000000
#define PPC_ROUTE_R2HA  0x04000000
#define PPC_ROUTE_R2HB  0x08000000
#define PPC_ROUTE_R2CA  0x10000000
#define PPC_ROUTE_R2CB  0x20000000
#define PPC_ROUTE_R2TA  0x40000000
#define PPC_ROUTE_R2TB  0x80000000

#define PPC_ROUTE_HX2R  0x000C0000
#define PPC_ROUTE_HX2CA 0x00000030
#define PPC_ROUTE_HX2CB 0x000000C0
#define PPC_ROUTE_HX2TA 0x00003000
#define PPC_ROUTE_HX2TB 0x0000C000

#define PPC_ROUTE_TA2CA 0x00000002
#define PPC_ROUTE_TB2CB 0x00000004

/* MB only */
#define PPC_ROUTE_CA2TB 0x00000100
#define PPC_ROUTE_CB2TA 0x00000800

/* PM only */
#define PPC_ROUTE_TC2R  0x00020000
#define PPC_ROUTE_R2TC  0x02000000

/* route flags */
#define PPC_ROUTF_TMUXA 0x00000001
#define PPC_ROUTF_TMUXB 0x00000002
#define PPC_ROUTF_TMUXC 0x00000004
#define PPC_ROUTF_TMUXD 0x00000008
#define PPC_ROUTF_HPFDP 0x00000010
#define PPC_ROUTF_HXFDP 0x00000020
#define PPC_ROUTF_CXFDP 0x00000040
#define PPC_ROUTF_TXFDP 0x00000080

#define PPC_ROUTF_CA2CB 0x00000100
#define PPC_ROUTF_CB2CA 0x00000200
#define PPC_ROUTF_TA2TB 0x00000400
#define PPC_ROUTF_TB2TA 0x00000800

#define PPC_ROUTF_TX2HX 0x00001000
#define PPC_ROUTF_CX2HX 0x00002000
#define PPC_ROUTF_TX2CX 0x00004000
#define PPC_ROUTF_CXAOB 0x00008000
#define PPC_ROUTF_HXAOB 0x00010000
#define PPC_ROUTF_TXAOB 0x00020000
#define PPC_ROUTF_TA2HX 0x00040000
#define PPC_ROUTF_TB2HX 0x00080000

#define PPC_ROUTF_DIO2P 0x00100000
#define PPC_ROUTF_P2DIO 0x00200000
#define PPC_ROUTF_CXMIX 0x00400000
#define PPC_ROUTF_TXTAG 0x00800000

#define PPC_ROUTF_NIO2M  0x01000000
#define PPC_ROUTF_M2NIO  0x02000000
#define PPC_ROUTF_NV2M   0x04000000
#define PPC_ROUTF_M2NV   0x08000000

#define PPC_ROUTF_NIO2HX 0x10000000
#define PPC_ROUTF_HX2NIO 0x20000000
#define PPC_ROUTF_NV2HX  0x40000000
#define PPC_ROUTF_HX2NV  0x80000000

#ifdef _NEWER
#define PPC_ROUTE_HAANY (PPC_ROUTE_HA2R|PPC_ROUTE_R2HA)
#define PPC_ROUTE_HBANY (PPC_ROUTE_HB2R|PPC_ROUTE_R2HB)
#define PPC_ROUTE_TAANY (PPC_ROUTE_TA2R|PPC_ROUTE_R2TA)
#define PPC_ROUTE_TBANY (PPC_ROUTE_TB2R|PPC_ROUTE_R2TB)
#else
#define PPC_ROUTE_HAANY (PPC_ROUTE_HA2R)
#define PPC_ROUTE_HBANY (PPC_ROUTE_HB2R)
#define PPC_ROUTE_TAANY (PPC_ROUTE_TA2R)
#define PPC_ROUTE_TBANY (PPC_ROUTE_TB2R)
#endif
#define PPC_ROUTE_XAANY (PPC_ROUTE_HA2R|PPC_ROUTE_TA2R|PPC_ROUTE_R2HA|PPC_ROUTE_R2TA)
#define PPC_ROUTE_XBANY (PPC_ROUTE_HB2R|PPC_ROUTE_TB2R|PPC_ROUTE_R2HB|PPC_ROUTE_R2TB)
#define PPC_ROUTE_HXANY (PPC_ROUTE_HA2R|PPC_ROUTE_HB2R|PPC_ROUTE_R2HA|PPC_ROUTE_R2HB)
#define PPC_ROUTE_TXANY (PPC_ROUTE_TA2R|PPC_ROUTE_TB2R|PPC_ROUTE_R2TA|PPC_ROUTE_R2TB)
#define PPC_ROUTE_XA2TA (PPC_ROUTE_HA2TA|PPC_ROUTE_CA2TA)
#define PPC_ROUTE_XB2TA (PPC_ROUTE_HB2TA|PPC_ROUTE_CB2TA)
#define PPC_ROUTE_XA2TB (PPC_ROUTE_HA2TB|PPC_ROUTE_CA2TB)
#define PPC_ROUTE_XB2TB (PPC_ROUTE_HB2TB|PPC_ROUTE_CB2TB)
#define PPC_ROUTE_X2TA  (PPC_ROUTE_HA2TA|PPC_ROUTE_HB2TA|PPC_ROUTE_R2TA|PPC_ROUTE_CA2TA|PPC_ROUTE_CB2TA)
#define PPC_ROUTE_X2TB  (PPC_ROUTE_HA2TB|PPC_ROUTE_HB2TB|PPC_ROUTE_R2TB|PPC_ROUTE_CA2TB|PPC_ROUTE_CB2TB)
#define PPC_ROUTE_TA2X  (PPC_ROUTE_TA2HA|PPC_ROUTE_TA2R)
#define PPC_ROUTE_TB2X  (PPC_ROUTE_TB2HB|PPC_ROUTE_TB2R)
#define PPC_ROUTE_HX2TX (PPC_ROUTE_HA2TA|PPC_ROUTE_HA2TB|PPC_ROUTE_HB2TA|PPC_ROUTE_HB2TB)
#define PPC_ROUTE_HA2X  (PPC_ROUTE_HA2CA|PPC_ROUTE_HA2CB|PPC_ROUTE_HA2TA|PPC_ROUTE_HA2TB|PPC_ROUTE_HA2R)
#define PPC_ROUTE_HB2X  (PPC_ROUTE_HB2CA|PPC_ROUTE_HB2CB|PPC_ROUTE_HB2TA|PPC_ROUTE_HB2TB|PPC_ROUTE_HB2R)
#define PPC_ROUTE_HX2X  (PPC_ROUTE_HA2X|PPC_ROUTE_HB2X)
#define PPC_ROUTE_X2HA  (PPC_ROUTE_CA2HA|PPC_ROUTE_CB2HA|PPC_ROUTE_TA2HA|PPC_ROUTE_R2HA)
#define PPC_ROUTE_X2HB  (PPC_ROUTE_CA2HB|PPC_ROUTE_CB2HB|PPC_ROUTE_TB2HB|PPC_ROUTE_R2HB)
#define PPC_ROUTE_X2HX  (PPC_ROUTE_X2HA|PPC_ROUTE_X2HB)
#define PPC_ROUTE_HX2CX (PPC_ROUTE_HA2CA|PPC_ROUTE_HB2CB)
#define PPC_ROUTE_CX2HX (PPC_ROUTE_CA2HA|PPC_ROUTE_CB2HB)

/* DMA buffer sizes */
#define DMA_BUFSZ_B4K   0x00000000
#define DMA_BUFSZ_B64K  0x00000004
#define DMA_BUFSZ_B256K 0x00000006
#define DMA_BUFSZ_B512K 0x00000007
#define DMA_BUFSZ_B1M   0x00000008
#define DMA_BUFSZ_B16M  0x0000000C
#define DMA_BUFSZ_B64M  0x0000000E
#define DMA_BUFSZ_B256M 0x00000010
#define DMA_BUFSZ_NP2   0x00000020

#define DMA_MASK_B4K    0x00000FC0
#define DMA_MASK_B64K   0x0000FFC0
#define DMA_MASK_B256K  0x0003FFC0
#define DMA_MASK_B512K  0x0007FFC0
#define DMA_MASK_B1M    0x000FFFC0
#define DMA_MASK_B16M   0x00FFFFC0
#define DMA_MASK_B64M   0x03FFFFC0
#define DMA_MASK_B256M  0x0FFFFFC0

#define B64   0x00000040
#define B256  0x00000100
#define B1K   0x00000400
#define B2K   0x00000800
#define B4K   0x00001000
#define B8K   0x00002000
#define B16K  0x00004000
#define B64K  0x00010000
#define B128K 0x00020000
#define B256K 0x00040000
#define B512K 0x00080000
#define B1M   0x00100000
#define B4M   0x00400000
#define B8M   0x00800000
#define B16M  0x01000000
#define B32M  0x02000000
#define B48M  0x03000000
#define B64M  0x04000000
#define B128M 0x08000000
#define B256M 0x10000000

#define PB1K   10
#define PB2K   11
#define PB4K   12
#define PB8K   13
#define PB16K  14
#define PB64K  16
#define PB128K 17
#define PB256K 18
#define PB1M   20
#define PB2M   21
#define PB4M   22
#define PB8M   23
#define PB16M  24
#define PB32M  25
#define PB64M  26
#define PB128M 27
#define PB256M 28

#define PIPE_MASK(A) ((0x1<<A)-64)
#define PIPE_SIZE(A) (0x1<<A)
#define PIPE_CFG(A)  (A-12)

#define NBANK 16

typedef struct {
  int_4 mchan;
  int_4 nchan;
  int_4 dec;
  int_4 cpc;
  int_4 xfer;
  int_4 cons;
  int_4 prod;
  int_4 chan;
  int_4 index;
  int_4 pcfg;
  int_4 pstrt;
  int_4 psize;
  int_4 pmask;
  int_4 pindx;
  int_4 phinc;
  int_4 dphinc;
  int_4 tphinc;
  int_4 tdphinc;
  int_4 tphase;
  int_4 tdphase;
} TBSTRUCT;

typedef struct {
  int_4 cur_addr;
  int_4 cfg_addr;
} DMACHANNEL;

typedef struct {
  int_4 stages;
  int_4 active;
  int_4 cstage;
  int_4 lstage;
  int_4 busy;
  int_4 chantop;
  int_4 bits;
  int_4 speed;
  int_4 taddr[16];
  TBSTRUCT tb[3];
  int_4 omoda;
  int_4 omodb;
  DMACHANNEL *dca[2];
} FTTSTRUCT;

int_4 dmaSync(DMACHANNEL *dc);

#endif
