#ifndef ICELIB_H_

#include "icedefs.h"
#include "icemap.h"
#include "iceioctl.h"

/*
  Definitions for ICELIB runtime functions
  Author: Jeff Schoen
*/

/* Software version number */
#define ICE_VERSION	399
#define ICE_MAGIC	0x01CE0123

/* Operational parameters */
#define MAX_PIC_TEMP    70
#define MAX_IOM_TEMP    70
#define MAX_PM_TEMP     70

/* Card Types */
#define ICEPIC1		1
#define ICEPIC2		2
#define ICEPIC3		3
#define ICEPIC4		4
#define ICEPIC5		5
#define ICEPIC6		6
#define ICEPIC7		7
#define ICEPIC8		8
#define ICEPIC9		9
/* out of service */
#define ICEMBT3		9
#define ICEMBT4		10
#define ICEMBT2		11
#define ICESLIC3	12

/* network attached devices */
#define ICENIC		13
#define ICENVM		14

/* SHARC Port Types */
#define IOPT_NONE	0
#define IOPT_STREAM	1
#define IOPT_LINK	2
#define IOPT_MODULE	3
#define IOPT_TUNER	4
#define IOPT_INTERNAL	5
#define IOPT_EXTERNAL	6
#define IOPT_CORE	7
#define IOPT_MCORE	8
#define IOPT_TBANK	9
#define IOPT_CBANK	10

/* I/O Module Types */
#define IOMT_NONE	0
#define IOMT_E2D	-1
#define IOMT_D2E	1
#define IOMT_LV2D	-2
#define IOMT_D2LV	2
#define IOMT_A2D	-3
#define IOMT_D2A	3
#define IOMT_CXD	-4
#define IOMT_DXC	4
#define IOMT_T2D	-5
#define IOMT_D2T	5
#define IOMT_UXD	-6
#define IOMT_DXU	6
#define IOMT_VXD	-7
#define IOMT_DXV	7
#define IOMT_DR2D	-8
#define IOMT_D2DR	8
#define IOMT_PSE2D	-9
#define IOMT_D2PSE	9
#define IOMT_FPDPXD	-10
#define IOMT_DXFPDP	10
#define IOMT_FPQCXD	-11
#define IOMT_DXFPQC	11
#define IOMT_UDPXD	-12
#define IOMT_DXUDP	12
#define IOMT_SNTXD	-13
#define IOMT_DXSNT	13
#define IOMT_SDDSXD	-14
#define IOMT_DXSDDS	14
#define IOMT_GXD	-15
#define IOMT_DXG	15
#define IOMT_NFXD	-16
#define IOMT_DXNF	16
#define IOMT_CDR2D	-17
#define IOMT_D2CDR	17
#define IOMT_LB2D       -18
#define IOMT_D2LB       18
#define IOMT_FLZRXD     -19
#define IOMT_DXFLZR     19
#define IOMT_RFXD       -20
#define IOMT_DXRF       20
#define IOMT_MSASXD     -21
#define IOMT_DXMSAS     21
#define IOMT_DSFPXD     -22
#define IOMT_DXDSFP     22
#define IOMT_SFPXD      -23
#define IOMT_DXSFP      23
#define IOMT_SFPPXD     -24
#define IOMT_DXSFPP     24
#define IOMT_ES2D       -25
#define IOMT_D2ES       25
#define IOMT_TGSDDSXD	-26
#define IOMT_DXTGSDDS	26
#define IOMT_AWG2D	-27
#define IOMT_D2AWG	27
#define IOMT_UFLZXD     -28
#define IOMT_DXUFLZ     28
#define IOMT_GPSXD      -29
#define IOMT_GPS        29
#define IOMT_DIODEI     -30
#define IOMT_DIODE      30
#define IOMT_TGVITAXD   -31
#define IOMT_DXTGVITA   31 
#define IOMT_RF2D       -32
#define IOMT_D2RF       32 
#define IOMT_QSFPXD     -33
#define IOMT_DXQSFP     33 
#define IOMT_UPACXD     -34
#define IOMT_DXUPAC     34 

#define IOMT_LOOP	-98
#define IOMT_AUTO	-99
#define IOMT_TEST	-100
#define IOMT_UNKNOWN	-101

/* Processor Module Types */
#define PMT_TEST	-100
#define PMT_LOOP	-2
#define PMT_AUTO	-1
#define PMT_NONE	0
#define PMT_DTDM	1
#define PMT_DTDMX	2
#define PMT_ZPPM	3
#define PMT_S6M		4
#define PMT_V5M		5
#define PMT_V6M		6
#define PMT_V7M		7
#define PMT_K8M		8
#define PMT_K8P		9
#define PMT_A8M		10
#define PMT_A9M		11

/* DMA Function Flags */
#define DMA_LOOPNSPIN	-23
#define DMA_INDEX	-22
#define DMA_ALLOCATE	-21
#define DMA_ENABLE	-20
#define DMA_LOADNSPIN	-19
#define DMA_STATUS_W64	-18
#define DMA_ONDEMAND	-17
#define DMA_BURY	-16
#define DMA_KILL	-15
#define DMA_LOOP	-14
#define DMA_POLL	-13
#define DMA_ACTIVE	-12
#define DMA_RESHOT	-11
#define DMA_LOST	-10
#define DMA_CANCEL	-9
#define DMA_RESET	-8
#define DMA_LOAD	-7
#define DMA_CYCLE	-5
#define DMA_STATUS	-4
#define DMA_WAIT	-3
#define DMA_CONTINUOUS	-2
#define DMA_SPIN	-1
#define DMA_STOP	 0
#define DMA_ONESHOT	 1

/* Algorithm types */
#define ALG_NULL	0
#define ALG_NOOP	1
#define ALG_USER	2
#define ALG_SWAP	3
#define ALG_LUT		4
#define ALG_AM		5
#define ALG_FM		6
#define ALG_PM		7
#define ALG_PSK		8
#define ALG_QPSK	9
#define ALG_TTDM	10

/* Operational Flags - Global */
#define FLG_TEST	0x00000001
#define FLG_NOBRK	0x00000002
#define FLG_BIGEND	0x00000004
#define FLG_HS		0x00000008
#define FLG_SSC		0x00000008
#define FLG_MUXCLK	0x00000010
#define FLG_INTCLK	0x00000020
#define FLG_TC		0x00000040
#define FLG_LOOP	0x00000080
#define FLG_NOLOCK	0x00000100
#define FLG_DISABLE	0x00000200
#define FLG_VERBOSE	0x00000400
#define FLG_NCCLK	0x00000800
#define FLG_BOOT	0x00001000
#define FLG_LSBX	0x00002000
#define FLG_CLKI	0x00004000
#define FLG_MSBI	0x00008000
#define FLG_VHS		0x00010000
#define FLG_DUAL	0x00020000
#define FLG_MUX		0x00040000
#define FLG_ALT		0x00080000
#define FLG_SGO		0x00100000
#define FLG_TGO		0x00200000
#define FLG_GGO		0x00400000
#define FLG_XGO		0x00800000
#define FLG_BIT3	0x00000000
#define FLG_BIT0	0x10000000
#define FLG_BIT4	0x20000000
#define FLG_BIT1	0x30000000
#define FLG_BITS	0x30000000
#define FLG_RGO		0x40000000
#define FLG_BACKEND	0x80000000
#define FLG_NIO		0x80000000
#define FLG_NVME	0x10000000

#define FLG_XSTGO       0x00B00000
#define FLG_STGO        0x00300000
#define FLG_XTGO        0x00A00000

/* Operational Flags - call specific - never global */
#define FLG_NOWAIT	0x01000000
#define FLG_NOWRAP	0x02000000
#define FLG_SYNC	0x02000000
#define FLG_TUNER	0x02000000
#define FLG_FTUN	0x01000000
#define FLG_DTUN	0x02000000
#define FLG_GTUN	0x04000000
#define FLG_MTUN	0x08000000
#define FLG_RTUN	0x10000000
#define FLG_ATUN	0x1F000000
#define FLG_TCINTERP	0x01000000
#define FLG_TCDELTA	0x02000000
#define FLG_INFO	0x01000000
#define FLG_XFER	0x02000000
#define FLG_ICLK	0x01000000
#define FLG_PCLK	0x02000000
#define FLG_EEEI	0x08000000
#define FLG_LONG	0x10000000
#define FLG_FLOAT	0x20000000

/* RF Option Flags */
#define RFOPT_ENABLE  0x00000001
#define RFOPT_XREF    0x00000002
#define RFOPT_XADCLK  0x00000004
#define RFOPT_ATTEN   0x00000008
#define RFOPT_LNA     0x00000010
#define RFOPT_DCS     0x00000020
#define RFOPT_AIS     0x00000040
#define RFOPT_AGC     0x00000080
#define RFOPT_BBAMP   0x00000100
#define RFOPT_MUTE    0x00000200

#define RFOPTLIST "ENABLE,XREF,XADCLK,ATTEN,LNA,DCS,AIS,AGC,BBAMP,MUTE"

/* TimeCode return status */
#define TC_DPTX		4
#define TC_OKNC		2
#define TC_OK		1
#define TC_INACTIVE	-1
#define TC_BADPARAM	-2
#define TC_NOBITS	-3
#define TC_NOBARKER	-4
#define TC_BCDERR	-5
#define TC_NAN		-6
#define TC_NOCLOCK	-7
#define TC_BADSTART	-8
#define TC_NTPERR	-9

/* TimeCode modes */
#define TCM_OFF 0
#define TCM_CPU 1
#define TCM_ZTC 2
#define TCM_SDN 3
#define TCM_SMS 4
#define TCM_DTL 5
#define TCM_IRB 6
#define TCM_SDD 7
#define TCM_ICE 8
#define TCM_VRT 9
#define TCM_FILE 10
#define TCM_STC 12

/* Timecode specific flags */
#define FLG_TC_VFTX	0x0001
#define FLG_TC_SWAP	0x0002
#define FLG_TC_OPPS	0x0004
#define FLG_TC_FILL	0x0008
#define FLG_TC_PREFILL	0x0008
#define FLG_TC_OVFT	0x0010

/* Tuner/Core specific flags */
#define FLG_UFILT	0x0001
#define FLG_UCFIR	0x0002
#define FLG_NCFIR	0x0004
#define FLG_PFIR4	0x0008
#define FLG_AOVSR	0x0010
#define FLG_POVSR	0x0020
#define FLG_ITDEC	0x0040
#define FLG_RESAMP	0x0080
#define FLG_DSYNC	0x0100
#define FLG_TALT1	0x0200
#define FLG_FSYNC	0x0400
#define FLG_GSYNC	0x0800
#define FLG_TBANK	0x1000
#define FLG_URFIR	0x2000
#define FLG_PRESAMP	0x4000
#define FLG_PREDELAY	0x8000
#define FLG_ITCPC      0x10000
#define FLG_ITFMT      0x20000
#define FLG_PMWBR      0x40000
#define FLG_PMWBT      0x80000
#define FLG_FIRONLY   0x100000
#define FLG_SWAPIO    0x200000

/* LoadFile flags */
#define FLG_SHARC	0x01000000
#define FLG_IOC		0x02000000
#define FLG_PM		0x03000000
#define FLG_FC		0x04000000
#define FLG_PPC		0x05000000
#define FLG_FLASH	0x06000000
#define FLG_COEF	0x07000000
#define FLG_COEF_FLIP	0x07010000
#define FLG_COEF_PP4	0x07020000
#define FLG_COEF_PP4F	0x07030000
#define FLG_COEF_PP8	0x07060000
#define FLG_MOD		0x08000000
#define FLG_STRATIX	0x09000000
#define FLG_VIRTEX	0x0A000000
#define FLG_LUT		0x0B000000
#define FLG_JVM		0x0C000000
#define FLG_JVM_NIO	0x0C010000
#define FLG_JVM_NVME	0x0C020000
#define FLG_SLR		0x0D000000
#define FLG_JVN		0x0E000000
#define FLG_BIOC	0x0F000000

/* Processor flags */
#define FLG_CTL_BUS	0x00000002
#define FLG_PPC_BUS	0x00000001
#define FLG_PPC_MEM	0x00000000

/* Feed flags */
#define FEED_PORT	0x000000FF
#define FEED_RCFLIP	0x00000100
#define FEED_TUNER	0x00000200

/* Driver status flags */
#define DRV_FLG_XBAR_10M	0x00000001
#define DRV_FLG_XBAR_1PPS	0x00000002
#define DRV_FLG_XBAR_IRIG	0x00000004
#define DRV_FLG_IOM1_10M	0x00000100
#define DRV_FLG_IOM1_1PPS	0x00000200
#define DRV_FLG_IOM2_10M	0x00001000
#define DRV_FLG_IOM2_1PPS	0x00002000

/* Device lock flags */
#define LOCK_OPEN   1
#define LOCK_CLOSE  2
#define LOCK_ALLOC  3
#define LOCK_FREE   4
#define LOCK_QUERY  5
#define LOCK_CLEAR  6

/* Key flags */
#define KEY_RATE	1
#define KEY_FREQ	2
#define KEY_DEC 	3
#define KEY_GAIN	4
#define KEY_OVSR	5
#define KEY_FRAME	6
#define KEY_ALG		7
#define KEY_FEED	8
#define KEY_TCOFF	9
#define KEY_FLAG	10
#define KEY_SWAP	11
#define KEY_CHAIN	12
#define KEY_RATIO	13
#define KEY_MCFG	14
#define KEY_BREAK	15
#define KEY_CHAN	16
#define KEY_ACTIVE	17
#define KEY_DELAY	18
#define KEY_PKTLEN	19
#define KEY_DFREQ	20

#define KEY_PRCCLK	21
#define KEY_IOCCLK	22
#define KEY_ICLK	23
#define KEY_CCLK	24
#define KEY_MCLK	25
#define KEY_PCLK	26
#define KEY_TUNCLK	27
#define KEY_FRAMEDEC	28
#define KEY_DRATE	29
#define KEY_FTTM	30

#define KEY_CLKI	31
#define KEY_MSBI	32
#define KEY_BIT		33
#define KEY_BITS	34
#define KEY_CLKDLY	35
#define KEY_ATCCALIB	36
#define KEY_NODMA	37
#define KEY_TODO	38
#define KEY_GAP		39
#define KEY_MUTE	40

#define KEY_FLGBSET	41
#define KEY_FLGBCLR	42
#define KEY_TFLGBSET	43
#define KEY_TFLGBCLR	44
#define KEY_CSIZE	45
#define KEY_CXFER	46
#define KEY_STIME	47

#define KEY_NFREQ	51
#define KEY_NDEC	52
#define KEY_HUNG	53
#define KEY_CRC32	54
#define KEY_SEQERR	55
#define KEY_SEQFILL	56
#define KEY_PKTHDR	57
#define KEY_NRATIO	58
#define KEY_INBYTE	59
#define KEY_INBYTE_W64	60

#define KEY_RFFREQ	61
#define KEY_RFGAIN	62
#define KEY_RFBW	63
#define KEY_RFOPTS	64
#define KEY_RFATTN	65
#define KEY_RFPWR	66
#define KEY_RFAIS	67
#define KEY_RFCAIS	68
#define KEY_TCXO	69
#define KEY_AGCTC	70
#define KEY_AGCCF	71
#define KEY_MGAIN	72
#define KEY_MFREQ	73
#define KEY_NTPOFF	74
#define KEY_GPS		75
#define KEY_SYSMON	76
#define KEY_GPSMODE	77
#define KEY_LOG		78
#define KEY_MCORES	79
#define KEY_OPTS	80
#define KEY_NFGAIN	81
#define KEY_NYBW	82
#define KEY_FBWF	83

#define KEY_CTYPE	101
#define KEY_PTYPE	102
#define KEY_PINDEX	103
#define KEY_MTYPE	104
#define KEY_MREV	105
#define KEY_TEMP	106
#define KEY_CHNS	107
#define KEY_CPC		108
#define KEY_STATUS	109
#define KEY_VERSION	110
#define KEY_MCHNS	111
#define KEY_CBUFSZ	112
#define KEY_IOCTYPE	113
#define KEY_IOCCODE	113
#define KEY_TINC	114
#define KEY_DRIVER	115
#define KEY_TCMODE	116
#define KEY_IDCODE	117
#define KEY_UIDCODE	118
#define KEY_USER1	119
#define KEY_USER2	120
#define KEY_PMSTAT	121
#define KEY_BSCAN	122
#define KEY_PMTYPE	125
#define KEY_PMCODE	126
#define KEY_CONFIG	127
#define KEY_DETECT	128
#define KEY_QALLOC	129
#define KEY_VERBOSE	130
#define KEY_RSTIME	131
#define KEY_TPOE	132
#define KEY_ROUTE	133
#define KEY_TIMEOUT	134
#define KEY_PMHBT	135
#define KEY_GCHIP	136
#define KEY_JTAGPORT	137
#define KEY_FPGAREV	138
#define KEY_FPGASIG	139
#define KEY_TPD		140
#define KEY_TPDX	141
#define KEY_RMPL	142
#define KEY_WMPL	143
#define KEY_ROUTF	144
#define KEY_DMACHK	145
#define KEY_DMAOFF	146
#define KEY_STATE	147
#define KEY_DUMP	148
#define KEY_MBITS	149
#define KEY_TPSR	150
#define KEY_MPCIO	151
#define KEY_CUID	152
#define KEY_EMT		153
#define KEY_EYESCAN	154
#define KEY_ADLM	155
#define KEY_NAME	156
#define KEY_HWREV	157
#define KEY_EVCNT	158
#define KEY_RTCLK	159
#define KEY_MGTDUMP	160
#define KEY_DRVFLG	161
#define KEY_DCSBN	162
#define KEY_CLKSTAT	163
#define KEY_MTOFF	164
#define KEY_ENABLE	165
#define KEY_SPEEDS	166
#define KEY_FPGADSG	167
#define KEY_ROUTIOS	168
#define KEY_DSGTYPE	169

#define KEY_NVME	170
#define KEY_NIOP	171
#define KEY_NVMS	172
#define KEY_NIOS	173
#define KEY_VCTL	174

#define KEY_NPM		175

#define KEY_DEVNO	190
#define KEY_MTYPE1	191
#define KEY_MTYPE2	192
#define KEY_MREV1	193
#define KEY_MREV2	194
#define KEY_PMTYPE1	195
#define KEY_PMTYPE2	196
#define KEY_PM1CODE	197
#define KEY_PM2CODE	198
#define KEY_PMINDEX	199

#define KEY_PCICLK	201
#define KEY_PCIBUS	202
#define KEY_PCIBW	203
#define KEY_PCIREV	204
#define KEY_PFIFO	205
#define KEY_AFIFO	206
#define KEY_PCINACK	207
#define KEY_PCIGEN	208
#define KEY_PCILANES	209

#define KEY_ARGS	300
#define KEY_TABLE	350
#define KEY_TRACE	360
#define KEY_CNAME	370
#define KEY_QSTAT	380

#define KEY_IPADDR	401
#define KEY_IPCONN	402
#define KEY_IPDISC	403
#define KEY_IPVLAN      404
#define KEY_IPSOCK      405
#define KEY_IPCFG       406
#define KEY_IPTC        407
#define KEY_PING        408

#define KEY_IOC		0x1000
#define KEY_IOCIOM	0x1000
#define KEY_IOCALG	0x1010
#define KEY_IOCRAM	0x1020

#define KEY_MOD		0x2000
#define KEY_APP		0x4000
#define KEY_FLASH	0x8000
#define KEY_CORE	0x01000000
#define KEY_MCORE	0x02000000
#define KEY_NULL	-1

/* Query flags */
#define QUERY_FPLAN	1

/* Application flags */
#define APP_STOP	0
#define APP_START	1
#define APP_PAUSE	2
#define APP_ABORT	-9

/* Send/Recv flags */
#define PKT_RMIF	0x00100000
#define PKT_ACMD	0x00200000
#define PKT_NOWAIT	0x00400000
#define PKT_ACMDNW	0x00600000
#define PKTF_BUS_RD	1
#define PKTF_BUS_WR	2
#define PKTF_MEM_RD	3
#define PKTF_MEM_WR	4
#define PKTF_MEM_TEST	5
#define PKTF_INIT	6
#define PKTF_MODIFY	7
#define PKTF_RUN	8
#define PKTF_STOP	9
#define PKTF_MEM_CAL	11
#define PKTF_LNK_CAL	12
#define PKTF_HYP_CAL	13
#define PKTF_BUS_RWM	14
#define PKTF_MEM_INIT	15
#define PKTF_GET16      0x00100000
#define PKTF_GET32      0x00200000
#define PKTF_GET512     0x02000000
#define PKTF_GET1K      0x04000000

/* Packet types */
#define PKTMODELIST "AUTO,NONE,ETH,UDP,ICE,SDDS,VRT,VRTL,VRTW,VRTX,VRTD"
#define PKT_AUTO	-1
#define PKT_NONE	0x0
#define PKT_ETH		0x1
#define PKT_UDP		0x2
#define PKT_ICE		0x3
#define PKT_SDDS	0x4
#define PKT_VRT		0x5
#define PKT_VRTL	0x6
#define PKT_VRTW	0x7
#define PKT_VRTX	0x8
#define PKT_VRTD	0x9

#define OUID_ICE	0x00104D77
#define OUID_DIFI	0x006A621E

/* Buffer flags */
#define BUF_EXPAND	1

/* type 2 name conversion modes */
#define T2N_CARD	1
#define T2N_IOM		2
#define T2N_PM		3
#define T2N_PORT	4

#define MAX_MCS		16
#define MAX_IOM		2
#define MAX_PM		4

/* DMA channel structures */
typedef struct {
  int_u4 haddr, hsize, hindex, hcycle;
  int_u4 caddr, csize, cindex, ccycle;
  int_4 stat,  dir,   todo,   block;
  int_4 cntrl, tcbp,  reg,    mcsr;
  int_4 enb,   enbx,  mcfg,   mcfgx;
  int_4 err,   miss,  iper,   multi;
  int_4 hxfer, hxfes, cxfer,  cxfes;
  int_4 cindp, hindp, cind,   hind;
  int_4 type,  port,  bits,   dec;
  int_4 rate,  gain,  fcny,   frame;
  int_4 master,slave, feed,   tcoff;
  int_4 alg,   appp,  chain,  flags;
} DMASTRUCT;

/* DMA CHAIN structures */
typedef struct {
  int_4 haddr, hsize, todo,   chain;
} DMACHAINSTRUCT;

/* APP channel structures */
typedef struct {
  int_4 ena,   alg,   xferi,  xfero;
  int_4 dmai,  dmao,  indi,   indo;
  int_4 args[4];
  int_4 ici,   ico,   tc,     tcbp;
} APPSTRUCT;

typedef  struct {
  int_4 ii, im, ic, cp, gp, db, da;
} TCBSTRUCT;

typedef  struct {
  int_4 xs, xi, xc, cp, yc, yi;
} TCB2DSTRUCT;

typedef  struct {
  int_4 dvid,pcsr,code,plat;
  int_4 bar0,bar1,bar2,bar3;
  int_4 bar4,bar5,cbcis,subs;
  int_4 exrom,resv1,resv2,lgir;
  int_4 fifo,fil1,mcsr,fil2;
  int_4 fil3,ioc,mdar,mdtc;
  int_4 omb1,omb2,omb3,omb4; 
  int_4 imb1,imb2,imb3,imb4; 
  int_4 hir[32];
} PCISTRUCT;

typedef  struct {
  int_4 fifo,pcsr,code,plat;
  int_4 bar0,bar1,bar2,bar3;
  int_4 bar4,bar5,cbcis,subs;
  int_4 exrom,resv1,resv2,lgir;
  int_4 dvid,fil1,mcsr,fil2;
  int_4 fil3,ioc,mdar,mdtc;
  int_4 omb1,omb2,omb3,omb4; 
  int_4 imb1,imb2,imb3,imb4; 
  int_4 hir[32];
} PCIADONSTRUCT;

typedef struct {
  int_8 paddr;			/* physical address in bytes */
  int_8 bytes;			/* physical size in bytes */
  int_8 offset;			/* offset to current virtual segment */
  int_8 vbytes;			/* current virtual segment size */
  char  devn[8];		/* backing device name - if not pmem */
  char *vaddr;			/* current virtual segment address */
  int_4 vfill;			/* for clearing upper INTEGER*8 vaddr on 32bit OS */
} DMAMAP;

typedef struct {
  int_4 dir;
  int_4 rate;
  int_4 gain;
  int_4 bits;
} IOMPARAM;

/* Structure used as a handle to the PIC device */
typedef struct {
  int_4 type;			/* card type */
  int_1 rev;			/* card revision */
  int_1 ptype;			/* IO port type */
  int_1 pindex;			/* IO port index */
  int_1 pmi;			/* Current Processor module index */
  int_1 pmif;			/* Current Processor module index flag */
  int_1 miom;			/* Max IO Modules */
  int_1 mtype[MAX_IOM];		/* IO Module types */
  int_1 mrev[MAX_IOM];		/* IO Module revision */
  int_1 mpm,npm;		/* Max and actual Processor Modules */
  int_1 pmtype[MAX_PM];		/* Processor Module types */
  int_1 pmrev[MAX_PM];		/* Processor Module revision */
  int_4 flags;			/* Modifier flags */
  int_4 mchan,nchan;		/* Max and current number of active channels */
  int_1 cpb,cpc;		/* ChipsPerBoard and ChannelsPerChip */
  int_1 gmcs,gmcr;		/* Global master clock source & register */
  int_4 magic;

  int_4 devno;			/* device number in system */
  int_4 lcount;			/* lock up/down counter */
  int_2 pbw,pbr;		/* PCI bus width (bytes) and rate (MHz) */
  int_1 state;			/* current card state 1=ok */
  int_1 side;			/* current card side 0,1,2,3 */
  int_1 timeout;		/* timeout (sec) */
  int_1 verbose;		/* debug verbose mode */
  int_4 int_clk,prc_clk;	/* on-board clocks */
  int_4 ioc_clk,ext_clk;	/* on-board clocks */

  int_2 ovsr[MAX_IOM];		/* Tuner oversampling factor */
  int_4 tflags;			/* Tuner mode flags */
  int_1 bphw,bpcl;		/* bytes per host word, Kintex-7 */
  int_1 inp,mbits;		/* INP and MBITS flags */
  int_1 isX,isY;		/* flags for PIC4 , PIC5 extensions */
  int_1 tcmode,tcflags;		/* timecode mode and flags */
  char  tc_mode[8];		/* timecode mode string */
  int_1 decb,dec2,dec5,deco;	/* AD6620|TunerBank parameters */
  int_2 bpp,ctaps;		/* bytes per DMA packet & CFIR number taps */
  double cfir_gain,pfir_gain,res_gain; /* GC4016 filter gains */
  double dfreq,ofreq;		/* channel spacing and offset for tuner banks */
  double res_ratio;		/* resampler ratio */
  double atccalib;		/* additional time code calibration offset */
  double feedrdec;		/* feed decimation for time code */
  double tc_wsec,tc_fsec,tc_soff;/* timestamp whole sec, fractional sec and status/offset */
  char  iocc[4];		/* IOC code signature */
  int_1 xmux,xinc;		/* mux mode counters */
  int_1 k7,ndmac;		/* Kintex version / # dma channels */
  DMASTRUCT dma[DMA_CHNSY];	/* DMA channel structures */
  APPSTRUCT app[APP_CHNS];	/* APP channel structures */
  TCB2DSTRUCT tcb[DMA_CHNSY];	/* TCB channel structures */
  double pretcount[DMA_CHNSY];	/* DMA channel presampler starting offset */
  int_8  dma_vindx[DMA_CHNSY];	/* DMA channel virtual buffer index */
  int_4  dma_vcycx[DMA_CHNSY];	/* DMA channel virtual buffer cycle */
  int_4 *dma_vaddr[DMA_CHNSY];	/* DMA channel virtual buffer addresses */
  int_4  dmaPkt[DMA_CHNSY];	/* DMA isfp assignment */
  int_4 *chantbl,*chiptbl,*linktbl,*tadrtbl;	/* multichan table pointers */
  int_4 *iom_state[MAX_IOM];
  int_4 *iom_buf;
  int_4 iom_cnt;
  int_2 iom_prt_jvm[MAX_IOM];
  int_4 _linktbl[34],_tadrtbl[18]; /* local tables */
  int_4 pkt[16];
  int_u4 basf,size;		/* Memory map parameters */
  int_8  base;			/* Memory map parameters */
  int_8  regX;
  int_4  *loc;
  PICIOCTL ioctl;		/* ioctl() call arguments */
#if _WIN
  int_4 ioctldata[65536];	/* IOCTL data buffer */
  void *fd;			/* Windows device handle */
#else
  int_4 fd;			/* File descriptor for VMS/UNIX drivers */
#endif
  int_4 X;			/* temp var for memmap driver calls */
  int_4 ims,ems;		/* SHARC internal memory start offset */
  int_4 *pbreak,vbreak;		/* Break pointers */
  char *config;			/* pointer to config string */
  void *ftt;			/* pointer to current FTT parameters */
  int_4 fttm;			/* number of passes in Fast Tuner Transform */
  int_4 tc_modl;		/* stupid hold of last tc_mode for BC */
  int_4 pktlen;			/* frame size for tuner banks */
  int_4 chan;			/* current tunerbank channel for KEY access */
  int_4 irbreg;			/* IRIG-B TimeCode confiuration register */
  int_4 ntaps,nres;		/* Number of taps in current tuner filter */
  int_2 d1,d2,d3,d4;		/* TunerBank stage decimation parameters */
  int_2 m1,m2,m3,m4;		/* TunerBank stage channels parameters */
  int_4 phinc,dphinc;		/* more TunerBank parameters */
  int_4 tun_clk;		/* Tuner oversampling clock */
  char  socc[4];		/* PIC5+ signature */
  int_4 crcs[7];		/* FPGA download checksums 0=IOC 1|2=IOM 3|4=PM */
  int_4 gchip;			/* Graychip Tuner Mode 1=GC4014 2=GC4016 3=GC5016 4=FPGA */
  int_4 res_mon;		/* resampler M over N ratio */
  int_4 res_inc;		/* resampler increment register */
  int_4 res_dec;		/* presampler decimation register */
  int_4 res_ovsr;		/* presampler oversampling register */
  int_4 pid,tid;		/* current process and thread id */
  int_4 oppsoffset;		/* OPPS Offset */
  int_4 tbflags;		/* TBank operational flags for DMA_RESET */
  int_1 pm0mt;			/* Number of base board multi-tuners */
  int_1 gaux;			/* Global Auxiliary register */
  int_1 pm0t;			/* Number of base board tuners */
  int_1 presamp;		/* indicates use of presampler path on this channel */
  int_1 pktmode;		/* Current packet mode */
  int_1 flashport;		/* spi flash port */
  int_1 qdrx;			/* Quad IOM port mode */
  int_1 mcs;			/* MultiCore Streaming Mode */
  int_1 dsgtype;		/* version of design load (PIC=0 PAC=1 NPAC=2 SPAC=3 PICQ=4 TRAY=5) */
  int_1 nio,niop,nioi;		/* NIO mode, ports, and invert */
  int_1 dio,nvme,up12;		/* DirectIO, NVME mode, UpConvert 12b modes */
  int_1 pktntoff;		/* PKT control buffer offset to TC words */
  float ntpoffset;
  int_4 leapsecsoy;		/* second of year at which to apply leap second */
  int_4 dmaci,dmaco;		/* internal input/output dma channels */
  DMAMAP mapi,mapo;
  int_4 todo;
  int_4 frame;
  int_4 driver;
  int_4 sock,nicno;
  int_4 ipsock,ipaddr,ipconn,ipvlan;
  int_4 pktseq;
  int_4 csig,tclk,order,mdw,nps,cfxo;
  IOMPARAM iomparam[MAX_IOM];
#ifdef _LINX
  pthread_t threadid;
  pthread_t threadid2;
#endif
  void *plan[MAX_MCS];		/* core plan structs */
  void *halo;			/* core plan structs */
  double joinTimer,joinTime;
  double vctlTimer,vctlTime;
  double pollTimer,vctlFill;
  double zeroTime;
  double xfreq,xcf,xcfb;	/* Xcvr tuned freq, output center freq and bypass point */
  float fbwf;
  char cname[16];
  char xname[16];
  int_1 xcvr,xchn,xena,xinv;	/* XCVR flag xcvr=present xchn=IoModN xena=notBypassed xinv=invertedSpectrum */
  int_4 tsig;
  int_4 isDUC;
  int_4 pcyc;
  int_4 ouid;
  int_4 coreAddrX;
  int_4 pktnctxoff;		/* Vita Contex packet data offset */
  int_2 nioctl[16];
  int_4 vctx;			/* Vita context field mask */
} PICSTRUCT;

typedef struct {
  int_4 block, total, sjump, bjump;
  float srate, brate;
} DMASPEC;

typedef struct {
  double offset, delta, soy, fsoy; 
} TIMECODE;

typedef struct { 
  int_4 type, index, dmac, dir, flags, pad;
  double rate, fcny, dec, gain;
} IOPORT;

typedef struct { 
  int_1 func, flag, info, rep, seq, try_, rpt, adj;
} PKT_RMIF_STRUCT;

typedef struct { 
  int_4 myip; 	/* HW NIC ip address */
  int_4 mcip; 	/* ip address to import from or export to */
  int_4 port;	/* port number */
  int_4 pkty;	/* packet type: ICE/SDDS/VITA */
  int_4 form;	/* packet format */
  int_4 size;	/* packet data size in bytes */
  int_4 tinc;	/* time code increment */
} NIC_PORT_STRUCT;

typedef struct { 
  int_2 func; 
  int_2 size;
  int_4 addr;
} PKT_ACMD_STRUCT;

typedef struct {
  char   e_ident[16];
  int_u2 e_type;
  int_u2 e_machine;
  int_u4 e_version;
  int_u4 e_entry; 
  int_u4 e_phoff;
  int_u4 e_shoff;
  int_u4 e_flags;
  int_u2 e_ehsize;
  int_u2 e_phentsize;
  int_u2 e_phnum;
  int_u2 e_shentsize;
  int_u2 e_shnum;
  int_u2 e_shstrndx;
} ELFHDRSTRUCT;

typedef struct {
  int_u4 p_type;
  int_u4 p_offset;
  int_u4 p_vaddr;
  int_u4 p_paddr;
  int_u4 p_filesz;
  int_u4 p_memsz;
  int_u4 p_flags;
  int_u4 p_align;
} ELFPGMSTRUCT;

typedef struct {
  int_u2 sh_name;
  int_u2 sh_type;
  int_u4 sh_flags;
  int_u4 sh_addr;
  int_u4 sh_offset;
  int_u4 sh_size;
  int_u4 sh_link;
  int_u4 sh_info;
  int_u4 sh_addralign;
  int_u4 sh_entsize;
} ELFSECSTRUCT;

#ifdef __cplusplus
extern "C" {
#endif

/* Function prototypes */
int_4 pic_open (PICSTRUCT *p, const char *str, int_4 *pbreak, int_4 flags);
int_4 pic_close (PICSTRUCT *p);
int_4 pic_lock (PICSTRUCT *p, int_4 mode);
int_4 pic_active (PICSTRUCT *p, int_4 flags);
int_4 pic_reset (PICSTRUCT *p, int_4 flags);
int_4 pic_sniff (PICSTRUCT *p, int_4 flags);
int_4 pic_test (PICSTRUCT *p, int_4 test, int_4 flags);
int_4 pic_read (PICSTRUCT *p, int_4 offset, int_4 *value);
int_4 pic_write (PICSTRUCT *p, int_4 offset, int_4 *value);
int_4 pic_writem (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 mask);
int_4 pic_rfifo (PICSTRUCT *p, int_4 *value, int_4 bytes, int_4 flags);
int_4 pic_wfifo (PICSTRUCT *p, int_4 *value, int_4 bytes, int_4 flags);
int_4 pic_rpm (PICSTRUCT *p, int_4 node, int_4 addr, int_4 *data, int_4 bytes, int_4 flags);
int_4 pic_wpm (PICSTRUCT *p, int_4 node, int_4 addr, int_4 *data, int_4 bytes, int_4 flags);
int_4 pic_msg (PICSTRUCT *p, int_4 node, int_4 func, int_4 addr, int_4 *data, int_4 sbytes, int_4 rbytes, int_4 timeout);
int_4 pic_sendrecv (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data, int_4 bytes, int_4 maxbytes, int_4 flags, int_4 timeout);
int_4 pic_send (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data, int_4 bytes, int_4 flags);
int_4 pic_recv (PICSTRUCT *p, int_4 node, int_4 *head, int_4 *data, int_4 bytes, int_4 flags);
int_4 pic_rapp (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 flag);
int_4 pic_wapp (PICSTRUCT *p, int_4 offset, int_4 *value, int_4 flag);
int_4 pic_setkey (PICSTRUCT *p, int_4 dmac, int_4 key, void *vval, int_4 len);
int_4 pic_getkey (PICSTRUCT *p, int_4 dmac, int_4 key, void *vval, int_4 len);
int_4 pic_setkeyl (PICSTRUCT *p, int_4 dmac, int_4 key, int_4 lval);
int_4 pic_getkeyl (PICSTRUCT *p, int_4 dmac, int_4 key);
int_4 pic_getkeyv (PICSTRUCT *p, int_4 dmac, int_4 key);
int_4 pic_setkeyd (PICSTRUCT *p, int_4 dmac, int_4 key, double dval);
double pic_getkeyd (PICSTRUCT *p, int_4 dmac, int_4 key);
int_4 pic_setkeyt (PICSTRUCT *p, int_4 dmac, char *tbl);
int_4 pic_getkeyt (PICSTRUCT *p, int_4 dmac, char *tbl, int_4 len);
int_1 pic_getkeytype (PICSTRUCT *p, int_4 key);
int_4 pic_getkeysize (PICSTRUCT *p, int_4 key);
int_4 pic_name2key (PICSTRUCT *p, char *name);
char* pic_key2name (PICSTRUCT *p, int_4 key);
int_4 pic_name2offset (PICSTRUCT *p, char *name, int_4 flags);
int_4 pic_tuner_dec (PICSTRUCT *p, int_4 dec, int_4 port, int_4 flags);
real_8 pic_tuner_freq (PICSTRUCT *p, real_8 fcny, int_4 flags);
int_4 pic_tuner_ovsr (PICSTRUCT *p, int_4 factor, int_4 port, int_4 flags);
int_4 setupTunerChip (PICSTRUCT *p, int_4 port, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags, real_8 ratio);
int_4 pic_parsetc (PICSTRUCT *p, char *string, int_4 flags);
int_4 pic_ioport (PICSTRUCT *p, int_4 ptype, int_4 pindex, int_4 dmac, 
  int_4 dir, int_4 bits, int_4 rate, real_8 fcny, int_4 dec, int_4 gain, int_4 flags);
int_4 pic_compress (int_u1 *ibuf, int_4 ilen, int_u1 *obuf, int_4 olen, int_4 dir);
#ifdef _XMIDAS
#define XM_CDEFS_H_
#include "headers.h"
int_4 pic_file (PICSTRUCT *p, HEADER *hcb, int_4 *nstart, int_4 *nbytes, int_4 mode);
int_4 pic_mapfile (PICSTRUCT *p, DMAMAP *map, HEADER *hcb, int_4 mode);
#endif
int_4 pic_ioctl (PICSTRUCT *p, int_4 function, int_4 offset, int_4* buffer, int_4 bytes); 
int_4 pic_map (PICSTRUCT *p, int_4 **vaddr,  int_4 *paddr, int_4 bytes, int_4 mode);
int_4 pic_mapmem (PICSTRUCT *p,  DMAMAP *map, int_4 bytes, int_4 mode);
int_4 pic_dma (PICSTRUCT *p, int_4 dmac, int_4 dir, int_4 buffer[], int_u4 nstart, int_4 nbytes, int_4 nblock, int_4 flags);
int_4 pic_dmasetup (PICSTRUCT *p, int_4 dmac, int_4 dir, DMAMAP *map, int_4 nblock, int_4 flags);
int_4 pic_dmafunc (PICSTRUCT *p, int_4 dmac, int_4 func);
int_8 pic_dmafuncx (PICSTRUCT *p, int_4 dmac, int_4 func);
int_4 pic_dmaxfer (PICSTRUCT *p, int_4 dmac, void *buffer, int_4 bytes, int_4 flags);
int_4 pic_dmaxptr (PICSTRUCT *p, int_4 dmac, void **buffer, int_4 bytes, int_4 flags);
int_4 pic_dmadump (PICSTRUCT *p, int_4 dmac, int_4 flags);
int_4 pic_dmastat (PICSTRUCT *p, int_4 dmac, int_4 *index, int_4 *cycle);
int_4 pic_dmachain (PICSTRUCT *p, int_4 dmac, int_4 index,  DMAMAP *map, int_4 todo, int_4 nextindex);
int_4 pic_load (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags);
int_4 pic_loadioc (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags);
int_4 pic_loadsharc (PICSTRUCT *p, int_u1 *loadbyte, int_4 nprog, int_4 flags);
int_4 pic_loadmod (PICSTRUCT *p, int_4 port, int_u1 *loadbyte, int_4 nprog, int_4 flags);
int_4 pic_loadfc (PICSTRUCT *p, int_4 port, int_2 *data, int_4 ndata, int_4 flags);
int_4 pic_setup_gps (PICSTRUCT *p, int_4 port);
int_4 pic_dump_gps (PICSTRUCT *p, int_4 port);
int_4 pic_loadfile (PICSTRUCT *p, char *name, int_4 flags);
int_4 pic_getstrflag (PICSTRUCT *p, char *name, char *str, int_4 flags);
int_4 pic_getintflag (PICSTRUCT *p, char *name);
int_4 pic_getintflagdef (PICSTRUCT *p, char *name, int_4 def);
real_8 pic_getdblflag (PICSTRUCT *p, char *name);
real_8 pic_getdblflagdef (PICSTRUCT *p, char *name, real_8 def);
int_4 mpc_read (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len);
int_4 mpc_write (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len);
int_4 mpc_writem (PICSTRUCT *p, int_4 addr, int_1 data, int_1 mask);
int_4 ampc_read (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len);
int_4 ampc_write (PICSTRUCT *p, int_4 addr, int_1 *data, int_4 len);
int_4 ampc_writem (PICSTRUCT *p, int_4 addr, int_1 data, int_1 mask);

int_4 pic_timer (PICSTRUCT *p, int_4 port, int_4 rate);
int_4 pic_nvram (PICSTRUCT *p, char *fname, int_4 mode);
int_4 pic_tc (PICSTRUCT *p, int_4 dmac, real_8 *offset, real_8 *delta, real_8 *soy, real_8 *fsoy, int_4 flags);
int_4 pic_timecode (PICSTRUCT *p, int_4 dmac, char *mode, real_8 offset, real_8 delta, int_4 dec, int_4 bits, int_2 *buffer, real_8 *soy, real_8 *fsoy);
int_4 pic_tc_insert (PICSTRUCT *p, int_2 *buffer, int_4 length, real_8 delta, real_8 soy, real_8 fsoy, int_4 flags);
int_4 pic_buffer (PICSTRUCT *p, DMAMAP *map, int_4 offset, int_1 *bbuf, int_4 length, int_4 mode);
int_4 pic_type2name (PICSTRUCT* p, int_4 mode, int_4 type, int_4 rev, char *name);
void pic_prettyprint (PICSTRUCT* p);
void pic_print_pm (PICSTRUCT* p, int_4 offset);
int_4 pic_spec (PICSTRUCT *p, int_4 *data, int_4 nstart, int_4 ndata, int_4 nblock,
  int_4 *Ntotal, float *Maxsrate, int_4 *Maxsjump, float *Maxbrate, int_4 *Maxbjump, int_4 flags);
int_4 pic_detect (int_4 flags);
int_4 pic_str2ip (PICSTRUCT *p, char *str);
int_4 pic_jpmemrd (PICSTRUCT *p, int_4 port, int_4 addr, int_4 len);
int_4 pic_jpmemwr (PICSTRUCT *p, int_4 port, int_4 addr, int_4 data, int_4 len);
int_4 pic_jpmemmsk (PICSTRUCT *p, int_4 port, int_4 addr, int_4 dand, int_4 dxor, int_4 len);
int_4 pic_setcore (PICSTRUCT *p, int_4 port, int_4 addr, int_4 *lval, int_4 bytes);
int_4 pic_setcorel (PICSTRUCT *p, int_4 port, int_4 addr, int_4 lval);
int_4 pic_test_upac (PICSTRUCT *p, int_4 test);
int_4 pic_getGPSrev (PICSTRUCT *p, int_4 port);
int_4 pic_query (PICSTRUCT *p, int_4 type, int_4 info, void *buf, int_4 len);
int_4 pic_xlock (int_4 mode);
char *pic_mapram (int_8 base, int_8 size);
void  pic_unmapram (int_8 base, int_8 size, char *addr);
double pic_time ();
char* pic_stime ();
void  udelay (int_4 usec);

#ifdef __cplusplus
}
#endif

#define ICELIB_H_
#endif /* ICELIB_H_ */

