/****************************************************************/
/*  ICE-PIC Device Driver for SOLARIS                           */
/*                                                              */
/*  Author:  Jeff Schoen - Innovative Computer Engineering      */
/*                                                              */
/****************************************************************/

#define ICE_DRIVER_VERSION 338

/*****************/
/* Include Files */
/*****************/
#include <sys/debug.h>
#include <sys/types.h>
#include <sys/buf.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/map.h>
#include <sys/cred.h>
#include <sys/open.h>
#include <sys/stat.h>
#include <sys/kmem.h>
#include <sys/cmn_err.h>
#include <sys/conf.h>
#include <sys/ddi.h>
#include <sys/devops.h>
#include <sys/sunddi.h>
#include <sys/modctl.h>
#include <sys/proc.h>
#include <sys/pci.h>
#include <sys/vmsystm.h>

/**********************************/
/* Function Prototype Definitions */
/**********************************/
/* Data access requirements. */
#ifdef _LITTLE_ENDIAN
static struct ddi_device_acc_attr endian_attr = {
        DDI_DEVICE_ATTR_V0,
        DDI_STRUCTURE_BE_ACC,
        DDI_STRICTORDER_ACC
};
#else
static struct ddi_device_acc_attr endian_attr = {
        DDI_DEVICE_ATTR_V0,
        DDI_STRUCTURE_LE_ACC,
        DDI_STRICTORDER_ACC
};
#endif
static struct ddi_device_acc_attr nosw_attr = {
        DDI_DEVICE_ATTR_V0,
        DDI_NEVERSWAP_ACC,
        DDI_STRICTORDER_ACC
};

/***********************/
/* General Definitions */
/***********************/
#define ESUCCESS 0
#define	SUCCESS(a)	(((a) & 1) == 1)
#define	FAILURE(a)	(((a) & 1) == 0)

#define vprintf(A) 		if (verbose==1) cmn_err(CE_NOTE,A)
#define vprintf1(A,B) 		if (verbose==1) cmn_err(CE_NOTE,A,B)
#define vprintf2(A,B,C) 	if (verbose==1) cmn_err(CE_NOTE,A,B,C)
#define vprintf3(A,B,C,D) 	if (verbose==1) cmn_err(CE_NOTE,A,B,C,D)
#define vprintf4(A,B,C,D,E) 	if (verbose==1) cmn_err(CE_NOTE,A,B,C,D,E)
#define vprintf5(A,B,C,D,E,F) 	if (verbose==1) cmn_err(CE_NOTE,A,B,C,D,E,F)

/*******************************/
/* Driver Specific Definitions */
/*******************************/

#define _UNIX 1
#define _SOL  1
#ifndef _LITTLE_ENDIAN
#define _EEEI 1
#else
#define _IEEE 1
#endif

#include "iceioctl.h"
#include "iceflex.h"

#define MAXPIC 16
#define MAXMAP 4096
#define MAXIOBUF 512
#define MAXBCNT (1024*1024*1024)

#define VENDOR_ID 0x1172
#define DEVICE_ID 0x7777

typedef struct {
    dev_t		device;
    caddr_t 		usradr;
    int			bytes;
    uint_t 		pciadr;
    struct buf*		bp;
    ddi_dma_handle_t	handle;   
    ddi_dma_cookie_t	cookie[8];
    uint_t		ccount;
} DMACHANNEL;

struct ddi_dma_attr dmaAttr = {
    DMA_ATTR_V0,
    0,
    0xffffffff,
    0xffffffff,
    1,
    1,
    1,
    0xffffffff,
    0xffffffff,
    1,
    512,
    0
};

typedef struct {
  int		   attached;
  int		   opened;
  char* 	   loc;
  char*		   cfg;
  dev_t	 	   device;
  long 		   memsize;
  DMACHANNEL	   dma[MAXIOBUF];
  dev_info_t*	   devi;
  ddi_acc_handle_t cfgHandle;
  ddi_acc_handle_t Handle;
} UCB;

static UCB 	    ucbs[MAXPIC];
static int 	    verbose=1;

/***********************************/
/* Device Driver Macro Definitions */
/***********************************/
#define _RD(A,B)  { \
    B = (int)ddi_get32 (ucb->Handle, (uint32_t*)(ucb->loc+A)); }
#define _WR(A,B)  { \
    ddi_put32 (ucb->Handle, (uint32_t*)(ucb->loc+A), B); }

/************************************/
/* Prototypes for External Routines */
/************************************/
static int      icepic_open(dev_t*, int, int, cred_t*);
static int      icepic_close(dev_t, int, int, cred_t*);
static int      icepic_ioctl(dev_t, int, intptr_t, int, cred_t*, int*);
static int	icepic_strategy (struct buf*);
static void 	icepic_minphys (struct buf*);

int icepic_mapio (UCB*, int, int);

static struct cb_ops icepic_cb_ops = {
        icepic_open,            /* open */
        icepic_close,           /* close */
        icepic_strategy,        /* strategy */
        nodev,                  /* print */
        nodev,                  /* dump */
        nodev,                  /* read */
        nodev,                  /* write */
        icepic_ioctl,           /* ioctl */
        nodev,                  /* devmap */
        nodev,            	/* mmap */
        nodev,          	/* segmap */
        nochpoll,               /* poll */
        ddi_prop_op,            /* cb_prop_op */
        0,                      /* streamtab */
        D_NEW | D_MP            /* Driver compatibility flag */
};

static int icepic_info(dev_info_t*, ddi_info_cmd_t, void*, void**);
static int icepic_attach(dev_info_t*, ddi_attach_cmd_t);
static int icepic_detach(dev_info_t*, ddi_detach_cmd_t);
static int icepic_probe(dev_info_t*);

static struct dev_ops icepic_ops = {
        DEVO_REV,               /* devo_rev, */
        0,                      /* refcnt */
        icepic_info,            /* info */
        nulldev,                /* identify */
        nulldev,           	/* probe */
        icepic_attach,          /* attach */
        icepic_detach,          /* detach */
        nodev,                  /* reset */
        &icepic_cb_ops,         /* driver operations */
        (struct bus_ops*) 0,    /* bus operations */
        NULL,                   /* power operations */
};

static int currentpid (void);
static void copy_to_user (void*, void*, int);
static void copy_from_user (void*, void*, int);
static void udelay (int);
static void reverse (void *, int);

/***********************************************/
/* Loadable Driver Configuration Definitions   */
/***********************************************/
extern struct mod_ops   mod_driverops;

static struct modldrv icepic_modldrv = {
        &mod_driverops,             /* Type of module.This one is a driver */
        "ICE icepic driver v310",   /* Name of the module. */
        &icepic_ops,                /* driver ops */
};

static struct modlinkage icepic_modlinkage = {
        MODREV_1,
        (void*) &icepic_modldrv,
        NULL,
};

int _init (void) 
{ 
    int error;
    int i, j;

    error = mod_install (&icepic_modlinkage); 
    if (error != 0) 
        return (error);

    for (i = 0; i < MAXPIC; i++) {
        ucbs[i].attached = 0;
        for (j = 0; j < MAXIOBUF; j++) {
	    ucbs[i].dma[j].pciadr  = 0;
	}
    }
    return error;
}

int _fini (void) 
{ 
    int error;

    error = mod_remove (&icepic_modlinkage); 

    return error;
}

int _info (struct modinfo* modinfop) 
{ 
    int error;
    error = mod_info (&icepic_modlinkage, modinfop); 
    return error;
}

static int icepic_attach (register dev_info_t* dip,
			 register ddi_attach_cmd_t cmd)
{
    int 	index,status;
    char 	name[20];
    UCB* 	ucb;
    caddr_t 	regptr;
  
    index         = ddi_get_instance(dip);
    ucb           = &ucbs[index];
    ucb->attached = 1;
    ucb->devi     = dip;

    sprintf (name, "icepic%d", index);
    vprintf1("ICEPIC Attaching %s", name);
    status = ddi_create_minor_node (dip, name, S_IFCHR, index, "icepic", 0); 
    if (status == DDI_FAILURE) {
	ddi_remove_minor_node (dip, NULL);
	vprintf1 ("icepic_attach: create_minor_node failed for %d", index);
	return (DDI_FAILURE);
    }

    status = pci_config_setup (dip, &ucb->cfgHandle);
    if (status == DDI_SUCCESS) ;
    else vprintf1("ICEPIC Config Setup status=%08x", status);

    ddi_dev_regsize (dip, 1, &ucb->memsize);
    status = ddi_regs_map_setup (dip, 1, &regptr, 0, ucb->memsize,
				&endian_attr, &ucb->Handle);
    if (status == DDI_SUCCESS) {
      ucb->loc = (char*) regptr;
      vprintf2("ICEPIC Attached at %08x %08x", ucb->loc, ucb->memsize);
      ddi_report_dev (dip);
    }
    else
      vprintf1("ICEPIC REG Map status=%08x", status);

    return (DDI_SUCCESS);
}

static int icepic_detach (register dev_info_t* dip,
			 register ddi_detach_cmd_t cmd)
{
    UCB* 	ucb;
    int index;

    index = ddi_get_instance (dip);
    vprintf1("ICEPIC Detaching %d", index);
    ucb = &ucbs[index];
    ucb->attached = 0;

    ddi_regs_map_free (&ucb->Handle);
    pci_config_teardown (&ucb->cfgHandle);

    return (DDI_SUCCESS);
}

static int icepic_info (dev_info_t* dip, ddi_info_cmd_t infocmd, 
			void* arg, void** result)
{
    register int index;
    register int error = DDI_SUCCESS;
    UCB* ucb;
    
    index = getminor ((dev_t) arg);
   
    ucb = &ucbs[index]; 

    switch (infocmd) {
    case DDI_INFO_DEVT2DEVINFO :
	*result = (void*) ucb->devi;
	break;

    case DDI_INFO_DEVT2INSTANCE:
	*result = (void*) index;
	break;

    default :
	error = DDI_FAILURE;
	break;
    }
    return (error);

}

/************************************/
/* pic open routine                 */
/************************************/
static int icepic_open (register dev_t* devp, register int flag,
        		register int otyp, register cred_t* cred)
{
  int index;
  UCB *ucb;

  index = getminor(*devp);
  vprintf1 ("icepic_open - dev %x",index);

  ucb = &ucbs[index];
  if (ucb->attached == 0) {
    vprintf("icepic_open - device unit not attached");
    return(-ENXIO);
  }
  ucb->opened++;

  return(ESUCCESS); 
}
 
/************************************/
/* pic close routine                */
/************************************/
static int icepic_close (register dev_t dev, register int flag,
        		 register int otyp, register cred_t* cred)
{
    int index = getminor(dev);
    UCB *ucb = &ucbs[index];;
    vprintf1 ("icepic_close - dev %d",index);
    if (ucb->attached == 0) {
        vprintf("icepic_close - device unit not attached");
    }

 /* make sure all DMA's from this PID have been terminated */
    icepic_cancel (index, 1);
    ucb->opened--;
    return 0;
}
    
/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_BUFADDR	(long) p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_STATUS	p->status	/* return status */

static int icepic_ioctl (dev_t dev, int cmd, intptr_t arg, int mode, 
			 cred_t* cred, int* rval)
{
    int 	index = getminor(dev);
    UCB*  	ucb = &ucbs[index];

    int		i,j,k;		/* Local counters */
    int		status,stat;	/* Local function return status */
    int		c;		/* Current dma channel */
    PICIOCTL*   p;
    PICIOCTL	ploc;		/* pic ioctl handle structure */
    uint_t	rvalue;
    int		wtmp[6];	/* kernel space temp data */
    int		worda,wordb;	/* load IOC temps */
#ifdef _MULTI_DATAMODEL
    PICIOCTL32	p32;
#endif

    ucb->device = dev;

    /* vprintf3("icepic_ioctl - dev %x, cmd %x, arg %x", index,cmd,arg); */

    p = &ploc;	/* ioctl data = the PICIOCTL */

#ifdef _MULTI_DATAMODEL
    switch (ddi_model_convert_from (mode & DATAMODEL_MASK)) {
    case DDI_MODEL_ILP32: 
	copy_from_user ((void*) &p32, (void*) arg, sizeof(PICIOCTL32));
	p->function = p32.function;
	p->offset   = p32.offset;
	p->bufaddr  = (void *) p32.bufaddr;
	p->bytes    = p32.bytes;
	p->status   = p32.status;
	break;
    case DDI_MODEL_NONE:
        copy_from_user ((void*) p, (void*) arg, sizeof(PICIOCTL));
	break;
    }
#else
    copy_from_user ((void*) p, (void*) arg, sizeof(PICIOCTL));
#endif

    rvalue = PIC_BYTES;			/* normal|default completion status */
    status = ESUCCESS; 			/* assume success */

    switch (PIC_FUNCTION) {
    case IOCTL_READ:			/* read a device register */

        if (PIC_OFFSET == REG_FIFO) {
            for (i = 0; i < PIC_BYTES; ) {
                _RD (REG_MCSR, stat);
                if (stat&MCSR_IFIFO_FULL && PIC_BYTES-i >= 24) j = 6;
                else if (stat&MCSR_IFIFO_AFULL && PIC_BYTES-i >= 16) j = 4;
                else if (stat&MCSR_IFIFO_HALF && PIC_BYTES-i >= 8) j = 2;
                else if (!(stat&MCSR_IFIFO_EMPTY)) j = 1;
                else break;

                for (k = 0; k < j; k++)
		    _RD(REG_FIFO, wtmp[k]);  

	        copy_to_user (((char*)PIC_BUFADDR)+i, (void*) wtmp, j*4);
	        i += (j*4);
            }
	    rvalue = i;
        }
        else if ((PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
            status = EINVAL;
        else {
            _RD (PIC_OFFSET, wtmp[0]);		
            copy_to_user ((char*) PIC_BUFADDR, (void*) wtmp, 4);
        }
        break;

    case IOCTL_WRITE:			/* write a device register */

        if (PIC_OFFSET == REG_FIFO) {
            for (i = 0; i < PIC_BYTES; ) {
                _RD (REG_MCSR, stat);
                if ((stat&MCSR_OFIFO_EMPTY) && PIC_BYTES-i >= 24) j = 6;
                else if (!(stat&MCSR_OFIFO_HALF) && PIC_BYTES-i >= 16) j = 4;
                else if (!(stat&MCSR_OFIFO_AFULL) && PIC_BYTES-i >= 8) j = 2;
                else if (!(stat&MCSR_OFIFO_FULL)) j = 1;
                else break;

	        copy_from_user ((void*) wtmp, ((char*) PIC_BUFADDR)+i, j*4);
                for (k = 0; k < j; k++)
	            _WR(REG_FIFO, wtmp[k]);  
		
	        i += (j*4);
            }
	    rvalue = i;
        }
        else if ((PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
            status = EINVAL;
        else {
            copy_from_user ((void*) wtmp, (void*) PIC_BUFADDR, 4);
            _WR(PIC_OFFSET, wtmp[0]);
        }
        break;

    case IOCTL_ACMD: 			/* command to SHARC processor */

        copy_from_user ((void*) wtmp, (void*) PIC_BUFADDR, 20);
	for (k = 4, j = 16; j >= 0; j -= 4) {
	    _WR(REG_MBX + j, wtmp[k]);
	    k--;
	}

        for (rvalue = 0, j = 0; j < 32; j++) {  
	 /* cant wait forever in kernel mode */
            _RD (REG_MCSR, stat);
            if ((stat & MCSR_IMB1F) != 0x0) {
                _RD (REG_IMB1, wtmp[0]);		
                copy_to_user ((void*) PIC_BUFADDR, (void*) wtmp, 4);
	        rvalue = 4; 
	        break;
            }
            if (j > 4) udelay (100);
        }
        break;

    case IOCTL_MAP:			/* map user buffer to PCI */

	for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr; c++);

     /* find an open dma handler slot */
	if (c >= MAXIOBUF) { 
	    status = EINVAL; 
    	    break; 
        }

     /* Validate user buffer */
        if ((PIC_BUFADDR % 4) != 0) 
	    status = EINVAL;
        else if (PIC_BYTES <= 0 || PIC_BYTES > MAXBCNT || (PIC_BYTES % 4) != 0) 
	    status = EINVAL;

        if (status != ESUCCESS) 
	    break;

	ucb->dma[c].device = dev;
        ucb->dma[c].usradr = (caddr_t) PIC_BUFADDR;
        ucb->dma[c].bytes  = PIC_BYTES;

     /* map buffer into PCI bus space */
        status = icepic_mapio (ucb, IOCTL_MAP, c);
        rvalue = (status == ESUCCESS) ? ucb->dma[c].pciadr : 0;
        break;
    
    case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

     vprintf1 ("icepic_ioctl - UNMAP offset=%08x", PIC_OFFSET);
		
     /* find the matching dma handler slot */
        for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr != PIC_OFFSET; c++);
        if (c >= MAXIOBUF) { 
	    status = EINVAL; 
	    break; 
        }

     /* Unmap user buffer from PCI bus space */
        status = icepic_mapio (ucb, IOCTL_UNMAP, c);
        break;

    case IOCTL_QMAP: 			/* query user buffer map */

        for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr != PIC_OFFSET; c++);
        rvalue = (c < MAXIOBUF) ? ucb->dma[c].bytes : 0;
        break;

    case IOCTL_LIOC:                      /* load IOC words */

        for (i = 0; i < PIC_BYTES; i += 4) {
            k = PIC_BYTES-i; 
	    if (k > 4) k = 4;
            copy_from_user ((void *)(&worda), ((char*)PIC_BUFADDR)+i, k);
#ifndef _LITTLE_ENDIAN
	    reverse (&worda,4);
#endif
            for (j = 0; j < (k * 8); j++) {
        	wordb = (worda >> j) & 0x1;
        	_WR(REG_IOC,wordb);
      	    }
        }
	rvalue = i;
        break;

    case IOCTL_VERSION:                 /* get driver version */

        wtmp[0] = ICE_DRIVER_VERSION;		
        copy_to_user ((char*) PIC_BUFADDR, (void*) wtmp, 4);
	break;

    default: 				/* handle unknown function codes */
        status = EINVAL;
        break;
    }

 /* Copy internal status to user space */
#ifdef _MULTI_DATAMODEL
    switch (ddi_model_convert_from (mode & DATAMODEL_MASK)) {
    case DDI_MODEL_ILP32:
        copy_to_user ((void*) &((PICIOCTL32*)arg)->status, (void*) &rvalue, 4);
	break;
    case DDI_MODEL_NONE:
        copy_to_user ((void*) &((PICIOCTL*)arg)->status, (void*) &rvalue, 4);
	break;
    }
#else
    copy_to_user ((void*) &((PICIOCTL*)arg)->status, (void*) &rvalue, 4);
#endif


    return(status);
}


/************************************/
/* pic cancel routine               */
/************************************/
int icepic_cancel (int index, int reason)
{
    int 	status, c;
    UCB*	ucb = &ucbs[index];

    vprintf1 ("icepic_cancel - dev %x",index);

    if (reason == 1) {	/* close */
        for (c = 0; c < MAXIOBUF; c++) {
            if (ucb->dma[c].pciadr) {
                vprintf1 ("icepic_cancel - stray DMA - resetting card %d", c);

             /* unexpected termination - reset card to avoid stray DMAs */
	        _WR (REG_MCSR, MCSR_RESET);

	     /* deallocate/unmap stray buffer */
	        status = icepic_mapio (ucb, IOCTL_UNMAP, c);
            }
        }
    }
    return(ESUCCESS); 
}

/************************************/
/* pic mapio routine                */
/************************************/
int icepic_mapio (UCB* ucb, int function, int ch)
{
    int status;
    struct buf*		bp;
    uio_t		uio;
    DMACHANNEL*		dmaP;

    status = ESUCCESS;	/* assume success */

    dmaP = &ucb->dma[ch];

    switch (function) {
    case IOCTL_MAP:

	uio.uio_iov = (iovec_t*) kmem_zalloc (sizeof(iovec_t), KM_NOSLEEP);
	uio.uio_iov->iov_base = dmaP->usradr;
	uio.uio_iov->iov_len  = dmaP->bytes;
	uio.uio_iovcnt  = 1;
	uio.uio_offset  = 0;
	uio.uio_loffset = 0;
	uio.uio_segflg  = UIO_USERSPACE;
	uio.uio_fmode   = 0;
	uio.uio_limit   = 0xffffffff;
	uio.uio_llimit  = 0xffffffff;
	uio.uio_resid   = dmaP->bytes;

	vprintf2 ("icepic_mapio - dmaP=%08x, uadr=%08x", dmaP,dmaP->usradr);

	bp = getrbuf (KM_NOSLEEP);

	dmaP->bp  = bp;
	bp->b_private = dmaP;

	status = physio (icepic_strategy, bp, ucb->device, 
				B_READ, icepic_minphys,&uio);
	if (status) vprintf1 ("icepic_mapio - physio error status=%x", status);
        break;

    case IOCTL_UNMAP:
	ddi_dma_unbind_handle (dmaP->handle);
	ddi_dma_free_handle (&dmaP->handle);
	freerbuf (dmaP->bp);
	dmaP->pciadr = 0;
        break;

    default:
        status = EINVAL;
	break;
    }
#if DEBUG
    for (i = 0; i <= MAXIOBUF; i++)
	if (ucb->dma[i].pciadr)
            vprintf3("DMA table i=%d addr=%x bytes=%d \n",i,
		    ucb->dma[i].pciadr,ucb->dma[i].bytes);
#endif
    return (status);
}

static int icepic_strategy (struct buf* bp)
{
    int status;
    int	index = getminor (bp->b_edev);

    UCB*	ucb=&ucbs[index];
    DMACHANNEL*	dmaP;

    vprintf1 ("icepic_strategy - bp=%08x", bp);
    vprintf2 ("icepic_strategy - b_iodone=%08lx, b_flags=%lx", bp->b_iodone,
							       bp->b_flags);

    if (bp->b_flags & B_DONE) return (0);

    dmaP  = (DMACHANNEL*) bp->b_private;

    status = ddi_dma_alloc_handle (ucb->devi, &dmaAttr, 
			       DDI_DMA_DONTWAIT, 0, &dmaP->handle);
    if (status != DDI_SUCCESS) {
	vprintf1 ("icepic_mapio - ddi_dma_alloc_handle status=%d", status);
	bioerror(bp,ENXIO);
	biodone(bp);
	return (0);
    }

 /* Bind buffer to dma handle */
    status = ddi_dma_buf_bind_handle (dmaP->handle, 
				      bp, 
				      DDI_DMA_READ|DDI_DMA_STREAMING,
				      DDI_DMA_DONTWAIT, 0, 
				      &dmaP->cookie[0],
				      &dmaP->ccount);
    if (status != DDI_DMA_MAPPED) {
	vprintf1 ("icepic_strategy - ddi_dma_buf_bind_handle status=%d", status);
	ddi_dma_free_handle (&dmaP->handle);
	bioerror(bp,ENOMEM);
	biodone(bp);
	return (0);
    }

    vprintf5 ("icepic_mapio - dma_cookie %d %08x %08x %08x %08x",
			dmaP->ccount,
			dmaP->cookie[0].dmac_laddress,
			dmaP->cookie[0].dmac_address,
			dmaP->cookie[0].dmac_size,
 			dmaP->cookie[0].dmac_type);

    if (dmaP->ccount != 1) {
	vprintf1 ("icepic_strategy - cookie count=%d", dmaP->ccount);
	bioerror(bp,EIO);
    }
    else {
 /* Set dma address for use by application/board */
        dmaP->pciadr = dmaP->cookie[0].dmac_address;
	bp->b_resid = 0;
    }

    biodone (bp);

    return (0);
}

static void icepic_minphys (struct buf* bp)
{
 /* Nothing to do here */
    return;
}

static void copy_to_user (void *dest, void *src, int bytes)
{
    ddi_copyout (src, dest, bytes, 0);
}

static void copy_from_user (void *dest, void *src, int bytes)
{
    ddi_copyin (src, dest, bytes, 0);
}

static void udelay (int usec)
{
    drv_usecwait (usec);
}

static int currentpid()
{
#ifdef _MULTI_DATAMODEL
    return (ddi_get_pid());
#else
    return (101);
#endif
}

static void reverse (void* pBuf, int pLen) {
  int i,j; 
  uchar_t temp, *tPtr;
  tPtr = (uchar_t *)pBuf;
  j=pLen; pLen >>= 1;
  for (i=0; i<pLen; i++) {
    temp = tPtr[--j]; tPtr[j]=tPtr[i]; tPtr[i]=temp;
  }
}
