/*
 *
 * Driver for the ICEPIC PCI card.
 *
 * Author: Tom Potthast, tom@espycorp.com
 * Module: picdrv.c
 * Creation Date: 2/10/2000
 *
 * $RCSfile:$
 * $Revision: $
 * $Author: $
 * $Date: $
 */

#define ICE_DRIVER_VERSION 338

#define KERNEL
#define INKERNEL

#include <sys/types.h>
#include <sys/debug.h>
#include <sys/param.h>
#include <sys/kmem.h>
#include <sys/ddi.h>
#include <sys/cred.h>
#include <sys/sema.h>
#include <sys/iograph.h>
#include <sys/hwgraph.h>
#include <sys/uio.h>
#include <sys/scsi.h>
#include <sys/edt.h>
#include <sys/cpu.h>
#include <sys/atomic_ops.h>
#include <sys/PCI/bridge.h>
#include <sys/PCI/PCI_defs.h>
#include <sys/PCI/pciio.h>
#include <sys/time.h>
#include <sys/mload.h>
#include <sys/mman.h>
#include <sys/immu.h>
#include <sys/systm.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/xlate.h>
#include <sys/proc.h>
#include <sys/errno.h>
#include <sys/alenlist.h>
#include <sys/pci_intf.h>
#include <sys/cmn_err.h>

#include "picdrv.h"

int icepic_devflag = D_MP;
char *icepic_mversion = M_VERSION;

#define SLEEPPRI ((PZERO+1)|PCATCH)

static int READ_WORD (caddr_t handle, int reg)
{
  /*
  cmn_err (CE_CONT, "READ_WORD: %x %x\n",handle, reg);
  return 0;
  */
  return * (int * ) (handle+reg);
}

void READ_IO(int a, caddr_t b, int c,int d,__uint32_t e) 
{
  int i;
  /*
  cmn_err (CE_CONT, "READIO: %x %x %x %x %x\n", a,b,c,d,e);
  return;
  */
  if (d == 4) { 
    *(int *) e = *(int *)(b+c);
  }else {
    cmn_err (CE_WARN, "Not supported\n");
  }
  /*
  for (i = 0; i < d; ++i) {
    *((caddr_t)e+i) = *(caddr_t)(b+c+i);
  } 
  */
}

static void WRITE_WORD (caddr_t handle, int reg,int wordb)
{
  /*
  cmn_err (CE_CONT, "WRITE_WORD: %x %x %x\n", handle, reg, wordb);
  return;
  */
  *(int *) (handle+reg) = wordb;
}

void WRITE_IO(int a, caddr_t b, int c,int d,__uint32_t e) 
{
  int i;
  
  /*
  cmn_err (CE_CONT, "WRITEIO: %x %x %x %x %x\n", a,b,c,d,e);
  return;
  */
  if (d == 4) { 
    *(int *)(b+c) = *(int *) e;
  }else {
    cmn_err (CE_WARN, "Not supported\n");
  }
  /*
  for (i = 0; i < d; ++i) {
  *(caddr_t)(b+c+i) = *((caddr_t) e+i);
  }
  */
}


/*
#define WRITE_IO(a,b,c,d,e) cmn_err (CE_CONT, "%x %x %x %x %x\n",a,b,c,d,e);
*/

void hex_dump (unsigned char *str, unsigned char *p, long s)
{
  int i;
  uint32_t *iptr = (uint32_t *) p;
  long len = (s+3)/4;

  printf ("%s: Dumping addr: %x, size: %d\n", str, p,s);
  for (i=0;i<len;++i) {
    printf ("%08x ", *iptr++);
    if ((i+1) % 4 == 0) printf ("\n");
  }
  printf ("\n");
}

static int icepic_mapio (vertex_hdl_t vertex, UCB *ucb, int function, int chan)
{
  int npairs;
  DMACHANNEL *dma = &(ucb->dma[chan]);
  alenaddr_t addr;
  size_t alength;
  int dir = B_READ;
  int total = 0;
  caddr_t kvaddr;
  paddr_t paddr;
  unsigned int flags;

  switch (function) {
  case IOCTL_MAP:
    dma->dir = dir;

#ifdef DEBUG
  cmn_err (CE_CONT, "Mapping: 0x%x size: %d\n",
	   dma->usradr, dma->bytes);
#endif 
   
    if (fast_userdma ((void *)dma->usradr, dma->bytes, dma->dir, &(dma->cookie))) {
      cmn_err (CE_WARN, "map_addr: fast_userdma failed " );
      return ENOMEM;
    }

    dma->Ualen = uvaddr_to_alenlist(NULL, dma->usradr, dma->bytes, 0);

    flags = PCIIO_INPLACE|PCIIO_PREFETCH|PCIIO_BYTE_STREAM;

    dma->dma_map = pciio_dmamap_alloc(vertex,0,dma->bytes, flags);
    if (!dma->dma_map) {
      cmn_err (CE_WARN, "map_addr: pciio_dmamap_alloc failed " );
      fast_undma((void *)dma->usradr, dma->bytes,dma->dir, &(dma->cookie));
      return ENOMEM;
    } 
  
    dma->Ualen = pciio_dmamap_list( dma->dma_map, dma->Ualen, flags);
    if (dma->Ualen == NULL) {
      cmn_err (CE_WARN, "map_addr: pciio_dmamap_list" );
      fast_undma((void *)dma->usradr, dma->bytes,dma->dir, &(dma->cookie));
      return ENOMEM;
    }
 
    dma->primed = TRUE;
    alenlist_cursor_init(dma->Ualen, 0, NULL);
    if (alenlist_get(dma->Ualen,NULL,dma->bytes,&addr,&alength,0) != ALENLIST_SUCCESS) {
      cmn_err (CE_WARN, "Cannot get pci addr");
      return ENOMEM;
    }
    dma->pciadr = (unsigned int) addr;

#if 0
    while (total < dma->bytes) {
      if (alenlist_get(dma->Ualen,NULL,dma->bytes,&addr,&alength,0) != ALENLIST_SUCCESS) {
	cmn_err (CE_WARN, "Cannot get pci addr");
	return ENOMEM;
      }

      /* NOT RIGHT */
      dma->pciadr = (unsigned int) addr;
      
#ifdef DEBUG
      cmn_err (CE_CONT, " PCI Mapping: 0x%x sz: %d, pciadr: %x\n", addr, alength, dma->pciadr);
#endif

      total += alength;
    }
   
    dma->bytes = alength;
#endif
    break;

  case IOCTL_UNMAP:
   
    if (dma->primed) {
#ifdef DEBUG
      cmn_err (CE_CONT, "UnMapping: 0x%x size: %d\n",
	       dma->usradr, dma->bytes);
#endif 
      pciio_dmamap_done (dma->dma_map);
      pciio_dmamap_free (dma->dma_map);
      alenlist_done (dma->Ualen);
      fast_undma((void *)dma->usradr, dma->bytes, dma->dir, &(dma->cookie));

      dma->primed = FALSE;
      dma->pciadr = 0;
      dma->bytes = 0;
      dma->pid = 0;
    }
    break;
  }

  return NO_ERROR;
}
void icepic_intr (intr_arg_t intr_arg)
{
}

int icepic_reg ()
{
 int ret;
#ifdef DEBUG
  cmn_err (CE_NOTE, "ICEPIC register");
#endif
 ret = pciio_driver_register (VENDOR_ID,DEVICE_ID,WHOAMI,0);
 return(ret);
}

int icepic_unreg ()
{
#ifdef DEBUG
  cmn_err (CE_NOTE, "ICEPIC unregister");
#endif
  pciio_driver_unregister (WHOAMI);
  return NO_ERROR;
}

static int already_displayed = 0;
#define MAX_PATH_NAME 500

int icepic_attach (vertex_hdl_t vh)
{
 int err;
 int i;
 vertex_hdl_t v_parent;
 vertex_hdl_t v_dev;
 DEVICE_INFO_PTR board = NULL;
 struct device_desc_s ddesc = {0};
 char name [MAX_PATH_NAME];
 pciio_info_t pci_info;
 pciio_slot_t slot;
 char *cfgReg;
 char *path = "ice";
 UCB *ucb;

 pci_info = pciio_info_get(vh);
 slot = pciio_info_slot_get(pci_info);

 if (!already_displayed) {
   cmn_err(CE_NOTE,PREFIX_STRING ": ICEPIC PCI device driver for Origin 2000\n");
   cmn_err(CE_NOTE,PREFIX_STRING ": Version: %s Date: %s\n", VERSION, DATE);
   already_displayed = 1;
 }
 cmn_err(CE_NOTE,PREFIX_STRING ": ICEPIC Board found in PCI slot %d\n", slot);

 bzero(name,MAX_PATH_NAME);
 sprintf (name, "icepic-%1d", slot);

/*
 ----------------------------------------------------------------------
| Add the path or check that it is there.
 ----------------------------------------------------------------------
*/
 err = hwgraph_path_add(
        GRAPH_VERTEX_NONE,      /* Start at /hw */
        path,
        &v_parent);

#ifdef DEBUG
 cmn_err(CE_NOTE, "Adding path %s : %d",path,err);
#endif

/*
 ----------------------------------------------------------------------
| Check for the proper edge or make it.
 ----------------------------------------------------------------------
*/
 
 err = hwgraph_edge_get( v_parent, name, &v_dev);
 if( err != GRAPH_SUCCESS) {
#ifdef DEBUG
  cmn_err(CE_NOTE, "Edge not here, make one");
#endif
   err = hwgraph_char_device_add(
         v_parent,
         name,
        PREFIX_STRING,
        &v_dev);
   if( err) {
        cmn_err (CE_WARN, "char device add(%s) error %d",name,err);
        return(err);
   }
   hwgraph_chmod(v_dev, (mode_t) 0777);

   NEW(board);
   if( board == NULL ) {
     cmn_err (CE_WARN, "cannot alloc space for board.");
     err= ENOMEM;
     return(err);
   }

   board->vertex = v_dev;
   board->parentVertex = vh;

#ifdef SWAP_ENDIAN
   pciio_endian_set(board->parentVertex,
                    PCIDMA_ENDIAN_LITTLE,
                    PCIDMA_ENDIAN_BIG);
#endif
   SetInfo(v_dev,board);

 } else {  /* Already there, must be a reload. */
   cmn_err (CE_NOTE, "found char device %s, vh 0x%x",name,v_dev);
   board = GetInfo(v_dev);
 }
 ucb = &(board->ucb);

 for (i=0; i<MAXIOBUF; i++) {
   ucb->dma[i].dma_map = NULL;
   ucb->dma[i].pciadr = 0;
   ucb->dma[i].bytes = 0;
   ucb->dma[i].pid = 0;
 }
 ucb->attached = 1;


/*
 ----------------------------------------------------------------------
| Map in Memory
 ----------------------------------------------------------------------
*/

  cfgReg = (char *) pciio_piotrans_addr(vh,0,PCIIO_SPACE_CFG,
                               (iopaddr_t)0,PCI_CFG_VEND_SPECIFIC,0);

  board->vendor =  * (short *) &cfgReg[0];
  board->device_id = * (short *) &cfgReg[2]; 
  board->InterruptPin = cfgReg[0x3E];
  board->InterruptLine = cfgReg[0x3F];

#ifdef DEBUG
  cmn_err(CE_NOTE, "Vendor: %x, Device ID: %x", board->vendor, board->device_id);
  cmn_err(CE_NOTE, "InterruptLine: %x, InterruptPin: %x", board->InterruptLine, board->InterruptPin);
#endif


  /* Map in I/O Base Register using Window 0*/
  ucb->addrMap = pciio_piomap_alloc(vh, NULL, PCIIO_SPACE_WIN(0),
                            (iopaddr_t) 0, IO_SPACE_SIZE, IO_SPACE_SIZE, 0);

  if (!ucb->addrMap) {
    cmn_err (CE_WARN, "cannot allocate PCI I/O map");
    err = ENOMEM;
    goto errorReturn;
  }
  ucb->membase  = (caddr_t) pciio_piomap_addr(ucb->addrMap,0, IO_SPACE_SIZE);

  if (!ucb->membase) {
    cmn_err (CE_WARN, "cannot map Base Address");
    err = ENOMEM;
    goto errorReturn;
  }

  
   
#ifdef DEBUG
  cmn_err (CE_CONT, "ICEPIC Base address: %x\n", ucb->membase);
#endif

  return (NO_ERROR);

errorReturn:
  hwgraph_vertex_destroy(v_dev);
  hwgraph_vertex_destroy(v_parent);
  DELETE (board);
  return(err);
}

int icepic_detach (vertex_hdl_t vh)
{
#ifdef DEBUG
  cmn_err (CE_NOTE, "ICEPIC detach");
#endif
 return (NO_ERROR);
}

int icepic_unload ()
{
#ifdef DEBUG
  cmn_err (CE_NOTE, "ICEPIC unload");
#endif
 return (NO_ERROR);
}


int icepic_open (dev_t *devp, int oflag, int otyp, cred_t *crp)
{
  vertex_hdl_t v_dev = dev_to_vhdl(*devp);
  DEVICE_INFO_PTR board = GetInfo(v_dev);
  UCB *ucb;	/* unit control block */

#ifdef DEBUG
  cmn_err (CE_NOTE, "pic_open - dev 0x%x, flag 0x%x\n",devp,oflag);
#endif
  ucb = &(board->ucb);

cmn_err (CE_CONT, "ucb: %x board: %x board->ucb: %x\n",
	 ucb, board, &board->ucb);

  if (ucb->attached == 0) {
    cmn_err (CE_WARN, "pic_open - device unit not attached\n");
    return(ENXIO);
  }
  ucb->opened++;
 
  return(NO_ERROR); 
}

int icepic_close (dev_t dev, int flag, int otyp, cred_t *crp)
{
  vertex_hdl_t v_dev = dev_to_vhdl(dev);
  DEVICE_INFO_PTR board = GetInfo(v_dev);
  UCB *ucb;	/* unit control block */
  int i;
 
#ifdef DEBUG
  cmn_err (CE_NOTE, "pic_close - dev 0x%x, flag 0x%x\n",dev,flag);
#endif

  ucb = &(board->ucb);
  if (ucb->attached == 0) {
    cmn_err(CE_WARN, "pic_close - device unit not attached\n");
    return(EINVAL);
  }

  for (i = 0; i < MAXIOBUF; ++i) {
    icepic_mapio (v_dev, ucb, IOCTL_UNMAP, i);
  }
  /* make sure all DMA's have been terminated */
  /*  icepic_cancel (dev, flag, format, 1); */
  ucb->opened--;
 
  return(NO_ERROR); 

}
/************************************/
/* pic read stub                    */
/************************************/
int icepic_read (dev_t dev, uio_t *uio, cred_t *crp)
{
#ifdef DEBUG
  cmn_err  (CE_NOTE, "pic_read - dev 0x%x",dev);
#endif
  return (NO_ERROR);
}

/************************************/
/* pic write stub                   */
/************************************/
int icepic_write (dev_t dev, uio_t *uiop, cred_t *crp)
{
#ifdef DEBUG
  cmn_err(CE_NOTE, "pic_write - dev 0x%x",dev);
#endif
  return (NO_ERROR);
}



/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_ADPADDR	0		/* user buffer addr */
#define PIC_IOHANDLE	ucb->membase	/* PCI controller base address */
#define PIC_BUFADDR	p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_STATUS	p->status	/* return status */

int icepic_ioctl (dev_t dev, int cmd, void *arg, int mode, cred_t *crp, int *rvalp)
{
  vertex_hdl_t v_dev = dev_to_vhdl(dev);
  DEVICE_INFO_PTR board = GetInfo(v_dev);
  UCB 		*ucb;		/* unit control block */
  int		i,j,k;		/* Local counters */
  int		status,stat;	/* Local function return status */
  int		cur_pri;	/* Holds the current process IPL */
  int		chan;		/* Current dma channel */
  PICIOCTL      data;           /* Copy of arg */
  PICIOCTL	*p;		/* pic ioctl handle structure */
  struct task 	*tsk;		/* current task pointer */
  int		usermap;	/* whether to map user space */
  unsigned char *bdata;		/* byte data reference */
  int		worda,wordb;	/* load IOC temps */
  int rc;
/*
  cmn_err (CE_NOTE, "pic_ioctl - dev 0x%x, cmd 0x%x, arg 0x%x",
	   dev,cmd,arg);
*/

  ucb = &(board->ucb);		/* ucb for this controller */

  /*
  hex_dump ("arg", (unsigned char *) arg, sizeof(PICIOCTL));
  */
  rc = copyin ((caddr_t) arg, (caddr_t) &data, sizeof (PICIOCTL));
  if (rc == -1) {
    cmn_err (CE_WARN, "copyin failed");
    return EIO;
  }
  
  p = (PICIOCTL *) &data;		/* ioctl data = the PICIOCTL */

  i = PIC_BYTES;			/* normal|default completion status */
  status = NO_ERROR; 			/* assume success */

  if (PIC_FUNCTION==IOCTL_MAP || PIC_FUNCTION==IOCTL_UNMAP || PIC_BYTES ==0) 
    usermap = 0;
  else 
    usermap = 1;

  if (usermap) {			/* Map user buffer */
#if 0
    tsk = current_task();	
    if (vm_map_pageable (tsk->map, (vm_offset_t)PIC_BUFADDR, 
	(vm_offset_t)(PIC_BUFADDR+PIC_BYTES), VM_PROT_READ|VM_PROT_WRITE)) {
      vprintf("Unable to lock user buffer in memory\n");
      status = EINVAL; p->status = i; return (status);
    }
#endif
  }

  switch (p->function) {

  case IOCTL_READ:			/* read a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {

	stat = READ_WORD (PIC_IOHANDLE, REG_MCSR);
       
             if ((stat&MCSR_IFIFO_FULL) && (PIC_BYTES-i>=24)) j=6;
        else if ((stat&MCSR_IFIFO_AFULL) && (PIC_BYTES-i>=16)) j=4;
        else if ((stat&MCSR_IFIFO_HALF) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_IFIFO_EMPTY)) j=1;
        else break;
        while (j>0) { 
	  READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_FIFO, 4, ( PIC_BUFADDR+i) );  
    	  i+=4; j--;
        }
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) {	/* invalid register ? */
      cmn_err (CE_WARN, "Invalid register (IOCTL_READ): %x", PIC_OFFSET);
      status = EINVAL;
    } else 
      READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			PIC_OFFSET, PIC_BYTES,PIC_BUFADDR );		
    break;

  case IOCTL_WRITE:			/* write a device register */

    if (PIC_OFFSET == REG_FIFO) {
      for (i=0; i<PIC_BYTES; ) {
	stat = READ_WORD (PIC_IOHANDLE, REG_MCSR);
  
             if ((stat&MCSR_OFIFO_EMPTY) && (PIC_BYTES-i>=24)) j=6;
        else if (!(stat&MCSR_OFIFO_HALF) && (PIC_BYTES-i>=16)) j=4;
        else if (!(stat&MCSR_OFIFO_AFULL) && (PIC_BYTES-i>=8)) j=2;
        else if (!(stat&MCSR_OFIFO_FULL)) j=1;
        else break;
        while (j>0) {
	  WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE,
                        REG_FIFO, 4, (PIC_BUFADDR+i) );
	  i+=4; j--;
        }
      }
    }
    else if ( (PIC_OFFSET&0XFFFFF000) != 0) { 	/* invalid register ? */
      cmn_err (CE_WARN, "Invalid register (IOCTL_WRITE): %x", PIC_OFFSET);
      status = EINVAL;
    } else 
      WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			PIC_OFFSET, PIC_BYTES, PIC_BUFADDR );		
    break;

  case IOCTL_ACMD: 			/* command to SHARC processor */

    for (j=16; j>=0; j-=4) 
      WRITE_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_MBX+j, 4, (PIC_BUFADDR+j));
    for (i=0,j=0; j<32; j++) {    	/* cant wait forever in kernel mode */
      stat = READ_WORD (PIC_IOHANDLE, REG_MCSR);
    
      if ( (stat&MCSR_IMB1F) != 0x0 ) {
        READ_IO (PIC_ADPADDR, PIC_IOHANDLE, 
			REG_IMB1, 4, PIC_BUFADDR );
	i=4; break;
      }
    }
    break;

  case IOCTL_MAP:			/* map user buffer to PCI */
    cmn_err (CE_NOTE, "IOCTL_MAP");
    /* find an open dma handler slot */
    for (chan=0; chan<MAXIOBUF && ucb->dma[chan].pciadr!=0; chan++);
    if (chan>=MAXIOBUF) { 
      cmn_err (CE_WARN, "MAXIOBUF");
      status = EINVAL;
      break; 
    }

    cmn_err (CE_CONT, "pic bufaddr: %x\n",  PIC_BUFADDR);
    /* Validate user buffer */
    if ( ((unsigned long) PIC_BUFADDR % 4) != 0 ){
       cmn_err (CE_WARN, "PIC_BUFADDR % 4");
       status = EINVAL;
    }
    if ( (PIC_BYTES <= 0)  || (PIC_BYTES > MAXBCNT) ||
        ((PIC_BYTES % 4) != 0) ) {
      cmn_err (CE_WARN, "PIC_BYTES: %d\n", PIC_BYTES);
      status = EINVAL;
    }
    if (status!=NO_ERROR) break;

    /* map user buffer into PCI bus space */
    ucb->dma[chan].usradr = (uvaddr_t) PIC_BUFADDR;
    ucb->dma[chan].bytes  = PIC_BYTES;
    status = icepic_mapio ( board->parentVertex, ucb, IOCTL_MAP, chan);
 
    i = ucb->dma[chan].pciadr;
    break;
    
  case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

    /* find the matching dma handler slot */
    for (chan=0; chan<MAXIOBUF && ucb->dma[chan].pciadr!=PIC_OFFSET; chan++);
    if (chan>=MAXIOBUF) { status = EINVAL; break; }

    /* unmap user buffer from PCI bus space */
    status = icepic_mapio (v_dev, ucb, IOCTL_UNMAP, chan);
    break;


  case IOCTL_QMAP: 			/* query user buffer map */
    for (i=0,chan=0; chan<MAXIOBUF && i==0; chan++) {
      if (PIC_OFFSET >= (ucb->dma[chan].pciadr) &&
	  PIC_OFFSET <= (ucb->dma[chan].pciadr+ucb->dma[chan].bytes))
	i = ucb->dma[chan].bytes;
    }
    break;
    

  case IOCTL_LIOC:                      /* load IOC words */

    bdata = (unsigned char *)PIC_BUFADDR;
    for (k=0; k<PIC_BYTES; k++) {
      worda = (int)bdata[k];
      for (j=0; j<8; j++) {
        wordb = worda>>j;
	WRITE_WORD (PIC_IOHANDLE, REG_IOC, wordb);
       
      }
    }
    break;

  case IOCTL_VERSION:                   /* get driver version */

    *(int*)PIC_BUFADDR = ICE_DRIVER_VERSION;
    break;

  default: 				/* handle unknown function codes */
    cmn_err (CE_CONT, "IOCTL: Invalid function code: %d", cmd);
    status = EINVAL;
    break;
  }

  if (usermap) {			/* Unmap user buffer */
#if 0
    if (vm_map_pageable (tsk->map, (vm_offset_t)PIC_BUFADDR, 
	(vm_offset_t)(PIC_BUFADDR+PIC_BYTES), VM_PROT_NONE)) {
      vprintf("Unable to unlock user buffer in memory\n");
    }
#endif
  }

  /* fill in IO status block and signal completion */
  p->status = i;

  copyout ((void *) &data, arg, sizeof (PICIOCTL));
  return(status);
}
