/****************************************************************/
/*  ICE-PIC Device Driver for HP-UX                             */
/*                                                              */
/*  Author:  Jeff Schoen - Innovative Computer Engineering      */
/*                                                              */
/****************************************************************/

#define ICE_DRIVER_VERSION 338

#define PCI_LITTLE_ENDIAN_ONLY

/*****************/
/* Include Files */
/*****************/
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/malloc.h>
#include <mod_conf.h>
#include <sys/moddefs.h>
#include <sys/io.h>
#include <sys/ioctl.h>
#include <sys/wsio.h>
#include <sys/mman.h>
#include <sys/pci.h>
#include <sys/user.h>

#define _UNIX 1
#define _HPUX 1
#define _IEEE 1

#include "iceioctl.h"
#include "iceflex.h"

/*****************/
/* Entry Points */
/*****************/
int icepic_install(void);
int icepic_load(void *drv_infop);
int icepic_unload(void *drv_infop);
int icepic_open(dev_t dev, int flags, intptr_t dummy, int mode);
int icepic_close(dev_t dev, int flags, int mode);
int icepic_ioctl(dev_t dev, int cmd, PICIOCTL *p, int flags);
int icepic_attach (uint32_t idparm, struct isc_table_type *isc);
int icepic_linked_attach (uint32_t idparm, struct isc_table_type *isc);

/*****************/
/* Wrapper Table */
/*****************/
extern struct mod_operations gio_mod_ops;
drv_info_t icepic_drv_info;
extern struct mod_conf_data icepic_conf_data;
/* module type specific data */
struct mod_type_data icepic_drv_link = {
  "icepic - Loadable/Unloadable ICEPIC Module",
  (void *)NULL
};
struct modlink icepic_mod_link[] = {
  { &gio_mod_ops, (void *)&icepic_drv_link }, /* WSIO */
  { NULL, (void *)NULL }
};
struct modwrapper icepic_wrapper = {
  MODREV,
  icepic_load,
  icepic_unload,
  (void (*)())NULL,
  (void *)&icepic_conf_data,
  icepic_mod_link
};

/*****************/
/* Driver Header */
/*****************/
drv_info_t icepic_drv_info = {
  "icepic", /* type */
  "pseudo", /* class */
  DRV_CHAR|DRV_PSEUDO|DRV_SAVE_CONF|DRV_MP_SAFE,/* flags */
  -1, /* b_major */
  -1, /* c_major */
  NULL, /* cdio */
  NULL, /* gio_private */
  NULL, /* cdio_private */
};
drv_ops_t icepic_drv_ops = {
  icepic_open, /* d_open */
  icepic_close, /* d_close */
  NULL, /* d_strategy */
  NULL, /* d_dump */
  NULL, /* d_psize */
  NULL, /* d_mount */
  NULL, /* d_read */
  NULL, /* d_write */
  icepic_ioctl, /* d_ioctl */
  NULL, /* d_select */
  NULL, /* d_option1 */
  NULL, /* reserved1 */
  NULL, /* reserved2 */
  NULL, /* reserved3 */
  NULL, /* reserved4 */
  0 /* d_flags */
};
wsio_drv_data_t icepic_wsio_data = {
  "pseudo_icepic", /* drv_path */
  T_DEVICE, /* drv_type */
  DRV_CONVERGED, /* drv_flags */
  NULL, /* dvr_minor_build - field not used */
  NULL, /* drv_minor_decode - field not used*/
};
wsio_drv_info_t icepic_wsio_info = {
  &icepic_drv_info,
  &icepic_drv_ops,
  &icepic_wsio_data,
};
int (*icepic_saved_attach)();

/**********************************/
/* Function Prototype Definitions */
/**********************************/

/***********************/
/* General Definitions */
/***********************/
#define ESUCCESS 0
#define	SUCCESS(a)	(((a) & 1) == 1)
#define	FAILURE(a)	(((a) & 1) == 0)

#define vprintf(A) 		if (icepic_verbose==1) printf(A)
#define vprintf1(A,B) 		if (icepic_verbose==1) printf(A,B)
#define vprintf2(A,B,C) 	if (icepic_verbose==1) printf(A,B,C)
#define vprintf3(A,B,C,D) 	if (icepic_verbose==1) printf(A,B,C,D)
#define vprintf4(A,B,C,D,E) 	if (icepic_verbose==1) printf(A,B,C,D,E)
#define vprintf5(A,B,C,D,E,F) 	if (icepic_verbose==1) printf(A,B,C,D,E,F)

#define PRINTD(S) if (icepic_debug) printf (S)

/*******************************/
/* Driver Specific Definitions */
/*******************************/

#define MAXPIC 16
#define MAXMAP 4096
#define MAXIOBUF 512
#define MAXBCNT (1024*1024*1024)

#define VENDOR_ID 0x1172
#define DEVICE_ID 0x7777

typedef struct {
    dev_t		device;
    caddr_t 		usradr;
    int			bytes;
    uint64_t 		pciadr;
    struct buf*		bp;
    uint64_t		ccount;
    void*               dma_handle;
    wsio_dma_map_t      dma_map;
} DMACHANNEL;

typedef struct {
  int		   attached;
  int		   opened;
  char* 	   loc;
  char*		   cfg;
  dev_t	 	   device;
  long 		   memsize;
  struct isc_table_type *isc;
  DMACHANNEL	   dma[MAXIOBUF];
} UCB;

UCB ucbs[MAXPIC];
extern int icepic_verbose;
extern int icepic_debug;
extern int icepic_no_unload;

/***********************************/
/* Device Driver Macro Definitions */
/***********************************/
uint32_t pci_get32 (struct isc_table_type *isc, uint32_t addr) {
  uint32_t data = 0;

  READ_REG_UINT32_ISC (isc, isc->if_reg_ptr + addr, &data);
  /*
  vprintf2 ("icepic> read 0x%08x from offset 0x%08x\n", data, addr);
  */
  return (data);
}

void pci_put32 (struct isc_table_type *isc,
		uint32_t addr, uint32_t datain) {
  uint32_t data = datain;
  /*
  vprintf2 ("icepic> writing 0x%08x to offset 0x%08x\n", data, addr);
  */
  WRITE_REG_UINT32_ISC (isc, isc->if_reg_ptr + addr, &data);

}

#define _RD(A,B)  { \
    B = (int)pci_get32 (ucb->isc, (uint32_t)(A)); }

#define _WR(A,B)  { \
    pci_put32 (ucb->isc, (uint32_t)(A), B); }

int icepic_mapio (UCB*, int, int);

int currentpid (void);
void copy_to_user (void*, void*, int);
void copy_from_user (void*, void*, int);
void udelay (int);
void reverse (void *, int);

/***********************************************/
/* Loadable Driver Configuration Definitions   */
/***********************************************/
int icepic_load(void *arg)
{
  int ret;

  PRINTD ("icepic> Loading\n");

  /* Use drv_info passed to us instead of static version */
  icepic_wsio_info.drv_info = (drv_info_t *) arg;

  /* Register with WSIO */
  ret = wsio_install_driver (&icepic_wsio_info);
  if (ret) {
    /*
     * The install worked; link into the pci_attach chain
     */

    if (mod_wsio_attach_list_add (MOD_WSIO_PCI, icepic_attach)) {
      printf ("icepic> Failed to link to attach chain\n");
    }

  }
  else {
    printf("icepic> wsio_install_driver failed!!\n");
    return (ENXIO);
  }

  return (0);
}
/* This function is only called when the administrator
* attempts to unload the module and there are no open
* devices using the module. If there is some reason that
* the module should not be unloaded, check it now and
* return non-zero.
*/
int icepic_unload(void *drv_infop)
{

  if (icepic_no_unload) {
    printf("icepic> Im BUSY\n");
    return (EBUSY);
  }

  /* Remove my attach from the list */
  mod_wsio_attach_list_remove (MOD_WSIO_PCI, icepic_attach);

  /* Unregister with WSIO */
  if ( wsio_uninstall_driver(&icepic_wsio_info) ) {
    /* Uninstall failed! Return to a loaded, functional state. */
    printf("icepic> wsio_uninstall_driver failed!!\n");
    return (ENXIO);
  }
  /* Cancel pending timeouts, free allocated memory and resources, etc. */
  PRINTD("icepic> Unloaded\n");
  return (0);
}
/*
* INSTALL
* This function is called if module is statically
* linked.
*/
int icepic_install(void)
{
  int ret;

  vprintf ("icepic> icepic_install()\n");

  ret = wsio_install_driver (&icepic_wsio_info);
  if (ret) {
    /*
     * The install worked; link into the pci_attach chain
     */
    icepic_saved_attach = pci_attach;
    pci_attach = icepic_linked_attach;
  }

  return (ret);
}

int icepic_linked_attach (uint32_t idparm, struct isc_table_type *isc)
{
  icepic_attach (idparm, isc);
  return (icepic_saved_attach (idparm, isc));
}

int icepic_attach (uint32_t idparm, struct isc_table_type *isc) 
{
    int 	index,status;
    char 	name[20];
    UCB* 	ucb;
    caddr_t 	regptr = 0;
    uint32_t    bar0;
    uint32_t    bar1;
    uint32_t    bar0size;

    PCI_ID *id = (PCI_ID *)&idparm;
  
    /*
     * Make sure this device is ours since we'll be called for all of them
     */
    if ((id->vendor_id != VENDOR_ID) || (id->device_id != DEVICE_ID))
        return 1;

    index         = 0;
    ucb           = &ucbs[index];
    ucb->attached = 1;

    ucb->isc = isc;
    isc_claim(isc,&icepic_wsio_info);

    pci_read_cfg_uint32_isc (isc, 0x10, &bar0);
    pci_read_cfg_uint32_isc (isc, 0x14, &bar1);

    /*
     * If the O/S didn't map the first memory region into virtual memory,
     * figure out how big it is and map it manually.  The O/S will only
     * map it automatically if the region is 8kb or less.
     */

    if (!isc->if_reg_ptr) {
      /*
       * See if we're using I/O space or memory space to make the
       * appropriate mapping calls.
       */
      if (bar0 & 1) {
	printf ("icepic> Using I/O space - ugh\n");
      }
      else {
	/*
	 * Get the size of the first memory region.
	 */
        pci_write_cfg_uint32_isc (isc, 0x10, 0xffffffff);
	pci_read_cfg_uint32_isc (isc, 0x10, &bar0size);
	pci_write_cfg_uint32_isc (isc, 0x10, bar0);

	bar0size &= 0xfffffff0;
	bar0size ^= 0xffffffff;
        bar0size += 1;

	regptr = bar0;

	/*
	 * If this is 64 bit space we need to use BAR1 for the upper half
	 * of the address.  I'm assuming at this point there's not enough
	 * register space used to take up more than the 4 Gb of space BAR0
	 * can specify.  Also, I don't like this.  regptr has to be of type
	 * caddr_t for the call to map_mem_to_host(), but what if we're on
	 * a 32 bit machine?  This will work fine for the 64 bit boxes, but
	 * will need to be kludged a little for 32 bit machines.
	 */
	if (bar0 & 7) {
	  regptr += (bar1 << 32);
	}

	isc->if_reg_ptr = map_mem_to_host (isc, regptr, bar0size);

	if (!isc->if_reg_ptr) {
	  printf ("icepic> Failed to map bus memory to host\n");
	}

      }
    }

    vprintf2 ("icepic> claimed device at 0x%016lx mapped at 0x%016lx\n",
	     regptr, isc->if_reg_ptr);

    return ESUCCESS;
}


/************************************/
/* pic open routine                 */
/************************************/
int icepic_open (dev_t dev, int flags, intptr_t dummy, int mode)
{
  int index;
  UCB *ucb;

  index = minor(dev);
  vprintf1 ("icepic_open - dev %x\n",index);

  ucb = &ucbs[index];
  if (ucb->attached == 0) {
    vprintf("icepic_open - device unit not attached");
    return(-ENXIO);
  }
  ucb->opened++;

  return(ESUCCESS); 
}
 
/************************************/
/* pic close routine                */
/************************************/
int icepic_close (dev_t dev, int flags, int mode)
{
    int index = minor(dev);
    UCB *ucb = &ucbs[index];;
    vprintf1 ("icepic_close - dev %d\n",index);
    if (ucb->attached == 0) {
        vprintf("icepic_close - device unit not attached");
    }

 /* make sure all DMA's from this PID have been terminated */
    icepic_cancel (index, 1);
    ucb->opened--;
    return 0;
}
    
/************************************/
/* pic ioctl routine                */
/************************************/
#define PIC_BUFADDR	(long) p->bufaddr	/* user buffer addr */
#define PIC_BYTES	p->bytes	/* bytes to transfer */
#define PIC_FUNCTION	p->function	/* user function code */
#define PIC_OFFSET	p->offset	/* register offset */
#define PIC_STATUS	p->status	/* return status */

int icepic_ioctl (dev_t dev, int cmd, PICIOCTL *p, int flags) 
{
    int 	index = minor(dev);
    UCB*  	ucb = &ucbs[index];

    int		i,j,k;		/* Local counters */
    int		status,stat;	/* Local function return status */
    int		c;		/* Current dma channel */
    uint32_t	rvalue;
    int		wtmp[6];	/* kernel space temp data */
    int		worda,wordb;	/* load IOC temps */

    ucb->device = dev;

    vprintf3("icepic_ioctl - dev %x, cmd %x, arg %x\n", index,cmd,p);

    rvalue = PIC_BYTES;			/* normal|default completion status */
    status = ESUCCESS; 			/* assume success */

    switch (PIC_FUNCTION) {
    case IOCTL_READ:			/* read a device register */
	if (PIC_OFFSET == REG_FIFO) {
            for (i = 0; i < PIC_BYTES; ) {
                _RD (REG_MCSR, stat);
                if (stat&MCSR_IFIFO_FULL && PIC_BYTES-i >= 24) j = 6;
                else if (stat&MCSR_IFIFO_AFULL && PIC_BYTES-i >= 16) j = 4;
                else if (stat&MCSR_IFIFO_HALF && PIC_BYTES-i >= 8) j = 2;
                else if (!(stat&MCSR_IFIFO_EMPTY)) j = 1;
                else break;

                for (k = 0; k < j; k++)
		    _RD(REG_FIFO, wtmp[k]);  

	        copy_to_user (((char*)PIC_BUFADDR)+i, (void*) wtmp, j*4);
	        i += (j*4);
            }
	    rvalue = i;
        }
        else if ((PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
            status = EINVAL;
        else {
            _RD (PIC_OFFSET, wtmp[0]);
	    copy_to_user ((char*) PIC_BUFADDR, (void*) wtmp, 4);
        }
        break;

    case IOCTL_WRITE:			/* write a device register */

        if (PIC_OFFSET == REG_FIFO) {
            for (i = 0; i < PIC_BYTES; ) {
                _RD (REG_MCSR, stat);
                if ((stat&MCSR_OFIFO_EMPTY) && PIC_BYTES-i >= 24) j = 6;
                else if (!(stat&MCSR_OFIFO_HALF) && PIC_BYTES-i >= 16) j = 4;
                else if (!(stat&MCSR_OFIFO_AFULL) && PIC_BYTES-i >= 8) j = 2;
                else if (!(stat&MCSR_OFIFO_FULL)) j = 1;
                else break;

	        copy_from_user ((void*) wtmp, ((char*) PIC_BUFADDR)+i, j*4);
                for (k = 0; k < j; k++)
	            _WR(REG_FIFO, wtmp[k]);  
		
	        i += (j*4);
            }
	    rvalue = i;
        }
        else if ((PIC_OFFSET&0XFFFFF000) != 0) 	/* invalid register ? */
            status = EINVAL;
        else {
            copy_from_user ((void*) wtmp, (void*) PIC_BUFADDR, 4);
            _WR(PIC_OFFSET, wtmp[0]);
        }
        break;

    case IOCTL_ACMD: 			/* command to SHARC processor */

        copy_from_user ((void*) wtmp, (void*) PIC_BUFADDR, 20);
	for (k = 4, j = 16; j >= 0; j -= 4) {
	    _WR(REG_MBX + j, wtmp[k]);
	    k--;
	}

        for (rvalue = 0, j = 0; j < 32; j++) {  
	 /* cant wait forever in kernel mode */
            _RD (REG_MCSR, stat);
            if ((stat & MCSR_IMB1F) != 0x0) {
                _RD (REG_IMB1, wtmp[0]);		
                copy_to_user ((void*) PIC_BUFADDR, (void*) wtmp, 4);
	        rvalue = 4; 
	        break;
            }
            if (j > 4) udelay (100);
        }
        break;

    case IOCTL_MAP:			/* map user buffer to PCI */

	for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr; c++);

     /* find an open dma handler slot */
	if (c >= MAXIOBUF) { 
	    status = EINVAL; 
    	    break; 
        }

     /* Validate user buffer */
        if ((PIC_BUFADDR % 4) != 0) 
	    status = EINVAL;
        else if (PIC_BYTES <= 0 || PIC_BYTES > MAXBCNT || (PIC_BYTES % 4) != 0) 
	    status = EINVAL;

        if (status != ESUCCESS) 
	    break;

	ucb->dma[c].device = dev;
        ucb->dma[c].usradr = (caddr_t) PIC_BUFADDR;
        ucb->dma[c].bytes  = PIC_BYTES;

     /* map buffer into PCI bus space */
        status = icepic_mapio (ucb, IOCTL_MAP, c);
        rvalue = (status == ESUCCESS) ? ucb->dma[c].pciadr : 0;
        break;
    
    case IOCTL_UNMAP: 			/* unmap user buffer from PCI */

     vprintf1 ("icepic_ioctl - UNMAP offset=%08x", PIC_OFFSET);
		
     /* find the matching dma handler slot */
        for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr != PIC_OFFSET; c++);
        if (c >= MAXIOBUF) { 
	    status = EINVAL; 
	    break; 
        }

     /* Unmap user buffer from PCI bus space */
        status = icepic_mapio (ucb, IOCTL_UNMAP, c);
        break;

    case IOCTL_QMAP: 			/* query user buffer map */

        for (c = 0; c < MAXIOBUF && ucb->dma[c].pciadr != PIC_OFFSET; c++);
        rvalue = (c < MAXIOBUF) ? ucb->dma[c].bytes : 0;
        break;

    case IOCTL_LIOC:                      /* load IOC words */

        for (i = 0; i < PIC_BYTES; i += 4) {
            k = PIC_BYTES-i; 
	    if (k > 4) k = 4;
            copy_from_user ((void *)(&worda), ((char*)PIC_BUFADDR)+i, k);
#ifndef _LITTLE_ENDIAN
	    reverse (&worda,4);
#endif
            for (j = 0; j < (k * 8); j++) {
        	wordb = (worda >> j) & 0x1;
        	_WR(REG_IOC,wordb);
      	    }
        }
	rvalue = i;
        break;

    case IOCTL_VERSION:                 /* get driver version */
	*(int*)PIC_BUFADDR = ICE_DRIVER_VERSION;
	break;

    default: 				/* handle unknown function codes */
        status = EINVAL;
        break;
    }

 /* Copy internal status to user space */
    PIC_STATUS = rvalue;

    return(status);
}


/************************************/
/* pic cancel routine               */
/************************************/
int icepic_cancel (int index, int reason)
{
    int 	status, c;
    UCB*	ucb = &ucbs[index];

    vprintf1 ("icepic_cancel - dev %x",index);

    if (reason == 1) {	/* close */
        for (c = 0; c < MAXIOBUF; c++) {
            if (ucb->dma[c].pciadr) {
                vprintf1 ("icepic_cancel - stray DMA - resetting card %d", c);

             /* unexpected termination - reset card to avoid stray DMAs */
	        _WR (REG_MCSR, MCSR_RESET);

	     /* deallocate/unmap stray buffer */
	        status = icepic_mapio (ucb, IOCTL_UNMAP, c);
            }
        }
    }
    return(ESUCCESS); 
}

/************************************/
/* pic mapio routine                */
/************************************/
int icepic_mapio (UCB* ucb, int function, int ch)
{
    int status;
    DMACHANNEL*		dmaP;
    wsio_dma_map_t      host_range;
    wsio_map_status_t   map_status;
    space_t             region_id;

    status = ESUCCESS;	/* assume success */

    dmaP = &ucb->dma[ch];

    switch (function) {
    case IOCTL_MAP:
      /*
       * device, usradr and bytes in ucb->dma[ch] set on entry
       */

	/*
	 * First get a DMA handle
	 */

	dmaP->dma_handle = wsio_allocate_dma_handle (ucb->isc);

	if (dmaP->dma_handle) {
	  
	  /*
	   * Set up the address range we want to map, and tell the O/S
	   * the range in in user space (as well as the address range's space_id).
	   * Eventually we should get rid of WSIO_DMA_CONTIGUOUS and check
	   * the return value to see if the entire range was mapped.  For now
	   * this should be sufficient.
	   *
	   */

	  host_range.iov_base = dmaP->usradr;
	  host_range.iov_len  = dmaP->bytes;

	  region_id = _MOV_FROM_RR ((dmaP->usradr));
	  map_status = wsio_map_dma_buffer (ucb->isc, dmaP->dma_handle,
					    NULL,
					    WSIO_DMA_CONTIGUOUS,
					    region_id,
					    &host_range,
					    &(dmaP->dma_map));

	  if (WSIO_MAP_OK != map_status) {
	    printf ("icepic_mapio> map buffer failed\n");
	    status = EFAULT;
	  }
	  else {
	    dmaP->pciadr = dmaP->dma_map.iov_base;
	    printf ("icepic> mapped 0x%016lx len %d to 0x%08x len %d\n",
		    dmaP->usradr, dmaP->bytes,
		    dmaP->dma_map.iov_base, dmaP->dma_map.iov_len);
	  }
	}
	else {
	  printf ("icepic_mapio> allocate dma handle failed\n");
	  status = EFAULT;
	}

        break;

    case IOCTL_UNMAP:
	dmaP->pciadr = 0;
	map_status = wsio_unmap_dma_buffer (ucb->isc, dmaP->dma_handle,
					    &(dmaP->dma_map));
	map_status = WSIO_MAP_OK;
	if (WSIO_MAP_OK == map_status) {

	  dmaP->dma_map.iov_base = 0;
	  dmaP->dma_map.iov_len  = 0;
	  dmaP->pciadr = 0;

	  wsio_free_dma_handle (ucb->isc, dmaP->dma_handle);

	  dmaP->dma_handle = NULL;

	}
	else {
	  printf ("icepic_mapio> couldn't unmap DMA!\n");
	  status = EFAULT;
	}

        break;

    default:
        status = EINVAL;
	break;
    }
    return (status);
}

void copy_to_user (void *dest, void *src, int bytes)
{
  int status;

  status = copyout (src, dest, bytes);
  if (status)
    printf ("icepic> copyout 0x%016lx -> 0x%016lx %d failed\n",
	    src, dest, bytes);
}

void copy_from_user (void *dest, void *src, int bytes)
{
  int status;

  status = copyin (src, dest, bytes);
  if (status)
    printf ("icepic> copyin 0x%016lx -> 0x%016lx %d failed\n",
	    src, dest, bytes);
}

void udelay (int usec)
{
    busywait (usec);
}

int currentpid()
{
    return (101);
}

void reverse (void* pBuf, int pLen) {
  int i,j; 
  char temp, *tPtr;
  tPtr = (char *)pBuf;
  j=pLen; pLen >>= 1;
  for (i=0; i<pLen; i++) {
    temp = tPtr[--j]; tPtr[j]=tPtr[i]; tPtr[i]=temp;
  }
}
