/*
 * This interfaces a Rumel IO dataflow to an Ice Core.
*/
`timescale 1 ps / 1 ps

`include "mdefs.h"

module usercode #(parameter PORT=1, parameter BW=64) (

  input  wire	reset_n,	//    global reset, active low
  input  wire	clk,		//    global system clock, used for data and configuration
  input  wire	fastclk,	//    fast clock for user logic

  input  wire	     cfg_wr,	//    configuration write enable, active high, config data valid should be latched when asserted
  input  wire [15:0] cfgwradr,	//    configuration local write address, used for write of configuration data
  input  wire [31:0] cfgwrdat,	//    configuration write data from C user interface, setkey() command

  input  wire        cfg_rd,	//    configuration read  enable, active high
  input  wire [15:0] cfgrdadr,	//    configuration local read address, used for read of configuration data
  output wire [31:0] cfgrddat,	//    configuration read data to be valid no more than 4 clocks after cfg_rd asserted

  input  wire [BW-1:0] din,	//    data input, from Sonet Rx Logic
  input  wire        dinvalid,	//    data input valid, din[63:0] should be latched when this signal high
		
  output wire [BW-1:0] dout,	//    data output to ICEPIC card
  output wire        doutvalid,	//    data output valid, asserted every clock cycle dout is valid, active high
  input  wire        doutavail,	//    data output space available for data, active high
		
  output wire [7:0]  test);

  wire sack;
  reg srst,scs,swr,srd;
  reg [31:0] swrbus;
  reg [31:0] saddr;
  wire [31:0] srdbus;

  wire istat,ostat;
  wire [7:0] isel,osel;
  wire iena;
  reg [BW-1:0] ibus;
  wire [BW-1:0] obus;
  reg oena,oval;

  wire ioclk = clk;
  wire sclk  = fastclk;

  reg [7:0] rstn,rstm;
  reg crst,drst,cfg_cyc,cfg_rw;
  wire cfg_row = cfg_rd || cfg_wr;
  always @(posedge clk) begin
//    rstn <= {rstn,reset_n};	// pick up external and create internal initial reset
    rstn <= {rstn,1'b1};	// create internal initial reset
    crst <= (rstn!=8'hFF);
    if (crst) cfg_cyc <= 0; else if (cfg_row) cfg_cyc <= !cfg_cyc;
    if (cfg_row) cfg_rw <= cfg_rd;
    if (cfg_row) saddr  <= cfg_rd? cfgrdadr : cfgwradr;
    if (cfg_row) swrbus <= cfgwrdat;
  end

  reg [1:0] s_dif;
  reg s_cyc,s_row;
  always @(posedge sclk) begin
    rstm <= {rstm,1'b1};	// create internal initial reset
    drst <= (rstm!=8'hFF);
    srst <= crst || drst;
    if (srst) s_cyc <= 0; else if (s_row) s_cyc <= !s_cyc;
    if (srst) s_dif <= 0; else s_dif <= {s_dif,(s_cyc^cfg_cyc)&&!s_row};// s_cyc^cfg_cyc has meta-instability
    if (srst) s_row <= 0; else s_row <= (s_dif==3) && !s_row;		// but not twice
    if (srst) swr <= 0; else swr <= s_row && !cfg_rw;			// cfg_rw not changing when s_row
    if (srst) srd <= 0; else srd <= s_row &&  cfg_rw;
    scs  <= 1'b1;
  end
  assign cfgrddat = srdbus;

  assign isel = 8'd0;
  assign iena = dinvalid;
  always @(posedge ioclk) begin
    ibus <= din;
    oena <= (doutavail & ostat);
    oval <= oena;
  end
  assign dout      = obus;
  assign doutvalid = oval;

  generate 
  if (PORT==1) begin:p1
    `USERENGINE1 #(PORT,BW) engine (sclk,srst, scs,saddr, swr,swrbus, srd,srdbus, sack,
				ioclk, istat,iena,isel,ibus, ostat,oena,osel,obus, test);
  end
  if (PORT==2) begin:p2
    `USERENGINE2 #(PORT,BW) engine (sclk,srst, scs,saddr, swr,swrbus, srd,srdbus, sack,
				ioclk, istat,iena,isel,ibus, ostat,oena,osel,obus, test);
  end
  endgenerate
			 
endmodule 			 
