/*
   Non-Midas acquisition routine for ICE card on the network

   Author: Jeff Schoen

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "icedefs.h"
#include "packet.h"
#include "niolib.h"

#define DBGTC 0

double pic_time();
int_4 control (char *addr, char *fname, char *flags, int_4 dir);
int_4 acquire (char *addr, char *fname, char *flags, int_8 size, int_4 pass, int_4 chan);
int_4 file    (char *addr, char *fname, char *aname, int_4 flags);

int main (int argc, char *argv[])
{
  char *func,*addr,*flags,*fname,naddr[80];
  int_4 i,i1,i2,loop,sock,status,chan;
  int_8 size;

  if (argc < 5) {
    BAIL:
    printf("Syntax: icenet <func> <addr> <file> <flags> <size> <loop>\n");
    printf("          <func>    = function = ACQ|CMD|SET|GET|MSG or FILE|IMPORT|EXPORT\n");
    printf("          <addr>    = network address localIF/remoteAddr (ex. udp:192.168.201.105:7777/224.1.40.%%d:7777) \n");
    printf("          <file>    = table or key=value for SET|GET|MSG, local filename for FILE|IMPORT|EXPORT|ACQ output (NULL to bypass)\n");
    printf("          <flags>   = packet type = ICE|SDDS or control type = RMIF|HTTP or archive name for IMPORT|EXPORT, remote filename for FILE\n");
    printf("          <size>    = length of acquire in bytes (ex. 1e6 is roughly 1000 SDDS packets) \n");
    printf("          <loop>    = loop count (index can be %%d in the <addr> string) (negative for 1-based) \n");
#if DBGTC
    printf("          <chan>    = channel number for time-code tests \n");
#endif
    printf("\n");
    exit(0);
  }

  func = argv[1]; 
  addr = argv[2]; 
  fname = argv[3];
  flags = argv[4];
  size = (argc>5)? (long)atof(argv[5]) : 1; 
  loop = (argc>6)? atoi(argv[6]) : 1; 
  chan = (argc>7)? atoi(argv[7]) : 0; 

  status = 0;

  if (loop<0) { i1=1; i2=-loop; }
  else { i1=0; i2=loop-1; }

  for (i=i1; i<=i2; i++) {
    sprintf(naddr,addr,i); /* possible index in loop address */
         if (strcasecmp(func,"ACQ")==0) status += acquire (naddr,fname,flags,size,i,chan);
    else if (strcasecmp(func,"SET")==0) status += control (naddr,fname,flags,1);
    else if (strcasecmp(func,"MSG")==0) status += control (naddr,fname,flags,0);
    else if (strcasecmp(func,"GET")==0) status += control (naddr,fname,flags,-1);
    else if (strcasecmp(func,"FILE")==0) status += file (naddr,fname,flags,0);
    else if (strcasecmp(func,"IMPORT")==0) status += file (naddr,fname,flags,1);
    else if (strcasecmp(func,"EXPORT")==0) status += file (naddr,fname,flags,-1);
    else printf("Function=%s not coded yet\n",func); 
  }
  if (loop>1) printf("Total errors=%d\n",status);

  exit (status);
}

int_8 I2E8 (int_8 a) {
  int_1 c, *b=(int_1*)(&a);
  c=b[0]; b[0]=b[7]; b[7]=c;
  c=b[1]; b[1]=b[6]; b[6]=c;
  c=b[2]; b[2]=b[5]; b[5]=c;
  c=b[3]; b[3]=b[4]; b[4]=c;
  return a;
}

#define PKTSIZE (8192+64)

int_4 acquire (char *addr, char *fname, char *flags, int_8 size, int_4 pass, int_4 chan) {

  int_4 sock,xfer,seq,seqNext=-1,errs=0,sdds,status;
  int_8 total,tics,ltc=0,ltcd=0;
  ICEPACKETSTRUCT *ipkt;
  SDDSPACKETSTRUCT *spkt;
  unsigned char *ucdata;
  double tdb,t1,t2,dtc,dltc,delta;
  char *pdata;
  FILE *fp = NULL;

  sock = nioOpen (addr,0,NIO_INPUT);
  if (sock<=0) { printf("Err opening socket=%s status=%d\n",addr,status); return -1; }

  if (fname!=NULL && strcmp(fname,"NULL")!=0) fp = fopen(fname,"w");

  printf("Transfer pass=%d addr=%s ",pass,addr);
  if (fp!=NULL) printf("file=%s ",fname);
  fflush(stdout);

  xfer  = PKTSIZE;
  pdata = nioAlloc(xfer);
  ipkt  = (ICEPACKETSTRUCT*)pdata;
  spkt  = (SDDSPACKETSTRUCT*)(pdata-8);  // Not an ICE/SDDS packet
  ucdata = (unsigned char *)&spkt->frame;
  sdds  = strcmp(flags,"SDDS")==0;

  t1 = pic_time();

  for (total=0; total<size; ) {

    status = nioRecv (sock, 0, pdata,xfer, 0);

    if (status==0)
      usleep(10000);
    else if (status<0) { 
      printf("nioRecv error\n");
      break;
    }
    else if (sdds) {
      seq = 256*ucdata[0] + ucdata[1];
      if (seqNext<0);
      else if (seq==seqNext);
      else printf("\n Seq errs=%d actual=%d expect=%d",errs++,seq,seqNext);
#if DBGTC
      tics = I2E8(spkt->tctics); 
      if ((spkt->tcinfo&0x40)!=0) printf("Seq actual=%d tcinfo=%04x tccorr=%04x tctics=%016lx tcdiff=%016lx dd=%ld\n",
				seq,spkt->tcinfo,spkt->tccorr,tics,tics-ltc,abs(tics-ltc-ltcd)); 
      if ((spkt->tcinfo&0x40)!=0) { ltcd=tics-ltc; ltc=tics; }
#endif
      seqNext = seq+1;
      if ((seqNext&0x1F)==0x1F) seqNext++;
      seqNext &= 0xFFFF;
      total += (status-56);
    }
    else {
      seq = ipkt->count;
      if (seqNext<0);
      else if (seq==seqNext);
      else printf("\n Seq errs=%d actual=%d expect=%d",errs++,seq,seqNext);
#if DBGTC
      delta = *(double *)(pdata+56); dtc = ipkt->tcwsec+ipkt->tcfsec-(delta*ipkt->tcoff);
      if (chan==0 || ipkt->channel==chan)
      if (ipkt->tcstatus>0) { printf("Seq=%d tcmode=%d tcstat=%d tcoff=%lf tcwsec=%lf tcfsec=%lf dtc=%lf diff=%12.9lf\n",
				seq,ipkt->tcmode,ipkt->tcstatus,ipkt->tcoff,ipkt->tcwsec,ipkt->tcfsec,dtc,dtc-dltc); dltc=dtc; }
#endif
      seqNext = seq+1;
      total += (status-64);
    }
    if (fp!=NULL && status>0) fwrite(pdata,1,status,fp);
  }

  t2 = pic_time();

  nioFree(pdata);
  nioClose(sock);
  if (fp!=NULL) fclose(fp);

  tdb = 1.e-6*total; t2-=t1;
  if (errs>0)  printf("\nTransfer pass=%d addr=%s ",pass,addr);
  printf(" size=%lfMby elapse=%lfs rate=%lfMby/s errs=%d\n",tdb,t2,tdb/t2,errs);

  return errs;
}

#define MAXMSG 8192
#define MAXMAT 1024

int_4 control (char *addr, char *fname, char *flags, int_4 dir) {

  int_4 sock,len,status,cmd,rmif,rtab,flag,ml=0;
  char msg[MAXMSG];

  strcpy(msg,fname); 
  len  = strlen(fname);
  rmif = strcasecmp(flags,"RMIF")==0;
  rtab = strcasecmp(flags,"RTAB")==0;
  flag = rtab? NIO_RTAB|NIO_HTTP : rmif? NIO_RMIF : NIO_HTTP;
  cmd  = (dir>0)?NIO_SET:(dir<0)?NIO_GET:NIO_MSG;

  sock = nioOpen(addr,0,flag);

  if (sock<=0) { printf("Err: Opening socket addr=%s status=%d\n",addr,status); return -1; }

  status = nioMsg(sock,0,cmd,msg,len,flag);

  nioClose(sock);

  if (status>0) printf("%s\n",msg);

  return status;
}


int_4 file (char *addr, char *fname, char *aname, int_4 dir) {
  int_4 sock = nioOpen(addr,0,NIO_HTTP);
  int_4 len = nioFile(sock,fname,aname,dir);
  nioClose(sock);
  return 0;
}

