/*
   Example non-Midas acquisition routine for an ICE-PIC card

   Author: Jeff Schoen

*/

#define DEVN "%s,ICE%s,DEVNO=%d,PORT=%s,IOM1=D2AWGM3,IOM2=NONE,MUXCLK=%s,"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "icedefs.h"
#include "icelib.h"

static int transfer (PICSTRUCT *p, char *fname, int_4 dir, int_4 bits, int_4 rate, int_8 size, int_4 dec, double freq, int_4 gain, int_4 nbuf);

int main (int argc, char *argv[])
{
  PICSTRUCT p;
  char devname[200],*devty,*func,*port,*fname,*clks,*flgs;
  char portstr[8]="MODULE1", nonestr[5]="NONE", clksint[2]="N";
  int_4 flags = 0;
  int_4 Mcbreak = 0;
  int_4 status,i;
  int_4 devno;
  int_4 dir,bits,rate,dec,gain,nbuf;
  int_8 size;
  double freq;

  if (argc <= 3) {
    BAIL:
    printf("Syntax:  ice <func> <devtype> <devno> <port> ... \n");
    printf("         <file> <bits> <clks> <rate> <dec> <freq> <gain> <flags> <nbuf>\n");
    printf("              <func>    = SNIFF|ACQUIRE|PLAY|DMAC|RESET \n");
    printf("              <devty>   = PIC|MBT \n");
    printf("              <devno>   = 0|1|2 ... \n");
    printf("              <port>    = MODULE1|TUNER2|MODULE3|TBANK3 ... \n");
    printf("              <file>    = filename for acquire or playback \n");
    printf("              <bits>    = bits per sample 1|4|8|16|-8|-16 (- for complex) \n");
    printf("              <clks>    = I|P|A|B|C|X|N \n");
    printf("              <rate>    = sample rate in Hz \n");
    printf("              <size>    = number of samples \n");
    printf("              <dec>     = decimation for tuners \n");
    printf("              <freq>    = frequency for tuners \n");
    printf("              <gain>    = gain for tuners \n");
    printf("              <flags>   = extra flags like VERBOSE|IOC=IIR etc\n");
    printf("              <nbuf>    = number of buffers for streaming to/from file\n");
    printf(" case sensitive \n");
    exit(0);
  }

  func = argv[1];
  devty = argv[2];
  if (strstr("PIC,MBT,SLIC",devty)==0) goto BAIL;
  devno = atoi(argv[3]); 
  port = (argc>4)? argv[4] : portstr;
  clks = (argc>7)? argv[7] : clksint;
  flgs = (argc>13)? argv[13] : nonestr;

  sprintf(devname,DEVN,flgs,devty,devno,port,clks);
  status = pic_open (&p, devname, &Mcbreak, flags);

  if (status<=0) printf ("Error opening device - aborted\n");
  else if (strcmp(func,"SNIFF")==0) status = pic_sniff(&p,0);
  else if (strcmp(func,"RESET")==0) status = pic_reset(&p,FLG_BOOT);
  else if (strcmp(func,"DMAC")==0) status = pic_dmadump(&p,0,0);
  else if (strcmp(func,"ACQUIRE")==0 || strcmp(func,"PLAY")==0) {
    fname = argv[5];
    dir  = (strcmp(func,"PLAY")==0)? 1 : -1;
    bits = atoi(argv[6]); 
    rate = atof(argv[8]); 
    size = atof(argv[9]); 
    dec  = atof(argv[10]); 
    freq = atof(argv[11]); 
    gain = atof(argv[12]); 
    nbuf = (argc>=15)? atoi(argv[14]) : 1; 
    status = pic_reset (&p, FLG_BOOT); /* full reset even though port specified */
    status = transfer (&p,fname,dir,bits,rate,size,dec,freq,gain,nbuf); 
    if (status == -1) printf("Err setting up ioport\n");
    if (status == -2) printf("Err allocating memory\n");
    if (status == -3) printf("Err setting up dma\n");
    status = pic_reset (&p, 0); /* reset the port - not necessary */
  }
  pic_close (&p);
  exit (status);
}


int transfer (PICSTRUCT *p, char *fname, int_4 dir, int_4 bits, int_4 rate, int_8 size,
	int_4 dec, double freq, int_4 gain, int_4 nbuf) {

  int_4 dmac,nstart,nxfer,status,fstat,nmap;
  int_8 nbytes;
  double ratio;
  FILE *fp;
  DMAMAP map;
  void *data;

  nbytes = size*bits/8;			/* size converted to bytes */
  if (nbytes<0) nbytes = -2*nbytes;	/* -bits means complex */

  if ((nbytes%4096)!=0) nbytes -= (nbytes%4096);

  if (nbytes>0x10000000) {
    nxfer = 0x00100000;
    nbytes -= (nbytes%nxfer);
    nmap = -nbytes/0x1000;
  } else {
    nmap = nbytes;
    nxfer = nbytes/16;			/* x16 buffering */
  }

  /* present normalized freq */
  if (bits<0) freq /= rate;
  else freq /= (rate/2);

  /* special tuner bank parameters */
  if (pic_getkeyl(p,0,KEY_PTYPE)==IOPT_TBANK) {
    pic_setkeyl(p,0,KEY_CHNS,   pic_getintflag(p,"CHNS"));
    pic_setkeyl(p,0,KEY_PKTLEN, pic_getintflag(p,"PKTLEN"));
    pic_setkeyd(p,0,KEY_DFREQ,  pic_getdblflag(p,"DFREQ")/rate);
    if (dir>0) pic_setkeyd(p,0,KEY_RATIO, pic_getdblflag(p,"DRATIO"));
  }

  /* set I/O port */
  dmac = pic_ioport (p, -1, -1, -1, dir, bits, rate, freq, dec, gain, 0);
  if (dmac<=0) return -1;

  /* special resampler handling */
  ratio = pic_getdblflag(p,"RATIO");
  if (ratio>0) pic_setkeyd (p,dmac,KEY_RATIO,ratio);

  /* map the buffer to physical memory */
  status = pic_mapmem (p, &map, nmap, 1);
  if (status<0) return -2;
  data = (void *)map.vaddr;

  /* setup the DMA handler */
  status = pic_dmasetup (p, dmac, dir, &map, -1, 0);
  if (status<0) return -3;

  /* open the input/output file */
  fp = fopen(fname, (dir>0)? "r":"w" );
  if (fp==NULL) { printf("Err opening data file: %s\n",fname); goto NOFILE; }

  /* preload output buffer from file */
  if (dir>0) {
    status = fread(data,1,nbytes,fp);
    if (status!=nbytes) printf("Err: Read status=%d of %d bytes from %s\n",status,nbytes,fname);
  }

  /* start transfer for nbuf passes (1==DMA_ONESHOT) */
  status = pic_dmafunc (p, dmac, nbuf); 

  /* multiple shot overlapped transfer loop */
  for (;;) {
    status = pic_dmaxptr (p, dmac, &data, nxfer, 0); /* blocks until nxfer bytes ready, returns status and data ptr */
    if (status<0) break; /* transfer complete */

    if (dir<0)       fstat = fwrite(data,1,status,fp);	/* flush next full region to disk */
    else if (nbuf>1) fstat = fread(data,1,status,fp);	/* load newly opened region */
    else             fstat = status;			/* no need to fill ahead */
    if (fstat!=status && dir>0) {
      if (fstat<0) fstat=0; 
      fseek(fp,0L,SEEK_SET); 			 	/* loop on input file */
      printf("Read loop status=%d of %d bytes from %s\n",fstat,status,fname);
      fstat += fread(data+fstat,1,status-fstat,fp);
    }
    if (fstat!=status) printf("Err: Read|Write status=%d of %d bytes from %s\n",fstat,status,fname);
  } 

  /* remove channel from queue */
  status = pic_dmafunc (p, dmac, DMA_CANCEL); 

  /* close the input/output file */
  fclose(fp);
  NOFILE:

  /* unmap the buffer from physical memory */
  status = pic_mapmem (p, &map, nmap, -1);

  return 1;
}
