/****************************************************************************/
/*                                                                          */
/*  part_stv6111.c                                                          */
/*                                                                          */
/****************************************************************************/

#include "part_stv6111.h"

/* Sane defaults for STV6111 registers                             */
/*                                                                 */
int32_t pt_stv6111_rgc_default(PT_STV6111_REG *R)
{
  static const PT_STV6111_REG stv6111_reg_defaults = {
    {0x7C,0xB1,0x40,0xA6,0xF4,0x00,0x00,0x90,
     0x46,0xBD,0x91,0x00,0x00,0x00,0x00,0x84} };

  int32_t i;
  for (i=0; i<STV6111_NUM_RG; i++){
    R->rg[i] = stv6111_reg_defaults.rg[i];
  }
  return (0);
}


/*                                                                  */
/* Modifies a STV6111 register set to tune to a given RF frequency  */
/* and returns the exact tuned frequency based on fref and dividers */
/*                                                                  */
int32_t pt_stv6111_rgc_rffreq(PT_STV6111_REG *S, UDB *bmhz, float64_t fref)
{
  /* Fref(MHz), R, K */
  static const uint8_t k_tbl[17][3] = {
    {31,0x01,0x10}, {30,0x01,0x0F}, {29,0x01,0x0E}, {28,0x01,0x0D}, 
    {27,0x01,0x0C}, {26,0x01,0x0B}, {25,0x00,0x0A}, {24,0x00,0x09}, 
    {23,0x00,0x08}, {22,0x00,0x07}, {21,0x00,0x06}, {20,0x00,0x05},
    {19,0x00,0x04}, {18,0x00,0x03}, {17,0x00,0x02}, {16,0x00,0x01}, 
    {15,0x00,0x00} };

  /* (VCOFREQ/100), PLL_ICP, VCO_AMP */
  static const uint8_t cp_tbl[7][3] = {
    {52,0x07,0x01}, {48,0x06,0x01}, {42,0x05,0x01}, {37,0x04,0x01}, 
    {33,0x02,0x01}, {30,0x01,0x01}, {27,0x00,0x01} };
  
  int32_t i,R,P,N,F;
  float64_t fmhz,fnorm,fvco;
  float64_t sdbias = 0.0;
  int32_t idx = 0;

  /* Set K, R, O dividers based on reference frequency */                    
  if ((fref < 15.0)||(fref > 32.0)) return (-1);
  idx = 0;
  for (i=0; i<17; i++) {
    idx = (fref <= ((float64_t)k_tbl[i][0])) ? (i) : (idx);
  }
  S->rg[0x00] = u8_setbits(S->rg[0x00],STV6111_RGM_00_R,k_tbl[idx][1]);
  S->rg[0x00] = u8_setbits(S->rg[0x00],STV6111_RGM_00_K,k_tbl[idx][2]);
  S->rg[0x00] = u8_setbits(S->rg[0x00],STV6111_RGM_00_O,0x00);
  R = k_tbl[idx][1];

  /* Set LO divider based on RF / VCO frequency */
  fmhz = bmhz->f64[0];
  if ((fmhz < 649.9)||(fmhz > 2600.1)) return (-2);
  P = (fmhz > 1300.0) ? (0x00) : (0x01);
  S->rg[0x03] = u8_setbits(S->rg[0x03],STV6111_RGM_03_LO,P);

  /* Compute normalized frequency accounting for sigma-delta bias */
  fnorm = (((float64_t)(R+1))/fref)*((float64_t)(2*(P+1)))*(fmhz);
  N = (int32_t) (fnorm);
  N = (N > 0) ? (N) : (1);
  F = (int32_t) ((262144.0*(fnorm-((float64_t)N)))+0.499999);

  if ((F <= 0) || (F >= 262144)) {
    N = (int32_t) (fnorm+0.4999999);
    N = (N > 0) ? (N) : (1);
    F = 0;
    sdbias = 0.0;
  }  
  else {
    N = (int32_t) (fnorm);
    N = (N > 0) ? (N) : (1);
    F = (int32_t) ((262144.0*(fnorm-((float64_t)N)))+0.499999);
    F = (F > 0) ? (F) : (1);
    F = (F < 262144) ? (F) : (262143);
    sdbias = 0.5;
  }  
  fmhz = ((float64_t)N)+((sdbias+((float64_t)F))/262144.0);
  fmhz = (fref/((float64_t)(R+1)))*(fmhz/((float64_t)(2*(P+1))));
  bmhz->f64[0] = fmhz;

  S->rg[0x04] = u8_setbits(S->rg[0x04],STV6111_RGM_04_N70,((0xFF)&(N)));
  S->rg[0x05] = u8_setbits(S->rg[0x05],STV6111_RGM_05_N88,((0x01)&(N>>8)));
  S->rg[0x05] = u8_setbits(S->rg[0x05],STV6111_RGM_05_F60,((0x7F)&(F)));
  S->rg[0x06] = u8_setbits(S->rg[0x06],STV6111_RGM_06_F147,((0xFF)&(F>>7)));
  S->rg[0x07] = u8_setbits(S->rg[0x07],STV6111_RGM_07_F1715,((0x07)&(F>>15)));

  /* If fractional component is non-zero, turn on PLL SD modulator */
  if (F == 0) 
    S->rg[0x08] = u8_setbits(S->rg[0x08],STV6111_RGM_08_PLLSD,0x00);
  else
    S->rg[0x08] = u8_setbits(S->rg[0x08],STV6111_RGM_08_PLLSD,0x01);

  /* Compute VCO frequency and CP current */
  fvco = fmhz*P; 
  idx = 0;
  for (i=0; i<7; i++) {
    idx = (fvco <= (100.0*(float64_t)cp_tbl[i][0])) ? (i) : (idx);
  }
  S->rg[0x07] = u8_setbits(S->rg[0x07],STV6111_RGM_07_ICP,cp_tbl[idx][1]);
  S->rg[0x07] = u8_setbits(S->rg[0x07],STV6111_RGM_07_VCOAMP,cp_tbl[idx][2]);

  /* Set VCO calibrator clock frequency */
  S->rg[0x09] = u8_setbits(S->rg[0x09],STV6111_RGM_09_TCAL,0x00);

  dbgprint(0,1,"   STV6111-RFREQ: N:%d F:%d R:%d P:%d FNORM: %21.17f FREQ: %21.16f FREF: %21.17f \n",
          N,F,R,P,fnorm,fmhz,fref);

  return (0);
}
int32_t pt_stv6111_rgc_rffreq_(PT_STV6111_REG *S, real_8 *bmhz, real_8 fref) {
  return pt_stv6111_rgc_rffreq(S, (UDB *)bmhz, (float64_t)fref);
}


/*                                                                 */
/* Modifies a STV6111 register set to change to a given RF BW      */
/*                                                                 */
int32_t pt_stv6111_rgc_rfbw(PT_STV6111_REG *S, UDB *bmhz)
{
  uint8_t ubw = 0x11;
  bmhz->f64[0] = 0.5*bmhz->f64[0];
  udb_f64_to_i32wf(bmhz);
  bmhz->i32[0] = (bmhz->i32[0] < 6) ? (6) : (bmhz->i32[0]);
  bmhz->i32[0] = (bmhz->i32[0] > 69) ? (69) : (bmhz->i32[0]);
  if (bmhz->i32[0] > 63) {
    ubw = 44;
    S->rg[0x08] = u8_setbits(S->rg[0x08],STV6111_RGM_08_BBFSEL,ubw); 
    S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_BBWB,0x01); 
    S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_DCLOOP,0x01);
  }
  else {
    ubw = (0x3F)&(bmhz->i32[0] - 6);
    S->rg[0x08] = u8_setbits(S->rg[0x08],STV6111_RGM_08_BBFSEL,ubw); 
    S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_BBWB,0x00); 
    S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_DCLOOP,0x00);
  }
  bmhz->i32[0] = 2*bmhz->i32[0];
  bmhz->i32[1] = 0;
  udb_i32wf_to_f64(bmhz);
  return (0);
}
int32_t pt_stv6111_rgc_rfbw_(PT_STV6111_REG *S, real_8 *bmhz) {
  return pt_stv6111_rgc_rfbw(S, (UDB *)bmhz);
}


/*                                                                 */
/* Modifies a STV6111 register set to set manual RF gain           */
/*   The PE43204 attenuator is included in this gain calculation   */
/*   since it's setting is dependent on the STV6111 gain.          */
/*                                                                 */
int32_t pt_stv6111_rgc_rfgain(PT_STV6111_REG *S, UDB *gn, uint32_t *atn)
{
  float64_t gaintot = 0;
  uint8_t stvlna  = 0x01;
  int32_t i;
  int32_t idx = 0;
  float64_t gerrabs;
  float64_t gerr = 1000.0;

  typedef struct {
    uint8_t   rgval; 
    float64_t dbgain;
  } STV_GAINTBL;

  static const STV_GAINTBL stvgntbl[32] = {
    {0x00, 0.00}, {0x01, 0.03}, {0x02, 0.06}, {0x03, 0.10}, {0x04, 0.13},
    {0x05, 0.16}, {0x06, 0.35}, {0x07, 0.63}, {0x08, 1.16}, {0x09, 1.80},
    {0x0A, 2.54}, {0x0B, 3.37}, {0x0C, 4.28}, {0x0D, 5.25}, {0x0E, 6.29},
    {0x0F, 7.39}, {0x10, 8.52}, {0x11, 9.70}, {0x12,10.90}, {0x13,12.12},
    {0x14,13.36}, {0x15,14.59}, {0x16,15.82}, {0x17,17.03}, {0x18,18.22},
    {0x19,19.38}, {0x1A,20.50}, {0x1B,21.56}, {0x1C,22.57}, {0x1D,23.52},
    {0x1E,24.38}, {0x1F,25.17} };

  /* Baseband settings */
  S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_BBMAG,0x03);
  S->rg[0x01] = u8_setbits(S->rg[0x01],STV6111_RGM_01_BBMODE,0x00);

  /* First, determine attenuator setting */
  *atn = PE43204_ATTN_18DB;
  gaintot = gn->f64[0];
  if (gaintot >= 18.0) {
    *atn = PE43204_ATTN_00DB;
    gaintot -= 18.0;
  }
  else if (gaintot >=  12.0) { 
    *atn = PE43204_ATTN_06DB;
    gaintot -= 12.0;
  }
  else if (gaintot >= 6.0) { 
    *atn = PE43204_ATTN_12DB;
    gaintot -= 6.0;
  }

  /* Turn on STVLNA if lots of gain requested */
  stvlna  = 0x01;
  if ((gaintot - stvgntbl[31].dbgain) > 0.0) {
    stvlna  = 0x00;
    gaintot -= 6.0;
  }
  S->rg[0x02] = u8_setbits(S->rg[0x02],STV6111_RGM_02_LNASEL,stvlna);

  /* Minimize absolute error */
  i = 0;
  idx = 0;
  gerrabs = 0.0;
  gerr = 1000.0;
  for (i=0; i<32; i++) {
    gerrabs = gaintot - stvgntbl[i].dbgain;
    gerrabs = (gerrabs > 0.0) ? (gerrabs) : (-1.0*gerrabs);
    if (gerrabs < gerr) {
      idx = i;
      gerr = gerrabs;
    } 
  }
  gaintot -= stvgntbl[idx].dbgain;
  S->rg[0x02] = u8_setbits(S->rg[0x02],STV6111_RGM_02_LNAAGC,stvgntbl[idx].rgval);

  dbgprint(0,1,"   STV6111-RFGAIN-> FG: %6.2f FE: %6.2f ATN: %2.2X LNA: %2.2X idx: %2.2d AGC: %2.2X \n",gn->f64[0],gaintot,*atn,stvlna,idx,stvgntbl[idx].rgval);

  /* Report warning if ABS(gain_error) > 3dB */
  gaintot = (gaintot > 0.0) ? (gaintot) : (-1.0*gaintot);
  if (gaintot > 3.0) 
    dbgwarnmsg("Requested RF Gain outside allowed bounds. \n");

  return (0);
}
int32_t pt_stv6111_rgc_rfgain_(PT_STV6111_REG *S, real_8 *gn, int_u4 *atn) {
  return pt_stv6111_rgc_rfgain(S, (UDB*)gn, (uint32_t*)atn);
}
