/****************************************************************************/
/*                                                                          */
/*  c3_target_icepic - C3 (Command,Control,Communication) interface part to */
/*                     a module on an ICEPIC device                         */
/*                                                                          */
/****************************************************************************/

#include "c3_target_icepic.h"    

#define _BUILD_DEBUG_ROUTINES 1

int_4 partaddrlookup(Part_Config_Obj *part_cfg, int_4 numparts, int_4 part_ID,
                     int_4 sub_ID)
{
  int_4 i = 0;
  Part_Config_Obj *partptr = part_cfg;

  /* Check if Root Part is requested - Always at address 0 */
  if(part_ID==BAD_ROOT_ID) return(0);

  for(i=0;i<numparts;++i){
    if(partptr->part_ID==part_ID && partptr->part_subID==sub_ID) {
      return(part_cfg[i].part_addr);
    }
    ++partptr;
  }
  return(-1);
}

int_4 cmdcfg_ctl(Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, 
                 int_4 func, int_4 addr, int_4 data, int_u4 mask)
{

  int_4 part_reg0;
  int_4 part_reg1;
  int_4 part_cntl,cmd_cntl;
  int_4 cmd_addr0 = 0;
  int_4 cmd_addr1 = 0;
  int_4 part_addr = 0;
  int_4 cmd_rtn;

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  /* Part Address Resolution */
  part_addr = partaddrlookup(&module_cfg->part_config[0],
                               module_cfg->RT.numparts,part_ID,sub_ID);
  if(part_addr<0){
    dbgerror("Part ID: %d (%d) Not On Module\n",part_ID,sub_ID); 
    return(-1); 
  }

  part_cntl = (((module_cfg->RT.cmdkey)&0xFF)<<24)|((part_addr&0xFF)<<16);
  cmd_cntl = (((module_cfg->RT.cmdkey)&0xFF)<<24)|((0x16&0xFF)<<16);
  part_reg0 = data;
  part_reg1 = part_cntl|(func<<12)|addr;

  cmd_addr0 = ((module_cfg->RT.numparts)-1)<<8;
  cmd_addr1 = cmd_addr0+4;

/* Check to see if Command Buffer is ready to handle the command, wait if not...*/
  cmd_rtn = 0;
  pic_jpmemwr(p,mport,cmd_addr1,cmd_cntl|(CFUNC_RRD<<12)|4,4);
  do{
    cmd_rtn = pic_jpmemrd(p,mport,cmd_addr0,4);
  }while(cmd_rtn>500);

  if(func < CFUNC_PWR){
    /* Write the Mask for Part Commands */
    pic_jpmemwr(p,mport,cmd_addr0,mask,4);
    pic_jpmemwr(p,mport,cmd_addr1,(part_cntl|(PFUNC_RWR<<12)|0x00000FFF),4);
  }

  /* Perform Write/Read Command */
  pic_jpmemwr(p,mport,cmd_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,cmd_addr1,part_reg1,4);

  return(0);
}

int_4 cmdcfg_stat(Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, int_4 *func, int_4 *pin, int_4 *addr, int_4 *data)
{
  int_4 cmd_addr0 = ((module_cfg->RT.numparts)-1)<<8;
  int_4 cmd_addr1 = cmd_addr0+4;
  int_4 cmd_stat0,cmd_stat1;

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  cmdcfg_ctl(module_cfg, part_ID,sub_ID,CFUNC_RRD,0,0,0);

  cmd_stat0 = 0;
  cmd_stat1 = 0;
  cmd_stat0 = pic_jpmemrd(p,mport,cmd_addr0,4);
  cmd_stat1 = pic_jpmemrd(p,mport,cmd_addr1,4);

  *func = (cmd_stat1>>12)&0xF;
  *pin = (cmd_stat1>>16)&0xFF;
  *addr = cmd_stat1&0xFFF;
  *data = cmd_stat0;

  return(0);
}

int_4 cmdcfg_getcmdkey(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 cmd_part_addr = ((module_cfg->RT.numparts-1)<<8);
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,10,0,0xFF);
  module_cfg->RT.cmdkey = pic_jpmemrd(p,mport,cmd_part_addr,4);
  dbgprint(module_cfg->dbg,DBG_PT,"Command Key = 0x%x\n",module_cfg->RT.cmdkey);

  return(0);
}

int_4 cmdcfg_checklock(Module_Config_Obj *module_cfg)
{
  int_4 locked,cmdkey;
  int_4 cmd_part_addr = ((module_cfg->RT.numparts-1)<<8);
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  /* Check if Session is already locked */
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,2,0,0xFF);
  locked = pic_jpmemrd(p,mport,cmd_part_addr,4);
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,8,0,0xFF);
  cmdkey = pic_jpmemrd(p,mport,cmd_part_addr,4);
  dbgprint(module_cfg->dbg,DBG_PT,"Session Status: lock=%d, active_key=%d, cmd_key=%d\n",locked,cmdkey,module_cfg->RT.cmdkey);

  return(0);
}

int_4 cmdcfg_locksession(Module_Config_Obj *module_cfg)
{
  int_4 rtn;
  int_4 cmd_part_addr = ((module_cfg->RT.numparts-1)<<8);
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  /* Check if Session is already locked */
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,2,0,0xFF);
  rtn = pic_jpmemrd(p,mport,cmd_part_addr,4);
  dbgprint(module_cfg->dbg,DBG_PT,"Lock Session: session_lock = %d\n",rtn);
  if(rtn) return(-1);

  /* Lock Session */
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,2,module_cfg->RT.cmdkey,0xFF);
  
  /* Verify Lock */
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,8,0,0xFF);
  rtn = pic_jpmemrd(p,mport,cmd_part_addr,4);
  dbgprint(module_cfg->dbg,DBG_PT,"Active Session Key: 0x%x\n",rtn);
  if(rtn != module_cfg->RT.cmdkey) return(-1);

  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,2,0,0xFF);
  rtn = pic_jpmemrd(p,mport,cmd_part_addr,4);
  dbgprint(module_cfg->dbg,DBG_PT,"Lock Session: session_lock = %d\n",rtn);

  return(0);
}

int_4 cmdcfg_delay(Module_Config_Obj *module_cfg, int_4 delay)
{
  int_4 delay_cnt = delay*module_cfg->RT.cfgclk;
  cmdcfg_ctl(module_cfg,BAD_DELAY_ID,0,PFUNC_RWR,1,delay_cnt,0xFFFFFFFF);
  cmdcfg_ctl(module_cfg,BAD_DELAY_ID,0,PFUNC_RWR,0,0,0xFFFFFFFF);
  return(0);
}

int_4 cmdcfg_bufclr(Module_Config_Obj *module_cfg)
{
  int_4 cmd_cnt_addr = 4;
  int_4 cmd_part_addr = ((module_cfg->RT.numparts-1)<<8);
  int_4 cmd_rtn,part_busy;
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;
  int_4 timeout;
  
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,cmd_cnt_addr,0,0);

  timeout = 5000;
  do{
    cmd_rtn = pic_jpmemrd(p,mport,cmd_part_addr,4);
    part_busy = (pic_jpmemrd(p,mport,cmd_part_addr+4,4)>>12)&0x8;
    if(timeout-- == 0){
      dbgerrormsg("COMMAND BUFFER!!!!!!\n");  
      return(-1);
    }
  }while((cmd_rtn>0) || part_busy);

  return(0);
}

int_4 getpartconfig(PICSTRUCT *p, int_4 mport, int_4 part_num, Module_Config_Obj *module_cfg, Part_Config_Obj *part_cfg)
{
  int_4 part_addr = MODCONFIG_ADDR+PARTCONFIG_BYTE*part_num; 
  int_4 part_word = pic_jpmemrd(p,mport,part_addr,4);

  part_cfg->part_ID =   (part_word>>8)&0xFFFF;
  part_cfg->part_subID = part_word&0xFF;
  part_cfg->part_addr = (part_word>>24)&0xFF;

  dbgprint(module_cfg->dbg,DBG_PT,"Part Config for Module: Part %d at addr 0x%x\n",part_num+1,part_addr);
  dbgprint(module_cfg->dbg,DBG_PT,"RAW = 0x%x\n",part_word);
  dbgprint(module_cfg->dbg,DBG_PT,"Part ID = %d\n",part_cfg->part_ID);
  dbgprint(module_cfg->dbg,DBG_PT,"Part SubID = %d\n",part_cfg->part_subID);
  dbgprint(module_cfg->dbg,DBG_PT,"Part Addr = 0x%x\n",part_cfg->part_addr);

  return(part_word);
}

int_4 getmoduleconfig(Module_Config_Obj *module_cfg)
{
  int_4 i,x,crc;
  UDB bb = udb_initializer;
  UDB bd = udb_initializer;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  bb.i32[0] = pic_jpmemrd(p,mport,MODCONFIG_ADDR,4);
  bd.i32[0] = pic_jpmemrd(p,mport,MODDETECT_ADDR,4);
  module_cfg->RT.modid    = bb.i8[0]; 
  module_cfg->RT.hw_ver   = bd.i8[0]; 
  module_cfg->RT.sw_ver   = bb.i8[2]; 
  module_cfg->RT.numparts = bb.i8[3]; 

  dbgprintmsg(module_cfg->dbg,DBG_PT,"Module Configuration : \n");
  dbgprint(module_cfg->dbg,DBG_PT,"ModID = 0x%x\n", module_cfg->RT.modid);
  dbgprint(module_cfg->dbg,DBG_PT,"HW Ver = %d.%d.%d\n",(module_cfg->RT.hw_ver>>6)&0x3,(module_cfg->RT.hw_ver>>3)&0x7,(module_cfg->RT.hw_ver)&0x7);
  dbgprint(module_cfg->dbg,DBG_PT,"SW Ver = %d.%d.%d\n",(module_cfg->RT.sw_ver>>6)&0x3,(module_cfg->RT.sw_ver>>3)&0x7,(module_cfg->RT.sw_ver)&0x7);
  dbgprint(module_cfg->dbg,DBG_PT,"Num Parts = %d\n",module_cfg->RT.numparts);

  /* Part Identification */
  /* crc = *(int_4*)module_cfg; */
  crc = bb.i32[0];
  for(i=1; i<module_cfg->RT.numparts; ++i){
    x = getpartconfig(p,mport,i,module_cfg,&module_cfg->part_config[i]);
    crc^=x;
  }

  module_cfg->RT.cmdkey = 0; /* Initialize Command Key to Zero */
  /* Set Session Timeout */
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,1,0xFFFFFFFF,0xFFFFFFFF);

  cmdcfg_getcmdkey(module_cfg);
  x = cmdcfg_locksession(module_cfg);
  if(x<0) return(-1);

  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,5,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,5,0x00,0xFF);

  /* Get the CRC, and Configuration Clock Frequency from the Module Part */
  cmdcfg_ctl(module_cfg,BAD_ROOT_ID,0,PFUNC_RRD,5,0,0xFFFFFFFF);

  cmdcfg_ctl(module_cfg,BAD_ROOT_ID,0,PFUNC_RRD,4,0,0xFFFFFFFF);

  if(cmdcfg_bufclr(module_cfg)<0) return(-1);

  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  module_cfg->RT.cfgclk = stat_data;

  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  module_cfg->RT.modcrc = stat_data;

  dbgprint(module_cfg->dbg,DBG_PT,"CRC = 0x%x\n",module_cfg->RT.modcrc);
  dbgprint(module_cfg->dbg,DBG_PT,"Calculated CRC = 0x%x\n",crc);
  dbgprint(module_cfg->dbg,DBG_PT,"Config Clock = %dMHZ\n",module_cfg->RT.cfgclk);

  if(crc != module_cfg->RT.modcrc) {
    dbgerrormsg("Checksum error in Module Configuration. Retrying...\n");
    dbgprintmsg(module_cfg->dbg,DBG_PT,"/* Release Session Lock */\n");
    cmdcfg_bufclr(module_cfg);
    cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,1,0x0,0xFFFFFFFF);
    return(-1);
  }
  return(0);
}


/*******************************************************************/
#ifdef _BUILD_DEBUG_ROUTINES


int_4 getrdbufstat(Module_Config_Obj *module_cfg)
{
  int_4 x1,x2;

  int_4 cmd_addr0 = ((module_cfg->RT.numparts)-1)<<8;
  int_4 cmd_addr1 = cmd_addr0+4;

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,15,0,0xFFFFFFFF);
  x1 = pic_jpmemrd(p,mport,cmd_addr0,4);  
  x2 = pic_jpmemrd(p,mport,cmd_addr1,4);  

  dbgprint(module_cfg->dbg,DBG_PT,"Read Buffer Status: Output Ready=0x%1.1x : Input Ready=0x%1.1x\n",x1&0xF,(x1>>4)&0xF);

  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RRD,5,0,0xFFFFFFFF);
  x1 = pic_jpmemrd(p,mport,cmd_addr0,4);  
  x2 = pic_jpmemrd(p,mport,cmd_addr1,4);  

  dbgprint(module_cfg->dbg,DBG_PT,"Read Commands in Buffer Status: 0x%8.8x : 0x%8.8x\n",x2,x1);

  return(0);
}


int_4 partcfg_write(Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, int_4 func, int_4 pin, int_4 addr, int_4 data, int_u4 mask)
{

  int_4 part_reg0 = data;
  int_4 part_reg1 = (pin<<16)|(func<<12)|addr;

  /* Part Address Resolution */
  int_4 part_addr = partaddrlookup(&module_cfg->part_config[0],module_cfg->RT.numparts,part_ID,sub_ID);
  if(part_addr<0){dbgerror("Part ID: %d Not On Module\n",part_ID); return(-1); }

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 part_addr0 = (part_addr<<8);
  int_4 part_addr1 = (part_addr<<8)+4;
  int_4 part_busy;

  /* Write the Mask */
  pic_jpmemwr(p,mport,part_addr0,mask,4);
  pic_jpmemwr(p,mport,part_addr1,((PFUNC_RWR<<12)|0x00000FFF),4);

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  }while(part_busy);

  /* Check for Busy */
  part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  if(part_busy) dbgerror("WRITING: WHY IS THE PART BUSY? 0x%x\n",part_addr);

  /* Perform Write */
  pic_jpmemwr(p,mport,part_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,part_addr1,part_reg1,4);

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  }while(part_busy);

  return(0);
}


int_4 partcfg_read(Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, int_4 func, int_4 *pin, int_4 addr, int_4 *data, int_u4 mask)
{

  int_4 part_reg0 = 0;
  int_4 part_reg1 = (func<<12)|addr;

  /* Part Address Resolution */
  int_4 part_addr = partaddrlookup(&module_cfg->part_config[0],module_cfg->RT.numparts,part_ID,sub_ID);
  if(part_addr<0){dbgerror("Part ID: %d Not On Module\n",part_ID); return(-1); }

  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 part_addr0 = (part_addr<<8);
  int_4 part_addr1 = (part_addr<<8)+4;
  int_4 part_busy;

  /* Write the Mask */
  if(func!=0 && func!=1) {
    pic_jpmemwr(p,mport,part_addr0,mask,4);
    pic_jpmemwr(p,mport,part_addr1,((PFUNC_RWR<<12)|0x00000FFF),4);
    do{
      part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
    }while(part_busy);
  }

  /* Check for Busy */
  part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  if(part_busy) dbgerror("READING: WHY IS THE PART BUSY? 0x%x\n",part_addr);

  /* Perform Write */
  pic_jpmemwr(p,mport,part_addr0,part_reg0,4);
  pic_jpmemwr(p,mport,part_addr1,part_reg1,4);

  /* Wait Until Ready */
  do{
    part_busy = (pic_jpmemrd(p,mport,part_addr1,4)&0x00008000)>>15;
  }while(part_busy);
  
  *data = pic_jpmemrd(p,mport,part_addr0,4);
  *pin  = (pic_jpmemrd(p,mport,part_addr1,4)>>16)&0xFFFF;

  return(0);
}






#endif /* _BUILD_DEBUG_ROUTINES */
