/****************************************************************************/
/*  QSFP module interface routines                                          */
/****************************************************************************/

#define SEND_BASE 0x100

#define ARP_PAD   20
#define ARP_SIZE  (16+28+ARP_PAD)

#define ICMP_PAD  20
#define ICMP_SIZE (16+20+8+ICMP_PAD)

#define IGMP_BASE 0x200
#define IGMP_PAD  16
#define IGMP_SIZE (16+24+8+IGMP_PAD)
#define IGMP_SEP  IGMP_SIZE

#define VCTL_BASE 0x400
#define VCTL_SIZE 18*4
#define VCTL_SEP  128

#define MAXARPTBL 2048
#define MAXNETCFG 256

int_u4 parseIPaddr (char *s) {
  int_4 i,oct=0,addr=0;
  char *sn,*se;
  se = s + strlen(s);
  for (i=0; i<4; i++,s=sn+1) {
    if (s>=se || *s=='\n') break;
    oct = strtol(s,&sn,10);
    /* for parsing the IPADDRX constructs */
    if (sn[0]=='+') oct += strtol(sn+1,&sn,10);
    if (sn[0]=='|') oct |= strtol(sn+1,&sn,10);
    if (sn[0]=='&') oct &= strtol(sn+1,&sn,10);
    addr = (addr<<8) | (oct&0xFF);
  }
  return (i==1)? oct : addr;
}
int_4 parseIPport (char *s, int_4 port) {
  char *sc = strstr(s,":");
  if (sc!=NULL) port = atol(sc+1);
  return port;
}
int_4 parseSID (char *s) {
  int i,id=0,sign=1; char c;
  if (strncmp(s,"ANY",3)==0) return -1;
  for (i=0; i<strlen(s); i++) {
    c = s[i];
    if (i==0 && c=='+') sign=1;
    if (i==0 && c=='-') sign=-1;
    if (c>='0' && c<='9') id = id*16 + (c-'0');
    if (c>='a' && c<='f') id = id*16 + (c-'a') + 10;
    if (c>='A' && c<='F') id = id*16 + (c-'A') + 10;
  }
  if (id==0xFFFF) return -1;
  return sign*id;
}
int_8 parseRF (char *s) {
  double d = strtod(s,NULL);
  return (int_8)(d*1e6+.5);
}
int_4 loadArray (PICSTRUCT *p, int_4 pm, int_4 iadr, int_4 index, int_4 value) {
  pic_wpb (p,pm,iadr|0x101,index);
  pic_wpb (p,pm,iadr|0x103,value);
  return 0;
}

int_8 getARPtable (PICSTRUCT *p, int_4 addr) {
  int_4 i,na,lat,tpl[6]; int_8 mac=0;
  char fn[40],saddr[20],data[MAXARPTBL],*s;
  FILE *fp;
  strcpy(fn,"/var/etc/arptbl");
  fp = fopen(fn,"r");
  if (fp==NULL) return -1;
  lat = fread(data,1,MAXARPTBL,fp);
  fclose(fp);
  if (lat<=0) { printf("Err: Reading arp table file=%s or file is empty.\n",fn); return -2; }
  data[lat]=0;
  sprintf(saddr,"%d.%d.%d.%d",(addr>>24)&0xFF,(addr>>16)&0xFF,(addr>>8)&0xFF,(addr>>0)&0xFF);
  s = strstr(data,saddr);
  if (s==NULL) return -1;
  s = strstr(s,"lladdr");
  if (s==NULL) { printf("Err: finding lladdr=%s keyword in arp table file=%s.\n",saddr,fn); return -2; }
  na = sscanf(s+7,"%02x:%02x:%02x:%02x:%02x:%02x ",tpl+5,tpl+4,tpl+3,tpl+2,tpl+1,tpl+0);
  if (na!=6) { printf("Err: converting %s in arp table file=%s.\n",s,fn); return -2; }
  for (i=5; i>=0; i--) mac = (mac<<8) | (tpl[i]&0xFF);
  vprint("ARP for ip=%08x returned eth=%012lx\n",addr,mac);
  return mac;
}

int_u4 getGateway (PICSTRUCT *p, int_4 isfp, int_u4 from, int_u4 to) {
  int_4 i,na,lc; int_u4 mask=0,gate=0;
  char fn[40],str[40],data[MAXNETCFG],*s;
  FILE *fp;
  strcpy(fn,"/var/etc/network.conf");
  fp = fopen(fn,"r"); if (fp==NULL) return -1;
  lc = fread(data,1,MAXNETCFG,fp);
  fclose(fp);
  if (lc<=0) { printf("Err: Reading network config file=%s or file is empty.\n",fn); return -2; }
  data[lc]=0;
  if (findstrflagx("qsfpmask",data,str,9,isfp)>0) mask = parseIPaddr(str);
  if (findstrflagx("qsfpgate",data,str,9,isfp)>0) gate = parseIPaddr(str);
  if ( (from&mask) == (to&mask)) return to;
  vprint("Link from=%08x to=%08x with netmask=%08x through gateway=%08x\n",from,to,mask,gate);
  return gate;
}

int_4 do_nio_command (PICSTRUCT *p, int_4 isfp, int_4 cmd) {
  int_4 i,status,timeout=100;
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  int_4 fullcmd = 0x1CE00000 | (cmd<<16) | ichn;
  jvm_wmem (p,pm,0x04,fullcmd,FLG_NIO);
  for (i=0; i<timeout; i++) {
    status = jvm_rmem (p,pm,0x04,FLG_NIO);
    if (status==(fullcmd|0x20000000)) return 0;
    pic_relock(p,10);
  }
  printf("Error status=%08x running NIO command=%08x on isfp=%d\n",status,fullcmd,isfp);
  return -1;
}

int_4 pic_igmp_qsfp (PICSTRUCT *p, PKTPlan *plan, int_4 isfp) {
  int_4 n,addr,cb[128];
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  n = PKT_ipeth(plan,cb,0,0,IGMP_PAD,PROTO_IGMP);	/* forms IGMP response packet */
  if (n*4 != IGMP_SEP) printf("IGMP padded size=%d not right=%d\n",n*4,IGMP_SEP);
  addr = IGMP_BASE + ichn*IGMP_SEP;
  jvm_rwmem (p,pm,addr,cb,IGMP_SEP,FLG_NIO);
  return 0;
}

int_4 pic_dump_send (PICSTRUCT *p, int_4 isfp) {
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  int_4 i,n=16,lval[16];
  jvm_rwmem (p,pm,SEND_BASE,lval,-64,FLG_NIO);
  for (i=0; i<n; i++) printf("NIO Send Buf i=%02x d=%08x\n",i,lval[i]);
  return 0;
}

int_8 pic_arp_qsfp (PICSTRUCT *p, PKTPlan *plan, int_4 isfp, int_4 ip) {
  int_4 i,n=0,addr=plan->mcip,cb[128],lval[2],ires; int_u8 xval[2]; int_8 eres=-1;
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  plan->mcip=ip;	/* in case this is just the gatewaay address */
  n = PKT_ipeth(plan,cb,0,0,ARP_PAD,PROTO_ARP);	/* forms ARP request packet */
  if (n*4 != ARP_SIZE) printf("ARP padded size=%d not right=%d\n",n*4,ARP_SIZE);
  jvm_rwmem (p,pm,SEND_BASE,cb,ARP_SIZE,FLG_NIO);
  do_nio_command(p,isfp,0x3);	/* Send */
  for (i=0; i<100; i++) {
    jvm_rwmem (p,pm,SEND_BASE,lval,-4,FLG_NIO);
    if (lval[0]==0x02000406) break;
    pic_relock(p,10);
  }
  if (lval[0]==0x02000406) { 
    jvm_rwmem (p,pm,SEND_BASE+4,(int_4*)xval,-16,FLG_NIO);
    ires = swapit( (int_4)((xval[0]>>48)|(xval[1]<<16)) );
    eres = swaplt( xval[0]<<16 );
    vprint("ARP response isfp=%d myip=%x mcip=%08x ires=%08x eres=%012lx wait=%d\n",isfp,plan->myip,plan->mcip,ires,eres,i);
  }
  plan->mcip=addr;
  return eres;
}

int_8 ip2eth (PICSTRUCT *p, PKTPlan *pp, int_4 isfp) {
  int_u4 myip=pp->myip, ip=pp->mcip; int_8 eth=pp->edes;
  eth = getARPtable(p,ip);
  if (eth>0) { vprint("ArpTable IP=%08x ETH=%012lx\n",ip,eth); return eth; }
  ip = getGateway(p,isfp,myip,ip);
  eth = pic_arp_qsfp (p,pp,isfp,ip);
  if (eth<0) { printf("Unable to ARP ip=%08x from ip=%08x on isfp=%d\n",ip,pp->myip,isfp); return -1; }
  return eth;
}

int_8 pic_icmp_qsfp (PICSTRUCT *p, PKTPlan *plan, int_4 isfp) {
  int_4 i,n=0,addr,cb[128],lval[2],ires; int_u8 xval[2]; int_8 eres=-1;
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  n = PKT_ipeth(plan,cb,0,0,ICMP_PAD,PROTO_ICMP);	/* forms ICMP request packet */
  if (n*4 != ICMP_SIZE) printf("ICMP padded size=%d not right=%d\n",n*4,ICMP_SIZE);
  jvm_rwmem (p,pm,SEND_BASE,cb,ICMP_SIZE,FLG_NIO);
  do_nio_command(p,isfp,0x3);	/* Send */
  for (i=0; i<100; i++) {
    jvm_rwmem (p,pm,SEND_BASE+4,lval,-4,FLG_NIO);
    if (lval[0]==0x0100ce01) return i*10;
    pic_relock(p,10);
  }
  return -1;
}

int_4 pic_send_qsfp (PICSTRUCT *p, int_4 isfp, int_4 ip, int_4 mode) {
  PKTPlan pp; int_4 time,ls,n; char s[40];
  if (isfp==0) isfp=1;
  ls = findstrflagx("IPADDRx",p->config,s,1,isfp);
  if (ls<0) { printf("Unable to determine ip=%08x for qsfp on isfp=%d\n",pp.mcip,isfp); return -1; }
  pp.myip = parseIPaddr(s);
  pp.mcip = swap(ip); pp.edes = 0;	/* str2ip is BE, and make pktlib do the right thing */
  if (mode==0) {
    pp.edes = ip2eth(p,&pp,isfp);
    time = pic_icmp_qsfp (p,&pp,isfp);
    if (time<0) printf("Ping unable to ICMP ip=%08x eth=%012lx on isfp=%d\n",pp.mcip,pp.edes,isfp);
    else printf("Ping of ip=%08x eth=%012lx from ip=%08x on isfp=%d took %d ms\n",pp.mcip,pp.edes,pp.myip,isfp,time);
  }
  else if (mode==1 || mode==2) {
    pic_igmp_qsfp (p,&pp,isfp);
    do_nio_command(p,isfp,mode);
  }
  return time;
}

static int_4 vctl_offset;
static int_4 vctl_sequence;

int_4 pic_vctl_qsfp (PICSTRUCT *p, int dmac, PKTPlan *pp, int_4 isfp) {
  DMASTRUCT *dma = p->dma+(dmac-1);
  int_4 i,n,nh,addr,cb[128];
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  int_4 sid = (pp->chan>0)? pp->chan : 0;
  n = nh = PKT_ipeth(pp,cb,0,VCTL_SIZE,0,PROTO_UDP);	/* forms VCTL response packet */
  vctl_offset = nh*4;
  vctl_sequence = 0;
  cb[n++] = 0x68600013;	/* VRT context header 14w total */
  cb[n++] = sid;	/* Stream ID */
  cb[n++] = OUID_DIFI;	/* DIFI Org ID */
  cb[n++] = 0x00030006;	/* Packet and Class codes */
  cb[n++] = 0;		/* Time Stamp Integer Seconds */
  cb[n++] = 0;		/* Time Stamp Fractional Seconds */
  cb[n++] = 0;		/* Time Stamp Fractional Seconds */
  cb[n++] = 0xA1081000;	/* Poulated Fields bitmask */
  cb[n++] = 0;		/* messageID - sequence count */
  cb[n++] = 0x00000000;	/* controllee ID */
  cb[n++] = 0x00000000;	/* controller ID */
  cb[n++] = 0x40100002;	/* Poulated Fields bitmask */
  cb[n++] = 0x00000002;	/* Poulated Fields bitmask */
  cb[n++] = 0x00000064;	/* Reference Point */
  cb[n++] = 0x00000000;	/* tsfempto */
  cb[n++] = 0x00000000;	/* tsfempto */
  cb[n++] = 0x00000000;	/* bufSize upper - not set yet */
  cb[n++] = 0x00000000;	/* bufSize lower - not set yet */
  cb[n++] = 0x00000000;	/* bufFill */
  for (i=nh; i<n; i++) cb[i]=swap(cb[i]);
  vprint("Enabling Vita flow control for isfp=%d at McIP=%08x\n",isfp,pp->mcip);
  addr = VCTL_BASE + ichn*VCTL_SEP;
  jvm_rwmem (p,pm,addr,cb,VCTL_SEP,FLG_NIO);
  p->vctlTime = pic_time();
  return 0;
}

int_4 pic_send_vctl (PICSTRUCT *p, int_4 dmac, double fill, double time) {
  DMASTRUCT *dma = p->dma+(dmac-1);
  int_4 dmapkt = p->dmaPkt[dmac-1];
  int_4 isfp = (dmapkt>>0)&0xF;
  int_4 inst = (dmapkt>>8)&0xF;
  int_4 pkty = (dmapkt>>16)&0xF;
  int_4 muxgbe = (dmapkt>>24)&0xF;
  int_4 pm   = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  int_4 addr = VCTL_BASE + ichn*VCTL_SEP + vctl_offset;
  int_4 bufsize_u = dma->hsize>>24;
  int_4 bufsize_l = dma->hsize<<8;
  int_4 ifill,jfill,isec;
  int_8 psec;
  ifill = 0xFFFF*fill; 
  jfill = (ifill==0)? 0x3 : (ifill<0x0100)? 0x2 : (ifill>=0xFFFF)? 0xC : (ifill>=0xFE00)? 0x4 : 0;
  ifill = ifill&0xFFF0;
  p->vctlTime = time;
  time -= 631152000.0;	/* now J1950 to J1970 */
  isec  = time;
  psec  = (time-isec)*1e12;
  jvm_wmem (p,pm,addr+(4<<2),swap(isec),FLG_NIO);
  jvm_wmem (p,pm,addr+(5<<2),swap(psec>>32),FLG_NIO);
  jvm_wmem (p,pm,addr+(6<<2),swap(psec),FLG_NIO);
  jvm_wmem (p,pm,addr+(8<<2),swap(vctl_sequence),FLG_NIO);
  jvm_wmem (p,pm,addr+(16<<2),swap(bufsize_u),FLG_NIO);
  jvm_wmem (p,pm,addr+(17<<2),swap(bufsize_l),FLG_NIO);
  jvm_wmem (p,pm,addr+(18<<2),swap(ifill|jfill),FLG_NIO);
  do_nio_command(p,isfp,0x4);	/* Update */
  vctl_sequence++;
  return 0;
}

int_4 pic_update_vctx (PICSTRUCT *p, int_4 dmac, int_4 key, double dval) {
  int_4 i,j,nioctl,iadr,pm,mg,ichn,inst,lval[2],nc=0,field,noff,ioff; int_8 xval;
  if (dmac<1 || dmac>16) return 0;
  nioctl = p->nioctl[dmac-1];
  if (nioctl==0 || p->pktnctxoff==0) return 0;

  if (key==KEY_RATE) {
    field = VCTX_SR;
    nc = vctx_field (p->vctx,field,lval,(int_8)(1.0e6*dval));
  }
  else if (key==KEY_RFFREQ) {
    field = VCTX_RF;
    nc = vctx_field (p->vctx,field,lval,(int_8)(1.0e6*dval));
  }
  else if (key==KEY_TEMP) {
    field = VCTX_TEMP;
    nc = vctx_field (p->vctx,field,lval,(int_8)dval);
  }
  else if (key==KEY_RFGAIN || key==KEY_MGAIN || key==KEY_GAIN) {
    field = VCTX_GAIN;
    nc = vctx_field (p->vctx,field,lval,(int_8)dval);
  }
  if (nc==0) return 0;
  noff = ((p->vctx>>VCTX_NOCID)&1)? 3:5;	/* offset to context data */
  ioff = noff + vctx_field_offset (p->vctx,field);
  pm   = (nioctl>>12)&0xF;
  mg   = (nioctl>>8)&0xF;		/* though CTX packets are only on root port */
  ichn = (nioctl>>4)&0xF;
  inst = (nioctl>>0)&0xF;
  iadr = PPC_NIO | ((ichn)<<12) | (inst<<10) | ((p->pktnctxoff+ioff)<<2);
  for (i=0; i<nc; i++) pic_wpb (p,pm,iadr+(i<<2),swapit(lval[i]));	/* download sequence */
  if ((p->vctx&1)==0) return 0;	/* Disabled onChange bit */
  iadr = PPC_NIO | (ichn<<12) | 0x1;	// wrena
  pic_wpb (p,pm,iadr,0x21);	/* kick the CALT bit */
  v2print("Kicking off a context packet for key=%d at iadr=%08x\n",key,iadr);
  return 0;
}

int_4 pic_setup_qsfp (PICSTRUCT *p, int_4 dmac, int_4 dir, int_4 bits, int_4 rate, int_4 flags) {
  PKTPlan pp;
  int_4 i,j,k,nio,pm,iadr,n,ichn,nchn,ndup,ls,dmap,isfp,inst,iside,lform,nc,cb[1024],chan,flgs=0,ttup,flg,osel,tbp,route,auxf,vcmax;
  int_4 ethhdrlen,pkthdrlen,muxgbe,muxgbesip,myip; int_8 maxbitrate,bitrate; 
  char *cform = (char*)(&lform), *cosel = (char*)(&osel);
  char s[40],*sc;

  nio = (flags&FLG_NIO)? 1 : 0;
  bitrate = ((int_8)bits)*rate; if (bitrate<0) bitrate *= -2;

  cform[0] = (bits<0)? 'C':'S';
  bits = abs(bits);
  cform[1] = (bits<=8)? 'B' : (bits<=12)? 'J' : (bits<=16)? 'I' : 'F';
  tbp = p->pindex%10;

  pp.pkty = (p->pktmode>0)? p->pktmode : (dir>0)? PKT_ICE : PKT_AUTO;
  pp.size = findintflagdef("PKTLEN",p->config,1024);

  /* calculate final bit rate */
  pkthdrlen = (pp.pkty==PKT_ICE)? 64:32;
  ethhdrlen = 66;
  bitrate *= (double)(pp.size+pkthdrlen+ethhdrlen)/pp.size;
  maxbitrate = (int_8)(finddblflagdef("MAXGBERATE",p->config,10.0) * 1e9);

  /* determine necessary GBE muxing for final bit rate */
  muxgbe = (bitrate>maxbitrate*2)? 4 : (bitrate>maxbitrate)? 2 : 1;
  muxgbe = findintflagdef("MUXGBE",p->config,muxgbe);
  muxgbesip = findintflagdef("MUXGBESIP",p->config,0);
  auxf = (muxgbe==4)? 2 : (muxgbe==2)? 1 : 0;
  if (auxf==0); 
  else if (pp.size==1024) auxf |= 0x0; else if (pp.size==2048) auxf |= 0x4;
  else if (pp.size==4096) auxf |= 0x8; else if (pp.size==8192) auxf |= 0xC;
  else printf("Illegal packet size=%d for MUXGBE mode=%d. Must be 1K|2K|4K|8K.\n",pp.size,muxgbe);
  vcmax = findintflagdef("VCMAX",p->config,4);
  if (vcmax==3 || vcmax>4) vcmax=4;

  pp.ouid = (p->ouid>0)? p->ouid:OUID_ICE;
  switch (pp.pkty) {
    case PKT_SDDS: flgs |= (PKTFLG_BIGE); break;
    case PKT_VRT:  flgs |= (PKTFLG_BIGE); break;
    case PKT_VRTL: flgs |= (PKTFLG_BIGE|PKTFLG_NOSID|PKTFLG_TRIM); break;
    case PKT_VRTW: flgs |= (PKTFLG_BIGE|PKTFLG_NOSID|PKTFLG_TRIM); break;
    case PKT_VRTX: flgs |= (PKTFLG_BIGE|PKTFLG_NOCID); break;
    case PKT_VRTD: flgs |= (PKTFLG_BIGE|PKTFLG_TRIM); pp.ouid=OUID_DIFI; break;
  }
  p->vctx = findintflagdef("VCTX",p->config,(pp.pkty==PKT_VRTX)? 1:0);
  if (p->vctx==1) p->vctx=(pp.pkty==PKT_VRTX)? 0x1010592 : 0x000513;	/* default AST/ICE fields */
  if (findintflag("DALT",p->config)>0) flgs |= PKTFLG_DALT;
  if (p->tcmode!=0) flgs |= PKTFLG_TC;

  chan = (dir>0)?1:-1;

  pp.muxm = muxgbe;
  pp.form = lform;
  pp.myip = 0;
  pp.mcip = -1;
  pp.port = findintflagdef("PKTPORT",p->config,(pp.pkty==PKT_SDDS)? 29495:7000);
  pp.temp = ptemp(p,0);
  pp.rate = rate;
  pp.freq = (int_8)(1e6*finddblflagdef("RFFREQ",p->config,0.0));
  pp.bw   = (int_8)(1e6*finddblflagdef("RFBW",p->config,0.0));
  pp.gain = findintflagdef("RFGAIN",p->config,0);
  pp.flgs = findintflagdef("PKTFLG",p->config,flgs);
  pp.wsec = (int_u4)(p->tc_wsec-631152000);	/* Vita is J1970, Midas is J1950 */
  pp.qsec = (int_u4)(p->tc_fsec*4e9+.5);		/* 250pS fractional */
  pp.ncps = (p->fttm>=0 && p->mcs==0)? p->m1 : 1;
  pp.ncps = findintflagdef("NCPS",p->config,pp.ncps);
  pp.nfps = findintflagdef("NFPS",p->config,1);
  pp.ncpa = findintflagdef("NCPA",p->config,1);

  if (findintflag("MTGO",p->config)>0) {
    pp.wsec++;
    pp.qsec=0;
    if (findintflag("MTGOOFF",p->config)>0) pp.wsec++;
  }
  pp.wsec += findintflagdef("NIOTCOFF",p->config,0);

  nchn = findintflagdef("NIOC",p->config,pp.nfps);
  ndup = findintflagdef("NIODUP",p->config,0);

  p->niochns = nchn;

for (n=0; n<nchn; n++) {

  isfp = (ndup>0)? n+1 : (dmac<=0)? 1 : p->dma[dmac-1].port;
  isfp = findintflagdef("ISFP",p->config,isfp);
  inst = 0;
  ls = findstrflagx("NIOAx",p->config,s,1,n+1);
  if (ls>0) {
    if (s[1]=='.' && s[3]=='/') {
      isfp = s[0]-'0';
      inst = s[2]-'0'-1;
      pp.mcip = parseIPaddr(s+4);
    } else {
      pp.mcip = parseIPaddr(s);
    } 
    pp.port = parseIPport(s,pp.port);
  } else {
    ls = findstrflagx((dir>0)?"IPDESTx":"IPCONNx",p->config,s,1,isfp);
    if (ls>0) { pp.mcip = parseIPaddr(s); pp.port = parseIPport(s,pp.port); }
  }
  if (pp.ncps>1 && nchn==2) isfp += n*2;
  if (pp.ncps>1 && nchn==4) isfp += n;
  if (pp.ncps>1) pp.mcip += n;

  pm   = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;

  iadr = (dir>0)? 0x20004 : 0x20008;	/* MUXGBE register */
  pic_wpb (p,pm,PPC_NIO|iadr,auxf);
  if (auxf!=0) printf("Using MuxGBE=%d maxbitrate=%lld for dir=%d bitrate=%lld pktlen=%d auxf=%1x isfp=%d.\n",muxgbe,maxbitrate*muxgbe,dir,bitrate,pp.size,auxf,isfp);
  iadr = 0x20020;	/* Virtual Channel MAX register */
  pic_wpb (p,pm,PPC_NIO|iadr,vcmax);

  ls = findstrflagx("IPADDRx",p->config,s,1,isfp);
  if (ls>0) { pp.myip = parseIPaddr(s); }

  ls = findstrflagx("SIDx",p->config,s,1,n+1);
  if (ls>0) chan = parseSID(s);

  ls = findstrflagx("SRx",p->config,s,1,n+1);
  if (ls>0) pp.rate = parseRF(s);

  ls = findstrflagx("RFx",p->config,s,1,n+1);
  if (ls>0) pp.freq = parseRF(s);

  ls = findstrflagx("BWx",p->config,s,1,n+1);
  if (ls>0) pp.bw = parseRF(s);

  if (pp.ncps>1) chan=n; /* handle case nfps>1 */
  pp.chan = findintflagdef("PKTCHN",p->config,chan);

  /* from DMA channel to DMA index */
  dmap = dmac-1;
  iside = dmap&0x1;
       if (dmap==8||dmap==9) dmap=6;
  else if (dmap==10||dmap==11) dmap=7;
  else if (dmap>=12) dmap-=12;
 
  p->dmaPkt[dmac-1] = (muxgbe<<24) | ((pp.pkty&0xF)<<16) | (inst<<8) | isfp;
  p->niochntbl[n] = (isfp<<4) | inst;
  /* Stream input - setup PKT2DAT core - or disable it from DMA_CANCEL */ 
  if ((flags&FLG_NIO)==0 && dir<0) {	
    flg = findintflagdef("PKT2DATFLAGS",p->config,0);
    if (findintflag("PKTASIS",p->config)>0) flg|=0x01;
    if (findintflag("NIOLOOP",p->config)>0) flg|=0x0A;	/* GateWay|AllEth */
    if (findintflag("PKTGATE",p->config)>0) flg|=0x0A;	/* GateWay|AllEth */
    i = findintflag("PKTTEST",p->config); if (i>0) { flg|=0x200; flg|=(i<<12); }
    if (p->ouid<0)   flg|=0x400;	/* Allow any OUID */
    if (pp.mcip==-1) flg|=0x10; 
    if (pp.chan==-1) flg|=0x20;
    if (flags&FLG_DISABLE) pp.mcip=-1;
   for (k=0; k<muxgbe; k++) {
    iadr = PPC_NIO|0x8102 | ((ichn+k)<<12);	/* PKT2DAT parameter address */
    osel = dmap; 				/* icetge's osel[6:4] per instance */
    vprint("QSFP dmac=%d dmap=%d dir=%d pkty=%d form=%.2s isfp=%d iadr=%08x ip=%08x chan=%04x inst=%d osel=%x flags=%08x pm=%d\n",
		    dmac,dmap,dir,pp.pkty,cform,isfp,iadr,pp.mcip,pp.chan,inst+1,osel,flg,pm);
    pic_wpb (p,pm,iadr+0x00,0);		/* pp.pkty temp disable */
    pic_wpb (p,pm,iadr+0x04,pp.myip);
    pic_wpb (p,pm,iadr+0x08,pp.mcip);
    if (flags&FLG_DISABLE) continue;
    pic_wpb (p,pm,iadr+0x0C,pp.chan);
    pic_wpb (p,pm,iadr+0x10,flg);
    pic_wpb (p,pm,iadr+0x14,osel);	/* osel[3:0] get << 4 in icetge for DMA controller logic */
    pic_wpb (p,pm,iadr+0x20,0);		/* ic */
    pic_wpb (p,pm,iadr+0x24,0);		/* nc */
    loadArray (p,pm,iadr+0x28, inst, pp.mcip);
    loadArray (p,pm,iadr+0x2C, inst, pp.chan);
    loadArray (p,pm,iadr+0x30, inst, osel);
    pic_wpb (p,pm,iadr+0x38,0);		/* nseq */
    pic_wpb (p,pm,iadr+0x3C,pp.ouid);	/* ouid */
    pic_igmp_qsfp (p,&pp,isfp+k);
    if (pm>0) {
      flg = pic_rpb (p,pm,PPC_DMAC_ROUTE_FLG);
      pic_wpb (p,pm,PPC_DMAC_ROUTE_FLG,flg|PPC_ROUTF_NIO2HX);
    } else {
      flg = pic_acmd (p, ADON_RD, DMA_ROUTF, 0, 0);
      pic_acmd (p, ADON_WR, DMA_ROUTF, flg|PPC_ROUTF_NIO2M, 0);
      p->dma[dmac-1].reg = PPC_OSTAT(PM_PROC);
      p->dma[dmac-1].enb |= PPC_ROUTE_PR2R;
    }
    ls = findstrflagx("NIOACTLx",p->config,s,1,n+1);
    if (ls<0) ls = findstrflagx("VFCMA",p->config,s,1,n+1);
    if (ls>0) {	sc=s;			/* enable Vita flow control packets */
      if      (sc[0]=='+') { sc++; pp.mcip += parseIPaddr(sc); }
      else if (sc[0]=='-') { sc++; pp.mcip -= parseIPaddr(sc); }
      else                         pp.mcip  = parseIPaddr(sc);
      pp.port = parseIPport(sc,pp.port);
      pic_vctl_qsfp (p,dmac,&pp,isfp);
    }
   }
    continue;
  }

  /* resolve UniCast Address */
  pp.edes=0;
  ttup = (pp.mcip>>24)&0xFF;
  if (ttup<224 || ttup>240) { 
    pp.edes = ip2eth(p,&pp,isfp);
    if (pp.edes<=0) printf("Err: No response to ARP for unicast address=%08x. Run $ICEROOT/cfg/mkarptbl and add entry to /var/etc/arptbl. Defaulting.\n",pp.mcip);
    if (pp.edes<=0) pp.edes = (0x1CE000L<<24)|pp.mcip;
    p->dmaPkt[dmac-1] |= (1<<28); /* isUniCast bit */
  }

  /* Stream or NIO output - setup dat2pkt sequencer */
  if (dmac<=16) p->nioctl[dmac-1] = (pm<<12)|(muxgbe<<8)|(ichn<<4)|(inst<<0);
  myip = pp.myip;
  for (k=0; k<muxgbe; k++) {
    pp.indx = k;
    if (k>0 && muxgbesip==0) { ls=findstrflagx("IPADDRx",p->config,s,1,isfp+k); if (ls>0) pp.myip=parseIPaddr(s); }
    if (pp.pkty==PKT_ETH) { nc=1; cb[0]=0x02; }			/* enable gated ETH mode */
    else nc = PKT_cblock(p,&pp,cb);
    iadr = PPC_NIO | ((ichn+k)<<12) | (inst<<10);
    pic_wpb (p,pm,iadr,0x0);
    for (i=1; i<nc; i++) pic_wpb (p,pm,iadr+(i<<2),cb[i]);	/* download sequence */
    j=cb[0];				/* enable word */
    if (dir<0) j |= 0x10;		/* feedthru bit */
    if (findintflag("PKTDBG",p->config)>0) j |= 0x80;
    iadr = PPC_NIO | ((ichn+k)<<12);	/* sysreg is inst=0 only */
    pic_wpb (p,pm,iadr,j);
  }
  pp.myip = myip;

  /* setup response channel */
  iadr = PPC_NIO|0x8102 | (ichn<<12);
  if (dir<0) pic_wpb (p,pm,iadr+0x08,pp.mcip);

  /* from DMA channel to DMA port 0=PPC,1=Host,2=HA,3=HB,4=CA,5=CB,6=TA,7=TB */
  dmap = dmac+1;
  if (dmap==10||dmap==11) dmap=6;
  else if (dmap==12||dmap==13) dmap=7;
  else while (dmap>7) dmap-=2;

  vprint("QSFP dmac=%d dmap=%d isfp=%d inst=%d dir=%d pkty=%d form=%.2s rate=%d nc=%d myip=%08x mcip=%08x sys=%x pm=%d\n",
		dmac,dmap,isfp,inst+1,dir,pp.pkty,cform,rate,nc,pp.myip,pp.mcip,j,pm);

  if (p->dsgtype==DSG_TRAY && pm>0 && dir>0) {
    flg = pic_rpb (p,pm,PPC_DMAC_ROUTE_FLG);
    route = 0;
    if (pm==1) route |= ((dmac==1)? PPC_ROUTE_HA2TA : (dmac==2)? PPC_ROUTE_HB2TA : (dmac==3)? PPC_ROUTE_CA2TA : (dmac==4)? PPC_ROUTE_CB2TA : 0);
    if (pm==2) route |= ((dmac==1)? PPC_ROUTE_HA2TB : (dmac==2)? PPC_ROUTE_HB2TB : (dmac==3)? PPC_ROUTE_CA2TB : (dmac==4)? PPC_ROUTE_CB2TB : 0);
    pic_wpb (p,pm,PPC_DMAC_ROUTE_FLG,flg|PPC_ROUTF_HX2NIO);
    p->dma[dmac-1].enb |= route;
  }
  if ((flags&FLG_NIO)==0 && dir>0) {			/* NIO Output */
    iadr = PPC_NIO|0x10000; 
    j = 0x10 << (ichn>>1); j = (j<<24)|(j<<16);		/* has to assume adjacent sfp pairs map to IO module */
    j = findintflagdef("NIOSYSREG",p->config,j);
    pic_wpb (p,pm,iadr,j);
    continue;
  } 
  for (i=0; i<8; i++) {					/* for each of 8 potential MCS channels */
    if (ndup==0 && nchn>1 && (i%nchn)!=n) continue;	/* not this individual channel */
    if (p->mcs==0 && p->fttm>0 && tbp!=3 && (i%2)!=(tbp-1)) continue;	/* not this tbank channel */
    if (pm>0) dmap = 2+iside;
    j = ((dmap<<3)|i)<<8; 				/* address of channel */
    k = (0x10<<ichn) | (iside<<3) | (inst-0);		/* select for channel */
    if (ndup>0) k |= 0xF0;
    pic_wpb (p,pm,PPC_DMAC_ROUTE_NIO,j|k);
    vprint("Set NIO ROUTE i=%d chn=%d:%d dma=%d:%d j=%04x k=%02x pindex=%d\n",i,n,nchn,dmac,dmap,j,k,p->pindex);
  }
  chan++;
}

  return 0;
}

int_4 pic_qsfp_stats (PICSTRUCT *p, int_4 isfp, char *str, int_4 flags) {
  int_4 pm,ichn,iadr,jadr,pkty,myip,mcip,cnts,cpkt,ipkt,opkt,cseq,ccmp,cgmp,carp,coth,csnd;
  int_u1 *mcip_ = (int_u1*)(&mcip), *myip_ = (int_u1*)(&myip);
  char xips[40];
  pm   = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  iadr = PPC_NIO|0x08100|(ichn<<12);
  jadr = PPC_NIO|0x20000|(ichn<<12);	/* opkts register */
  if (flags==0) {
    pic_wpb (p,pm,iadr+0x1C+2,0);
    pic_wpb (p,pm,iadr+0x34+2,0);
    pic_wpb (p,pm,jadr+0x10,0);
  }
  pkty = pic_rpb (p,pm,iadr+0x00);
  myip = pic_rpb (p,pm,iadr+0x04);
  mcip = pic_rpb (p,pm,iadr+0x08);
  ipkt = pic_rpb (p,pm,iadr+0x1C);
  cnts = pic_rpb (p,pm,iadr+0x34);
  cpkt = (cnts>> 0)&0xFFF;
  cseq = (cnts>>12)&0xF;
  ccmp = (cnts>>16)&0xF;
  cgmp = (cnts>>20)&0xF;
  carp = (cnts>>24)&0xF;
  coth = (cnts>>28)&0xF;
  opkt = pic_rpb (p,pm,jadr);
//  csnd = jvm_rmem (p,pm,0x60+(ichn<<2),FLG_NIO);
  if (flags==16) {
    sprintf(str,"QSFP-%d pkty=%d myip=%08x xip=%08x cpkt=%03x icmp=%01x igmp=%01x arp=%01x oth=%01x ipkt=%08x opkt=%08x",
			isfp,pkty,myip,mcip, cpkt,ccmp,cgmp,carp,coth, ipkt,opkt);
  } else {
    if (mcip>=-2 && mcip<=0) sprintf(xips,"%d",mcip);
    else sprintf(xips,"%d.%d.%d.%d",mcip_[3],mcip_[2],mcip_[1],mcip_[0]);
    sprintf(str,"QSFP-%d pkty=%d myip=%d.%d.%d.%d xip=%s cpkt=%d icmp=%d igmp=%d arp=%d oth=%d ipkt=%d opkt=%d",
			isfp,pkty, myip_[3],myip_[2],myip_[1],myip_[0], xips, cpkt,ccmp,cgmp,carp,coth, ipkt,opkt);
  }
  return strlen(str);
}

int_4 pic_dump_qsfps (PICSTRUCT *p, int_4 flags) {
  int_4 ls,isfp,ichn,pm,lval[12]; char str[128];
  for (isfp=1; isfp<=p->niop; isfp++) {
    pm   = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
    ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
    if (flags==0) { pic_qsfp_stats(p,isfp,str,0); continue; }
    if (ichn==0) {
      jvm_rwmem (p,pm,0x00,lval,-32,FLG_NIO);
      printf("NIOJVM pm=%d   Cmd=%08x   FPass=%08x FArg=%08x   BStat=%08x BArg=%08x BCnt=%08x\n",
		     pm,      lval[1],     lval[2],  lval[3],     lval[5],  lval[6],  lval[7]);
      if (p->verbose) {
        jvm_rwmem (p,pm,0x100,lval,-48,FLG_NIO);
        printf("NIOJVM (%08x %08x %08x %08x  %08x %08x %08x %08x  %08x %08x %08x %08x)\n",
		lval[0],lval[1],lval[2],lval[3],lval[4],lval[5],lval[6],lval[7],lval[8],lval[9],lval[10],lval[11]);
      }
    }
    ls = pic_qsfp_stats(p,isfp,str,0x10);
    printf("%s\n",str);
  }
  return 0;
}

int_4 pic_enable_qsfp (PICSTRUCT *p, int_4 dmac, int_4 ena) {
  int_4 dmapkt = p->dmaPkt[dmac-1];
  int_4 isfp = (dmapkt>>0)&0xF;
  int_4 inst = (dmapkt>>8)&0xF;
  int_4 pkty = (dmapkt>>16)&0xF;
  int_4 muxgbe = (dmapkt>>24)&0xF;
  int_4 isUC = (dmapkt>>28)&0xF;
  int_4 pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  int_4 ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  int_4 j,k,nc,iadr;
  if (pkty==0xF) pkty=-1;
for (nc=0; nc<p->niochns; nc++) {
  if (p->niochns>1) { j=p->niochntbl[nc]; isfp=j>>4; inst=j&0xF; }
for (k=0; k<muxgbe; k++) {
  iadr = PPC_NIO|0x8102|((ichn+k)<<12);
  if (ena>0) { 
    if (isUC==0) do_nio_command(p,isfp+k,0x1);	/* Join if not UniCast */
    pic_wpb (p,pm,iadr+0x00,pkty);	/* pkty=!none enables */
    p->joinTime = pic_time();
  } else {
    if (isUC==0) do_nio_command(p,isfp+k,0x2);	/* Leave if not UniCast */
    pic_wpb (p,pm,iadr+0x00,0);		/* pkty=none */
    pic_wpb (p,pm,iadr+0x10,0);		/* flags=0 */
    p->joinTime = 0.0;
  }
}
}
  return 0;
}

int_4 pic_ifup_qsfp (PICSTRUCT *p, int_4 nsfp, int_4 updn) {
  int_4 gw,flg,ls,iadr,isfp,ichn,pm,myip=-1; char s[40];
  for (isfp=1; isfp<=nsfp; isfp++) {
    pm = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
    gw = (findintflag("NIOLOOP",p->config)>0 && pm==1);	/* GateWay|AllEth */
    ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
    if (ichn==0) {
      flg = findintflagdef("RXRAWBITS",p->config,0);
      pic_wpb (p,pm,PPC_NIO|0x10000,flg);
      pic_wpb (p,pm,PPC_NIO|0x2000C,0);
    }
    if (updn) {
      iadr = PPC_NIO|0x8000|(ichn<<12);
      pic_wpb (p,pm,iadr,1);		/* enable last config */
    } else {
      flg = gw? 0xA:0;
      ls = findstrflagx("IPADDRx",p->config,s,1,isfp);
      if (ls>0) myip = parseIPaddr(s);
      iadr = PPC_NIO|0x8000|(ichn<<12);	/* core enable reg */
      pic_wpb (p,pm,iadr,0);		/* configure when down */
      iadr = PPC_NIO|0x8102|(ichn<<12);	/* core param regs */
      pic_wpb (p,pm,iadr+0x00, gw? 1:0);	/* pkty=0 */
      pic_wpb (p,pm,iadr+0x04,myip); /* myip */
      pic_wpb (p,pm,iadr+0x08,-1); /* mcip */
      pic_wpb (p,pm,iadr+0x0C,-1); /* mcid */
      pic_wpb (p,pm,iadr+0x10, flg); /* flag */
      pic_wpb (p,pm,iadr+0x14, 0); /* osel */
      pic_wpb (p,pm,iadr+0x18, 0); /* ipkt */
      pic_wpb (p,pm,iadr+0x1C, 0); /* opkt */
      pic_wpb (p,pm,iadr+0x34, 0); /* cnts */
      vprint("QSFP ifup isfp=%d myip=%08x flgs=%04x\n",isfp,myip,flg);
    }
  }
  return nsfp;
}

int_4 pic_tcupx_qsfp (PICSTRUCT *p, int_4 dmac, real_8 soy, real_8 fsoy) {
  DMASTRUCT *dma = p->dma+(dmac-1);
  int_4 dmapkt = p->dmaPkt[dmac-1];
  int_4 isfp = (dmapkt>>0)&0xF;
  int_4 inst = (dmapkt>>8)&0xF;
  int_4 pkty = (dmapkt>>16)&0xF;
  int_4 muxgbe = (dmapkt>>24)&0xF;
  int_8 tinc = p->regX;			/* stored here for now */
  int_4 i,j,k,n,nc,cb[8],ichn,iadr,pm;
  int_u4 wsec,qsec,psecu,psecl; int_8 psec;

  wsec = (int_u4)(soy-631152000);	/* Vita is J1970, Midas is J1950 */
  qsec = (int_u4)(fsoy*4e9+.5);
  psec = (int_8)qsec*250;
for (nc=0; nc<p->niochns; nc++) {
  if (p->niochns>1) { j=p->niochntbl[nc]; isfp=j>>4; inst=j&0xF; }
 for (k=0; k<muxgbe; k++) {
  ichn = (p->dsgtype>0)? (isfp-1)%4 : isfp-1;
  iadr = PPC_NIO | (ichn<<12) | (inst<<10);
  pm   = (p->dsgtype<DSG_TRAY)? 0 : (isfp>4)? 2 : 1;
  n=0;
  psecu = (int_4)(psec>>32);
  psecl = (int_4)(psec&B4GM);
  if (pkty>=PKT_VRT) {
    cb[n++] = swap(wsec);	/* time stamp int sec */
    cb[n++] = swap(psecu);	/* time stamp frac sec upper */
    cb[n++] = swap(psecl);	/* time stamp frac sec lower */
  }
  for (i=0; i<n; i++) pic_wpb (p,pm,iadr+((p->pktntoff+i)<<2),cb[i]);	/* updated TC initial fields */
  psec += tinc; if (psec>=1e12) { wsec++; psec-=1e12; }
  isfp++;
 }
}
  return 0;
}

int_4 pic_tcup_qsfp (PICSTRUCT *p, int_4 port, int_4 stat, real_8 offset, real_8 delta, real_8 soy, real_8 fsoy) {
  int_u4 koff,kfoff,ksoy,kfsoy;
  int_4 i,n,na,nt,isfp,nfps,isfpl,inst,iadr,jadr,buf[32];
  int_4 ice_nt=24,sdds_nt=20,vrt_nt=25; 
  if (p->tc_wsec==soy) return 0;	/* just once per second */
  isfp = (port<=0)? 1 : (port&1)? 1 : 2;
  isfp = findintflagdef("ISFP",p->config,isfp);
  nfps = findintflagdef("NFPS",p->config,1);
  inst = 0;
  kfoff = ((int_8)(offset*B4G))&B4GM;
  koff  = ((int_8)offset)&B4GM;
  ksoy  = ((int_8)soy)&B4GM;
  kfsoy = ((int_8)(fsoy*B4G))&B4GM;
  if (kfoff>0) { kfoff=B4G-kfoff; koff-=1; }
  nt = (p->pktmode>=PKT_VRT)? vrt_nt : (p->pktmode==PKT_SDDS)? sdds_nt : ice_nt;
  i=0; 
  buf[i++]=stat;
  buf[i++]=kfoff;
  buf[i++]=koff;
  buf[i++]=ksoy;
  buf[i++]=kfsoy;
  pic_transfer (p,buf,i*4,DMA_INT_ADDR,1,1);
  for (n=0;n<nfps;n++) {
    iadr = PPC_NIO | ((isfp-1)<<12) | (inst<<10);
    if (p->pktmode==PKT_ICE) pic_acmd (p, ADON_TCUP, DMA_INT_ADDR, iadr+1, iadr+nt*4);
    isfp += ((nfps==2)?2:1);
  }
#if DEBUG
  printf("QSFP TCUP port=%d off=%f dt=%f soy=%f fsoy=%f  ko=%08x kfo=%08x ksoy=%08x kfsoy=%08x\n",
			port,offset,delta*1e9,soy,fsoy,koff,kfoff,ksoy,kfsoy);
#endif
  p->tc_wsec=soy; 
  p->tc_fsec=fsoy; 
  p->tc_soff=offset; 
  return 0;
}

int_4 pic_dump_qsfp_buf (PICSTRUCT *p, int_4 port, int_4 *data) {
  int_4 i,j,d,status;
  WR(REG_IOC, REG_IOC_ADDR|0x500|PPC_BADDR_WR);
  for (i=0; i<16; i+=8) WR(REG_IOC, 0x0000>>i);
  WR(REG_IOC, REG_IOC_ADDR|0x500|PPC_BDATA);
  for (j=0; j<32; j++) {
    for (d=i=0; i<32; i+=8) { RD(REG_IOC, status);  d |= ((status&0xFF)<<i); }
    data[j]=d;
  }
  return 0;
}

int_4 pic_dump_qsfp (PICSTRUCT *p, int_4 port) {
  int_4 i, data[32];
  pic_dump_qsfp_buf (p,port,data);
  for (i=0; i<32; i++) {
    if (i==0) printf("NIO JVM - ? ? pass arg   ? resync arg bad\n");
    if (i==8) printf("NIO JVM - ipkt cnt per chan\n");
    if (i==16) printf("NIO JVM - opkt cnt per chan\n");
    if (i==24) printf("NIO JVM - other \n");
    printf(" a=%d b=%04x d=%08x\n",i,i*4,data[i]);
  }
  return 0;
}

int_4 pic_test_qsfp (PICSTRUCT *p, int_4 port) {
  int_4 n, errors=0, status, np=4, nc=4, tmp[32], dmac, dir=-1, nbytes=1024*1024*4;
  DMAMAP map;

  printf("Testing QSFP config=%s\n",p->config);
  pic_reset (p,0);

  pic_dump_qsfp_buf (p,port,tmp);
  printf("NIO preDMA  port=%d  npkt1=%d npkt2=%d npkt3=%d npkt4=%d\n",port,tmp[8],tmp[9],tmp[10],tmp[11]);
  
  dmac = pic_ioport (p, IOPT_MODULE, 1, -1, dir, -16, 10000000, 0.0, 1, 0, 0);
  status = pic_mapmem (p, &map, nbytes, 1); 
  status = pic_dmasetup (p, dmac, dir, &map, -1, 0);
  status = pic_dmafunc (p, dmac, 1);
  status = pic_dmafunc (p, dmac, DMA_WAIT);
  printf("Snapshot of IIR status=%d\n",status);
  status = pic_dmafunc (p, dmac, DMA_CANCEL);
  status = pic_mapmem (p, &map, nbytes, -1);

  pic_dump_qsfp_buf (p,port,tmp);
  printf("NIO postDMA  port=%d  npkt1=%d npkt2=%d npkt3=%d npkt4=%d\n",port,tmp[8],tmp[9],tmp[10],tmp[11]);

  for (n=0; n<np; n++) if (tmp[n+8]!=4000) errors++;

  return errors;
}

