/****************************************************************************/
/*                                                                          */
/*   LB2DM3 module interface routines for the ICEPIC device target          */
/*                                                                          */
/****************************************************************************/

#include "iomlib_lb2dm3.h"            

/*---------------------------------------------------------------------------*/
/* Internal Private Interfaces                                               */
/*                                                                           */
/*   NOTE: These routines should NEVER be called by external libraries.      */
/*         The API to these routines is fluid and is only self-consistent    */
/*         within a release series.                                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/

int32_t lb2dm3_hbf_loadcoef(Module_Config_Obj *modcfg, int32_t part_ID,
                                 int32_t sub_ID, int32_t coeset)
{
  static const int32_t lb2dm3_hbfir_coefs0 [] = {
    -1,2,-3,5,-8,12,-17,23,-31,41,-54,69,-87,109,-135,165,-201,242,
    -291,347,-413,489,-579,684,-810,963,-1153,1394,-1715,2168,-2862,
    4087,-6902,20844
  };
  static const int32_t lb2dm3_hbfir_coefs1 [] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  };
  static const int32_t lb2dm3_hbfir_coefs2 [] = {
    0,1,-1,2,-3,5,-7,11,-16,22,-30,41,-55,71,-92,118,-148,185,-230,283,
    -346,420,-509,615,-743,898,-1092,1339,-1666,2126,-2828,4063,-6887,20839
  };

  int32_t i;
  int32_t size = sizeof(lb2dm3_hbfir_coefs0)/sizeof(lb2dm3_hbfir_coefs0[0]);

  for(i=0; i<size; ++i) {
    if(coeset==2){
      cmdcfg_ctl(modcfg,part_ID,sub_ID,PFUNC_RWR,i,
                  lb2dm3_hbfir_coefs2[i]&0xFFFF,0xFFFF);
    }
    else if(coeset==1){
      cmdcfg_ctl(modcfg,part_ID,sub_ID,PFUNC_RWR,i,
                  lb2dm3_hbfir_coefs1[i]&0xFFFF,0xFFFF);
    }
    else{
      cmdcfg_ctl(modcfg,part_ID,sub_ID,PFUNC_RWR,i,
                  lb2dm3_hbfir_coefs0[i]&0xFFFF,0xFFFF);
    }
  }
  return(0);
}

int32_t lb2dm3_cfir_loadcoef(Module_Config_Obj *modcfg, int32_t part_ID,
                                 int32_t sub_ID, float64_t srate)
{
  int32_t i;
  int32_t coefval, coefset;
  static const int32_t cfir_nfilt = 25;
  static const int32_t cfir_taps  =  7;
  static const int32_t lb2dm3_cfir_coefs[25][7] = {
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,   0,    0,     0,    0,     0, 32767},
    {  0,  -4,    3,   -27,   12,  -304, 32767},
    {  0,  -7,    7,   -42,   29,  -407, 32767},
    {  2,  -7,    7,   -42,   23,  -501, 32767},
    {  2,  -6,    6,   -43,   19,  -590, 32767},
    {  1,  -6,    4,   -49,   15,  -672, 32767},
    {  1,  -8,    5,   -59,   18,  -759, 32767},
    {  2,  -11,  13,   -75,   39,  -873, 32767},
    {  6,  -16,  27,   -92,   76, -1007, 32767},
    {  0,  -16,  16,  -107,   63, -1091, 32767},
    {  1,  -20,  25,  -129,   91, -1208, 32767},
    {  6,  -32,  46,  -165,  152, -1349, 32767},
    {  5,  -38,  61,  -204,  203, -1506, 32767},
    { 14,  -58, 101,  -270,  321, -1730, 32767},
    { 16,  -66, 122,  -316,  376, -1907, 32767},
    { 25,  -87, 166,  -385,  503, -2153, 32767},
    { 16,  -90, 184,  -470,  598, -2402, 32767},
    { 33, -128, 260,  -577,  797, -2705, 32767},
    { 25, -125, 270,  -650,  859, -3011, 32767},
    { 71, -190, 419,  -810, 1191, -3677, 32767},
    { 74, -239, 419,  -836, 1442, -4089, 32767},
    { 11, -168, 408, -1039, 1666, -4513, 32767},
    {-33,  -48, 237,  -956, 1708, -4781, 32767}
  };

  /* Choose equalization index (zero based) to nearest 5MHz sample rate */
  coefset = (int32_t) (0.5 + ((1.0/5.0)*srate));
  coefset -= 1;
  coefset = (coefset >= 0) ? (coefset) : (0);
  coefset = (coefset <= (cfir_nfilt-1)) ? (coefset) : (cfir_nfilt-1);

  for(i=0; i<cfir_taps; ++i) {
    coefval = lb2dm3_cfir_coefs[coefset][i];
    cmdcfg_ctl(modcfg,part_ID,sub_ID,PFUNC_RWR,i,(coefval&0xFFFF),0xFFFF);
  }

  cmdcfg_ctl(modcfg,part_ID,sub_ID,PFUNC_RWR,64,86408,0xFFFFFF);

  return(0);
}

int32_t lb2dm3_presel_decode(Module_Config_Obj *mco, uint32_t *psel0, 
                             uint32_t *psel1)
{
  int32_t pselrg; 
  uint32_t (*pseltbl_p)[2];
  enum LB2DM3_PRESEL_LIST {
    LB2DM3_PRESEL_AUTO      =  0, /* Auto select filter based on Fc     */
    LB2DM3_PRESEL_LOWBAND   =  1, /* Force Select lowband filter        */
    LB2DM3_PRESEL_MIDBAND   =  2, /* Force Select midband filter        */
    LB2DM3_PRESEL_HIGHBAND  =  3, /* Force Select highband filter       */
    LB2DM3_PRESEL_BYPASS    =  4, /* Force Bypass all filters           */
    LB2DM3_PRESEL_TERMINATE =  5  /* Force Terminate input (no signal)  */
  };
  uint32_t pseltbl[6][2] = { 
    {0x02,0x02},{0x00,0x03},{0x01,0x00},{0x00,0x01},{0x02,0x02},{0x03,0x00} 
  };
  uint32_t pseltblv3[6][2] = { 
/*    {0x01,0x02},{0x02,0x01},{0x00,0x03},{0x02,0x00},{0x01,0x02},{0x03,0x00}  broke */
    {0x01,0x02},{0x01,0x02},{0x00,0x03},{0x01,0x00},{0x02,0x01},{0x03,0x00} /* fixed */
  };
 pseltbl_p = ((mco->RT.hw_ver&0xF) == 0x7)? pseltblv3 : pseltbl;
  
  *psel0 = pseltbl_p[LB2DM3_PRESEL_BYPASS][0];
  *psel1 = pseltbl_p[LB2DM3_PRESEL_BYPASS][1];
  pselrg = (LB2DM3_TGLSS_PRESEL & mco->LS.u32[LB2DM3_RG_TGLSS]) ?
             (mco->LS.u32[LB2DM3_RG_PRESEL]) : (LB2DM3_PRESEL_BYPASS);

  switch (pselrg) {
      case (LB2DM3_PRESEL_LOWBAND):
        *psel0 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][0];
        *psel1 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][1];
        break;
      case (LB2DM3_PRESEL_MIDBAND):
        *psel0 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][0];
        *psel1 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][1];
        break;
      case (LB2DM3_PRESEL_HIGHBAND):
        *psel0 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][0];
        *psel1 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][1];
        break;
      case (LB2DM3_PRESEL_BYPASS):
        *psel0 = pseltbl_p[LB2DM3_PRESEL_BYPASS][0];
        *psel1 = pseltbl_p[LB2DM3_PRESEL_BYPASS][1];
        break;
      case (LB2DM3_PRESEL_TERMINATE):
        *psel0 = pseltbl_p[LB2DM3_PRESEL_TERMINATE][0];
        *psel1 = pseltbl_p[LB2DM3_PRESEL_TERMINATE][1];
        break;
      default:
        if (mco->LS.i32[LB2DM3_RG_RFFREQW] < 3000) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_HIGHBAND][1];
        }
        if (mco->LS.i32[LB2DM3_RG_RFFREQW] < 1705) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_MIDBAND][1];
        }
        if (mco->LS.i32[LB2DM3_RG_RFFREQW] < 1100) {
          *psel0 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][0];
          *psel1 = pseltbl_p[LB2DM3_PRESEL_LOWBAND][1];
        }
  }
  dbgprint(mco->dbg,DBG_LB,"-------PRESEL: %4.4i  0x%8.8X  0x%8.8X \n",
      pselrg,*psel0,*psel1);
  return (0);
}

int32_t lb2dm3_tglsw(Module_Config_Obj *mco, uint32_t tglswreg,  
                       uint32_t tglswmask)
{
  uint32_t pinreg = LB2DM3_PIN_NULL;
  uint32_t pinval = LB2DM3_PIN_NULL;
  uint32_t psel0,psel1;
  int32_t cmfunc,cmpin,cmaddr,cmdata;

  /* First, get current module state of toggle switches */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,CMD_RG_FIFO_RBCNT,0,0xFF);
  cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RRD,LB2DM3_RG_TGLSS,0,0xFFFFFFFF);
  cmdcfg_bufclr(mco);
  cmdata = 0x00000000;
  cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  mco->LS.i32[LB2DM3_RG_TGLSS] = cmdata;

  dbgprint(mco->dbg,DBG_LB,
    "lb2dm3_tglsw INP: LS: 0x%8.8X  TS: 0x%8.8X  TMSK: 0x%8.8X\n",
     mco->LS.u32[LB2DM3_RG_TGLSS],tglswreg,tglswmask);

  /* PTID_ROOT pin controls - THESE MUST BE FIRST!!! */
  pinreg = ROOT_PIN_NULL;
  if (LB2DM3_TGLSS_RSTFPGA & tglswmask) {  
    pinval = 0x01;
    pinreg = u32_setbits(ROOT_PIN_NULL,ROOT_PIN_RESET,pinval);
    cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    cmdcfg_delay(mco,10000);
    pinval = 0x00;
    pinreg = u32_setbits(ROOT_PIN_NULL,ROOT_PIN_RESET,pinval);
    cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    cmdcfg_delay(mco,10000);
    cmdcfg_bufclr(mco);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_RSTFPGA,0x00);
  }
  if (LB2DM3_TGLSS_DF & tglswmask) {
    pinval = (LB2DM3_TGLSS_DF & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,ROOT_PIN_ENABLE,pinval);
    cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_DF,pinval);
  }

  /* PTID_STATE pin controls */
  pinreg = LB2DM3_PIN_NULL;
  if (LB2DM3_TGLSS_PPSTAG & tglswmask) {
    pinval = (LB2DM3_TGLSS_PPSTAG & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,LB2DM3_PIN_PPSTAG,pinval);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_PPSTAG,pinval);
  }
  if (LB2DM3_TGLSS_LSBP & tglswmask) {
    pinval = (LB2DM3_TGLSS_LSBP & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,LB2DM3_PIN_LSBP,pinval);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_LSBP,pinval);
  }
  if (LB2DM3_TGLSS_RAMP & tglswmask) {
    pinval = (LB2DM3_TGLSS_RAMP & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,LB2DM3_PIN_RSEL,pinval);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_RAMP,pinval);
  }
  if (LB2DM3_TGLSS_MTGO & tglswmask) {
    pinval = (LB2DM3_TGLSS_MTGO & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,LB2DM3_PIN_MTGO,pinval);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_MTGO,pinval);
  }
  if (LB2DM3_TGLSS_HBF & tglswmask) {
    pinval = (LB2DM3_TGLSS_HBF & tglswreg) ? (0x01) : (0x00);
    pinreg = u32_setbits(pinreg,LB2DM3_PIN_HBFBYP,pinval);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_HBF,pinval);
  }
  if ((LB2DM3_TGLSS_PPSTAG & tglswmask) ||
      (LB2DM3_TGLSS_LSBP & tglswmask)   ||
      (LB2DM3_TGLSS_RAMP & tglswmask)   ||
      (LB2DM3_TGLSS_HBF  & tglswmask)   ||
      (LB2DM3_TGLSS_MTGO & tglswmask))     {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
  }

  /* Various part pin controls */
  if (LB2DM3_TGLSS_PRESEL & tglswmask) {
    pinval = (LB2DM3_TGLSS_PRESEL & tglswreg) ? (0x01) : (0x00);
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_PRESEL,pinval);
    lb2dm3_presel_decode(mco,&psel0,&psel1);
    cmdcfg_ctl(mco,PTID_SKY13384,0,PFUNC_PWR,0,psel0,0xFFFF);
    cmdcfg_ctl(mco,PTID_SKY13384,1,PFUNC_PWR,0,psel1,0xFFFF);
  }
  if (LB2DM3_TGLSS_LNA & tglswmask) {
    pinval = (LB2DM3_TGLSS_LNA & tglswreg) ? (0x00) : (0x01);
    pinreg = u32_setbits(VMMK2103_PIN_NULL,VMMK2103_PIN_BYPASS,((mco->RT.hw_ver&0xF)==0x7)? !pinval : pinval);
    cmdcfg_ctl(mco,PTID_VMMK2103,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    pinval = (pinval == 0x01) ? (0x00) : (0x01);  /* Negate bypass */
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_LNA,pinval);
  }
  if (LB2DM3_TGLSS_DCS & tglswmask) {
    pinval = (LB2DM3_TGLSS_DCS & tglswreg) ? (0x00) : (0x01);
    pinreg = u32_setbits(DCS_PIN_NULL,DCS_PIN_BYPASS,pinval);
    cmdcfg_ctl(mco,PTID_DCS,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    pinval = (pinval == 0x01) ? (0x00) : (0x01);  /* Negate bypass */
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_DCS,pinval);
  }
  if (LB2DM3_TGLSS_AIS & tglswmask) {
    pinval = (LB2DM3_TGLSS_AIS & tglswreg) ? (0x00) : (0x01);
    pinreg = u32_setbits(AIS_PIN_NULL,AIS_PIN_BYPASS,pinval);
    cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    pinval = (pinval == 0x01) ? (0x00) : (0x01);  /* Negate bypass */
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_AIS,pinval);
  }
#ifdef BOGUS
  if (LB2DM3_TGLSS_HBF & tglswmask) {
    pinval = (LB2DM3_TGLSS_HBF & tglswreg) ? (0x00) : (0x01);
    pinreg = u32_setbits(HBDECFIR_PIN_NULL,HBDECFIR_PIN_BYPASS,pinval);
    cmdcfg_ctl(mco,PTID_HBDECFIR,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    pinval = (pinval == 0x01) ? (0x00) : (0x01);  /* Negate bypass */
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_HBF,pinval);
  }
#endif
  if (LB2DM3_TGLSS_EQ & tglswmask) {
    pinval = (LB2DM3_TGLSS_EQ & tglswreg) ? (0x00) : (0x01);
    pinreg = u32_setbits(FIR_PIN_NULL,FIR_PIN_BYPASS,pinval);
    cmdcfg_ctl(mco,PTID_FIR,0,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_FIR,1,PFUNC_PWR,0x00,pinreg,0xFFFFFFFF);
    pinval = (pinval == 0x01) ? (0x00) : (0x01);  /* Negate bypass */
    mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(mco->LS.u32[LB2DM3_RG_TGLSS],
      LB2DM3_TGLSS_EQ,pinval);
  }

  /* Flush the commands */
  cmdcfg_bufclr(mco);
  cmdcfg_delay(mco,1000);

  dbgprint(mco->dbg,DBG_LB,
    "lb2dm3_tglsw OUT: LS: 0x%8.8X  TS: 0x%8.8X  TMSK: 0x%8.8X\n",
     mco->LS.u32[LB2DM3_RG_TGLSS],tglswreg,tglswmask);

  return (0);
}

int32_t lb2dm3_mtgo(Module_Config_Obj *mco)
{
  int32_t mtgooff = mco->LS.i32[LB2DM3_RG_MTGOOFF];
  cmdcfg_ctl(mco,PTID_MTGO,0,PFUNC_RWR,0x00,mtgooff,0xFFFF);
  lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_MTGO);
  return (0);
}

int32_t lb2dm3_dcs(Module_Config_Obj *mco, int32_t dcsBn)
{
  cmdcfg_ctl(mco,PTID_DCS,0,PFUNC_RWR,0x00,dcsBn,0x0F);
  lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_DCS);
  return (0);
}

int32_t lb2dm3_statistics(Module_Config_Obj *mco)
{
  int32_t stat_Bn = findintflagdef("STATBN",mco->config,22)&0xFF;
  cmdcfg_ctl(mco,PTID_STATIS,0,PFUNC_RWR,0x00,stat_Bn,0x0F);
  return(0);
}

int32_t lb2dm3_ais(Module_Config_Obj *mco)
{
  int32_t ais_cf1 = 0;
  int32_t ais_cf2 = 0x80000000;
  int32_t ais_cf1_ll  = (int32_t) (-0.09*2147483648.0);
  int32_t ais_cf1_ul  = (int32_t) (0.09*2147483648.0);
  uint32_t ais_cf2_ll = (uint32_t) (0.93*2147483648.0);
  uint32_t ais_cf2_ul = (uint32_t) (1.07*2147483648.0);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x00,ais_cf1,0xFFFFFFFF);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x01,ais_cf2,0xFFFFFFFF);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x05,ais_cf1_ul,0xFFFFFFFF);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x06,ais_cf1_ll,0xFFFFFFFF);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x07,ais_cf2_ul,0xFFFFFFFF);
  cmdcfg_ctl(mco,PTID_AIS,0,PFUNC_RWR,0x08,ais_cf2_ll,0xFFFFFFFF);
  lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_AIS);
  return (0);
}

int32_t lb2dm3_adclm(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t adclm_B1  =  (9)&0xFF;
  int32_t adclm_B2  = (15)&0xFF;
  int32_t adclm_dec = (10)&0xFF;
  int32_t stat_func, stat_ID, stat_addr, stat_val;

  if (flags & LB2DM3_FCN_SET) {
    cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,0x01,adclm_B1,0xFF);
    cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,0x02,adclm_B2,0xFF);
    cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,0x03,adclm_dec,0xFF);
    cmdcfg_delay(mco,2000);
    cmdcfg_bufclr(mco);
  }
  
  if (flags & LB2DM3_FCN_GET) {
    cmdcfg_ctl(mco, PTID_ADCLM,0,PFUNC_RRD,0,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_val);
    cmdcfg_bufclr(mco);
    mco->LS.i32[LB2DM3_RG_ADCLM] = adclm_to_cf_i32(stat_val);
  }
  return (0);
}

int32_t lb2dm3_rfpwr(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t stat_func, stat_ID, stat_addr, stat_val;
  float32_t adcval;

  if (flags & LB2DM3_FCN_GET) {
    cmdcfg_ctl(mco,PTID_LTC2362,0,PFUNC_PWR,0,1,0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_LTC2362,0,PFUNC_RRD,0,0,0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_LTC2362,0,PFUNC_PWR,0,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_val);
    cmdcfg_bufclr(mco);

    /* Three segment piecewise linear fit to dBm */
    adcval = (float32_t) stat_val;
    adcval = (adcval > 0.0) ? (adcval) : (0.0);
    adcval = (adcval < 3000.0) ? (adcval) : (3000.0);
    if (adcval > 2513.0) {
      adcval = (0.05836689*adcval) - 146.35;
    }
    else if (adcval > 356.0) {
      adcval = (0.02300209*adcval) - 57.95;
    }
    else {
      adcval = (0.09740266*adcval) - 85.68;
    }

    if (mco->LS.u32[LB2DM3_RG_TGLSS] & LB2DM3_TGLSS_LNA)
      adcval = adcval - 0.5; 
    else 
      adcval = adcval + 16.3;

    adcval = (adcval > -45.0) ? (adcval) : (-45.0);
    adcval = (adcval < 10.0) ? (adcval) : (10.0);
    adcval = (adcval < 0.0) ? (adcval-0.5) : (adcval+0.5);

    mco->LS.i32[LB2DM3_RG_RFPWR] = (int32_t) adcval;
  }
  return (0);
}

int32_t lb2dm3_hbf(Module_Config_Obj *mco)
{
  int32_t coeset = 0;
  lb2dm3_hbf_loadcoef(mco,PTID_HBDECFIR,0, coeset);
  lb2dm3_hbf_loadcoef(mco,PTID_HBDECFIR,1, coeset);
  cmdcfg_bufclr(mco);
  /* lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_HBF); */
  return (0);
}

int32_t lb2dm3_cfir(Module_Config_Obj *mco)
{
  UDB rate = udb_initializer;

  rate.i32[0] = mco->LS.i32[LB2DM3_RG_SRATEW];
  rate.i32[1] = mco->LS.i32[LB2DM3_RG_SRATEF];
  udb_i32wf_to_f64(&rate);

  lb2dm3_cfir_loadcoef(mco,PTID_FIR,0,rate.f64[0]);
  lb2dm3_cfir_loadcoef(mco,PTID_FIR,1,rate.f64[0]);
  cmdcfg_bufclr(mco);
  return (0);
}

int32_t lb2dm3_si5338(Module_Config_Obj *mco)
{
  UDB rate = udb_initializer;
  PT_SI5338_REG SIR[SI5338_MAX_CFG_RG];
  PT_SI5338_MSCFG MS;
  int32_t i,rtn;
  int32_t stat_func, stat_pin, stat_addr, stat_data;
  int32_t reg_data45, reg_data46, reg_data47;

  rate.i32[0] = mco->LS.i32[LB2DM3_RG_SRATEW];
  rate.i32[1] = mco->LS.i32[LB2DM3_RG_SRATEF];
  udb_i32wf_to_f64(&rate);

  /* Check for digital HBF in setup */
  if (u32_getbits(LB2DM3_TGLSS_HBF,mco->LS.u32[LB2DM3_RG_TGLSS]) == 0x00) {
  /*if (rate.f64[0] <= 125.0) { */
    rate.f64[0] = 2.0*rate.f64[0];
  }

  /* Initialize the Si5338 configuration */
  pt_si5338_rgc_default(&SIR[0], FPGA_MODID_LB2DM3);

  /* Configure internal / external reference (assume internal) */
  if (mco->LS.u32[LB2DM3_RG_CLK] & LB2DM3_RGM_CLK_EXTREF) {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x00)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x00 >> 3)),0);
  }
  else {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x0A)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x0A >> 3)),0);
  }
  pt_si5338_rgc_setbits(&SIR[0],28,0x03,0x00,0);

  /* Configure input to P2_DIVIDER (Zerodly clock reference) */
  pt_si5338_rgc_setbits(&SIR[0],28,0x20,((0x01)&(0x00)),0);
  pt_si5338_rgc_setbits(&SIR[0],30,0x18,((0x03)&(0x00 >> 1)),0);

  /* Configure a divider solution for the MSx outputs                */
  /*   By preselecting only VCO multiples of 5MHz, guarantee Zerodly */
  /*   path is integer multiple, so we use Samplerate path as the    */
  /*   critical path choice                                          */
  MS.dbglvl = 0;            /* MS.dbglvl = DBG_LB for debug output */
  MS.inp_ref_frq    = 10.0; /* BAD BAD BAD - don't assume 10MHz ref clock */
  MS.clk_frq_des[0] = 30.0;            /* STV6111 reference     */
  MS.clk_frq_des[1] = MS.inp_ref_frq;  /* Unused output         */
  MS.clk_frq_des[2] = rate.f64[0];     /* ADC clock             */
  MS.clk_frq_des[3] = MS.inp_ref_frq;  /* Zero delay mode clock */
  MS.clk_frq_des[4] = MS.inp_ref_frq;  /* Unused output         */
  MS.ms_critical    = 2;               /* Critical path is sample rate */
  rtn = pt_si5338_msx_soln(&SIR[0], &MS);
  if (rtn<0) {
    dbgerrormsg("Invalid clock configuration specified. Bad... \n");
    return(-2); 
  }

  /* Update actual rates / references for precision queries */
  rate = udb_initializer;
  rate.f64[0] = MS.clk_frq_act[0];
  udb_f64_to_i32wf(&rate);
  mco->LS.i32[LB2DM3_RG_VCOREFW] = rate.i32[0];
  mco->LS.i32[LB2DM3_RG_VCOREFF] = rate.i32[1];
  if (MS.clk_frq_act[0] != MS.clk_frq_des[0]) {
    dbgwarnmsg("Requested ADC sample rate requires non-integer RF VCO reference. Degraded RF phase noise may result.\n");
  }

  rate = udb_initializer;
  rate.f64[0] = MS.clk_frq_act[2];
  /* Digital HBF in setup */
  if (u32_getbits(LB2DM3_TGLSS_HBF,mco->LS.u32[LB2DM3_RG_TGLSS]) == 0x00) {
  /* if (rate.f64[0] <= 125.0) { */
    rate.f64[0] = 0.5*rate.f64[0];
  }
  if (MS.clk_frq_act[2] != MS.clk_frq_des[2]) {
    dbgwarn("Unable to obtain exact sample rate. True sample rate is: %21.16f MHz \n",rate.f64[0]);
  }
  udb_f64_to_i32wf(&rate);
  mco->LS.i32[LB2DM3_RG_SRATEW] = rate.i32[0];
  mco->LS.i32[LB2DM3_RG_SRATEF] = rate.i32[1];

  for (i=0; i<SI5338_MAX_CFG_RG; i++) { 
    dbgprint(0,DBG_LB,
      " SI5338_REG: %3.3u  %2.2X  0x%2.2X  0x%2.2X  0x%2.2X\n", 
      SIR[i].ra,(SIR[i].rv ^ si5338_lb2dm3_reg_defaults[i].rv),
      si5338_lb2dm3_reg_defaults[i].rv,SIR[i].rv,SIR[i].rm);
  }

  /* Disable outputs and pause LOL */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,230,1<<4,1<<4); 
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,241,1<<7,1<<7);

  /* Configure all the registers on the part */
  for(i=0; i<SI5338_MAX_CFG_RG; i++){
    cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,SIR[i].ra,SIR[i].rv,SIR[i].rm);
  }

  /* Cfg PLL for Lock, initiate PLL lock, Restart LOL, Wait 25ms */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,49,0,1<<7);     
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,246,1<<1,1<<1);
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,241,0x65,0xFF);
  cmdcfg_delay(mco,25000);                             

  /* Validate Input Clock Status */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RRD,218,0,0xFFFFFFFF);

  /* Confirm PLL Lock */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RRD,218,0,0xFF);

  /* Copy FCAL Values To Active Registers */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RRD,237,0,0xFF);
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RRD,236,0,0xFF);
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RRD,235,0,0xFF);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  /* Check lock status */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  dbgprint(mco->dbg,DBG_LB,"Si5338 Clock Status = %x\n",stat_data);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  dbgprint(mco->dbg,DBG_LB,"Si5338 PLL Lock Status = %x\n",stat_data);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data47 = stat_data & 0x03;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data46 = stat_data & 0xFF;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data45 = stat_data & 0xFF;
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,47,reg_data47,0x03);
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,46,reg_data46,0xFF);
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,45,reg_data45,0xFF);

  /* Set PLL To Use FCAL Values */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,49,1<<7,1<<7);

  /* Enable Outputs */
  cmdcfg_ctl(mco,PTID_SI5338,0,PFUNC_RWR,230,0,1<<4);
  cmdcfg_delay(mco,25000);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return (0);
}


int32_t lb2dm3_ads42b49(Module_Config_Obj *mco)
{
  int32_t addr,data;

  /* Reset ADC */
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_PWR,0,0,0xFFFF);
  cmdcfg_delay(mco,10000);
  dbgprintmsg(mco->dbg,DBG_LB,"DBG: ADC \n");

  /* Configure ADC */
/* JJB */
  addr = 0x3D;  /* TPH setting */
  data = 0x20;
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
/* JJB */
  addr = 0x3F;  /* TPH setting */
  data = 0x12;
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
  addr = 0x40;  /* TPL setting */
  data = 0x34;
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
  /* Set ADC Channel-A gain to maximum (6dB) */
  addr = 0x25; /* TPMA: 0=Normal, 3=Fixed Pat, 4=Ramp */
  data = 0xC0;  
  /* Enable ADC CH-A Output */
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
  /* Set ADC Channel-B gain to maximum (6dB) */
  addr = 0x2B;
  data = 0xC0; /* TPMB: 0=Normal, 3=Fixed Pat, 4=Ramp */
  /* Enable ADC CH-B Output */
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
  addr = 0x44;
  data = 0x0001; /* data = 0x0000; Don't enable test ramp... */
  /* Enable ADC Output */
  cmdcfg_ctl(mco,PTID_ADS42B49,0,PFUNC_RWR,addr,data,0xFFFF);
  cmdcfg_bufclr(mco);

  return (0);
}

int32_t lb2dm3_stv6111(Module_Config_Obj *mco)
{
  int32_t stat_func, stat_pin, stat_addr, stat_data;
  PT_STV6111_REG STV;
  int32_t i;
  uint32_t rfatn = PE43204_ATTN_00DB;
  UDB rffreq = udb_initializer;
  UDB rfgain = udb_initializer;
  UDB rfbw   = udb_initializer;
  UDB vcoref = udb_initializer;

  pt_stv6111_rgc_default(&STV);
  rffreq.i32[0] = mco->LS.i32[LB2DM3_RG_RFFREQW];
  rffreq.i32[1] = mco->LS.i32[LB2DM3_RG_RFFREQF];
  udb_i32wf_to_f64(&rffreq);

  /* RF Frequency configuration */
  vcoref.i32[0] = mco->LS.i32[LB2DM3_RG_VCOREFW];
  vcoref.i32[1] = mco->LS.i32[LB2DM3_RG_VCOREFF];
  udb_i32wf_to_f64(&vcoref);

  if (pt_stv6111_rgc_rffreq(&STV, &rffreq, vcoref.f64[0]) != 0) {
    rffreq.f64[0] = 900.0;
    dbgwarn("Invalid RFFREQ. Using %f \n",rffreq.f64[0]);
    pt_stv6111_rgc_rffreq(&STV, &rffreq, vcoref.f64[0]);
  }
  udb_f64_to_i32wf(&rffreq);
  mco->LS.i32[LB2DM3_RG_RFFREQW] = rffreq.i32[0];
  mco->LS.i32[LB2DM3_RG_RFFREQF] = rffreq.i32[1];

  /* RF Gain configuration */
  rfgain.i32[0] = mco->LS.i32[LB2DM3_RG_RFGAINW];
  rfgain.i32[1] = mco->LS.i32[LB2DM3_RG_RFGAINF]; 

  udb_i32wf_to_f64(&rfgain);
  if (pt_stv6111_rgc_rfgain(&STV, &rfgain, &rfatn) != 0) {
    rfgain.f64[0] = 0.0;
    dbgwarn("Invalid RFGAIN. Using %f \n",rfgain.f64[0]);
    pt_stv6111_rgc_rfgain(&STV, &rfgain, &rfatn);
  }

  /* Configure the attenuator */
  cmdcfg_ctl(mco,PTID_PE43204,0,PFUNC_PWR,0x00,rfatn,0xFFFF);
  dbgprint(mco->dbg,DBG_LB,"PE43204(RFATTN)-> RG: 0x%2.2X  RV: 0x%2.2X \n", 0,rfatn);

  /* Configure the tuner */
  for (i=0; i<STV6111_NUM_RG; i++)
    dbgprint(mco->dbg,DBG_LB, "STV6111(RFFREQ)-> RG: 0x%2.2X  RV: 0x%2.2X \n",i,STV.rg[i]);

  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x00,STV.rg[0x00],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x01,STV.rg[0x01],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x02,STV.rg[0x02],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x03,STV.rg[0x03],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x04,STV.rg[0x04],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x05,STV.rg[0x05],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x06,STV.rg[0x06],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x07,STV.rg[0x07],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x08,STV.rg[0x08],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0A,STV.rg[0x0A],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0B,STV.rg[0x0B],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0C,STV.rg[0x0C],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0D,STV.rg[0x0D],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0E,STV.rg[0x0E],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x0F,STV.rg[0x0F],0xFF);
  cmdcfg_delay(mco,2000);

  /* Initiate VCO calibration */
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_VCOCAL,0x01);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  cmdcfg_delay(mco,20000);
  /* Initiate VCO loop filter calibration (BW must = 23MHz!) */
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_BBFCAL,0x01);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  cmdcfg_delay(mco,5000);

  /* Check VCO calibration status */
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RRD,0x09,0,0xFF);
  cmdcfg_delay(mco,25000);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  dbgprint(mco->dbg,DBG_LB,"STV6111 REG 0x09 = 0x%8.8X\n",stat_data);
  /* Now set RF BW and initiate calibration */       
  rfbw.i32[0] = mco->LS.i32[LB2DM3_RG_RFBWW];
  rfbw.i32[1] = mco->LS.i32[LB2DM3_RG_RFBWF];
  udb_i32wf_to_f64(&rfbw);
  pt_stv6111_rgc_rfbw(&STV, &rfbw);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x01,STV.rg[0x01],0xFF);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x08,STV.rg[0x08],0xFF);
  cmdcfg_delay(mco,5000);
  STV.rg[0x09] = u8_setbits(STV.rg[0x09],STV6111_RGM_09_BBFCAL,0x01);
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RWR,0x09,STV.rg[0x09],0x0F);
  cmdcfg_delay(mco,5000);
 
  /* Check lock status */ 
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RRD,0x09,0,0xFF);
  cmdcfg_delay(mco,25000);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  dbgprint(mco->dbg,DBG_LB,"STV6111 REG 0x09 = 0x%8.8X\n",stat_data);
  /* Check PLL lock status */
  cmdcfg_ctl(mco,PTID_STV6111,0,PFUNC_RRD,0x08,0,0xFF);
  cmdcfg_delay(mco,25000);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  dbgprint(mco->dbg,DBG_LB,"STV6111 PLL LOCK = %x\n",((0x01)&(stat_data)));
  cmdcfg_delay(mco,2000);
  dbgprintmsg(mco->dbg,DBG_LB,"DBG 01 \n");

  return (0);
}

int32_t lb2dm3_fwload(Module_Config_Obj *mco, uint32_t flags) 
{
  return (0);
}

int32_t lb2dm3_hh_reset(Module_Config_Obj *mco, uint32_t flags) 
{
  /* FPGA_RESET = 0x0001, HH_RESET = 0x0004 */
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,0x0005,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,0x0000,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_bufclr(mco);
  return (0);
}

int32_t lb2dm3_get_state(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  rtn = module_get_state(mco);
  return (rtn);
}

int32_t lb2dm3_set_state(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  rtn = module_set_state(mco);
  return (rtn);
}


int32_t lb2dm3_master_reset(Module_Config_Obj *mco)
{
  uint32_t tglssreg = mco->LS.u32[LB2DM3_RG_TGLSS];
  int32_t dcsBn = findintflagdef("DCSBN",mco->config,14)&0xFF;

  /* Unconditionally reset the FPGA and disable output data flow */
  lb2dm3_tglsw(mco,LB2DM3_TGLSS_RSTFPGA,LB2DM3_TGLSS_RSTFPGA);
  mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(tglssreg,LB2DM3_TGLSS_RSTFPGA,0x00);
  lb2dm3_tglsw(mco,LB2DM3_TGLSS_NULL,LB2DM3_TGLSS_DF);
  mco->LS.u32[LB2DM3_RG_TGLSS] = u32_setbits(tglssreg,LB2DM3_TGLSS_DF,0x00);

  /* Save desired toggle switches and configuration to state part */
  module_set_state(mco);

  /* Configure the module */
  lb2dm3_mtgo(mco);
  lb2dm3_dcs(mco,dcsBn);
  lb2dm3_statistics(mco);
  lb2dm3_ais(mco);
  lb2dm3_adclm(mco,LB2DM3_FCN_SET);
  lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],0xFFFFFFFF);
  lb2dm3_hbf(mco);   
  lb2dm3_cfir(mco);   
  lb2dm3_si5338(mco);   
  lb2dm3_ads42b49(mco);
  lb2dm3_stv6111(mco);
  cmdcfg_bufclr(mco);

  /* Commit the final module configuration to the state part */
  module_set_state(mco);

  return (0);
}




/*---------------------------------------------------------------------------*/
/* External Public Interfaces                                                */
/*                                                                           */
/*   NOTE: These routines define the public interface to the module and      */
/*         should be called by external libraries. The API to these          */
/*         routines will be stable.                                          */
/*                                                                           */
/*---------------------------------------------------------------------------*/

/* This function generates the commands to queue up for an LB2DM3 module */
int32_t module_gen_qcmd_lb2dm3(Module_Config_Obj *mco, uint32_t flags) 
{

  return (0);
}


/* This function decodes the status queue returned from an LB2DM3 module */
int32_t module_dec_qsta_lb2dm3(Module_Config_Obj *mco, uint32_t flags) 
{

  return (0);
}


/* This function is an ICEPIC stop gap. It actually does the work. */
int32_t module_fcn_lb2dm3(Module_Config_Obj *mco, uint32_t flags, UDB *buf) 
{
  int32_t rtn = 0;
  int32_t i;        
  uint32_t fcnlist[16];
  int32_t MAX_FCNLIST = sizeof(fcnlist)/sizeof(fcnlist[0]);

  /* mco->dbg = DBG_LB|DBG_MO; */
  dbgprint(mco->dbg,DBG_LB,"MODLIB: module_fcn_lb2dm3: 0x%8.8X\n", 
    mco->LS.u32[LB2DM3_RG_FCN]);

  /* Based on the requested function, build a function list to execute */
  for (i=0; i<MAX_FCNLIST; i++) fcnlist[i] = LB2DM3_FCN_NULL;
  switch (mco->LS.u32[LB2DM3_RG_FCN]) {
    case (LB2DM3_FCN_MASTER_RESET):
      fcnlist[0] = LB2DM3_FCN_MASTER_RESET;
      break;
    case (LB2DM3_FCN_HHRESET):
      fcnlist[0] = LB2DM3_FCN_HHRESET;
      break;
    case (LB2DM3_FCN_RFFREQ|LB2DM3_FCN_SET):  
      fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;
      fcnlist[1] = LB2DM3_FCN_RFFREQ|LB2DM3_FCN_SET;
      fcnlist[2] = LB2DM3_FCN_PRESEL;
      fcnlist[3] = LB2DM3_FCN_STATE|LB2DM3_FCN_SET;   
      break;
    case (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_ENABLE):
      fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;
      fcnlist[1] = LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_ENABLE;
      fcnlist[2] = LB2DM3_FCN_STATE|LB2DM3_FCN_SET;
      break;
    case (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_DISABLE):
      fcnlist[0] = LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_DISABLE;
      break;
    case (LB2DM3_FCN_TGLSW|LB2DM3_FCN_GET):
      fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;
      fcnlist[1] = LB2DM3_FCN_TGLSW|LB2DM3_FCN_GET;
      break;
    case (LB2DM3_FCN_TGLSW|LB2DM3_FCN_SET):
      fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;
      fcnlist[1] = LB2DM3_FCN_TGLSW|LB2DM3_FCN_SET;
      fcnlist[2] = LB2DM3_FCN_STATE|LB2DM3_FCN_SET;
      break;
    default:
      fcnlist[0] = mco->LS.u32[LB2DM3_RG_FCN];
      if ((mco->LS.u32[LB2DM3_RG_FCN] & 0xFFFF) != LB2DM3_FCN_STATE) {
        if (mco->LS.u32[LB2DM3_RG_FCN] & LB2DM3_FCN_GET) {
          fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;      
          fcnlist[1] = mco->LS.u32[LB2DM3_RG_FCN];
        }
        if (mco->LS.u32[LB2DM3_RG_FCN] & LB2DM3_FCN_SET) {
          fcnlist[0] = LB2DM3_FCN_STATE|LB2DM3_FCN_GET;      
          fcnlist[1] = mco->LS.u32[LB2DM3_RG_FCN];
          fcnlist[2] = LB2DM3_FCN_STATE|LB2DM3_FCN_SET;      
        }
      }
  }

  /* Execute the requested functions in order */
  for (i=0; i<MAX_FCNLIST; i++) {
    if (fcnlist[i] != LB2DM3_FCN_NULL) {
      dbgprint(mco->dbg,DBG_LB,"****** EXECUTING FCN: %3.3i  0x%8.8X \n",
                i,fcnlist[i]);
    }
    switch (fcnlist[i]) {

      /* Module initialization methods */
      case (LB2DM3_FCN_FWLOAD):
        rtn |= lb2dm3_fwload(mco, flags);
        break;
      case (LB2DM3_FCN_MASTER_RESET):
        rtn |= lb2dm3_master_reset(mco);
        break;
      case (LB2DM3_FCN_HHRESET):       
        rtn |= lb2dm3_hh_reset(mco,0);
        break;
      case (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_ENABLE):   
        rtn |= lb2dm3_tglsw(mco,LB2DM3_TGLSS_DF,LB2DM3_TGLSS_DF);
        break;
      case (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_DISABLE):   
        rtn |= lb2dm3_tglsw(mco,LB2DM3_TGLSS_NULL,LB2DM3_TGLSS_DF);
        break;

      /* Module status ("get") methods */
      case (LB2DM3_FCN_STATE|LB2DM3_FCN_GET):    
        rtn |= lb2dm3_get_state(mco, 0);
        break;
      case (LB2DM3_FCN_ADCLM|LB2DM3_FCN_GET):
        rtn |= lb2dm3_adclm(mco,LB2DM3_FCN_GET);
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_ADCLM];
        buf->i32[1] = 0;
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFPWR|LB2DM3_FCN_GET):
        rtn |= lb2dm3_rfpwr(mco,LB2DM3_FCN_GET);
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFPWR];  
        buf->i32[1] = 0;
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFFREQ|LB2DM3_FCN_GET):
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFFREQW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFFREQF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFBW|LB2DM3_FCN_GET):
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFBWW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFBWF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFGAIN|LB2DM3_FCN_GET):
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFGAINW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFGAINF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_GET):
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_SRATEW]; 
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_SRATEF]; 
        udb_i32wf_to_f64(buf);
        if (u32_getbits(LB2DM3_TGLSS_HBF,mco->LS.u32[LB2DM3_RG_TGLSS]) == 0x00) {
          buf->f64[0] = 0.5*(buf->f64[0]);
        }
        break;
      case (LB2DM3_FCN_PRESEL|LB2DM3_FCN_GET):
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_PRESEL];
        buf->i32[1] = 0;
        break;
      case (LB2DM3_FCN_TGLSW|LB2DM3_FCN_GET):
        buf->u32[0] = mco->LS.i32[LB2DM3_RG_TGLSS];    
        buf->u32[1] = 0;
        break;
 
      /* Module configuration methods */
      case (LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_SET):
        udb_f64_to_i32wf(buf);
        mco->LS.i32[LB2DM3_RG_SRATEW] = buf->i32[0];
        mco->LS.i32[LB2DM3_RG_SRATEF] = buf->i32[1];
        rtn |= lb2dm3_si5338(mco);   
        rtn |= lb2dm3_ads42b49(mco);
        rtn |= lb2dm3_stv6111(mco);   
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_SRATEW]; 
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_SRATEF]; 
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFFREQ|LB2DM3_FCN_SET):
        udb_f64_to_i32wf(buf);
        mco->LS.i32[LB2DM3_RG_RFFREQW] = buf->i32[0];
        mco->LS.i32[LB2DM3_RG_RFFREQF] = buf->i32[1];
        rtn |= lb2dm3_stv6111(mco);
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFFREQW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFFREQF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFGAIN|LB2DM3_FCN_SET):
        udb_f64_to_i32wf(buf);
        mco->LS.i32[LB2DM3_RG_RFGAINW] = buf->i32[0];
        mco->LS.i32[LB2DM3_RG_RFGAINF] = buf->i32[1];
        rtn |= lb2dm3_stv6111(mco);
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFGAINW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFGAINF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_RFBW|LB2DM3_FCN_SET):
        udb_f64_to_i32wf(buf);
        mco->LS.i32[LB2DM3_RG_RFBWW] = buf->i32[0];
        mco->LS.i32[LB2DM3_RG_RFBWF] = buf->i32[1];
        rtn |= lb2dm3_stv6111(mco);
        buf->i32[0] = mco->LS.i32[LB2DM3_RG_RFBWW];
        buf->i32[1] = mco->LS.i32[LB2DM3_RG_RFBWF];
        udb_i32wf_to_f64(buf);
        break;
      case (LB2DM3_FCN_PRESEL|LB2DM3_FCN_SET):
        mco->LS.i32[LB2DM3_RG_PRESEL] = buf->i32[0];
        rtn |= lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_PRESEL);
        break;
      case (LB2DM3_FCN_PRESEL):
        rtn |= lb2dm3_tglsw(mco,mco->LS.u32[LB2DM3_RG_TGLSS],LB2DM3_TGLSS_PRESEL);
        break;
      case (LB2DM3_FCN_TGLSW|LB2DM3_FCN_SET):
        rtn |= lb2dm3_tglsw(mco,buf->u32[0],buf->u32[1]);
        break;
      case (LB2DM3_FCN_STATE|LB2DM3_FCN_SET):
        rtn |= lb2dm3_set_state(mco, 0);
        break;
 
      /* Default and unknown methods */ 
      case (LB2DM3_FCN_NULL):
        break;
      default:
        dbgwarn("module_fcn_lb2dm3 unknown function: 0x%8.8X\n",fcnlist[i]);
    } 
  } /* End MAX_FCNLIST loop */

  if (rtn != 0) {
    dbgerrormsg("Function failure in module_fcn_lb2dm3... \n");
    return (-1);
  }
  return (0);
}








int_4 lb2dm3_init_registers(Module_Config_Obj *LBM, UDB *bin)
{
  int_4 rtn = 0;
  int_4 rfopts = 0;
  uint32_t tglsw;  
  UDB b  = udb_initializer;
  UDB rg = udb_initializer;

  /* Original calling parameters */
  PICSTRUCT *p = LBM->TGT.icepic.pdev;
  int_4 mport  = LBM->TGT.icepic.mport;
  int_4 dir    = bin[0].i32[0];
  int_4 bits   = bin[1].i32[0];
  int_4 rate   = bin[2].i32[0];
  int_4 gain   = bin[3].i32[0];
  int_4 flags  = bin[4].i32[0];

  dbgprint(LBM->dbg,DBG_LB,"lb2dm3_init_registers: %d %d %d %d %d\n",
            bin[0].i32[0],bin[1].i32[0],bin[2].i32[0],bin[3].i32[0],
            bin[4].i32[0]);

  /* Clocking and triggering configuration */
  if (findflag("PREFX",p->config) >= 0)
    LBM->LS.u32[LB2DM3_RG_CLK] |= LB2DM3_RGM_CLK_EXTREF;
  else
    LBM->LS.u32[LB2DM3_RG_CLK] &= (~LB2DM3_RGM_CLK_EXTREF);

  /* Sample rate (convert to MHz) */
  b = udb_initializer;
  b.f64[0] = ((real_8)rate)/(1000000.0);
  if (rate > 0) {
    udb_f64_to_i32wf(&b);
    LBM->LS.i32[LB2DM3_RG_SRATEW] = b.i32[0];
    LBM->LS.i32[LB2DM3_RG_SRATEF] = b.i32[1];
  }  
  if (bits != -16) {
    printf("ERROR: LB2DM3 data format must be complex 16-bit integer.\n");
    return (-1);
  }

  /* MTGO Offset Samples */
  b = udb_initializer;
  b.i32[0] = findintflagdef("MTGOOFF",p->config,0);
  LBM->LS.i32[LB2DM3_RG_MTGOOFF] = b.i32[0]+0x00;

  /* RF Bandwidth in MHz */
  b = udb_initializer;
  b.f64[0] = finddblflagdef("RFBW",p->config,20.0);
  if (b.f64[0] > 0.0) {
    udb_f64_to_i32wf(&b);
    LBM->LS.i32[LB2DM3_RG_RFBWW] = b.i32[0];
    LBM->LS.i32[LB2DM3_RG_RFBWF] = b.i32[1];
  }

  /* RF Gain (in dB) */
  b = udb_initializer;
  b.f64[0] = finddblflagdef("RFGAIN",p->config,0.0);
  udb_f64_to_i32wf(&b);
  LBM->LS.i32[LB2DM3_RG_RFGAINW] = b.i32[0];
  LBM->LS.i32[LB2DM3_RG_RFGAINF] = b.i32[1];

  /* RF Center frequency in MHz */
  b = udb_initializer;
  b.f64[0] = finddblflagdef("RFFREQ",p->config,900.0);
  if (b.f64[0] > 0.0) {
    udb_f64_to_i32wf(&b);
    LBM->LS.i32[LB2DM3_RG_RFFREQW] = b.i32[0];
    LBM->LS.i32[LB2DM3_RG_RFFREQF] = b.i32[1];
  }

  /* RF Preselector configuration */
  LBM->LS.i32[LB2DM3_RG_PRESEL] = findintflagdef("RFOPT_PRESEL",p->config,0);

  /* Midas "RFOPTS" controls */   
  rfopts = findmaskflagdef("RFOPTS",p->config,RFOPTLIST,0);
  rfopts = (findflag("RFOPT_LNA",p->config)>=0) ? (rfopts|RFOPT_LNA) : rfopts;
  rfopts = (findflag("RFOPT_DCS",p->config)>=0) ? (rfopts|RFOPT_DCS) : rfopts;
  rfopts = (findflag("RFOPT_AIS",p->config)>=0) ? (rfopts|RFOPT_AIS) : rfopts;
  rfopts = (findflag("RFOPT_AGC",p->config)>=0) ? (rfopts|RFOPT_AGC) : rfopts;
  rfopts = (findflag("RFOPT_XREF",p->config)>=0) ? (rfopts|RFOPT_XREF) : rfopts;
  rfopts = (findflag("PREFX",p->config)>=0) ? (rfopts|RFOPT_XREF) : rfopts;
  rfopts = (rfopts != 0) ? (rfopts|RFOPT_ENABLE) : (rfopts);

  /* Default configuration of toggle switches */
  rg.u32[0]  = LB2DM3_TGLSS_NULL;
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_LNA,   0x01); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_DCS,   0x01); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_AIS,   0x00); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_HBF,   0x01); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_SPINV, 0x00); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_EXTREF,0x00); 
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_PRESEL,0x01); 

  /* Set toggle switch controls */
  tglsw = (RFOPT_LNA & rfopts) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_LNA,tglsw); 
  tglsw = (RFOPT_DCS & rfopts) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_DCS,tglsw); 
  tglsw = (RFOPT_AIS & rfopts) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_AIS,tglsw); 
  tglsw = (RFOPT_AGC & rfopts) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_AGC,tglsw); 
  tglsw = (RFOPT_XREF & rfopts) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_EXTREF,tglsw); 
  tglsw = (findintflag("PPSTAG",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_PPSTAG,tglsw); 
  tglsw = (findintflag("LSBP",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_LSBP,tglsw); 
  tglsw = (findintflag("HBFBYP",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_HBF,tglsw); 
  tglsw = (findintflag("SPINV",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_SPINV,tglsw); 
  tglsw = (findintflag("MTGO",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_MTGO,tglsw); 
  tglsw = (findintflag("PRESELBYP",p->config) > 0) ? (0x00) : (0x01);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_PRESEL,tglsw); 
  tglsw = (findintflag("RAMPOUT",p->config) > 0) ? (0x01) : (0x00);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_RAMP,tglsw); 
  tglsw = (findintflag("LB2D_NEQ",p->config) > 0) ? (0x00) : (0x01);
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_EQ,tglsw); 

  /* Disable dataflow - wait for call to lb2dm3_enable() */
  rg.u32[0] = u32_setbits(rg.u32[0],LB2DM3_TGLSS_DF,0x00);

  LBM->LS.u32[LB2DM3_RG_TGLSS] = rg.u32[0];

  dbgprint(LBM->dbg,DBG_LB,"lb2dm3_init_registers: RG: 0x%8.8X \n",
              LBM->LS.u32[LB2DM3_RG_TGLSS]);

  return (rtn); 
}

int_4 lb2dm3_sg_fcn(PICSTRUCT *p, int_4 mport, int_4 fcnkey, UDB *b)
{
  int_4 rtn;
  Module_Config_Obj *LBM = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid        = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  LBM = module_alloc_obj(FPGA_MODID_LB2DM3, &T, 0);
  if (LBM == (Module_Config_Obj *)NULL) return (-2);

  /* Default cases */
  LBM->dbg = p->verbose;
  LBM->config = p->config;
  /* LBM->dbg = DBG_LB|DBG_MO; */     
  dbgprint(LBM->dbg,DBG_LB,"IOM: lb2dm3_sg_fcn:        0x%8.8X\n", fcnkey);
  LBM->LS.i32[LB2DM3_RG_FCN] = fcnkey;

  /* Special case of module master reset */
  if (fcnkey == LB2DM3_FCN_MASTER_RESET) {
    rtn = lb2dm3_init_registers(LBM, &b[0]);
    LBM->LS.u32[LB2DM3_RG_FCN] = LB2DM3_FCN_MASTER_RESET;
  }

  /* Execute the specified function */
  pic_lock(p,LOCK_ALLOC);
  rtn = module_fcn(LBM, 0, b);
  pic_lock(p,LOCK_FREE);

  module_free_obj(LBM, 0);
  return (rtn);
}


int_4 lb2dm3_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, 
                   int_4 rate, int_4 gain, int_4 flags) 
{
  int_4 rtn = 0;
  int_4 i;
  UDB bb[5];
  /* Get orignal calling function parameters into UDB array */
  for (i=0; i<5; i++) bb[i] = udb_initializer;
  bb[0].i32[0] = dir;
  bb[1].i32[0] = bits;
  bb[2].i32[0] = rate;
  bb[3].i32[0] = gain;
  bb[4].i32[0] = flags;
  /* Call the module setup function */
  rtn = lb2dm3_sg_fcn(p, mport, LB2DM3_FCN_MASTER_RESET, &bb[0]);
  return (rtn); 
}


int_4 lb2dm3_enable(PICSTRUCT *p, int_4 mport, int_4 ena)
{
  UDB b = udb_initializer;
  if (ena != 0)
    lb2dm3_sg_fcn(p, mport, (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_ENABLE), &b); 
  else
    lb2dm3_sg_fcn(p, mport, (LB2DM3_FCN_DATAFLOW|LB2DM3_FCN_DISABLE), &b);
  return (0);
}


/* WARNING: This is *** NOT *** a module reset. It is an HH interface retrain */
/*   You probably want LB2DM3_FCN_MASTER_RESET in lb2dm3_sg_fcn()             */
/*   The "_reset" function name is used for legacy compatibility in icelib.c  */
int_4 lb2dm3_reset(PICSTRUCT *p, int_4 mport)
{
  UDB b = udb_initializer;
  lb2dm3_sg_fcn(p, mport, LB2DM3_FCN_HHRESET, &b); 
  return (0);
}

int_4 lb2dm3_set_dcsbn(PICSTRUCT *p, int_4 mport, int_4 dcsBn)
{
  int_4 rtn;
  TGT_Config_Obj T;
  Module_Config_Obj *mco;
  T.tgtid        = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  mco = module_alloc_obj(FPGA_MODID_LB2DM3, &T, 0);
  rtn = module_lock(mco, 0);
  cmdcfg_ctl(mco,PTID_DCS,0,PFUNC_RWR,0x00,dcsBn,0x0F);
  cmdcfg_bufclr(mco);
  rtn = module_unlock(mco, 0);
  module_free_obj(mco, 0);
  return 0;
}

int_4 lb2dm3_get_opts(PICSTRUCT *p, int_4 mport)
{
  int_4 rtn = 0;
  int_4 rfopts = 0;
  UDB b = udb_initializer;

  /* Get state of toggle switches from module */
  rtn = lb2dm3_sg_fcn(p, mport, (LB2DM3_FCN_TGLSW|LB2DM3_FCN_GET), &b);

  /* Decode TGLSW register to ICEPIC options */
  rfopts = RFOPT_ENABLE;
  rfopts = (u32_getbits(b.u32[0],LB2DM3_TGLSS_LNA)) ? (RFOPT_LNA|rfopts) : 
           (rfopts);
  rfopts = (u32_getbits(b.u32[0],LB2DM3_TGLSS_DCS)) ? (RFOPT_DCS|rfopts) : 
           (rfopts);     
  rfopts = (u32_getbits(b.u32[0],LB2DM3_TGLSS_AIS)) ? (RFOPT_AIS|rfopts) : 
           (rfopts);     
  rfopts = (u32_getbits(b.u32[0],LB2DM3_TGLSS_AGC)) ? (RFOPT_AGC|rfopts) : 
           (rfopts);     
  rfopts = (u32_getbits(b.u32[0],LB2DM3_TGLSS_EXTREF)) ? (RFOPT_XREF|rfopts) : 
           (rfopts);     
  return (rfopts);
}

int_4 lb2dm3_set_opts(PICSTRUCT *p, int_4 mport, int_4 rfopts)
{
  int_4 rtn = 0;
  int_4 rfold = 0;
  int_u4 tglsw;
  UDB b = udb_initializer;


  /* Return if no options enabled */
  if ((rfopts & RFOPT_ENABLE) != RFOPT_ENABLE) return (0);

  /* Get currently configured options */
  rfold = lb2dm3_get_opts(p, mport);
  if (rfold == rfopts) return (0);

  /* Map requested changes */
  if ((rfold & RFOPT_LNA) != (rfopts & RFOPT_LNA)) {
    tglsw = (RFOPT_LNA & rfopts) ? (0x01) : (0x00);
    b.u32[0]  = u32_setbits(b.u32[0],LB2DM3_TGLSS_LNA,tglsw); 
    b.u32[1] |= LB2DM3_TGLSS_LNA;
  } 
  if ((rfold & RFOPT_DCS) != (rfopts & RFOPT_DCS)) {
    tglsw = (RFOPT_DCS & rfopts) ? (0x01) : (0x00);
    b.u32[0]  = u32_setbits(b.u32[0],LB2DM3_TGLSS_DCS,tglsw); 
    b.u32[1] |= LB2DM3_TGLSS_DCS;
  } 
  if ((rfold & RFOPT_AIS) != (rfopts & RFOPT_AIS)) {
    tglsw = (RFOPT_AIS & rfopts) ? (0x01) : (0x00);
    b.u32[0]  = u32_setbits(b.u32[0],LB2DM3_TGLSS_AIS,tglsw); 
    b.u32[1] |= LB2DM3_TGLSS_AIS;
  } 
  if ((rfold & RFOPT_AGC) != (rfopts & RFOPT_AGC)) {
    tglsw = (RFOPT_AGC & rfopts) ? (0x01) : (0x00);
    b.u32[0]  = u32_setbits(b.u32[0],LB2DM3_TGLSS_AGC,tglsw); 
    b.u32[1] |= LB2DM3_TGLSS_AGC;
  } 
  if ((rfold & RFOPT_XREF) != (rfopts & RFOPT_XREF)) {
    tglsw = (RFOPT_XREF & rfopts) ? (0x01) : (0x00);
    b.u32[0]  = u32_setbits(b.u32[0],LB2DM3_TGLSS_EXTREF,tglsw); 
    b.u32[1] |= LB2DM3_TGLSS_EXTREF;
  } 

  /* Set toggle switches */
  rtn = lb2dm3_sg_fcn(p, mport, (LB2DM3_FCN_TGLSW|LB2DM3_FCN_SET), &b);
  return (rtn);
}

int_4 lb2dm3_get_clkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 rtn, status;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 extref;

  /* Create a module handle and configure device transport */
  Module_Config_Obj *LB2DM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  LB2DM3_module = module_alloc_obj(FPGA_MODID_LB2DM3, &T, 0);
  if (LB2DM3_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(LB2DM3_module, 0);
  if(rtn<0) { 
    printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); 
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return(-1);
   }

  lb2dm3_get_state(LB2DM3_module, 0);
  cmdcfg_ctl(LB2DM3_module,BAD_MOD_ID,0,PFUNC_PRD,1,0,0xFF);
  cmdcfg_bufclr(LB2DM3_module);
  cmdcfg_stat(LB2DM3_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

  status = 0;
  extref = LB2DM3_module->LS.u32[LB2DM3_RG_CLK] & LB2DM3_RGM_CLK_EXTREF;

  if (extref && (stat_data&0x1)==0) status |= 0x1;
  if ((stat_data&0x2)==0) status |= 0x2;

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(LB2DM3_module);

  /* Release Command Lock */
  rtn = module_unlock(LB2DM3_module, 0);
  module_free_obj(LB2DM3_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return (status);
}

int_4 lb2dm3_getkeyl(PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 rtn = 0;
  UDB b = udb_initializer;
  uint32_t fcnkey = LB2DM3_FCN_NULL;

  if (key == KEY_CLKSTAT) return lb2dm3_get_clkstat (p,mport);
  if (key == KEY_RFOPTS) return lb2dm3_get_opts (p,mport);
 
  /* Convert ICEPIC keys to LB2DM3 function keys and call SET/GET function */
  if (key == KEY_RFGAIN) fcnkey = LB2DM3_FCN_RFGAIN|LB2DM3_FCN_GET;
  if (key == KEY_ADLM)   fcnkey = LB2DM3_FCN_ADCLM|LB2DM3_FCN_GET;
  if (key == KEY_RFPWR)  fcnkey = LB2DM3_FCN_RFPWR|LB2DM3_FCN_GET;
  if (fcnkey != LB2DM3_FCN_NULL) {
    rtn = lb2dm3_sg_fcn(p, mport, fcnkey, &b);
    if (rtn != 0) { 
      fcnkey = LB2DM3_FCN_NULL;
      b.i32[0] = 0;
    }
  }
  /* Reformat and scale return value for calling routine */
  switch (fcnkey) {
    case (LB2DM3_FCN_RFGAIN|LB2DM3_FCN_GET):
      udb_f64_to_i32rnd(&b);
      break;
    case (LB2DM3_FCN_ADCLM|LB2DM3_FCN_GET):
      udb_f64_to_i32rnd(&b);
      break;
    case (LB2DM3_FCN_RFPWR|LB2DM3_FCN_GET):
      udb_f64_to_i32rnd(&b);
      break;
    default:
      dbgwarn("Unknown KEY in lb2dm3_getkeyl: 0x%8.8X \n",key);
      b.i32[0] = 0;
  }
  return (b.i32[0]);
}


real_8 lb2dm3_getkeyd(PICSTRUCT *p, int_4 mport, int_4 key)
{
  int_4 rtn = 0;
  UDB b = udb_initializer;
  uint32_t fcnkey = LB2DM3_FCN_NULL;

  /* Convert ICEPIC keys to LB2DM3 function keys and call SET/GET function */
  if (key == KEY_RFFREQ) fcnkey = LB2DM3_FCN_RFFREQ|LB2DM3_FCN_GET;
  if (key == KEY_RFBW)   fcnkey = LB2DM3_FCN_RFBW|LB2DM3_FCN_GET; 
  if (key == KEY_DRATE)  fcnkey = LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_GET;
  if (fcnkey != LB2DM3_FCN_NULL) {
    rtn = lb2dm3_sg_fcn(p, mport, fcnkey, &b);
    if (rtn != 0) {
      fcnkey = LB2DM3_FCN_NULL;
      b.f64[0] = 0.0;
    }
  }
  /* Reformat and scale return value for calling routine */
  switch (fcnkey) {
    case (LB2DM3_FCN_RFFREQ|LB2DM3_FCN_GET):
      break;
    case (LB2DM3_FCN_RFBW|LB2DM3_FCN_GET):  
      break;
    case (LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_GET):
      break;
    default:
      dbgwarn("Unknown KEY in lb2dm3_getkeyd: 0x%8.8X \n",key);
      b.f64[0] = 0.0;
  }
  return (b.f64[0]);
}

int_4 lb2dm3_setkeyl(PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)
{
  int_4 rtn = 0;
  UDB b = udb_initializer;
  uint32_t fcnkey = LB2DM3_FCN_NULL;

  if (key == KEY_RFOPTS) return lb2dm3_set_opts (p,mport,i4val);
  if (key == KEY_DCSBN) return lb2dm3_set_dcsbn (p,mport,i4val);
 
  /* Convert ICEPIC keys to LB2DM3 function keys and call SET/GET function */
  if (key == KEY_RFGAIN) fcnkey = LB2DM3_FCN_RFGAIN|LB2DM3_FCN_SET;
  if (key == KEY_RFBW)   fcnkey = LB2DM3_FCN_RFBW|LB2DM3_FCN_SET;

  /* Reformat and scale input value to correct form for the key */
  switch (fcnkey) {
    case (LB2DM3_FCN_RFGAIN|LB2DM3_FCN_SET):
      b.i32[0] = i4val;     
      udb_i32wf_to_f64(&b);
      break;
    case (LB2DM3_FCN_RFBW|LB2DM3_FCN_SET):
      b.i32[0] = i4val;     
      udb_i32wf_to_f64(&b);
      break;
    case (LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_SET):
      b.i32[0] = i4val;     
      udb_i32wf_to_f64(&b);
      break;
    default:
      dbgwarn("Unknown KEY in lb2dm3_setkeyl: 0x%8.8X \n",key);
      b.i32[0] = 0;
  }
  /* Call the sg function */
  rtn = lb2dm3_sg_fcn(p, mport, fcnkey, &b);
  return (rtn);
}


int_4 lb2dm3_setkeyd(PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)
{
  int_4 rtn = 0;
  UDB b = udb_initializer;
  uint32_t fcnkey = LB2DM3_FCN_NULL;

  /* Convert ICEPIC keys to LB2DM3 function keys and call SET/GET function */
  if (key == KEY_RFFREQ) fcnkey = LB2DM3_FCN_RFFREQ|LB2DM3_FCN_SET;
  if (key == KEY_RFBW)   fcnkey = LB2DM3_FCN_RFBW|LB2DM3_FCN_SET;

  /* Reformat and scale input value to correct form for the key */
  switch (fcnkey) {
    case (LB2DM3_FCN_RFFREQ|LB2DM3_FCN_SET):
      b.f64[0] = r8val;     
      break;
    case (LB2DM3_FCN_RFBW|LB2DM3_FCN_SET):  
      b.f64[0] = r8val;     
      break;
    case (LB2DM3_FCN_SAMPRATE|LB2DM3_FCN_SET):
      b.f64[0] = r8val;     
      break;
    default:
      dbgwarn("Unknown KEY in lb2dm3_setkeyd: 0x%8.8X \n",key);
      b.i32[0] = 0;
  }
  /* Call the sg function */
  rtn = lb2dm3_sg_fcn(p, mport, fcnkey, &b);
  return (rtn);
}

