/****************************************************************************/
/*                                                                          */
/*   A2DM1X module interface routines                                       */
/*                                                                          */
/*   Note: all jpmem access must be to port 2 as this dual site module      */
/*         is only controlled from the B sides JTAG connection              */
/*                                                                          */
/****************************************************************************/

#include "iomlib_a2dm1x.h"

int32_t mod_a2dm18_hhreset(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,5,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,0,0xFFFF);
  cmdcfg_delay(mco,10000);

  return(0);
}

int32_t mod_a2dm18_reset(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,0,0xFFFF);
  cmdcfg_delay(mco,10000);

  return(0);
}

int32_t mod_a2dm18_set_gain(PT_PE4302_REG *S1, PT_PE4302_REG *S2, UDB *gain)
{
  int32_t rtn = 0;
  uint8_t rgvalA, rgvalB;
  const float64_t max_pga_gain = 28.0; /* 15.0 + 15.0 - 2.0 */
  const float64_t max_pga_attn = 63.0; /* 31.5 + 31.5       */

  UDB idx = udb_initializer;
  UDB pga_des = udb_initializer;
  UDB pga_gain_max = udb_initializer;
  UDB pga_gain_min = udb_initializer;
  UDB total_attn = udb_initializer;

  const uint8_t pga_gain_tbl[64][2] = {
   {0x00,0x00}, {0x00,0x02}, {0x00,0x04}, {0x00,0x06}, {0x00,0x08}, {0x00,0x0A},
   {0x00,0x0C}, {0x00,0x0E}, {0x00,0x10}, {0x00,0x12}, {0x00,0x14}, {0x00,0x16},
   {0x00,0x18}, {0x00,0x1A}, {0x00,0x1C}, {0x00,0x1E}, {0x00,0x20}, {0x02,0x20},
   {0x04,0x20}, {0x06,0x20}, {0x08,0x20}, {0x0A,0x20}, {0x0C,0x20}, {0x0E,0x20},
   {0x10,0x20}, {0x12,0x20}, {0x14,0x20}, {0x16,0x20}, {0x18,0x20}, {0x1A,0x20},
   {0x1C,0x20}, {0x1E,0x20}, {0x20,0x20}, {0x22,0x20}, {0x24,0x20}, {0x28,0x20},
   {0x2A,0x20}, {0x2C,0x20}, {0x2E,0x20}, {0x30,0x20}, {0x32,0x20}, {0x34,0x20},
   {0x36,0x20}, {0x38,0x20}, {0x3A,0x20}, {0x3C,0x20}, {0x3E,0x20}, {0x3F,0x21},
   {0x3F,0x23}, {0x3F,0x25}, {0x3F,0x27}, {0x3F,0x29}, {0x3F,0x2B}, {0x3F,0x2D},
   {0x3F,0x2F}, {0x3F,0x31}, {0x3F,0x33}, {0x3F,0x35}, {0x3F,0x37}, {0x3F,0x39},
   {0x3F,0x3B}, {0x3F,0x3D}, {0x3F,0x3F}, {0x3F,0x3F} };

  /* First, see if max attenuation is requested (pga <= -100dB) */
  pga_des.i32[0] = gain->i32[0];

  if (pga_des.i32[0] <= -100) {
    total_attn.f64[0] = -100.0;
    rgvalA = pga_gain_tbl[63][0];
    rgvalB = pga_gain_tbl[63][1];
  }
  else {
    pga_gain_max.f64[0] = max_pga_gain + 0.5;
    pga_gain_min.f64[0] = max_pga_gain - max_pga_attn;
    udb_i32wf_to_f64(&pga_des);
    total_attn.f64[0] = pga_gain_max.f64[0] - pga_des.f64[0];
    total_attn.f64[0] = (total_attn.f64[0] >= 0.0) ? (total_attn.f64[0]) : (0.0);
    total_attn.f64[0] = (total_attn.f64[0] <= max_pga_attn) ? (total_attn.f64[0]) : (max_pga_attn);
    idx.f64[0] = total_attn.f64[0];
    udb_f64_to_i32wf(&idx);
    idx.i32[0] = (idx.i32[0] >=  0) ? (idx.i32[0]) : (0);
    idx.i32[0] = (idx.i32[0] <= 63) ? (idx.i32[0]) : (63);
    rgvalA = pga_gain_tbl[idx.i32[0]][0];
    rgvalB = pga_gain_tbl[idx.i32[0]][1];
  }

  S1->rg[0x00] = u32_setbits(S1->rg[0x00],PE4302_RGM_00_ATTN,rgvalA);
  S2->rg[0x00] = u32_setbits(S2->rg[0x00],PE4302_RGM_00_ATTN,rgvalB);

  return(rtn);
}

int32_t mod_a2dm18_set_rate(Module_Config_Obj *mco)
{
  int32_t rtn,i;
  UDB srate = udb_initializer;
  UDB refclk = udb_initializer;
  int32_t stat_func,stat_ID;
  int32_t stat_addr,stat_data;
  PT_ADF4350_REG SYN;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  /* Define sample rate */
  srate = udb_initializer;
  srate.i32[0] = mco->LS.i32[A2DM18_RG_SRATEW];
  srate.i32[1] = mco->LS.i32[A2DM18_RG_SRATEF];
  udb_i32wf_to_f64(&srate);

  /* Define reference frequency */
  refclk = udb_initializer;
  refclk.f64[0] = 10.0; 

  /* Setup clock */
  pt_adf4350_rgc_default(&SYN);

  rtn = pt_adf4350_setrate(&SYN, &srate, &refclk);

  /* Enable ADF4350 chip */
  cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PWR,0,0x0000,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PWR,0,0x0001,0xFFFF);

  for(i=0; i<10; ++i){
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,5,SYN.rg[5],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,4,SYN.rg[4],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,3,SYN.rg[3],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,2,SYN.rg[2],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,1,SYN.rg[1],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,0,SYN.rg[0],0xFFFFFFFF);
    cmdcfg_delay(mco,10000);
  
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PRD,1,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_data);
    /* printf("ADF4350 Lock Detect = 0x%0.4x\n",stat_data); */
    if ((stat_data & 0x0001) == 0x0001) {
      break;
    }else if (i == 9 ){
      printf("WARNING: A2DM1X clock synth failed to lock. Check reference.\n");
      rtn = -1;
    }
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return(rtn);
}

int32_t mod_a2dm18_get_adclm(Module_Config_Obj *mco, int32_t chan, int32_t *adclm)
{
  int32_t rtn;
  int32_t stat_func, stat_ID, stat_addr, stat_val;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  if (chan == 1) {
    cmdcfg_ctl(mco, PTID_ADCLM,0,PFUNC_RRD,0,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_val);
  }else if (chan == 2) {
    cmdcfg_ctl(mco, PTID_ADCLM,1,PFUNC_RRD,0,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_val);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);
  *adclm = adclm_to_cf_i32(stat_val);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);

}

int32_t mod_a2dm18_get_temp(Module_Config_Obj *mco, int32_t chan, int32_t *temp)
{
  int32_t rtn;
  int32_t stat_func, stat_ID, stat_addr;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  cmdcfg_ctl(mco, PTID_TMP112,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,temp);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);

}

int32_t mod_a2dm18_set_rfgain(Module_Config_Obj *mco, int32_t chan)
{
  int32_t rtn;
  UDB gain_cha = udb_initializer;
  UDB gain_chb = udb_initializer;
  PT_PE4302_REG ATN_1A, ATN_1B;
  PT_PE4302_REG ATN_2A, ATN_2B;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  /* Set Gain to maximum for calibration */
  gain_cha = udb_initializer;
  gain_chb = udb_initializer;

  /* Set gain to requested value */
  gain_cha.i32[0] = mco->LS.u32[A2DM18_RG_CHAGAIN];
  gain_chb.i32[0] = mco->LS.u32[A2DM18_RG_CHBGAIN];
  
  if ((chan == 1) || (chan == 3)) {
    mod_a2dm18_set_gain(&ATN_1A, &ATN_1B, &gain_cha);
    cmdcfg_ctl(mco, PTID_PE4302, 0, PFUNC_RWR, PE4302_RG_00, ATN_1A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
    cmdcfg_ctl(mco, PTID_PE4302, 1, PFUNC_RWR, PE4302_RG_00, ATN_1B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  }else if ((chan == 2) || (chan == 3)) {
    mod_a2dm18_set_gain(&ATN_2A, &ATN_2B, &gain_chb);
    cmdcfg_ctl(mco, PTID_PE4302, 2, PFUNC_RWR, PE4302_RG_00, ATN_2A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
    cmdcfg_ctl(mco, PTID_PE4302, 3, PFUNC_RWR, PE4302_RG_00, ATN_2B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);

}

int32_t mod_a2dm18_cfg_tuner(Module_Config_Obj *mco, int32_t chan)
{
  int32_t i;
  UDB srate = udb_initializer;
  PT_PTFD_REG TUN;  
  UDB tfreq = udb_initializer;
  int32_t samples = 256;

  /* Define sample rate */
  srate = udb_initializer;
  srate.i32[0] = mco->LS.i32[A2DM18_RG_SRATEW];
  srate.i32[1] = mco->LS.i32[A2DM18_RG_SRATEF];
  udb_i32wf_to_f64(&srate);

  /* Setup tuner */
  pt_ptfd_rgc_default(&TUN);

  /* Set tuner decimation */
  TUN.nm.dec = mco->LS.i32[A2DM18_RG_TDEC];

  /* Define tuner freq */
  tfreq = udb_initializer;
  if(chan == 2) {
    tfreq.i32[0] = mco->LS.i32[A2DM18_RG_T2FREQW];
    tfreq.i32[1] = mco->LS.i32[A2DM18_RG_T2FREQF];
  } else {
    tfreq.i32[0] = mco->LS.i32[A2DM18_RG_T1FREQW];
    tfreq.i32[1] = mco->LS.i32[A2DM18_RG_T1FREQF];
  }

  udb_i32wf_to_f64(&tfreq);

  /* Reset tuner */
  /* uint32_t ppsp = u32_getbits(A2DM18_PIN_PPSP,mco->LS.i32[A2DM18_RG_PIN]);*/
  /* TUN.nm.system = (ppsp<<1)|0; */
  TUN.nm.system = 0;

  if (chan == 2) 
    cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_SYS,TUN.nm.system,0xFFFFFFFF);
  else 
    cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_SYS,TUN.nm.system,0xFFFFFFFF);
 
  /* Set tuner frequency */
  if(TUN.nm.dec != 0) pt_ptfd_tune(&TUN, &srate, &tfreq);
  udb_f64_to_i32wf(&tfreq);
  
  if (chan == 2) {
    cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_TFREQW,tfreq.i32[0],0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_TFREQF,tfreq.i32[1],0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_DEC,TUN.nm.dec,0xFFFFFFFF);
  } else {
    cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_TFREQW,tfreq.i32[0],0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_TFREQF,tfreq.i32[1],0xFFFFFFFF);
    cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_DEC,TUN.nm.dec,0xFFFFFFFF);
  }

  samples = 256;
  for(i=0; i<samples; ++i){
    if (chan == 2)
      cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_TOFF+i,TUN.nm.twave[i],0xFFFFFFFF);
    else 
      cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_TOFF+i,TUN.nm.twave[i],0xFFFFFFFF);
  }

  /* TUN.nm.system = (ppsp<<1)|1; */
  TUN.nm.system = 1;

  if (chan == 2)
    cmdcfg_ctl(mco,PTID_PTFD,1,PFUNC_RWR,PTFD_RG_SYS,TUN.nm.system,0xFFFFFFFF);
  else 
    cmdcfg_ctl(mco,PTID_PTFD,0,PFUNC_RWR,PTFD_RG_SYS,TUN.nm.system,0xFFFFFFFF);

  if (chan == 2) {
    mco->LS.i32[A2DM18_RG_T2FREQW] = tfreq.i32[0];
    mco->LS.i32[A2DM18_RG_T2FREQF] = tfreq.i32[1];
  } else {
    mco->LS.i32[A2DM18_RG_T1FREQW] = tfreq.i32[0];
    mco->LS.i32[A2DM18_RG_T1FREQF] = tfreq.i32[1];
  }

  return (0);
}


int32_t mod_a2dm18_set_rffreq(Module_Config_Obj *mco, int32_t chan)
{
  int32_t rtn;

  if( mco->LS.i32[A2DM18_RG_TDEC] == 0) {
    printf("WARNING: Tuners Not Currently Enabled!\n");
    return(0);
  }
    
  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  mod_a2dm18_cfg_tuner(mco, chan);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);
}

int32_t mod_a2dm18_setup(Module_Config_Obj *mco)
{
  int32_t rtn,i;
  UDB srate = udb_initializer;
  UDB refclk = udb_initializer;
  int32_t chan;
  PT_PE4302_REG ATN_1A, ATN_1B;
  PT_PE4302_REG ATN_2A, ATN_2B;
  UDB gain_cha = udb_initializer;
  UDB gain_chb = udb_initializer;
  PT_ADF4350_REG SYN;
  int32_t stat_func,stat_ID;
  int32_t stat_addr,stat_data;
  PT_ADC12D_REG ADC;
  int32_t ical;

  /* Define sample rate */
  srate = udb_initializer;
  srate.i32[0] = mco->LS.i32[A2DM18_RG_SRATEW];
  srate.i32[1] = mco->LS.i32[A2DM18_RG_SRATEF];
  udb_i32wf_to_f64(&srate);

  /* Define reference frequency */
  refclk = udb_initializer;
  refclk.f64[0] = 10.0; 

  /* Get current channel */
  chan = 1;
  if ((mco->LS.u32[A2DM18_RG_FCN])&(A2DM18_FCN_SIDEB))
    chan = 2;
  else
    chan = 1;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  /* Setup pin configuration */
  cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_PWR,0,mco->LS.u32[A2DM18_RG_PIN],0xFFFFFFFF);
  cmdcfg_delay(mco,10000);

  /* Set ADLM */
  cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,ADCLM_RG_B1,mco->LS.u32[A2DM18_RG_ADLMB1],ADCLM_RGM_B1);
  cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,ADCLM_RG_B2,mco->LS.u32[A2DM18_RG_ADLMB2],ADCLM_RGM_B2);
  cmdcfg_ctl(mco,PTID_ADCLM,0,PFUNC_RWR,ADCLM_RG_DEC,mco->LS.u32[A2DM18_RG_ADLMD],ADCLM_RGM_DEC);

  cmdcfg_ctl(mco,PTID_ADCLM,1,PFUNC_RWR,ADCLM_RG_B1,mco->LS.u32[A2DM18_RG_ADLMB1],ADCLM_RGM_B1);
  cmdcfg_ctl(mco,PTID_ADCLM,1,PFUNC_RWR,ADCLM_RG_B2,mco->LS.u32[A2DM18_RG_ADLMB2],ADCLM_RGM_B2);
  cmdcfg_ctl(mco,PTID_ADCLM,1,PFUNC_RWR,ADCLM_RG_DEC,mco->LS.u32[A2DM18_RG_ADLMD],ADCLM_RGM_DEC);

  /* Set Gain to maximum for calibration */
  gain_cha = udb_initializer;
  gain_chb = udb_initializer;

  gain_cha.i8[0] = -100;
  gain_chb.i8[0] = -100;

  mod_a2dm18_set_gain(&ATN_1A, &ATN_1B, &gain_cha);
  mod_a2dm18_set_gain(&ATN_2A, &ATN_2B, &gain_chb);

  cmdcfg_ctl(mco, PTID_PE4302, 0, PFUNC_RWR, PE4302_RG_00, ATN_1A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 1, PFUNC_RWR, PE4302_RG_00, ATN_1B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 2, PFUNC_RWR, PE4302_RG_00, ATN_2A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 3, PFUNC_RWR, PE4302_RG_00, ATN_2B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);

  /* Setup clock */
  pt_adf4350_rgc_default(&SYN);
  rtn = pt_adf4350_setrate(&SYN, &srate, &refclk);

  /* Enable ADF4350 chip */
  cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PWR,0,0x0000,0xFFFF);
  cmdcfg_delay(mco,10000);
  cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PWR,0,0x0001,0xFFFF);

  for(i=0; i<10; ++i){
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,5,SYN.rg[5],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,4,SYN.rg[4],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,3,SYN.rg[3],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,2,SYN.rg[2],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,1,SYN.rg[1],0xFFFFFFFF);
    cmdcfg_delay(mco,10);
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_RWR,0,SYN.rg[0],0xFFFFFFFF);
    cmdcfg_delay(mco,10000);
  
    cmdcfg_ctl(mco,PTID_ADF4350,0,PFUNC_PRD,1,0,0xFFFFFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_ID,&stat_addr,&stat_data);
    /* printf("ADF4350 Lock Detect = 0x%0.4x\n",stat_data); */
    if ((stat_data & 0x0001) == 0x0001) {
      break;
    }else if (i == 9 ){
      printf("WARNING: A2DM1X clock synth failed to lock. Check reference.\n");
      rtn = -1;
    }
  }

  /* Setup ADC */
  pt_adc12d_rgc_default(&ADC);

  ADC.rg[ADC12D_RG_CONFIG1] = u32_setbits(ADC.rg[ADC12D_RG_CONFIG1],ADC12D_RGM_CONFIG1_PDI,1);
  ADC.rg[ADC12D_RG_CONFIG1] = u32_setbits(ADC.rg[ADC12D_RG_CONFIG1],ADC12D_RGM_CONFIG1_PDQ,1);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CONFIG1    , ADC.rg[ADC12D_RG_CONFIG1]   , 0xFFFF);
  cmdcfg_delay(mco,1000);
  ADC.rg[ADC12D_RG_CONFIG1] = u32_setbits(ADC.rg[ADC12D_RG_CONFIG1],ADC12D_RGM_CONFIG1_PDI,0);
  ADC.rg[ADC12D_RG_CONFIG1] = u32_setbits(ADC.rg[ADC12D_RG_CONFIG1],ADC12D_RGM_CONFIG1_PDQ,0);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CONFIG1    , ADC.rg[ADC12D_RG_CONFIG1]   , 0xFFFF);
  cmdcfg_delay(mco,1000);

  pt_adc12d_cfg(&ADC,&srate);

  /* Turn on / off test pattern mode */
  if (mco->LS.u32[A2DM18_RG_ADCTPM] == 1)
    ADC.rg[ADC12D_RG_CONFIG1] = u32_setbits(ADC.rg[ADC12D_RG_CONFIG1],ADC12D_RGM_CONFIG1_TPM,0x01);

  /* Configure ADC Hardware */
  cmdcfg_ctl(mco,PTID_ADC12D,0,PFUNC_PWR,0,ADC.rg[ADC12D_RG_PIN],0xFFFF);

  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CONFIG1    , ADC.rg[ADC12D_RG_CONFIG1]   , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_RES1       , ADC.rg[ADC12D_RG_RES1]      , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_I_OFF_ADJ  , ADC.rg[ADC12D_RG_I_OFF_ADJ] , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_I_FS_ADJ   , ADC.rg[ADC12D_RG_I_FS_ADJ]  , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CAL_ADJ    , ADC.rg[ADC12D_RG_CAL_ADJ]   , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CAL_VAL    , ADC.rg[ADC12D_RG_CAL_VAL]   , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_RES2       , ADC.rg[ADC12D_RG_RES2]      , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_DES_ADJ    , ADC.rg[ADC12D_RG_DES_ADJ]   , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_RES3       , ADC.rg[ADC12D_RG_RES3]      , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_RES4       , ADC.rg[ADC12D_RG_RES4]      , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_Q_OFF_ADJ  , ADC.rg[ADC12D_RG_Q_OFF_ADJ] , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_Q_FS_ADJ   , ADC.rg[ADC12D_RG_Q_FS_ADJ]  , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_APT_DEL_CA , ADC.rg[ADC12D_RG_APT_DEL_CA], 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_APT_DEL_FA , ADC.rg[ADC12D_RG_APT_DEL_FA], 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_AUTOSYNC   , ADC.rg[ADC12D_RG_AUTOSYNC]  , 0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_RES5       , ADC.rg[ADC12D_RG_RES5]      , 0xFFFF);

  /* Run a calibration cycle and wait for it to complete */
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CAL_ADJ, ADC12D_RGM_CAL_ADJ_CSS, ADC12D_RGM_CAL_ADJ_CSS);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CONFIG1, 0, ADC12D_RGM_CONFIG1_CAL);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RRD, ADC12D_RG_CONFIG1, 0,0xFFFF);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RWR, ADC12D_RG_CONFIG1, ADC12D_RGM_CONFIG1_CAL,ADC12D_RGM_CONFIG1_CAL);
  cmdcfg_delay(mco,100);
  cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_RRD, ADC12D_RG_CONFIG1, 0, 0xFFFF);

  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco, PTID_CMD, 0, &stat_func, &stat_ID, &stat_addr, &stat_data);
  cmdcfg_stat(mco, PTID_CMD, 0, &stat_func, &stat_ID, &stat_addr, &stat_data);

  for (ical=0; ical<100; ical++) {
    cmdcfg_ctl(mco, PTID_ADC12D, 0, PFUNC_PRD, 1, 0, 0xFFFF);
    cmdcfg_bufclr(mco);
    cmdcfg_stat(mco, PTID_CMD,0, &stat_func, &stat_ID, &stat_addr, &stat_data);
    if((stat_data&0x0001)==0) break;
    cmdcfg_delay(mco,10000);
  }
  if(ical == 100) {
    printf("ERROR: ADC Failed to Calibrate Properly\n");
    return(-3);
  }

  /* Set gain to requested value */
  gain_cha.u32[0] = mco->LS.u32[A2DM18_RG_CHAGAIN];
  gain_chb.u32[0] = mco->LS.u32[A2DM18_RG_CHBGAIN];
  
  mod_a2dm18_set_gain(&ATN_1A, &ATN_1B, &gain_cha);
  mod_a2dm18_set_gain(&ATN_2A, &ATN_2B, &gain_chb);

  cmdcfg_ctl(mco, PTID_PE4302, 0, PFUNC_RWR, PE4302_RG_00, ATN_1A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 1, PFUNC_RWR, PE4302_RG_00, ATN_1B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 2, PFUNC_RWR, PE4302_RG_00, ATN_2A.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);
  cmdcfg_ctl(mco, PTID_PE4302, 3, PFUNC_RWR, PE4302_RG_00, ATN_2B.rg[PE4302_RG_00], PE4302_RGM_00_ATTN);

  /* Configure Tuners */
  mod_a2dm18_cfg_tuner(mco, chan);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return(rtn);
}


int32_t mod_a2dm18_enable_output(Module_Config_Obj *mco)
{
  int32_t sysreg = 0x00000000;
  int32_t rtn;

  rtn = getmoduleconfig(mco);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  /* If PIN controls are enabled for module, copy them to sysreg */
  sysreg = 0x00000000;
  if ((mco->LS.u32[A2DM18_RG_FCN])&(A2DM18_FCN_PINCTL)) {
     sysreg = mco->LS.i32[A2DM18_RG_PIN];
     /* printf("PINCTL: %8.8X \n",sysreg); */
  }

  /* Enable Output */
  if (A2DM18_PIN_ENABLE & mco->LS.u32[A2DM18_RG_PIN]) {
    cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,ROOT_PIN_ENABLE,0xFFFF);
  }
  else {
    cmdcfg_ctl(mco,PTID_ROOT,0,PFUNC_PWR,0,ROOT_PIN_NULL,0xFFFF);

    mod_a2dm18_reset(mco);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  /* Release Command Lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);
}


int32_t mod_a2dm18_get_state(Module_Config_Obj *mco)
{
  /* Get module configuration */
  int32_t rtn = 0;
  rtn = mod_get_state(mco);
  if (rtn != 0) {
    printf("ERROR: Unable to get module state\n");
    return (-1);
  }

  /* Confirm we're dealing with A2DM18 module */
  /* printf("MODLIB: mod_a2dm18_get_state: %8.8X\n", mco->LS.u32[A2DM18_RG_FCN]); */

  return (0);
}


int32_t mod_a2dm18_set_state(Module_Config_Obj *mco)
{
  int32_t rtn = 0x00000000;

  /* printf("EXEC: mod_a2dm18_set_state...\n"); */

  /* Setup / configure the module */
  if (A2DM18_FCN_SETUP & mco->LS.u32[A2DM18_RG_FCN])
  rtn |= mod_a2dm18_setup(mco);

  /* Set Gain */
  if (A2DM18_FCN_CHG_RFGAINA & mco->LS.u32[A2DM18_RG_FCN])
  rtn |= mod_a2dm18_set_rfgain(mco,1);

  if (A2DM18_FCN_CHG_RFGAINB & mco->LS.u32[A2DM18_RG_FCN])
  rtn |= mod_a2dm18_set_rfgain(mco,2);

#ifdef BOGUS
  /* Set ADC Clock */
  if (A2DM18_FCN_CHG_SRATE & mco->LS.u32[A2DM18_RG_FCN])
#endif

  /* Set RF Freq */
  if (A2DM18_FCN_CHG_RFFREQA & mco->LS.u32[A2DM18_RG_FCN])
  rtn |= mod_a2dm18_set_rffreq(mco,1);

  if (A2DM18_FCN_CHG_RFFREQB & mco->LS.u32[A2DM18_RG_FCN])
  rtn |= mod_a2dm18_set_rffreq(mco,2);

  /* Enable or disable module output */
  if (A2DM18_FCN_CHG_ENABLE & mco->LS.u32[A2DM18_RG_FCN])
    rtn |= mod_a2dm18_enable_output(mco);

  /* Update module state to configured running state */
  if (rtn == 0) {
    rtn = mod_set_state(mco);
  }
  else {
    printf("ERROR: SET_STATE failed.\n");
    return (-1);
  }

  return (0);
}

int32_t mod_a2dm18_init(Module_Config_Obj *modcfg)
{
  /* printf("EXEC: mod_a2dm18_init...\n"); */
  /* If necessary, load firmware, etc. */


  /* Configure the module object */

  /* Establish communications */


  /* Get current state */

  return (0);
}

int_4 a2dm1x_reset(PICSTRUCT *p)
{
  int_4 rtn;
  int_4 a2dports;

  /* Create a module handle and configure device transport */
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);
  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM;

  rtn = getmoduleconfig(A2DM18);
  if(rtn<0){printf("ERROR: Command Module is Already Locked!!\n"); return(-1); }

  mod_a2dm18_hhreset(A2DM18);

  a2dports = (p->qdrx>0)?4:2;
  a2dports = findintflagdef("A2DPORTS",p->config,a2dports);
  if(a2dports==3) {
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_DUAL;
  }else if(a2dports==4){ 
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_DWIDE;
  }

  /* Setup pin configuration */
  cmdcfg_ctl(A2DM18,PTID_STATE,0,PFUNC_PWR,0,A2DM18->LS.u32[A2DM18_RG_PIN],0xFFFFFFFF);
  cmdcfg_delay(A2DM18,10000);

  /* Wait for Command Buffer to Execute */
  cmdcfg_bufclr(A2DM18);

  /* Release Command Lock */
  cmdcfg_ctl(A2DM18,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return(0);
}

int_4 a2dm1x_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, 
		     real_8 rate, int_4 gain, int_4 flags)
{ 
  /* Create a module handle and configure device transport */
  int_4 rtn,bpa;
  real_8 tfreq = 0.0;
  UDB tf = udb_initializer;
  int_4 busrate;
  int_4 tdec = 0;
  UDB b = udb_initializer;
  int_4 byte_loading = 0;
  int_4 a2dports = 0;
  int_4 adclmB1 = 0; 
  int_4 adclmB2 = 0; 
  int_4 adclmD  = 0; 
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;

  mod_force_init_obj(A2DM18);

  /* printf("IOM: a2dm18_setup: %8.8X\n",flags); */

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM;

  /* Get module's current state into local struct, then modify */
  rtn = mod_a2dm18_get_state(A2DM18);

  /* Reset Module */
  mod_a2dm18_reset(A2DM18);

  /* Tuner Setup */
  /* tfreq = finddblflagdef("RFFREQ",p->config,150.0) * 1000000.0; */
/*
  tfreq = finddblflagdef("RFFREQ",p->config,150.0 * 1000000.0);
  if(fabs(tfreq) < 10000.0) tfreq = tfreq * 1000000.0;
  if(tfreq > 2.147e9) {
     printf("WARNING: Cannot tune higher than 2.147GHz, setting tune frequency to 2.147GHz\n");
     tfreq = 2.147e9;
  }
*/

  tfreq = finddblflagdef("RFFREQ",p->config,150.0);

  tf = udb_initializer;
  tf.f64[0] = tfreq;	/* negative to mix with conjugate */
  udb_f64_to_i32wf(&tf);
  if(mport == 2) {
    A2DM18->LS.u32[A2DM18_RG_T2FREQW] = tf.u32[0];
    A2DM18->LS.u32[A2DM18_RG_T2FREQF] = tf.u32[1];
  } else {
    A2DM18->LS.u32[A2DM18_RG_T1FREQW] = tf.u32[0];
    A2DM18->LS.u32[A2DM18_RG_T1FREQF] = tf.u32[1];
  }

  bpa = (bits<0)? -2*bits/8 : bits/8;
  tdec = (bits<0)? 4:0;
  tdec = findintflagdef("RFDEC",p->config,tdec);
  if(tdec==4)
    A2DM18->LS.u32[A2DM18_RG_TDEC] = 1;
  else if(tdec==8)
    A2DM18->LS.u32[A2DM18_RG_TDEC] = 2;
  else
    A2DM18->LS.u32[A2DM18_RG_TDEC] = 0;

  /* Sample rate (convert to MHz) */
  b = udb_initializer;
  busrate = (int_4)(rate/1e6) * bpa;
  /* if(p->verbose) printf("LVDS Rate = %d MBPS\n",busrate); */
  rate = (bits<0 && tdec!=0)? rate*tdec : rate;
  if (rate < 200000000.0) {
    rate = rate*4.0;
    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDEC,3);
    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDECD,2);
  } else if (rate < 400000000.0) {
    rate = rate*2.0;
    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDEC,1);
    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDECD,0);
  } else {

    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDEC,0);
    A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_HDECD,0);
  }
  b.f64[0] = ((real_8)rate)/(1000000.0);
  if (rate > 0) {
    udb_f64_to_i32wf(&b);
    A2DM18->LS.i32[A2DM18_RG_SRATEW] = b.i32[0];
    A2DM18->LS.i32[A2DM18_RG_SRATEF] = b.i32[1];
  }  

  /* Gain */
  if(mport == 1)
    A2DM18->LS.u32[A2DM18_RG_CHAGAIN] = gain;
  else
    A2DM18->LS.u32[A2DM18_RG_CHBGAIN] = gain;

  /* Clocking and triggering configuration */
  if (findflag("PREFX",p->config) >= 0) {
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_EXTREF;
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_TXPD;
  } else {
    A2DM18->LS.u32[A2DM18_RG_PIN] &= (~A2DM18_PIN_EXTREF);
    A2DM18->LS.u32[A2DM18_RG_PIN] &= (~A2DM18_PIN_TXPD);
  }

  if(abs(bits) == 16)
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_FMT; 
  else
    A2DM18->LS.u32[A2DM18_RG_PIN] &= (~A2DM18_PIN_FMT); 

  byte_loading = findintflagdef("BSEL",p->config,0);
  A2DM18->LS.i32[A2DM18_RG_PIN] = u32_setbits(A2DM18->LS.i32[A2DM18_RG_PIN],A2DM18_PIN_BSEL,byte_loading);

  if(findintflag("RSEL",p->config) > 0)
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_RSEL;

  if(findintflag("PPST",p->config) > 0)
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_PPST;

  if(findintflag("LSBP",p->config) > 0)
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_LSBP;

  if(findintflag("PPSP",p->config) > 0)
    A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_PPSP;

  a2dports = (p->qdrx>0)? 4:2;
  /* check if lvds rate is > 1000 - on PIC6 need a2dports=3, on PIC8 need QDRX (a2dports=4) */
  /* check if lvds rate is > 2000 - not possible on PIC6, on PIC8 use a2dports=5 (along with QDRX) */
  if(p->type==ICEPIC6 && busrate>1000) a2dports=3;
  if(p->type>ICEPIC6 && busrate > 2000) a2dports=5;
  a2dports = findintflagdef("A2DPORTS",p->config,a2dports);
  if(a2dports==5) A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_PWIDE;
  else A2DM18->LS.u32[A2DM18_RG_PIN] &= (~A2DM18_PIN_PWIDE);
  if(a2dports==4 || a2dports==5) A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_DWIDE;
  else A2DM18->LS.i32[A2DM18_RG_PIN] &= (~A2DM18_PIN_DWIDE);
  if(a2dports==3) A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_DUAL;
  else A2DM18->LS.i32[A2DM18_RG_PIN] &= (~A2DM18_PIN_DUAL);

  if (a2dports == 5 && mport == 2) A2DM18->LS.u32[A2DM18_RG_PIN] |= A2DM18_PIN_VSEL;
  else A2DM18->LS.u32[A2DM18_RG_PIN] &= (~A2DM18_PIN_VSEL);
  vprint("M18 a2dports = %d (pin=0x%x)\n",a2dports,A2DM18->LS.u32[A2DM18_RG_PIN]);

  /* ADCLM Setup */
  adclmB1 = findintflagdef("ADCLMB1",p->config,9)&0xFF;
  adclmB2 = findintflagdef("ADCLMB2",p->config,15)&0xFF;
  adclmD  = findintflagdef("ADCLMD", p->config,10)&0xFF;
  A2DM18->LS.u32[A2DM18_RG_ADLMB1] = adclmB1;
  A2DM18->LS.u32[A2DM18_RG_ADLMB2] = adclmB2;
  A2DM18->LS.u32[A2DM18_RG_ADLMD]  = adclmD;

  /* Set state change keys (Disable output and force module setup) */
  A2DM18->LS.u32[A2DM18_RG_FCN]  = A2DM18_FCN_NULL;
  A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_PINCTL; 
  A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_SETUP;
  if (mport == 2) 
    A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_SIDEB;
  
  /* Synchronize the module state with the local state */
  rtn = mod_a2dm18_set_state(A2DM18); 

  return (a2dports);
}

int_4 a2dm1x_enable(PICSTRUCT *p, int_4 mport, int_4 ena)
{
  /* Create a module handle and configure device transport */
  int_4 rtn;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

  /* printf("IOM: a2dm18 enable: %8.8X\n",ena); */

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);

  /* Update the enable / disable PIN control for the module */
  if (ena == 0) 
    A2DM18->LS.i32[A2DM18_RG_PIN] &= (~A2DM18_PIN_ENABLE);
  else
    A2DM18->LS.i32[A2DM18_RG_PIN] |= A2DM18_PIN_ENABLE;

  /* Clear state key reg, force pin ctrl update, enable output */
  A2DM18->LS.u32[A2DM18_RG_FCN]  = A2DM18_FCN_NULL;
  A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_PINCTL; 
  A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_CHG_ENABLE;

  /* Set module's state to local state */
  rtn = mod_a2dm18_set_state(A2DM18);

  return(0);
}

int_4 a2dm1x_get_gain(PICSTRUCT *p, int_4 mport)
{ 
  /* Create a module handle and configure device transport */
  int_4 rtn = 0;
  int_4 gain = 0;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);

  if(mport==2)
    gain = A2DM18->LS.i32[A2DM18_RG_CHBGAIN];
  else
    gain = A2DM18->LS.i32[A2DM18_RG_CHAGAIN];

#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif

  return (gain);
}

int_4 a2dm1x_set_gain(PICSTRUCT *p, int_4 mport, int_4 gain)
{ 
  /* Create a module handle and configure device transport */
  int_4 rtn;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);
  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);
  if(rtn<0) return(rtn);

  if(mport==2) {
    A2DM18->LS.i32[A2DM18_RG_CHBGAIN] = gain;
    A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_CHG_RFGAINB;
  }else {
    A2DM18->LS.i32[A2DM18_RG_CHAGAIN] = gain;
    A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_CHG_RFGAINA;
  }

  rtn = mod_a2dm18_set_state(A2DM18);

  return (0);
}

real_8 a2dm1x_get_freq (PICSTRUCT *p, int_4 mport)
{
  /* Create a module handle and configure device transport */
  int_4 rtn = 0;
  UDB rffreq = udb_initializer;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);

  rffreq = udb_initializer;
  if(mport == 2) {
    rffreq.i32[0] = A2DM18->LS.i32[A2DM18_RG_T2FREQW];
    rffreq.i32[1] = A2DM18->LS.i32[A2DM18_RG_T2FREQF];
  } else {
    rffreq.i32[0] = A2DM18->LS.i32[A2DM18_RG_T1FREQW];
    rffreq.i32[1] = A2DM18->LS.i32[A2DM18_RG_T1FREQF];
  }

  udb_i32wf_to_f64(&rffreq);

#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif

  /* return (rffreq.f64[0]/1000000.0); */
  return (rffreq.f64[0]);
}

int_4 a2dm1x_set_freq (PICSTRUCT *p, int_4 mport, real_8 freq)
{
  /* Create a module handle and configure device transport */
  int_4 rtn = 0;
  UDB tfreq = udb_initializer;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);
  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);
  if(rtn<0) return(rtn);

  tfreq = udb_initializer;
  /* tfreq.f64[0] = freq * 1000000.0; */
  tfreq.f64[0] = freq;
  udb_f64_to_i32wf(&tfreq);

  if(mport==2) {
    A2DM18->LS.i32[A2DM18_RG_T2FREQW] = tfreq.i32[0];
    A2DM18->LS.i32[A2DM18_RG_T2FREQF] = tfreq.i32[1];
    A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_CHG_RFFREQB;
  }else {
    A2DM18->LS.i32[A2DM18_RG_T1FREQW] = tfreq.i32[0];
    A2DM18->LS.i32[A2DM18_RG_T1FREQF] = tfreq.i32[1];
    A2DM18->LS.u32[A2DM18_RG_FCN] |= A2DM18_FCN_CHG_RFFREQA;
  }

  rtn = mod_a2dm18_set_state(A2DM18);

  return (0);
}

int_4 a2dm1x_get_adclm (PICSTRUCT *p, int_4 mport)
{
  /* Create a module handle and configure device transport */
  int_4 rtn = 0; 
  int_4 adclm = 0;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_adclm(A2DM18,mport,&adclm);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return (adclm);
}

int_4 a2dm1x_get_temp (PICSTRUCT *p, int_4 mport)
{
  /* Create a module handle and configure device transport */
  int_4 rtn = 0;
  int_4 temp = 0;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  temp = 0;
  rtn = mod_a2dm18_get_temp(A2DM18,mport,&temp);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return (temp);
}

real_8 a2dm1x_get_rate (PICSTRUCT *p, int_4 mport)
{
  /* Create a module handle and configure device transport */
  int_4 rtn = 0;
  UDB rffreq = udb_initializer;
  Module_Config_Obj tmp;
  Module_Config_Obj *A2DM18 = (Module_Config_Obj *) &tmp;
  mod_force_init_obj(A2DM18);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  if (A2DM18 == (Module_Config_Obj *)NULL) {
    printf("ERROR: Allocation of module object failed!!! \n");
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return (-2);
  }
  A2DM18->TGT.tgtid = C3_TGT_DEV_ICEPIC;
  A2DM18->TGT.icepic.pdev  = p;
  A2DM18->TGT.icepic.mport = A2DM1X_PORT_DSM; 

  /* Get module's current state into local struct */
  rtn = mod_a2dm18_get_state(A2DM18);

  rffreq = udb_initializer;
  rffreq.i32[0] = A2DM18->LS.i32[A2DM18_RG_SRATEW];
  rffreq.i32[1] = A2DM18->LS.i32[A2DM18_RG_SRATEF];

  udb_i32wf_to_f64(&rffreq);

#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif

  /* return (rffreq.f64[0]/1000000.0); */
  return (rffreq.f64[0]);
}

