/*
  Native entries for the ICE FileIO interface 

  @author Jeff Schoen 
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

FIOSTRUCT *fioOpen (const char *fname, int_4 flags) {
  int_4 iofunc,fd,stat,lfn; int_8 p8s[2]; char stmp[120]; FIOSTRUCT *f;
  if ((flags&FIO_INOUT)==FIO_INOUT) iofunc = O_RDWR;
  else if (flags&FIO_OUTPUT) iofunc = O_RDWR|O_CREAT;
  else iofunc = O_RDONLY;
  strcpy(stmp,fname);
  if (strstr(stmp,"ramd:")==stmp) { strstr(stmp,fname+5); flags |= FIO_ICERAM; }
  if (strstr(stmp,"file:")==stmp) { strstr(stmp,fname+5); }
  lfn = strlen(stmp);
  fd = open (stmp,iofunc,0666);
  if (fd<=0) return NULL;
  f = (FIOSTRUCT*) calloc(sizeof(FIOSTRUCT),1);
  if (f==NULL) return NULL;
  f->fd = f->fdh = fd;
  f->flags = flags;
  f->start = 0;
  if (flags&FIO_BLUE) {
    stat = read (fd, (void *)(&f->hcb), 512);
    if (strncmp(f->hcb.version,"BLUE",4)!=0) { printf("fioOpen name=%s not a BLUE file\n",fname); fioClose(f); return NULL; }
    f->start = (int_8)f->hcb.data_start;
    f->length = (int_8)f->hcb.data_size;
    f->offset = 0;
    if (flags&FIO_VERBOSE) printf("fioOpen BLUE fname=%s len=%lld\n",stmp,f->length);
    if (f->hcb.detached>0) {
      strcpy(stmp+lfn-3,"det");
      fd = open (stmp,iofunc,0666);
      if (fd<=0) { fioClose(f); return NULL; }
      if (flags&FIO_ICERAM) {	// read ICERAM buffer location
	stat = read (fd, p8s, 16);
	if (stat!=16) { fioClose(f); return NULL; }
	f->paddr=p8s[0]; f->psize=p8s[1];
	if (flags&FIO_VERBOSE) printf("fioOpen BLUE dname=%s off=%llx len=%ld\n",stmp,f->paddr,f->psize);
	f->pbuffer = pic_mapram(f->paddr,f->psize);
	if (f->pbuffer==NULL) { fioClose(f); return NULL; }
	close(fd); fd=-2;
      } else {
	if (flags&FIO_VERBOSE) printf("fioOpen BLUE dname=%s len=%lld\n",stmp,f->length);
      }
      f->fd = fd;
    }
  } else {
    if (f->flags&FIO_VERBOSE) printf("fioOpen fname=%s fd=%d\n",fname,fd);
  }
  return f;
}

int_4 fioClose (FIOSTRUCT *f) {
  if (f->flags&FIO_VERBOSE) printf("fioClose fd=%d fdh=%d\n",f->fd,f->fdh);
  if (f->pbuffer!=NULL) pic_unmapram(f->paddr,f->psize,f->pbuffer);
  if (f->fd>0) close (f->fd);
  if (f->fdh>0 && f->fdh!=f->fd) close (f->fdh);
  free(f);
  return 0;
}

int_4 fioSeek (FIOSTRUCT *f, int_8 offset) {
  int_8 loff=offset+f->start,lret;
  lret = lseek (f->fd, loff, SEEK_SET);
  f->offset = offset;
  return (loff==lret)? 0 : -1;
}

int_8 fioAvail (FIOSTRUCT *f, int_4 len, int_4 dir) {
  int_8 lcur,loff,avail=0; int_4 stat,seek=0;
  if (f->flags&FIO_RTFILE) {
    lcur = (int_8)f->hcb.in_byte;
    if (dir<0 && lcur >= f->offset+f->length) {	// fell behind a buffer or initial sync up
      f->offset = lcur - (f->length>>1);	// skip to halfway back
      f->offset = (f->offset/len)*len;		// round to element
      seek++;
    }
    if (dir>0 && lcur >= f->offset) {		// fell behind a buffer or initial sync up
      f->offset = lcur;				// skip to current back
      f->offset = (f->offset/len)*len;		// round to element
      seek++;
    }
    loff = f->offset % f->length;
    if (loff==0 && f->loff!=0) {
      seek++;
      if (f->flags&FIO_VERBOSE) printf("fioAvail at RTF top fd=%d fdh=%d offset=%llx\n",f->fd,f->fdh,f->offset);
    }
    if (loff+len > f->length) { printf("fioAvail: RTF length=%ld not multiple of xfer=%d\n",f->length, len); return -1; }
    f->loff = loff;
    if (seek && f->fd>0) lseek (f->fd, loff+f->start, SEEK_SET);
    if (dir<0 && lcur-f->offset>=len) return len;
    lseek (f->fdh, 80L, SEEK_SET);				//  seek to header
    if (dir<0) {
      stat = read (f->fdh, (void *)(&f->hcb.in_byte), 8);	//  get new inbyte
      if (stat!=8) printf("fioAvail: err reading header bytes=8 stat=%d\n",stat);
    } else {
      f->hcb.in_byte += len;
      stat = write (f->fdh, (void *)(&f->hcb.in_byte), 8);	//  set new inbyte
      if (stat!=8) printf("fioAvail: err writing header bytes=8 stat=%d\n",stat);
      avail = len;
    }
    if (f->fd>0) lseek (f->fd, loff+f->start, SEEK_SET);	// seek back to current
  } else {
    if (dir>0) avail = len;
    else avail = f->length - f->offset;
  }
  return avail;
}

int_4 fioWrite (FIOSTRUCT *f, void *buffer, int_4 len) {
  int_4 stat=0; int_8 avail = fioAvail (f,len,1);
  if (avail<=0) return avail;
  if (len>avail) len=avail;
  if (f->fd<0) { stat = len; memcpy((char*)f->pbuffer+f->loff,buffer,len); }
  else stat = write (f->fd, buffer, len);
  if (stat>0) f->offset += stat;
  return stat;
}

int_4 fioRead (FIOSTRUCT *f, void *buffer, int_4 len) {
  int_4 stat=0; int_8 avail = fioAvail (f,len,-1);
  if (avail<=0) return avail;
  if (len>avail) len=avail;
  if (f->fd<0) { stat = len; memcpy(buffer,(char*)f->pbuffer+f->loff,len); }
  else stat = read (f->fd, buffer, len);
  if (stat>0) f->offset += stat;
  return stat;
}




int_4 fioGetKey (const char *table, const char *key, char *value, int_4 flag) {
  int_4 lk,lv,curly=0; char c,match[40],*sp,*sm;
  sp = strchr(key,'.');	/* sub table ? */
  lk = (sp!=NULL)? sp-key : strlen(key);
  strncpy(match,key,lk); match[lk]='='; match[lk+1]=0;
  sm = strstr(table,match);
#if DEBUG
  printf("fioGetKey tbl=%s key=%s lk=%d match=%s sm=%p\n",table,key,lk,match,sm);
#endif
  if (sm==NULL) { value[0]=0; return -1; }
  sm += (lk+1);
  for (lv=0; ;lv++) {
    c = sm[lv];
    if (c=='{') curly++;
    if (c==','||c=='}'||c==0) { if (curly==0) break; }
    if (c=='}') curly--;
  }
  if (sp!=NULL) return fioGetKey(sm,key+lk+1,value,flag);
  strncpy(value,sm,lv); value[lv]=0;
  return lv;
}

int_4 fioGetKeyState (const char *table, const char *key, int_4 def) {
  char value[40]; int i, lv = fioGetKey(table,key,value,0);
  if (lv<=0) return def;
  for (i=0; i<lv; i++) value[i] = toupper(value[i]);
  if (strstr("1,T,TRUE,Y,YES,ON",value)!=NULL) return 1;
  if (strstr("0,F,FALSE,Y,NO,OFF",value)!=NULL) return 0;
  return -1;
}
int_4 fioGetKeyL (const char *table, const char *key, int_4 def) {
  char value[40]; int lv = fioGetKey(table,key,value,0);
  return (lv<=0)? def : atoi(value);
}
int_8 fioGetKeyX (const char *table, const char *key, int_8 def) {
  char value[40]; int lv = fioGetKey(table,key,value,0);
  return (lv<=0)? def : atol(value);
}
real_8 fioGetKeyD (const char *table, const char *key, real_8 def) {
  char value[40]; int lv = fioGetKey(table,key,value,0);
  return (lv<=0)? def : atof(value);
}
int_4 fioGetKeyCmp (const char *table, const char *key, const char *match) {
  char value[40]; int lv = fioGetKey(table,key,value,0);
  return (lv<=0)? -1 : strncmp(value,match,lv);
}
