package nxm.ice.lib;

import nxm.sys.inc.*;
import nxm.sys.lib.*;

/**
  Reference implementation of various timecode modes processed by ICE cards.

  These are not optimized for speed.
  They are primarily used to test the ICE library functions.

  SMS/SDN/DTL Timecode formats

 Time code is the time at the sample ONE BIT PRIOR to the Barker code [SDN],
 at the start of the first Barker bit [SMS], or at the midpoint (high-to-low
 edge of clock) of the first Barker bit [Datolite].

       Bit(s)          Data
       ------          -------------------------------
       .. 1            Fill bits: SMS = 0, SDN = 1
        1-7            7 bit Barker code, 1110010 = 114
        8-14           BCD seconds, LSB first
       15-21           BCD minutes
       22-27           BCD hours
       28-37           BCD days
  For SDN only:
       38-49           BCD milliseconds
       50-60           Binary VFT (very fine time), MSB first, units = 5 nsec
  VFT Extended Precision/Range:
       61-63           Bits -1,-2,-3 of VFT; inverted polarity
          64           Spare
       65-67           Bits 13,12,11 of VFT; inverted polarity

    * NOTE - this scheme can only address 81us above a 1ms hack, min sample rate = 12345 Hz
    * NOTE - barker can easily reappear in timecode string, require 4 bits of fill for SDN/DTL
    * NOTE - with 4 bit fill, minimum supported rate is 71kHz


  IRIG-B Timecode format

 Time code is once per second.  Format is pairs of BCD numbers

       Bit(s)          Data
       ------          -------------------------------
       02-08           BCD seconds, LSBit first, LSDigit first
       11-18           BCD minutes
       21-28           BCD hours
       31-38           BCD days
       41-44           BCD days*100


  SDDS Timecode format

 Time code is once per packet.  Format is binary numbers

       Bit(s)          Data Big Endian
       ------          -------------------------------
       00-04           status bits
       05-15
       16-31           # (250psec) hacks from 1ms ptr to SSC
       32-96           # (250psec) time increments

 IOC tvalid bits: 7=val 6=cyc 5=hit 4=nclk 3=den 2=ply 1=enb 0=ena

  @author Jeff Schoen
  @version %I%, %G%
*/
public class TimeCode {

  public final static int
        TCM_OFF=0, TCM_CPU=1, TCM_ZTC=2, TCM_SDN=3, TCM_SMS=4,
        TCM_DTL=5, TCM_IRB=6, TCM_SDDS=7;
  public final static String tcmodeList = "CPU,ZTC,SDN,SMS,DTL,IRB,SDDS";

  public  Time time = new Time();

  private int tcmode;
  private int tcbit;
  private int tcbits;
  private int tcflags;
  private byte[] buffer = new byte[128];
  private byte fill,mask;
  private final byte B1=1, B0=0;
  private double hack;
  private double yis; // year in seconds 

  private int OPPS=0x1, FILL=0x2, VFT=0x4, EXT=0x8, TDC=0x10;

  private int BARKER = 0x27;

  private boolean debug = false;

  public TimeCode() {
  }

  public void setTime (Time time) {
    this.time.fromTime(time);
    yis = time.getYiS();
  }

  public void setMode (String tcmodestr) {
    tcbit = 0;
    tcmode = 0;
    tcflags = 0;
    int len = tcmodestr.length();
    while (len<3) { tcmodestr += ' '; len++; }
    String s = tcmodestr.substring(0,3);
    tcmode = Parser.find(tcmodeList,s,0);
    if (tcmode<=0) throw new MidasException ("Mode string="+tcmodestr+" not found in "+tcmodeList);
    if (tcmode==TCM_SMS) tcbit=1;
    if (tcmode==TCM_SDN) { tcbit=4; tcflags |= VFT; }
    if (len<4) return;
    char c = tcmodestr.charAt(3);
         if (c=='X') ;
    else if (c=='S') ;
    else if (c=='0') tcbit = 0;
    else if (c=='1') tcbit = 1;
    else if (c=='3') tcbit = 3;
    else if (c=='4') tcbit = 4;
    else throw new MidasException ("Bad tcmode string bit field: "+tcmodestr);
    fill = (byte)(0x1<<tcbit);
    mask = (byte)(~fill);
    for (int i=4; i<len; i++) {
      c = tcmodestr.charAt(i);
           if (c=='P') tcflags |= OPPS;
      else if (c=='F') tcflags |= FILL;
      else if (c=='X') tcflags |= EXT;
      else if (c=='Y') tcflags &= ~VFT;
      else if (c=='D') tcflags |= TDC;
      else throw new MidasException ("Bad tcmode string flag field: "+tcmodestr);
    }
    hack = 0.001;
    if (tcmode==TCM_SDN) { tcbits = 67; }
    if (tcmode==TCM_DTL) { tcbits = 67; }
    if (tcmode==TCM_SMS) { tcbits = 37; hack=1; }
    if (tcmode==TCM_IRB) { tcbits = 48; hack=1; }
    // no extended precision ?
    if ((tcflags&EXT)==0) tcbits = Math.min(tcbits,60); 
    // init the buffer with fill and barker bits
    for (int i=0; i<128; i+=32) setBits (buffer,i,32,-1);
    setBits (buffer,1,7,BARKER);
  }

  public void setMode (int mode) {
    tcmode = mode;
  }

  public void setBit (int bit) {
    tcbit = bit;
  }

  public void setFlags (int flags) {
    tcflags = flags;
  }

  private void getTC() {
    if (debug) {
      System.out.print("TcBits=[");
      for (int i=0; i<tcbits; i++) System.out.print(buffer[i]);
      System.out.println("]");
    }
    if (tcmode==TCM_IRB)
      getIRB (buffer,-1,tcbits);
    else 
      getSDN (buffer,-1,tcbits);
  }

  private void setTC() {
    if (tcmode==TCM_IRB) setIRB (buffer,0,tcbits);
    else setSDN (buffer,0,tcbits);
  }

  public int nextHack (double delta) {
    int samples = (int)(hack/delta);
    time.addSec (samples*delta);
    setTC();
    return samples;
  }

  public int getHack (double delta) {
    int samples = (int)(hack/delta);
    return samples;
  }

  private void getIRB (byte[] buf, int bit, int nbits) {
    int i; double tc=0, ftc=0;
    bit += 8; // skip generated barker code
    i = getbcd (buf, bit+1,  4, 9);   tc += i;             // seconds
    i = getbcd (buf, bit+6,  3, 5);   tc += i*10;          // seconds*10
    i = getbcd (buf, bit+10, 4, 9);   tc += i*60;          // minutes
    i = getbcd (buf, bit+15, 4, 5);   tc += i*600;         // minutes*10
    i = getbcd (buf, bit+20, 4, 9);   tc += i*3600;        // hours
    i = getbcd (buf, bit+25, 4, 2);   tc += i*36000;       // hours*10
    i = getbcd (buf, bit+30, 4, 9);   tc += (i-1)*86400;   // day
    i = getbcd (buf, bit+35, 4, 9);   tc += i*864000;      // day*10
    i = getbcd (buf, bit+40, 4, 3);   tc += i*8640000;     // day*100
    tc += 1;    // decoded value is attached to the next seconds marker
    if (is(OPPS)) ftc = 0.011;  // double sync is 11mS past the nominal 1PPS
    time.fromJ1950 (tc,ftc);
  }

  private void setIRB (byte[] buf, int bit, int nbits) {
    double tc = time.getWSec();
    bit += 8; // skip generated barker code
    int day  = (int)(tc / 86400);	tc -= day*86400;
    int hour = (int)(tc / 3600);	tc -= hour*3600;
    int min  = (int)(tc / 60);		tc -= min*60;
    int sec  = (int)(tc);
    day--;	// one based
    setbcd (buf, bit+1,  4, sec%10);   // seconds
    setbcd (buf, bit+6,  3, sec/10);   // seconds*10
    setbcd (buf, bit+10, 4, min%10);   // minutes
    setbcd (buf, bit+15, 4, min/10);   // minutes*10
    setbcd (buf, bit+20, 4, hour%10);  // hours
    setbcd (buf, bit+25, 4, hour/10);  // hours*10
    setbcd (buf, bit+30, 4, day%10);   // day
    setbcd (buf, bit+35, 4, (day/10)%10); // day*10
    setbcd (buf, bit+40, 4, day/100);  // day*100
  }

  private void getSDN (byte[] buf, int bit, int nbits) {
    int i,k; double tc=0, ftc=0;

    if (nbits>38) { 		// SMS
      i = getbcd (buf, bit+8,   7,  59); tc += i;              // seconds
      i = getbcd (buf, bit+15,  7,  59); tc += i*60;           // minutes
      i = getbcd (buf, bit+22,  6,  23); tc += i*3600;         // hours
      i = getbcd (buf, bit+28, 10, 366); tc += (i-1)*86400;    // days
    }
    if (nbits>48) { 		// SDN
      i = getbcd (buf, bit+38, 12, 999);  ftc += i*1.0e-3;     // millisec */
    }
    if (nbits>59 && is(VFT)) {		// VFT
      for (i=0,k=50; k<=60; k++) i = (i<<1) + buf[bit+k];   ftc += i*5.0e-9;
    }
    if (nbits>66 && is(EXT)) { 	// extended VFT
      for (i=0,k=61; k<=63; k++) i = (i<<1) + (1-buf[bit+k]);   ftc += i*625.0e-12;
      for (i=0,k=65; k<=67; k++) i = (i<<1) + (1-buf[bit+k]);   ftc += i*10240.0e-9;
    }
    time.fromJ1950 (tc,ftc);
  }

  private void setSDN (byte[] buf, int bit, int nbits) {
    int i,k; double tc = time.getWSec()-yis, ftc=time.getFSec();

//System.out.println("Set SDN "+tc+" bits="+nbits);
    int day  = (int)(tc / 86400);	tc -= day*86400;
    int hour = (int)(tc / 3600);	tc -= hour*3600;
    int min  = (int)(tc / 60);		tc -= min*60;
    int sec  = (int)(tc);
    day++;	// one based

    // handle the standard SMS timecode fields
    setbcd (buf, bit+8,   7,  sec); // seconds
    setbcd (buf, bit+15,  7,  min); // minutes
    setbcd (buf, bit+22,  6, hour); // hours
    setbcd (buf, bit+28, 10,  day); // days


    if (nbits<49) return;    // no SDN
    int msec = (int)(ftc*1000);
    setbcd (buf, bit+38, 12, msec);   // millisec
//System.out.println("Set TC "+day+" "+hour+" "+min+" "+sec+" "+msec);

    if (nbits<60) return;;    // no VFT
    int tics = (int)( (ftc-msec*1e-3)/5e-9 );	// 5ns tics
    for (k=50,i=1<<10; k<=60; k++,i>>=1) buf[bit+k] = ((tics&i)!=0)? B1:B0;

    if (nbits<67 || !is(EXT)) return; // no extended VFT
    for (k=61; k<=63; k++) buf[bit+k] = B1;	// 625ps tics inverted
    for (k=65,i=1<<13; k<=67; k++,i>>=1) buf[bit+k] = ((tics&i)!=0)? B0:B1; // 5ns tics inverted
  }

  private boolean is (int flag) {
    return (flag&tcflags) != 0;
  }

  // least significant digit 1st
  private int getbcd (byte[] buf, int bit, int nbits, int maxval) {
    int i,j,bcd=0,mult=1,val=0;
    for (i=j=0; i<nbits; i++,j++) {
      bcd += (buf[bit+i] << j);
      if ( j==3 || i==nbits-1 ) {
        if (bcd>9) throw new MidasException("Bad BCD digit="+bcd+" at offset="+(bit+i));
        val += bcd*mult;        // new total
        mult *= 10; bcd = 0; j = -1;
      }
    }
    if (val > maxval) throw new MidasException("Bad BCD value="+val+" max="+maxval);
    return val;
  }

  private void setbcd (byte[] buf, int bit, int nbits, int val) {
    int i,bcd,xx,mult=1;
    for (i=0; i<nbits; ) {
      bcd = (val/mult) % 10;
      if (i<nbits) buf[bit+i] = ((bcd&0x1)!=0)? B1 : B0; i++;
      if (i<nbits) buf[bit+i] = ((bcd&0x2)!=0)? B1 : B0; i++;
      if (i<nbits) buf[bit+i] = ((bcd&0x4)!=0)? B1 : B0; i++;
      if (i<nbits) buf[bit+i] = ((bcd&0x8)!=0)? B1 : B0; i++;
      mult *= 10;
    }
  }

  private void setBits (byte[] buf, int bit, int nbits, int val) {
    for (int i=0; i<nbits; i++) {
      buf[bit+i] = (((val>>i)&0x1)!=0)? B1 : B0;
    }
  }

  public void setFillWords (byte[] data, int offset, int words) {
    for (int i=0; i<words; i++, offset+=2) data[offset] |= fill;
  }

  public void setTimeCodeWords (byte[] data, int offset, int words) {
    for (int i=0; i<words; i++, offset+=2) {
      data[offset] = (buffer[i]!=0)? (byte)(data[offset]|fill) : (byte)(data[offset]&mask);
    }
  }

  public int findTimeCodeWords (byte[] data, int offset, int words) {
    boolean tdc = (tcflags&TDC) != 0;
    if (tcmode==TCM_SDN) offset += 2; /* so I indicates one bit prior to barker */
    for (int i=0; i<words-tcbits; i++, offset+=2) {
     if (tdc) {
      if ( (data[offset+0] &fill) != 0 
        && (data[offset+4] &fill) != 0 
        && (data[offset+8] &fill) != 0 
        && (data[offset+12]&fill) == 0 
        && (data[offset+16]&fill) == 0 
        && (data[offset+20]&fill) != 0 
        && (data[offset+24]&fill) == 0
	// now check for pre fill
	&& (offset>=4  && (data[offset-4] &fill) != 0)
	&& (offset>=8  && (data[offset-8] &fill) != 0)
	&& (offset>=12 && (data[offset-12]&fill) != 0) ) {
        parseTimeCodeWords (data,offset,words-i);
        return i;
      }
     } else {
      if ( (data[offset+0] &fill) != 0 
        && (data[offset+2] &fill) != 0 
        && (data[offset+4] &fill) != 0 
        && (data[offset+6] &fill) == 0 
        && (data[offset+8] &fill) == 0 
        && (data[offset+10]&fill) != 0 
        && (data[offset+12]&fill) == 0
	// now check for pre fill
	&& (offset>=2 && (data[offset-2] &fill) != 0)
	&& (offset>=4 && (data[offset-4] &fill) != 0)
	&& (offset>=6 && (data[offset-6] &fill) != 0) ) {
        parseTimeCodeWords (data,offset,words-i);
        return i;
      }
     }
    }
    return -1;
  }

  private void parseTimeCodeWords (byte[] data, int offset, int words) {
    int inc = ((tcflags&TDC)!=0)? 4 : 2;
    for (int i=0; i<tcbits; i++, offset+=inc) {
      buffer[i] = ((data[offset]&fill)!=0)? B1 : B0;
    }
    getTC();
  }

}
