package nxm.ice.lib;

import nxm.sys.lib.*;
import java.io.File;
import nxm.ice.lib.DevIce.DmaMap;

/** 
  Resource for interacting with an ICE RamDisk using native code.
 
  @author Jeff Schoen
  @version $Id: RamDiskResource.java,v 1.5 2003/10/27 22:30:08 schoenj Exp $
*/
public class RamDiskResource extends IOResource {

  private int fd,fb;
  private File file;
  private static int PGSZ=8192, PGMSK=PGSZ-1;
  private long paddr=-1, psize;
  private long rdaddr, rdsize;
  private String rdpath;
  public  MDevIce pic;
  public  DmaMap map;

  public RamDiskResource() { super(); }

  /** Test for the existence of the resource. */
  public boolean exists () { return file.exists(); }

  public boolean isFile() { return true; }

  /** Set the URL that this resource points to. */
  public void setURL (String str) {
    super.setURL(str);
    if (str.startsWith("ramd:")) str = str.substring(5);
    file = new File (str);
  }

  /** open the resource */
  public boolean open () { 
    isOpen = false;
    String auxstr=null;
    Table auxtbl = (Table)M.results.get("AUX");
    String path = urlstr.substring(0,urlstr.lastIndexOf('/')+1);
    for (Table.Iterator ti=auxtbl.iterator(); auxstr==null && ti.getNext(); ) {
      String s = ti.value.toString(); if (s.startsWith(path)) auxstr = s; 
    }
    if (auxstr==null) M.error("RamDisk AUX entry not found for: "+urlstr);
    Parser p = new Parser(auxstr);
    rdpath = p.get(1).substring(5);
    String devn = p.get(2);
    rdaddr = Convert.s2xl (p.get(3));
    rdsize = Convert.s2xl (p.get(4));
    if (urlstr.endsWith("/")) return false; // just the directory
    int devno = (devn.length()==4)? (devn.charAt(3)-'0') : 0;
    pic = new MDevIce(M,"ICERAM,DEVNO="+devno+",IOM=NONE,PM=NONE");
    long maxalloc = (long)M.results.getL("ENV.ICE.MAXALLOC",-1);
    if (maxalloc>0) pic.setMaxAlloc(maxalloc<<20);
    if (rdaddr==0 && rdsize!=0) {
      pic.open();
      rdaddr = pic.phys2long( pic.getKeyL(0,MDevIce.KEY_QALLOC) );
      pic.close();
    }
    fd = Native.fopen (file.toString(),dir);
    byte[] detbuf = new byte[16];
    if (dir<=0) {
      Native.freadb (fd,detbuf,0,16);
      paddr = Convert.unpackX(detbuf,0);
      psize = Convert.unpackX(detbuf,8);
      length = Math.min(length,psize);
    } else {
      lockFileAlloc();
      Convert.packX(detbuf,0,-1L);
      Convert.packX(detbuf,8,1L);
      // add temp allocation to keep overwrite algorithm simple
      Native.fwriteb (fd,detbuf,0,16);
      Native.fseek(fd,0L);
      psize = (length+PGMSK)/PGSZ*PGSZ;
      getAllocation();
      Convert.packX(detbuf,0,paddr);
      Convert.packX(detbuf,8,psize);
      Native.fwriteb (fd,detbuf,0,16);
      lockFileFree();
    }
    if (devn.startsWith("RAM") || devn.startsWith("ram")) {
      map = pic.getMap();
      map.mode  = DmaMap.RAMD;
      map.bytes = length;
      map.paddr = getPhysicalAddress();
      map.dev   = pic;
      map.devn  = "iceram"; if (devn.length()==4) map.devn = "iceram"+devno;
      map.setPaged(Shell.ostag.startsWith("lin"));
      isOpen = true;
    } else {
      fb = Native.fopen ("/dev/"+devn,dir);
      Native.fseek(fb,getPhysicalAddress());
      isOpen = (fb > 0);
    }
    return isOpen;
  }
 
  /** close the resource */
  public boolean close() { 
    if (!isOpen) return false;
    if (fd!=0) Native.fclose(fd); 
    if (fb!=0) Native.fclose(fb); 
    if (map!=null) map.close(DmaMap.NOCHK);
    isOpen = false; 
    return true;
  }

  /** delete the resource */
  public boolean delete() { return file.delete(); }

  /** get length of resource */
  public long getLength() { return length; }

  /** seek to the specified offset in file */
  public boolean seek (long offset) { 
    if (offset==this.offset) return true;
    this.offset = offset;
    if (fb!=0) Native.fseek(fb,getPhysicalAddress()+offset);
    return true; 
  }

  /** get currently available length of resource */
  public long avail() { return length-offset; } 

  /** read data from resource */
  public int read (byte[] buf, int boff, int bytes) { 
    bytes = (int)Math.min(bytes,length-offset);
    if (map!=null) {
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2ja (vaddr,0, buf,boff, bytes);
    } else {
      Native.freadb(fb,buf,boff,bytes);
    }
    offset += bytes;
    return bytes;
  }

  /** write data to resource */
  public int write (byte[] buf, int boff, int bytes) { 
    if (map!=null) {
      bytes = (int)Math.min(bytes,length-offset);
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.ja2p (buf,boff, vaddr,0, bytes);
    } else {
      bytes = checkAllocation(bytes);
      Native.fwriteb(fb,buf,boff,bytes);
    }
    offset += bytes;
    return bytes;
  }

  /** read data from resource */
  public int read (long lbuf, int boff, int bytes) { 
    bytes = (int)Math.min(bytes,length-offset);
    if (map!=null) {
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2p (vaddr,0, lbuf,boff, bytes);
    } else {
      Native.fread(fb,lbuf,boff,bytes);
    }
    offset += bytes;
    return bytes;
  }

  /** write data to resource */
  public int write (long lbuf, int boff, int bytes) { 
    if (map!=null) {
      bytes = (int)Math.min(bytes,length-offset);
      long vaddr = map.getVirtualAddress(offset,bytes);
      Native.p2p (lbuf,boff, vaddr,0, bytes);
    } else {
//      long next = offset + bytes;
      bytes = checkAllocation(bytes);
      Native.fwrite(fb,lbuf,boff,bytes);
    }
    offset += bytes;
    return bytes;
  }

  /** Return a printable string containing information about this object. */
  public String toString () { return "RamDiskResource : " + getURL(); }

  private synchronized void getAllocation() {
    String[] lf = M.io.listFiles(rdpath,"*","det");
    int ndet = lf.length;
    long[] paddrs = new long[ndet+1];
    long[] psizes = new long[ndet+1];
    byte[] detbuf = new byte[16];
    for (int i=0; i<ndet; i++) {
      String fname = rdpath+lf[i];
      int fd = Native.fopen (fname,-1);
      Native.freadb (fd,detbuf,0,16);
      Native.fclose(fd);
      long paddri = Convert.unpackX(detbuf,0);
      long psizei = Convert.unpackX(detbuf,8);
      // now sort into ascending order
      int j=0; for (; j<i && paddri>paddrs[j]; j++);
      for (int k=i-1; k>=j; k--) { paddrs[k+1]=paddrs[k]; psizes[k+1]=psizes[k]; }
      paddrs[j]=paddri; psizes[j]=psizei;
    }
    paddr = -1; 
    long minsize = paddrs[ndet] = rdsize;
    for (int n=ndet-1; n>=0; n--) {
      long size = paddrs[n+1] - (paddrs[n]+psizes[n]);
      if (size>=psize && size<=minsize) { paddr=paddrs[n]+psizes[n]; minsize=size; }
    }
    if (paddr<0) M.error("Could not allocate RamDisk buffer of "+psize+" bytes");
  }

  private long extent = 0x01000000-1;

  public int checkAllocation (int bytes) {
    long next = offset + bytes;
    if (length >= next) return bytes;
    length = next;
    if (psize >= length) return bytes;
    psize = (next+bytes+extent) & ~extent;
    byte[] detbuf = new byte[16];
    Convert.packX(detbuf,0,paddr);
    Convert.packX(detbuf,8,psize);
    Native.fseek (fd,0);
    Native.fwriteb (fd,detbuf,0,16);
    return bytes;
  }

  public long getPhysicalAddress() { return rdaddr+paddr; }

  public Table getStatFS() {
    if (!isOpen) open();
    String[] lf = M.io.listFiles(rdpath,"*","det");
    int files = lf.length;
    long total = 0;
    byte[] detbuf = new byte[16];
    for (int i=0; i<files; i++) {
      String fname = rdpath+lf[i];
      int fd = Native.fopen (fname,-1);
      Native.freadb (fd,detbuf,0,16);
      Native.fclose(fd);
//    long paddri = Convert.unpackX(detbuf,0);
      long psizei = Convert.unpackX(detbuf,8);
      total += psizei;
    }
    int tmby = (int)(rdsize/1000000);
    int fmby = tmby - (int)(total/1000000);
    int used = (100-100*fmby/tmby);
    String stats = "{PATH="+rdpath+",FILES="+files+",TMBY="+tmby+",FMBY="+fmby+",USED="+used+"}";
    return new Table(stats);
  }

  File lockFile;
  private void lockFileAlloc() {
    lockFile = new File(rdpath+"allocate.LOCK");
    for (int i=0; i<100; i++) {
      try { if (lockFile.createNewFile()) return; } catch (Exception e) { }
      Time.sleep(0.010);
    }
    System.out.println("Overriding stale lock file "+lockFile);
  }

  private void lockFileFree() {
    lockFile.delete();
  }

}
