package nxm.ice.lib;

import nxm.sys.lib.*;
import nxm.sys.inc.PacketHandler;

/**
  Defines the header for ICE packetized data.

  TimeCode Modes: OFF=0,CPU=1,ZTC=2,SDN=3,SMS=4,DTL=5,IRB=6,SDD=7,ICE=8,VRT=9,FILE=10,STC=12
  TimeCode Status: OKNC=2,OK=1,INACTIVE=-1,BADPARAM=-2,NOBITS=-3,NOBARKER=-4,BCDERR=-5,NAN=-6,NOCLOCK=-7,BADSTART=-8,NTPERR=-9

  @author Jeff Schoen
  @author Jay Hallam
  @author Neon Ngo - use specified data rep (byte order) when packing/unpacking to/from data buffer
  @version $Id: ICEPacket.java,v 1.10 2011/02/02 22:04:13 ntn Exp $
*/
public class ICEPacket implements PacketHandler {

  /** Flag for using adjunct bytes as xdelta and xstart */
  public static final int ABSC=0x01;
  /** Flag for fixed size transfer length (constant packet size) */
  public static final int FIXED=0x02;
  /** Flag for packet header only debug mode */
  public static final int PKHO=0x04;
  /** The length of an ICE packet */
  public static final int headerLength=64;

  private int mhdrs=8;
  /** Byte buffer holding one or more ICE packets.  Meant to be read, not written */
  public byte[] buf = new byte[mhdrs*headerLength];

  private int rhdrs=0, whdrs=0;
  private int count=0, fixedSize=0, bpa=1, ipoff=0, flags;

  private double npktoff;
  /** data representation/byte order of data/header in packet (defaults to local machine rep) */
  private byte   rep = Shell.rep;
  private boolean pkho = false;

  /* The ICE packet sync bytes */
  private static final int ICE_SYNC_1 = 101;
  private static final int ICE_SYNC_2 = 102;
  private static final int ICE_SYNC_3 = 103;

  /** Construct an ICE packet 1024 bytes long */
  public ICEPacket() {
    this("SB",1024,0);
  }

  /** Construct an ICE packet from the specified byte array.
    @param ibuf Byte array containing header
    @param offset Byte offset of header within byte array
  */
  public ICEPacket(byte[] ibuf,int offset) {
    setBytes(ibuf,offset);
  }

  /** set ICE packet from the specified byte array
    @param ibuf Byte array containing header
    @param offset Byte offset of header within byte array
  */
  public void setBytes (byte[] ibuf,int offset) {
    System.arraycopy(ibuf,offset,buf,0,headerLength);
    // set internal fields (rep, bpa, count) based on packet header in byte array
    rep = getRep();
    bpa = getBPA();
    count = getCount();
  }

  /** Construct an ICE packet
    @param format Midas format digraph used to set bpa, mode and type
    @param size The size in elements (of data)
    @param flags Flags for this function
  */
  public ICEPacket (String format, int size, int flags) {
    this.flags = flags;
    setRep(Shell.rep); // default new packets to local machine rep
    setKeys((byte)0);
    setCount(0);
    if ((flags&FIXED)!=0) setFixedSize(size);
    else setSize(size);
    setUser((short)0);
    setChannel(1);
    setID((byte)0,(byte)0);
    setFormat(format);
    setTC((byte)0,(byte)0,0.0,0.0,0.0);
    if ((flags&ABSC)!=0) setAbscissa(0.0,1.0);
    if ((flags&PKHO)!=0) pkho = true;
  }

  /** Set barker syncs and the user key.  Barker keys must be the same for all packets.
    @param userKey User defined key
  */
  public void setKeys (byte userKey) {
    buf[0]=ICE_SYNC_1; buf[1]=ICE_SYNC_2; buf[2]=ICE_SYNC_3;
    setUserKey(userKey);
  }

  /** Set the user defined key
    @param val User field value
  */
  public void setUserKey (byte val ) {
    buf[3] = val;
  }
  /** Get the user defined key
    @return User key value
  */
  public byte getUserKey () {
    return buf[3];
  }

  /** Does this packet have synchronization barker codes?
    @return True if header in this instance has a sync
  */
  public boolean hasSync() {
    return hasSync(this.buf, 0);
  }

  /** Does the packet have synchronization barker codes?
    @param buf The buffer to check
    @param boffset Offset into the buffer
    @return True if header in this buffer has a sync
  */
  public static boolean hasSync(byte[] buf, int boffset) {
    return (buf[boffset+0] == ICE_SYNC_1) &&
           (buf[boffset+1] == ICE_SYNC_2) &&
           (buf[boffset+2] == ICE_SYNC_3);
  }

  /** packet number since start of xfer
    @param count Next packet number (zero based)
  */
  public void setCount (int count) {
    this.count = count;
    Convert.packL(buf,4,count,rep);
  }
  /** up the count by one */
  public void upCount() {
    count++;
    Convert.packL(buf,4,count,rep);
  }
  /** up the count by N
    @param n Amount to increase current count by
  */
  public void upCount (int n) {
    count+=n;
    Convert.packL(buf,4,count,rep);
  }
  /** data elements in packet
    @param size Size in atoms of current packet
  */
  public void setSize (int size) {
    Convert.packL(buf,8,size,rep);
  }
  /** user defined field
    @param user User field value
  */
  public void setUser (short user) {
    Convert.packI(buf,12,user,rep);
  }
  /** port channel number
    @param channel Channel field value
  */
  public void setChannel (int channel) {
    Convert.packI(buf,14,(short)channel,rep);
  }
  /** subsystem IDs
    @param id1 Subsystem ID field 1
    @param id2 Subsystem ID field 2
  */
  public void setID (byte id1, byte id2) {
    buf[16]=id1;
    buf[17]=id2;
  }

  /** set data representation ('I'=Ieee,'E'=Eeei,'V'=Vax), i.e. data byte order 
    @param rep Current data representation
  */
  public void setRep (byte rep) {
    buf[18]=rep;
  }
  /** get data representation ('I'=Ieee,'E'=Eeei,'V'=Vax), i.e. data byte order
    @return Current data representation
  */
  public byte getRep () {
    return buf[18];
  }
  /** get data representation in specified byte array.
    @param buf Byte array containing header
    @param boffset Byte offset of header within byte array
    @return Data rep
   */
  public static byte getRep (byte[] buf, int boffset) {
    return buf[boffset+18];
  }

  /** Get bytes per atom (BPA)
    @return Bytes per atom
  */
  public int getBPA () { return buf[19]; }
  /** Set bytes per atom (BPA). For advanced usage only as {@link #setFormat(String)}
      calculates and call this method to set bpa appropriately.
      Note this will downcast int to a byte to put into packet header.
    @param val BPA value
   */
  public void setBPA (int val) { bpa = buf[19] = (byte)val; }

  /** Set data format
      @param format Midas format digraph (e.g. "SB", "SI", etc.)
   */
  public void setFormat (String format) {
    setBPA(Data.getBPA(format));
    buf[20] = (byte)format.charAt(0);
    buf[21] = (byte)format.charAt(1);
  }

  /** Set timecode mode, status, offset whole seconds and fractional seconds
    @param mode Timecode mode
    @param status Timecode mode
    @param offset Offset at which timecode applies
    @param wsec Timecode whole seconds
    @param fsec Timecode fractional seconds
  */
  public void setTC (int mode, int status, double offset, double wsec, double fsec) {
    buf[22]=(byte)mode;
    buf[23]=(byte)status;
    Convert.packD(buf,24,offset,rep);
    Convert.packD(buf,32,wsec,rep);
    Convert.packD(buf,40,fsec,rep);
  }

  public void setTC (ICEPacket source) {
   System.arraycopy(source.buf,22, buf,22, 26);
  }

  /** Get timecode as a Time object
    @return Current time code object
  */
  public Time getTC () {
    double wsec = Convert.unpackD(buf,32,rep);
    double fsec = Convert.unpackD(buf,40,rep);
    return new Time (wsec,fsec);
  }

  /** get TC mode
    @return TimeCode mode
  */
  public byte getTCMode () {
    return buf[23];
  }

  /** get TC status
    @return TimeCode status
  */
  public byte getTCStatus () {
    return buf[23];
  }

  /** get TC offset
    @return TimeCode offset
  */
  public double getTCO () {
    return Convert.unpackD(buf,24,rep);
  }

  /** decrement TC offset
    @param xfer Number of samples to decrement offset by
  */
  public void decTCO (int xfer) {
    double tcoff = Convert.unpackD(buf,24,rep);
    Convert.packD(buf,24,tcoff-xfer,rep);
  }

  /** Get timecode
    @param tc TimeCode object to fill in
    @param offset Sample offset
    @param delta Sample delta
    @return Timecode status
  */
  public int getTC (Time tc, double offset, double delta) {
    double wsec = Convert.unpackD(buf,32,rep);
    double fsec = Convert.unpackD(buf,40,rep);
    tc.fromJ1950( wsec,fsec );
    double tcoff = offset - Convert.unpackD(buf,24,rep);
    tc.addSec( tcoff*delta );
    return (int)buf[23]; // timecode status
  }

/* Start of set/get User Data */
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-1)
  */
  public void setUserData (double val, int pos) {
    Convert.packD(buf,48+(8*pos),val,rep);
  }
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-1)
  */
  public void setUserData (long val, int pos) {
    Convert.packX(buf,48+(8*pos),val,rep);
  }
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-3)
  */
  public void setUserData (float val, int pos) {
    Convert.packF(buf,48+(4*pos),val,rep);
  }
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-3)
  */
  public void setUserData (int val, int pos) {
    Convert.packL(buf,48+(4*pos),val,rep);
  }
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-8)
  */
  public void setUserData (short val, int pos) {
    Convert.packI(buf,48+(2*pos),val,rep);
  }
  /** Set the user data.
    @param val The value
    @param pos The element position in the user data area (0-16)
  */
  public void setUserData (byte val, int pos) {
    buf[48+pos] = val;
  }

  /** Get the user data as a double
    @param pos The element position in the user data area (0-1)
    @return User data value
  */
  public double getUserDataD (int pos) {
    return Convert.unpackD(buf,48+(8*pos),rep);
  }
  /** Get the user data as a Midas X extra long type (Java long)
    @param pos The element position in the user data area (0-1)
    @return User data value
  */
  public long getUserDataX (int pos) {
    return Convert.unpackX(buf,48+(8*pos),rep);
  }
  /** Get the user data as a float
    @param pos The element position in the user data area (0-3)
    @return User data value
  */
  public float getUserDataF (int pos) {
    return Convert.unpackF(buf,48+(4*pos),rep);
  }
  /** Get the user data as a Midas L type (Java int)
    @param pos The element position in the user data area (0-3)
    @return User data value
  */
  public int getUserDataL (int pos) {
    return Convert.unpackL(buf,48+(4*pos),rep);
  }
  /** Get the user data as a Midas I type (Java short)
    @param pos The element position in the user data area (0-8)
    @return User data value
  */
  public short getUserDataI (int pos) {
    return Convert.unpackI(buf,48+(2*pos),rep);
  }
  /** Get the user data as byte
    @param pos The element position in the user data area (0-16)
    @return User data value
  */
  public byte getUserDataB (int pos) {
    return buf[48+pos];
  }

/* End of set/get User Data */

  /** set abscissa xstart and xdelta
    @param xstart The x start
    @param xdelta The x delta
  */
  public void setAbscissa (double xstart, double xdelta) {
    Convert.packD(buf,48,xstart,rep);
    Convert.packD(buf,56,xdelta,rep);
  }

  /** set abscissa mode
    @param mode The special header mode
  */
  public void setMode (String mode) {
    if (mode.equals("ABSC")) flags |= ABSC; else flags &= ~ABSC;
  }

  /** Set fixed size mode
    @param size The desired fixed size
  */
  public void setFixedSize (int size) {
    fixedSize = size;
    setSize(size);
  }

  /** Get Channel number of this packet
    @return Channel number
  */
  public int getChannel() {
    return (int)Convert.unpackI(buf,14,rep);
  }

  /** Get the current packet count
    @return Packet count
  */
  public int getCount() {
    count = Convert.unpackL(buf,4,rep);
    return count;
  }

  /** get the current Midas digraph format
    @return Packet data format
  */
  public String getFormat() {
    return Convert.unpackS(buf,20,2);
  }

  /** Get Size of packet data in elements
    @return Packet data size
  */
  public int getSize() {
    int size;
    if (fixedSize!=0) size = fixedSize;
    else size = Convert.unpackL(buf,8,rep);
    return size;
  }

  /** Get size of packet data in bytes
    @return Packet bytes
  */
  public int getBytes() {
    int size=fixedSize;
    if (size==0) size = Convert.unpackL(buf,8,rep);
    return bpa*size;
  }
  public int getFixedSize() {
    return fixedSize;
  }

  // Not Used - Use user data as a ram buffer
  public void setRBData (int off, int value) {
    Convert.packL(buf,48+off*4,value,rep);
  }
  // Not Used - Use user data as a ram buffer
  public int getRBData (int off) {
    return Convert.unpackL(buf,48+off*4,rep);
  }

  /* packet Handler methods */

  /** Get the configuration string from the data file
    @return Config string
  */
  public String getConfiguration (DataFile df) {
    String config = "ICE";
    if (fixedSize!=0) config += "/FS="+fixedSize;
    if ((flags&ABSC)!=0) config += "/MODE=ABSC";
    if (df.hp!=null) {
      config += "/DET";
      if (df.hp.aux!=null && df.hp.aux.length()>0) config += "="+df.hp.aux;
    }
    return config;
  }

  /** set filename for separate packet header file
    @param df Midas data file
    @param filename The file name
  */
  public void setFileName (DataFile df, FileName filename) {
    DataFile hp = new DataFile();
    hp.init(df.cmd,filename,"3000","NH", df.getPacketHandlerFlags() );
    hp.setSubRecords("KEYS|4B,CNT|SL,ELEM|SL,USER|SI,CHAN|SI,SID|2B,"+
        "REP|SB,BPA|SB,MODE|SB,TYPE|SB,TCM|SB,TCS|SB,TCO|SD,TCWS|SD,TCFS|SD");
    if ((flags&ABSC)!=0) hp.setSubRecords("+XDEL|SD,XSTA|SD");
    else  hp.setSubRecords("+RBST|SL,RBSZ|SL,RBDO|SL,RBDS|SL");
    if (fixedSize>0) hp.setSize((int)(df.getSize()/fixedSize));
    df.hp=hp;
  }

  /** Get the aligned data offset given a reasonable offset guesstimate.
    @param df Midas data file
    @param boffset Approximate offset
    @return data offset
  */
  public double naturalDataOffset (DataFile df, double boffset) {
    int bpp = fixedSize*bpa;
    if (bpp==0) bpp = getBytes();
    double npkt = Math.floor(boffset/bpp);
    return npkt*bpp;
  }

  /** translate seek data byte offset to the packet byte offset
    @param df The Midas data file
    @param boffset The offset
    @return packet offset
  */
  public double dataToPacketOffset (DataFile df, double boffset) {
    int bpp = fixedSize*bpa;
    if (bpp==0) bpp = getBytes();
    double npkt = Math.floor(boffset/bpp);
    double bpkthdr = npkt*headerLength;
    ipoff = (int)(boffset-npkt*bpp);            // save offset into packet
    if (df.hp==null) {
      boffset += bpkthdr;
      if (ipoff>0) boffset += headerLength;     // skip header
    } else {
      if (ipoff>0) npkt++;                      // skip header
      npktoff = npkt;
    }
    return boffset;
  }

  /** translate seek packet byte offset to the data byte offset
    @param df The Midas data file
    @param boffset The offset
    @return data offset
  */
  public double packetToDataOffset (DataFile df, double boffset) {
    if (df.hp!=null) return boffset;
    int bpp = fixedSize*bpa;
    if (bpp==0) bpp = getBytes();
    double npkt = Math.floor(boffset/(bpp+headerLength));
    return npkt*bpp;
  }

  /** open the packet handler
    @param df The Midas data file
  */
  public void open (DataFile df) {
    setFormat(df.getFormat());
    if (df.hp!=null) {
      Object rg = df.fn.getQualifier("RG");
      if (rg!=null) df.hp.setQualifier("RG",rg);
      df.hp.open();
    }
  }

  /** seek to the given data offset (or find valid trim value)
    @param df The Midas data file
    @param boffset The offset
  */
  public void seek (DataFile df, double boffset) {
    boffset = dataToPacketOffset (df,boffset);
    if (df.hp!=null) df.hp.seek(npktoff);
    if (!pkho) df.io.seek((long)(df.getDataStart()+boffset));
  }

  /** read packet header and data
    @param df The Midas data file
    @param buf The buffer in which to read the header and data
    @param boff Offset into buf
    @param bytes number of bytes to read
    @param lbuf Read into this if buf is null
    @return number of bytes read
  */
  public int read (DataFile df, byte[] buf, int boff, int bytes, long lbuf) {
    int ndo=bytes, nbp=getBytes(), nb;
    for (rhdrs = 0; ndo>0; ndo-=nb) {
      if (ipoff==0) {             // read packet header
        if (rhdrs==mhdrs) setMaxHeaders(mhdrs*2);
        if (df.hp!=null) df.hp.read (this.buf,rhdrs*headerLength,headerLength);
        else             df.io.read (this.buf,rhdrs*headerLength,headerLength);
        rhdrs++;
      }
      nb = Math.min(ndo,nbp-ipoff);
      if (buf!=null) nb = df.io.read(buf,boff,nb);
      else           nb = df.io.read(lbuf,boff,nb);
      if (nb<=0) return nb;
      boff+=nb; ipoff+=nb; if (ipoff==nbp) ipoff=0; // signal next packet header
    }
    return bytes;
  }

  /** Write packet header and data
    @param df The Midas data file
    @param buf The buffer from which to write
    @param boff Offset into buf
    @param bytes number of bytes to write
    @param lbuf Write this if buf is null
    @return number of bytes read
  */
  public int write (DataFile df, byte[] buf, int boff, int bytes, long lbuf) {
    int ndo=bytes, nbp=getBytes(), nb;
    for (whdrs = 0; ndo>0; ndo-=nb) {
      if (ipoff==0) {    // write packet header
             if (df.hp!=null) df.hp.write (this.buf,whdrs*headerLength,headerLength);
        else if (df.io!=null) df.io.write (this.buf,whdrs*headerLength,headerLength);
        whdrs++;
      }
      nb = Math.min(ndo,nbp-ipoff);
      if (pkho);
      else if (buf!=null) nb = df.io.write(buf,boff,nb);
      else                nb = df.io.write(lbuf,boff,nb);
      if (nb<0) return nb;
      boff+=nb; ipoff+=nb; if (ipoff==nbp) ipoff=0; // signal next packet header
    }
    return bytes;
  }

  /** Close the detached data file if not null
    @param df The Midas data file
  */
  public void close (DataFile df) {
    if (df.hp!=null) df.hp.close();
  }

  /** Does this packet have any headers
    @return True if has header
  */
  public boolean hasHeader() {
    return rhdrs>0;
  }

  /** List the packet(s) header count, size, channel, format and timecode
    @return list string
  */
  public String listHeader() {
    String list = "ICE-Packet Count="+getCount()+" Elem="+getSize()+
      " Chan="+getChannel()+" Form="+getFormat()+" TC="+getTC()+" at "+getTCO();
    return list;
  }

  /** Make a clone of the packet handler 
    @return packe handler clone
  */
  public PacketHandler cloneOf() {
    PacketHandler ph=null;
    try { ph = (PacketHandler)this.clone(); }
    catch (Exception e) {}
    return ph;
  }

  /** Copy from another ICE packet object
    @param source ICEPacket class to copy from
  */
  public void copyFrom (ICEPacket source) {
    setBytes(source.buf,0);
  }

  /** Set the maximum number of headers (should to be 1 or more)
    @param nmhdrs Maximum nuber of headers 
  */
  private void setMaxHeaders (int nmhdrs) {
    byte[] nbuf = new byte[nmhdrs*headerLength];
    if (rhdrs>0) System.arraycopy(buf,0, nbuf,0, rhdrs*headerLength);
    this.buf = nbuf;
    this.mhdrs = nmhdrs;
  }

  /** set ICE packet from the specified byte array
    @param ibuf Byte array containing header
    @param offset Byte offset of header within byte array
  */
  public void setBuffer (byte[] ibuf,int offset) {
    System.arraycopy(ibuf,offset,buf,0,headerLength);
  }

}
