package nxm.ice.lib;

import nxm.sys.inc.DataTypes;
import nxm.sys.lib.Args;
import nxm.sys.lib.Parser;
import nxm.sys.lib.Convert;
import nxm.sys.lib.Shell;

/**
  Structure for a Code-Once-Run-Everywhere library function
  Structure for a Hardware Abstraction Library Object 

  This function may have implementations in more than one language.

  The NeXtMidas policy is to code CORE functions in C or Fortran to allow
  code sharing with other versions of Midas.  All other framework and GUI
  code is in Java for portability and deployment options.

  Typically, math functions needed by GUIs are included in the SYS tree.
  Some functions are coded here in Java and in C for performance comparisons.

  The Java code for each function library exposes methods for setting and
  getting each of the library's properties.  If native code is used, the
  Java wrapper loads the native code and calls the standard library interface
  routines as appropriate.

  This class implements a factory for instantiating alternate implementations.
  The innerclasses of this factory comprise the supported algorithm implementations.

  The executable names will adhere to the following naming conventions when compiled by NeXtMidas:

   Foo$JVM.class   - the JDK implementation of the function Foo (Java/Lime)
   Foo$CPU_xxx.so  - the CPU native implementation of Foo (C)
   Foo$PHI_xxx.so  - the Intel Xeon Phi accelerator card (C)
   Foo$GPU_xxx.so  - the GPU implementation of Foo (CUDA/Lime)
   Foo$VHS_xxx.so  - the Verilator Hardware Simulator implementation of Foo (Verilog/Lime)
   Foo$MLS_xxx.so  - the MatLab Simulink implementation of Foo (FPGA)
   Foo$ICE_xxx.so  - the ICE FPGA implementation of Foo (FPGA/Lime)

  Where xxx is an architecture dependent tag such as lin32 or arm32.

  These will typically be conditionally compiled from common source code Foo.jv
  in NeXtMidas but may be compiled differently with custom build scripts.

  @author Jeff Schoen
  @version $Id: CoreFactory.java,v 1.2 2011/05/24 13:20:30 jgs Exp $
*/
public class CoreFactory {

  public final static String factoryList = "JVM,CPU,PHI,GPU,MLS,VHS,ICE";
  public final static int JVM=0, CPU=1, PHI=2, GPU=3, MLS=4, VHS=5, ICE=6;
  public final static int CORE_TYPE_FPGA=5, CORE_TYPE_ICE=6, VERBOSE=0;			// FPGA core
  public final static int ARRAY=0, STREAM=1, BUFFER=2, BUFSTR=3, PACKET=4;		// buffer type
  public final static int FRAW=0, FCXI=8, FBIT=2, FAXI=3, FALL=4, FSWP=5;		// buffer formats
  public final static int BCIO=0, BCI=1, BCO=2, BCIIO=3, BCIOO=4, BCIIOO=5;		// buffer config
  public final static int BC_IO=0, BC_I=1, BC_O=2, BC_IIO=3, BC_IOO=4, BC_IIOO=5;	// buffer config
  public final static int BF_RAW=0, BF_CXI=8, BF_BIT=2, BF_AXI=3, BF_ALL=4;		// buffer format
  public static String defKeys=null;

  /** Create an instance of the function specified by name and parsing of the Args class
     @param name	Name of core
     @param args	Command Args object
     @return		Core object
  */
  public static Core getCore (String name, Args args) {
    String config = null;
    String cflags = null;
    String opt = args.option;
    int i=name.indexOf(";");
    if (i>0) {
      opt = name.substring(i+1);
      name = name.substring(0,i);
    } 
    opt = args.getS("/COREOPT",opt);
    String path = "nxm."+opt.toLowerCase()+".core";
    int impl=ICE, coreio=1;
    if (args.getS("/CORE").equals("ICEX")) coreio=0;
    else impl = args.getSelectionIndex("/CORE",factoryList,0,-1);
    if (impl==VHS) {
      config = "VHS-CORE,NOLOCK,IOM=NONE,PM=NONE,";
    }
    if (impl==ICE) {
      String alias  = args.getS("/COREDEV","PIC1AUTO:11");
      int ib = alias.indexOf('|');
      int ic = alias.indexOf(':');
      if (ib>0) { cflags = alias.substring(ib); alias = alias.substring(0,ib); }
      int cindex = 11;
      if (ic>0) {
	cindex=0;
	String cport = alias.substring(ic+1);
	if (cport.startsWith("PM")) { cindex=Convert.s2l(cport.substring(2,3))*10; cport = cport.substring(3); }
	if (cport.startsWith("CORE")) cport = cport.substring(4);
	cindex+=Convert.s2l(cport); if (cindex<0) cindex=40-cindex;
	alias=alias.substring(0,ic);
      }
      int pmi    = cindex/10;
      int ci     = cindex%10;
      int side   = (ci+1)%2+1;
      config = MDevIce.getConfig(args,alias);
      if (config.indexOf(",IPORT=")<0) config += ",IPORT=PM0CORE"+side;
      if (config.indexOf(",OPORT=")<0) config += ",OPORT=PM"+pmi+"CORE"+ci;
      config += ",COREIO="+coreio;
      if (cflags!=null) config += cflags;
    }
    cflags  = args.getS("/COREFLG");
    if (cflags!=null) config += ","+cflags+",";
    Core core = getCore(path,name,impl);
    core.halo.mcid = args.getL("/MCID",-1);
    core.halo.vbpr = args.getL("/COREVB",0);
    if (core!=null && args.getState("/XAB")) core.halo.flgs |= 0x08;
    if (core!=null && args.getState("/XIO")) core.halo.flgs |= 0x10;
    if (impl>JVM) core.allocNative(config);
    else core.init();
    return core;
  }

  /** Create an instance of the function specified by path, name, and implementation
     @param path	Jave package path name
     @param name	Name of core
     @param impl	Implementation index
     @return		Core object
  */
  public static Core getCore (String path, String name, int impl) {
    String cname = path+"."+name+"$"+Parser.get(factoryList,impl,-1); 	// add inner class $xxx suffix
    Core core = (Core)Shell.getInstanceForClassName(cname);
    core.name = name;
    core.halo.impl = impl;
    return core;
  }

  /** get the natural data format type
     @param impl	Implementation index
     @return		Data type code
  */
  public static byte getDataType (int impl) {
    if (impl==VHS) return DataTypes.INT;
    if (impl==ICE) return DataTypes.INT;
    return DataTypes.FLOAT;
  }

  public static class TypeDef {
  }
}
