/*
  Verilog modules for simulating various Xilinx primitives
  This also allows similar primitives to be inferred on Altera devices

  Note: In verilator, counters must be in their own always block

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

*/
`include "../lib/ramdefs.h"


module RAMBXX (CLKA,ENA,WEA_,SSRA,ADDRA,DIA_,DIPA_,DOA,DOPA,
               CLKB_,ENB,WEB_,SSRB,ADDRB,DIB_,DIPB_,DOB,DOPB);

  parameter WIDTH_A=8;
  parameter WIDTH_B=8;
  parameter DEPTH=15; // ram bits = 2**DEPTH
  parameter FLG_A=0;
  parameter FLG_B=0;

  localparam NB=WIDTH_A/8, ND=(NB==0)?WIDTH_A:NB*8, NP=(NB==0)?1:NB;
  localparam MB=WIDTH_B/8, MD=(MB==0)?WIDTH_B:MB*8, MP=(MB==0)?1:MB;

  localparam LND=(ND>=64)? 6 : (ND>=32)? 5 : (ND>=16)? 4 : (ND>=8)? 3 : (ND==4)? 2 : (ND==2)? 1 : 0; // log2 port width
  localparam LMD=(MD>=64)? 6 : (MD>=32)? 5 : (MD>=16)? 4 : (MD>=8)? 3 : (MD==4)? 2 : (MD==2)? 1 : 0;

  localparam NA=DEPTH-LND;
  localparam MA=DEPTH-LMD;

  localparam NEL=(2**NA);
  localparam MEL=(2**MA);

  localparam NBE=(FLG_A&`BENA)? NP:1;
  localparam MBE=(FLG_B&`BENA)? MP:1;

  input CLKA,ENA,SSRA;
  input [NBE:0] WEA_;
  input [NA-1:0] ADDRA;
  input [ND-1:0] DIA_;
  input [NP-1:0] DIPA_;
  output [ND-1:0] DOA;
  output [NP-1:0] DOPA;

  input CLKB_,ENB,SSRB;
  input [MBE-1:0] WEB_;
  input [MA-1:0] ADDRB;
  input [MD-1:0] DIB_;
  input [MP-1:0] DIPB_;
  output [MD-1:0] DOB;
  output [MP-1:0] DOPB;

  wire CLKB=CLKA;

  // handle write byte enables
  wire WEA,WEB;
  wire [ND-1:0] DIA;
  wire [NP-1:0] DIPA;
  wire [MD-1:0] DIB;
  wire [MP-1:0] DIPB;
  wire [ND-1:0] CDOA;
  wire [MD-1:0] CDOB;
  genvar i;
  generate
  if (FLG_A&`BENA) begin:aa
    assign WEA  = (WEA_!=0);
    for (i=0; i<NP; i=i+1) begin:c assign DIA[7+i*8:i*8]  = WEA_[i]? DIA_[7+i*8:i*8] : CDOA[7+i*8:i*8]; end
    assign DIPA = DIPA_;
  end
  else begin:ab
    assign WEA  = WEA_;
    assign DIA  = DIA_;
    assign DIPA = DIPA_;
  end
  if (FLG_B&`BENA) begin:ba
    assign WEB  = (WEB_!=0);
    for (i=0; i<MP; i=i+1) begin:c assign DIB[7+i*8:i*8]  = WEB_[i]? DIB_[7+i*8:i*8] : CDOB[7+i*8:i*8]; end
    assign DIPB = DIPB_;
  end
  else begin:bb
    assign WEB  = WEB_;
    assign DIB  = DIB_;
    assign DIPB = DIPB_;
  end
  endgenerate

  // infer from verilog

 /* verilator lint_off MULTIDRIVEN */
  reg [ND-1:0] ram [NEL-1:0];
  reg [NP-1:0] par [NEL-1:0];
 /* verilator lint_on MULTIDRIVEN */

  wire [ND-1:0] UDOA;
  wire [NP-1:0] UDOPA;
  wire [MD-1:0] UDOB;
  wire [MP-1:0] UDOPB;

  reg [ND-1:0] RDOA;
  reg [NP-1:0] RDOPA;
  reg [MD-1:0] RDOB;
  reg [MP-1:0] RDOPB;

  reg [NA-1:0] RADDRA;
  reg [MA-1:0] RADDRB;

  always @(posedge CLKA) begin
    if (ENA) begin
      if (WEA) ram[ADDRA] <= DIA;
      if (WEA) par[ADDRA] <= DIPA;
      RADDRA <= ADDRA;
    end
  end
  assign CDOA  = ram[ADDRA];
  assign UDOA  = ram[RADDRA];
  assign UDOPA = par[RADDRA];

generate

if (MD==ND) begin:c1to1
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[ADDRB] <= DIB;
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = ram[ ADDRB];
  assign UDOB  = ram[RADDRB];
end

if (MD==ND && NB!=0 && MB!=0) begin:c1to1p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[ADDRB] <= DIPB;
    end
  end
  assign UDOPB = par[RADDRB];
end

if (MD==2*ND) begin:c1to2
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[2*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[2*ADDRB+1] <= DIB[2*ND-1:1*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = {ram[2* ADDRB+1],ram[2* ADDRB+0]};
  assign UDOB  = {ram[2*RADDRB+1],ram[2*RADDRB+0]};
end

if (MD==2*ND && NB!=0 && MB!=0) begin:c1to2p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[2*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[2*ADDRB+1] <= DIPB[2*NP-1:1*NP];
    end
  end
  assign UDOPB = {par[2*RADDRB+1],par[2*RADDRB+0]};
end

if (MD==4*ND) begin:c1to4
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[4*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[4*ADDRB+1] <= DIB[2*ND-1:1*ND];
      if (WEB) ram[4*ADDRB+2] <= DIB[3*ND-1:2*ND];
      if (WEB) ram[4*ADDRB+3] <= DIB[4*ND-1:3*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB  = {ram[4* ADDRB+3],ram[4* ADDRB+2],ram[4* ADDRB+1],ram[4* ADDRB+0]};
  assign UDOB  = {ram[4*RADDRB+3],ram[4*RADDRB+2],ram[4*RADDRB+1],ram[4*RADDRB+0]};
end

if (MD==4*ND && NB!=0 && MB!=0) begin:c1to4p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[4*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[4*ADDRB+1] <= DIPB[2*NP-1:1*NP];
      if (WEB) par[4*ADDRB+2] <= DIPB[3*NP-1:2*NP];
      if (WEB) par[4*ADDRB+3] <= DIPB[4*NP-1:3*NP];
    end
  end
  assign UDOPB = {par[4*RADDRB+3],par[4*RADDRB+2],par[4*RADDRB+1],par[4*RADDRB+0]};
end

if (MD==8*ND) begin:c1to8
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) ram[8*ADDRB+0] <= DIB[1*ND-1:0*ND];
      if (WEB) ram[8*ADDRB+1] <= DIB[2*ND-1:1*ND];
      if (WEB) ram[8*ADDRB+2] <= DIB[3*ND-1:2*ND];
      if (WEB) ram[8*ADDRB+3] <= DIB[4*ND-1:3*ND];
      if (WEB) ram[8*ADDRB+4] <= DIB[5*ND-1:4*ND];
      if (WEB) ram[8*ADDRB+5] <= DIB[6*ND-1:5*ND];
      if (WEB) ram[8*ADDRB+6] <= DIB[7*ND-1:6*ND];
      if (WEB) ram[8*ADDRB+7] <= DIB[8*ND-1:7*ND];
      RADDRB <= ADDRB;
    end
  end
  assign CDOB = {ram[8* ADDRB+7],ram[8* ADDRB+6],ram[8* ADDRB+5],ram[8* ADDRB+4],
                 ram[8* ADDRB+3],ram[8* ADDRB+2],ram[8* ADDRB+1],ram[8* ADDRB+0]};
  assign UDOB = {ram[8*RADDRB+7],ram[8*RADDRB+6],ram[8*RADDRB+5],ram[8*RADDRB+4],
                 ram[8*RADDRB+3],ram[8*RADDRB+2],ram[8*RADDRB+1],ram[8*RADDRB+0]};
end
if (MD==8*ND && NB!=0 && MB!=0) begin:c1to8p
  always @(posedge CLKB) begin
    if (ENB) begin
      if (WEB) par[8*ADDRB+0] <= DIPB[1*NP-1:0*NP];
      if (WEB) par[8*ADDRB+1] <= DIPB[2*NP-1:1*NP];
      if (WEB) par[8*ADDRB+2] <= DIPB[3*NP-1:2*NP];
      if (WEB) par[8*ADDRB+3] <= DIPB[4*NP-1:3*NP];
      if (WEB) par[8*ADDRB+4] <= DIPB[5*NP-1:4*NP];
      if (WEB) par[8*ADDRB+5] <= DIPB[6*NP-1:5*NP];
      if (WEB) par[8*ADDRB+6] <= DIPB[7*NP-1:6*NP];
      if (WEB) par[8*ADDRB+7] <= DIPB[8*NP-1:7*NP];
    end
  end
  assign UDOPB = {par[8*RADDRB+7],par[8*RADDRB+6],par[8*RADDRB+5],par[8*RADDRB+4],
                  par[8*RADDRB+3],par[8*RADDRB+2],par[8*RADDRB+1],par[8*RADDRB+0]};
end

endgenerate

  always @(posedge CLKA) if (ENA) RDOA <= UDOA;
  always @(posedge CLKA) if (ENA) RDOPA <= UDOPA;
  assign DOA  = (FLG_A&`OREG)? RDOA : UDOA;
  assign DOPA = (FLG_A&`OREG)? RDOPA : UDOPA;

  always @(posedge CLKB) if (ENA) RDOB <= UDOB;
  always @(posedge CLKB) if (ENA) RDOPB <= UDOPB;
  assign DOB  = (FLG_B&`OREG)? RDOB : UDOB;
  assign DOPB = (FLG_B&`OREG)? RDOPB : UDOPB;

endmodule

