/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  DDR Network DRAM Controller (64Mby x 32b)

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

  This module uses 4 BDRAMs to interface the 800Mby/s input/output streams 
  to the 1600Mby/sec DDR interface.  All transfers must be naturally aligned.
  Burst transfers are 4, 16, 32, or 64 bytes in length.

  This memory is 4 bytes wide, 16by banks, 64Mby total

  Memory control word:
  [31] nCS on 2nd cycle (config register)
  [30] FN on 1st cycle
  [29:28]=xfer 0-4by 1-16by 2-32by 3-64by

  [27:26]=ADDR alias 
  [25:0]=ADDR [5:4]=BANK [1:0]=0

  addr to cntrl bits are:  ED CBA9 8765 4321 0|654 32bb 10xx

  All 64by access start on bank=0
  All 32by access start on bank=0 or bank=2
  Same bank access needs 5 cycle seperation requiring 1 wait state before and after 32by access

  The read clock should calibrate to near the write clock at 270 deg.

  Memory Chacke Line Mapping:

    IoBus DdCc BbAa
          HhGg FfEe

    ReRam FfEe BbAa
    FeRam HhGg DdCc

    Chip0 DdCc / BbAa
    Chip1 HhGg / FfEe

    acl
    fcmd1
    fcmd2
    rd     wr
    rd1    wack
    rd2
    rd3
    rd4
    rack
    renl   rdata,rdqs

*/
`define WC1 3
`define WC2 1
`define WC4 2

//`define SIBUF  IBUF_SSTL2_I
//`define SOBUF  OBUF_SSTL2_I
//`define SOBUFT OBUFT_SSTL2_II

`define SIBUF  IBUF_LVCMOS25
`define SOBUF  OBUF_LVCMOS25_F_16
`define SOBUFT OBUFT_LVCMOS25_F_16

module ndram (
  aclk,rclk,wclk, rst, ago,apd,acmd,afull, qrdym,qselm,
  ioclk, irdy,iena,ibus, ordy,oena,obus,
  mckp,mckn,mpd,mcs,mfn,mba,ma,mdqs,mdq,
  mclk,trs,sclk, test);

  parameter MBW=64;
  parameter IBW=32;

  parameter IBB=10;
  parameter MBB=9;

  parameter RACKDELAY=8;
  parameter WACKDELAY=4;

  input aclk,rclk,wclk;
  input rst,ago,apd;
  input [35:0] acmd;
  output afull;
  output [23:0] qrdym;
  input  [23:0] qselm;

  input  ioclk, iena,oena;
  input  irdy,ordy;
  input  [IBW-1:0] obus;
  output [IBW-1:0] ibus;

  output mckp,mckn,mpd,mcs,mfn;
  output [1:0] mba;
  output [14:0] ma;
  inout [3:0] mdqs;
  inout [31:0] mdq;
  input mclk,sclk;
  output [2:0] trs;

  output [7:0] test;

  wire H=1, L=0;
  wire [1:0] L2 = {2{L}};
  wire [2:0] L3 = {3{L}};
  wire [3:0] L4 = {4{L}};
  wire [15:0] L16 = {16{L}};
  wire [31:0] L32 = {32{L}};

  reg rack,wack;
  wire [MBW-1:0] rdata,wdata;
  wire renl,renu,wenl,wenu,ienax,oenax;
  wire [IBB-1:0] iadrx,oadrx;
  wire [MBB-1:0] rcntl,rcntu,wcntl,wcntu;

  // MEM side counters
  wire acl,refresh;
  wire [35:0] qcmd;
  scheduler #(1,IBB,MBB,RACKDELAY,WACKDELAY) cnt (
        ioclk,rst, ago,acmd,afull, qrdym,qselm,
        ioclk,irdy,iena,ienax,iadrx, ioclk,ordy,oena,oenax,oadrx,
        aclk,acl,qcmd,refresh, rclk,rack,renl,rcntl,renu,rcntu, wclk,wack,wenl,wcntl,wenu,wcntu,
        sclk,16'h0000,test);

  // DMAR data array
  RAMB16_S18_S36 ram1r (
    .ADDRA(iadrx), .DIA(L16), .DIPA(L2), .CLKA(ioclk), .ENA(H), .WEA(L), .SSRA(L), .DOA(ibus[15:00]),
    .ADDRB(rcntl), .DIB(rdata[31:00]), .DIPB(L4), .CLKB(rclk), .ENB(H), .WEB(renl), .SSRB(L)
  );
  RAMB16_S18_S36 ram2r (
    .ADDRA(iadrx), .DIA(L16), .DIPA(L2), .CLKA(ioclk), .ENA(H), .WEA(L), .SSRA(L), .DOA(ibus[31:16]),
    .ADDRB(rcntu), .DIB(rdata[63:32]), .DIPB(L4), .CLKB(~rclk), .ENB(H), .WEB(renu), .SSRB(L)
  );

  // DMAW data array
  RAMB16_S18_S36 ram1w (
    .ADDRA({L3,oadrx}), .DIA(obus[15:00]), .DIPA(L2), .CLKA(ioclk), .ENA(H), .WEA(oenax), .SSRA(L),
    .ADDRB({L3,wcntl}), .DIB(L32), .DIPB(L4), .CLKB(wclk), .ENB(H), .WEB(L), .SSRB(L), .DOB(wdata[31:00])
  );
  RAMB16_S18_S36 ram2w (
    .ADDRA({L3,oadrx}), .DIA(obus[31:16]), .DIPA(L2), .CLKA(ioclk), .ENA(H), .WEA(oenax), .SSRA(L),
    .ADDRB({L3,wcntu}), .DIB(L32), .DIPB(L4), .CLKB(~wclk), .ENB(H), .WEB(L), .SSRB(L), .DOB(wdata[63:32])
  );

  // register the output data for routing purposes
  reg [MBW-1:0] xdata;
  always @(posedge wclk) xdata[31:00] <= wdata[31:00];
  always @(negedge wclk) xdata[63:32] <= wdata[63:32];

  reg [14:0] fma;
  reg [1:0] fmba;
  reg fmpd,fmfn,fmcs;

  reg wr,wr1,wr2,wr3,wr4,wen;
  reg rd,rd1,rd2,rd3,rd4,rd5;
  reg fcmd1,fcmd2,fcmd3,fcmd4,fcmd5,fcmd6,fcmd7,fcmd8;
  reg arw,acfg,active2,acl2; 
  reg [1:0] awc,burst,bank,acnt;
  reg [25:2] addr;

  // burst DMA state machine
  always @(posedge aclk) begin
    if (acl) arw   <= qcmd[35];		// read/write
    if (acl) acnt  <= qcmd[33:32];     // xfer cnt
    if (acl) acfg  <= qcmd[31];        // configuration command
    if (acl) addr  <= qcmd[25:2];      // start RAM address
    if (acl) bank  <= qcmd[5:4];
    else if (active2) bank <= bank+1;
    awc <= (acnt==0)? `WC1 : `WC4;
    acl2 <= acl;
  end

  wire active1 = (fcmd1 || (acnt>1 && fcmd3) || (acnt>2 && (fcmd5|fcmd7)));

  // command state machine
  always @ (posedge aclk) begin
    fcmd1 <= acl;
    fcmd2 <= fcmd1;
    fcmd3 <= fcmd2;
    fcmd4 <= fcmd3;
    fcmd5 <= fcmd4;
    fcmd6 <= fcmd5;
    fcmd7 <= fcmd6;
    fcmd8 <= fcmd7;
    fmpd  <= apd; 
    active2 <= active1;
    if (active1) begin
      fmfn <= arw || acfg;
      fmcs <= L;
      fmba <= bank;
      fma  <= addr[25:11];
    end else if (active2) begin
      fmfn <= H;    
      fmcs <= !acfg;
      fma  <= {awc,awc, L,L,L,L, addr[10:6],addr[3:2]};
    end else if (refresh) begin  // autorefresh
      fmfn <= L;    
      fmcs <= L;    
    end else begin    
      fmfn <= H;    
      fmcs <= H;    
    end
    rd  <= (fcmd2|fcmd4|fcmd6|fcmd8) && arw;
    rd1 <= rd; rd2 <= rd1; rd3 <= rd2; rd4 <= rd3; rd5 <= rd4;    
    wr  <= (fcmd2|fcmd4|fcmd6|fcmd8) && !arw;
    wr1 <= wr; wr2 <= wr1; wr3 <= wr2; wr4 <= wr3;
    wen <= (wr2||wr3);
  end

  // output enables - one early last half, one late first half
  reg oen,oeni,oenj,woeni,woenj;
  always @(negedge aclk) oen <= (wr1|wr2|wr3);
  always @(negedge aclk) oenj <= oen;
  always @(posedge aclk) oeni <= oenj;
  always @(negedge wclk) woenj <= oen;
  always @(posedge wclk) woeni <= woenj;

  // controller acks - one cycle early
  reg rdx,mack,macken;
  always @ (posedge wclk) begin
    if (fcmd6) macken <= (acnt==3);
    if (!macken) mack <= 0; else mack <= rdx;
  end
  always @ (posedge aclk) begin
    wack <= wr||wr1; // also one cycle pipeline
    rdx  <= rd3||rd4;
    rack <= rdx;
  end

  // timing recovery signals
  wire [7:0] rdqs;
  wire trs_re; vote3 rev (trs_re, rdqs[0],rdqs[1],rdqs[3]);
  wire trs_fe; vote3 fev (trs_fe, rdqs[4],rdqs[5],rdqs[7]);
  reg  trs_en; always @ (posedge mclk) trs_en <= mack;
  assign trs = {trs_fe,trs_re,trs_en};

  // DDR clocks (with feedback from mckp)
  ddr_o bmckp (mckp,H,H, aclk, H,L);
  ddr_o bmckn (mckn,H,H, aclk, L,H);

  // control signals (180 phase)
  sdr_oj bmcs (mcs,H, aclk, fmcs);
  sdr_oj bmfn (mfn,H, aclk, fmfn);
  sdr_oj bmpd (mpd,H, aclk, fmpd);

  genvar i;
  generate

  // address signals (180 phase)
  for (i=0; i<15; i=i+1) begin:bma
    sdr_oj inst (ma[i],H, aclk, fma[i]);
  end

  // bank signals (180 phase)
  for (i=0; i<2; i=i+1) begin:bmba
    sdr_oj inst (mba[i],H, aclk, fmba[i]);
  end

  // data strobes
  for (i=0; i<4; i=i+1) begin:bmdqs
    ddr_io inst (mdqs[i],oeni,oenj, mclk,rdqs[i+0],rdqs[i+4], aclk, wen,L);
  end

  // data buffers
  for (i=0; i<32; i=i+1) begin:dq
    ddr_io inst  (mdq[i], woeni,woenj, rclk,rdata[i+0],rdata[i+32], wclk,xdata[i+0],xdata[i+32]);  
  end

  endgenerate

//  assign test = {oenax,ienax,wack,rack,acl,ago,refresh,rst};

endmodule

// single data rate clocked output buffer
module sdr_o (out,ena, clk,in);
  output out;
  input in,clk,ena;

  wire ff;
  FD f0 (.Q(ff),.C(clk),.D(in)) /* synthesis syn_useioff=1 */; 
  `SOBUFT b0 (.I(ff), .T(~ena), .O(out) );
   //synthesis attribute IOB of f0 is "TRUE"

endmodule

// single data rate clocked output buffer
module sdr_oj (out,ena, clk,in);
  output out;
  input in,clk,ena;

  wire ff;
  FD f0 (.Q(ff),.C(~clk),.D(in)) /* synthesis syn_useioff=1 */; 
  `SOBUFT b0 (.I(ff), .T(~ena), .O(out) );
   //synthesis attribute IOB of f0 is "TRUE"

endmodule

// double data rate clocked output buffer
module ddr_o (out,ena0,ena1, clk,in0,in1);
  output out;
  input ena0,ena1,clk,in0,in1;

  wire H=1, L=0;
  wire ff,tt;
  FDDRRSE f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(in0),.D1(in1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  FDDRRSE t0 (.Q(tt),.C0(clk),.C1(~clk),.CE(H),.D0(~ena0),.D1(~ena1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  `SOBUFT b0 (.I(ff), .T(tt), .O(out) );
  //synthesis attribute IOB of f0 is "TRUE"
  //synthesis attribute IOB of t0 is "TRUE"

endmodule

// double data rate clocked output buffer with feedback
module ddr_o_fb (io,ena0,ena1, clk,in0,in1,fb);
  inout io;
  input ena0,ena1,clk,in0,in1;
  output fb;

  wire H=1, L=0;
  wire ff,tt;
  FDDRRSE f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(in0),.D1(in1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  FDDRRSE t0 (.Q(tt),.C0(clk),.C1(~clk),.CE(H),.D0(~ena0),.D1(~ena1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  `SOBUFT b0 (.I(ff), .T(tt), .O(io) );
  `SIBUF fb0 (.I(io), .O(fb) );
  //synthesis attribute IOB of f0 is "TRUE"
  //synthesis attribute IOB of t0 is "TRUE"

endmodule

// double data rate clocked input/output buffer
module ddr_io (io,ena0,ena1, rclk,rdata0,rdata1, wclk,wdata0,wdata1);
  inout io;
  input ena0,ena1,rclk,wclk;
  output rdata0,rdata1;
  input wdata0,wdata1;

  wire H=1, L=0;
  wire rff,wff,tff;

  FDDRRSE bwff (.Q(wff),.C0(wclk),.C1(~wclk),.CE(H),.D0(wdata0),.D1(wdata1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  FDDRRSE btff (.Q(tff),.C0(wclk),.C1(~wclk),.CE(H),.D0(~ena0),.D1(~ena1),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  `SOBUFT odq (.I(wff),.T(tff),.O(io));
  `SIBUF idq (.I(io),.O(rff));
  FD brff0 (.D(rff),.C(rclk),.Q(rdata0)) /* synthesis syn_useioff=1 */; 
  FD brff1 (.D(rff),.C(~rclk),.Q(rdata1)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOB of bwff is "TRUE"
  //synthesis attribute IOB of btff is "TRUE"
  //synthesis attribute IOB of brff0 is "TRUE"
  //synthesis attribute IOB of brff1 is "TRUE"

endmodule

