/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  MultiChannel Core Skeleton
  Handles individual and banked DMA modes

  Addressing: CCXXaaaa
  CC   = chip select
  XX   = system[7], group[6], channel[5:0]
  aaaa = individual channel

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  11/15/2009

*/
 
module mcengine (
  sclk,srst, scs,saddr, swr,swrbus, srd,srdbus, sack,
  ioclk, istat,iena,isel,ibus, ostat,oena,osel,obus,
`ifdef MC_FEEDBACK
  ostatfb,oenafb,oselfb,obusfb,
`endif
  test
);

  parameter PORT=1;	// engine index
  parameter IBW=64;	// I/O input bus width
  parameter OBW=IBW;	// I/O output bus width

`ifdef MC_COMSTAT
  localparam COMSTAT=`MC_COMSTAT;
`else
  localparam COMSTAT=0;
`endif
`ifdef MC_FEEDBACK
  localparam FEEDBACK=`MC_FEEDBACK;
`else
  localparam FEEDBACK=0;
`endif
  localparam SBW=32;	// system bus width 
  localparam NC=(PORT==4)?`MC_ENGINE4S:(PORT==3)?`MC_ENGINE3S:(PORT==2)?`MC_ENGINE2S:`MC_ENGINE1S;	// Number of cores
  localparam IC=1;	// input channels
`ifdef ICE_K8
  localparam JBW=64;	// internal core I bus width 
`else
  localparam JBW=32;	// internal core I bus width 
`endif
  localparam KBW=JBW;	// internal core O bus width 
  localparam UPS=(PORT==4)?`MC_UPS4:(PORT==3)?`MC_UPS3:(PORT==2)?`MC_UPS2:`MC_UPS1;	// Output UPS mode
  localparam NPER=(UPS>0)? UPS:1;	// output samples per transfer
  localparam LBW=(UPS>0)? 32:KBW;	// output core width to summer
  localparam IAW=IBW/8;	// internal auxiliary bus width
  localparam JAW=JBW/8;	// internal auxiliary bus width
  localparam KAW=KBW/8;	// internal auxiliary bus width
  localparam LAW=LBW/8;	// internal auxiliary bus width
  localparam OAW=OBW/8;	// internal auxiliary bus width
  localparam NCO=(NC>16)?5:(NC>8)?4:(NC>4)?3:(NC>2)?2:1;	// num chan order
  localparam NCOM=(NCO<2)?2:NCO;				// num chan order minimum
  localparam FLGO=(`ICE_CARD=="S6M")? `PACKET+`OREG : `PACKET;	// input fifo flags
  localparam FLGI=(UPS>0)? `NEMPTY:0;	
  localparam MC=(PORT<=2)?1:0;
  localparam MT=(PORT>=3)?1:0;
  localparam CLKX=`SYS_CLKX;
`include "../lib/functions.h"
  localparam CLKF=clkf(PORT);	// MHz
  localparam FCLK=fclk(PORT);	// Hz
  localparam MIC=(IAW==8)? 5 : (IAW==16)? 4 : (IAW==32)? 3 : 2;	// order cycles per 256by
  localparam MJC=(JAW==4)? 6 : (JAW==8)? 5 : 4;			// order cycles per 256by
`ifdef MC_NR
  localparam NR=`MC_NR;
`else
  localparam NR=8;
`endif
`ifdef MC_NRI
  localparam NRI=`MC_NRI;
`else
  localparam NRI=0;
`endif
  localparam IKR=(UPS>0)? 8:4;
  localparam OKR=4;

  input sclk;		// system clock
  input srst;		// system reset
  input scs;		// system select (qualifies swr and srd for this core)
  input swr;		// register write
  input srd;		// register read
  input [31:0] saddr;	// register address
  input [31:0] swrbus;	// register data
  output [31:0] srdbus;	// register data
  output sack;		// acknowledge

  input ioclk;		// I/O data bus clock

  output istat;		// istat - ready to accept 64 by burst
  input iena;		// input enable (one cycle early) (continuous 64 byte bursts)
  input [7:0] isel;	// input channel selection
  input [IBW-1:0] ibus;	// input data bus

  output ostat;		// ostat - ready to provide 64 byte burst
  input  oena;		// output enable (one cycle early) (continuous 64 byte bursts)
  output [17:0] osel;	// output channel selection
  output [OBW-1:0] obus;// output data bus

`ifdef MC_FEEDBACK
  output ostatfb;	// ostat - ready to provide 64 byte burst
  input  oenafb;	// output enable (one cycle early) (continuous 64 byte bursts)
  output [7:0] oselfb;	// output channel selection
  output [OBW-1:0] obusfb;// output data bus
`endif

  output [7:0] test;	// 8 test signals for scope probes (test bit=0 of sub-cores must be enable)

  wire L=0, H=1;
  assign sack=H;
//  wire ioclk_ = ((CLKF>CLKX) && (UPS<=0))? sclk : ioclk;
  wire ioclk_ = (CLKF>CLKX)? sclk : ioclk;
  wire [1:0] sclks_ = {sclk,sclk};

  // core variables 
  wire [3:0]     sbank = saddr[19:16];		// specific channel bank
  wire [1:0]     gsys  = saddr[23:22];		// system|group addressing
  wire           sys   = saddr[23];		// system addressing
  wire           grp   = saddr[22];		// group addressing
  wire [NCO-1:0] schn  = saddr[NCO-1+16:16];	// specific channel address
  wire [31:0]    srdbus_[NC-1:0],srdbusg,srdbusc;
  wire [JBW-1:0] jbus;
  reg  [JBW-1:0] kbus;
  reg  [JBW-1:0] ibus_[NC-1:0];
  wire [LBW-1:0] obus_[NC-1:0];
  reg  [3:0]     kchn;
  wire [JAW-1:0] jsel;
  wire [NC-1:0]  istat_,jstat_,ostat_;
  reg  [NC-1:0]  ena_;
  reg  [NCOM-1:0] wchn;
  wire [NC*LBW-1:0] wbusc;
  wire [NC*LAW-1:0] wselc;
  reg  ival,ena,ienb,ienc,wena,wenaz,sena,frst,fupd,jstat,kstat,prime;
  wire [31:0] tracer;
  wire [JBW-1:0] obusfb_[NC-1:0];
  wire [NC-1:0] ostatfb_;
  reg  [MIC-1:0] icnt;
  reg  [MJC-1:0] jcnt;

  // system status ram with parameterized ID 
`ifdef MC_DBG
  statusblkdbg #("MCOR",NC,IC,32'h1CE,UPS,0,CLKX,CLKF,32,12) ss (sclk,scs&sys,saddr,swr,swrbus,srd,srdbusg, ioclk_,tracer);
`else
  portstat #(PORT,"MCOR",NC,IC,32'h1CE,UPS,0,CLKX,CLKF) ss (sclk,scs&sys,swr,saddr,swrbus,srdbusg);
`endif

  // global multicore settings
  reg [5:0] mchn; reg [4:0] lchn; reg [3:0] shft; reg [15:0] scale,nfscl; reg [31:0] rate;
  reg sbase,stag,split,bulk,otag,itag,ibnk,obnk,rsas,icsel,tbnk,swrsys,swrscl,swrnfs,swrate,swrtap,swrena;
  reg scsc,swrc,swap,noop,ilrs,debug,throt,ups,mcfb;
  always @(posedge sclk) begin
    sbase  <= (saddr[10:0]==0);			// the base register of each core
    swrsys <= scs && swr && (gsys>1) && (saddr[5:2]==0);
    swrscl <= scs && swr && (gsys>1) && (saddr[5:2]==1);
    swrnfs <= scs && swr && (gsys>1) && (saddr[5:2]==2);
    swrate <= scs && swr && (gsys>1) && (saddr[5:2]==3);
    swrtap <= scs && swr && (gsys>1) && (saddr[5:2]==4);
    swrena <= scs && swr && (gsys>1) && (saddr[5:2]==6);
    scsc   <= scs &&        (gsys<2);
    swrc   <= scs && swr && (gsys<2);
    if (swrsys) sena  <= swrbus[0];
    if (swrsys) stag  <= swrbus[1];		// old disable startup tags
    if (swrsys) split <= swrbus[2];
    if (swrsys) bulk  <= swrbus[3];		// 512by|1kby chunks
    if (swrsys) rsas  <= swrbus[4];		// reset at start and stop
    if (swrsys) swap  <= swrbus[5];
    if (swrsys) noop  <= swrbus[6];
    if (swrsys) ilrs  <= swrbus[7];
    if (swrsys) itag  <= swrbus[8];		// data tag input
    if (swrsys) otag  <= swrbus[9];		// data tag output
    if (swrsys) ibnk  <= swrbus[10];		// direct tbank input slicing
    if (swrsys) obnk  <= swrbus[11];		// direct tbank output slicing
    if (swrsys) debug <= swrbus[12];
    if (swrsys) throt <= swrbus[13];
    if (swrsys) ups   <= swrbus[14];		// enable output upsample 
    if (swrsys) mcfb  <= swrbus[15];		// multichannel feedback
    if (swrsys) mchn  <= swrbus[21:16];		// max channels to use 
    if (swrsys) icsel <= swrbus[22];		// icsel mode - Input Core Select
    if (swrsys) tbnk  <= swrbus[23];		// tbank mode - Tuner Bank Specifics
    if (swrsys) shft  <= swrbus[27:24];
    if (swrscl) scale <= swrbus[15:0];
    if (swrnfs) nfscl <= swrbus[15:0];
    if (swrate) rate  <= swrbus;
    if (swrena) sena  <= swrbus[0];
    lchn <= (mchn==0)? NC-1 : mchn-1;		// last channel
  end

  //		 jcnt	itcnt
  //  ienbX	 30	126
  //  ienbX jchn 31	127	itagok	ibnkok	
  //  ienb  jchn 0	0	itcnt
  //  ienc  jbus kchn	1 	itagme	ibnkme
  //  iena_ kbus 
  //        ibus_
  //

  // input bus distributor - slow to system clock for easier routing
  always @(posedge ioclk) ival <= iena;
  wire fswr = ival && (icnt==0);
  wire fsrd = ienb && (jcnt==30);
  wire fstat,fszs,irdy;
  wire [3:0] ichnx = icsel? isel[7:4] : {ibus[48],ibus[32],ibus[16],ibus[0]};
  queMxNa #(9,8) fs (ioclk,ioclk_, frst,L, fswr,fsrd, {ichnx,isel[3:0]},jsel, fstat,fszs);
  wire [3:0] jchn = jsel[7:4];
  fifoNxM #(IBW,JBW,NRI,3,0,FLGI) fi (ioclk,frst, istat,ival,ibus, ioclk_,frst, irdy,ienb,jbus);

  wire wrdy,wrdyn;
  wire enad; delaypipe #(8) obc (sclk,ena,enad);
  always @(posedge sclk) begin
    ena  <= (ena_!=0);				// any cores enabled
    fupd <= srst || (rsas? ena^enad : !ena);	// reset at restart of channels
//    fupd <= srst || (rsas? ena^enad : ena&!enad);	// reset at restart of channels
  end
  reg [6:0] itcnt;
  reg [NCOM-1:0] itchn;
  wire itnxt = (itcnt==127) && ienb;
  wire itrst = frst || (itnxt && (itchn==lchn) );
  wire chnoor = (jchn > lchn);
  wire itagok = istat_[jchn] || chnoor || icsel;
  wire ibnkok = istat_[itchn];
  wire ichn = jsel[0];
  wire tsen = jsel[1];				// timecode sync enable
  wire tcen = jsel[2] && !itag;			// use timecode synchronization
  wire crdy = (MC>0) || (UPS>0) || (tcen?tsen:H);
  wire itagx = ienb && (jcnt==0);
  reg irdyd; 
  wire irdys = (irdy && irdyd);

  always @(posedge ioclk_) begin
    irdyd <= irdy;
    frst  <= (UPS>0)? !sena || (tbnk&fupd) : fupd;	// reset fifos when not enabled
    jstat <= (ena_&!istat_)==0;				// cumulative input ready
    kstat <= (ena_&!ostat_)==0;				// cumulative output ready
    if (!irdys) ienb <= 0; else ienb <= ibnk? ibnkok : itag? itagok : noop? wrdyn : jstat; 	// feed data in chunks
    ienc  <= ienb;					// data available on jbus
    kbus  <= jbus;					// register for broadcast route timing
    if (frst) prime <= 1;  else if (fstat) prime <= 0;
    if (frst) kchn <= 0;   else if (itagx|prime) kchn  <= jchn;
    if (itrst) itchn <= 0; else if (itnxt) itchn <= itchn+1;
    if (frst)  itcnt <= 0; else if (ienb) itcnt <= itcnt+1;
    if (frst) jcnt <= 0;   else if (ienb) jcnt <= jcnt+1;
  end
  always @(posedge ioclk) begin
    if (frst) icnt <= 0; else if (ival) icnt <= icnt+1;
  end

  // core instantiations 
  genvar i,l,n;
  generate
  for (i=0; i<NC; i=i+1) begin:cor
    wire sack_;
    wire odd = (i&1)? H:L;
    wire [7:0] osel_;
    wire [32:0] test_;
    reg  srst_,scs_,swr_,iena_,oena_,ienb_,oenafb_,frst_;
    reg [15:0] saddr_;
    reg [31:0] swrbus_;
    wire [1:0] swrs_ = {swr_,swr_};
    always @(posedge sclk) begin
      srst_  <= srst; 				// channel select by group bits or index
      scs_   <= scsc && ( grp? (sbank[0] || sbank[odd+1]) && (!sbank[3] || i<=lchn) : (schn==i) );
      saddr_ <= saddr[15:0];
      if (scs_) swrbus_ <= swrbus;
      if (!scs_) swr_ <= 0; else swr_ <= swr; 	// makes sure scs_ is not optimized to SR if swr_ = scs && swr is used
      if (srst) ena_[i] <= 0; else if (swr_ && sbase) ena_[i] <= swrbus_[0];	// core enable bit
    end

    wire ichnme  = !split || !(odd^ichn);
    wire itagme  = !itag  || (kchn==i);
    reg ibnkme; always @(posedge ioclk_) if (ienb) ibnkme <= !ibnk  || (itchn==i);

    always @(posedge ioclk_) begin
      ienb_ <= ena_[i] && (iena_ || ienb_);
      iena_ <= ena_[i] && ienc && (crdy || ienb_) && ichnme && itagme && ibnkme;
      ibus_[i] <= kbus;				// register for broadcast route timing
      oena_ <= (UPS>0)? wenaz && ena_[i] : wena && (wchn==i);
      oenafb_ <= mcfb && wena && (wchn==i);
      frst_ <= !ena_[i];
    end
    if (PORT==1) begin:p1
     `MC_ENGINE1 #(PORT,JBW,LBW) engi ( sclks_,srst_, scs_,saddr_, swrs_,swrbus_, srd,srdbus_[i], sack_,
		ioclk_, {jstat_[i],istat_[i]},iena_,8'h0,ibus_[i], ostat_[i],oena_,osel_,obus_[i], test_);
    end
    if (PORT==2) begin:p2
     `MC_ENGINE2 #(PORT,JBW,LBW) engi ( sclks_,srst_, scs_,saddr_, swrs_,swrbus_, srd,srdbus_[i], sack_,
		ioclk_, {jstat_[i],istat_[i]},iena_,8'h0,ibus_[i], ostat_[i],oena_,osel_,obus_[i], test_);
    end
    if (PORT==3) begin:p3
     `MC_ENGINE3 #(PORT,JBW,LBW) engi ( sclks_,srst_, scs_,saddr_, swrs_,swrbus_, srd,srdbus_[i], sack_,
		ioclk_, {jstat_[i],istat_[i]},iena_,8'h0,ibus_[i], ostat_[i],oena_,osel_,obus_[i], test_);
    end
    if (PORT==4) begin:p4
     `MC_ENGINE4 #(PORT,JBW,LBW) engi ( sclks_,srst_, scs_,saddr_, swrs_,swrbus_, srd,srdbus_[i], sack_,
		ioclk_, {jstat_[i],istat_[i]},iena_,8'h0,ibus_[i], ostat_[i],oena_,osel_,obus_[i], test_);
    end
    assign wbusc[LBW-1+LBW*i:LBW*i] = obus_[i]; 	// one big output vector
    assign wselc[LAW-1+LAW*i:LAW*i] = osel_; 		// one big select vector
`ifdef MC_FEEDBACK
    wire irdyfb;	// copies the testVector output to multichannel FIFOs for output feedback monitoring
    fifoNxM #(32,KBW,0,3) fbf (sclk,frst_, irdyfb,test_[32],test_[31:0], ioclk_,frst_, ostatfb_[i],oenafb_,obusfb_[i]);
`endif
  end
  endgenerate

  generate

  if (UPS>0) begin
// wena
// wenb	oena_
// wenc	obus_ wbusc
// ...
// wvalx wbusx
  reg wenb,wenc,hold; 
  wire wrdyz,whold;
  always @(posedge ioclk_) begin
    if (!wrdyz) wenaz <= 0; else wenaz <= sena && kstat && !whold;
    wenb <= wenaz; wenc <= wenb;
  end

  // throttle circuit
  reg [9:0] rcnt;  // rate count
  wire [31:0] fclkb = FCLK*NPER;
  wire rstr = (rcnt==fclkb[29:20]);
  always @(posedge sclk) begin
    if (!throt) hold <= 0; else hold <= (rcnt>rate[29:20]);
    if (rstr) rcnt <= 1; else rcnt <= rcnt+1;
  end
  delayp #(4) dpz (ioclk_,hold,whold);	// resync

  // adder circuit
  wire wvalz; wire [LBW*UPS-1:0] wbusz;
  mcaddups #(NC,LBW,UPS) oad (ioclk_,sena,ena_, ups,scale,nfscl, sclk,swrtap,swrbus, wenc,wselc,wbusc, wvalz,wbusz);
  reg wvalx; reg [LBW*UPS-1:0] wbusx;
  always @(posedge ioclk_) begin
    wvalx <= (MC>0 && noop)? ienc : wvalz;
    wbusx <= (MC>0 && noop)? jbus : wbusz;
  end
  fifoNxM #(LBW*UPS,OBW,NR,4,0,FLGO) fo (ioclk_,frst, wrdyz,wvalx,wbusx, ioclk,frst, ostat,oena,obus);
  assign osel[15:0] = jstat_[NC-1:0];	// for dmacontroller to decide which channel needs data
  assign osel[16] = ienb;		// fifo output enabled
  assign osel[17] = icsel;		// enable input channel select mode
  assign wrdyn = wrdyz;

  end

  if (UPS==0 || FEEDBACK) begin

  // collector variables
  reg wstr,wenb,wenc,wval,wvalx,wpha,wsamd;
  reg [NCOM-1:0] wchk, wchnb,wchnc;
  reg [3:0] wchnd; 
  reg [6:0] wcnt; 
  reg [KAW-1:0] wchnv,wchnx; 
  reg [KBW-1:0] wbus,wbusx;
  wire [7:0] ochn = wchnc;
  wire wrap = (wchk>lchn);
  wire wsam = (wchk==wchn) && wena;
  wire ordy = (FEEDBACK)? ostatfb_[wchk] : ostat_[wchk];

  // collector logic
  always @(posedge ioclk_) begin
    wsamd <= wsam;
    if (wrap) wchk <= 0; else if ((!ordy)||wstr) wchk <= wchk+1;
    if (wstr) wchn <= wchk;
    if (wstr) wpha <= 0; else wpha <= !wpha;
`ifdef ICE_K8
    wstr <= ordy && wrdy && !wstr && !wsamd && !(wena && (bulk? (wcnt<126):(wcnt[4:0]<30)));
    wena <= wstr || (wena && (bulk? (wcnt!=127):(wcnt[4:0]!=31)));
`else
    wstr <= ordy && wrdy && !wstr && !wsamd && !(wena && (bulk? (wcnt<126):(wcnt[3:0]<14)));
    wena <= wstr || (wena && (bulk? (wcnt!=127):(wcnt[3:0]!=15)));
`endif
    if (!wena) wcnt <= 0; else wcnt <= wcnt+1;
    wenb  <= wena; wenc <= wenb; wval <= wenc;
    wchnb <= wchn; wchnc <= wchnb; wchnd <= wchnc; 
    wchnv <= otag? 0 : (KAW>4)? ochn : wpha? ochn[7:4] : ochn[3:0];
    wbus  <= (FEEDBACK)? obusfb_[wchnc] : obus_[wchnc];
    wchnx <= wchnv;
    wvalx <= (MC>0 && noop)? ienc : wval;
    wbusx <= (MC>0 && noop)? jbus
          : (KAW>4 && otag)? {wbus[63:49],wchnd[3], wbus[47:33],wchnd[2], wbus[31:17],wchnd[1], wbus[15:1],wchnd[0]}
          : (KAW<5 && swap)? {wbus[23:16],wbus[31:24],wbus[7:0],wbus[15:8]}
          : wbus;
  end

`ifdef MC_FEEDBACK
  assign oselfb[7:6] = 2;
  fifoNxM #(KBW+KAW,OBW+OAW,NR,4,`BPAR,FLGO) fofb (ioclk_,frst, wrdy,wvalx,{wchnx,wbusx}, ioclk,frst, ostatfb,oenafb,{oselfb[5:0],obusfb});
`else
  fifoNxM #(KBW+KAW,OBW+OAW,NR,4,`BPAR,FLGO) fo (ioclk_,frst, wrdy,wvalx,{wchnx,wbusx}, ioclk,frst, ostat,oena,{osel[7:0],obus});
  assign osel[17:8] = 0;
  assign wrdyn = wrdy;
`endif
  assign tracer = {wbusx[31:1],wvalx};
  end
  endgenerate

  reg c0; always @(posedge sclk) c0 <= grp || (schn==0);
  generate
  if (COMSTAT) begin
    // common status ram for write/readback only status needs (256B*16 = 4kB)
    spram #(4,32,0) rs (sclk,scsc, {grp? 4'd0:schn,saddr[7:2]},swrc, swrbus,srdbusc);
    assign srdbus = sys? srdbusg : c0? srdbus_[0] : srdbusc;
  end
  if (!COMSTAT) begin
    // mux of individual core status busses
    reg [NCO-1:0] rchn; always @(posedge sclk) if (c0) rchn <= 0; else rchn <= schn;
    assign srdbusc = srdbus_[rchn];
    assign srdbus = sys? srdbusg : srdbusc;
  end
  endgenerate
 
  assign test = {istat,ostat,iena,oena, wena,wrdy,irdy,ienb};

endmodule

/*
    wstr
    0 wena 0  wchn
    1 wenb 1  wchnb
    0 wenc 2  wchnc
    1 wval 3  wchnv

      wena 14
      wena 15


    ienb
    ienc  jbus
          kbus  iena_
	        ibus_
*/
