/**
  Verilog modules for implementing core tasks

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  11/15/2013
*/

import CORE_funcs::*;

/* verilator lint_off MULTITOP */

module coretasks();

endmodule

// MC is Multi Cycle count
module sumf #(parameter N=1, parameter MC=0) (
  input clk, input rst,
  input str, input ena, input fptx a[N], output fptx sum);

  localparam ORD = (N<=2)? 1 : (N<=4)? 2 : (N<=8)? 3 : (N<=16)? 4 : (N<=32)? 5 : 6;
  localparam M = 1<<ORD;
  localparam H = N/2;
  localparam HP = (N%2==1)? H:0;
  localparam L = M/2;
  wire T=1, F=0;

  fptx s[M];
  fptx sumz,suml,sumx;

  // input data is 1 cyc delayed from str & ena
  wire zero; delaypipe #(ORD) dps (clk,str,zero);
  wire dena; delaypipe #(ORD) dpe (clk,ena,dena);

  genvar i,j;
  generate 
  for (i=0; i<H; i=i+1) begin:tap			// symmetric duals
    always @(posedge clk) s[L+i] <= a[i] + a[N-1-i];
  end
  for (i=H; i<L; i=i+1) begin:tapp
    always @(posedge clk) s[L+i] <= (i==HP)? a[i] : 0;	// center tap OR pad non-power2 sums
  end
  for (j=ORD-2; j>0; j=j-1) begin:suma
    localparam TROW=1<<j;
    localparam TOFF=2<<j;
    for (i=0; i<TROW; i++) always @(posedge clk) s[TROW+i] <= s[TOFF+i] + s[TOFF+TOFF-1-i];
  end
  endgenerate

  assign sumz = zero? 0 : sumx;
/* verilator lint_off SELRANGE */
  assign suml = (N<=1)? a[0] : (N<=2)? a[0]+a[1] : (M<4)? s[2] : s[2]+s[3];
/* verilator lint_on SELRANGE */
  always @(posedge clk) if (rst) sumx <= 0; else if (dena) sumx <= sumz + suml;

  generate
  if (MC>0) begin
    // count off the delay offset the first time through the loop - actual mc
    reg prime; always @(posedge clk) if (rst) prime <= F; else if (str) prime <= T;
    reg primed; always @(posedge clk) if (rst) primed <= F; else if (prime && (str||!ena)) primed <= T;
    reg [3:0] off; always @(posedge clk) if (!prime) off <= MC; else if (!primed) off <= off-1;
    srlMxN #(4,32,0) srl (clk,T,off,sumx,sum);
  end
  if (MC<=0) begin
    assign sum = sumx;
  end
  endgenerate

endmodule

module msumf #(parameter N=1) (
  input clk, input rst,
  input fptx a[N+1], input fptx b[N],  output fptx sum, input intx n);
  import CORE_funcs::*;
  wire T=1, F=0;
  fptx p[N];
  genvar i;
  generate 
  for (i=0; i<N; i=i+1) begin:tap
    always @(posedge clk) p[i] <= mulff(a[i],b[i]);
  end
  endgenerate
  sumf #(N) sp (clk,rst, T,T, p,sum);
endmodule

module divff_task (
  input clk, input rst, input en, input xn,
  input fptx a, input fptx b, output fptx c);
  import CORE_funcs::*;
  fptx a2,b2;
  always @(posedge clk) begin
    if (en) a2 <= a;
    if (en) b2 <= invf(b);
    if (xn) c  <= mulff(b2,a2);
  end
endmodule


module divii_task (	// output must be < 1
  input clk, input rst, input en, input xn,
  input int32 a, input int32 b, output dptx c);
  import CORE_funcs::*;
  reg go;
  reg[5:0] ii;
  reg [63:0] ax,bx,bm;
  wire [63:0] bp=bm+bx;
  always @(posedge clk) begin
    go <= en || (go && ii>0);
    if (en) begin ii<=31; c<=0; ax<={a,32'b0}; bx<={b,31'b0}; bm<=0; end
    if (go) begin ii<=ii-1; bx<=(bx>>1); end
    if (go && ax[61:0]>=bp[61:0]) begin c[ii]<=1'b1; bm<=bp; end
    //if (go) $write("divii ii=%d a=%x ax=%x b=%x bx=%x bm=%x bp=%x c=%x\n",ii,a,ax,b,bx,bm,bp,c);
  end
endmodule

module sincf_task (
  input clk, input rst, input en, input xn,
  input fptx p, output fptx sc);
  import CORE_funcs::*;
  fptx t1;
  always @(posedge clk) begin
    if (en) t1 <= sincos_part1(p,0);
    if (xn) sc <= sincos_part3(sincos_part2(t1));
  end
endmodule

module coscf_task (
  input clk, input rst, input en, input xn,
  input fptx p, output fptx sc);
  import CORE_funcs::*;
  fptx t1;
  always @(posedge clk) begin
    if (en) t1 <= sincos_part1(p,1);
    if (xn) sc <= sincos_part3(sincos_part2(t1));
  end
endmodule

module sincoscf_task (
  input clk, input rst, input en, input xn,
  input fptx p, output CxFptx sc);
  import CORE_funcs::*;
  CxFptx t1;
  uinty t2x,t2y;
  always @(posedge clk) begin
    if (en) t1.x <= sincos_part1(p,1);
    if (en) t1.y <= sincos_part1(p,0);
    if (xn) sc.x <= sincos_part3(sincos_part2(t1.x));
    if (xn) sc.y <= sincos_part3(sincos_part2(t1.y));
  end
endmodule

module sincoscfx_task (
  input clk, input rst, input en, input xn,
  input fptx p, output CxFptx sc);
  import CORE_funcs::*;
  logic fn;
  CxFptx t1;
  uinty t2x,t2y;
  always @(posedge clk) begin
    fn <= en;
    if (en) t1.x <= sincos_part1(p,1);
    if (en) t1.y <= sincos_part1(p,0);
    if (fn) t2x  <= sincos_part2(t1.x);
    if (fn) t2y  <= sincos_part2(t1.y);
    if (xn) sc.x <= sincos_part3(t2x);
    if (xn) sc.y <= sincos_part3(t2y);
  end
endmodule

module pol2recf_task (
  input clk, input rst, input en, input xn,
  input fptx amp, input fptx pha, output CxFptx sc);
  import CORE_funcs::*;
  logic fn;
  CxFptx t1;
  uinty t2x,t2y;
  fptx scl,sclx,scly;
  always @(posedge clk) begin
    fn <= en;
    if (en) scl  <= amp;
    if (en) t1.x <= sincos_part1(pha,1);
    if (en) t1.y <= sincos_part1(pha,0);
    if (fn) t2x  <= sincos_part2(t1.x);
    if (fn) t2y  <= sincos_part2(t1.y);
    if (fn) sclx <= t1.x[15]? -scl : scl;
    if (fn) scly <= t1.y[15]? -scl : scl;
    if (xn) sc.x <= mulff(sclx,t2x[16:0]);
    if (xn) sc.y <= mulff(scly,t2y[16:0]);
  end
endmodule

module mulccx_task (
  input clk, input rst, input en, input xn,
  input CxFptx a, input CxFptx b, output CxFptx c);
  import CORE_funcs::*;
  CxFptw t1,t2;
  always @(posedge clk) begin
    if (en) begin t1.x <= mulff(a.x,b.x); t1.y <= mulff(a.y,b.y); end
    if (en) begin t2.x <= mulff(a.y,b.x); t2.y <= mulff(a.x,b.y); end
    if (xn) c.x <= t1.x - t1.y;
    if (xn) c.y <= t2.x + t2.y;
  end
endmodule

module mulccjx_task (
  input clk, input rst, input en, input xn,
  input CxFptx a, input CxFptx b, output CxFptx c);
  import CORE_funcs::*;
  CxFptw t1,t2;
  always @(posedge clk) begin
    if (en) begin t1.x <= mulff(a.x,b.x); t1.y <= mulff(a.y,b.y); end
    if (en) begin t2.x <= mulff(a.y,b.x); t2.y <= mulff(a.x,b.y); end
    if (xn) c.x <= t1.x + t1.y;
    if (xn) c.y <= t2.x - t2.y;
  end
endmodule

module mag2fx_task (
  input clk, input rst, input en, input xn,
  input CxFptx a, output fptx b);
  import CORE_funcs::*;
  fptw t1,t2;
  always @(posedge clk) begin
    if (en) begin t1 <= mulff(a.x,a.x); t2 <= mulff(a.y,a.y); end
    if (xn) b <= t1 + t2;
  end
endmodule

module mulccA_task #(parameter N=1) (
  input clk, input rst, input en, input xn,
  input CxFptx a, input CxFptx b[N], output CxFptx c[N]);
  genvar i;
  generate 
  for (i=0; i<N; i=i+1) begin
    mulccx_task inst (clk,rst,en,xn, a,b[i],c[i]);
  end
  endgenerate
endmodule

module atan2cf_task (
  input clk, input rst, input en, input xn,
  input fptx y, input fptx x, output fptx a);
  import CORE_funcs::*;
  wire T=1, F=0;
  logic[35:0] luta[0:1023] = '{
 36'hFFFFFF000,36'h800FFF000,36'h4B9FFF000,36'h347FFF000,36'h27EFFF000,36'h202FFF000,36'h1AEFFF000,36'h172FFF000,36'h144FFF000,36'h120FFF000,36'h104FFF000,36'h0ECFFF000,36'h0D8FFF000,36'h0C8FFF000,36'h0BAFFF000,36'h0ADFFF000,
 36'h0A2FFF000,36'h099FFF000,36'h090FFF000,36'h089FFF000,36'h082FFF000,36'h07CFFF000,36'h076FFF000,36'h071FFF000,36'h06CFFF000,36'h068FFF000,36'h064FFF000,36'h060FFF000,36'h05DFFF000,36'h05AFFF000,36'h057FFF000,36'h054FFF000,
 36'h000800FFF,36'h347CB9800,36'h347E8E4B9,36'h2B6F37347,36'h23AF8427E,36'h1DDFAC202,36'h198FC31AE,36'h163FD2172,36'h13AFDC144,36'h119FE3120,36'h0FEFE8104,36'h0E8FEC0EC,36'h0D5FEF0D8,36'h0C5FF10C8,36'h0B8FF30BA,36'h0ACFF50AD,
 36'h0A1FF60A2,36'h098FF7099,36'h08FFF8090,36'h088FF9089,36'h081FF9082,36'h07BFFA07C,36'h076FFA076,36'h071FFB071,36'h06CFFB06C,36'h068FFC068,36'h064FFC064,36'h060FFC060,36'h05CFFC05D,36'h059FFD05A,36'h056FFD057,36'h054FFD054,
 36'h000B47FFF,36'h172CB9B47,36'h202DFD800,36'h202EBB5FD,36'h1D5F274B9,36'h1A1F663E0,36'h172F8E347,36'h14AFA92D5,36'h128FBB27E,36'h10CFC823A,36'h0F5FD2202,36'h0E1FD91D5,36'h0D0FDF1AE,36'h0C1FE418E,36'h0B4FE7172,36'h0A9FEA159,
 36'h09FFED144,36'h096FEF131,36'h08EFF1120,36'h087FF2111,36'h080FF3104,36'h07AFF40F7,36'h075FF50EC,36'h070FF60E2,36'h06BFF70D8,36'h067FF80D0,36'h063FF80C8,36'h05FFF90C0,36'h05CFF90BA,36'h059FFA0B3,36'h056FFA0AD,36'h053FFA0A8,
 36'h000CB9FFF,36'h0C8D49CB9,36'h144DFDA02,36'h172E8E800,36'h172EF368E,36'h15EF37581,36'h144F664B9,36'h12AF8741F,36'h111F9F3A7,36'h0FBFB1347,36'h0E8FBE2F8,36'h0D7FC82B6,36'h0C8FD027E,36'h0BBFD724F,36'h0AFFDC226,36'h0A4FE0202,
 36'h09BFE41E3,36'h093FE71C7,36'h08BFE91AE,36'h084FEC198,36'h07EFED184,36'h078FEF172,36'h073FF0161,36'h06EFF2152,36'h06AFF3144,36'h066FF4137,36'h062FF512B,36'h05FFF5120,36'h05BFF6116,36'h058FF710C,36'h055FF7104,36'h053FF80FB,
 36'h000D81FFF,36'h07CDC5D81,36'h0D8E2BB47,36'h10CE8E972,36'h120EDF800,36'h120F1D6DF,36'h116F4C5FD,36'h107F6F549,36'h0F7F894B9,36'h0E7F9D442,36'h0D8FAD3E0,36'h0CBFB938D,36'h0BEFC3347,36'h0B3FCB30A,36'h0A8FD12D5,36'h09FFD72A7,
 36'h097FDB27E,36'h08FFDF25A,36'h088FE223A,36'h082FE521D,36'h07CFE8202,36'h076FEA1EA,36'h071FEC1D5,36'h06DFED1C1,36'h069FEF1AE,36'h065FF019D,36'h061FF118E,36'h05EFF217F,36'h05AFF3172,36'h057FF4165,36'h055FF5159,36'h052FF514E,
 36'h000DFDFFF,36'h054E22DFD,36'h099E5FC1F,36'h0C8EA1A7E,36'h0E2EDF920,36'h0ECF13800,36'h0ECF3D713,36'h0E6F5F651,36'h0DDF795B0,36'h0D3F8E52A,36'h0C8F9F4B9,36'h0BDFAD458,36'h0B3FB8405,36'h0AAFC13BD,36'h0A1FC837E,36'h099FCE347,
 36'h091FD4315,36'h08AFD82EA,36'h084FDC2C2,36'h07EFDF29F,36'h079FE227E,36'h074FE5261,36'h06FFE7246,36'h06BFE922E,36'h067FEB217,36'h063FEC202,36'h060FEE1EF,36'h05CFEF1DD,36'h059FF01CC,36'h056FF11BD,36'h054FF21AE,36'h051FF31A1,
 36'h000E51FFF,36'h03CE67E51,36'h071E8ECB9,36'h099EBBB47,36'h0B3EE9A02,36'h0C2F138EC,36'h0C8F37800,36'h0C8F56737,36'h0C4F6F68E,36'h0BEF845FD,36'h0B7F95581,36'h0AFFA3516,36'h0A8FAE4B9,36'h0A0FB8467,36'h099FC041F,36'h092FC73E0,
 36'h08BFCD3A7,36'h085FD2374,36'h080FD6347,36'h07AFDA31D,36'h076FDD2F8,36'h071FE02D5,36'h06DFE32B6,36'h069FE5299,36'h065FE727E,36'h061FE9266,36'h05EFEA24F,36'h05BFEC23A,36'h058FED226,36'h055FEE214,36'h053FEF202,36'h050FF01F2,
 36'h000E8EFFF,36'h02DE9CE8E,36'h057EB6D2A,36'h078ED6BE0,36'h090EF8AB6,36'h0A0F199AE,36'h0A9F378C8,36'h0ADF52800,36'h0ADF69752,36'h0ABF7C6BB,36'h0A7F8D638,36'h0A1F9B5C5,36'h09CFA7561,36'h096FB1508,36'h090FB94B9,36'h08BFC0472,
 36'h085FC7433,36'h080FCC3FA,36'h07BFD13C7,36'h076FD5398,36'h072FD936E,36'h06EFDC347,36'h06AFDF323,36'h066FE1302,36'h063FE42E4,36'h05FFE62C8,36'h05CFE72AD,36'h059FE9295,36'h057FEA27E,36'h054FEC269,36'h051FED255,36'h04FFEE243,
 36'h000EBBFFF,36'h023EC5EBB,36'h044ED7D81,36'h060EEEC58,36'h076F08B47,36'h086F22A4F,36'h090F3B972,36'h096F528AD,36'h099F66800,36'h099F78766,36'h097F886DF,36'h094F95667,36'h090FA15FD,36'h08CFAB59E,36'h088FB4549,36'h083FBB4FD,
 36'h07FFC24B9,36'h07AFC747B,36'h076FCC442,36'h072FD140F,36'h06EFD53E0,36'h06AFD83B5,36'h067FDB38D,36'h063FDE368,36'h060FE0347,36'h05DFE2327,36'h05AFE430A,36'h057FE62EF,36'h055FE82D5,36'h052FE92BE,36'h050FEB2A7,36'h04EFEC292,
 36'h000EDFFFF,36'h01CEE6EDF,36'h037EF3DC5,36'h04FF04CB9,36'h062F18BBD,36'h071F2DAD5,36'h07CF41A02,36'h083F55944,36'h087F66899,36'h089F77800,36'h089F85777,36'h087F926FC,36'h085F9D68E,36'h082FA662B,36'h07FFAF5D1,36'h07CFB7581,
 36'h078FBD538,36'h075FC34F5,36'h071FC84B9,36'h06DFCD481,36'h06AFD144E,36'h067FD441F,36'h064FD83F4,36'h060FDB3CC,36'h05EFDD3A7,36'h05BFE0385,36'h058FE2365,36'h056FE4347,36'h053FE532B,36'h051FE7310,36'h04FFE82F8,36'h04DFEA2E0,
 36'h000EFCFFF,36'h017F01EFC,36'h02DF0ADFD,36'h041F17D08,36'h052F27C1F,36'h060F37B47,36'h06BF48A7E,36'h072F599C7,36'h077F68920,36'h07AF77889,36'h07CF84800,36'h07CF8F784,36'h07BF9A713,36'h079FA36AD,36'h077FAC651,36'h074FB35FD,
 36'h072FBA5B0,36'h06FFBF56A,36'h06CFC552A,36'h069FC94EF,36'h066FCD4B9,36'h063FD1486,36'h060FD5458,36'h05DFD842D,36'h05BFDA405,36'h058FDD3E0,36'h056FDF3BD,36'h054FE139D,36'h051FE337E,36'h04FFE5362,36'h04DFE6347,36'h04BFE832D,
 36'h000F13FFF,36'h013F17F13,36'h026F1EE2B,36'h037F28D49,36'h046F35C72,36'h053F42BA7,36'h05DF50AEA,36'h064F5EA3A,36'h06AF6B998,36'h06EF78904,36'h070F8487C,36'h071F8E800,36'h071F9878E,36'h070FA1727,36'h06FFA96C8,36'h06DFB0672,
 36'h06BFB7622,36'h069FBC5D9,36'h067FC2596,36'h064FC6558,36'h062FCB51F,36'h05FFCE4EA,36'h05DFD24B9,36'h05AFD548B,36'h058FD8460,36'h056FDA438,36'h054FDD413,36'h051FDF3F0,36'h04FFE13D0,36'h04DFE33B1,36'h04BFE4394,36'h04AFE6379,
 36'h000F27FFF,36'h010F2AF27,36'h020F2FE51,36'h02FF37D81,36'h03CF41CB9,36'h048F4CBFA,36'h051F58B47,36'h059F63A9F,36'h05EF6FA02,36'h063F7A972,36'h065F858EC,36'h067F8E871,36'h068F97800,36'h068FA0797,36'h067FA7737,36'h066FAE6DF,
 36'h065FB468E,36'h063FBA642,36'h061FBF5FD,36'h05FFC45BD,36'h05DFC8581,36'h05BFCC549,36'h059FCF516,36'h057FD34E6,36'h055FD64B9,36'h053FD848F,36'h051FDB467,36'h04FFDD442,36'h04DFDF41F,36'h04BFE13FF,36'h04AFE23E0,36'h048FE43C3,
 36'h000F37FFF,36'h00EF3AF37,36'h01CF3EE72,36'h029F45DB0,36'h034F4DCF5,36'h03FF56C42,36'h047F5FB98,36'h04FF69AF8,36'h054F73A61,36'h059F7D9D5,36'h05CF86952,36'h05EF8F8D8,36'h060F97868,36'h060F9F800,36'h060FA679F,36'h060FAD746,
 36'h05FFB36F3,36'h05EFB86A6,36'h05CFBD65F,36'h05BFC261C,36'h059FC65DF,36'h057FCA5A5,36'h056FCD56F,36'h054FD153D,36'h052FD450E,36'h050FD64E2,36'h04FFD94B9,36'h04DFDB492,36'h04BFDD46D,36'h04AFDF44B,36'h048FE142A,36'h046FE240B,
 36'h000F46FFF,36'h00CF48F46,36'h018F4BE8E,36'h023F50DD9,36'h02EF57D2A,36'h037F5EC81,36'h03FF66BE0,36'h046F6FB47,36'h04CF78AB6,36'h050F80A2E,36'h054F889AE,36'h056F90937,36'h058F988C8,36'h059F9F860,36'h05AFA6800,36'h05AFAC7A6,
 36'h059FB2752,36'h058FB7704,36'h057FBC6BB,36'h056FC0678,36'h055FC4638,36'h054FC85FD,36'h052FCC5C5,36'h051FCF591,36'h04FFD2561,36'h04EFD4533,36'h04CFD7508,36'h04BFD94DF,36'h049FDB4B9,36'h048FDD494,36'h046FDF472,36'h045FE1452,
 36'h000F52FFF,36'h00AF54F52,36'h015F57EA6,36'h01FF5BDFD,36'h028F60D58,36'h031F66CB9,36'h038F6DC1F,36'h03FF75B8D,36'h044F7CB02,36'h049F84A7E,36'h04CF8BA02,36'h04FF9298E,36'h051F99920,36'h052FA08BA,36'h053FA685A,36'h054FAC800,
 36'h054FB17AC,36'h053FB675D,36'h053FBB713,36'h052FBF6CE,36'h051FC368E,36'h050FC7651,36'h04FFCA618,36'h04EFCD5E3,36'h04CFD05B0,36'h04BFD3581,36'h04AFD5554,36'h048FD852A,36'h047FDA502,36'h046FDC4DC,36'h044FDE4B9,36'h043FDF497,
 36'h000F5DFFF,36'h009F5EF5D,36'h013F61EBB,36'h01BF64E1C,36'h024F69D81,36'h02CF6ECEA,36'h032F74C58,36'h038F7ABCC,36'h03EF81B47,36'h042F87AC8,36'h046F8EA4F,36'h049F949DD,36'h04BF9A972,36'h04CFA090C,36'h04EFA68AD,36'h04EFAC854,
 36'h04FFB1800,36'h04FFB57B1,36'h04EFBA766,36'h04EFBE721,36'h04DFC26DF,36'h04CFC66A1,36'h04BFC9667,36'h04AFCC631,36'h049FCF5FD,36'h048FD25CC,36'h047FD459E,36'h046FD6573,36'h045FD9549,36'h044FDB522,36'h042FDC4FD,36'h041FDE4DA,
 36'h000F66FFF,36'h008F67F66,36'h010F6AECE,36'h018F6DE38,36'h020F70DA5,36'h027F75D16,36'h02DF7AC8B,36'h033F7FC05,36'h038F85B85,36'h03CF8BB0A,36'h040F91A95,36'h043F96A26,36'h045F9C9BD,36'h047FA2959,36'h048FA78FB,36'h049FAC8A2,
 36'h04AFB184F,36'h04AFB5800,36'h04AFB97B5,36'h04AFBD76F,36'h049FC172D,36'h049FC56EE,36'h048FC86B3,36'h047FCB67B,36'h047FCE647,36'h046FD0615,36'h045FD35E6,36'h044FD55B9,36'h043FD758E,36'h042FD9566,36'h040FDB540,36'h03FFDD51C,
 36'h000F6FFFF,36'h007F70F6F,36'h00FF72EDF,36'h016F74E51,36'h01DF77DC5,36'h023F7BD3D,36'h029F7FCB9,36'h02EF84C38,36'h033F89BBD,36'h037F8EB47,36'h03BF94AD5,36'h03EF99A69,36'h040F9EA02,36'h042FA39A1,36'h043FA8944,36'h045FAC8EC,
 36'h045FB1899,36'h046FB584A,36'h046FB9800,36'h046FBD7B9,36'h046FC1777,36'h046FC4737,36'h045FC76FC,36'h044FCA6C3,36'h044FCD68E,36'h043FCF65B,36'h042FD262B,36'h041FD45FD,36'h040FD65D1,36'h040FD85A8,36'h03FFDA581,36'h03EFDC55B,
 36'h000F77FFF,36'h006F77F77,36'h00DF79EEE,36'h014F7BE67,36'h01AF7EDE3,36'h020F81D61,36'h025F85CE2,36'h02AF89C67,36'h02FF8DBF0,36'h033F92B7E,36'h036F96B10,36'h039F9BAA7,36'h03BFA0A43,36'h03DFA49E3,36'h03FFA9988,36'h040FAD931,
 36'h041FB18DF,36'h042FB5890,36'h042FB9846,36'h042FBD800,36'h042FC07BD,36'h042FC377D,36'h042FC6741,36'h042FC9708,36'h041FCC6D2,36'h040FCF69E,36'h040FD166D,36'h03FFD363F,36'h03EFD5612,36'h03EFD75E8,36'h03DFD95C0,36'h03CFDB599,
 36'h000F7DFFF,36'h006F7EF7D,36'h00CF7FEFC,36'h012F81E7B,36'h018F84DFD,36'h01DF86D81,36'h022F8AD08,36'h027F8DC92,36'h02BF91C1F,36'h02EF95BB1,36'h032F99B47,36'h035F9EAE0,36'h037FA2A7E,36'h039FA6A21,36'h03BFAA9C7,36'h03CFAE972,
 36'h03DFB2920,36'h03EFB68D3,36'h03FFB9889,36'h03FFBD842,36'h03FFC0800,36'h03FFC37C0,36'h03FFC6784,36'h03FFC974A,36'h03EFCB713,36'h03EFCE6DF,36'h03DFD06AD,36'h03DFD367E,36'h03CFD5651,36'h03CFD7626,36'h03BFD85FD,36'h03AFDA5D6,
 36'h000F84FFF,36'h005F84F84,36'h00BF85F08,36'h010F87E8E,36'h015F89E15,36'h01AF8BD9E,36'h01FF8ED2A,36'h023F92CB9,36'h027F95C4B,36'h02BF99BE0,36'h02EF9CB79,36'h031FA0B15,36'h033FA4AB6,36'h035FA8A5A,36'h037FACA02,36'h039FAF9AE,
 36'h03AFB395E,36'h03BFB6911,36'h03BFBA8C8,36'h03CFBD882,36'h03CFC083F,36'h03CFC3800,36'h03CFC67C3,36'h03CFC8789,36'h03CFCB752,36'h03CFCD71D,36'h03BFD06EB,36'h03BFD26BB,36'h03AFD468E,36'h03AFD6662,36'h039FD8638,36'h038FD9610,
 36'h000F89FFF,36'h005F8AF89,36'h00AF8BF13,36'h00FF8CE9E,36'h014F8EE2B,36'h018F90DB9,36'h01CF93D49,36'h021F95CDC,36'h024F99C72,36'h028F9CC0B,36'h02BF9FBA7,36'h02DFA3B47,36'h030FA6AEA,36'h032FAAA90,36'h034FADA3A,36'h035FB09E7,
 36'h036FB4998,36'h037FB794C,36'h038FBA904,36'h039FBD8BE,36'h039FC087C,36'h039FC383C,36'h03AFC6800,36'h03AFC87C6,36'h039FCB78E,36'h039FCD759,36'h039FCF727,36'h039FD16F6,36'h038FD36C8,36'h038FD569C,36'h037FD7672,36'h037FD9649,
 36'h000F8EFFF,36'h004F8FF8E,36'h009F90F1D,36'h00EF91EAD,36'h012F92E3F,36'h016F94DD1,36'h01AF97D66,36'h01EF99CFD,36'h021F9CC97,36'h025F9FC33,36'h028FA2BD2,36'h02AFA5B74,36'h02DFA8B1A,36'h02FFABAC2,36'h030FAEA6E,36'h032FB2A1D,
 36'h033FB59CF,36'h034FB8984,36'h035FBB93C,36'h036FBE8F7,36'h036FC08B5,36'h037FC3876,36'h037FC683A,36'h037FC8800,36'h037FCB7C8,36'h037FCD793,36'h037FCF760,36'h037FD172F,36'h036FD3701,36'h036FD56D4,36'h035FD76A9,36'h035FD8680,
 36'h000F93FFF,36'h004F93F93,36'h008F94F27,36'h00CF95EBB,36'h011F97E51,36'h014F98DE8,36'h018F9AD81,36'h01CF9DD1C,36'h01FF9FCB9,36'h022FA2C58,36'h025FA4BFA,36'h027FA7B9F,36'h02AFAAB47,36'h02CFADAF1,36'h02DFB0A9F,36'h02FFB3A4F,
 36'h030FB6A02,36'h031FB99B9,36'h032FBB972,36'h033FBE92D,36'h034FC18EC,36'h034FC38AD,36'h035FC6871,36'h035FC8837,36'h035FCA800,36'h035FCD7CA,36'h035FCF797,36'h035FD1766,36'h034FD3737,36'h034FD470A,36'h034FD66DF,36'h033FD86B6,
 36'h000F97FFF,36'h004F98F97,36'h008F98F2F,36'h00BF99EC8,36'h00FF9BE62,36'h013F9CDFD,36'h016F9ED99,36'h01AFA0D38,36'h01DFA2CD8,36'h020FA4C7B,36'h022FA7C1F,36'h025FAABC7,36'h027FACB71,36'h029FAFB1D,36'h02BFB2ACC,36'h02CFB4A7E,
 36'h02EFB7A33,36'h02FFBA9EA,36'h030FBC9A4,36'h031FBF961,36'h031FC1920,36'h032FC48E2,36'h032FC68A6,36'h032FC886C,36'h033FCA835,36'h033FCD800,36'h033FCF7CD,36'h033FD079B,36'h033FD276C,36'h032FD473F,36'h032FD6713,36'h032FD76E9,
 36'h000F9BFFF,36'h003F9CF9B,36'h007F9CF37,36'h00BF9DED4,36'h00EF9EE72,36'h012FA0E10,36'h015FA1DB0,36'h018FA3D52,36'h01BFA5CF5,36'h01EFA7C9B,36'h020FA9C42,36'h022FACBEC,36'h025FAEB98,36'h027FB1B47,36'h028FB3AF8,36'h02AFB6AAB,
 36'h02BFB8A61,36'h02CFBBA1A,36'h02DFBD9D5,36'h02EFBF992,36'h02FFC2952,36'h030FC4914,36'h030FC68D8,36'h030FC889F,36'h031FCA868,36'h031FCD833,36'h031FCE800,36'h031FD07CE,36'h031FD279F,36'h031FD4772,36'h030FD5746,36'h030FD771B,
 36'h000F9FFFF,36'h003F9FF9F,36'h006FA0F3F,36'h00AFA1EDF,36'h00DFA2E80,36'h010FA3E22,36'h013FA4DC5,36'h016FA6D6A,36'h019FA8D10,36'h01CFAACB9,36'h01EFACC63,36'h020FAEC0F,36'h022FB0BBD,36'h024FB2B6E,36'h026FB5B20,36'h027FB7AD5,
 36'h029FB9A8D,36'h02AFBCA46,36'h02BFBEA02,36'h02CFC09C1,36'h02DFC2981,36'h02DFC5944,36'h02EFC7909,36'h02EFC98D0,36'h02FFCB899,36'h02FFCD864,36'h02FFCE831,36'h02FFD0800,36'h02FFD27D0,36'h02FFD47A3,36'h02FFD5777,36'h02FFD774C,
 36'h000FA3FFF,36'h003FA3FA3,36'h006FA3F46,36'h009FA4EE9,36'h00CFA5E8E,36'h00FFA6E33,36'h012FA7DD9,36'h015FA9D81,36'h017FAAD2A,36'h01AFACCD5,36'h01CFAEC81,36'h01EFB0C30,36'h020FB2BE0,36'h022FB4B92,36'h024FB6B47,36'h025FB8AFD,
 36'h027FBBAB6,36'h028FBDA71,36'h029FBFA2E,36'h02AFC19ED,36'h02BFC39AE,36'h02BFC5972,36'h02CFC7937,36'h02CFC98FE,36'h02DFCB8C8,36'h02DFCD893,36'h02DFCE860,36'h02DFD082F,36'h02DFD2800,36'h02DFD37D2,36'h02DFD57A6,36'h02DFD777B,
 36'h000FA6FFF,36'h003FA6FA6,36'h006FA6F4C,36'h009FA7EF3,36'h00BFA8E9A,36'h00EFA9E42,36'h011FAADEC,36'h013FABD96,36'h016FADD42,36'h018FAECEF,36'h01BFB0C9E,36'h01DFB2C4E,36'h01EFB4C01,36'h020FB6BB5,36'h022FB8B6B,36'h023FBAB23,
 36'h025FBCADD,36'h026FBEA99,36'h027FC0A57,36'h028FC2A17,36'h029FC49D9,36'h029FC699D,36'h02AFC7963,36'h02AFC992B,36'h02BFCB8F5,36'h02BFCD8C0,36'h02BFCF88E,36'h02CFD085D,36'h02CFD282D,36'h02CFD3800,36'h02CFD57D3,36'h02CFD67A9,
 36'h000FA9FFF,36'h002FA9FA9,36'h005FA9F52,36'h008FAAEFC,36'h00BFABEA6,36'h00DFACE51,36'h010FADDFD,36'h012FAEDAA,36'h015FAFD58,36'h017FB1D08,36'h019FB2CB9,36'h01BFB4C6B,36'h01DFB6C1F,36'h01EFB7BD5,36'h020FB9B8D,36'h021FBBB47,
 36'h023FBDB02,36'h024FBFABF,36'h025FC1A7E,36'h026FC3A3F,36'h027FC4A02,36'h028FC69C7,36'h028FC898E,36'h029FCA956,36'h029FCB920,36'h02AFCD8EC,36'h02AFCF8BA,36'h02AFD0889,36'h02AFD285A,36'h02AFD382C,36'h02AFD5800,36'h02AFD67D5,
 36'h000FACFFF,36'h002FACFAC,36'h005FACF58,36'h007FADF04,36'h00AFADEB1,36'h00CFAEE5F,36'h00FFAFE0D,36'h011FB0DBD,36'h013FB1D6D,36'h015FB3D1F,36'h018FB4CD2,36'h019FB6C86,36'h01BFB7C3D,36'h01DFB9BF4,36'h01EFBBBAD,36'h020FBCB68,
 36'h021FBEB25,36'h022FC0AE4,36'h023FC2AA4,36'h024FC3A66,36'h025FC5A2A,36'h026FC79EF,36'h026FC99B6,36'h027FCA97F,36'h028FCC94A,36'h028FCD916,36'h028FCF8E4,36'h029FD18B3,36'h029FD2884,36'h029FD3857,36'h029FD582A,36'h029FD6800
  };
  fptx qxy;
  intx lut,lutdx,lutdy,b,c,d;
  assign qxy = atan2cf_pre(y,x);
  wire[15:0] qx = qxy[15:0];
  wire[15:0] qy = qxy[31:16];
  wire[9:0] luti = {qy[15:11],qx[15:11]};
  wire [35:0] lutad = luta[luti];
  logic[10:0] dx,dy;
  logic quadx,quady;
  assign b = lut + idu11(lutdx,dx) + idu11(lutdy,dy);
  assign c = {b[15],quadx^b[14],b[13:0]};
  always @(posedge clk) begin
    lut   <= {2'h0,lutad[11:0], 2'h0};
    lutdx <= {2'h3,lutad[23:12],2'h0};
    lutdy <= {2'h0,lutad[35:24],2'h0};
    quadx <= qx[0];
    quady <= qy[0];
    dx <= qx[10:0];
    dy <= qy[10:0];
//    if (xn) d <= quady? -c : c;	// accurate
    if (xn) d <= quady? ~c : c;		// faster
  end
  assign a = {{16{d[15]}},d};
endmodule

module CIB2CFA_task #(N=1,NA=N) (input clk, input rst, input en, input xn, input [N*32-1:0] a, output CxFptx b[NA]);
genvar i;
generate
  for (i=0; i<N; i=i+1) always @(posedge clk) if (xn) b[i] <= CI2CF(a[31+i*32:i*32]);
endgenerate
endmodule

module CFA2CIB_task #(N=1,NA=N) (input clk, input rst, input en, input xn, input CxFptx a[NA], output logic [N*32-1:0] b);
genvar i;
generate
  for (i=0; i<N; i=i+1) always @(posedge clk) if (xn) b[31+i*32:i*32] <= CF2CI(a[i]);
endgenerate
endmodule

module CIB2CFx_task #(N=1) (input clk, input rst, input en, input xn, input [N*32-1:0] a, output logic [N*64-1:0] b);
genvar i;
generate
  for (i=0; i<N; i=i+1) always @(posedge clk) if (xn) b[63+i*64:i*64] <= CI2CF(a[31+i*32:i*32]);
endgenerate
endmodule

module CFx2CIB_task #(N=1) (input clk, input rst, input en, input xn, input [N*64-1:0] a, output logic [N*32-1:0] b);
genvar i;
generate
  for (i=0; i<N; i=i+1) always @(posedge clk) if (xn) b[31+i*32:i*32] <= CF2CI(a[63+i*64:i*64]);
endgenerate
endmodule

module CFA2CBB_task #(N=1,NA=N) (input clk, input rst, input en, input xn, input CxFptx a[NA], output logic [N*16-1:0] b);
genvar i;
generate
  for (i=0; i<N; i=i+1) always @(posedge clk) if (xn) b[15+i*16:i*16] <= CF2CB(a[i]);
endgenerate
endmodule

/* verilator lint_on MULTITOP */
