/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  Various I/O Buffers

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

  IOBDELAY = IBUF buffer not in IOB
             IFD  ff in IOB
             BOTH either of the above

*/
`include "mdefs.h"
`ifdef VIRTEX2
`define ISTD #(.IOSTANDARD("LVDSEXT_25_DT"))
`define OSTD #(.IOSTANDARD("LVDSEXT_25"))
`elsif SPARTAN6
`define ISTD #(.IOSTANDARD("LVDS_25"),.DIFF_TERM("TRUE"))
`define OSTD #(.IOSTANDARD("LVDS_25"))
`elsif VIRTEX7
`define ISTD #(.IOSTANDARD("LVDS_25"),.DIFF_TERM("TRUE"),.IBUF_LOW_PWR("FALSE"))
`define OSTD #(.IOSTANDARD("LVDS_25"),.SLEW("FAST"))
`elsif VIRTEX8
`define ISTD #(.IOSTANDARD("LVDS"),.DIFF_TERM("TRUE"),.IBUF_LOW_PWR("FALSE"))
`define OSTD #(.IOSTANDARD("LVDS"),.SLEW("MEDIUM"))
`else
`define ISTD #(.IOSTANDARD("LVDSEXT_25"),.DIFF_TERM("TRUE"))
`define OSTD #(.IOSTANDARD("LVDSEXT_25"))
`endif
`define SISTD #(.IOSTANDARD("LVCMOS25"))
`define OSTDB #(.IOSTANDARD("BLVDS_25"),.SLEW("FAST"))

module buffers();

endmodule

module ibff (pin,clk,idat);
  output idat;
  input pin,clk;
  FD f0 (.D(pin),.C(clk),.Q(idat)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
endmodule

module ibffN (pin,clk,idat);
  parameter N=1;
  output [N-1:0] idat;
  input [N-1:0] pin;
  input clk;
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    ibff inst(pin[i],clk,idat[i]);
  end
  endgenerate
endmodule


module ibffd (pin,clk,idat);
  output idat;
  input pin,clk;
  FD f0 (.D(pin),.C(clk),.Q(idat)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
  //synthesis attribute IOBDELAY of f0 is "BOTH"
endmodule

module ibffdN (pin,clk,idat);
  parameter N=1;
  output [N-1:0] idat;
  input [N-1:0] pin;
  input clk;
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    ibffd inst(pin[i],clk,idat[i]);
  end
  endgenerate
endmodule


module sibuf (pin,idat);
  output idat;
  input pin;
  IBUF b0 (.I(pin),.O(idat));
endmodule

module ibufd (pin,idat);
  output idat;
  input pin;
  IBUF b0 (.I(pin),.O(idat));
  //synthesis attribute IOBDELAY of b0 is "IBUF"
endmodule

module ibufdN (pin,idat);
  parameter N=1;
  output [N-1:0] idat;
  input [N-1:0] pin;
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    ibufd inst(pin[i],idat[i]);
  end
  endgenerate
endmodule


module sobuf (pout,odat);
  output pout;
  input odat;
  OBUF b0 (.O(pout),.I(odat));
endmodule

module obff (pout,clk,odat);
  output pout;
  input clk,odat;
  wire ff;
  FD f0 (.D(odat),.C(clk),.Q(ff)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOB of f0 is "TRUE"
  OBUF b0 (.I(ff), .O(pout) );
endmodule

module obffN (pout,clk,odat);
  parameter N=1;
  output [N-1:0] pout;
  input [N-1:0] odat;
  input clk;
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    obff inst (pout[i],clk,odat[i]);
  end
  endgenerate
endmodule

module obfft (pout,clk,oe,odat);
  output pout;
  input clk,oe,odat;
  wire ff;
  FD f0 (.D(odat),.C(clk),.Q(ff)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
  OBUFT b0 (.I(ff),.T(~oe),.O(pout));
endmodule

module obfftN (pout,clk,oe,odat);
  parameter N=1;
  output [N-1:0] pout;
  input [N-1:0] odat;
  input clk,oe;
  genvar i;
  generate
  for (i=0; i<N; i=i+1) begin:blk
    obfft inst (pout[i],clk,oe,odat[i]);
  end
  endgenerate
endmodule

// avoid name conflict with IDDR
module iddr_ (pin, clk, idatp, idatn);
  input clk;
  input pin;
  output idatp, idatn;
  FD bf0p (.D(pin),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */; 
  FD bf0n (.D(pin),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOB of bf0p is "TRUE"
  //synthesis attribute IOB of bf0n is "TRUE"
endmodule

module iddrc (pin, idat, clk, idatp, idatn);
  input clk;
  input pin;
  output idat, idatp, idatn;
  IBUF b0 (.I(pin),.O(idat));
  FD bf0p (.D(idat),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */; 
  FD bf0n (.D(idat),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOB of bf0p is "TRUE"
  //synthesis attribute IOB of bf0n is "TRUE"
endmodule

module diddrc (pin_p, pin_n, idat, clk, idatp, idatn);
  input clk;
  input pin_p, pin_n;
  output idat, idatp, idatn;
  IBUFDS b0 (.O(idat),.I(pin_p),.IB(pin_n));
`ifdef SPARTAN6
  IDDR2 #(.DDR_ALIGNMENT("NONE"),.SRTYPE("SYNC")) brff (.D(idat),.C0(clk),.C1(~clk),.CE(1'b1),.Q0(idatp),.Q1(idatn));
`else
  FD bf0p (.D(idat),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */; 
  FD bf0n (.D(idat),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOB of bf0p is "TRUE"
  //synthesis attribute IOB of bf0n is "TRUE"
`endif
endmodule

module iddrd (pin, clk, idatp, idatn);
  input clk;
  input pin;
  output idatp, idatn;
  FD bf0p (.D(pin),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */; 
  FD bf0n (.D(pin),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */; 
  //synthesis attribute IOBDELAY of bf0p is "BOTH"
  //synthesis attribute IOBDELAY of bf0n is "BOTH"
  //synthesis attribute IOB of bf0p is "TRUE"
  //synthesis attribute IOB of bf0n is "TRUE"
endmodule

module iddrN (pin, clk, idatp,idatn);
  parameter N=1;
  input [N-1:0] pin;
  input clk;
  output [N-1:0] idatp,idatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    iddr_ inst (pin[i], clk, idatp[i],idatn[i]);
   end
  endgenerate
endmodule

module obuftN (pout, ts, odat);
  parameter N=1;
  output [N-1:0] pout;
  input ts;
  input [N-1:0] odat;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    OBUFT inst (.I(odat[i]),.T(ts),.O(pout[i]));
   end
  endgenerate
endmodule

module iobuftN (pin, ts, idat,odat);
  parameter N=1;
  output [N-1:0] pin;
  input ts;
  output [N-1:0] idat;
  input [N-1:0] odat;
`ifdef ALTERA
   altera_gpio #(
           .SIZE                (N),
           .PIN_TYPE            ("bidir"),
           .REGISTER_MODE       ("none"),
           .HALF_RATE           ("false"),
           .DDIO_WITH_DELAY     ("false"),
           .SEPARATE_I_O_CLOCKS ("false"),
           .BUFFER_TYPE         ("single-ended"),
           .PSEUDO_DIFF         ("false"),
           .ARESET_MODE         ("none"),
           .SRESET_MODE         ("none"),
           .OPEN_DRAIN          ("false"),
           .BUS_HOLD            ("false"),
           .ENABLE_OE           ("false"),
           .ENABLE_CKE          ("false"),
           .ENABLE_TERM         ("false")
   ) core (
           .dout               (odat),
           .din                (idat),
           .oe                 (~ts), 
           .pad_io             (pin), 
           .cke                (1'b1),
           .ck_fr_in           (1'b0),
           .ck_fr_out          (1'b0),
           .ck_in              (1'b0),
           .ck_out             (1'b0),
           .ck_fr              (1'b0),
           .ck                 (1'b0),
           .ck_hr_in           (1'b0),
           .ck_hr_out          (1'b0),
           .ck_hr              (1'b0),
           .pad_io_b           (),    
           .pad_in             (4'b0000),
           .pad_in_b           (4'b0000),
           .pad_out            (),       
           .pad_out_b          (),       
           .terminationcontrol (1'b0),   
           .aclr               (1'b0),   
           .aset               (1'b0),   
           .sclr               (1'b0),   
           .sset               (1'b0)    
   );
`else	
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    IBUF  insti (.I(pin[i]),.O(idat[i]));
    OBUFT insto (.I(odat[i]),.T(ts),.O(pin[i]));
   end
  endgenerate
`endif  
endmodule

module iobuftrN (pin, ts, idat,odat, clk,qdat);
  parameter N=1;
  output [N-1:0] pin;
  input ts,clk;
  output [N-1:0] idat;
  output [N-1:0] qdat;
  input [N-1:0] odat;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    IBUF  insti (.I(pin[i]),.O(idat[i]));
    FD    instf (.D(idat[i]),.C(clk),.Q(qdat[i])) /* synthesis syn_useioff=1 */; 
    //synthesis attribute IOB of instf is "TRUE"
    OBUFT insto (.I(odat[i]),.T(ts),.O(pin[i]));
   end
  endgenerate
endmodule

module oddr (pout, clk, odatp,odatn);
  parameter PIPE=0;
  input clk;
  output pout;
  input odatp, odatn;

  wire H=1, L=0;
  wire ff;
`ifdef VIRTEX8
  ODDRE1 f0 (.Q(ff),.C(clk),.D1(odatp),.D2(odatn),.SR(L));
`elsif SPARTAN6
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatn),.R(L),.S(L));
`else
  FDDRRSE f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatn),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
`endif
  OBUF b0 (.I(ff), .O(pout) );

endmodule

module oddrN (pout, clk, odatp,odatn);
  parameter N=1;
  output [N-1:0] pout;
  input clk;
  input [N-1:0] odatp,odatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    oddr inst (pout[i], clk, odatp[i],odatn[i]);
   end
  endgenerate
endmodule


module oddrt (pout, clk,oe, odatp,odatn);
  parameter PIPE=0;
  input clk,oe;
  output pout;
  input odatp, odatn;

  wire H=1, L=0;
  wire ff;
`ifdef VIRTEX8
  ODDRE1 f0 (.Q(ff),.C(clk),.D1(odatp),.D2(odatn),.SR(L));
`elsif SPARTAN6
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatn),.R(L),.S(L));
`else
  FDDRRSE f0 (.Q(ff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatn),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
`endif
  OBUFT b0 (.I(ff), .T(~oe), .O(pout) );

endmodule

module oddrtN (pout, clk,oe, odatp,odatn);
  parameter N=1;
  inout [N-1:0] pout;
  input oe,clk;
  input [N-1:0] odatp,odatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    oddrt inst (pout[i], clk,oe, odatp[i],odatn[i]);
   end
  endgenerate
endmodule


// clocked single data rate output, double data rate input buffer
module ioddr (pio, iclk, idatp,idatn, oclk,oe, odatp,odatn);
  parameter PIPE=0;
  inout pio;
  input iclk,oe,oclk;
  output idatp,idatn;
  input odatp,odatn;

  wire H=1, L=0;
  wire rff,wff,wts;

  IBUF idq (.I(pio),.O(rff));
`ifdef VIRTEX2
  wire idatpb,idatnb,odatnb;
  reg idatpc,idatnc;
  reg idatpd,idatnd,odatnd;
  FD brffp (.D(rff),.C(iclk),.Q(idatpb)) /* synthesis syn_useioff=1 */;
  FD brffn (.D(rff),.C(~iclk),.Q(idatnb)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of brffp is "TRUE"
  //synthesis attribute IOB of brffn is "TRUE"
  always @(posedge iclk) idatpc <= idatpb;
  always @(posedge iclk) idatpd <= idatpc;
  always @(negedge iclk) idatnc <= idatnb;
  always @(posedge iclk) idatnd <= idatnc;
  assign idatp = PIPE? idatpd : idatpb;
  assign idatn = PIPE? idatnd : idatnb;
  always @(negedge iclk) odatnd <= odatn;
  assign odatnb = PIPE? odatnd : odatn;
  FDDRRSE bwff (.Q(wff),.C0(oclk),.C1(~oclk),.CE(H),.D0(odatp),.D1(odatnb),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of bwff is "TRUE"
  OBUFT odq (.I(wff),.T(~oe),.O(pio));
`elsif SPARTAN6
  IDDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) brff (.D(rff),.C0(iclk),.C1(~iclk),.Q0(idatp),.Q1(idatn));
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) bwff (.Q(wff),.C0(oclk),.C1(~oclk),.D0(odatp),.D1(odatn),.R(L),.S(L));
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) bwts (.Q(wts),.C0(oclk),.C1(~oclk),.D0(~oe),.D1(~oe),.R(L),.S(L));
  OBUFT odq (.I(wff),.T(wts),.O(pio));
`elsif VIRTEX8
  IDDRE1 #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE_PIPELINED":"OPPOSITE_EDGE")) brff (.D(rff),.C(iclk),.CB(~iclk),.Q1(idatp),.Q2(idatn));
  ODDRE1 bwff (.Q(wff),.C(oclk),.D1(odatp),.D2(odatn),.SR(L));
  OBUFT odq (.I(wff),.T(~oe),.O(pio));
`else
  IDDR #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE_PIPELINED":"OPPOSITE_EDGE")) brff (.D(rff),.C(iclk),.Q1(idatp),.Q2(idatn));
  ODDR #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE":"OPPOSITE_EDGE")) bwff (.Q(wff),.C(oclk),.D1(odatp),.D2(odatn));
  OBUFT odq (.I(wff),.T(~oe),.O(pio));
`endif

endmodule

module ioddrN (pio, iclk, idatp,idatn, oclk,oe, odatp,odatn);
  parameter N=1;
  parameter PIPE=0;
  inout [N-1:0] pio;
  input iclk,oe,oclk;
  output [N-1:0] idatp,idatn;
  input [N-1:0] odatp,odatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    ioddr #(PIPE) inst (pio[i], iclk, idatp[i],idatn[i], oclk,oe, odatp[i],odatn[i]);
   end
  endgenerate
endmodule


// clocked DDR differential I/O buffer
module dioddr (pio_p,pio_n, clk,oe, idatp,idatn, odatp,odatn);
  inout pio_p,pio_n;
  input oe,clk;
  output idatp,idatn;
  input odatp,odatn;

  wire H=1, L=0;
  wire rff,wff;

  FDDRRSE bwff (.Q(wff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatn),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  IOBUFDS `ISTD dq (.I(wff),.O(rff),.T(~oe),.IO(pio_p),.IOB(pio_n));
  FD brffp (.D(rff),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */;
  FD brffn (.D(rff),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of bwff is "TRUE"
  //synthesis attribute IOB of brffp is "TRUE"
  //synthesis attribute IOB of brffn is "TRUE"

endmodule

module dioddrN (pio_p,pio_n, clk,oe, idatp,idatn, odatp,odatn);
  parameter N=1;
  inout [N-1:0] pio_p,pio_n;
  input oe,clk;
  output [N-1:0] idatp,idatn;
  input [N-1:0] odatp,odatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    dioddr inst (pio_p[i],pio_n[i], clk,oe, idatp[i],idatn[i], odatp[i],odatn[i]);
   end
  endgenerate
endmodule

module dobuf (pio_p,pio_n, oe,odat);
  inout pio_p,pio_n;
  input oe,odat;
  OBUFTDS `OSTD dq (.I(odat),.T(!oe),.O(pio_p),.OB(pio_n));
endmodule

module dibuf (pio_p,pio_n, idat);
  input pio_p,pio_n;
  output idat;
  IBUFDS `ISTD iob (idat,pio_p,pio_n);
endmodule

module dibff (pio_p,pio_n, clk, idat);
  input pio_p,pio_n;
  input clk;
  output idat;
  wire pin;
  IBUFDS `ISTD iob (pin,pio_p,pio_n);
  FD f0 (.D(pin),.C(clk),.Q(idat)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of f0 is "TRUE"
endmodule

module dibufgh (pio_p,pio_n, idat);
  input pio_p,pio_n;
  output idat;
  wire idata,idatb;
  IBUFDS `ISTD iob (idata,pio_p,pio_n);
//  BUFIO2 bio (.I(idata),.DIVCLK(idatb));
  BUFH bufh (idatb,idata);
  reg idatc; always @(posedge idatb) idatc <= !idatc;
  BUFG bufg (idat,idatc);
endmodule

module diobuf (pio_p,pio_n, oe, idat,odat);
  inout pio_p,pio_n;
  input oe,odat;
  output idat;
  IOBUFDS `ISTD dq (.I(odat),.O(idat),.T(~oe),.IO(pio_p),.IOB(pio_n));
endmodule

// clocked DDR differential input buffer
module diddr (pio_p,pio_n, clk, idatp,idatn);
  parameter PIPE=0;
  parameter DELAY=0;
  parameter INVERT=0;
  input pio_p,pio_n;
  input clk;
  output idatp,idatn;

  wire H=1, L=0;
  wire rffi,rff;
  wire inv = (INVERT!=0);

  IBUFDS `ISTD dq (.O(rffi),.I(pio_p),.IB(pio_n));

  generate
  if (DELAY>0) begin:wd
    IDELAY dly (.I(rffi),.O(rff));
    defparam dly.IOBDELAY_TYPE = "FIXED";
    defparam dly.IOBDELAY_VALUE = DELAY-1;
  end
  else begin:wo
    assign rff = rffi;
  end
  endgenerate

  wire idatpb,idatnb;
`ifdef VIRTEX2
  reg idatpc,idatnc;
  reg idatpd,idatnd;
  FD brffp (.D(rff),.C(clk),.Q(idatpb)) /* synthesis syn_useioff=1 */;
  FD brffn (.D(rff),.C(~clk),.Q(idatnb)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of brffp is "TRUE"
  //synthesis attribute IOB of brffn is "TRUE"
  always @(posedge clk) idatpc <= idatpb^inv;
  always @(posedge clk) idatpd <= idatpc^inv;
  always @(negedge clk) idatnc <= idatnb;
  always @(posedge clk) idatnd <= idatnc;
`elsif SPARTAN6
  IDDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) brff (.D(rff),.C0(clk),.C1(~clk),.CE(H),.R(L),.Q0(idatpb),.Q1(idatnb));
`elsif VIRTEX8
  IDDRE1 #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE_PIPELINED":"OPPOSITE_EDGE")) brff (.D(rff),.C(clk),.CB(~clk),.Q1(idatpb),.Q2(idatnb));
`else
  IDDR #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE_PIPELINED":"OPPOSITE_EDGE")) brff (.D(rff),.C(clk),.CE(H),.Q1(idatpb),.Q2(idatnb));
`endif
`ifdef VIRTEX2
  assign idatp = PIPE? idatpd : idatpb;
  assign idatn = PIPE? idatnd : idatnb;
`else
  assign idatp = idatpb^inv;
  assign idatn = idatnb^inv;
`endif
endmodule

module diddrN (pio_p,pio_n, clk, idatp,idatn);
  parameter N=1;
  parameter PIPE=0;
  parameter DELAY=0;
  parameter INVERT=0;
  input [N-1:0] pio_p,pio_n;
  input clk;
  output [N-1:0] idatp,idatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    diddr #(PIPE,DELAY,((INVERT>>i)&1)) inst (pio_p[i],pio_n[i], clk, idatp[i],idatn[i]);
   end
  endgenerate
endmodule

// clocked DDR differential input buffer with source and serdes with feedback - for 1PPS sampling
module diddrws (pio_p,pio_n,pio_fb, idat, clk,idatp, pclk,qclk,qrst, qdat,rdat);
  input pio_p,pio_n;
  inout pio_fb;
  input clk,qrst,pclk,qclk;
  output idatp,idat;
  output [3:0] qdat,rdat;

  wire H=1, L=0;
  wire pio,piofb;

  IBUFDS `ISTD dq (.O(pio),.I(pio_p),.IB(pio_n));
`ifdef SPARTAN6
  FD f0 (.D(pio),.C(clk),.Q(idatp));
  //synthesis attribute IOB of f0 is "FALSE"
  ISERDES2 #(.DATA_RATE("SDR"),.DATA_WIDTH("4"),.INTERFACE_TYPE("RETIMED")) 
	qser (.D(pio),.CLKDIV(qclk),.CLK0(pclk),.RST(qrst), .Q1(qdat[0]),.Q2(qdat[1]),.Q3(qdat[2]),.Q4(qdat[3]));
  OBUFT b1 (.I(idatp),.T(L),.O(pio_fb));
  IBUF  b2 (.I(pio_fb),.O(piofb));
  ISERDES2 #(.DATA_RATE("SDR"),.DATA_WIDTH("4"),.INTERFACE_TYPE("RETIMED")) 
	rser (.D(piofb),.CLKDIV(qclk),.CLK0(pclk),.RST(qrst), .Q1(rdat[0]),.Q2(rdat[1]),.Q3(rdat[2]),.Q4(rdat[3]));
`else
  IDDR #(.DDR_CLK_EDGE("OPPOSITE_EDGE")) brff (.D(pio),.C(clk),.CE(H),.Q1(qdat[0]),.Q2(qdat[1]));
`endif
 assign idat = pio;

endmodule



// QDR for hypertransport ports
 
module diqdrc (pio_p,pio_n, qclk,qrst,pclk,rrst,ss, vadj,test);
  parameter SPEED=400;	// in MHz
  parameter signed DELAY=0;
  parameter OPTION=0;
  localparam PERIOD=1000.0/SPEED;
  input pio_p,pio_n;
  input qrst;		// reset for the downstream ISERDES
  input [7:0] vadj;
  output qclk,pclk,rrst;
  output [7:0] ss;
  output [7:0] test;

  wire L=0;
  wire pio,piod,pios,piot,piob;
  IBUFDS `ISTD iob (pio,pio_p,pio_n);

  wire sclk = vadj[0];
  wire srst = vadj[1];	// reset for the clock
  wire sadj = vadj[2];
  wire supd = vadj[3];
  wire scyc = vadj[4];	// cycle on ultrascale - track on spartan
  wire rclk = vadj[5];	// fifo reset clock
  wire good = vadj[6];	// clock OK
  wire csel = vadj[7];	// clock select

  // adjustment circuit
  wire dclk,vclk,vclk4,vclk1,vfbi,vfbo,vlock,lock,ssr,cdat;
  reg prst,scalw,scal,slock,spha,squpd; 
  reg [15:0] scnt;
  always @(posedge sclk) begin
    if (srst) scnt <= 0; else scnt <= scnt+1;
//    if (srst) scnt <= 0; else if (!scnt[15]) scnt <= scnt+1;
`ifdef SPARTAN6
    if (srst) scalw <= 1; else scalw <= scalw && !scnt[7];
`else
    if (srst) scalw <= 1; else scalw <= scalw && !scnt[4];
`endif
    scal <= srst;
    // scal <= scalw && scnt[5]; // dn32, up32, dn - note that DATAOUTs are junk during calibration cycles
    prst <= (!vlock && (scnt[11:4]==0)) || scalw;	// internal reset if input PLL clock becomes unlocked at startup
    slock <= !(qrst || srst || sadj);
    if (srst) spha <= 0; else if (sadj) spha <= !spha;
    if (sadj) squpd <= supd;
  end

  // reset circuit
`ifdef VIRTEX8
  wire xclk = rclk;
  delaypipe #(16,1) rdp (rclk,qrst|prst,rrst);
`else
  wire xclk = qclk;
  delaypipe #(64,1) rdp (qclk,qrst|prst,rrst);
`endif

  // retimed adjustment circuit
  reg [4:0] qcnt;
  reg qrst1,qadj,qupd,qpha,qcyc,qcal;
  always @(posedge xclk) begin
    qrst1 <= srst;
    qcal  <= scal;
    if (qrst1) qpha <= 0; else if (qcnt==4) qpha <= !qpha;
    qcyc <= (spha^qpha) || (qcyc && qcnt!=31);
    if (!qcyc) qcnt <= 0; else qcnt <= qcnt+1;
`ifdef VIRTEX8
    qupd <= squpd;
    qadj <= (qcnt==14 || qcnt==16);
`else
    qupd <= (qcnt==15) && squpd;
    qadj <= (qcnt==15);
`endif
  end

`ifdef VIRTEX2
  wire lclk1,lclk2;
  DCM dcm (.PSCLK(sclk),.PSEN(sadj),.PSINCDEC(supd),.RST(srst),.LOCKED(vlock),.CLKIN(pio),.CLKFB(pclk),.CLK0(lclk2),.CLKDV(lclk1));
  defparam dcm.CLK_FEEDBACK = "1X";
  defparam dcm.DLL_FREQUENCY_MODE = "HIGH";
  defparam dcm.DUTY_CYCLE_CORRECTION = "TRUE";
  defparam dcm.CLKIN_PERIOD = 7.5;
  defparam dcm.CLKOUT_PHASE_SHIFT = "VARIABLE";
  defparam dcm.PHASE_SHIFT = 0;
  defparam dcm.STARTUP_WAIT = "FALSE";
  BUFG bufr (qclk,lclk1);
  BUFG bufs (pclk,lclk2);
  assign ss = 1'b0;
  assign piob = pio;

`elsif SPARTAN6
  assign ss[7:1] = {csel,scyc,qcal,qupd,qadj,qrst1,qclk};
  IODELAY2 dly (.IDATAIN(pio),.DATAOUT(piod),.DATAOUT2(piot));
  defparam dly.DELAY_SRC = "IDATAIN";
  defparam dly.IDELAY_TYPE = "FIXED";
  defparam dly.IDELAY_VALUE = 0;
generate
if (OPTION>0) begin
  wire bio = (OPTION==2)? piod : pio;
  BUFIO2 #(.DIVIDE(1),.DIVIDE_BYPASS("TRUE")) bufd (.I(bio),.IOCLK(),.DIVCLK(dclk),.SERDESSTROBE());
end else begin
  BUFG     bufd(.O(dclk),.I(piot)); 
end
endgenerate
  BUFG     bfb (.O(vfbi),.I(vfbo)); 
  PLL_BASE pll (.CLKIN(dclk),.RST(prst),.CLKOUT0(vclk4),.CLKOUT1(vclk1),.CLKFBIN(vfbi),.CLKFBOUT(vfbo),.LOCKED(vlock));
  defparam pll.CLK_FEEDBACK   = "CLKFBOUT";
  defparam pll.CLKIN_PERIOD   = PERIOD;
  defparam pll.COMPENSATION   = "SOURCE_SYNCHRONOUS";
  defparam pll.DIVCLK_DIVIDE  = 1;
  defparam pll.CLKFBOUT_MULT  = 2;
  defparam pll.CLKFBOUT_PHASE = DELAY*1.0;
  defparam pll.REF_JITTER     = 0.1;
  defparam pll.BANDWIDTH      = "HIGH";
  defparam pll.CLKOUT0_DIVIDE = 1;
  defparam pll.CLKOUT1_DIVIDE = 4;
  BUFPLL   puf (.PLLIN(vclk4),.LOCKED(vlock),.GCLK(qclk),.IOCLK(pclk),.LOCK(lock),.SERDESSTROBE(ss[0]));
  BUFG     bufr (.O(qclk),.I(vclk1));
  assign piob = L;

`elsif VIRTEX7 
  IDELAYE2 dly (.C(sclk),.CE(sadj),.INC(supd),.LD(srst), .IDATAIN(pio),.DATAOUT(piod) );
  defparam dly.HIGH_PERFORMANCE_MODE = "TRUE";
  defparam dly.IDELAY_TYPE = "VARIABLE";
  defparam dly.IDELAY_VALUE = 0;
  BUFR  bufr (.O(qclk),.I(piod),.CLR(1'b0),.CE(1'b1));
  defparam bufr.BUFR_DIVIDE = "2";
  BUFIO bufs (.O(pclk),.I(piod));
  assign ss = 1'b0;
  assign vlock = slock;
  assign piob = piod;

`elsif VIRTEX8
  wire x_clk0,x_clk1,sena,fempty,femptyd;
  PLLE3_ADV x_dcm (.RST(srst),.PWRDWN(1'b0),.LOCKED(vlock),.CLKIN(pio),.CLKOUT0(x_clk0),.CLKOUT1(x_clk1));
  defparam x_dcm.CLKIN_PERIOD = 1000.0/SPEED;
  defparam x_dcm.REF_JITTER = 0.5;
  defparam x_dcm.CLKFBOUT_MULT = (SPEED>=500)? 2:3;
  defparam x_dcm.CLKOUT0_DIVIDE = (SPEED>=500)? 2:3;
  defparam x_dcm.CLKOUT1_DIVIDE = (SPEED>=500)? 2:3;
  defparam x_dcm.CLKOUT1_PHASE = (SPEED>=500)? 180.0:120.0;
  delaypipe #(5,1) ssb (rclk,!rrst,sena);	// disable 2x clock until N=5 clocks after reset is lifted
  BUFGCTRL bufs (.O(pclk),.I0(x_clk0),.I1(x_clk1),.S0(!csel),.S1(csel),.CE0(sena),.CE1(sena));
  IDELAYE3 dly (.IDATAIN(pio),.DATAOUT(piod));
  ISERDESE3 #(.DATA_WIDTH(4),.FIFO_ENABLE("TRUE"),.FIFO_SYNC_MODE("TRUE")) 
    ser (.D(piod),.CLKDIV(rclk),.CLK(pclk),.CLK_B(~pclk),.RST(rrst), .FIFO_RD_CLK(rclk),.FIFO_RD_EN(ssr),.FIFO_EMPTY(fempty));
  delaypipe #(1,1) ssd (rclk,fempty,femptyd);
  delaypipe #(1,1) sse (rclk,!femptyd,ssr);	// all should synchronize on this
  assign ss = {qcyc|qrst1,qupd,qadj,qrst1,rclk,ssr};
  assign piob = piod;

`else
  IDELAY dly (.C(sclk),.CE(sadj),.INC(supd),.RST(srst), .I(pio),.O(piod) );
  defparam dly.IOBDELAY_TYPE = "VARIABLE";
  defparam dly.IOBDELAY_VALUE = 0;
  BUFR  bufr (.O(qclk),.I(pclk));
  defparam bufr.BUFR_DIVIDE = "2";
  BUFIO bufs (.O(pclk),.I(piod));
  assign ss[0] = 1'b0;
  assign vlock = slock;
  assign piob = piod;
`endif

  // the PIOB in test[0] is needed by IOMODULE.v
  reg jjj; always @(posedge qclk) jjj <= !jjj;
  assign test = {srst,sadj,supd,scal,prst,srst|vlock,jjj,piob};

endmodule

module diqdr (pio_p,pio_n, qclk,pclk,rrst,sync, qdat0,qdat1,qdat2,qdat3
`ifdef QDR_TRACK
  ,track
`endif
);
  parameter PIPE=0;
  parameter INVERT=0;
  parameter DELAY=0;
  localparam DEPTH=4;
  localparam IOBD=(DELAY==0)? "NONE" : "IFD";
  input pio_p,pio_n;
  input qclk,pclk,rrst;
  input [7:0] sync;
  output qdat0,qdat1,qdat2,qdat3;
`ifdef QDR_TRACK
  output [8:0] track;
`else
  wire [8:0] track;
`endif

  wire ss = sync[0];	// bit-slip for V4 and S6
  wire pio,piod,pioc;
  wire [3:0] cdat,sdat;
  wire inv = (INVERT!=0);
  wire ssd; delayp #(1) ssdly (qclk,ss,ssd);

`ifdef DTDM
  IBUFDS #(.IOSTANDARD("LVDSEXT_25")) dq (.O(pio),.I(pio_p),.IB(pio_n));
`else
  IBUFDS `ISTD dq (.O(pio),.I(pio_p),.IB(pio_n));
`endif
`ifdef QDR_TRACK
  assign track[8] = pio;
`endif

`ifdef VIRTEX2
  wire pdat0,pdat1;
  FD brffp (.D(pio),.C(pclk),.Q(pdat0)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of brffp is "TRUE"
  FD brffn (.D(pio),.C(~pclk),.Q(pdat1)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of brffn is "TRUE"
  reg rdat0; always @(posedge pclk) rdat0 <= pdat0;
  reg rdat1; always @(negedge pclk) rdat1 <= pdat1;
  reg [3:0] sdats; always @(posedge pclk) sdats <= {rdat1,rdat0,sdats[3:2]};
  assign sdat=sdats;
`elsif VIRTEX4 
  // IDELAY is in the SERDES
  ISERDES #(.DATA_RATE("DDR"),.DATA_WIDTH(DEPTH),.INTERFACE_TYPE("NETWORKING"),.IOBDELAY(IOBD),.BITSLIP_ENABLE("TRUE"))
	ser (.D(pio),.CLKDIV(qclk),.CLK(pclk),.SR(rrst), .BITSLIP(ss), .Q4(sdat[0]),.Q3(sdat[1]),.Q2(sdat[2]),.Q1(sdat[3]));
  defparam ser.IOBDELAY_TYPE = (DELAY==0)? "DEFAULT" : "FIXED";
  defparam ser.IOBDELAY_VALUE = DELAY;
`elsif SPARTAN6
  wire sclk,srst,sadj,sadjc,supd,scal;
  diqdrTrack (qclk,sdat,cdat, sync, sclk,srst,sadj,sadjc,supd,scal, track[7:0]);
  IODELAY2 #(.DELAY_SRC("IDATAIN"),.IDELAY_TYPE("VARIABLE_FROM_ZERO")) 
	dly (.CLK(sclk),.CE(sadj),.INC(supd),.RST(srst), .CAL(scal), .IOCLK0(pclk), .IDATAIN(pio),.DATAOUT(piod));
  IODELAY2 #(.DELAY_SRC("IDATAIN"),.IDELAY_TYPE("VARIABLE_FROM_ZERO")) 
	dlyc (.CLK(sclk),.CE(sadjc),.INC(supd),.RST(srst), .CAL(scal), .IOCLK0(pclk), .IDATAIN(pio),.DATAOUT(pioc));
  ISERDES2 #(.DATA_RATE("SDR"),.DATA_WIDTH(DEPTH),.INTERFACE_TYPE("RETIMED")) 
	ser (.D(piod),.CLKDIV(qclk),.CLK0(pclk),.RST(rrst), .IOCE(ss), .Q1(sdat[0]),.Q2(sdat[1]),.Q3(sdat[2]),.Q4(sdat[3]));
  ISERDES2 #(.DATA_RATE("SDR"),.DATA_WIDTH(DEPTH),.INTERFACE_TYPE("RETIMED")) 
	serc (.D(pioc),.CLKDIV(qclk),.CLK0(pclk),.RST(rrst), .IOCE(ss), .Q1(cdat[0]),.Q2(cdat[1]),.Q3(cdat[2]),.Q4(cdat[3]));
`elsif VIRTEX6
  IDELAY #(.IOBDELAY_TYPE("FIXED"),.IOBDELAY_VALUE(DELAY)) 
	dly (.I(pio),.O(piod));
  ISERDESE1 #(.DATA_RATE("DDR"),.DATA_WIDTH(DEPTH),.INTERFACE_TYPE("NETWORKING"),.IOBDELAY(IOBD))  // note Q outputs are MSB ordered
	ser (.D(pio),.DDLY(piod),.CLKDIV(qclk),.CLK(pclk),.CLKB(~pclk),.RST(rrst), .Q4(sdat[0]),.Q3(sdat[1]),.Q2(sdat[2]),.Q1(sdat[3]));
`elsif VIRTEX7
  IDELAYE2 #(.HIGH_PERFORMANCE_MODE("TRUE"),.IDELAY_TYPE("FIXED"),.IDELAY_VALUE(DELAY))
	dly (.IDATAIN(pio),.DATAOUT(piod) );
  ISERDESE2 #(.DATA_RATE("DDR"),.DATA_WIDTH(DEPTH),.INTERFACE_TYPE("NETWORKING"),.IOBDELAY(IOBD))  // note Q outputs are MSB ordered
	ser (.D(pio),.DDLY(piod),.CLKDIV(qclk),.CLK(pclk),.CLKB(~pclk),.RST(rrst), .Q4(sdat[0]),.Q3(sdat[1]),.Q2(sdat[2]),.Q1(sdat[3]));
`elsif VIRTEX8
  wire sclk=sync[1]; wire srst=sync[2]; wire sadj=sync[3]; wire supd=sync[4]; wire scyc=sync[5];
  IDELAYE3 #(.DELAY_TYPE("VARIABLE"),.DELAY_FORMAT("COUNT"))
	dly (.CLK(sclk),.CE(sadj),.INC(supd),.RST(srst),.EN_VTC(!scyc), .IDATAIN(pio),.DATAOUT(piod));
  ISERDESE3 #(.DATA_WIDTH(DEPTH),.FIFO_ENABLE((PIPE==1)?"TRUE":"FALSE"),.FIFO_SYNC_MODE("TRUE"))  // note Q outputs are LSB ordered
	ser (.D(piod),.CLKDIV(qclk),.CLK(pclk),.CLK_B(~pclk),.RST(rrst), .Q(sdat), .FIFO_RD_CLK(qclk),.FIFO_RD_EN(ssd));
`endif

  ff #(4,PIPE) pr ({qdat3,qdat2,qdat1,qdat0},inv?~sdat:sdat,qclk);
endmodule

/*
  This algorithm walks a second delay block +-16 taps and checks to see if the test tap data matches the actual tap data
  If not, the left and right edges of the eye are compared and if <-2 or >2 the test and actual taps are moved +-1 tap

  The data match is only tested for QDR buffers with a transition (which the default comma/tst pattern insures).
  The sequencer seq is stepped every 256 cnt clocks with transitions
  The 256 state sequencer is as described below:

  seq[0]	0=set test tap  1=meas cycle		meas
  seq[4:1]	1-16 tap window offset			tap
  seq[6:5]	0->toUp 1->fromUp 2->toLo 3->fromLo	phase
  seq[7]	adjust actual tap cycle			mode
*/

module diqdrTrack (qclk,qdata,cdata, sync, sclk,srst,sadj,sadjc,supd,scal, status);
  input qclk;
  input [3:0] qdata,cdata;
  input [7:0] sync;
  output sclk;
  output reg srst,sadj,sadjc,supd,scal;
  output reg [7:0] status;

  wire L=0, H=1;
  assign sclk = sync[1];

  reg done,ena,enb,bad,okt,okd,lock,fixup,fixdn;
  reg [7:0] cnt,seq;
  reg [3:0] lstat,ustat;

  wire       meas  = seq[0];
  wire [3:0] tap   = seq[4:1];
  wire [1:0] phase = seq[6:5];
  wire       mode  = seq[7];
  wire       mid   = (cnt==8'h40);
  wire [3:0] max   = 4'hF;
  wire	     set   = !mode && meas && done && bad;

  always @(posedge qclk) begin
    ena <= sync[6];
    enb <= sync[7];
    okt <= (qdata!=0) && (qdata!=15);	// has transitions
    okd <= (qdata==cdata);		// matches center
    done <= (cnt==0 && okt);
    if (okt) cnt <= cnt+1;
    if (done) bad <= 0; else bad <= bad || (okt && !okd);
    if (!ena) seq <= 0; else if (done) seq <= seq+1;
    lock <= done && (seq==8'h81);
    if (lock) ustat <= max; else if (set&&(phase==0)&&(ustat==max)) ustat <= tap;
    if (lock) lstat <= max; else if (set&&(phase==2)&&(lstat==max)) lstat <= tap;
    if (lock) status <= {ustat,lstat};
    fixup <= enb && lock && (ustat>4) && ((ustat-2)>lstat);
    fixdn <= enb && lock && (lstat>4) && ((lstat-2)>ustat);
    srst  <= !ena? sync[2] : L;
    sadj  <= !ena? sync[3] : (fixup||fixdn);
    sadjc <= !ena? sync[3] : (!mode && !meas && mid && okt) || (fixup||fixdn);
    supd  <= !ena? sync[4] : mode? fixup : (phase==0||phase==3);
    scal  <= !ena? sync[5] : L;
  end

endmodule

module diqdrN (pio_p,pio_n, qclk,pclk,rrst,sync, qdat0,qdat1,qdat2,qdat3
`ifdef QDR_TRACK
  ,tadr,track
`endif
);
  parameter N=1;
  parameter PIPE=0;
  parameter INVERT=0;
  parameter DELAY=0;
  input [N-1:0] pio_p,pio_n;
  input qclk,pclk,rrst;
  input [7:0] sync;
  output [N-1:0] qdat0,qdat1,qdat2,qdat3;

`ifdef QDR_TRACK
  input [3:0] tadr;
  output [7:0] track;
  wire [7:0] tracks [N-1:0];
  assign track = tracks[tadr];
`endif

  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
`ifdef QDR_TRACK
    diqdr #(PIPE,(INVERT>>i)&1,DELAY) inst (pio_p[i],pio_n[i], qclk,pclk,rrst,sync, qdat0[i],qdat1[i],qdat2[i],qdat3[i], tracks[i]);
`else
    diqdr #(PIPE,(INVERT>>i)&1,DELAY) inst (pio_p[i],pio_n[i], qclk,pclk,rrst,sync, qdat0[i],qdat1[i],qdat2[i],qdat3[i]);
`endif
   end
  endgenerate

endmodule


module doqdr (pio_p,pio_n, qclk,pclk,prst,ss,ts, qdat0,qdat1,qdat2,qdat3);
  parameter PIPE=0;
  parameter INVERT=0;
  parameter DELAY=0;
  localparam DELAYM=(DELAY==0)?0:DELAY-1;
  output pio_p,pio_n;
  input qclk,pclk,prst,ss,ts;
  input qdat0,qdat1,qdat2,qdat3;
  wire inv = (INVERT!=0);

  wire pio,piod,pts;
  wire sdat0,sdat1,sdat2,sdat3,prstd;
  ff #(5,PIPE) pr ({sdat0,sdat1,sdat2,sdat3,prstd},{qdat0^inv,qdat1^inv,qdat2^inv,qdat3^inv,prst},qclk);
`ifdef VIRTEX2
  reg pdat0;  always @(posedge pclk) pdat0  <= ss? sdat2 : sdat0;
  reg pdat1;  always @(posedge pclk) pdat1  <= ss? sdat3 : sdat1;
  reg pdat1n; always @(negedge pclk) pdat1n <= pdat1;
  assign pts = ts;
  FDDRRSE ser (.Q(pio),.C0(pclk),.C1(~pclk),.CE(1'b1),.D0(pdat0),.D1(pdat1n),.R(1'b0),.S(1'b0));
`elsif VIRTEX4
  OSERDES #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    ser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.SR(prst), .OCE(1'b1),
         .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(sdat0),.D2(sdat1),.D3(sdat2),.D4(sdat3));
`elsif VIRTEX6
  OSERDESE1 #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    ser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst), .OCE(1'b1),
         .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(sdat0),.D2(sdat1),.D3(sdat2),.D4(sdat3));
`elsif VIRTEX7
  OSERDESE2 #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    ser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst), .OCE(1'b1),
         .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(sdat0),.D2(sdat1),.D3(sdat2),.D4(sdat3));
`elsif VIRTEX8
  OSERDESE3 #(.DATA_WIDTH(4)) 
    ser (.OQ(pio),.T_OUT(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst),
         .T(ts), .D({sdat3,sdat2,sdat1,sdat0}));
  generate
  if (DELAY>0) begin:blk
    ODELAYE3 #(.DELAY_TYPE("FIXED"),.DELAY_FORMAT("COUNT"),.DELAY_VALUE(DELAYM)) dly (.ODATAIN(pio),.DATAOUT(piod),.EN_VTC(1'b0));
  end
  endgenerate
`elsif SPARTAN6
  OSERDES2 #(.DATA_RATE_OQ("SDR"),.DATA_RATE_OT("SDR"),.DATA_WIDTH(4),.OUTPUT_MODE("SINGLE_ENDED")) 
    ser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK0(pclk),.RST(prstd), .OCE(1'b1),.IOCE(ss), 
         .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(sdat0),.D2(sdat1),.D3(sdat2),.D4(sdat3));
`endif
  OBUFTDS `OSTD dq (.I((DELAY>0)?piod:pio),.T(pts),.O(pio_p),.OB(pio_n));  // the ts is necessay - using a OBUFDS will produce no drive

endmodule

module doqdrN (pio_p,pio_n, qclk,pclk,prst,ss,ts, qdat0,qdat1,qdat2,qdat3);
  parameter N=1;
  parameter PIPE=0;
  parameter INVERT=0;
  output [N-1:0] pio_p,pio_n;
  input qclk,pclk,prst,ss,ts;
  input [N-1:0] qdat0,qdat1,qdat2,qdat3;

  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    doqdr #(PIPE,(INVERT>>i)&1) inst (pio_p[i],pio_n[i], qclk,pclk,prst,ss,ts, qdat0[i],qdat1[i],qdat2[i],qdat3[i]);
   end
  endgenerate

endmodule

module doqdrc (pio_p,pio_n, qclk,qrst,pclk,rrst,ss,ts);
  parameter SPEED=400;	// in MHz
  parameter PIPE=0;
  localparam PERIOD=1000.0/SPEED;
  output pio_p,pio_n;
  input qclk,qrst,ts;
  output rrst,ss;
`ifdef SPARTAN6
  output [1:0] pclk;
`else
  input pclk;
`endif
  wire H=1, L=0;
  wire pts,pio;

`ifndef SPARTAN6
  wire prst; delaypipe #(63,1) vdp (qclk,qrst,prst);
  reg ss1; always @(posedge qclk) ss1 <= !ss1;
  reg ss2; always @(negedge qclk) ss2 <=  ss1;
  reg ss;  always @(posedge pclk) ss  <=  ss1^ss2;
  assign rrst = prst;
`endif

`ifdef VIRTEX2
  FDDRRSE ser (.Q(pio),.C0(pclk),.C1(~pclk),.CE(1'b1),.D0(H),.D1(L),.R(1'b0),.S(1'b0));
  assign pts = ts;

`elsif VIRTEX4
  OSERDES #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    ser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.SR(prst), .OCE(1'b1),
         .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(H),.D2(L),.D3(H),.D4(L));

`elsif VIRTEX6
  OSERDESE1 #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    qser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst), .OCE(1'b1),
          .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(H),.D2(L),.D3(H),.D4(L));

`elsif VIRTEX7
  OSERDESE2  #(.DATA_RATE_OQ("DDR"),.DATA_RATE_TQ("DDR"),.DATA_WIDTH(4)) 
    qser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst), .OCE(1'b1),
          .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(H),.D2(L),.D3(H),.D4(L));

`elsif VIRTEX8
  OSERDESE3  #(.DATA_WIDTH(4)) 
    qser (.OQ(pio),.T_OUT(pts), .CLKDIV(qclk),.CLK(pclk),.RST(prst),
	  .T(ts), .D({L,H,L,H}));

`elsif SPARTAN6
  wire vclk1,vclk4,vfb,vlock,lock;
  reg [11:0] qcnt; always @(posedge qclk) if (qrst) qcnt <= 0; else qcnt <= qcnt+1;
  reg prst,prstc,prstd;  always @(posedge qclk) begin prst <= !vlock && (qcnt[11:4]==0); prstc <= prst; prstd <= prstc; end
  reg rrst_;       always @(posedge qclk) rrst_<= !vlock || prst;
  delaypipe #(62,1) rdp (qclk,rrst_,rrst);
  PLL_BASE pll (.CLKIN(qclk),.RST(prst),.CLKOUT0(vclk4),.CLKOUT1(vclk1),.CLKFBIN(vfb),.CLKFBOUT(vfb),.LOCKED(vlock));
  defparam pll.CLKIN_PERIOD   = PERIOD*2;	// this is the quarter rate clock speed not the DDR clock
  defparam pll.COMPENSATION   = "INTERNAL";
  defparam pll.CLKFBOUT_MULT  = 4;
  defparam pll.CLKOUT0_DIVIDE = 1;
  defparam pll.CLKOUT1_DIVIDE = 4;
  BUFPLL   puf (.PLLIN(vclk4),.LOCKED(vlock),.GCLK(qclk),.IOCLK(pclk),.LOCK(lock),.SERDESSTROBE(ss));
  defparam puf.DIVIDE = 4;
  OSERDES2 #(.DATA_RATE_OQ("SDR"),.DATA_RATE_OT("SDR"),.DATA_WIDTH(4),.OUTPUT_MODE("SINGLE_ENDED")) 
    qser (.OQ(pio),.TQ(pts), .CLKDIV(qclk),.CLK0(pclk),.RST((PIPE>0)?prstd:prst), .OCE(1'b1),.IOCE(ss), 
          .T1(ts),.T2(ts),.T3(ts),.T4(ts), .D1(H),.D2(L),.D3(H),.D4(L));
`endif

  OBUFTDS `OSTD dq (.I(pio),.T(pts),.O(pio_p),.OB(pio_n));

endmodule

module dioqdr (pio_p,pio_n, iclk,iclk2,irst,iss, idat0,idat1,idat2,idat3,
                            oclk,oclk2,orst,ts, odat0,odat1,odat2,odat3); 
  parameter PIPE=0;
  parameter INVERT=0;
  parameter DELAY=0;
  inout pio_p,pio_n;
  input iclk,iclk2,irst;
  input [7:0] iss;
  output idat0,idat1,idat2,idat3;
  input oclk,oclk2,orst,ts;
  input odat0,odat1,odat2,odat3;
  wire oss=0;

  diqdr #(PIPE,INVERT,DELAY) diq (pio_p,pio_n, iclk,iclk2,irst,iss,    idat0,idat1,idat2,idat3);
  doqdr #(PIPE,INVERT)       doq (pio_p,pio_n, oclk,oclk2,orst,oss,ts, odat0,odat1,odat2,odat3);

endmodule

module dioqdrN (pio_p,pio_n, iclk,iclk2,irst,ss, idat0,idat1,idat2,idat3, oclk,oclk2,orst,ts, odat0,odat1,odat2,odat3);
  parameter N=1;
  parameter PIPE=0;
  parameter INVERT=0;
  parameter DELAY=0;
  inout [N-1:0] pio_p,pio_n;
  input iclk,iclk2,irst;
  input [7:0] ss;
  output [N-1:0] idat0,idat1,idat2,idat3;
  input oclk,oclk2,orst,ts;
  input [N-1:0] odat0,odat1,odat2,odat3;

  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    dioqdr #(PIPE,(INVERT>>i)&1,DELAY) inst (pio_p[i],pio_n[i], 
       iclk,iclk2,irst,ss, idat0[i],idat1[i],idat2[i],idat3[i],
       oclk,oclk2,orst,ts, odat0[i],odat1[i],odat2[i],odat3[i]);
   end
  endgenerate

endmodule


// clocked DDR differential input buffer external termination
module diddrxt (pio_p,pio_n, clk, idatp,idatn);
  input pio_p,pio_n;
  input clk;
  output idatp,idatn;

  wire rff;

  IBUFDS #(.IOSTANDARD("LVDSEXT_25")) dq (.O(rff),.I(pio_p),.IB(pio_n));
  FD brffp (.D(rff),.C(clk),.Q(idatp)) /* synthesis syn_useioff=1 */;
  FD brffn (.D(rff),.C(~clk),.Q(idatn)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of brffp is "TRUE"
  //synthesis attribute IOB of brffn is "TRUE"

endmodule

module diddrxtN (pio_p,pio_n, clk, idatp,idatn);
  parameter N=1;
  input [N-1:0] pio_p,pio_n;
  input clk;
  output [N-1:0] idatp,idatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    diddrxt inst (pio_p[i],pio_n[i], clk, idatp[i],idatn[i]);
   end
  endgenerate
endmodule

module doddrb (pio_p,pio_n, clk,oe, odatp,odatn);
  parameter PIPE=0;
  inout pio_p,pio_n;
  input oe,clk;
  input odatp,odatn;

  wire H=1, L=0;
  wire wff,wts;

`ifdef VIRTEX8
  ODDRE1 bwff (.Q(wff),.C(clk),.D1(odatp),.D2(odatn),.SR(L));
`else
  ODDR #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE":"OPPOSITE_EDGE")) bwff (.Q(wff),.C(clk),.CE(H),.D1(odatp),.D2(odatn));
`endif
  OBUFTDS `OSTDB dq (.I(wff),.T(~oe),.O(pio_p),.OB(pio_n));

endmodule

// clocked DDR differential output buffer
module doddr (pio_p,pio_n, clk,oe, odatp,odatn);
  parameter PIPE=0;
  parameter INVERT=0;
  inout pio_p,pio_n;
  input oe,clk;
  input odatp,odatn;

  wire H=1, L=0;
  wire wff,wts;
  wire inv = INVERT;

`ifdef VIRTEX2
  reg odatnd; always @(negedge clk) odatnd <= odatn;
  wire odatnb = PIPE? odatnd:odatn;
  FDDRRSE bwff (.Q(wff),.C0(clk),.C1(~clk),.CE(H),.D0(odatp),.D1(odatnb),.R(L),.S(L)) /* synthesis syn_useioff=1 */;
  //synthesis attribute IOB of bwff is "TRUE"
  OBUFTDS `OSTD dq (.I(wff),.T(~oe),.O(pio_p),.OB(pio_n));
`elsif SPARTAN6
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) bwff (.Q(wff),.C0(clk),.C1(~clk),.CE(H),.D0(inv^odatp),.D1(inv^odatn),.R(L),.S(L));
  ODDR2 #(.DDR_ALIGNMENT(PIPE?"C0":"NONE"),.SRTYPE(PIPE?"ASYNC":"SYNC")) bwts (.Q(wts),.C0(clk),.C1(~clk),.CE(H),.D0(~oe),.D1(~oe),.R(L),.S(L));
  OBUFTDS `OSTD dq (.I(wff),.T(wts),.O(pio_p),.OB(pio_n));
`elsif VIRTEX8
  ODDRE1 bwff (.Q(wff),.C(clk),.D1(odatp),.D2(odatn),.SR(L));
  OBUFTDS `OSTD dq (.I(wff),.T(~oe),.O(pio_p),.OB(pio_n));
`else
  ODDR #(.DDR_CLK_EDGE(PIPE?"SAME_EDGE":"OPPOSITE_EDGE")) bwff (.Q(wff),.C(clk),.CE(H),.D1(odatp),.D2(odatn));
  OBUFTDS `OSTD dq (.I(wff),.T(~oe),.O(pio_p),.OB(pio_n));
`endif

endmodule

module doddrN (pio_p,pio_n, clk,oe, odatp,odatn);
  parameter N=1;
  parameter PIPE=0;
  parameter INVERT=0;
  inout [N-1:0] pio_p,pio_n;
  input oe,clk;
  input [N-1:0] odatp,odatn;
  genvar i;
  generate
   for (i=0; i<N; i=i+1) begin:blk
    doddr #(PIPE,((INVERT>>i)&1)) inst (pio_p[i],pio_n[i], clk,oe, odatp[i],odatn[i]);
   end
  endgenerate
endmodule

/*
  Digitize the strobe of a DDR input clock with a 270 phase shifted version
  of the data collection clock to keep the clock centered on the data window
*/
module timing_recovery (clk,rst, tune,data, sclk,trse,trsa,trsu, out);

  parameter BW=12;
  input clk,rst;
  input [7:0] tune;
  input [2:0] data;
  input sclk,trse;
  output trsa,trsu;
  output [31:0] out;

  wire env = data[0];
  wire rev = data[1];
  wire fev = data[2];

  reg adj,adjup,adjdn,done,donex,doney;
  reg [BW-1:0] rec,fec,cnt;
  reg [7:0] tot,srec,sfec;

  // buffers for timing
  reg enw,rew,few;
  always @(posedge clk) enw <= env;
  always @(posedge clk) rew <= rev;
  always @(negedge clk) few <= fev;
  reg enx,rex,fex;
  always @(posedge clk) enx <= enw;
  always @(posedge clk) rex <= rew;
  always @(posedge clk) fex <= few;

  // calibration logic
  reg eny,rey,fey;
  always @(posedge clk) begin
    eny <= enx;
    if (!enx) rey <= 0; else rey <= rex;
    if (!enx) fey <= 0; else fey <= fex;
    if (rst) cnt <= 0; else if (eny) cnt <= cnt+1;
    done  <= eny && (cnt==0); donex <= done; doney <= donex;
    if (done) rec  <= 0; else if (rey) rec <= rec+1;
    if (done) fec  <= 0; else if (fey) fec <= fec+1;
    if (done) srec <= rec[BW-1:BW-8];
    if (done) sfec <= fec[BW-1:BW-8];
    adjdn <= (srec[7:4]>tune[7:4]);
    adjup <= (srec[7:4]<tune[3:0]);
    adj   <= (doney && (adjup ^ adjdn)) ^ adj;
  end

  reg trsz,trsx,trsa,trsu,sadj,updn,ooru,oord;
  always @(posedge sclk) begin
    updn  <= adjup;
    if (!trse) sadj <= adj; else if (trsx) sadj <= !sadj;
    trsx <= (sadj != adj) && !trsx;
    trsa <= trsx && !trsz && (updn? !ooru : !oord);
    trsz <= trsa || (trsz && !trsx); // wait one cycle for settling
    if (!trse) tot <= 0; else if (trsa) tot <= trsu? tot+1 : tot-1;
    trsu <= updn;
    ooru <= tot[7:4]==4'h2;
    oord <= tot[7:4]==4'hD;
  end

  assign out = {sfec, srec, tot, 3'd0,adj, 3'd0,trse};

endmodule

