/**********************************************
 ICE Proprietary Software - do NOT disseminate
 **********************************************/
/*
  Ice NVME x4 Controller

  Jeff Schoen
  Innovative Computer Engineering, Inc.
  1/15/2003

*/
module nvmcontroller (
  bsclk,bscs,bsrd,bswr,bspage,bsaddr,bsdata,bsdatao,	// Boundary Scan Interface
  cclk, cs,crd,cwr,cpage,caddr,cdata,cdatao,		// JVM control bus
  rclk, rrst, rcena, rcbus,				// RIO command bus
  pxclk, pxrxp,pxrxn, pxtxp,pxtxn, 			// Physical 3GIO ports
  ioclk,iorst, irdy,iena,isel,ibus, ordy,oena,osel,obus,// DMA Controller IO bus
`ifdef NVM_DBG
  sclk,tic,ticx, drdy,dena,dbus, test);			
`else
  sclk,tic,ticx,test);			
`endif

  parameter PORTS=4;
  parameter IBW=32;

  localparam NC=PORTS/4;
  localparam DBW=IBW;
  localparam UBW=128;
  localparam FBW=8+UBW;
  localparam BSPO=(PORTS==4)?4:0;			// boundary scan port offset
  localparam IBV=IBW/8;
  localparam UBV=UBW/8;

  // BSCAN ports
  input bsclk,bscs,bsrd,bswr;
  input [7:0] bspage,bsaddr,bsdata;
  output [7:0] bsdatao;

  // JVM control IF
  input cs;
  input cclk,crd,cwr;
  input [7:0] cpage,caddr,cdata;
  output [7:0] cdatao;
        
  // RIO command bus
  input rclk,rrst,rcena;
  input [35:0] rcbus;

  // MGT ports
  inout [3:0] pxclk;
  input [PORTS-1:0] pxrxp,pxrxn;
  output [PORTS-1:0] pxtxp,pxtxn;

  // IO port busses
  input ioclk,iorst;
  input iena,oena;
  output irdy,ordy;
  input [7:0] isel;
  output [15:0] osel;
  input [IBW-1:0] ibus;
  output [IBW-1:0] obus;

  // test port
  input sclk;
  input tic,ticx;
`ifdef NVM_DBG
  output drdy;
  input  dena;
  output [IBW-1:0] dbus;
`endif
  output [7:0] test;

  // global signals 
  wire F=0, T=1;
  wire csel = (NC==2)? caddr[7] : F;
  wire [NC-1:0] ics = (NC==2)? {cs&csel,cs&!csel} : cs;

  // boundary scan to DRP monitor
  wire drp_en;
  wire [15:0] drp_di,drp_addr;
  wire [PORTS-1:0] drp_wex,drp_rdyx;
  wire [PORTS*16-1:0] drp_dox;
  wire [PORTS*17-1:0] dmon_dox;
  wire drp_clk = sclk;
  bs2drp #(PORTS,0,BSPO) (bsclk,bscs,bsrd,bswr,bspage,bsaddr,bsdata,bsdatao,
                   drp_clk,drp_en,drp_wex,drp_addr,drp_di,drp_rdyx,drp_dox,dmon_dox);

  wire [7:0] cdatao_[NC-1:0];
  assign cdatao = cdatao_[csel];

  reg [1:0] toks;
  wire [1:0] loks;
  assign test = {loks,toks};

genvar i;
generate
for (i=0; i<NC; i=i+1) begin

  wire user_clk,user_reset,user_lnk_up;
  wire nvclk = user_clk;

  assign loks[i] = user_lnk_up;
  always @(posedge nvclk) toks[i] <= !toks[i];

  // processor interface
  wire srst=F;
  wire srwx,srdx,swrx,sackx,swrs,srds,swrf,srdf; wire [31:0] saddrx,swrbusx,srdbusx,srdbusy;
  wire bwr,brd; wire [19:0] baddr; wire [31:0] bwrbus,brdbus,brdbusx,brdbusy;
  wire umem = baddr[16];
  assign brdbus = umem? brdbusy : brdbusx;
  wire bwrx = bwr && !umem;
  wire bwry = bwr &&  umem;
  wire [7:0] testp;
  processorLite #(32,4,4) prc (cclk,crd,cwr,cpage,caddr,cdata,cdatao_[i],ics[i],
                nvclk,baddr, brd,brdbus, bwr,bwrbus,
                nvclk,srst, srwx,saddrx, srdx,srdbusx, swrx,swrbusx, sackx,
                nvclk,srds,swrs, nvclk,srdf,swrf, testp);

  // RIO command bus
  wire crdy,srdy; wire selcf = saddrx[17]; wire selcfrd = srdx && selcf && saddrx[4];
  wire [31:0] rcstat,rcdata; assign srdbusy = saddrx[3]? rcstat : rcdata;
  cmdfifo #(4) cf0 (rrst,rcstat, rclk,crdy,rcena,rcbus, nvclk,srdy,selcfrd,rcdata);

  // packet processor system register
  wire [15:0] nadr  = saddrx;
  wire [31:0] nwbus = swrbusx;
  wire nwc = swrx;
  wire nrc = srdx;
  reg [31:0] nsys,ncfgr;
  wire [9:0] adr = nsys[11:0];
  wire [4:0] len = nsys[16:12];
  wire lbe = nsys[19];		// last BE
  wire [3:0] lst = nsys[23:20];	// last quad enable
  wire rvw = nsys[24];		// read | write
  wire rvc = nsys[25];		// requester | completer
  wire cyc = nsys[26];		// start cycle
  wire cfg = nsys[27];		// configuration access
  wire rst = nsys[28];		// reset
  wire ack = nsys[29];		// signal ack
  wire xio = nsys[30];		// external IO mode
  wire peek = nsys[31];		// peek at buffer - dont actually read
  wire cfga = nadr[15];		// rd/wr config
  wire sysa = nadr[14];		// rd/wr system
  wire nws_ = nwc && sysa;	// write nsys
  wire [31:0] nstat,nrbus;

  wire prstn; delaypipe #(32) startup (nvclk,T,prstn);

/*
	nws	nwsd
	nsys	xcyc	xcycd

xlen	0	N	N-1	N-2
xadr	X	0	adr	adr+1
wbus			d0	dadr	dadr+1

*/

  // packet cycle logic
  reg [9:0] xadr,yadr;
  reg [4:0] xlen;
  wire [UBW-1:0] xrbus,xwbus,yrbus,ywbus,zwbus,rbus;
  wire wready,rlasti;
  reg nws,nwsd,pgpm;
  reg xcyc,xcycd,xiovd,lastd;
  wire last = (xlen<=1) || (rvw && rlasti);
  wire xiov = xio && (xcyc && !nwsd);

  // requester IF logic
  always @(posedge nvclk) begin
    nws <= nws_; nwsd <= nws; 
    if (nws_) nsys <= nwbus;
    xcyc <= (cyc && nws) || (xcyc && !last);
    xcycd <= xcyc;
    xiovd <= xiov;
    lastd <= last;
    if (nws) xlen <= len; else if (xcyc) xlen <= xlen-1;
    if (nws) xadr <= 0;   else if (nwsd) xadr <= adr; else if (xcyc) xadr <= xadr+1; 
  end
  assign srdbusx = selcf? srdbusy : cfga? ncfgr : nstat;
  wire xwr = xcyc && rvw && !xio &&  rvc;
  wire [3:0] wkeep = lastd? lst : 4'hF;
  wire sack = nws && ack;

  // completer IF logic
  wire [31:0] caddrl = yrbus[31+ 0: 0+ 0];
  wire [31:0] caddru = yrbus[31+32: 0+32];
  wire [10:0] clen   = yrbus[10+64: 0+64];
  wire [3:0]  ctype  = yrbus[14+64:11+64];
  wire [15:0] crid_  = yrbus[31+64:16+64];
  wire [7:0]  ctag_  = yrbus[ 7+96: 0+96];
  wire [23:0] ccid   = 24'h010000;
  wire cgo,lastc,ccrdy;
  wire [9:0]  cadr   = caddru[13:4] | caddrl[13:4];	// using OR so NVME thinks queues are 4K page aligned
  reg [10:0] ccnt;
  reg [2:0] ccut;
  reg [6:0] caddp; 
  reg [31:0] caddq; 
  reg [3:0] ckeep;
  reg [15:0] crid;
  reg [7:0] ctag;
  wire cntgt4 = (ccnt>4);
  wire crcmp = (ccnt<=32);
  wire [5:0] clenx = crcmp? ccnt : 32;
  wire crcyc_ = (ctype==0);
  wire cwcyc_ = (ctype==1);
  wire cxcyc_ = caddru[13];
  wire cxpgp_ = caddru[14];
  reg cgoon,cgoh,cgor,cwcyc,crcyc,cxcyc,cxsel;
  reg ok_credits;
  //                                    cgo  1
  //                                    cgoh    1  1  1  1  1  1  1  1     1  1  1
  //                                    cgor       1  1  1  1  1  1  1  1     1  1  1
  //  cgo -> cgoh  -> cgor 		ccnt	   42 38 34 30 26 22 18 14 10 10 6  2
  //         cadr  -> cadr+1		ccut	   0  1  2  3  4  5  6  7     0  1  2
  //         ccnt  -> ccnt-4		caddr
  // 
  wire cval;
  wire cgorv = cgor && cval;
  wire cgon = ((ccut==7) || !cntgt4) && cval;
  wire [9:0] yadrp = yadr + cgorv;
  always @(posedge nvclk) begin
    if (cgo) yadr  <= cadr; else if (cwcyc||cgorv) yadr <= yadr+1; 
    if (cgo) cxcyc <= cxcyc_ && !rst;	// to keep cxsel seperate from cxcyc
    if (cgo) cxsel <= cxcyc_;
    if (cgo) ctag  <= ctag_;
    if (cgo) crid  <= crid_;
    if (cgo) ccnt  <= clen; else if (cgorv) ccnt <= ccnt-4;
    if (cgo) caddp <= caddrl; else if (cgorv) caddp <= caddp+16;
    if (cgo) caddq <= caddrl; 
    if (cgo) pgpm  <= cxpgp_; 	// PGP List
    if (!cgor) ccut<= 0; else if (cval) ccut <= ccut+1;
    if (rst) crcyc <= 0; else crcyc <= (cgo && crcyc_) || (crcyc && !(cgorv && !cntgt4));
    if (rst) cgoh  <= 0; else cgoh  <= (cgo && crcyc_) || (cgoh && (!ccrdy||!cval)) || (cgorv && (ccut==7) && cntgt4);
    if (rst) cgor  <= 0; else cgor  <= (cgoh && ccrdy && cval)  || (cgor && !cgon);
    if (rst) cwcyc <= 0; else cwcyc <= (cgo && cwcyc_) || (cwcyc && cgoon);
    if (rst) cgoon <= 0; else cgoon <= (cgo || cgoon) && !lastc;
    ckeep <= (ccnt>=4)? 4'hF : (ccnt[1:0]==3)? 4'h7 : (ccnt[1:0]==2)? 4'h3 : 4'h1;
  end
  wire [9:0] yadrp1 = {yadr,1'b1};
  wire [9:0] yadrp2 = {yadr+1,1'b0};
  wire [127:0] pgp  = { 32'h2000, 10'h0,yadrp2,12'b0 , 32'h2000, 10'h0,yadrp1,12'b0 };	// PGP list
  wire [127:0] rcmd = { 32'h0 , ccid,ctag , crid,10'b0,clenx , 1'b0,crcmp,1'b0,ccnt,11'h0,caddp };
  wire ywr = cgoon && cwcyc && !cxcyc;

  // PCIe root controller RAM
  dpram #(8,32,UBW) nvmrx (nvclk,T, baddr[13:2],bwrx, bwrbus,brdbusx, nvclk,T, xadr,xwr, xrbus,xwbus);
  dpram #(8,32,UBW) nvmry (nvclk,T, baddr[13:2],bwry, bwrbus,brdbusy, nvclk,T, yadrp,ywr, yrbus,ywbus);

  // PCIe IO RAMs
  wire xwrdy,xrrdy;
  wire xrena = cxcyc && cwcyc && cgoon;
  wire xwena = cxcyc && cgor && cval;
  reg ival; always @(posedge ioclk) ival <= iena;
  reg frst; always @(posedge ioclk) frst <= iorst || rst;
  wire irdy_, ordy_; wire [IBW-1:0] obus_; wire [IBV-1:0] osel_;
  if (i==0) begin assign obus = obus_; assign osel = osel_; assign irdy = irdy_; assign ordy = ordy_; end
  // 16K data fifos
  fifoNxM #(IBW,UBW,8,5,0,`PACKET|`RAHEAD|`BURST16) xiow (ioclk,frst, irdy_,ival,ibus,    nvclk,frst, xwrdy,xwena,zwbus);
  fifoNxM #(UBW+UBV,IBW+IBV,8,5,0,`PACKET|`BPAR) xior (nvclk,frst, xrrdy,xrena,{caddq[23:8],yrbus}, ioclk,frst, ordy_,oena,{osel_,obus_});
  wire [3:0] fsta = {ordy,xrrdy,xwrdy,irdy};	// FIFO status bits

  // reset logic
  reg rstm; always @(posedge sclk) rstm <= (tic && !user_lnk_up) || (xio&rst);
  reg rstn; always @(posedge sclk) rstn <= !rstm;

  // PCIe root core signals
  wire sys_clk    = pxclk[1];
  wire sys_clk_gt = pxclk[0];
  wire sys_rst_n  = rstn;

  wire s_axis_cc_tvalid = (cgoh && ccrdy) || cgor;
  wire s_axis_cc_tlast  = cgor && cgon;
  wire s_axis_cc_tready;
  wire [32:0]  s_axis_cc_tuser = 0;             // Parity:32|Discontinue
  wire [127:0] s_axis_cc_tdata = cgoh? rcmd : pgpm? pgp : cxsel? zwbus : ywbus;
  wire [3:0]   s_axis_cc_tkeep = ckeep;

  wire cqs,cqe;
  wire m_axis_cq_tvalid;
  wire m_axis_cq_tlast;
  wire m_axis_cq_tready = cgo || cgoon;
  wire [84:0]  m_axis_cq_tuser;
  wire [127:0] m_axis_cq_tdata;
  wire [3:0]   m_axis_cq_tkeep;

  wire setc = (s_axis_cc_tvalid && s_axis_cc_tready);
  wire getc = (m_axis_cq_tvalid && m_axis_cq_tready);
  assign lastc = (m_axis_cq_tvalid && m_axis_cq_tready && m_axis_cq_tlast);
  assign cgo   = m_axis_cq_tvalid && m_axis_cq_tuser[40] && !crcyc && xrrdy;
  assign ccrdy = ok_credits && (xwrdy || !cxcyc);
  assign cval = s_axis_cc_tready;

//  assign wdbg = s_axis_cc_tvalid;
//  assign wdbgbus = {ccrdy,ok_credits,s_axis_cc_tready,s_axis_cc_tlast, s_axis_cc_tdata[123:0]};

  wire m_axis_rc_tvalid;
  wire m_axis_rc_tlast;
  wire m_axis_rc_tready = xcyc && rvw && rvc;
  wire [74:0]  m_axis_rc_tuser;
  wire [127:0] m_axis_rc_tdata;
  wire [3:0]   m_axis_rc_tkeep;

  wire s_axis_rq_tvalid = xcycd && !rvw && rvc;
  wire s_axis_rq_tlast  = lastd;
  wire s_axis_rq_tready;
  wire [11:0]  s_axis_rq_tuser = {4'h0,{4{lbe}},4'hF};       // Discontinue|AddOff:3|LastBE:4|FirstBE:4
  wire [127:0] s_axis_rq_tdata = xwbus;
  wire [3:0]   s_axis_rq_tkeep = wkeep;

  wire setr = (s_axis_rq_tvalid && s_axis_rq_tready);
  wire getr = (m_axis_rc_tvalid && m_axis_rc_tready);
  wire lastr = (m_axis_rc_tvalid && m_axis_rc_tready && m_axis_rc_tlast);

  reg getbusy; always @(posedge user_clk) if (rst) getbusy <= F; 
               else if (sack) getbusy <= T; else if (rvc?getr:getc) getbusy <= F;

  assign xrbus = m_axis_rc_tdata;
  assign yrbus = m_axis_cq_tdata;
  assign rlasti = rvc? m_axis_rc_tlast : m_axis_cq_tlast;
  assign wready = s_axis_rq_tready;

  wire [7:0] lsta = {s_axis_rq_tvalid,m_axis_rc_tvalid,m_axis_cq_tvalid,s_axis_cc_tvalid,
		     s_axis_rq_tready,m_axis_rc_tready,m_axis_cq_tready,s_axis_cc_tready};

  // PCIe configuration interface
  wire [2:0]  fc_sel=3'b100;                    // Transmit Credits Available Space
  wire [7:0]  fc_ph,fc_nph,fc_cplh;
  wire [11:0] fc_pd,fc_npd,fc_cpld;
  wire [18:0] cfg_mgmt_dwaddr = {nadr[1],4'h0,1'b0,nadr[14:2]}; // CFG MGMT 19-bit address port
  wire [31:0] cfg_mgmt_di = nwbus;              // CFG MGMT data in port
  wire [31:0] cfg_mgmt_do;                      // CFG MGMT data out port
  wire [3:0]  cfg_mgmt_byte_en = 4'hF;          // CFG MGMT byte enables
  wire cfg_mgmt_type1_cfg_reg_access = nadr[0];
  wire cfg_mgmt_rd_wr_done;
  reg cfg_mgmt_rd_en;
  reg cfg_mgmt_wr_en;
  reg ok_cplh,ok_cpld,ok_nph,ok_npd,ok_ph,ok_pd;
  always @(posedge user_clk) begin
    cfg_mgmt_rd_en <= (cfga && nrc) || (cfg_mgmt_rd_en && !cfg_mgmt_rd_wr_done);
    cfg_mgmt_wr_en <= (cfga && nwc) || (cfg_mgmt_wr_en && !cfg_mgmt_rd_wr_done);
    if (cfg_mgmt_rd_en && cfg_mgmt_rd_wr_done) ncfgr <= cfg_mgmt_do;
    ok_cplh <= (fc_cplh>=4);
    ok_cpld <= (fc_cpld>=32);
    ok_nph  <= (fc_nph>=4);
    ok_npd  <= (fc_npd>=4);
    ok_ph   <= (fc_ph>=4);
    ok_pd   <= (fc_pd>=32);
    ok_credits <= ok_pd && ok_ph && ok_npd && ok_nph && ok_cpld && ok_cplh;
  end

  wire [5:0] cfg_ltssm_state;
  wire [2:0] pl_sel_lnk_rate;
  wire [3:0] pl_sel_lnk_width;
  wire pl_directed_change_done;
  wire [3:0] gt_rxcdrlock,gt_rxelecidle;
  wire [7:0] cfg_lnk_status = {pl_sel_lnk_width,1'b0,pl_sel_lnk_rate};

  assign nstat = { cfg_lnk_status, 4'h0,fsta, lsta, user_lnk_up,getbusy,cfg_ltssm_state };

  // CommonMode != GND work around
  reg cm_detect,cm_state,cm_set,cm_ltssm;
  reg rxelecidle,rxelecidled;
  reg [1:0] tics;
  reg [3:0] cm_cnt;
  reg [5:0] rxelecidles;
  reg [15:0] rxelecidlecnt;
  always @(posedge drp_clk) begin
    tics <= {tics,tic};
    rxelecidle  <= |gt_rxelecidle;
    rxelecidles <= {rxelecidles,rxelecidle};
    rxelecidled <= rxelecidled? |rxelecidles : &rxelecidles;	// 6 clock hysteresis
    if (!rxelecidled) rxelecidlecnt <= 0; else rxelecidlecnt <= rxelecidlecnt+1;
    cm_ltssm <= (cfg_ltssm_state<2);
    cm_detect <= (rxelecidlecnt[14:12]!=0);
    cm_set <= cm_set? F : (cm_cnt==0) && ((cm_state!=cm_detect) || (tics==1));
    if (cm_set) cm_state <= cm_detect;
    if (cm_set) cm_cnt <= 15; else if (cm_cnt!=0) cm_cnt <= cm_cnt-1;
  end
  wire [8:0] cm_addr = 9'h061;
  wire [15:0] cm_val = { 5'h0,1'h0,4'hA,4'hA,cm_detect? 2'h1:2'h3}; // {rxph_mon,cm_buf_pd,cm_buf_cfg,cm_trim,cm_sel}

`ifdef NVM_DBG
  reg dbg_val;
  reg [95:0] dbg_dat;
  reg [27:0] dbg_cnt;
  reg [3:0]  dbg_sta;
  wire dbg_cgo = cgo && !rst;
  always @(posedge user_clk) begin
    dbg_val <= cgo || (cgoh && ccrdy && s_axis_cc_tready);
    dbg_sta <= {!xrrdy,xwrdy,cgoh,cgo};
    dbg_cnt <= dbg_cnt+1;
    dbg_dat <= dbg_cgo? m_axis_cq_tdata : rcmd;
  end
  wire dbg_rdy;
  fifoNxM #(128,IBW) dbgr (user_clk,frst, dbg_rdy,dbg_val,{dbg_cnt,dbg_sta,dbg_dat} , ioclk,frst, drdy,dena,dbus);
`endif

  // Xilinx PCIe LociCore instantiation
  pcif_0 #() pcif_i (
  .pci_exp_txn                                ( pxtxn[3+i*4:i*4] ),
  .pci_exp_txp                                ( pxtxp[3+i*4:i*4] ),
  .pci_exp_rxn                                ( pxrxn[3+i*4:i*4] ),
  .pci_exp_rxp                                ( pxrxp[3+i*4:i*4] ),
  .user_clk                                   ( user_clk ),
  .user_reset                                 ( user_reset ),
  .user_lnk_up                                ( user_lnk_up ),
  // Tx Host
  .s_axis_cc_tready                           ( s_axis_cc_tready ),
  .s_axis_cc_tdata                            ( s_axis_cc_tdata ),
  .s_axis_cc_tkeep                            ( s_axis_cc_tkeep ),
  .s_axis_cc_tuser                            ( s_axis_cc_tuser ),
  .s_axis_cc_tlast                            ( s_axis_cc_tlast ),
  .s_axis_cc_tvalid                           ( s_axis_cc_tvalid ),
  // Rx Host
  .m_axis_cq_tready                           ( m_axis_cq_tready ),
  .m_axis_cq_tdata                            ( m_axis_cq_tdata ),
  .m_axis_cq_tkeep                            ( m_axis_cq_tkeep ),
  .m_axis_cq_tuser                            ( m_axis_cq_tuser ),
  .m_axis_cq_tlast                            ( m_axis_cq_tlast ),
  .m_axis_cq_tvalid                           ( m_axis_cq_tvalid ),
  // Rx Card
  .m_axis_rc_tready                           ( m_axis_rc_tready ),
  .m_axis_rc_tdata                            ( m_axis_rc_tdata ),
  .m_axis_rc_tkeep                            ( m_axis_rc_tkeep ),
  .m_axis_rc_tuser                            ( m_axis_rc_tuser ),
  .m_axis_rc_tlast                            ( m_axis_rc_tlast ),
  .m_axis_rc_tvalid                           ( m_axis_rc_tvalid ),
  // Tx Card
  .s_axis_rq_tready                           ( s_axis_rq_tready ),
  .s_axis_rq_tdata                            ( s_axis_rq_tdata ),
  .s_axis_rq_tkeep                            ( s_axis_rq_tkeep ),
  .s_axis_rq_tuser                            ( s_axis_rq_tuser ),
  .s_axis_rq_tlast                            ( s_axis_rq_tlast ),
  .s_axis_rq_tvalid                           ( s_axis_rq_tvalid ),
  // Flow Control
  .cfg_fc_cpld                                ( fc_cpld ),
  .cfg_fc_cplh                                ( fc_cplh ),
  .cfg_fc_npd                                 ( fc_npd ),
  .cfg_fc_nph                                 ( fc_nph ),
  .cfg_fc_pd                                  ( fc_pd ),
  .cfg_fc_ph                                  ( fc_ph ),
  .cfg_fc_sel                                 ( fc_sel ),
  // Management Interface
  .cfg_mgmt_type1_cfg_reg_access              ( cfg_mgmt_type1_cfg_reg_access ),
  .cfg_mgmt_write_data                        ( cfg_mgmt_di ),
  .cfg_mgmt_byte_enable                       ( cfg_mgmt_byte_en ),
  .cfg_mgmt_addr                              ( cfg_mgmt_dwaddr ),
  .cfg_mgmt_write                             ( cfg_mgmt_wr_en ),
  .cfg_mgmt_read                              ( cfg_mgmt_rd_en ),
  .cfg_mgmt_read_data                         ( cfg_mgmt_do ),
  .cfg_mgmt_read_write_done                   ( cfg_mgmt_rd_wr_done ),
  .cfg_current_speed                          ( pl_sel_lnk_rate ),
  .cfg_negotiated_width                       ( pl_sel_lnk_width ),
  .cfg_ltssm_state                            ( cfg_ltssm_state ),
//  .cfg_hot_reset_in   ( rstm ),
  // DRP Interface 
  .free_run_clock       ( drp_clk ),            // this is an input
  .ext_ch_gt_drpdo      ( drp_dox[63+i*64:i*64] ),
  .ext_ch_gt_drprdy     ( drp_rdyx[3+i*4:i*4] ),
  .ext_ch_gt_drpaddr    ( {4{cm_set?cm_addr[8:0]:drp_addr[8:0]}} ),
  .ext_ch_gt_drpen      ( cm_set? 4'hF:{4{drp_en}} ),
  .ext_ch_gt_drpdi      ( {4{cm_set?cm_val:drp_di}} ),
  .ext_ch_gt_drpwe      ( cm_set? 4'hF:drp_wex[3+i*4:i*4] ),
  .pcie_cq_np_req       (1'b1),                 // allow CQ to use credits
  .gt_pcieuserratedone  (8'B0),
  .gt_loopback          (24'B0),
  .gt_txprbsforceerr    (8'B0),
  .gt_txprbssel         (32'B0),
  .gt_rxprbssel         (32'B0),
  .gt_dmonfiforeset     (8'h00),
  .gt_dmonitorclk       ({4{drp_clk}}),
  .gt_dmonitorout       ( dmon_dox[67+i*68:i*68] ),
  .gt_rxcdrlock         (gt_rxcdrlock),
  .gt_txelecidle        (gt_rxelecidle),        // using ICE_SWAP_RXTX_ELECIDLE=1
  // System  (SYS) Interface
  .sys_clk              ( sys_clk ),
  .sys_clk_gt           ( sys_clk_gt ),
  .sys_reset            ( sys_rst_n )
 );

end
endgenerate

endmodule
