package nxm.ice.prim;

import nxm.sys.inc.*;
import nxm.sys.lib.*;
import nxm.sys.libm.*;
import nxm.ice.lib.ICEPacket;

/**
  Performs cross correlation between channels of packetized data.

  @author Jeff Schoen
  @version $Id: picmeasure.java,v 1.3 2001/04/25 17:08:12 kingm Exp $
*/
public class picmeasure extends Primitive {

  private DataFile hi,ho;
  private Data dbi,dbo;
  private byte ftype;
  private int bpa,nchan,nfft,nfit,nffti,npkts;
  private ICEPacket pkt;
  private ChanData[] chns;
  private double tdelta,scale;
  private float[] fwin,fbuf,fbufa,fbufb,fbufi,fbufp;
  private Fft fftf, ffti;

  public int open() {

    hi = MA.getDataFile ("IN");
    hi.open();
    tdelta = hi.getXDelta();
    bpa = hi.getBPA();
    ftype = hi.getFormatType();

    pkt = (ICEPacket)hi.getPacketHandler();
    if (pkt==null) M.error("Input file is not packetized: "+hi.getTag());

    ho = MA.getDataFile ("OUT");
    ho.setSubRecords("CHAN|SI,STAT|SI,DIFF|SL,TIME|SD,TDIF|SD,SDIF|SD");
    ho.open(ho.OUTPUT|ho.OPTIONAL);

    nchan = MA.getL("NCHAN");
    npkts = MA.getL("/NPKT",5);
    nfft  = MA.getL("/NFFT",16384);
    nfit  = MA.getL("/NFIT",16384);
    nffti = MA.getL("/NFFTI",131072);
    scale = 1.0/nfft;

    xfer = MA.getL("/TL",pkt.getFixedSize());
    dbi = hi.getDataBuffer(xfer);
    dbo = ho.getDataBuffer(1);

    chns = new ChanData[nchan];
    for (int i=0; i<nchan; i++) {
      chns[i] = new ChanData();
      chns[i].chan = (short)(i+1);
      chns[i].clear();
    }
    fbuf  = new float[nfft];
    fbufi = new float[nfft];
    fbufa = new float[nfft];
    fbufb = new float[nfft];
    fwin  = Window.get("HANN",nfft,scale);

    fftf = new Fft(nfft,Fft.REAL|Fft.FORWARD|Fft.NONYQUIST);
    // fftf.setDebugFileOut(M,"test_picm_f");
    ffti = new Fft(nffti,Fft.COMPLEX|Fft.INVERSE);
    // ffti.setDebugFileOut(M,"test_picm_i");

    fbufp = new float[nffti*2]; // complex array

    return (NORMAL);
  }

  public int process() {
    int n, nc;

    // get the next packet header
    n = hi.read(dbi);
    if (n==0) return (NOOP);
    if (n<0) return (FINISH);

    // validate the channel number
    nc = pkt.getChannel();
    if (nc<=0 || nc>nchan) {
      M.warning("Channel "+nc+" out of range [1-"+nchan+"]");
      return(NOOP);
    }

    // get one-based channel from zero-based array 
    ChanData cd = chns[nc-1]; 

    // process data ?
    if (nc==1 && cd.ipkts==npkts) {
      MT.writeln("At time "+cd.tc);
      for (int i=0; i<nchan; i++) {
        ChanData ci = chns[i];
        if (ci.ipkts==0) continue;
        measure (cd,ci);
	MT.writeln(ci.toString());
	ci.setDataBuffer(dbo);
	ho.write(dbo);
        // skip all data held up in buffer
        for (n=0; hi.isStream() && hi.avail()>0; n++) hi.read(dbi);
      }
      if (hi.isStream()) {
        Time.sleep(0.1);
        for (int i=0; i<nchan; i++) chns[i].clear();
      }
    }
    else cd.addData(dbi,pkt,xfer);

    return (NORMAL);
  }

  public int close() {
    fftf.free();
    ffti.free();
    hi.close();
    ho.close();
    return (NORMAL);
  }

  public int getChannels() { return nchan; }

  public void measure (ChanData ca, ChanData cb) {
    int ioff=0,ia=0,ib=0,iover,navg,nfft=16384;

    // find time difference between channels
    cb.time = cb.tc.getSec();
    cb.tdif = cb.tc.diff(ca.tc);
    ioff = (int)Math.round(cb.tdif/tdelta);
    if (ioff>0) ia=ioff; else ib=-ioff;
    cb.tdif -= (double)ioff*tdelta;
    if (Math.abs(cb.tdif)<1e-12) cb.tdif = 0;

    // calculate overlap and muber to average
    iover = Math.min (ca.ipkts*xfer-ia, cb.ipkts*xfer-ib);
    navg = Math.max(0,iover/nfft);

    // clear
    cb.sdif = 0;
    cb.difn = 0;
    cb.stat = (short)navg;
    cb.navg = navg;
    if (navg<=0) return;

    // compute difference
    Fill.S (fbufi,nfft,0.0);
    for (int n=0; n<navg; n++) {
      Convert.bb2ja (ca.bbuf,n*nfft+ia,ftype, fbufa,0,FLOAT, nfft);
      Convert.bb2ja (cb.bbuf,n*nfft+ib,ftype, fbufb,0,FLOAT, nfft);
      Multiply.SSS (fbufa,fwin,fbufa,nfft);
      Multiply.SSS (fbufb,fwin,fbufb,nfft);
      fftf.work (fbufa);
      fftf.work (fbufb);
      Multiply.GCC (fbufa,fbufb,fbuf,nfft/2);
      Add.SSS (fbufi,fbuf,fbufi,nfft);
    }
    if (navg>1) {
      double rnavg = 1.0/navg;
      Scale.SS(fbufi,fbufi, nfft, rnavg);
    }

    // compute the phase fit
    Fill.C (fbufp,nffti,0.0);
    Noop.CC (fbufi,fbufp,nfft/2);
    ffti.work (fbufp);
    Magnitude.CS (fbufp,fbufp,nffti);
    // find the peak
    int imax=0; float fmax=fbufp[0];
    for (int i=1; i<nffti; i++) if (fbufp[i]>fmax) { imax=i; fmax=fbufp[i]; }
    // zero delay at [0], negative delays are wrapped to top half of buffer
    if (imax>=(nffti/2)) imax -= nffti;
    double a0 = Math.sqrt( fbufp[ (imax-1+nffti)%nffti ] );
    double a1 = Math.sqrt( fbufp[ (imax+0+nffti)%nffti ] );
    double a2 = Math.sqrt( fbufp[ (imax+1+nffti)%nffti ] );
    double x = (a2+a0-2*a1);
    if (Math.abs(x)<1e-20) x=1e-20;
    x = ((double)imax-(a2-a0)/(2*x))*(double)nfft/(double)nffti;
    cb.sdif = x*tdelta;
    cb.difn = (int)((cb.sdif+cb.tdif)*1e9);
  }

  private class ChanData {
    int ipkts;
    short chan;
    short stat;
    int difn,navg,count; 
    double time;
    double tdif;
    double sdif;
    double delta = tdelta;
    Time tc = new Time();
    byte[] bbuf = new byte[xfer*npkts*bpa];
    public void setDataBuffer (Data db) {
      Convert.packI (db.buf, 0, chan);
      Convert.packI (db.buf, 2, stat);
      Convert.packL (db.buf, 4, difn);
      Convert.packD (db.buf, 8, time);
      Convert.packD (db.buf, 16, tdif);
      Convert.packD (db.buf, 24, sdif);
    }
    public int addData (Data db, ICEPacket pkt, int xfer) {
      int pcount = pkt.getCount();
      if (ipkts>0 && ++count!=pcount) ipkts=0;
      if (ipkts<npkts) {
        System.arraycopy(db.buf,0, bbuf,ipkts*xfer*bpa, xfer*bpa);
        if (ipkts==0) {
	  int status = pkt.getTC(tc,0.0,tdelta);
	  if (status<0) M.warning ("Bad TimeCode status: "+status);
	}
        ipkts++;
      }
      count = pcount;
      return ipkts;
    }
    public void clear() {
      ipkts=0;
      time=0;
      tdif=0;
    }
    public String toString() {
      return "Chan "+chan+" Pkts "+ipkts+" NAvg "+navg+" Diff "+difn;
    }
  }

}
