/*

  Description:  Rumel supplied common routines for the ICE-PIC modules

  Author:  Bill Owen 

*/
       
/* Clock Syn chip (ICD-2051) Critical Parameters */ 
#define MEG           1000000.0      /* Value of 1 MegaHertz              */ 
#define P_LO                4        /* Low  P counter value              */ 
#define P_HI              130        /* High P counter value              */ 
#define Q_LO                3        /* Low  Q counter value              */ 
#define Q_HI              129        /* High Q counter value              */ 
#define FVCO_LO      40000000.0      /* Low  Fvco limit of Prgrammed Clock*/ 
#define Q_CON_LO       200000.0      /* Low  Q design constraint value    */ 
#define Q_CON_HI      1000000.0      /* High Q design constraint value    */ 
 
/*
  find_pq-This function determines the values of P and Q and checks
        the constraint that 200KHz < Fref/Q < 1MHz 
*/
static int_u4 find_pq (float out, float ref, int_u4 *p, int_u4 *q) 
{ 
     int_u4 workp      = P_LO; 
     int_u4 workq      = Q_LO; 
     float constraint; 
     float new_freq; 
     float diff       = MEG; 
     float hold_diff  = MEG ; 
     int_u4 ppm        = 0; 
 
  /* From the valid values for P and Q determine what    */ 
  /* values should be sent out for clock configuration   */
      while (workp <= P_HI) 
       { 
         workq = Q_LO; 
         while (workq<=Q_HI) 
           { 
             new_freq   = (float)(2.0 * ref * ((float)workp)/((float)workq)); 
             diff       = new_freq - out; 
             if (diff < 0.0)
               diff = (float)(0.0 - diff); 
             constraint = ref/((float)workq); 
             if (diff<hold_diff) 
               {                           /* Constraint #2 */ 
                 if ((Q_CON_LO < constraint) && (Q_CON_HI > constraint)) 
                   { 
                      hold_diff = diff; 
                      *p        = workp; 
                      *q        = workq; 
                      ppm       = (int_u4)( (diff * MEG/out) ); 
                   } 
               } 
             ++workq; 
           } 
         ++workp; 
       } 
 
      return(ppm);      /* Return the clock frequency error in PPM */ 
} 

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&*/
/*--------------------------------------------------------------*/
#define PQMASK        0x0000007f 
#define MAX_CLK_INDEX         14   /* Number of Index frequency Values  */ 
 
static int_u4 calc_synthetic_clkbits (int_4 fref, int_u4 freq, int_u4 *actual) 
{ 
   /* Clock Index frequency Values-Specific to ICD-2051 Prog clock      */ 
    float vco_index[MAX_CLK_INDEX] = {42.5F, 47.5F, 53.5F, 58.5F, 62.5F, 
 68.5F, 69.0F, 82.0F, 87.0F, 92.0F, 92.1F, 105.0F, 115.0F, 120.0F }; 
    int_u4 clk_div     = 0; 
    int_u4 clk_index   = 0; 
    int_u4 p           = 0; 
    int_u4 q           = 0; 
    int_u4 program_out = 0; 
    float f_out;       
    int_u4 ppm_error   = 0; 
  
    f_out = (float) freq;   
       
  /*  This code takes the user desired output frequencies        */ 
  /*  and brings them into the range of 40MHz < Fvco < 120MHz    */ 
  /*  As the output frequency is continually doubled to be       */ 
  /*  brought into range, it keeps track of the divisor.         */ 
  /*  Ex. Fout = 39.5MHz, Double this to 79MHz so that it is     */ 
  /*      brought into range and set the output frequency divisor*/ 
  /*      to be equal to 2.                                      */ 
   
  /* clk_div is initially 0, Continually double the desired  */ 
  /* output frequency until the Fvco is within the range     */ 
  /* of 40 MHz <= Fvco <= 120 MHz.  Each time we double      */ 
  /* our desired output we add 1 to the variable clk_div-this*/ 
  /* value is then sent out in the clock serial config       */ 
  /* bits div is returned by the function & *new_fout is     */ 
  /* changed to return Fvco when it is brought into range    */ 
 
   while (f_out!=0 && f_out<FVCO_LO) { f_out *= 2.0; clk_div++; } 
    
  /* Determine the Fvco index for clock configuration    */ 
  /* After Fvco is returned determine its index among    */ 
  /* the valid ranges in the clock sythesis specs        */ 
  /* These ranges are listed in the array Clk_index[]    */ 
   while (f_out >= (vco_index[clk_index] * MEG)) 
     ++clk_index; 
      
  /* PPM is the clock error introduced in parts per million  */ 
   ppm_error = find_pq(f_out,(float)fref,&p,&q); 
   if (actual!=0) *actual = (int_u4)( (double)freq*(1.0+ppm_error*.000001) );
#if DEBUG
   printf("Synthetic Clock div=%d index=%d P=%d Q=%d err=%d\n", 
  clk_div,clk_index,p,q,ppm_error); 
#endif 
  /* Calculate P=(130 - p)'for clock config information      */ 
  /* Calculate Q=(130 - q)'for clock config information      */ 
     p = (p - 3) & PQMASK; 
     q = (q - 2) & PQMASK; 
 
  /* Place the clock config infor into a int_u4              */ 
  /* program_out = Index (4 bits), ~P (7 bits), R (1 bit),   */ 
  /*               M (3 bits), ~Q (7 bits)--total = 22 bits  */ 
   
   program_out |= q; 
   program_out |= (clk_div   << 7); 
   program_out |= 0x00000400; 
   program_out |= (p         << 11); 
   program_out |= (clk_index << 18); 
    
  /* Leave Lower 10 Bits Alone                   */
  /* 22 Upper bits of 32 Stores Programming Bits */
   program_out = program_out << 10;
   return(program_out); 
} 

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&*/
/*--------------------------------------------------------------*/       
#define FOUTLOWVAL   40000000  /*  40  MHz */    /* PLL Does Not Respond Below 40MHz */   /* 1500000 */ /* 1.5  MHz */
#define FOUTHIVAL   450000000  /* 450  MHz */
#define VCOLOWVAL   300000000  /* 300  MHz */
#define VCOHIVAL   1040000000  /* 1.04 GHz */
#define FINLOWVAL     1500000  /* 1.5  MHz */
#define FINHIVAL    450000000  /* 450  MHz */
#define FINPFDLOVAL   1500000  /* 1.5  Mhz */
#define FINPFDHIVAL 300000000  /* 300  MHz */

int_4 calc_enhancedpll_values(int_u4 freq,int_u4 reffreq,int_u4 maxrate,int_u4 *pllcfgval)
{
    int ii;
    int_u4 ndivcnt    = 1;  /* Input Ref Divide      Counter */
    int_u4 multcnt    = 1;  /* Input Ref Multiply    Counter */ 
    int_u4 cpostcnt   = 1;  /* Post Scale VCO Divide Counter */
    int_u4 desireddif = 10000000;
    int_u4 actualdif  = 0;
    int_u4 holdmcnt   = 1;
    int_u4 holdncnt   = 1;
    int_u4 holdccnt   = 1;
    int_u4 ccnthigh   = 0;
    int_u4 ccntlow    = 0;
    int_u4 ccntbypass = 0;
    int_u4 ccntodd    = 0;
 
    int_u4 holdval    = 0;   
    int_u4 holdrevclo = 0;
    int_u4 holdrevchi = 0;
    int_u4 cdata      = 0;
    int_u4 ccnt       = 0;
    int_u4 pllcfgcnt  = 0;
    int_u4 bitcnt     = 0; 
   
   if (FOUTHIVAL < maxrate)
     maxrate = FOUTHIVAL;

   if ((freq < FOUTLOWVAL) ||  (freq > maxrate))
     {
       printf("ERROR-Programmable Clock Range %d MHZ to %d MHz\n",FOUTLOWVAL/1000000,maxrate/1000000);
       return(-1);
     }  

   if ((reffreq < FINLOWVAL) || (reffreq > FINHIVAL))
     {
       printf("ERROR-Input Reference Frequency Range 1.5MHz to 450MHz\n");
       return(-1);
     }   


  
   ndivcnt = 1;
   while (ndivcnt<2)   /* & BILL - This should be (ndivcnt < 511) But N >1 Adds LOTS Of Jitter */               
     {     
       if ((reffreq/ndivcnt) < FINPFDLOVAL)
         break;
       else if (((reffreq/ndivcnt) >= FINPFDLOVAL) && ((reffreq/ndivcnt) <= FINPFDHIVAL))
         {
            multcnt = 1;
            while (multcnt < 511)     
              {
                if (((reffreq/ndivcnt) * multcnt) > VCOHIVAL)                       
                  break;  
                else if ((((reffreq/ndivcnt) * multcnt) > VCOLOWVAL) && (((reffreq/ndivcnt) * multcnt) < VCOHIVAL))
                  { 
                    for(cpostcnt=1;cpostcnt<512;cpostcnt++)
                      {

                        if ((freq*cpostcnt) == ((reffreq/ndivcnt) * multcnt))
                          {
                             desireddif = 0;
                             holdmcnt   = multcnt;
                             holdncnt   = ndivcnt;
                             holdccnt   = cpostcnt;  
                             break;   
                          }
                        else
                          {
                             if (freq >  (((reffreq/ndivcnt) * multcnt)/cpostcnt))
                               actualdif = freq - (((reffreq/ndivcnt) * multcnt)/cpostcnt);
                             else
                               actualdif = (((reffreq/ndivcnt) * multcnt)/cpostcnt) - freq;

                             if (actualdif < desireddif)
                               {   
                                 desireddif = actualdif;
                                 holdmcnt   = multcnt;
                                 holdncnt   = ndivcnt;
                                 holdccnt   = cpostcnt; 
                               }  
                          }
                      } /* For cpostcnt */
                  }
                if (desireddif == 0)
                  break;           
                ++multcnt;    
              } /* while (multcnt < 512) */   
          }

       if (desireddif == 0)
         break;
       ++ndivcnt;  
     } /* while (ndivcnt < 512) */
         
        
     
   if (desireddif != 0)
     printf("\nWARN: Desired Frequency = %d Actual Frequency = %d\n",freq,((reffreq/holdncnt) * holdmcnt)/holdccnt);
  
   if (holdccnt == 1)
     cdata      = 0x00000400;    /* Set Bypass Bit For Multiply By 1 */ 
   else
     {
       ccnthigh   = holdccnt/2;
       ccntlow    = holdccnt/2;
       ccntbypass = 0;
       ccntodd    = 0; 

       if ((holdccnt%2) == 1)
         {
           ccnthigh   = ccnthigh + 1;
           ccntodd    = 1;
         } 


        /* Bit Flip C Low Count  */
        for (ii=0;ii<8;ii++)
          holdrevclo  = (holdrevclo << 1) | ((ccntlow >> ii) & 0x00000001);
    
        /* Bit Flip C High Count */
        for (ii=0;ii<8;ii++)
          holdrevchi  = (holdrevchi << 1) | ((ccnthigh >> ii) & 0x00000001);

        cdata = (holdrevchi << 10) | (ccntbypass << 9) | (holdrevclo << 1) | ccntodd;
     } 

 /*
 printf(" Programmable PLL M Value = %d\n",holdmcnt);
 printf(" Programmable PLL N Value = %d\n",holdncnt);
 printf(" Programmable PLL C Value = %d\n",holdccnt);
 printf(" Programmable PLL C High Value = %d\n",ccnthigh);
 printf(" Programmable PLL C Low  Value = %d\n",ccntlow);
 */
  for(ii=0;ii<11;ii++)
    *(pllcfgval+ii) = 0;   

  /* Set M Counter Value */ 
  if (holdmcnt == 1)
    {
      *(pllcfgval + 0) = 0x00001000;             /* Set M Count = 0 */
      *(pllcfgval + 1) = 0x00001801;             /* Set Bypass = 1: addr[3..0] = b"0001",parm[2..0] = b"100",data[8..0] = 1 */
    }  
  else
    {
      *(pllcfgval + 0) = 0x00001000 | holdmcnt;  /* Set M Count = val */
      *(pllcfgval + 1) = 0x00001800;             /* Set Bypass = 0; addr[3..0] = b"0001",parm[2..0] = b"100",data[8..0] = 0 */
    }

 /* Set N Counter Value */ 
  if (holdncnt == 1)
    {
      *(pllcfgval + 2) = 0;                      /* Set N Count = 0 */
      *(pllcfgval + 3) = 0x00000801;             /* Set Bypass = 1: addr[3..0] = 0,parm[2..0] = b"100",data[8..0] = 1 */
    }  
  else
    {
      *(pllcfgval + 2) = holdncnt;               /* Set N Count = val */
      *(pllcfgval + 3) = 0x00000800;             /* Set Bypass = 0: addr[3..0] = 0,parm[2..0] = b"100",data[8..0] = 0 */
    }

 /* Set C Hi Counter Value */
  if (ccnthigh == 0)
    {
      *(pllcfgval + 4) = 0x00004000;             /* Set C Count High = 0 */
      *(pllcfgval + 7) = 0x00004801;             /* Set Bypass = 1: addr[3..0] = b"0100),parm[2..0] = b"100",data[8..0] = 1 */
    }  
  else
    {
      *(pllcfgval + 4) = 0x00004000 | ccnthigh;  /* Set C Count High Count = val */
      *(pllcfgval + 7) = 0x00004800;             /* Set Bypass = 0: addr[3..0] = b"0100",parm[2..0] = b"100",data[8..0] = 0 */
    }

 /* Set C Low Counter Value */
  if (ccntlow != ccnthigh)
   {
      *(pllcfgval + 5) = 0x00004200 | ccntlow;  /* Set C Count Low Count = val */
      *(pllcfgval + 6) = 0x00004A01;            /* Set Countodd = 1: addr[3..0] = b"0100",parm[2..0] = b"101",data[8..0] = 1 */
   }
  else
    {
      *(pllcfgval + 5) = 0x00004200 | ccntlow;  /* Set C Count Low Count = val */
      *(pllcfgval + 6) = 0x00004A00;            /* Set Countodd = 0: addr[3..0] = b"0100",parm[2..0] = b"101",data[8..0] = 0 */
   }

 /* Charge Pump Current*/ 

  if (freq > 40000000)
    {  
      *(pllcfgval +  8) = 0x00002000 | 0x00000001;  /* Set Charge Pump Current 12uA  */
      *(pllcfgval +  9) = 0x00002200 | 0x0000000F;  /* Set Resistance 8.5K Ohm       */
      *(pllcfgval + 10) = 0x00002400 | 0x00000003;  /* Loop Filter Capacitance = 5pF */
    }
  else
    {
      *(pllcfgval +  8) = 0x00002000 | 0x00000005;  /* Set Charge Pump Current 12uA  */
      *(pllcfgval +  9) = 0x00002200 | 0x00000003;  /* Set Resistance 8.5K Ohm       */
      *(pllcfgval + 10) = 0x00002400 | 0x00000003;  /* Loop Filter Capacitance = 5pF */
    }
   
  return(0);  /* & BILL */ 

 /* Code For Direct PLL Load Without Going Through Altera Interface */
  if (holdncnt == 1)
   *(pllcfgval + 0) = 0x00000400;                     /* Set Bit Position 10, N Count Bypass */ 
  else
    { 
      /* Reverse N Count Value */
       holdval = 0;
       for (ii=0;ii<9;ii++)
         holdval  = (holdval << 1) | ((holdncnt >> ii) & 0x00000001);  
         
      *(pllcfgval + 0) = (holdval << 11) & 0x0000F800; 
      *(pllcfgval + 1) = (holdval >>  5) & 0x0000000F;
    }

  if (holdmcnt == 1)
    *(pllcfgval + 1) = *(pllcfgval + 1) | 0x00004000; /* Set Bit Position 14, M Count Bypass */   
  else
    {
     /* Reverse M Count Value */
      holdval = 0;
      for (ii=0;ii<9;ii++)
        holdval  = (holdval << 1) | ((holdmcnt >> ii) & 0x00000001);
       
     *(pllcfgval + 1) = *(pllcfgval + 1) | ((holdval << 15) & 0x00008000);
     *(pllcfgval + 2) = (holdval >> 1) & 0x000000FF;
    }
  
  bitcnt    = 8;
  pllcfgcnt = 2;
  for (ii=0;ii<6;ii++)
   {
     ccnt = 0;
     while (ccnt < 18)    
      {
        if (bitcnt > ccnt)
          *(pllcfgval+pllcfgcnt) =  *(pllcfgval+pllcfgcnt) | ((cdata << (bitcnt - ccnt)) & (0x00000001 << bitcnt));
        else
          *(pllcfgval+pllcfgcnt) =  *(pllcfgval+pllcfgcnt) | ((cdata >> (ccnt - bitcnt)) & (0x00000001 << bitcnt)); 
  
         bitcnt = bitcnt + 1;
         ccnt   = ccnt   + 1;    
         if (bitcnt == 16)
           {
             bitcnt = 0;
             pllcfgcnt = pllcfgcnt + 1;
           }
      }
   }
  
  pllcfgval[10] = 0x000021EC;  /* Set Charge Pump Current = 12uA, Resistance = 12K, Capacitance = 5pF */
 
  return 0;
}
