/****************************************************************************/
/*                                                                          */
/*  module_common.c                                                         */
/*                                                                          */
/****************************************************************************/

#include "module_common.h" 


int32_t mod_validate_obj(Module_Config_Obj *mco)
{
  /* We should probably check for a few more items here */
  if (mco->initmagic != MOBJ_INITMAGIC) return (-1);
  return (0);
}


int32_t mod_init_obj(Module_Config_Obj *mco)
{
  int32_t i;

  mco->initmagic     = MOBJ_INITMAGIC;
  mco->dbg           = 0;             
  mco->RT.modid      = 0x00;
  mco->RT.hw_ver     = 0x00;
  mco->RT.sw_ver     = 0x00;
  mco->RT.numparts   = 0;
  mco->RT.modcrc     = 0x00000000;
  mco->RT.cfgclk     = 0;
  mco->RT.cmdkey     = 0;

  for (i=0; i<PT_CFG_OBJ_MAX; i++) {
    mco->part_config[i].part_subID = 0x00;
    mco->part_config[i].part_ID    = 0x0000;
    mco->part_config[i].part_addr  = 0x00;
  }
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    mco->LS.u32[i] = 0x00000000;
  }
  mco->TGT.tgtid = C3_TGT_DEV_NULL;
  for (i=0; i<MOBJ_CMD_QUEUE_MAX; i++) {
    mco->QCMD[i] = cmdbuf_initializer;
  }
  mco->qcmdcnt = 0;
  for (i=0; i<MOBJ_CMD_QUEUE_MAX; i++) {
    mco->QSTA[i] = cmdbuf_initializer;
  }
  mco->qstacnt = 0;
  return (0);
}


int32_t mod_force_init_obj(Module_Config_Obj *mco)
{
  int32_t rtn;
  mco->initmagic = 0x00000000;
  rtn = mod_init_obj(mco);
  return (rtn);
}


Module_Config_Obj *mod_alloc_obj()
{
  int32_t rtn = 0;
  Module_Config_Obj *p = malloc(sizeof(Module_Config_Obj));
  if (p != (Module_Config_Obj *)NULL) {
    rtn = mod_init_obj(p);
    if (rtn != 0) {
      free(p);
      return (NULL);
    }
  }
  return (p);  
}


int32_t mod_free_obj(Module_Config_Obj *mco)
{
  if (mod_validate_obj(mco) == 0) {
    if (mco != (Module_Config_Obj *)NULL) {
      /* We must kill the initmagic - static heap allocation preserves it! */
      mco->initmagic = 0x00000000;
      free(mco);
    }
  }
  else {
    dbgerrormsg("Attempt to free uninitialized module \n");
    return (-1);  
  }
  return (0);
}


/* Updates the local state object to reflect the module state part */
int32_t mod_get_state(Module_Config_Obj *mco)
{
  int32_t rtn;
  int32_t i;
  int32_t cmfunc,cmpin,cmaddr,cmdata;
  uint32_t crc = 0xFFFFFFFF;

  if (mod_validate_obj(mco) != 0) return (-1);

  /* Get module configuration from the module root part and lock CMD queue */
  rtn = getmoduleconfig(mco);
  if (rtn<0) {
    dbgerrormsg("Command Module is Already Locked!!\n");
    return(-1);
  }

  /* Get module state - clear stat queue, issue rg reads, store stat queue */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,CMD_RG_FIFO_RBCNT,0,0xFF);
  cmdcfg_bufclr(mco);
  /* This only works if PT_STATE_MAX_RG < CMD_RG_FIFO_DEPTH !!! */
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RRD,i,0,0xFFFFFFFF);
  }
  cmdcfg_bufclr(mco);
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdata = 0x00000000;
    cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
    mco->LS.i32[i] = cmdata;
  }
  cmdcfg_bufclr(mco);

  /* Get PIN controls into a register for state checks / control */
  cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_PRD,0,0,0xFFFFFFFF);
  cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  cmdcfg_bufclr(mco);
  mco->LS.i32[STATE_RG_PIN] = cmdata;

  /* Update CRC (really checksum, not CRC) register */
  crc = 0xFFFFFFFF;
  mco->LS.i32[STATE_RG_CRC] = crc;
  for(i=0; i<PT_STATE_MAX_RG; i++) {
    crc ^= mco->LS.u32[i];
  }
  mco->LS.i32[STATE_RG_CRC] = crc;

  for (i=0; i<PT_STATE_MAX_RG; i++) {
    dbgprint(mco->dbg,DBG_MO,"GET_STATE-> RG: %8.8X  ST: %8.8X\n",
               i,mco->LS.u32[i]);
  }

  /* Release command lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);
}


/* Updates the module state part to reflect local state object */
int32_t mod_set_state(Module_Config_Obj *mco) 
{
  int32_t rtn;
  int32_t i;
  uint32_t crc = 0xFFFFFFFF;

  if (mod_validate_obj(mco) != 0) return (-1);

  /* Get module configuration from the module root part and lock CMD queue */
  rtn = getmoduleconfig(mco);
  if (rtn<0) {
    dbgerrormsg("Command Module is Already Locked!!\n");
    return(-1);
  }

  /* Set module state - update CRC, update PIN, write to state part rg */
  mco->LS.i32[STATE_RG_FCN]  = 0x00000000;
  crc = 0xFFFFFFFF;
  mco->LS.i32[STATE_RG_CRC] = crc;
  for(i=0; i<PT_STATE_MAX_RG; i++) {
    crc ^= mco->LS.u32[i];
  }
  mco->LS.i32[STATE_RG_CRC] = crc;

  /* Write to STATE part */
  /* This only works if PT_STATE_MAX_RG < CMD_RG_FIFO_DEPTH !!! */
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RWR,i,mco->LS.i32[i],0xFFFFFFFF);
  }
  cmdcfg_bufclr(mco);

  for (i=0; i<PT_STATE_MAX_RG; i++) {
    dbgprint(mco->dbg,DBG_MO,"SET_STATE-> RG: %8.8X  ST: %8.8X\n",
               i,mco->LS.u32[i]);
  }

  /* Release command lock */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);

  return (0);
}

/****************************************************************************/
/*                                                                          */
/*  Module Private routines                                                 */
/*                                                                          */
/****************************************************************************/

int32_t module_validate_obj(Module_Config_Obj *mco)
{
  /* We should probably check for a few more items here */
  if (mco->initmagic != MOBJ_INITMAGIC) return (-1);
  return (0);
}


int32_t module_clear_qcmd(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t i;
  if (module_validate_obj(mco) != 0) return (-1);
  for (i=0; i<MOBJ_CMD_QUEUE_MAX; i++) mco->QCMD[i] = cmdbuf_initializer;
  mco->qcmdcnt = 0;
  return (0);
}


int32_t module_add_qcmd(Module_Config_Obj *mco, CMDBUF *cbuf, uint32_t flags)
{
  if (module_validate_obj(mco) != 0) return (-1);
  if ((mco->qcmdcnt >= 0) && (mco->qcmdcnt < MOBJ_CMD_QUEUE_MAX)) {
    mco->QCMD[mco->qcmdcnt] = *cbuf;
    mco->qcmdcnt += 1;
  }
  else {
    return (-1);
  }
  return (0);
}


int32_t module_clear_qsta(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t i;
  if (module_validate_obj(mco) != 0) return (-1);
  for (i=0; i<MOBJ_CMD_QUEUE_MAX; i++) mco->QSTA[i] = cmdbuf_initializer;
  mco->qstacnt = 0;
  return (0);
}

int32_t module_init_obj(Module_Config_Obj *mco, uint32_t typeid,
                        TGT_Config_Obj *T, uint32_t flags)
{
  int32_t i;
  mco->initmagic     = MOBJ_INITMAGIC;
  mco->mod_typeid    = typeid;
  mco->dbg           = 0;             
  mco->RT.modid      = 0x00;
  mco->RT.hw_ver     = 0x00;
  mco->RT.sw_ver     = 0x00;
  mco->RT.numparts   = 0;
  mco->RT.modcrc     = 0x00000000;
  mco->RT.cfgclk     = 0;
  mco->RT.cmdkey     = 0;

  for (i=0; i<PT_CFG_OBJ_MAX; i++) {
    mco->part_config[i].part_subID = 0x00;
    mco->part_config[i].part_ID    = 0x0000;
    mco->part_config[i].part_addr  = 0x00;
  }

  /* Transfer C3 target configuration */
  mco->TGT.tgtid = C3_TGT_DEV_NULL;
  if (T != (TGT_Config_Obj *)NULL) {
    mco->TGT.tgtid        = T->tgtid;
    mco->TGT.icepic.pdev  = T->icepic.pdev;
    mco->TGT.icepic.mport = T->icepic.mport;
  }

  /* Initialize local command and status queues */
  module_clear_qcmd(mco,0);
  module_clear_qsta(mco,0);

  return (0);
}


int32_t module_dispatch_qcmd(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  if (module_validate_obj(mco) != 0) return (-1);
  switch (mco->TGT.tgtid) {
    case (C3_TGT_DEV_ICEPIC):
      rtn = 0;
      break;

    default:
      rtn = -1;
      dbgerrormsg("Invalid C3 target in module_lock() \n");
  }
  return (rtn);
}


int32_t module_retrieve_qsta(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  if (module_validate_obj(mco) != 0) return (-1);
  switch (mco->TGT.tgtid) {
    case (C3_TGT_DEV_ICEPIC):
      rtn = 0;
      break;

    default:
      rtn = -1;
      dbgerrormsg("Invalid C3 target in module_lock() \n");
  }
  return (rtn);
}

/****************************************************************************/
/*                                                                          */
/*  Module Public API routines                                              */
/*                                                                          */
/****************************************************************************/


Module_Config_Obj *module_alloc_obj(uint32_t typeid, TGT_Config_Obj *T, uint32_t flags)
{
  int32_t rtn = 0;
  Module_Config_Obj *p = (Module_Config_Obj *)malloc(sizeof(Module_Config_Obj));
  if (p == (Module_Config_Obj *)NULL) {
    dbgerrormsg("Allocation of module object failed!!! \n");
  }
  else {
    rtn = module_init_obj(p, typeid, T, flags);
    if (rtn != 0) {
      free(p);
      p = (Module_Config_Obj *)NULL;
    }
  }
  return (p);  
}

int32_t module_free_obj(Module_Config_Obj *mco, uint32_t flags)
{
  if (module_validate_obj(mco) == 0) {
    if (mco != (Module_Config_Obj *)NULL) {
      /* Prevent prior values from Java future static heap allocation */
      mco->initmagic = 0x00000000;
      mco->RT.cmdkey = 0x00000000;
      free(mco);
    }
  }
  else {
    dbgerrormsg("Attempt to free uninitialized module \n");
    return (-1);  
  }
  return (0);
}

int32_t module_lock(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  int32_t i;
  int32_t usleepval;
  int32_t usleeptbl[] = {1000,2000,5000,12000,31000,111000,0};
  switch (mco->TGT.tgtid) {
    case (C3_TGT_DEV_ICEPIC):
      i = 0;
      do {
        rtn = getmoduleconfig(mco);
        usleepval = (rtn != 0) ? (usleeptbl[i]) : (0);
        usleep(usleepval);
        i++;
        dbgprint(mco->dbg,DBG_MO,"module_lock()   status: %8.8X %8.8X \n",
                   mco->RT.cmdkey, rtn);
      } while (usleepval != 0);
      if (rtn != 0) {
        dbgerrormsg("Unable to obtain CMD lock in module_lock() \n");
        rtn = -1;
      }
      break;
    default:
      rtn = -2;
      dbgerrormsg("Invalid C3 target in module_lock() \n");
  }
  return (rtn);
}


int32_t module_unlock(Module_Config_Obj *mco, uint32_t flags)
{
  int32_t rtn = 0;
  switch (mco->TGT.tgtid) {
    case (C3_TGT_DEV_ICEPIC):
      cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,1,0,0xFFFFFFFF);
      dbgprint(mco->dbg,DBG_MO,"module_unlock() status: %8.8X %8.8X \n",
                 mco->RT.cmdkey, rtn);
      if (rtn != 0) {
        dbgerrormsg("Unable to release CMD lock in module_unlock() \n");
        rtn = -1;
      }
      mco->RT.cmdkey = 0x00000000;
      break;
    default:
      rtn = -2;
      dbgerrormsg("Invalid C3 target in module_unlock() \n");
  }
  return (rtn);

}


int32_t module_fcn(Module_Config_Obj *mco, uint32_t flags, UDB *data)
{
  int32_t rtn = 0;
  int32_t lcllock = 0;
  UDB lcldata = udb_initializer;
  UDB *datbuf = (UDB *) NULL;

  /* Debug stuff */
  /* mco->dbg = DBG_MO|DBG_LB; */

  /* Make certain that we have a valid data buffer handle */
  datbuf = (UDB *) &lcldata; 
  if (data != ((UDB *) NULL)) datbuf = data;

  /* Clear the local command and status queues */
  module_clear_qcmd(mco,0);
  module_clear_qsta(mco,0);

  /* Obtain C3 lock if not called with an active lock */
  if (mco->RT.cmdkey == 0) {
    lcllock = 1;
    rtn = module_lock(mco,0);
  }
  if (rtn != 0) {
    dbgerrormsg("Unable to obtain C3 lock in module_fcn \n");
    rtn = -2;
  }

  /* Generate the commands to be sent to the module                       */  
  /*   NOTE: Right now, this is a placeholder that actually does the work */
  /*         In the future, it will only generate the command list        */
  if (rtn == 0) {
    switch (mco->mod_typeid) {
#if !NEWSTUFF
      case (FPGA_MODID_LB2DM3):
        /* module_gen_qcmd_lb2dm3(mco,flags); */
        module_fcn_lb2dm3(mco,flags,datbuf);
        break;
      case (FPGA_MODID_D2AWGM3):
        /* module_gen_qcmd_d2awgm3(mco,flags); */
        /* module_fcn_d2awgm3(mco,flags,datbuf); */
        break;
#endif
      case (FPGA_MODID_D2RF):  
        /* module_gen_qcmd_d2rf(mco,0,datbuf); */
        break;
      case (FPGA_MODID_A2DM18):
        /* module_gen_qcmd_a2dm18(mco,0,datbuf); */
        break;
      default:
        rtn = -3;
        dbgerrormsg("Invalid module TYPEID in module_qcmd_qsta()  \n");
    }
  }

  /* Dispatch the commands to the module using proper transport */
  if (rtn == 0) {
    rtn = module_dispatch_qcmd(mco,0);
    if (rtn != 0) {
      dbgwarnmsg("Unsuccesful command dispatch \n");
      rtn = -4;
    }
  }

  /* Retrieve any status from the module */ 
  if (rtn == 0) {
    rtn = module_retrieve_qsta(mco,0);
    if (rtn != 0) {
      dbgwarnmsg("Unsuccesful command dispatch \n");
      rtn = -5;
    }
  }

  /* Process and decode the return status based on module type */
  if (rtn == 0) {
    switch (mco->mod_typeid) {
      case (FPGA_MODID_LB2DM3):
        /* module_dec_qsta_lb2dm3(mco,0,datbuf); */
        break;
      case (FPGA_MODID_D2AWGM3):
        /* module_dec_qsta_d2awgm3(mco,0,datbuf); */
        break;
      case (FPGA_MODID_D2RF):  
        /* module_dec_qsta_d2rf(mco,0,datbuf); */
        break;
      case (FPGA_MODID_A2DM18):
        /* module_dec_qsta_a2dm18(mco,0,datbuf); */
        break;
      default:
        rtn = -6;
        dbgerrormsg("Invalid module TYPEID in module_qcmd_qsta()  \n");
    }
  }

  /* Finally, release the module lock if set locally */
  if (lcllock == 1) rtn = module_unlock(mco,0);
  if (rtn != 0) {
    dbgwarnmsg("Unsuccesful C3 lock release in module_qcmd_qsta \n");
    rtn = -7;
  }

  return (rtn);
}


int32_t module_get_state(Module_Config_Obj *mco)
{
  int32_t i;
  int32_t cmfunc,cmpin,cmaddr,cmdata;
  uint32_t crc;

  /* Validate module object and command lock */
  if (module_validate_obj(mco) != 0) return (-1);

  /* Get module state - clear stat queue, issue rg reads, store stat queue */
  cmdcfg_ctl(mco,PTID_CMD,0,CFUNC_RWR,CMD_RG_FIFO_RBCNT,0,0xFF);
  cmdcfg_bufclr(mco);
  /* This only works if PT_STATE_MAX_RG < CMD_RG_FIFO_DEPTH !!! */
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RRD,i,0,0xFFFFFFFF);
  }
  cmdcfg_bufclr(mco);
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdata = 0x00000000;
    cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
    mco->LS.i32[i] = cmdata;
  }
  cmdcfg_bufclr(mco);

#ifdef BOGUS
  /* Get PIN controls into a register for state checks / control */
  cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_PRD,0,0,0xFFFFFFFF);
  cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  cmdcfg_bufclr(mco);
  mco->LS.i32[STATE_RG_PIN] = cmdata;
#endif

  /* Update CRC (really checksum, not CRC) register */
  crc = 0xFFFFFFFF;
  mco->LS.i32[STATE_RG_CRC] = crc;
  for(i=0; i<PT_STATE_MAX_RG; i++) {
    crc ^= mco->LS.u32[i];
  }
  mco->LS.i32[STATE_RG_CRC] = crc;

  for (i=0; i<PT_STATE_MAX_RG; i++) {
    dbgprint(mco->dbg,DBG_MO,"GET_STATE-> RG: %3.3d  ST: %8.8X\n",
               i,mco->LS.u32[i]);
  }
  return (0);
}


/* Updates the module state part to reflect local state object */
int32_t module_set_state(Module_Config_Obj *mco) 
{
  int32_t i;
  uint32_t crc;

  /* Valid module object with command lock */
  if (mod_validate_obj(mco) != 0) return (-1);

  /* Set module state - update CRC, update PIN, write to state part rg */
  mco->LS.i32[STATE_RG_FCN]  = 0x00000000;
  crc = 0xFFFFFFFF;
  mco->LS.i32[STATE_RG_CRC] = crc;
  for(i=0; i<PT_STATE_MAX_RG; i++) {
    crc ^= mco->LS.u32[i];
  }
  mco->LS.i32[STATE_RG_CRC] = crc;

  /* Write to STATE part */
  /* This only works if PT_STATE_MAX_RG < CMD_RG_FIFO_DEPTH !!! */
  for (i=0; i<PT_STATE_MAX_RG; i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RWR,i,mco->LS.i32[i],0xFFFFFFFF);
  }
  cmdcfg_bufclr(mco);

  for (i=0; i<PT_STATE_MAX_RG; i++) {
    dbgprint(mco->dbg,DBG_MO,"SET_STATE-> RG: %3.3d  ST: %8.8X\n",
               i,mco->LS.u32[i]);
  }

  return (0);
}
