/*  
   Native network utilites for the ICE libraries

   Address specification format is    local IF:port / MCast Address:port

   send   udp:192.168.201.105:7777/224.1.40.87:7777

   recv   udp:192.168.201.105:7777/224.1.40.87:7777

   need to set System parameters to allow large network buffers for UDP on Linux:
      > sysctl -a | grep net.core
      > sysctl -w net.core.rmem_max=16777216
      > sysctl -w net.core.wmem_max=16777216

*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define NIO_DEBUG 0
#define dprintf if (NIO_DEBUG>=1) printf
#define d2printf if (NIO_DEBUG>=2) printf

#if _LINX

#include <unistd.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <pthread.h>

/* TX buf > 128K causes output packet reordering on newer Intel Cards */
#define MAXSOCKET 1024
#define TXBUFSIZE 0x0020000
#define TXBUFSIZX 0x1000000
#define RXBUFSIZE 0x0020000
#define RXBUFSIZX 0x1000000

#include "icedefs.h"
#include "niolib.h"

#define PT_NONE 0
#define PT_UDP  1
#define PT_TCP  2

int_4 sock2s[MAXSOCKET];
int_4 protocols[MAXSOCKET];
int_4 openflags[MAXSOCKET];
struct sockaddr_in clientaddrs[MAXSOCKET];

int_4 getaddr (int_4 sock, int_4 chan, struct sockaddr_in *sa);
int_4 addr2str (const struct sockaddr_in *sa, char *str, int_4 len);
int_4 parseAddr (const char *url,char *name,int_4 *port,char *oname,int_4 *oport,int_4 proto);
int_4 loadAddr (const char *host, int port, struct sockaddr_in *address);
int_4 needMatch (char *a, int_4 alen, char*b);
int_4 hasMatch (char *a, int_4 alen, char *b, int_4 blen);
int_4 formTable (char *a, int_4 alen, char *b);
void nudelay (int usec);

char tmpstr[80];
char *saddr (const struct sockaddr_in *sa) { 
  addr2str((const struct sockaddr_in *)sa,tmpstr,80);
  return tmpstr; 
}
void perrn (const char *msg) {
  printf("nioErr %s : %s\n",msg,strerror(errno));
}
void perrnu (const char *msg, char *url) {
  printf("nioErr %s on url=%s : %s\n",msg,url,strerror(errno));
}

int_4 nioOpen (char *url, int_4 chans, int_4 flags)
{
  int_4 i,status,reuse=1,unblock=0,sock=-1,sock2=0,net,port,oport,ttl=20;
  unsigned int bufsize,txbufsize,rxbufsize;
  char name[64],oname[64];
  struct sockaddr_in clientaddr,serveraddr;
  struct ip_mreq group;
  socklen_t rlen,len;
  int_4 haslocal = strchr(url,'/')!=NULL;
  int_4 openflag = flags;
  int_4 protocol = (flags&NIO_HTTP)? PT_TCP:PT_UDP;
  protocol = parseAddr (url,name,&port,oname,&oport,protocol);

  if (flags&NIO_THRSLV) port += 10;
  if (flags&NIO_SERVER) { strcpy(name,oname); port=oport; }

  dprintf("nioOpen url=%s chans=%d flags=%x addr=%s port=%d oaddr=%s oport=%d \n",
	url,chans,flags,name,port,oname,oport);
  
  if ((flags&NIO_THRSLV)==0 && (flags&NIO_THREAD)!=0) 
    sock2 = nioOpen (url,chans,flags|NIO_THRSLV);

  loadAddr (name,port,&serveraddr);
  loadAddr (oname,oport,&clientaddr);

  if (protocol==PT_UDP) sock = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  else if (protocol==PT_TCP) sock = socket (AF_INET, SOCK_STREAM, 0);
  if (sock<0) { perrnu("opening socket",url); return -1; }

  if (protocol==PT_UDP) {
    if (flags&NIO_SERVER) {
      unblock = 1; /* unblock the socket */
    }
    else if (flags&NIO_INPUT) {
      unblock = 1; /* if input, unblock the sockets */
      inet_aton(oname,&group.imr_multiaddr);
      inet_aton( name,&group.imr_interface);
      net = group.imr_multiaddr.s_addr & 0xFF;
      if (net>=224 && net<=240) { /* join multicast group ? */
        status = setsockopt (sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, &group, sizeof(group));
        dprintf("Joining multicast group=%s on %s status=%d\n",oname,name,status); 
        if (status<0) { perrnu("joining multicast group",url); return -5; }
        serveraddr.sin_addr.s_addr = group.imr_multiaddr.s_addr; /* pull interface from here strangely enough */
      }
    } else {
      if (sock<=0 || sock>=MAXSOCKET) { printf("Socket number=%d out of range\n",sock); return -5; }
    }
    /* store output address */
    clientaddrs[sock] = clientaddr;
  } 
  sock2s[sock]=sock2;
  openflags[sock]=openflag;
  protocols[sock]=protocol;

  if (protocol==PT_UDP) {
    status = setsockopt (sock, IPPROTO_IP, IP_MULTICAST_TTL, &ttl, sizeof(ttl));
    if (status<0) { perrnu("setting port ttl",url); return -5; }
  }

  if (protocol!=PT_TCP || haslocal!=0) {
    status = bind (sock, (struct sockaddr *)&serveraddr, sizeof(serveraddr));
    if (status<0) { perrnu("binding socket",url); return -4; }
  }

  if (protocol==PT_TCP) {
    status = connect (sock, (struct sockaddr *)&clientaddr, sizeof(clientaddr));
    if (status<0) { perrnu("connecting socket",url); return -4; }
    clientaddrs[0] = clientaddr;
  }

  if (flags&NIO_RMIF) unblock=2;
  if (flags&NIO_HTTP) unblock=2;

  status = ioctl (sock, FIONBIO, &unblock);
  if (status<0) { perrnu("unblocking socket",url); return -2; }

  len = sizeof(bufsize);
  status = setsockopt (sock, SOL_SOCKET, SO_REUSEADDR, &reuse, len);
  if (status<0) { perrnu("setting port reuse",url); return -3; }

  if (flags&NIO_RMIF) goto NOBUFRESIZE;
  if (flags&NIO_HTTP) goto NOBUFRESIZE;

  rxbufsize=(flags&(NIO_THREAD|NIO_BIGBUF))?RXBUFSIZX:RXBUFSIZE;
  status = setsockopt (sock, SOL_SOCKET, SO_RCVBUF, &rxbufsize, len);
  if (status<0) { perrnu("sizing receive socket buffer",url); return -3; }

  txbufsize=(flags&(NIO_THREAD|NIO_BIGBUF))?TXBUFSIZX:TXBUFSIZE;
  status = setsockopt (sock, SOL_SOCKET, SO_SNDBUF, &txbufsize, len);
  if (status<0) { perrnu("sizing send socket buffer",url); return -3; }

  rlen=len; status = getsockopt (sock, SOL_SOCKET, SO_RCVBUF, &bufsize, &rlen);
  if (bufsize<rxbufsize) printf ("RCV buffer is only %d bytes\n",bufsize);

  rlen=len; status = getsockopt (sock, SOL_SOCKET, SO_SNDBUF, &bufsize, &rlen);
  if (bufsize<txbufsize) printf ("SND buffer is only %d bytes\n",bufsize);

  NOBUFRESIZE:

  return sock;
}

int_4 nioRecon (int_4 sock, int_4 dir)
{
  int_4 status,unblock;
  struct sockaddr_in clientaddr = clientaddrs[0];
  if (dir<0) clientaddr.sin_family = AF_UNSPEC; /* disconnect */
  unblock=0; ioctl (sock, FIONBIO, &unblock);
  status = connect (sock, (struct sockaddr *)&clientaddr, sizeof(clientaddr));
  unblock=2; ioctl (sock, FIONBIO, &unblock);
  if (status<0) perrn("reconnecting socket"); 
  return status;
}

int_4 nioClose (int_4 sock)
{
  int_4 status = close (sock);
  if (sock<MAXSOCKET && sock2s[sock]>0) close (sock2s[sock]);
  dprintf("Closing sock=%d status=%d\n",sock,status); 
  return status;
}

typedef struct {
  int_1  keys[4];       /* barker syncs[2:0] and packet type[3] */
  int_4  count;         /* packet number since start of xfer */
  int_4  elem;          /* data elements in packet */
  int_2  user;          /* user defined field */
  int_2  channel;       /* port channel number */
  int_4  dummy[12];
} ICEPACKETSTR;

typedef struct {
  int_4 sock;
  int_4 chns;
  int_4 ncpa;
  void *haddr;
  int_4 hlen;
  int_4 dlen;
  void* vaddr;
  int_4 bytes;
  int_4 flags;
  int_4 status;
  ICEPACKETSTR ipkt;
} BANKPARAM;

void *nioSendBankRun (void *p) {
  BANKPARAM *bp = (BANKPARAM*)p; 
  bp->status = nioSendBank (bp->sock, bp->chns, bp->ncpa, bp->haddr, bp->hlen, bp->dlen, bp->vaddr, bp->bytes, NIO_THRSLV);
  return NULL;
}
 
int_4 nioSendBank (int_4 sock, int_4 chns, int_4 ncpa, void *haddr, int_4 hlen, int_4 dlen, void* vaddr, int_4 bytes, int_4 flags)
{
  int_4 i,j,ii,ic=0,chn0=0,icpa=0,status=0;
  int_4 nper=bytes/dlen/chns;
  int_4 plen=hlen+dlen;
  ICEPACKETSTR* ipkt = (ICEPACKETSTR*)haddr;
  int_4 *pdata, *hdata=(int_4*)haddr;
  int_4 count0 = ipkt->count;
  int_4 sock2 = (sock<MAXSOCKET)? sock2s[sock]:0;
  pthread_t threadid=0;
  BANKPARAM bp;
  if (sock2>0) {		// thread by 2 - 1st half
    bp.ipkt = *ipkt;
    bp.sock=sock2;  bp.chns=chns; bp.ncpa=ncpa;
    bp.haddr=(void*)(&bp.ipkt); bp.hlen=hlen; bp.dlen=dlen; 
    bp.vaddr=(void*)((char*)vaddr+(bytes/2)); bp.bytes=bytes; bp.flags=-2;
    status = pthread_create(&threadid,NULL,nioSendBankRun,(void*)(&bp));
    chns = chns/2;
  }
  if (flags==NIO_THRSLV) {	// thread by 2 - 2nd half
    chn0 = chns/2;
    ic = chn0/ncpa;
  }
  for (i=chn0; i<chns; i++) {
    ipkt->channel = i+1;
    for (j=0; j<nper; j++) {
      if (i==0 && j==0) {
	pdata = (int_4*)haddr;
	memcpy ((char*)haddr+hlen,vaddr,dlen);
      } else {
	pdata = (int_4*)((char*)vaddr-hlen);
/*	memcpy (pdata,hdata,hlen); */
	for (ii=0; ii<16; ii++) pdata[ii]=hdata[ii];
      }
      status += nioSend (sock, ic, (void*)pdata, plen, 0);
      ipkt->count++;
      vaddr = (char*)vaddr+dlen;
    }
    if (++icpa==ncpa) { icpa=0; ic++; ipkt->count=count0; }
  }
  if (threadid!=0) status = pthread_join(threadid,NULL);

//  printf("Ok chns=%d ncpa=%d h=%d d=%d bytes=%d\n",chns,ncpa,hlen,dlen,bytes);
  return status;
}

int_4 nioSend (int_4 sock, int_4 chan, void* vaddr, int_4 bytes, int_4 flags)
{
  int_4 status,protocol=protocols[sock],openflag=openflags[sock];
  socklen_t addrlen;
  struct sockaddr_in address;
  addrlen = getaddr(sock,chan,&address);
  TRYAGAIN:
  if (protocol==PT_UDP) {
    status = sendto (sock, vaddr, bytes, 0,  (struct sockaddr *)&address, addrlen);
  } else {
    status = write (sock, vaddr, bytes);
  }
  if (status<0 && errno==EAGAIN) { nudelay((flags<0)?-flags:10000); goto TRYAGAIN; }
  d2printf("nioSend sock=%d chan=%d vaddr=%p bytes=%d flags=%x status=%d to=%s\n",sock,chan,vaddr,bytes,flags,status,saddr(&address));
  /* always blocking, so all errors are bad but not catastrophic */
  if (status<0) { perrn("sendto"); return -1; }
  if (flags>0) nudelay (flags);
  return status;
}

int_4 nioRecv (int_4 sock, int_4 chan, void* vaddr, int_4 bytes, int_4 flags)
{
  int_4 status,protocol=protocols[sock],openflag=openflags[sock];
  socklen_t addrlen = sizeof(struct sockaddr_in);
  struct sockaddr_in *address = &clientaddrs[sock];
  if (protocol==PT_UDP) {
   if (openflag&NIO_SERVER)
    status = recvfrom (sock, vaddr, bytes, 0, (struct sockaddr *)address, &addrlen);
   else
    status = recvfrom (sock, vaddr, bytes, 0,  NULL, NULL);
  } else {
    status = read (sock, vaddr, bytes);
  }
  if (status>0) d2printf("nioRecv sock=%d chan=%d vaddr=%p bytes=%d flags=%x status=%d from=%s\n",sock,chan,vaddr,bytes,flags,status,saddr(address));
  /* always unblocking, so check for empty buffer */
  if (status<0) { if (errno==EAGAIN) return 0; perrn("recvfrom"); return -1; }
  return status;
}

#define MAXMSG 8192
/*
 * this routine is called for SET, GET, or MSG.
 * if a GET has aa value, it is polled 10x per second until it matches
 * an internal timeout of 5 sec applies to all responses
 */
int_4 nioMsg (int_4 sock, int_4 chan, int_4 func, char* vaddr, int_4 bytes, int_4 flags)
{
  int_4 status,pfunc,i,j,k,timeout,ml=0; 
  char  bi[MAXMSG],bo[MAXMSG],match[40],bc;
  char *fi=vaddr, *s; 
  FILE *fp = NULL;
  if (strncmp(vaddr+bytes-4,".tbl",4)==0) {  /* input from file */
    fp = fopen(vaddr,"r");
    if (fp==NULL) { printf("Err: Opening file=%s for read\n",vaddr); return -1; }
    fi = bo; /* use as temp input buffer */
    bytes = fread(fi,1,MAXMSG,fp);
    fclose(fp);
  }
  if (func==NIO_GET) ml=needMatch(fi,bytes,match);
  if (flags&NIO_RMIF) {
    pfunc = (func==NIO_SET)? 101 : (func==NIO_GET)? 102 : (func==NIO_RET)? 103 : (func==NIO_ACK)? 104 : 0;
    /* func,flag,info,rep,seq,try,rpt,adj */
    bi[0]=pfunc; bi[1]=0; bi[2]=0; bi[3]='E'; bi[4]=-1; bi[5]=0; bi[6]=0; bi[7]=0;
    j=formTable(fi,bytes,bi+8);
    bytes=j+8;
  }
  else if (flags&NIO_HTTP) {
    s = strchr(fi,'?'); k = (s==NULL)?0:(int)(s-fi)+1;
    if (flags&NIO_RTAB)     strcpy(bi,"GET /Results/Get/?");
    else if (func==NIO_MSG) strcpy(bi,"GET /Message/Send?");
    else if (func==NIO_MSGR) strcpy(bi,"GET /Message/Recv?");
    else if (fi[0]=='/')   sprintf(bi,"GET %.*s/Get/?",k-1,fi);
    else                    strcpy(bi,"GET /Controls/Get/?");
    j = strlen(bi);
    if (func==NIO_SET) bi[j-5]='S'; /* Get to Set */
    j+=formTable(fi+k,bytes-k,bi+j);
    strcpy(bi+j," HTTP/1.1\r\n"); j+=11;
    strcpy(bi+j,"\r\n"); j+=2;
    bytes=j;
  }
  RETRY: i=0; timeout=500;
  status = nioSend (sock,chan, (void*)bi,bytes, 0);
  READMORE:
  status = nioRecv (sock,chan, (void*)(bo+i),MAXMSG-i, 0);

  if (status>0) i+=status; /* add previous reads */
  if (status==0 && --timeout>0) { nudelay(10000); goto READMORE; }
  if (status>0 && bo[i-1]!='}') goto READMORE;
  if (i<=0);
  else if (flags&NIO_RMIF) { bc=bo[0];
    if (bo[4]>=0) { bo[0]=123; nioSend (sock,chan, (void*)bo,8, 0); } /* send RECEIPT */
    strcpy(vaddr,"Rmif:"); 
    strcpy(vaddr+5, (bc==101)?"SET":(bc==102)?"GET":(bc==103)?"RET":(bc==104)?"ACK":"UNK" );
    if (i>8) { strncpy(vaddr+8,bo+8,i-8); vaddr[i]=0; }
    status=i;
  }
  else if (flags&NIO_HTTP) {
    for (j=i; i>0 && bo[i]!='?'; i--);
    if (i>0) i-=3; j-=i;
    strncpy(vaddr,bo+i,j); vaddr[j]=0;
    status=j;
  }
  if (ml>0 && hasMatch(vaddr,status,match,ml)<0) { 
    /* no match, try again */
    if (flags&NIO_HTTP) nioRecon(sock,-1);
    usleep(100000);
    if (flags&NIO_HTTP) nioRecon(sock,1);
    goto RETRY;
  }
  return status;
}

int_4 nioMsgRecv (int_4 sock, int_4 *func, char* vaddr, int_4 bytes, int_4 flags)
{
  int_4 status,pfunc,i=0,j; 
  socklen_t addrlen = sizeof(struct sockaddr_in);
  char  bo[MAXMSG];

  if (flags&NIO_RMIF) { 
    status = recvfrom (sock, bo, MAXMSG, 0,  (struct sockaddr *)(clientaddrs+sock), &addrlen);
    if (status<0) { if (errno==EAGAIN) return 0; perrn("recvfrom-msg"); return -1; }
    pfunc=bo[0]; *func=pfunc;
    if (status>8) { status-=8; strncpy(vaddr,bo+8,status); vaddr[status]=0; }
  }
  else if (flags&NIO_HTTP) {
    for (j=i; i>0 && bo[i]!='?'; i--);
    if (i>0) i-=3; j-=i;
    strncpy(vaddr,bo+i,j); vaddr[j]=0;
    status=j;
  }
  return status;
}

int_4 nioMsgSend (int_4 sock, int_4 func, char* vaddr, int_4 bytes, int_4 flags)
{
  int_4 status,pfunc,i,j,k;
  char  bi[MAXMSG], *fi=vaddr, *s; 
  if (flags&NIO_RMIF) {
    pfunc = (func==NIO_SET)? 101 : (func==NIO_GET)? 102 : (func==NIO_RET)? 103 : (func==NIO_ACK)? 104 : 0;
    /* func,flag,info,rep,seq,try,rpt,adj */
    bi[0]=pfunc; bi[1]=0; bi[2]=0; bi[3]='E'; bi[4]=-1; bi[5]=0; bi[6]=0; bi[7]=0; j=8;
    j+=formTable(fi,bytes,bi+j);
    bytes=j;
    status = sendto (sock, bi, bytes, 0,  (struct sockaddr *)(clientaddrs+sock), sizeof(struct sockaddr));
  }
  else if (flags&NIO_HTTP) {
    s = strchr(fi,'?'); k = (s==NULL)?0:(int)(s-fi)+1;
    if (flags&NIO_RTAB)     strcpy(bi,"GET /Results/Get/?");
    else if (func==NIO_MSG) strcpy(bi,"GET /Message/Send?");
    else if (func==NIO_MSGR) strcpy(bi,"GET /Message/Recv?");
    else if (fi[0]=='/')   sprintf(bi,"GET %.*s/Get/?",k-1,fi);
    else                    strcpy(bi,"GET /Controls/Get/?");
    j = strlen(bi);
    if (func==NIO_SET) bi[j-5]='S'; /* Get to Set */
    j+=formTable(fi+k,bytes-k,bi+j);
    strcpy(bi+j," HTTP/1.1\r\n"); j+=11;
    strcpy(bi+j,"\r\n"); j+=2;
    bytes=j;
  }
  return status;
}

int_4 formTable (char *a, int_4 alen, char *b) {
  int_4 i=0,j=0;
  if (a[i]!='{') b[j++]='{';
  for (; i<alen; i++) {
    if (a[i]==10) { if (b[j-1]!='{') b[j++]=','; } /* convert LF to comma */
    else if (a[i]==',' && b[j-1]==','); /* no duplicate commas */
    else b[j++]=a[i];
  }
  if (b[j-1]!='}') b[j++]='}';
  return j;
}

int_4 parseAddr (const char *url,char *name,int_4 *port,char *oname,int_4 *oport,int_4 proto)
{
  int_4 i,ls,lc; char *u,*s,*c;
  u = (char*)url;
  /* special case for hardware alias */
  if (strncasecmp(u,"NIC",3)==0 || strncasecmp(u,"ICENIC",6)==0 || strncasecmp(u,"ICESRV",6)==0) { 
    s=strstr(u,"ADDR="); c=strchr(s,',');
    u=s+5; c[0]=0;
  }
       if (strncasecmp(u,"udp:",4)==0) { proto=PT_UDP; u+=4; }
  else if (strncasecmp(u,"tcp:",4)==0) { proto=PT_TCP; u+=4; }
  else if (strncasecmp(u,"http:",5)==0) { proto=PT_TCP; u+=5; }
  s = strchr(u,'/'); ls = (s==NULL)?0:(int)(s-u);
  c = strchr(u,':'); lc = (c==NULL)?0:(int)(c-u);
  if (ls==0) {
    gethostname(name,64);
    *port = 7000;
  } else {
    if (lc==0 || lc>ls) { strncpy(name,u,ls); name[ls]=0; *port=7000; }
    else { strncpy(name,u,lc); name[lc]=0; *port=atoi(c+1); }
    u = s+1;
  }
  c = strchr(u,':'); lc = (c==NULL)?0:(int)(c-u);
  if (lc==0) { strcpy(oname,u); *oport=9000; }
  else { strncpy(oname,u,lc); oname[lc]=0; *oport=atoi(c+1); }

  return proto;
}

int_4 loadAddr (const char *host, int_4 port, struct sockaddr_in *address) 
{
  char name[64];
  struct hostent *hostent;
  hostent = gethostbyname(host);
  if (hostent==NULL || hostent->h_addr==NULL) {
    printf("formAddr(): gethostbyname host=%s returned null\n",host);
    return -1;
  }
  address->sin_family = hostent->h_addrtype;
  address->sin_port = htons(port);
  memcpy(&address->sin_addr, hostent->h_addr, hostent->h_length);
  addr2str(address,name,64);
  dprintf("FormAddr host=%s port=%d address=%s\n",host,port,name); 
  return 0;
}


int_4 addr2str (const struct sockaddr_in *sa, char *str, int_4 lstr) 
{
  int_4 retstrlen = 0;
  const struct in_addr *sin_addr = &sa->sin_addr; /* in network byte-order */
  const int_u1 *addr = (int_u1 *) sin_addr; /* cast to byte array to easily get IP address */
  u_int16_t port = ntohs(sa->sin_port);  /* convert from network byte-order to host */
  retstrlen = (int_4) snprintf(str, (size_t)lstr, "%u.%u.%u.%u:%d", addr[0], addr[1], addr[2], addr[3], port);
  return retstrlen;
}

int_4 getaddr (int_4 sock, int_4 chan, struct sockaddr_in *sa) 
{
  int_4 i, addrlen=sizeof(*sa);
  int_u1 *addr = (int_u1 *)&sa->sin_addr;
  memcpy (sa,clientaddrs+sock,addrlen); 
  if (chan>0) {
    i = addr[3]+chan;
    addr[3] = i&0xFF; 			/* adjust addr for channel number */
    addr[2] += (i>>8);			/* adjust addr subnet for blocks of 256 channels */
  }
  return addrlen;
}

void nudelay (int usec) {
  int ksec,udelay; 
#ifdef XXX__USE_POSIX199309
  struct timespec tv,tw;
  tv.tv_sec = 0;
  tv.tv_nsec = usec*1000;
  while(nanosleep(&tv,&tv));
#else
  struct timeval tv,tw;
  if (usec>=10000) { ksec = usec/10000*10000; usleep(ksec); usec-=ksec; }
  if (usec<=0) return;
  gettimeofday(&tv,0);
  TRYAGAIN:
  gettimeofday(&tw,0);
  udelay = (tw.tv_sec-tv.tv_sec)*1000000 + tw.tv_usec - tv.tv_usec;
  if (udelay < usec) goto TRYAGAIN;
#endif
}

int_4 needMatch (char *a, int_4 alen, char*b) {
  int_4 i,br=0,be=0,bl=0,bt;
  for (i=0; i<alen; i++) {
    if (a[i]=='=') be=i;
    if (a[i]=='{') bl=i;
    if (a[i]=='}') br=i;
  }
  if (br==0) br=alen;
  if (a[be+1]=='"' && a[br-1]=='"') { be++; br--; }
  if (be==0 || be+1==br || a[be+1]==',') return 0;
  bt = br-be-1;
  if (bt>0) { strncpy(b,a+be+1,bt); b[bt]=0; }
  return bt;
}

int_4 hasMatch (char *a, int_4 alen, char *b, int_4 blen) {
  int i,aeqi=0;
  char ca,cb;
  for (i=0; i<alen; i++) {
    ca = a[i];
    if (ca=='=') { if (a[i+1]=='"') i++; aeqi=i+1; continue; }
    if (aeqi==0) continue;
    cb = b[i-aeqi];
    if (ca=='}' || ca==','|| ca=='"' || (i-aeqi)>=blen) return 1;   /* end of shorter string */
    if (ca==cb) continue;       /* case match */
    if (ca>='a'&&ca<='z') ca-=32;
    if (cb>='a'&&cb<='z') cb-=32;
    if (ca!=cb) return -1;      /* anycase nomatch */
  }
  return -1;
}

double nioTime() {
  struct timeval tv;
  gettimeofday(&tv,0);
  return (double)tv.tv_sec + (double)tv.tv_usec*1.e-6;
}

#else

int_4 nioOpen (char *url, int_4 chans, int_4 flags)
{
  printf("nioOpen stub url=%s chans=%d flags=%x\n",url,chans,flags);
  return 1;
}

int_4 nioClose (int_4 sock)
{
  printf("nioClose stub sock=%d\n",sock);
  return 0;
}

int_4 nioSend (int_4 sock, int_4 chan, void* vaddr, int_4 bytes, int_4 flags)
{
  printf("nioSend stub sock=%d chan=%d vaddr=%p bytes=%d flags=%x\n",sock,chan,vaddr,bytes,flags);
  return 0;
}

int_4 nioRecv (int_4 sock, int_4 chan, void* vaddr, int_4 bytes, int_4 flags)
{
  printf("nioRecv stub sock=%d chan=%d vaddr=%p bytes=%d flags=%x\n",sock,chan,vaddr,bytes,flags);
  return 0;
}

int_4 nioMsg (int_4 sock, int_4 chan, int_4 func, char* msg, int_4 bytes, int_4 flags)
{
  printf("nioMsg stub sock=%d chan=%d func=%d msg=%p bytes=%d flags=%x\n",sock,chan,func,msg,bytes,flags);
  return 0;
}

double nioTime() {
  return 0.0;
}


#endif

void* nioAlloc (int_4 bytes)
{
  void* pdata = malloc(bytes);
  return pdata;
}

int_4 nioFree (void* pdata)
{
  free(pdata);
  return 0;
}

#define MAXDATA 1048576
int_4 nioFile (int_4 sock, char* fname, char* aname, int_4 flags)
{
  FILE *fp; char *bo,*bi,*s; double t;
  int_4 i,j,k,lhdr=0,timeout,status,bytes,dir=flags,stats=(dir!=0); 
  int_8 xfer,length=0,offset=0;

  bo = (char*) nioAlloc(MAXDATA); bi=bo;
  dprintf("nioFile sock=%d fname=%s aname=%s flags=%d\n",sock,fname,aname,flags);
  fp = fopen(fname, (dir>0)? "r":"w");
  if (fp==NULL) { printf("nioFile err opening file=%s\n",fname); return -1; }
  sprintf(bi,"GET /Archive/%s%s HTTP/1.1\r\n\r\n",(dir>0)?"Import/":(dir<0)?"Export/":"",aname); bytes=strlen(bi);
  if (dir>0) bytes += fread(bi+bytes,1,512,fp); /* attach BLUE header */
  status = nioSend (sock,0, (void*)bi,bytes, 0);
  i=0; xfer=256; timeout=500;
  READMORE:
  status = nioRecv (sock,0, (void*)(bo+i),(int_4)xfer-i, 0);
  if (status>0) i+=status; /* add previous reads */
  if (i<xfer && --timeout>0) { nudelay(10000); goto READMORE; }
  bo[xfer]=0;
  s = strstr(bo,"Content-Length:");
  if (s!=NULL) length = atol(s+16);
  lhdr = (int_4)(s-bo)+16+4+5;	/* def 5 bytes after content */
  s = strstr(bo,"BLUE");
  if (s!=NULL && (int_4)(s-bo)<=256) lhdr = (int_4)(s-bo);
  t = nioTime();
  dprintf("nioFile status=%d dir=%d len=%ld BLUE at lhdr=%d\n",status,dir,(long int)length,lhdr);
  if (dir<=0 && length>0) {
    i = i-lhdr; memcpy(bo, bo+lhdr, i); /* copy partial file data to beginning of buffer */
    READMOREDATA:
    xfer = MAXDATA;
    if (length-offset<xfer) xfer=length-offset;
    status = nioRecv (sock,0, (void*)(bo+i),(int_4)xfer-i, 0);
    if (status>0) i += status; else timeout--;
    if (i<xfer && timeout>0) { nudelay(10000); goto READMOREDATA; }
    if (i>0) fwrite(bo,1,i,fp);
    offset += i; i=0;
    if (stats) printf("\rnioFile fn=%s sz=%ld off=%ld pc=%d",fname,(long int)length,(long int)offset,(int)(100*offset/length));
    if (status>=0 && offset<length && timeout>0) { timeout=500; goto READMOREDATA; }
  }
  if (dir>0 && length>0) {
    offset+=512; /* already sent header */
    WRITEMOREDATA:
    xfer = MAXDATA;
    if (length-offset<xfer) xfer=length-offset;
    status = fread(bi,1,xfer,fp);
    status = nioSend (sock,0, (void*)bi,xfer, 0);
    if (status>0) offset+=status; 
    if (stats) printf("\rnioFile fn=%s sz=%ld off=%ld pc=%d",fname,(long int)length,(long int)offset,(int)(100*offset/length));
    if (status>=0 && offset<length && timeout>0) { timeout=500; goto WRITEMOREDATA; }
  }
  t = nioTime()-t; if (t<.01) t=.01;
  if (stats) printf(" time=%.1lfs rate=%.1lfMB/s\n",t,length/t/1.0e6);
  dprintf("nioFile close fname=%s at len=%ld\n",fname,length);
  fclose(fp);
  nioFree(bo);
  return 0;
}
