#include "stdio.h"
#include "math.h"

// process 
// 1. read power from FPGA
// 2. convert to log scale (dBm)
// 3. add current gain setting to compute real input power (dBm)
// 4. RSSI reading to verify no overload on input out of band

// format of the AGC table is;
// signal level, attenuator settting, LNA setting, gain

char agctbl [] = {  \
	  0,  54,  4, 11, \
	 -6,  54,  4, 11 , \
	-12,  54,  4, 11, \
	-18,  48,  4, 17, \
	-24,  42,  4, 23, \
	-30,  36,  4, 29, \
	-36,  42,  3, 36, \
	-42,  36,  3, 42, \
	-48,  30,  3, 48, \
	-54,  24,  3, 54, \
	-60,  18,  3, 60, \
	-66,  12,  3, 66, \
	-72,  12,  3, 66, \
	-78,  12,  3, 66, \
	-84,  12,  3, 66, \
	-90,  12,  3, 66 
};


char agcgaintbl [] = {  \
	 0,  54,  4, 12, \
	 6,  54,  4, 12, \
	12,  54,  4, 12, \
	18,  48,  4, 18, \
	24,  42,  4, 24, \
	30,  36,  4, 30, \
	36,  42,  3, 36, \
	42,  36,  3, 42, \
	48,  30,  3, 48, \
	54,  24,  3, 54, \
	60,  18,  3, 60, \
	66,  12,  3, 66, \
	72,  12,  3, 66, \
	78,  12,  3, 66, \
	84,  12,  3, 66, \
	90,  12,  3, 66 
};
//
int_4 headroom = -0;
//#define LNA_GAIN 13
#define LNA_GAIN 12
#define AMP_GAIN 65
#define CHIP_GAIN 78
#define GAIN_CORRECTION 8.5

int_4 reset_minmax(PICSTRUCT *p, int_4 mport) {
  // reset the min, max registers
  rfxd_mmsk( p, mport, FEMTO_RESET, ~FEMTO_MINMAXRES,FEMTO_MINMAXRES);
  rfxd_mmsk( p, mport, FEMTO_RESET, ~FEMTO_MINMAXRES,0);
  return 0;
}

int_4 get_minmax(PICSTRUCT *p, int_4 mport, int_4 * imax, int_4 * imin, int_4 * qmax, int_4 * qmin){
  int_4 imaxhi;
  imaxhi =  rfxd_mrd(p, mport, I_MAXHI) << 24;		// latch data
  *imax  = (imaxhi                           ) | (rfxd_mrd(p, mport, I_MAXLO) << 16);
  *imin  = (rfxd_mrd(p, mport, I_MINHI) << 24) | (rfxd_mrd(p, mport, I_MINLO) << 16);
  *qmax  = (rfxd_mrd(p, mport, Q_MAXHI) << 24) | (rfxd_mrd(p, mport, Q_MAXLO) << 16);
  *qmin  = (rfxd_mrd(p, mport, Q_MINHI) << 24) | (rfxd_mrd(p, mport, Q_MINLO) << 16);
  return 0;
}

int_4 agc_get_lna(PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	return (int_4) l->LNA;
#else
	return (int_4)RFXD_RD(p, mport, LGSSTRUCT, LNA);
#endif
}

int_4 agc_get_attn(PICSTRUCT *p, int_4 mport) {
	int_4 atten;
	atten = 72-rfxd_get_gain(p,mport);
	return atten;
}

int_4 agc_get_saved_pwr(PICSTRUCT *p, int_4 mport) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	return (int_4) l->SAVED_POWER;
#else
	return (int_4)RFXD_RD2(p, mport, LGSSTRUCT, SAVED_POWER);
#endif
}

int_4 agc_set_saved_pwr(PICSTRUCT *p, int_4 mport, int_4 power) {
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->SAVED_POWER = (int_2)power;
	return 0;
#else
	RFXD_WR2(p, mport, LGSSTRUCT, SAVED_POWER, power);
	return 0;
#endif
}

int_4 agc_set_gain (PICSTRUCT *p, int_4 mport, int_4 gain) {
	int ptr, lna, attn, old_lna, old_attn;

	if( (gain > 90) || (gain < 0)) return -1;
	old_lna = agc_get_lna(p, mport);
	old_attn = agc_get_attn(p, mport);		// attenuation
	ptr = gain/6;
	ptr = ptr*4;
	lna = agcgaintbl[ptr+2];
	attn = agcgaintbl[ptr+1];
	// don't adjust radio if params are the same
	if( (old_lna == lna) && (old_attn == attn)) return -2;
	radlna_noupdate(p, mport, lna);		// update structure but not RFIC
	rfxd_set_gain(p, mport, 72-attn);	// this updates both LNA and gain
	return 0;
}

int_4 agc_get_gain (PICSTRUCT *p, int_4 mport) {
	int_4 gain, lna, attn;

	lna = agc_get_lna(p, mport);
	attn = agc_get_attn(p, mport);		// attenuation
	if (lna == 3) {
		gain = LNA_GAIN;
	} else {
		gain = 0;
	}
	gain = gain + AMP_GAIN - attn;
	//v3print("AGC_GAIN- lna: %d attn: %d gain: %d\n", lna, attn,gain); 
	return gain;
} 

/* the following routine adjusts the gain in 6dB steps trying to keep the signal within
   a voltage range. It is currently dependent on the min-max detector which is sampled 
   by the processor over time. Be careful of changing the gain table as the gains
   appear not to be consistant. (The code should change to use values in the tables).
*/

int_4 radagcv (PICSTRUCT *p, int_4 mport, int_4 volts) {
	int_4 i, ptr, gain, lna, attn, prev_gain, current_gain, prev_pwr;
	int inc = 0;
	
	if (volts > 8192) {
		inc = -6;
	} else if (volts < 2048){
		inc = 6;
	}
	if(inc != 0) {
		prev_gain = agc_get_gain(p, mport);
		gain = prev_gain + inc +6;  // compensate for table error
		v3print("RADAGCV- volts: %d prev gain: %d new_gain: %d\n", volts, prev_gain, gain);
		agc_set_gain(p, mport, gain);		// this can fail if at limits	
		current_gain = agc_get_gain(p, mport);
		lna = agc_get_lna(p, mport);
		attn = agc_get_attn(p, mport);		// attenuation
		v3print( "RADAGCV- volts: %d inc: %d attn: %d lna: %d prev gain: %d new_gain: %d\n", volts, inc, attn, lna, prev_gain, current_gain);
	} else {
		v3print( "RADAGCV- no change\n");
	}
	return 0;
}

int_4 radagcp (PICSTRUCT *p, int_4 mport, int_4 pwr) {
	int_4 i, ptr, lna, attn, previous_gain, current_gain, previous_pwr;
	
	previous_pwr = (int_4)agc_get_saved_pwr(p, mport);
	if ( abs(pwr - previous_pwr) < 9 ) {
		//v3print("RADAGCP- Not enough change power\n");
		return -1; 
	}
	previous_gain = agc_get_gain(p, mport);
	lna = 4;
	attn = 72;
	for (i = 0; i < 16; i++) {
		ptr = i * 4;
		if ( (pwr+headroom) < agctbl[ptr]) {
			attn         = agctbl[ptr+1];
			lna          = agctbl[ptr+2];
			//current_gain = agctbl[ptr+3];
		}
	}	
	agc_set_saved_pwr(p, mport, pwr);
	radlna_noupdate(p, mport, lna);		// update structure but not RFIC
	rfxd_set_gain(p, mport, 72-attn);	// wants gain relative to 72
	current_gain = agc_get_gain(p, mport);
	v3print( "RADAGCP- Prev_pwr %d pwr: %d attn: %d lna: %d old_gain: %d new_gain: %d\n", previous_pwr, pwr, attn, lna, previous_gain, current_gain);
	return current_gain;
}

int_4 agc_reset (PICSTRUCT *p, int_4 mport) {
	rfxd_mmsk(p, mport, FEMTO_RESET, ~FEMTO_AGC_RESET, FEMTO_AGC_RESET);
	rfxd_mmsk(p, mport, FEMTO_RESET, ~FEMTO_AGC_RESET, 0);
	return 0;
}

// return raw agc value
int_4 agc_read (PICSTRUCT *p, int_4 mport) {
	int_4 dat_pwr_h, dat_pwr_l, dat_pwr;

	dat_pwr_h = rfxd_mrd(p, mport, FEMTO_PWRHI);
	dat_pwr_l = rfxd_mrd(p, mport, FEMTO_PWRLO);

	dat_pwr = dat_pwr_l + dat_pwr_h * 256;
	//v3print("H: %02x L: %02x PWR: dat_pwr\n", dat_pwr_h, dat_pwr_l, dat_pwr);
	return dat_pwr;
}

// convert voltage input to dBm output reference to 50 ohms
real_4 agc_convert (PICSTRUCT *p, int_4 mport, int_4 adc) {
	real_4 vin, pwr, pwrdBm;
	
	//convert to volts
	vin = adc/32768.;
	pwr = (vin*vin)/50.;
	// log conversion
	pwrdBm = (10*log10(pwr))+30.;
	v3print("AGCCONVERT: AD: %d VOLTS: %f PWR: %f\n", adc, vin, pwrdBm); 	
	return pwrdBm;
}	

int_4 rfxd_agc_get_pwr (PICSTRUCT *p, int_4 mport) {
	//short adc;
	int_4 adc;
	int_4 gain, lna, attn;
	real_4 pwr_in, pwr_out_dBm;
	real_4 pwr_in_vpp, pwr_out_dBm_vpp;
	int_4 imax, imin, qmax, qmin, iavg, qavg, iqavg, ipp, qpp, vpp;
	
	lna = agc_get_lna(p, mport);
	attn = agc_get_attn(p, mport);
	gain = agc_get_gain(p, mport);
	// get input power
	adc = agc_read(p, mport);
	// convert to logarithmic
	pwr_out_dBm = agc_convert(p, mport, adc);
	// add current gain and correction
	pwr_in = pwr_out_dBm - agc_get_gain(p, mport) + GAIN_CORRECTION;
	//reset the agc mechanism
	//agc_reset(p, mport);
	agc_reset(p, mport);
	get_minmax(p, mport, &imax, &imin, &qmax, &qmin);
        imax = imax/65536; imin = imin/65536;
        qmax = qmax/65536; qmin = qmin/65536;
        iavg = (imax + imin)/2;
        qavg = (qmax + qmin)/2;
	iqavg= (abs(iavg) > abs(qavg)) ? (abs(iavg)) : (abs(qavg));
        ipp  = (imax - imin);
        qpp  = (qmax - qmin);
	vpp = (ipp > qpp) ? (ipp) : (qpp);
	// convert to logarithmic
	pwr_out_dBm_vpp = agc_convert(p, mport, vpp);
	// add current gain and correction
	pwr_in_vpp = pwr_out_dBm_vpp - agc_get_gain(p, mport) + GAIN_CORRECTION;
	v3print("ADC: %d PWROUT: %f PWRIN: %f GAIN: %d ATTN: %d LNA: %d PWR_IN_VPP: %f\n", adc, pwr_out_dBm, pwr_in, gain, attn, lna, pwr_in_vpp);	
        v3print("IMAX: %d IMIN: %d IAVG: %d QMAX: %d QMIN: %d QAVG: %d IPP: %d QPP: %d\n", imax, imin, iavg, qmax, qmin, qavg, ipp, qpp);
	reset_minmax(p, mport);
	//rssi(p, mport);
	return pwr_in;
}

int_4 rfxd_agc_update (PICSTRUCT *p, int_4 mport) {
	//short adc;
	int_4 adc;
	int_4 old_gain, new_gain;
	real_4 pwr_out_dBm, pwr_in;
	int_4 imax, imin, qmax, qmin, iavg, qavg, iqavg, ipp, qpp, vpp, ip, qp, vp;
	int_4 adjust =0;
	char agc_onoff;

#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	agc_onoff = l->agc_enable;
#else
	agc_onoff = RFXD_RD(p, mport, LGSSTRUCT, agc_enable);
#endif
	get_minmax(p, mport, &imax, &imin, &qmax, &qmin);
        imax = imax/65536; imin = imin/65536;
        qmax = qmax/65536; qmin = qmin/65536;
        iavg = (imax + imin)/2;
        qavg = (qmax + qmin)/2;
	iqavg= (abs(iavg) > abs(qavg)) ? (abs(iavg)) : (abs(qavg));
        ipp  = (imax - imin);
        qpp  = (qmax - qmin);
	vpp = (ipp > qpp) ? (ipp) : (qpp);
	ip = (abs(imax) > abs(imin)) ? (abs(imax)) : (abs(imin));
	qp = (abs(qmax) > abs(qmin)) ? (abs(qmax)) : (abs(qmin));
	vp = (ip > qp) ? (ip) : (qp);

	old_gain = agc_get_gain(p, mport);
	// get input power
	adc = agc_read(p, mport);
	// if the adc reading is dominated by DC, use the p-p reading
        //if(adc > vpp){
        if(iqavg > adc){
          v3print("SET_AGC- Using vpp for ADC reading, ADC: %d, VPP: %d\n", adc, vpp);
	  adc = vpp/2;
	}
	// convert to logarithmic
	pwr_out_dBm = agc_convert(p, mport, adc);
	// add current gain and correction
	pwr_in = pwr_out_dBm - agc_get_gain(p, mport) + GAIN_CORRECTION;
	// do the agc setting
	if (agc_onoff) {
#if 1
		radagcv(p, mport, vpp/2);
#else
		if (radagcp(p, mport, (int_4)(pwr_in+.5)) >= 0) {
			adjust = 1;
			new_gain = agc_get_gain(p, mport);
			v3print("SET_AGC- ADC: %d PWROUT: %f PWRIN: %f old_gain(78): %d new_gain(78): %d\n", adc, pwr_out_dBm, pwr_in, old_gain, new_gain);
		} else {
			v3print("SET_AGC- NO CHANGE ADC: %d PWROUT: %f PWRIN: %f old_gain(78): %d\n", adc, pwr_out_dBm, pwr_in, old_gain);
		}
#endif
	} else {
		//v3print("AGC DISABLED ADC: %d PWROUT: %f PWRIN: %f old_gain(78): %d\n", adc, pwr_out_dBm, pwr_in, old_gain);
		v3print("AGC DISABLED\n");
	}
	
	// reset the agc mechanism
	agc_reset(p, mport);
        
        v3print("IMAX: %d IMIN: %d IAVG: %d QMAX: %d QMIN: %d QAVG: %d IPP: %d QPP: %d\n", imax, imin, iavg, qmax, qmin, qavg, ipp, qpp);
	reset_minmax(p, mport);
	//rssi(p, mport);
	return 0;
}

int_4 rfxd_agc_set(PICSTRUCT *p, int_4 mport, int_4 onoff, int_4 a)
{
#if MEMSTRUCT
	LGSSTRUCT *l;
	l = &femto[p->devno][mport];
	l->agc_enable = onoff;
#else
	RFXD_WR(p, mport, LGSSTRUCT, agc_enable, onoff);
#endif
	return 0;
}

// setup constants and reset AGC filter
int_4 rfxd_agc_init (PICSTRUCT *p, int_4 mport, int_4 alpha, int_4 beta) {
	rfxd_mwr(p, mport, FEMTO_ALPHA, alpha);
	rfxd_mwr(p, mport, FEMTO_BETA, beta);
	agc_set_saved_pwr(p, mport, -200);
	agc_reset(p, mport);
	reset_minmax(p, mport);
	return 0;
}

