
/****************************************************************************/
/*                                                                          */
/*   A2DM1X module interface routines                                       */
/*                                                                          */
/*   Note: all jpmem access must be to port 2 as this dual site module      */
/*         is only controlled from the B sides JTAG connection              */
/*                                                                          */
/****************************************************************************/

#include "iomlib_d2rf.h" 

static int_4 d2rf_set_trx_rate(Module_Config_Obj *module_cfg, real_8 tx_rate, int_4 rx, int_4 state);
/* static int_4 ad9361_config(Module_Config_Obj *module_cfg); */
static int_4 ad9361_config(Module_Config_Obj *module_cfg, int_4 dir, real_8 samprate, real_8 rxlo, real_8 txlo, int_4 bandwidth, int_4 tx_attn);
static int_4 d2rf_si5338(Module_Config_Obj *mco, real_8 rate, int_4 extref);
static int_4 d2rf_set_rffreq(Module_Config_Obj *module_cfg, real_8 rf_freq, int_4 rx, int_4 state, int_4 runcal);
static int_4 d2rf_set_tx_attn(Module_Config_Obj *module_cfg, int_4 rf_atten, int_4 state);
static int_4 ad9361_tracking(Module_Config_Obj *module_cfg, int_u4 bbdc_track, int_u4 rfdc_track, int_u4 rxquad_track);
static int_4 d2rf_tx_quad_calib(Module_Config_Obj *module_cfg, int_4 tx_attn, int_4 txq_attn);
static int_4 d2rf_get_txlo(Module_Config_Obj *module_cfg, real_8 *txlo);

int_4 d2rf_dbgsetup(Module_Config_Obj *mco, int_4 dbgsel, int_4 trigsel, int_4 datsel)
{
  int_4 regword;

  regword = ((trigsel&0xF)<<8) | ((datsel&0xF)<<4);

  cmdcfg_ctl(mco,BAD_MEM_ID,dbgsel,PFUNC_PWR,0,regword|0x01,0xFFFF);
  cmdcfg_ctl(mco,BAD_MEM_ID,dbgsel,PFUNC_PWR,0,regword,0xFFFF);

  return(0);
}

int_4 d2rf_get_dbg(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  int_4 i,k,l;
  int_4 memdat;
  int_4 dbgsel,dbgrst,trigsel,datasel,mdepth;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 CMDDEPTH = 64;
  FILE *file0,*file1;
 
  int_4 dbgSig_I[250],dbgSig_Q[250];
  union dbgMem dbgArray[500];

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  dbgrst  = findintflagdef("DBGRST",p->config,1);
  dbgsel  = findintflagdef("DBGSEL",p->config,0);
  trigsel = findintflagdef("TRIGSEL",p->config,0);
  datasel = findintflagdef("DATASEL",p->config,0);
  mdepth  = findintflagdef("DBGDEPTH",p->config,500);

  if(mdepth>500) mdepth=500;

/* printf("JJB::: Debug sel=%d, mdepth=%d\n",dbgsel,mdepth); */

  if(dbgrst) {
    d2rf_dbgsetup(D2RF_module, dbgsel, trigsel, datasel);
    cmdcfg_delay(D2RF_module,2000);
  }

  cmdcfg_ctl(D2RF_module,BAD_MEM_ID,dbgsel,PFUNC_PRD,0,0,0xFFFFFFFF);
  k = 0;
  for(i=0;i<mdepth;i++) {
    if(i%CMDDEPTH == CMDDEPTH-1) {
      /* printf("JJB:: clearing command buffer, %d...\n",i); */
      cmdcfg_bufclr(D2RF_module);
    }
    cmdcfg_ctl(D2RF_module,BAD_MEM_ID,dbgsel,PFUNC_RRD,i,0,0xFFFFFFFF);
    k+=2;
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  cmdcfg_stat(D2RF_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
/* printf("JJB::: MEM Pin Stat = 0x%x\n",stat_data); */
  file0 = fopen("d2rf_dbgI","wb");
  file1 = fopen("d2rf_dbgQ","wb");
  k = 0;
  l = 0;
  for(i=0;i<mdepth;i+=2) {
    cmdcfg_stat(D2RF_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
    dbgArray[k].dbgWord[0] = stat_data;
    dbgSig_I[l] = dbgArray[k].dbgSig.datI;
    dbgSig_Q[l++] = dbgArray[k].dbgSig.datQ;

    cmdcfg_stat(D2RF_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
    dbgArray[k++].dbgWord[1] = stat_data;

    /*fprintf(file,"MEMDAT:: %d = 0x%0.4x, 0x%0.4x, 0x%0.8x\n",k-1,dbgArray[k-1].dbgSig.datI,dbgArray[k-1].dbgSig.datQ,dbgArray[k-1].dbgSig.dummy);*/
  }

  fwrite(&dbgSig_I,sizeof(int_4),250,file0);
  fwrite(&dbgSig_Q,sizeof(int_4),250,file1);

  fclose(file0);
  fclose(file1);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  /* Check if Session is still locked */
  return (0);
}

real_8 d2rf_get_BBPLL(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  real_8 ref_clk = (real_8)REFCLK_FREQ;
  real_8 bbpll_mod = 2088960;
  real_8 bbpll_freq;
  int_4 stat_func,stat_pin,stat_addr;
  int_4 f3,f2,f1,i1;

  real_8 bbpll_frac = 0.0;
  real_8 bbpll_int  = 0.0;

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

/* Determine BBPLL Clock Frequency */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_FRAC_BB_FREQ_WORD_1,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_FRAC_BB_FREQ_WORD_2,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_FRAC_BB_FREQ_WORD_3,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_INTEGER_BB_FREQ_WORD,0,0xFF);
  cmdcfg_bufclr(module_cfg);

  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f3);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f2);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i1);

  bbpll_frac = (real_8)((f3<<16)|(f2<<8)|f1);
  bbpll_int = (real_8)i1;

  bbpll_freq = ref_clk*bbpll_int + (ref_clk*bbpll_frac)/bbpll_mod;

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return(bbpll_freq);
}

real_8 d2rf_get_freq(PICSTRUCT *p, int_4 mport)
{
  int_4 txlo_div,f1,f2,f3,i1,i2;
  real_8 txlo_frac,txlo_int,txlo_freq;
  int_4 stat_func,stat_pin,stat_addr;
  real_8 ref_clk = (real_8)REFCLK_FREQ;
  real_8 txlo_mod = 8388593;

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(D2RF_module, 0);
  if( rtn < 0 ) {
    printf("ERROR: GET TX RATE: Command Module Lock Failed!!\n");
    #ifdef PICLOCK_BLD 
      pic_lock(p,LOCK_FREE);
    #endif

    return(-1);
   }

/* Determine TX LO Frequency */
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RFPLL_DIVIDERS,0,0xF0);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_2,0,0x7F);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_1,0,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_0,0,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_1,0,0x07);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_0,0,0xFF);
  cmdcfg_bufclr(D2RF_module);

  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&txlo_div);
  txlo_div = txlo_div>>4;
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f3);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f2);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f1);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i1);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i2);

  txlo_frac = (real_8)((f3<<16)|(f2<<8)|f1);
  txlo_int = (real_8)((i1<<8)|i2);
  txlo_freq = (ref_clk*txlo_int + (ref_clk*txlo_frac)/txlo_mod)/(1<<(txlo_div+1));
  if(p->verbose==3) printf("Current TXLO = %f, int=%f, frac=%f\n",txlo_freq,txlo_int,txlo_frac);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return(txlo_freq);
}

real_8 d2rf_get_txrate(PICSTRUCT *p, int_4 mport)
{
  real_8 bbpll_rate,rate;
  int_4 tx_filters,tx_fir_int;
  int_4 tx_div,txclk_div;
  int_4 stat_func, stat_pin, stat_addr;

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(D2RF_module, 0);
  if( rtn < 0 ) {
    printf("ERROR: GET TX RATE: Command Module Lock Failed!!\n");
    #ifdef PICLOCK_BLD 
      pic_lock(p,LOCK_FREE);
    #endif

    return(-1);
   }

  bbpll_rate = d2rf_get_BBPLL(D2RF_module);

  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_ENABLE_FILTER_CTRL,0,0x3F);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_BBPLL,0,0x0F);
  cmdcfg_bufclr(D2RF_module);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&tx_filters);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&tx_div);

  if     (tx_filters==0x1C) tx_fir_int = 3;
  else if(tx_filters==0x0C) tx_fir_int = 2;
  else if(tx_filters==0x04) tx_fir_int = 1;
  else if(tx_filters==0x00) tx_fir_int = 0;

  if(tx_div>7) txclk_div = tx_div+1;
  else txclk_div = tx_div;
  
  rate = bbpll_rate/(real_8)(pow(2,tx_fir_int+txclk_div))/2;

  if(p->verbose==3) printf("Current TX Rate = %f, tx_filters=0x%x, tx_div=0x%x, tx_fir_int=%d, txclk_div=%d\n",rate, tx_filters, tx_div, tx_fir_int, txclk_div);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return(rate);
}

int_4 d2rf_get_state(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 ensm_config;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_STATE,0x00,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

  if(p->verbose==3) printf("AD9361 State = 0x%x\n",stat_data);

  return(0);
}

int_4 d2rf_set_state(Module_Config_Obj *module_cfg, int_4 state)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 ensm_config;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  if(state){
/* Go to FDD State */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ENSM_CONFIG_1,0x28,0xFF);
  }else{
/* Go to ALERT State */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_STATE,0x00,0xFF);
    cmdcfg_bufclr(module_cfg);
    cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

    ensm_config = stat_data&0x9F;
    ensm_config |= 0x05;
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ENSM_CONFIG_1,ensm_config,0xFF);
  }

  return(0);
}
#ifdef BOGUS
int_4 d2rf_debug(D2RF_State_Obj *DRFST)
{ 
  /* printf("You are now debugging the D2RF...\n"); */
  return (0);
}
#endif

int_4 d2rf_reset(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;


  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x05,0xFF);
  cmdcfg_delay(D2RF_module,500);
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_delay(D2RF_module,500);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  /* Check if Session is still locked */
  return (0);
}

int_4 d2rf_dreset(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  /* Disable TX/RX Chains */
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x01,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_SYNTH_PD_OVERRIDE,0x1F,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_SYNTH_PD_OVERRIDE,0x1F,0xFF);

  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0x01,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x01,0xFF);
  cmdcfg_delay(D2RF_module,500);
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_delay(D2RF_module,500);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* test_memory(p,JTAG_PORT_IOOFF+mport,MTEST_IO2MB,0); */

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  /* Check if Session is still locked */
  return (0);
}

int_4 d2rf_loadcoefs (Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, int_4 coeset)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 i;
  int_4 size; /*  = sizeof(hbfir_coefs0)/sizeof(hbfir_coefs0[0]);*/
  real_8 coe_tmp;

  cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,256,0,0xFFFF);

  if(coeset==1)      size = sizeof(hbfir_coefs1)/sizeof(hbfir_coefs1[0]);
  else if(coeset==2) size = sizeof(hbfir_coefs2)/sizeof(hbfir_coefs2[0]);
  else               size = sizeof(hbfir_coefs0)/sizeof(hbfir_coefs0[0]);

  for(i=0;i<size;++i) {
    if(coeset==1) {
      cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,0,0xFFFF);
    } else if(coeset==1) {
      cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,hbfir_coefs1[i]&0xFFFF,0xFFFF);
    } else if(coeset==2) {
      coe_tmp = (real_8)hbfir_coefs2[i];
      coe_tmp = 16383.0 * (coe_tmp/32767.0);
      cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,(int_4)coe_tmp,0xFFFF);
    } else {
      coe_tmp = (real_8)hbfir_coefs0[i];
      cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,(int_4)coe_tmp,0xFFFF);
    }
  }

  cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,256,1,0xFFFF);

  size = sizeof(hbpfir_coefs)/sizeof(hbpfir_coefs[0]);
  for(i=0;i<size;++i) {
    cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,hbpfir_coefs[i],0xFFFF);
  } 

  return(0);
}

int_4 d2rf_loadprogcoefs (Module_Config_Obj *module_cfg, int_4 coeset)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 i;
  int_4 size = sizeof(txprog_coefs)/sizeof(txprog_coefs[0]);

  int_u4 txfirconfig;
  int_u4 txfirC,txfirH,txfirL;
  int_4  coefx;

  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(1);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
  for(i=0;i<size;++i) {
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x060,i,0xFF);
    coefx = txprog_coefs_64[i];
    txfirC = coefx;
    txfirH = (coefx&0xFF00)>>8;
    txfirL = coefx&0x00FF;
/*    txfirC = txprog_coefs[i]; */
/*    txfirH = (txprog_coefs[i]&0xFF00)>>8; */
/*    txfirL = txprog_coefs[i]&0x00FF; */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x061,txfirL,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x062,txfirH,0xFF);
    
    txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(1)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(1);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x064,0,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x064,0,0xFF);
  }

  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(1);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(0)|TX_FIR_GAIN(1);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);

  return(0);
}

/*
int_4 d2rf_loadprogcoefs (Module_Config_Obj *module_cfg, int_4 coeset)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 i;
  int_4 size = sizeof(txprog_coefs)/sizeof(txprog_coefs[0]);

  int_u4 txfirconfig;
  int_u4 txfirC,txfirH,txfirL;

  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
  for(i=0;i<size;++i) {
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x060,i,0xFF);
    txfirC = txprog_coefs[i];
    txfirH = (txprog_coefs[i]&0xFF00)>>8;
    txfirL = txprog_coefs[i]&0x00FF;
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x061,txfirL,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x062,txfirH,0xFF);
    
    txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(1)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(0);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x064,0,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x064,0,0xFF);
  }

  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(1)|TX_FIR_GAIN(0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);
  txfirconfig = TX_FIR_NUMTAPS(3)|TX_FIR_SEL(1)|TX_FIR_WRTX(0)|TX_FIR_TXCLK(0)|TX_FIR_GAIN(0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x065,txfirconfig,0xFF);

  return(0);
}
*/

/* int_4 d2rf_implement(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags, D2RF_State_Obj *DRFST) */
int_4 d2rf_implement(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{

  int_4 rtn,i;
  int_4 extref;
  int_4 refclk = 50e6;
  int_4 delay_cnt; 
  int_4 dio_tr,dio_lfsr,dio_hh,dio_preg;
  int_4 dio_mode,dio_dbg;
  real_8 ratef;
  int_4 coesel;
  int_u4 samp_freq;
  int_u4 rf_bandwidth;
  real_8 rf_freq;
  real_8 samprate,txlo,rxlo;
  int_4 hbby;
  int_4  dis_pa;
  int_4 rf_atten;
  int_u4 lna;
  int_u4 tia;    
  int_u4 mixer;
  int_u4 lpf; 
  int_u4 digital;
  int_4 trx_reg;
  int_4 rxmod;
  int_4 ais_bypass;
  int_4 ais_cf1;
  int_4 ais_cf2;
  int_4 sysreg;
  int_4  mtgooff = 0;
  int_u4 dio_offcnt = 0;
  int_u4 rf_offcnt = 0;
  int_4 hbint,fs4rot;
  int_4 hbint_reg;
  int_4 hb_prescale,hb_maxclip,hb_minclip;
 
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: IMPLEMENT: Command Module Lock Failed!!\n"); return(-1); }

  /* Reset Module */
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_delay(D2RF_module,500);
  cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0,0xFFFF);

  /* Wait for Command Buffer to Execute */
  cmdcfg_bufclr(D2RF_module);

  dio_mode = findintflagdef("D2RFDIO",p->config,0);
  extref = findintflagdef("PREFX",p->config,0);
  if(dio_mode) {
    ratef = (real_8)rate / 1000000.0;
    rate = 40.0e6;  
  }else {
    ratef = (real_8)10.0;
    /* Limit Playback to 40MHz Max */
    if(rate > 61440000) {
      printf("WARNING: Maximum Supported TX Rate is 61.44Msps, currently set to %fMsps!!\n", rate/1e6 );
      rate = 61440000;
    }
  }

  d2rf_si5338(D2RF_module,ratef,extref);

  cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_PE43204_ID,0,PFUNC_PWR,0,3,0xFFFF);

  hb_prescale = findintflagdef("HBPRE",p->config,0x00FF);
  hb_maxclip  = findintflagdef("HBCMAX",p->config,32700)<<8;
  hb_minclip  = findintflagdef("HBCMIN",p->config,-32700)<<8;

/* Set Prescalar Value */
  cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,1,hb_prescale,0xFFFF);

/* Set Clip Value */
  cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,2,hb_maxclip,0xFFFFFFFF);
  cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,3,hb_minclip,0xFFFFFFFF);

  coesel = findintflagdef("HBCOEF",p->config,0);
  d2rf_loadcoefs (D2RF_module, BAD_HBINTFIR_ID, 0, coesel);
#ifdef BOGUS
  d2rf_loadcoefs (D2RF_module, BAD_HBINTFIR_ID, 1, coesel);
#endif

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */ 
/*              Setup the Tranceiver Chip                */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */ 
  /* Reset the Chip */
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_delay(D2RF_module,500);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0,0xFFFF);

  /* Enable TX/RX Chains */
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_SYNTH_PD_OVERRIDE,0x00,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_SYNTH_PD_OVERRIDE,0x00,0xFF);

  /* samp_freq = rate; */
  rf_freq  = finddblflagdef("RFFREQ",p->config,900);
  dis_pa   = findintflagdef("PAOFF",p->config,0);
  rf_atten = findintflagdef("RFATTN",p->config,-1);
  lna      = findintflagdef("G1",p->config,0);
  tia      = findintflagdef("G2",p->config,0);
  mixer    = findintflagdef("G3",p->config,0);
  lpf      = findintflagdef("G4",p->config,0);
  digital  = findintflagdef("G5",p->config,0);
  fs4rot   = findintflagdef("FS4ROT",p->config,0);

  hbint_reg = HBINT_BYROT;
  if(fs4rot) hbint_reg &= ~HBINT_BYROT;

  hbint = 1;
  if(rate <= 25.0e6) hbint = 2;
  hbint = findintflagdef("HBINT",p->config,hbint);

  if(hbint==0) {
    hbint_reg |= (HBINT_BYP1|HBINT_BYP2);
    samprate = ((real_8)(rate))/1e6;
    cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,hbint_reg,0xFFFF);
  }else if(hbint==1) {
    hbint_reg |= (HBINT_BYP2);
    samprate = ((real_8)(2*rate))/1e6;
    cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,hbint_reg,0xFFFF);
  }else {
    samprate = ((real_8)(4*rate))/1e6;
    cmdcfg_ctl(D2RF_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,hbint_reg,0xFFFF);
  }

  if(dio_mode && (samprate > 120.0)) samprate = 120.0;

  rxlo = (dir==-1)? (real_8)rf_freq : 6000.0;
  txlo = (dir==1)? (real_8)rf_freq : 6000.0;

  rf_bandwidth = (int_u4)findintflagdef("RFBW",p->config,(int_u4)(1.1765*samprate*1e6)); 

  ad9361_config(D2RF_module,dir,samprate,rxlo,txlo,(int_4)rf_bandwidth,(int_4)rf_atten);

/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */
/*     Determine if this is a RX or TX Module      */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */
/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */
  trx_reg=0;
  if(p->verbose==3) printf("THIS IS AN %s MODULE!!\n",(dir==1)? "OUTPUT" : "INPUT");
  rxmod = (dir==-1)? 1 : 0;

  /* Power Down TX2 */
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ANALOG_PD_OVERRIDE,0x3E,0xFF);

  if(p->verbose) printf("\nD2RF: Sampling Frequency = %d MHz\n",(int_u4)samprate);
  /* d2rf_set_trx_rate(D2RF_module, ((real_8)(2*samp_freq))/1e6,rxmod, 0); */

  /* if(p->verbose) printf("D2RF: Module %d TX Attenuation = %d dB\n",mport,rf_atten); */
  /* d2rf_set_tx_attn(D2RF_module,rf_atten, 0); */ 

  if(p->verbose) printf("D2RF: Module %d RF LO = %f MHz\n",mport,rf_freq);
  /* d2rf_set_rffreq(D2RF_module,(real_8)6000,1, 0, 0); */

  if(p->verbose) printf("D2RF: Module %d RF Bandwidth = %d MHz\n",mport,rf_bandwidth/(int_u4)1e6);
  /* d2rf_set_bandwidth(D2RF_module,rf_bandwidth,rf_bandwidth, 0,0); */

  /*if(p->verbose==3) printf("D2RF: Module %d RX Gain = %d:%d:%d:%d:%d dB\n",mport,lna,mixer,tia,lpf,digital);
  d2rf_set_rx_gain(D2RF_module,lna,mixer,tia,lpf,digital); */

  d2rf_loadprogcoefs (D2RF_module, 0);

  d2rf_set_state(D2RF_module, 0);

#ifdef BOGUS
  /* Setup the Image Correction Module */
  ais_bypass = 1;
  ais_cf1 = 0;
  ais_cf2 = 0x80000000;
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_PWR,0x00,ais_bypass,0xFFFFFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x00,ais_cf1,0xFFFFFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x01,ais_cf2,0xFFFFFFFF);
#endif

  sysreg = 0;
/*
  if (findintflag("RXSWAPI",p->config) > 0) sysreg |= 1<<8; 
  if (findintflag("RXSWAPQ",p->config) > 0) sysreg |= 1<<9; 
  if (findintflag("RXSWAPF",p->config) > 0) sysreg |= 1<<10; 
  if (findintflag("RSEL",p->config) > 0) sysreg |= 1<<2; 
  if (findintflag("TSEL",p->config) > 0) sysreg |= 1<<3; 
  if (findintflag("MTGO",p->config) > 0) sysreg |= 1<<5; 
*/
  if (findintflag("RXSWAPI",p->config) > 0) sysreg |= 1<<6; 
  if (findintflag("RXSWAPQ",p->config) > 0) sysreg |= 1<<7; 
  if (findintflag("RXSWAPF",p->config) > 0) sysreg |= 1<<8; 
  if (findintflag("RSEL",p->config) > 0) sysreg |= 1<<0; 
  if (findintflag("TSEL",p->config) > 0) sysreg |= 1<<1; 
  if (findintflag("MTGO",p->config) > 0) sysreg |= 1<<3; 
  if (findintflag("HLDINV",p->config) > 0) sysreg |= 1<<9; 
  if (findintflag("HLDRDY",p->config) > 0) sysreg |= 1<<10; 
  if (findintflag("HLDNRDY",p->config) > 0) sysreg |= 1<<11; 

  mtgooff = findintflagdef("MTGOOFF",p->config,0);
  rf_offcnt = (int_u4)(samprate*1e6 - 84 + mtgooff); 

  cmdcfg_ctl(D2RF_module,BAD_MTGO_ID,1,PFUNC_RWR,0,rf_offcnt,0xFFFFFFFF);

  if(dio_mode) {
    dio_dbg = findintflagdef("DIODBG",p->config,0);
    dio_tr = 1;
    dio_lfsr = 0;
    dio_hh = 1;
    dio_preg = (dio_dbg<<6|dio_hh<<5|dio_lfsr<<3|dio_tr<<2);
    dio_offcnt = (int_u4)(199999988 + mtgooff);
    cmdcfg_ctl(D2RF_module,BAD_MTGO_ID,0,PFUNC_RWR,0,dio_offcnt,0xFFFFFFFF);
    cmdcfg_ctl(D2RF_module,BAD_MTGO_ID,0,PFUNC_PWR,0,dio_preg,0xFFFF);
  }
  else if(rxmod){

  /* Choose the RX RF switch and ADC input depending on tuned frequency */
    if(rf_freq < 500){
      cmdcfg_ctl(D2RF_module,BAD_SKY13317_ID,0,PFUNC_PWR,0,0x02,0xFF);
      cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x30,0xFF);
    }
    else if(rf_freq < 3000){
      cmdcfg_ctl(D2RF_module,BAD_SKY13317_ID,0,PFUNC_PWR,0,0x01,0xFF);
      cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x0C,0xFF);
    }
    else if(rf_freq < 6000){
      cmdcfg_ctl(D2RF_module,BAD_SKY13317_ID,0,PFUNC_PWR,0,0x04,0xFF);
      cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x03,0xFF);
    }
    else {
      printf("/* ERROR: Invalid RFFREQ parameter  */ \n");
      cmdcfg_ctl(D2RF_module,BAD_SKY13317_ID,0,PFUNC_PWR,0,0x01,0xFF);
      cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x0C,0xFF);
    }

  /* Put RF Switch into RX Mode  */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);

  /* Disable TX PA  */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x01,0xFF);

  /* Enable RX LNA  */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,0,PFUNC_PWR,0,0x00,0xFF);

  /* Remove RF RX attenuation  */ 
    cmdcfg_ctl(D2RF_module,BAD_PE43204_ID,0,PFUNC_PWR,0,0x00,0xFF);

    d2rf_set_rffreq(D2RF_module,rf_freq,1,0,1);

  /* Setup for Recieve */    
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0x00,0xFF);

  /* Put in FDD State */
    d2rf_set_state(D2RF_module, 1);

  }else{

    dio_dbg = findintflagdef("DIODBG",p->config,0);
    dio_tr = 1;
    dio_preg = (dio_dbg<<6|dio_tr<<2);
    cmdcfg_ctl(D2RF_module,BAD_MTGO_ID,0,PFUNC_PWR,0,dio_preg,0xFFFF);
/*
    dio_dbg = findintflagdef("DIODBG",p->config,0);
    dio_tr = 1;
    dio_lfsr = 1;
    dio_hh = 0;
    dio_preg = (dio_dbg<<6|dio_hh<<5|dio_lfsr<<3|dio_tr<<2);
*/
    cmdcfg_delay(D2RF_module,25000);                             

    d2rf_set_rffreq(D2RF_module,rf_freq,0, 0,1); 

    if(rf_atten >= 0) {
      if(p->verbose) printf("D2RF: Module %d TX Attenuation = %d dB\n",mport,rf_atten);
      d2rf_set_tx_attn(D2RF_module,rf_atten, 0);
    }

  /* Disable RX LNA  */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,0,PFUNC_PWR,0,0x01,0xFF);

  /* Enable TX PA  */
    /* cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,dis_pa,0xFF); */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,1,0xFF);

  /* Put RF Switch into TX Mode  */
    /* cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x00,0xFF); */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);

  /* Put in FDD State */
    d2rf_set_state(D2RF_module, 1);
  }
 
  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return (0);
}
 
/*---------------------------------------------------------------------------*/
/* D2RF - Public Interfaces for Module Initialization, Setup, and Adjust   */
/*---------------------------------------------------------------------------*/
int_4 d2rf_enable(PICSTRUCT *p, int_4 mport, int_4 ena)
{

  /* Module Identification */
  int_4 rtn;
  int_4 tx;
  int_4 trx_reg;
  int_4 sysreg;
  int_4 hh_phase;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  tx = (p->mtype[mport-1]>0)? 1 : 0;
  trx_reg = 0;
  sysreg = 0;
  if (findintflag("RXSWAPI",p->config) > 0) sysreg |= 1<<6; 
  if (findintflag("RXSWAPQ",p->config) > 0) sysreg |= 1<<7; 
  if (findintflag("RXSWAPF",p->config) > 0) sysreg |= 1<<8; 
  if (findintflag("RSEL",p->config) > 0) sysreg |= 1<<0; 
  if (findintflag("TSEL",p->config) > 0) sysreg |= 1<<1; 
  if (findintflag("MTGO",p->config) > 0) sysreg |= 1<<3; 
  if (findintflag("HLDINV",p->config) > 0) sysreg |= 1<<9; 
  if (findintflag("HLDRDY",p->config) > 0) sysreg |= 1<<10; 
  if (findintflag("HLDNRDY",p->config) > 0) sysreg |= 1<<11; 

  hh_phase = findintflagdef("HH_PHASE",p->config,0);
  sysreg |= (hh_phase&0x3)<<4; 

  if(ena == 0){
    cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x01,0xFF);
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_SYNTH_PD_OVERRIDE,0x1F,0xFF);
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_SYNTH_PD_OVERRIDE,0x1F,0xFF);
    /* Disable Digital IO Port */
    cmdcfg_ctl(D2RF_module,BAD_MTGO_ID,0,PFUNC_PWR,0,0x02,0xFFFF);
  }else {
    /* Configure System Register */
    cmdcfg_ctl(D2RF_module,BAD_MOD_ID,0,PFUNC_PWR,0,sysreg,0xFF);

    /* Enable Module */
    cmdcfg_ctl(D2RF_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x02,0xFF);
    if(tx) { 
      /* Setting TX_FRAME to Start Transmission */
      trx_reg = 0x06;
    }
    /* Set TX/RX Switch and Enable TX PA */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x00,0xFF);
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x00,0xFF);

    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,trx_reg,0xFF);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}

int_4 d2rf_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{ 
  int_4 rtn; 
/*   D2RF_State_Obj DRFST; */

if(p->verbose==3) printf("D2RF_SETUP: Calling Reset\n");
  /* Reset module */
  rtn = d2rf_dreset(p,mport);
  if(rtn<0) return(-1);

if(p->verbose==3) printf("D2RF_SETUP: Calling Implement\n");
  /* implement with enable */
  /* d2rf_implement(p,mport,dir,bits,rate,gain,flags,&DRFST); */
  d2rf_implement(p,mport,dir,bits,rate,gain,flags);

  return (0);
}

int_4 d2rf_set_rx_gain(Module_Config_Obj *module_cfg, int_u4 lna, int_u4 mixer, int_u4 tia, int_u4 lpf, int_u4 digital)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;
  int_u4 g1;
  int_u4 g2;
  int_u4 g3;

  g1    = (lna<<5)|mixer;
  g2    = (tia<<5)|lpf;
  g3    = 0x20|digital;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x12,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_ADDRESS,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA1,g1,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA2,g2,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA3,g3,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x16,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x00,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX2_MANUAL_GAIN,0x00,0xFF);

  return(0);
}

int_4 d2rf_load_gt_gain(Module_Config_Obj *module_cfg, int_u4 band)
{
  const uint8_t(*tab)[3];
  int_4 index_max = SIZE_FULL_TABLE;
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;
  int_4 i;

  tab = &full_gain_table[band][0];

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x12,0xFF);

  for (i = 0; i < index_max; i++) {
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_ADDRESS,i,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA1,tab[i][0],0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA2,tab[i][1],0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_WRITE_DATA3,tab[i][2],0xFF);
  
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x1E,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_CONFIG,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GAIN_TABLE_READ_DATA1,0x00,0xFF);

  return(0);
}

int_4 d2rf_si5338(Module_Config_Obj *mco, real_8 rate, int_4 extref)
{
  PT_SI5338_REG SIR[SI5338_MAX_CFG_RG];
  PT_SI5338_MSCFG MS;
  int_4 i,rtn;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 reg_data45, reg_data46, reg_data47;
  real_8 rate_act;

  /* Initialize the Si5338 configuration */
  pt_si5338_rgc_default(&SIR[0], FPGA_MODID_D2RF);

  /* Configure internal / external reference (assume internal) */
  if (extref) {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x00)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x00 >> 3)),0);
  }
  else {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x0A)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x0A >> 3)),0);
  }
  pt_si5338_rgc_setbits(&SIR[0],28,0x03,0x00,0);

  /* Configure input to P2_DIVIDER (Zerodly clock reference) */
  pt_si5338_rgc_setbits(&SIR[0],28,0x20,((0x01)&(0x00)),0);
  pt_si5338_rgc_setbits(&SIR[0],30,0x18,((0x03)&(0x00 >> 1)),0);

  /* Configure a divider solution for the MSx outputs */
  MS.dbglvl = 0;     
  MS.inp_ref_frq    = 10.0; /* BAD BAD BAD - don't assume 10MHz ref clock */
  MS.clk_frq_des[0] = 40.0;            /* AD9361 reference      */
  MS.clk_frq_des[1] = MS.inp_ref_frq;  /* Unused output         */
  MS.clk_frq_des[2] = rate;            /* ADC clock             */
  MS.clk_frq_des[3] = MS.inp_ref_frq;  /* Zero delay mode clock */
  MS.clk_frq_des[4] = MS.inp_ref_frq;  /* Unused output         */
  MS.ms_critical    = 3;               /* Zerodly is critical path */
  rtn = pt_si5338_msx_soln(&SIR[0], &MS);
  if (rtn<0) {
    dbgerrormsg("Invalid clock configuration specified. Bad... \n");
    return(-2); 
  }

  /* Disable outputs and pause LOL */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,230,1<<4,1<<4); 
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,241,1<<7,1<<7);

  /* Configure all the registers on the part */
  for(i=0; i<SI5338_MAX_CFG_RG; i++){
    cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,SIR[i].ra,SIR[i].rv,SIR[i].rm);
  }

  /* Cfg PLL for Lock, initiate PLL lock, Restart LOL, Wait 25ms */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,49,0,1<<7);     
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,246,1<<1,1<<1);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,241,0x65,0xFF);
  cmdcfg_delay(mco,25000);                             

  /* Validate Input Clock Status */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,218,0,0xFFFFFFFF);

  /* Confirm PLL Lock */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,218,0,0xFF);

  /* Copy FCAL Values To Active Registers */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,237,0,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,236,0,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,235,0,0xFF);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  /* Check lock status */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  /* printf("Si5338 Clock Status = %x\n",stat_data); */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  /* printf("Si5338 PLL Lock Status = %x\n",stat_data); */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data47 = stat_data & 0x03;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data46 = stat_data & 0xFF;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data45 = stat_data & 0xFF;
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,47,reg_data47,0x03);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,46,reg_data46,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,45,reg_data45,0xFF);

  /* Set PLL To Use FCAL Values */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,49,1<<7,1<<7);

  /* Enable Outputs */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,230,0,1<<4);
  cmdcfg_delay(mco,25000);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return (0);
}


int_4 d2rf_set_gain(PICSTRUCT *p, int_4 mport, int_u4 gain)
{
  int_u4 lna = (gain>>28)&0x0F;
  int_u4 tia = (gain>>24)&0x0F;
  int_u4 mixer = (gain>>16)&0xFF;
  int_u4 lpf = (gain>>8)&0xFF;
  int_u4 digital = gain&0xFF;

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); return(-1); }

if(p->verbose==3) printf("Setting RX Gain: %d:%d:%d:%d:%d\n",lna,tia,mixer,lpf,digital);
  d2rf_set_rx_gain(D2RF_module, lna, mixer, tia, lpf, digital);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}

int_4 d2rf_set_attn(PICSTRUCT *p, int_4 mport, int_4 rf_atten)
{

/* TX attenuation tied to RF frequency */

if(p->verbose==3) printf("D2RF Set Attn\n");

  /* Module Identification */
  int_4 rtn;
  int_4 rx = 0;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 txpa_state;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); return(-1); }

/*  int_4  dis_pa = findintflagdef("PAOFF",p->config,0); */
/*  cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,dis_pa,0xFF); */
  rx = (p->mtype[mport-1]<0)? 1 : 0;

  if(rx == 0) {
    /* Determine if TX PA was enabled */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PRD,0,0,0xFF);
    cmdcfg_bufclr(D2RF_module);
    cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

    txpa_state = stat_data;
 
    /* Disable TX PA  */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x01,0xFF);

    /* Put RF Switch into RX Mode  */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);
  }

  d2rf_set_tx_attn(D2RF_module,rf_atten,1);

  if(rx == 0) {
    /* Put RF Switch into RX Mode  */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x00,0xFF);

    /* Return TX PA State */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,txpa_state,0xFF);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}

#ifdef BOGUS
int_4 d2rf_run_tx_cal(Module_Config_Obj *module_cfg, int_4 txq_attn, real_8 txlo)
{
  int_4 cal_phase, cal_gain;
  int_4 cal_dcI, cal_dcQ;
  int_4 rtn;

  /* Run SSB Calibration with atten = 0 */
  rtn = d2rf_tx_quad_cal2(module_cfg, txlo, 0, 0x01);
  if(rtn<0) return(-1);

  /* Get SSB Calibration results */
  d2rf_get_imcal(module_cfg, &cal_phase, &cal_gain);

  /* Run DC Calibration with atten = 2*rf_atten */
  rtn = d2rf_tx_quad_cal2(module_cfg, txlo, txq_attn, 0x00);
  if(rtn<0) return(-1);

  /* Get DC Calibration results */
  d2rf_get_dccal(module_cfg, &cal_dcI, &cal_dcQ);

  /* Force LO and SSB Calibration results */
  d2rf_tx_quad_cal_force(module_cfg, cal_dcI, cal_dcQ, cal_phase, cal_gain);

  return(0);
}
#endif

int_4 set_tx_attn(Module_Config_Obj *module_cfg, int_4 rf_atten)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 x1;
  int_u4 x0;
  int_u4 rf_attenx;

  rf_attenx = (int_u4)(1e3*rf_atten/250);

  x1 = ((int_u4)rf_attenx>>8) & 0x000000FF;
  x0 = (int_u4)rf_attenx & 0x000000FF;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_ATTN_1,x1,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_ATTN_0,x0,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x075,0xFF,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x076,0x01,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX2_DIG_ATTN,0x40,0xFF);

  return (0);
}

int_4 d2rf_set_tx_attn(Module_Config_Obj *module_cfg, int_4 rf_atten, int_4 state)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 cal_phase,cal_gain,cal_dcI,cal_dcQ;
  real_8 txlo;
  int_u4 x1;
  int_u4 x0;
  int_u4 rf_attenx;
  int_u4 txq_attn;

  rf_attenx = (int_u4)(1e3*rf_atten/250);

/* Disable Transmisson */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0,0xFFFF);

/* Go to ALERT State */
  d2rf_set_state(module_cfg,0);

/* Disable Tracking Control */
  ad9361_tracking(module_cfg,0,0,0);

  x1 = ((int_u4)rf_attenx>>8) & 0x000000FF;
  x0 = (int_u4)rf_attenx & 0x000000FF;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_ATTN_1,x1,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_ATTN_0,x0,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x075,0xFF,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x076,0x01,0xFF);

/* Update Immediately */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX2_DIG_ATTN,0x40,0xFF);

#ifdef BOGUS
  d2rf_get_txlo(module_cfg, &txlo);
  txq_attn = findintflagdef("TXQA",p->config,TXQC_ATTN_DEF);
  printf("Using TXQC Attn = %d\n",txq_attn);

  /* Run SSB Calibration with atten = 0 */
  d2rf_tx_quad_cal2(module_cfg, (txlo/1e6), 0, 0x01);

  /* Get SSB Calibration results */
  d2rf_get_imcal(module_cfg, &cal_phase, &cal_gain);

  /* Run DC Calibration with atten = 2*rf_atten */
  d2rf_tx_quad_cal2(module_cfg, (txlo/1e6), (int_4)(rf_attenx/2), 0x00);

  /* Get DC Calibration results */
  d2rf_get_dccal(module_cfg, &cal_dcI, &cal_dcQ);

  /* Force LO and SSB Calibration results */
  d2rf_tx_quad_cal_force(module_cfg, cal_dcI, cal_dcQ, cal_phase, cal_gain);
#endif

  d2rf_tx_quad_calib(module_cfg, rf_atten,-1);

/* Enable Tracking Control */
  ad9361_tracking(module_cfg,1,1,1);

/* Go to FDD State */
  d2rf_set_state(module_cfg,state);

/* Enable Transmission */
  if(state) cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0x0006,0xFFFF);

  return (0);
}

int_4 d2rf_get_attn(PICSTRUCT *p, int_4 mport)
{
  int_4 txlo_div,f1,f2,f3,i1,i2;
  real_8 txlo_frac,txlo_int,txlo_freq;
  int_4 stat_func,stat_pin,stat_addr;
  int_4 attn,attn0,attn1;

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(D2RF_module, 0);
  if( rtn < 0 ) {
    printf("ERROR: GET TX RATE: Command Module Lock Failed!!\n");
    #ifdef PICLOCK_BLD 
      pic_lock(p,LOCK_FREE);
    #endif

    return(-1);
   }

  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_ATTN_1,0,0xFF);
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_ATTN_0,0,0xFF);
  cmdcfg_bufclr(D2RF_module);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&attn1);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&attn0);

  attn = ((attn1<<8)+attn0)>>2;

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return(attn);
}

int_4 bbpll_calc_rate(int_u8 rf_freq, int_u8 *vco_freq, int_u1 *div, int_u4 *vco_int, int_u4 *vco_frac, int_u1 *fir_interp, int_u1 *clk_div)
{
/* Check for valid BBPLL Range with Dividers... */
  int_u8 ref_clk = (int_u8)REFCLK_FREQ;
  int_u4 i;
  int_u4 min_plldiv, max_plldiv;
  int_u8 min_pll,max_pll;

RECALC_ERROR:
  min_plldiv = 2;
  max_plldiv = 256;
  for(i=3;i>0;--i){
   min_pll = pow(2,i)*(int_u8)rf_freq*min_plldiv;
   max_pll = pow(2,i)*(int_u8)rf_freq*max_plldiv;
   if((min_pll <= MAX_BBPLL_FREQ) && (max_pll >= MIN_BBPLL_FREQ)) goto BBPLL_VALID;
  }

  printf("ERROR: CAN'T FIND VALID BBPLL RANGE - Setting TX Sample Rate to 1MHz!!!\n");
  rf_freq = 1e6; goto RECALC_ERROR;

BBPLL_VALID:
  *fir_interp = i;

if(*fir_interp==3) *fir_interp=2;

  for(i=1;i<=8;++i){
    *vco_freq = (int_u8)rf_freq*pow(2,*fir_interp)*pow(2,i);
    if((*vco_freq <= MAX_BBPLL_FREQ) && (*vco_freq >= MIN_BBPLL_FREQ) && ((int_u4)(*vco_freq/(int_u8)ref_clk) < 255)) break;
  }

  if(i==8){ *clk_div=1; *div = 7; }
  else    { *clk_div=0; *div = i; }
 
/* Determine Integer Word */
  *vco_int = (int_u4)(*vco_freq/(int_u8)ref_clk);

/* Determine Fractional Word */
  *vco_frac = (int_u4)(rint(2088960*((real_8)*vco_freq/(real_8)ref_clk - (real_8)*vco_int)));

  return(0);
}

int_4 rfpll_calc_rate(int_8 rf_freq, int_8 ref_clk, int_8 *vco_freq, int_4 *div, int_4 *vco_int, int_4 *vco_frac)
{
  
/* Determine VCO Divider */
  if(rf_freq < 93.75e6) {
    if(rf_freq<50e6){
      printf("WARNING: Cannot Tune lower than 50MHz, Tuning now to 50MHz\n");
      rf_freq = 50e6;
    }
    *div = 6;
  }
  else if(rf_freq < 187.5e6) *div = 5;
  else if(rf_freq < 375e6)   *div = 4;
  else if(rf_freq < 750e6)   *div = 3;
  else if(rf_freq < 1500e6)  *div = 2;
  else if(rf_freq < 3000e6)  *div = 1;
  else {
    if(rf_freq > 6000e6){
      printf("WARNING: Cannot Tune higher than 6GHz, Tuning now to 6GHz\n");
      rf_freq = 6000e6;
    }
    *div = 0;
  }
 
  *vco_freq = rf_freq * (int_8)pow(2,*div+1);

/* Determine Integer Word */
  *vco_int = (int_4)(*vco_freq/ref_clk);

/* Determine Fractional Word */
  *vco_frac = (int_4)(rint(8388593*((real_8)(*vco_freq)/(real_8)ref_clk - (real_8)(*vco_int))));

  return(0);
}

int_4 d2rf_vco_init(Module_Config_Obj *module_cfg, int_4 vco_freq_mhz, int_4 rx)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 offs = rx ? 0 : 0x40;
  int_u4 tmp;
  int_u1 i = 0;

  int_u4 a1  = AD9361_REG_RX_VCO_OUTPUT;
  int_u4 a2  = AD9361_REG_RX_ALC_VARACTOR;
  int_u4 a3  = AD9361_REG_RX_VCO_BIAS_1;
  int_u4 a4  = AD9361_REG_RX_FORCE_VCO_TUNE_1;
  int_u4 a5  = AD9361_REG_RX_VCO_VARACTOR_CTRL_1;
  int_u4 a6  = AD9361_REG_RX_VCO_CAL_REF;
  int_u4 a7  = AD9361_REG_RX_VCO_VARACTOR_CTRL_0;
  int_u4 a8  = AD9361_REG_RX_CP_CURRENT;
  int_u4 a9  = AD9361_REG_RX_LOOP_FILTER_1;
  int_u4 a10 = AD9361_REG_RX_LOOP_FILTER_2;
  int_u4 a11 = AD9361_REG_RX_LOOP_FILTER_3;

  while (i < SYNTH_LUT_SIZE && rfpll_loop_table[i].VCO_MHz > vco_freq_mhz) i++;
 
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a1+offs),VCO_OUTPUT_LEVEL(rfpll_loop_table[i].VCO_Output_Level)|PORB_VCO_LOGIC,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a2+offs),VCO_VARACTOR(rfpll_loop_table[i].VCO_Varactor),VCO_VARACTOR(~0));
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a3+offs),VCO_BIAS_REF(rfpll_loop_table[i].VCO_Bias_Ref)|VCO_BIAS_TCF(rfpll_loop_table[i].VCO_Bias_Tcf),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a4+offs),VCO_CAL_OFFSET(rfpll_loop_table[i].VCO_Cal_Offset),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a5+offs),VCO_VARACTOR_REFERENCE(rfpll_loop_table[i].VCO_Varactor_Reference),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a6+offs),VCO_CAL_REF_TCF(0),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a7+offs),VCO_VARACTOR_OFFSET(0)|VCO_VARACTOR_REFERENCE_TCF(7),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a8+offs),CHARGE_PUMP_CURRENT(rfpll_loop_table[i].Charge_Pump_Current),CHARGE_PUMP_CURRENT(~0));
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a9+offs),LOOP_FILTER_C2(rfpll_loop_table[i].LF_C2)|LOOP_FILTER_C1(rfpll_loop_table[i].LF_C1),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a10+offs),LOOP_FILTER_R1(rfpll_loop_table[i].LF_R1)|LOOP_FILTER_C3(rfpll_loop_table[i].LF_C3),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a11+offs),LOOP_FILTER_R3(rfpll_loop_table[i].LF_R3),0xFF);

  return(0);
}

int_4 rf_synth_cp_cal(Module_Config_Obj *module_cfg, int_4 rx)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 offs = rx ? 0 : 0x40;
  int_u4 vco_cal_cnt;
  int_4 stat_pin; /* Old, need to remove and change cmdfg_stat to return cmd_key, and part_ID information */
  int_4 stat_func, stat_key, stat_ID, stat_addr, stat_data;

  int_u4 a1 = AD9361_REG_RX_LO_GEN_POWER_MODE;
  int_u4 a2 = AD9361_REG_RX_VCO_LDO;
  int_u4 a3 = AD9361_REG_RX_VCO_PD_OVERRIDES;
  int_u4 a4 = AD9361_REG_RX_CP_CURRENT;
  int_u4 a5 = AD9361_REG_RX_CP_CONFIG;
  int_u4 a6 = AD9361_REG_RX_VCO_CAL;
  int_u4 a7 = AD9361_REG_RX_CAL_STATUS;

  d2rf_get_state(module_cfg);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a1+offs),0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a2+offs),0x0B,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a3+offs),0x02,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a4+offs),0x80,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a5+offs),0x00,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a6+offs),0x8E,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ENSM_CONFIG_2,0x04,0xFF);
  d2rf_set_state(module_cfg,0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ENSM_MODE,0x01,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a5+offs),0x04,0xFF);
  cmdcfg_delay(module_cfg,1000); /* Charge Pump Cal shoud be less than 921.6us */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,(a7+offs),0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("%s CP CAL: 0x%x\n",(rx)? "RX" : "TX",stat_data);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(a5+offs),0x00,0xFF);

  return(0);
}

int_4 d2rf_rx_quad_cal(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 cnt = 500;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x20,0xFF);
RXQUAD_RETRY:
  cmdcfg_delay(module_cfg,25000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("RX QUAD CAL : 0x%x\n",stat_data&0x20);
  if(stat_data & 0x20) {
    if(p->verbose==3) printf("RX QUAD CAL Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: RX QUAD FAILED TO LOCK!!\n"); return(-1); }
    else goto RXQUAD_RETRY;
  }

  return(0);
}

int_4 d2rf_tx_quad_cal_force(Module_Config_Obj *module_cfg, int_4 dcI, int_4 dcQ, int_4 phase, int_4 gain)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

#ifdef BOGUS
  printf("Quad Cal Phase = 0x%x\n",phase);
  printf("Quad Cal Gain = 0x%x\n",gain);
  printf("Quad Cal DCI = 0x%x\n",dcI);
  printf("Quad Cal DCQ = 0x%x\n",dcQ);
#endif

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TXQC_FORCE_BITS,0x05,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_OUT1_PHASE_CORR,phase,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_OUT1_GAIN_CORR,gain,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_OUT1_OFFSET_I,dcI,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX1_OUT1_OFFSET_Q,dcQ,0xFF);

  return(0);
}

int_4 d2rf_get_dccal(Module_Config_Obj *module_cfg, int_4 *dcI, int_4 *dcQ)
{
  int_4 stat_func,stat_pin,stat_addr;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_OUT1_OFFSET_I,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_OUT1_OFFSET_Q,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,dcI);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,dcQ);
  
  return(0);
}

int_4 d2rf_get_imcal(Module_Config_Obj *module_cfg, int_4 *phase, int_4 *gain)
{
  int_4 stat_func,stat_pin,stat_addr;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_OUT1_PHASE_CORR,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX1_OUT1_GAIN_CORR,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,phase);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,gain);
  
  return(0);
}

int_4 d2rf_tx_quad_cal2(Module_Config_Obj *module_cfg, real_8 rf_freq, int_u4 txq_attn, int_4 caltype)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 thresh1 = 1;
  int_u4 thresh2 = 1;
  int_u4 nco_freq;
  int_u4 cnt=0,band,cal_lock;
  const uint8_t(*tab)[3];
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 calmask,calrun;
  int_u4 M,rx_nco_freq,tx_nco_freq;
  real_8 rate = 0.0;
  real_8 bw = 0.0;
  real_8 nco_freq2 = 0.0;
  int_4 index_max = 0;
  int_4 lpf_tia_mask = 0;
  int_4 i=0;
  int_4 nco_phase_start = 0; 
  int_4 nco_phase_stop  = 0; 
  int_4 cal_run_cnt = 0;
  int_4 txq_coff = 0;
  int_4 retrycnt = 0;
  int_u4 timeout;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TXQC_FORCE_BITS,0x00,0xFF);
  /*cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x00,0,0xFFFF);*/
  /*cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x01,0,0xFFFF);*/
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B2,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B3,0x00,0xFF);

  nco_freq = (caltype)? 1 : 1; 
  rx_nco_freq = (nco_freq<<5);
  tx_nco_freq = (nco_freq<<6);
  M = (caltype)? 3 : 0;

  if(rf_freq < 1300.0) band=0;
  else if(rf_freq < 4000.0) band=1;
  else band=2;

  calmask = (caltype)? 0x01 : 0x02;
  calrun = (caltype)? 0x03<<3 : 0x04<<3;

  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,1,0,0xFFFFFFFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  rate = (real_8)stat_data;
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  bw = (real_8)stat_data;
  nco_freq2 = (4*bw/rate) - 1;

  tab = &full_gain_table[band][0];
  index_max = SIZE_FULL_TABLE;
  lpf_tia_mask = 0x3F;
  i=0;
  for (i = 0; i < index_max; i++){
    if ((tab[i][1] & lpf_tia_mask) == 0x20) {
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_FULL_LMT_GAIN,i,0xFF);
      break;
    }
  }

  nco_phase_start = (caltype)? 10 : 15; /*21;*/
  nco_phase_stop  = 24; /*24;*/
  cal_run_cnt = 0;
  txq_coff = nco_phase_start;
  retrycnt = 0;
  do{
    ++cal_run_cnt;
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP1,0x7F,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP2,(tx_nco_freq|0x0F),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(1<<6|calrun|1<<2|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(1<<6|calrun|0<<2|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_NCO_FREQ_PHASE_OFFSET,(rx_nco_freq|txq_coff),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_SETTLE_COUNT,0xF0,0xFF);
    /* cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_SETTLE_COUNT,0x10,0xFF);*/
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_THRES_ACCUM,0x0F,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_COUNT,0xFE,0xFF);
    /* cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_COUNT,0xA0,0xFF);*/
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES1,thresh1,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES2,thresh2,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_LPF_GAIN,0x18,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_CAL_ATTN,txq_attn,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SQUARER_CONFIG,0x00,0xFF);
  
    /* Initiate Cal */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x10,0xFF);
    /* cmdcfg_delay(module_cfg,15000); */
    /* Wait for cal to complete, checking the complete bit */
    timeout = 1000;
    do {
       /* Read 4th bit of cal_control */
       cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,(1<<4),0xFF);
       cmdcfg_bufclr(module_cfg);
       cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
       /* printf("stat_data = %d \n",stat_data); */
       
       /* Delay a little bit */
       cmdcfg_delay(module_cfg,10);
       
    } while (timeout-- && stat_data != 0);  /* if reach timeout # of loops, or cal finished then finish */    
  
    /* Check Cal Done Signal, Status, and Cal Results */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_QUAD_CAL_STATUS,0,0xFF);
    cmdcfg_bufclr(module_cfg);
    cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  
    cal_lock = stat_data & calmask;

    if( (thresh1 < 15) && (thresh2 < 15) && (txq_coff == nco_phase_stop) && (cal_lock != calmask) ) {
      if(caltype) thresh2 += 1;
      else        thresh1 += 1;
      cal_run_cnt = 0;
      txq_coff = nco_phase_start;
    }else if((cal_lock != calmask) && (txq_coff == nco_phase_stop) && (retrycnt <= 1)) {
      cmdcfg_delay(module_cfg,200);
      thresh1 = 1;
      thresh2 = 1;
      cal_run_cnt = 0;
      txq_coff = nco_phase_start;
      if(p->verbose==3) printf("Failed TX Quad Cal - Retrying: %d\n",retrycnt);
      ++retrycnt;
    }else if (cal_lock != calmask) {
      ++txq_coff;
    }
  }while((cal_lock != calmask) && (txq_coff<=nco_phase_stop));

  if(cal_lock != calmask) {
    printf("ERROR: TX Failed to Properly Calibrate!! : 0x%x, 0x%x\n",calmask,stat_data);
    if(p->verbose==3) printf("thres1 = %d, thres2 = %d, atten = %d\n",thresh1,thresh2,txq_attn);
  }else {
    if(p->verbose==3) printf("TX Quad Cal Lock: status = 0x%x, thresh1 = %d, thresh2 = %d, phase offset = %d, tx atten = %d\n",stat_data,thresh1,thresh2,txq_coff,txq_attn);
  }

  return(0);
}

#ifdef BOGUS
int_4 d2rf_tx_quad_cal2(Module_Config_Obj *module_cfg, real_8 rf_freq, int_u4 txq_attn, int_4 caltype)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 thresh1 = 1;
  int_u4 thresh2 = 1;
  int_u4 nco_freq;
  int_u4 cnt=0,band,cal_lock;
  const uint8_t(*tab)[3];
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 calmask,calrun;
  int_u4 M,rx_nco_freq,tx_nco_freq;
  real_8 rate = 0.0;
  real_8 bw = 0.0;
  real_8 nco_freq2 = 0.0;
  int_4 index_max = 0;
  int_4 lpf_tia_mask = 0;
  int_4 i=0;
  int_4 nco_phase_start = 0; 
  int_4 nco_phase_stop  = 0; 
  int_4 cal_run_cnt = 0;
  int_4 txq_coff = 0;
  int_4 retrycnt = 0;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TXQC_FORCE_BITS,0x00,0xFF);
  /*cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x00,0,0xFFFF);*/
  /*cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x01,0,0xFFFF);*/
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B2,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B3,0x00,0xFF);

  nco_freq = (caltype)? 1 : 1; 
  rx_nco_freq = (nco_freq<<5);
  tx_nco_freq = (nco_freq<<6);
  M = (caltype)? 3 : 0;

  if(rf_freq < 1300.0) band=0;
  else if(rf_freq < 4000.0) band=1;
  else band=2;

  calmask = (caltype)? 0x01 : 0x02;
  calrun = (caltype)? 0x03<<3 : 0x04<<3;

  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,1,0,0xFFFFFFFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  rate = (real_8)stat_data;
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  bw = (real_8)stat_data;
  nco_freq2 = (4*bw/rate) - 1;

  tab = &full_gain_table[band][0];
  index_max = SIZE_FULL_TABLE;
  lpf_tia_mask = 0x3F;
  i=0;
  for (i = 0; i < index_max; i++){
    if ((tab[i][1] & lpf_tia_mask) == 0x20) {
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_FULL_LMT_GAIN,i,0xFF);
      break;
    }
  }

  nco_phase_start = (caltype)? 15 : 24; /*21;*/
  nco_phase_stop  = 24; /*24;*/
  cal_run_cnt = 0;
  txq_coff = nco_phase_start;
  retrycnt = 0;
  do{
    ++cal_run_cnt;
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP1,0x7F,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP2,(tx_nco_freq|0x0F),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(1<<6|calrun|1<<2|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(1<<6|calrun|0<<2|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_NCO_FREQ_PHASE_OFFSET,(rx_nco_freq|txq_coff),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_SETTLE_COUNT,0xF0,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_THRES_ACCUM,0x0F,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_COUNT,0xFE,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES1,thresh1,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES2,thresh2,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_LPF_GAIN,0x18,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_CAL_ATTN,txq_attn,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SQUARER_CONFIG,0x00,0xFF);
  
    /* Initiate Cal */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x10,0xFF);
    cmdcfg_delay(module_cfg,15000);
  
    /* Check Cal Done Signal, Status, and Cal Results */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_QUAD_CAL_STATUS,0,0xFF);
    cmdcfg_bufclr(module_cfg);
    cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  
    cal_lock = stat_data & calmask;

    if( (thresh1 < 10) && (thresh2 < 10) && (txq_coff == nco_phase_stop) && (cal_lock != calmask) ) {
      if(caltype) thresh2 += 1;
      else        thresh1 += 1;
      cal_run_cnt = 0;
      txq_coff = nco_phase_start;
    }else if((cal_lock != calmask) && (txq_coff == nco_phase_stop) && (retrycnt <= 1)) {
      cmdcfg_delay(module_cfg,2000);
      cmdcfg_bufclr(module_cfg);
      thresh1 = 1;
      thresh2 = 1;
      cal_run_cnt = 0;
      txq_coff = nco_phase_start;
      if(p->verbose==3) printf("Failed TX Quad Cal - Retrying: %d\n",retrycnt);
      ++retrycnt;
    }else if (cal_lock != calmask) {
      ++txq_coff;
    }
  }while((cal_lock != calmask) && (txq_coff<=nco_phase_stop));

  if(cal_lock != calmask) {
    printf("ERROR: TX Failed to Properly Calibrate!! : 0x%x, 0x%x\n",calmask,stat_data);
    if(p->verbose==3) printf("thres1 = %d, thres2 = %d, atten = %d\n",thresh1,thresh2,txq_attn);
  }else {
    if(p->verbose==3) printf("TX Quad Cal Lock: status = 0x%x, thresh1 = %d, thresh2 = %d, phase offset = %d, tx atten = %d\n",stat_data,thresh1,thresh2,txq_coff,txq_attn);
  }

  return(0);
}
#endif

int_4 d2rf_tx_quad_calib(Module_Config_Obj *module_cfg, int_4 tx_attn, int_4 txq_attn)
{
  int_4 cal_phase,cal_gain,cal_dcI,cal_dcQ;
  real_8 txlo;
  int_4 txqa;

  d2rf_get_txlo(module_cfg, &txlo);

  /* Run SSB Calibration with atten = 0 */
  d2rf_tx_quad_cal2(module_cfg, (txlo/1e6), 0, 0x01);

  /* Get SSB Calibration results */
  d2rf_get_imcal(module_cfg, &cal_phase, &cal_gain);

  /* Run DC Calibration with atten = 2*tx_atten */
  txqa = (txq_attn==-1)? 2*tx_attn : txq_attn;
  d2rf_tx_quad_cal2(module_cfg, (txlo/1e6), txqa, 0x00);

  /* Get DC Calibration results */
  d2rf_get_dccal(module_cfg, &cal_dcI, &cal_dcQ);

  /* Force LO and SSB Calibration results */
  d2rf_tx_quad_cal_force(module_cfg, cal_dcI, cal_dcQ, cal_phase, cal_gain);
  
  return(0);
}

int_4 d2rf_tx_quad_cal(Module_Config_Obj *module_cfg, real_8 rf_freq, int_u4 txq_off, int_u4 thresh1, int_u4 thresh2, int_u4 nco_freq, int_u4 txq_attn)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 cnt=0,band,cal_lock;
  const uint8_t(*tab)[3];
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_u4 rx_nco_freq = 0;            
  int_u4 tx_nco_freq = 0;            
  int_u4 M = 0;
  real_8 rate = 0.0;               
  real_8 bw = 0.0;              
  real_8 nco_freq2 = 0.0;            
  int_4 index_max = 0;              
  int_4 lpf_tia_mask = 0;   
  int_4 i=0;
  int_4 nco_phase_start = 0;
  int_4 nco_phase_stop  = 0;
  int_4 cal_run_cnt = 0;
  int_4 txq_coff = 0;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TXQC_FORCE_BITS,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x00,0,0xFFFF);
  cmdcfg_ctl(module_cfg,BAD_AIS_ID,0,PFUNC_RWR,0x01,0,0xFFFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B2,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x0B3,0x00,0xFF);

  rx_nco_freq = (nco_freq<<5);
  tx_nco_freq = (nco_freq<<6);
  /* M = (nco_freq)? 0 : 1; */
  M = 3;

  if(rf_freq < 1300) band=0;
  else if(rf_freq < 4000) band=1;
  else band=2;

  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RRD,1,0,0xFFFFFFFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  rate = (real_8)stat_data;
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  bw = (real_8)stat_data;
  nco_freq2 = (4*bw/rate) - 1;

  tab = &full_gain_table[band][0];
  index_max = SIZE_FULL_TABLE;
  lpf_tia_mask = 0x3F;
  i=0;
  for (i = 0; i < index_max; i++)
    if ((tab[i][1] & lpf_tia_mask) == 0x20) {
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_FULL_LMT_GAIN,i,0xFF);
      break;
    }

  nco_phase_start = 20;
  nco_phase_stop  = 26;
  cal_run_cnt = 0;
  txq_coff = nco_phase_start;
  do{
    ++cal_run_cnt;
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP1,0x7F,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_KEXP2,(tx_nco_freq|0x0F),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(0x7C|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_CONTROL,(0x78|M),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_NCO_FREQ_PHASE_OFFSET,(rx_nco_freq|txq_coff),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_SETTLE_COUNT,0xF0,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_CAL_COUNT,0xFE,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES1,thresh1,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_QUAD_MAG_FTEST_THRES2,thresh2,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_LPF_GAIN,0x18,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_QUAD_CAL_ATTN,txq_attn,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SQUARER_CONFIG,0x00,0xFF);
  
    /* Initiate Cal */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x10,0xFF);
    cmdcfg_delay(module_cfg,15000);
  
    /* Check Cal Done Signal, Status, and Cal Results */
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_QUAD_CAL_STATUS,0,0xFF);
    cmdcfg_bufclr(module_cfg);
    cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  
    cal_lock = stat_data&0x03;

    ++txq_coff;
    if( (thresh2 < 30) && (txq_coff == nco_phase_stop) && (cal_lock != 3) ) {
      thresh2 += 1;
      cal_run_cnt = 0;
      txq_coff = nco_phase_start;
      /* printf("thresh1 = %d, thresh2 = %d\n",thresh1,thresh2); */
    }
  }while((cal_lock != 3) && (txq_coff<=nco_phase_stop));

  if(cal_lock != 3) {
    printf("ERROR: TX Failed to Properly Calibrate!! : 0x%x\n",stat_data);
    if(p->verbose==3) printf("thres1 = %d, thres2 = %d, atten = %d\n",thresh1,thresh2,txq_attn);
  }else {
    if(p->verbose==3) printf("TX Quad Cal Lock: thresh1 = %d, thresh2 = %d, phase offset = %d, tx atten = %d\n",thresh1,thresh2,txq_coff,txq_attn);
  }

  return(0);
}

int_4 d2rf_bbpll_init(Module_Config_Obj *module_cfg, real_8 bbpll_freq, real_8 ref_clk)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

/* Determine CP Current Scale Factor: */
  real_8 scale = bbpll_freq/(ref_clk*32);
  int_u1 bbpll_cp = (int_u1)rint(scale*32);

  if(bbpll_cp < 1) bbpll_cp = 1;
  else if(bbpll_cp > 64) bbpll_cp = 64;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CP_CURRENT,bbpll_cp,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LOOP_FILTER_3,0xE8,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LOOP_FILTER_2,0x5B,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LOOP_FILTER_1,0x35,0xFF);
  
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_VCO_CONTROL,0xE0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SDM_CONTROL,0x10,0xFF);

  return(0);
}

int_4 d2rf_set_trx_rate(Module_Config_Obj *module_cfg, real_8 tx_rate, int_4 rx, int_4 state)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u8 ref_clk = (int_u8)REFCLK_FREQ;
  int_u1 bbpll_div,fir_interp,clk_div,tx_fir_int,rx_fir_int;
  int_u4 bbpll_int,bbpll_frac;
  int_u8 bbpll_freq;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_u8 rate_act = 0;

/* Disable Transmisson */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0x0000,0xFFFF);
  
/* Go to ALERT State */
  d2rf_set_state(module_cfg,0);

  bbpll_calc_rate((int_u8)(tx_rate*1e6), &bbpll_freq, &bbpll_div, &bbpll_int, &bbpll_frac, &fir_interp, &clk_div);

  d2rf_bbpll_init(module_cfg,(real_8)bbpll_freq, (real_8)ref_clk);

/*
  if(fir_interp==3) tx_fir_int = 0x1C;
  else if(fir_interp==2) tx_fir_int = 0x0C;
  else if(fir_interp==1) tx_fir_int = 0x04;
  else if(fir_interp==0) tx_fir_int = 0x00;
*/

  if(fir_interp==3) tx_fir_int = 0x1D;
  else if(fir_interp==2) tx_fir_int = 0x0D;
  else if(fir_interp==1) tx_fir_int = 0x05;
  else if(fir_interp==0) tx_fir_int = 0x01;

  if(clk_div==0) rx_fir_int = tx_fir_int;
  else if(fir_interp==3) rx_fir_int = 0x0C;
  else if(fir_interp==2) rx_fir_int = 0x04;
  else if(fir_interp==1) rx_fir_int = 0x00;
  else if(fir_interp==0) { printf("WARNING: TX Rates May Not Be Correct!!\n");  rx_fir_int = 0x00; }

  /* Write TX Rate to State */
  rate_act = bbpll_freq/(int_u8)(bbpll_div * pow(2,fir_interp+1));

  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RWR,0,(int_u4)rate_act,0xFFFFFFFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_ENABLE_FILTER_CTRL,0x40|tx_fir_int,0xFF);
  /*cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_ENABLE_FILTER_CTRL,0x80|rx_fir_int,0xFF);*/
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_ENABLE_FILTER_CTRL,0xC0|rx_fir_int,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BBPLL,(clk_div<<3|bbpll_div),0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INTEGER_BB_FREQ_WORD,bbpll_int,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_FRAC_BB_FREQ_WORD_3,(bbpll_frac & 0xFF),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_FRAC_BB_FREQ_WORD_2,(bbpll_frac>>8 & 0xFF),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_FRAC_BB_FREQ_WORD_1,(bbpll_frac>>16 & 0xFF),0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SDM_CONTROL_1,0x5,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_SDM_CONTROL_1,0x1,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_VCO_PROGRAM_1,0x86,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_VCO_PROGRAM_2,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_VCO_PROGRAM_2,0x05,0xFF);

  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CH_1_OVERFLOW,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("BBPLL CAL: 0x%x\n",stat_data);

/* Go to FDD State */
  d2rf_set_state(module_cfg,state);

/* Enable Transmission for TX Module */
  if(rx==0 && state) cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0x0006,0xFFFF);

  return(0);
}

int_4 ad9361_tracking(Module_Config_Obj *module_cfg, int_u4 bbdc_track, int_u4 rfdc_track, int_u4 rxquad_track)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 qtrack = 0;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CALIBRATION_CONFIG_2,0x60|K_EXP_PHASE(0x15),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CALIBRATION_CONFIG_3,0x80|K_EXP_AMPLITUDE(0x15),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_CONFIG_2,0x85|(bbdc_track ? 0x20 : 0)|(rfdc_track ? 0x08 : 0),0xFF);

  if(rxquad_track) qtrack = 0x03;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CALIBRATION_CONFIG_1,0xC0|(rxquad_track ? 0x03 : 0),0xFF);

return 0;
}

#ifdef BOGUS
int_4 d2rf_set_ais(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  int_4 dc_I = findintflagdef("DCI",p->config,-12);
  int_4 dc_Q = findintflagdef("DCQ",p->config,4);
  int_4 cxr_A = findintflagdef("CXRA",p->config,0x7FFF);
  int_4 cxr_C = findintflagdef("CXRC",p->config,0x0000);
  int_4 cxr_D = findintflagdef("CXRD",p->config,0x7FFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x00,dc_I,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x01,dc_Q,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x02,cxr_A,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x03,cxr_C,0xFFFF);
  cmdcfg_ctl(D2RF_module,BAD_AIS_ID,0,PFUNC_RWR,0x04,cxr_D,0xFFFF);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}
#endif

int_4 d2rf_get_opts(PICSTRUCT *p, int_4 mport)
{
  int_4 rtn = 0;
  int_4 rfopts = 0;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);


  rfopts = RFOPT_ENABLE;

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); return(-1); }

  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PRD,0,0,0xFF);

  cmdcfg_bufclr(D2RF_module);

  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

  if((stat_data & 0x04) == 0) rfopts |= RFOPT_MUTE;
  
  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(rfopts);
}

int_4 d2rf_set_opts(PICSTRUCT *p, int_4 mport, int_4 rfopts)
{
  int_4 rtn = 0;

  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  if((rfopts & RFOPT_ENABLE) != RFOPT_ENABLE) return (0);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); return(-1); }

  if(rfopts & RFOPT_MUTE)
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0x02,0xFF);
  else
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_PWR,0,0x06,0xFF);
  
  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}

int_4 d2rf_set_freq(PICSTRUCT *p, int_4 mport, real_8 rf_freq)
{
  /* Module Identification */
  int_4 rx = 0;
  int_4 rtn;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 txpa_state;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); return(-1); }

  rx = (p->mtype[mport-1]<0)? 1 : 0;

  if(rx == 0) {
    /* Determine if TX PA was enabled */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PRD,0,0,0xFF);
    cmdcfg_bufclr(D2RF_module);
    cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

    txpa_state = stat_data;
 
    /* Disable TX PA  */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,0x01,0xFF);

    /* Put RF Switch into RX Mode  */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x01,0xFF);
  }

  d2rf_set_rffreq(D2RF_module,rf_freq,rx,1,1);

  if(rx == 0) {
    /* Put RF Switch into RX Mode  */
    cmdcfg_ctl(D2RF_module,BAD_PE42423_ID,0,PFUNC_PWR,0,0x00,0xFF);

    /* Return TX PA State */
    cmdcfg_ctl(D2RF_module,BAD_VMMK2103_ID,1,PFUNC_PWR,0,txpa_state,0xFF);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

  return(0);
}

int_4 d2rf_set_rffreq(Module_Config_Obj *module_cfg, real_8 rf_freq, int_4 rx, int_4 state, int_4 runcal)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_8 ref_clk = (int_8)REFCLK_FREQ;
  int_4 div,vco_int,vco_frac; 
  int_8 vco_freq;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_u4 txq_off;
  int_u4 txq_attn;
  int_u4 thresh1;
  int_u4 thresh2;
  int_u4 nco_freq;
  int_4 tx_attn;
  int_4 offs;
  const int_4 (*tab)[3];
  int_4 index_max = SIZE_TXATN_TBL;
  int_4 i=0;
  int_4 rf_ifreq = (int_4) rf_freq;
  int_4 vco_freq_mhz;
  int_4 addr_f2,addr_f1,addr_f0;
  int_u4 addr_i1;
  int_u4 addr_i0;
  int_4 x1;
  int_4 x0;
  int_u4 div_mask = 0;
  int_u4 div_data = 0;
  int_u4 band = 0;
  int_4 addr_vcocal = 0;
  int_4 cnt = 0;


 if(p->verbose==3) printf("Setting %s LO to %fMHz\n",(rx)? "RX" : "TX", rf_freq); 

  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,5,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_CMD_ID,0,CFUNC_RWR,5,0x00,0xFF);

  rfpll_calc_rate((int_u8)(rf_freq*1e6), ref_clk, &vco_freq, &div, &vco_int, &vco_frac);

/* Update TX attenuation depending on LO */
  txq_attn = findintflagdef("TXQA",p->config,-1);
  tab = &tx_attn_tbl[0];
  for (i = 0; i < index_max; i++){
    if (rf_ifreq < tab[i][0]) {
      tx_attn = tab[i][1];
      if(txq_attn==-1){
        txq_attn = tab[i][2];
      }
      break;
    }
  }

/* Disable Transmisson */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0,0xFFFF);

#ifdef BOGUS
  txq_off = findintflagdef("TXQC",p->config,TXQC_OFF_DEF);
  txq_attn = findintflagdef("TXQA",p->config,-1);
  thresh1 = findintflagdef("TXQTH1",p->config,1);
  thresh2 = findintflagdef("TXQTH2",p->config,1);
  nco_freq = findintflagdef("TXQNCO",p->config,1);
#endif

  offs = (rx)? 0 : 0x40;

/* Go to ALERT State */
  d2rf_set_state(module_cfg,0);

/* Disable Tracking Control */
  ad9361_tracking(module_cfg,0,0,0);

  set_tx_attn(module_cfg, tx_attn);

/* Write New PLL Loop Values */
  vco_freq_mhz = (int_4)(vco_freq/1e6); 
  d2rf_vco_init(module_cfg, vco_freq_mhz, rx);

/* Write Fractional Value */
  addr_f2 = AD9361_REG_RX_FRACT_BYTE_2;
  addr_f1 = AD9361_REG_RX_FRACT_BYTE_1;
  addr_f0 = AD9361_REG_RX_FRACT_BYTE_0;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(addr_f2+offs),((vco_frac>>16)&0x0000007F),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(addr_f1+offs),((vco_frac>>8)&0x000000FF),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(addr_f0+offs),(vco_frac&0x000000FF),0xFF);

/* Write Integer Value */
  addr_i1 = AD9361_REG_RX_INTEGER_BYTE_1;
  addr_i0 = AD9361_REG_RX_INTEGER_BYTE_0;

  x1 = (vco_int>>8)&0xFF;
  x0 = vco_int&0xFF;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(addr_i1+offs),x1,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,(addr_i0+offs),x0,0xFF);

/* Write New Divider Value */
  div_mask = (rx)? RX_VCO_DIVIDER(~0) : TX_VCO_DIVIDER(~0);
  div_data = (rx)? RX_VCO_DIVIDER(div) : TX_VCO_DIVIDER(div);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RFPLL_DIVIDERS,div_data,div_mask);

/* Load Gain Table */
  if(rf_freq <= 650) band=0;
  else if(rf_freq < 4000) band=1;
  else band=2;

  if(rx) d2rf_load_gt_gain(module_cfg, band);

/* Check for VCO Cal Lock */
  addr_vcocal = AD9361_REG_RX_CP_OVR_VCO_LOCK;
  cnt = 500;  
VCO_RETRY:
  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,(addr_vcocal+offs),0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("VCO CAL LOCK: 0x%x\n",stat_data);
  if((stat_data & 0x02) == 0) {
    if(p->verbose==3) printf("VCO Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: RF VCO FAILED TO LOCK!!\n"); return(-1); }
    else goto VCO_RETRY;
  }

if(runcal) {
/* Run RF DC Calibration: */
  cnt = 500;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x02,0xFF);
RFDC_RETRY:
  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("RF DC CAL LOCK: 0x%x\n",stat_data);
  if(stat_data & 0x02) {
    if(p->verbose==3) printf("RF DC Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: RF VCO FAILED TO LOCK!!\n"); return(-1); }
    else goto RFDC_RETRY;
  }

#ifdef BOGUS
if(rx) {
/* Run RX Quad Calibration */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ANALOG_PD_OVERRIDE,0x3F,0xFF);
  d2rf_rx_quad_cal(module_cfg);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ANALOG_PD_OVERRIDE,0x3E,0xFF);
}
#endif

/* Run TX Quad Calibration: */
  /* if(rx==0) d2rf_tx_quad_cal(module_cfg,rf_freq,txq_off,thresh1,thresh2,nco_freq,txq_attn); */
  if(rx==0) d2rf_tx_quad_calib(module_cfg, tx_attn,txq_attn);
}

if(state) {
/* Enable Tracking Control */
  ad9361_tracking(module_cfg,1,1,1);
}

/* Go to FDD State */
  d2rf_set_state(module_cfg,state);

/* Enable Transmission for TX Module */
  if(rx==0 && state) cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_PWR,0,0x0006,0xFFFF);
  
  return(0);
}

int_4 bb_rx_analog_filter_cal(Module_Config_Obj *module_cfg, int_4 rx_bb_bw, int_4 bbpll_freq)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 rxbbf_div,target,tmp;
  int_4 ret;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 cnt = 0;
  int_4 cal_lock = 0;
  int_4 cal_cnt = 0;

  if(rx_bb_bw<200000) rx_bb_bw = 200000;
  else if(rx_bb_bw>28000000) rx_bb_bw = 28000000;

  /* 1.4 * BBBW * 2PI / ln(2) */
  target = 126906 * (rx_bb_bw / 10000);
  rxbbf_div = (bbpll_freq + target - 1)/target;
  if(rxbbf_div > 511) rxbbf_div = 511;

  /* Set RX baseband filter divide value */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_BBF_TUNE_DIVIDE,rxbbf_div,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_BBF_TUNE_CONFIG,(rxbbf_div>>8),0x01);

  /* Write the BBBW into registers 0x1FB and 0x1FC */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_BBBW_MHZ,rx_bb_bw/1e6,0xFF);
  tmp = ((rx_bb_bw % (int_4)1e6) * 128 + 1e6/2)/1e6;
  if(tmp>127) tmp=127;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_BBBW_KHZ,tmp,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_MIX_LO_CM,RX_MIX_LO_CM(0x3F),0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_MIX_GM_CONFIG,RX_MIX_GM_PLOAD(3),0xFF);

  /* Enable the RX BBF tune circuit by writing 0x1E2=0x02 and 0x1E3=0x02 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX1_TUNE_CTRL,0x02,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX2_TUNE_CTRL,0x02,0xFF);

  /* Start the RX Baseband Filter calibration in register 0x016[7] */
  cnt=0;
  cal_lock=0;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x80,0xFF);

  cal_cnt = 0;
/* RETRY: */
  do{
    cmdcfg_delay(module_cfg,100);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0x80);
    cmdcfg_bufclr(module_cfg);
    cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&cal_lock);
  /* if(cal_lock & 0x80) goto RETRY; */
  }while(cal_lock & 0x80);
  if(p->verbose==3) printf("RX BB FILTER CAL: 0x%x, cnt=%d\n",cal_lock,cal_cnt);

  /* Disable the RX baseband filter tune circuit, write 0x1E2=3, 0x1E3=3 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX1_TUNE_CTRL,0x03,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX2_TUNE_CTRL,0x03,0xFF);

  return(0);
}

int_4 bb_tx_analog_filter_cal(Module_Config_Obj *module_cfg, int_4 tx_bb_bw, int_4 bbpll_freq)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 target, txbbf_div;
  int_4 ret;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 cnt = 0;
  int_4 cal_lock = 0;

  if(tx_bb_bw<625000) tx_bb_bw = 625000;
  else if(tx_bb_bw>32000000) tx_bb_bw = 32000000; 

  /* 1.6 * BBBW * 2PI / ln(2) */
  target = 145036 * (tx_bb_bw / 10000);

  txbbf_div = (bbpll_freq + target - 1)/target;
  if(txbbf_div > 511) txbbf_div = 511;

  /* Set TX baseband filter divide value */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_BBF_TUNE_DIVIDE,txbbf_div,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_BBF_TUNE_MODE,(txbbf_div>>8),0x01);

  /* Enable the TX baseband filter tune circuit by setting 0x0CA=0x22. */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_TUNE_CTRL,0x22,0xFF);

  /* Start the TX Baseband Filter calibration in register 0x016[6] */
  cnt=0;
  cal_lock=0;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x40,0xFF);
  cmdcfg_delay(module_cfg,1000);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0x40);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&cal_lock);
  if(p->verbose==3) printf("TX BB FILTER CAL: 0x%x\n",cal_lock);

  /* Disable the TX baseband filter tune circuit by writing 0x0CA=0x26. */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_TUNE_CTRL,0x26,0xFF);

  return(0);
}

int_4 tia_cal(Module_Config_Obj *module_cfg, int_4 bb_bw_Hz)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 Cbbf, R2346;
  int_8 CTIA_fF;
 
  int_4 stat_func,stat_pin,stat_addr; 
  int_4 reg1EB,reg1EC,reg1E6;
  int_4 reg1DB, reg1DF, reg1DD, reg1DC, reg1DE, temp;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_C3_MSB,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_C3_LSB,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_R2346 ,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&reg1EB);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&reg1EC);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&reg1E6);

  if(bb_bw_Hz<20000) bb_bw_Hz = 200000;
  else if(bb_bw_Hz>20000000) bb_bw_Hz = 20000000;

  Cbbf = (reg1EB * 160) + (reg1EC * 10) + 140; /* fF */
  R2346 = 18300 * (reg1E6 & 0x07);

  CTIA_fF = (Cbbf * R2346 * 560)/3500000;

  if (bb_bw_Hz <= 3000000)  reg1DB = 0xE0;
  else if (bb_bw_Hz <= 10000000) reg1DB = 0x60;
  else reg1DB = 0x20;

  if (CTIA_fF > 2920) {
    reg1DC = 0x40;
    reg1DE = 0x40;
    temp = (CTIA_fF-400 + 320/2)/320;
    if(temp>127) temp = 127;
    reg1DD = temp;
    reg1DF = temp;
  }
  else {
    temp = (CTIA_fF-400 + 40/2)/40;
    reg1DC = temp;
    reg1DE = temp;
    reg1DD = 0;
    reg1DF = 0;
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_TIA_CONFIG,reg1DB,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TIA1_C_LSB,reg1DC,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TIA1_C_MSB,reg1DD,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TIA2_C_LSB,reg1DE,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TIA2_C_MSB,reg1DF,0xFF);

  return 0;
}

int_u8 do_div(int_u8* n, int_u8 base)
{
        int_u8 mod = 0;

        mod = *n % base;
        *n = *n / base;

        return mod;
}

#define BITS_PER_LONG           32
int_u4 int_sqrt(int_u4 x)
{
        int_u4 b, m, y = 0;

        if (x <= 1)
                return x;

        m = 1UL << (BITS_PER_LONG - 2);
        while (m != 0) {
                b = y + m;
                y >>= 1;

                if (x >= b) {
                        x -= b;
                        y += m;
                }
                m >>= 2;
        }

        return y;
}

int_4 bb_tx_filter2_cal(Module_Config_Obj *module_cfg, int_4 tx_rf_bw)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u8 cap;
  int_u4 corner, res, div;
  int_u4 reg_conf, reg_res;
  int32_t ret, i;
  
  if(tx_rf_bw < 530000UL) tx_rf_bw = 530000UL;
  else if(tx_rf_bw > 20000000UL) tx_rf_bw = 20000000UL;

  /* 5 * BBBW/2 * 2PI */
  corner = 15708  * (tx_rf_bw / 10000UL);
  
  for (i = 0, res = 1; i < 4; i++) {
    div = corner * res;
    cap = (500000000ULL) + (div >> 1);
    do_div(&cap, div);
    cap -= 12ULL;
    if (cap < 64ULL) break;
    res <<= 1;
  }

  if (cap > 63ULL) cap = 63ULL;
  
  if (tx_rf_bw <= 4500000UL) reg_conf = 0x59;
  else if (tx_rf_bw <= 12000000UL) reg_conf = 0x56;
  else reg_conf = 0x57;
  
  switch (res) {
    case 1:
      reg_res = 0x0C;
      break;
    case 2:
      reg_res = 0x04;
      break;
    case 4:
      reg_res = 0x03;
      break;
    case 8:
      reg_res = 0x01;
      break;
    default:
      reg_res = 0x01;
  }
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CONFIG0,reg_conf,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RESISTOR,reg_res,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAPACITOR,(int_u4)cap,0xFF);

  return(0);
}

int_4 adc_setup(Module_Config_Obj *module_cfg, int_u4 bbpll_freq, int_u4 adc_sampl_freq_Hz)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_u4 scale_snr_1e3, maxsnr, sqrt_inv_rc_tconst_1e3, tmp_1e3,scaled_adc_clk_1e6, inv_scaled_adc_clk_1e3, sqrt_term_1e3,min_sqrt_term_1e3, bb_bw_Hz;
  int_u8 tmp, invrc_tconst_1e6;
  int_u1 data[40];
  int_u4 i;
  int_4 c3_msb,c3_lsb,r2346,tmp1,tmp2;
  int_4 stat_func,stat_pin,stat_addr;
  int_u4 rxbbf_div = 0;

  /* Reset ADC */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x226,0xFF,0xFF);
  cmdcfg_delay(module_cfg,500);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x226,0x00,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_C3_MSB,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_C3_LSB,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_R2346 ,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_TUNE_DIVIDE,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RX_BBF_TUNE_CONFIG,0,0x01);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&c3_msb);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&c3_lsb);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&r2346);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&tmp1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&tmp2);

  rxbbf_div = (tmp2<<8)|tmp1;

  tmp = bbpll_freq * 10000ULL;
  do_div(&tmp, 126906UL * rxbbf_div);
  bb_bw_Hz = tmp;

  bb_bw_Hz = clamp(bb_bw_Hz, 200000UL, 28000000UL);

  if (adc_sampl_freq_Hz < 80000000)
    scale_snr_1e3 = 1000;
  else
    scale_snr_1e3 = 1585; /* pow(10, scale_snr_dB/10); */

  if (bb_bw_Hz >= 18000000) {
    invrc_tconst_1e6 = (160975ULL * r2346 * (160 * c3_msb + 10 * c3_lsb + 140) * (bb_bw_Hz)* (1000 + (10 * (bb_bw_Hz - 18000000) / 1000000)));
    do_div(&invrc_tconst_1e6, 1000UL);
  }
  else {
    invrc_tconst_1e6 = (160975ULL * r2346 * (160 * c3_msb + 10 * c3_lsb + 140) * (bb_bw_Hz));
  }

  do_div(&invrc_tconst_1e6, 1000000000UL);

  sqrt_inv_rc_tconst_1e3 = int_sqrt((int_u4)invrc_tconst_1e6);
  maxsnr = 640 / 160;
  scaled_adc_clk_1e6 = DIV_ROUND_CLOSEST(adc_sampl_freq_Hz, 640);
  inv_scaled_adc_clk_1e3 = DIV_ROUND_CLOSEST(640000000, DIV_ROUND_CLOSEST(adc_sampl_freq_Hz, 1000));
  tmp_1e3 = DIV_ROUND_CLOSEST(980000 + 20 * max_t(int_u4, 1000U, DIV_ROUND_CLOSEST(inv_scaled_adc_clk_1e3, maxsnr)), 1000);
  sqrt_term_1e3 = int_sqrt(scaled_adc_clk_1e6);
  min_sqrt_term_1e3 = min_t(int_u4, 1000U, int_sqrt(maxsnr * scaled_adc_clk_1e6));

  data[0] = 0;
  data[1] = 0;
  data[2] = 0;
  data[3] = 0x24;
  data[4] = 0x24;
  data[5] = 0;
  data[6] = 0;

  tmp = -50000000 + 8ULL * scale_snr_1e3 * sqrt_inv_rc_tconst_1e3 * min_sqrt_term_1e3;
  do_div(&tmp, 100000000UL);
  data[7] = min_t(int_u8, 124U, tmp);

  tmp = (invrc_tconst_1e6 >> 1) + 20 * inv_scaled_adc_clk_1e3 * data[7] / 80 * 1000ULL;
  do_div(&tmp, invrc_tconst_1e6);
  data[8] = min_t(int_u8, 255U, tmp);

  tmp = (-500000 + 77ULL * sqrt_inv_rc_tconst_1e3 * min_sqrt_term_1e3);
  do_div(&tmp, 1000000UL);
  data[10] = min_t(int_u8, 127U, tmp);

  data[9] = min_t(int_u4, 127U, ((800 * data[10]) / 1000));
  tmp = ((invrc_tconst_1e6 >> 1) + (20 * inv_scaled_adc_clk_1e3 * data[10] * 1000ULL));
  do_div(&tmp, invrc_tconst_1e6 * 77);
  data[11] = min_t(int_u8, 255U, tmp);
  data[12] = min_t(int_u4, 127U, (-500000 + 80 * sqrt_inv_rc_tconst_1e3 * min_sqrt_term_1e3) / 1000000UL);

  tmp = -3 * (long)(invrc_tconst_1e6 >> 1) + inv_scaled_adc_clk_1e3 * data[12] * (1000ULL * 20 / 80);
  do_div(&tmp, invrc_tconst_1e6);
  data[13] = min_t(int_u8, 255, tmp);

  data[14] = 21 * (inv_scaled_adc_clk_1e3 / 10000);
  data[15] = min_t(int_u4, 127U, (500 + 1025 * data[7]) / 1000);
  data[16] = min_t(int_u4, 127U, (data[15] * tmp_1e3) / 1000);
  data[17] = data[15];
  data[18] = min_t(int_u4, 127U, (500 + 975 * data[10]) / 1000);
  data[19] = min_t(int_u4, 127U, (data[18] * tmp_1e3) / 1000);
  data[20] = data[18];
  data[21] = min_t(int_u4, 127U, (500 + 975 * data[12]) / 1000);
  data[22] = min_t(int_u4, 127, (data[21] * tmp_1e3) / 1000);
  data[23] = data[21];
  data[24] = 0x2E;
  data[25] = (128 + min_t(int_u4, 63000U, DIV_ROUND_CLOSEST(63 * scaled_adc_clk_1e6, 1000)) / 1000);
  data[26] = min_t(int_u4, 63U, 63 * scaled_adc_clk_1e6 / 1000000 * (920 + 80 * inv_scaled_adc_clk_1e3 / 1000) / 1000);
  data[27] = min_t(int_u4, 63, (32 * sqrt_term_1e3) / 1000);
  data[28] = data[25];
  data[29] = data[26];
  data[30] = data[27];
  data[31] = data[25];
  data[32] = data[26];
  data[33] = min_t(int_u4, 63U, 63 * sqrt_term_1e3 / 1000);
  data[34] = min_t(int_u4, 127U, 64 * sqrt_term_1e3 / 1000);
  data[35] = 0x40;
  data[36] = 0x40;
  data[37] = 0x2C;
  data[38] = 0x00;
  data[39] = 0x00;

  for (i = 0; i < 40; i++) {
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,0x200+i,data[i],0xFF);
  }

  return(0);
}

int_4 d2rf_set_bw(PICSTRUCT *p, int_4 mport, int_4 bw)
{
#ifdef BOGUS
if(p->verbose==3) printf("D2RF Set BW\n");

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0){printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); return(-1); }

  d2rf_set_bandwidth(D2RF_module,bw,bw,1,1);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);
#endif

  return(0);
}

int_4 d2rf_get_txlo(Module_Config_Obj *module_cfg, real_8 *txlo)
{
  real_8 ref_clk = (real_8)REFCLK_FREQ;
  real_8 txlo_mod = 8388593;
  int_4 txlo_div,f1,f2,f3,i1,i2;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  real_8 txlo_frac = 0.0;
  real_8 txlo_int = 0.0;
  real_8 txlo_freq = 0.0;

/* Determine TX LO Frequency */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RFPLL_DIVIDERS,0,0xF0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_2,0,0x7F);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_1,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_0,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_1,0,0x07);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_0,0,0xFF);
  cmdcfg_bufclr(module_cfg);

  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&txlo_div);
  txlo_div = txlo_div>>4;
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f3);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f2);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i2);

  txlo_frac = (real_8)((f3<<16)|(f2<<8)|f1);
  txlo_int = (real_8)((i1<<8)|i2);
  txlo_freq = (ref_clk*txlo_int + (ref_clk*txlo_frac)/txlo_mod)/(1<<(txlo_div+1));
  
  *txlo = txlo_freq;

  return(0);
}


int_4 d2rf_set_bandwidth(Module_Config_Obj *module_cfg, int_4 tx_bw, int_4 rx_bw, int_4 state, int_4 runcal)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  real_8 ref_clk = (real_8)REFCLK_FREQ;
  real_8 bbpll_mod = 2088960;
  real_8 txlo_mod = 8388593;
  real_8 bbpll_freq;
  int_4 txlo_div,f1,f2,f3,i1,i2;
  int_4 stat_func,stat_pin,stat_addr;

  int_4 txq_off = findintflagdef("TXQC",p->config,TXQC_OFF_DEF);
  int_4 txq_attn = findintflagdef("TXQA",p->config,TXQC_ATTN_DEF);
  int_4 thresh1 = findintflagdef("TXQTH1",p->config,1);
  int_4 thresh2 = findintflagdef("TXQTH2",p->config,1);
  int_4 nco_freq = findintflagdef("TXQNCO",p->config,1);
  real_8 txlo_frac = 0.0;
  real_8 txlo_int = 0.0;
  real_8 txlo_freq = 0.0;
  int_4 adc_div = 0;
  int_4 adc_sampl_freq_Hz = 0;

  /* Write Bandwidth to State */
  cmdcfg_ctl(module_cfg,BAD_MOD_ID,0,PFUNC_RWR,1,tx_bw,0xFFFFFFFF);

/* Determine BBPLL Clock Frequency */
  bbpll_freq = d2rf_get_BBPLL(module_cfg);

/* Determine TX LO Frequency */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_RFPLL_DIVIDERS,0,0xF0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_2,0,0x7F);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_1,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_FRACT_BYTE_0,0,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_1,0,0x07);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TX_INTEGER_BYTE_0,0,0xFF);
  cmdcfg_bufclr(module_cfg);

  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&txlo_div);
  txlo_div = txlo_div>>4;
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f3);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f2);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&f1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i1);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&i2);

  txlo_frac = (real_8)((f3<<16)|(f2<<8)|f1);
  txlo_int = (real_8)((i1<<8)|i2);
  txlo_freq = (ref_clk*txlo_int + (ref_clk*txlo_frac)/txlo_mod)/(1<<(txlo_div+1));

/* Disable Tracking Control */
  ad9361_tracking(module_cfg,0,0,0);

/* Go To Alert State */
  d2rf_set_state(module_cfg,0);

/* Run RX BB Filter Cal */
  bb_rx_analog_filter_cal(module_cfg,rx_bw/2,(int_4)bbpll_freq);

/* Run TX BB Filter Cal */
  bb_tx_analog_filter_cal(module_cfg,tx_bw/2,(int_4)bbpll_freq);

/* Run RX TIA Filter Cal */
  tia_cal(module_cfg,rx_bw/2);

/* Run TX 2nd BB Filter Cal */
  bb_tx_filter2_cal(module_cfg,tx_bw/2);

/* Setup RX ADC */
  adc_div = 0;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_BBPLL,0,0x07);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&adc_div);

  adc_sampl_freq_Hz = (int_4)(bbpll_freq/(1<<adc_div));
  adc_setup(module_cfg,bbpll_freq,adc_sampl_freq_Hz);

if(runcal) { 
/* Run RX Quad Calibration */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ANALOG_PD_OVERRIDE,0x3F,0xFF);
  d2rf_rx_quad_cal(module_cfg);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_ANALOG_PD_OVERRIDE,0x3E,0xFF);

/* Run TX Quad Cal */
  d2rf_tx_quad_cal(module_cfg,txlo_freq,txq_off,thresh1,thresh2,nco_freq,txq_attn);
}

/* Enable Tracking Control */
 ad9361_tracking(module_cfg,1,1,1);

/* Go To FDD State */
  d2rf_set_state(module_cfg,state);

  return(0);
}

int_4 ad9361_load_gm_table(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 i;
  int_4 size = sizeof(gm_gain)/sizeof(gm_gain[0]);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_CONFIG,0x02,0xFF);
  for(i=0; i<size; ++i){
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_ADDRESS,(15-i),0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_BIAS_WRITE,0x00,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_GAIN_WRITE,gm_gain[i],0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_CTRL_WRITE,gm_ctrl[i],0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_CONFIG,0x06,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_GAIN_READ,0x00,0xFF);
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_GAIN_READ,0x00,0xFF);    
  }
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_CONFIG,0x02,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_GAIN_READ,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_GAIN_READ,0x00,0xFF);    
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_GM_SUB_TABLE_CONFIG,0x00,0xFF);

  return(0);
}

int_4 ad9361_config(Module_Config_Obj *module_cfg, int_4 dir, real_8 samprate, real_8 rxlo, real_8 txlo, int_4 bandwidth, int_4 tx_attn)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 cnt;
  real_8 bbpll_freq;
  int_4 adc_div;
  int_4 adc_sampl_freq_Hz;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CTRL,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BANDGAP_CONFIG0,0x0E,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BANDGAP_CONFIG1,0x0E,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_1,0x06,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x10,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x13,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x73,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CLOCK_ENABLE,0x17,0xFF);

  d2rf_set_trx_rate(module_cfg, samprate,1, 0);
  d2rf_set_trx_rate(module_cfg, samprate,0, 0);

/* Enable RX/TX Channels */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_ENABLE_FILTER_CTRL,0x40,0xC0);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_ENABLE_FILTER_CTRL,0x40,0xC0);

  /* rf_port_setup */
  if(dir == 1)
    cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x03,0xFF);
  else {
  /* Choose the RX RF switch and ADC input depending on tuned frequency */
    if(rxlo < 500.0){
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x30,0xFF);
    }
    else if(rxlo < 3000.0){
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x0C,0xFF);
    }
    else if(rxlo < 6000.0){
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x03,0xFF);
    }
    else {
      printf("/* ERROR: Invalid RFFREQ parameter  */ \n");
      cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_INPUT_SELECT,0x0C,0xFF);
    }
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_1,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_2,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_3,0x10,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_CLOCK_DATA_DELAY,0x40,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_CLOCK_DATA_DELAY,0x07,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_BIAS_CTRL,0x22,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DIGITAL_IO_CTRL,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_INVERT_CTRL1,0x57,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_INVERT_CTRL2,0x3F,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REFERENCE_CLOCK_CYCLES,0x27,0xFF);

  rf_synth_cp_cal(module_cfg,1);
  rf_synth_cp_cal(module_cfg,0);

  /* Set RX and TX LO */
  d2rf_set_rffreq(module_cfg, rxlo, 1, 0, 0);
  d2rf_set_rffreq(module_cfg, txlo, 0, 0, 0);

  ad9361_load_gm_table(module_cfg);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_COUNT,0x3F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_SHIFT,0x0F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_ATTN,0x01,0xFF);

  bbpll_freq = d2rf_get_BBPLL(module_cfg);
  bb_rx_analog_filter_cal(module_cfg, bandwidth/2, (int_4)bbpll_freq);
  bb_tx_analog_filter_cal(module_cfg, bandwidth/2, (int_4)bbpll_freq);
  tia_cal(module_cfg,bandwidth/2);
  bb_tx_filter2_cal(module_cfg,bandwidth/2);

/* Setup RX ADC */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_BBPLL,0,0x07);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&adc_div);

  adc_sampl_freq_Hz = (int_4)(bbpll_freq/(1<<adc_div));
  adc_setup(module_cfg,bbpll_freq,adc_sampl_freq_Hz);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_COUNT,0x3F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_SHIFT,0x0F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_ATTN,0x01,0xFF);

/* Run BB DC Calibration: */
  cnt = 500;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x01,0xFF);
BBDC_RETRY:
  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("BB DC CAL LOCK: 0x%x\n",stat_data);
  if(stat_data & 0x01) {
    if(p->verbose==3) printf("BB DC Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: BB DC CAL FAILED TO LOCK!!\n"); return(-1); }
    else goto BBDC_RETRY;
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_WAIT_CNT,0x20,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_CNT,0x32,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_CONFIG_1,0x24,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_ATTN,0x05,0xFF);

/* Run RF DC Calibration: */
  cnt = 500;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x02,0xFF);
RFDC_RETRY:
  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("RF DC CAL LOCK: 0x%x\n",stat_data);
  if(stat_data & 0x02) {
    if(p->verbose==3) printf("RF DC Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: RF VCO FAILED TO LOCK!!\n"); return(-1); }
    else goto RFDC_RETRY;
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_ATTN_OFFSET,0x00,0xFF);

  return(0);
}

#ifdef BOGUS
int_4 ad9361_config(Module_Config_Obj *module_cfg)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 cnt;

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CTRL,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BANDGAP_CONFIG0,0x0E,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BANDGAP_CONFIG1,0x0E,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_1,0x06,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x10,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x13,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REF_DIVIDE_CONFIG_2,0x73,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CLOCK_ENABLE,0x17,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_1,0x01,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_2,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_PARALLEL_PORT_CONF_3,0x10,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_CLOCK_DATA_DELAY,0x40,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_CLOCK_DATA_DELAY,0x07,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_BIAS_CTRL,0x22,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DIGITAL_IO_CTRL,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_INVERT_CTRL1,0x57,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_LVDS_INVERT_CTRL2,0x3F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_REFERENCE_CLOCK_CYCLES,0x27,0xFF);

  rf_synth_cp_cal(module_cfg,1);
  rf_synth_cp_cal(module_cfg,0);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_OUTPUT,0x0A,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_OUTPUT,0x4A,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_ALC_VARACTOR,0x80,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_BIAS_1,0x04,0xFF); /* 0x0D */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_FORCE_VCO_TUNE_1,0x00,0xFF); /* 0x70 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_VARACTOR_CTRL_1,0x08,0xFF); /* 0x09 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_CAL_REF,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_RX_VCO_VARACTOR_CTRL_0,0x63,0xFF); /* 0x70 */

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_OUTPUT,0x0A,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_OUTPUT,0x4A,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_ALC_VARACTOR,0x80,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_BIAS_1,0x04,0xFF); /* 0x0D */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_FORCE_VCO_TUNE_1,0x00,0xFF); /* 0x70 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_VARACTOR_CTRL_1,0x08,0xFF); /* 0x09 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_CAL_REF,0x00,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_VCO_VARACTOR_CTRL_0,0x63,0xFF); /* 0x70 */

  ad9361_load_gm_table(module_cfg);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_COUNT,0x3F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_SHIFT,0x0F,0xFF);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_BB_DC_OFFSET_ATTN,0x01,0xFF);

/* Run RF BB DC Calibration: */
  cnt = 500;
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_CAL_CONTROL,0x01,0xFF);
BBDC_RETRY:
  cmdcfg_delay(module_cfg,2000);
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_CAL_CONTROL,0,0xFF);
  cmdcfg_bufclr(module_cfg);
  cmdcfg_stat(module_cfg,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  if(p->verbose==3) printf("BB DC CAL LOCK: 0x%x\n",stat_data);
  if(stat_data & 0x01) {
    if(p->verbose==3) printf("BB DC Locking Count = %d\n",cnt);
    if(cnt-- == 0) { printf("ERROR: BB DC CAL FAILED TO LOCK!!\n"); return(-1); }
    else goto BBDC_RETRY;
  }

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_WAIT_CNT,0x20,0xFF);        /* 0x20 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_CNT,0xFF,0xFF);   /* 0x32 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_CONFIG_1,0x3F,0xFF); /* 0x24 */
  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_DC_OFFSET_ATTN,0x05,0xFF);

  cmdcfg_ctl(module_cfg,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TX_ATTN_OFFSET,0x00,0xFF);

  return(0);
}
#endif

int_4 d2rf_get_temp (PICSTRUCT *p, int_4 mport)
{
  int_4 rtn, temp;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  /* Create a module handle and configure device transport */
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0) { 
    printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); 
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return(-1);
   }

  /* Get module's current state into local struct */
  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RWR,AD9361_REG_TEMP_START_READ,0x01,0xFF);
  do{
    cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TEMP_START_READ,0,0xFF);
    cmdcfg_bufclr(D2RF_module);
    cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  }while((stat_data&0x02)==0);

  cmdcfg_ctl(D2RF_module,BAD_AD9361_ID,0,PFUNC_RRD,AD9361_REG_TEMP,0,0xFF);
  cmdcfg_bufclr(D2RF_module);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

  temp = stat_data;

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return (temp);
}

int_4 d2rf_get_clkstat (PICSTRUCT *p, int_4 mport)
{
  int_4 rtn, status;
  int_4 stat_func,stat_pin,stat_addr,stat_data;

  /* Create a module handle and configure device transport */
  Module_Config_Obj *D2RF_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2RF_module = module_alloc_obj(FPGA_MODID_D2RF, &T, 0);
  if (D2RF_module == (Module_Config_Obj *)NULL) return (-2);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_ALLOC);
#endif

  rtn = module_lock(D2RF_module, 0);
  if(rtn<0) { 
    printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); 
#ifdef PICLOCK_BLD 
    pic_lock(p,LOCK_FREE);
#endif
    return(-1);
   }

  cmdcfg_ctl(D2RF_module,BAD_MOD_ID,0,PFUNC_PRD,1,0,0xFF);
  cmdcfg_bufclr(D2RF_module);
  cmdcfg_stat(D2RF_module,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);

  status = 0;
  if (findintflag("PREFX",p->config)>0 && (stat_data&0x1)==0) status |= 0x1;
  if ((stat_data&0x2)==0) status |= 0x2;

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2RF_module);

  /* Release Command Lock */
  rtn = module_unlock(D2RF_module, 0);
  module_free_obj(D2RF_module, 0);

#ifdef PICLOCK_BLD 
  pic_lock(p,LOCK_FREE);
#endif

  return (status);
}

