
/****************************************************************************/
/*                                                                          */
/*   D2AWGM3 module interface routine                                       */
/*                                                                          */
/****************************************************************************/

#include "iomlib_d2awgm3.h" 

int_4 d2awgm3_set_state(Module_Config_Obj *mco)
{
  int_4 i;

  for(i=0;i<D2AWGM3_STATE_SZ;i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RWR,i,mco->LS.i32[i],0xFFFFFFFF);
  }

  return(0);
}

int_4 d2awgm3_get_state(Module_Config_Obj *mco)
{
  int_4 cmfunc,cmpin,cmaddr,cmdata;
  int_4 i;

  for(i=0;i<D2AWGM3_STATE_SZ;i++) {
    cmdcfg_ctl(mco,PTID_STATE,0,PFUNC_RRD,i,0,0xFFFFFFFF);
  }

  cmdcfg_bufclr(mco);

  for(i=0;i<D2AWGM3_STATE_SZ;i++) {
    cmdata = 0x00000000;
    cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
    mco->LS.i32[i] = cmdata;
  }

  return(0);
}

int_4 d2awgm3_dbgsetup(Module_Config_Obj *mco, int_4 dbgsel, int_4 trigsel, int_4 datsel)
{
  int_4 regword;

  regword = ((trigsel&0xF)<<8) | ((datsel&0xF)<<4);

  /* printf("JJB::: Calling dbgsetup, %d,%d,%d, 0x%x\n",dbgsel,trigsel,datsel,regword); */
  cmdcfg_ctl(mco,BAD_MEM_ID,dbgsel,PFUNC_PWR,0,regword|0x01,0xFFFF);
  cmdcfg_ctl(mco,BAD_MEM_ID,dbgsel,PFUNC_PWR,0,regword,0xFFFF);

  return(0);
}

int_4 max5868_reset(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x01,0xFFFF);
  
  return(0);
}

int_4 max5868_resetn(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x00,0xFFFF);
  
  return(0);
}

int_4 max5868_clkdis(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x10,0x10);
  
  return(0);
}

int_4 max5868_clken(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x00,0x10);
  
  return(0);
}

int_4 max5868_datdis(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x20,0x20);
  
  return(0);
}

int_4 max5868_daten(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,0x00,0x20);
  
  return(0);
}

int_4 max5868_mute(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x00,0x04,0x04);
  
  return(0);
}

int_4 max5868_unmute(Module_Config_Obj *mco)
{
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x00,0x00,0x04);
  
  return(0);
}

int_4 max5868_clkinv(Module_Config_Obj *mco, int_4 invert)
{
/* printf("JJB: max5868_clkinv = %d : 0x%x\n",invert,(invert<<12)); */
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,(invert<<12),0x1000);
  
  return(0);
}

int_4 max5868_synctype(Module_Config_Obj *mco, int_4 type)
{
/* printf("JJB: max5868_synctype = %d : 0x%x\n",type,(type<<13)); */
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_PWR,0x00,(type<<13),0x2000);
  
  return(0);
}

int_4 d2awgm3_calculate_rfpwr(int_4 adc_rfpwr, int_4 atten_qdb, real_8 freqmhz)
{
  /* RF power meter is log-linear meter, but the solution curve varies with */
  /* frequency. This table is used to get the linear (slope / y-intercept)  */
  /* fit coefficients, and freqmhz is used to interpolate the surface       */
  /* between frequencies. The result is accurate to about 1dB.              */
  enum { RFPWR_TBL_SZ = 26 };
  static real_8 d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ][3] = {
     {   0.0, 4.6821315E-02, -59.22189}, { 100.0, 4.7154218E-02, -61.85349}, 
     { 200.0, 4.7427673E-02, -62.97015}, { 300.0, 4.7674309E-02, -63.54081}, 
     { 400.0, 4.8336752E-02, -64.10522}, { 500.0, 4.8874997E-02, -64.74007}, 
     { 600.0, 4.9617149E-02, -65.32240}, { 700.0, 5.0067808E-02, -65.99976}, 
     { 800.0, 5.0547868E-02, -66.55621}, { 900.0, 5.0625544E-02, -66.93533}, 
     {1000.0, 5.0322998E-02, -67.01509}, {1100.0, 4.9655993E-02, -66.91844}, 
     {1200.0, 4.8734497E-02, -66.74814}, {1300.0, 4.7813214E-02, -66.36892}, 
     {1400.0, 4.6602238E-02, -65.97525}, {1500.0, 4.5890063E-02, -65.36910}, 
     {1600.0, 4.5359418E-02, -64.87202}, {1700.0, 4.4941839E-02, -64.26146}, 
     {1800.0, 4.4212226E-02, -63.76649}, {1900.0, 4.3545038E-02, -63.30266}, 
     {2000.0, 4.2666040E-02, -62.93183}, {2100.0, 4.2282421E-02, -62.67810}, 
     {2200.0, 4.1794870E-02, -62.33878}, {2300.0, 4.1251764E-02, -61.96976}, 
     {2400.0, 4.0923487E-02, -61.50374}, {2500.0, 4.0676787E-02, -61.17500} 
  };

  int_4  i,idx_low,idx_high;
  real_8 x1, x2, y1, y2, ym, yb;
  real_8 m_rf, b_rf, rfpwr_dbm, rffreq;

  /* Constrain input frequency to prevent overflow */
  rffreq = (freqmhz < d2awg_m3_rfpwr_tbl[0][0]) ? 
             (d2awg_m3_rfpwr_tbl[0][0]) : (freqmhz);
  rffreq = (freqmhz > d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ-1][0]) ? 
             (d2awg_m3_rfpwr_tbl[RFPWR_TBL_SZ-1][0]) : (freqmhz);

  /* Find frequency range of interest (in 1st nyquist zone) */
  idx_low = 0;
  for (i=0; i<RFPWR_TBL_SZ; i++) { 
      idx_low = (rffreq > d2awg_m3_rfpwr_tbl[i][0]) ? (i) : (idx_low);
  }
  idx_high = idx_low + 1;
  if (idx_high >= RFPWR_TBL_SZ) {
      idx_high = RFPWR_TBL_SZ - 1;
      idx_low  = RFPWR_TBL_SZ - 2;
  }

  /* Linearly interpolate the slope coefficient */
  x2 = d2awg_m3_rfpwr_tbl[idx_high][0];
  x1 = d2awg_m3_rfpwr_tbl[idx_low][0];
  y2 = d2awg_m3_rfpwr_tbl[idx_high][1];
  y1 = d2awg_m3_rfpwr_tbl[idx_low][1];
  ym = (y2-y1)/(x2-x1);
  yb = y2 - ym*x2; 
  m_rf = ym*rffreq + yb;

  /* Linearly interpolate the intercept coefficient */
  x2 = d2awg_m3_rfpwr_tbl[idx_high][0];
  x1 = d2awg_m3_rfpwr_tbl[idx_low][0];
  y2 = d2awg_m3_rfpwr_tbl[idx_high][2];
  y1 = d2awg_m3_rfpwr_tbl[idx_low][2];
  ym = (y2-y1)/(x2-x1);
  yb = y2 - ym*x2; 
  b_rf = ym*rffreq + yb;

  /* Calculate RF power at input of the attenuator */
  rfpwr_dbm = m_rf*((real_8) adc_rfpwr) + b_rf;

  /* Translate the measurement to output of attenuator (4x in dB) */
  rfpwr_dbm = rfpwr_dbm - (0.25*((real_8) atten_qdb));

  /* Limit the result to realistic integer range to prevent overflow */
  rfpwr_dbm = (rfpwr_dbm < -100.0) ? (100.0) : (rfpwr_dbm);
  rfpwr_dbm = (rfpwr_dbm >   40.0) ?  (40.0) : (rfpwr_dbm);

#ifdef DBG_PRINT
  printf("D2AWGM3_RFPWR_DEBUG: %d %d %d %20.15f %20.15f %20.15f \n",
           idx_low,idx_high,adc_rfpwr,atten_qdb,rffreq,m_rf,b_rf);
#endif

  return ((int_4) rfpwr_dbm);
}
 
int_4 d2awgm3_get_rfpwr(PICSTRUCT *p, int_4 mport)
{
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;
  real_8 freqmhz;
  int_4  cmfunc, cmpin, cmaddr, cmdata;
  int_4  adc_rfpwr, atten_db, rfpwrdbm;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) {
      printf("D2AWGM3_ERROR: GET RFPWR: Module object allocation failed! \n");
      return (-2);
  }
  pic_lock(p,LOCK_ALLOC);
  if (module_lock(D2AWGM3_module,0) < 0) {
      printf("D2AWGM3_ERROR: GET RFPWR: Command Module Lock Failed! \n");
      pic_lock(p,LOCK_FREE);
      return (-1);
  }

  /* Get current module state */
  d2awgm3_get_state(D2AWGM3_module);

  /* Convert current digital tuned output frequency */
  freq_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW];
  freq_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF];
  udb_i32wf_to_f64(&freq_state);
  freqmhz = freq_state.f64[0];

  /* Get current attenuator setting */
  cmdcfg_ctl(D2AWGM3_module,BAD_PE43704_ID,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_bufclr(D2AWGM3_module);
  cmaddr = 0;
  cmdata = 0;
  cmdcfg_stat(D2AWGM3_module,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  cmdcfg_bufclr(D2AWGM3_module);
  /* printf("CM: 0x%8.8X  0x%8.8X \n",cmaddr,cmdata); */
  atten_db = (cmaddr == 0x00000000) ? (cmdata & 0x000000FF) : (0);

  /* Read RF power meter ADC */
  cmdcfg_ctl(D2AWGM3_module,BAD_MCP3421_ID,0,PFUNC_RRD,0,0,0xFFFFFFFF);
  cmdcfg_bufclr(D2AWGM3_module);
  cmaddr = 0;
  cmdata = 0;
  cmdcfg_stat(D2AWGM3_module,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  cmdcfg_bufclr(D2AWGM3_module);
  /* printf("CM: 0x%8.8X  0x%8.8X \n",cmaddr,cmdata); */
  adc_rfpwr = (cmaddr == 0x00000000) ? ((cmdata & 0x0FFF0000)>>16) : (0);

  /* Calculate RF output power in dBm using calibrated table */
  rfpwrdbm = d2awgm3_calculate_rfpwr(adc_rfpwr, atten_db, freqmhz); 

  /* Set module state */
  /* D2AWGM3_module->LS.i32[D2AWGM3_RG_RFPWR] = rfpwrdbm; */
  d2awgm3_set_state(D2AWGM3_module);

  /* Release Command Lock */
  if (module_unlock(D2AWGM3_module, 0) < 0) {
      printf("D2AWGM3_ERROR: GET RFPWR: Command Module Unlock Failed!!\n");
      return (-1);
  }
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(rfpwrdbm);
}

int_4 d2awgm3_rdtemp(Module_Config_Obj *mco)
{
  int_4 cmfunc,cmpin,cmaddr,cmdata;

  cmdcfg_ctl(mco,BAD_MIC280_ID,0,PFUNC_RRD,1,0,0xFFFFFFFF);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,PTID_CMD,0,&cmfunc,&cmpin,&cmaddr,&cmdata);
  
  return(cmdata);
}

int_4 d2awgm3_get_temp(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB gain_state = udb_initializer;
  int_4 cmfunc,cmpin,cmaddr,cmdata;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: GET TEMP: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  cmdata = d2awgm3_rdtemp(D2AWGM3_module);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);
  return(cmdata);
}

int_4 d2awgm3_setgain(Module_Config_Obj *mco, int_4 gain, real_8 freq)
{
  int_4  gain_index, rgval;
  real_8 gain_correction;

  /* Simple 3rd order curve fit to gain slope correction: freq in MHz */
  gain_correction = (-1.750257E-3) + (1.610926E-2)*(freq) +
    (-7.883655E-6)*(freq*freq) + (1.6156237E-9)*(freq*freq*freq); 

  gain_correction = (gain_correction <  0.0) ?  (0.0) : (gain_correction);
  gain_correction = (gain_correction > 18.0) ? (18.0) : (gain_correction);

  gain_index = gain + ((int_4)(gain_correction + 0.5));
  gain_index = (gain_index < 0) ? (0) : gain_index;

  if(gain_index > (GAINTBLSIZE-1)) {
    printf("D2AWG_WARNING!! Maximum Normalized Gain Reached\n");
    gain_index = GAINTBLSIZE-1;
  }

#ifdef DBG_PRINT
  printf("D2AWG_GAIN_DEBUG: %20.5f %d %20.5f %d %d \n",
           freq, gain, gain_correction,
           d2awgm3_gaintbl[gain_index][0], 
           d2awgm3_gaintbl[gain_index][1]);
#endif

  /* Set attenuator register value */
  rgval = (d2awgm3_gaintbl[gain_index][0])<<2;
  cmdcfg_ctl(mco,BAD_PE43704_ID,0,PFUNC_RWR,0x00,rgval,0xFFFF);
  mco->LS.i32[D2AWGM3_RG_ATTN] = rgval;

  /* Set amplifier register value */
  rgval =  d2awgm3_gaintbl[gain_index][1];
  cmdcfg_ctl(mco,BAD_LMH6401_ID,0,PFUNC_RWR,0x02,rgval,0xFFFF);
  mco->LS.i32[D2AWGM3_RG_VGA] = rgval;

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(mco);

  return(0);
}

int_4 d2awgm3_set_gain(PICSTRUCT *p, int_4 mport, int_4 gain)
{

  /* Module Identification */
  int_4 rtn;
  real_8 m3attn,m3vga;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){ 
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n");
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  m3attn = finddblflagdef("M3ATTN",p->config,-1.0);
  m3vga  = finddblflagdef("M3VGA",p->config,-1.0);

  if((m3attn >= 0) && (m3vga >= 0)) {
    cmdcfg_ctl(D2AWGM3_module,BAD_LMH6401_ID,0,PFUNC_RWR,0x02,(int_4)(m3vga),0xFFFF);
    cmdcfg_ctl(D2AWGM3_module,BAD_PE43704_ID,0,PFUNC_RWR,0x00,(int_4)(m3attn*4),0xFFFF);
    D2AWGM3_module->LS.i32[D2AWGM3_RG_ATTN] = (int_4)(m3attn*4); 
    D2AWGM3_module->LS.i32[D2AWGM3_RG_VGA]  = (int_4)(m3vga); 
  } else {
    d2awgm3_get_state(D2AWGM3_module);
  
    freq_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW];
    freq_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF];
  
    udb_i32wf_to_f64(&freq_state);
  
    d2awgm3_setgain(D2AWGM3_module, gain, freq_state.f64[0]);
  
    D2AWGM3_module->LS.i32[D2AWGM3_RG_GAIN] = gain; /* Gain Frac Part */ 
  }
  d2awgm3_set_state(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

int_4 d2awgm3_set_mgain(PICSTRUCT *p, int_4 mport, int_4 gain)
{

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){ 
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n");
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  if(p->verbose) printf("SETTING D2AWGM3 MGAIN = %d\n",gain);
  cmdcfg_ctl(D2AWGM3_module,BAD_LMH6401_ID,0,PFUNC_RWR,0x02,gain,0xFFFF);
  D2AWGM3_module->LS.i32[D2AWGM3_RG_VGA]  = gain; 
  d2awgm3_set_state(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

int_4 d2awgm3_get_gain(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB gain_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  d2awgm3_get_state(D2AWGM3_module);

  gain_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_GAIN];

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(gain_state.i32[0]);
}

int_4 d2awgm3_get_mgain(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB gain_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  d2awgm3_get_state(D2AWGM3_module);

  gain_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_VGA];

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(gain_state.i32[0]);
}

int_4 d2awgm3_set_attn(PICSTRUCT *p, int_4 mport, int_4 attn)
{

  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){ 
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n");
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  if(p->verbose) printf("SETTING D2AWGM3 ATTN = %d\n",attn);
  cmdcfg_ctl(D2AWGM3_module,BAD_PE43704_ID,0,PFUNC_RWR,0x00,attn,0xFFFF);
  D2AWGM3_module->LS.i32[D2AWGM3_RG_ATTN]  = attn; 
  d2awgm3_set_state(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

int_4 d2awgm3_get_attn(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB gain_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
  }

  d2awgm3_get_state(D2AWGM3_module);

  gain_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_ATTN];

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(gain_state.i32[0]);
}

real_8 d2awgm3_max5868_tune(Module_Config_Obj *mco, int_8 rate, real_8 tfreq, int_4 specinv)
{
  real_8 fcw_full,fcw_rem;
  int_8 fcw,nfw,dfw;
  int_8 scale;
  int_4 fcw_0,fcw_1,fcw_2,fcw_3;
  int_4 nfw_0,nfw_1,nfw_2;
  int_4 dfw_0,dfw_1,dfw_2;
  int_4 cfncoe_0,cfncoe_1,cfncoe_2,cfncoe_3,cfncoe_4;
  int_4 cfgdsp;
  real_8 act_freq,act_vco;

  scale = pow(2,33);

  fcw_full = (tfreq / (real_8)rate)*scale;
  fcw = (int_8)fcw_full;
  fcw_rem = fcw_full - (real_8)fcw;
  nfw = (int_8)(fcw_rem * pow(2,18)-1);
  dfw = (int_8)((real_8)nfw/fcw_rem);

  act_vco = (real_8)fcw + (real_8)nfw / (real_8)dfw;
  act_freq = (((real_8)fcw + (real_8)nfw/(real_8)dfw) * (real_8)rate / scale);

  fcw_0 = fcw & 0xFF;
  fcw_1 = (fcw & 0xFF00)>>8;
  fcw_2 = (fcw & 0xFF0000)>>16;
  fcw_3 = (fcw & 0xFF000000)>>24;

  nfw_0 = nfw & 0xFF;
  nfw_1 = (nfw & 0xFF00)>>8;
  nfw_2 = (nfw & 0x030000)>>16;

  dfw_0 = dfw & 0xFF;
  dfw_1 = (dfw & 0xFF00)>>8;
  dfw_2 = (dfw & 0x070000)>>16;

  cfncoe_0 = nfw & 0xFF;
  cfncoe_1 = (nfw & 0xFF00)>>8;
  cfncoe_2 = ((dfw & 0x0003F)<<2) | ((nfw & 0x030000)>>16);
  cfncoe_3 = ((dfw & 0x03FC0)>>6);
  cfncoe_4 = ((dfw & 0x7C000)>>14);

/*
  printf("MAX5868 Tune: Desired Freq = %f, Actual Freq = %f\n", tfreq,act_freq);
  printf("MAX5868 Tune: fcw_full=%f, fcw=%ld, fcw_rem=%f, nfw=%ld, dfw=%ld\n",fcw_full,fcw,fcw_rem,nfw,dfw);
  printf("MAX5868 Tune FCW: fcw=%x, fcw_0=%x, fcw_1=%x, fcw_2=%x, fcw_3=%x\n",fcw,fcw_0,fcw_1,fcw_2,fcw_3);
  printf("MAX5868 Tune NFW: nfw=%x, nfw_0=%x, nfw_1=%x, nfw_2=%x\n",nfw,nfw_0,nfw_1,nfw_2);
  printf("MAX5868 Tune DCW: dfw=%x, dfw_0=%x, dfw_1=%x, dfw_2=%x\n",dfw,dfw_0,dfw_1,dfw_2);
  printf("MAX5868 Tune CFNCO: cfncoe_0=%x, cfncoe_1=%x, cfncoe_2=%x, cfncoe_3=%x, cfncoe_4=0x%x\n",cfncoe_0,cfncoe_1,cfncoe_2,cfncoe_3,cfncoe_4);
*/

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x02,fcw_0,0xFFFF);
  cmdcfg_delay(mco,10000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x03,fcw_1,0xFFFF);
  cmdcfg_delay(mco,10000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x04,fcw_2,0xFFFF);
  cmdcfg_delay(mco,10000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x05,fcw_3,0xFFFF);
  cmdcfg_delay(mco,10000);                             

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x60,0x18,0xFFFF);
  cmdcfg_delay(mco,10000);                             

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x6B,cfncoe_0,0xFFFF);
  cmdcfg_delay(mco,1000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x6C,cfncoe_1,0xFFFF);
  cmdcfg_delay(mco,1000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x6D,cfncoe_2,0xFFFF);
  cmdcfg_delay(mco,1000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x6E,cfncoe_3,0xFFFF);
  cmdcfg_delay(mco,1000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x6F,cfncoe_4,0xFFFF);
  cmdcfg_delay(mco,1000);                             

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x60,0x00,0xFFFF);
  cmdcfg_delay(mco,1000);                             

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x01,0x01,0x0001);
  cmdcfg_delay(mco,1000);                             

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x00,0x00,0x01);
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x51,specinv<<1,0x02);

  cmdcfg_delay(mco,1000);                             

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return(act_freq);
}

real_8 d2awgm3_max5868(Module_Config_Obj *mco, int_8 rate, real_8 tfreq, int_4 interp, int_4 specinv, int_4 mute_offset)
{
  real_8 fcw_full,fcw_rem;
  int_8 fcw,nfw,dfw;
  int_8 scale;
  int_4 fcw_0,fcw_1,fcw_2,fcw_3;
  int_4 nfw_0,nfw_1,nfw_2;
  int_4 dfw_0,dfw_1,dfw_2;
  int_4 cfncoe_0,cfncoe_1,cfncoe_2,cfncoe_3,cfncoe_4;
  int_4 cfgdsp;
  real_8 act_freq;

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x100,mute_offset,0xFFFF);

  cfgdsp = 0x00;
  if(interp==24) cfgdsp = 0xF0;
  else if(interp==20) cfgdsp = 0xE0;
  else if(interp==16) cfgdsp = 0xD0;
  else if(interp==12) cfgdsp = 0xC0;
  else if(interp==10) cfgdsp = 0x00;
  else if(interp==8) cfgdsp = 0xB0;
  else if(interp==6) cfgdsp = 0xA0;
  else if(interp==5) cfgdsp = 0x90;
  else cfgdsp = 0x80;

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x01,(cfgdsp|0x0C),0xFFFF);
  cmdcfg_delay(mco,10000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x01,(cfgdsp|0x04),0xFFFF);
  cmdcfg_delay(mco,10000);                             
  max5868_clken(mco);
  cmdcfg_delay(mco,10000);                             
  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x01,(cfgdsp|0x00),0xFFFF);
  cmdcfg_delay(mco,10000);                             

  act_freq = d2awgm3_max5868_tune(mco, rate, tfreq, specinv);

  cmdcfg_ctl(mco,BAD_MAX5868_ID,0,PFUNC_RWR,0x01,(cfgdsp|0x01),0xFFFF);
  cmdcfg_delay(mco,1000);                             

  max5868_daten(mco);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return(act_freq);
}

int_4 d2awgm3_lmx2582_mix(Module_Config_Obj *mco, int_4 mixpd)
{
  int_4 (*lmx2582_regs)[2];

  lmx2582_regs = &lmx2582_base_table[0];

  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xFF7F) | (mixpd<<7);
  lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xFBFF) | (mixpd<<10);

/* printf("JJB::: Setting mixpd to %d\n",mixpd); */
  cmdcfg_ctl(mco,BAD_LMX2582_ID,0,PFUNC_RWR,lmx2582_regs[6][0],lmx2582_regs[6][1],0xFFFF);
  /* cmdcfg_ctl(mco,BAD_LMX2582_ID,0,PFUNC_RWR,lmx2582_regs[21][0],lmx2582_regs[21][1],0xFFFF); */
  /* cmdcfg_ctl(mco,BAD_LMX2582_ID,0,PFUNC_RWR,lmx2582_regs[42][0],lmx2582_regs[42][1],0xFFFF); */

  return(0);
}

int_4 d2awgm3_setfreq(Module_Config_Obj *mco, real_8 datfreq)
{
  int_8 dac_rate;
  int_4 mixpd;
  int_4 dblen;
  int_4 specinv;
  UDB rate_state = udb_initializer;
  UDB freq_state = udb_initializer;

  d2awgm3_get_state(mco);

  rate_state.i32[0] = mco->LS.i32[D2AWGM3_RG_RATEL];
  rate_state.i32[1] = mco->LS.i32[D2AWGM3_RG_RATEU];

  dac_rate = rate_state.i64[0];
  
  if(datfreq > 3*(dac_rate/2)) {
    mixpd = 0;
    dblen = 1;
    specinv = (datfreq > 2*dac_rate)? 0 : 1;
    datfreq = fabs(2*(real_8)dac_rate - datfreq);
  } else if(datfreq > (dac_rate/2)) {
    mixpd = 0;
    dblen = 0;
    specinv = (datfreq > dac_rate)? 0 : 1;
    datfreq = fabs((real_8)dac_rate - datfreq);
  } else {
    mixpd = 1;
    dblen = 0;
    specinv = 0;
  }

  if(mixpd) {
    /* First Switch */ 
    cmdcfg_ctl(mco,BAD_PGEN_ID,0,PFUNC_PWR,0x00,0x00,0xFFFF);

    /* Second Switch */ 
    cmdcfg_ctl(mco,BAD_PGEN_ID,1,PFUNC_PWR,0x00,0x01,0xFFFF);

    /* Mixer */
    cmdcfg_ctl(mco,BAD_PGEN_ID,2,PFUNC_PWR,0x00,0x00,0xFFFF);
  } else {
    /* First Switch */ 
    cmdcfg_ctl(mco,BAD_PGEN_ID,0,PFUNC_PWR,0x00,0x01,0xFFFF);

    /* Second Switch */ 
    cmdcfg_ctl(mco,BAD_PGEN_ID,1,PFUNC_PWR,0x00,0x00,0xFFFF);

    /* Mixer */
    cmdcfg_ctl(mco,BAD_PGEN_ID,2,PFUNC_PWR,0x00,(dblen<<1)|0x01,0xFFFF);
  }

  d2awgm3_lmx2582_mix(mco,mixpd);
  freq_state.f64[0] = d2awgm3_max5868_tune(mco, dac_rate, datfreq, specinv);
  
  udb_f64_to_i32wf(&freq_state);
  mco->LS.i32[2] = freq_state.i32[0];
  mco->LS.i32[3] = freq_state.i32[1];

  return(0);
}

int_4 d2awgm3_set_freq(PICSTRUCT *p, int_4 mport, real_8 freq)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET FREQ: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  d2awgm3_setfreq(D2AWGM3_module, freq*1e6);

  d2awgm3_set_state(D2AWGM3_module);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

real_8 d2awgm3_get_freq(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB freq_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  d2awgm3_get_state(D2AWGM3_module);

  freq_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW];
  freq_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF];

  udb_i32wf_to_f64(&freq_state);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(freq_state.f64[0]/1e6);
}

int_4 d2awgm3_rdclkstat (Module_Config_Obj *mco)
{
  int_4 status;
  int_4 stat_func,stat_pin,stat_addr,stat_data;
  int_4 extref;

  cmdcfg_ctl(mco,BAD_MOD_ID,0,PFUNC_PRD,1,0,0xFF);
  cmdcfg_ctl(mco,BAD_PGEN_ID,3,PFUNC_PRD,0,0,0xFF);
  cmdcfg_bufclr(mco);
  cmdcfg_stat(mco,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  cmdcfg_stat(mco,BAD_CMD_ID,0,&stat_func,&stat_pin,&stat_addr,&extref);

  status = 0;
  if ((extref==0) && (stat_data&0x1)==0) status |= 0x1;
  if ((stat_data&0x2)==0) status |= 0x2;

  return(status);
}

int_4 d2awgm3_reset(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x05,0xFF);
  cmdcfg_delay(D2AWGM3_module,500);
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_delay(D2AWGM3_module,500);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  /* Check if Session is still locked */
  return (0);
}

int_4 d2awgm3_dreset(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  /* Disable TX/RX Chains */
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);

  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x01,0xFF);
  cmdcfg_delay(D2AWGM3_module,500);
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  cmdcfg_delay(D2AWGM3_module,500);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  /* Check if Session is still locked */
  return (0);
}

int_4 d2awgm3_findrate(int_4 rate, int_8 *dac_rate, int_8 *actrate, int_8 *fpga_srate, int_4 *dac_interp, int_4 *fpga_interp, int_4 hsmode, int_4 polyfir)
{
  /* const int_8 MAX_DACRATE = 4960000000; */
  /* const int_8 MIN_DACRATE = 3550000000; */
  int_8 max_fpgarate;

  int_4 i,j,tblszd,tblszf;
  int_4 *dac_interp_; 
  int_4 *fpga_interp_;
  int_8 interp_rate;
  int_8 cur_interp_rate;
  int_8 fpga_rate_max;
  int_8 inrate,cur_rate;
  int_8 error,cur_error;
  int_8 fpga_rate;

  /* max_fpgarate = (hsmode)? 500000000 : 315000000; */
  max_fpgarate = (hsmode)? MAXFPGARATEHS : MAXFPGARATELS;
  dac_interp_ = d2awgm3_dac_interp;
  fpga_interp_ = (polyfir)? d2awgm3_fpga_interp_pp : d2awgm3_fpga_interp; 

  tblszd = sizeof(d2awgm3_dac_interp)/sizeof(d2awgm3_dac_interp[0]);
  tblszf = (polyfir)? sizeof(d2awgm3_fpga_interp_pp)/sizeof(d2awgm3_fpga_interp_pp[0]) : 
                      sizeof(d2awgm3_fpga_interp)/sizeof(d2awgm3_fpga_interp[0]);

  if(hsmode) { 
    fpga_interp_ = (polyfir)? &d2awgm3_fpga_interp_pp[tblszf-1] :
                              &d2awgm3_fpga_interp[tblszf-1];
    fpga_interp_ = &d2awgm3_fpga_interp[tblszf-1];
  }

  inrate = rate;
  interp_rate = 0;
  fpga_rate_max = 0;
  error = -1;
  for(i=0; i<tblszd; i++) {
    for(j=0; j<tblszf; j++) {
      cur_rate = inrate;
      cur_error = 0;

      fpga_rate = inrate * fpga_interp_[j];
      if (fpga_rate > max_fpgarate) {
         if (error == 0) continue;
         fpga_rate = max_fpgarate;
         cur_rate = fpga_rate / fpga_interp_[j];
         cur_error = inrate-cur_rate;
      }

      cur_interp_rate = fpga_rate * dac_interp_[i];
      if (cur_interp_rate > MAXDACRATE) {
         if (error == 0) continue;
         cur_interp_rate = MAXDACRATE;
         cur_rate = cur_interp_rate / dac_interp_[i];
         cur_rate /= fpga_interp_[j];
         cur_error = inrate-cur_rate;
      } else if (cur_interp_rate < MINDACRATE) {
         if (error == 0) continue;
         cur_interp_rate = MINDACRATE;
         cur_rate = cur_interp_rate / dac_interp_[i];
         if (cur_rate > max_fpgarate) continue;
         cur_rate /= fpga_interp_[j];
         cur_error = cur_rate-inrate;
      }

      /* printf("JJB ::: error = %d, rate = %d, dac_interp = %d, fpga_interp = %d, interp_rate = %lu, fpga_rate = %lu\n", cur_error, cur_rate, dac_interp_[i], fpga_interp_[j], cur_interp_rate, fpga_rate); */
      if (cur_error != 0) {
         if (error > 0 && cur_error>=error) continue;
      } else if (error <= 0) {
         if (fpga_rate <= fpga_rate_max) continue;
      }

      error = cur_error;
      interp_rate = cur_interp_rate;
      if (error == 0) fpga_rate_max = fpga_rate;
      *dac_interp = dac_interp_[i];
      *fpga_interp = fpga_interp_[j];
    }
  }

  *dac_rate = interp_rate;
  *actrate = interp_rate/(*dac_interp)/(*fpga_interp);
  *fpga_srate = *dac_rate/(*dac_interp);

  /* printf("JJB::: D2AWGM3, DAC Rate = %lu, DAC Interp = %d, FPGA Interp = %d, FPGA Rate = %lu, %lu\n", *dac_rate, *dac_interp, *fpga_interp, max_fpgarate, *dac_rate/(*dac_interp)); */
  if(*actrate != rate) {
    printf("WARNING!!! Unable to meet rate requirement of %f Msps, setting rate to %f Msps\n", ((double)rate)/1e6, ((double)(*actrate))/1e6);
  }
/*  printf("JJB::: dac_rate = %lu, act_rate = %ld, dac_interp = %d, fpga_interp = %d, fpga_rate = %ld\n", *dac_rate, *actrate, *dac_interp,*fpga_interp,fpga_rate); */

  return(0);

}

int_4 d2awgm3_findrate_fixdac(int_8 dac_rate, int_4 rate, int_8 *actrate, int_8 *fpga_srate, int_4 *dac_interp, int_4 *fpga_interp, int_4 hsmode, int_4 polyfir)
{
  int_4 (*d2awgm3_interp)[2];
  int_4 interp;
  int_8 max_fpgarate;
  int_4 tblszf,tblszd;
  int_4 *dac_interp_; 
  int_4 *fpga_interp_;
  int_8 fpga_rate;
  int_4 i,j;
  int_4 rdiff,rdiff_,ind;
  int_4 tmin1,tmin2;
  int_4 dac_int,fpga_int,arate;


  max_fpgarate = (hsmode)? MAXFPGARATEHS : MAXFPGARATELS;
  dac_interp_ = d2awgm3_dac_interp;
  fpga_interp_ = (polyfir)? d2awgm3_fpga_interp_pp : d2awgm3_fpga_interp; 
  rdiff = 500000000;

  tblszd = sizeof(d2awgm3_dac_interp)/sizeof(d2awgm3_dac_interp[0]);
  tblszf = (polyfir)? sizeof(d2awgm3_fpga_interp_pp)/sizeof(d2awgm3_fpga_interp_pp[0]) : 
                      sizeof(d2awgm3_fpga_interp)/sizeof(d2awgm3_fpga_interp[0]);

  if(hsmode) { 
    fpga_interp_ = (polyfir)? &d2awgm3_fpga_interp_pp[tblszf-1] :
                              &d2awgm3_fpga_interp[tblszf-1];
    tblszf = 1;
  }

  for(i=0;i<tblszd;i++) {
    if((dac_rate/dac_interp_[i]) > max_fpgarate) continue;    
    for(j=0;j<tblszf;j++) {

      interp = dac_interp_[i] * fpga_interp_[j]; 
      rdiff_ = abs(rate - dac_rate/interp);
      /* printf("JJB::: %d : desired = %d, calculated = %d, interp = %d, mindiff = %d, diff = %d\n",i,rate,dac_rate/interp,interp,rdiff,rdiff_); */
      if(rdiff_ < rdiff) {
        rdiff = rdiff_;
        *dac_interp = dac_interp_[i];
        *fpga_interp = fpga_interp_[j];
      }

    }
  }

  arate = dac_rate/((*dac_interp) * (*fpga_interp));

  if(arate != rate) printf("\n!!!WARNING!!! - Desired Rate of %f Msps cannot be met, using %f Msps\n\n",rate/1e6,arate/1e6);

  /* printf("JJB::: %d: dac_interp = %d, fpga_interp = %d, desired = %d, actual = %d\n",ind,d2awgm3_interp[ind][1],d2awgm3_interp[ind][0],rate,arate);  */
  /* printf("JJB::: D2AWGM3, DAC Rate = %lu, DAC Interp = %d, FPGA Interp = %d, FPGA Rate = %lu\n", dac_rate, *dac_interp, *fpga_interp, dac_rate/(*dac_interp)); */

  *actrate = arate;
  *fpga_srate = dac_rate/(*dac_interp);

  return(0);
}

int_4 d2awgm3_coewr(Module_Config_Obj *module_cfg, int_4 part_addr, int_4 *coe, int_4 taps, int_4 interp)
{

  int_4 part_addr0 = 0;
  int_4 part_addr1 = 0;
  int_4 part_reg0;
  int_4 part_reg1;
  int_4 *coeptr;
  int_4 i,j;
  double time;
 
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  part_addr0 = (part_addr<<8);
  part_addr1 = (part_addr<<8)+4;
  part_reg1 = (PFUNC_RWR<<12)|14;

  iom_init(p,mport);
  time = gettime();
  coeptr = coe;
  for(i=0;i<taps;i++){
    for(j=0;j<interp;j++){
      if (p->type==ICEPIC7 || p->type==ICEPIC8) {
	iom_prt_wr(p,mport, part_addr, part_reg1, (*(coeptr++) & 0x3FFFF)<<8|(i<<26)|j );
      } else {
	pic_jpmemwr(p,mport,part_addr0,(*(coeptr++) & 0x3FFFF)<<8|(i<<26)|j,4);
	pic_jpmemwr(p,mport,part_addr1,part_reg1,4);
      }
    }
  }
  iom_prt_flush(p,mport,part_addr);
  time = gettime()-time;
  if (p->verbose>1) printf("PolyFilt load ntap=%d interp=%d took=%f\n",taps,interp,time);
  return(0);
}

int_4 d2awgm3_polyfilt(Module_Config_Obj *mco, int_4 interp)
{
  int_4 i,j;
  int_4 coe_wradr,coe_wrdat,coe_wr;
  int_4 *coefs;
  int_4 taps = 22;
  int_4 stat_func, stat_pin, stat_addr, stat_data;

  if(interp==1) coefs = d2awgm3_polyfir_1;
  else if(interp==2) coefs = d2awgm3_polyfir_2;
  else if(interp==3) coefs = d2awgm3_polyfir_3;
  else if(interp==4) coefs = d2awgm3_polyfir_4;
  else if(interp==8) coefs = d2awgm3_polyfir_8;
  else if(interp==10) coefs = d2awgm3_polyfir_10;
  else if(interp==12) coefs = d2awgm3_polyfir_12;
  else if(interp==16) coefs = d2awgm3_polyfir_16;
  else if(interp==20) coefs = d2awgm3_polyfir_20;
  else if(interp==32) coefs = d2awgm3_polyfir_32;
  else if(interp==48) coefs = d2awgm3_polyfir_48;
  else if(interp==50) coefs = d2awgm3_polyfir_50;
  else if(interp==64) coefs = d2awgm3_polyfir_64;
  else if(interp==72) coefs = d2awgm3_polyfir_72;
  else if(interp==84) coefs = d2awgm3_polyfir_84;
  else if(interp==96) coefs = d2awgm3_polyfir_96;
  else if(interp==128) coefs = d2awgm3_polyfir_128;
  else coefs = d2awgm3_polyfir_2;

  cmdcfg_ctl(mco,BAD_HBINTFIR_ID,0,PFUNC_RWR,10,16,0xFFFF);
  cmdcfg_ctl(mco,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,0,0xFFFF);
  cmdcfg_ctl(mco,BAD_HBINTFIR_ID,0,PFUNC_RWR,0,interp,0xFFFF);
  d2awgm3_coewr(mco, HBINTFIR_ADDR, coefs, taps,interp);

  return(0);
}


int_4 d2awgm3_get_dbg(PICSTRUCT *p, int_4 mport)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  int_4 i,k,l;
  int_4 memdat;
  int_4 dbgrst,dbgsel,trigsel,datasel,mdepth,displ,aoff,hbidbg;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 CMDDEPTH = 64;
  FILE *file0,*file1,*file2,*file3,*file4,*file5;
 
  int_4 dbgSig0[250],dbgSig1[250];
  int_4 dbgSig2[250],dbgSig3[250],dbgSig5[250];
  int_8 dbgSig4[250];
  union dbgMem dbgArray[500];

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  dbgrst  = findintflagdef("DBGRST",p->config,1);
  dbgsel  = findintflagdef("DBGSEL",p->config,0);
  aoff    = findintflagdef("DBGAOFF",p->config,0);
  trigsel = findintflagdef("TRIGSEL",p->config,0);
  datasel = findintflagdef("DATASEL",p->config,0);
  mdepth  = findintflagdef("DBGDEPTH",p->config,500);
  displ   = findintflagdef("DBGDISPL",p->config,0);
  hbidbg  = findintflagdef("HBDBG",p->config,7);

  if(mdepth>500) mdepth=500;

/* printf("JJB::: Debug mdepth=%d\n",mdepth); */

  cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_RWR,2,hbidbg,0xFF);

  if(dbgrst) {
    d2awgm3_dbgsetup(D2AWGM3_module, dbgsel, trigsel, datasel);
    cmdcfg_delay(D2AWGM3_module,2000);
  }

  cmdcfg_ctl(D2AWGM3_module,BAD_MEM_ID,dbgsel,PFUNC_PRD,0,0,0xFFFFFFFF);
  k = 0;
  for(i=0;i<mdepth;i++) {
    if(i%CMDDEPTH == CMDDEPTH-1) {
      /* printf("JJB:: clearing command buffer, %d...\n",i); */
      cmdcfg_bufclr(D2AWGM3_module);
    }
    cmdcfg_ctl(D2AWGM3_module,BAD_MEM_ID,dbgsel,PFUNC_RRD,i+aoff,0,0xFFFFFFFF);
    k+=2;
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  cmdcfg_stat(D2AWGM3_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
/* printf("JJB::: MEM Pin Stat = 0x%x\n",stat_data); */
  file0 = fopen("d2awgm3_dbg0","ab");
  file1 = fopen("d2awgm3_dbg1","ab");
  file2 = fopen("d2awgm3_dbg2","ab");
  file3 = fopen("d2awgm3_dbg3","ab");
  file4 = fopen("d2awgm3_dbg4","ab");
  file5 = fopen("d2awgm3_dbg5","ab");

  k = 0;
  l = 0;
  for(i=0;i<mdepth;i+=2) {
    cmdcfg_stat(D2AWGM3_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
    dbgArray[k].dbgWord[0] = stat_data;
    cmdcfg_stat(D2AWGM3_module,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
    dbgArray[k].dbgWord[1] = stat_data;

    /*dbgSig_I[l] = dbgArray[k].dbgSig.datI;*/
    /*dbgSig_Q[l++] = dbgArray[k].dbgSig.datQ;*/

    dbgSig0[l] = dbgArray[k].dbgSig16.dat0;
    dbgSig1[l] = dbgArray[k].dbgSig16.dat1;
    dbgSig2[l] = dbgArray[k].dbgSig16.dat2;
    dbgSig3[l] = dbgArray[k].dbgSig16.dat3;
    dbgSig4[l] = dbgArray[k].dbgSig1.dat0;
    dbgSig5[l] = dbgArray[k].dbgSig1.dat2;
    k=k+1; l=l+1;

    /* printf("MEMDAT:: %d = 0x%0.4x, 0x%0.4x, 0x%0.8x\n",k-1,dbgArray[k-1].dbgSig.datI,dbgArray[k-1].dbgSig.datQ,dbgArray[k-1].dbgSig.dummy); */
    if(displ==0){ 
      printf("MEMDAT:: %d = 0x%0.4x, 0x%0.4x, 0x%0.4x, 0x%0.4x\n",k-1,dbgArray[k-1].dbgSig16.dat0,dbgArray[k-1].dbgSig16.dat1,
                                                                    dbgArray[k-1].dbgSig16.dat2,dbgArray[k-1].dbgSig16.dat3);
    }else if(displ==1){
      printf("MEMDAT:: %d = 0x%0.8x, 0x%0.5x, 0x%0.4x, 0x%0.4x\n",k-1,dbgArray[k-1].dbgSig0.dat0,dbgArray[k-1].dbgSig0.dat1,
                                                                      dbgArray[k-1].dbgSig0.dat2,dbgArray[k-1].dbgSig0.dat3);
    }else if(displ==2){
      printf("MEMDAT:: %d = 0x%0.9x, 0x%0.4x, 0x%0.4x\n",k-1,dbgArray[k-1].dbgSig1.dat0,dbgArray[k-1].dbgSig1.dat1,
                                                             dbgArray[k-1].dbgSig1.dat2);
    }
  }

  /*fwrite(&dbgSig_I,sizeof(int_4),250,file0);*/
  /*fwrite(&dbgSig_Q,sizeof(int_4),250,file1);*/
  fwrite(&dbgSig0,sizeof(int_4),250,file0);
  fwrite(&dbgSig1,sizeof(int_4),250,file1);
  fwrite(&dbgSig2,sizeof(int_4),250,file2);
  fwrite(&dbgSig3,sizeof(int_4),250,file3);
  fwrite(&dbgSig4,sizeof(int_8),250,file4);
  fwrite(&dbgSig5,sizeof(int_4),250,file5);

  fclose(file0);
  fclose(file1);
  fclose(file2);
  fclose(file3);
  fclose(file4);
  fclose(file5);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  /* Check if Session is still locked */
  return (0);
}
int_4 d2awgm3_lmx2582(Module_Config_Obj *mco, int_8 rate, int_4 mixpd)
{
  int_4 i;
  /* int_4 (*lmx2582_regs)[2]; */
  int_4 lmx2582_regs[LMX2582_CONFIG_REGS][2];
  int_4 Fpd = 160000000;
  int_4 PLL_N_PRE = 2;
  int_4 PLL_DEN = 1000000;
  real_8 Fvco_, Fvco_rem;
  int_4 PLL_N,PLL_NUM;

  /* lmx2582_regs = &lmx2582_base_table[0]; */
  for(i=0;i<LMX2582_CONFIG_REGS;++i) {
    lmx2582_regs[i][0] = lmx2582_base_table[i][0];
    lmx2582_regs[i][1] = lmx2582_base_table[i][1];
  }

  cmdcfg_ctl(mco,BAD_LMX2582_ID,0,PFUNC_PWR,0,0x01,0xFFFF);

  if(rate > MAXDACRATE){ /*4960000000){ */
    printf("WARNING!!! %ld Msps is an Invalid DAC Rate, Maximum DAC Rate is %d Msps\n",rate/1e6,MAXDACRATE/1e6);
    rate = MAXDACRATE; /*4960000000;*/
  }else if(rate < MINDACRATE){ /*3550000000){*/
    printf("WARNING!!! %ld Msps is an Invalid DAC Rate, Minimum DAC Rate is %d Msps\n",rate/1e6,MINDACRATE/1e6);
    rate = MINDACRATE; /* 3550000000;*/
  }

  if(rate < MINVCORATE) {
    /* printf("JJB:: Using Channel Divider Path, Rate = %ld\n",rate); */

    rate = 2*rate;

    /* Power Down VCO to Output Buffer : Reg 31, bit 9=1 A, bit 10=1 B */
    /* Power Up VCO to Channel Divider : Reg 31, bit 7=0 */
    lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xF97F) | (3<<9);

    /* Enable Channel Divider : Reg 34, bit 5=1 */
    lmx2582_regs[18][1] = (lmx2582_regs[18][1] & 0xFFDF) | (1<<5);

    /* Enable SEG1 Channel Divider : Reg 35, bit 1=1 */
    /* Disable SEG2 Channel Divider : Reg 35, bit 7=0 */
    /* Disable SEG3 Channel Divider : Reg 35, bit 8=0 */
    /* Set Channel Divider (SEG1) to Divide by 2 : Reg 35, bit 2=0 */
    lmx2582_regs[17][1] = (lmx2582_regs[17][1] & 0xFE79) | (1<<1);

    /* Enable Buffer from Channel Divider to Selection Mux : Reg 36, bit 10=1 A, bit 11=1 B*/
    /* Configure Channel Divider Mux for SEG1 : Reg 36, bit 4:6=1 */
    lmx2582_regs[16][1] = (lmx2582_regs[16][1] & 0xFF8F) | (1<<4);

    /* Select Ouput Mux A : Reg 47, bit 11:12=0 */
    lmx2582_regs[5][1] = (lmx2582_regs[5][1] & 0xE7FF);

    /* Select Ouput Mux B : Reg 48, bit 0:1=0 */
    lmx2582_regs[4][1] = (lmx2582_regs[4][1] & 0xFFFC);
  }

  Fvco_ = ((real_8)rate) / ((real_8)(Fpd*PLL_N_PRE));
  Fvco_rem = Fvco_ - (int_4)Fvco_;

  PLL_N = (int_4)Fvco_;
  PLL_NUM = (int_4)(Fvco_rem * PLL_DEN);

  lmx2582_regs[14][1] = (PLL_N & 0xFFF)<<1;
  lmx2582_regs[7][1]  = (PLL_NUM & 0x0000FFFF);
  lmx2582_regs[8][1]  = (PLL_NUM & 0xFFFF0000)>>16;

  /* Handle the Mix Path Powerdown Case */
  lmx2582_regs[6][1]  = (lmx2582_regs[6][1] & 0xFF7F) | (mixpd<<7);
  lmx2582_regs[21][1] = (lmx2582_regs[21][1] & 0xFBFF) | (mixpd<<10);

/* printf("JJB::: lmx reg 0x%.2x = 0x%.4x \n",lmx2582_regs[6][0],lmx2582_regs[6][1]); */
/* printf("JJB::: lmx reg 0x%.2x = 0x%.4x \n",lmx2582_regs[21][0],lmx2582_regs[21][1]); */

  /* printf("JJB::: Fvco_ = %f, Fvco_rem = %f, PLL_N = %d, PLL_NUM = %d\n",Fvco_,Fvco_rem,PLL_N,PLL_NUM); */
  /* printf("JJB::: %d, %d, %d\n",lmx2582_regs[14][1],lmx2582_regs[7][1],lmx2582_regs[8][1]); */

  for (i=0; i<LMX2582_CONFIG_REGS; i++) {
    /* printf("JJB::: LMX2582 : ADDR=0x%x, Data=0x%x\n",lmx2582_regs[i][0],lmx2582_regs[i][1]); */
    cmdcfg_ctl(mco,BAD_LMX2582_ID,0,PFUNC_RWR,lmx2582_regs[i][0],lmx2582_regs[i][1],0xFFFF);
  } 

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return(0);
}

int_4 d2awgm3_si5344(Module_Config_Obj *mco, real_8 rate, int_4 extref)
{
  PT_SI5344_REG SIR[SI5344_MAX_CFG_RG];
  int_4 i,rtn,cmdcnt;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  uint8_t last_page;

  /* Initialize the Si5344 configuration */
  pt_si5344_rgc_default(&SIR[0], FPGA_MODID_D2AWGM3);

  /* Configure all the registers on the part */
  cmdcnt = 0;
  last_page = 255;
  for(i=0; i<SI5344_MAX_CFG_RG; i++){
    if(SIR[i].rp != last_page) {
      /* Need to update page register */
      /* printf("UPDATE PAGE to 0x%x\n",SIR[i].rp); */
      cmdcfg_ctl(mco,BAD_SI5344_ID,0,PFUNC_RWR,SI5344_RG_PAGE,SIR[i].rp,0xFF);
      last_page = SIR[i].rp;
      cmdcnt++;
    }

    /* printf("SI5344: 0x%0.2x:0x%0.2x\n",SIR[i].ra,SIR[i].rv); */
    cmdcfg_ctl(mco,BAD_SI5344_ID,0,PFUNC_RWR,SIR[i].ra,SIR[i].rv,0xFF);

    if(++cmdcnt > 255) {
      /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
      /* printf("CLEARING CMD BUFFER...\n"); */
      cmdcfg_bufclr(mco);
      cmdcnt = 0;
    }
  }

  if (extref) {
    /* printf("Enable External Reference...\n"); */
    cmdcfg_ctl(mco,BAD_SI5344_ID,0,PFUNC_RWR,SI5344_RG_PAGE,0x05,0xFF);
    cmdcfg_ctl(mco,BAD_SI5344_ID,0,PFUNC_RWR,0x2A,0x05,0xFF);
  }

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return (0);
}


int_4 d2awgm3_si5338(Module_Config_Obj *mco, real_8 rate, int_4 extref)
{
  PT_SI5338_REG SIR[SI5338_MAX_CFG_RG];
  PT_SI5338_MSCFG MS;
  int_4 i,rtn;
  int_4 stat_func, stat_pin, stat_addr, stat_data;
  int_4 reg_data45, reg_data46, reg_data47;
  real_8 rate_act;

/* printf("JJB: Si5338 Rate = %f\n",rate); */

  /* Initialize the Si5338 configuration */
  pt_si5338_rgc_default(&SIR[0], FPGA_MODID_D2AWGM3);

  /* Configure internal / external reference (assume internal) */
  if (extref) {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x00)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x00 >> 3)),0);
  }
  else {
    pt_si5338_rgc_setbits(&SIR[0],28,0x1C,((0x07)&(0x0A)),0);
    pt_si5338_rgc_setbits(&SIR[0],29,0x18,((0x03)&(0x0A >> 3)),0);
  }
  pt_si5338_rgc_setbits(&SIR[0],28,0x03,0x00,0);

  /* Configure input to P2_DIVIDER (Zerodly clock reference) */
  pt_si5338_rgc_setbits(&SIR[0],28,0x20,((0x01)&(0x00)),0);
  pt_si5338_rgc_setbits(&SIR[0],30,0x18,((0x03)&(0x00 >> 1)),0);

  /* Configure a divider solution for the MSx outputs */
  MS.dbglvl = 0;     
  MS.inp_ref_frq    = 10.0; /* BAD BAD BAD - don't assume 10MHz ref clock */
  MS.clk_frq_des[0] = 200.0;           /* RF VCO reference      */
  MS.clk_frq_des[1] = MS.inp_ref_frq;  /* Unused output         */
  MS.clk_frq_des[2] = rate;            /* RF Synth clock        */
  MS.clk_frq_des[3] = MS.inp_ref_frq;  /* Zero delay mode clock */
  MS.clk_frq_des[4] = MS.inp_ref_frq;  /* Unused output         */
  MS.ms_critical    = 2;               /* Zerodly is critical path */
  rtn = pt_si5338_msx_soln(&SIR[0], &MS);
  if (rtn<0) {
    dbgerrormsg("Invalid clock configuration specified. Bad... \n");
    return(-2); 
  }

  /* Disable outputs and pause LOL */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,230,1<<4,1<<4); 
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,241,1<<7,1<<7);

  /* Configure all the registers on the part */
  for(i=0; i<SI5338_MAX_CFG_RG; i++){
    cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,SIR[i].ra,SIR[i].rv,SIR[i].rm);
  }

  /* Cfg PLL for Lock, initiate PLL lock, Restart LOL, Wait 25ms */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,49,0,1<<7);     
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,246,1<<1,1<<1);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,241,0x65,0xFF);
  cmdcfg_delay(mco,25000);                             

  /* Validate Input Clock Status */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,218,0,0xFFFFFFFF);

  /* Confirm PLL Lock */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,218,0,0xFF);

  /* Copy FCAL Values To Active Registers */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,237,0,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,236,0,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RRD,235,0,0xFF);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  /* Check lock status */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  /* printf("Si5338 Clock Status = %x\n",stat_data); */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  /* printf("Si5338 PLL Lock Status = %x\n",stat_data); */
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data47 = stat_data & 0x03;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data46 = stat_data & 0xFF;
  cmdcfg_stat(mco,PTID_CMD,0,&stat_func,&stat_pin,&stat_addr,&stat_data);
  reg_data45 = stat_data & 0xFF;
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,47,reg_data47,0x03);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,46,reg_data46,0xFF);
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,45,reg_data45,0xFF);

  /* Set PLL To Use FCAL Values */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,49,1<<7,1<<7);

  /* Enable Outputs */
  cmdcfg_ctl(mco,BAD_SI5338_ID,0,PFUNC_RWR,230,0,1<<4);
  cmdcfg_delay(mco,25000);

  /* Wait for Command Buffer to Execute Before Reading from Status FIFO */
  cmdcfg_bufclr(mco);

  return (0);
}

int_4 d2awgm3_loadcoefs (Module_Config_Obj *module_cfg, int_4 part_ID, int_4 sub_ID, int_4 coeset)
{
  PICSTRUCT *p = module_cfg->TGT.icepic.pdev;
  int_4 mport = module_cfg->TGT.icepic.mport;

  int_4 i;
  int_4 size; 
  real_8 coe_tmp;

  cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,256,0,0xFFFF);

  size = sizeof(d2awgm3_hbfir0)/sizeof(d2awgm3_hbfir0[0]);

  for(i=0;i<size;++i) {
    coe_tmp = (real_8)d2awgm3_hbfir0[i];
    cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,(int_4)coe_tmp,0xFFFF);
  }

  cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,256,1,0xFFFF);

  size = sizeof(d2awgm3_hbfir1)/sizeof(d2awgm3_hbfir1[0]);
  for(i=0;i<size;++i) {
    cmdcfg_ctl(module_cfg,part_ID,sub_ID,PFUNC_RWR,i,d2awgm3_hbfir1[i],0xFFFF);
  } 

  return(0);
}

int_4 d2awgm3_implement(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{
  int_4 rtn,i;
  int_4 extref;
  int_4 refclk = 50e6;
  real_8 ratef;
  int_4 rfgain;
  int_u4 samp_freq;
  real_8 samprate,txlo,rxlo,datfreq;
  int_4  dis_pa;
  int_u4 lna;
  int_4 rxmod;
  int_4 sysreg;
  /*int_4 lmxrate; */
  int_8 lmxrate;
  int_4 dac_clkinv,dac_synct;
  int_4 hbinter,hbistage,hbidbg;
  int_4 hb_prescale,hb_maxclip,hb_minclip;
  int_4 dbgena,trigsel,datasel;
  int_4 specinv,mixpd,dblen;
  int_4 hsmode;
  int_4 mtgo;
  int_4 mtgooffns = 0;
  real_8 mtgooffns_act;
  int_4 mtgooff = 0;
  int_4 mtgooff_, mtgo_fir_ind;
  int_4 awgoutdel;
  int_4 dac_interp,fpga_interp;
  const real_8 DACRATE = 4960.0; /* 4800.0; */
  int_8 actrate,dac_rate,fpga_srate;
  int_4 d2arev,lmxdbg;
  int_8 tdac_rate, tactrate;
  int_4 tdac_interp, tfpga_interp;
  int_4 test_tone;
  UDB rate_state = udb_initializer;
  UDB fpga_rate_state = udb_initializer;
  UDB freq_state = udb_initializer;
  UDB mtgo_state = udb_initializer;
  int_4 polyfir;
  int_4 max_dlatency;
  int_4 mute_offset;

  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){printf("ERROR: IMPLEMENT: Command Module Lock Failed!!\n"); return(-1); }

  /* Reset Module */
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,1,0xFFFF);
  cmdcfg_delay(D2AWGM3_module,500);
  cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0,0xFFFF);

  /* Wait for Command Buffer to Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  d2awgm3_get_state(D2AWGM3_module);

  d2arev  = findintflagdef("D2AREV",p->config,1);
  /* lmxrate = findintflagdef("LMXRATE",p->config,0); */
  lmxrate = (int_8)(finddblflagdef("DACRATE",p->config,0.0)*1e6);
  hsmode  = findintflagdef("QDRX",p->config,0);
  polyfir = hsmode?0:1; /* findintflagdef("POLYFIR",p->config,1); */
  if((hsmode==0) && (rate > MAXFPGARATELS)) {
    printf("!!!WARNING!!! - For Rates > %f MSPS, Enable Flag QDRX.  Setting Rate to %f MSPS\n",(real_8)MAXFPGARATELS/1e6,(real_8)MAXFPGARATELS/1e6);
    rate = 240e6;
  }

  if(lmxrate) {
    /* dac_rate = ((int_8)lmxrate) * 1e6; */
    dac_rate = lmxrate;
    d2awgm3_findrate_fixdac(dac_rate, rate, &actrate, &fpga_srate, &dac_interp, &fpga_interp, hsmode,polyfir);
  } else {
    d2awgm3_findrate(rate, &dac_rate, &actrate, &fpga_srate, &dac_interp, &fpga_interp, hsmode,polyfir);
  }
  hbistage = (int_4)round(log2(fpga_interp));

  dac_interp  = findintflagdef("DACINT",p->config,dac_interp);
  hbinter     = findintflagdef("HBINT",p->config,fpga_interp);
  if(hbinter != fpga_interp) hbistage = (int_4)round(log2(hbinter));
  hbidbg      = findintflagdef("HBDBG",p->config,7);
  hb_prescale = findintflagdef("HBPRE",p->config,0x00FF);
  hb_maxclip  = findintflagdef("HBCMAX",p->config,32700)<<8;
  hb_minclip  = findintflagdef("HBCMIN",p->config,-32700)<<8;
  extref      = findintflagdef("PREFX",p->config,0);
  ratef       = finddblflagdef("CRATE",p->config,320.0);
  rfgain      = findintflagdef("RFGAIN",p->config,0);
  test_tone   = findintflagdef("AWGTONE",p->config,0);
  mtgooffns   = findintflagdef("MTOFF",p->config,0);
  mtgooffns   = findintflagdef("MTGOOFF",p->config,mtgooffns);
  awgoutdel   = findintflagdef("AWGOUTDEL",p->config,-1);

 /* printf("JJB:::dacinter=%d, hbinter=%d, hbistage=%d\n",dac_interp,hbinter,hbistage); */
  if(p->verbose) printf("SRATE = %lu, DACRATE = %lu, FPGARATE = %lu, FPGA_INTERP = %d, DAC_INTERP = %d\n",actrate,dac_rate,fpga_srate,hbinter,dac_interp);

  if(polyfir) {
    if(fpga_interp == 1) mtgo_fir_ind = 0;
    else if(fpga_interp == 2) mtgo_fir_ind = 1;
    else if(fpga_interp == 3) mtgo_fir_ind = 2;
    else if(fpga_interp == 4) mtgo_fir_ind = 3;
    else if(fpga_interp == 8) mtgo_fir_ind = 4;
    else if(fpga_interp == 10) mtgo_fir_ind = 5;
    else if(fpga_interp == 12) mtgo_fir_ind = 6;
    else if(fpga_interp == 16) mtgo_fir_ind = 7;
    else if(fpga_interp == 20) mtgo_fir_ind = 8;
    else if(fpga_interp == 32) mtgo_fir_ind = 9;
    else if(fpga_interp == 48) mtgo_fir_ind = 10;
    else if(fpga_interp == 50) mtgo_fir_ind = 11;
    else if(fpga_interp == 64) mtgo_fir_ind = 12;
    else if(fpga_interp == 72) mtgo_fir_ind = 13;
    else if(fpga_interp == 84) mtgo_fir_ind = 14;
    else if(fpga_interp == 96) mtgo_fir_ind = 15; 
    else if(fpga_interp == 128) mtgo_fir_ind = 16;

    if(awgoutdel >= 0) {
      mtgooff_ = fpga_interp*awgoutdel;
      mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
    }
    else {
      mtgooff = (int_4)((real_8)(mtgooffns)/1e9*fpga_srate+(mtgooffns<0?-0.5:0.5)); /* ns to samples */
      mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
      if(p->verbose) {
        /* printf("MTGO OFFSET = %d ns, FPGA Rate = %lu\n",mtgooffns,fpga_srate); */
        /* printf("MTGO OFFSET = %d samples, FPGA Rate = %lu, Actual MTGO OFFSET = %f ns\n",mtgooff,fpga_srate,mtgooffns_act); */
        printf("Requested MTGO OFFSET = %d ns, Actual MTGO OFFSET = %f ns\n",mtgooffns,mtgooffns_act);
      }
      mtgooff_ = fpga_srate - 131 - d2awgm3_mtgo_tbl_pp[mtgo_fir_ind] + mtgooff;
    }
  } else {
    if(fpga_interp == 1) mtgo_fir_ind = 0;
    else if(fpga_interp == 2) mtgo_fir_ind = 1;
    else if(fpga_interp == 4) mtgo_fir_ind = 2;
    else if(fpga_interp == 8) mtgo_fir_ind = 3;
    else if(fpga_interp == 16) mtgo_fir_ind = 4;
    else if(fpga_interp == 32) mtgo_fir_ind = 5;
    else if(fpga_interp == 64) mtgo_fir_ind = 6;
    else if(fpga_interp == 128) mtgo_fir_ind = 7;

    if(awgoutdel >= 0) {
      mtgooff_ = fpga_interp*awgoutdel;
      mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
    }
    else {
      mtgooff = (int_4)round(((real_8)(mtgooffns)/1e9*(fpga_srate/4)+(mtgooffns<0?-0.5:0.5))); /* ns to samples */
      mtgooffns_act = (real_8)(1e9/fpga_srate)*mtgooff;
      mtgooff_ = (fpga_srate/4) - 23 - d2awgm3_mtgo_tbl[mtgo_fir_ind] + mtgooff;
    }
  }
  if( (findflag("MTGOOFF",p->config)<0) && (findflag("AWGOUTDEL",p->config)<0) ) {
    mtgooff_ = 0;
    mtgooffns_act = 0;
  }
 /* printf("JJB::: POLYFIR = %d, MTGO OFFSET = %d, MTGO FIR OFFSET = %d, FPGA SRATE = %lu\n", polyfir,mtgooff_, d2awgm3_mtgo_tbl[mtgo_fir_ind], fpga_srate); */

  mtgo_state.f64[0] = mtgooffns_act;
  udb_f64_to_i32wf(&mtgo_state);

  cmdcfg_ctl(D2AWGM3_module,BAD_MTGO_ID,0,PFUNC_RWR,0,mtgooff_,0xFFFFFFFF);

  D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOW] = mtgo_state.i32[0]; /* Freq Whole Part */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOF] = mtgo_state.i32[1]; /* Freq Frac Part */ 

  if(polyfir) {
    d2awgm3_polyfilt(D2AWGM3_module, hbinter);
  }else if(hsmode==0) {
    /* Set Prescalar Value */
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,1,hb_prescale,0xFFFF);
    
    /* Set Clip Value */
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,2,hb_maxclip,0xFFFFFFFF);
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,3,hb_minclip,0xFFFFFFFF);
    
    /* Set Oversample Factor to 4 */
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_RWR,0,hbinter,0xFF);
    
    /* Set Output Stage Select to 2 - always log2(OversampFactor) */
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_RWR,1,hbistage,0xFF);
    
    cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_RWR,2,hbidbg,0xFF);
  }

  if(d2arev==1) {
    if(extref) {
      cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,3,PFUNC_PWR,0,0x00,0xFFFF);
      cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,4,PFUNC_PWR,0,0x03,0xFFFF);
    }else {
      cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,3,PFUNC_PWR,0,0x04,0xFFFF);
      cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,4,PFUNC_PWR,0,0x00,0xFFFF);
    }
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,5,PFUNC_PWR,0,0x05,0xFFFF);
    cmdcfg_delay(D2AWGM3_module,25000);
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,5,PFUNC_PWR,0,0x01,0xFFFF);
    cmdcfg_delay(D2AWGM3_module,25000);
  }else {
    d2awgm3_si5344(D2AWGM3_module,ratef,extref);
  }

  datfreq = finddblflagdef("RFFREQ",p->config,160.0)*1e6;

  if(datfreq > 3*(dac_rate/2)) {
    mixpd = 0;
    dblen = 1;
    specinv = (datfreq > 2*dac_rate)? 0 : 1;
    datfreq = abs(2*dac_rate - datfreq);
  } else if(datfreq > (dac_rate/2)) {
    mixpd = 0;
    dblen = 0;
    specinv = (datfreq > dac_rate)? 0 : 1;
    datfreq = abs(dac_rate - datfreq);
  } else {
    mixpd = 1;
    dblen = 0;
    specinv = 0;
  }

  /* printf("JJB::: trying to set rate %ld, %d\n", ((int_8)lmxrate)*1000000,lmxrate); */
  /* d2awgm3_lmx2582(D2AWGM3_module,((int_8)lmxrate)*1000000); */

  /* printf("JJB::: trying to set rate %ld, mixpd = %d\n", dac_rate,mixpd); */
  d2awgm3_lmx2582(D2AWGM3_module,dac_rate,mixpd);

  max5868_mute(D2AWGM3_module);
  max5868_reset(D2AWGM3_module);
  max5868_clkdis(D2AWGM3_module);
  max5868_datdis(D2AWGM3_module);
  max5868_resetn(D2AWGM3_module);
  cmdcfg_delay(D2AWGM3_module,10000);

  dac_clkinv = findintflagdef("DACCLKINV",p->config,0);
  dac_synct  = findintflagdef("DACSYNCT",p->config,1);
  max5868_clkinv(D2AWGM3_module,dac_clkinv);
  max5868_synctype(D2AWGM3_module,dac_synct);
  
  if(dac_interp==24) max_dlatency = 1317;
  else if(dac_interp==20) max_dlatency = 1137;
  else if(dac_interp==16) max_dlatency = 864;
  else if(dac_interp==12) max_dlatency = 764;
  else if(dac_interp==10) max_dlatency = 673;
  else if(dac_interp==8) max_dlatency = 496;
  else if(dac_interp==6) max_dlatency = 420;
  else if(dac_interp==5) max_dlatency = 382;
  else max_dlatency = 288;

  /* mute_offset = (int_4)(((real_8)max_dlatency/(real_8)dac_rate) * fpga_srate); */
  mute_offset = (max_dlatency * fpga_srate)/dac_rate;

/* printf("JJB:: dac_interp = %d, max_dlatency = %d, dac_rate = %ld, fpga_srate = %ld, mute_offset = %d\n",dac_interp,max_dlatency,dac_rate,fpga_srate,mute_offset); */

  /* printf("JJB::: dac_rate = %ld, datfreq = %f, dac_interp = %d\n",dac_rate,datfreq,dac_interp);  */
  rate_state.i64[0] = dac_rate;
  fpga_rate_state.i64[0] = fpga_srate;
  freq_state.f64[0] = d2awgm3_max5868(D2AWGM3_module, dac_rate, datfreq, dac_interp, specinv, mute_offset);
  udb_f64_to_i32wf(&freq_state);

  D2AWGM3_module->LS.i32[D2AWGM3_RG_RATEL] = rate_state.i32[0]; /* Rate Lower 32-bits */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_RATEU] = rate_state.i32[1]; /* Rate Upper 32-bits */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGA_RATEL] = fpga_rate_state.i32[0]; /* FPGA Rate Lower 32-bits */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGA_RATEU] = fpga_rate_state.i32[1]; /* FPGA Rate Upper 32-bits */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW] = freq_state.i32[0]; /* Freq Whole Part */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF] = freq_state.i32[1]; /* Freq Frac Part */ 
  D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGAINT] = fpga_interp; /* FPGA Interpolation Value */ 
 
  cmdcfg_ctl(D2AWGM3_module,BAD_LMH6401_ID,0,PFUNC_RWR,0x02,0x20,0xFFFF);
  cmdcfg_ctl(D2AWGM3_module,BAD_PE43704_ID,0,PFUNC_RWR,0x00,0x30,0xFFFF);

  if(mixpd) {
    /* First Switch */ 
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,0,PFUNC_PWR,0x00,0x00,0xFFFF);

    /* Second Switch */ 
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,1,PFUNC_PWR,0x00,0x01,0xFFFF);

    /* Mixer */
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,2,PFUNC_PWR,0x00,0x00,0xFFFF);
  } else {
    /* First Switch */ 
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,0,PFUNC_PWR,0x00,0x01,0xFFFF);

    /* Second Switch */ 
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,1,PFUNC_PWR,0x00,0x00,0xFFFF);

    /* Mixer */
    cmdcfg_ctl(D2AWGM3_module,BAD_PGEN_ID,2,PFUNC_PWR,0x00,(dblen<<1)|0x01,0xFFFF);
  }

  trigsel = findintflagdef("TRIGSEL",p->config,1);
  datasel = findintflagdef("DATASEL",p->config,2);
  dbgena  = findintflagdef("DBGENA",p->config,0);
  if(dbgena) d2awgm3_dbgsetup(D2AWGM3_module,0,trigsel,datasel);

  udb_i32wf_to_f64(&freq_state);

  d2awgm3_setgain(D2AWGM3_module, rfgain, freq_state.f64[0]);

  D2AWGM3_module->LS.i32[D2AWGM3_RG_GAIN] = rfgain; /* Gain Frac Part */ 
  d2awgm3_set_state(D2AWGM3_module);

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  return (0);
}
 
/*---------------------------------------------------------------------------*/
/* D2AWGM3 - Public Interfaces for Module Initialization, Setup, and Adjust   */
/*---------------------------------------------------------------------------*/
int_4 d2awgm3_enable(PICSTRUCT *p, int_4 mport, int_4 ena)
{

  /* Module Identification */
  int_4 rtn;
  int_4 tss;
  int_4 parI,parQ;
  int_4 hsmode;
  int_4 polyfir;
  int_4 sysreg;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){printf("ERROR: ENABLE: Command Module Lock Failed!!\n"); return(-1); }

  sysreg = 0;
  if (findintflag("RSEL",p->config) > 0) sysreg |= 1<<0; 
  if (findintflag("TSEL",p->config) > 0) sysreg |= 1<<1; 
  if (findintflag("MTGO",p->config) > 0) sysreg |= 1<<3; 
  if (findintflag("QDRX",p->config) > 0) sysreg |= 1<<14;
  if (findintflag("AWGTONE",p->config) > 0) sysreg |= 1<<13; 

  hsmode = findintflagdef("QDRX",p->config,0);
  polyfir = hsmode?0:1; /*findintflagdef("POLYFIR",p->config,1);*/

  tss = findintflagdef("TSS",p->config,0); 
  sysreg |= (tss&0x0F)<<4;

  if (findintflag("CALEN",p->config) > 0) sysreg |= 1<<8; 
  if (findintflag("TLRS",p->config) > 0)  sysreg |= 1<<9; 

  parI = findintflagdef("PARI",p->config,0);
  sysreg |= (parI&0x03)<<12;

  parQ = findintflagdef("PARQ",p->config,0);
  sysreg |= (parQ&0x03)<<14;

  if(ena == 0){
    max5868_mute(D2AWGM3_module);
    if(polyfir) cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,0,0xFFFF);
    cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
    cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x05,0xFF);
    cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x00,0xFF);
  }else {
    /* Configure System Register */
    cmdcfg_ctl(D2AWGM3_module,BAD_MOD_ID,0,PFUNC_PWR,0,sysreg,0xFF);

    /* Enable Module */
    cmdcfg_ctl(D2AWGM3_module,BAD_ROOT_ID,0,PFUNC_PWR,0,0x02,0xFF);

    /* Enable Filter */
    if(polyfir) cmdcfg_ctl(D2AWGM3_module,BAD_HBINTFIR_ID,0,PFUNC_PWR,0,1,0xFFFF);

    max5868_unmute(D2AWGM3_module);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  return(0);
}

int_4 d2awgm3_setup(PICSTRUCT *p, int_4 mport, int_4 dir, int_4 bits, int_4 rate, int_4 gain, int_4 flags)
{ 
  int_4 rtn; 
/*   D2AWGM3_State_Obj DRFST; */

if(p->verbose==3) printf("D2AWGM3_SETUP: Calling Reset\n");
  /* Reset module */
  rtn = d2awgm3_dreset(p,mport);
  if(rtn<0) return(-1);

if(p->verbose==3) printf("D2AWGM3_SETUP: Calling Implement\n");
  /* implement with enable */
  /* d2awgm3_implement(p,mport,dir,bits,rate,gain,flags,&DRFST); */
  d2awgm3_implement(p,mport,dir,bits,rate,gain,flags);

  return (0);
}

int_4 d2awgm3_getkeyl(PICSTRUCT *p, int_4 mport, int_4 key)
{
  /* Module Identification */
  int_4 rtn;
  int_4 keyval;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB key_state = udb_initializer;

  if(key == KEY_RFPWR) return d2awgm3_get_rfpwr(p,mport);

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  if(key == KEY_RFGAIN) {
    d2awgm3_get_state(D2AWGM3_module);
    keyval = D2AWGM3_module->LS.i32[D2AWGM3_RG_GAIN];
  }
  else if(key == KEY_TEMP) {
    keyval = d2awgm3_rdtemp(D2AWGM3_module);
  }
  else if(key == KEY_MGAIN) {
    d2awgm3_get_state(D2AWGM3_module);
    keyval = D2AWGM3_module->LS.i32[D2AWGM3_RG_VGA];
  }
  else if(key == KEY_RFATTN) {
    d2awgm3_get_state(D2AWGM3_module);
    keyval = D2AWGM3_module->LS.i32[D2AWGM3_RG_ATTN];
  }
  else if(key == KEY_CLKSTAT) {    
    keyval = d2awgm3_rdclkstat (D2AWGM3_module);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(keyval);
}

real_8 d2awgm3_getkeyd(PICSTRUCT *p, int_4 mport, int_4 key)
{
  /* Module Identification */
  int_4 rtn;
  real_8 keyval;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  UDB key_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  if(key == KEY_MTOFF) {
    d2awgm3_get_state(D2AWGM3_module);
  
    key_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOW];
    key_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOF];
  
    udb_i32wf_to_f64(&key_state);
    keyval = key_state.f64[0];    
  }
  else if(key == KEY_RFFREQ) {
    d2awgm3_get_state(D2AWGM3_module);
  
    key_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW];
    key_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF];
  
    udb_i32wf_to_f64(&key_state);
    keyval = key_state.f64[0]/1e6;    
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(keyval);
}

int_4 d2awgm3_setkeyl(PICSTRUCT *p, int_4 mport, int_4 key, int_4 i4val)
{
  /* Module Identification */
  int_4 rtn;
  UDB key_state = udb_initializer;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  if(key == KEY_RFGAIN) {
    d2awgm3_get_state(D2AWGM3_module);
    
    key_state.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQW];
    key_state.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FREQF];
    udb_i32wf_to_f64(&key_state);
    d2awgm3_setgain(D2AWGM3_module, i4val, key_state.f64[0]);
    
    D2AWGM3_module->LS.i32[D2AWGM3_RG_GAIN] = i4val; /* Gain Frac Part */ 
    d2awgm3_set_state(D2AWGM3_module);
  }
  else if(key == KEY_RFATTN) {
    cmdcfg_ctl(D2AWGM3_module,BAD_PE43704_ID,0,PFUNC_RWR,0x00,i4val,0xFFFF);
    D2AWGM3_module->LS.i32[D2AWGM3_RG_ATTN]  = i4val; 
    d2awgm3_set_state(D2AWGM3_module);
  }
  else if(key == KEY_MGAIN) {
    cmdcfg_ctl(D2AWGM3_module,BAD_LMH6401_ID,0,PFUNC_RWR,0x02,i4val,0xFFFF);
    D2AWGM3_module->LS.i32[D2AWGM3_RG_VGA]  = i4val; 
    d2awgm3_set_state(D2AWGM3_module);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

int_4 d2awgm3_setkeyd(PICSTRUCT *p, int_4 mport, int_4 key, real_8 r8val)
{
  /* Module Identification */
  int_4 rtn;
  Module_Config_Obj *D2AWGM3_module = (Module_Config_Obj *) NULL;
  TGT_Config_Obj T;
  int_4 fpga_interp,mtgo_fir_ind,mtgooff,mtgooff_;
  real_8 mtgooffns_act;
  UDB fpga_srate = udb_initializer;
  UDB mtgo_state = udb_initializer;

  /* Configure transport and allocate module object */
  T.tgtid = C3_TGT_DEV_ICEPIC;
  T.icepic.pdev  = p;
  T.icepic.mport = mport;
  D2AWGM3_module = module_alloc_obj(FPGA_MODID_D2AWGM3, &T, 0);
  if (D2AWGM3_module == (Module_Config_Obj *)NULL) return (-2);

  pic_lock(p,LOCK_ALLOC);
  rtn = module_lock(D2AWGM3_module, 0);
  if(rtn<0){
    printf("ERROR: SET GAIN: Command Module Lock Failed!!\n"); 
    pic_lock(p,LOCK_FREE);
    return(-1);
   }

  if(key == KEY_MTOFF) {
    d2awgm3_get_state(D2AWGM3_module);
    fpga_interp = D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGAINT];
    fpga_srate.i32[0] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGA_RATEL];
    fpga_srate.i32[1] = D2AWGM3_module->LS.i32[D2AWGM3_RG_FPGA_RATEU];
  
    if(fpga_interp == 1) mtgo_fir_ind = 0;
    else if(fpga_interp == 2) mtgo_fir_ind = 1;
    else if(fpga_interp == 3) mtgo_fir_ind = 2;
    else if(fpga_interp == 4) mtgo_fir_ind = 3;
    else if(fpga_interp == 8) mtgo_fir_ind = 4;
    else if(fpga_interp == 10) mtgo_fir_ind = 5;
    else if(fpga_interp == 12) mtgo_fir_ind = 6;
    else if(fpga_interp == 16) mtgo_fir_ind = 7;
    else if(fpga_interp == 20) mtgo_fir_ind = 8;
    else if(fpga_interp == 32) mtgo_fir_ind = 9;
    else if(fpga_interp == 48) mtgo_fir_ind = 10;
    else if(fpga_interp == 50) mtgo_fir_ind = 11;
    else if(fpga_interp == 64) mtgo_fir_ind = 12;
    else if(fpga_interp == 72) mtgo_fir_ind = 13;
    else if(fpga_interp == 84) mtgo_fir_ind = 14;
    else if(fpga_interp == 96) mtgo_fir_ind = 15; 
    else if(fpga_interp == 128) mtgo_fir_ind = 16;

    mtgooff = (int_4)((real_8)(r8val)/1e9*fpga_srate.i64[0]+(r8val<0?-0.5:0.5)); /* ns to samples */
    mtgo_state.f64[0] = (real_8)(1e9/fpga_srate.i64[0])*mtgooff;
    mtgooff_ = fpga_srate.i64[0] - 131 - d2awgm3_mtgo_tbl_pp[mtgo_fir_ind] + mtgooff;

    cmdcfg_ctl(D2AWGM3_module,BAD_MTGO_ID,0,PFUNC_RWR,0,mtgooff_,0xFFFFFFFF);

    udb_f64_to_i32wf(&mtgo_state);
    D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOW] = mtgo_state.i32[0]; /* Freq Whole Part */ 
    D2AWGM3_module->LS.i32[D2AWGM3_RG_MTGOF] = mtgo_state.i32[1]; /* Freq Frac Part */ 
    d2awgm3_set_state(D2AWGM3_module);
  }
  else if(key == KEY_RFFREQ) {
    d2awgm3_setfreq(D2AWGM3_module, r8val*1e6);
    d2awgm3_set_state(D2AWGM3_module);
  }

  /* Let Command Buffer Execute */
  cmdcfg_bufclr(D2AWGM3_module);

  /* Release Command Lock */
  rtn = module_unlock(D2AWGM3_module, 0);
  module_free_obj(D2AWGM3_module, 0);

  pic_lock(p,LOCK_FREE);

  return(0);
}

