/****************************************************************************/
/*                                                                          */
/*   A2DM14 module interface routines                                       */
/*                                                                          */
/****************************************************************************/
/* A2DM14 Defines */
#define A2DM14_BASE       0x00000400  /* Base configuration reg bit[4:0]=(complex,testclk,testramp,tcxo_pd,enable) */
#define A2DM14_FOFF       0x00000404  /* Base configuration complex output freq offset */
#define A2DM14_STATE      0xFD000000  /* Base address for state structure register (See A2DM14_STATE_* for mapping) */
#define A2DM14_SLC        0x00000100  /* Silicon Labs chip register bit[6:0]=(INC,DEC,CS,LOL,C2B,C1B,NRS) pins */
#define A2DM14_SLC_SPI    0x0000C100  /* Silicon Labs chip (SPI mode=2) */
#define A2DM14_ADC        0x00000200  /* A to D chip register bit[0:0]=(MODE) pins */
#define A2DM14_ADC_SPI    0x00008200  /* A to D chip (SPI mode=1) */
#define A2DM14_PGA        0x00000300  /* Prog. Gain Amp register bit[7:0]=(NPD,GS,B[5:0]) pins */
#define A2DM14_FTR        0x00001000  /* Fine Time Register */
#define A2DM14_PPSR       0x00002000  /* 1 PPS Rate */

#define A2DM14_STATE_TRUE      0x01
#define A2DM14_STATE_FALSE     0x00
#define A2DM14_STATE_ON        0x01
#define A2DM14_STATE_OFF       0x00

typedef struct {
    int_u4 chksum;    /* 32 bit checksum over packet ensure packet integrity */
    int_4  opsize;    /* size of operations queue to implement current state */
    real_8 samprate;  /* A/D sample rate in Hz                               */
    int_4  refclk;    /* Reference clock rate in Hz                          */
    int_4  foffset;   /* complex rotation in output mode                     */
    int_1  enable;    /* Module enable/disable flag {TRUE,FALSE}             */
    int_1  bits;      /* Module output bits {-16,-8,8,16}                    */
    int_1  pga;       /* Gain of programmable gain amplifier {-100,20}       */
    int_1  tcxo;      /* TCXO power state {ON,OFF}                           */
    int_1  test;      /* A2D test waveform                                   */
    int_1  opts;      /* Options mask                                        */
    int_1  dual;      /* DDR output mode                                     */
    int_1  sysreg;    /* System Register                                     */
} A2DM14_State;

#define A2DM14_SIZE sizeof(A2DM14_State)

int_4 a2dm14_dump_state (PICSTRUCT *p, int_4 mport, A2DM14_State *ADST)
{ 
  /* If debug output is on, display unfiltered state info */
  if (p->verbose == 3) {
    printf("ADST.chksum:    %8.8X  \n",ADST->chksum);
    printf("ADST.opsize  :  %i     \n",ADST->opsize);
    printf("ADST.refclk  :  %i     \n",ADST->refclk);
    printf("ADST.samprate:  %17.6f \n",ADST->samprate);
    printf("ADST.foffset:   %8.8x  \n",ADST->foffset);
    printf("ADST.enable:    %2.2X  \n",ADST->enable);
    printf("ADST.bits:      %i     \n",ADST->bits);   
    printf("ADST.pga:       %i     \n",ADST->pga);   
    printf("ADST.tcxo:      %2.2X  \n",ADST->tcxo);  
    printf("ADST.test:      %2.2X  \n",ADST->test);  
    printf("ADST.opts:      %2.2X  \n",ADST->opts);  
    printf(" \n");
  }
  return 0;
}

int_4 a2dm14_cfg_clocks (PICSTRUCT *p, int_4 mport, A2DM14_State *ADST)
{
  Si5326_Reg_Obj Si5326_Reg;
  int_u1 regadd, regval;
  int_4  i,ii;

  /* Si5326 Ref-Clock and A/D clock configuration    */
  Si5326_Reg.debuglevel = -1;
  Si5326_Reg.fref   = (double)ADST->refclk;
  Si5326_Reg.dclk1  = ADST->samprate;   
  Si5326_Reg.dclk2  = ADST->samprate;  
  Si5326_Reg.dwght1 = 1.0;
  Si5326_Reg.dwght2 = 1.0;   
  ii = si5326_regcfg_init(&Si5326_Reg,SI5326_REGCFG_MT_A2DM14);
  ii = si5326_regcfg_bwsel(&Si5326_Reg,7);

  if (ADST->opts & RFOPT_XADCLK) {
    /* Configure Si5326 for bypass */
    si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_1);
    si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_BYP);
  }
  else {
    /* Configure Si5326 to generate desired A/D clock rate */
    if (ADST->opts & RFOPT_XREF) {
      si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_1);
    }
    else {
      si5326_regcfg_clkin(&Si5326_Reg,SI5326_REGCFG_CLKIN_2);
    }
    ii = si5326_regcfg_dividers(&Si5326_Reg);
    if (ii != 0) {
      ADST->samprate = Si5326_Reg.fclk2;
      printf("A2DM14 Warning: A/D sample rate not possible. Using %f\n", Si5326_Reg.fclk2);
    }
  }
  si5326_regcfg_debug(&Si5326_Reg);

  /* Reset the Si5326 */
  iom_wrb (p,mport,A2DM14_SLC,0x00);
  iom_dly(p,mport,10000);
  iom_wrb (p,mport,A2DM14_SLC,0x01);

  if (ADST->opts & RFOPT_XADCLK) {
    iom_wrb(p,mport,(A2DM14_SLC_SPI|0),0x22); 
  }
  else

  /* Now setup the Si5326 chip registers */
  for(i=0; i<Si5326_Reg.numreg; i++) {
    regadd = ((Si5326_Reg.regval[i] & 0x0000FF00)>>8);
    regval =  (Si5326_Reg.regval[i] & 0x000000FF);
    iom_wrb(p,mport,(A2DM14_SLC_SPI|regadd),regval); 
    /* ICAL could take up to 53msec worst case */
    if (Si5326_Reg.regval[i] == 0x00008840) {
      ii = iom_assert(p,mport,A2DM14_SLC_SPI|0x82,1,0x00,200000);
      if (ii<0) printf("A2DM14 Warning: Si5326 synthesizer not locked. Check external clock.\n");
    }
  }
  iom_dly (p,mport,10000); /* Reset DCM */
  iom_wrb (p,mport,A2DM14_SLC,0x01);

  return (0);
}

int_4 a2dm14_cfg_pga (PICSTRUCT *p, int_4 mport, A2DM14_State *ADST)
{ 
  /* Gain Block chip register bit[7:0]=(NPD,N/A,B[5:0]) pins - GS strobe now pulsed on reg write */
  int_u1 regval = 0x00;
  real_8 didx; 
  int_4 gain = ADST->pga;

  if (gain <= -100) {
    /* Turn off input amplifier output */
    iom_wrb (p,mport,A2DM14_PGA,0x00);
    ADST->pga = -100;
    vprint("A2DM14: Turn PGA870 OFF \n");
  }
  else if ((gain >= -11) && (gain <= 20)) {
    /* Convert gain to integer index for PGA870 gain */
    didx = (real_8) gain;
    didx = 2.0*(didx + 11.5);
    didx = (didx >= 0.0) ? (didx) : (0.0);
    didx = (didx <= 63.0) ? (didx) : (63.0);
    regval = (int_u1) didx;
    regval = regval | 0x80;
    iom_wrb (p,mport,A2DM14_PGA,regval);
    vprint("A2DM14: PGA870 gain set %i  %f  0x%2.2X \n",gain,didx,regval);
    ADST->pga = gain;
  }
  else {
    /* Invalid gain requested - return error status */
    vprint("A2DM14: Invalid gain request %i \n",gain);
    return (-1);
  }
  return (0);
}
  
int_4 a2dm14_cfg_ads61b49 (PICSTRUCT *p, int_4 mport, A2DM14_State *ADST)
{
  int_u1 regadd, regval;
  double rate;

  rate = ADST->samprate;
  if (ADST->bits<0) rate *= 2;

  /* Reset the device */
  regadd = 0x00; regval=0x80;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Select high/low speed mode */
  regadd = 0x20; regval = (rate >= 75000000.0) ? 0x00 : 0x04;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Remove standby */
  regadd = 0x3F; regval=0x00;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Select LVDS data output */
  regadd = 0x41; regval=0x80;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Select clock edge position */
  regadd = 0x44; regval=0xD8;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Select output data format (2s compliment) */
  regadd = 0x50; regval=0x04;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Select output data content */
  regadd = 0x51; regval=0x00;            /* Upper 8-bits custom pat */
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 
  regadd = 0x52; regval=0x00;            /* Lower 6-bits custom pat */
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 
  /* 0x00=Normal, 0x01=AllZeros 0x02=AllOnes, 0x04=Ramp, 0x05=CustomPat */
  regadd = 0x62; regval=ADST->test;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* ADC Fine Gain (0-6dB) and DC offset correction time constant */
  regadd = 0x55; regval=0x08;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* DC offset pedestal */                      
  regadd = 0x63; regval=0x00;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  /* Enable / disable DC offset correction */
  regadd = 0x53; regval=(ADST->opts&RFOPT_DCS)?0x40:0x00;
  iom_wrb(p,mport,(A2DM14_ADC_SPI|regadd),regval); 

  return (0);
}

int_4 a2dm14_implement (PICSTRUCT *p, int_4 mport, A2DM14_State *ADST, int_4 flags)
{ 
  int_4 sysreg; 

  /* consistency and range checks */
  ADST->samprate = max(1000000.0,min(300000000.0,ADST->samprate));
  if (ADST->opts&RFOPT_XADCLK) ADST->opts &= ~RFOPT_XREF;
  ADST->tcxo = (ADST->opts&(RFOPT_XREF|RFOPT_XADCLK))? 0 : 1;

  /* system register */
  sysreg = 0;
  if (ADST->bits<0)  sysreg |= 0x10; /* complex output */
  if (iom_getcfgb(p,mport,"DCMTEST")) sysreg |= 0x08; /* DCM version of half speed clock */
  if (ADST->test!=0) sysreg |= 0x04; /* internal test ramp */
  if (ADST->tcxo==0) sysreg |= 0x02; /* power down TCXO */
  if (iom_getcfgb(p,mport,"MTGO")) sysreg |= 0x20;
  if (ADST->dual!=0) sysreg |= 0x40; /* DDR output mode */
  if (iom_getcfgb(p,mport,"LBTEST")) sysreg |= 0x80; /* LoopBack  Test */
  iom_wrb (p,mport,A2DM14_BASE,sysreg);
  iom_wr (p,mport,A2DM14_FOFF,ADST->foffset,4);

  /* A/D clock configuration    */   
  a2dm14_cfg_clocks(p,mport,ADST);

  /* programmable gain configuration     */   
  a2dm14_cfg_pga(p,mport,ADST);

  /* ADS61B49 configuration     */   
  a2dm14_cfg_ads61b49(p,mport,ADST);

  if (ADST->enable)  sysreg |= 0x01; /* enable transport */
  iom_wrb (p,mport,A2DM14_BASE,sysreg);
  ADST->sysreg = sysreg;

  /* update state */
  iom_exec (p,mport);

  return (0);
}
 
/*---------------------------------------------------------------------------*/
/* A2DM14 - Public Interfaces for Module Initialization, Setup, and Adjust   */

int_4 a2dm14_setup (PICSTRUCT *p, int_4 mport, int_4 flags)
{ 
  A2DM14_State *ADST;
  int_4 i,bits,gain; double d,rate;

  if (mport==3) return a2dm14_setup(p,1,flags) + a2dm14_setup(p,2,flags);

  iom_init(p,mport);

  bits = iom_getcfgl(p,mport,"BITS",16);
  rate = iom_getcfgd(p,mport,"RATE",100000000.0);
  gain = iom_getcfgl(p,mport,"GAIN",0);

  /* Initialize state information on module */
  ADST = (A2DM14_State*)iom_alloc_state(p,mport,A2DM14_SIZE);
  ADST->refclk	= 10000000;
  ADST->foffset	= 0x80000000;
  ADST->enable	= A2DM14_STATE_FALSE;
  ADST->tcxo	= A2DM14_STATE_ON;
  ADST->bits 	= bits; 
  ADST->pga	= gain; 
  ADST->test	= iom_getcfgl(p,mport,"A2DTEST",0);
  ADST->samprate = (bits<0)? rate*2 : rate;
  ADST->opts	= iom_getcfgm(p,mport,"A2DOPTS",0,RFOPTLIST);
  ADST->dual	= (flags&FLG_DUAL) || (bits<0);

  /* adjust center freq of hilbert output ? */
  d = iom_getcfgd(p,mport,"MFREQ",-1.0);
  if (d != -1.0) { 
    d /= (-ADST->samprate/4); 
    if (d> 1.0) d-=2.0;
    if (d<-1.0) d+=2.0;
    ADST->foffset = R2L(d);
  }

  /* setup base register modes - disable module output */
  iom_wrb (p,mport,A2DM14_BASE,0);

  /* Determine if external or internal reference is to be used */
  i = iom_getcfgl(p,mport,"PREFX",-1);
  if (i > 0) ADST->opts |= RFOPT_XREF;
  if (i > 100) ADST->refclk = i; /* in Hz */
  else if (i > 1) ADST->refclk = i*1000000; /* in MHz */

  /* Determine if external clock is to be applied */
  if (iom_getcfgx(p,mport,"MUXCLK=N")) ADST->opts |= RFOPT_XADCLK;
  if (iom_getcfgx(p,mport,"MUXCLK=A")) {
    if (mport==1) ADST->opts |= RFOPT_XADCLK;
    else printf ("Cannot use MUXCLK=A with A2DM14 on Module=2\n");
  }
  if (iom_getcfgx(p,mport,"MUXCLK=B")) {
    if (mport==2) ADST->opts |= RFOPT_XADCLK;
    else printf ("Cannot use MUXCLK=B with A2DM14 on Module=1\n");
  }

  /* implement with enable */
  ADST->enable = 0;
  a2dm14_implement(p,mport,ADST,-1);
  iom_set_state(p,mport);

  return (0);
}

int_4 a2dm14_enable (PICSTRUCT *p, int_4 mport, int_4 ena) {
  A2DM14_State* ADST = (A2DM14_State*) iom_get_state (p,mport);
  ADST->enable = ena;
  if (ena) ADST->sysreg |= 0x01;
  else     ADST->sysreg &= 0xFE;
  iom_wrb (p,mport,A2DM14_BASE,ADST->sysreg);
  iom_set_state (p,mport);
  return 0;
}

int_4 a2dm14_get_clkstat (PICSTRUCT *p, int_4 mport) {
  int_4 i,stat=0;
  i=pic_jpmemrd (p,mport,A2DM14_PPSR,1);
  if (i>0x55 && i<0x70) stat |= 0x2;	/* 1PPS valid */
  return stat;
}

int_4 a2dm14_getkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data) {
  int_4 stat=0,status=-1;
  int_4* lval=(int_4*)data;
  real_8* dval=(real_8*)data;
  A2DM14_State* ADST = (A2DM14_State*) iom_get_state (p,mport);
  switch (key) {
    case KEY_GAIN:
    *lval = (int_4)ADST->pga;
    status = 4;
    break;
    case KEY_OPTS:
    *lval = (int_4)ADST->opts;
    status = 4;
    break;
    case KEY_CLKSTAT:
    *lval = a2dm14_get_clkstat(p,mport);
    break;
  }
  return status;
}

int_4 a2dm14_setkey (PICSTRUCT *p, int_4 mport, int_4 key, void *data) {
  int_4 stat=0,status=-1;
  int_4* lval=(int_4*)data;
  real_8* dval=(real_8*)data;
  A2DM14_State* ADST = (A2DM14_State*) iom_get_state (p,mport);
  switch (key) {
    case KEY_GAIN: 
    ADST->pga = *lval;
    stat = a2dm14_cfg_pga(p,mport,ADST);
    if (stat>=0) status=4;
    break;
    case KEY_OPTS:
    ADST->opts = *lval;
    if (ADST->opts&RFOPT_ENABLE) a2dm14_implement(p,mport,ADST,0);
    status=4;
    break;
  }
  iom_set_state (p,mport);
  return status;
}

